# A script which demonstrates how R can be used to plot the results.
# See also the shell script "build" in the same directory.
#
# Usage:
#   R --vanilla --slave < plot.R
#
# Requirements:
#   R: http://www.r-project.org/
#
# Tested with R version 2.4.1 (2006-12-18).

legend.p.values <- c("0.0001", "0.001", "0.01", "0.1")
colors <- c("#f7faf7", "#eff7ef", "#e7f3e7", "#dfefdf")
borders <- c("#000000", "#000000", "#000000", "#000000")
ltys <- c("solid", "solid", "solid", "12")
lwds <- c(0.2, 0.2, 0.2, 1.0)

columnstotal <- 11
columns <- c(2:4,6, 7,9:11)  # Omit the 0.05 level

legend <- sapply(
    legend.p.values,
    function(p) eval(bquote(expression(italic(p) < .(p))))
)
levels <- length(legend)

read.input <- function(file) {
    raw.data <- matrix(
        scan(file=file, quiet=TRUE),
        ncol=columnstotal, byrow=TRUE
    )
    items <- raw.data[,1]
    bounds <- raw.data[,columns]

    list(items=items, bounds=bounds, maxitems=max(items), maxbounds=max(bounds))
}

get.polygon <- function(input, level) {
    # Both upper and lower bound for this level
    x <- list(low=input$items, up=rev(input$items))
    y <- list(low=input$bounds[,level], up=rev(input$bounds[,2*levels+1-level]))

    # Remove redundant points
    for (curve in 1:2) {
        l <- length(y[[curve]])
        skip <- which(
            y[[curve]][c(-1,-2)] == y[[curve]][c(-1,-l)] &
            y[[curve]][c(-1,-l)] == y[[curve]][c(-l+1, -l)]
        ) + 1
        if (length(skip) > 0) {
            x[[curve]] <- x[[curve]][-skip]
            y[[curve]] <- y[[curve]][-skip]
        }
    }

    # Close the polygon
    list(x=c(x$low, x$up, x$low[1]), y=c(y$low, y$up, y$low[1]))
}

plot.area <- function(input, step, level) {
    p <- get.polygon(input, level)
    if (step == "fill") {
        polygon(p$x, p$y, col=colors[level], border=NA)
    } else if (step == "line") {
        polygon(p$x, p$y, col=NA, border=borders[level], lty=ltys[level], lwd=lwds[level])
    } else {
        stop("invalid step")
    }
}

plot.areas <- function(input, step) {
    for (level in 1:levels) {
        plot.area(input, step, level)
    }
}

plot.prepare <- function(outfile, input) {
    postscript(outfile, width=5, height=5, horizontal=FALSE, onefile=FALSE, paper="special", pointsize=12)
    op <- par(no.readonly=TRUE)
    par(mar=c(2.5, 2.5, 0.5, 1.5))
    plot(0., 0., xlim=c(0, input$maxitems), ylim=c(0, input$maxbound), type='n', main=NA, xlab=NA, ylab=NA)
    op
}

plot.finish <- function(op) {
    par(op)
    invisible(dev.off())
}

plot.legend <- function(input) {
    x0 <- 0.99 * input$maxitems
    y0 <- 0.01 * input$maxbound
    w <- max(strwidth(legend))
    h <- max(strheight(legend))
    xmargin <- 1.0 * strwidth("x")
    ymargin <- 0.3 * h
    wfull <- w + 2*xmargin
    hfull <- h + 2*ymargin
    x <- x0 - wfull
    for (i in 0:levels) {
        y <- y0 + hfull * i
        if (i > 0) {
            rect(x, y, x + wfull, y + hfull, col=colors[i], border=NA)
        } else {
            rect(x, y, x + wfull, y + hfull, col="#ffffff", border=NA)
        }
    }
    for (i in 1:levels) {
        y <- y0 + hfull * i
        lines(c(x, x + wfull), c(y, y), type='l', col=borders[i], lty=ltys[i], lwd=lwds[i])
        text(x + xmargin, y - (h + ymargin), legend[i], adj=c(0,0))
    }
}

process.file <- function(file) {
    input <- read.input(file)
    op <- plot.prepare(sprintf("%s.eps", file), input)
    plot.legend(input)
    plot.areas(input, "fill")
    plot.areas(input, "line")
    plot.finish(op)
}

process.file("type")
process.file("hapax")
