# The MIT License (MIT) # Copyright (c) 2014-2017 University of Bristol
#
#  Permission is hereby granted, free of charge, to any person obtaining a copy
#  of this software and associated documentation files (the "Software"), to deal
#  in the Software without restriction, including without limitation the rights
#  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
#  copies of the Software, and to permit persons to whom the Software is
#  furnished to do so, subject to the following conditions:
#
#  The above copyright notice and this permission notice shall be included in all
#  copies or substantial portions of the Software.
#
#  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
#  MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
#  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
#  DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
#  OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE
#  OR OTHER DEALINGS IN THE SOFTWARE.

import unittest
import sys

from hyperstream import TimeInterval
from hyperstream.utils import datetime2unix
from .helpers import *

RANDOM_VALUES = {
    'betavariate': [
        0.039981834097823, 0.981304494343183, 0.103808554963057, 0.138446289876271,
        0.155847790152019, 0.936165436117980, 0.691777297709685, 0.202711794550541,
        0.439187581262989, 0.854411325237021, 0.012871422905004, 0.445711666122735,
        0.173612011072429, 0.294213597937781, 0.571754488117056, 0.414207690302475,
        0.952829471212172, 0.554866330696849, 0.853597747841655, 0.726548827146225,
        0.434566859684085, 0.845728292811518, 0.755650779991056, 0.967643717137684,
        0.855246132463383, 0.622668240929598, 0.715418188650215, 0.110575048120036,
        0.975782622203252, 0.911339685810993, 0.770410055493041, 0.494632012614062,
        0.214519001820389, 0.245253365269930, 0.615395718030917, 0.768849947883717,
        0.143943873499559, 0.553601557863770, 0.280901407530050, 0.772551703947893,
        0.782327973677141, 0.486333485714099, 0.750803107845182, 0.102266594101819,
        0.068535570781350, 0.914774771182072, 0.586769119311220, 0.233320009269012,
        0.233797285521539, 0.276737112045214, 0.218306114082117, 0.162816644080913,
        0.953114601742251, 0.666718604507751, 0.754286487729222, 0.633736098912436,
        0.745322580574953, 0.775601376437185, 0.869843144466489, 0.652589692264996
    ],
    'expovariate': [
        3.394823806874659, 0.581127564523488, 0.007519672058314, 2.418848860875040,
        2.801300959473452, 0.872818427870305, 1.113411706270757, 0.087671478631061,
        1.454491546927796, 0.270247967319917, 0.031298757464487, 1.554820555740570,
        0.424783969801393, 0.976256988533290, 0.956632880871021, 0.160819948534567,
        0.202227141819855, 0.121504542781544, 0.014726687798474, 0.666995224752983,
        3.349598620046617, 0.066740710553357, 0.778897212763763, 0.627169493658980,
        0.919956064910344, 0.093134261821545, 0.865128825812668, 0.314143102991765,
        0.812905426912728, 1.034607693670155, 0.655921477976220, 0.438875802168443,
        0.286552206139466, 2.710786411749832, 0.604016087463631, 0.755365636542648,
        0.019488234358730, 0.709483750170804, 0.005797007645201, 0.155214410916815,
        0.640226369456586, 0.473766605605160, 0.055697926496315, 0.885588186216424,
        0.179130522055541, 0.814931243906700, 0.155772001995568, 2.769506831287718,
        1.473945818640017, 3.145001077306131, 0.152251542816020, 0.364408641013038,
        0.040394611099021, 0.324046924239758, 1.642542394521078, 0.195215975982500,
        0.167910514134237, 3.094860156976556, 0.167887905027271, 1.795100810202958
    ],
    'gammavariate': [
        0.034122056303517, 0.819316938516747, 4.893990231207050, 0.093238767965875,
        0.062653368633307, 0.540893888677075, 0.398146730647281, 2.477674145470838,
        0.265945456658999, 1.440498095730080, 3.479785441733883, 0.237277063974075,
        1.061059410972392, 0.472757056386849, 0.484807556477113, 1.906802450982214,
        1.697773907743712, 2.167940835057879, 4.225448242167752, 0.720001469650587,
        0.035729192259713, 2.740124924523351, 0.614173008703897, 0.763787353084732,
        0.508389510898236, 2.419918892177294, 0.546448162063484, 1.310869670605297,
        0.586209300594369, 0.439072201829716, 0.731812395588524, 1.034964061541275,
        1.389691639897196, 0.068797714057458, 0.791006964997359, 0.634574154700323,
        3.947672654322900, 0.677073210230625, 5.153310521407328, 1.939551415551206,
        0.749025865807436, 0.974588922379475, 2.915532064817145, 0.531830509662681,
        1.807869194387956, 0.584597328850929, 1.936237033639271, 0.064744318636450,
        0.260092883032779, 0.044021772573587, 1.957381341407605, 1.186156762423377,
        3.229188209498897, 1.284518963073033, 0.215035876124057, 1.729669369589433,
        1.867104880870748, 0.046338583263660, 1.867228551311800, 0.181654626260094
    ],
    'gauss': [
        01.054219641927238, -0.225557255750686, 02.197040548376180, 00.103491789727369,
        01.226197529624492, -0.492012676750713, -0.198114993904518, -0.368908398457307,
        00.075994371191587, -0.731246053108685, 01.730469790846852, 00.339286625801242,
        -0.793301565330494, 01.150298484529457, -0.423484998531862, -0.377227190281404,
        00.201191163960159, 00.450034666561860, 01.150116230720591, 00.105939158666782,
        00.356502892758491, -0.079919387895188, -1.082857059855480, -0.285936313257175,
        -0.346814969321444, -0.256881102258723, -0.696979366413084, -0.377499627520273,
        -1.348994444204497, -0.499429050862792, -0.930241287708008, 00.111367638837469,
        00.012404354114906, 02.328394072209139, -1.176780706969193, 00.354849884867363,
        01.182456826105767, 00.144094950423267, 00.556793983695921, 00.020230708184767,
        -0.959260150198715, 00.165387954371103, 01.254495799595116, 00.444315947531775,
        00.656742549085982, 01.094783865442602, 01.451280543224706, 01.852781273500150,
        00.329648919441743, -2.486228819823942, 00.539083396938225, 00.661971580335303,
        00.780264495057645, 00.198194768427309, 00.217236306069111, -0.585867168640169,
        01.403963278880950, 02.053924883121816, 01.069438836743639, 01.564129852304573
    ],
    'lognormvariate': [
        4.182149649506653, 6.072683633963685, 1.378906052287930, 1.820308315387552,
        0.496143003914426, 1.262823784820494, 0.541232922428520, 0.197427904210337,
        1.141079547621132, 1.210524252689620, 1.203885159410419, 1.313149056265397,
        0.950794910474111, 0.843500437187102, 0.187032058314804, 0.927866832248184,
        0.156570492107917, 0.271953010190621, 0.412265059596156, 0.335500854109232,
        1.894937540967587, 0.277297951845252, 0.675726657696681, 2.817759974006620,
        2.502088419825137, 0.622939011683825, 2.152480312223666, 0.452792026701372,
        0.882873194607882, 1.349444237752270, 0.869975688840489, 2.111560218917180,
        0.409725015624480, 1.879234230032750, 2.606256173070972, 6.229504939045913,
        0.906235497770029, 0.828016527006029, 0.682915284714766, 1.012706478986566,
        0.367098765722139, 0.278910825237462, 0.428440501893463, 1.004907017932176,
        0.185565283388187, 1.090536358789101, 0.261012404269340, 0.162505658588734,
        1.124937908343475, 2.828519961139446, 0.720248041371186, 17.481108541304152,
        2.289963461381308, 1.681143365809301, 0.424940732580820, 1.668947117554719,
        2.447378048054388, 2.258635444051391, 0.653795316276048, 0.388972244244800
    ],
    'normalvariate': [
        01.430825384561959, 01.803800621694465, 00.321290469070466, 00.599005890800551,
        -0.700891079468200, 00.233350312510460, -0.613905552258328, -1.622381803257973,
        00.131974785909151, 00.191053532461377, 00.185553959787939, 00.272428112308738,
        -0.050456896398401, -0.170194858695000, -1.676475241982295, -0.074867056225568,
        -1.854248941609945, -1.302125984262414, -0.886088787958202, -1.092130776793514,
        00.639185878079636, -1.282662712448799, -0.391966637231399, 01.035942233839799,
        00.917125751078616, -0.473306659555168, 00.766620810808078, -0.792322361123261,
        -0.124573696156307, 00.299692831949558, -0.139290011585451, 00.747427114481840,
        -0.892269037898180, 00.630864369412055, 00.957914775126571, 01.829296865604353,
        -0.098456075401950, -0.188722164643868, -0.381384461196793, 00.012626429070964,
        -1.002124350723792, -1.276863171055819, -0.847603402642404, 00.004895017760377,
        -1.684348526836401, 00.086669647496766, -1.343187346852594, -1.817042455734101,
        00.117727841549639, 01.039753592922873, -0.328159624394767, 02.861120785911318,
        00.828535861710167, 00.519474136815591, -0.855805572533170, 00.512192959069697,
        00.895017267032754, 00.814760845114164, -0.424960948565885, -0.944247289465118
    ],
    'paretovariate': [
        29.809400800717510, 01.7880534401352315, 1.0075480157929015, 11.232921214436221,
        16.466054512384030, 02.3936476787662757, 3.0447284136688280, 1.0916294395161450,
        04.282305560549401, 01.3102893193840228, 1.0317937139180504, 4.7342369172954900,
        01.529260017630925, 02.6545017930933260, 2.6029173687084950, 1.1744734841034520,
        01.224126026748782, 01.1291944955974207, 1.0148356597400270, 1.9483740899647228,
        28.491295513537708, 01.0690182570835713, 2.1790678917276303, 1.8723035050106858,
        02.509180146459839, 01.0976090923717308, 2.3753120677749830, 1.3690856427811218,
        02.254448616097609, 02.8140020682287330, 1.9269173113984270, 1.5509626492274813,
        01.331827695661731, 15.0410993623286920, 1.8294513029924269, 2.1283895981144103,
        01.019679369608408, 02.0329414820037095, 1.0058138428094914, 1.1679083465608209,
        01.896910233245502, 01.6060321043903114, 1.0572782597459870, 2.4244099767209266,
        01.196176861442035, 02.2590203455962850, 1.1685597434253272, 15.950765651183254,
        04.366430336403290, 23.2197004879965800, 1.1644531094097856, 1.4396623990728115,
        01.041221570722970, 01.3827121883154585, 5.1682926620010985, 1.2155734928205928,
        01.182830759277237, 22.0841500072887450, 1.1828040168323900, 6.0200815768637320
    ],
    'randint2': [
        5, 3, 1, 5, 5, 3, 4, 1, 4, 2, 1, 4, 2, 4, 4, 1, 1, 1, 1, 3, 5, 1, 3, 3, 4, 1, 3, 2, 3, 4,
        3, 2, 2, 5, 3, 3, 1, 3, 1, 1, 3, 2, 1, 3, 1, 3, 1, 5, 4, 5, 1, 2, 1, 2, 5, 1, 1, 5, 1, 5
    ],
    'randint3': [
        4, 1, 1, 1, 5, 1, 1, 1, 3, 2, 1, 1, 1, 3, 5, 4, 5, 4, 2, 1, 2, 1, 1, 5, 4, 2, 1, 5, 4, 1,
        5, 1, 5, 5, 1, 3, 5, 1, 3, 4, 3, 3, 2, 2, 4, 5, 1, 5, 1, 1, 3, 2, 4, 4, 2, 1, 1, 5, 4, 2
    ],
    'random': [
        0.966453535692138, 0.440732599175352, 0.007491470058587, 0.910975962449124, 0.939268997363764,
        0.582227573058949, 0.671563481487985, 0.083938226837084, 0.766480932791796, 0.236809775363117,
        0.030814021726610, 0.788772717236283, 0.346088965597123, 0.623281475039168, 0.615815695103615,
        0.148554638708287, 0.183090647409931, 0.114412969688687, 0.014618780486909, 0.486751540604758,
        0.964901560916215, 0.064562280977186, 0.541088185551130, 0.465898559008309, 0.601463449561051,
        0.088928829990662, 0.579002686187366, 0.269585503819448, 0.556432560556215, 0.644634234178282,
        0.481036371366518, 0.355239147442983, 0.249152121361209, 0.933515498042346, 0.453388019476493,
        0.530161206911590, 0.019299566309716, 0.508101925779792, 0.005780237417743, 0.143768427595595,
        0.472826925347406, 0.377347440772596, 0.054175198646142, 0.587528508131069, 0.164003223741961,
        0.557330237441468, 0.144245721601908, 0.937307084696224, 0.770979971519774, 0.956933122349405,
        0.141227764416499, 0.305392708287698, 0.039589624227961, 0.276783694791692, 0.806512505115603,
        0.177343035278254, 0.154570514710789, 0.954718655702394, 0.154551400089043, 0.833889294151231
    ],
    'randrange2': [
        18, 14, 10, 18, 18, 14, 16, 10, 16, 12, 10, 16, 12, 16, 16, 10, 10, 10, 10, 14, 18, 10, 14, 14, 16,
        10, 14, 12, 14, 16, 14, 12, 12, 18, 14, 14, 10, 14, 10, 10, 14, 12, 10, 14, 10, 14, 10, 18, 16, 18,
        10, 12, 10, 12, 18, 10, 10, 18, 10, 18
    ],
    'randrange3': [
        16, 10, 10, 10, 18, 10, 10, 10, 14, 12, 10, 10, 10, 14, 18, 16, 18, 16, 12, 10, 12, 10, 10, 18, 16,
        12, 10, 18, 16, 10, 18, 10, 18, 18, 10, 14, 18, 10, 14, 16, 14, 14, 12, 12, 16, 18, 10, 18, 10, 10,
        14, 12, 16, 16, 12, 10, 10, 18, 16, 12
    ],
    'triangular': [
        4.682762875874238, 3.626159769226909, 2.212011368448777, 4.483209798223082,
        4.573159271028750, 3.869054691108233, 4.007372398360772, 2.709668486705239,
        4.163007048043646, 3.191997756784259, 2.429981546533871, 4.203958639082648,
        3.441018318267585, 3.933827513051516, 3.922210750833969, 2.944101600596951,
        3.048114442443949, 2.828539569442598, 2.296163270716432, 3.708949748713680,
        4.675507600626219, 2.622393513673718, 3.801812729810393, 3.671942389572636,
        3.899679103787349, 2.730460799731220, 3.863871271607618, 3.271814854024236,
        3.827182356344678, 3.966673690541899, 3.698887350061536, 3.459943452554892,
        3.222666237436551, 4.553397821464158, 3.649341722281638, 3.783526630434640,
        2.340290167149009, 3.746027363668379, 2.186229494190525, 2.928768305646555,
        3.684328219820720, 3.504687557147854, 2.570132609027809, 3.877543887313001,
        2.991977490899752, 3.828655633149338, 2.930308728117419, 4.566319534782433,
        4.171109123321607, 4.640555104429366, 2.920525168856885, 3.353645540651684,
        2.487378441632136, 3.288682338185075, 4.238119113868061, 3.031531973168803,
        2.963028082801710, 4.631429744970086, 2.962968535588914, 4.294073574976609
    ],
    'uniform': [
        4.899360607076416, 3.322197797526058, 2.022474410175761, 4.732927887347373, 4.817806992091292,
        3.746682719176847, 4.014690444463955, 2.251814680511251, 4.299442798375389, 2.710429326089353,
        2.092442065179829, 4.366318151708850, 3.038266896791369, 3.869844425117505, 3.847447085310846,
        2.445663916124863, 2.549271942229794, 2.343238909066063, 2.043856341460727, 3.460254621814275,
        4.894704682748647, 2.193686842931558, 3.623264556653390, 3.397695677024928, 3.804390348683154,
        2.266786489971987, 3.737008058562099, 2.808756511458345, 3.669297681668646, 3.933902702534848,
        3.443109114099555, 3.065717442328950, 2.747456364083627, 4.800546494127040, 3.360164058429480,
        3.590483620734771, 2.057898698929151, 3.524305777339376, 2.017340712253229, 2.431305282786786,
        3.418480776042219, 3.132042322317789, 2.162525595938428, 3.762585524393208, 2.492009671225883,
        3.671990712324404, 2.432737164805725, 4.811921254088673, 4.312939914559324, 4.870799367048216,
        2.423683293249498, 2.916178124863096, 2.118768872683885, 2.830351084375076, 4.419537515346811,
        2.532029105834762, 2.463711544132369, 4.864155967107185, 2.463654200267129, 4.501667882453695
    ],
    'vonmisesvariate': [
        3.357856054826835, 5.157489756582599, 4.479739182645804, 1.907971117523622,
        5.055141604545795, 6.107096891938155, 2.915387721029356, 5.460182454227138,
        1.116271426329938, 4.960737570079505, 5.887441356819019, 0.981316492756079,
        6.274362166448599, 0.632521528607341, 6.028546694550388, 3.418698935619459,
        0.060489178998973, 0.276154477688722, 6.043683015273760, 5.632838807299208,
        2.045567387571900, 0.834502386051994, 0.763471240922435, 5.805628236451655,
        0.755672107602623, 2.112922851993352, 4.920322771565935, 4.749565803242515,
        3.723684053158859, 4.352031602899314, 5.614316866936048, 6.225137698980708,
        5.766737554488466, 5.873170966841112, 0.316277234794306, 2.436297088323581,
        0.600638552593339, 1.532669183971546, 0.488588701880799, 0.903538039802433,
        0.547776055749545, 6.176392770219052, 1.428120214925881, 6.053689328434808,
        1.672377583279894, 0.640266865453971, 5.282504365451881, 1.467161497219122,
        6.140657913249537, 4.016127529818116, 0.628520977491434, 1.141278277115692,
        5.517591788868223, 5.783409984284813, 5.543137983515277, 5.603426529843089,
        5.666252112039157, 5.331163516794262, 1.405021421913882, 4.657615032038910
    ],
    'weibullvariate': [
        3.394823806874659, 0.581127564523488, 0.007519672058314, 2.418848860875040,
        2.801300959473452, 0.872818427870305, 1.113411706270757, 0.087671478631061,
        1.454491546927796, 0.270247967319917, 0.031298757464487, 1.554820555740570,
        0.424783969801393, 0.976256988533290, 0.956632880871021, 0.160819948534567,
        0.202227141819855, 0.121504542781544, 0.014726687798474, 0.666995224752983,
        3.349598620046617, 0.066740710553357, 0.778897212763763, 0.627169493658980,
        0.919956064910344, 0.093134261821545, 0.865128825812668, 0.314143102991765,
        0.812905426912728, 1.034607693670155, 0.655921477976220, 0.4388758021684434,
        0.286552206139466, 2.710786411749832, 0.604016087463631, 0.755365636542648,
        0.019488234358730, 0.709483750170804, 0.005797007645201, 0.155214410916815,
        0.640226369456586, 0.473766605605160, 0.055697926496315, 0.885588186216424,
        0.179130522055541, 0.814931243906700, 0.155772001995568, 2.769506831287718,
        1.473945818640017, 3.145001077306131, 0.152251542816020, 0.364408641013038,
        0.040394611099021, 0.324046924239758, 1.642542394521078, 0.195215975982500,
        0.167910514134237, 3.094860156976556, 0.167887905027271, 1.795100810202958
    ],
    'custom': [
        -0.987982323681744, -0.403745561951705, 00.551693007468676, 00.999907570085015, 00.528811724075207,
        -0.428471182309132, -0.991819659674552, -0.643293715945895, 00.296673503522449, 00.963880472032213,
        00.744900179718102, -0.158937902545648, -0.916649210188621, -0.831597461328594, 00.018021158368760,
        00.851071208170706, 00.901650314096472, 00.123256279415406, -0.768459010134729, -0.953656629697290,
        -0.262066741929040, 00.670466099786067, 00.986575501370702, 00.395631936821145, -0.559053805891616,
        -0.999748057676344, -0.521278555807829, 00.436452046271176, 00.992910649810199, 00.636491780955790,
        -0.305114695977134, -0.966200128537194, -0.738965618780344, 00.167670472968601, 00.920151105122217,
        00.826649054720701, -0.026870324303591, -0.855685251082012, -0.897787104210346, -0.114467634085048,
        00.774092850923502, 00.950955938705069, 00.253514521999197, -0.677006977090617, -0.985091383620956,
        -0.387487315031860, 00.566370804008219, 00.999510217795927, 00.513704546819593, -0.444398715432779,
        -0.993923848165933, -0.629639978609997, 00.313531983546438, 00.968444085957101, 00.732973161947531,
        -0.176389906877676, -0.923580908783280, -0.821635882465121, 00.035717385023463, 00.860232253440635
    ]
}

SEA_ICE_SUMS = [
    15.48, 16.15, 17.26, 18.21, 19.38, 18.61, 18.87, 18.80, 20.38, 20.82,
    17.96, 15.88, 15.16, 15.89, 17.89, 19.21, 19.47, 18.78, 19.10, 18.84,
    21.40, 20.54, 17.67, 15.66, 15.30, 15.80, 17.67, 18.90, 19.79, 19.62,
    19.43, 19.41, 21.45, 20.78, 17.66, 15.28, 14.84, 15.86, 17.77, 19.15,
    19.48, 18.80, 18.48, 19.14, 21.07, 21.21, 17.90, 16.06, 15.63, 16.60,
    18.22, 19.72, 20.25, 19.88, 19.49, 19.60, 21.15, 21.29, 18.38, 16.34,
    15.42, 16.02, 17.67, 18.99, 19.23, 18.52, 18.77, 18.80, 19.38, 20.33,
    18.05, 15.50, 14.59, 15.39, 17.51, 19.79, 20.78, 20.15, 19.45, 20.03,
    21.34, 19.84, 17.00, 14.95, 15.02, 15.47, 17.24, 18.99, 19.26, 19.07,
    19.03, 19.33, 20.40, 20.49, 18.03, 15.64, 15.51, 16.14, 17.80, 19.17,
    19.36, 18.94, 18.59, 19.00, 21.10, 20.58, 17.46, 15.53, 15.08, 15.98,
    18.13, 19.60, 19.92, 19.46, 18.82, 18.75, 20.90, 20.61, 17.55, 15.01,
    14.65, 15.70, 17.74, 19.65, 19.94, 19.20, 19.26, 19.29, 20.80, 20.52,
    17.67, 15.82, 15.52, 16.87, 18.21, 19.24, 19.48, 18.71, 18.53, 18.82,
    19.83, 19.90, 16.65, 15.00, 15.11, 15.65, 16.84, 18.20, 18.73, 18.76,
    17.88, 17.98, 19.93, 20.24, 18.05, 16.22, 15.76, 16.52, 17.86, 19.17,
    20.12, 19.14, 18.10, 18.06, 19.71, 19.86, 17.77, 15.88, 15.15, 16.03,
    17.41, 18.95, 19.98, 19.50, 18.72, 18.86, 20.38, 20.84, 17.67, 14.86,
    14.18, 15.33, 17.28, 18.70, 18.94, 18.44, 18.00, 18.79, 19.63, 20.42,
    16.63, 14.24, 14.00, 14.38, 16.17, 17.47, 18.83, 18.50, 18.69, 19.06,
    20.32, 19.57, 16.38, 14.66, 14.41, 14.97, 16.37, 18.15, 18.54, 17.55,
    17.05, 18.07, 18.71, 20.03, 18.39, 15.80, 15.13, 16.79, 18.56, 19.34,
    19.78, 18.50, 17.39, 17.20, 19.40, 20.51, 17.75, 15.24, 14.71, 15.98,
    18.40, 19.40, 19.83, 18.66, 18.08, 18.34, 19.04, 19.62, 17.04, 14.75,
    14.55, 15.69, 17.31, 18.67, 19.49, 18.81, 18.32, 17.26, 19.39, 19.81,
    16.69, 14.25, 14.12, 14.58
]


def rng_helper(hs, ticker, ti, tool_name, **kwargs):
    random = hs.channel_manager.memory.get_or_create_stream(tool_name)
    random.purge()
    tool = getattr(hs.plugins.data_generators.tools, tool_name)
    tool(**kwargs).execute(sources=[], sink=random, interval=ti, alignment_stream=ticker)
    values = random.window().values()
    print(values)
    # tester.assertListEqual(values, RANDOM_VALUES[tool_name])

    # some values are different between python 2.x and python 3.x
    if tool_name not in RANDOM_VALUES:
        if sys.version_info[0] < 3:
            rvs = RANDOM_VALUES[tool_name + '2']
        else:
            rvs = RANDOM_VALUES[tool_name + '3']
    else:
        rvs = RANDOM_VALUES[tool_name]

    assert_all_close(values, rvs, 1e-15)
    return random


class TestTools(unittest.TestCase):
    def run(self, result=None):
        with resource_manager() as resource:
            self.hs = resource
            super(TestTools, self).run(result)

    def test_data_generators(self):
        ti = TimeInterval(t1, t1 + minute)

        # Create a clock stream to align the random numbers to
        ticker = self.hs.channel_manager.memory.get_or_create_stream("ticker")
        try:
            self.hs.tools.clock().execute(sources=[], sink=ticker, interval=ti)
        except AttributeError:
            raise

        # Test random number generators
        rng_helper(self.hs, ticker, ti, "betavariate", alpha=1.0, beta=1.0, seed=1234)
        rng_helper(self.hs, ticker, ti, "expovariate", lambd=1.0, seed=1234)
        rng_helper(self.hs, ticker, ti, "gammavariate", alpha=1.0, beta=1.0, seed=1234)
        rng_helper(self.hs, ticker, ti, "gauss", seed=1234)
        rng_helper(self.hs, ticker, ti, "lognormvariate", mu=0.0, sigma=1.0, seed=1234)
        rng_helper(self.hs, ticker, ti, "normalvariate", mu=0.0, sigma=1.0, seed=1234)
        rng_helper(self.hs, ticker, ti, "paretovariate", alpha=1.0, seed=1234)
        rng_helper(self.hs, ticker, ti, "randint", a=1, b=5, seed=1234)
        rng_helper(self.hs, ticker, ti, "random", seed=1234)
        rng_helper(self.hs, ticker, ti, "randrange", start=10, stop=20, step=2, seed=1234)
        rng_helper(self.hs, ticker, ti, "triangular", low=2, high=5, mode=4, seed=1234)
        rng_helper(self.hs, ticker, ti, "uniform", a=2, b=5, seed=1234)
        rng_helper(self.hs, ticker, ti, "vonmisesvariate", mu=0.0, kappa=1.0, seed=1234)
        rng_helper(self.hs, ticker, ti, "weibullvariate", alpha=1.0, beta=1.0, seed=1234)

        # Test custom random function
        import math
        rng_helper(self.hs, ticker, ti, "custom", func=lambda dt: math.sin(datetime2unix(dt)))

    def test_combine_generators(self):
        with HyperStream(file_logger=False, console_logger=False, mqtt_logger=None) as hs:
            ti = TimeInterval(t1, t1 + minute)

            # Create a clock stream to align the random numbers to
            ticker = self.hs.channel_manager.memory.get_or_create_stream("ticker")
            hs.tools.clock().execute(sources=[], sink=ticker, interval=ti)

            gauss = rng_helper(hs, ticker, ti, "gauss", seed=1234)
            import math
            custom = rng_helper(hs, ticker, ti, "custom", func=lambda dt: math.sin(datetime2unix(dt)))

            merged = self.hs.channel_manager.memory.get_or_create_stream("merged")
            hs.tools.aligned_merge().execute(sources=[gauss, custom], sink=merged, interval=ti)

            summed = self.hs.channel_manager.memory.get_or_create_stream("summed")
            hs.tools.list_sum().execute(sources=[merged], sink=summed, interval=ti)

            self.assertListEqual(
                summed.window().values(),
                list(map(sum, zip(gauss.window().values(), custom.window().values())))
            )

    def test_data_importers(self):
        with HyperStream(file_logger=False, console_logger=False, mqtt_logger=None) as hs:
            reader = hs.plugins.data_importers.tools.csv_reader('plugins/data_importers/data/sea_ice.csv')
            # noinspection PyTypeChecker
            ti = TimeInterval(datetime(1990, 1, 1).replace(tzinfo=UTC), datetime(2011, 4, 1).replace(tzinfo=UTC))

            # TODO: More complicated tests, including headers, different delimiters, messy data etc etc.
            sea_ice = hs.channel_manager.memory.get_or_create_stream("sea_ice")

            reader.execute(sources=[], sink=sea_ice, interval=ti)

            sea_ice_sums = hs.channel_manager.mongo.get_or_create_stream("sea_ice_sums")
            hs.tools.list_sum().execute(sources=[sea_ice], sink=sea_ice_sums, interval=ti)

            # print(sea_ice_sums.window().values())

            # TODO: the below assertion is causing travis to fail - why?
            # assert_all_close(sea_ice_sums.window().values(), list(map(sum, sea_ice.window().values())), 1e-5)
