/*************************************************************
 *
 *  Copyright (c) 2019-2023 The MathJax Consortium
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * @fileoverview  Implements a subclass of ContextMenu specific to MathJax
 *
 * @author dpvc@mathjax.org (Davide Cervone)
 */

import {MathItem} from '../../core/MathItem.js';
import {JaxList} from './Menu.js';

import {ContextMenu, SubMenu, Submenu, Menu, Item} from './mj-context-menu.js';

/*==========================================================================*/

/**
 * The subclass of ContextMenu that handles the needs of the MathJax
 *   contextual menu (in particular, tying it to a MathItem).
 */
export class MJContextMenu extends ContextMenu {

  /**
   * Static map to hold methods for re-computing dynamic submenus.
   * @type {Map<string, (menu: MJContextMenu, sub: Submenu)}
   */
  public static DynamicSubmenus: Map<string, (menu: MJContextMenu, sub: Submenu) => SubMenu> = new Map();

  /**
   * The MathItem that has posted the menu
   */
  public mathItem: MathItem<HTMLElement, Text, Document> = null;

  /**
   * The error message for the current MathItem
   */
  public errorMsg: string = '';

  /**
   * The jax object from the parent menu item
   */
  protected jax: JaxList;

  /*======================================================================*/

  /**
   * Before posting the menu, set the name for the ShowAs and CopyToClipboard menus,
   *   enable/disable the semantics check item, and set the dynamic submenus.
   *
   * @override
   */
  public post(x?: any, y?: number) {
    if (this.mathItem) {
      if (y !== undefined) {
        this.getOriginalMenu();
        this.getSemanticsMenu();
        this.getSpeechMenu();
        this.getSvgMenu();
        this.getErrorMessage();
        this.dynamicSubmenus();
      }
      super.post(x, y);
    }
  }

  /**
   * Clear the stored MathItem when the menu is removed
   *
   * @override
   */
  public unpost() {
    super.unpost();
    this.mathItem = null;
  }

  /*======================================================================*/

  /**
   * Find an item in the menu (recursively descending into submenus, if needed)
   *
   * @param {string[]} names   The menu IDs to look for
   * @returns {Item}         The menu item (or null if not found)
   */
  public findID(...names: string[]): Item {
    let menu = this as Menu;
    let item = null as Item;
    for (const name of names) {
      if (menu) {
        item = menu.find(name);
        menu = (item instanceof Submenu ? item.submenu : null);
      } else {
        item = null;
      }
    }
    return item;
  }

  /*======================================================================*/

  /**
   * @param {JaxList} jax   The jax being maintained by the parent Menu item
   */
  public setJax(jax: JaxList) {
    this.jax = jax;
  }

  /*======================================================================*/

  /**
   * Set up original-form menu
   */
  protected getOriginalMenu() {
    const input = this.mathItem.inputJax.name;
    const original = this.findID('Show', 'Original');
    original.content = (input === 'MathML' ? 'Original MathML' : input + ' Commands');
    const clipboard = this.findID('Copy', 'Original');
    clipboard.content = original.content;
  }

  /**
   * Enable/disable the semantics settings item
   */
  protected getSemanticsMenu() {
    const semantics = this.findID('Settings', 'semantics');
    this.mathItem.inputJax.name === 'MathML' ? semantics.disable() : semantics.enable();
  }

  /**
   * Enable/disable the speech menus
   */
  protected getSpeechMenu() {
    const speech = this.mathItem.outputData.speech;
    this.findID('Show', 'Speech')[speech ? 'enable' : 'disable']();
    this.findID('Copy', 'Speech')[speech ? 'enable' : 'disable']();
  }

  /**
   * Enable/disable the svg menus
   */
  protected getSvgMenu() {
    const svg = this.jax.SVG;
    this.findID('Show', 'SVG')[svg ? 'enable' : 'disable']();
    this.findID('Copy', 'SVG')[svg ? 'enable' : 'disable']();
  }

  /**
   * Get any error message and enable/disable the menu items for it
   */
  protected getErrorMessage() {
    const children = this.mathItem.root.childNodes[0].childNodes;
    let disable = true;
    this.errorMsg = '';
    if (children.length === 1 && children[0].isKind('merror')) {
      const attributes = children[0].attributes;
      this.errorMsg = (attributes.get('data-mjx-error') || attributes.get('data-mjx-message') || '') as string;
      disable = !this.errorMsg;
    }
    this.findID('Show', 'Error')[disable ? 'disable' : 'enable']();
    this.findID('Copy', 'Error')[disable ? 'disable' : 'enable']();
  }

  /*======================================================================*/

  /**
   * Renews the dynamic submenus.
   */
  public dynamicSubmenus() {
    for (const [id, method] of MJContextMenu.DynamicSubmenus) {
      const menu = this.find(id) as Submenu;
      if (!menu) continue;
      const sub = method(this, menu);
      menu.submenu = sub;
      if (sub.items.length) {
        menu.enable();
      } else {
        menu.disable();
      }
    }
  }

}
