MODULE letkf_tools
!=======================================================================
!
! [PURPOSE:] Module for LETKF with SPEEDY
!
! [HISTORY:]
!   01/26/2009 Takemasa Miyoshi  created
!
!=======================================================================
!
! HOOPE-EnKF-RTC
! by Y.Sawada 20230914
!
  USE common
  USE common_mpi
  USE common_speedy
  USE common_mpi_speedy
  USE common_letkf
  USE letkf_obs

  IMPLICIT NONE

  PRIVATE
  PUBLIC ::  das_letkf

  INTEGER,SAVE :: nobstotal

  REAL(r_size),PARAMETER :: cov_infl_mul = -1.01d0 !multiplicative inflation
  REAL(r_size),PARAMETER :: cov_infl_mul_para = 3.00d0 !multiplicative inflation
! > 0: globally constant covariance inflation
! < 0: 3D inflation values input from a GPV file "infl_mul.grd"
!TVS  LOGICAL,PARAMETER :: msw_vbc = .FALSE.

! change by Y.Saw including QTHR as 3d variables
  REAL(r_size),PARAMETER :: var_local(nv3d+nv2d,nid_obs) = RESHAPE( &
!           U      V      T      Q   QTHR     PS   RAIN
   & (/ 1.0d0, 1.0d0, 1.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0,  & ! U
   &    1.0d0, 1.0d0, 1.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0,  & ! V
   &    1.0d0, 1.0d0, 1.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0,  & ! T
   &    1.0d0, 1.0d0, 1.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0,  & ! Q
   &    1.0d0, 1.0d0, 1.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0,  & ! RH
   &    1.0d0, 1.0d0, 1.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0,  & ! PS
   &    1.0d0, 1.0d0, 1.0d0, 0.0d0, 1.0d0, 1.0d0, 1.0d0,  & ! RAIN
   &    0.0d0, 0.0d0, 0.0d0, 0.0d0, 0.0d0, 0.0d0, 0.0d0 /)& ! QTHR
   & ,(/nv3d+nv2d,nid_obs/))
  INTEGER,SAVE :: var_local_n2n(nv3d+nv2d)

CONTAINS
!-----------------------------------------------------------------------
! Data Assimilation
!-----------------------------------------------------------------------
SUBROUTINE das_letkf(gues3d,gues2d,anal3d,anal2d)
  IMPLICIT NONE
  CHARACTER(12) :: inflfile='infl_mul.grd'
  REAL(r_size),INTENT(INOUT) :: gues3d(nij1,nlev,n_ens,nv3d) ! background ensemble
  REAL(r_size),INTENT(INOUT) :: gues2d(nij1,n_ens,nv2d)      !  output: destroyed
  REAL(r_size),INTENT(OUT) :: anal3d(nij1,nlev,n_ens,nv3d) ! analysis ensemble
  REAL(r_size),INTENT(OUT) :: anal2d(nij1,n_ens,nv2d)
  REAL(r_size),ALLOCATABLE :: mean3d(:,:,:)
  REAL(r_size),ALLOCATABLE :: mean2d(:,:)
  REAL(r_size),ALLOCATABLE :: hdxf(:,:)
  REAL(r_size),ALLOCATABLE :: rdiag(:)
  REAL(r_size),ALLOCATABLE :: rloc(:)
  REAL(r_size),ALLOCATABLE :: dep(:)
  REAL(r_size),ALLOCATABLE :: work3d(:,:,:)
  REAL(r_size),ALLOCATABLE :: work2d(:,:)
  REAL(r_sngl),ALLOCATABLE :: work3dg(:,:,:,:)
  REAL(r_sngl),ALLOCATABLE :: work2dg(:,:,:)
  REAL(r_size),ALLOCATABLE :: logpfm(:,:)
  REAL(r_size) :: parm
  REAL(r_size) :: parm_dummy = 1.0d0 !YSaw for RTC 20230914
  REAL(r_size) :: obserr_para = 2.12d0 !YSaw for RTC 20230914
  REAL(r_size) :: obs_para = 9.93d0 !YSaw for RTC 20230914
  REAL(r_size) :: trans(n_ens,n_ens,nv3d+nv2d)
  LOGICAL :: ex
  INTEGER :: ij,ilev,n,m,i,j,k,nobsl,ierr
  LOGICAL,PARAMETER :: RTCswitch = .FALSE.

  WRITE(6,'(A)') 'Hello from das_letkf'
  nobstotal = nobs !+ ntvs
  WRITE(6,'(A,I8)') 'Target observation numbers : NOBS=',nobs!,', NTVS=',ntvs
  !
  ! In case of no obs
  !
  IF(nobstotal == 0) THEN
    WRITE(6,'(A)') 'No observation assimilated'
    anal3d = gues3d
    anal2d = gues2d
    RETURN
  END IF
  !
  ! Variable localization
  !
  var_local_n2n(1) = 1
  DO n=2,nv3d+nv2d
    DO i=1,n
      var_local_n2n(n) = i
      IF(MAXVAL(ABS(var_local(i,:)-var_local(n,:))) < TINY(var_local)) EXIT
    END DO
  END DO
  WRITE(6,*) 'var_local_n2n ', var_local_n2n

  !
  ! multiplicative inflation
  !
  IF(cov_infl_mul > 0.0d0) THEN ! fixed multiplicative inflation parameter
    ALLOCATE( work3d(nij1,nlev,nv3d) )
    ALLOCATE( work2d(nij1,nv2d) )
    work3d = cov_infl_mul
    work3d(:,:,iv3d_qthr) = cov_infl_mul_para ! YSaw different inflation for parameters
    work2d = cov_infl_mul
    work3d(:,nlev,:) = 1.00d0
  END IF
  IF(cov_infl_mul <= 0.0d0) THEN ! 3D parameter values are read-in
    ALLOCATE( work3dg(nlon,nlat,nlev,nv3d) )
    ALLOCATE( work2dg(nlon,nlat,nv2d) )
    ALLOCATE( work3d(nij1,nlev,nv3d) )
    ALLOCATE( work2d(nij1,nv2d) )
    INQUIRE(FILE=inflfile,EXIST=ex)
    IF(ex) THEN
      IF(myrank == 0) THEN
        WRITE(6,'(A,I3.3,2A)') 'MYRANK ',myrank,' is reading.. ',inflfile
        CALL read_grd4(inflfile,work3dg,work2dg)
      END IF
      CALL scatter_grd_mpi(0,work3dg,work2dg,work3d,work2d)
    ELSE
      WRITE(6,'(2A)') '!!WARNING: no such file exist: ',inflfile
      work3d = -1.0d0 * cov_infl_mul
      work2d = -1.0d0 * cov_infl_mul
    END IF
  END IF
  !
  ! FCST PERTURBATIONS
  ! Move this here after inflation factors are determined. YSaw 20230914
  !
  !work3d (:,:,iv3d_qthr) = cov_infl_mul_para ! YSaw different inflation for parameters
  IF(RTCswitch) THEN
    DO ilev=1,2
      CALL hoopeot(gues3d(:,ilev,:,iv3d_qthr),obs_para,obserr_para,nij1,work3d(:,ilev,iv3d_qthr)) ! RTC YSaw 20230914
    END DO
  !ELSE ! RTC without modifying mean
  !  DO ilev=1,2
  !    CALL hoopeot_nomeanchange(gues3d(:,ilev,:,iv3d_qthr),obs_para,obserr_para,nij1,work3d(:,ilev,iv3d_qthr)) ! RTC YSaw 20230914
  !  END DO
  ENDIF
  ALLOCATE(mean3d(nij1,nlev,nv3d))
  ALLOCATE(mean2d(nij1,nv2d))
  CALL ensmean_grd(n_ens,nij1,gues3d,gues2d,mean3d,mean2d)
  DO m=1,n_ens
    gues3d(:,:,m,:) = gues3d(:,:,m,:) - mean3d
    gues2d(:,m,:) = gues2d(:,m,:) - mean2d
  END DO
  !
  ! p_full for background ensemble mean
  !
  ALLOCATE(logpfm(nij1,nlev))
  CALL calc_pfull(nij1,1,mean2d(:,iv2d_ps),logpfm)
  logpfm = DLOG(logpfm)
  !
  ! MAIN ASSIMILATION LOOP
  !
  ALLOCATE( hdxf(1:nobstotal,1:n_ens),rdiag(1:nobstotal),rloc(1:nobstotal),dep(1:nobstotal) )
  DO ilev=1,nlev
    WRITE(6,'(A,I3)') 'ilev = ',ilev
    DO ij=1,nij1
      DO n=1,nv3d
        IF(n == iv3d_qthr .AND. RTCswitch) THEN ! Regression-to-Climatology for parameters
          CALL obs_local(ij,ilev,n,hdxf,rdiag,rloc,dep,nobsl,logpfm)
          CALL letkf_core(n_ens,nobstotal,nobsl,hdxf,rdiag,rloc,dep,parm_dummy,trans(:,:,n)) !no inflation
          parm_dummy = 1.0d0 ! YSaw 20230914
          work3d(ij,ilev,n) = work3d(ij,ilev,iv3d_u)
        ELSE ! the other state variables
          IF(var_local_n2n(n) < n) THEN
            trans(:,:,n) = trans(:,:,var_local_n2n(n))
            work3d(ij,ilev,n) = work3d(ij,ilev,var_local_n2n(n))
          ELSE
            CALL obs_local(ij,ilev,n,hdxf,rdiag,rloc,dep,nobsl,logpfm)
            parm = work3d(ij,ilev,n)
            CALL letkf_core(n_ens,nobstotal,nobsl,hdxf,rdiag,rloc,dep,parm,trans(:,:,n))
            work3d(ij,ilev,n) = parm
          END IF
        END IF
        DO m=1,n_ens
          anal3d(ij,ilev,m,n) = mean3d(ij,ilev,n)
          DO k=1,n_ens
            anal3d(ij,ilev,m,n) = anal3d(ij,ilev,m,n) &
              & + gues3d(ij,ilev,k,n) * trans(k,m,n)
          END DO
        END DO
      END DO
      IF(ilev >= 5) THEN !no analysis for upper-level Q
        DO m=1,n_ens
          anal3d(ij,ilev,m,iv3d_q) = mean3d(ij,ilev,iv3d_q) &
                                 & + gues3d(ij,ilev,m,iv3d_q)
        END DO
      END IF
      !IF(lat1(ij) <= -20.0d0 .OR. lat1(ij) >= 20.0d0)THEN
      !  DO m=1,n_ens
      !    anal3d(ij,ilev,m,iv3d_qthr) = mean3d(ij,ilev,iv3d_qthr) &
      !                           & + gues3d(ij,ilev,m,iv3d_qthr)
      !  END DO
      !ENDIF
      IF(RTCswitch) THEN
      IF(ilev >= 3) THEN !no analysis for upper-level QTHR
        DO m=1,n_ens
          anal3d(ij,ilev,m,iv3d_qthr) = mean3d(ij,ilev,iv3d_qthr) &
                                 & + gues3d(ij,ilev,m,iv3d_qthr)
        END DO
      END IF
      ELSE ! RTC is not called, qthr is not adjusted Y.Saw 20230922
        DO m=1,n_ens
          anal3d(ij,ilev,m,iv3d_qthr) = mean3d(ij,ilev,iv3d_qthr) &
                                 & + gues3d(ij,ilev,m,iv3d_qthr)
        END DO
      ENDIF
      IF(ilev == 1) THEN !update 2d variable at ilev=1
        DO n=1,nv2d
          IF(var_local_n2n(nv3d+n) <= nv3d) THEN
            trans(:,:,nv3d+n) = trans(:,:,var_local_n2n(nv3d+n))
            work2d(ij,n) = work2d(ij,var_local_n2n(nv3d+n))
          ELSE IF(var_local_n2n(nv3d+n) < nv3d+n) THEN
            trans(:,:,nv3d+n) = trans(:,:,var_local_n2n(nv3d+n))
            work2d(ij,n) = work2d(ij,var_local_n2n(nv3d+n)-nv3d)
          ELSE
            CALL obs_local(ij,ilev,nv3d+n,hdxf,rdiag,rloc,dep,nobsl,logpfm)
            parm = work2d(ij,n)
            CALL letkf_core(n_ens,nobstotal,nobsl,hdxf,rdiag,rloc,dep,parm,trans(:,:,nv3d+n))
            work2d(ij,n) = parm
          END IF
          DO m=1,n_ens
            anal2d(ij,m,n)  = mean2d(ij,n)
            DO k=1,n_ens
              anal2d(ij,m,n) = anal2d(ij,m,n) + gues2d(ij,k,n) * trans(k,m,nv3d+n)
            END DO
          END DO
        END DO
      END IF
    END DO
  END DO
  DEALLOCATE(hdxf,rdiag,rloc,dep)
  IF(cov_infl_mul < 0.0d0) THEN
    CALL gather_grd_mpi(0,work3d,work2d,work3dg,work2dg)
    IF(myrank == 0) THEN
      WRITE(6,'(A,I3.3,2A)') 'MYRANK ',myrank,' is writing.. ',inflfile
      CALL write_grd4(inflfile,work3dg,work2dg)
    END IF
    DEALLOCATE(work3dg,work2dg,work3d,work2d)
  END IF

  DEALLOCATE(logpfm,mean3d,mean2d)
  RETURN
END SUBROUTINE das_letkf
!-----------------------------------------------------------------------
! Project global observations to local
!     (hdxf_g,dep_g,rdiag_g) -> (hdxf,dep,rdiag)
!-----------------------------------------------------------------------
SUBROUTINE obs_local(ij,ilev,nvar,hdxf,rdiag,rloc,dep,nobsl,logpfm)
  IMPLICIT NONE
  INTEGER,INTENT(IN) :: ij,ilev,nvar
  REAL(r_size),INTENT(IN) :: logpfm(nij1,nlev)
  REAL(r_size),INTENT(OUT) :: hdxf(nobstotal,n_ens)
  REAL(r_size),INTENT(OUT) :: rdiag(nobstotal)
  REAL(r_size),INTENT(OUT) :: rloc(nobstotal)
  REAL(r_size),INTENT(OUT) :: dep(nobstotal)
  INTEGER,INTENT(OUT) :: nobsl
  REAL(r_size) :: minlon,maxlon,minlat,maxlat,dist,dlev
  REAL(r_size) :: tmplon,tmplat,tmperr,tmpwgt(nlev)
  INTEGER :: tmpqc
  INTEGER,ALLOCATABLE:: nobs_use(:)
  INTEGER :: imin,imax,jmin,jmax,im,ichan
  INTEGER :: n,nn,tvnn,iobs
!
! INITIALIZE
!
  IF( nobs > 0 ) THEN
    ALLOCATE(nobs_use(nobs))
  END IF
!
! data search
!
  minlon = lon1(ij) - dlon_zero(ij)
  maxlon = lon1(ij) + dlon_zero(ij)
  minlat = lat1(ij) - dlat_zero
  maxlat = lat1(ij) + dlat_zero
  IF(maxlon - minlon >= 360.0d0) THEN
    minlon = 0.0d0
    maxlon = 360.0d0
  END IF

  DO jmin=1,nlat-2
    IF(minlat < lat(jmin+1)) EXIT
  END DO
  DO jmax=1,nlat-2
    IF(maxlat < lat(jmax+1)) EXIT
  END DO
  nn = 1
!TVS  tvnn = 1
  IF(minlon >= 0 .AND. maxlon <= 360.0) THEN
    DO imin=1,nlon-1
      IF(minlon < lon(imin+1)) EXIT
    END DO
    DO imax=1,nlon-1
      IF(maxlon < lon(imax+1)) EXIT
    END DO
    IF( nobs > 0 ) &
    & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
  ELSE IF(minlon >= 0 .AND. maxlon > 360.0) THEN
    DO imin=1,nlon-1
      IF(minlon < lon(imin+1)) EXIT
    END DO
    maxlon = maxlon - 360.0d0
    IF(maxlon > 360.0d0) THEN
      imin = 1
      imax = nlon
      IF( nobs > 0 ) &
      & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
    ELSE
      DO imax=1,nlon-1
        IF(maxlon < lon(imax+1)) EXIT
      END DO
      IF(imax > imin) THEN
        imin = 1
        imax = nlon
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
      ELSE
        imin = 1
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
        DO imin=1,nlon-1
          IF(minlon < lon(imin+1)) EXIT
        END DO
        imax = nlon
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
      END IF
    END IF
  ELSE IF(minlon < 0 .AND. maxlon <= 360.0d0) THEN
    DO imax=1,nlon-1
      IF(maxlon < lon(imax+1)) EXIT
    END DO
    minlon = minlon + 360.0d0
    IF(minlon < 0) THEN
      imin = 1
      imax = nlon
      IF( nobs > 0 ) &
      & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
    ELSE
      DO imin=1,nlon-1
        IF(minlon < lon(imin+1)) EXIT
      END DO
      IF(imin < imax) THEN
        imin = 1
        imax = nlon
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
      ELSE
        imin = 1
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
        DO imin=1,nlon-1
          IF(minlon < lon(imin+1)) EXIT
        END DO
        imax = nlon
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
      END IF
    END IF
  ELSE
    maxlon = maxlon - 360.0d0
    minlon = minlon + 360.0d0
    IF(maxlon > 360.0 .OR. minlon < 0) THEN
      imin = 1
      imax = nlon
      IF( nobs > 0 ) &
      & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
    ELSE
      DO imin=1,nlon-1
        IF(minlon < lon(imin+1)) EXIT
      END DO
      DO imax=1,nlon-1
        IF(maxlon < lon(imax+1)) EXIT
      END DO
      IF(imin > imax) THEN
        imin = 1
        imax = nlon
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
      ELSE
        IF( nobs > 0 ) &
        & CALL obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
      END IF
    END IF
  END IF
  nn = nn-1
  IF(nn < 1) THEN
    nobsl = 0
    RETURN
  END IF
!
! CONVENTIONAL
!
  nobsl = 0
  IF(nn > 0) THEN
    DO n=1,nn
      !
      ! vertical localization
      !
      IF(NINT(obselm(nobs_use(n))) == id_ps_obs .AND. ilev > 1) THEN
        dlev = ABS(LOG(obsdat(nobs_use(n))) - logpfm(ij,ilev))
      ELSE IF(NINT(obselm(nobs_use(n))) /= id_ps_obs) THEN
        dlev = ABS(LOG(obslev(nobs_use(n))) - logpfm(ij,ilev))
      ELSE
        dlev = 0.0d0
      END IF
      !IF(NINT(obselm(nobs_use(n))) == id_u_obs .OR. NINT(obselm(nobs_use(n))) == id_qthr_obs) THEN
      !  WRITE(6,*) 'id ',obselm(nobs_use(n)), dlev, LOG(obslev(nobs_use(n))), dist_zerov !YSaw 20230831
      !ENDIF

      IF(dlev > dist_zerov) CYCLE
      !
      ! horizontal localization
      !
      tmplon=obslon(nobs_use(n))
      tmplat=obslat(nobs_use(n))
      CALL com_distll_1( tmplon, tmplat,lon1(ij), lat1(ij), dist)
      IF(dist > dist_zero ) CYCLE
      !
      ! variable localization
      !
      SELECT CASE(NINT(obselm(nobs_use(n))))
      CASE(id_u_obs)
        iobs=1
      CASE(id_v_obs)
        iobs=2
      CASE(id_t_obs)
        iobs=3
      CASE(id_q_obs)
        iobs=4
      CASE(id_rh_obs)
        iobs=5
      CASE(id_ps_obs)
        iobs=6
      CASE(id_rain_obs)
        iobs=7
      CASE(id_qthr_obs)
        iobs=8
      END SELECT
      !WRITE(6,*) 'id ',obselm(nobs_use(n)), id_qthr_obs !YSaw
      !
      ! Pseudo-observation has the differernt spatial localization 
      !
      !IF(iobs == 8) THEN ! Pseudo QTHR observation
        !WRITE(6,*)'dist lev for QTHR = ',dist, dlev
      !  IF(dist > 0.01d0) CYCLE
      !  IF(dlev > 0.01d0) CYCLE
      !ENDIF
      IF(iobs == 4 .AND. ilev == 1)THEN !YSaw 20230908
        WRITE(6,*) 'assimilating surface humidity', tmplon,tmplat
      ENDIF
      ! end pseudo-observation localization
      !WRITE(6,*) 'nvar, ',nvar, iobs, var_local(nvar,iobs)
      !WRITE(6,*) 'observation spec', n, obserr(nobs_use(n)), obslev(nobs_use(n)), obslon(nobs_use(n)), obslat(nobs_use(n))
      IF(var_local(nvar,iobs) < TINY(var_local)) CYCLE

      nobsl = nobsl + 1
      hdxf(nobsl,:) = obshdxf(nobs_use(n),:)
      dep(nobsl)    = obsdep(nobs_use(n))
      !
      ! Observational localization
      !
      tmperr=obserr(nobs_use(n))
      rdiag(nobsl) = tmperr * tmperr
      rloc(nobsl) =EXP(-0.5d0 * ((dist/sigma_obs)**2 + (dlev/sigma_obsv)**2)) &
                  & * var_local(nvar,iobs)
    END DO
  END IF

  IF( nobsl > nobstotal ) THEN
    WRITE(6,'(A,I5,A,I5)') 'FATAL ERROR, NOBSL=',nobsl,' > NOBSTOTAL=',nobstotal
    WRITE(6,*) 'IJ,NN,TVNN=', ij, nn, tvnn
    STOP 99
  END IF

  IF( nobs > 0 ) THEN
    DEALLOCATE(nobs_use)
  END IF

  RETURN
END SUBROUTINE obs_local

SUBROUTINE obs_local_sub(imin,imax,jmin,jmax,nn,nobs_use)
  INTEGER,INTENT(IN) :: imin,imax,jmin,jmax
  INTEGER,INTENT(INOUT) :: nn, nobs_use(nobs)
  INTEGER :: j,n,ib,ie,ip

  DO j=jmin,jmax
    IF(imin > 1) THEN
      ib = nobsgrd(imin-1,j)+1
    ELSE
      IF(j > 1) THEN
        ib = nobsgrd(nlon,j-1)+1
      ELSE
        ib = 1
      END IF
    END IF
    ie = nobsgrd(imax,j)
    n = ie - ib + 1
    IF(n == 0) CYCLE
    DO ip=ib,ie
      IF(nn > nobs) THEN
        WRITE(6,*) 'FATALERROR, NN > NOBS', NN, NOBS
      END IF
      nobs_use(nn) = ip
      nn = nn + 1
    END DO
  END DO

  RETURN
END SUBROUTINE obs_local_sub

SUBROUTINE hoopeot(theta_bk,theta_c,C_std,np,inflation)
!
! Sawada & Duc
! HOOPE-EnKF with optimal transport (1-D) version
!
  USE common
  USE common_letkf

  IMPLICIT NONE

  REAL(r_size),INTENT(INOUT):: theta_bk(np,n_ens) !ensembles of parameter
  REAL(r_size),INTENT(IN)   :: theta_c          !climatological parameter
  REAL(r_size),INTENT(IN)   :: C_std            !climatological para's sd
  REAL(r_size),INTENT(IN)   :: inflation(np)    !prior MI hyperparameter
  REAL(r_size)              :: theta_g(np), theta_b(np), B_std(np), ptbmultip, tmp1, tmp2
  INTEGER, INTENT(IN)       :: np
  INTEGER :: i,j,k

  DO i = 1, np
     CALL com_mean(n_ens, theta_bk(i,:), theta_b(i))
     CALL com_stdev(n_ens, theta_bk(i,:), B_std(i))
  ENDDO
  !
  ! Posterior mean of theta after integrating N(theta_c,C_std**2)
  !
  DO i = 1, np
     theta_g(i) = theta_c*inflation(i)*(B_std(i)**2.0d0)/(C_std**2.0d0+inflation(i)*(B_std(i)**2.0d0)) &
             & + theta_b(i)*(C_std**2.0d0)/(C_std**2.0d0+inflation(i)*(B_std(i)**2.0d0))
  ENDDO
  !
  ! Calculating each ensembles
  !
  DO i = 1, np
   !ptbmultip = C_std*(inflation(i)**(1/2))/sqrt(C_std**2+inflation(i)*((B_std(i)**2)))
   tmp1 = C_std*(inflation(i)**(0.5d0))
   tmp2 = sqrt(C_std**2.0d0+inflation(i)*((B_std(i)**2.0d0)))
   ptbmultip = tmp1/tmp2
   WRITE(6,*) 'inflation, multiplicative ', i, inflation(i), ptbmultip, tmp1, tmp2, B_std(i), C_std, theta_g(i), theta_b(i)
   DO j = 1, n_ens
     theta_bk(i,j) = theta_g(i) + ptbmultip*(theta_bk(i,j)-theta_b(i))
   ENDDO
  ENDDO

END SUBROUTINE

SUBROUTINE hoopeot_nomeanchange(theta_bk,theta_c,C_std,np,inflation)
!
! Sawada & Duc
! HOOPE-EnKF with optimal transport (1-D) version
!
  USE common
  USE common_letkf

  IMPLICIT NONE

  REAL(r_size),INTENT(INOUT):: theta_bk(np,n_ens) !ensembles of parameter
  REAL(r_size),INTENT(IN)   :: theta_c          !climatological parameter
  REAL(r_size),INTENT(IN)   :: C_std            !climatological para's sd
  REAL(r_size),INTENT(IN)   :: inflation(np)    !prior MI hyperparameter
  REAL(r_size)              :: theta_g(np), theta_b(np), B_std(np), ptbmultip, tmp1, tmp2
  INTEGER, INTENT(IN)       :: np
  INTEGER :: i,j,k

  DO i = 1, np
     CALL com_mean(n_ens, theta_bk(i,:), theta_b(i))
     CALL com_stdev(n_ens, theta_bk(i,:), B_std(i))
  ENDDO
  !
  ! Calculating each ensembles
  !
  DO i = 1, np
   !ptbmultip = C_std*(inflation(i)**(1/2))/sqrt(C_std**2+inflation(i)*((B_std(i)**2)))
   tmp1 = C_std*(inflation(i)**(0.5d0))
   tmp2 = sqrt(C_std**2.0d0+inflation(i)*((B_std(i)**2.0d0)))
   ptbmultip = tmp1/tmp2
   WRITE(6,*) 'inflation, multiplicative ', i, inflation(i), ptbmultip, tmp1, tmp2, B_std(i), C_std
   DO j = 1, n_ens
     theta_bk(i,j) = theta_b(i) + ptbmultip*(theta_bk(i,j)-theta_b(i))  ! mean does not change
   ENDDO
  ENDDO

END SUBROUTINE



END MODULE letkf_tools
