MODULE lorenz96
!=======================================================================
!
! [PURPOSE:] Lorenz 1996 model
!
! [PUBLIC:]
!   SUBROUTINE tinteg_rk4(kt,xin,xout)
!   SUBROUTINE TL_tinteg_rk4(kt,x9,xin,xout)
!   SUBROUTINE TL_tinteg_rk4_x9out(kt,x9,xin,xout)
!   SUBROUTINE AD_tinteg_rk4(kt,x9,xin,xout)
!   SUBROUTINE tinteg_rk4_ptbmtx(alpha,kt,x9,pa,pf)
!   SUBROUTINE TL_tinteg_rk4_ptbmtx(kt,x9,pa,pf)
!
! [FIRST CREATED:] 08/23/2003 Takemasa MIYOSHI
!
! [HISTORY:]
!   08/23/2003 Takemasa Miyoshi  Initial Creation
!   10/17/2003 Takemasa Miyoshi  Tangent Linear Model is added
!   03/20/2004 Takemasa Miyoshi  Covariance matrix forecast is added
!   03/30/2004 Takemasa Miyoshi  Adjoint model is added
!   03/31/2004 Takemasa Miyoshi  Cleaned up
!   06/08/2009 Takemasa Miyoshi  Orography model is separated
!   07/12/2022 Yohei Sawada      Multi-scale model 
!
!=======================================================================
  USE common

  PRIVATE

  PUBLIC :: tinteg_rk4 ! TL_tinteg_rk4, TL_tinteg_rk4_x9out, AD_tinteg_rk4,&
          !& tinteg_rk4_ptbmtx, TL_tinteg_rk4_ptbmtx

  ! --- config of case study 1 in Pathiraja and van Leeuwen 2022
!  INTEGER,PARAMETER,PUBLIC :: nx=9         ! number of grid points
!  INTEGER,PARAMETER,PUBLIC :: ny=128         ! number of sub-grid points
!  REAL(r_size),SAVE,PUBLIC :: dt=0.0005d0    ! time of one time step
!  REAL(r_size),SAVE,PUBLIC :: force=10.0d0   ! F term
!  REAL(r_size),SAVE,PUBLIC :: hx=-0.8d0      ! hx term
!  REAL(r_size),SAVE,PUBLIC :: hy=1.0d0      ! hy term
!  REAL(r_size),SAVE,PUBLIC :: oneday=0.2d0  ! time for one day

  ! --- config of case study 2 in Pathiraja and van leeuwen 2022
  INTEGER,PARAMETER,PUBLIC :: nx=9         ! number of grid points
  INTEGER,PARAMETER,PUBLIC :: ny=20         ! number of sub-grid points
  REAL(r_size),SAVE,PUBLIC :: dt=0.0005d0    ! time of one time step
  REAL(r_size),SAVE,PUBLIC :: force=14.0d0   ! F term
  REAL(r_size),SAVE,PUBLIC :: hx=-2.0d0      ! hx term
  REAL(r_size),SAVE,PUBLIC :: hy=1.0d0      ! hy term
  REAL(r_size),SAVE,PUBLIC :: oneday=0.2d0  ! time for one day


CONTAINS
!=======================================================================
! [1] Methods of Lorenz96
!=======================================================================
!-----------------------------------------------------------------------
! [1.1] Time integration of Lorenz96
!-----------------------------------------------------------------------
SUBROUTINE tinteg_rk4(kt,xin,xout,yin,yout,force_coupled)
  IMPLICIT NONE

  INTEGER,INTENT(IN) :: kt
  REAL(r_size),INTENT(IN)  :: xin(1:nx)
  REAL(r_size),INTENT(OUT) :: xout(1:nx)
  REAL(r_size),INTENT(IN)  :: yin(1:ny,1:nx)
  REAL(r_size),INTENT(OUT) :: yout(1:ny,1:nx)
  REAL(r_size),INTENT(OUT) :: force_coupled(1:nx)
  REAL(r_size),ALLOCATABLE :: x(:),xtmp(:),qx1(:),qx2(:),qx3(:),qx4(:)
  REAL(r_size),ALLOCATABLE :: y(:,:),ytmp(:,:),qy1(:,:),qy2(:,:),qy3(:,:),qy4(:,:)
  INTEGER :: k
!--[1.1.1] allocation --------------------------------------------------
  ALLOCATE( x(1:nx) )
  ALLOCATE( xtmp(1:nx) )
  ALLOCATE( qx1(1:nx) )
  ALLOCATE( qx2(1:nx) )
  ALLOCATE( qx3(1:nx) )
  ALLOCATE( qx4(1:nx) )
  ALLOCATE( y(1:ny,1:nx) )
  ALLOCATE( ytmp(1:ny,1:nx) )
  ALLOCATE( qy1(1:ny,1:nx) )
  ALLOCATE( qy2(1:ny,1:nx) )
  ALLOCATE( qy3(1:ny,1:nx) )
  ALLOCATE( qy4(1:ny,1:nx) )

!--[1.1.2] time integration --------------------------------------------
  x(:) = xin(:)
  y(:,:) = yin(:,:)
  !print*, 'yin = ', yin(:,1)
  !print*, 'xin = ', xin(:)
!>>>>> TIME INTEGRATION START
  DO k=1,kt
    xtmp(:) = x(:)
    ytmp(:,:) = y(:,:)
    CALL lorenz96_core(xtmp,ytmp,qx1,qy1)
    xtmp(:) = x(:) + 0.5d0 * qx1(:)
    ytmp(:,:) = y(:,:) + 0.5d0 * qy1(:,:)
    CALL lorenz96_core(xtmp,ytmp,qx2,qy2)
    xtmp(:) = x(:) + 0.5d0 * qx2(:)
    ytmp(:,:) = y(:,:) + 0.5d0 * qy2(:,:)
    CALL lorenz96_core(xtmp,ytmp,qx3,qy3)
    xtmp(:) = x(:) + qx3(:)
    ytmp(:,:) = y(:,:) + qy3(:,:)
    CALL lorenz96_core(xtmp,ytmp,qx4,qy4)
    x(:) = x(:) + ( qx1(:) + 2.0d0 * qx2(:) + 2.0d0 * qx3(:) + qx4(:) ) / 6.0d0
    y(:,:) = y(:,:) + ( qy1(:,:) + 2.0d0 * qy2(:,:) + 2.0d0 * qy3(:,:) + qy4(:,:) ) / 6.0d0
  !print*, 'xout = ', x(:)
  !print*, 'yout = ', y(:,1)
  END DO
!<<<<< TIME INTEGRATION END
  xout(:) = x(:)
  yout(:,:) = y(:,:)
  DO k = 1, nx
   force_coupled(k) = force + hx*sum(y(:,k))/dble(ny)
  ENDDO

  !print*, 'yout = ', yout(1,:)
  !print*, 'xout = ', xout(:)
!--[1.1.3] tidy up -----------------------------------------------------
  DEALLOCATE( xtmp,qx1,qx2,qx3,qx4,ytmp,qy1,qy2,qy3,qy4 )

  RETURN
END SUBROUTINE tinteg_rk4
!=======================================================================
! [2] core part of Lorenz96
!=======================================================================
!--[2.1] NL ------------------------------------------------------------
SUBROUTINE lorenz96_core(xin,yin,xout,yout)
  IMPLICIT NONE

  REAL(r_size),INTENT(IN) :: xin(1:nx)
  REAL(r_size),INTENT(IN) :: yin(1:ny,1:nx)
  REAL(r_size),INTENT(OUT) :: xout(1:nx)
  REAL(r_size),INTENT(OUT) :: yout(1:ny,1:nx)
  REAL(r_size):: force_coupled(1:nx)
  INTEGER :: i,j

  DO i = 1, nx
   force_coupled(i) = force + hx*sum(yin(:,i))/dble(ny)
  ENDDO

  xout(1) = xin(nx) * ( xin(2) - xin(nx-1) ) - xin(1) + force_coupled(1)
  xout(2) = xin(1) * ( xin(3) - xin(nx) ) - xin(2) + force_coupled(2)
  DO i=3,nx-1
    xout(i) = xin(i-1) * ( xin(i+1) - xin(i-2) ) - xin(i) + force_coupled(i)
  END DO
  xout(nx) = xin(nx-1) * ( xin(1) - xin(nx-2) ) - xin(nx) + force_coupled(nx)

  xout(:) = dt * xout(:)

  ! j = 1
  yout(1,1) = yin(2,1) * (yin(ny,nx) - yin(3,1)) - yin(1,1) + hy * xin(1)
  DO i=2,ny-2
   yout(i,1) = yin(i+1,1) * (yin(i-1,1) - yin(i+2,1)) - yin(i,1) + hy * xin(1)
  ENDDO
  yout(ny-1,1) = yin(ny,1) * (yin(ny-2,1) - yin(1,2)) - yin(ny-1,1) + hy * xin(1)
  yout(ny,1) = yin(1,2) * (yin(ny-1,1) - yin(2,2)) - yin(ny,1) + hy * xin(1)

  DO j = 2, nx-1
   yout(1,j) = yin(2,j) * (yin(ny,j-1) - yin(3,j)) - yin(1,j) + hy * xin(j)
   DO i=2,ny-2
     yout(i,j) = yin(i+1,j) * (yin(i-1,j) - yin(i+2,j)) - yin(i,j) + hy * xin(j)
   ENDDO
   yout(ny-1,j) = yin(ny,j) * (yin(ny-2,j) - yin(1,j+1)) - yin(ny-1,j) + hy * xin(j)
   yout(ny,j) = yin(1,j+1) * (yin(ny-1,j) - yin(2,j+1)) - yin(ny,j) + hy * xin(j)
  ENDDO ! j loop

  ! j = nx
  yout(1,nx) = yin(2,nx) * (yin(ny,nx-1) - yin(3,nx)) - yin(1,nx) + hy * xin(nx)
  DO i=2,ny-2
   yout(i,nx) = yin(i+1,nx) * (yin(i-1,nx) - yin(i+2,nx)) - yin(i,nx) + hy * xin(nx)
  ENDDO
  yout(ny-1,nx) = yin(ny,nx) * (yin(ny-2,nx) - yin(1,1)) - yin(ny-1,nx) + hy * xin(nx)
  yout(ny,nx) = yin(1,1) * (yin(ny-1,nx) - yin(2,1)) - yin(ny,nx) + hy * xin(nx)

  !yout(:,:) = dt * yout(:,:) * 128.0d0 ! case study 1
  yout(:,:) = dt * yout(:,:) / 0.70d0 ! case study 2
  !print*, 'y input', yin(:,1)
  !print*, 'y increment', yout(:,1)
  !STOP

  RETURN
END SUBROUTINE lorenz96_core

END MODULE lorenz96
