#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Thu Feb 23 16:33:07 2023


@author: pdavid

MAIN MODULE FOR THE SOLUTION SPLIT COUPLING MODEL IN 2D!

This is the first coupling module that I manage to succeed with some type of coupling

the coupling with the negihbouring FV works quite well.

The problem arises when coupling two contiguous source blocks. Since there is no
continuity enforced explicitly the solution does not respec C1 nor C0 continuity.

Furthermore,
"""
import os
directory_script = os.path.dirname(__file__)
with open(os.path.join(directory_script,'../current_directory.txt'), 'r') as file:
    file_contents = file.read()
# Now, file_contents contains the entire content of the file as a string

source_directory=os.path.join(file_contents[:-1], 'src')
import sys
sys.path.append(source_directory)

import numpy as np
import scipy as sp

from Assembly_diffusion import Lapl_2D_FD_sparse

# from Green import  grad_Green_norm, grad_Green, Green, Sampson_grad_Green, Sampson_Green, block_grad_Green_norm_array,kernel_green_neigh, from_pos_get_green,kernel_integral_grad_green_face,kernel_integral_Green_face
from Green import *
from Neighbourhood import get_neighbourhood, get_uncommon
from Small_functions import (
    get_4_blocks,
    get_boundary_vector,
    pos_to_coords,
    v_linear_interpolation,
)

print("MODULE")
print(os.getcwd())


class assemble_SS_2D_FD_sparse:
    """Create a local, potential based linear system"""

    def __init__(self, pos_s, Rv, h, L, K_eff, D, directness):
        """Class that will assemble the localized operator splitting approach.
        Its arguments are:
            - pos_s: an array containing the position of each sources
            - Rv: an array containing the radius of each source
            - h: the size of the cartesian mesh
            - L: the size of the full square domain
            - K_eff: an array with the effective permeability of each source
            - D: the diffusion coefficient in the parenchyma
            - directness: how many levels of adjacent FV cells are considered for the splitting"""
        x = np.linspace(h / 2, L - h / 2, int(np.around(L / h)))
        y = x.copy()
        self.x = x
        self.y = y
        self.K_0 = K_eff * np.pi * Rv**2
        self.pos_s = pos_s  # exact position of each source
        self.L = L
        self.n_sources = self.pos_s.shape[0]
        self.Rv = Rv
        self.h = h
        self.D = D
        self.boundary = get_boundary_vector(len(x), len(y))
        self.directness = directness

        self.no_interpolation = (
            0  # The variable is set to 0 by default so the complex interpolation
        )
        # function will be used for the estimation of the wall concentration for q

    def solve_problem(self, BC_types, BC_values, C_v_array):
        """Function that after initialisation solves the entire problem"""
        self.pos_arrays()  # Creates the arrays with the geometrical position of the sources

        # Assembly of the Laplacian and other arrays for the linear system
        LIN_MAT = self.assembly_linear_problem_sparse(BC_types, BC_values)
        self.set_intravascular_terms(
            C_v_array
        )  # Sets up the intravascular concentration as BC
        sol = sp.sparse.linalg.spsolve(LIN_MAT, -self.H0)
        self.sol = sol
        s_FV = sol[: -self.S].reshape(len(self.x), len(self.y))
        q = sol[-self.S :]

        # initial guesses -> solution of the linear system
        self.s_FV = s_FV
        self.q_linear = q

        self.set_phi_bar_linear(C_v_array)
        self.C_v_array = C_v_array
        return self.q_linear

    def pos_arrays(self):
        """This function is the pre processing step. It is meant to create the s_blocks
        and uni_s_blocks arrays which will be used extensively throughout. s_blocks represents
        the block where each source is located, uni_s_blocks contains all the source blocks
        in a given order that will be respected throughout the resolution"""
        # pos_s will dictate the ID of the sources by the order they are kept in it!
        source_FV = np.array([]).astype(int)
        uni_s_blocks = np.array([], dtype=int)
        for u in self.pos_s:
            r = np.argmin(np.abs(self.x - u[0]))
            c = np.argmin(np.abs(self.y - u[1]))
            source_FV = np.append(
                source_FV, c * len(self.x) + r
            )  # block where the source u is located
            if c * len(self.x) + r not in uni_s_blocks:
                uni_s_blocks = np.append(uni_s_blocks, c * len(self.x) + r)

        self.FV_DoF = np.arange(len(self.x) * len(self.y))
        self.s_blocks = source_FV  # for each entry it shows the block of that source
        self.uni_s_blocks = uni_s_blocks

        total_sb = len(np.unique(self.s_blocks))  # total amount of source blocks
        self.total_sb = total_sb

        # We initialize the independent vector
        self.H0 = np.zeros(len(self.s_blocks) + len(self.FV_DoF))

    def assembly_linear_problem_sparse(self, BC_types, BC_values):
        # First of all it is needed to remove the source_blocks from the main matrix
        # pdb.set_trace()
        self.A_matrix = Lapl_2D_FD_sparse(len(self.x), len(self.y))
        # self.set_Dirichlet(values_Dirich)

        b_data = self.assemble_b_matrix_sparse()

        self.b_matrix = sp.sparse.csc_matrix(
            (b_data[0], (b_data[1], b_data[2])),
            shape=(len(self.x) * len(self.y), len(self.pos_s)),
        )

        self.A_matrix_virgin, self.b_matrix_virgin = (
            self.A_matrix.copy(),
            self.b_matrix.copy(),
        )

        self.set_BCs(BC_types, BC_values)  # For now let's do it the slow way
        c_data, d_data = self.assemble_c_d_matrix_sparse()

        self.c_matrix = sp.sparse.csc_matrix(
            (c_data[0], (c_data[1], c_data[2])),
            shape=(len(self.pos_s), len(self.x) * len(self.y)),
        )

        self.d_matrix = sp.sparse.csc_matrix(
            (d_data[0], (d_data[1], d_data[2])),
            shape=(len(self.pos_s), len(self.pos_s)),
        )

        Up = sp.sparse.hstack((self.A_matrix, self.b_matrix))
        Down = sp.sparse.hstack((self.c_matrix, self.d_matrix))

        self.Up = Up
        self.Down = Down

        LIN_MAT = sp.sparse.vstack((Up, Down))
        self.LIN_MAT = LIN_MAT
        return LIN_MAT

    def set_intravascular_terms(self, C_v_array):
        """Be careful, it operates over the class variable H0 which is the independent term
        Must be called only once!!"""
        self.S = len(C_v_array)
        self.H0[-len(C_v_array) :] -= C_v_array
        self.C_v_array = C_v_array

    def set_BCs(self, BC_type, BC_values):
        array_opposite = np.array([1, 0, 3, 2])
        b_prime = np.zeros((self.b_matrix.shape))
        for i in range(4):
            # Goes through each boundary {north, south, east, west} = {0,1,2,3,4}
            c = 0
            for k in self.boundary[i, :]:
                # Goes through each of the cells in the boundary i
                normal = np.array([[0, 1], [0, -1], [1, 0], [-1, 0]])[i]
                m = self.boundary[array_opposite[i], c]
                # The division by h is because the kernel calculates the integral, what we
                # need is an average value per full cell
                (
                    r_k_face_kernel,
                    r_k_grad_face_kernel,
                    r_m_face_kernel,
                    r_m_grad_face_kernel,
                ) = self.get_interface_kernels(k, normal, m)

                if BC_type[i] == "Dirichlet":
                    self.A_matrix[k, k] -= 2
                    self.b_matrix[k, :] -= r_k_face_kernel * 2 / self.h
                    self.H0[k] += BC_values[i] * 2
                elif BC_type[i] == "Neumann":
                    self.b_matrix[k, :] -= r_k_grad_face_kernel
                    self.H0[k] += BC_values[i] * self.h
                elif BC_type[i] == "Periodic":

                    jump_r_km = -r_k_grad_face_kernel / 2 - r_k_face_kernel / self.h
                    jump_r_mk = (
                        r_m_grad_face_kernel / 2 + r_m_face_kernel / self.h
                    )  # The negative sign is because th normal is opposite

                    self.b_matrix[k, :] += jump_r_mk + jump_r_km
                    self.A_matrix[k, k] -= 1
                    self.A_matrix[k, m] += 1
                elif BC_type[i] == "Infinite":
                    # Only works if all are infinite
                    b_prime[k, :] -= r_k_grad_face_kernel
                    # pdb.set_trace()
                else:
                    print("Error in the assignation of the BC_type")

                c += 1
        if np.all(BC_type == "Infinite"):
            self.b_prime = np.sum(b_prime, axis=0) / (self.boundary.size)
            self.b_matrix = self.b_matrix + self.b_prime

    # =============================================================================
    #
    #         def set_BCs_sparse(self, BC_type, BC_values):
    #             array_opposite = np.array([1, 0, 3, 2])
    #             data_A=np.array([])
    #             row_A=np.array([])
    #             col_A=np.array([])
    #
    #             data_B=np.array([])
    #             row_B=np.array([])
    #             col_B=np.array([])
    #             b_prime=np.zeros(len(pos_s))
    #             for i in range(4):
    #                 # Goes through each boundary {north, south, east, west} = {0,1,2,3,4}
    #                 c = 0
    #                 for k in self.boundary[i, :]:
    #                     # Goes through each of the cells in the boundary i
    #                     normal = np.array([[0, 1], [0, -1], [1, 0], [-1, 0]])[i]
    #                     m = self.boundary[array_opposite[i], c]
    #                     # The division by h is because the kernel calculates the integral, what we
    #                     # need is an average value per full cell
    #                     (
    #                         r_k_face_kernel,
    #                         r_k_grad_face_kernel,
    #                         r_m_face_kernel,
    #                         r_m_grad_face_kernel,
    #                     ) = self.get_interface_kernels(k, normal, m)
    #
    #                     if BC_type[i] == "Dirichlet":
    #
    #                         self.A_matrix[k, k] -= 2
    #                         self.b_matrix[k, :] -= r_k_face_kernel * 2/self.h
    #                         self.H0[k] += BC_values[i] * 2
    #                     elif BC_type[i] == "Neumann":
    #                         self.b_matrix[k, :] -= r_k_grad_face_kernel
    #                         self.H0[k] -= BC_values[i] *self.h
    #                     elif BC_type[i] == "Periodic":
    #
    #                         jump_r_km = -r_k_grad_face_kernel / 2 - r_k_face_kernel / self.h
    #                         jump_r_mk = (
    #                             r_m_grad_face_kernel / 2 + r_m_face_kernel / self.h
    #                         )  # The negative sign is because th normal is opposite
    #
    #                         self.b_matrix[k, :] += jump_r_mk + jump_r_km
    #                         self.A_matrix[k, k] -= 1
    #                         self.A_matrix[k, m] += 1
    #                     elif BC_type[i]=="Infinite":
    #                         pdb.set_trace()
    #                         b_prime+=r_k_face_kernel
    #                     else:
    #                         print("Error in the assignation of the BC_type")
    #                     c += 1
    #             if BC_type[i]=="Infinite":
    #                 for i in np.unique(self.boundary):
    #                     self.b_matrix[i]-=b_prime/(self.boundary.size-4)
    #
    # =============================================================================

    def assemble_b_matrix_sparse(self):
        b_data = np.array([[0], [0], [0]])
        for k in self.FV_DoF:
            # if np.any(k==self.s_blocks): pdb.set_trace()
            c = 0
            neigh = np.array([len(self.x), -len(self.x), 1, -1])

            if k in self.boundary[0]:
                neigh = np.delete(neigh, np.where(neigh == len(self.x))[0])
            if k in self.boundary[1]:
                neigh = np.delete(neigh, np.where(neigh == -len(self.x))[0])
            if k in self.boundary[2]:
                neigh = np.delete(neigh, np.where(neigh == 1)[0])
            if k in self.boundary[3]:
                neigh = np.delete(neigh, np.where(neigh == -1)[0])

            for i in neigh:
                m = k + i
                pos_k = pos_to_coords(self.x, self.y, k)
                pos_m = pos_to_coords(self.x, self.y, m)
                normal = (pos_m - pos_k) / self.h
                (
                    r_k_face_kernel,
                    r_k_grad_face_kernel,
                    r_m_face_kernel,
                    r_m_grad_face_kernel,
                ) = self.get_interface_kernels(k, normal, m)

                # pdb.set_trace()
                kernel_m = r_m_face_kernel / self.h + r_m_grad_face_kernel / 2

                kernel_k = r_k_face_kernel / self.h + r_k_grad_face_kernel / 2

                # =============================================================================
                #                 self.b_matrix[k, :] += kernel_m
                #                 self.b_matrix[k, :] -= kernel_k
                # =============================================================================
                # Assemble the sparse vectors:
                total_kernel = kernel_m - kernel_k
                c = np.nonzero(total_kernel)[0]
                r = k + np.zeros(len(c))
                d = total_kernel[c]
                b_data = append_sparse(b_data, d, r, c)
                c += 1
        return b_data

    def assemble_c_d_matrix_sparse(self):
        c_matrix = np.array([[0], [0], [0]])
        d_matrix = np.array([[0], [0], [0]])

        no_interpolation = self.no_interpolation
        # pdb.set_trace()
        for block_ID in self.uni_s_blocks:
            # Loop that goes through each of the source containing blocks
            k = block_ID
            # We obtain the neighbourhood of the block
            neigh_base = get_neighbourhood(self.directness, len(self.x), block_ID)
            # We obtain the sources contained in that neighbourhood
            # sources_neigh is the set: E(\widehat{V_k})
            sources_neigh = np.arange(len(self.s_blocks))[
                np.in1d(self.s_blocks, neigh_base)
            ]

            # E(V_k):
            sources_in = np.where(self.s_blocks == block_ID)[
                0
            ]  # sources that need to be solved in this block
            for i in sources_in:
                # Now we are gonna assemble the equation for the source i
                # Firstly we make the set of all the sources in the neighbourhood except i
                other = np.delete(sources_neigh, np.where(sources_neigh == i))
                #############################################################################################
                #############################################################################################
                if no_interpolation:

                    # self.c_matrix[i, k] = 1  # regular term
                    c_matrix = append_sparse(c_matrix, 1, i, k)
                    # self.d_matrix[i, i] = 1 / self.K_0[i]
                    d_matrix = append_sparse(d_matrix, 1 / self.K_0[i], i, i)
                    print("old version flux estimation")
                    for j in other:
                        """Goes through each of the other sources (j != i)"""
                        # =============================================================================
                        #                         self.d_matrix[i, j] += Green(
                        #                             self.pos_s[i], self.pos_s[j], self.Rv[j], self.D
                        #                         )
                        # =============================================================================
                        d_matrix = append_sparse(
                            d_matrix,
                            Green(self.pos_s[i], self.pos_s[j], self.Rv[j], self.D),
                            i,
                            j,
                        )
                else:
                    # print("new version flux estimation")
                    if (
                        self.pos_s[i, 0] > self.x[-1]
                        or self.pos_s[i, 0] < self.x[0]
                        or self.pos_s[i, 1] > self.y[-1]
                        or self.pos_s[i, 1] < self.y[0]
                    ):
                        # Basically we do the same as if there was no interpolation if the source falls near the boundary
                        """The condition will be satisfied if the source falls within the dual boundary, that is,
                        it would need to be interpolated with the boundary values of the regular term. That is difficult,
                        so for these sources that lie closer than h/2 from the boundary, no interpolation is performed"""
                        # self.c_matrix[i, k] = 1  # regular term
                        c_matrix = append_sparse(c_matrix, 1, i, k)
                        # self.d_matrix[i, i] = 1 / self.K_0[i]
                        d_matrix = append_sparse(d_matrix, 1 / self.K_0[i], i, i)
                        for j in other:
                            """Goes through each of the other sources (j != i)"""
                            d_matrix = append_sparse(
                                d_matrix,
                                Green(self.pos_s[i], self.pos_s[j], self.Rv[j], self.D),
                                i,
                                j,
                            )
                    # =============================================================================
                    #                             self.d_matrix[i, j] += Green(
                    #                                 self.pos_s[i], self.pos_s[j], self.Rv[j], self.D
                    #                             )
                    # =============================================================================

                    else:
                        # Now is when we calculate properly the interpolation function \mathcal{I}_\phi
                        """The source does not belong to the dual boundary, therefore, it can be
                        interpolated without issues"""
                        blocks = get_4_blocks(
                            self.pos_s[i], self.x, self.y, self.h
                        )  # gets the ID of each of the 4 blocks

                        c_kernel = np.zeros(len(self.x) * len(self.y))
                        d_kernel = np.zeros(len(self.pos_s))

                        coord_blocks = np.zeros((0, 2))
                        for k in blocks:

                            coord_block = pos_to_coords(self.x, self.y, k)
                            coord_blocks = np.vstack((coord_blocks, coord_block))
                        center = np.sum(coord_blocks, axis=0) / 4
                        # pdb.set_trace()
                        c_kernel[blocks] = v_linear_interpolation(
                            center, self.pos_s[i], self.h
                        )  # This gets the value of each of the shape functions

                        for (
                            k
                        ) in (
                            blocks
                        ):  # Loop through each of the 4 nearest blocks to the source
                            # The following is the neighbourhood of each of the four blocks
                            neigh_m = get_neighbourhood(self.directness, len(self.x), k)
                            # c_kernel[k] represents the value of the linear shape functions for that block
                            # at the position of the source i
                            d_kernel += (
                                kernel_green_neigh(
                                    coord_block,
                                    neigh_m,
                                    self.pos_s,
                                    self.s_blocks,
                                    self.Rv,
                                    self.D,
                                )
                                * c_kernel[k]
                            )
                            d_kernel -= (
                                kernel_green_neigh(
                                    coord_block,
                                    neigh_base,
                                    self.pos_s,
                                    self.s_blocks,
                                    self.Rv,
                                    self.D,
                                )
                                * c_kernel[k]
                            )
                        # self.c_matrix[i, :] = c_kernel  # regular term
                        p = np.nonzero(c_kernel)[0]
                        if len(p):
                            c_matrix = append_sparse(
                                c_matrix, c_kernel[p], np.zeros(len(p)) + i, p
                            )
                        q = np.nonzero(d_kernel)[0]
                        # self.d_matrix[i, i] = 1 / self.K_0[i]
                        # self.d_matrix[i] += d_kernel
                        if len(q):
                            d_matrix = append_sparse(
                                d_matrix, d_kernel[q], np.zeros(len(q)) + i, q
                            )
                        d_matrix = append_sparse(d_matrix, 1 / self.K_0[i], i, i)
                        for j in other:
                            """Goes through each of the other sources (j != i)"""
                            # =============================================================================
                            #                             self.d_matrix[i, j] += (
                            #                                 Green(self.pos_s[i], self.pos_s[j], self.Rv[j], self.D)
                            #                             )
                            # =============================================================================
                            d_matrix = append_sparse(
                                d_matrix,
                                Green(self.pos_s[i], self.pos_s[j], self.Rv[j], self.D),
                                i,
                                j,
                            )
        return (c_matrix, d_matrix)

    def get_interface_kernels(self, block_ID, normal, neighbour):
        """Complicated function that is meant to ease the coding of the b_matrix
        The jump in flux and concentration across different interfaces is dependent on the
        average values of the gradient and value of the rapid potential on each side of the
        interface (side k and side m)
            Returns:
            - the kernel for the Sampson integral over the face of the rapid potential
            - the kernel of the Sampson integral over the face of the normal gradient of the rapid potential
            - The kernel for the jump between neighbours"""
        k = block_ID  # the block in questino for reference
        perp_dir = np.array([[0, -1], [1, 0]]).dot(normal)
        # First the kernels for r_k

        N_k = get_neighbourhood(self.directness, len(self.x), k)

        pos_k = pos_to_coords(self.x, self.y, k)  # cell center
        pos_a = pos_k + (perp_dir + normal) * self.h / 2
        pos_b = pos_k + (-perp_dir + normal) * self.h / 2

        pos_m = pos_to_coords(self.x, self.y, neighbour)
        pos_a_m = pos_a - normal * self.L
        pos_b_m = pos_b - normal * self.L
        m_neigh = get_neighbourhood(self.directness, len(self.x), neighbour)
        if np.all((pos_m - pos_k) / self.h == normal):
            # This is a real neighbour and they have common sources
            unc_k_m = get_uncommon(N_k, m_neigh)
            unc_m_k = get_uncommon(m_neigh, N_k)

            # sources in the neighbourhood of m that are not in the neigh of k
            Em = np.arange(len(self.s_blocks))[np.in1d(self.s_blocks, unc_m_k)]
            # sources in the neighbourhood of k that are not in the neigh of m
            Ek = np.arange(len(self.s_blocks))[np.in1d(self.s_blocks, unc_k_m)]

            # pdb.set_trace()
            r_m_grad_face_kernel = kernel_integral_grad_Green_face(
                self.pos_s, Em, pos_a, pos_b, normal, self.Rv, self.D
            )
            r_m_face_kernel = kernel_integral_Green_face(
                self.pos_s, Em, pos_a, pos_b, self.Rv, self.D
            )

            r_k_grad_face_kernel = kernel_integral_grad_Green_face(
                self.pos_s, Ek, pos_a, pos_b, normal, self.Rv, self.D
            )
            r_k_face_kernel = kernel_integral_Green_face(
                self.pos_s, Ek, pos_a, pos_b, self.Rv, self.D
            )

        else:
            # it is a "fake" neighbour caused by the periodic BCs
            # The division by h is because the kernel calculates the integral, what we
            # need is an average value per full cell
            Em = np.arange(len(self.s_blocks))[np.in1d(self.s_blocks, m_neigh)]
            # sources in the neighbourhood of k that are not in the neigh of m
            Ek = np.arange(len(self.s_blocks))[np.in1d(self.s_blocks, N_k)]
            r_k_face_kernel = kernel_integral_Green_face(
                self.pos_s, Ek, pos_a, pos_b, self.Rv, self.D
            )
            r_k_grad_face_kernel = kernel_integral_grad_Green_face(
                self.pos_s, Ek, pos_a, pos_b, normal, self.Rv, self.D
            )
            r_m_grad_face_kernel = kernel_integral_grad_Green_face(
                self.pos_s, Em, pos_a_m, pos_b_m, normal, self.Rv, self.D
            )
            r_m_face_kernel = kernel_integral_Green_face(
                self.pos_s, Em, pos_a_m, pos_b_m, self.Rv, self.D
            )

        return (
            r_k_face_kernel,
            r_k_grad_face_kernel,
            r_m_face_kernel,
            r_m_grad_face_kernel,
        )

    def set_phi_bar_linear(self, C_v_array):
        import pdb
        self.phi_bar = C_v_array - self.q_linear / self.K_0
        self.C_v_array = C_v_array
        self.phi_bar2 = self.q_linear / (self.K_0) - self.Down.dot(self.sol)

        if np.sum(np.abs(self.phi_bar - self.phi_bar2)) < 10**-4:
            print("ERROR ERROR ERROR ERRROR")
            print("ERROR ERROR ERROR ERRROR")
            print("ERROR ERROR ERROR ERRROR")

        return self.phi_bar

    def get_corners(self):
        # This function is often used in the reconstruction
        corners = np.array(
            [
                0,
                len(self.x) - 1,
                len(self.x) * (len(self.y) - 1),
                len(self.x) * len(self.y) - 1,
            ]
        )
        self.corners = corners
        return corners


class non_linear_metab_sparse(assemble_SS_2D_FD_sparse):
    """Creates a non linear, potential based
    diffusion 2D problem with circular sources"""

    def __init__(self, pos_s, Rv, h, L, K_eff, D, directness):
        assemble_SS_2D_FD_sparse.__init__(self, pos_s, Rv, h, L, K_eff, D, directness)
        self.pos_arrays()
        self.D = D

    def solve_linear_prob(self, BC_types, BC_values, C_v_array):
        """Solves the problem without metabolism, necessary to provide an initial guess
        DIRICHLET ARRAY ARE THE VALUES OF THE CONCENTRATION VALUE AT EACH BOUNDARY, THE CODE
        IS NOT YET MADE TO CALCULATE ANYTHING OTHER THAN dirichlet_array=np.zeros(4)"""

        self.solve_problem(BC_types, BC_values, C_v_array)
        self.S
        self.K_0

        # initial guesses
        self.s_FV_linear = self.s_FV
        self.q_linear = self.q_linear

        self.set_phi_bar_linear(C_v_array)

    # =============================================================================
    #         self.phi_bar2 = q_linear / (K0) - np.dot(self.Down, sol)
    #
    #         if np.any(np.abs(self.phi_bar - self.phi_bar2)) < 10**-4:
    #             print("ERROR ERROR ERROR ERRROR")
    #             print("ERROR ERROR ERROR ERRROR")
    #             print("ERROR ERROR ERROR ERRROR")
    #             pdb.set_trace()
    # =============================================================================

    def reconstruct_field_kk(self, s_FV, q):
        G_rec = from_pos_get_green(
            self.x,
            self.y,
            self.pos_s,
            self.Rv,
            np.array([0, 0]),
            self.directness,
            self.s_blocks,
            self.D,
        )
        return (np.dot(G_rec[:, :], q) + np.ndarray.flatten(s_FV)).reshape(
            len(self.y), len(self.x)
        )

    def assemble_it_matrices_Sampson(self, s_FV, q):
        """Assembles the important matrices for the iterative problem through
        Sampson integration within each cell. The matrices:
            - Upper Jacobian: [\partial_{u} I_m  \partial_{q} I_m]"""
        phi_0 = self.phi_0
        w_i = np.array([1, 4, 1, 4, 16, 4, 1, 4, 1]) / 36
        corr = (
            np.array(
                [
                    [
                        -1,
                        -1,
                    ],
                    [0, -1],
                    [1, -1],
                    [-1, 0],
                    [0, 0],
                    [1, 0],
                    [-1, 1],
                    [0, 1],
                    [1, 1],
                ]
            )
            * self.h
            / 2
        )
        I_m = np.zeros(s_FV.size)
        rec_sing = np.zeros(s_FV.size)  # integral of the rapid term over the cell
        part_Im_s_FV = np.zeros(s_FV.size)  # \frac{\partial I_m}{\partial u}
        part_Im_q = np.zeros((s_FV.size, len(self.pos_s)))
        for k in range(s_FV.size):
            # pdb.set_trace()
            for i in range(len(w_i)):

                pos_xy = pos_to_coords(self.x, self.y, k) + corr[i]
                kernel_G = kernel_green_neigh(
                    pos_xy,
                    get_neighbourhood(self.directness, len(self.x), k),
                    self.pos_s,
                    self.s_blocks,
                    self.Rv,
                    self.D,
                )
                SL = np.dot(kernel_G, q)
                rec_sing[k] += w_i[i] * SL

                I_m[k] += phi_0 * w_i[i] / (phi_0 + SL + s_FV[k])
                part_Im_s_FV[k] -= phi_0 * w_i[i] / (phi_0 + SL + s_FV[k]) ** 2
                for l in range(len(self.pos_s)):
                    # part_Im_q[k,l]+=part_Im_u[k]*kernel_G[l]
                    part_Im_q[k, l] -= (
                        kernel_G[l] * phi_0 * w_i[i] / (phi_0 + SL + s_FV[k]) ** 2
                    )

        self.I_m = I_m
        self.rec_sing = rec_sing  # integral of the Single Layer potential over the cell
        self.part_Im_q = part_Im_q
        self.part_Im_s_FV = part_Im_s_FV
        return ()

    def Full_Newton(self, s_linear, q_linear, rel_error, M, phi_0):

        M_D = M / self.D
        iterations = 0

        self.phi_0 = phi_0
        stabilization = 1
        self.stabilization = stabilization
        if phi_0 < 0.2:
            self.stabilization /= 2
        if M_D * 50**2 > 0.25:
            self.stabilization /= 2
        print("stabilization= ", self.stabilization)
        rl = np.array([1])
        arr_unk = np.array(
            [np.concatenate((s_linear, q_linear))]
        )  # This is the array where the arrays of u through iterations will be kept
        S = self.S

        stabilization = self.stabilization
        reduce = False
        while np.abs(rl[-1]) > rel_error and iterations < 300:
            if iterations % 75 == 74 or reduce:
                stabilization /= 2
                print("stabilization has been reduced")
                arr_unk = np.array([np.concatenate((s_linear, q_linear))])
                rl = np.array([1])  # restart the computation
                reduce = False
            self.assemble_it_matrices_Sampson(arr_unk[-1, :-S], arr_unk[-1, -S:])
            # average phi field
            s_field = arr_unk[-1, :-S]
            r_field = self.rec_sing
            phi = s_field + r_field
            if np.any(phi < 0):
                s_field[phi < 0] = -r_field[phi < 0]
                self.assemble_it_matrices_Sampson(s_field, arr_unk[-1, -S:])
                phi = s_field + r_field

            metab = M_D * (1 - self.I_m) * self.h**2
            self.metab = metab
            metab[phi < 0] = 0

            part_FV = self.part_Im_s_FV * M_D * self.h**2
            part_q = self.part_Im_q * M_D * self.h**2
            Jacobian = sp.sparse.hstack(
                (sp.sparse.diags(part_FV) + self.A_matrix, part_q + self.b_matrix)
            )

            Jacobian = sp.sparse.vstack((Jacobian, self.Down))
            # Compute the new value of u:
            F = self.LIN_MAT.dot(arr_unk[-1]) + self.H0 - np.pad(metab, [0, self.S])
            Jacobian = sp.sparse.csc_matrix(
                Jacobian
            )  # I do this to improve the efficiency of the following resolution
            inc = sp.sparse.linalg.spsolve(Jacobian, -F)
            inc[:-S][(inc[:-S] + phi) < 0] = -phi[(inc[:-S] + phi) < 0]
            arr_unk = np.concatenate(
                (arr_unk, np.array([arr_unk[-1] + inc * stabilization]))
            )

            rl = np.append(rl, np.sum(np.abs(inc)) / len(inc))
            print("residual", np.sum(np.abs(inc)) / len(inc))
            iterations += 1
            if len(rl) > 8 and rl[-1] > rl[-8]:
                reduce = True
        self.arr_unk_metab = arr_unk
        self.s_FV_metab = self.arr_unk_metab[-1, :-S]
        self.q_metab = self.arr_unk_metab[-1, -S:]

        self.residual = np.sum(np.abs(inc)) / len(inc)
        if iterations < 300:
            return (self.s_FV_metab, self.q_metab)
        else:
            return (np.zeros(len(self.s_FV_metab)), np.zeros(len(self.q_metab)))


def append_sparse(arr, d, r, c):
    data_arr = np.append(arr[0], d)
    row_arr = np.append(arr[1], r)
    col_arr = np.append(arr[2], c)
    return (data_arr, row_arr, col_arr)
