# -*- coding: utf-8 -*-

import numpy as np
import scipy as sp


def A_assembly_Dirich(cxlen, cylen):
    """This function assembles the laplacian operator for cartesian coordinates in 2D
    - It does not take into account the discretization size h, therefore, it must be homogeneous
    - For the same reason, if the discretization size is different than 1 there will be a factor to
      multiply/divide the operator
    - As well, there is no diffusion coefficient considered

    INPUT -> the x and y length
    OUTPUT -> FD Laplacian operator
    """
    A = np.zeros([cxlen * cylen, cxlen * cylen])
    north = np.arange(cxlen * (cylen - 1), cxlen * cylen)
    south = np.arange(cxlen)
    west = np.arange(0, cxlen * cylen, cxlen)
    east = np.arange(cxlen - 1, cxlen * cylen, cxlen)

    boundary = np.concatenate([north, south, east, west])

    for i in range(cxlen * cylen):
        if i not in boundary:
            A[i, i] -= 4
            A[i, i + 1] += 1
            A[i, i - 1] += 1
            A[i, i + cxlen] += 1
            A[i, i - cxlen] += 1

        else:
            if i in north:
                others = [1, -1, -cxlen]
            if i in south:
                others = [1, -1, cxlen]
            if i in east:
                others = [cxlen, -cxlen, -1]
            if i in west:
                others = [cxlen, -cxlen, 1]

            if i == 0:
                # corner sudwest
                others = [1, cxlen]
            if i == cxlen - 1:
                # sud east
                others = [-1, cxlen]
            if i == cxlen * (cylen - 1):
                # north west
                others = [1, -cxlen]
            if i == cxlen * cylen - 1:
                others = [-1, -cxlen]

            A[i, i] = -4
            for n in others:
                A[i, i + n] = 1

    return A


def A_assembly(cxlen, cylen):
    """This function assembles the laplacian operator for cartesian coordinates in 2D
    - It does not take into account the discretization size h, therefore, it must be homogeneous
    - For the same reason, if the discretization size is different than 1 there will be a factor to
      multiply/divide the operator
    - As well, there is no diffusion coefficient considered

    INPUT -> the x and y length
    OUTPUT -> FD Laplacian operator
    """
    A = np.zeros([cxlen * cylen, cxlen * cylen])
    north = np.arange(cxlen * (cylen - 1), cxlen * cylen)
    south = np.arange(cxlen)
    west = np.arange(0, cxlen * cylen, cxlen)
    east = np.arange(cxlen - 1, cxlen * cylen, cxlen)

    boundary = np.concatenate([north, south, east, west])

    for i in range(cxlen * cylen):
        if i not in boundary:
            A[i, i] -= 4
            A[i, i + 1] += 1
            A[i, i - 1] += 1
            A[i, i + cxlen] += 1
            A[i, i - cxlen] += 1

        else:
            if i in north:
                others = [1, -1, -cxlen]
            if i in south:
                others = [1, -1, cxlen]
            if i in east:
                others = [cxlen, -cxlen, -1]
            if i in west:
                others = [cxlen, -cxlen, 1]

            if i == 0:
                # corner sudwest
                others = [1, cxlen]
            if i == cxlen - 1:
                # sud east
                others = [-1, cxlen]
            if i == cxlen * (cylen - 1):
                # north west
                others = [1, -cxlen]
            if i == cxlen * cylen - 1:
                others = [-1, -cxlen]

            A[i, i] = -len(others)
            for n in others:
                A[i, i + n] = 1

    return A


def Lapl_2D_FD_sparse(cxlen, cylen):
    """This function assembles the laplacian operator for cartesian coordinates in 2D
    - It does not take into account the discretization size h, therefore, it must be homogeneous
    - For the same reason, if the discretization size is different than 1 there will be a factor to
      multiply/divide the operator
    - As well, there is no diffusion coefficient considered

    INPUT -> the x and y length
    OUTPUT -> FD Laplacian operator
    """
    north = np.arange(cxlen * (cylen - 1), cxlen * cylen)
    south = np.arange(cxlen)
    west = np.arange(0, cxlen * cylen, cxlen)
    east = np.arange(cxlen - 1, cxlen * cylen, cxlen)

    boundary = np.concatenate([north, south, east, west])

    row = np.array([], dtype=int)
    col = np.array([], dtype=int)
    data = np.array([])

    for i in range(cxlen * cylen):
        if i not in boundary:
            c = np.array([0, 1, -1, cxlen, -cxlen])
        else:
            if i in north:
                c = np.array([0, 1, -1, -cxlen])
            if i in south:
                c = np.array([0, 1, -1, cxlen])
            if i in east:
                c = np.array([0, cxlen, -cxlen, -1])
            if i in west:
                c = np.array([0, cxlen, -cxlen, 1])

            if i == 0:
                # corner sudwest
                c = np.array([0, 1, cxlen])
            if i == cxlen - 1:
                # sud east
                c = np.array([0, -1, cxlen])
            if i == cxlen * (cylen - 1):
                # north west
                c = np.array([0, 1, -cxlen])
            if i == cxlen * cylen - 1:
                c = np.array([0, -1, -cxlen])

        d = np.ones(len(c))
        d[0] = -len(c) + 1
        row = np.concatenate((row, np.zeros(len(c)) + i))
        col = np.concatenate((col, c + i))
        data = np.concatenate((data, d))

    operator = sp.sparse.csc_matrix(
        (data, (row, col)), shape=(cylen * cxlen, cxlen * cylen)
    )
    return operator
