function hcli=p_to_comp_hcli(point,varargin)
%% convert point to connecting orbit with complex conjugated eigenpairs
% M.Stöhr 2023/03/02: based on
% (c) DDE-BIFTOOL v. 2.02, 16/6/2002
%
% $Id: p_tohcli.m 308 2018-10-28 15:08:12Z jansieber $
%
% adapted to saddles with complex conjugated unstable eigendirections
%
% INPUT:
%     point a periodic solution near a homoclinic solution
%           alternatively an initial point in a hcli structure,
%           where a good starting guess for the profile and steady
%           states are available
%     named (but mandatory) 'funcs': problem functions
% OUTPUT:
%     comp_hcli a starting value to compute the exact homoclinic or
%     heteroclinic solution  
%
%%
%% for backward compatibility re-organize arguments
args=varargin;
if ~isfield(point,'kind')
    funcs=point;
    point=args{1};
    args=[args(2:end),{'funcs',funcs}];
end
default={'funcs',[]};
options=dde_set_options(default,args,'pass_on'); 
sys_tau=options.funcs.sys_tau;
sys_deri=options.funcs.sys_deri;

if mod(length(point.mesh),point.degree)~=1
    err=[length(point.mesh) point.degree];
    error('P_TOHCLI: psol does not contain L=%d intervals of m=% points!',...
        err(1),err(2));
end

hcli.kind='comp_hcli';
hcli.parameter=point.parameter;
hcli.mesh=point.mesh;
hcli.degree=point.degree;

switch point.kind
    case 'psol'
        ntst=size(point.profile,2);
        test=NaN(1,ntst-1);
        for i=1:ntst-1
            test(i)=norm(point.profile(:,i)-point.profile(:,i+1));
        end
        [minval, pos]=min(abs(test)); %#ok<ASGLU>
        stst.kind='stst';
        stst.parameter=hcli.parameter;
        stst.x=point.profile(:,pos);
        x_profile=NaN(1,ntst);
        for i=1:size(point.profile,2)
            x_profile(1,i)=norm(point.profile(:,i)-stst.x);
        end
        
        [peak, peak_pos]=max(x_profile); %#ok<ASGLU>
        [hole, hole_pos]=min(x_profile); %#ok<ASGLU>
        left_part=point.profile(:,1:peak_pos);
        right_part=point.profile(:,peak_pos+1:end);
        hole_begin=hole_pos-mod(hole_pos,point.degree)+1;
        hole_end=hole_begin+point.degree;
        
        if hole_pos<peak_pos
            right_part=[right_part left_part(:,2:hole_begin)];
            left_part=left_part(:,hole_end:end);
            hcli.mesh=[hcli.mesh(hole_end:end) ...
                (hcli.mesh(2:hole_begin)+1)];
        else
            left_part=[right_part(:,hole_end-peak_pos:end-1) left_part];
            right_part=right_part(:,1:hole_begin-peak_pos);
            hcli.mesh= ...
                [(hcli.mesh(hole_end:end-1)-1)...
                hcli.mesh(1:hole_begin)];
        end
        
        nb_of_points=length(hcli.mesh);
        rest=mod(nb_of_points,point.degree);
        hcli.profile=[left_part right_part];
        
        if rest>1
            hcli.profile=point.profile(:,1+floor((rest-1)/2):end-ceil((rest-1)/2));
            hcli.mesh=hcli.mesh(1+floor((rest-1)/2):end-ceil((rest-1)/2));
        end
        if rest==0
            rest=point.degree;
            hcli.profile=point.profile(:,1+floor((rest-1)/2):end-ceil((rest-1)/2));
            hcli.mesh=hcli.mesh(1+floor((rest-1)/2):end-ceil((rest-1)/2));
        end
        hcli.mesh=hcli.mesh-hcli.mesh(1);
        hcli.period=point.period*hcli.mesh(end);
        hcli.mesh=hcli.mesh/hcli.mesh(end);
        hcli.x1=stst.x;
        hcli.x2=stst.x;
        stst1=stst;
        stst2=stst;
    case {'hcli'}
        warning('case: not implemented/debugged yet')
        hcli=point;
        hcli.kind='comp_hcli';
        stst1.kind='stst';
        stst1.parameter=point.parameter;
        stst1.x=point.x1;
        stst2=stst1;
        stst2.x=point.x2;
    case 'comp_hcli'
        hcli=point;
        stst1.kind='stst';
        stst1.parameter=point.parameter;
        stst1.x=point.x1;
        stst2=stst1;
        stst2.x=point.x2;
    otherwise
        error(['P_TOHCLI: not a valid conversion for other than psol' ...
            ' or hcli type points']);
end

m=df_mthod('stst','cheb');

%% compute stability and right eigenvectors at stst1

stst1.stability=p_stabil(options.funcs,stst1,m.stability);

if isempty(stst1.stability.l1) || max(real(stst1.stability.l1))<0
    error('P_TOHCLI: no unstable eigenmodes found');
end

gamma_v=real(stst1.stability.l1(:));
omega_v=imag(stst1.stability.l1(:));
v_sel=real(gamma_v)>0;
gamma_v=gamma_v(v_sel);
omega_v=omega_v(v_sel);
hcli.gamma_v=gamma_v;
hcli.omega_v=omega_v;

% normalize eigenvectors
v=stst1.stability.v(:,v_sel);
vn=sqrt(sum(v.*conj(v),1));
v=v./vn(ones(length(v(:,1)),1),:);

% set first coordinate of imag(v_k) to 0 
for k=1:length(v(1,:))
    theta = atan2(imag(v(1,k)),real(v(1,k))); %angle(v(:,k))
    v(:,k)=v(:,k)*exp(-1i.*theta);
end

hcli.v_r=real(v);
hcli.v_i=imag(v);


%% compute stability and left eigenvectors at stst2

stst2.stability=p_stabil(options.funcs,stst2,m.stability);

gamma_w=real(stst2.stability.l1(:));
omega_w=imag(stst2.stability.l1(:));
w_sel=real(gamma_w)>0;
gamma_w=gamma_w(w_sel);
omega_w=omega_w(w_sel);
hcli.gamma_w=gamma_w;
hcli.omega_w=omega_w;

%normalize eigenvectors
w=stst1.stability.w(:,w_sel);
wn=sqrt(sum(w.*conj(w),1));
w=w./wn(ones(length(w(:,1)),1),:);

% set first coordinate of imag(w_k) to 0 
for k=1:length(w(1,:))
    theta = atan2(imag(w(1,k)),real(w(1,k))); %angle(w(:,k))
    w(:,k)=w(:,k)*exp(-1i.*theta);
end

hcli.w_r=real(w);
hcli.w_i=imag(w);


%% compute alphas and epsilon
% left boundary condition: x(0)= x_1 + epsilon*sum(alpha_k*v_k)
% now adapted to x(0)= x_1 + epsilon*sum(alpha_k*real(v_k)+alpha_(k+1)*imag(v_k))

v_real=[hcli.v_r(:,1), hcli.v_i(:,1),...
        hcli.v_r(:,3), hcli.v_i(:,3)];

hcli.alpha=v_real\(hcli.profile(:,1)-hcli.x1);
hcli.epsilon=norm(hcli.alpha);
hcli.alpha=hcli.alpha/hcli.epsilon;

end
