# PACKAGES ####
library(car)
library(tidyverse)
library(survival)
library(caTools)
library(ggtext)
library(gridExtra)

# DATA ####
# Set working directory
setwd("WORKING DIRECTORY")

# Load data for Analysis 1
nests <- read.csv("nest_data_final.csv", header= TRUE, stringsAsFactors = TRUE)
str(nests)
nests$ant_species <- relevel(nests$ant_species, ref = "RRB")
nests$height_m <- nests$height_cm/100

# Load data for Analysis 2
nests_tmp <- read.csv("nest_data_tmp.csv", header= TRUE, stringsAsFactors = TRUE)

# ANALYSIS ####
# First analysis
Mlr <- glm(case_control ~ alive + height_m + canopy_area + ant_species, data = nests)
summary(Mlr)
confint(Mlr)
vif(Mlr)

Mlr_results <- data.frame(variable=c("Intercept", "Live", "Height", "Canopy Area", 
                                     "C. sjostedti", "C. nigriceps", "No ants", "T. penzigi"))
Mlr_results$est <- exp(summary(Mlr)$coefficients[,1])
Mlr_results$lcl <- exp(confint(Mlr)[,1])
Mlr_results$ucl <- exp(confint(Mlr)[,2])
Mlr_results <- Mlr_results[2:8,]

Mlr_results <- Mlr_results %>%
  mutate(name = fct_relevel(variable, 
                            "Live", "Height", "Canopy Area", 
                            "No ants", "T. penzigi", "C. sjostedti", "C. nigriceps"))

# Second analysis
nests_tmp <- nests_tmp %>%
  mutate(ant_species = fct_relevel(ant_species, 
                            "RRB", "BBR", "AB", "TP", "TP/AB"))

Mlr2 <- glm(case_control ~ height_m + ant_species, data = nests_tmp)
summary(Mlr2)
confint(Mlr2)
vif(Mlr2)

Mlr2_results <- data.frame(variable=c("Intercept", "Height", "C. nigriceps", 
                                     "C. sjostedti", "T. penzigi", 
                                     "T. penzigi + C. sjostedti", "No ants"))
Mlr2_results$est <- exp(summary(Mlr2)$coefficients[,1])
Mlr2_results$lcl <- exp(confint(Mlr2)[,1])
Mlr2_results$ucl <- exp(confint(Mlr2)[,2])
Mlr2_results <- Mlr2_results[2:7,]

Mlr2_results <- Mlr2_results %>%
  mutate(name = fct_relevel(variable, 
                            "Height",
                            "No ants", "T. penzigi + C. sjostedti", 
                            "T. penzigi", "C. sjostedti", "C. nigriceps"))


# GRAPHS ####
# Graph of regression coefficients
# Analysis 1
xlabels <- c("*C. nigriceps*",
  "*C. sjostedti*",
  "*T. penzigi*",
  "No ants",
  "Canopy Area (m<sup>2</sup>)",
  "Height (m)",
  "Live")
  
fig1a <- Mlr_results %>%
  mutate(name = fct_relevel(variable, 
                            "C. nigriceps", "C. sjostedti", "T. penzigi", "No ants", 
                            "Canopy Area", "Height", "Live")) %>%
ggplot(aes(x=name, y=est)) + 
  geom_pointrange(aes(ymin = lcl, ymax = ucl), size=1.25) +
  geom_hline(yintercept = 1, linetype="dashed", size=1.25) +
  scale_y_continuous(trans='log', limits = c(0.3, 1.6), breaks = c(0.4, 0.8, 1.2)) +
  scale_x_discrete(labels = xlabels) +
  coord_flip() +
  ylab("Odds Ratio") +
  xlab("") +
  theme_classic() +
  theme(text=element_text(size=30), axis.text.y = element_markdown())

# Analysis 2
xlabels2 <- c("*C. nigriceps*",
             "*C. sjostedti*",
             "*T. penzigi*",
             "*T. penzigi* + *C. sjostedti*",
             "No ants",
             "Height (m)")


fig1b <- Mlr2_results %>%
  mutate(name = fct_relevel(variable, 
                            "C. nigriceps",  "C. sjostedti", "T. penzigi", 
                            "T. penzigi + C. sjostedti", "No ants", "Height")) %>%
  ggplot(aes(x=name, y=est)) + 
  geom_pointrange(aes(ymin = lcl, ymax = ucl), size=1.25) +
  geom_hline(yintercept = 1, linetype="dashed", size=1.25) +
  scale_y_continuous(trans='log', limits = c(0.3, 1.6), breaks = c(0.4, 0.8, 1.2)) +
  scale_x_discrete(labels = xlabels2) +
  coord_flip() +
  ylab("Odds Ratio") +
  xlab("") +
  theme_classic() +
  theme(text=element_text(size=30), axis.text.y = element_markdown())

grid.arrange(fig1a, fig1b, nrow = 1)
