function [r,stats] = registerRegions(warpImage,PC,rPrev,frame,options)
debug = false; % turn this on to plot each registration
r = rPrev; % preallocate r
warpingFcn = options.fwdWarpingFcn; % grab the warping function for convenience
composeP = options.composition; % grab the composition function for convenience
hTemp = []; hWarp = []; hDiff = []; %#ok<NASGU> % set a null hTemp for debugging
collectStats = nargout == 2; % if the program is asking for 2 outputs, also collect stats on the optimization
if collectStats % stats are requested, preallocate the arrays for them
    dPAll = r; % should be same size as r
    iterAll = zeros(1,numel(PC)); % just an array of numbers
    imerrorAll = zeros(1,numel(PC)); % just an array of numbers
    converged = ones(1,numel(PC)); % preallocate an array for convergence
end
%% loop through each point and process it
loopTime = tic; % calculate the time it takes to loop
parfor i = 1:numel(PC) % loop through each point
    Hinv = PC(i).Hinv; % get the inverse Hessian
    VTdWdP = PC(i).VTdWdP; % get VTdWdP
    P = rPrev(i,:); % grab the previous parameters
    X = PC(i).X.x; % get the original X coordinates
    Y = PC(i).X.y; % get the original Y coordinates
    temp = PC(i).template; % get template
    iterCount = 0; % initialize an iteration counter
    stop = false; % set stop to off so that the opimization initializes
    failed = false; % set failed to false so that the opimization initializes
    if debug % if debugging initialize the debug plot
        [hTemp,hWarp,hDiff] = initializeDebugPlot(temp,hTemp,hWarp,hDiff); %#ok<UNRCH>
    end
    while ~stop && ~failed
        % increment the iteration counter
        iterCount = iterCount+1;       
        [x,y] = transform2D(X, Y, warpingFcn, P , PC(i).N);
        % then take those x y coordinates and grab what they are from the
        % warped image.
        warped = warpImage(x, y);        
        imerror = warped-temp; % calculate the image error     
        % calculate a parameter update:
        %      dP = H^-1 * VTdWdP^T * (warped - template)
        dP =  Hinv * VTdWdP' * imerror(:);
        % produce a set of new parameters by composing P and dP -> P
        P = feval(composeP,P,dP);
        % check if the optimization is complete
        [stop,failed] = checkOptimzation(iterCount,dP,options);       
        if debug % if debugging is enabled, update the plots
            updateDebugPlots(warped,imerror,hWarp,hDiff); %#ok<UNRCH>
            pause;
        end
    end
    % collect results for point
    if failed && collectStats
        converged(i) = false;
        %keyboard;
    elseif iterCount == 1 % if only one iteration passed, consider the regions identical
        r(i,:) = rPrev(i,:); % and use the previous values of P
    else % otherwise use the updated value
        r(i,:)= P;
    end
    % gather optimization statistics for point
    if collectStats
        dPAll(i,:) = dP';
        iterAll(i) = iterCount;
        imerrorAll(i) = sum(abs(imerror(:)));
    end
end
%% cleaning up and displaying the results
frameTime = toc(loopTime); % save the frame time
if nargout == 1 % check how many output arguments we have, if there's only one output a simple result
    fprintf('Frame : %d \t Processed %d regions in %0.3f seconds\n',frame,numel(PC),frameTime);
else % otherwise collect optimization statistics
    epsilonMean = mean(sum(abs(dPAll),2)); % get the mean of the epsilon values
    epsilonMin = min(sum(abs(dPAll),2));
    iterMean = mean(iterAll); % get the mean number of iterations
    iterMax = max(iterAll); % get hte max number of iterations
    meanImError = mean(imerrorAll); % find the mean image error
    %fprintf('Frame: %d \t Elapsed: %0.3f \t Epsilon: %f \t Iterations: %0.1f \t Max: %d \t ImageError: %f\n',frame,frameTime,epsilonMean,iterMean,iterMax,meanImError);
    fprintf('Frame: %d \t Elapsed: %0.3f \t Epsilon: %0.4f \t Min: %0.4f \t Iterations: %0.1f \t Max: %d \t ImageError: %0.0f\n',frame,frameTime,epsilonMean,epsilonMin,iterMean,iterMax,meanImError);
    stats.dP = dPAll; % save the dP values
    stats.iterations = iterAll; % save the iterations
    stats.imageError = imerrorAll; % save the imageerrors
    stats.converged = converged; % save if the region converged
%     stats.pixPerRegion = numel(imerror); % LGW - for convenience
%     stats.maxAbsIntensity = 
end
end

function [stop,failed] = checkOptimzation(iterCount,dP,options)
o = options.Optimization; % optimoptions
% -------------------------------------
% check if the optimization is complete
% -------------------------------------
convergence = sum(abs(dP)) < o.epsilon; % check if dP < epsilon

% -------------------------------------
% check if the opimization has failed
% -------------------------------------
maxIters = iterCount > o.maximumIterations; % check if we're above maximum interations

% -------------------------------------
% compile the results
% -------------------------------------
stop = convergence;
failed = maxIters;
end

function [hTemp,hWarp,hDiff] = initializeDebugPlot(temp,hTemp,hWarp,hDiff)
if isempty(hTemp) % set up a loop to display the image registration
    subplot(1,3,1); hTemp = imagesc(temp); % use temp as a default to initialize
    subplot(1,3,2); hWarp = imagesc(temp); % ""
    subplot(1,3,3); hDiff = imagesc(temp); % ""
else
    set(hTemp,'CData',temp) % if the loop is already active just update the template image
end
end

function updateDebugPlots(warped,imerror,hWarp,hDiff)
set(hWarp,'CData',warped)
set(hDiff,'CData',imerror)
drawnow;
pause(0.01);
end

