function gif(varargin)
% gif is the simplest way to make gifs. Simply call
% 
%   gif('myfile.gif') 
% 
% to write the first frame, and then call 
% 
%   gif
% 
% to write each subsequent frame. That's it. 
% 
%% Syntax
% 
%  gif('filename.gif') 
%  gif(...,'DelayTime',DelayTimeValue,...) 
%  gif(...,'LoopCount',LoopCountValue,...) 
%  gif(...,'frame',handle,...) 
%  gif(...,'nodither') 
%  gif 
%  gif('clear') 
% 
%% Description 
% 
% gif('filename.gif') writes the first frame of a new gif file by the name filename.gif. 
% 
% gif(...,'DelayTime',DelayTimeValue,...) specifies a the delay time in seconds between
% frames. Default delay time is 1/15. 
% 
% gif(...,'LoopCount',LoopCountValue,...) specifies the number of times the gif animation 
% will play. Default loop count is Inf. 
% 
% gif(...,'frame',handle,...) uses the frame of the given figure or set of axes. The default 
% frame handle is gca, meaning the current axes. To turn an entire figure window into a gif, 
% use 'frame',gcf to use the current figure. 
% 
% gif(...,'nodither') maps each color in the original image to the closest color in the new 
% without dithering. Dithering is performed by default to achieve better color resolution, 
% albeit at the expense of spatial resolution.
% 
% gif adds a frame to the current gif file. 
% 
% gif('clear') clears the persistent variables associated with the most recent gif. 
% 
%% Example 
% 
%    % Some sample data: 
%    t = sin(linspace(0,2*pi,30)); 
%    [X,Y,Z] = peaks(500); 
% 
%    % Plot the first frame: 
%    h = surf(X,Y,Z*t(1)); 
%    shading interp
%    axis([-3 3 -3 3 -9 9])
% 
%    % Create a new gif file and write the first frame: 
%    gif('myfile.gif') % Or specify options like this: gif('myfile.gif','DelayTime',0.2,'LoopCount',5,'frame',gcf)
% 
%    % Loop through every other frame. 
%    for k = 2:length(t) 
%       set(h,'Zdata',Z*t(k))
%       gif
%    end
% 
%    % View the gif in Matlab if you'd like: 
%    web('myfile.gif') 
% 
%% Author Information 
% This function was written by Chad A. Greene of the University of Texas 
% Institute for Geophysics (UTIG), June 2017. 
% 
% See also: imwrite, getframe, and rgb2ind. 

% Define persistent variables: 
persistent gif_filename firstframe DelayTime DitherOption LoopCount frame

%% Parse Inputs

if nargin>0 
   
   % The user may want to clear things and start over: 
   if any(strcmpi(varargin,'clear'))
            
      % Clear persistent variables associated with this function: 
      clear gif_filename firstframe DelayTime DitherOption LoopCount frame
   end
   
   % If the first input ends in .gif, assume this is the first frame:
   if strcmpi(varargin{1}(end-3:end),'.gif')
      
      % This is what the user wants to call the new .gif file: 
      gif_filename = varargin{1}; 
      
      % Check for an existing .gif file by the same name: 
      if exist(gif_filename,'file')==2
         
         % Ask the user if (s)he wants to overwrite the existing file: 
         choice = questdlg(['The file  ',gif_filename,' already exists. Overwrite it?'], ...
            'The file already exists.','Overwrite','Cancel','Cancel');
         
         % Overwriting basically means deleting and starting from scratch: 
         if strcmp(choice,'Overwrite')
            delete(gif_filename) 
         else 
            clear gif_filename firstframe DelayTime DitherOption LoopCount frame
            error('The giffing has been canceled.') 
         end
         
      end
      
      firstframe = true; 
      
      % Set defaults: 
      DelayTime = 1/15; 
      DitherOption = 'dither'; 
      LoopCount = Inf; 
      frame = gca; 
   end
   
   tmp = strcmpi(varargin,'DelayTime'); 
   if any(tmp) 
      DelayTime = varargin{find(tmp)+1}; 
      assert(isscalar(DelayTime)==1,'Error: DelayTime must be a scalar value.') 
   end
   
   if any(strcmpi(varargin,'nodither'))
      DitherOption = 'nodither'; 
   end
   
   tmp = strcmpi(varargin,'LoopCount'); 
   if any(tmp) 
      LoopCount = varargin{find(tmp)+1}; 
      assert(isscalar(LoopCount)==1,'Error: LoopCount must be a scalar value.') 
   end
   
   tmp = strcmpi(varargin,'frame'); 
   if any(tmp) 
      frame = varargin{find(tmp)+1}; 
      assert(ishandle(frame)==1,'Error: frame must be a figure handle or axis handle.') 
   end
   
else
   assert(isempty(gif_filename)==0,'Error: The first call of the gif function requires a filename ending in .gif.') 
end

%% Perform work: 

% Get frame: 
f = getframe(frame); 

% Convert the frame to a colormap and corresponding indices: 
[imind,cmap] = rgb2ind(f.cdata,256,DitherOption);    

% Write the file:     
if firstframe
   imwrite(imind,cmap,gif_filename,'gif','LoopCount',LoopCount,'DelayTime',DelayTime)
   firstframe = false;
else
   imwrite(imind,cmap,gif_filename,'gif','WriteMode','append','DelayTime',DelayTime)
end

end