
options(stringsAsFactors = FALSE)

# ------------------------------------------------------------------------------------
# load data

dir <- "file-path-to-data"
setwd(dir)

dat <- read.csv("MHH_data.csv")
str(dat)
     
     
# ------------------------------------------------------------------------------------
# load and install binary packages

ipak <- function(pkg){
    new.pkg <- pkg[!(pkg %in% installed.packages()[, "Package"])]
    if(length(new.pkg)) install.packages(new.pkg, dependencies = TRUE)
    sapply(pkg, require, character.only = TRUE)
}

packages <- c("ggplot2", "scales", "grid", "gridExtra", "reshape2", "plyr",  
                         "texreg", "lsmeans", "RColorBrewer", "effects", "xtable")
ipak(packages)

# devtools::install_github("lionel-/ggstance")
library(ggstance)


# ------------------------------------------------------------------------------------
# reshape data

dat_long <- melt(dat, measure.vars = c("npp_max_15km", "npp_mean_15km"), 
    variable.name = "max_mean_NPP", value.name = "npp_15km")
str(dat_long)

dat_long$max_mean_NPP <- factor(gsub(".+\\_(\\w+)\\_.+", "\\1", dat_long$max_mean_NPP))
dat_long$max_mean_NPP <- factor(dat_long$max_mean_NPP, 
    levels = c("mean", "max"), labels = c("Mean", "Max"))
    

dat_long120 <- melt(dat, measure.vars = c("npp_max_120km", "npp_mean_120km"), 
    variable.name = "max_mean_NPP", value.name = "npp_120km")
str(dat_long120)

dat_long120$max_mean_NPP <- factor(gsub(".+\\_(\\w+)\\_.+", "\\1", dat_long120$max_mean_NPP))
dat_long120$max_mean_NPP <- factor(dat_long120$max_mean_NPP, 
     levels = c("mean", "max"), labels = c("Mean", "Max"))

 
# ------------------------------------------------------------------------------------
# set global ggplot theme
    
theme_nice <- theme_set(theme_bw())
theme_nice <- theme_update(
    axis.title.x = element_text(size = 10, margin = margin(t=10)),
    axis.title.y = element_text(size = 10, angle = 90, margin = margin(r=10)),
    axis.text.x = element_text(size = 8, angle = 0, margin = margin(t=2), color = "black"),
    axis.text.y = element_text(size = 8, hjust = 1, margin = margin(r=2), color = "black"),
    # strip.background = element_rect(color = "grey", fill = "white"),
    strip.text = element_text(size = 10),
    strip.background = element_blank(),
    panel.border = element_rect(color = "grey20"),
    panel.spacing.x = unit(0.2, "lines"),
    panel.spacing.y = unit(0.05,"lines"),
    panel.grid.minor = element_blank(),
    panel.grid.major.y = element_blank(),
    # explicitly set the horizontal lines (or they will disappear too)
    panel.grid.major.x = element_line(size=.5, color="#f0f0f0"),
    legend.title = element_text(size = 8, face = "plain"),
    legend.key.size = unit(.8, units = "line"),
    legend.text = element_text(size = 8),
    legend.background = element_rect(fill = "gray95", color = "gray20",
                                     size = 0.5, linetype = "dotted"))



##############################################################################
##############################################################################
##############################################################################
# 15km radius OLS models for figure 2 split by mean and max NPP

# -------------------------------------------
model_mm15c <- lm(npp_15km ~ subsistence + 
                             #climate +
                             ET + 
                             mean_annual_precip_mm + 
                             latitude_abs +
                             #I(latitude_abs^2) +
                             max_mean_NPP +
                             #marine_dist +
                             #mobility_bin +
                             #fishing_bin +
                             subsistence:max_mean_NPP,
              data = dat_long)
summary(model_mm15c)

eff_mm15c <- allEffects(model_mm15c)
eff_df_mm15c <- as.data.frame(eff_mm15c[["subsistence:max_mean_NPP"]])


# -------------------------------------------
model_mm15b <- lm(npp_15km ~ subsistence_ternary + 
                                  #climate +
                                  ET + 
                                  mean_annual_precip_mm + 
                                  latitude_abs +
                                  #I(latitude_abs^2) +
                                  max_mean_NPP +
                                  #marine_dist +
                                  #mobility_bin +
                                  #fishing_bin +
                                  subsistence_ternary:max_mean_NPP,
              data = dat_long)
summary(model_mm15b)

eff_mm15b <- allEffects(model_mm15b)
eff_df_mm15b <- as.data.frame(eff_mm15b[["subsistence_ternary:max_mean_NPP"]])


# -------------------------------------------
model_mm15a <- lm(npp_15km ~ subsistence_binary + 
                             #climate +
                             ET + 
                             mean_annual_precip_mm + 
                             latitude_abs +         
                             #I(latitude_abs^2)                   
                             max_mean_NPP +
                             #marine_dist +
                             #mobility_bin +
                             #fishing_bin +
                             subsistence_binary:max_mean_NPP,
              data = dat_long)
summary(model_mm15a)

eff_mm15a <- allEffects(model_mm15a)
eff_df_mm15a <- as.data.frame(eff_mm15a[["subsistence_binary:max_mean_NPP"]])


# -------------------------------------------
# effects combined

eff_df_mm15a$model <- "A"
eff_df_mm15b$model <- "B"
eff_df_mm15c$model <- "C"
colnames(eff_df_mm15a)[1] <- "subsistence"
colnames(eff_df_mm15b)[1] <- "subsistence"

eff_df_comb_mean_max_15 <- rbind(eff_df_mm15a, eff_df_mm15b, eff_df_mm15c)

eff_df_comb_mean_max_15$model <- factor(eff_df_comb_mean_max_15$model, levels = c("A", "B", "C"))

eff_df_comb_mean_max_15$max_mean_NPP <- factor(eff_df_comb_mean_max_15$max_mean_NPP, 
    levels = c("Mean", "Max"), labels = c("mean", "max"))

eff_df_comb_mean_max_15$subsistence <- factor(eff_df_comb_mean_max_15$subsistence,
    levels = rev(c("Foraging", "Agriculture", "Horticulture", "Intensive agriculture", "Pastoralism")))


# -------------------------------------------
# plot 

mean_max_npp15 <- ggplot(eff_df_comb_mean_max_15, aes(x = fit,
        y = subsistence, group = max_mean_NPP, linetype = max_mean_NPP)) +
    geom_point(shape = 16, size = 2.5, position = position_dodgev(height = 0.4)) +
    geom_linerangeh(aes(xmin = lower, xmax = upper), size = 0.7, position = position_dodgev(height = 0.4)) +
    scale_x_continuous(breaks = seq(100, 1200, by = 100)) +
    facet_grid(model ~ ., scales = "free", space = "free") +
    labs(x = bquote("NPP 15 km radius (g C/" *m^2* "/year)"), y = "") +
    guides(linetype = guide_legend(title = "NPP", reverse = TRUE)) +
    theme(legend.position = c(0.18, 0.89),
          strip.text.y = element_text(angle = 0))
ggsave(mean_max_npp15, file = "figure_2.pdf", height = 4, width = 5)


# -------------------------------------------
# tables using lsmeans

preds_mm15a <- lsmeans(model_mm15a, specs = ~ subsistence, at = list(max_mean_NPP = "max"))
contr_mm15a <- contrast(preds_mm15a, method = "pairwise", adjust = "holm")
confint(contr_mm15a)
contr_mm15a <- as.data.frame(summary(contr_mm15a))

contr_mm15a_tab <- xtable(    
    x = contr_mm15a, 
    caption = "Subsistence 4-way contrasts, warm and cold climate combined.", 
    align = c("l", "l", "r", "r", "r", "r", "r"), 
    digits = c(0, 0, 0, 1, 0, 2, 4)
)

print(
    x = contr_mm15a_tab,
    type = "html",
    file = "reg_contr_mm15a.doc",
    caption.placement = "top"
)

# 
preds_mm15c <- lsmeans(model_mm15c, specs = ~ subsistence_binary, at = list(max_mean_NPP = "max"))
contr_mm15c <- contrast(preds_mm15c, method = "pairwise", adjust = "holm")
confint(contr_mm15c)
contr_mm15c <- as.data.frame(summary(contr_mm15c))
#confint(contr_mm15c)

contr_mm15c_tab <- xtable(    
    x = contr_mm15c, 
    caption = "Subsistence 2-way contrasts, warm and cold climate combined.", 
    align = c("l", "l", "r", "r", "r", "r", "r"), 
    digits = c(0, 0, 0, 1, 0, 2, 4)
)

print(
    x = contr_mm15c_tab,
    type = "html",
    file = "reg_contr_mm15c.doc",
    caption.placement = "top"
)

#
preds_mm15b <- lsmeans(model_mm15b, specs = ~ subsistence_ternary, at = list(max_mean_NPP = "max"))
contr_mm15b <- contrast(preds_mm15b, method = "pairwise", adjust = "holm")
confint(contr_mm15b)
contr_mm15b <- as.data.frame(summary(contr_mm15b))
#confint(contr_mm15b)

contr_mm15b_tab <- xtable(    
    x = contr_mm15b, 
    caption = "Subsistence 3-way contrasts, warm and cold climate combined.", 
    align = c("l", "l", "r", "r", "r", "r", "r"), 
    digits = c(0, 0, 0, 1, 0, 2, 4)
)

print(
    x = contr_mm15b_tab,
    type = "html",
    file = "reg_contr_mm15b.doc",
    caption.placement = "top"
)





##############################################################################
##############################################################################
##############################################################################
# 120km radius OLS models for figure 2 split by mean and max NPP

# -------------------------------------------
model_mm120c <- lm(npp_120km ~ subsistence + 
                             #climate +
                             ET + 
                             mean_annual_precip_mm + 
                             latitude_abs +
                             #I(latitude_abs^2) +
                             max_mean_NPP +
                             #marine_dist +
                             #mobility_bin +
                             #fishing_bin +
                             subsistence:max_mean_NPP,
              data = dat_long120)
summary(model_mm120c)

eff_mm120c <- allEffects(model_mm120c)
eff_df_mm120c <- as.data.frame(eff_mm120c[["subsistence:max_mean_NPP"]])


# -------------------------------------------
model_mm120b <- lm(npp_120km ~ subsistence_ternary + 
                                  #climate +
                                  ET + 
                                  mean_annual_precip_mm + 
                                  latitude_abs +
                                  #I(latitude_abs^2) +
                                  max_mean_NPP +
                                  #marine_dist +
                                  #mobility_bin +
                                  #fishing_bin +
                                  subsistence_ternary:max_mean_NPP,
              data = dat_long120)
summary(model_mm120b)

eff_mm120b <- allEffects(model_mm120b)
eff_df_mm120b <- as.data.frame(eff_mm120b[["subsistence_ternary:max_mean_NPP"]])


# -------------------------------------------
model_mm120a <- lm(npp_120km ~ subsistence_binary + 
                             #climate +
                             ET + 
                             mean_annual_precip_mm + 
                             latitude_abs +     
                             #I(latitude_abs^2)                       
                             max_mean_NPP +
                             #marine_dist +
                             #mobility_bin +
                             #fishing_bin +
                             subsistence_binary:max_mean_NPP,
              data = dat_long120)
summary(model_mm120a)

eff_mm120a <- allEffects(model_mm120a)
eff_df_mm120a <- as.data.frame(eff_mm120a[["subsistence_binary:max_mean_NPP"]])


# -------------------------------------------
# effects combined

eff_df_mm120a$model <- "A"
eff_df_mm120b$model <- "B"
eff_df_mm120c$model <- "C"
colnames(eff_df_mm120a)[1] <- "subsistence"
colnames(eff_df_mm120b)[1] <- "subsistence"

eff_df_comb_mean_max_120 <- rbind(eff_df_mm120a, eff_df_mm120b, eff_df_mm120c)

eff_df_comb_mean_max_120$model <- factor(eff_df_comb_mean_max_120$model, levels = c("A", "B", "C"))

eff_df_comb_mean_max_120$max_mean_NPP <- factor(eff_df_comb_mean_max_120$max_mean_NPP, 
    levels = c("Mean", "Max"), labels = c("mean", "max"))

eff_df_comb_mean_max_120$subsistence <- factor(eff_df_comb_mean_max_120$subsistence,
    levels = rev(c("Foraging", "Agriculture", "Horticulture", "Intensive agriculture", "Pastoralism")))


# -------------------------------------------
# plot 

mean_max_npp120 <- ggplot(eff_df_comb_mean_max_120, aes(x = fit,
        y = subsistence, group = max_mean_NPP, linetype = max_mean_NPP)) +
    geom_point(shape = 16, size = 2.5, position = position_dodgev(height = 0.4)) +
    geom_linerangeh(aes(xmin = lower, xmax = upper), size = 0.7, position = position_dodgev(height = 0.4)) +
    scale_x_continuous(breaks = seq(100, 1500, by = 100)) +
    facet_grid(model ~ ., scales = "free", space = "free") +
    labs(x = bquote("NPP 120 km radius (g C/" *m^2* "/year)"), y = "") +
    guides(linetype = guide_legend(title = "NPP", reverse = TRUE)) +
    theme(legend.position = c(0.16, 0.89),
          strip.text.y = element_text(angle = 0))
ggsave(mean_max_npp120, file = "figure_S1.pdf", height = 4, width = 6)


# -------------------------------------------
# tables using lsmeans

preds_mm120a <- lsmeans(model_mm120a, specs = ~ subsistence, at = list(max_mean_NPP = "max"))
contr_mm120a <- contrast(preds_mm120a, method = "pairwise", adjust = "holm")
confint(contr_mm120a)
contr_mm120a <- as.data.frame(summary(contr_mm120a))

contr_mm120a_tab <- xtable(    
    x = contr_mm120a, 
    caption = "Subsistence 4-way contrasts, warm and cold climate combined.", 
    align = c("l", "l", "r", "r", "r", "r", "r"), 
    digits = c(0, 0, 0, 1, 0, 2, 4)
)

print(
    x = contr_mm120a_tab,
    type = "html",
    file = "reg_contr_mm120a.doc",
    caption.placement = "top"
)

#
preds_mm120c <- lsmeans(model_mm120c, specs = ~ subsistence_binary, at = list(max_mean_NPP = "max"))
contr_mm120c <- contrast(preds_mm120c, method = "pairwise", adjust = "holm")
confint(contr_mm120c)
contr_mm120c <- as.data.frame(summary(contr_mm120c))

contr_mm120c_tab <- xtable(    
    x = contr_mm120c, 
    caption = "Subsistence 2-way contrasts, warm and cold climate combined.", 
    align = c("l", "l", "r", "r", "r", "r", "r"), 
    digits = c(0, 0, 0, 1, 0, 2, 4)
)

print(
    x = contr_mm120c_tab,
    type = "html",
    file = "reg_contr_mm120c.doc",
    caption.placement = "top"
)

#
preds_mm120b <- lsmeans(model_mm120b, specs = ~ subsistence_ternary, at = list(max_mean_NPP = "max"))
contr_mm120b <- contrast(preds_mm120b, method = "pairwise", adjust = "holm")
confint(contr_mm120b)
contr_mm120b <- as.data.frame(summary(contr_mm120b))

contr_mm120b_tab <- xtable(    
    x = contr_mm120b, 
    caption = "Subsistence 3-way contrasts, warm and cold climate combined.", 
    align = c("l", "l", "r", "r", "r", "r", "r"), 
    digits = c(0, 0, 0, 1, 0, 2, 4)
)

print(
    x = contr_mm120b_tab,
    type = "html",
    file = "reg_contr_mm120b.doc",
    caption.placement = "top"
)


