
options(stringsAsFactors = FALSE)

# ------------------------------------------------------------------------------------
# load data

dir <- "file-path-to-data"
setwd(dir)

dat <- read.csv("MHH_data.csv")
str(dat)


# ------------------------------------------------------------------------------------
# load and install binary packages

ipak <- function(pkg){
    new.pkg <- pkg[!(pkg %in% installed.packages()[, "Package"])]
    if(length(new.pkg)) install.packages(new.pkg, dependencies = TRUE)
    sapply(pkg, require, character.only = TRUE)
}

packages <- c("ggplot2", "scales", "reshape2", "plyr", "multcomp", "rgeos",
              "sp", "maps", "maptools", "rgdal", "raster", "mapproj", "RColorBrewer")
ipak(packages)

                           
# ------------------------------------------------------------------------------------
# projection strings

# proj4 strings
proj4_long_lat <- "+proj=longlat +datum=WGS84 +no_defs +ellps=WGS84 +towgs84=0,0,0"
proj4_Mollweide <- "+proj=moll +lon_0=0 +x_0=0 +y_0=0 +ellps=WGS84 +datum=WGS84 +units=m +no_defs"   


# ------------------------------------------------------------------------------------
# biome data

# read in biome shape file
biomes <- readOGR(".", layer = "tnc_terr_ecoregions")

# rename long strings
biomes@data$WWF_MHTNAM[biomes@data$WWF_MHTNAM %in% c("Temperate Grasslands, Savannas and Shrublands", 
    "Tropical and Subtropical Grasslands, Savannas and Shrublands")] <- c("Temperate Grasslands, Savannas and Shrub", 
    "Tropical and Subtropical Grasslands, Savannas and Shrub")

# project biomes
biomes <- spTransform(biomes, CRS(proj4_Mollweide)) 

biome_pal <- c("#659D32", "#F3B962", "#8DD3C7", "#6495ed",
               "#F0D1E1", "#55AE3A", "#A2C257", "#eee9e9",
               "#458B00", "#3D9140", "#FFED6F", "#228b22",
               "#215E21", "#BCD868", "#6b8e23", "#C191C2")

biome_palette <- with(dat,
    data.frame(biomes = levels(factor(biomes@data$WWF_MHTNAM)),
               color = biome_pal))

pch_palette <- with(dat,
    data.frame(subsistence = sort(unique(subsistence)),
               pch = c(8, 0, 2, 1)))

# ------------------------------------------------------------------------------------
# society points

# create SpatialPointsDataFrame object (unprojected)
dat_sp <- SpatialPointsDataFrame(
    coords = dat[, c("longitude", "latitude")], 
        data = dat[, !(colnames(dat) %in% c("longitude", "latitude"))],
        proj4string = CRS(proj4_long_lat)
        )

# project coordinates
dat_sp <- spTransform(dat_sp, CRS(proj4_Mollweide)) 

data(wrld_simpl) # maptools
wrld_simpl_Mollweide <- spTransform(wrld_simpl, CRS(proj4_Mollweide))

png(file = "figure_1.png", height = 5.4, width = 7.5, units = "in", res = 400)
    # plot biomes
    plot(coordinates(dat_sp), type = "n", asp = 1, xaxt = "n", yaxt = "n", xlab = "", ylab = "", xlim = c(-1.71e+07, 1.74e+07))
    title(xlab = "Longitude", ylab = "Latitude", line = 0.3, cex.lab = 0.8)
    llgridlines(dat_sp, plotLabels = TRUE, col = "grey25", cex = 0.5, offset = 0.35, lwd = 0.7)
    plot(wrld_simpl_Mollweide, add = TRUE, asp = 1)
    plot(biomes, col = biome_palette$color[match(biomes@data$WWF_MHTNAM, biome_palette$biomes)],
         border = "grey30", lwd = 0.2, add = TRUE, asp = 1) # need character strings, not factors
    # add points
    points(x = coordinates(dat_sp)[dat_sp@data$climate %in% "warm", "longitude"], 
               y = coordinates(dat_sp)[dat_sp@data$climate %in% "warm", "latitude"], 
           pch = pch_palette$pch[match(dat_sp@data$subsistence[dat_sp@data$climate %in% "warm"], pch_palette$subsistence)], 
           col = "#D55E00", cex = 0.4, asp = 1)
    points(x = coordinates(dat_sp)[dat_sp@data$climate %in% "cold", "longitude"], 
               y = coordinates(dat_sp)[dat_sp@data$climate %in% "cold", "latitude"], 
           pch = pch_palette$pch[match(dat_sp@data$subsistence[dat_sp@data$climate %in% "cold"], pch_palette$subsistence)], 
           col = "#56B4E9", cex = 0.4, asp = 1)
    # text(dat_sp, labels = dat_sp@data$society_name, adj = c(1, 1), cex = 0.4, col = "black")
    # legend
    legend("topleft", inset = c(0, 0), cex = 0.4, pt.cex = 0.8, legend = biome_palette$biomes,
      pch = 15, col = biome_palette$color, bty = "n", ncol = 4, xpd = TRUE, 
      text.width = rep(c(0, 6.7e+06, 7.6e+06, 7.9e+06), each = 4))
     legend(x = -1.8e+07, y = 8e+06, cex = 0.45, pt.cex = 0.8, legend = c("Cold climate", "Warm climate"),
      pch = 16, col = c("#56B4E9", "#D55E00"), bty = "n", ncol = 1, xpd = TRUE)  
    legend(x = -1.8e+07, y = -7e+06, cex = 0.45, pt.cex = 0.7, legend = pch_palette$subsistence,
      pch = pch_palette$pch, col = "grey30", bty = "n", ncol = 1, xpd = TRUE) 
dev.off()


