/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2022 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <test/test.hh>
#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <cmath>
#include <complex>

using namespace test;
using namespace eos;
using namespace agv_2019a;

class LoopParameterTest :
    public TestCase
{
    public:
        LoopParameterTest() :
            TestCase("loop_parameter_test")
        {
        }

        virtual void run() const
        {
            {
                static const double eps = 1e-12;
                agv_2019a::CharmLoopsParameters testclp = {/*muhat =*/ 1.0, /*s =*/ -4.0, /*z =*/ 0.15, /*feynepsilonhat*/ 1e-10};

                /* Check, that the square root in C++ of complex arguments is handled correctly */

                static const complex<double> m1(-1.0, 0.0);
                TEST_CHECK_NEARLY_EQUAL(0.0, std::sqrt(m1).real(), eps);
                TEST_CHECK_NEARLY_EQUAL(1.0, std::sqrt(m1).imag(), eps);

                /* Comparison with Mathematica results */

                TEST_CHECK_NEARLY_EQUAL(1.5811388300841895,     testclp.xa.real(), eps);
                TEST_CHECK_NEARLY_EQUAL(0.7745966689726898,     testclp.xb.real(), eps);
                TEST_CHECK_NEARLY_EQUAL(1.5811388300841895,     testclp.xc.real(), eps);
                TEST_CHECK_NEARLY_EQUAL(0.7745966689726898,     testclp.xd.real(), eps);
                TEST_CHECK_NEARLY_EQUAL(1.5811388300841895,     testclp.xe.real(), eps);

                TEST_CHECK_NEARLY_EQUAL(-6.719840027857805e-10, testclp.xa.imag(), eps);
                TEST_CHECK_NEARLY_EQUAL(0.6324555318142068,     testclp.xb.imag(), eps);
                TEST_CHECK_NEARLY_EQUAL(-6.719840027857805e-10, testclp.xc.imag(), eps);
                TEST_CHECK_NEARLY_EQUAL(0.6324555318142068,     testclp.xd.imag(), eps);
                TEST_CHECK_NEARLY_EQUAL(-6.719840027857805e-10, testclp.xe.imag(), eps);

                /* Check, that the logarithm of complex numbers with negative real part is consistent with Mathematica */

                const double s_list[9] = {-1.0, -7.0 / 9.0, -5.0 / 9.0, -1.0 / 3.0, -1.0 / 9.0, 1.0 / 9.0, 1.0 / 3.0, 5.0 / 9.0, 7.0 / 9.0};
                const complex<double> lnms_ref[9] = {5.0000000000000005e-21 + 1.e-10 * 1.0i,-0.25131442828090605 + 1.e-10 * 1.0i,-0.587786664902119 + 1.e-10 * 1.0i,-1.0986122886681096 + 1.0000000000000002e-10 * 1.0i,-2.197224577336219 + 1.e-10 * 1.0i,-2.19722457733622 - 3.141592653489793 * 1.0i,-1.09861228866811 - 3.141592653489793 * 1.0i,-0.5877866649021192 - 3.141592653489793 * 1.0i,-0.2513144282809062 - 3.141592653489793 * 1.0i};

                for (unsigned int i = 0; i < 9; i++)
                {
                    const double s = s_list[i];
                    agv_2019a::CharmLoopsParameters testclp = {/*muhat =*/ 1.0, /*s =*/ s, /*z =*/ 0.15, /*feynepsilonhat*/ 1e-10};
                    const complex<double> lnms = std::log(-testclp.s_eps);

                    TEST_CHECK_NEARLY_EQUAL(lnms_ref[i].real(), lnms.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(lnms_ref[i].imag(), lnms.imag(), eps);
                }
            }
        }
} loop_parameter_test ;

class TwoLoopTest :
    public TestCase
{
    public:
        TwoLoopTest() :
            TestCase("two_loop_test")
        {
        }

        virtual void run() const
        {
            static double eps = 1e-12;
            const double z_s_list[45][2] = {{0.05, -1.0},     {0.05, -7.0 / 9.0},    {0.05, -5.0 / 9.0},    {0.05, -1.0 / 3.0},    {0.05, -1.0 / 9.0},  {0.05, 1.0 / 9.0},  {0.05, 1.0 / 3.0},  {0.05, 5.0 / 9.0},  {0.05, 7.0 / 9.0},
                                            {0.075, -1.0},    {0.075, -7.0 / 9.0},   {0.075, -5.0 / 9.0},   {0.075, -1.0 / 3.0},   {0.075, -1.0 / 9.0}, {0.075, 1.0 / 9.0}, {0.075, 1.0 / 3.0}, {0.075, 5.0 / 9.0}, {0.075, 7.0 / 9.0},
                                            {0.1, -1.0},      {0.1, -7.0 / 9.0},     {0.1, -5.0 / 9.0},     {0.1, -1.0 / 3.0},     {0.1, -1.0 / 9.0},   {0.1, 1.0 / 9.0},   {0.1, 1.0 / 3.0},   {0.1, 5.0 / 9.0},   {0.1, 7.0 / 9.0},
                                            {0.125, -1.0},    {0.125, -7.0 / 9.0},   {0.125, -5.0 / 9.0},   {0.125, -1.0 / 3.0},   {0.125, -1.0 / 9.0}, {0.125, 1.0 / 9.0}, {0.125, 1.0 / 3.0}, {0.125, 5.0 / 9.0}, {0.125, 7.0 / 9.0},
                                            {0.15, -1.0},     {0.15, -7.0 / 9.0},    {0.15, -5.0 / 9.0},    {0.15, -1.0 / 3.0},    {0.15, -1.0 / 9.0},  {0.15, 1.0 / 9.0},  {0.15, 1.0 / 3.0},  {0.15, 5.0 / 9.0},  {0.15, 7.0 / 9.0}};

            // Counterterms

            {
                // The computation took 1.090361s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f17ct_ref[45] = {0.007508380484939292 + 6.873191015900809e-13 * 1.0i, 0.005952604401909534 + 5.547258909565289e-13 * 1.0i, 0.004336911891155662 + 4.1182723464905446e-13 * 1.0i, 0.002656161338996113 + 2.572749847338921e-13 * 1.0i, 0.0009044920478106094 + 8.946448429403098e-14 * 1.0i, -0.0009248201293365621 - 9.353157026853862e-14 * 1.0i, -0.002839546035419166 - 2.940497627051733e-13 * 1.0i, -0.004848738820221368 - 5.14954318682398e-13 * 1.0i, -0.006963041259226838 - 7.598075046419174e-13 * 1.0i, 0.007508380484939292 + 6.873191015900809e-13 * 1.0i, 0.005952604401909534 + 5.547258909565289e-13 * 1.0i, 0.004336911891155662 + 4.1182723464905446e-13 * 1.0i, 0.002656161338996113 + 2.572749847338921e-13 * 1.0i, 0.0009044920478106094 + 8.946448429403098e-14 * 1.0i, -0.0009248201293365621 - 9.353157026853862e-14 * 1.0i, -0.002839546035419166 - 2.940497627051733e-13 * 1.0i, -0.004848738820221368 - 5.14954318682398e-13 * 1.0i, -0.006963041259226838 - 7.598075046419174e-13 * 1.0i, 0.007508380484939292 + 6.873191015900809e-13 * 1.0i, 0.005952604401909534 + 5.547258909565289e-13 * 1.0i, 0.004336911891155662 + 4.1182723464905446e-13 * 1.0i, 0.002656161338996113 + 2.572749847338921e-13 * 1.0i, 0.0009044920478106094 + 8.946448429403098e-14 * 1.0i, -0.0009248201293365621 - 9.353157026853862e-14 * 1.0i, -0.002839546035419166 - 2.940497627051733e-13 * 1.0i, -0.004848738820221368 - 5.14954318682398e-13 * 1.0i, -0.006963041259226838 - 7.598075046419174e-13 * 1.0i, 0.007508380484939292 + 6.873191015900809e-13 * 1.0i, 0.005952604401909534 + 5.547258909565289e-13 * 1.0i, 0.004336911891155662 + 4.1182723464905446e-13 * 1.0i, 0.002656161338996113 + 2.572749847338921e-13 * 1.0i, 0.0009044920478106094 + 8.946448429403098e-14 * 1.0i, -0.0009248201293365621 - 9.353157026853862e-14 * 1.0i, -0.002839546035419166 - 2.940497627051733e-13 * 1.0i, -0.004848738820221368 - 5.14954318682398e-13 * 1.0i, -0.006963041259226838 - 7.598075046419174e-13 * 1.0i, 0.007508380484939292 + 6.873191015900809e-13 * 1.0i, 0.005952604401909534 + 5.547258909565289e-13 * 1.0i, 0.004336911891155662 + 4.1182723464905446e-13 * 1.0i, 0.002656161338996113 + 2.572749847338921e-13 * 1.0i, 0.0009044920478106094 + 8.946448429403098e-14 * 1.0i, -0.0009248201293365621 - 9.353157026853862e-14 * 1.0i, -0.002839546035419166 - 2.940497627051733e-13 * 1.0i, -0.004848738820221368 - 5.14954318682398e-13 * 1.0i, -0.006963041259226838 - 7.598075046419174e-13 * 1.0i};

                // The computation took 5.23546s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f19ct_ref[45] = {-2.7747020877105797+5.626346304045955e-9*1.0i,-3.678411078930253+6.551498509238362e-9*1.0i,-5.189029077982212+7.622673015584751e-9*1.0i,-8.082810669902756+8.358442082775868e-9*1.0i,-15.295431631415227+4.10848830837457e-9*1.0i,-54.58909756932087-0.2962036019425178*1.0i,70.55799350353001-17.653390450324757*1.0i,30.565735956813924+5.7665720781398235*1.0i,18.123518226961437+8.16568053415471*1.0i,-3.893962954910876+3.6667442484171744e-9*1.0i,-4.9435312452224025+4.047484854303513e-9*1.0i,-6.590976326253326+4.3029628743453725e-9*1.0i,-9.469670526543581+3.875115198659835e-9*1.0i,-15.549077574087107-2.327794826955606e-10*1.0i,-34.693573875853275-0.29620348722185535*1.0i,167.92305609469395-169.05788651872777*1.0i,52.46869793945905-5.200967465309851*1.0i,29.684631469279182+5.682142901544971*1.0i,-4.6121577307674215+2.4686952677874505e-9*1.0i,-5.708816327234217+2.5808340482679426e-9*1.0i,-7.354933349386755+2.490392924813179e-9*1.0i,-10.05051056855254+1.734163658380889e-9*1.0i,-15.153979279758971-1.5672712814544838e-9*1.0i,-27.418279308218146-0.2962034663408389*1.0i,-106.01160307383219-0.18257783824640594*1.0i,84.59645934900874-33.95940219738226*1.0i,43.98892291368844-0.5735255289875738*1.0i,-5.072667438223817+1.6824488080230685e-9*1.0i,-6.166265303521357+1.655473734962947e-9*1.0i,-7.751673931711565+1.4166070071917854e-9*1.0i,-10.22260815512277+6.081296126569895e-10*1.0i,-14.548968500514661-1.9781302920086277e-9*1.0i,-23.33010199275789-0.2962034587959605*1.0i,-53.38314519171202-0.18257745555047408*1.0i,153.5213258190528-140.2100221981614*1.0i,62.635938975007846-13.392627642880836*1.0i,-5.357791346984698+1.1439296343338859e-9*1.0i,-6.42217138552391+1.0446350252241993e-9*1.0i,-7.922209335544608+7.481806136477017e-10*1.0i,-10.170139590163245-1.6911948663200337e-11*1.0i,-13.881352337142118-2.052076503827342e-9*1.0i,-20.571120817214762-0.29620345513394997*1.0i,-37.880978841112146-0.18257740232203398*1.0i,-180.21214469438945-0.1297454551302265*1.0i,89.84030337463037-40.709145314441294*1.0i};

                // The computation took 1.08506s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f27ct_ref[45] = {-0.04505028290963531 - 4.123914609540503e-12 * 1.0i, -0.03571562641145776 - 3.3283553457391752e-12 * 1.0i, -0.026021471346933556 - 2.470963407894314e-12 * 1.0i, -0.015936968033976706 - 1.543649908403359e-12 * 1.0i, -0.005426952286864184 - 5.36786905764194e-13 * 1.0i, 0.005548920776019206 + 5.611360414563721e-13 * 1.0i, 0.017037276212515384 + 1.7643509111750194e-12 * 1.0i, 0.029092432921328126 + 3.089723044977223e-12 * 1.0i, 0.0417782475553613 + 4.558784583472264e-12 * 1.0i, -0.04505028290963531 - 4.123914609540503e-12 * 1.0i, -0.03571562641145776 - 3.3283553457391752e-12 * 1.0i, -0.026021471346933556 - 2.470963407894314e-12 * 1.0i, -0.015936968033976706 - 1.543649908403359e-12 * 1.0i, -0.005426952286864184 - 5.36786905764194e-13 * 1.0i, 0.005548920776019206 + 5.611360414563721e-13 * 1.0i, 0.017037276212515384 + 1.7643509111750194e-12 * 1.0i, 0.029092432921328126 + 3.089723044977223e-12 * 1.0i, 0.0417782475553613 + 4.558784583472264e-12 * 1.0i, -0.04505028290963531 - 4.123914609540503e-12 * 1.0i, -0.03571562641145776 - 3.3283553457391752e-12 * 1.0i, -0.026021471346933556 - 2.470963407894314e-12 * 1.0i, -0.015936968033976706 - 1.543649908403359e-12 * 1.0i, -0.005426952286864184 - 5.36786905764194e-13 * 1.0i, 0.005548920776019206 + 5.611360414563721e-13 * 1.0i, 0.017037276212515384 + 1.7643509111750194e-12 * 1.0i, 0.029092432921328126 + 3.089723044977223e-12 * 1.0i, 0.0417782475553613 + 4.558784583472264e-12 * 1.0i, -0.04505028290963531 - 4.123914609540503e-12 * 1.0i, -0.03571562641145776 - 3.3283553457391752e-12 * 1.0i, -0.026021471346933556 - 2.470963407894314e-12 * 1.0i, -0.015936968033976706 - 1.543649908403359e-12 * 1.0i, -0.005426952286864184 - 5.36786905764194e-13 * 1.0i, 0.005548920776019206 + 5.611360414563721e-13 * 1.0i, 0.017037276212515384 + 1.7643509111750194e-12 * 1.0i, 0.029092432921328126 + 3.089723044977223e-12 * 1.0i, 0.0417782475553613 + 4.558784583472264e-12 * 1.0i, -0.04505028290963531 - 4.123914609540503e-12 * 1.0i, -0.03571562641145776 - 3.3283553457391752e-12 * 1.0i, -0.026021471346933556 - 2.470963407894314e-12 * 1.0i, -0.015936968033976706 - 1.543649908403359e-12 * 1.0i, -0.005426952286864184 - 5.36786905764194e-13 * 1.0i, 0.005548920776019206 + 5.611360414563721e-13 * 1.0i, 0.017037276212515384 + 1.7643509111750194e-12 * 1.0i, 0.029092432921328126 + 3.089723044977223e-12 * 1.0i, 0.0417782475553613 + 4.558784583472264e-12 * 1.0i};

                // The computation took 5.27316s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f29ct_ref[45] = {-1.608986716471776+3.934295499979576e-9*1.0i,-2.3693283777971317+4.271854880838442e-9*1.0i,-3.7800190331181796+4.402372917703126e-9*1.0i,-6.599376801954662+3.5067631829492385e-9*1.0i,-13.404671606463694-3.753818639220557e-9*1.0i,-50.98081440086062+1.7772205299757942*1.0i,52.76153897156211-38.65385854966077*1.0i,34.66130927641784-12.829931777947536*1.0i,29.017679457734744-5.407202046720414*1.0i,-2.401870228870557+2.652835236941801e-9*1.0i,-3.1817692982717034+2.698718235901337e-9*1.0i,-4.538488326070166+2.4507032774885e-9*1.0i,-7.036633206233375+1.244260771205401e-9*1.0i,-12.213990799607984-3.984122385193444e-9*1.0i,-30.90700561930492+1.7772206377857764*1.0i,107.16140038893903-148.86783796939537*1.0i,46.114443570219464-23.70169235520142*1.0i,35.328110799930535-9.590736456099371*1.0i,-2.9414233745675844+1.8750311194276108e-9*1.0i,-3.6918220890152753+1.7933824848964267e-9*1.0i,-4.938908656081523+1.4262694196122121e-9*1.0i,-7.0986719328511185+2.821701102354999e-10*1.0i,-11.102215787231103-3.4649922257079255e-9*1.0i,-23.22044639006853+1.7772206593931228*1.0i,-89.78355190008688+1.0954637858202039*1.0i,62.98816795440386-46.82941116582578*1.0i,42.92255440219149-16.42890162479344*1.0i,-3.3211858277453743+1.3656343000405952e-9*1.0i,-4.021094256914381+1.2293665602458048e-9*1.0i,-5.1442919083031455+8.41362748745472e-10*1.0i,-6.997705859671058-1.5657051029408877e-10*1.0i,-10.142144919941671-2.905822344058433e-9*1.0i,-18.936989330517427+1.7772206675965885*1.0i,-45.21119817675006+1.0954640964488*1.0i,102.8561882200495-124.64515483844737*1.0i,52.751369875880215-27.83701399063691*1.0i,-3.5939711464536828+1.0157641805518302e-9*1.0i,-4.235378448263781+8.599867522896628e-10*1.0i,-5.23645786793729+4.89817231583082e-10*1.0i,-6.824324716574154-3.572393189764593e-10*1.0i,-9.323332766521304-2.4259036057229785e-9*1.0i,-16.137135778031848+1.777220671669935*1.0i,-31.4613064877505+1.0954641421398201*1.0i,-146.15081645857504+0.7784645027656028*1.0i,67.4668671122318-49.36283092562938*1.0i};

                for (unsigned int i = 0; i < 45; i++)
                {
                    const double z = z_s_list[i][0];
                    const complex<double> s = z_s_list[i][1];
                    agv_2019a::CharmLoopsParameters testclp = {/*muhat =*/ 1.0, /*s =*/ s, /*z =*/ z, /*feynepsilonhat*/ 1e-10};

                    const complex<double> f17ct = agv_2019a::f17ctQs(testclp) + agv_2019a::f17ctQc(testclp) + agv_2019a::f17ctQb(testclp);
                    const complex<double> f19ct = agv_2019a::f19ctQs(testclp) + agv_2019a::f19ctQc(testclp) + agv_2019a::f19ctQb(testclp);
                    const complex<double> f27ct = agv_2019a::f27ctQs(testclp) + agv_2019a::f27ctQc(testclp) + agv_2019a::f27ctQb(testclp);
                    const complex<double> f29ct = agv_2019a::f29ctQs(testclp) + agv_2019a::f29ctQc(testclp) + agv_2019a::f29ctQb(testclp);

                    TEST_CHECK_NEARLY_EQUAL(f17ct_ref[i].real(), f17ct.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(f17ct_ref[i].imag(), f17ct.imag(), eps);

                    TEST_CHECK_NEARLY_EQUAL(f19ct_ref[i].real(), f19ct.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(f19ct_ref[i].imag(), f19ct.imag(), eps);

                    TEST_CHECK_NEARLY_EQUAL(f27ct_ref[i].real(), f27ct.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(f27ct_ref[i].imag(), f27ct.imag(), eps);

                    TEST_CHECK_NEARLY_EQUAL(f29ct_ref[i].real(), f29ct.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(f29ct_ref[i].imag(), f29ct.imag(), eps);
                }
            }

            // Two loop functions

            {
                static double eps = 1e-12;

                // The computation took 19.5591s AbsoluteTiming with the Methematica-Ginac script from [AGV:2019A]
                const complex<double> f27a_ref[45] = {0.22820842548329867+0.03284371205569114*1.0i,0.24322769157762436+0.038807446284946145*1.0i,0.26125394694869497+0.047508301945062836*1.0i,0.28369187897119896+0.061560652401266236*1.0i,0.31282262823987966+0.08948335331359147*1.0i,0.30793554542001145+0.15190390991685027*1.0i,0.2835394815474132+0.17453482839483492*1.0i,0.2659117064548151+0.18807482777462914*1.0i,0.25167739857336713+0.19779897979563899*1.0i,0.1942262099395171+0.015747105165950348*1.0i,0.20751249514080594+0.01884930884765368*1.0i,0.22372458623749508+0.02351831767405843*1.0i,0.24463371479392249+0.03143542261205623*1.0i,0.274534554628007+0.0487104444737394*1.0i,0.28542163932296316+0.10497461448205425*1.0i,0.2675801944480435+0.13175126609782584*1.0i,0.2534960136717421+0.1482279346961768*1.0i,0.241627863938783+0.16025051932819934*1.0i,0.16645325509067144+0.0070655760260136535*1.0i,0.17797119045536447+0.008554220180702288*1.0i,0.19206401194042821+0.01085624811041852*1.0i,0.21046491233817016+0.014932462684399517*1.0i,0.23802694089995924+0.024626783050882728*1.0i,0.25871995614499+0.07169968659932635*1.0i,0.24716924227864856+0.0998914003429483*1.0i,0.23666032853125607+0.11777793089896793*1.0i,0.22725464768356218+0.13105242565722863*1.0i,0.14385120056625744+0.0028418587109217963*1.0i,0.15383165223752537+0.0034769130379967983*1.0i,0.16597783790901066+0.004483920247051687*1.0i,0.18180478887241847+0.006341870696572438*1.0i,0.205882958222052+0.011154491652198506*1.0i,0.23162741198184045+0.04861818766051762*1.0i,0.22551840625100988+0.07638961331501815*1.0i,0.21826240395351126+0.09461115090738242*1.0i,0.21116191302573561+0.10838289879031443*1.0i,0.1253602314267599+0.0009632892835105267*1.0i,0.13409970793541337+0.0011904116273360188*1.0i,0.14463317059892322+0.0015593946514974588*1.0i,0.1582094609654583+0.002268564706348397*1.0i,0.17872815784755386+0.004277619489111539*1.0i,0.20604953478302296+0.033090915231659634*1.0i,0.20431183133982378+0.05927954183803186*1.0i,0.19983162152652945+0.07710369130760691*1.0i,0.1947612479177847+0.09084011783500079*1.0i};

                // The computation took 22.2867s AbsoluteTiming with the Methematica-Ginac script from [AGV:2019A]
                const complex<double> f29a_ref[45] = {-1.327593865389298-0.26354311133041985*1.0i,-1.4752272048905066-0.31892150923336793*1.0i,-1.701417762302468-0.4053802100753181*1.0i,-2.1007213493562373-0.5630064622825302*1.0i,-3.150248273544651-0.9902966237413686*1.0i,-0.7597102025321296-4.134375257724479*1.0i,0.16753628934200737-2.9100769956755577*1.0i,0.4974848467549766-2.3381370294447197*1.0i,0.6773412441390434-1.9597798499797952*1.0i,-1.174311190527999-0.15514452828030834*1.0i,-1.3103448154388344-0.1897059207632319*1.0i,-1.5228195635238784-0.24487728930679786*1.0i,-1.9067438800032237-0.34893822649070927*1.0i,-2.9484978085140274-0.6491865159295083*1.0i,-0.8585268855028432-3.8355451575958712*1.0i,0.09347127701492153-2.6969010669564253*1.0i,0.4332264060021709-2.1601946401686285*1.0i,0.6189732755843638-1.8032387227327886*1.0i,-1.031851997909416-0.08601215472262158*1.0i,-1.1541823598664762-0.10610702883207601*1.0i,-1.3485806069093482-0.13881082239841191*1.0i,-1.707045931604111-0.20240218784562547*1.0i,-2.708647789225609-0.39708733657481154*1.0i,-0.8966424700433189-3.5458869195213674*1.0i,0.05895489858151093-2.4918355485129613*1.0i,0.4005894972674273-1.9895284697137834*1.0i,0.5876647077034924-1.6533316758674381*1.0i,-0.9033591077645449-0.04355378332207671*1.0i,-1.0118657632051313-0.05415494340115856*1.0i,-1.1870941854763062-0.07171553361522243*1.0i,-1.5160352669914974-0.1068565395336103*1.0i,-2.459127048591247-0.22100616808667084*1.0i,-0.8860758954385597-3.274680785067058*1.0i,0.055600033360480694-2.2993133350758126*1.0i,0.392636146665593-1.8290654279801188*1.0i,0.5773484745524833-1.5122277422367745*1.0i,-0.7895963007241877-0.019228145658293005*1.0i,-0.8852409765960578-0.02408278061223923*1.0i,-1.042123061348065-0.032260072893188774*1.0i,-1.3413611068846678-0.04909068922306581*1.0i,-2.2182923911288244-0.10724895973649762*1.0i,-0.8371655238898433-3.0282708948147423*1.0i,0.07680072001682103-2.122363664368087*1.0i,0.40411706671191006-1.6807994286205883*1.0i,0.5835471540035471-1.3813992013466887*1.0i};

                // The computation took 83.3218s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f27b_ref[45] = {0.22116104744579745-5.5334126169981346e-11*1.0i,0.20735428622243157-5.7376103868023165e-11*1.0i,0.19305033438721408-5.954226001136931e-11*1.0i,0.1782084657024352-6.184797118891083e-11*1.0i,0.16278245487666587-6.430883603414372e-11*1.0i,0.1467195039633833-6.694622634029201e-11*1.0i,0.1299588861684262-6.978229105669698e-11*1.0i,0.11243021094760941-7.285905212484067e-11*1.0i,0.09405117317643885-7.645439836778678e-11*1.0i,0.23600156921419213-4.4674319799042905e-11*1.0i,0.22239592222915672-4.651884433215514e-11*1.0i,0.20830218848609006-4.847799939255992e-11*1.0i,0.19368042539307972-5.056499663425029e-11*1.0i,0.17848529636420096-5.2794324467697606e-11*1.0i,0.16266501708188308-5.518474566201803e-11*1.0i,0.14616002416645912-5.775502298632773e-11*1.0i,0.12890127174735078-6.055067558463634e-11*1.0i,0.11080802098258857-6.400568963726982e-11*1.0i,0.24882632335598132-3.7410519126979125e-11*1.0i,0.23538728178064572-3.912170587483388e-11*1.0i,0.22146752951941262-4.0941139367589585e-11*1.0i,0.20702774985262873-4.2879033657072796e-11*1.0i,0.19202331511608017-4.495293026707259e-11*1.0i,0.17640324854303957-4.717759516381648e-11*1.0i,0.1601089124094912-4.9573900540167415e-11*1.0i,0.14307232922789126-5.2166271302667155e-11*1.0i,0.12521400279035788-5.5422333389287814e-11*1.0i,0.2602318289764167-3.200084641719059e-11*1.0i,0.24693559950846455-3.3614222516575865e-11*1.0i,0.23316485807265486-3.533040526804143e-11*1.0i,0.21888080713768954-3.7159830768018765e-11*1.0i,0.20403940647623764-3.9118042138852616e-11*1.0i,0.18859034771783278-4.122013841367789e-11*1.0i,0.17247575836516482-4.348699178535753e-11*1.0i,0.1556285430290103-4.5986325858393684e-11*1.0i,0.13797023009533405-4.843059286940843e-11*1.0i,0.2705632944288101-2.7750690634320563e-11*1.0i,0.2573922497380883-2.928768338961163e-11*1.0i,0.24375203037357562-3.092315292718695e-11*1.0i,0.22960428059676508-3.266831249959523e-11*1.0i,0.2149054595512967-3.4537039894644295e-11*1.0i,0.19960582646857494-3.65427688109321e-11*1.0i,0.18364815797124834-3.871214460104966e-11*1.0i,0.1669661060679175-4.1026737562788185e-11*1.0i,0.14948206622557336-4.359890226623975e-11*1.0i};

                // The computation took 108.757s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f29b_ref[45] = {-0.5199119415599585-1.4654322200158276e-10*1.0i,-0.5079622656422487-1.4769074851983532e-10*1.0i,-0.49599108866826214-1.489528500542292e-10*1.0i,-0.4840162125816292-1.503384083889614e-10*1.0i,-0.4720598373043323-1.5189627333711542e-10*1.0i,-0.460149740874094-1.5361578675765486e-10*1.0i,-0.44832084337332745-1.5553514032262683e-10*1.0i,-0.4366173073315984-1.5758061522319622e-10*1.0i,-0.4250954011656063-1.5927525964798406e-10*1.0i,-0.48529093847569627-1.256079684708311e-10*1.0i,-0.472775719200496-1.263318338828867e-10*1.0i,-0.46020878323730496-1.2712941810377743e-10*1.0i,-0.4476049181150774-1.2803980098397005e-10*1.0i,-0.43498284742866034-1.290487716687494e-10*1.0i,-0.4223663122083252-1.3022827261011116e-10*1.0i,-0.40978550820379667-1.3154366484968705e-10*1.0i,-0.39727902173666507-1.3297096757014515e-10*1.0i,-0.3848964761693878-1.3426415534922853e-10*1.0i,-0.45456645876278046-1.1091660923057134e-10*1.0i,-0.4415583724370755-1.113260594820531e-10*1.0i,-0.42847239058854125-1.1179590586607446e-10*1.0i,-0.4153206959735485-1.1234924102154764e-10*1.0i,-0.40211900886071206-1.1298340041321353e-10*1.0i,-0.388887583260777-1.1377032649306784e-10*1.0i,-0.37565253505635354-1.1464074134437396e-10*1.0i,-0.3624476360516695-1.155981976808107e-10*1.0i,-0.34931677349220536-1.1789325071731582e-10*1.0i,-0.4266432358479504-9.969003400556176e-11*1.0i,-0.4131944254001585-9.985168247794718e-11*1.0i,-0.3996444444458893-1.0007283890445251e-10*1.0i,-0.3860031666055681-1.0035439146349745e-10*1.0i,-0.3722836503243183-1.00714103723476e-10*1.0i,-0.35850305985416675-1.0116885107436246e-10*1.0i,-0.3446838969318833-1.0172129805141594e-10*1.0i,-0.3308556695551772-1.0232525937681203e-10*1.0i,-0.3170571863802252-1.0311396181350574e-10*1.0i,-0.40087824318121124-9.065548312037208e-11*1.0i,-0.3870286799616509-9.062439687568258e-11*1.0i,-0.37305689677621334-9.063505501671898e-11*1.0i,-0.35897068488351636-9.069545114925859e-11*1.0i,-0.3447807028992944-9.0812690700659e-11*1.0i,-0.33050133013493443-9.102318898612793e-11*1.0i,-0.3161518115891475-9.131895239988808e-11*1.0i,-0.30175781416141945-9.164580205833772e-11*1.0i,-0.2873535727203489-9.276845958083868e-11*1.0i};

                // The computation took 38.0936s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f27c_ref[45] = {1.6791188931480354+0.176035647094692*1.0i,1.8445207172828009+0.22491670644554373*1.0i,2.039637263807971+0.29730968194630775*1.0i,2.272896086709195+0.41084782274089404*1.0i,2.549536785805979+0.601808734589645*1.0i,2.827806363319306+0.9245760968901244*1.0i,3.0754663141059595+1.7691695215566483*1.0i,2.6058034231749705+2.181421117441232*1.0i,2.279797531379785+2.350344220275119*1.0i,1.484743676467711+0.11361500496449817*1.0i,1.6269481151672336+0.14452145433548835*1.0i,1.793156719131109+0.18988086952137584*1.0i,1.9904759809253152+0.2601102467789501*1.0i,2.2257045229267924+0.37613871047342284*1.0i,2.4857658838282046+0.5765911372455763*1.0i,3.1416997323708813+0.7601944755598984*1.0i,2.8440086840698187+1.725913168485203*1.0i,2.4961456676789524+2.046984350070474*1.0i,1.3168845149308028+0.07081490948937363*1.0i,1.4401326832344288+0.08983704116886254*1.0i,1.582817619753906+0.11762279270003301*1.0i,1.7508368251793547+0.16039324744905237*1.0i,1.9511183497668894+0.23067496202722326*1.0i,2.1833585326977936+0.35426411305347916*1.0i,2.3452983398140606+0.5171740670821414*1.0i,2.977072733932623+1.1206774043534402*1.0i,2.6724361288312366+1.6750448942107994*1.0i,1.16957799802036+0.04166580412826543*1.0i,1.2770492135049047+0.052777116063518335*1.0i,1.4002120709097787+0.06897286248920559*1.0i,1.5438511429916377+0.09385314550921109*1.0i,1.714290631468566+0.13474026970042674*1.0i,1.9165961456360656+0.20774456244040174*1.0i,2.120944344168538+0.34227779599615715*1.0i,2.8233710535070475+0.26928476879405494*1.0i,2.7856818014716183+1.2184027574251957*1.0i,1.0390818597513611+0.02237301526240776*1.0i,1.1333078877013394+0.028317910732284512*1.0i,1.2401166856635362+0.036977789310918*1.0i,1.3632657540420463+0.05028109592823782*1.0i,1.5080618886040846+0.07219171884337117*1.0i,1.6810035995398285+0.11178669807703567*1.0i,1.8794068499956147+0.19171297494068718*1.0i,1.8932930155997258+0.26923528871960656*1.0i,2.7836870826023272+0.6523003925748014*1.0i};

                // The computation took 42.7845s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f29c_ref[45] = {4.513856529852124+0.0207355852668627*1.0i,5.361223602411098+0.046867419372780716*1.0i,6.624108607271655+0.10225601541733553*1.0i,8.65620349138604+0.23447062307073452*1.0i,12.433907173691708+0.6243478653382832*1.0i,22.821909202485084+2.567179581454365*1.0i,10.692084535966496+33.60298442326314*1.0i,-2.8798399539731907+26.463237013362225*1.0i,-7.824065464593247+21.343149376344343*1.0i,4.0358639153977585-0.01672342427738016*1.0i,4.704428485527288-0.011223111902779692*1.0i,5.677680935841327+0.0045368523830759955*1.0i,7.181239815332177+0.048603859585540476*1.0i,9.762725143675464+0.18599482897810304*1.0i,15.33406642968113+0.7673801354773317*1.0i,29.98647118915443+28.318563553995926*1.0i,3.1848420350278523+28.184157826656254*1.0i,-4.593443179030643+23.166621144288627*1.0i,3.6495067319840047-0.029840490762190086*1.0i,4.181776694049864-0.03333773896234149*1.0i,4.942720134717311-0.0352004613257364*1.0i,6.081961266976778-0.03016274884331871*1.0i,7.929399611129133+0.0019154474534452959*1.0i,11.429867994081569+0.16184551665261893*1.0i,21.880073581287125+1.51626737880229*1.0i,11.389018404101098+28.215748987292642*1.0i,-0.5552453106487576+24.507107317068886*1.0i,3.337499192173418-0.02912944629051495*1.0i,3.7645367447525793-0.03510936110315299*1.0i,4.366798023150222-0.04261492246199339*1.0i,5.2465829209420045-0.05131691634560376*1.0i,6.612458463566568-0.057668184411678425*1.0i,8.985004117418619-0.03886656472050731*1.0i,14.315559518816258+0.2047904690843776*1.0i,23.63360332122115+24.05165030208523*1.0i,4.454180994470647+25.124699774631836*1.0i,3.0857372777635668-0.021835153585378464*1.0i,3.430723648432444-0.02704941225633206*1.0i,3.9127406310524773-0.0342688089151153*1.0i,4.603748413145579-0.0445117754490898*1.0i,5.640933373917424-0.059131123402437424*1.0i,7.331718576160419-0.07784223163139359*1.0i,10.596647403375428-0.0739379390605599*1.0i,21.682218300103465+0.7364793070766922*1.0i,10.808478393410638+24.52304908051292*1.0i};

                // The computation took 503.891s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f27d_ref[45] = {1.2213628974054718+6.716973643960955e-10*1.0i,1.3153370476170805+7.399241219729902e-10*1.0i,1.4207719714813623+8.27648172219142e-10*1.0i,1.5412020734441825+9.4655683291478e-10*1.0i,1.6812863317012376+1.1198123539557514e-9*1.0i,1.832844774916787+1.3294361167481838e-9*1.0i,2.3572328317154003-0.12108026221133983*1.0i,2.636420899983733+0.38495505251790263*1.0i,2.6668203992044255+0.7701469182572964*1.0i,1.0471034426878543+5.674845038328158e-10*1.0i,1.1268902863537464+6.177836020526684e-10*1.0i,1.214399092923526+6.798328566759437e-10*1.0i,1.3111813889654584+7.589130435192715e-10*1.0i,1.418392261847579+8.629017500538794e-10*1.0i,1.5301077246801165+9.850613658102247e-10*1.0i,1.5662886044720707-0.3264140678416929*1.0i,2.268364891785666-0.043016267790648044*1.0i,2.455135031896134+0.3643482531427935*1.0i,0.8938431264618147+4.909077588877153e-10*1.0i,0.9629194280222135+5.297002836357478e-10*1.0i,1.0373804480005493+5.760685262146126e-10*1.0i,1.1178443194017826+6.325689971831139e-10*1.0i,1.204210051026454+7.020806158664072e-10*1.0i,1.2919029790760503+7.795488699002817e-10*1.0i,1.3257759359845238+5.805755876053809e-10*1.0i,1.771457034804655-0.35215964420208934*1.0i,2.1639134801243927+0.006146051877067293*1.0i,0.7563840852514563+4.312020740471212e-10*1.0i,0.8170575014298684+4.620523963438927e-10*1.0i,0.8815691167864177+4.980130197118626e-10*1.0i,0.9500701968246825+5.403630876088528e-10*1.0i,1.0219938095556986+5.900790966961722e-10*1.0i,1.0937791973413495+6.434106580854859e-10*1.0i,1.1437532165780624+6.368541249912596e-10*1.0i,1.1268520933991795-0.43306473569107595*1.0i,1.7938435502501306-0.287901158430941*1.0i,0.6314081602346882+3.8286385173336157e-10*1.0i,0.685305952604859+4.0797187850216687e-10*1.0i,0.7419765011407211+4.3663272997207514e-10*1.0i,0.8013313771542134+4.694964417240044e-10*1.0i,0.8626492718346379+5.067679609283005e-10*1.0i,0.9231207094047371+5.456604057485492e-10*1.0i,0.9709209565257595+5.609575026710445e-10*1.0i,0.9001531324828989-9.71880353972665e-11*1.0i,1.3400969618951422-0.4874382314092145*1.0i};

                 // The computation took 481.037s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> f29d_ref[45] = {1.5190076758183597+8.08713096489555e-10*1.0i,1.963280719685434+1.227643764423192e-9*1.0i,2.657221736658631+1.9516317451007126e-9*1.0i,3.812491043816973+3.3839682167524643e-9*1.0i,6.020981537256716+7.069178131757781e-9*1.0i,12.413457656382846+2.846968172320885e-8*1.0i,7.959392847454609+22.383912062586504*1.0i,-2.7294754253877613+18.744592355495456*1.0i,-6.971134142648225+15.37638905381391*1.0i,1.2942412409528465+6.06011241188753e-10*1.0i,1.6364746945808513+9.091722930065771e-10*1.0i,2.163845396029396+1.4110561608049466e-9*1.0i,3.01136269189972+2.33333086185894e-9*1.0i,4.5053668011779315+4.370386608165688e-9*1.0i,7.8034238557341595+1.1298265434334098e-8*1.0i,20.588005710464387+16.515872882194824*1.0i,1.6730766827030834+19.399285962978297*1.0i,-4.519606403589023+16.430543443753777*1.0i,1.1195880760323114+4.587299429203995e-10*1.0i,1.3844245401297286+6.849960598742655e-10*1.0i,1.7908698679095778+1.0479563883336596e-9*1.0i,2.430000880045803+1.6790613344142002e-9*1.0i,3.499952492241121+2.9372841936492478e-9*1.0i,5.5631893528027945+6.242999006644823e-9*1.0i,11.88131131476203+2.8868271328974515e-8*1.0i,7.329977737338481+18.744043733235912*1.0i,-1.5812619941721096+17.109569343689145*1.0i,0.9830955710404261+3.482423238665433e-10*1.0i,1.1875243203338908+5.20834930739511e-10*1.0i,1.5028448848838831+7.907994259426232e-10*1.0i,1.9931084690107461+1.2409975269633833e-9*1.0i,2.7868772672227067+2.07511785532688e-9*1.0i,4.198852665613984+3.949331528474431e-9*1.0i,7.392460902291987+1.0718327558834062e-8*1.0i,15.31300731249246+14.928903630674114*1.0i,1.9476652309408564+17.25393106734873*1.0i,0.876555361208279+2.6356250515391366e-10*1.0i,1.0328205882663974+3.971827311488596e-10*1.0i,1.2775565782516445+6.021365628328113e-10*1.0i,1.6572977244847653+9.331881933860586e-10*1.0i,2.259666754885231+1.5138201803210904e-9*1.0i,3.277299453333164+2.68812883064129e-9*1.0i,5.271425291834532+5.881233278159925e-9*1.0i,12.012800453932869+3.455512853633991e-8*1.0i,6.298563906296295+16.51665885914099*1.0i};

                // The computation took 5.55759s AbsoluteTiming with the Mathematica-Ginac script from [AGV:2019A]
                // feynepsilonhat = 1e-10
                const complex<double> f29e_ref[45] = {2.3021828027409255 - 6.463426460712185e-10 * 1.0i, 2.385948579526878 - 1.044983460873894e-9 * 1.0i, 2.700100154127618 - 1.5445331680297159e-9 * 1.0i, 3.6163362509571346 - 1.932085602618372e-9 * 1.0i, 6.666877489981057 + 2.8917135352912737e-10 * 1.0i, 28.623209566833367 + 9.837709313842424e-8 * 1.0i, -48.40967194198186 - 5.577487933474039 * 1.0i, -16.874156506724486 - 14.088669006050232 * 1.0i, -8.120489514625849 - 13.065461479444517 * 1.0i, 2.3891251350058544 - 1.0545575523934758e-10 * 1.0i, 2.53201813513229 - 2.763216322421158e-10 * 1.0i, 2.905754685469725 - 4.0957903735261425e-10 * 1.0i, 3.8132032921355905 - 2.3579538321882865e-10 * 1.0i, 6.262937168090792 + 1.854425946135052e-9 * 1.0i, 16.09708865543818 + 2.5199682340826257e-8 * 1.0i, -139.6018251283025 + 89.57165539021763 * 1.0i, -34.10106085600972 - 12.77883576501469 * 1.0i, -16.18476485749127 - 15.329293002930044 * 1.0i, 2.360596786243271 + 2.0758150753863447e-10 * 1.0i, 2.5208807225145855 + 1.4876189169399368e-10 * 1.0i, 2.8920338899305498 + 1.6927081958328927e-10 * 1.0i, 3.710589948855498 + 4.959286314942801e-10 * 1.0i, 5.655080738092669 + 2.130152054746759e-9 * 1.0i, 11.553725232419755 + 1.2162026052919828e-8 * 1.0i, 58.30623682360957 + 3.0637617953743757e-7 * 1.0i,-62.76950940050142 + 0.031852335768391526 * 1.0i, -27.510523556251457 - 15.546163093825761 * 1.0i, 2.2611611038878188 + 3.9498049275721314e-10 * 1.0i, 2.417885207731062 + 3.929780945099992e-10 * 1.0i, 2.7603235654499256 + 4.773212936015625e-10 * 1.0i, 3.4718374685672275 + 8.230784942497849e-10 * 1.0i, 5.022765191422876 + 2.0671784284331807e-9 * 1.0i, 8.99999956749037 + 7.480778663193632e-9 * 1.0i, 25.235965138100266 + 5.594193908109446e-8 * 1.0i, -129.21960318321723 + 67.6483365255224 * 1.0i, -43.80351847021929 - 11.716589533222155 * 1.0i, 2.1167522450187013 + 5.073221842621933e-10 * 1.0i, 2.2593251623889987 + 5.336877606509915e-10 * 1.0i, 2.5628117869741036 + 6.408615860209466e-10 * 1.0i, 3.1698959791101298 + 9.60874046995741e-10 * 1.0i, 4.4158702116722175 + 1.9111201510213505e-9 * 1.0i, 7.274234695054613 + 5.2121578164587845e-9 * 1.0i, 15.966659039855655 + 2.29688197350697e-8 * 1.0i, 104.86751778726 + 8.100933825971879e-7 * 1.0i, -69.40585654227208 + 1.840951225135683 * 1.0i};

                // The computation took 529.971s Absolute Timing  with the Mathematica-Ginac script from [AGV:2019A]
                const complex<double> weight4_ref[45] = {-19.943420811418328+69.12220075921215*1.0i,-19.416443825764926+61.44724076199987*1.0i,-17.984247025545407+51.31949556440109*1.0i,-13.777863810921685+2.733071662104102*1.0i,8.674632363281113+19.542554412182277*1.0i,30.07316975834337+19.542554456149478*1.0i,63.6707962441056+38.33290778438321*1.0i,55.85473719298716+58.28664153474111*1.0i,50.59673164867681+64.65080087804398*1.0i,-14.111520882997851+52.742811453410425*1.0i,-13.64355405351835+46.19062797386781*1.0i,-12.313747867972491-8.363570364325646*1.0i,-8.231462396319344+4.263599579897788*1.0i,10.045531919479885+14.61047900343665*1.0i,24.199606613103537+14.61047902261622*1.0i,43.32220638388199+10.053515836533045*1.0i,52.112918509506244+38.11746664591408*1.0i,47.63463065732847+46.04664983323184*1.0i,-10.949530350509928+42.892826885112825*1.0i,-10.506343564100487+37.07697858601813*1.0i,-9.225342467632995-6.374641319575888*1.0i,-5.171235437378367+4.73670584517861*1.0i,10.220766158584777+11.54533957725176*1.0i,20.808273263431772+11.545339587931041*1.0i,31.358041265731465+11.545339600173271*1.0i,48.49447980896905+24.05487143714152*1.0i,44.93020246591302+34.23123868770222*1.0i,-8.978763604506938+36.19307599339755*1.0i,-8.547588424121663-12.080904732324846*1.0i,-7.294800435494734-5.189794855432012*1.0i,-3.224482270673864+4.83276111583443*1.0i,10.022034006640979+9.416050201902173*1.0i,18.487991466320267+9.416050208698277*1.0i,26.557089960840663+9.416050216357085*1.0i,43.73252756441579+9.964783641489575*1.0i,42.51684699294689+25.570152629372146*1.0i,-7.646273276764368-15.619145116421054*1.0i,-7.221608252046043-10.753312955346782*1.0i,-5.987487396834586-4.416920363980466*1.0i,-1.8712748698202644+4.773981287308398*1.0i,9.68865087707498+7.837272221559653*1.0i,16.747772443711817+7.837272226263501*1.0i,23.25903826271488+7.837272231208386*1.0i,30.89186482276262+7.837272238464681*1.0i,40.29674647809264+18.441714179671777*1.0i};

                /*
                    The tests checks, whether the two-loop functions equals the reference values up to an absolute error epsilon. For s -> 1 this
                    is not the best way to go since the functions become large and a check for relative error would be better suited. The method
                    might need to be adapted in the future to the specific value of s.
                */
                for (unsigned int i = 0; i < 45; i++)
                {
                    std::cout << i+1 << std::endl;

                    const double z = z_s_list[i][0];
                    const complex<double> s = z_s_list[i][1];
                    agv_2019a::CharmLoopsParameters testclp = {/*muhat =*/ 1.0, /*s =*/ s, /*z =*/ z, /*feynepsilonhat*/ 1e-10};

                    const complex<double> f27a_res = agv_2019a::f27a(testclp);
                    const complex<double> f29a_res = agv_2019a::f29a(testclp);

                    const complex<double> f27b_res = agv_2019a::f27b(testclp);
                    const complex<double> f29b_res = agv_2019a::f29b(testclp);

                    const complex<double> f27c_res = agv_2019a::f27c(testclp);
                    const complex<double> f29c_res = agv_2019a::f29c(testclp);

                    const complex<double> f27d_res = agv_2019a::f27d(testclp);
                    const complex<double> f29d_res = agv_2019a::f29d(testclp);

                    const complex<double> f29e_res = agv_2019a::f29e(testclp);

                    const complex<double> weight4_res = agv_2019a::GPLweight4Parts(testclp);

                    TEST_CHECK_NEARLY_EQUAL(f27a_ref[i].real(), f27a_res.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(f27a_ref[i].imag(), f27a_res.imag(), eps);

                    TEST_CHECK_NEARLY_EQUAL(f29a_ref[i].real(), f29a_res.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(f29a_ref[i].imag(), f29a_res.imag(), eps);

                    TEST_CHECK_NEARLY_EQUAL(f27b_ref[i].real(), f27b_res.real(), 15*eps);
                    TEST_CHECK_NEARLY_EQUAL(f27b_ref[i].imag(), f27b_res.imag(), 15*eps);

                    TEST_CHECK_NEARLY_EQUAL(f29b_ref[i].real(), f29b_res.real(), 30*eps);
                    TEST_CHECK_NEARLY_EQUAL(f29b_ref[i].imag(), f29b_res.imag(), 30*eps);

                    TEST_CHECK_NEARLY_EQUAL(f27c_ref[i].real(), f27c_res.real(), 5*eps);
                    TEST_CHECK_NEARLY_EQUAL(f27c_ref[i].imag(), f27c_res.imag(), 5*eps);

                    TEST_CHECK_NEARLY_EQUAL(f29c_ref[i].real(), f29c_res.real(), 150*eps);
                    TEST_CHECK_NEARLY_EQUAL(f29c_ref[i].imag(), f29c_res.imag(), 150*eps);

                    TEST_CHECK_NEARLY_EQUAL(f27d_ref[i].real(), f27d_res.real(), 190*eps);
                    TEST_CHECK_NEARLY_EQUAL(f27d_ref[i].imag(), f27d_res.imag(), 190*eps);

                    TEST_CHECK_NEARLY_EQUAL(f29d_ref[i].real(), f29d_res.real(), 1200*eps);
                    TEST_CHECK_NEARLY_EQUAL(f29d_ref[i].imag(), f29d_res.imag(), 1200*eps);

                    TEST_CHECK_NEARLY_EQUAL(f29e_ref[i].real(), f29e_res.real(), eps);
                    TEST_CHECK_NEARLY_EQUAL(f29e_ref[i].imag(), f29e_res.imag(), eps);

                    TEST_CHECK_NEARLY_EQUAL(weight4_ref[i].real(), weight4_res.real(), 5.1*eps);
                    TEST_CHECK_NEARLY_EQUAL(weight4_ref[i].imag(), weight4_res.imag(), 5.1*eps);
                }
            }
        }
} two_loop_test ;

class deltaC79Test :
    public TestCase
{
    public:
        deltaC79Test() :
            TestCase("delta_C79_test")
        {
        }


        virtual void run() const
        {
            static const double eps = 1e-10;
            static const double m_b = 4.18;
            static const double m_c = 1.47;
            static const double mu = m_b;
            static const double alpha_s = 0.6;
            static const complex<double> s = -4.0 + 1.5i;

            Parameters p = Parameters::Defaults();
            p["b->s::c1"] = -0.5;
            p["b->s::c2"] = 2.0;

            auto m = Model::make("WET", p, Options());
            const auto wc = m->wilson_coefficients_b_to_s(mu, "mu", false);

            const complex<double> deltaC7_ref = -0.080829295482612 - 0.047566420817994316 * 1.0i;
            const complex<double> deltaC9_ref = -0.7022411014307881 + 0.1880322661854362 * 1.0i;
            const complex<double> deltaC7_Qc_ref = -0.04771664722865329 - 0.04728788790918982 * 1.0i;
            const complex<double> deltaC9_Qc_ref = -0.7827669035741187 + 0.1870105731065536 * 1.0i;

            const complex<double> deltaC7_res = agv_2019a::delta_c7(s, mu, alpha_s, m_c, m_b, wc);
            const complex<double> deltaC9_res = agv_2019a::delta_c9(s, mu, alpha_s, m_c, m_b, wc);
            const complex<double> deltaC7_Qc_res = agv_2019a::delta_c7_Qc(s, mu, alpha_s, m_c, m_b, wc);
            const complex<double> deltaC9_Qc_res = agv_2019a::delta_c9_Qc(s, mu, alpha_s, m_c, m_b, wc);

            TEST_CHECK_NEARLY_EQUAL(deltaC7_ref.real(), deltaC7_res.real(), eps);
            TEST_CHECK_NEARLY_EQUAL(deltaC7_ref.imag(), deltaC7_res.imag(), eps);

            TEST_CHECK_NEARLY_EQUAL(deltaC9_ref.real(), deltaC9_res.real(), eps);
            TEST_CHECK_NEARLY_EQUAL(deltaC9_ref.imag(), deltaC9_res.imag(), eps);

            TEST_CHECK_NEARLY_EQUAL(deltaC7_Qc_ref.real(), deltaC7_Qc_res.real(), eps);
            TEST_CHECK_NEARLY_EQUAL(deltaC7_Qc_ref.imag(), deltaC7_Qc_res.imag(), eps);

            TEST_CHECK_NEARLY_EQUAL(deltaC9_Qc_ref.real(), deltaC9_Qc_res.real(), eps);
            TEST_CHECK_NEARLY_EQUAL(deltaC9_Qc_ref.imag(), deltaC9_Qc_res.imag(), eps);

        }
} delta_C79_test ;

