/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2022 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/multiplepolylog-li22.hh>
#include <eos/maths/polylog.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>

#include <cmath>
#include <complex>

namespace eos
{
    using std::complex;
    using std::log;
    using std::sqrt;
    using std::real;

    namespace agv_2019a
    {
        // Loop variables from [AGV:2019A] p. 12 eq. (3.9) and p. 15 eq. (3.20)

        inline complex<double> s_eps(const complex<double> & s, const double & feynepsilonhat)
        {
            return s / (1.0 - feynepsilonhat * 1.0i);
        }

        inline complex<double> z_eps(const double & z, const double & feynepsilonhat)
        {
            return (z - feynepsilonhat * 1.0i) / (1.0 - feynepsilonhat * 1.0i);
        }

        inline complex<double> x_a(const double & z, const double & feynepsilonhat)
        {
            if (z_eps(z, feynepsilonhat) == 0.25)
            {
                throw InternalError("0 in denominator in x_a");
            }
            else
            {
                return 1.0 / sqrt(1.0 - 4.0 * z_eps(z, feynepsilonhat));
            }
        }

        inline complex<double> x_b(const double & z, const double & feynepsilonhat)
        {
            return sqrt(4.0 * z_eps(z, feynepsilonhat)) - sqrt(4.0 * z_eps(z, feynepsilonhat) - 1.0);
        }

        inline complex<double> y_a(const complex<double> & s, const double & z, const double & feynepsilonhat)
        {
            if (s_eps(s, feynepsilonhat) == 1.0)
            {
                return 0.0;
            }
            else if (4.0 * z_eps(z, feynepsilonhat) == (1.0 - s_eps(s, feynepsilonhat)))
            {
                throw InternalError("0 in denominator in y_a");
            }
            else
            {
                return 1.0 / sqrt(1.0 - 4.0 * z_eps(z, feynepsilonhat) / (1.0 - s_eps(s, feynepsilonhat)));
            }
        }

        inline complex<double> y_b(const complex<double> & s, const double & feynepsilonhat)
        {
            if (s_eps(s, feynepsilonhat) == 0.0)
            {
                return 0.0;
            }
            else if (s_eps(s, feynepsilonhat) == 4.0)
            {
                throw InternalError("0 in denominator in y_b");
            }
            else
            {
                return 1.0 / sqrt(1.0 - 4.0 / s_eps(s, feynepsilonhat));
            }
        }

        inline complex<double> y_c(const complex<double> & s, const double & z, const double & feynepsilonhat)
        {
            if (s_eps(s, feynepsilonhat) == 0.0)
            {
                return 0.0;
            }
            else if (4.0 * z_eps(z, feynepsilonhat) == s_eps(s, feynepsilonhat))
            {
                throw InternalError("0 in denominator in y_c");
            }
            else
            {
                return 1.0 / sqrt(1.0 - 4.0 * z_eps(z, feynepsilonhat) / s_eps(s, feynepsilonhat));
            }
        }

        // Constructor of the loop parameters
        CharmLoopsParameters::CharmLoopsParameters(const double & muhat, const complex<double> & s, const double & z, const double & feynepsilonhat) :
            muhat(muhat)
        {
            s_eps = agv_2019a::s_eps(s, feynepsilonhat);
            z_eps = agv_2019a::z_eps(z, feynepsilonhat);

            xa = agv_2019a::x_a(z, feynepsilonhat);
            xb = agv_2019a::x_b(z, feynepsilonhat);
            xc = xa;
            xd = xb;
            xe = xa;

            ya = agv_2019a::y_a(s, z, feynepsilonhat);
            yb = agv_2019a::y_b(s, feynepsilonhat);
            yc = agv_2019a::y_c(s, z, feynepsilonhat);
            yd = yc;
            ye = yc;
        }

        // LO one-loop functions, ref. [AGV:2019A] p. 6 and p. 31

        complex<double> f290(const CharmLoopsParameters & clp)
        {
            const double q_c = 2.0 / 3.0; // charm quark charge

            const complex<double> result = 2.0 / 3.0 + 4.0 * clp.z_eps / clp.s_eps + log(power_of<2>(clp.muhat) / clp.z_eps)
                + (1.0 - 3.0 * power_of<2>(clp.yc)) / (2.0 * power_of<3>(clp.yc)) * (log(1.0 + clp.yc) - log(1.0 - clp.yc));

            return 2.0 / 3.0 * q_c * result;
        }

        complex<double> f190(const CharmLoopsParameters & clp)
        {
            const double c_F = 4.0 / 3.0; // SU(3) color factor

            return c_F * f290(clp);
        }

        // Counterterms to the two-loop functions

        complex<double> f17ctQs(const CharmLoopsParameters & )
        {
            return 0.0;
        }

        complex<double> f17ctQc(const CharmLoopsParameters & )
        {
            return 0.0;
        }

        complex<double> f17ctQb(const CharmLoopsParameters & clp)
        {
            const double muhat = clp.muhat;
            const complex<double> yb = clp.yb;

            const complex<double> result = - 8.0 + 4.0 * (log(1.0 + yb) - log(1.0 - yb)) / yb - 8.0 * log(muhat);

            return result / 81.0;
        }

        complex<double> f27ctQs(const CharmLoopsParameters & )
        {
            return 0.0;
        }

        complex<double> f27ctQc(const CharmLoopsParameters & )
        {
            return 0.0;
        }

        complex<double> f27ctQb(const CharmLoopsParameters & clp)
        {
            return - 6.0 * f17ctQb(clp);
        }

        complex<double> f19ctQs(const CharmLoopsParameters & clp)
        {
            const complex<double> s = clp.s_eps;
            const complex<double> lnms = log(-s);
            const double lnmuhat = log(clp.muhat);

            // Mathematica: -104/2187 + (2*Pi^2)/729 - (32*Log[mub])/729 - (16*Log[mub]^2)/243 + (16*Log[-s])/729 + (16*Log[mub]*Log[-s])/243 - (4*Log[-s]^2)/243;
            const complex<double> result = - 104.0 / 9.0 + 2.0 * pisqu / 3.0 - 32.0 * lnmuhat / 3.0 - 16.0 * power_of<2>(lnmuhat) + 16.0 * lnms / 3.0
                + 16.0 * lnmuhat * lnms - 4.0 * power_of<2>(lnms);

            return result / 243.0;
        }

        complex<double> f19ctQc(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xa = clp.xa;
            const complex<double> yc = clp.yc;

            const complex<double> yc2 = power_of<2>(yc);
            const complex<double> yc3 = power_of<3>(yc);

            const complex<double> lnz    = log(clp.z_eps);
            const complex<double> lnxa   = log(xa);
            const complex<double> ln1pxa = log(1.0 + xa);
            const complex<double> ln1mxa = log(1.0 - xa);
            const complex<double> ln1pyc = log(1.0 + yc);
            const complex<double> ln1myc = log(1.0 - yc);

            const complex<double> result = 1792.0 + (768.0 * 1.0i) * M_PI - 20.0 * pisqu - 1536.0 / yc2 - ((624.0 * 1.0i) * M_PI) / yc2
                + 96.0 * dilog(1.0 / 2.0) - 48.0 * dilog((1.0 - xa) / 2.0) - 48.0 * dilog((1.0 + xa) / 2.0) + (-312.0 / yc3 + 648.0 / yc - 288.0 * yc) * dilog((1.0 - yc) / 2.0)
                + (312.0 / yc3 - 648.0 / yc + 288.0 * yc) * dilog((1.0 + yc) / 2.0) + 1536.0 * ln2 + (96.0 * 1.0i) * M_PI * ln2 - (1248.0 * ln2) / yc2 + 96.0 * power_of<2>(ln2) + 2688.0 * lnmuhat + (96.0 * 1.0i) * M_PI * lnmuhat
                - (2400.0 * lnmuhat) / yc2 + 192.0 * ln2 * lnmuhat + 96.0 * power_of<2>(lnmuhat) + 24.0 * power_of<2>(ln1mxa) + ln1mxa * (-768.0 - (48.0 * 1.0i) * M_PI + 624.0 / yc2 - 48.0 * ln2 - 96.0 * lnmuhat - 96.0 * lnxa)
                + 96.0 * power_of<2>(lnxa) + lnxa * (1536.0 + (96.0 * 1.0i) * M_PI - 1248.0 / yc2 + 192.0 * ln2 + 192.0 * lnmuhat - 96.0 * ln1pxa) + (-768.0 - (48.0 * 1.0i) * M_PI + 624.0 / yc2 - 48.0 * ln2 - 96.0 * lnmuhat) * ln1pxa
                + 24.0 * power_of<2>(ln1pxa) + (-156.0 / yc3 + 324.0 / yc - 144.0 * yc) * power_of<2>(ln1myc) + (156.0 / yc3 - 324.0 / yc + 144.0 * yc) * power_of<2>(ln1pyc) - 576.0 * lnz + (576.0 * lnz) / yc2
                + ln1pyc * (768.0 / yc3 + ((312.0 * 1.0i) * M_PI) / yc3 - 1008.0 / yc - ((648.0 * 1.0i) * M_PI) / yc + 96.0 * yc + (288.0 * 1.0i) * M_PI * yc + (312.0 * ln2) / yc3 - (648.0 * ln2) / yc + 288.0 * yc * ln2
                    + (1200.0 * lnmuhat) / yc3 - (2448.0 * lnmuhat) / yc + 1152.0 * yc * lnmuhat + (-312.0 / yc3 + 648.0 / yc - 288.0 * yc) * ln1mxa + (624.0 / yc3 - 1296.0 / yc + 576.0 * yc) * lnxa
                    + (-312.0 / yc3 + 648.0 / yc - 288.0 * yc) * ln1pxa - (288.0 * lnz) / yc3 + (576.0 * lnz) / yc - 288.0 * yc * lnz)
                + ln1myc * (-768.0 / yc3 - ((312.0 * 1.0i) * M_PI) / yc3 + 1008.0 / yc + ((648.0 * 1.0i) * M_PI) / yc - 96.0 * yc - (288.0 * 1.0i) * M_PI * yc - (312.0 * ln2) / yc3 + (648.0 * ln2) / yc - 288.0 * yc * ln2
                    - (1200.0 * lnmuhat) / yc3 + (2448.0 * lnmuhat) / yc - 1152.0 * yc * lnmuhat + (312.0 / yc3 - 648.0 / yc + 288.0 * yc) * ln1mxa + (-624.0 / yc3 + 1296.0 / yc - 576.0 * yc) * lnxa
                    + (312.0 / yc3 - 648.0 / yc + 288.0 * yc) * ln1pxa + (288.0 * lnz) / yc3 - (576.0 * lnz) / yc + 288.0 * yc * lnz);

            return result / 81.0;
        }

        complex<double> f19ctQb(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> yb = clp.yb;

            const complex<double> yb3 = power_of<3>(yb);

            const complex<double> ln1pyb    = log(1.0 + yb);
            const complex<double> ln1myb    = log(1.0 - yb);
            const complex<double> dilog1pybhalf = dilog((1.0 + yb) / 2.0);
            const complex<double> dilog1mybhalf = dilog((1.0 - yb) / 2.0);

            const complex<double> result = -224.0 / 27.0 - (2.0 * power_of<2>(M_PI)) / 9.0 + 40.0 / (9.0 * power_of<2>(yb))
                - (80.0 * lnmuhat) / 9.0 + (16.0 * lnmuhat) / (3.0 * power_of<2>(yb)) - (16.0 * power_of<2>(lnmuhat)) / 3.0
                + (4.0 / (3.0 * yb3) - 4.0 / yb) * dilog1mybhalf + (-4.0 / (3.0 * yb3) + 4.0 / yb) * dilog1pybhalf
                + (20.0 / (9.0 * yb3) + (8.0 * lnmuhat) / (3.0 * yb3) - (4.0 + 8.0 * lnmuhat) / yb - (4.0 * ln2) / (3.0 * yb3) + (4.0 * ln2) /  yb) * ln1myb
                + (2.0 / (3.0 * yb3) - 2.0 / yb) * power_of<2>(ln1myb)
                + (-20.0 / (9.0 * yb3) + 4.0 / yb - (8.0 * lnmuhat) / (3.0 * yb3) + (8.0 * lnmuhat) / yb + (4.0 * ln2) / (3.0 * yb3) - (4.0 * ln2) / yb) * ln1pyb
                + (-2.0 / (3.0 * yb3) + 2.0 / yb) * power_of<2>(ln1pyb);

            return result / 81.0;
        }

        complex<double> f29ctQs(const CharmLoopsParameters & clp)
        {
            return - 6.0 * f19ctQs(clp);
        }

        complex<double> f29ctQc(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xa = clp.xa;
            const complex<double> yc = clp.yc;

            const complex<double> yc2 = power_of<2>(yc);
            const complex<double> yc3 = power_of<3>(yc);

            const complex<double> lnz     = log(clp.z_eps);
            const complex<double> lnxa    = log(xa);
            const complex<double> ln1pxa  = log(1.0 + xa);
            const complex<double> ln1mxa  = log(1.0 - xa);
            const complex<double> ln1pyc  = log(1.0 + yc);
            const complex<double> ln1myc  = log(1.0 - yc);
            const complex<double> ln1pyc2 = power_of<2>(ln1pyc);
            const complex<double> ln1myc2 = power_of<2>(ln1myc);

            const complex<double> result = 208.0 + (48.0 * 1.0i * M_PI) + 40.0 * power_of<2>(M_PI) - 240.0 / yc2 - (48.0 * 1.0i * M_PI) / yc2
                - 192.0 * dilog(1.0 / 2.0) + 96.0 * dilog((1.0 - xa) / 2.0) + 96.0 * dilog((1.0 + xa) / 2.0)
                + (-24.0 / yc3 - 72.0 * yc) * dilog((1.0 - yc) / 2.0) + (24.0 / yc3 + 72.0 * yc) * dilog((1.0 + yc) / 2.0) + 96.0 * ln2 - (192.0 * 1.0i * M_PI *  ln2)
                - (96.0 * ln2) / yc2 - 192.0 * power_of<2>(ln2) + 384.0 * lnmuhat - (192.0 * 1.0i * M_PI *  lnmuhat) - (384.0 * lnmuhat) / yc2 - (384.0 * ln2 *  lnmuhat)
                - 192.0 * power_of<2>(lnmuhat) - 48.0 * power_of<2>(ln1mxa) - 192.0 * power_of<2>(lnxa) + ln1mxa * (-48.0 + (96.0 * 1.0i *  M_PI) + 48.0 / yc2 + 96.0 * ln2 + 192.0 * lnmuhat
                + 192.0 * lnxa) + (-48.0 + (96.0 * 1.0i *  M_PI) + 48.0 / yc2 + 96.0 * ln2 + 192.0 * lnmuhat) * ln1pxa - 48.0 * power_of<2>(ln1pxa)
                + lnxa * (96.0 - (192.0 * 1.0i * M_PI) - 96.0 / yc2 - 384.0 * ln2 - 384.0 * lnmuhat + 192.0 * ln1pxa)
                + (-12.0 / yc3 - 36.0 * yc) * ln1myc2 + (12.0 / yc3 + 36.0 * yc) * ln1pyc2 - 144.0 * lnz + (144.0 * lnz) / yc2
                + ln1pyc * (120.0 / yc3 + (24.0 * 1.0i * M_PI) / yc3 - 144.0 / yc + 24.0 * yc + (72.0 * 1.0i * M_PI *  yc) + (24.0 * ln2) / yc3 + (72.0 * yc *  ln2)
                    + (192.0 * lnmuhat) / yc3 - (288.0 * lnmuhat) / yc + (288.0 * yc *  lnmuhat) + (-24.0 / yc3 - 72.0 * yc) * ln1mxa
                    + (48.0 / yc3 + 144.0 * yc) * lnxa + (-24.0 / yc3 - 72.0 * yc) * ln1pxa - (72.0 * lnz) / yc3 + (144.0 * lnz) / yc - (72.0 * yc *  lnz))
                + ln1myc * (-120.0 / yc3 - (24.0 * 1.0i * M_PI) / yc3 + 144.0 / yc - 24.0 * yc - (72.0 * 1.0i * M_PI *  yc) - (24.0 * ln2) / yc3 - (72.0 * yc *  ln2)
                    - (192.0 * lnmuhat) / yc3 + (288.0 * lnmuhat) / yc - (288.0 * yc *  lnmuhat) + (24.0 / yc3 + 72.0 * yc) * ln1mxa
                    + (-48.0 / yc3 - 144.0 * yc) * lnxa + (24.0 / yc3 + 72.0 * yc) * ln1pxa + (72.0 * lnz) / yc3 - (144.0 * lnz) / yc + (72.0 * yc *  lnz));

            return result / 27.0;
        }

        complex<double> f29ctQb(const CharmLoopsParameters & clp)
        {
            return -6.0 * f19ctQb(clp);
        }

        // NLO two-loop functions

        complex<double> f17a(const CharmLoopsParameters & clp)
        {
            // f17a = - f27a / (2 * N_c)
            return - f27a(clp) / 6.0;
        }

        complex<double> f19a(const CharmLoopsParameters & clp)
        {
            // f19a = - f29a / (2 * N_c)
            return - f29a(clp) / 6.0;
        }

        complex<double> f27a(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xa = clp.xa;
            const complex<double> ya = clp.ya;

            const complex<double> xa2 = power_of<2>(xa);
            const complex<double> xa3 = power_of<3>(xa);
            const complex<double> xa4 = power_of<4>(xa);

            const complex<double> ya2 = power_of<2>(ya);
            const complex<double> ya3 = power_of<3>(ya);
            const complex<double> ya4 = power_of<4>(ya);
            const complex<double> ya5 = power_of<5>(ya);
            const complex<double> ya6 = power_of<6>(ya);
            const complex<double> ya7 = power_of<7>(ya);
            const complex<double> ya8 = power_of<8>(ya);

            const complex<double> xainv = 1.0 / xa;
            const complex<double> yainv = 1.0 / ya;

            const double imxa = imag(xa);
            const double imyainv = imag(yainv);

            const complex<double> ln1pxa  = log(1.0 + xa);
            const complex<double> ln1mxa  = log(1.0 - xa);
            const complex<double> ln1pya  = log(1.0 + ya);
            const complex<double> ln1mya  = log(1.0 - ya);

            const complex<double> num1 = (-1.0 + ya) * (1.0 + ya) * (3.0 * ya4 - 3.0 * xa * ya4 + xa2 * (9.0 * ya2 - 106.0 * ya4) + xa3 * (9.0 * ya2 - 38.0 * ya4) + 2.0 * xa4 * (-9.0 + 72.0 * ya2 + 4.0 * ya4));
            const complex<double> num2 = 4.0 * (-1.0 + ya) * (1.0 + ya) * (9.0 * ya2 - 72.0 * ya4 + xa2 * (-9.0 + 72.0 * ya2 + 4.0 * ya4));
            const complex<double> num3 = (-1.0 + ya) * (1.0 + ya) * (3.0 * ya4 + 3.0 * xa * ya4 + xa3 * ya2 * (-9.0 + 38.0 * ya2) + xa2 * (9.0 * ya2 - 106.0 * ya4) + 2.0 * xa4 * (-9.0 + 72.0 * ya2 + 4.0 * ya4));
            const complex<double> num4 = 2.0 * (xa - ya) * (xa + ya) * (9.0 - 81.0 * ya2 + 68.0 * ya4);
            const complex<double> num5 = (-1.0 + ya) * (1.0 + ya) * (ya4 + xa2 * ya2 * (3.0 + 7.0 * ya2) + xa4 * (-6.0 - 15.0 * ya2 + 10.0 * ya4));

            const complex<double> factor1 = (1.0 / (9.0 * (-1.0 + xa2) * ya7)) * 2.0 * (xa - ya) * (-1.0 + ya) * (1.0 + ya) * (xa + ya) * (-1.0 + 3.0 * ya2);
            const complex<double> factor2 = (1.0 / (9.0 * (-1.0 + xa2) * ya8)) * (xa - ya) * (xa + ya) * power_of<2>(-1.0 + ya2) * (1.0 + 3.0 * ya2);

            const complex<double> logs1 = 2.0 * (xa - ya) * (xa + ya) * (9.0 * ya - 81.0 * ya3 + 68.0 * ya5 - 18.0 * 1.0i * M_PI * (1.0 - 4.0 * ya2 + 3.0 * ya4) - 18.0 * ln2 + 72.0 * ya2 * ln2 - 54.0 * ya4 * ln2);
            const complex<double> logs2 = 2.0 * (xa - ya) * (xa + ya) * (9.0 * ya - 81.0 * ya3 + 68.0 * ya5 + 18.0 * 1.0i * M_PI * (1.0 - 4.0 * ya2 + 3.0 * ya4) + 18.0 * ln2 - 72.0 * ya2 * ln2 + 54.0 * ya4 * ln2);
            const complex<double> logs3 = 2.0 * (xa - ya) * (-1.0 + ya) * (1.0 + ya) * (xa + ya) * (ya - 3.0 * ya3 - 1.0i * M_PI * (-1.0 - 2.0 * ya2 + 3.0 * ya4) + ln2 + ya2 * ln4 - ya4 * log(8.0));
            const complex<double> logs4 = 2.0 * (xa - ya) * (-1.0 + ya) * (1.0 + ya) * (xa + ya) * (-ya + 3.0 * ya3 - 1.0i * M_PI * (-1.0 - 2.0 * ya2 + 3.0 * ya4) + ln2 + ya2 * ln4 - ya4 * log(8.0));
            const complex<double> logs5 = 2.0 * (xa - ya) * (-1.0 + ya) * (1.0 + ya) * (xa + ya) * (ya - 3.0 * ya3 + 1.0i * M_PI * (-1.0 - 2.0 * ya2 + 3.0 * ya4) - ln2 - ya2 * ln4 + ya4 * log(8.0));
            const complex<double> logs6 = 2.0 * (xa - ya) * (-1.0 + ya) * (1.0 + ya) * (xa + ya) * (-ya + 3.0 * ya3 + 1.0i * M_PI * (-1.0 - 2.0 * ya2 + 3.0 * ya4) - ln2 - ya2 * ln4 + ya4 * log(8.0));

            const complex<double> logs7 = 2.0 * (xa - ya) * (xa + ya) * (9.0 * ya - 81.0 * ya3 + 68.0 * ya5 - 18.0 * ln2 + 72.0 * ya2 * ln2 - 54.0 * ya4 * ln2);
            const complex<double> logs8 = 2.0 * (xa - ya) * (xa + ya) * (9.0 * ya - 81.0 * ya3 + 68.0 * ya5 + 18.0 * ln2 - 72.0 * ya2 * ln2 + 54.0 * ya4 * ln2);
            const complex<double> logs9 = 2.0 * (xa - ya) * (-1.0 + ya) * (1.0 + ya) * (xa + ya) * (ya - 3.0 * ya3 - ln2 - ya2 * ln4 + ya4 * log(8.0));
            const complex<double> logs10 = 2.0 * (xa - ya) * (-1.0 + ya) * (1.0 + ya) * (xa + ya) * (-ya + 3.0 * ya3 - ln2 - ya2 * ln4 + ya4 * log(8.0));

            const complex<double> term1 = (1.0 / (243.0 * xa2 * (-1.0 + xa2) * ya6)) * (9.0 * ya4 * (-1.0 + ya2) + 12.0 * 1.0i * M_PI * xa2 * (-9.0 * ya2 + 81.0 * ya4 - 70.0 * ya6 + xa2 * (9.0 - 81.0 * ya2 + 68.0 * ya4 + 2.0 * ya6))
                + xa2 * ya2 * (27.0 + ya4 * (217.0 - 864.0 * ln2) - 108.0 * ln2 + 6.0 * ya2 * (-53.0 + 162.0 * ln2)) + xa4 * (108.0 * ln2 - 27.0 * ya2 * (1.0 + 36.0 * ln2) + ya6 * (-226.0 + 48.0 * ln2)
                    + ya4 * (327.0 + 816.0 * ln2))) + (16.0 * lnmuhat) / 81.0;
            const complex<double> term2 = (1.0 / (243.0 * xa2 * (-1.0 + xa2) * ya6)) * (9.0 * ya4 * (-1.0 + ya2) + xa2 * ya2 * (27.0 + ya4 * (217.0 - 24.0 * 1.0i * M_PI - 864.0 * ln2) - 108.0 * ln2 + 6.0 * ya2 * (-53.0 + 162.0 * ln2))
                + xa4 * (108.0 * ln2 - 27.0 * ya2 * (1.0 + 36.0 * ln2) + ya6 * (-226.0 + 24.0 * 1.0i * M_PI + 48.0 * ln2) + ya4 * (327.0 + 816.0 * ln2))) + (16.0 * lnmuhat) / 81.0;

            const complex<double> denom1 = (81.0 * (-1.0 + xa) * xa3 * ya6);
            const complex<double> denom2 = (81.0 * (-1.0 + xa2) * ya6);
            const complex<double> denom3 = (81.0 * xa3 * (1.0 + xa) * ya6);
            const complex<double> denom4 = (54.0 * xa4 * ya6);
            const complex<double> denom5 = (81.0  * (-1.0 + xa2) * ya7);
            const complex<double> denom6 = (9.0 * (-1.0 + xa2) * ya8);

            // Heaviside theta related functions
            const double t11mya1mxa = T(1.0, 1.0 - ya, 1.0 - xa); // MyT[1, 1 - ya, 1 - xa]
            const double t11pya1mxa = T(1.0, 1.0 + ya, 1.0 - xa); // MyT[1, 1 + ya, 1 - xa]
            const double t11mya1pxa = T(1.0, 1.0 - ya, 1.0 + xa); // MyT[1, 1 - ya, 1 + xa]
            const double t11pya1pxa = T(1.0, 1.0 + ya, 1.0 + xa); // MyT[1, 1 + ya, 1 + xa]

            complex<double> result;

            if (imag(clp.s_eps) > 0)
            {
                // F27aupper
                result = term2 - (num3 * ln1mxa) / denom3 - (num1 * ln1pxa) / denom1 + (2.0 * num5 * li2half - num5 * dilog((1.0 - xa) / 2.0) - num5 * dilog((1.0 + xa) / 2.0) + num5 * ln2 * ln1mxa - (num5 * power_of<2>(ln1mxa)) / 2.0
                        + num5 * ln2 * ln1pxa - (num5 * power_of<2>(ln1pxa)) / 2.0) / denom4 + (-(logs7 * ln1mya) - logs8 * ln1pya) / denom5
                    + ((-logs10 - logs9) * li2half + logs10 * dilog((1.0 - ya) / 2.0) + logs9 * dilog((1.0 + ya) / 2.0) - logs10 * ln2 * ln1mya + (logs9 * power_of<2>(ln1mya)) / 2.0 - logs9 * ln2 * ln1pya + (logs10 * power_of<2>(ln1pya)) / 2.0) / denom6
	                + (num2 * log(xa) + num4 * log((xa + ya) * xainv) + num4 * log(1.0 - ya * xainv)) / denom2 + factor1 * (dilog((xa - ya) / (-1.0 + xa)) - dilog((xa - ya) / (1.0 + xa)) - dilog((xa + ya) / (-1.0 + xa))
                        + dilog((xa + ya) / (1.0 + xa)) + (-ln1mxa + 2.0 * log(xa) - ln1pxa) * ln1mya + (ln1mxa - 2.0 * log(xa) + ln1pxa) * ln1pya
                        + (-ln1mxa + ln1pxa + log((1.0 - ya) / (1.0 + xa)) - log((1.0 + ya) / (1.0 - xa))) * log((xa + ya) * xainv) + (ln1mxa - ln1pxa + log((-1.0 + ya) / (-1.0 + xa)) - log((1.0 + ya) / (1.0 + xa))) * log(1.0 - ya * xainv)
	                + log(1.0 + xainv) * ((2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + log((-1.0 + xa) * xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                    + factor2 * ((pisqu * ln2) / 6.0 - 4.0 * trilog(-1.0) - 2.0 * li3half + 2.0 * trilog(1.0 / (1.0 - xa)) - 2.0 * trilog(1.0 - xa) + 2.0 * trilog((-1.0 + xa) / (2.0 * xa)) + 2.0 * trilog(1.0 / (1.0 + xa)) + 2.0 * trilog((1.0 - xa) / (1.0 + xa))
                        - 2.0 * trilog(1.0 + xa) + 2.0 * trilog((1.0 + xa) / (1.0 - xa)) + 2.0 * trilog((1.0 + xa) / (2.0 * xa)) + trilog((1.0 + xa) / (1.0 - ya)) + 4.0 * trilog((1.0 - ya) / 2.0) - trilog((1.0 - ya) / (1.0 + xa)) + trilog((-1.0 + xa) / (-1.0 + ya))
                        - trilog((-1.0 + ya) / (-1.0 + xa)) - trilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) + trilog((1.0 - xa) / (1.0 + ya)) + trilog((1.0 + xa) / (1.0 + ya)) + 2.0 * trilog((-1.0 + ya) / (1.0 + ya))
                        - trilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) - trilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) + 4.0 * trilog((1.0 + ya) / 2.0) - trilog((1.0 + ya) / (1.0 - xa)) - trilog((1.0 + ya) / (1.0 + xa))
                        - trilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) + 2.0 * trilog((1.0 + ya) / (-1.0 + ya)) - trilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya)))
                        - trilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) + (pisqu * ln2) / 3.0 - power_of<3>(ln2) - 2.0 * dilog((1.0 + xa) / (1.0 - xa)) * log(1.0 / (1.0 - xa))
                        - 2.0 * dilog((1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 - xa)) - power_of<3>(ln1mxa) / 3.0 - 2.0 * dilog((-1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 + xa)) - 2.0 * dilog((1.0 - xa) / (1.0 + xa)) * log(1.0 / (1.0 + xa))
                        + dilog(-1.0) * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * log(1.0 / (1.0 + xa))) + dilog((xa - ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa) + dilog((xa + ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa)
                        - power_of<3>(ln1pxa) / 3.0 + dilog((xa - ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + dilog((xa + ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + (pisqu * log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0
                        + power_of<3>(log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0 + (pisqu * log((-1.0 + xa2) / (2.0 * xa))) / 3.0 + power_of<3>(log((-1.0 + xa2) / (2.0 * xa))) / 3.0 - (pisqu * log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0 - power_of<3>(log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 - ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 - ya))) / 6.0 - power_of<3>(ln1mya) / 6.0 + dilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) * log((1.0 - ya) / (1.0 + xa))
                        + dilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) * log((1.0 - ya) / (1.0 + xa)) - (pisqu * log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 + (pisqu * log((-1.0 + xa) / (-1.0 + ya))) / 6.0
                        + power_of<3>(log((-1.0 + xa) / (-1.0 + ya))) / 6.0 + dilog((-1.0 + ya) / (1.0 + ya)) * (-log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa))) + dilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) * log((-1.0 + ya) / (-1.0 + xa))
                        + dilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) * log((-1.0 + ya) / (-1.0 + xa)) - (power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * log(2.0 / (1.0 + ya))) / 2.0 + (pisqu * log((1.0 - xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 - xa) / (1.0 + ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 + ya))) / 6.0 - power_of<3>(ln1pya) / 6.0 + dilog(1.0 / (1.0 - xa)) * (ln1mya + ln1pya) + dilog(1.0 / (1.0 + xa)) * (ln1mya + ln1pya)
                        + dilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 - xa)) + dilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) * log((1.0 + ya) / (1.0 - xa))
                        + dilog((1.0 + ya) / (-1.0 + ya)) * (-log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa))) + dilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) * log((1.0 + ya) / (1.0 + xa))
                        + dilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 + xa)) - (log(-2.0 / (-1.0 + ya)) * power_of<2>(log((1.0 + ya) / (1.0 + xa)))) / 2.0 - (pisqu * log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0
                        - power_of<3>(log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0 - (pisqu * log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0
                        + li2half * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 4.0 * log(xa) + 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + 2.0 * ln1mya + 2.0 * ln1pya - 2.0 * log((xa + ya) * xainv) - 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 - ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa)) - 2.0 * ln1pya - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + dilog((1.0 + ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - 2.0 * ln1pya - log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa)) - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + power_of<2>(ln1pya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((1.0 - ya) / 2.0) / 2.0 - log((xa - ya) / (1.0 + xa)) / 2.0 - log((xa + ya) / (-1.0 + xa)) / 2.0 + log((xa + ya) * xainv) / 2.0
                            + log(1.0 - ya * xainv) / 2.0) + power_of<2>(ln1mya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((xa - ya) / (-1.0 + xa)) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log((xa + ya) * xainv) / 2.0
                            - log((xa + ya) / (1.0 + xa)) / 2.0 + log(1.0 - ya * xainv) / 2.0) + dilog((1.0 - xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 + xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv)) + log((1.0 - ya) / 2.0) * ((-0.5) * power_of<2>(log((1.0 + ya) / (1.0 - xa))) - power_of<2>(log((1.0 + ya) / (1.0 + xa))) / 2.0
                            - log((1.0 + ya) / (1.0 - xa)) * log((xa + ya) * xainv) - log((1.0 + ya) / (1.0 + xa)) * log(1.0 - ya * xainv)) - (2.0 * 1.0i) * M_PI * H1((1.0 + xa) / (-1.0 + xa), 1.0 + xainv) * power_of<2>(log(-1.0 / (1.0 + xa))) * my_sign(-imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) * xainv, 1.0 + xainv) * power_of<2>(log((2.0 * xa) / (-1.0 + xa2))) * my_sign(-imag(xainv)) - (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) * xainv) * power_of<2>(log(1.0 / (-1.0 + xa))) * my_sign(imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1(1.0 + xainv, (-1.0 + xa) * xainv) * power_of<2>(log((-2.0 * xa) / (-1.0 + xa2))) * my_sign(imag(xainv)) + 1.0i * M_PI * H1((1.0 + xa) / (xa - ya), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((-2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((1.0 + ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa - ya))) + 1.0i * M_PI * H1((-1.0 + xa) / (xa - ya), (1.0 + xa) / (xa - ya)) * power_of<2>(log((2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa - ya)))
	                    - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa - ya)) * power_of<2>(log((-1.0 + ya) / (1.0 + xa))) * my_sign(imag((1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((1.0 - ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa + ya))) + 1.0i * M_PI * H1((1.0 + xa) / (xa + ya), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((-2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa + ya)))
	                    - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa + ya)) * power_of<2>(log(-((1.0 + ya) / (1.0 + xa)))) * my_sign(imag((1.0 + xa) / (xa + ya)))
                        + 1.0i * M_PI * H1((-1.0 + xa) / (xa + ya), (1.0 + xa) / (xa + ya)) * power_of<2>(log((2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa + ya)))
                        + power_of<2>(log(1.0 / (1.0 + xa))) * (lnhalf + ln2 - log((2.0 * xa) / (1.0 + xa)) - log((1.0 + xa) / (2.0 * xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xa / (1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), 1.0 / (1.0 + xa)))
                        + power_of<2>(log(1.0 / (1.0 - xa))) * (lnhalf + ln2 - log((-1.0 + xa) / (2.0 * xa)) - log((2.0 * xa) / (-1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xa / (1.0 - xa))) * T(1.0, 1.0 / (1.0 + xa), 1.0 / (1.0 - xa)))
                        + ln1mya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0 + (2.0 * ln2 - log((1.0 - ya) / 2.0)) * ln1pya - log((1.0 + ya) / 2.0) * ln1pya
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1pxa))
                        + power_of<2>(log((1.0 + ya) / (1.0 - xa))) * ((-0.5) * log(-2.0 / (-1.0 + ya)) + log((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) / 2.0 + log((-2.0 * (xa + ya)) / ((-1.0 + xa) * (-1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (-1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), (1.0 + ya) / (1.0 - xa))) + power_of<2>(log((1.0 + ya) / (1.0 + xa))) * (log((-2.0 * (xa - ya)) / ((1.0 + xa) * (-1.0 + ya))) / 2.0
                            + log((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) / 2.0 - 1.0i * M_PI * my_sign(imag((xa - ya) / (1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), (1.0 + ya) / (1.0 + xa)))
                        + power_of<2>(ln1mxa) * ((-1.0i) * M_PI * my_sign(imxa) * t11mya1mxa - 1.0i * M_PI * my_sign(imxa) * t11pya1mxa) + power_of<2>(ln1pxa) * ((-1.0i) * M_PI * my_sign(-imxa) * t11mya1pxa
                            - 1.0i * M_PI * my_sign(-imxa) * t11pya1pxa) + ln1pya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1pxa))
                        + power_of<2>(log(-2.0 / (-1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), -2.0 / (-1.0 + xa)) + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa))) + log((xa + ya) * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0
                            - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa + ya))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa + ya))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa)))
                        + power_of<2>(log((1.0 - ya) / (1.0 + xa))) * ((-0.5) * log(2.0 / (1.0 + ya)) - log((1.0 + ya) / 2.0) / 2.0 + log(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) / 2.0 + log((2.0 * (xa + ya)) / ((1.0 + xa) * (1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), (1.0 - ya) / (1.0 + xa))) + power_of<2>(log(2.0 / (1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, 1.0 / (1.0 + xa), 2.0 / (1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + log(1.0 - ya * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0 - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa - ya))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
	                    - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa - ya))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * (log((2.0 * (xa - ya)) / ((-1.0 + xa) * (1.0 + ya))) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa - ya) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), (-1.0 + ya) / (-1.0 + xa))) + log((1.0 + ya) / (1.0 - xa)) * ((ln1mxa - ln1pxa) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)) + log((1.0 - ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log((xa + ya) * xainv) - log((1.0 + ya) / 2.0) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)) + dilog(xa / (-1.0 + xa)) * (-2.0 * ln1mxa - 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + log((1.0 - ya) / (1.0 + xa))
                            + log((1.0 + ya) / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + dilog((1.0 + xa) / (-1.0 + xa)) * (2.0 * log(1.0 / (1.0 + xa)) - log((1.0 - ya) / (1.0 + xa)) - log((1.0 + ya) / (1.0 + xa)) + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + log((-1.0 + ya) / (-1.0 + xa)) * ((ln1mxa - ln1pxa) * log(1.0 - ya * xainv) - log((1.0 + ya) / 2.0) * log(1.0 - ya * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + dilog(xa / (1.0 + xa)) * (-2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 2.0 * ln1pxa + log((-1.0 + ya) / (-1.0 + xa))
                            + log((1.0 + ya) / (1.0 - xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + dilog((-1.0 + xa) / (1.0 + xa)) * (2.0 * log(1.0 / (1.0 - xa)) - log((-1.0 + ya) / (-1.0 + xa)) - log((1.0 + ya) / (1.0 - xa)) + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)) + log((1.0 + ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log(1.0 - ya * xainv) + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                            + ln1pxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) + (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + ln1mxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + log(1.0 + xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, 1.0 + xainv) * T(p(-xainv, (xa + ya) * xainv), (xa + ya) * xainv, (-1.0 + xa) * xainv) + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, 1.0 + xainv) * T(p(-xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, (-1.0 + xa) * xainv))
	                    + log((-1.0 + xa) * xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) * T(p(xainv, (xa + ya) * xainv), (xa + ya) * xainv, 1.0 + xainv)
                            + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv) * T(p(xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, 1.0 + xainv)) - (21.0 * zeta3) / 4.0);

                return result;
            }
            else
            {
                // F27alower
                result = term1 - (num3 * ln1mxa) / denom3 - (num1 * ln1pxa) / denom1 + (2.0 * num5 * li2half - num5 * dilog((1.0 - xa) / 2.0) - num5 * dilog((1.0 + xa) / 2.0) + num5 * ln2 * ln1mxa - (num5 * power_of<2>(ln1mxa)) / 2.0
                        + num5 * ln2 * ln1pxa - (num5 * power_of<2>(ln1pxa)) / 2.0) / denom4 + (-(logs1 * ln1mya) - logs2 * ln1pya) / denom5
                    + ((logs3 + logs4) * li2half - logs3 * dilog((1.0 - ya) / 2.0) - logs4 * dilog((1.0 + ya) / 2.0) + logs3 * ln2 * ln1mya + (logs5 * power_of<2>(ln1mya)) / 2.0 + logs4 * ln2 * ln1pya + (logs6 * power_of<2>(ln1pya)) / 2.0) / denom6
                    + (num2 * log(xa) + num4 * log((xa + ya) * xainv) + num4 * log(1.0 - ya * xainv)) / denom2 + factor1 * (dilog((xa - ya) / (-1.0 + xa)) - dilog((xa - ya) / (1.0 + xa)) - dilog((xa + ya) / (-1.0 + xa))
                        + dilog((xa + ya) / (1.0 + xa)) + (-ln1mxa + 2.0 * log(xa) - ln1pxa) * ln1mya + (ln1mxa - 2.0 * log(xa) + ln1pxa) * ln1pya
                        + (-ln1mxa + ln1pxa + log((1.0 - ya) / (1.0 + xa)) - log((1.0 + ya) / (1.0 - xa))) * log((xa + ya) * xainv) + (ln1mxa - ln1pxa + log((-1.0 + ya) / (-1.0 + xa)) - log((1.0 + ya) / (1.0 + xa))) * log(1.0 - ya * xainv)
                        + log(1.0 + xainv) * ((2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + log((-1.0 + xa) * xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                    + factor2 * ((pisqu * ln2) / 6.0 - 4.0 * trilog(-1.0) - 2.0 * li3half + 2.0 * trilog(1.0 / (1.0 - xa)) - 2.0 * trilog(1.0 - xa) + 2.0 * trilog((-1.0 + xa) / (2.0 * xa)) + 2.0 * trilog(1.0 / (1.0 + xa)) + 2.0 * trilog((1.0 - xa) / (1.0 + xa))
                        - 2.0 * trilog(1.0 + xa) + 2.0 * trilog((1.0 + xa) / (1.0 - xa)) + 2.0 * trilog((1.0 + xa) / (2.0 * xa)) + trilog((1.0 + xa) / (1.0 - ya)) + 4.0 * trilog((1.0 - ya) / 2.0) - trilog((1.0 - ya) / (1.0 + xa)) + trilog((-1.0 + xa) / (-1.0 + ya))
                        - trilog((-1.0 + ya) / (-1.0 + xa)) - trilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) + trilog((1.0 - xa) / (1.0 + ya)) + trilog((1.0 + xa) / (1.0 + ya)) + 2.0 * trilog((-1.0 + ya) / (1.0 + ya))
                        - trilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) - trilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) + 4.0 * trilog((1.0 + ya) / 2.0) - trilog((1.0 + ya) / (1.0 - xa)) - trilog((1.0 + ya) / (1.0 + xa))
                        - trilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) + 2.0 * trilog((1.0 + ya) / (-1.0 + ya)) - trilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya)))
                        - trilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) + (pisqu * ln2) / 3.0 - power_of<3>(ln2) - 2.0 * dilog((1.0 + xa) / (1.0 - xa)) * log(1.0 / (1.0 - xa))
                        - 2.0 * dilog((1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 - xa)) - power_of<3>(ln1mxa) / 3.0 - 2.0 * dilog((-1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 + xa)) - 2.0 * dilog((1.0 - xa) / (1.0 + xa)) * log(1.0 / (1.0 + xa))
                        + dilog(-1.0) * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * log(1.0 / (1.0 + xa))) + dilog((xa - ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa) + dilog((xa + ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa)
                        - power_of<3>(ln1pxa) / 3.0 + dilog((xa - ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + dilog((xa + ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + (pisqu * log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0
                        + power_of<3>(log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0 + (pisqu * log((-1.0 + xa2) / (2.0 * xa))) / 3.0 + power_of<3>(log((-1.0 + xa2) / (2.0 * xa))) / 3.0 - (pisqu * log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0 - power_of<3>(log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 - ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 - ya))) / 6.0 - power_of<3>(ln1mya) / 6.0 + dilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) * log((1.0 - ya) / (1.0 + xa))
                        + dilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) * log((1.0 - ya) / (1.0 + xa)) - (pisqu * log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 + (pisqu * log((-1.0 + xa) / (-1.0 + ya))) / 6.0
                        + power_of<3>(log((-1.0 + xa) / (-1.0 + ya))) / 6.0 + dilog((-1.0 + ya) / (1.0 + ya)) * (-log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa))) + dilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) * log((-1.0 + ya) / (-1.0 + xa))
                        + dilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) * log((-1.0 + ya) / (-1.0 + xa)) - (power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * log(2.0 / (1.0 + ya))) / 2.0 + (pisqu * log((1.0 - xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 - xa) / (1.0 + ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 + ya))) / 6.0 - power_of<3>(ln1pya) / 6.0 + dilog(1.0 / (1.0 - xa)) * (ln1mya + ln1pya) + dilog(1.0 / (1.0 + xa)) * (ln1mya + ln1pya)
                        + dilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 - xa)) + dilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) * log((1.0 + ya) / (1.0 - xa))
                        + dilog((1.0 + ya) / (-1.0 + ya)) * (-log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa))) + dilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) * log((1.0 + ya) / (1.0 + xa))
                        + dilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 + xa)) - (log(-2.0 / (-1.0 + ya)) * power_of<2>(log((1.0 + ya) / (1.0 + xa)))) / 2.0 - (pisqu * log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0
                        - power_of<3>(log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0 - (pisqu * log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0
                        + li2half * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 4.0 * log(xa) + 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + 2.0 * ln1mya + 2.0 * ln1pya - 2.0 * log((xa + ya) * xainv) - 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 - ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa)) - 2.0 * ln1pya - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + dilog((1.0 + ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - 2.0 * ln1pya - log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa)) - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + power_of<2>(ln1pya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((1.0 - ya) / 2.0) / 2.0 - log((xa - ya) / (1.0 + xa)) / 2.0 - log((xa + ya) / (-1.0 + xa)) / 2.0 + log((xa + ya) * xainv) / 2.0
                            + log(1.0 - ya * xainv) / 2.0) + power_of<2>(ln1mya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((xa - ya) / (-1.0 + xa)) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log((xa + ya) * xainv) / 2.0
                            - log((xa + ya) / (1.0 + xa)) / 2.0 + log(1.0 - ya * xainv) / 2.0) + dilog((1.0 - xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 + xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv)) + log((1.0 - ya) / 2.0) * ((-0.5) * power_of<2>(log((1.0 + ya) / (1.0 - xa))) - power_of<2>(log((1.0 + ya) / (1.0 + xa))) / 2.0
                            - log((1.0 + ya) / (1.0 - xa)) * log((xa + ya) * xainv) - log((1.0 + ya) / (1.0 + xa)) * log(1.0 - ya * xainv)) - (2.0 * 1.0i) * M_PI * H1((1.0 + xa) / (-1.0 + xa), 1.0 + xainv) * power_of<2>(log(-1.0 / (1.0 + xa))) * my_sign(-imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) * xainv, 1.0 + xainv) * power_of<2>(log((2.0 * xa) / (-1.0 + xa2))) * my_sign(-imag(xainv)) - (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) * xainv) * power_of<2>(log(1.0 / (-1.0 + xa))) * my_sign(imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1(1.0 + xainv, (-1.0 + xa) * xainv) * power_of<2>(log((-2.0 * xa) / (-1.0 + xa2))) * my_sign(imag(xainv)) + 1.0i * M_PI * H1((1.0 + xa) / (xa - ya), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((-2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((1.0 + ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa - ya))) + 1.0i * M_PI * H1((-1.0 + xa) / (xa - ya), (1.0 + xa) / (xa - ya)) * power_of<2>(log((2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa - ya)) * power_of<2>(log((-1.0 + ya) / (1.0 + xa))) * my_sign(imag((1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((1.0 - ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa + ya))) + 1.0i * M_PI * H1((1.0 + xa) / (xa + ya), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((-2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa + ya)))
                        - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa + ya)) * power_of<2>(log(-((1.0 + ya) / (1.0 + xa)))) * my_sign(imag((1.0 + xa) / (xa + ya)))
                        + 1.0i * M_PI * H1((-1.0 + xa) / (xa + ya), (1.0 + xa) / (xa + ya)) * power_of<2>(log((2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa + ya)))
                        + power_of<2>(log(1.0 / (1.0 + xa))) * (lnhalf + ln2 - log((2.0 * xa) / (1.0 + xa)) - log((1.0 + xa) / (2.0 * xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xa / (1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), 1.0 / (1.0 + xa)))
                        + power_of<2>(log(1.0 / (1.0 - xa))) * (lnhalf + ln2 - log((-1.0 + xa) / (2.0 * xa)) - log((2.0 * xa) / (-1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xa / (1.0 - xa))) * T(1.0, 1.0 / (1.0 + xa), 1.0 / (1.0 - xa)))
                        + ln1mya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0 + (2.0 * ln2 - log((1.0 - ya) / 2.0)) * ln1pya - log((1.0 + ya) / 2.0) * ln1pya
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1pxa))
                        + power_of<2>(log((1.0 + ya) / (1.0 - xa))) * ((-0.5) * log(-2.0 / (-1.0 + ya)) + log((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) / 2.0 + log((-2.0 * (xa + ya)) / ((-1.0 + xa) * (-1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (-1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), (1.0 + ya) / (1.0 - xa))) + power_of<2>(log((1.0 + ya) / (1.0 + xa))) * (log((-2.0 * (xa - ya)) / ((1.0 + xa) * (-1.0 + ya))) / 2.0
                            + log((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) / 2.0 - 1.0i * M_PI * my_sign(imag((xa - ya) / (1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), (1.0 + ya) / (1.0 + xa)))
                        + power_of<2>(ln1mxa) * ((-1.0i) * M_PI * my_sign(imxa) * t11mya1mxa - 1.0i * M_PI * my_sign(imxa) * t11pya1mxa) + power_of<2>(ln1pxa) * ((-1.0i) * M_PI * my_sign(-imxa) * t11mya1pxa
                            - 1.0i * M_PI * my_sign(-imxa) * t11pya1pxa) + ln1pya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1pxa))
                        + power_of<2>(log(-2.0 / (-1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), -2.0 / (-1.0 + xa)) + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa))) + log((xa + ya) * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0
                            - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa + ya))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa + ya))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa)))
                        + power_of<2>(log((1.0 - ya) / (1.0 + xa))) * ((-0.5) * log(2.0 / (1.0 + ya)) - log((1.0 + ya) / 2.0) / 2.0 + log(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) / 2.0 + log((2.0 * (xa + ya)) / ((1.0 + xa) * (1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), (1.0 - ya) / (1.0 + xa))) + power_of<2>(log(2.0 / (1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, 1.0 / (1.0 + xa), 2.0 / (1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + log(1.0 - ya * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0 - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa - ya))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
                            - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa - ya))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * (log((2.0 * (xa - ya)) / ((-1.0 + xa) * (1.0 + ya))) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa - ya) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), (-1.0 + ya) / (-1.0 + xa))) + log((1.0 + ya) / (1.0 - xa)) * ((ln1mxa - ln1pxa) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)) + log((1.0 - ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log((xa + ya) * xainv) - log((1.0 + ya) / 2.0) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)) + dilog(xa / (-1.0 + xa)) * (-2.0 * ln1mxa - 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + log((1.0 - ya) / (1.0 + xa))
                            + log((1.0 + ya) / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + dilog((1.0 + xa) / (-1.0 + xa)) * (2.0 * log(1.0 / (1.0 + xa)) - log((1.0 - ya) / (1.0 + xa)) - log((1.0 + ya) / (1.0 + xa)) + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + log((-1.0 + ya) / (-1.0 + xa)) * ((ln1mxa - ln1pxa) * log(1.0 - ya * xainv) - log((1.0 + ya) / 2.0) * log(1.0 - ya * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + dilog(xa / (1.0 + xa)) * (-2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 2.0 * ln1pxa + log((-1.0 + ya) / (-1.0 + xa))
                            + log((1.0 + ya) / (1.0 - xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + dilog((-1.0 + xa) / (1.0 + xa)) * (2.0 * log(1.0 / (1.0 - xa)) - log((-1.0 + ya) / (-1.0 + xa)) - log((1.0 + ya) / (1.0 - xa)) + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)) + log((1.0 + ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log(1.0 - ya * xainv) + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                            + ln1pxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) + (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + ln1mxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + log(1.0 + xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, 1.0 + xainv) * T(p(-xainv, (xa + ya) * xainv), (xa + ya) * xainv, (-1.0 + xa) * xainv) + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, 1.0 + xainv) * T(p(-xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                            + log((-1.0 + xa) * xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) * T(p(xainv, (xa + ya) * xainv), (xa + ya) * xainv, 1.0 + xainv)
                            + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv) * T(p(xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, 1.0 + xainv)) - (21.0 * zeta3) / 4.0);

                return result;
            }
        }

        complex<double> f29a(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xa = clp.xa;
            const complex<double> ya = clp.ya;

            const complex<double> xa2 = power_of<2>(xa);
            const complex<double> xa3 = power_of<3>(xa);
            const complex<double> xa4 = power_of<4>(xa);

            const complex<double> ya2 = power_of<2>(ya);
            const complex<double> ya3 = power_of<3>(ya);
            const complex<double> ya4 = power_of<4>(ya);
            const complex<double> ya5 = power_of<5>(ya);
            const complex<double> ya6 = power_of<6>(ya);
            const complex<double> ya7 = power_of<7>(ya);
            const complex<double> ya8 = power_of<8>(ya);

            const complex<double> xainv = 1.0 / xa;
            const complex<double> yainv = 1.0 / ya;

            const double imxa = imag(xa);
            const double imyainv = imag(yainv);

            const complex<double> ln1pxa  = log(1.0 + xa);
            const complex<double> ln1mxa  = log(1.0 - xa);
            const complex<double> ln1pya  = log(1.0 + ya);
            const complex<double> ln1mya  = log(1.0 - ya);

            const complex<double> num1 = 4.0 * ((-ya2) * (3.0 - 22.0 * ya2 + 15.0 * ya4 + 4.0 * ya5) + xa2 * (9.0 - 42.0 * ya2 + 41.0 * ya4 - 12.0 * ya6 + 4.0 * ya7));
            const complex<double> num2 = -3.0 * ya4 * (-1.0 + ya2) + 3.0 * xa2 * (ya2 - 4.0 * ya4 + 3.0 * ya6) + xa4 * (18.0 + 15.0 * ya2 - 81.0 * ya4 + 16.0 * ya6);
            const complex<double> num3 = (-1.0 + ya) * (1.0 + ya) * (-ya4 + xa2 * ya2 * (-1.0 + 3.0 * ya2) + xa4 * (-6.0 - 11.0 * ya2 + 16.0 * ya4));
            const complex<double> num4 = 4.0 * (3.0 * ya2 - 22.0 * ya4 + 15.0 * ya6 - 4.0 * ya7 + xa2 * (-9.0 + 42.0 * ya2 - 41.0 * ya4 + 12.0 * ya6 + 4.0 * ya7));
            const complex<double> num5 = 2.0 * power_of<2>(-1.0 + ya2) * (ya2 + 5.0 * ya4 + xa2 * (-3.0 - 7.0 * ya2 + 4.0 * ya4));

            const complex<double> denom1 = 81.0 * (-1.0 + xa2) * ya7;
            const complex<double> denom2 = 81.0 * xa4 * ya6;
            const complex<double> denom3 = 27.0 * xa4 * ya6;
            const complex<double> denom4 = 27.0 * (-1.0 + xa2) * ya8;
            const complex<double> denom5 = 81.0 * xa3 * (1.0 + xa) * ya6;
            const complex<double> denom6 = 81.0 * (-1.0 + xa2) * ya8;
            const complex<double> denom7 = 81.0 * (-1.0 + xa) * xa3 * ya6;
            const complex<double> denom8 = 81.0 * (-1.0 + xa2) * ya6;
            const complex<double> denom9 = 243.0 * (-1.0 + xa2) * ya6;
            const complex<double> denom10 = 243.0 * (-1.0 + xa2) * ya7;

            const complex<double> logs1 = 2.0 * (-3.0 * ya4 * (-1.0 + ya2) - 3.0 * xa * ya4 * (-1.0 + ya2) + xa2 * (3.0 * ya2 + 53.0 * ya4 - 56.0 * ya6)
                + 2.0 * xa4 * (9.0 - 87.0 * ya2 + 96.0 * ya4 + ya6 * (-18.0 - 8.0 * 1.0i * M_PI - 8.0 * ln2))
                + xa3 * ya2 * (9.0 - 53.0 * ya2 + 4.0 * ya4 * (11.0 - 4.0 * 1.0i * M_PI - 4.0 * ln2)));
            const complex<double> logs2 = 4.0 * (3.0 * 1.0i * M_PI * power_of<2>(-1.0 + ya2) * (ya2 + 5.0 * ya4 + xa2 * (-3.0 - 7.0 * ya2 + 4.0 * ya4))
                + ya2 * (3.0 * ya - 22.0 * ya3 + 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (-9.0 * ya + 42.0 * ya3 - 41.0 * ya5 + 12.0 * ya7 - 9.0 * ln2 + 45.0 * ya4 * ln2 - 45.0 * ya6 * ln2 + 4.0 * ya8 * (-1.0 + log(8.0))
                    - ya2 * log(8.0)));
            const complex<double> logs3 = 2.0 * (-3.0 * ya4 * (-1.0 + ya2) + 3.0 * xa * ya4 * (-1.0 + ya2) + xa2 * (3.0 * ya2 + 53.0 * ya4 - 56.0 * ya6)
                + 2.0 * xa4 * (9.0 - 87.0 * ya2 + 96.0 * ya4 + ya6 * (-18.0 - 8.0 * 1.0i * M_PI - 8.0 * ln2))
                + xa3 * ya2 * (-9.0 + 53.0 * ya2 + 4.0 * ya4 * (-11.0 + 4.0 * 1.0i * M_PI + log(16.0))));
            const complex<double> logs4 = 8.0 * (ya2 * (3.0 - 53.0 * ya2 + ya4 * (50.0 - 8.0 * 1.0i * M_PI - 8.0 * ln2))
                + xa2 * (-9.0 + 87.0 * ya2 - 96.0 * ya4 + 2.0 * ya6 * (9.0 + 4.0 * 1.0i * M_PI + log(16.0))));
            const complex<double> logs5 = 4.0 * (9.0 * ya2 - 159.0 * ya4 - 8.0 * ya6 * (-14.0 + log(8.0)) + xa2 * (-27.0 + 261.0 * ya2 - 288.0 * ya4 + 4.0 * ya6 * (23.0 + ln64)));
            const complex<double> logs6 = 4.0 * (-6.0 * 1.0i * M_PI * (ya2 * (3.0 - 22.0 * ya2 + 15.0 * ya4) + xa2 * (-9.0 + 42.0 * ya2 - 41.0 * ya4 + 12.0 * ya6))
                + ya2 * (9.0 * ya - 159.0 * ya3 - 18.0 * ln2 + 132.0 * ya2 * ln2 - 90.0 * ya4 * ln2 - 8.0 * ya5 * (-14.0 + log(8.0)))
                + xa2 * (-27.0 * ya + 261.0 * ya3 - 288.0 * ya5 + 54.0 * ln2 - 252.0 * ya2 * ln2 + 246.0 * ya4 * ln2 - 72.0 * ya6 * ln2
                    + 4.0 * ya7 * (23.0 + ln64)));
            const complex<double> logs7 = 4.0 * (6.0 * 1.0i * M_PI * (ya2 * (3.0 - 22.0 * ya2 + 15.0 * ya4) + xa2 * (-9.0 + 42.0 * ya2 - 41.0 * ya4 + 12.0 * ya6))
                + ya2 * (9.0 * ya - 159.0 * ya3 + 18.0 * ln2 - 132.0 * ya2 * ln2 + 90.0 * ya4 * ln2 - 8.0 * ya5 * (-14.0 + log(8.0)))
                + xa2 * (-27.0 * ya + 261.0 * ya3 - 288.0 * ya5 - 54.0 * ln2 + 252.0 * ya2 * ln2 - 246.0 * ya4 * ln2 + 72.0 * ya6 * ln2
                    + 4.0 * ya7 * (23.0 + ln64)));
            const complex<double> logs8 = 4.0 * (-3.0 * 1.0i * M_PI * power_of<2>(-1.0 + ya2) * (ya2 + 5.0 * ya4 + xa2 * (-3.0 - 7.0 * ya2 + 4.0 * ya4))
                - ya2 * (3.0 * ya - 22.0 * ya3 + 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (-4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (9.0 * ya - 42.0 * ya3 + 41.0 * ya5 - 12.0 * ya7 - 45.0 * ya4 * ln2 + 45.0 * ya6 * ln2 + ya2 * log(8.0) - 4.0 * ya8 * (1.0 + log(8.0))
                    + log(512.0)));
            const complex<double> logs9 = 4.0 * (-3.0 * 1.0i * M_PI * power_of<2>(-1.0 + ya2) * (ya2 + 5.0 * ya4 + xa2 * (-3.0 - 7.0 * ya2 + 4.0 * ya4))
                - ya2 * (-3.0 * ya + 22.0 * ya3 - 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (-4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (-9.0 * ya + 42.0 * ya3 - 41.0 * ya5 + 12.0 * ya7 - 45.0 * ya4 * ln2 + 45.0 * ya6 * ln2 + ya2 * log(8.0) - 4.0 * ya8 * (1.0 + log(8.0))
                    + log(512.0)));
            const complex<double> logs10 = 4.0 * (3.0 * 1.0i * M_PI * power_of<2>(-1.0 + ya2) * (ya2 + 5.0 * ya4 + xa2 * (-3.0 - 7.0 * ya2 + 4.0 * ya4))
                + ya2 * (-3.0 * ya + 22.0 * ya3 - 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (9.0 * ya - 42.0 * ya3 + 41.0 * ya5 - 12.0 * ya7 - 9.0 * ln2 + 45.0 * ya4 * ln2 - 45.0 * ya6 * ln2 + 4.0 * ya8 * (-1.0 + log(8.0))
                    - ya2 * log(8.0)));

            const complex<double> logsp1 = 4.0 * (ya2 * (-3.0 * ya + 22.0 * ya3 - 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (9.0 * ya - 42.0 * ya3 + 41.0 * ya5 - 12.0 * ya7 - 9.0 * ln2 + 45.0 * ya4 * ln2 - 45.0 * ya6 * ln2 + 4.0 * ya8 * (-1.0 + log(8.0))
                    - ya2 * log(8.0)));
            const complex<double> logsp2 = 4.0 * (ya2 * (3.0 * ya - 22.0 * ya3 + 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (-9.0 * ya + 42.0 * ya3 - 41.0 * ya5 + 12.0 * ya7 - 9.0 * ln2 + 45.0 * ya4 * ln2 - 45.0 * ya6 * ln2 + 4.0 * ya8 * (-1.0 + log(8.0))
                    - ya2 * log(8.0)));
            const complex<double> logsp3 = 4.0 * (ya2 * (-3.0 * ya + 22.0 * ya3 - 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (-4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (9.0 * ya - 42.0 * ya3 + 41.0 * ya5 - 12.0 * ya7 - 9.0 * ln2 + 45.0 * ya4 * ln2 - 45.0 * ya6 * ln2 - ya2 * log(8.0)
                    + 4.0 * ya8 * (1.0 + log(8.0))));
            const complex<double> logsp4 = 4.0 * (ya2 * (3.0 * ya - 22.0 * ya3 + 15.0 * ya5 + 9.0 * ya2 * ln2 - 27.0 * ya4 * ln2 + ya6 * (-4.0 + 15.0 * ln2) + log(8.0))
                + xa2 * (-9.0 * ya + 42.0 * ya3 - 41.0 * ya5 + 12.0 * ya7 - 9.0 * ln2 + 45.0 * ya4 * ln2 - 45.0 * ya6 * ln2 - ya2 * log(8.0)
                  + 4.0 * ya8 * (1.0 + log(8.0))));
            const complex<double> logsp5 = 4.0 * (xa2 * (27.0 * ya - 261.0 * ya3 + 288.0 * ya5 + ya7 * (-92.0 - 24.0 * 1.0i * M_PI - 24.0 * ln2) + 54.0 * ln2 - 252.0 * ya2 * ln2
                + 246.0 * ya4 * ln2 - 72.0 * ya6 * ln2) + ya2 * (-9.0 * ya + 159.0 * ya3 - 18.0 * ln2 + 132.0 * ya2 * ln2 - 90.0 * ya4 * ln2
                + 8.0 * ya5 * (-14.0 + 3.0 * 1.0i * M_PI + log(8.0))));
            const complex<double> logsp6 = 4.0 * (xa2 * (27.0 * ya - 261.0 * ya3 + 288.0 * ya5 + ya7 * (-92.0 - 24.0 * 1.0i * M_PI - 24.0 * ln2) - 54.0 * ln2 + 252.0 * ya2 * ln2
                - 246.0 * ya4 * ln2 + 72.0 * ya6 * ln2) + ya2 * (-9.0 * ya + 159.0 * ya3 + 18.0 * ln2 - 132.0 * ya2 * ln2 + 90.0 * ya4 * ln2
                + 8.0 * ya5 * (-14.0 + 3.0 * 1.0i * M_PI + log(8.0))));
            const complex<double> logsp7 = 8.0 * (3.0 * ya2 - 53.0 * ya4 + ya6 * (50.0 - 8.0 * ln2) + xa2 * (-9.0 + 87.0 * ya2 - 96.0 * ya4 + 2.0 * ya6 * (9.0 + log(16.0))));
            const complex<double> logsp8 = 2.0 * (3.0 * ya4 * (-1.0 + ya2) - 3.0 * xa * ya4 * (-1.0 + ya2)
                + xa2 * ya2 * (-3.0 - 53.0 * ya2 + 56.0 * ya4) + xa3 * (9.0 * ya2 - 53.0 * ya4 + ya6 * (44.0 - 16.0 * ln2))
                + 2.0 * xa4 * (-9.0 + 87.0 * ya2 - 96.0 * ya4 + 2.0 * ya6 * (9.0 + log(16.0))));
            const complex<double> logsp9 = 2.0 * (3.0 * ya4 * (-1.0 + ya2) + 3.0 * xa * ya4 * (-1.0 + ya2) + xa2 * ya2 * (-3.0 - 53.0 * ya2 + 56.0 * ya4)
                + xa3 * ya2 * (-9.0 + 53.0 * ya2 + 4.0 * ya4 * (-11.0 + log(16.0))) + 2.0 * xa4 * (-9.0 + 87.0 * ya2 - 96.0 * ya4 + 2.0 * ya6 * (9.0 + log(16.0))));
            const complex<double> logsp10 = 4.0 * (ya2 * (9.0 - 159.0 * ya2 + 8.0 * ya4 * (14.0 - 3.0 * 1.0i * M_PI - log(8.0)))
                + xa2 * (-27.0 + 261.0 * ya2 - 288.0 * ya4 + 4.0 * ya6 * (23.0 + 6.0 * 1.0i * M_PI + ln64)));

            const complex<double> termp1 = (1.0 / (729.0 * xa2 * (-1.0 + xa2) * ya6)) * 2.0 * (27.0 * ya4 * (-1.0 + ya2) + xa4 * (-324.0 * ln2 + 27.0 * ya2 * (3.0 + 116.0 * ln2)
            	- 9.0 * ya4 * (115.0 + 384.0 * ln2) + ya6 * (491.0 - 228.0 * 1.0i * M_PI + 78.0 * pisqu + 648.0 * ln2 + 144.0 * ln2squ))
                + xa2 * ya2 * (-18.0 * ya2 * (-59.0 + 106.0 * ln2) - 2.0 * ya4 * (259.0 - 114.0 * 1.0i * M_PI + 39.0 * pisqu - 900.0 * ln2 + 72.0 * ln2squ)
                    + 27.0 * (-3.0 + log(16.0))));

            const complex<double> term1 = 2.0 * ((66.0 * pisqu * xa2 * (-1.0 + xa2) * ya6 - 27.0 * ya4 * (-1.0 + ya2)
                + xa2 * ya2 * (81.0 - 108.0 * ln2 + 18.0 * ya2 * (-59.0 + 106.0 * ln2) + 2.0 * ya4 * (259.0 - 900.0 * ln2 + 72.0 * ln2squ))
                - xa4 * (-324.0 * ln2 + 27.0 * ya2 * (3.0 + 116.0 * ln2) - 9.0 * ya4 * (115.0 + 384.0 * ln2)
                    + ya6 * (491.0 + 648.0 * ln2 + 144.0 * ln2squ)) - 12.0 * 1.0i * M_PI * xa2 * (9.0 * ya2 - 159.0 * ya4 + ya6 * (131.0 - 24.0 * ln2)
                    + xa2 * (-27.0 + 261.0 * ya2 - 288.0 * ya4 + ya6 * (73.0 + 24.0 * ln2)))) / (729.0 * xa2 * (-1.0 + xa2) * ya6));

            const double factor1 = 16.0 / 81;
            const double factor2 = 16.0 / 243;

            // Heaviside theta related functions
            const double t11mya1mxa = T(1.0, 1.0 - ya, 1.0 - xa); // MyT[1, 1 - ya, 1 - xa]
            const double t11pya1mxa = T(1.0, 1.0 + ya, 1.0 - xa); // MyT[1, 1 + ya, 1 - xa]
            const double t11mya1pxa = T(1.0, 1.0 - ya, 1.0 + xa); // MyT[1, 1 - ya, 1 + xa]
            const double t11pya1pxa = T(1.0, 1.0 + ya, 1.0 + xa); // MyT[1, 1 + ya, 1 + xa]

            complex<double> result;

            if (imag(clp.s_eps) > 0)
            {
                // F29aupper
                result = termp1 - (64.0 * power_of<2>(lnmuhat)) / 81.0 - (logsp9 * ln1mxa) / denom5 + (logsp7 * log(xa)) / denom8 - (logsp8 * ln1pxa) / denom7
                    + (-2.0 * num2 * li2half + num2 * dilog((1.0 - xa) / 2.0) + num2 * dilog((1.0 + xa) / 2.0) - num2 * ln2 * ln1mxa - num2 * ln2 * ln1pxa) / denom2 + ((num3 * power_of<2>(ln1mxa)) / 2.0 + (num3 * power_of<2>(ln1pxa)) / 2.0) / denom3
                    + (logsp6 * ln1mya + logsp5 * ln1pya) / denom10 + ((-logsp3 - logsp4) * li2half + logsp4 * dilog((1.0 - ya) / 2.0) + logsp3 * dilog((1.0 + ya) / 2.0) - logsp4 * ln2 * ln1mya + (logsp1 * power_of<2>(ln1mya)) / 2.0
                        - logsp3 * ln2 * ln1pya + (logsp2 * power_of<2>(ln1pya)) / 2.0) / denom6 + (logsp10 * log((xa + ya) * xainv) + logsp10 * log(1.0 - ya * xainv)) / denom9
                    + factor2 * (-19.0 * lnmuhat - (12.0 * 1.0i) * M_PI * lnmuhat - 12.0 * lnmuhat * ln1mya - 12.0 * lnmuhat * ln1pya + 12.0 * lnmuhat * log((xa + ya) * xainv) + 12.0 * lnmuhat * log(1.0 - ya * xainv))
                    + factor1 * (2.0 * li2half - 2.0 * dilog(1.0 / (1.0 - xa)) - 2.0 * dilog(1.0 / (1.0 + xa)) + dilog((1.0 - ya) / (1.0 + xa)) - dilog((xa - ya) / (2.0 * xa)) + dilog((-1.0 + ya) / (-1.0 + xa)) + dilog((1.0 + ya) / (1.0 - xa))
                        + dilog((1.0 + ya) / (1.0 + xa)) - dilog((xa + ya) / (2.0 * xa)) + 2.0 * power_of<2>(log(xa)) - 2.0 * log(xa) * ln1pxa + log((xa - ya) / (1.0 + xa)) * ln1pya + ln1pya * log((xa + ya) / (-1.0 + xa))
                        + (2.0 * log(xa) - log((xa - ya) / (2.0 * xa))) * log((xa + ya) * xainv) - power_of<2>(log((xa + ya) * xainv)) / 2.0 + ln1mya * (log((xa - ya) / (-1.0 + xa)) + log((xa + ya) / (1.0 + xa)))
                        + (2.0 * log(xa) - log((xa + ya) / (2.0 * xa))) * log(1.0 - ya * xainv) - power_of<2>(log(1.0 - ya * xainv)) / 2.0 + ln1mxa * (-2.0 * log(xa) + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * t11mya1mxa
                        + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * t11pya1mxa) + ln1pxa * ((2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * t11mya1pxa + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * t11pya1pxa))
                    + ((-num1 - num4) * dilog(xa / (-1.0 + xa)) + (-num1 - num4) * dilog(xa / (1.0 + xa)) + num4 * dilog((xa - ya) / (-1.0 + xa)) + num1 * dilog((xa - ya) / (1.0 + xa)) + num1 * dilog((xa + ya) / (-1.0 + xa))
                        + num4 * dilog((xa + ya) / (1.0 + xa)) + (num1 * ln1mxa - 2.0 * num1 * log(xa) + num1 * ln1pxa) * ln1mya + (num4 * ln1mxa - 2.0 * num4 * log(xa) + num4 * ln1pxa) * ln1pya
                        + (-(num4 * ln1mxa) - num1 * ln1pxa + num4 * log((1.0 - ya) / (1.0 + xa)) + num1 * log((1.0 + ya) / (1.0 - xa))) * log((xa + ya) * xainv)
                        + (-(num1 * ln1mxa) - num4 * ln1pxa + num4 * log((-1.0 + ya) / (-1.0 + xa)) + num1 * log((1.0 + ya) / (1.0 + xa))) * log(1.0 - ya * xainv)
                        + log(1.0 + xainv) * ((2.0 * 1.0i) * num4 * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) + (2.0 * 1.0i) * num1 * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + log((-1.0 + xa) * xainv) * ((2.0 * 1.0i) * num1 * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * num4 * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))) / denom1
                    + num5 * ((pisqu * ln4) / 12.0 - 4.0 * trilog(-1.0) - 2.0 * li3half + 2.0 * trilog(1.0 / (1.0 - xa)) - 2.0 * trilog(1.0 - xa) + 2.0 * trilog((-1.0 + xa) / (2.0 * xa)) + 2.0 * trilog(1.0 / (1.0 + xa)) + 2.0 * trilog((1.0 - xa) / (1.0 + xa))
                        - 2.0 * trilog(1.0 + xa) + 2.0 * trilog((1.0 + xa) / (1.0 - xa)) + 2.0 * trilog((1.0 + xa) / (2.0 * xa)) + trilog((1.0 + xa) / (1.0 - ya)) + 4.0 * trilog((1.0 - ya) / 2.0) - trilog((1.0 - ya) / (1.0 + xa)) + trilog((-1.0 + xa) / (-1.0 + ya))
                        - trilog((-1.0 + ya) / (-1.0 + xa)) - trilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) + trilog((1.0 - xa) / (1.0 + ya)) + trilog((1.0 + xa) / (1.0 + ya)) + 2.0 * trilog((-1.0 + ya) / (1.0 + ya))
                        - trilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) - trilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) + 4.0 * trilog((1.0 + ya) / 2.0) - trilog((1.0 + ya) / (1.0 - xa)) - trilog((1.0 + ya) / (1.0 + xa))
                        - trilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) + 2.0 * trilog((1.0 + ya) / (-1.0 + ya)) - trilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya)))
                        - trilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) + (pisqu * ln2) / 3.0 - power_of<3>(ln2) - 2.0 * dilog((1.0 + xa) / (1.0 - xa)) * log(1.0 / (1.0 - xa))
                        - 2.0 * dilog((1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 - xa)) - power_of<3>(ln1mxa) / 3.0 - 2.0 * dilog((-1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 + xa)) - 2.0 * dilog((1.0 - xa) / (1.0 + xa)) * log(1.0 / (1.0 + xa))
                        + dilog(-1.0) * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * log(1.0 / (1.0 + xa))) + dilog((xa - ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa) + dilog((xa + ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa)
                        - power_of<3>(ln1pxa) / 3.0 + dilog((xa - ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + dilog((xa + ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + (pisqu * log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0
                        + power_of<3>(log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0 + (pisqu * log((-1.0 + xa2) / (2.0 * xa))) / 3.0 + power_of<3>(log((-1.0 + xa2) / (2.0 * xa))) / 3.0 - (pisqu * log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0 - power_of<3>(log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 - ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 - ya))) / 6.0 - power_of<3>(ln1mya) / 6.0 + dilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) * log((1.0 - ya) / (1.0 + xa))
                        + dilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) * log((1.0 - ya) / (1.0 + xa)) - (pisqu * log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 + (pisqu * log((-1.0 + xa) / (-1.0 + ya))) / 6.0
                        + power_of<3>(log((-1.0 + xa) / (-1.0 + ya))) / 6.0 + dilog((-1.0 + ya) / (1.0 + ya)) * (-log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa))) + dilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) * log((-1.0 + ya) / (-1.0 + xa))
                        + dilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) * log((-1.0 + ya) / (-1.0 + xa)) - (power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * log(2.0 / (1.0 + ya))) / 2.0 + (pisqu * log((1.0 - xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 - xa) / (1.0 + ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 + ya))) / 6.0 - power_of<3>(ln1pya) / 6.0 + dilog(1.0 / (1.0 - xa)) * (ln1mya + ln1pya) + dilog(1.0 / (1.0 + xa)) * (ln1mya + ln1pya)
                        + dilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 - xa)) + dilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) * log((1.0 + ya) / (1.0 - xa))
                        + dilog((1.0 + ya) / (-1.0 + ya)) * (-log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa))) + dilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) * log((1.0 + ya) / (1.0 + xa))
                        + dilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 + xa)) - (log(-2.0 / (-1.0 + ya)) * power_of<2>(log((1.0 + ya) / (1.0 + xa)))) / 2.0 - (pisqu * log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0
                        - power_of<3>(log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0 - (pisqu * log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0
                        + li2half * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 4.0 * log(xa) + 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + 2.0 * ln1mya + 2.0 * ln1pya - 2.0 * log((xa + ya) * xainv) - 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 - ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa)) - 2.0 * ln1pya - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + dilog((1.0 + ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - 2.0 * ln1pya - log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa)) - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + power_of<2>(ln1pya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((1.0 - ya) / 2.0) / 2.0 - log((xa - ya) / (1.0 + xa)) / 2.0 - log((xa + ya) / (-1.0 + xa)) / 2.0 + log((xa + ya) * xainv) / 2.0
                            + log(1.0 - ya * xainv) / 2.0) + power_of<2>(ln1mya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((xa - ya) / (-1.0 + xa)) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log((xa + ya) * xainv) / 2.0
                            - log((xa + ya) / (1.0 + xa)) / 2.0 + log(1.0 - ya * xainv) / 2.0) + dilog((1.0 - xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 + xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv)) + log((1.0 - ya) / 2.0) * ((-0.5) * power_of<2>(log((1.0 + ya) / (1.0 - xa))) - power_of<2>(log((1.0 + ya) / (1.0 + xa))) / 2.0
                            - log((1.0 + ya) / (1.0 - xa)) * log((xa + ya) * xainv) - log((1.0 + ya) / (1.0 + xa)) * log(1.0 - ya * xainv)) - (2.0 * 1.0i) * M_PI * H1((1.0 + xa) / (-1.0 + xa), 1.0 + xainv) * power_of<2>(log(-1.0 / (1.0 + xa))) * my_sign(-imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) * xainv, 1.0 + xainv) * power_of<2>(log((2.0 * xa) / (-1.0 + xa2))) * my_sign(-imag(xainv)) - (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) * xainv) * power_of<2>(log(1.0 / (-1.0 + xa))) * my_sign(imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1(1.0 + xainv, (-1.0 + xa) * xainv) * power_of<2>(log((-2.0 * xa) / (-1.0 + xa2))) * my_sign(imag(xainv)) + 1.0i * M_PI * H1((1.0 + xa) / (xa - ya), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((-2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((1.0 + ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa - ya))) + 1.0i * M_PI * H1((-1.0 + xa) / (xa - ya), (1.0 + xa) / (xa - ya)) * power_of<2>(log((2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa - ya)) * power_of<2>(log((-1.0 + ya) / (1.0 + xa))) * my_sign(imag((1.0 + xa) / (xa - ya))) - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((1.0 - ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa + ya)))
                        + 1.0i * M_PI * H1((1.0 + xa) / (xa + ya), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((-2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa + ya)))
                        - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa + ya)) * power_of<2>(log(-((1.0 + ya) / (1.0 + xa)))) * my_sign(imag((1.0 + xa) / (xa + ya))) + 1.0i * M_PI * H1((-1.0 + xa) / (xa + ya), (1.0 + xa) / (xa + ya)) * power_of<2>(log((2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa + ya)))
                        + power_of<2>(log(1.0 / (1.0 + xa))) * (lnhalf + ln2 - log((2.0 * xa) / (1.0 + xa)) - log((1.0 + xa) / (2.0 * xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xa / (1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), 1.0 / (1.0 + xa)))
                        + power_of<2>(log(1.0 / (1.0 - xa))) * (lnhalf + ln2 - log((-1.0 + xa) / (2.0 * xa)) - log((2.0 * xa) / (-1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xa / (1.0 - xa))) * T(1.0, 1.0 / (1.0 + xa), 1.0 / (1.0 - xa)))
                        + ln1mya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0 + (2.0 * ln2 - log((1.0 - ya) / 2.0)) * ln1pya - log((1.0 + ya) / 2.0) * ln1pya
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1pxa))
                        + power_of<2>(log((1.0 + ya) / (1.0 - xa))) * ((-0.5) * log(-2.0 / (-1.0 + ya)) + log((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) / 2.0 + log((-2.0 * (xa + ya)) / ((-1.0 + xa) * (-1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (-1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), (1.0 + ya) / (1.0 - xa))) + power_of<2>(log((1.0 + ya) / (1.0 + xa))) * (log((-2.0 * (xa - ya)) / ((1.0 + xa) * (-1.0 + ya))) / 2.0
                            + log((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) / 2.0 - 1.0i * M_PI * my_sign(imag((xa - ya) / (1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), (1.0 + ya) / (1.0 + xa)))
                        + power_of<2>(ln1mxa) * ((-1.0i) * M_PI * my_sign(imxa) * t11mya1mxa - 1.0i * M_PI * my_sign(imxa) * t11pya1mxa) + power_of<2>(ln1pxa) * ((-1.0i) * M_PI * my_sign(-imxa) * t11mya1pxa
                            - 1.0i * M_PI * my_sign(-imxa) * t11pya1pxa) + ln1pya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1pxa))
                        + power_of<2>(log(-2.0 / (-1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), -2.0 / (-1.0 + xa)) + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa))) + log((xa + ya) * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0
                            - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa + ya))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa + ya))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa)))
                        + power_of<2>(log((1.0 - ya) / (1.0 + xa))) * ((-0.5) * log(2.0 / (1.0 + ya)) - log((1.0 + ya) / 2.0) / 2.0 + log(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) / 2.0 + log((2.0 * (xa + ya)) / ((1.0 + xa) * (1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), (1.0 - ya) / (1.0 + xa))) + power_of<2>(log(2.0 / (1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, 1.0 / (1.0 + xa), 2.0 / (1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + log(1.0 - ya * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0 - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa - ya))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
                        - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa - ya))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * (log((2.0 * (xa - ya)) / ((-1.0 + xa) * (1.0 + ya))) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa - ya) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), (-1.0 + ya) / (-1.0 + xa))) + log((1.0 + ya) / (1.0 - xa)) * ((ln1mxa - ln1pxa) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)) + log((1.0 - ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log((xa + ya) * xainv) - log((1.0 + ya) / 2.0) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)) + dilog(xa / (-1.0 + xa)) * (-2.0 * ln1mxa - 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + log((1.0 - ya) / (1.0 + xa))
                            + log((1.0 + ya) / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + dilog((1.0 + xa) / (-1.0 + xa)) * (2.0 * log(1.0 / (1.0 + xa)) - log((1.0 - ya) / (1.0 + xa)) - log((1.0 + ya) / (1.0 + xa)) + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + log((-1.0 + ya) / (-1.0 + xa)) * ((ln1mxa - ln1pxa) * log(1.0 - ya * xainv) - log((1.0 + ya) / 2.0) * log(1.0 - ya * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + dilog(xa / (1.0 + xa)) * (-2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 2.0 * ln1pxa + log((-1.0 + ya) / (-1.0 + xa))
                            + log((1.0 + ya) / (1.0 - xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + dilog((-1.0 + xa) / (1.0 + xa)) * (2.0 * log(1.0 / (1.0 - xa)) - log((-1.0 + ya) / (-1.0 + xa)) - log((1.0 + ya) / (1.0 - xa)) + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)) + log((1.0 + ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log(1.0 - ya * xainv) + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + ln1pxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) + (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + ln1mxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + log(1.0 + xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, 1.0 + xainv) * T(p(-xainv, (xa + ya) * xainv), (xa + ya) * xainv, (-1.0 + xa) * xainv) + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, 1.0 + xainv) * T(p(-xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + log((-1.0 + xa) * xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) * T(p(xainv, (xa + ya) * xainv), (xa + ya) * xainv, 1.0 + xainv)
                            + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv) * T(p(xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, 1.0 + xainv)) - (21.0 * zeta3) / 4.0) / denom4;

                return result;
            }
            else
            {
                // F29alower
                result = -term1 - (64.0 * power_of<2>(lnmuhat)) / 81.0 + (logs1 * ln1mxa) / denom5 + (logs4 * log(xa)) / denom8 + (logs3 * ln1pxa) / denom7
                    + (-2.0 * num2 * li2half + num2 * dilog((1.0 - xa) / 2.0) + num2 * dilog((1.0 + xa) / 2.0) - num2 * ln2 * ln1mxa - num2 * ln2 * ln1pxa) / denom2 + ((num3 * power_of<2>(ln1mxa)) / 2.0 + (num3 * power_of<2>(ln1pxa)) / 2.0) / denom3
                    + (-(logs6 * ln1mya) - logs7 * ln1pya) / denom10 + ((logs8 + logs9) * li2half - logs8 * dilog((1.0 - ya) / 2.0) - logs9 * dilog((1.0 + ya) / 2.0) + logs8 * ln2 * ln1mya + (logs10 * power_of<2>(ln1mya)) / 2.0
                        + logs9 * ln2 * ln1pya + (logs2 * power_of<2>(ln1pya)) / 2.0) / denom6 + (logs5 * log((xa + ya) * xainv) + logs5 * log(1.0 - ya * xainv)) / denom9
                    + factor2 * (-19.0 * lnmuhat + (12.0 * 1.0i) * M_PI * lnmuhat - 12.0 * lnmuhat * ln1mya - 12.0 * lnmuhat * ln1pya + 12.0 * lnmuhat * log((xa + ya) * xainv) + 12.0 * lnmuhat * log(1.0 - ya * xainv))
                    + factor1 * (2.0 * li2half - 2.0 * dilog(1.0 / (1.0 - xa)) - 2.0 * dilog(1.0 / (1.0 + xa)) + dilog((1.0 - ya) / (1.0 + xa)) - dilog((xa - ya) / (2.0 * xa)) + dilog((-1.0 + ya) / (-1.0 + xa)) + dilog((1.0 + ya) / (1.0 - xa))
                        + dilog((1.0 + ya) / (1.0 + xa)) - dilog((xa + ya) / (2.0 * xa)) + 2.0 * power_of<2>(log(xa)) - 2.0 * log(xa) * ln1pxa + log((xa - ya) / (1.0 + xa)) * ln1pya + ln1pya * log((xa + ya) / (-1.0 + xa))
                        + (2.0 * log(xa) - log((xa - ya) / (2.0 * xa))) * log((xa + ya) * xainv) - power_of<2>(log((xa + ya) * xainv)) / 2.0 + ln1mya * (log((xa - ya) / (-1.0 + xa)) + log((xa + ya) / (1.0 + xa)))
                        + (2.0 * log(xa) - log((xa + ya) / (2.0 * xa))) * log(1.0 - ya * xainv) - power_of<2>(log(1.0 - ya * xainv)) / 2.0 + ln1mxa * (-2.0 * log(xa) + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * t11mya1mxa
                            + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * t11pya1mxa) + ln1pxa * ((2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * t11mya1pxa + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * t11pya1pxa))
                    + ((-num1 - num4) * dilog(xa / (-1.0 + xa)) + (-num1 - num4) * dilog(xa / (1.0 + xa)) + num4 * dilog((xa - ya) / (-1.0 + xa)) + num1 * dilog((xa - ya) / (1.0 + xa)) + num1 * dilog((xa + ya) / (-1.0 + xa))
                        + num4 * dilog((xa + ya) / (1.0 + xa)) + (num1 * ln1mxa - 2.0 * num1 * log(xa) + num1 * ln1pxa) * ln1mya + (num4 * ln1mxa - 2.0 * num4 * log(xa) + num4 * ln1pxa) * ln1pya
                        + (-(num4 * ln1mxa) - num1 * ln1pxa + num4 * log((1.0 - ya) / (1.0 + xa)) + num1 * log((1.0 + ya) / (1.0 - xa))) * log((xa + ya) * xainv)
                        + (-(num1 * ln1mxa) - num4 * ln1pxa + num4 * log((-1.0 + ya) / (-1.0 + xa)) + num1 * log((1.0 + ya) / (1.0 + xa))) * log(1.0 - ya * xainv)
                        + log(1.0 + xainv) * ((2.0 * 1.0i) * num4 * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) + (2.0 * 1.0i) * num1 * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + log((-1.0 + xa) * xainv) * ((2.0 * 1.0i) * num1 * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * num4 * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))) / denom1
                    + (num5 * ((pisqu * ln4) / 12.0 - 4.0 * trilog(-1.0) - 2.0 * li3half + 2.0 * trilog(1.0 / (1.0 - xa)) - 2.0 * trilog(1.0 - xa) + 2.0 * trilog((-1.0 + xa) / (2.0 * xa)) + 2.0 * trilog(1.0 / (1.0 + xa)) + 2.0 * trilog((1.0 - xa) / (1.0 + xa))
                        - 2.0 * trilog(1.0 + xa) + 2.0 * trilog((1.0 + xa) / (1.0 - xa)) + 2.0 * trilog((1.0 + xa) / (2.0 * xa)) + trilog((1.0 + xa) / (1.0 - ya)) + 4.0 * trilog((1.0 - ya) / 2.0) - trilog((1.0 - ya) / (1.0 + xa)) + trilog((-1.0 + xa) / (-1.0 + ya))
                        - trilog((-1.0 + ya) / (-1.0 + xa)) - trilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) + trilog((1.0 - xa) / (1.0 + ya)) + trilog((1.0 + xa) / (1.0 + ya)) + 2.0 * trilog((-1.0 + ya) / (1.0 + ya))
                        - trilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) - trilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) + 4.0 * trilog((1.0 + ya) / 2.0) - trilog((1.0 + ya) / (1.0 - xa)) - trilog((1.0 + ya) / (1.0 + xa))
                        - trilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) + 2.0 * trilog((1.0 + ya) / (-1.0 + ya)) - trilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya)))
                        - trilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) - trilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) + (pisqu * ln2) / 3.0 - power_of<3>(ln2) - 2.0 * dilog((1.0 + xa) / (1.0 - xa)) * log(1.0 / (1.0 - xa))
                        - 2.0 * dilog((1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 - xa)) - power_of<3>(ln1mxa) / 3.0 - 2.0 * dilog((-1.0 + xa) / (2.0 * xa)) * log(1.0 / (1.0 + xa)) - 2.0 * dilog((1.0 - xa) / (1.0 + xa)) * log(1.0 / (1.0 + xa))
                        + dilog(-1.0) * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * log(1.0 / (1.0 + xa))) + dilog((xa - ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa) + dilog((xa + ya) / (-1.0 + xa)) * (ln1mxa - ln1pxa)
                        - power_of<3>(ln1pxa) / 3.0 + dilog((xa - ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + dilog((xa + ya) / (1.0 + xa)) * (-ln1mxa + ln1pxa) + (pisqu * log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0
                        + power_of<3>(log((-0.5) * ((-1.0 + xa2) * xainv))) / 3.0 + (pisqu * log((-1.0 + xa2) / (2.0 * xa))) / 3.0 + power_of<3>(log((-1.0 + xa2) / (2.0 * xa))) / 3.0 - (pisqu * log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0 - power_of<3>(log((1.0 - xa2) / (2.0 * xa - 2.0 * ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 - ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 - ya))) / 6.0 - power_of<3>(ln1mya) / 6.0 + dilog(((-1.0 + xa) * (-1.0 + ya)) / ((1.0 + xa) * (1.0 + ya))) * log((1.0 - ya) / (1.0 + xa))
                        + dilog((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) * log((1.0 - ya) / (1.0 + xa)) - (pisqu * log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa - ya)))) / 6.0 + (pisqu * log((-1.0 + xa) / (-1.0 + ya))) / 6.0
                        + power_of<3>(log((-1.0 + xa) / (-1.0 + ya))) / 6.0 + dilog((-1.0 + ya) / (1.0 + ya)) * (-log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa))) + dilog((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) * log((-1.0 + ya) / (-1.0 + xa))
                        + dilog(((1.0 + xa) * (-1.0 + ya)) / ((-1.0 + xa) * (1.0 + ya))) * log((-1.0 + ya) / (-1.0 + xa)) - (power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * log(2.0 / (1.0 + ya))) / 2.0 + (pisqu * log((1.0 - xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 - xa) / (1.0 + ya))) / 6.0
                        + (pisqu * log((1.0 + xa) / (1.0 + ya))) / 6.0 + power_of<3>(log((1.0 + xa) / (1.0 + ya))) / 6.0 - power_of<3>(ln1pya) / 6.0 + dilog(1.0 / (1.0 - xa)) * (ln1mya + ln1pya) + dilog(1.0 / (1.0 + xa)) * (ln1mya + ln1pya)
                        + dilog(((1.0 + xa) * (1.0 + ya)) / ((-1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 - xa)) + dilog(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) * log((1.0 + ya) / (1.0 - xa))
                        + dilog((1.0 + ya) / (-1.0 + ya)) * (-log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa))) + dilog(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) * log((1.0 + ya) / (1.0 + xa))
                        + dilog(((-1.0 + xa) * (1.0 + ya)) / ((1.0 + xa) * (-1.0 + ya))) * log((1.0 + ya) / (1.0 + xa)) - (log(-2.0 / (-1.0 + ya)) * power_of<2>(log((1.0 + ya) / (1.0 + xa)))) / 2.0 - (pisqu * log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0
                        - power_of<3>(log((-0.5) * (((-1.0 + xa) * (1.0 + xa)) / (xa + ya)))) / 6.0 - (pisqu * log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0 - power_of<3>(log((-1.0 + xa2) / (2.0 * (xa + ya)))) / 6.0
                        + li2half * (2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 4.0 * log(xa) + 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + 2.0 * ln1mya + 2.0 * ln1pya - 2.0 * log((xa + ya) * xainv) - 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 - ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - log((1.0 - ya) / (1.0 + xa)) - log((-1.0 + ya) / (-1.0 + xa)) - 2.0 * ln1pya - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + dilog((1.0 + ya) / 2.0) * (-ln1mxa + 2.0 * log(xa) - ln1pxa - 2.0 * ln1mya - 2.0 * ln1pya - log((1.0 + ya) / (1.0 - xa)) - log((1.0 + ya) / (1.0 + xa)) - log((xa + ya) * xainv) - log(1.0 - ya * xainv))
                        + power_of<2>(ln1pya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((1.0 - ya) / 2.0) / 2.0 - log((xa - ya) / (1.0 + xa)) / 2.0 - log((xa + ya) / (-1.0 + xa)) / 2.0 + log((xa + ya) * xainv) / 2.0 + log(1.0 - ya * xainv) / 2.0)
                        + power_of<2>(ln1mya) * (ln64 / 12.0 - ln1mxa / 2.0 + log(xa) - ln1pxa / 2.0 - log((xa - ya) / (-1.0 + xa)) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log((xa + ya) * xainv) / 2.0
                            - log((xa + ya) / (1.0 + xa)) / 2.0 + log(1.0 - ya * xainv) / 2.0) + dilog((1.0 - xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv))
                        + dilog((1.0 + xa) / 2.0) * (-ln1mya - ln1pya + 2.0 * log((xa + ya) * xainv) + 2.0 * log(1.0 - ya * xainv)) + log((1.0 - ya) / 2.0) * ((-0.5) * power_of<2>(log((1.0 + ya) / (1.0 - xa))) - power_of<2>(log((1.0 + ya) / (1.0 + xa))) / 2.0
                            - log((1.0 + ya) / (1.0 - xa)) * log((xa + ya) * xainv) - log((1.0 + ya) / (1.0 + xa)) * log(1.0 - ya * xainv)) - (2.0 * 1.0i) * M_PI * H1((1.0 + xa) / (-1.0 + xa), 1.0 + xainv) * power_of<2>(log(-1.0 / (1.0 + xa))) * my_sign(-imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) * xainv, 1.0 + xainv) * power_of<2>(log((2.0 * xa) / (-1.0 + xa2))) * my_sign(-imag(xainv)) - (2.0 * 1.0i) * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) * xainv) * power_of<2>(log(1.0 / (-1.0 + xa))) * my_sign(imag(xainv))
                        + (2.0 * 1.0i) * M_PI * H1(1.0 + xainv, (-1.0 + xa) * xainv) * power_of<2>(log((-2.0 * xa) / (-1.0 + xa2))) * my_sign(imag(xainv)) + 1.0i * M_PI * H1((1.0 + xa) / (xa - ya), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((-2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa - ya)) * power_of<2>(log((1.0 + ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa - ya))) + 1.0i * M_PI * H1((-1.0 + xa) / (xa - ya), (1.0 + xa) / (xa - ya)) * power_of<2>(log((2.0 * (xa - ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa - ya)))
                        - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa - ya)) * power_of<2>(log((-1.0 + ya) / (1.0 + xa))) * my_sign(imag((1.0 + xa) / (xa - ya))) - 1.0i * M_PI * H1((-1.0 + xa) / (1.0 + xa), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((1.0 - ya) / (-1.0 + xa))) * my_sign(imag((-1.0 + xa) / (xa + ya)))
                        + 1.0i * M_PI * H1((1.0 + xa) / (xa + ya), (-1.0 + xa) / (xa + ya)) * power_of<2>(log((-2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((-1.0 + xa) / (xa + ya)))
                        - 1.0i * M_PI * H1((1.0 + xa) / (-1.0 + xa), (1.0 + xa) / (xa + ya)) * power_of<2>(log(-((1.0 + ya) / (1.0 + xa)))) * my_sign(imag((1.0 + xa) / (xa + ya))) + 1.0i * M_PI * H1((-1.0 + xa) / (xa + ya), (1.0 + xa) / (xa + ya)) * power_of<2>(log((2.0 * (xa + ya)) / (-1.0 + xa2))) * my_sign(imag((1.0 + xa) / (xa + ya)))
                        + power_of<2>(log(1.0 / (1.0 + xa))) * (lnhalf + ln2 - log((2.0 * xa) / (1.0 + xa)) - log((1.0 + xa) / (2.0 * xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xa / (1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), 1.0 / (1.0 + xa)))
                        + power_of<2>(log(1.0 / (1.0 - xa))) * (lnhalf + ln2 - log((-1.0 + xa) / (2.0 * xa)) - log((2.0 * xa) / (-1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xa / (1.0 - xa))) * T(1.0, 1.0 / (1.0 + xa), 1.0 / (1.0 - xa)))
                        + ln1mya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0 + (2.0 * ln2 - log((1.0 - ya) / 2.0)) * ln1pya - log((1.0 + ya) / 2.0) * ln1pya
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(imyainv) * t11mya1pxa))
                        + power_of<2>(log((1.0 + ya) / (1.0 - xa))) * ((-0.5) * log(-2.0 / (-1.0 + ya)) + log((-0.5) * (((-1.0 + xa) * (-1.0 + ya)) / (xa + ya))) / 2.0 + log((-2.0 * (xa + ya)) / ((-1.0 + xa) * (-1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (-1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), (1.0 + ya) / (1.0 - xa))) + power_of<2>(log((1.0 + ya) / (1.0 + xa))) * (log((-2.0 * (xa - ya)) / ((1.0 + xa) * (-1.0 + ya))) / 2.0
                            + log((-0.5) * (((1.0 + xa) * (-1.0 + ya)) / (xa - ya))) / 2.0 - 1.0i * M_PI * my_sign(imag((xa - ya) / (1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), (1.0 + ya) / (1.0 + xa)))
                        + power_of<2>(ln1mxa) * ((-1.0i) * M_PI * my_sign(imxa) * t11mya1mxa - 1.0i * M_PI * my_sign(imxa) * t11pya1mxa) + power_of<2>(ln1pxa) * ((-1.0i) * M_PI * my_sign(-imxa) * t11mya1pxa
                            - 1.0i * M_PI * my_sign(-imxa) * t11pya1pxa) + ln1pya * (pisqu / 6.0 - ln2squ / 2.0 - ln2squ / 2.0 - power_of<2>(ln1mxa) / 2.0 - 2.0 * ln2 * log(xa) - power_of<2>(ln1pxa) / 2.0
                            + ln1mxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1mxa) + ln1pxa * (2.0 * ln2 - (2.0 * 1.0i) * M_PI * my_sign(-imyainv) * t11pya1pxa))
                        + power_of<2>(log(-2.0 / (-1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, 1.0 / (1.0 - xa), -2.0 / (-1.0 + xa)) + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((1.0 + xa) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa))) + log((xa + ya) * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0
                            - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa + ya))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa + ya))) * T(1.0, (1.0 + ya) / (1.0 - xa), -2.0 / (-1.0 + xa)))
                        + power_of<2>(log((1.0 - ya) / (1.0 + xa))) * ((-0.5) * log(2.0 / (1.0 + ya)) - log((1.0 + ya) / 2.0) / 2.0 + log(((1.0 + xa) * (1.0 + ya)) / (2.0 * (xa + ya))) / 2.0 + log((2.0 * (xa + ya)) / ((1.0 + xa) * (1.0 + ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa + ya) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 - xa), (1.0 - ya) / (1.0 + xa))) + power_of<2>(log(2.0 / (1.0 + xa))) * ((-2.0 * 1.0i) * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, 1.0 / (1.0 + xa), 2.0 / (1.0 + xa))
                            + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 - ya) / (1.0 + xa), 2.0 / (1.0 + xa)) + 1.0i * M_PI * my_sign(imag((-1.0 + xa) / (1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + log(1.0 - ya * xainv) * (-(ln2 * ln1mxa) + power_of<2>(ln1mxa) / 2.0 - ln2 * ln1pxa + power_of<2>(ln1pxa) / 2.0 - (2.0 * 1.0i) * M_PI * log(-2.0 / (-1.0 + xa)) * my_sign(imag((-1.0 + xa) / (xa - ya))) * T(1.0, (-1.0 + ya) / (-1.0 + xa), -2.0 / (-1.0 + xa))
                        - (2.0 * 1.0i) * M_PI * log(2.0 / (1.0 + xa)) * my_sign(imag((1.0 + xa) / (xa - ya))) * T(1.0, (1.0 + ya) / (1.0 + xa), 2.0 / (1.0 + xa)))
                        + power_of<2>(log((-1.0 + ya) / (-1.0 + xa))) * (log((2.0 * (xa - ya)) / ((-1.0 + xa) * (1.0 + ya))) / 2.0 - log((1.0 + ya) / 2.0) / 2.0 + log(((-1.0 + xa) * (1.0 + ya)) / (2.0 * (xa - ya))) / 2.0
                            - 1.0i * M_PI * my_sign(imag((xa - ya) / (-1.0 + xa))) * T(1.0, (1.0 + ya) / (1.0 + xa), (-1.0 + ya) / (-1.0 + xa))) + log((1.0 + ya) / (1.0 - xa)) * ((ln1mxa - ln1pxa) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)) + log((1.0 - ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log((xa + ya) * xainv) - log((1.0 + ya) / 2.0) * log((xa + ya) * xainv)
                            + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)) + dilog(xa / (-1.0 + xa)) * (-2.0 * ln1mxa - 2.0 * log(1.0 / (1.0 + xa)) + 2.0 * ln1pxa + log((1.0 - ya) / (1.0 + xa))
                            + log((1.0 + ya) / (1.0 + xa)) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                        + dilog((1.0 + xa) / (-1.0 + xa)) * (2.0 * log(1.0 / (1.0 + xa)) - log((1.0 - ya) / (1.0 + xa)) - log((1.0 + ya) / (1.0 + xa)) + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, (xa + ya) * xainv, 1.0 + xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + log((-1.0 + ya) / (-1.0 + xa)) * ((ln1mxa - ln1pxa) * log(1.0 - ya * xainv) - log((1.0 + ya) / 2.0) * log(1.0 - ya * xainv)
                            + (2.0 * 1.0i) * M_PI * log(1.0 + xainv) * my_sign(-imag(xainv)) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv)) + dilog(xa / (1.0 + xa)) * (-2.0 * log(1.0 / (1.0 - xa)) + 2.0 * ln1mxa - 2.0 * ln1pxa + log((-1.0 + ya) / (-1.0 + xa))
                            + log((1.0 + ya) / (1.0 - xa)) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + dilog((-1.0 + xa) / (1.0 + xa)) * (2.0 * log(1.0 / (1.0 - xa)) - log((-1.0 + ya) / (-1.0 + xa)) - log((1.0 + ya) / (1.0 - xa)) + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv)
                            + (2.0 * 1.0i) * M_PI * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)) + log((1.0 + ya) / (1.0 + xa)) * ((-ln1mxa + ln1pxa) * log(1.0 - ya * xainv) + (2.0 * 1.0i) * M_PI * log((-1.0 + xa) * xainv) * my_sign(imag(xainv)) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + ln1pxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) + (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) - (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + ln1mxa * ((-1.0 / 3.0) * pisqu + log(1.0 + xainv) * ((-2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, (xa + ya) * xainv, 1.0 + xainv) - (2.0 * 1.0i) * M_PI * my_sign(-imxa) * T(1.0, 1.0 - ya * xainv, 1.0 + xainv))
                            + log((-1.0 + xa) * xainv) * ((2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) + (2.0 * 1.0i) * M_PI * my_sign(imxa) * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv)))
                        + log(1.0 + xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, 1.0 + xainv) * T(p(-xainv, (xa + ya) * xainv), (xa + ya) * xainv, (-1.0 + xa) * xainv) + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, 1.0 + xainv) * T(p(-xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, (-1.0 + xa) * xainv))
                        + log((-1.0 + xa) * xainv) * (4.0 * pisqu * T(1.0, (xa + ya) * xainv, (-1.0 + xa) * xainv) * T(p(xainv, (xa + ya) * xainv), (xa + ya) * xainv, 1.0 + xainv)
                        + 4.0 * pisqu * T(1.0, 1.0 - ya * xainv, (-1.0 + xa) * xainv) * T(p(xainv, 1.0 - ya * xainv), 1.0 - ya * xainv, 1.0 + xainv)) - (21.0 * zeta3) / 4.0)) / denom4;

                return result;
            }
        }

        complex<double> f17c(const CharmLoopsParameters & clp)
        {
            // f17c = - f27c / (2 * N_c)
            return - f27c(clp) / 6.0;
        }

        complex<double> f19c(const CharmLoopsParameters & clp)
        {
            // f19c = - f29c / (2 * N_c)
            return - f29c(clp) / 6.0;
        }

        complex<double> f27c(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xc = clp.xc;
            const complex<double> yc = clp.yc;
            const complex<double> w3 = (2.0 * xc * xc) / (1.0 + xc * xc);

            const complex<double> xcinv = 1.0 / xc;
            const complex<double> ycinv = 1.0 / yc;
            const complex<double> w3inv = 1.0 / w3;

            // Imaginary parts
            const double imxc = imag(xc);
            const double imw3 = imag(w3);
            const double imycinv = imag(ycinv);

            // Polynomials in xc and yc
            const complex<double> xc2 = power_of<2>(xc);
            const complex<double> xc3 = power_of<3>(xc);
            const complex<double> xc4 = power_of<4>(xc);
            const complex<double> xc6 = power_of<6>(xc);

            const complex<double> yc2 = power_of<2>(yc);
            const complex<double> yc3 = power_of<3>(yc);
            const complex<double> yc4 = power_of<4>(yc);
            const complex<double> yc5 = power_of<5>(yc);
            const complex<double> yc6 = power_of<6>(yc);

            const complex<double> factorp0 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc);
            const complex<double> factorp1 = power_of<2>(-1.0 + xc) * power_of<2>(1.0 + xc) * power_of<2>(-1.0 + yc) * power_of<2>(1.0 + yc) * (-2.0 * yc2 + xc2 * (-1.0 + 3.0 * yc2));
            const complex<double> termp1 = (2.0 * (3.0 + yc2 * (-18.0 - (4.0 * 1.0i) * M_PI - 8.0 * ln2) + xc2 * (12.0 + (4.0 * 1.0i) * M_PI + 3.0 * yc2 + ln256))) / (9.0 * (xc - yc) * (xc + yc));

            const complex<double> num0 = (1.0 + xc) * (2.0 * xc3 + 2.0 * yc4 + xc2 * (-3.0 + 4.0 * yc2 - 3.0 * yc4) + xc * (3.0 - 8.0 * yc2 + 3.0 * yc4));
            const complex<double> num1 = (-1.0 + xc) * (2.0 * xc3 - 2.0 * yc4 + xc * (3.0 - 8.0 * yc2 + 3.0 * yc4) + xc2 * (3.0 - 4.0 * yc2 + 3.0 * yc4));
            const complex<double> num2 = (-1.0 + xc) * (1.0 + xc) * (3.0 * yc2 - 7.0 * yc4 + xc2 * (3.0 - 5.0 * yc2 + 6.0 * yc4));
            const complex<double> num3 = (-1.0 + xc) * (1.0 + xc) * power_of<2>(1.0 + yc) * (yc3 * (-1.0 - 2.0 * yc + 13.0 * yc2) + xc2 * yc * (1.0 - 6.0 * yc - 4.0 * yc2 + 14.0 * yc3 + 15.0 * yc4) + xc4 * (2.0 - 3.0 * yc - 2.0 * yc2 - 27.0 * yc3 - 6.0 * yc4 + 6.0 * yc5));
            const complex<double> num4 = (-1.0 + xc) * (1.0 + xc) * yc * (3.0 * xc6 * (11.0 - 6.0 * yc2 + yc4) + 2.0 * (yc2 + 2.0 * yc4) + 2.0 * xc4 * (-5.0 - 13.0 * yc2 + 3.0 * yc4) + xc2 * (1.0 - 6.0 * yc2 + 11.0 * yc4));
            const complex<double> num5 = (-1.0 + xc) * (1.0 + xc) * (yc2 * (3.0 + yc4) + xc2 * (yc2 + 3.0 * yc6) + xc6 * (-4.0 + 51.0 * yc2 - 44.0 * yc4 + 9.0 * yc6) + xc4 * (-4.0 - 15.0 * yc2 - 12.0 * yc4 + 11.0 * yc6));
            const complex<double> num6 = (-1.0 + xc) * (1.0 + xc) * power_of<2>(-1.0 + yc) * (yc3 * (-1.0 + 2.0 * yc + 13.0 * yc2) + xc2 * yc * (1.0 + 6.0 * yc - 4.0 * yc2 - 14.0 * yc3 + 15.0 * yc4) + xc4 * (-2.0 - 3.0 * yc + 2.0 * yc2 - 27.0 * yc3 + 6.0 * yc4 + 6.0 * yc5));
            const complex<double> num7 = (-1.0 + xc) * (1.0 + xc) * yc * (2.0 * yc2 + 2.0 * xc4 * (-5.0 - yc2 + yc4) + xc6 * (17.0 - 14.0 * yc2 + 3.0 * yc4) + xc2 * (1.0 - 2.0 * yc2 + 3.0 * yc4));
            const complex<double> num8 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc) * (xc2 * (-1.0 + 38.0 * yc2 - 5.0 * yc4) + xc4 * (-15.0 - 4.0 * yc2 + 3.0 * yc4) - 2.0 * (yc2 + 7.0 * yc4));
            const complex<double> num9 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc) * (2.0 * yc2 * (-1.0 + 5.0 * yc2) + xc4 * (9.0 - 4.0 * yc2 + 3.0 * yc4) - xc2 * (1.0 + 10.0 * yc2 + 5.0 * yc4));
            const complex<double> num10 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc) * (-2.0 * yc2 + 6.0 * yc4 + xc4 * (5.0 - 4.0 * yc2 + 3.0 * yc4) - xc2 * (1.0 + 2.0 * yc2 + 5.0 * yc4));

            const complex<double> denom0 = (9.0 * power_of<2>(xc - yc) * power_of<2>(xc + yc)); // can b e replaced by denom1
            const complex<double> denom1 = (9.0 * power_of<2>(xc - yc) * power_of<2>(xc + yc));
            const complex<double> denom2 = (9.0 * power_of<2>(xc - yc) * yc * power_of<2>(xc + yc));
            const complex<double> denom3 = (9.0 * power_of<2>(xc - yc) * yc2 * power_of<2>(xc + yc) * (yc + xc2 * (2.0 + yc)));
            const complex<double> denom4 = (9.0 * power_of<2>(xc - yc) * power_of<2>(xc + yc) * (xc2 * (-2.0 + yc) + yc) * (yc + xc2 * (2.0 + yc)));
            const complex<double> denom5 = (9.0 * power_of<2>(xc - yc) * yc2 * power_of<2>(xc + yc) * (xc2 * (-2.0 + yc) + yc));
            const complex<double> denom6 = (9.0 * power_of<3>(xc - yc) * power_of<3>(xc + yc));
            const complex<double> denom7 = (9.0 * (xc - yc) * (xc + yc));

            // Polylogarithms in xc and yc
            const complex<double> ln1pxc = log(1.0 + xc);
            const complex<double> ln1mxc = log(1.0 - xc);
            const complex<double> ln1pyc = log(1.0 + yc);
            const complex<double> ln1myc = log(1.0 - yc);
            const complex<double> ln1pxchalf = log((1.0 + xc) / 2.0); // MyLog[(1 + xc)/2]
            const complex<double> ln1mxchalf = log((1.0 - xc) / 2.0); // MyLog[(1 - xc)/2]
            const complex<double> ln1pychalf = log((1.0 + yc) / 2.0); // MyLog[(1 + yc)/2]
            const complex<double> ln1mychalf = log((1.0 - yc) / 2.0); // MyLog[(1 - yc)/2]

            const complex<double> lnr_xc1 = log(xc / (1.0 + xc));
            const complex<double> lnr_xc2 = log(xc / (-1.0 + xc));

            const complex<double> li2r_xc = dilog(1.0 / (1.0 + xc));

            // Heaviside theta related functions
            const double t11myc1mxc = T(1.0, 1.0 - yc, 1.0 - xc); // MyT[1, 1 - yc, 1 - xc]
            const double t11pyc1mxc = T(1.0, 1.0 + yc, 1.0 - xc); // MyT[1, 1 + yc, 1 - xc]
            const double t11myc1pxc = T(1.0, 1.0 - yc, 1.0 + xc); // MyT[1, 1 - yc, 1 + xc]
            const double t11pyc1pxc = T(1.0, 1.0 + yc, 1.0 + xc); // MyT[1, 1 + yc, 1 + xc]

            const complex<double> f27cPart1 = termp1 + (-(num3 / denom3) + (4.0 * num5) / denom4 - num6 / denom5) * dilog(1.0 / 2.0)
                + (num6 * dilog((1.0 - yc) / 2.0)) / denom5 + (num3 * dilog((1.0 + yc) / 2.0)) / denom3 + (16.0 * lnmuhat) / 9.0 - (4.0 * num1 * ln1mxc) / denom1 + (16.0 * log(xc)) / 9.0 - (4.0 * num0 * ln1pxc) / denom0
                - (num6 * ln2 * ln1myc) / denom5 + (num6 * power_of<2>(ln1myc)) / (2.0 * denom5) - (num3 * ln2 * ln1pyc) / denom3 + (num3 * power_of<2>(ln1pyc)) / (2.0 * denom3) - (2.0 * num2 * (-ln1myc + ln1pyc)) / denom2
                + ( - (2.0 * num5 * dilog((1.0 + xc) / 2.0)) - (4.0 * num7 * li2r_xc) + (4.0 * num7 * dilog(1.0 / (1.0 - xc))) - (2.0 * num5 * dilog((1.0 - xc) / 2.0)) - (4.0 * num7 * dilog(-1.0 / (-1.0 + xc)))
                + (4.0 * num7 * li2r_xc) - (4.0 * num7 * dilog((-1.0 + yc) / (-1.0 - xc))) - (4.0 * num7 * dilog((-1.0 + yc) / (-1.0 + xc))) + (4.0 * num4 * dilog(-yc)) - (4.0 * num4 * dilog(yc))
                + (4.0 * num7 * dilog((1.0 + yc) / (1.0 - xc))) + (4.0 * num7 * dilog((1.0 + yc) / (1.0 + xc))) + (2.0 * num5 * ln2 * ln1mxc) - (num5 * power_of<2>(ln1mxc)) + (2.0 * num5 * ln2 * ln1pxc) - (num5 * power_of<2>(ln1pxc))
                + (4.0 * num7 * log((-xc - yc) / (1.0 - xc)) * ln1pyc) + (4.0 * num7 * log((xc - yc) / (1.0 + xc)) * ln1pyc) - (4.0 * num7 * ln1myc * log((-xc - yc) / (-1.0 - xc)))
                - (4.0 * num7 * ln1myc * log((xc - yc) / (-1.0 + xc))) + (4.0 * num7 * ln1mxc * log(1.0 - yc / xc)) - (4.0 * num7 * ln1pxc * log(1.0 - yc / xc))
                - (4.0 * num7 * ln1mxc * log(1.0 + yc / xc)) + (4.0 * num7 * ln1pxc * log(1.0 + yc / xc)) - ((8.0 * 1.0i) * num7 * M_PI * ln1mxc * my_sign(imag(xcinv)) * t11myc1mxc)
                - ((8.0 * 1.0i) * num7 * M_PI * ln1pxc * my_sign(-imag(xcinv)) * t11myc1pxc) + ((8.0 * 1.0i) * num7 * M_PI * ln1mxc * my_sign(imag(xcinv)) * t11pyc1mxc)
                + ((8.0 * 1.0i) * num7 * M_PI * ln1pxc * my_sign(-imag(xcinv)) * t11pyc1pxc)) / denom4;

            const complex<double> f27cPart2 = (2.0 * num8 * trilog(1.0 / 2.0) + (-3.0 * num8 - 4.0 * num9) * trilog((1.0 - yc) / 2.0) - 2.0 * num9 * trilog(1.0 - yc) - 2.0 * num9 * trilog(-yc) - 2.0 * num9 * trilog(yc)
                - 2.0 * num9 * trilog(yc / (-1.0 + yc)) + 4.0 * num9 * trilog((2.0 * yc) / (-1.0 + yc)) - 2.0 * num9 * trilog(yc / (1.0 + yc)) + 4.0 * num9 * trilog((2.0 * yc) / (1.0 + yc)) + (-3.0 * num8 - 4.0 * num9) * trilog((1.0 + yc) / 2.0)
                - 2.0 * num9 * trilog(1.0 + yc) - (num8 * pisqu * ln2) / 3.0 + (2.0 * num8 * power_of<3>(ln2)) / 3.0 - (num8 * pisqu * ln1myc) / 12.0 + 4.0 * num9 * dilog(1.0 - yc) * ln1myc
                + (num8 * ln2squ * ln1myc) / 2.0 + 2.0 * num9 * power_of<2>(ln1myc) * log(yc) + (3.0 * num8 * power_of<2>(ln1myc) * ln1pychalf) / 2.0 - (num8 * pisqu * ln1pyc) / 12.0
                + 4.0 * num9 * dilog(1.0 + yc) * ln1pyc + (num8 * ln2squ * ln1pyc) / 2.0 - 2.0 * num8 * ln2 * ln1myc * ln1pyc + (3.0 * num8 * ln1mychalf * power_of<2>(ln1pyc)) / 2.0
                + 2.0 * num9 * log(-yc) * power_of<2>(ln1pyc) + dilog((1.0 - yc) / 2.0) * (3.0 * num8 * ln1myc + num8 * ln1pyc) + dilog((1.0 + yc) / 2.0) * (num8 * ln1myc + 3.0 * num8 * ln1pyc)
                + dilog(-yc) * (2.0 * num9 * ln1myc + 2.0 * num9 * ln1pyc) + dilog(yc) * (2.0 * num9 * ln1myc + 2.0 * num9 * ln1pyc)
                - (num9 * (-4.0 * power_of<3>(ln2) + pisqu * ln4 - ln1myc * (pisqu - 6.0 * ln2squ + ln64 * ln1myc) + (pisqu + 6.0 * ln2squ) * ln1pyc + 2.0 * (-3.0 * log(-yc) + log((1.0 + yc) / 8.0)) * power_of<2>(ln1pyc))) / 6.0
                - (num9 * (-4.0 * power_of<3>(ln2) + pisqu * ln4 + (pisqu + 6.0 * ln2squ) * ln1myc + 2.0 * power_of<2>(ln1myc) * (log((1.0 - yc) / 8.0) - 3.0 * log(yc)) - ln1pyc * (pisqu - 6.0 * ln2squ + ln64 * ln1pyc))) / 6.0
                + (7.0 * num8 * zeta3) / 2.0 + 11.0 * num9 * zeta3) / denom6;

            const complex<double> f27cPart3 = 2.0 * num10 * (((pisqu * ln4) / 12.0 - 6.0 * li3half + 4.0 * trilog(1.0 - xcinv) + 4.0 * trilog(1.0 + xcinv) + 2.0 * trilog(-1.0 / (-1.0 - xc)) + 2.0 * trilog(1.0 / (1.0 - xc)) + 2.0 * trilog(1.0 - xc) + 2.0 * trilog(-1.0 / (-1.0 + xc)) + 4.0 * trilog(-xcinv) + 4.0 * trilog(xcinv) - 4.0 * trilog((-0.5) * ((1.0 - xc) * xcinv)) + 2.0 * trilog(-xc) + 2.0 * trilog(xc) + 2.0 * trilog(xc / (-1.0 + xc)) - 2.0 * trilog((2.0 * xc) / (-1.0 + xc)) + 2.0 * trilog(1.0 / (1.0 + xc)) - 4.0 * trilog((1.0 - xc) / (1.0 + xc)) + 2.0 * trilog(xc / (1.0 + xc)) - 2.0 * trilog((2.0 * xc) / (1.0 + xc)) + 2.0 * trilog(1.0 + xc)
                - 4.0 * trilog((1.0 + xc) / (1.0 - xc)) - 4.0 * trilog((1.0 + xc) / (2.0 * xc)) + 2.0 * trilog((-1.0 - yc) / (-1.0 - xc)) + 2.0 * trilog((-1.0 - yc) / (-1.0 + xc)) + 2.0 * trilog(((-1.0 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (1.0 - yc))) + 2.0 * trilog(((-1.0 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (1.0 - yc))) + 4.0 * trilog((1.0 - yc) / 2.0) + 2.0 * trilog((1.0 - yc) / (1.0 - xc)) + 2.0 * trilog((1.0 - yc) / (1.0 + xc)) + 2.0 * trilog(((1.0 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-1.0 - yc))) + 2.0 * trilog(((1.0 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-1.0 - yc))) - 2.0 * trilog((-1.0 - xc) / (-xc - yc)) - 2.0 * trilog((1.0 - xc) / (-xc - yc))
                - 2.0 * trilog((-1.0 - yc) / (-xc - yc)) + 2.0 * trilog((-0.5) * (((-1.0 - xc) * (-1.0 - yc)) / (-xc - yc))) - 2.0 * trilog((1.0 - yc) / (-xc - yc)) + 2.0 * trilog(((1.0 - xc) * (1.0 - yc)) / (2.0 * (-xc - yc))) - 2.0 * trilog((-1.0 + xc) / (xc - yc)) - 2.0 * trilog((1.0 + xc) / (xc - yc)) - 2.0 * trilog((-1.0 - yc) / (xc - yc)) + 2.0 * trilog((-0.5) * (((-1.0 + xc) * (-1.0 - yc)) / (xc - yc))) - 2.0 * trilog((1.0 - yc) / (xc - yc)) + 2.0 * trilog(((1.0 + xc) * (1.0 - yc)) / (2.0 * (xc - yc))) - 4.0 * trilog((-1.0 + yc) / (-1.0 - xc)) - 4.0 * trilog((-1.0 + yc) / (-1.0 + xc)) + 4.0 * trilog((1.0 + yc) / 2.0) - 4.0 * trilog((1.0 + yc) / (1.0 - xc))
                - 4.0 * trilog((1.0 + yc) / (1.0 + xc)) - 4.0 * dilog(-xcinv) * lnhalf - 4.0 * dilog(xcinv) * lnhalf + 4.0 * dilog((-0.5) * ((1.0 - xc) * xcinv)) * lnhalf + 4.0 * dilog((1.0 - xc) / (1.0 + xc)) * lnhalf + 4.0 * dilog((1.0 + xc) / (1.0 - xc)) * lnhalf + 4.0 * dilog((1.0 + xc) / (2.0 * xc)) * lnhalf - (pisqu * ln2) / 3.0 + power_of<3>(ln2) / 3.0 - 2.0 * power_of<2>(lnhalf) * log(1.0 - xcinv) - 2.0 * power_of<2>(lnhalf) * log(1.0 + xcinv) - power_of<3>(log((2.0 * (-1.0 - xc)) / (1.0 - xc))) / 3.0 - (pisqu * ln1mxc) / 12.0 - 2.0 * dilog(1.0 - xc) * ln1mxc - 2.0 * dilog(-xc) * ln1mxc
                + (ln2squ * ln1mxc) / 2.0 - 2.0 * (pisqu / 12.0 - ln2squ / 2.0) * ln1mxc - ln2 * power_of<2>(ln1mxc) + power_of<3>(ln1mxc) / 6.0 + (2.0 * power_of<3>(log(-2.0 * xcinv))) / 3.0 + (2.0 * power_of<3>(log(2.0 * xcinv))) / 3.0 + 2.0 * power_of<2>(lnhalf) * log((-0.5) * ((-1.0 - xc) * xcinv)) + 2.0 * power_of<2>(lnhalf) * log((-1.0 + xc) / (2.0 * xc)) - power_of<2>(ln1mxc) * log(xc) - power_of<2>(lnhalf) * lnr_xc1 + 2.0 * power_of<2>(lnhalf) * log((-2.0 * xc) / (1.0 - xc)) - power_of<2>(lnhalf) * lnr_xc2 - power_of<2>(lnhalf) * lnr_xc2 - power_of<3>(log((-2.0 * (1.0 - xc)) / (1.0 + xc))) / 3.0 - power_of<3>(log((2.0 * (-1.0 + xc)) / (1.0 + xc))) / 3.0
                - power_of<2>(lnhalf) * lnr_xc1 + 2.0 * power_of<2>(lnhalf) * log((2.0 * xc) / (1.0 + xc)) + (3.0 * power_of<2>(ln1mxc) * ln1pxchalf) / 2.0 + dilog(-(xc / (1.0 - xc))) * (ln1mxc - ln1pxc) + dilog(xc / (-1.0 + xc)) * (ln1mxc - ln1pxc) + dilog((-xc + yc) / (-1.0 - xc)) * (ln1mxc - ln1pxc) + dilog((xc + yc) / (1.0 + xc)) * (ln1mxc - ln1pxc) - (pisqu * ln1pxc) / 12.0 - 2.0 * dilog(xc) * ln1pxc - 2.0 * dilog(1.0 + xc) * ln1pxc + (ln2squ * ln1pxc) / 2.0 - 2.0 * (pisqu / 12.0 - ln2squ / 2.0) * ln1pxc
                - 2.0 * ln2 * ln1mxc * ln1pxc - ln2 * power_of<2>(ln1pxc) + (3.0 * ln1mxchalf * power_of<2>(ln1pxc)) / 2.0 - log(-xc) * power_of<2>(ln1pxc) + power_of<3>(ln1pxc) / 6.0 + dilog(-(xc / (-1.0 - xc))) * (-ln1mxc + ln1pxc) + dilog(xc / (1.0 + xc)) * (-ln1mxc + ln1pxc) + dilog((-xc + yc) / (1.0 - xc)) * (-ln1mxc + ln1pxc) + dilog((xc + yc) / (-1.0 + xc)) * (-ln1mxc + ln1pxc) + (ln1pxc * (pisqu - 6.0 * ln2squ - 12.0 * ln1mxc * ln1pxchalf + ln64 * ln1pxc)) / 12.0
                + (ln1mxc * (pisqu - 6.0 * ln2squ + ln64 * ln1mxc - 12.0 * ln1mxchalf * ln1pxc)) / 12.0 - power_of<3>(log((-2.0 * (1.0 + xc)) / (1.0 - xc))) / 3.0 + power_of<3>(log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc)))) / 3.0 + power_of<3>(log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc)))) / 3.0 + power_of<3>(log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc)))) / 3.0 + power_of<3>(log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc)))) / 3.0 - 2.0 * dilog(((1.0 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-1.0 - yc))) * ln1mychalf - 2.0 * dilog(((1.0 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-1.0 - yc))) * ln1mychalf
                + 2.0 * dilog((1.0 - yc) / (-xc - yc)) * ln1mychalf - 2.0 * dilog(((1.0 - xc) * (1.0 - yc)) / (2.0 * (-xc - yc))) * ln1mychalf + 2.0 * dilog((1.0 - yc) / (xc - yc)) * ln1mychalf - 2.0 * dilog(((1.0 + xc) * (1.0 - yc)) / (2.0 * (xc - yc))) * ln1mychalf - 2.0 * dilog(2.0 / (1.0 - xc)) * ln1myc - 2.0 * dilog(2.0 / (1.0 + xc)) * ln1myc - dilog((1.0 - yc) / (1.0 - xc)) * ln1myc - dilog((1.0 - yc) / (1.0 + xc)) * ln1myc - ln2 * ln1mxc * ln1myc + (power_of<2>(ln1mxc) * ln1myc) / 2.0 - ln2 * ln1pxc * ln1myc
                + (power_of<2>(ln1pxc) * ln1myc) / 2.0 - 2.0 * pisqu * ((-1.0 / 6.0) * log(2.0 * xcinv) + log((-2.0 * (1.0 - xc)) / (1.0 + xc)) / 6.0 - log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (-xc - yc)) / 6.0) - power_of<3>(log(-2.0 / (-xc - yc))) / 3.0 - 2.0 * pisqu * (log((2.0 * (-1.0 - xc)) / (1.0 - xc)) / 6.0 - log(-2.0 * xcinv) / 6.0 - log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc))) / 6.0 + log(2.0 / (-xc - yc)) / 6.0) - power_of<3>(log(2.0 / (-xc - yc))) / 3.0 + power_of<2>(ln1mychalf) * log((-1.0 - xc) / (-xc - yc)) - power_of<2>(ln1mychalf) * log(((1.0 + xc) * (-1.0 - yc)) / (2.0 * (-xc - yc))) + power_of<2>(ln1mychalf) * log((-xc - yc) / (-1.0 - xc))
                - power_of<2>(ln1mychalf) * log((-2.0 * (-xc - yc)) / ((-1.0 - xc) * (-1.0 - yc))) - 2.0 * pisqu * ((-1.0 / 6.0) * log(-2.0 * xcinv) + log((-2.0 * (1.0 + xc)) / (1.0 - xc)) / 6.0 - log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (xc - yc)) / 6.0) - power_of<3>(log(-2.0 / (xc - yc))) / 3.0 - 2.0 * pisqu * ((-1.0 / 6.0) * log(2.0 * xcinv) + log((2.0 * (-1.0 + xc)) / (1.0 + xc)) / 6.0 - log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc))) / 6.0 + log(2.0 / (xc - yc)) / 6.0) - power_of<3>(log(2.0 / (xc - yc))) / 3.0 + power_of<2>(ln1mychalf) * log((-1.0 + xc) / (xc - yc)) - power_of<2>(ln1mychalf) * log(((1.0 - xc) * (-1.0 - yc)) / (2.0 * (xc - yc)))
                + power_of<2>(ln1mychalf) * log((xc - yc) / (-1.0 + xc)) - power_of<2>(ln1mychalf) * log((-2.0 * (xc - yc)) / ((-1.0 + xc) * (-1.0 - yc))) - 2.0 * dilog(((-1.0 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (1.0 - yc))) * ln1pychalf - 2.0 * dilog(((-1.0 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (1.0 - yc))) * ln1pychalf + 2.0 * dilog((-1.0 - yc) / (-xc - yc)) * ln1pychalf - 2.0 * dilog((-0.5) * (((-1.0 - xc) * (-1.0 - yc)) / (-xc - yc))) * ln1pychalf + 2.0 * dilog((-1.0 - yc) / (xc - yc)) * ln1pychalf - 2.0 * dilog((-0.5) * (((-1.0 + xc) * (-1.0 - yc)) / (xc - yc))) * ln1pychalf
                + 2.0 * ln1myc * log((-xc - yc) / (1.0 - xc)) * ln1pychalf + 2.0 * ln1myc * log((xc - yc) / (1.0 + xc)) * ln1pychalf + log((1.0 - xc) / (-xc - yc)) * power_of<2>(ln1pychalf) - log((-0.5) * (((-1.0 + xc) * (1.0 - yc)) / (-xc - yc))) * power_of<2>(ln1pychalf) + log((-xc - yc) / (1.0 - xc)) * power_of<2>(ln1pychalf) - log((2.0 * (-xc - yc)) / ((1.0 - xc) * (1.0 - yc))) * power_of<2>(ln1pychalf) + log((1.0 + xc) / (xc - yc)) * power_of<2>(ln1pychalf) - log((-0.5) * (((-1.0 - xc) * (1.0 - yc)) / (xc - yc))) * power_of<2>(ln1pychalf) + log((xc - yc) / (1.0 + xc)) * power_of<2>(ln1pychalf)
                - log((2.0 * (xc - yc)) / ((1.0 + xc) * (1.0 - yc))) * power_of<2>(ln1pychalf) + dilog((1.0 + yc) / (1.0 - xc)) * (ln1myc + 2.0 * ln1pychalf) + dilog((1.0 + yc) / (1.0 + xc)) * (ln1myc + 2.0 * ln1pychalf) + li2r_xc * (-2.0 * lnhalf + ln1myc - ln1pyc) + dilog(-1.0 / (-1.0 + xc)) * (-2.0 * lnhalf + ln1myc - ln1pyc) - 2.0 * dilog(-2.0 / (-1.0 - xc)) * ln1pyc - 2.0 * dilog(-2.0 / (-1.0 + xc)) * ln1pyc - dilog((-1.0 - yc) / (-1.0 - xc)) * ln1pyc - dilog((-1.0 - yc) / (-1.0 + xc)) * ln1pyc
                - ln2 * ln1mxc * ln1pyc + (power_of<2>(ln1mxc) * ln1pyc) / 2.0 - ln2 * ln1pxc * ln1pyc + (power_of<2>(ln1pxc) * ln1pyc) / 2.0 + 2.0 * ln1mychalf * log((-xc - yc) / (-1.0 - xc)) * ln1pyc - ln1myc * log((-xc - yc) / (-1.0 - xc)) * ln1pyc - ln1myc * log((-xc - yc) / (1.0 - xc)) * ln1pyc + 2.0 * ln1mychalf * log((xc - yc) / (-1.0 + xc)) * ln1pyc - ln1myc * log((xc - yc) / (-1.0 + xc)) * ln1pyc - ln1myc * log((xc - yc) / (1.0 + xc)) * ln1pyc
                + dilog((-1.0 + yc) / (-1.0 - xc)) * (2.0 * ln1mychalf + ln1pyc) + dilog((-1.0 + yc) / (-1.0 + xc)) * (2.0 * ln1mychalf + ln1pyc) + dilog(1.0 / (1.0 - xc)) * (-2.0 * lnhalf - ln1myc + ln1pyc) + li2r_xc * (-2.0 * lnhalf - ln1myc + ln1pyc) + ln1mxc * log((-1.0 - yc) / (-1.0 - xc)) * log(1.0 - yc * xcinv) - ln1pxc * log((-1.0 - yc) / (-1.0 - xc)) * log(1.0 - yc * xcinv) - ln1mxc * log((1.0 - yc) / (1.0 - xc)) * log(1.0 - yc * xcinv) + ln1pxc * log((1.0 - yc) / (1.0 - xc)) * log(1.0 - yc * xcinv)
                - ln1mxc * log((-1.0 - yc) / (-1.0 + xc)) * log(1.0 + yc * xcinv) + ln1pxc * log((-1.0 - yc) / (-1.0 + xc)) * log(1.0 + yc * xcinv) + ln1mxc * log((1.0 - yc) / (1.0 + xc)) * log(1.0 + yc * xcinv) - ln1pxc * log((1.0 - yc) / (1.0 + xc)) * log(1.0 + yc * xcinv) - (2.0 * 1.0i) * M_PI * H1((-1.0 + xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 - xc) * (-1.0 - yc)) / (2.0 * (1.0 - xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) - (2.0 * 1.0i) * M_PI * H1((-1.0 - xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 + xc) * (-1.0 - yc)) / (2.0 * (1.0 + xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))
                + (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 - xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((-xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) + (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) - (2.0 * 1.0i) * M_PI * H1((1.0 + xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 - xc) * (1.0 - yc)) / (-1.0 - xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc))) - (2.0 * 1.0i) * M_PI * H1((1.0 - xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 + xc) * (1.0 - yc)) / (-1.0 + xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc))) + (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + xc), 2.0 / (1.0 - yc)) * power_of<2>(log((-xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))
                + (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 - xc), 2.0 / (1.0 - yc)) * power_of<2>(log((xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))
                - (2.0 * 1.0i) * M_PI * power_of<2>(ln1mxchalf) * my_sign(imxc / 2.0) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0) + (4.0 * 1.0i) * M_PI * ln1mxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0) - (2.0 * 1.0i) * M_PI * power_of<2>(ln1pxchalf) * my_sign((-0.5) * imxc) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0) + (4.0 * 1.0i) * M_PI * ln1pxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0)
                + (2.0 * 1.0i) * M_PI * power_of<2>(log((-xc - yc) / (1.0 - xc))) * my_sign(imag((1.0 + yc) / (1.0 - xc))) * T(1.0, (1.0 - yc) / 2.0, (-xc - yc) / (1.0 - xc)) + (2.0 * 1.0i) * M_PI * power_of<2>(log((xc - yc) / (1.0 + xc))) * my_sign(imag((1.0 + yc) / (1.0 + xc))) * T(1.0, (1.0 - yc) / 2.0, (xc - yc) / (1.0 + xc)) - (2.0 * 1.0i) * M_PI * ln1mxc * ln1pyc * my_sign(imag(xcinv)) * t11myc1mxc + (2.0 * 1.0i) * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11myc1mxc - (4.0 * 1.0i) * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11myc1mxc + (2.0 * 1.0i) * M_PI * ln1mxc * ln1myc * my_sign(imycinv) * t11myc1mxc
                - (2.0 * 1.0i) * M_PI * ln1pxc * ln1pyc * my_sign(-imag(xcinv)) * t11myc1pxc - (4.0 * 1.0i) * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11myc1pxc + (2.0 * 1.0i) * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11myc1pxc + (2.0 * 1.0i) * M_PI * ln1pxc * ln1myc * my_sign(imycinv) * t11myc1pxc - (2.0 * 1.0i) * M_PI * power_of<2>(ln1mxchalf) * my_sign(imxc / 2.0) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0) + (4.0 * 1.0i) * M_PI * ln1mxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0)
                - (2.0 * 1.0i) * M_PI * power_of<2>(ln1pxchalf) * my_sign((-0.5) * imxc) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0) + (4.0 * 1.0i) * M_PI * ln1pxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0) + (2.0 * 1.0i) * M_PI * power_of<2>(log((-xc - yc) / (-1.0 - xc))) * my_sign(imag((-1.0 + yc) / (-1.0 - xc))) * T(1.0, (1.0 + yc) / 2.0, (-xc - yc) / (-1.0 - xc)) + (2.0 * 1.0i) * M_PI * power_of<2>(log((xc - yc) / (-1.0 + xc))) * my_sign(imag((-1.0 + yc) / (-1.0 + xc))) * T(1.0, (1.0 + yc) / 2.0, (xc - yc) / (-1.0 + xc)) - (2.0 * 1.0i) * M_PI * ln1mxc * ln1myc * my_sign(imag(xcinv)) * t11pyc1mxc
                + (2.0 * 1.0i) * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11pyc1mxc - (4.0 * 1.0i) * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11pyc1mxc + (2.0 * 1.0i) * M_PI * ln1mxc * ln1pyc * my_sign(-imycinv) * t11pyc1mxc + dilog((1.0 - xc) / 2.0) * (2.0 * ln1mxc - 2.0 * lnr_xc2 - 2.0 * lnr_xc2 + ln1myc + 2.0 * log((-xc - yc) / (1.0 - xc)) + 2.0 * log((xc - yc) / (-1.0 + xc)) + ln1pyc - (4.0 * 1.0i) * M_PI * my_sign(imxc) * t11myc1mxc - (4.0 * 1.0i) * M_PI * my_sign(imxc) * t11pyc1mxc)
                - (2.0 * 1.0i) * M_PI * ln1pxc * ln1myc * my_sign(-imag(xcinv)) * t11pyc1pxc - (4.0 * 1.0i) * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11pyc1pxc + (2.0 * 1.0i) * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11pyc1pxc + (2.0 * 1.0i) * M_PI * ln1pxc * ln1pyc * my_sign(-imycinv) * t11pyc1pxc
                + dilog((1.0 + xc) / 2.0) * (-2.0 * lnr_xc1 - 2.0 * lnr_xc1 + 2.0 * ln1pxc + ln1myc + 2.0 * log((-xc - yc) / (-1.0 - xc)) + 2.0 * log((xc - yc) / (1.0 + xc)) + ln1pyc - (4.0 * 1.0i) * M_PI * my_sign(-imxc) * t11myc1pxc - (4.0 * 1.0i) * M_PI * my_sign(-imxc) * t11pyc1pxc) + li2half * (2.0 * lnr_xc1 + 2.0 * lnr_xc2 + 2.0 * lnr_xc2 + 2.0 * lnr_xc1 - 2.0 * ln1myc - 2.0 * log((-xc - yc) / (-1.0 - xc)) - 2.0 * log((-xc - yc) / (1.0 - xc)) - 2.0 * log((xc - yc) / (-1.0 + xc))
                  - 2.0 * log((xc - yc) / (1.0 + xc)) - 2.0 * ln1pyc + (4.0 * 1.0i) * M_PI * my_sign(imxc) * t11myc1mxc + (4.0 * 1.0i) * M_PI * my_sign(-imxc) * t11myc1pxc + (4.0 * 1.0i) * M_PI * my_sign(imxc) * t11pyc1mxc + (4.0 * 1.0i) * M_PI * my_sign(-imxc) * t11pyc1pxc) - (2.0 * 1.0i) * M_PI * log(1.0 - xcinv) * ln1mxc * my_sign(imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 - xcinv) + (2.0 * 1.0i) * M_PI * log(1.0 - xcinv) * ln1pxc * my_sign(imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 - xcinv) + (2.0 * 1.0i) * M_PI * log(1.0 + xcinv) * ln1mxc * my_sign(-imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 + xcinv)
                - (2.0 * 1.0i) * M_PI * log(1.0 + xcinv) * ln1pxc * my_sign(-imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 + xcinv) - (2.0 * 1.0i) * M_PI * log(1.0 - xcinv) * ln1mxc * my_sign(imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 - xcinv) + (2.0 * 1.0i) * M_PI * log(1.0 - xcinv) * ln1pxc * my_sign(imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 - xcinv) + (2.0 * 1.0i) * M_PI * log(1.0 + xcinv) * ln1mxc * my_sign(-imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 + xcinv) - (2.0 * 1.0i) * M_PI * log(1.0 + xcinv) * ln1pxc * my_sign(-imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 + xcinv) + (2.0 * pisqu * ln2 - 4.0 * power_of<3>(ln2) - 21.0 * zeta3) / 6.0 - (9.0 * zeta3) / 4.0) / denom6);

            const complex<double> f27cPart4 = (-8.0 * factorp0 * (-4.0 * li3half + 2.0 * trilog(1.0 - xcinv) + 2.0 * trilog(1.0 + xcinv) + trilog(-1.0 / (-1.0 - xc)) + trilog(1.0 / (1.0 - xc)) + trilog(-1.0 / (-1.0 + xc)) + 2.0 * trilog(-xcinv) + 2.0 * trilog(xcinv) - 2.0 * trilog((-0.5) * ((1.0 - xc) * xcinv)) + trilog(1.0 / (1.0 + xc)) - 2.0 * trilog((1.0 - xc) / (1.0 + xc)) - 2.0 * trilog((1.0 + xc) / (1.0 - xc)) - 2.0 * trilog((1.0 + xc) / (2.0 * xc)) + trilog(((-1.0 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (1.0 - yc))) + trilog(((-1.0 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (1.0 - yc))) + 2.0 * trilog((1.0 - yc) / 2.0)
                + trilog(((1.0 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-1.0 - yc))) + trilog(((1.0 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-1.0 - yc))) - trilog((-1.0 - xc) / (-xc - yc)) - trilog((1.0 - xc) / (-xc - yc)) - trilog((-1.0 - yc) / (-xc - yc)) + trilog((-0.5) * (((-1.0 - xc) * (-1.0 - yc)) / (-xc - yc))) - trilog((1.0 - yc) / (-xc - yc)) + trilog(((1.0 - xc) * (1.0 - yc)) / (2.0 * (-xc - yc))) - trilog((-1.0 + xc) / (xc - yc)) - trilog((1.0 + xc) / (xc - yc)) - trilog((-1.0 - yc) / (xc - yc)) + trilog((-0.5) * (((-1.0 + xc) * (-1.0 - yc)) / (xc - yc))) - trilog((1.0 - yc) / (xc - yc))
                + trilog(((1.0 + xc) * (1.0 - yc)) / (2.0 * (xc - yc))) - trilog((-1.0 + yc) / (-1.0 - xc)) - trilog((-1.0 + yc) / (-1.0 + xc)) + 2.0 * trilog((1.0 + yc) / 2.0) - trilog((1.0 + yc) / (1.0 - xc)) - trilog((1.0 + yc) / (1.0 + xc)) - li2r_xc * lnhalf - dilog(1.0 / (1.0 - xc)) * lnhalf - dilog(-1.0 / (-1.0 + xc)) * lnhalf - 2.0 * dilog(-xcinv) * lnhalf - 2.0 * dilog(xcinv) * lnhalf + 2.0 * dilog((-0.5) * ((1.0 - xc) * xcinv)) * lnhalf - li2r_xc * lnhalf + 2.0 * dilog((1.0 - xc) / (1.0 + xc)) * lnhalf
                + 2.0 * dilog((1.0 + xc) / (1.0 - xc)) * lnhalf + 2.0 * dilog((1.0 + xc) / (2.0 * xc)) * lnhalf - power_of<2>(lnhalf) * log(1.0 - xcinv) - power_of<2>(lnhalf) * log(1.0 + xcinv) - power_of<3>(log((2.0 * (-1.0 - xc)) / (1.0 - xc))) / 6.0 + power_of<3>(log(-2.0 * xcinv)) / 3.0 + power_of<3>(log(2.0 * xcinv)) / 3.0 + power_of<2>(lnhalf) * log((-0.5) * ((-1.0 - xc) * xcinv)) + power_of<2>(lnhalf) * log((-1.0 + xc) / (2.0 * xc)) - (power_of<2>(lnhalf) * lnr_xc1) / 2.0 + power_of<2>(lnhalf) * log((-2.0 * xc) / (1.0 - xc)) - (power_of<2>(lnhalf) * lnr_xc2) / 2.0 - (power_of<2>(lnhalf) * lnr_xc2) / 2.0 - power_of<3>(log((-2.0 * (1.0 - xc)) / (1.0 + xc))) / 6.0
                - power_of<3>(log((2.0 * (-1.0 + xc)) / (1.0 + xc))) / 6.0 - (power_of<2>(lnhalf) * lnr_xc1) / 2.0 + power_of<2>(lnhalf) * log((2.0 * xc) / (1.0 + xc)) - power_of<3>(log((-2.0 * (1.0 + xc)) / (1.0 - xc))) / 6.0 + power_of<3>(log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc)))) / 6.0 + power_of<3>(log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc)))) / 6.0 + power_of<3>(log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc)))) / 6.0 + power_of<3>(log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc)))) / 6.0 - dilog(((1.0 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-1.0 - yc))) * ln1mychalf - dilog(((1.0 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-1.0 - yc))) * ln1mychalf
                + dilog((1.0 - yc) / (-xc - yc)) * ln1mychalf - dilog(((1.0 - xc) * (1.0 - yc)) / (2.0 * (-xc - yc))) * ln1mychalf + dilog((1.0 - yc) / (xc - yc)) * ln1mychalf - dilog(((1.0 + xc) * (1.0 - yc)) / (2.0 * (xc - yc))) * ln1mychalf - dilog(2.0 / (1.0 - xc)) * ln1myc - dilog(2.0 / (1.0 + xc)) * ln1myc - pisqu * ((-1.0 / 6.0) * log(2.0 * xcinv) + log((-2.0 * (1.0 - xc)) / (1.0 + xc)) / 6.0 - log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (-xc - yc)) / 6.0) - power_of<3>(log(-2.0 / (-xc - yc))) / 6.0
                - pisqu * (log((2.0 * (-1.0 - xc)) / (1.0 - xc)) / 6.0 - log(-2.0 * xcinv) / 6.0 - log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc))) / 6.0 + log(2.0 / (-xc - yc)) / 6.0) - power_of<3>(log(2.0 / (-xc - yc))) / 6.0 + (power_of<2>(ln1mychalf) * log((-1.0 - xc) / (-xc - yc))) / 2.0 - (power_of<2>(ln1mychalf) * log(((1.0 + xc) * (-1.0 - yc)) / (2.0 * (-xc - yc)))) / 2.0 + (power_of<2>(ln1mychalf) * log((-xc - yc) / (-1.0 - xc))) / 2.0 - (power_of<2>(ln1myc) * log((-xc - yc) / (-1.0 - xc))) / 2.0 - (power_of<2>(ln1mychalf) * log((-2.0 * (-xc - yc)) / ((-1.0 - xc) * (-1.0 - yc)))) / 2.0
                - pisqu * ((-1.0 / 6.0) * log(-2.0 * xcinv) + log((-2.0 * (1.0 + xc)) / (1.0 - xc)) / 6.0 - log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (xc - yc)) / 6.0) - power_of<3>(log(-2.0 / (xc - yc))) / 6.0 - pisqu * ((-1.0 / 6.0) * log(2.0 * xcinv) + log((2.0 * (-1.0 + xc)) / (1.0 + xc)) / 6.0 - log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc))) / 6.0 + log(2.0 / (xc - yc)) / 6.0) - power_of<3>(log(2.0 / (xc - yc))) / 6.0 + (power_of<2>(ln1mychalf) * log((-1.0 + xc) / (xc - yc))) / 2.0 - (power_of<2>(ln1mychalf) * log(((1.0 - xc) * (-1.0 - yc)) / (2.0 * (xc - yc)))) / 2.0 + (power_of<2>(ln1mychalf) * log((xc - yc) / (-1.0 + xc))) / 2.0
                - (power_of<2>(ln1myc) * log((xc - yc) / (-1.0 + xc))) / 2.0 - (power_of<2>(ln1mychalf) * log((-2.0 * (xc - yc)) / ((-1.0 + xc) * (-1.0 - yc)))) / 2.0 - dilog(((-1.0 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (1.0 - yc))) * ln1pychalf - dilog(((-1.0 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (1.0 - yc))) * ln1pychalf + dilog((-1.0 - yc) / (-xc - yc)) * ln1pychalf - dilog((-0.5) * (((-1.0 - xc) * (-1.0 - yc)) / (-xc - yc))) * ln1pychalf + dilog((-1.0 - yc) / (xc - yc)) * ln1pychalf - dilog((-0.5) * (((-1.0 + xc) * (-1.0 - yc)) / (xc - yc))) * ln1pychalf
                + ln1myc * log((-xc - yc) / (1.0 - xc)) * ln1pychalf + ln1myc * log((xc - yc) / (1.0 + xc)) * ln1pychalf + (log((1.0 - xc) / (-xc - yc)) * power_of<2>(ln1pychalf)) / 2.0 - (log((-0.5) * (((-1.0 + xc) * (1.0 - yc)) / (-xc - yc))) * power_of<2>(ln1pychalf)) / 2.0 + (log((-xc - yc) / (1.0 - xc)) * power_of<2>(ln1pychalf)) / 2.0 - (log((2.0 * (-xc - yc)) / ((1.0 - xc) * (1.0 - yc))) * power_of<2>(ln1pychalf)) / 2.0 + (log((1.0 + xc) / (xc - yc)) * power_of<2>(ln1pychalf)) / 2.0 - (log((-0.5) * (((-1.0 - xc) * (1.0 - yc)) / (xc - yc))) * power_of<2>(ln1pychalf)) / 2.0
                + (log((xc - yc) / (1.0 + xc)) * power_of<2>(ln1pychalf)) / 2.0 - (log((2.0 * (xc - yc)) / ((1.0 + xc) * (1.0 - yc))) * power_of<2>(ln1pychalf)) / 2.0 + dilog((1.0 + yc) / (1.0 - xc)) * (ln1myc + ln1pychalf - ln1pyc) + dilog((1.0 + yc) / (1.0 + xc)) * (ln1myc + ln1pychalf - ln1pyc) - dilog(-2.0 / (-1.0 - xc)) * ln1pyc - dilog(-2.0 / (-1.0 + xc)) * ln1pyc + ln1mychalf * log((-xc - yc) / (-1.0 - xc)) * ln1pyc + ln1mychalf * log((xc - yc) / (-1.0 + xc)) * ln1pyc
                - (log((-xc - yc) / (1.0 - xc)) * power_of<2>(ln1pyc)) / 2.0 - (log((xc - yc) / (1.0 + xc)) * power_of<2>(ln1pyc)) / 2.0 + dilog((-1.0 + yc) / (-1.0 - xc)) * (ln1mychalf - ln1myc + ln1pyc) + dilog((-1.0 + yc) / (-1.0 + xc)) * (ln1mychalf - ln1myc + ln1pyc) - ln2 * ln1mxc * log(1.0 - yc * xcinv) + (power_of<2>(ln1mxc) * log(1.0 - yc * xcinv)) / 2.0 - ln2 * ln1pxc * log(1.0 - yc * xcinv) + (power_of<2>(ln1pxc) * log(1.0 - yc * xcinv)) / 2.0 - ln2 * ln1mxc * log(1.0 + yc * xcinv) + (power_of<2>(ln1mxc) * log(1.0 + yc * xcinv)) / 2.0
                - ln2 * ln1pxc * log(1.0 + yc * xcinv) + (power_of<2>(ln1pxc) * log(1.0 + yc * xcinv)) / 2.0 - 1.0i * M_PI * H1((-1.0 + xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 - xc) * (-1.0 - yc)) / (2.0 * (1.0 - xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) - 1.0i * M_PI * H1((-1.0 - xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 + xc) * (-1.0 - yc)) / (2.0 * (1.0 + xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) + 1.0i * M_PI * H1(2.0 / (1.0 - xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((-xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) + 1.0i * M_PI * H1(2.0 / (1.0 + xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))
                - 1.0i * M_PI * H1((1.0 + xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 - xc) * (1.0 - yc)) / (-1.0 - xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc))) - 1.0i * M_PI * H1((1.0 - xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 + xc) * (1.0 - yc)) / (-1.0 + xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc))) + 1.0i * M_PI * H1(-2.0 / (-1.0 + xc), 2.0 / (1.0 - yc)) * power_of<2>(log((-xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc))) + 1.0i * M_PI * H1(-2.0 / (-1.0 - xc), 2.0 / (1.0 - yc)) * power_of<2>(log((xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))
                - 1.0i * M_PI * power_of<2>(ln1mxchalf) * my_sign(imxc / 2.0) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0)
                + (2.0 * 1.0i) * M_PI * ln1mxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0) - 1.0i * M_PI * power_of<2>(ln1pxchalf) * my_sign((-0.5) * imxc) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0) + (2.0 * 1.0i) * M_PI * ln1pxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0) + 1.0i * M_PI * power_of<2>(log((-xc - yc) / (1.0 - xc))) * my_sign(imag((1.0 + yc) / (1.0 - xc))) * T(1.0, (1.0 - yc) / 2.0, (-xc - yc) / (1.0 - xc)) + 1.0i * M_PI * power_of<2>(log((xc - yc) / (1.0 + xc))) * my_sign(imag((1.0 + yc) / (1.0 + xc))) * T(1.0, (1.0 - yc) / 2.0, (xc - yc) / (1.0 + xc))
                + 1.0i * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11myc1mxc - (2.0 * 1.0i) * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11myc1mxc - (2.0 * 1.0i) * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11myc1pxc + 1.0i * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11myc1pxc - 1.0i * M_PI * power_of<2>(ln1mxchalf) * my_sign(imxc / 2.0) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0) + (2.0 * 1.0i) * M_PI * ln1mxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0) - 1.0i * M_PI * power_of<2>(ln1pxchalf) * my_sign((-0.5) * imxc) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0)
                + (2.0 * 1.0i) * M_PI * ln1pxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0) + 1.0i * M_PI * power_of<2>(log((-xc - yc) / (-1.0 - xc))) * my_sign(imag((-1.0 + yc) / (-1.0 - xc))) * T(1.0, (1.0 + yc) / 2.0, (-xc - yc) / (-1.0 - xc)) + 1.0i * M_PI * power_of<2>(log((xc - yc) / (-1.0 + xc))) * my_sign(imag((-1.0 + yc) / (-1.0 + xc))) * T(1.0, (1.0 + yc) / 2.0, (xc - yc) / (-1.0 + xc)) + 1.0i * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11pyc1mxc - (2.0 * 1.0i) * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11pyc1mxc
                + dilog((1.0 - xc) / 2.0) * (-lnr_xc2 - lnr_xc2 + log((-xc - yc) / (1.0 - xc)) + log((xc - yc) / (-1.0 + xc)) + log(1.0 - yc * xcinv) + log(1.0 + yc * xcinv) - (2.0 * 1.0i) * M_PI * my_sign(imxc) * t11myc1mxc - (2.0 * 1.0i) * M_PI * my_sign(imxc) * t11pyc1mxc) - (2.0 * 1.0i) * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11pyc1pxc + 1.0i * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11pyc1pxc
                + dilog((1.0 + xc) / 2.0) * (-lnr_xc1 - lnr_xc1 + log((-xc - yc) / (-1.0 - xc)) + log((xc - yc) / (1.0 + xc)) + log(1.0 - yc * xcinv) + log(1.0 + yc * xcinv) - (2.0 * 1.0i) * M_PI * my_sign(-imxc) * t11myc1pxc - (2.0 * 1.0i) * M_PI * my_sign(-imxc) * t11pyc1pxc) + li2half * (lnr_xc1 + lnr_xc2 + lnr_xc2 + lnr_xc1 - log((-xc - yc) / (-1.0 - xc)) - log((-xc - yc) / (1.0 - xc)) - log((xc - yc) / (-1.0 + xc)) - log((xc - yc) / (1.0 + xc)) - 2.0 * log(1.0 - yc * xcinv) - 2.0 * log(1.0 + yc * xcinv)
                  + (2.0 * 1.0i) * M_PI * my_sign(imxc) * t11myc1mxc + (2.0 * 1.0i) * M_PI * my_sign(-imxc) * t11myc1pxc + (2.0 * 1.0i) * M_PI * my_sign(imxc) * t11pyc1mxc + (2.0 * 1.0i) * M_PI * my_sign(-imxc) * t11pyc1pxc))) / denom7;

            const complex<double> f27cPart5 = factorp1 * ((-(pisqu * ln4) / 12.0 + 2.0 * li3half - 4.0 * trilog(1.0 - w3inv) + 2.0 * trilog(1.0 + w3inv) + trilog(-1.0 / (-1.0 - w3)) - 2.0 * trilog(1.0 / (1.0 - w3)) - 2.0 * trilog(-1.0 / (-1.0 + w3)) - 2.0 * trilog(w3inv) + 2.0 * trilog(-w3) - 2.0 * trilog(w3) + trilog(1.0 / (1.0 + w3)) + 2.0 * trilog((1.0 + w3) / (1.0 - w3)) + 2.0 * trilog((1.0 + w3) / (2.0 * w3)) - trilog(((-1.0 - w3) * xc) / (w3 * (-1.0 - xc))) + trilog(-(((-1.0 + w3) * xc) / (w3 * (-1.0 - xc)))) + trilog(((1.0 - w3) * xc) / (w3 * (1.0 - xc))) - trilog(-(((1.0 + w3) * xc) / (w3 * (1.0 - xc))))
                - trilog(((-1.0 - w3) * xc) / (-w3 - xc)) - trilog(-(((1.0 + w3) * xc) / (-w3 - xc))) + trilog(-(((1.0 - w3) * xc) / (w3 - xc))) + trilog(((-1.0 + w3) * xc) / (w3 - xc)) - trilog(-(((-1.0 - w3) * xc) / (w3 * (-1.0 + xc)))) + trilog(((-1.0 + w3) * xc) / (w3 * (-1.0 + xc))) + trilog(-(((1.0 - w3) * xc) / (w3 * (1.0 + xc)))) - trilog(((1.0 + w3) * xc) / (w3 * (1.0 + xc))) - trilog(-(((-1.0 - w3) * xc) / (-w3 + xc))) - trilog(((1.0 + w3) * xc) / (-w3 + xc)) + trilog(((1.0 - w3) * xc) / (w3 + xc)) + trilog(-(((-1.0 + w3) * xc) / (w3 + xc))) - trilog(((-1.0 - w3) * (-1.0 - yc)) / ((1.0 - w3) * (1.0 - yc)))
                - trilog((1.0 - yc) / 2.0) + trilog(1.0 - yc) - trilog(((1.0 + w3) * (1.0 - yc)) / ((-1.0 + w3) * (-1.0 - yc))) - trilog((-1.0 - w3) / (-w3 - yc)) + 2.0 * trilog((1.0 - w3) / (-w3 - yc)) + 2.0 * trilog((-1.0 - yc) / (-w3 - yc)) - trilog((-0.5) * (((-1.0 - w3) * (-1.0 - yc)) / (-w3 - yc))) - trilog((1.0 - yc) / (-w3 - yc)) + trilog(-((w3 * (1.0 - yc)) / (-w3 - yc))) + 2.0 * trilog((-1.0 + w3) / (w3 - yc)) - trilog((1.0 + w3) / (w3 - yc)) - trilog((-1.0 - yc) / (w3 - yc)) + 2.0 * trilog((1.0 - yc) / (w3 - yc)) - trilog((w3 * (1.0 - yc)) / (w3 - yc)) - trilog(((1.0 + w3) * (1.0 - yc)) / (2.0 * (w3 - yc)))
                - trilog(((1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (-1.0 - yc))) + trilog(((1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (-1.0 - yc))) + trilog(((-1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (1.0 - yc))) - trilog(((-1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (1.0 - yc))) + trilog(((-1.0 - w3) * (-xc - yc)) / ((-1.0 - xc) * (-w3 - yc))) - trilog(((1.0 - w3) * (-xc - yc)) / ((1.0 - xc) * (-w3 - yc))) - trilog(((-1.0 + w3) * (-xc - yc)) / ((-1.0 - xc) * (w3 - yc))) + trilog(((1.0 + w3) * (-xc - yc)) / ((1.0 - xc) * (w3 - yc))) - trilog(((1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (-1.0 - yc)))
                + trilog(((1.0 + w3) * (xc - yc)) / ((w3 - xc) * (-1.0 - yc))) + trilog(((-1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (1.0 - yc))) - trilog(((-1.0 + w3) * (xc - yc)) / ((w3 - xc) * (1.0 - yc))) + trilog(((-1.0 - w3) * (xc - yc)) / ((-1.0 + xc) * (-w3 - yc))) - trilog(((1.0 - w3) * (xc - yc)) / ((1.0 + xc) * (-w3 - yc))) - trilog(((-1.0 + w3) * (xc - yc)) / ((-1.0 + xc) * (w3 - yc))) + trilog(((1.0 + w3) * (xc - yc)) / ((1.0 + xc) * (w3 - yc))) - trilog((-1.0 + yc) / (-1.0 - w3)) + 2.0 * trilog((-1.0 + yc) / (-1.0 + w3)) + trilog(-yc) + trilog(yc) + trilog(yc / (-1.0 + yc)) - trilog((2.0 * yc) / (-1.0 + yc))
                + trilog(yc / (1.0 + yc)) - trilog((2.0 * yc) / (1.0 + yc)) - trilog((1.0 + yc) / 2.0) + trilog(1.0 + yc) + 2.0 * trilog((1.0 + yc) / (1.0 - w3)) - trilog((1.0 + yc) / (1.0 + w3)) + trilog((w3 * (-1.0 - yc)) / (-w3 + yc)) - trilog(-((w3 * (-1.0 - yc)) / (w3 + yc))) + dilog(1.0 / (1.0 - w3)) * lnhalf + dilog(-1.0 / (-1.0 + w3)) * lnhalf + 2.0 * dilog(w3inv) * lnhalf - 2.0 * dilog((1.0 + w3) / (1.0 - w3)) * lnhalf - 2.0 * dilog((1.0 + w3) / (2.0 * w3)) * lnhalf + power_of<3>(ln2) / 3.0 + power_of<2>(lnhalf) * log(1.0 - w3inv) + power_of<3>(log((2.0 * (-1.0 - w3)) / (1.0 - w3))) / 6.0 - power_of<3>(log(-2.0 / w3)) / 3.0
                - power_of<2>(lnhalf) * log((-1.0 + w3) / (2.0 * w3)) - power_of<2>(lnhalf) * log((-2.0 * w3) / (1.0 - w3)) + (power_of<2>(lnhalf) * log(-(w3 / (1.0 - w3)))) / 2.0 + (power_of<2>(lnhalf) * log(w3 / (-1.0 + w3))) / 2.0 + power_of<3>(log((-2.0 * (1.0 + w3)) / (1.0 - w3))) / 6.0 - power_of<3>(log(-((-1.0 - xc) / w3))) / 6.0 + power_of<3>(log((-1.0 - xc) / w3)) / 6.0 + power_of<3>(log(-((1.0 - xc) / w3))) / 6.0 - power_of<3>(log((1.0 - xc) / w3)) / 6.0 - power_of<3>(log(((-1.0 - w3) * (1.0 - xc)) / (-w3 - xc))) / 6.0 + power_of<3>(log(((-1.0 + w3) * (1.0 - xc)) / (w3 - xc))) / 6.0 - power_of<3>(log(-((-1.0 + xc) / w3))) / 6.0 + power_of<3>(log((-1.0 + xc) / w3)) / 6.0 - power_of<3>(log(((1.0 + w3) * (-1.0 + xc)) / (-w3 - xc))) / 6.0
                + power_of<3>(log(((1.0 - w3) * (-1.0 + xc)) / (w3 - xc))) / 6.0 + dilog(((-1.0 - w3) * xc) / (w3 * (-1.0 - xc))) * lnr_xc1 - dilog(-(((-1.0 + w3) * xc) / (w3 * (-1.0 - xc)))) * lnr_xc1 + dilog(-(((-1.0 - w3) * xc) / (-w3 + xc))) * lnr_xc1 - dilog(-(((-1.0 + w3) * xc) / (w3 + xc))) * lnr_xc1 - (log((-w3 - xc) / (w3 * (-1.0 - xc))) * power_of<2>(lnr_xc1)) / 2.0 + (log(-((w3 - xc) / (w3 * (-1.0 - xc)))) * power_of<2>(lnr_xc1)) / 2.0 - dilog(((1.0 - w3) * xc) / (w3 * (1.0 - xc))) * lnr_xc2
                + dilog(-(((1.0 + w3) * xc) / (w3 * (1.0 - xc)))) * lnr_xc2 + dilog(-(((1.0 + w3) * xc) / (-w3 - xc))) * lnr_xc2 - dilog(-(((1.0 - w3) * xc) / (w3 - xc))) * lnr_xc2 + (log((w3 * (-1.0 + xc)) / (-w3 - xc)) * power_of<2>(lnr_xc2)) / 2.0 - (log(-((w3 * (-1.0 + xc)) / (w3 - xc))) * power_of<2>(lnr_xc2)) / 2.0 + dilog(((-1.0 - w3) * xc) / (-w3 - xc)) * lnr_xc2 - dilog(((-1.0 + w3) * xc) / (w3 - xc)) * lnr_xc2 + dilog(-(((-1.0 - w3) * xc) / (w3 * (-1.0 + xc)))) * lnr_xc2
                - dilog(((-1.0 + w3) * xc) / (w3 * (-1.0 + xc))) * lnr_xc2 - (log(-(w3 / (-w3 - xc))) * power_of<2>(lnr_xc2)) / 2.0 + (log(-((w3 * (1.0 - xc)) / (-w3 - xc))) * power_of<2>(lnr_xc2)) / 2.0 + (log(w3 / (w3 - xc)) * power_of<2>(lnr_xc2)) / 2.0 - (log((w3 * (1.0 - xc)) / (w3 - xc)) * power_of<2>(lnr_xc2)) / 2.0 + dilog(xc / w3) * (-lnr_xc1 + lnr_xc2 + lnr_xc2 - lnr_xc1) - dilog(-(((1.0 - w3) * xc) / (w3 * (1.0 + xc)))) * lnr_xc1 + dilog(((1.0 + w3) * xc) / (w3 * (1.0 + xc))) * lnr_xc1
                + dilog(((1.0 + w3) * xc) / (-w3 + xc)) * lnr_xc1 - dilog(((1.0 - w3) * xc) / (w3 + xc)) * lnr_xc1 + (log(-(w3 / (-w3 - xc))) * power_of<2>(lnr_xc1)) / 2.0 - (log(w3 / (w3 - xc)) * power_of<2>(lnr_xc1)) / 2.0 - (log(-((-w3 - xc) / (w3 * (1.0 + xc)))) * power_of<2>(lnr_xc1)) / 2.0 + (log((w3 - xc) / (w3 * (1.0 + xc))) * power_of<2>(lnr_xc1)) / 2.0 + dilog(-(xc / w3)) * (lnr_xc1 - lnr_xc2 - lnr_xc2 + lnr_xc1)
                + dilog(-(xc / (-w3 - xc))) * (ln1mxc + lnr_xc1 - lnr_xc2 - ln1pxc) + dilog(xc / (w3 + xc)) * (ln1mxc - lnr_xc2 + lnr_xc1 - ln1pxc) + dilog(-(xc / (w3 - xc))) * (-ln1mxc - lnr_xc1 + lnr_xc2 + ln1pxc) + dilog(xc / (-w3 + xc)) * (-ln1mxc + lnr_xc2 - lnr_xc1 + ln1pxc) + power_of<3>(log(-((1.0 + xc) / w3))) / 6.0 - power_of<3>(log((1.0 + xc) / w3)) / 6.0 - (power_of<2>(lnr_xc1) * log(-(w3 / (-w3 + xc)))) / 2.0
                + (power_of<2>(lnr_xc2) * log(-(w3 / (-w3 + xc)))) / 2.0 + (power_of<2>(lnr_xc1) * log((w3 * (-1.0 - xc)) / (-w3 + xc))) / 2.0 - power_of<3>(log(((1.0 + w3) * (-1.0 - xc)) / (-w3 + xc))) / 6.0 - power_of<3>(log(((-1.0 - w3) * (1.0 + xc)) / (-w3 + xc))) / 6.0 + (power_of<2>(lnr_xc1) * log(-((w3 * (1.0 + xc)) / (-w3 + xc)))) / 2.0 - (power_of<2>(lnr_xc2) * log(-((-w3 + xc) / (w3 * (1.0 - xc))))) / 2.0 - (power_of<2>(lnr_xc2) * log((-w3 + xc) / (w3 * (-1.0 + xc)))) / 2.0 + (power_of<2>(lnr_xc1) * log(w3 / (w3 + xc))) / 2.0 - (power_of<2>(lnr_xc2) * log(w3 / (w3 + xc))) / 2.0
                + power_of<3>(log(((1.0 - w3) * (-1.0 - xc)) / (w3 + xc))) / 6.0 - (power_of<2>(lnr_xc1) * log(-((w3 * (-1.0 - xc)) / (w3 + xc)))) / 2.0 + power_of<3>(log(((-1.0 + w3) * (1.0 + xc)) / (w3 + xc))) / 6.0 - (power_of<2>(lnr_xc1) * log((w3 * (1.0 + xc)) / (w3 + xc))) / 2.0 + (power_of<2>(lnr_xc2) * log((w3 + xc) / (w3 * (1.0 - xc)))) / 2.0 + (power_of<2>(lnr_xc2) * log(-((w3 + xc) / (w3 * (-1.0 + xc))))) / 2.0 - (power_of<2>(lnr_xc1) * log(1.0 - xc / w3)) / 2.0 + (power_of<2>(lnr_xc2) * log(1.0 - xc / w3)) / 2.0 + (power_of<2>(lnr_xc2) * log(1.0 - xc / w3)) / 2.0 - (power_of<2>(lnr_xc1) * log(1.0 - xc / w3)) / 2.0
                + (power_of<2>(lnr_xc1) * log(1.0 + xc / w3)) / 2.0 - (power_of<2>(lnr_xc2) * log(1.0 + xc / w3)) / 2.0 - (power_of<2>(lnr_xc2) * log(1.0 + xc / w3)) / 2.0 + (power_of<2>(lnr_xc1) * log(1.0 + xc / w3)) / 2.0 - power_of<3>(log((-2.0 * (1.0 + w3)) / ((-1.0 + w3) * (-1.0 - yc)))) / 6.0 - power_of<3>(log(((1.0 - w3) * (-1.0 - xc)) / ((-w3 - xc) * (-1.0 - yc)))) / 6.0 + power_of<3>(log(((1.0 + w3) * (-1.0 - xc)) / ((w3 - xc) * (-1.0 - yc)))) / 6.0 - power_of<3>(log(((1.0 - w3) * (-1.0 + xc)) / ((-w3 + xc) * (-1.0 - yc)))) / 6.0 + power_of<3>(log(((1.0 + w3) * (-1.0 + xc)) / ((w3 + xc) * (-1.0 - yc)))) / 6.0 - power_of<3>(log((2.0 * (-1.0 - w3)) / ((1.0 - w3) * (1.0 - yc)))) / 6.0
                + power_of<3>(log(((-1.0 - w3) * (1.0 - xc)) / ((-w3 - xc) * (1.0 - yc)))) / 6.0 - power_of<3>(log(((-1.0 + w3) * (1.0 - xc)) / ((w3 - xc) * (1.0 - yc)))) / 6.0 + power_of<3>(log(((-1.0 - w3) * (1.0 + xc)) / ((-w3 + xc) * (1.0 - yc)))) / 6.0 - power_of<3>(log(((-1.0 + w3) * (1.0 + xc)) / ((w3 + xc) * (1.0 - yc)))) / 6.0 + dilog(((1.0 + w3) * (1.0 - yc)) / ((-1.0 + w3) * (-1.0 - yc))) * ln1mychalf + dilog(((1.0 + w3) * (1.0 - yc)) / (2.0 * (w3 - yc))) * ln1mychalf + dilog((1.0 - yc) / (w3 - yc)) * (-ln1mychalf - ln1myc) + dilog(2.0 / (1.0 - w3)) * ln1myc - dilog((1.0 - xc) / (-w3 - xc)) * ln1myc
                + dilog((1.0 - xc) / (w3 - xc)) * ln1myc - dilog((1.0 + xc) / (-w3 + xc)) * ln1myc + dilog((1.0 + xc) / (w3 + xc)) * ln1myc - dilog(1.0 - yc) * ln1myc + dilog((1.0 - yc) / (-w3 - yc)) * ln1myc - dilog(-((w3 * (1.0 - yc)) / (-w3 - yc))) * ln1myc + dilog((w3 * (1.0 - yc)) / (w3 - yc)) * ln1myc - dilog(-yc) * ln1myc - (pisqu / 12.0 - ln2squ / 2.0) * ln1myc - (ln2 * power_of<2>(ln1myc)) / 2.0 - power_of<3>(ln1myc) / 6.0 - (pisqu * log(-1.0 / (-w3 - yc))) / 6.0 - power_of<3>(log(-1.0 / (-w3 - yc))) / 6.0
                + (pisqu * log(1.0 / (-w3 - yc))) / 6.0 + power_of<3>(log(1.0 / (-w3 - yc))) / 6.0 + pisqu * (log((2.0 * (-1.0 - w3)) / (1.0 - w3)) / 6.0 - log(-2.0 / w3) / 6.0 - log((2.0 * (-1.0 - w3)) / ((1.0 - w3) * (1.0 - yc))) / 6.0 + log(2.0 / (-w3 - yc)) / 6.0) + power_of<3>(log(2.0 / (-w3 - yc))) / 6.0 + (power_of<2>(ln1myc) * log((-1.0 - w3) / (-w3 - yc))) / 2.0 + power_of<3>(log((-1.0 - xc) / (-w3 - yc))) / 6.0 - power_of<3>(log((1.0 - xc) / (-w3 - yc))) / 6.0 + power_of<3>(log((-1.0 + xc) / (-w3 - yc))) / 6.0 - power_of<3>(log((1.0 + xc) / (-w3 - yc))) / 6.0
                + pisqu * ((-1.0 / 6.0) * log(-2.0 / w3) + log((-2.0 * (1.0 + w3)) / (1.0 - w3)) / 6.0 - log((-2.0 * (1.0 + w3)) / ((-1.0 + w3) * (-1.0 - yc))) / 6.0 + log(-2.0 / (w3 - yc)) / 6.0) + power_of<3>(log(-2.0 / (w3 - yc))) / 6.0 + (pisqu * log(-1.0 / (w3 - yc))) / 6.0 + power_of<3>(log(-1.0 / (w3 - yc))) / 6.0 - (pisqu * log(1.0 / (w3 - yc))) / 6.0 - power_of<3>(log(1.0 / (w3 - yc))) / 6.0 - (power_of<2>(ln1mychalf) * log((-1.0 + w3) / (w3 - yc))) / 2.0 - (power_of<2>(ln1myc) * log((-1.0 + w3) / (w3 - yc))) / 2.0 - power_of<3>(log((-1.0 - xc) / (w3 - yc))) / 6.0 + power_of<3>(log((1.0 - xc) / (w3 - yc))) / 6.0 - power_of<3>(log((-1.0 + xc) / (w3 - yc))) / 6.0 + power_of<3>(log((1.0 + xc) / (w3 - yc))) / 6.0
                + (power_of<2>(ln1mychalf) * log(((1.0 - w3) * (-1.0 - yc)) / (2.0 * (w3 - yc)))) / 2.0 - (power_of<2>(ln1mychalf) * log((w3 - yc) / (-1.0 + w3))) / 2.0 + (power_of<2>(ln1myc) * log((w3 - yc) / (-1.0 + w3))) / 2.0 + (power_of<2>(ln1mychalf) * log((-2.0 * (w3 - yc)) / ((-1.0 + w3) * (-1.0 - yc)))) / 2.0 - dilog(((-1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (1.0 - yc))) * log((-xc - yc) / (-1.0 - xc)) + dilog(((-1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (1.0 - yc))) * log((-xc - yc) / (-1.0 - xc)) - dilog(((-1.0 - w3) * (-xc - yc)) / ((-1.0 - xc) * (-w3 - yc))) * log((-xc - yc) / (-1.0 - xc))
                + dilog(((-1.0 + w3) * (-xc - yc)) / ((-1.0 - xc) * (w3 - yc))) * log((-xc - yc) / (-1.0 - xc)) + ln1myc * log((-w3 - yc) / (-w3 + xc)) * log((-xc - yc) / (-1.0 - xc)) - ln1myc * log((w3 - yc) / (w3 + xc)) * log((-xc - yc) / (-1.0 - xc)) + (log((-w3 + xc) / (-w3 - yc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0 - (log(((w3 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-w3 - yc))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0 + (log((-w3 - yc) / (-w3 + xc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0
                - (log(-(((-1.0 - xc) * (-w3 - yc)) / ((-w3 + xc) * (1.0 - yc)))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0 - (log((w3 + xc) / (w3 - yc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0 + (log(((-w3 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (w3 - yc))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0 - (log((w3 - yc) / (w3 + xc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0 + (log(-(((-1.0 - xc) * (w3 - yc)) / ((w3 + xc) * (1.0 - yc)))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / 2.0 + dilog((-xc - yc) / (-w3 - yc)) * (log((-xc - yc) / (-1.0 - xc)) - log((-xc - yc) / (1.0 - xc)))
                + dilog(((1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (-1.0 - yc))) * log((-xc - yc) / (1.0 - xc)) - dilog(((1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (-1.0 - yc))) * log((-xc - yc) / (1.0 - xc)) + dilog(((1.0 - w3) * (-xc - yc)) / ((1.0 - xc) * (-w3 - yc))) * log((-xc - yc) / (1.0 - xc)) - dilog(((1.0 + w3) * (-xc - yc)) / ((1.0 - xc) * (w3 - yc))) * log((-xc - yc) / (1.0 - xc)) - (log((-w3 + xc) / (-w3 - yc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0 + (log(((w3 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (-w3 - yc))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0
                - (log((-w3 - yc) / (-w3 + xc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0 + (log(-(((1.0 - xc) * (-w3 - yc)) / ((-w3 + xc) * (-1.0 - yc)))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0 + (log((w3 + xc) / (w3 - yc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0 - (log(((-w3 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (w3 - yc))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0 + (log((w3 - yc) / (w3 + xc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0 - (log(-(((1.0 - xc) * (w3 - yc)) / ((w3 + xc) * (-1.0 - yc)))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / 2.0
                + dilog((-xc - yc) / (w3 - yc)) * (-log((-xc - yc) / (-1.0 - xc)) + log((-xc - yc) / (1.0 - xc))) - dilog(((-1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (1.0 - yc))) * log((xc - yc) / (-1.0 + xc)) + dilog(((-1.0 + w3) * (xc - yc)) / ((w3 - xc) * (1.0 - yc))) * log((xc - yc) / (-1.0 + xc)) - dilog(((-1.0 - w3) * (xc - yc)) / ((-1.0 + xc) * (-w3 - yc))) * log((xc - yc) / (-1.0 + xc)) + dilog(((-1.0 + w3) * (xc - yc)) / ((-1.0 + xc) * (w3 - yc))) * log((xc - yc) / (-1.0 + xc)) + ln1myc * log((-w3 - yc) / (-w3 - xc)) * log((xc - yc) / (-1.0 + xc))
                - ln1myc * log((w3 - yc) / (w3 - xc)) * log((xc - yc) / (-1.0 + xc)) + (log((-w3 - xc) / (-w3 - yc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0 - (log(((w3 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-w3 - yc))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0 + (log((-w3 - yc) / (-w3 - xc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0 - (log(-(((-1.0 + xc) * (-w3 - yc)) / ((-w3 - xc) * (1.0 - yc)))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0 - (log((w3 - xc) / (w3 - yc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0 + (log(((-w3 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (w3 - yc))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0
                - (log((w3 - yc) / (w3 - xc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0 + (log(-(((-1.0 + xc) * (w3 - yc)) / ((w3 - xc) * (1.0 - yc)))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / 2.0 + dilog((xc - yc) / (-w3 - yc)) * (log((xc - yc) / (-1.0 + xc)) - log((xc - yc) / (1.0 + xc))) + dilog(((1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (-1.0 - yc))) * log((xc - yc) / (1.0 + xc)) - dilog(((1.0 + w3) * (xc - yc)) / ((w3 - xc) * (-1.0 - yc))) * log((xc - yc) / (1.0 + xc)) + dilog(((1.0 - w3) * (xc - yc)) / ((1.0 + xc) * (-w3 - yc))) * log((xc - yc) / (1.0 + xc))
                - dilog(((1.0 + w3) * (xc - yc)) / ((1.0 + xc) * (w3 - yc))) * log((xc - yc) / (1.0 + xc)) - (log((-w3 - xc) / (-w3 - yc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0 + (log(((w3 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (-w3 - yc))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0 - (log((-w3 - yc) / (-w3 - xc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0 + (log(-(((1.0 + xc) * (-w3 - yc)) / ((-w3 - xc) * (-1.0 - yc)))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0 + (log((w3 - xc) / (w3 - yc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0 - (log(((-w3 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (w3 - yc))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0
                + (log((w3 - yc) / (w3 - xc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0 - (log(-(((1.0 + xc) * (w3 - yc)) / ((w3 - xc) * (-1.0 - yc)))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / 2.0 + dilog((xc - yc) / (w3 - yc)) * (-log((xc - yc) / (-1.0 + xc)) + log((xc - yc) / (1.0 + xc))) - (power_of<2>(ln1myc) * log(yc)) / 2.0 - (power_of<2>(ln1myc) * log(((-1.0 - w3) * yc) / (-w3 - yc))) / 2.0 + (power_of<2>(ln1myc) * log(((-1.0 + w3) * yc) / (w3 - yc))) / 2.0 + dilog(((-1.0 - w3) * (-1.0 - yc)) / ((1.0 - w3) * (1.0 - yc))) * ln1pychalf + dilog((-0.5) * (((-1.0 - w3) * (-1.0 - yc)) / (-w3 - yc))) * ln1pychalf
                - ln1myc * log((-w3 - yc) / (1.0 - w3)) * ln1pychalf - (log((1.0 - w3) / (-w3 - yc)) * power_of<2>(ln1pychalf)) / 2.0 + (log((-0.5) * (((-1.0 + w3) * (1.0 - yc)) / (-w3 - yc))) * power_of<2>(ln1pychalf)) / 2.0 - (log((-w3 - yc) / (1.0 - w3)) * power_of<2>(ln1pychalf)) / 2.0 + (log((2.0 * (-w3 - yc)) / ((1.0 - w3) * (1.0 - yc))) * power_of<2>(ln1pychalf)) / 2.0 + dilog((-1.0 + yc) / (-1.0 + w3)) * (-ln1mychalf + ln1myc - ln1pyc)
                + dilog((xc + yc) / (-w3 + xc)) * (ln1mxc - ln1pxc + ln1myc + log((-xc - yc) / (-1.0 - xc)) - log((-xc - yc) / (1.0 - xc)) - ln1pyc) + dilog((-xc + yc) / (-w3 - xc)) * (-ln1mxc + ln1pxc + ln1myc + log((xc - yc) / (-1.0 + xc)) - log((xc - yc) / (1.0 + xc)) - ln1pyc) + dilog((-1.0 - yc) / (-w3 - yc)) * (-ln1pychalf - ln1pyc) + dilog(-2.0 / (-1.0 + w3)) * ln1pyc + dilog((-1.0 - xc) / (-w3 - xc)) * ln1pyc - dilog((-1.0 - xc) / (w3 - xc)) * ln1pyc
                + dilog((-1.0 + xc) / (-w3 + xc)) * ln1pyc - dilog((-1.0 + xc) / (w3 + xc)) * ln1pyc + dilog((-1.0 - yc) / (w3 - yc)) * ln1pyc - dilog(yc) * ln1pyc - dilog(1.0 + yc) * ln1pyc - dilog((w3 * (-1.0 - yc)) / (-w3 + yc)) * ln1pyc + dilog(-((w3 * (-1.0 - yc)) / (w3 + yc))) * ln1pyc - (pisqu / 12.0 - ln2squ / 2.0) * ln1pyc - ln1mychalf * log((w3 - yc) / (-1.0 + w3)) * ln1pyc - log((-w3 - yc) / (-w3 + xc)) * log((-xc - yc) / (1.0 - xc)) * ln1pyc
                + log((w3 - yc) / (w3 + xc)) * log((-xc - yc) / (1.0 - xc)) * ln1pyc - log((-w3 - yc) / (-w3 - xc)) * log((xc - yc) / (1.0 + xc)) * ln1pyc + log((w3 - yc) / (w3 - xc)) * log((xc - yc) / (1.0 + xc)) * ln1pyc - (ln2 * power_of<2>(ln1pyc)) / 2.0 - (log((1.0 - w3) / (-w3 - yc)) * power_of<2>(ln1pyc)) / 2.0 + (log((-w3 - yc) / (1.0 - w3)) * power_of<2>(ln1pyc)) / 2.0 + (log((1.0 + w3) / (w3 - yc)) * power_of<2>(ln1pyc)) / 2.0 - (log(-yc) * power_of<2>(ln1pyc)) / 2.0 - power_of<3>(ln1pyc) / 6.0 + dilog((1.0 - yc) / 2.0) * (ln1myc + ln1pyc)
                + dilog((1.0 + yc) / 2.0) * (ln1myc + ln1pyc) + dilog((xc + yc) / (w3 + xc)) * (-ln1mxc + ln1pxc - ln1myc - log((-xc - yc) / (-1.0 - xc)) + log((-xc - yc) / (1.0 - xc)) + ln1pyc) + dilog((-xc + yc) / (w3 - xc)) * (ln1mxc - ln1pxc - ln1myc - log((xc - yc) / (-1.0 + xc)) + log((xc - yc) / (1.0 + xc)) + ln1pyc) + dilog((1.0 + yc) / (1.0 - w3)) * (-ln1myc - ln1pychalf + ln1pyc)
                - (ln1pyc * (pisqu - 6.0 * ln2squ - 12.0 * ln1myc * ln1pychalf + ln64 * ln1pyc)) / 12.0 - (ln1myc * (pisqu - 6.0 * ln2squ + ln64 * ln1myc - 12.0 * ln1mychalf * ln1pyc)) / 12.0 - (power_of<2>(ln1pyc) * log(((1.0 + w3) * yc) / (-w3 + yc))) / 2.0 + (power_of<2>(ln1pyc) * log(((1.0 - w3) * yc) / (w3 + yc))) / 2.0 + 2.0 * ln2 * ln1mxc * log(1.0 - yc / w3) - power_of<2>(ln1mxc) * log(1.0 - yc / w3) + 2.0 * ln2 * ln1pxc * log(1.0 - yc / w3) - power_of<2>(ln1pxc) * log(1.0 - yc / w3)
                + dilog((1.0 - xc) / 2.0) * (-2.0 * log(1.0 - yc / w3) - 2.0 * log(1.0 + yc / w3)) + dilog((1.0 + xc) / 2.0) * (-2.0 * log(1.0 - yc / w3) - 2.0 * log(1.0 + yc / w3)) + dilog(w3) * (-log(1.0 - yc / w3) - log(1.0 + yc / w3)) + 2.0 * ln2 * ln1mxc * log(1.0 + yc / w3) - power_of<2>(ln1mxc) * log(1.0 + yc / w3) + 2.0 * ln2 * ln1pxc * log(1.0 + yc / w3) - power_of<2>(ln1pxc) * log(1.0 + yc / w3) + dilog(-w3) * (log(1.0 - yc / w3) + log(1.0 + yc / w3)) - ln1mxc * log((-w3 - yc) / (-w3 - xc)) * log(1.0 - yc * xcinv) + ln1pxc * log((-w3 - yc) / (-w3 - xc)) * log(1.0 - yc * xcinv)
                + ln1mxc * log((w3 - yc) / (w3 - xc)) * log(1.0 - yc * xcinv) - ln1pxc * log((w3 - yc) / (w3 - xc)) * log(1.0 - yc * xcinv) + ln1mxc * log((-w3 - yc) / (-w3 + xc)) * log(1.0 + yc * xcinv) - ln1pxc * log((-w3 - yc) / (-w3 + xc)) * log(1.0 + yc * xcinv) - ln1mxc * log((w3 - yc) / (w3 + xc)) * log(1.0 + yc * xcinv) + ln1pxc * log((w3 - yc) / (w3 + xc)) * log(1.0 + yc * xcinv) + 1.0i * M_PI * H1((-1.0 - xc) / (-1.0 - w3), 1.0 + xc) * power_of<2>(log(-(w3 / (-1.0 - xc)))) * my_sign(-imxc) - 1.0i * M_PI * H1((-1.0 - xc) / (-1.0 + w3), 1.0 + xc) * power_of<2>(log(w3 / (-1.0 - xc))) * my_sign(-imxc)
                + 1.0i * M_PI * H1(1.0 - w3, 1.0 + xc) * power_of<2>(log((-w3 - xc) / ((1.0 - w3) * (-1.0 - xc)))) * my_sign(-imxc) - 1.0i * M_PI * H1(1.0 + w3, 1.0 + xc) * power_of<2>(log((w3 - xc) / ((1.0 + w3) * (-1.0 - xc)))) * my_sign(-imxc) - 1.0i * M_PI * H1((1.0 + xc) / (1.0 - w3), 1.0 + xc) * power_of<2>(log(-(w3 / (1.0 + xc)))) * my_sign(-imxc) + 1.0i * M_PI * H1((1.0 + xc) / (1.0 + w3), 1.0 + xc) * power_of<2>(log(w3 / (1.0 + xc))) * my_sign(-imxc) + 1.0i * M_PI * H1(1.0 - w3, 1.0 + xc) * power_of<2>(log((-w3 - xc) / ((-1.0 + w3) * (1.0 + xc)))) * my_sign(-imxc) - 1.0i * M_PI * H1(1.0 + w3, 1.0 + xc) * power_of<2>(log((w3 - xc) / ((-1.0 - w3) * (1.0 + xc)))) * my_sign(-imxc)
                - 1.0i * M_PI * H1((1.0 - xc) / (1.0 - w3), 1.0 - xc) * power_of<2>(log(-(w3 / (1.0 - xc)))) * my_sign(imxc) + 1.0i * M_PI * H1((1.0 - xc) / (1.0 + w3), 1.0 - xc) * power_of<2>(log(w3 / (1.0 - xc))) * my_sign(imxc) + 1.0i * M_PI * H1((-1.0 + xc) / (-1.0 - w3), 1.0 - xc) * power_of<2>(log(-(w3 / (-1.0 + xc)))) * my_sign(imxc) - 1.0i * M_PI * H1((-1.0 + xc) / (-1.0 + w3), 1.0 - xc) * power_of<2>(log(w3 / (-1.0 + xc))) * my_sign(imxc) + 1.0i * M_PI * H1(1.0 - w3, 1.0 - xc) * power_of<2>(log((-w3 + xc) / ((-1.0 + w3) * (1.0 - xc)))) * my_sign(imxc) + 1.0i * M_PI * H1(1.0 - w3, 1.0 - xc) * power_of<2>(log((-w3 + xc) / ((1.0 - w3) * (-1.0 + xc)))) * my_sign(imxc)
                - 1.0i * M_PI * H1(1.0 + w3, 1.0 - xc) * power_of<2>(log((w3 + xc) / ((-1.0 - w3) * (1.0 - xc)))) * my_sign(imxc) - 1.0i * M_PI * H1(1.0 + w3, 1.0 - xc) * power_of<2>(log((w3 + xc) / ((1.0 + w3) * (-1.0 + xc)))) * my_sign(imxc) + 1.0i * M_PI * H1((-1.0 - w3) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 + w3) * (-1.0 - yc)) / (2.0 * (1.0 + w3)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) - 1.0i * M_PI * H1(2.0 / (1.0 + w3), -2.0 / (-1.0 - yc)) * power_of<2>(log((w3 - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) + 1.0i * M_PI * H1((-1.0 + w3) / (-1.0 - yc), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log(-(((-w3 - xc) * (-1.0 - yc)) / ((1.0 - w3) * (-1.0 - xc))))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc)))
                - 1.0i * M_PI * H1((-1.0 - w3) / (-1.0 - yc), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log(-(((w3 - xc) * (-1.0 - yc)) / ((1.0 + w3) * (-1.0 - xc))))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc))) - 1.0i * M_PI * H1((1.0 + xc) / (1.0 - w3), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log((-w3 - yc) / (1.0 + xc))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc))) + 1.0i * M_PI * H1((1.0 + xc) / (1.0 + w3), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log((w3 - yc) / (1.0 + xc))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc))) + 1.0i * M_PI * H1((-1.0 + w3) / (-1.0 - yc), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log(-(((-w3 + xc) * (-1.0 - yc)) / ((1.0 - w3) * (-1.0 + xc))))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc)))
                - 1.0i * M_PI * H1((-1.0 - w3) / (-1.0 - yc), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log(-(((w3 + xc) * (-1.0 - yc)) / ((1.0 + w3) * (-1.0 + xc))))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc))) - 1.0i * M_PI * H1((1.0 - xc) / (1.0 - w3), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log((-w3 - yc) / (1.0 - xc))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc))) + 1.0i * M_PI * H1((1.0 - xc) / (1.0 + w3), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log((w3 - yc) / (1.0 - xc))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc))) + 1.0i * M_PI * H1((1.0 + w3) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 - w3) * (1.0 - yc)) / (-1.0 - w3)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))
                - 1.0i * M_PI * H1(-2.0 / (-1.0 - w3), 2.0 / (1.0 - yc)) * power_of<2>(log((w3 + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc))) - 1.0i * M_PI * H1((1.0 + w3) / (1.0 - yc), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log(-(((-w3 - xc) * (1.0 - yc)) / ((-1.0 - w3) * (1.0 - xc))))) * my_sign(imag((1.0 - xc) / (1.0 - yc))) + 1.0i * M_PI * H1((1.0 - w3) / (1.0 - yc), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log(-(((w3 - xc) * (1.0 - yc)) / ((-1.0 + w3) * (1.0 - xc))))) * my_sign(imag((1.0 - xc) / (1.0 - yc))) + 1.0i * M_PI * H1((-1.0 + xc) / (-1.0 - w3), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log((-w3 - yc) / (-1.0 + xc))) * my_sign(imag((1.0 - xc) / (1.0 - yc)))
                - 1.0i * M_PI * H1((-1.0 + xc) / (-1.0 + w3), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log((w3 - yc) / (-1.0 + xc))) * my_sign(imag((1.0 - xc) / (1.0 - yc))) - 1.0i * M_PI * H1((1.0 + w3) / (1.0 - yc), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log(-(((-w3 + xc) * (1.0 - yc)) / ((-1.0 - w3) * (1.0 + xc))))) * my_sign(imag((1.0 + xc) / (1.0 - yc))) + 1.0i * M_PI * H1((1.0 - w3) / (1.0 - yc), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log(-(((w3 + xc) * (1.0 - yc)) / ((-1.0 + w3) * (1.0 + xc))))) * my_sign(imag((1.0 + xc) / (1.0 - yc))) + 1.0i * M_PI * H1((-1.0 - xc) / (-1.0 - w3), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log((-w3 - yc) / (-1.0 - xc))) * my_sign(imag((1.0 + xc) / (1.0 - yc)))
                - 1.0i * M_PI * H1((-1.0 - xc) / (-1.0 + w3), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log((w3 - yc) / (-1.0 - xc))) * my_sign(imag((1.0 + xc) / (1.0 - yc))) + 1.0i * M_PI * H1(-w3inv, -ycinv) * power_of<2>(log(-w3 + yc)) * my_sign(-imycinv) - 1.0i * M_PI * H1(w3inv, -ycinv) * power_of<2>(log(w3 + yc)) * my_sign(-imycinv) + 1.0i * M_PI * H1(-w3inv, ycinv) * power_of<2>(log(-w3 - yc)) * my_sign(imycinv) - 1.0i * M_PI * H1(w3inv, ycinv) * power_of<2>(log(w3 - yc)) * my_sign(imycinv)
                + 1.0i * M_PI * power_of<2>(log(-(w3 / (-1.0 - w3)))) * my_sign(imag(1.0 / (-1.0 - w3))) * T(1.0, -(xc / (-1.0 - xc)), -(w3 / (-1.0 - w3))) - 1.0i * M_PI * power_of<2>(log(w3 / (-1.0 + w3))) * my_sign(imag(1.0 / (-1.0 + w3))) * T(1.0, -(xc / (-1.0 - xc)), w3 / (-1.0 + w3)) - 1.0i * M_PI * power_of<2>(log((-w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 + w3) / (-1.0 - xc))) * T(1.0, -(xc / (-1.0 - xc)), (-w3 - xc) / (-1.0 - xc)) + 1.0i * M_PI * power_of<2>(log((w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 - w3) / (-1.0 - xc))) * T(1.0, -(xc / (-1.0 - xc)), (w3 - xc) / (-1.0 - xc))
                - 1.0i * M_PI * power_of<2>(log(-(w3 / (1.0 - w3)))) * my_sign(-imag(1.0 / (1.0 - w3))) * T(1.0, -(xc / (1.0 - xc)), -(w3 / (1.0 - w3))) + 1.0i * M_PI * power_of<2>(log(w3 / (1.0 + w3))) * my_sign(-imag(1.0 / (1.0 + w3))) * T(1.0, -(xc / (1.0 - xc)), w3 / (1.0 + w3)) + 1.0i * M_PI * power_of<2>(log((-w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 + w3) / (1.0 - xc))) * T(1.0, -(xc / (1.0 - xc)), (-w3 - xc) / (1.0 - xc)) - 1.0i * M_PI * power_of<2>(log((w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 - w3) / (1.0 - xc))) * T(1.0, -(xc / (1.0 - xc)), (w3 - xc) / (1.0 - xc)) + 1.0i * M_PI * power_of<2>(log(-(w3 / (-1.0 - w3)))) * my_sign(imag(1.0 / (-1.0 - w3))) * T(1.0, xc / (-1.0 + xc), -(w3 / (-1.0 - w3)))
                - 1.0i * M_PI * power_of<2>(log(w3 / (-1.0 + w3))) * my_sign(imag(1.0 / (-1.0 + w3))) * T(1.0, xc / (-1.0 + xc), w3 / (-1.0 + w3)) - 1.0i * M_PI * power_of<2>(log((-w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 + w3) / (-1.0 + xc))) * T(1.0, xc / (-1.0 + xc), (-w3 + xc) / (-1.0 + xc)) + 1.0i * M_PI * power_of<2>(log((w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 - w3) / (-1.0 + xc))) * T(1.0, xc / (-1.0 + xc), (w3 + xc) / (-1.0 + xc)) - 1.0i * M_PI * power_of<2>(log(-(w3 / (1.0 - w3)))) * my_sign(-imag(1.0 / (1.0 - w3))) * T(1.0, xc / (1.0 + xc), -(w3 / (1.0 - w3))) + 1.0i * M_PI * power_of<2>(log(w3 / (1.0 + w3))) * my_sign(-imag(1.0 / (1.0 + w3))) * T(1.0, xc / (1.0 + xc), w3 / (1.0 + w3))
                + 1.0i * M_PI * power_of<2>(log((-w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 + w3) / (1.0 + xc))) * T(1.0, xc / (1.0 + xc), (-w3 + xc) / (1.0 + xc)) - 1.0i * M_PI * power_of<2>(log((w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 - w3) / (1.0 + xc))) * T(1.0, xc / (1.0 + xc), (w3 + xc) / (1.0 + xc)) + 1.0i * M_PI * power_of<2>(log((1.0 - w3) / 2.0)) * my_sign(imw3 / 2.0) * T(1.0, (1.0 - yc) / 2.0, (1.0 - w3) / 2.0) - (2.0 * 1.0i) * M_PI * log((1.0 - w3) / 2.0) * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 - w3) / 2.0) - 1.0i * M_PI * power_of<2>(log((w3 - yc) / (1.0 + w3))) * my_sign(imag((1.0 + yc) / (1.0 + w3))) * T(1.0, (1.0 - yc) / 2.0, (w3 - yc) / (1.0 + w3))
                - 1.0i * M_PI * power_of<2>(log(1.0 - w3)) * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * log(1.0 - w3) * log((-w3 - xc) / (-1.0 - xc)) * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * log(1.0 - w3) * log((-w3 + xc) / (-1.0 + xc)) * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3) + dilog((-1.0 + w3) / (-1.0 - xc)) * (log(w3 / (-1.0 + w3)) - log((w3 - yc) / (-1.0 + w3)) + (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) + dilog((-1.0 + w3) / (-1.0 + xc)) * (log(w3 / (-1.0 + w3)) - log((w3 - yc) / (-1.0 + w3)) + (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3))
                + (2.0 * 1.0i) * M_PI * log((1.0 - w3) / 2.0) * log(1.0 + w3) * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3) - (2.0 * 1.0i) * M_PI * log(1.0 + w3) * log((w3 - xc) / (-1.0 - xc)) * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3) - (2.0 * 1.0i) * M_PI * log(1.0 + w3) * log((w3 + xc) / (-1.0 + xc)) * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3) + dilog((-1.0 - w3) / (-1.0 - xc)) * (-log(-(w3 / (-1.0 - w3))) + log((-w3 - yc) / (-1.0 - w3)) - (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) + dilog((-1.0 - w3) / (-1.0 + xc)) * (-log(-(w3 / (-1.0 - w3))) + log((-w3 - yc) / (-1.0 - w3)) - (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3))
                + li2r_xc * (log(-(w3 / (-1.0 - w3))) - log(w3 / (-1.0 + w3)) - log((-w3 - yc) / (-1.0 - w3)) + log((w3 - yc) / (-1.0 + w3)) - (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) + dilog(-1.0 / (-1.0 + xc)) * (log(-(w3 / (-1.0 - w3))) - log(w3 / (-1.0 + w3)) - log((-w3 - yc) / (-1.0 - w3)) + log((w3 - yc) / (-1.0 + w3)) - (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3))
                + (2.0 * 1.0i) * M_PI * ln1mxc * log((-w3 - xc) / (-1.0 - w3)) * my_sign(imxc) * t11myc1mxc - (2.0 * 1.0i) * M_PI * ln1mxc * log((w3 - xc) / (-1.0 + w3)) * my_sign(imxc) * t11myc1mxc - (2.0 * 1.0i) * M_PI * ln1mxc * log((-w3 - yc) / (-1.0 - w3)) * my_sign(imxc) * t11myc1mxc + (2.0 * 1.0i) * M_PI * ln1mxc * log((w3 - yc) / (-1.0 + w3)) * my_sign(imxc) * t11myc1mxc + (2.0 * 1.0i) * M_PI * ln1pxc * log((-w3 + xc) / (-1.0 - w3)) * my_sign(-imxc) * t11myc1pxc - (2.0 * 1.0i) * M_PI * ln1pxc * log((w3 + xc) / (-1.0 + w3)) * my_sign(-imxc) * t11myc1pxc
                - (2.0 * 1.0i) * M_PI * ln1pxc * log((-w3 - yc) / (-1.0 - w3)) * my_sign(-imxc) * t11myc1pxc + (2.0 * 1.0i) * M_PI * ln1pxc * log((w3 - yc) / (-1.0 + w3)) * my_sign(-imxc) * t11myc1pxc - 1.0i * M_PI * power_of<2>(log(1.0 - yc / w3)) * my_sign(imag(yc / w3)) * T(1.0, 1.0 - yc, 1.0 - yc / w3) + 1.0i * M_PI * power_of<2>(log(1.0 + yc / w3)) * my_sign(-imag(yc / w3)) * T(1.0, 1.0 - yc, 1.0 + yc / w3) + 1.0i * M_PI * power_of<2>(log((-w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 + w3) / (-1.0 - xc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (-w3 - xc) / (-1.0 - xc))
                - (2.0 * 1.0i) * M_PI * log((-w3 - xc) / (-1.0 - xc)) * ln1myc * my_sign(imag((1.0 + xc) / (1.0 - yc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (-w3 - xc) / (-1.0 - xc)) - 1.0i * M_PI * power_of<2>(log((w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 - w3) / (-1.0 - xc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (w3 - xc) / (-1.0 - xc)) + (2.0 * 1.0i) * M_PI * log((w3 - xc) / (-1.0 - xc)) * ln1myc * my_sign(imag((1.0 + xc) / (1.0 - yc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (w3 - xc) / (-1.0 - xc)) + 1.0i * M_PI * power_of<2>(log((-w3 - yc) / (-1.0 - w3))) * my_sign(imag((-1.0 + yc) / (-1.0 - w3))) * T(1.0, (-xc - yc) / (-1.0 - xc), (-w3 - yc) / (-1.0 - w3))
                - 1.0i * M_PI * power_of<2>(log((w3 - yc) / (-1.0 + w3))) * my_sign(imag((-1.0 + yc) / (-1.0 + w3))) * T(1.0, (-xc - yc) / (-1.0 - xc), (w3 - yc) / (-1.0 + w3)) - 1.0i * M_PI * power_of<2>(log((-w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 + w3) / (1.0 - xc))) * T(1.0, (-xc - yc) / (1.0 - xc), (-w3 - xc) / (1.0 - xc)) + (2.0 * 1.0i) * M_PI * log((-w3 - xc) / (1.0 - xc)) * ln1pyc * my_sign(imag((-1.0 + xc) / (-1.0 - yc))) * T(1.0, (-xc - yc) / (1.0 - xc), (-w3 - xc) / (1.0 - xc)) + 1.0i * M_PI * power_of<2>(log((w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 - w3) / (1.0 - xc))) * T(1.0, (-xc - yc) / (1.0 - xc), (w3 - xc) / (1.0 - xc))
                - (2.0 * 1.0i) * M_PI * log((w3 - xc) / (1.0 - xc)) * ln1pyc * my_sign(imag((-1.0 + xc) / (-1.0 - yc))) * T(1.0, (-xc - yc) / (1.0 - xc), (w3 - xc) / (1.0 - xc)) - 1.0i * M_PI * power_of<2>(log((-w3 - yc) / (1.0 - w3))) * my_sign(imag((1.0 + yc) / (1.0 - w3))) * T(1.0, (-xc - yc) / (1.0 - xc), (-w3 - yc) / (1.0 - w3)) + 1.0i * M_PI * power_of<2>(log((w3 - yc) / (1.0 + w3))) * my_sign(imag((1.0 + yc) / (1.0 + w3))) * T(1.0, (-xc - yc) / (1.0 - xc), (w3 - yc) / (1.0 + w3)) + 1.0i * M_PI * power_of<2>(log((-w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 + w3) / (-1.0 + xc))) * T(1.0, (xc - yc) / (-1.0 + xc), (-w3 + xc) / (-1.0 + xc))
                - (2.0 * 1.0i) * M_PI * log((-w3 + xc) / (-1.0 + xc)) * ln1myc * my_sign(imag((1.0 - xc) / (1.0 - yc))) * T(1.0, (xc - yc) / (-1.0 + xc), (-w3 + xc) / (-1.0 + xc)) - 1.0i * M_PI * power_of<2>(log((w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 - w3) / (-1.0 + xc))) * T(1.0, (xc - yc) / (-1.0 + xc), (w3 + xc) / (-1.0 + xc)) + (2.0 * 1.0i) * M_PI * log((w3 + xc) / (-1.0 + xc)) * ln1myc * my_sign(imag((1.0 - xc) / (1.0 - yc))) * T(1.0, (xc - yc) / (-1.0 + xc), (w3 + xc) / (-1.0 + xc)) + 1.0i * M_PI * power_of<2>(log((-w3 - yc) / (-1.0 - w3))) * my_sign(imag((-1.0 + yc) / (-1.0 - w3))) * T(1.0, (xc - yc) / (-1.0 + xc), (-w3 - yc) / (-1.0 - w3))
                - 1.0i * M_PI * power_of<2>(log((w3 - yc) / (-1.0 + w3))) * my_sign(imag((-1.0 + yc) / (-1.0 + w3))) * T(1.0, (xc - yc) / (-1.0 + xc), (w3 - yc) / (-1.0 + w3)) - 1.0i * M_PI * power_of<2>(log((-w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 + w3) / (1.0 + xc))) * T(1.0, (xc - yc) / (1.0 + xc), (-w3 + xc) / (1.0 + xc)) + (2.0 * 1.0i) * M_PI * log((-w3 + xc) / (1.0 + xc)) * ln1pyc * my_sign(imag((-1.0 - xc) / (-1.0 - yc))) * T(1.0, (xc - yc) / (1.0 + xc), (-w3 + xc) / (1.0 + xc)) + 1.0i * M_PI * power_of<2>(log((w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 - w3) / (1.0 + xc))) * T(1.0, (xc - yc) / (1.0 + xc), (w3 + xc) / (1.0 + xc))
                - (2.0 * 1.0i) * M_PI * log((w3 + xc) / (1.0 + xc)) * ln1pyc * my_sign(imag((-1.0 - xc) / (-1.0 - yc))) * T(1.0, (xc - yc) / (1.0 + xc), (w3 + xc) / (1.0 + xc)) - 1.0i * M_PI * power_of<2>(log((-w3 - yc) / (1.0 - w3))) * my_sign(imag((1.0 + yc) / (1.0 - w3))) * T(1.0, (xc - yc) / (1.0 + xc), (-w3 - yc) / (1.0 - w3)) + 1.0i * M_PI * power_of<2>(log((w3 - yc) / (1.0 + w3))) * my_sign(imag((1.0 + yc) / (1.0 + w3))) * T(1.0, (xc - yc) / (1.0 + xc), (w3 - yc) / (1.0 + w3)) + 1.0i * M_PI * power_of<2>(log((1.0 - w3) / 2.0)) * my_sign(imw3 / 2.0) * T(1.0, (1.0 + yc) / 2.0, (1.0 - w3) / 2.0) - (2.0 * 1.0i) * M_PI * log((1.0 - w3) / 2.0) * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 - w3) / 2.0)
                - 1.0i * M_PI * power_of<2>(log((-w3 - yc) / (-1.0 - w3))) * my_sign(imag((-1.0 + yc) / (-1.0 - w3))) * T(1.0, (1.0 + yc) / 2.0, (-w3 - yc) / (-1.0 - w3)) - 1.0i * M_PI * power_of<2>(log(1.0 - w3)) * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * log(1.0 - w3) * log((w3 - xc) / (1.0 - xc)) * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * log(1.0 - w3) * log((w3 + xc) / (1.0 + xc)) * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3) + dilog((1.0 - w3) / (1.0 - xc)) * (log(-(w3 / (1.0 - w3))) - log((-w3 - yc) / (1.0 - w3)) + (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3))
                + dilog((1.0 - w3) / (1.0 + xc)) * (log(-(w3 / (1.0 - w3))) - log((-w3 - yc) / (1.0 - w3)) + (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) + (2.0 * 1.0i) * M_PI * log((1.0 - w3) / 2.0) * log(1.0 + w3) * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3) - (2.0 * 1.0i) * M_PI * log(1.0 + w3) * log((-w3 - xc) / (1.0 - xc)) * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3) - (2.0 * 1.0i) * M_PI * log(1.0 + w3) * log((-w3 + xc) / (1.0 + xc)) * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3) + dilog((1.0 + w3) / (1.0 - xc)) * (-log(w3 / (1.0 + w3)) + log((w3 - yc) / (1.0 + w3)) - (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3))
                + dilog((1.0 + w3) / (1.0 + xc)) * (-log(w3 / (1.0 + w3)) + log((w3 - yc) / (1.0 + w3)) - (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) + li2half * (-log(-(w3 / (-1.0 - w3))) - log(w3 / (1.0 + w3)) + log((-w3 - yc) / (-1.0 - w3)) + log((w3 - yc) / (1.0 + w3)) + 4.0 * log(1.0 - yc / w3) + 4.0 * log(1.0 + yc / w3) - (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3) - (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3))
                + dilog((1.0 + w3) / 2.0) * (log(-(w3 / (-1.0 - w3))) + log(w3 / (1.0 + w3)) - log((-w3 - yc) / (-1.0 - w3)) - log((w3 - yc) / (1.0 + w3)) + (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3) + (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) + dilog(1.0 / (1.0 - xc)) * (-log(-(w3 / (1.0 - w3))) + log(w3 / (1.0 + w3)) + log((-w3 - yc) / (1.0 - w3)) - log((w3 - yc) / (1.0 + w3)) - (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3))
                + li2r_xc * (-log(-(w3 / (1.0 - w3))) + log(w3 / (1.0 + w3)) + log((-w3 - yc) / (1.0 - w3)) - log((w3 - yc) / (1.0 + w3)) - (2.0 * 1.0i) * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3) + (2.0 * 1.0i) * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) - (2.0 * 1.0i) * M_PI * ln1mxc * log((-w3 + xc) / (1.0 - w3)) * my_sign(imxc) * t11pyc1mxc + (2.0 * 1.0i) * M_PI * ln1mxc * log((w3 + xc) / (1.0 + w3)) * my_sign(imxc) * t11pyc1mxc + (2.0 * 1.0i) * M_PI * ln1mxc * log((-w3 - yc) / (1.0 - w3)) * my_sign(imxc) * t11pyc1mxc
                - (2.0 * 1.0i) * M_PI * ln1mxc * log((w3 - yc) / (1.0 + w3)) * my_sign(imxc) * t11pyc1mxc - (2.0 * 1.0i) * M_PI * log((-w3 - xc) / (1.0 - w3)) * ln1pxc * my_sign(-imxc) * t11pyc1pxc + (2.0 * 1.0i) * M_PI * log((w3 - xc) / (1.0 + w3)) * ln1pxc * my_sign(-imxc) * t11pyc1pxc + (2.0 * 1.0i) * M_PI * ln1pxc * log((-w3 - yc) / (1.0 - w3)) * my_sign(-imxc) * t11pyc1pxc - (2.0 * 1.0i) * M_PI * ln1pxc * log((w3 - yc) / (1.0 + w3)) * my_sign(-imxc) * t11pyc1pxc + 1.0i * M_PI * power_of<2>(log(1.0 - yc / w3)) * my_sign(imag(yc / w3)) * T(1.0, 1.0 + yc, 1.0 - yc / w3)
                - 1.0i * M_PI * power_of<2>(log(1.0 + yc / w3)) * my_sign(-imag(yc / w3)) * T(1.0, 1.0 + yc, 1.0 + yc / w3) + (2.0 * 1.0i) * M_PI * log(1.0 - w3 * xcinv) * ln1mxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 - w3 * xcinv) - (2.0 * 1.0i) * M_PI * log(1.0 - w3 * xcinv) * ln1pxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 - w3 * xcinv) - (2.0 * 1.0i) * M_PI * log(1.0 + w3 * xcinv) * ln1mxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 + w3 * xcinv) + (2.0 * 1.0i) * M_PI * log(1.0 + w3 * xcinv) * ln1pxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 + w3 * xcinv) + (2.0 * 1.0i) * M_PI * log(1.0 - w3 * xcinv) * ln1mxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 - w3 * xcinv)
                - (2.0 * 1.0i) * M_PI * log(1.0 - w3 * xcinv) * ln1pxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 - w3 * xcinv) - (2.0 * 1.0i) * M_PI * log(1.0 + w3 * xcinv) * ln1mxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 + w3 * xcinv) + (2.0 * 1.0i) * M_PI * log(1.0 + w3 * xcinv) * ln1pxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 + w3 * xcinv) + pisqu * ((-1.0 / 6.0) * log((1.0 + xc) / w3) + log(((-1.0 + w3) * (1.0 + xc)) / (w3 + xc)) / 6.0 - log(((-1.0 + w3) * (1.0 + xc)) / ((w3 + xc) * (1.0 - yc))) / 6.0 + log((1.0 + xc) / (w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11myc1pxc * T(p(-xc, 1.0 - yc), 1.0 - yc, 1.0 - w3))
                - pisqu * ((-1.0 / 6.0) * log(-((1.0 + xc) / w3)) + log(((-1.0 - w3) * (1.0 + xc)) / (-w3 + xc)) / 6.0 - log(((-1.0 - w3) * (1.0 + xc)) / ((-w3 + xc) * (1.0 - yc))) / 6.0 + log((1.0 + xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11myc1pxc * T(p(-xc, 1.0 - yc), 1.0 - yc, 1.0 + w3)) + pisqu * ((-1.0 / 6.0) * log(-((-1.0 - xc) / w3)) + log(((1.0 - w3) * (-1.0 - xc)) / (w3 + xc)) / 6.0 - log(((1.0 - w3) * (-1.0 - xc)) / ((-w3 - xc) * (-1.0 - yc))) / 6.0 + log((-1.0 - xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11pyc1pxc * T(p(-xc, 1.0 + yc), 1.0 + yc, 1.0 - w3))
                - pisqu * ((-1.0 / 6.0) * log((-1.0 - xc) / w3) + log(((1.0 + w3) * (-1.0 - xc)) / (-w3 + xc)) / 6.0 - log(((1.0 + w3) * (-1.0 - xc)) / ((w3 - xc) * (-1.0 - yc))) / 6.0 + log((-1.0 - xc) / (w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11pyc1pxc * T(p(-xc, 1.0 + yc), 1.0 + yc, 1.0 + w3)) + pisqu * ((-1.0 / 6.0) * log((1.0 - xc) / w3) + log(((-1.0 + w3) * (1.0 - xc)) / (w3 - xc)) / 6.0 - log(((-1.0 + w3) * (1.0 - xc)) / ((w3 - xc) * (1.0 - yc))) / 6.0 + log((1.0 - xc) / (w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11myc1mxc * T(p(xc, 1.0 - yc), 1.0 - yc, 1.0 - w3))
                - pisqu * ((-1.0 / 6.0) * log(-((1.0 - xc) / w3)) + log(((-1.0 - w3) * (1.0 - xc)) / (-w3 - xc)) / 6.0 - log(((-1.0 - w3) * (1.0 - xc)) / ((-w3 - xc) * (1.0 - yc))) / 6.0 + log((1.0 - xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11myc1mxc * T(p(xc, 1.0 - yc), 1.0 - yc, 1.0 + w3)) + pisqu * ((-1.0 / 6.0) * log(-((-1.0 + xc) / w3)) + log(((1.0 - w3) * (-1.0 + xc)) / (w3 - xc)) / 6.0 - log(((1.0 - w3) * (-1.0 + xc)) / ((-w3 + xc) * (-1.0 - yc))) / 6.0 + log((-1.0 + xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11pyc1mxc * T(p(xc, 1.0 + yc), 1.0 + yc, 1.0 - w3))
                - pisqu * ((-1.0 / 6.0) * log((-1.0 + xc) / w3) + log(((1.0 + w3) * (-1.0 + xc)) / (-w3 - xc)) / 6.0 - log(((1.0 + w3) * (-1.0 + xc)) / ((w3 + xc) * (-1.0 - yc))) / 6.0 + log((-1.0 + xc) / (w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11pyc1mxc * T(p(xc, 1.0 + yc), 1.0 + yc, 1.0 + w3)) + (2.0 * pisqu * ln2 - 4.0 * power_of<3>(ln2) - 21.0 * zeta3) / 12.0 - zeta3 / 4.0)) / denom6;

            return f27cPart1 + f27cPart2 + f27cPart3 + f27cPart4 + f27cPart5;
        }

        complex<double> f29c(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xc = clp.xc;
            const complex<double> yc = clp.yc;
            const complex<double> w3 = (2.0 * xc * xc) / (1.0 + xc * xc);

            const complex<double> xcinv = 1.0 / xc;
            const complex<double> ycinv = 1.0 / yc;
            const complex<double> w3inv = 1.0 / w3;

            // Imaginary parts
            const double imxc = imag(xc);
            const double imycinv = imag(ycinv);
            const double imw3 = imag(w3);

            // Polynomials in xc and yc
            const complex<double> xc2 = power_of<2>(xc);
            const complex<double> xc3 = power_of<3>(xc);
            const complex<double> xc4 = power_of<4>(xc);
            const complex<double> xc6 = power_of<6>(xc);
            const complex<double> xc8 = power_of<8>(xc);

            const complex<double> yc2 = power_of<2>(yc);
            const complex<double> yc3 = power_of<3>(yc);
            const complex<double> yc4 = power_of<4>(yc);
            const complex<double> yc5 = power_of<5>(yc);
            const complex<double> yc6 = power_of<6>(yc);
            const complex<double> yc7 = power_of<7>(yc);
            const complex<double> yc8 = power_of<8>(yc);

            const complex<double> term1 = (-4.0 + yc2 * (9.0 + (4.0 * 1.0i) * M_PI + ln256));
            const complex<double> lognum1 = (xc2 * yc2 * (-27.0 - 13.0 * yc4 + (8.0 * 1.0i) * M_PI * (-1.0 + 3.0 * yc2) - 16.0 * ln2 + 12.0 * yc2 * (5.0 + log(16.0))) + yc4 * (11.0 - (4.0 * 1.0i) * M_PI * (-1.0 + 3.0 * yc2) + ln256 - 3.0 * yc2 * (7.0 + ln256)) + xc4 * (10.0 - 5.0 * yc4 + 6.0 * yc6 - (4.0 * 1.0i) * M_PI * (-1.0 + 3.0 * yc2) + ln256 - 3.0 * yc2 * (7.0 + ln256)));
            const complex<double> lognum2 = (xc3 * (-2.0 + 3.0 * yc2 - 4.0 * yc4 + 3.0 * yc6) - xc * (yc2 - 6.0 * yc4 + 5.0 * yc6) + xc4 * (4.0 + yc2 * (-9.0 - (4.0 * 1.0i) * M_PI - 8.0 * ln2)) + yc4 * (4.0 + yc2 * (-9.0 - (4.0 * 1.0i) * M_PI - 8.0 * ln2)) + 2.0 * xc2 * yc2 * term1);
            const complex<double> lognum3 = (xc3 * (2.0 - 3.0 * yc2 + 4.0 * yc4 - 3.0 * yc6) + xc * (yc2 - 6.0 * yc4 + 5.0 * yc6) + xc4 * (4.0 + yc2 * (-9.0 - (4.0 * 1.0i) * M_PI - 8.0 * ln2)) + yc4 * (4.0 + yc2 * (-9.0 - (4.0 * 1.0i) * M_PI - 8.0 * ln2)) + 2.0 * xc2 * yc2 * term1);
            const complex<double> lognum4 = (xc2 * (22.0 * pisqu * yc2 + 3.0 * (48.0 + 6.0 * yc4 + 32.0 * ln2 - yc2 * (57.0 + 72.0 * ln2 + 32.0 * ln2squ)) - (12.0 * 1.0i) * M_PI * (-4.0 + yc2 * (9.0 + ln256))) + yc2 * (-22.0 * pisqu * yc2 + 3.0 * (-42.0 - 32.0 * ln2 + yc2 * (45.0 + 72.0 * ln2 + 32.0 * ln2squ)) + (12.0 * 1.0i) * M_PI * (-4.0 + yc2 * (9.0 + ln256))));

            const complex<double> num1 = (-(yc4 * (1.0 + 6.0 * yc + 8.0 * yc2 + 2.0 * yc3 + 15.0 * yc4)) + xc4 * yc * (-2.0 + 34.0 * yc + 34.0 * yc2 - 57.0 * yc3 + 40.0 * yc4 + 70.0 * yc5 + 32.0 * yc6 + 9.0 * yc7) + xc2 * (yc2 + 14.0 * yc3 - 8.0 * yc4 - 46.0 * yc5 + 23.0 * yc6 - 8.0 * yc7 - 8.0 * yc8) + xc6 * (-12.0 - 10.0 * yc + 25.0 * yc2 + 4.0 * yc3 - 26.0 * yc4 - 56.0 * yc5 - 33.0 * yc6 + 6.0 * yc7 + 6.0 * yc8));
            const complex<double> num2 = (yc6 - 9.0 * yc8 + 2.0 * xc4 * yc4 * (-5.0 + 28.0 * yc2 + yc4) - 2.0 * xc2 * yc4 * (2.0 - 9.0 * yc2 + 7.0 * yc4) + 2.0 * xc6 * yc2 * (10.0 - 38.0 * yc2 - 7.0 * yc4 + 3.0 * yc6) + xc8 * (-4.0 + 4.0 * yc2 + 42.0 * yc4 - 21.0 * yc6 + 3.0 * yc8));
            const complex<double> num3 = (-(yc2 * (1.0 + 15.0 * yc4)) + xc2 * (-2.0 + 34.0 * yc4 - 32.0 * yc6) + 2.0 * xc6 * (9.0 - 92.0 * yc2 + 15.0 * yc4 + 4.0 * yc6) - 2.0 * xc4 * (-3.0 + 11.0 * yc2 - 65.0 * yc4 + 9.0 * yc6) + xc8 * (26.0 + 63.0 * yc2 - 50.0 * yc4 + 9.0 * yc6));
            const complex<double> num4 = (-1.0 + yc) * (1.0 + yc) * (yc2 + yc4 + 2.0 * xc2 * (1.0 + yc2) - 2.0 * xc4 * (3.0 + yc4) + 2.0 * xc6 * (-9.0 + 3.0 * yc2 + 4.0 * yc4) + xc8 * (38.0 - 41.0 * yc2 + 9.0 * yc4));
            const complex<double> num5 = (yc4 * (-1.0 + 6.0 * yc - 8.0 * yc2 + 2.0 * yc3 - 15.0 * yc4) + xc4 * yc * (2.0 + 34.0 * yc - 34.0 * yc2 - 57.0 * yc3 - 40.0 * yc4 + 70.0 * yc5 - 32.0 * yc6 + 9.0 * yc7) + xc2 * (yc2 - 14.0 * yc3 - 8.0 * yc4 + 46.0 * yc5 + 23.0 * yc6 + 8.0 * yc7 - 8.0 * yc8) + xc6 * (-12.0 + 10.0 * yc + 25.0 * yc2 - 4.0 * yc3 - 26.0 * yc4 + 56.0 * yc5 - 33.0 * yc6 - 6.0 * yc7 + 6.0 * yc8));
            const complex<double> num6 = (yc6 - 5.0 * yc8 - 2.0 * xc4 * yc4 * (5.0 - 18.0 * yc2 + yc4) - 2.0 * xc2 * yc4 * (2.0 - 7.0 * yc2 + 5.0 * yc4) + 2.0 * xc6 * yc2 * (10.0 - 30.0 * yc2 + 3.0 * yc4 + yc6) + xc8 * (-4.0 + 4.0 * yc2 + 26.0 * yc4 - 17.0 * yc6 + 3.0 * yc8));
            const complex<double> num7 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc) * (yc2 - 17.0 * yc4 - 2.0 * xc2 * (2.0 - 19.0 * yc2 + yc4) + xc4 * (-12.0 - 7.0 * yc2 + 3.0 * yc4));
            const complex<double> num8 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc) * (yc2 + 7.0 * yc4 - 2.0 * xc2 * (2.0 + 5.0 * yc2 + yc4) + xc4 * (12.0 - 7.0 * yc2 + 3.0 * yc4));
            const complex<double> num9 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc) * (yc2 + 3.0 * yc4 - 2.0 * xc2 * (2.0 + yc2 + yc4) + xc4 * (8.0 - 7.0 * yc2 + 3.0 * yc4));
            const complex<double> num10 = (-1.0 + xc) * (1.0 + xc) * (-1.0 + yc) * (1.0 + yc);
            const complex<double> num11 = power_of<2>(-1.0 + xc) * power_of<2>(1.0 + xc) * power_of<2>(-1.0 + yc) * power_of<2>(1.0 + yc) * (yc2 + xc2 * (-4.0 + 3.0 * yc2));

            const complex<double> denom1 = (9.0 * power_of<2>(xc - yc) * yc3 * power_of<2>(xc + yc) * (yc + xc2 * (2.0 + yc)));
            const complex<double> denom2 = (9.0 * power_of<2>(xc - yc) * yc3 * power_of<2>(xc + yc) * (xc2 * (-2.0 + yc) + yc) * (yc + xc2 * (2.0 + yc)));
            const complex<double> denom3 = (9.0 * power_of<2>(xc - yc) * power_of<2>(xc + yc) * (xc2 * (-2.0 + yc) + yc) * (yc + xc2 * (2.0 + yc)));
            const complex<double> denom4 = (9.0 * power_of<2>(xc - yc) * yc3 * power_of<2>(xc + yc) * (xc2 * (-2.0 + yc) + yc));
            const complex<double> denom5 = (9.0 * power_of<2>(xc - yc) * yc3 * power_of<2>(xc + yc) * (xc2 * (-2.0 + yc) + yc) * (yc + xc2 * (2.0 + yc)));
            const complex<double> denom6 = (9.0 * power_of<3>(xc - yc) * power_of<3>(xc + yc));
            const complex<double> denom7 = (9.0 * (xc - yc) * (xc + yc));
            const complex<double> denom8 = (9.0 * power_of<2>(xc - yc) * yc3 * power_of<2>(xc + yc));
            const complex<double> denom9 = (9.0 * power_of<2>(xc - yc) * yc2 * power_of<2>(xc + yc));

            // Polylogarithms in xc and yc

            const complex<double> ln1pxc = log(1.0 + xc);
            const complex<double> ln1mxc = log(1.0 - xc);
            const complex<double> ln1pyc = log(1.0 + yc);
            const complex<double> ln1myc = log(1.0 - yc);
            const complex<double> ln1pxchalf = log((1.0 + xc) / 2.0); // MyLog[(1 + xc)/2]
            const complex<double> ln1mxchalf = log((1.0 - xc) / 2.0); // MyLog[(1 - xc)/2]
            const complex<double> ln1pychalf = log((1.0 + yc) / 2.0); // MyLog[(1 + yc)/2]
            const complex<double> ln1mychalf = log((1.0 - yc) / 2.0); // MyLog[(1 - yc)/2]

            const complex<double> lnr_xc1 = log(xc / (1.0 + xc));
            const complex<double> lnr_xc2 = log(xc / (-1.0 + xc));

            // Heaviside theta related functions
            const double t11myc1mxc = T(1.0, 1.0 - yc, 1.0 - xc); // MyT[1, 1 - yc, 1 - xc]
            const double t11pyc1mxc = T(1.0, 1.0 + yc, 1.0 - xc); // MyT[1, 1 + yc, 1 - xc]
            const double t11myc1pxc = T(1.0, 1.0 - yc, 1.0 + xc); // MyT[1, 1 - yc, 1 + xc]
            const double t11pyc1pxc = T(1.0, 1.0 + yc, 1.0 + xc); // MyT[1, 1 + yc, 1 + xc]

            const complex<double> f29cPart1 = -(2.0 * lognum4) / (3.0 * (xc - yc) * yc2 * (xc + yc)) + (3.0 * num11 * pisqu * ln4) / (2.0 * denom6) - (3.0 * num9 * pisqu * ln4) / denom6 + ((-576.0 * num10) / denom7 - (36.0 * num11) / denom6 - (36.0 * num7) / denom6 + (216.0 * num9) / denom6) * li3half + (72.0 * num11 * trilog(1.0 - w3inv)) / denom6 - (36.0 * num11 * trilog(1.0 + w3inv)) / denom6 - (18.0 * num11 * trilog(-1.0 / (-1.0 - w3))) / denom6 + (36.0 * num11 * trilog(1.0 / (1.0 - w3))) / denom6 + (36.0 * num11 * trilog(-1.0 / (-1.0 + w3))) / denom6 + (36.0 * num11 * trilog(w3inv)) / denom6 - (36.0 * num11 * trilog(-w3)) / denom6
                + (36.0 * num11 * trilog(w3)) / denom6 - (18.0 * num11 * trilog(1.0 / (1.0 + w3))) / denom6 - (36.0 * num11 * trilog((1.0 + w3) / (1.0 - w3))) / denom6 - (36.0 * num11 * trilog((1.0 + w3) / (2.0 * w3))) / denom6 + ((288.0 * num10) / denom7 - (144.0 * num9) / denom6) * trilog(1.0 - xcinv) + ((288.0 * num10) / denom7 - (144.0 * num9) / denom6) * trilog(1.0 + xcinv) + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(-1.0 / (-1.0 - xc)) + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(1.0 / (1.0 - xc)) - (72.0 * num9 * trilog(1.0 - xc)) / denom6
                + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(-1.0 / (-1.0 + xc)) + ((288.0 * num10) / denom7 - (144.0 * num9) / denom6) * trilog(-xcinv) + ((288.0 * num10) / denom7 - (144.0 * num9) / denom6) * trilog(xcinv) + ((-288.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((-0.5) * ((1.0 - xc) * xcinv)) - (72.0 * num9 * trilog(-xc)) / denom6 - (72.0 * num9 * trilog(xc)) / denom6 + (18.0 * num11 * trilog(((-1.0 - w3) * xc) / (w3 * (-1.0 - xc)))) / denom6 - (18.0 * num11 * trilog(-(((-1.0 + w3) * xc) / (w3 * (-1.0 - xc))))) / denom6
                - (18.0 * num11 * trilog(((1.0 - w3) * xc) / (w3 * (1.0 - xc)))) / denom6 + (18.0 * num11 * trilog(-(((1.0 + w3) * xc) / (w3 * (1.0 - xc))))) / denom6 + (18.0 * num11 * trilog(((-1.0 - w3) * xc) / (-w3 - xc))) / denom6 + (18.0 * num11 * trilog(-(((1.0 + w3) * xc) / (-w3 - xc)))) / denom6 - (18.0 * num11 * trilog(-(((1.0 - w3) * xc) / (w3 - xc)))) / denom6 - (18.0 * num11 * trilog(((-1.0 + w3) * xc) / (w3 - xc))) / denom6 - (72.0 * num9 * trilog(xc / (-1.0 + xc))) / denom6 + (72.0 * num9 * trilog((2.0 * xc) / (-1.0 + xc))) / denom6
                + (18.0 * num11 * trilog(-(((-1.0 - w3) * xc) / (w3 * (-1.0 + xc))))) / denom6 - (18.0 * num11 * trilog(((-1.0 + w3) * xc) / (w3 * (-1.0 + xc)))) / denom6 + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(1.0 / (1.0 + xc)) + ((-288.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((1.0 - xc) / (1.0 + xc)) - (72.0 * num9 * trilog(xc / (1.0 + xc))) / denom6 + (72.0 * num9 * trilog((2.0 * xc) / (1.0 + xc))) / denom6 - (18.0 * num11 * trilog(-(((1.0 - w3) * xc) / (w3 * (1.0 + xc))))) / denom6 + (18.0 * num11 * trilog(((1.0 + w3) * xc) / (w3 * (1.0 + xc)))) / denom6
                - (72.0 * num9 * trilog(1.0 + xc)) / denom6 + ((-288.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((1.0 + xc) / (1.0 - xc)) + ((-288.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((1.0 + xc) / (2.0 * xc)) + (18.0 * num11 * trilog(-(((-1.0 - w3) * xc) / (-w3 + xc)))) / denom6 + (18.0 * num11 * trilog(((1.0 + w3) * xc) / (-w3 + xc))) / denom6 - (18.0 * num11 * trilog(((1.0 - w3) * xc) / (w3 + xc))) / denom6 - (18.0 * num11 * trilog(-(((-1.0 + w3) * xc) / (w3 + xc)))) / denom6 - (72.0 * num9 * trilog((-1.0 - yc) / (-1.0 - xc))) / denom6
                - (72.0 * num9 * trilog((-1.0 - yc) / (-1.0 + xc))) / denom6 + (18.0 * num11 * trilog(((-1.0 - w3) * (-1.0 - yc)) / ((1.0 - w3) * (1.0 - yc)))) / denom6 + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(((-1.0 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (1.0 - yc))) + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(((-1.0 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (1.0 - yc))) + ((288.0 * num10) / denom7 + (18.0 * num11) / denom6 + (54.0 * num7) / denom6 + (72.0 * num8) / denom6 - (144.0 * num9) / denom6) * trilog((1.0 - yc) / 2.0) + ((-18.0 * num11) / denom6 + (36.0 * num8) / denom6) * trilog(1.0 - yc)
                - (72.0 * num9 * trilog((1.0 - yc) / (1.0 - xc))) / denom6 - (72.0 * num9 * trilog((1.0 - yc) / (1.0 + xc))) / denom6 + (18.0 * num11 * trilog(((1.0 + w3) * (1.0 - yc)) / ((-1.0 + w3) * (-1.0 - yc)))) / denom6 + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(((1.0 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-1.0 - yc))) + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(((1.0 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-1.0 - yc))) + (18.0 * num11 * trilog((-1.0 - w3) / (-w3 - yc))) / denom6 - (36.0 * num11 * trilog((1.0 - w3) / (-w3 - yc))) / denom6
                - (36.0 * num11 * trilog((-1.0 - yc) / (-w3 - yc))) / denom6 + (18.0 * num11 * trilog((-0.5) * (((-1.0 - w3) * (-1.0 - yc)) / (-w3 - yc)))) / denom6 + (18.0 * num11 * trilog((1.0 - yc) / (-w3 - yc))) / denom6 - (18.0 * num11 * trilog(-((w3 * (1.0 - yc)) / (-w3 - yc)))) / denom6 - (36.0 * num11 * trilog((-1.0 + w3) / (w3 - yc))) / denom6 + (18.0 * num11 * trilog((1.0 + w3) / (w3 - yc))) / denom6 + (18.0 * num11 * trilog((-1.0 - yc) / (w3 - yc))) / denom6 - (36.0 * num11 * trilog((1.0 - yc) / (w3 - yc))) / denom6
                + (18.0 * num11 * trilog((w3 * (1.0 - yc)) / (w3 - yc))) / denom6 + (18.0 * num11 * trilog(((1.0 + w3) * (1.0 - yc)) / (2.0 * (w3 - yc)))) / denom6 + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((-1.0 - xc) / (-xc - yc)) + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((1.0 - xc) / (-xc - yc)) + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((-1.0 - yc) / (-xc - yc)) + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog((-0.5) * (((-1.0 - xc) * (-1.0 - yc)) / (-xc - yc))) + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((1.0 - yc) / (-xc - yc))
                + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(((1.0 - xc) * (1.0 - yc)) / (2.0 * (-xc - yc))) + (18.0 * num11 * trilog(((1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (-1.0 - yc)))) / denom6 - (18.0 * num11 * trilog(((1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (-1.0 - yc)))) / denom6 - (18.0 * num11 * trilog(((-1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (1.0 - yc)))) / denom6 + (18.0 * num11 * trilog(((-1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (1.0 - yc)))) / denom6 - (18.0 * num11 * trilog(((-1.0 - w3) * (-xc - yc)) / ((-1.0 - xc) * (-w3 - yc)))) / denom6
                + (18.0 * num11 * trilog(((1.0 - w3) * (-xc - yc)) / ((1.0 - xc) * (-w3 - yc)))) / denom6 + (18.0 * num11 * trilog(((-1.0 + w3) * (-xc - yc)) / ((-1.0 - xc) * (w3 - yc)))) / denom6 - (18.0 * num11 * trilog(((1.0 + w3) * (-xc - yc)) / ((1.0 - xc) * (w3 - yc)))) / denom6 + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((-1.0 + xc) / (xc - yc)) + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((1.0 + xc) / (xc - yc)) + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((-1.0 - yc) / (xc - yc))
                + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog((-0.5) * (((-1.0 + xc) * (-1.0 - yc)) / (xc - yc))) + ((-144.0 * num10) / denom7 + (72.0 * num9) / denom6) * trilog((1.0 - yc) / (xc - yc)) + ((144.0 * num10) / denom7 - (72.0 * num9) / denom6) * trilog(((1.0 + xc) * (1.0 - yc)) / (2.0 * (xc - yc))) + (18.0 * num11 * trilog(((1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (-1.0 - yc)))) / denom6 - (18.0 * num11 * trilog(((1.0 + w3) * (xc - yc)) / ((w3 - xc) * (-1.0 - yc)))) / denom6 - (18.0 * num11 * trilog(((-1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (1.0 - yc)))) / denom6
                + (18.0 * num11 * trilog(((-1.0 + w3) * (xc - yc)) / ((w3 - xc) * (1.0 - yc)))) / denom6 - (18.0 * num11 * trilog(((-1.0 - w3) * (xc - yc)) / ((-1.0 + xc) * (-w3 - yc)))) / denom6 + (18.0 * num11 * trilog(((1.0 - w3) * (xc - yc)) / ((1.0 + xc) * (-w3 - yc)))) / denom6 + (18.0 * num11 * trilog(((-1.0 + w3) * (xc - yc)) / ((-1.0 + xc) * (w3 - yc)))) / denom6 - (18.0 * num11 * trilog(((1.0 + w3) * (xc - yc)) / ((1.0 + xc) * (w3 - yc)))) / denom6 + (18.0 * num11 * trilog((-1.0 + yc) / (-1.0 - w3))) / denom6 - (36.0 * num11 * trilog((-1.0 + yc) / (-1.0 + w3))) / denom6
                + ((-144.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((-1.0 + yc) / (-1.0 - xc)) + ((-144.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((-1.0 + yc) / (-1.0 + xc)) + ((-18.0 * num11) / denom6 + (36.0 * num8) / denom6) * trilog(-yc) + ((-18.0 * num11) / denom6 + (36.0 * num8) / denom6) * trilog(yc) + ((-18.0 * num11) / denom6 + (36.0 * num8) / denom6) * trilog(yc / (-1.0 + yc)) + ((18.0 * num11) / denom6 - (72.0 * num8) / denom6) * trilog((2.0 * yc) / (-1.0 + yc)) + ((-18.0 * num11) / denom6 + (36.0 * num8) / denom6) * trilog(yc / (1.0 + yc)) + ((18.0 * num11) / denom6 - (72.0 * num8) / denom6) * trilog((2.0 * yc) / (1.0 + yc))
                + ((288.0 * num10) / denom7 + (18.0 * num11) / denom6 + (54.0 * num7) / denom6 + (72.0 * num8) / denom6 - (144.0 * num9) / denom6) * trilog((1.0 + yc) / 2.0) + ((-18.0 * num11) / denom6 + (36.0 * num8) / denom6) * trilog(1.0 + yc) - (36.0 * num11 * trilog((1.0 + yc) / (1.0 - w3))) / denom6 + (18.0 * num11 * trilog((1.0 + yc) / (1.0 + w3))) / denom6 + ((-144.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((1.0 + yc) / (1.0 - xc)) + ((-144.0 * num10) / denom7 + (144.0 * num9) / denom6) * trilog((1.0 + yc) / (1.0 + xc)) - (18.0 * num11 * trilog((w3 * (-1.0 - yc)) / (-w3 + yc))) / denom6
                + (18.0 * num11 * trilog(-((w3 * (-1.0 - yc)) / (w3 + yc)))) / denom6 - (18.0 * num11 * dilog(1.0 / (1.0 - w3)) * lnhalf) / denom6 - (18.0 * num11 * dilog(-1.0 / (-1.0 + w3)) * lnhalf) / denom6 - (36.0 * num11 * dilog(w3inv) * lnhalf) / denom6 + (36.0 * num11 * dilog((1.0 + w3) / (1.0 - w3)) * lnhalf) / denom6 + (36.0 * num11 * dilog((1.0 + w3) / (2.0 * w3)) * lnhalf) / denom6 + dilog((-0.5) * ((1.0 - xc) * xcinv)) * ((288.0 * num10 * lnhalf) / denom7 - (144.0 * num9 * lnhalf) / denom6)
                + dilog((1.0 - xc) / (1.0 + xc)) * ((288.0 * num10 * lnhalf) / denom7 - (144.0 * num9 * lnhalf) / denom6) + dilog((1.0 + xc) / (1.0 - xc)) * ((288.0 * num10 * lnhalf) / denom7 - (144.0 * num9 * lnhalf) / denom6) + dilog((1.0 + xc) / (2.0 * xc)) * ((288.0 * num10 * lnhalf) / denom7 - (144.0 * num9 * lnhalf) / denom6) + dilog(-xcinv) * ((-288.0 * num10 * lnhalf) / denom7 + (144.0 * num9 * lnhalf) / denom6) + dilog(xcinv) * ((-288.0 * num10 * lnhalf) / denom7 + (144.0 * num9 * lnhalf) / denom6) + (6.0 * num7 * pisqu * ln2) / denom6 + (12.0 * num9 * pisqu * ln2) / denom6
                - (6.0 * num11 * power_of<3>(ln2)) / denom6 - (12.0 * num7 * power_of<3>(ln2)) / denom6 - (12.0 * num9 * power_of<3>(ln2)) / denom6 + 144.0 * lnmuhat + (64.0 * 1.0i) * M_PI * lnmuhat - (64.0 * lnmuhat) / yc2 + 128.0 * ln2 * lnmuhat + 64.0 * power_of<2>(lnmuhat) - (18.0 * num11 * power_of<2>(lnhalf) * log(1.0 - w3inv)) / denom6 - (3.0 * num11 * power_of<3>(log((2.0 * (-1.0 - w3)) / (1.0 - w3)))) / denom6 + (6.0 * num11 * power_of<3>(log(-2.0 / w3))) / denom6 + (18.0 * num11 * power_of<2>(lnhalf) * log((-1.0 + w3) / (2.0 * w3))) / denom6 + (18.0 * num11 * power_of<2>(lnhalf) * log((-2.0 * w3) / (1.0 - w3))) / denom6
                - (9.0 * num11 * power_of<2>(lnhalf) * log(-(w3 / (1.0 - w3)))) / denom6 - (9.0 * num11 * power_of<2>(lnhalf) * log(w3 / (-1.0 + w3))) / denom6 - (3.0 * num11 * power_of<3>(log((-2.0 * (1.0 + w3)) / (1.0 - w3)))) / denom6 - (144.0 * num10 * power_of<2>(lnhalf) * log(1.0 - xcinv)) / denom7 + (72.0 * num9 * power_of<2>(lnhalf) * log(1.0 - xcinv)) / denom6 - (144.0 * num10 * power_of<2>(lnhalf) * log(1.0 + xcinv)) / denom7 + (72.0 * num9 * power_of<2>(lnhalf) * log(1.0 + xcinv)) / denom6 + (3.0 * num11 * power_of<3>(log(-((-1.0 - xc) / w3)))) / denom6 - (3.0 * num11 * power_of<3>(log((-1.0 - xc) / w3))) / denom6
                - (24.0 * num10 * power_of<3>(log((2.0 * (-1.0 - xc)) / (1.0 - xc)))) / denom7 + (12.0 * num9 * power_of<3>(log((2.0 * (-1.0 - xc)) / (1.0 - xc)))) / denom6 + (72.0 * lognum2 * ln1mxc) / denom9 + (3.0 * num9 * pisqu * ln1mxc) / denom6 + (72.0 * num9 * dilog(1.0 - xc) * ln1mxc) / denom6 + (72.0 * num9 * dilog(-xc) * ln1mxc) / denom6 - (36.0 * num3 * ln2 * ln1mxc) / denom3 - (18.0 * num9 * ln2squ * ln1mxc) / denom6 + (72.0 * num9 * (pisqu / 12.0 - ln2squ / 2.0) * ln1mxc) / denom6 - 64.0 * lnmuhat * ln1mxc + (18.0 * num4 * power_of<2>(ln1mxc)) / denom3 + (36.0 * num9 * ln2 * power_of<2>(ln1mxc)) / denom6
                - (6.0 * num9 * power_of<3>(ln1mxc)) / denom6 - (3.0 * num11 * power_of<3>(log(-((1.0 - xc) / w3)))) / denom6 + (3.0 * num11 * power_of<3>(log((1.0 - xc) / w3))) / denom6 + (3.0 * num11 * power_of<3>(log(((-1.0 - w3) * (1.0 - xc)) / (-w3 - xc)))) / denom6 - (3.0 * num11 * power_of<3>(log(((-1.0 + w3) * (1.0 - xc)) / (w3 - xc)))) / denom6 + (3.0 * num11 * power_of<3>(log(-((-1.0 + xc) / w3)))) / denom6 - (3.0 * num11 * power_of<3>(log((-1.0 + xc) / w3))) / denom6 + (3.0 * num11 * power_of<3>(log(((1.0 + w3) * (-1.0 + xc)) / (-w3 - xc)))) / denom6 - (3.0 * num11 * power_of<3>(log(((1.0 - w3) * (-1.0 + xc)) / (w3 - xc)))) / denom6
                + (48.0 * num10 * power_of<3>(log(-2.0 * xcinv))) / denom7 - (24.0 * num9 * power_of<3>(log(-2.0 * xcinv))) / denom6 + (48.0 * num10 * power_of<3>(log(2.0 * xcinv))) / denom7 - (24.0 * num9 * power_of<3>(log(2.0 * xcinv))) / denom6 + (144.0 * num10 * power_of<2>(lnhalf) * log((-0.5) * ((-1.0 - xc) * xcinv))) / denom7 - (72.0 * num9 * power_of<2>(lnhalf) * log((-0.5) * ((-1.0 - xc) * xcinv))) / denom6 + (144.0 * num10 * power_of<2>(lnhalf) * log((-1.0 + xc) / (2.0 * xc))) / denom7 - (72.0 * num9 * power_of<2>(lnhalf) * log((-1.0 + xc) / (2.0 * xc))) / denom6 + (16.0 * term1 * log(xc)) / yc2 + 128.0 * lnmuhat * log(xc) - 64.0 * ln1mxc * log(xc) + (36.0 * num9 * power_of<2>(ln1mxc) * log(xc)) / denom6 + 64.0 * power_of<2>(log(xc))
                - (18.0 * num11 * dilog(((-1.0 - w3) * xc) / (w3 * (-1.0 - xc))) * log(-(xc / (-1.0 - xc)))) / denom6 + (18.0 * num11 * dilog(-(((-1.0 + w3) * xc) / (w3 * (-1.0 - xc)))) * log(-(xc / (-1.0 - xc)))) / denom6 - (18.0 * num11 * dilog(-(((-1.0 - w3) * xc) / (-w3 + xc))) * log(-(xc / (-1.0 - xc)))) / denom6 + (18.0 * num11 * dilog(-(((-1.0 + w3) * xc) / (w3 + xc))) * log(-(xc / (-1.0 - xc)))) / denom6 - (72.0 * num10 * power_of<2>(lnhalf) * log(-(xc / (-1.0 - xc)))) / denom7 + (36.0 * num9 * power_of<2>(lnhalf) * log(-(xc / (-1.0 - xc)))) / denom6
                + (9.0 * num11 * log((-w3 - xc) / (w3 * (-1.0 - xc))) * power_of<2>(log(-(xc / (-1.0 - xc))))) / denom6 - (9.0 * num11 * log(-((w3 - xc) / (w3 * (-1.0 - xc)))) * power_of<2>(log(-(xc / (-1.0 - xc))))) / denom6 + (144.0 * num10 * power_of<2>(lnhalf) * log((-2.0 * xc) / (1.0 - xc))) / denom7 - (72.0 * num9 * power_of<2>(lnhalf) * log((-2.0 * xc) / (1.0 - xc))) / denom6 + (18.0 * num11 * dilog(((1.0 - w3) * xc) / (w3 * (1.0 - xc))) * log(-(xc / (1.0 - xc)))) / denom6 - (18.0 * num11 * dilog(-(((1.0 + w3) * xc) / (w3 * (1.0 - xc)))) * log(-(xc / (1.0 - xc)))) / denom6
                - (18.0 * num11 * dilog(-(((1.0 + w3) * xc) / (-w3 - xc))) * log(-(xc / (1.0 - xc)))) / denom6 + (18.0 * num11 * dilog(-(((1.0 - w3) * xc) / (w3 - xc))) * log(-(xc / (1.0 - xc)))) / denom6 - (72.0 * num10 * power_of<2>(lnhalf) * log(-(xc / (1.0 - xc)))) / denom7 + (36.0 * num9 * power_of<2>(lnhalf) * log(-(xc / (1.0 - xc)))) / denom6 - (9.0 * num11 * log((w3 * (-1.0 + xc)) / (-w3 - xc)) * power_of<2>(log(-(xc / (1.0 - xc))))) / denom6 + (9.0 * num11 * log(-((w3 * (-1.0 + xc)) / (w3 - xc))) * power_of<2>(log(-(xc / (1.0 - xc))))) / denom6
                - (18.0 * num11 * dilog(((-1.0 - w3) * xc) / (-w3 - xc)) * lnr_xc2) / denom6 + (18.0 * num11 * dilog(((-1.0 + w3) * xc) / (w3 - xc)) * lnr_xc2) / denom6 - (18.0 * num11 * dilog(-(((-1.0 - w3) * xc) / (w3 * (-1.0 + xc)))) * lnr_xc2) / denom6 + (18.0 * num11 * dilog(((-1.0 + w3) * xc) / (w3 * (-1.0 + xc))) * lnr_xc2) / denom6 - (72.0 * num10 * power_of<2>(lnhalf) * lnr_xc2) / denom7 + (36.0 * num9 * power_of<2>(lnhalf) * lnr_xc2) / denom6;

            const complex<double> f29cPart2 = + (9.0 * num11 * log(-(w3 / (-w3 - xc))) * power_of<2>(lnr_xc2)) / denom6 - (9.0 * num11 * log(-((w3 * (1.0 - xc)) / (-w3 - xc))) * power_of<2>(lnr_xc2)) / denom6 - (9.0 * num11 * log(w3 / (w3 - xc)) * power_of<2>(lnr_xc2)) / denom6 + (9.0 * num11 * log((w3 * (1.0 - xc)) / (w3 - xc)) * power_of<2>(lnr_xc2)) / denom6 - (24.0 * num10 * power_of<3>(log((-2.0 * (1.0 - xc)) / (1.0 + xc)))) / denom7 + (12.0 * num9 * power_of<3>(log((-2.0 * (1.0 - xc)) / (1.0 + xc)))) / denom6 - (24.0 * num10 * power_of<3>(log((2.0 * (-1.0 + xc)) / (1.0 + xc)))) / denom7 + (12.0 * num9 * power_of<3>(log((2.0 * (-1.0 + xc)) / (1.0 + xc)))) / denom6
                + (18.0 * num11 * dilog(-(((1.0 - w3) * xc) / (w3 * (1.0 + xc)))) * lnr_xc1) / denom6 - (18.0 * num11 * dilog(((1.0 + w3) * xc) / (w3 * (1.0 + xc))) * lnr_xc1) / denom6 - (18.0 * num11 * dilog(((1.0 + w3) * xc) / (-w3 + xc)) * lnr_xc1) / denom6 + (18.0 * num11 * dilog(((1.0 - w3) * xc) / (w3 + xc)) * lnr_xc1) / denom6 - (72.0 * num10 * power_of<2>(lnhalf) * lnr_xc1) / denom7 + (36.0 * num9 * power_of<2>(lnhalf) * lnr_xc1) / denom6 - (9.0 * num11 * log(-(w3 / (-w3 - xc))) * power_of<2>(lnr_xc1)) / denom6
                + (9.0 * num11 * log(w3 / (w3 - xc)) * power_of<2>(lnr_xc1)) / denom6 + (9.0 * num11 * log(-((-w3 - xc) / (w3 * (1.0 + xc)))) * power_of<2>(lnr_xc1)) / denom6 - (9.0 * num11 * log((w3 - xc) / (w3 * (1.0 + xc))) * power_of<2>(lnr_xc1)) / denom6 + dilog(-(xc / w3)) * ((-18.0 * num11 * log(-(xc / (-1.0 - xc)))) / denom6 + (18.0 * num11 * log(-(xc / (1.0 - xc)))) / denom6 + (18.0 * num11 * lnr_xc2) / denom6 - (18.0 * num11 * lnr_xc1) / denom6)
                + dilog(xc / w3) * ((18.0 * num11 * log(-(xc / (-1.0 - xc)))) / denom6 - (18.0 * num11 * log(-(xc / (1.0 - xc)))) / denom6 - (18.0 * num11 * lnr_xc2) / denom6 + (18.0 * num11 * lnr_xc1) / denom6) + (144.0 * num10 * power_of<2>(lnhalf) * log((2.0 * xc) / (1.0 + xc))) / denom7 - (72.0 * num9 * power_of<2>(lnhalf) * log((2.0 * xc) / (1.0 + xc))) / denom6 - (54.0 * num9 * power_of<2>(ln1mxc) * ln1pxchalf) / denom6 + (72.0 * lognum3 * ln1pxc) / denom9 + (3.0 * num9 * pisqu * ln1pxc) / denom6 + (72.0 * num9 * dilog(xc) * ln1pxc) / denom6 + (72.0 * num9 * dilog(1.0 + xc) * ln1pxc) / denom6
                - (36.0 * num3 * ln2 * ln1pxc) / denom3 - (18.0 * num9 * ln2squ * ln1pxc) / denom6 + (72.0 * num9 * (pisqu / 12.0 - ln2squ / 2.0) * ln1pxc) / denom6 - 64.0 * lnmuhat * ln1pxc + (72.0 * num9 * ln2 * ln1mxc * ln1pxc) / denom6 - 64.0 * log(xc) * ln1pxc + (18.0 * num4 * power_of<2>(ln1pxc)) / denom3 + (36.0 * num9 * ln2 * power_of<2>(ln1pxc)) / denom6 - (54.0 * num9 * ln1mxchalf * power_of<2>(ln1pxc)) / denom6 + (36.0 * num9 * log(-xc) * power_of<2>(ln1pxc)) / denom6 - (6.0 * num9 * power_of<3>(ln1pxc)) / denom6
                + dilog(-(xc / (w3 - xc))) * ((18.0 * num11 * ln1mxc) / denom6 + (18.0 * num11 * log(-(xc / (-1.0 - xc)))) / denom6 - (18.0 * num11 * log(-(xc / (1.0 - xc)))) / denom6 - (18.0 * num11 * ln1pxc) / denom6) + dilog(xc / (-w3 + xc)) * ((18.0 * num11 * ln1mxc) / denom6 - (18.0 * num11 * lnr_xc2) / denom6 + (18.0 * num11 * lnr_xc1) / denom6 - (18.0 * num11 * ln1pxc) / denom6)
                + dilog(-(xc / (-w3 - xc))) * ((-18.0 * num11 * ln1mxc) / denom6 - (18.0 * num11 * log(-(xc / (-1.0 - xc)))) / denom6 + (18.0 * num11 * log(-(xc / (1.0 - xc)))) / denom6 + (18.0 * num11 * ln1pxc) / denom6) + dilog(xc / (w3 + xc)) * ((-18.0 * num11 * ln1mxc) / denom6 + (18.0 * num11 * lnr_xc2) / denom6 - (18.0 * num11 * lnr_xc1) / denom6 + (18.0 * num11 * ln1pxc) / denom6) + dilog(-(xc / (-1.0 - xc))) * ((36.0 * num9 * ln1mxc) / denom6 - (36.0 * num9 * ln1pxc) / denom6)
                + dilog(xc / (1.0 + xc)) * ((36.0 * num9 * ln1mxc) / denom6 - (36.0 * num9 * ln1pxc) / denom6) + dilog((-xc + yc) / (1.0 - xc)) * ((36.0 * num9 * ln1mxc) / denom6 - (36.0 * num9 * ln1pxc) / denom6) + dilog((xc + yc) / (-1.0 + xc)) * ((36.0 * num9 * ln1mxc) / denom6 - (36.0 * num9 * ln1pxc) / denom6) + dilog(-(xc / (1.0 - xc))) * ((-36.0 * num9 * ln1mxc) / denom6 + (36.0 * num9 * ln1pxc) / denom6) + dilog(xc / (-1.0 + xc)) * ((-36.0 * num9 * ln1mxc) / denom6 + (36.0 * num9 * ln1pxc) / denom6)
                + dilog((-xc + yc) / (-1.0 - xc)) * ((-36.0 * num9 * ln1mxc) / denom6 + (36.0 * num9 * ln1pxc) / denom6) + dilog((xc + yc) / (1.0 + xc)) * ((-36.0 * num9 * ln1mxc) / denom6 + (36.0 * num9 * ln1pxc) / denom6) - (3.0 * num9 * ln1pxc * (pisqu - 6.0 * ln2squ - 12.0 * ln1mxc * ln1pxchalf + ln64 * ln1pxc)) / denom6 - (3.0 * num9 * ln1mxc * (pisqu - 6.0 * ln2squ + ln64 * ln1mxc - 12.0 * ln1mxchalf * ln1pxc)) / denom6 - (3.0 * num11 * power_of<3>(log(-((1.0 + xc) / w3)))) / denom6 + (3.0 * num11 * power_of<3>(log((1.0 + xc) / w3))) / denom6
                - (24.0 * num10 * power_of<3>(log((-2.0 * (1.0 + xc)) / (1.0 - xc)))) / denom7 + (12.0 * num9 * power_of<3>(log((-2.0 * (1.0 + xc)) / (1.0 - xc)))) / denom6 + (9.0 * num11 * power_of<2>(log(-(xc / (-1.0 - xc)))) * log(-(w3 / (-w3 + xc)))) / denom6 - (9.0 * num11 * power_of<2>(log(-(xc / (1.0 - xc)))) * log(-(w3 / (-w3 + xc)))) / denom6 - (9.0 * num11 * power_of<2>(lnr_xc1) * log((w3 * (-1.0 - xc)) / (-w3 + xc))) / denom6 + (3.0 * num11 * power_of<3>(log(((1.0 + w3) * (-1.0 - xc)) / (-w3 + xc)))) / denom6 + (3.0 * num11 * power_of<3>(log(((-1.0 - w3) * (1.0 + xc)) / (-w3 + xc)))) / denom6
                - (9.0 * num11 * power_of<2>(log(-(xc / (-1.0 - xc)))) * log(-((w3 * (1.0 + xc)) / (-w3 + xc)))) / denom6 + (9.0 * num11 * power_of<2>(log(-(xc / (1.0 - xc)))) * log(-((-w3 + xc) / (w3 * (1.0 - xc))))) / denom6 + (9.0 * num11 * power_of<2>(lnr_xc2) * log((-w3 + xc) / (w3 * (-1.0 + xc)))) / denom6 - (9.0 * num11 * power_of<2>(log(-(xc / (-1.0 - xc)))) * log(w3 / (w3 + xc))) / denom6 + (9.0 * num11 * power_of<2>(log(-(xc / (1.0 - xc)))) * log(w3 / (w3 + xc))) / denom6 - (3.0 * num11 * power_of<3>(log(((1.0 - w3) * (-1.0 - xc)) / (w3 + xc)))) / denom6
                + (9.0 * num11 * power_of<2>(lnr_xc1) * log(-((w3 * (-1.0 - xc)) / (w3 + xc)))) / denom6 - (3.0 * num11 * power_of<3>(log(((-1.0 + w3) * (1.0 + xc)) / (w3 + xc)))) / denom6 + (9.0 * num11 * power_of<2>(log(-(xc / (-1.0 - xc)))) * log((w3 * (1.0 + xc)) / (w3 + xc))) / denom6 - (9.0 * num11 * power_of<2>(log(-(xc / (1.0 - xc)))) * log((w3 + xc) / (w3 * (1.0 - xc)))) / denom6 - (9.0 * num11 * power_of<2>(lnr_xc2) * log(-((w3 + xc) / (w3 * (-1.0 + xc))))) / denom6 + (9.0 * num11 * power_of<2>(log(-(xc / (-1.0 - xc)))) * log(1.0 - xc / w3)) / denom6
                - (9.0 * num11 * power_of<2>(log(-(xc / (1.0 - xc)))) * log(1.0 - xc / w3)) / denom6 - (9.0 * num11 * power_of<2>(lnr_xc2) * log(1.0 - xc / w3)) / denom6 + (9.0 * num11 * power_of<2>(lnr_xc1) * log(1.0 - xc / w3)) / denom6 - (9.0 * num11 * power_of<2>(log(-(xc / (-1.0 - xc)))) * log(1.0 + xc / w3)) / denom6 + (9.0 * num11 * power_of<2>(log(-(xc / (1.0 - xc)))) * log(1.0 + xc / w3)) / denom6 + (9.0 * num11 * power_of<2>(lnr_xc2) * log(1.0 + xc / w3)) / denom6 - (9.0 * num11 * power_of<2>(lnr_xc1) * log(1.0 + xc / w3)) / denom6 + (3.0 * num11 * power_of<3>(log((-2.0 * (1.0 + w3)) / ((-1.0 + w3) * (-1.0 - yc))))) / denom6
                + (24.0 * num10 * power_of<3>(log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc))))) / denom7 - (12.0 * num9 * power_of<3>(log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc))))) / denom6 + (3.0 * num11 * power_of<3>(log(((1.0 - w3) * (-1.0 - xc)) / ((-w3 - xc) * (-1.0 - yc))))) / denom6 - (3.0 * num11 * power_of<3>(log(((1.0 + w3) * (-1.0 - xc)) / ((w3 - xc) * (-1.0 - yc))))) / denom6 + (24.0 * num10 * power_of<3>(log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc))))) / denom7 - (12.0 * num9 * power_of<3>(log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc))))) / denom6 + (3.0 * num11 * power_of<3>(log(((1.0 - w3) * (-1.0 + xc)) / ((-w3 + xc) * (-1.0 - yc))))) / denom6
                - (3.0 * num11 * power_of<3>(log(((1.0 + w3) * (-1.0 + xc)) / ((w3 + xc) * (-1.0 - yc))))) / denom6 + (3.0 * num11 * power_of<3>(log((2.0 * (-1.0 - w3)) / ((1.0 - w3) * (1.0 - yc))))) / denom6 + (24.0 * num10 * power_of<3>(log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc))))) / denom7 - (12.0 * num9 * power_of<3>(log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc))))) / denom6 - (3.0 * num11 * power_of<3>(log(((-1.0 - w3) * (1.0 - xc)) / ((-w3 - xc) * (1.0 - yc))))) / denom6 + (3.0 * num11 * power_of<3>(log(((-1.0 + w3) * (1.0 - xc)) / ((w3 - xc) * (1.0 - yc))))) / denom6 + (24.0 * num10 * power_of<3>(log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc))))) / denom7
                - (12.0 * num9 * power_of<3>(log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc))))) / denom6 - (3.0 * num11 * power_of<3>(log(((-1.0 - w3) * (1.0 + xc)) / ((-w3 + xc) * (1.0 - yc))))) / denom6 + (3.0 * num11 * power_of<3>(log(((-1.0 + w3) * (1.0 + xc)) / ((w3 + xc) * (1.0 - yc))))) / denom6 - (18.0 * num11 * dilog(((1.0 + w3) * (1.0 - yc)) / ((-1.0 + w3) * (-1.0 - yc))) * ln1mychalf) / denom6 - (18.0 * num11 * dilog(((1.0 + w3) * (1.0 - yc)) / (2.0 * (w3 - yc))) * ln1mychalf) / denom6 + dilog((1.0 - yc) / (-xc - yc)) * ((144.0 * num10 * ln1mychalf) / denom7 - (72.0 * num9 * ln1mychalf) / denom6)
                + dilog((1.0 - yc) / (xc - yc)) * ((144.0 * num10 * ln1mychalf) / denom7 - (72.0 * num9 * ln1mychalf) / denom6) + dilog(((1.0 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-1.0 - yc))) * ((-144.0 * num10 * ln1mychalf) / denom7 + (72.0 * num9 * ln1mychalf) / denom6) + dilog(((1.0 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-1.0 - yc))) * ((-144.0 * num10 * ln1mychalf) / denom7 + (72.0 * num9 * ln1mychalf) / denom6) + dilog(((1.0 - xc) * (1.0 - yc)) / (2.0 * (-xc - yc))) * ((-144.0 * num10 * ln1mychalf) / denom7 + (72.0 * num9 * ln1mychalf) / denom6)
                + dilog(((1.0 + xc) * (1.0 - yc)) / (2.0 * (xc - yc))) * ((-144.0 * num10 * ln1mychalf) / denom7 + (72.0 * num9 * ln1mychalf) / denom6) - (36.0 * lognum1 * ln1myc) / denom8 + (3.0 * num7 * pisqu * ln1myc) / (2.0 * denom6) - (18.0 * num11 * dilog(2.0 / (1.0 - w3)) * ln1myc) / denom6 + (18.0 * num11 * dilog((1.0 - xc) / (-w3 - xc)) * ln1myc) / denom6 - (18.0 * num11 * dilog((1.0 - xc) / (w3 - xc)) * ln1myc) / denom6 + (18.0 * num11 * dilog((1.0 + xc) / (-w3 + xc)) * ln1myc) / denom6 - (18.0 * num11 * dilog((1.0 + xc) / (w3 + xc)) * ln1myc) / denom6;

            const complex<double> f29cPart3 = + (36.0 * num9 * dilog((1.0 - yc) / (1.0 - xc)) * ln1myc) / denom6 + (36.0 * num9 * dilog((1.0 - yc) / (1.0 + xc)) * ln1myc) / denom6 - (18.0 * num11 * dilog((1.0 - yc) / (-w3 - yc)) * ln1myc) / denom6 + (18.0 * num11 * dilog(-((w3 * (1.0 - yc)) / (-w3 - yc))) * ln1myc) / denom6 - (18.0 * num11 * dilog((w3 * (1.0 - yc)) / (w3 - yc)) * ln1myc) / denom6 + (18.0 * num5 * ln2 * ln1myc) / denom4 - (9.0 * num7 * ln2squ * ln1myc) / denom6 + (18.0 * num11 * (pisqu / 12.0 - ln2squ / 2.0) * ln1myc) / denom6 - (32.0 * lnmuhat * ln1myc) / yc3
                + (96.0 * lnmuhat * ln1myc) / yc + (16.0 * ln1mxc * ln1myc) / yc3 - (48.0 * ln1mxc * ln1myc) / yc + (36.0 * num9 * ln2 * ln1mxc * ln1myc) / denom6 - (18.0 * num9 * power_of<2>(ln1mxc) * ln1myc) / denom6 - (32.0 * log(xc) * ln1myc) / yc3 + (96.0 * log(xc) * ln1myc) / yc + (16.0 * ln1pxc * ln1myc) / yc3 - (48.0 * ln1pxc * ln1myc) / yc + (36.0 * num9 * ln2 * ln1pxc * ln1myc) / denom6 - (18.0 * num9 * power_of<2>(ln1pxc) * ln1myc) / denom6 - (9.0 * num5 * power_of<2>(ln1myc)) / denom4
                + (9.0 * num11 * ln2 * power_of<2>(ln1myc)) / denom6 + (3.0 * num11 * power_of<3>(ln1myc)) / denom6 + dilog((1.0 - yc) / (w3 - yc)) * ((18.0 * num11 * ln1mychalf) / denom6 + (18.0 * num11 * ln1myc) / denom6) + dilog(1.0 - yc) * ((18.0 * num11 * ln1myc) / denom6 - (72.0 * num8 * ln1myc) / denom6) + dilog(2.0 / (1.0 - xc)) * ((-144.0 * num10 * ln1myc) / denom7 + (72.0 * num9 * ln1myc) / denom6) + dilog(2.0 / (1.0 + xc)) * ((-144.0 * num10 * ln1myc) / denom7 + (72.0 * num9 * ln1myc) / denom6) + (3.0 * num11 * pisqu * log(-1.0 / (-w3 - yc))) / denom6
                + (3.0 * num11 * power_of<3>(log(-1.0 / (-w3 - yc)))) / denom6 - (3.0 * num11 * pisqu * log(1.0 / (-w3 - yc))) / denom6 - (3.0 * num11 * power_of<3>(log(1.0 / (-w3 - yc)))) / denom6 - (18.0 * num11 * pisqu * (0. + log((2.0 * (-1.0 - w3)) / (1.0 - w3)) / 6.0 - log(-2.0 / w3) / 6.0 - log((2.0 * (-1.0 - w3)) / ((1.0 - w3) * (1.0 - yc))) / 6.0 + log(2.0 / (-w3 - yc)) / 6.0)) / denom6 - (3.0 * num11 * power_of<3>(log(2.0 / (-w3 - yc)))) / denom6 - (9.0 * num11 * power_of<2>(ln1myc) * log((-1.0 - w3) / (-w3 - yc))) / denom6 - (3.0 * num11 * power_of<3>(log((-1.0 - xc) / (-w3 - yc)))) / denom6
                + (3.0 * num11 * power_of<3>(log((1.0 - xc) / (-w3 - yc)))) / denom6 - (3.0 * num11 * power_of<3>(log((-1.0 + xc) / (-w3 - yc)))) / denom6 + (3.0 * num11 * power_of<3>(log((1.0 + xc) / (-w3 - yc)))) / denom6 - (18.0 * num11 * pisqu * (0. - log(-2.0 / w3) / 6.0 + log((-2.0 * (1.0 + w3)) / (1.0 - w3)) / 6.0 - log((-2.0 * (1.0 + w3)) / ((-1.0 + w3) * (-1.0 - yc))) / 6.0 + log(-2.0 / (w3 - yc)) / 6.0)) / denom6 - (3.0 * num11 * power_of<3>(log(-2.0 / (w3 - yc)))) / denom6 - (3.0 * num11 * pisqu * log(-1.0 / (w3 - yc))) / denom6 - (3.0 * num11 * power_of<3>(log(-1.0 / (w3 - yc)))) / denom6
                + (3.0 * num11 * pisqu * log(1.0 / (w3 - yc))) / denom6 + (3.0 * num11 * power_of<3>(log(1.0 / (w3 - yc)))) / denom6 + (9.0 * num11 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-1.0 + w3) / (w3 - yc))) / denom6 + (9.0 * num11 * power_of<2>(ln1myc) * log((-1.0 + w3) / (w3 - yc))) / denom6 + (3.0 * num11 * power_of<3>(log((-1.0 - xc) / (w3 - yc)))) / denom6 - (3.0 * num11 * power_of<3>(log((1.0 - xc) / (w3 - yc)))) / denom6 + (3.0 * num11 * power_of<3>(log((-1.0 + xc) / (w3 - yc)))) / denom6 - (3.0 * num11 * power_of<3>(log((1.0 + xc) / (w3 - yc)))) / denom6
                - (9.0 * num11 * power_of<2>(log((1.0 - yc) / 2.0)) * log(((1.0 - w3) * (-1.0 - yc)) / (2.0 * (w3 - yc)))) / denom6 + (9.0 * num11 * power_of<2>(log((1.0 - yc) / 2.0)) * log((w3 - yc) / (-1.0 + w3))) / denom6 - (9.0 * num11 * power_of<2>(ln1myc) * log((w3 - yc) / (-1.0 + w3))) / denom6 - (9.0 * num11 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-2.0 * (w3 - yc)) / ((-1.0 + w3) * (-1.0 - yc)))) / denom6 - (144.0 * num10 * pisqu * (0. - log(2.0 * xcinv) / 6.0 + log((-2.0 * (1.0 - xc)) / (1.0 + xc)) / 6.0 - log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (-xc - yc)) / 6.0)) / denom7
                + (72.0 * num9 * pisqu * (0. - log(2.0 * xcinv) / 6.0 + log((-2.0 * (1.0 - xc)) / (1.0 + xc)) / 6.0 - log((-2.0 * (1.0 - xc)) / ((-1.0 - xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (-xc - yc)) / 6.0)) / denom6 - (24.0 * num10 * power_of<3>(log(-2.0 / (-xc - yc)))) / denom7 + (12.0 * num9 * power_of<3>(log(-2.0 / (-xc - yc)))) / denom6 - (144.0 * num10 * pisqu * (0. + log((2.0 * (-1.0 - xc)) / (1.0 - xc)) / 6.0 - log(-2.0 * xcinv) / 6.0 - log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc))) / 6.0 + log(2.0 / (-xc - yc)) / 6.0)) / denom7
                + (72.0 * num9 * pisqu * (0. + log((2.0 * (-1.0 - xc)) / (1.0 - xc)) / 6.0 - log(-2.0 * xcinv) / 6.0 - log((2.0 * (-1.0 - xc)) / ((1.0 - xc) * (1.0 - yc))) / 6.0 + log(2.0 / (-xc - yc)) / 6.0)) / denom6 - (24.0 * num10 * power_of<3>(log(2.0 / (-xc - yc)))) / denom7 + (12.0 * num9 * power_of<3>(log(2.0 / (-xc - yc)))) / denom6 + (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-1.0 - xc) / (-xc - yc))) / denom7 - (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-1.0 - xc) / (-xc - yc))) / denom6 - (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log(((1.0 + xc) * (-1.0 - yc)) / (2.0 * (-xc - yc)))) / denom7
                + (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log(((1.0 + xc) * (-1.0 - yc)) / (2.0 * (-xc - yc)))) / denom6 + (18.0 * num11 * dilog(((-1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (1.0 - yc))) * log((-xc - yc) / (-1.0 - xc))) / denom6 - (18.0 * num11 * dilog(((-1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (1.0 - yc))) * log((-xc - yc) / (-1.0 - xc))) / denom6 + (18.0 * num11 * dilog(((-1.0 - w3) * (-xc - yc)) / ((-1.0 - xc) * (-w3 - yc))) * log((-xc - yc) / (-1.0 - xc))) / denom6
                - (18.0 * num11 * dilog(((-1.0 + w3) * (-xc - yc)) / ((-1.0 - xc) * (w3 - yc))) * log((-xc - yc) / (-1.0 - xc))) / denom6 + (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-xc - yc) / (-1.0 - xc))) / denom7 - (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-xc - yc) / (-1.0 - xc))) / denom6 + (72.0 * num6 * ln1myc * log((-xc - yc) / (-1.0 - xc))) / denom5 - (72.0 * num10 * power_of<2>(ln1myc) * log((-xc - yc) / (-1.0 - xc))) / denom7 - (18.0 * num11 * ln1myc * log((-w3 - yc) / (-w3 + xc)) * log((-xc - yc) / (-1.0 - xc))) / denom6
                + (18.0 * num11 * ln1myc * log((w3 - yc) / (w3 + xc)) * log((-xc - yc) / (-1.0 - xc))) / denom6 - (9.0 * num11 * log((-w3 + xc) / (-w3 - yc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6 + (9.0 * num11 * log(((w3 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (-w3 - yc))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6 - (9.0 * num11 * log((-w3 - yc) / (-w3 + xc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6 + (9.0 * num11 * log(-(((-1.0 - xc) * (-w3 - yc)) / ((-w3 + xc) * (1.0 - yc)))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6
                + (9.0 * num11 * log((w3 + xc) / (w3 - yc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6 - (9.0 * num11 * log(((-w3 - xc) * (1.0 - yc)) / ((-1.0 - xc) * (w3 - yc))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6 + (9.0 * num11 * log((w3 - yc) / (w3 + xc)) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6 - (9.0 * num11 * log(-(((-1.0 - xc) * (w3 - yc)) / ((w3 + xc) * (1.0 - yc)))) * power_of<2>(log((-xc - yc) / (-1.0 - xc)))) / denom6 - (18.0 * num11 * dilog(((1.0 - w3) * (-xc - yc)) / ((-w3 + xc) * (-1.0 - yc))) * log((-xc - yc) / (1.0 - xc))) / denom6
                + (18.0 * num11 * dilog(((1.0 + w3) * (-xc - yc)) / ((w3 + xc) * (-1.0 - yc))) * log((-xc - yc) / (1.0 - xc))) / denom6 - (18.0 * num11 * dilog(((1.0 - w3) * (-xc - yc)) / ((1.0 - xc) * (-w3 - yc))) * log((-xc - yc) / (1.0 - xc))) / denom6 + (18.0 * num11 * dilog(((1.0 + w3) * (-xc - yc)) / ((1.0 - xc) * (w3 - yc))) * log((-xc - yc) / (1.0 - xc))) / denom6 + (9.0 * num11 * log((-w3 + xc) / (-w3 - yc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6 - (9.0 * num11 * log(((w3 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (-w3 - yc))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6
                + (9.0 * num11 * log((-w3 - yc) / (-w3 + xc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6 - (9.0 * num11 * log(-(((1.0 - xc) * (-w3 - yc)) / ((-w3 + xc) * (-1.0 - yc)))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6 - (9.0 * num11 * log((w3 + xc) / (w3 - yc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6 + (9.0 * num11 * log(((-w3 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (w3 - yc))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6 - (9.0 * num11 * log((w3 - yc) / (w3 + xc)) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6
                + (9.0 * num11 * log(-(((1.0 - xc) * (w3 - yc)) / ((w3 + xc) * (-1.0 - yc)))) * power_of<2>(log((-xc - yc) / (1.0 - xc)))) / denom6 + dilog((-xc - yc) / (w3 - yc)) * ((18.0 * num11 * log((-xc - yc) / (-1.0 - xc))) / denom6 - (18.0 * num11 * log((-xc - yc) / (1.0 - xc))) / denom6) + dilog((-xc - yc) / (-w3 - yc)) * ((-18.0 * num11 * log((-xc - yc) / (-1.0 - xc))) / denom6 + (18.0 * num11 * log((-xc - yc) / (1.0 - xc))) / denom6) - (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-2.0 * (-xc - yc)) / ((-1.0 - xc) * (-1.0 - yc)))) / denom7
                + (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-2.0 * (-xc - yc)) / ((-1.0 - xc) * (-1.0 - yc)))) / denom6 - (144.0 * num10 * pisqu * (0. - log(-2.0 * xcinv) / 6.0 + log((-2.0 * (1.0 + xc)) / (1.0 - xc)) / 6.0 - log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (xc - yc)) / 6.0)) / denom7 + (72.0 * num9 * pisqu * (0. - log(-2.0 * xcinv) / 6.0 + log((-2.0 * (1.0 + xc)) / (1.0 - xc)) / 6.0 - log((-2.0 * (1.0 + xc)) / ((-1.0 + xc) * (-1.0 - yc))) / 6.0 + log(-2.0 / (xc - yc)) / 6.0)) / denom6 - (24.0 * num10 * power_of<3>(log(-2.0 / (xc - yc)))) / denom7 + (12.0 * num9 * power_of<3>(log(-2.0 / (xc - yc)))) / denom6
                - (144.0 * num10 * pisqu * (0. - log(2.0 * xcinv) / 6.0 + log((2.0 * (-1.0 + xc)) / (1.0 + xc)) / 6.0 - log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc))) / 6.0 + log(2.0 / (xc - yc)) / 6.0)) / denom7 + (72.0 * num9 * pisqu * (0. - log(2.0 * xcinv) / 6.0 + log((2.0 * (-1.0 + xc)) / (1.0 + xc)) / 6.0 - log((2.0 * (-1.0 + xc)) / ((1.0 + xc) * (1.0 - yc))) / 6.0 + log(2.0 / (xc - yc)) / 6.0)) / denom6 - (24.0 * num10 * power_of<3>(log(2.0 / (xc - yc)))) / denom7 + (12.0 * num9 * power_of<3>(log(2.0 / (xc - yc)))) / denom6 + (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-1.0 + xc) / (xc - yc))) / denom7 - (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-1.0 + xc) / (xc - yc))) / denom6
                - (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log(((1.0 - xc) * (-1.0 - yc)) / (2.0 * (xc - yc)))) / denom7 + (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log(((1.0 - xc) * (-1.0 - yc)) / (2.0 * (xc - yc)))) / denom6 + (18.0 * num11 * dilog(((-1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (1.0 - yc))) * log((xc - yc) / (-1.0 + xc))) / denom6 - (18.0 * num11 * dilog(((-1.0 + w3) * (xc - yc)) / ((w3 - xc) * (1.0 - yc))) * log((xc - yc) / (-1.0 + xc))) / denom6 + (18.0 * num11 * dilog(((-1.0 - w3) * (xc - yc)) / ((-1.0 + xc) * (-w3 - yc))) * log((xc - yc) / (-1.0 + xc))) / denom6
                - (18.0 * num11 * dilog(((-1.0 + w3) * (xc - yc)) / ((-1.0 + xc) * (w3 - yc))) * log((xc - yc) / (-1.0 + xc))) / denom6 + (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log((xc - yc) / (-1.0 + xc))) / denom7 - (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log((xc - yc) / (-1.0 + xc))) / denom6 + (72.0 * num6 * ln1myc * log((xc - yc) / (-1.0 + xc))) / denom5 - (72.0 * num10 * power_of<2>(ln1myc) * log((xc - yc) / (-1.0 + xc))) / denom7 - (18.0 * num11 * ln1myc * log((-w3 - yc) / (-w3 - xc)) * log((xc - yc) / (-1.0 + xc))) / denom6
                + (18.0 * num11 * ln1myc * log((w3 - yc) / (w3 - xc)) * log((xc - yc) / (-1.0 + xc))) / denom6 - (9.0 * num11 * log((-w3 - xc) / (-w3 - yc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6 + (9.0 * num11 * log(((w3 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (-w3 - yc))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6 - (9.0 * num11 * log((-w3 - yc) / (-w3 - xc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6 + (9.0 * num11 * log(-(((-1.0 + xc) * (-w3 - yc)) / ((-w3 - xc) * (1.0 - yc)))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6
                + (9.0 * num11 * log((w3 - xc) / (w3 - yc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6 - (9.0 * num11 * log(((-w3 + xc) * (1.0 - yc)) / ((-1.0 + xc) * (w3 - yc))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6 + (9.0 * num11 * log((w3 - yc) / (w3 - xc)) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6 - (9.0 * num11 * log(-(((-1.0 + xc) * (w3 - yc)) / ((w3 - xc) * (1.0 - yc)))) * power_of<2>(log((xc - yc) / (-1.0 + xc)))) / denom6 - (18.0 * num11 * dilog(((1.0 - w3) * (xc - yc)) / ((-w3 - xc) * (-1.0 - yc))) * log((xc - yc) / (1.0 + xc))) / denom6
                + (18.0 * num11 * dilog(((1.0 + w3) * (xc - yc)) / ((w3 - xc) * (-1.0 - yc))) * log((xc - yc) / (1.0 + xc))) / denom6 - (18.0 * num11 * dilog(((1.0 - w3) * (xc - yc)) / ((1.0 + xc) * (-w3 - yc))) * log((xc - yc) / (1.0 + xc))) / denom6 + (18.0 * num11 * dilog(((1.0 + w3) * (xc - yc)) / ((1.0 + xc) * (w3 - yc))) * log((xc - yc) / (1.0 + xc))) / denom6 + (9.0 * num11 * log((-w3 - xc) / (-w3 - yc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6 - (9.0 * num11 * log(((w3 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (-w3 - yc))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6
                + (9.0 * num11 * log((-w3 - yc) / (-w3 - xc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6 - (9.0 * num11 * log(-(((1.0 + xc) * (-w3 - yc)) / ((-w3 - xc) * (-1.0 - yc)))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6 - (9.0 * num11 * log((w3 - xc) / (w3 - yc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6 + (9.0 * num11 * log(((-w3 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (w3 - yc))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6 - (9.0 * num11 * log((w3 - yc) / (w3 - xc)) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6
                + (9.0 * num11 * log(-(((1.0 + xc) * (w3 - yc)) / ((w3 - xc) * (-1.0 - yc)))) * power_of<2>(log((xc - yc) / (1.0 + xc)))) / denom6 + dilog((xc - yc) / (w3 - yc)) * ((18.0 * num11 * log((xc - yc) / (-1.0 + xc))) / denom6 - (18.0 * num11 * log((xc - yc) / (1.0 + xc))) / denom6) + dilog((xc - yc) / (-w3 - yc)) * ((-18.0 * num11 * log((xc - yc) / (-1.0 + xc))) / denom6 + (18.0 * num11 * log((xc - yc) / (1.0 + xc))) / denom6) - (72.0 * num10 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-2.0 * (xc - yc)) / ((-1.0 + xc) * (-1.0 - yc)))) / denom7
                + (36.0 * num9 * power_of<2>(log((1.0 - yc) / 2.0)) * log((-2.0 * (xc - yc)) / ((-1.0 + xc) * (-1.0 - yc)))) / denom6 + (9.0 * num11 * power_of<2>(ln1myc) * log(yc)) / denom6 - (36.0 * num8 * power_of<2>(ln1myc) * log(yc)) / denom6 + (9.0 * num11 * power_of<2>(ln1myc) * log(((-1.0 - w3) * yc) / (-w3 - yc))) / denom6 - (9.0 * num11 * power_of<2>(ln1myc) * log(((-1.0 + w3) * yc) / (w3 - yc))) / denom6 - (18.0 * num11 * dilog(((-1.0 - w3) * (-1.0 - yc)) / ((1.0 - w3) * (1.0 - yc))) * ln1pychalf) / denom6 - (18.0 * num11 * dilog((-0.5) * (((-1.0 - w3) * (-1.0 - yc)) / (-w3 - yc))) * ln1pychalf) / denom6
                - (27.0 * num7 * power_of<2>(ln1myc) * ln1pychalf) / denom6 + (18.0 * num11 * ln1myc * log((-w3 - yc) / (1.0 - w3)) * ln1pychalf) / denom6 + (144.0 * num10 * ln1myc * log((-xc - yc) / (1.0 - xc)) * ln1pychalf) / denom7 - (72.0 * num9 * ln1myc * log((-xc - yc) / (1.0 - xc)) * ln1pychalf) / denom6 + (144.0 * num10 * ln1myc * log((xc - yc) / (1.0 + xc)) * ln1pychalf) / denom7 - (72.0 * num9 * ln1myc * log((xc - yc) / (1.0 + xc)) * ln1pychalf) / denom6
                + (9.0 * num11 * log((1.0 - w3) / (-w3 - yc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 - (9.0 * num11 * log((-0.5) * (((-1.0 + w3) * (1.0 - yc)) / (-w3 - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 + (9.0 * num11 * log((-w3 - yc) / (1.0 - w3)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 - (9.0 * num11 * log((2.0 * (-w3 - yc)) / ((1.0 - w3) * (1.0 - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 + (72.0 * num10 * log((1.0 - xc) / (-xc - yc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7 - (36.0 * num9 * log((1.0 - xc) / (-xc - yc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6
                - (72.0 * num10 * log((-0.5) * (((-1.0 + xc) * (1.0 - yc)) / (-xc - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7 + (36.0 * num9 * log((-0.5) * (((-1.0 + xc) * (1.0 - yc)) / (-xc - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 + (72.0 * num10 * log((-xc - yc) / (1.0 - xc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7 - (36.0 * num9 * log((-xc - yc) / (1.0 - xc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 - (72.0 * num10 * log((2.0 * (-xc - yc)) / ((1.0 - xc) * (1.0 - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7 + (36.0 * num9 * log((2.0 * (-xc - yc)) / ((1.0 - xc) * (1.0 - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6
                + (72.0 * num10 * log((1.0 + xc) / (xc - yc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7 - (36.0 * num9 * log((1.0 + xc) / (xc - yc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 - (72.0 * num10 * log((-0.5) * (((-1.0 - xc) * (1.0 - yc)) / (xc - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7 + (36.0 * num9 * log((-0.5) * (((-1.0 - xc) * (1.0 - yc)) / (xc - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 + (72.0 * num10 * log((xc - yc) / (1.0 + xc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7 - (36.0 * num9 * log((xc - yc) / (1.0 + xc)) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 - (72.0 * num10 * log((2.0 * (xc - yc)) / ((1.0 + xc) * (1.0 - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom7
                + (36.0 * num9 * log((2.0 * (xc - yc)) / ((1.0 + xc) * (1.0 - yc))) * power_of<2>(log((1.0 + yc) / 2.0))) / denom6 + dilog((-1.0 - yc) / (-xc - yc)) * ((144.0 * num10 * ln1pychalf) / denom7 - (72.0 * num9 * ln1pychalf) / denom6) + dilog((-1.0 - yc) / (xc - yc)) * ((144.0 * num10 * ln1pychalf) / denom7 - (72.0 * num9 * ln1pychalf) / denom6) + dilog(((-1.0 - xc) * (-1.0 - yc)) / ((1.0 - xc) * (1.0 - yc))) * ((-144.0 * num10 * ln1pychalf) / denom7 + (72.0 * num9 * ln1pychalf) / denom6);

            const complex<double> f29cPart4 = + dilog(((-1.0 + xc) * (-1.0 - yc)) / ((1.0 + xc) * (1.0 - yc))) * ((-144.0 * num10 * ln1pychalf) / denom7 + (72.0 * num9 * ln1pychalf) / denom6) + dilog((-0.5) * (((-1.0 - xc) * (-1.0 - yc)) / (-xc - yc))) * ((-144.0 * num10 * ln1pychalf) / denom7 + (72.0 * num9 * ln1pychalf) / denom6) + dilog((-0.5) * (((-1.0 + xc) * (-1.0 - yc)) / (xc - yc))) * ((-144.0 * num10 * ln1pychalf) / denom7 + (72.0 * num9 * ln1pychalf) / denom6) + (36.0 * lognum1 * ln1pyc) / denom8 + (3.0 * num7 * pisqu * ln1pyc) / (2.0 * denom6)
                - (18.0 * num11 * dilog(-2.0 / (-1.0 + w3)) * ln1pyc) / denom6 - (18.0 * num11 * dilog((-1.0 - xc) / (-w3 - xc)) * ln1pyc) / denom6 + (18.0 * num11 * dilog((-1.0 - xc) / (w3 - xc)) * ln1pyc) / denom6 - (18.0 * num11 * dilog((-1.0 + xc) / (-w3 + xc)) * ln1pyc) / denom6 + (18.0 * num11 * dilog((-1.0 + xc) / (w3 + xc)) * ln1pyc) / denom6 + (36.0 * num9 * dilog((-1.0 - yc) / (-1.0 - xc)) * ln1pyc) / denom6 + (36.0 * num9 * dilog((-1.0 - yc) / (-1.0 + xc)) * ln1pyc) / denom6 - (18.0 * num11 * dilog((-1.0 - yc) / (w3 - yc)) * ln1pyc) / denom6
                + (18.0 * num11 * dilog((w3 * (-1.0 - yc)) / (-w3 + yc)) * ln1pyc) / denom6 - (18.0 * num11 * dilog(-((w3 * (-1.0 - yc)) / (w3 + yc))) * ln1pyc) / denom6 + (18.0 * num1 * ln2 * ln1pyc) / denom1 - (9.0 * num7 * ln2squ * ln1pyc) / denom6 + (18.0 * num11 * (pisqu / 12.0 - ln2squ / 2.0) * ln1pyc) / denom6 + (32.0 * lnmuhat * ln1pyc) / yc3 - (96.0 * lnmuhat * ln1pyc) / yc - (16.0 * ln1mxc * ln1pyc) / yc3 + (48.0 * ln1mxc * ln1pyc) / yc + (36.0 * num9 * ln2 * ln1mxc * ln1pyc) / denom6
                - (18.0 * num9 * power_of<2>(ln1mxc) * ln1pyc) / denom6 + (32.0 * log(xc) * ln1pyc) / yc3 - (96.0 * log(xc) * ln1pyc) / yc - (16.0 * ln1pxc * ln1pyc) / yc3 + (48.0 * ln1pxc * ln1pyc) / yc + (36.0 * num9 * ln2 * ln1pxc * ln1pyc) / denom6 - (18.0 * num9 * power_of<2>(ln1pxc) * ln1pyc) / denom6 + (36.0 * num7 * ln2 * ln1myc * ln1pyc) / denom6 + (18.0 * num11 * ln1mychalf * log((w3 - yc) / (-1.0 + w3)) * ln1pyc) / denom6 + (144.0 * num10 * ln1mychalf * log((-xc - yc) / (-1.0 - xc)) * ln1pyc) / denom7
                - (72.0 * num9 * ln1mychalf * log((-xc - yc) / (-1.0 - xc)) * ln1pyc) / denom6 + (36.0 * num9 * ln1myc * log((-xc - yc) / (-1.0 - xc)) * ln1pyc) / denom6 - (72.0 * num6 * log((-xc - yc) / (1.0 - xc)) * ln1pyc) / denom5 + (36.0 * num9 * ln1myc * log((-xc - yc) / (1.0 - xc)) * ln1pyc) / denom6 + (18.0 * num11 * log((-w3 - yc) / (-w3 + xc)) * log((-xc - yc) / (1.0 - xc)) * ln1pyc) / denom6 - (18.0 * num11 * log((w3 - yc) / (w3 + xc)) * log((-xc - yc) / (1.0 - xc)) * ln1pyc) / denom6
                + (144.0 * num10 * ln1mychalf * log((xc - yc) / (-1.0 + xc)) * ln1pyc) / denom7 - (72.0 * num9 * ln1mychalf * log((xc - yc) / (-1.0 + xc)) * ln1pyc) / denom6 + (36.0 * num9 * ln1myc * log((xc - yc) / (-1.0 + xc)) * ln1pyc) / denom6 - (72.0 * num6 * log((xc - yc) / (1.0 + xc)) * ln1pyc) / denom5 + (36.0 * num9 * ln1myc * log((xc - yc) / (1.0 + xc)) * ln1pyc) / denom6 + (18.0 * num11 * log((-w3 - yc) / (-w3 - xc)) * log((xc - yc) / (1.0 + xc)) * ln1pyc) / denom6
                - (18.0 * num11 * log((w3 - yc) / (w3 - xc)) * log((xc - yc) / (1.0 + xc)) * ln1pyc) / denom6 - (9.0 * num1 * power_of<2>(ln1pyc)) / denom1 + (9.0 * num11 * ln2 * power_of<2>(ln1pyc)) / denom6 - (27.0 * num7 * ln1mychalf * power_of<2>(ln1pyc)) / denom6 + (9.0 * num11 * log((1.0 - w3) / (-w3 - yc)) * power_of<2>(ln1pyc)) / denom6 - (9.0 * num11 * log((-w3 - yc) / (1.0 - w3)) * power_of<2>(ln1pyc)) / denom6 - (9.0 * num11 * log((1.0 + w3) / (w3 - yc)) * power_of<2>(ln1pyc)) / denom6 - (72.0 * num10 * log((-xc - yc) / (1.0 - xc)) * power_of<2>(ln1pyc)) / denom7
                - (72.0 * num10 * log((xc - yc) / (1.0 + xc)) * power_of<2>(ln1pyc)) / denom7 + (9.0 * num11 * log(-yc) * power_of<2>(ln1pyc)) / denom6 - (36.0 * num8 * log(-yc) * power_of<2>(ln1pyc)) / denom6 + (3.0 * num11 * power_of<3>(ln1pyc)) / denom6 + dilog((1.0 + yc) / (1.0 - xc)) * ((-72.0 * num6) / denom5 + (144.0 * num10 * ln1myc) / denom7 - (36.0 * num9 * ln1myc) / denom6 + (144.0 * num10 * ln1pychalf) / denom7 - (72.0 * num9 * ln1pychalf) / denom6 - (144.0 * num10 * ln1pyc) / denom7)
                + dilog((1.0 + yc) / (1.0 + xc)) * ((-72.0 * num6) / denom5 + (144.0 * num10 * ln1myc) / denom7 - (36.0 * num9 * ln1myc) / denom6 + (144.0 * num10 * ln1pychalf) / denom7 - (72.0 * num9 * ln1pychalf) / denom6 - (144.0 * num10 * ln1pyc) / denom7) + dilog((xc + yc) / (w3 + xc)) * ((18.0 * num11 * ln1mxc) / denom6 - (18.0 * num11 * ln1pxc) / denom6 + (18.0 * num11 * ln1myc) / denom6 + (18.0 * num11 * log((-xc - yc) / (-1.0 - xc))) / denom6 - (18.0 * num11 * log((-xc - yc) / (1.0 - xc))) / denom6 - (18.0 * num11 * ln1pyc) / denom6)
                + dilog((-xc + yc) / (w3 - xc)) * ((-18.0 * num11 * ln1mxc) / denom6 + (18.0 * num11 * ln1pxc) / denom6 + (18.0 * num11 * ln1myc) / denom6 + (18.0 * num11 * log((xc - yc) / (-1.0 + xc))) / denom6 - (18.0 * num11 * log((xc - yc) / (1.0 + xc))) / denom6 - (18.0 * num11 * ln1pyc) / denom6) + dilog((1.0 + yc) / (1.0 - w3)) * ((18.0 * num11 * ln1myc) / denom6 + (18.0 * num11 * ln1pychalf) / denom6 - (18.0 * num11 * ln1pyc) / denom6)
                + dilog((-1.0 + yc) / (-1.0 + w3)) * ((18.0 * num11 * ln1mychalf) / denom6 - (18.0 * num11 * ln1myc) / denom6 + (18.0 * num11 * ln1pyc) / denom6) + dilog((xc + yc) / (-w3 + xc)) * ((-18.0 * num11 * ln1mxc) / denom6 + (18.0 * num11 * ln1pxc) / denom6 - (18.0 * num11 * ln1myc) / denom6 - (18.0 * num11 * log((-xc - yc) / (-1.0 - xc))) / denom6 + (18.0 * num11 * log((-xc - yc) / (1.0 - xc))) / denom6 + (18.0 * num11 * ln1pyc) / denom6)
                + dilog((-xc + yc) / (-w3 - xc)) * ((18.0 * num11 * ln1mxc) / denom6 - (18.0 * num11 * ln1pxc) / denom6 - (18.0 * num11 * ln1myc) / denom6 - (18.0 * num11 * log((xc - yc) / (-1.0 + xc))) / denom6 + (18.0 * num11 * log((xc - yc) / (1.0 + xc))) / denom6 + (18.0 * num11 * ln1pyc) / denom6) + dilog((-1.0 - yc) / (-w3 - yc)) * ((18.0 * num11 * ln1pychalf) / denom6 + (18.0 * num11 * ln1pyc) / denom6)
                + dilog((1.0 + yc) / 2.0) * ((-18.0 * num1) / denom1 - (18.0 * num11 * ln1myc) / denom6 - (18.0 * num7 * ln1myc) / denom6 - (18.0 * num11 * ln1pyc) / denom6 - (54.0 * num7 * ln1pyc) / denom6) + dilog((1.0 - yc) / 2.0) * ((-18.0 * num5) / denom4 - (18.0 * num11 * ln1myc) / denom6 - (54.0 * num7 * ln1myc) / denom6 - (18.0 * num11 * ln1pyc) / denom6 - (18.0 * num7 * ln1pyc) / denom6) + dilog(1.0 + yc) * ((18.0 * num11 * ln1pyc) / denom6 - (72.0 * num8 * ln1pyc) / denom6)
                + dilog(-yc) * ((-72.0 * num2) / denom2 + (18.0 * num11 * ln1myc) / denom6 - (36.0 * num8 * ln1myc) / denom6 - (36.0 * num8 * ln1pyc) / denom6) + dilog(yc) * ((72.0 * num2) / denom2 - (36.0 * num8 * ln1myc) / denom6 + (18.0 * num11 * ln1pyc) / denom6 - (36.0 * num8 * ln1pyc) / denom6) + dilog((-1.0 + yc) / (-1.0 - xc)) * ((72.0 * num6) / denom5 + (144.0 * num10 * ln1mychalf) / denom7 - (72.0 * num9 * ln1mychalf) / denom6 - (144.0 * num10 * ln1myc) / denom7 + (144.0 * num10 * ln1pyc) / denom7 - (36.0 * num9 * ln1pyc) / denom6)
                + dilog((-1.0 + yc) / (-1.0 + xc)) * ((72.0 * num6) / denom5 + (144.0 * num10 * ln1mychalf) / denom7 - (72.0 * num9 * ln1mychalf) / denom6 - (144.0 * num10 * ln1myc) / denom7 + (144.0 * num10 * ln1pyc) / denom7 - (36.0 * num9 * ln1pyc) / denom6) + dilog(-2.0 / (-1.0 - xc)) * ((-144.0 * num10 * ln1pyc) / denom7 + (72.0 * num9 * ln1pyc) / denom6) + dilog(-2.0 / (-1.0 + xc)) * ((-144.0 * num10 * ln1pyc) / denom7 + (72.0 * num9 * ln1pyc) / denom6)
                + (3.0 * num11 * ln1pyc * (pisqu - 6.0 * ln2squ - 12.0 * ln1myc * ln1pychalf + ln64 * ln1pyc)) / (2.0 * denom6) + (3.0 * num11 * ln1myc * (pisqu - 6.0 * ln2squ + ln64 * ln1myc - 12.0 * ln1mychalf * ln1pyc)) / (2.0 * denom6) - (3.0 * num8 * (4.0 * power_of<3>(ln2) - pisqu * ln4 + ln1myc * (pisqu - 6.0 * ln2squ + ln64 * ln1myc) - (pisqu + 6.0 * ln2squ) * ln1pyc - 2.0 * (-3.0 * log(-yc) + log((1.0 + yc) / 8.0)) * power_of<2>(ln1pyc))) / denom6
                - (3.0 * num8 * (4.0 * power_of<3>(ln2) - pisqu * ln4 - (pisqu + 6.0 * ln2squ) * ln1myc - 2.0 * power_of<2>(ln1myc) * (log((1.0 - yc) / 8.0) - 3.0 * log(yc)) + ln1pyc * (pisqu - 6.0 * ln2squ + ln64 * ln1pyc))) / denom6 + (9.0 * num11 * power_of<2>(ln1pyc) * log(((1.0 + w3) * yc) / (-w3 + yc))) / denom6 - (9.0 * num11 * power_of<2>(ln1pyc) * log(((1.0 - w3) * yc) / (w3 + yc))) / denom6 - (36.0 * num11 * ln2 * ln1mxc * log(1.0 - yc / w3)) / denom6 + (18.0 * num11 * power_of<2>(ln1mxc) * log(1.0 - yc / w3)) / denom6
                - (36.0 * num11 * ln2 * ln1pxc * log(1.0 - yc / w3)) / denom6 + (18.0 * num11 * power_of<2>(ln1pxc) * log(1.0 - yc / w3)) / denom6 - (36.0 * num11 * ln2 * ln1mxc * log(1.0 + yc / w3)) / denom6 + (18.0 * num11 * power_of<2>(ln1mxc) * log(1.0 + yc / w3)) / denom6 - (36.0 * num11 * ln2 * ln1pxc * log(1.0 + yc / w3)) / denom6 + (18.0 * num11 * power_of<2>(ln1pxc) * log(1.0 + yc / w3)) / denom6 + dilog(-w3) * ((-18.0 * num11 * log(1.0 - yc / w3)) / denom6 - (18.0 * num11 * log(1.0 + yc / w3)) / denom6)
                + dilog(w3) * ((18.0 * num11 * log(1.0 - yc / w3)) / denom6 + (18.0 * num11 * log(1.0 + yc / w3)) / denom6) - (72.0 * num6 * ln1mxc * log(1.0 - yc * xcinv)) / denom5 - (144.0 * num10 * ln2 * ln1mxc * log(1.0 - yc * xcinv)) / denom7 + (72.0 * num10 * power_of<2>(ln1mxc) * log(1.0 - yc * xcinv)) / denom7 + (72.0 * num6 * ln1pxc * log(1.0 - yc * xcinv)) / denom5 - (144.0 * num10 * ln2 * ln1pxc * log(1.0 - yc * xcinv)) / denom7 + (72.0 * num10 * power_of<2>(ln1pxc) * log(1.0 - yc * xcinv)) / denom7 - (36.0 * num9 * ln1mxc * log((-1.0 - yc) / (-1.0 - xc)) * log(1.0 - yc * xcinv)) / denom6;

            const complex<double> f29cPart5 = + (36.0 * num9 * ln1pxc * log((-1.0 - yc) / (-1.0 - xc)) * log(1.0 - yc * xcinv)) / denom6 + (36.0 * num9 * ln1mxc * log((1.0 - yc) / (1.0 - xc)) * log(1.0 - yc * xcinv)) / denom6 - (36.0 * num9 * ln1pxc * log((1.0 - yc) / (1.0 - xc)) * log(1.0 - yc * xcinv)) / denom6 + (18.0 * num11 * ln1mxc * log((-w3 - yc) / (-w3 - xc)) * log(1.0 - yc * xcinv)) / denom6 - (18.0 * num11 * ln1pxc * log((-w3 - yc) / (-w3 - xc)) * log(1.0 - yc * xcinv)) / denom6 - (18.0 * num11 * ln1mxc * log((w3 - yc) / (w3 - xc)) * log(1.0 - yc * xcinv)) / denom6
                + (18.0 * num11 * ln1pxc * log((w3 - yc) / (w3 - xc)) * log(1.0 - yc * xcinv)) / denom6 + (72.0 * num6 * ln1mxc * log(1.0 + yc * xcinv)) / denom5 - (144.0 * num10 * ln2 * ln1mxc * log(1.0 + yc * xcinv)) / denom7 + (72.0 * num10 * power_of<2>(ln1mxc) * log(1.0 + yc * xcinv)) / denom7 - (72.0 * num6 * ln1pxc * log(1.0 + yc * xcinv)) / denom5 - (144.0 * num10 * ln2 * ln1pxc * log(1.0 + yc * xcinv)) / denom7 + (72.0 * num10 * power_of<2>(ln1pxc) * log(1.0 + yc * xcinv)) / denom7 + (36.0 * num9 * ln1mxc * log((-1.0 - yc) / (-1.0 + xc)) * log(1.0 + yc * xcinv)) / denom6
                - (36.0 * num9 * ln1pxc * log((-1.0 - yc) / (-1.0 + xc)) * log(1.0 + yc * xcinv)) / denom6 - (36.0 * num9 * ln1mxc * log((1.0 - yc) / (1.0 + xc)) * log(1.0 + yc * xcinv)) / denom6 + (36.0 * num9 * ln1pxc * log((1.0 - yc) / (1.0 + xc)) * log(1.0 + yc * xcinv)) / denom6 - (18.0 * num11 * ln1mxc * log((-w3 - yc) / (-w3 + xc)) * log(1.0 + yc * xcinv)) / denom6 + (18.0 * num11 * ln1pxc * log((-w3 - yc) / (-w3 + xc)) * log(1.0 + yc * xcinv)) / denom6 + (18.0 * num11 * ln1mxc * log((w3 - yc) / (w3 + xc)) * log(1.0 + yc * xcinv)) / denom6
                - (18.0 * num11 * ln1pxc * log((w3 - yc) / (w3 + xc)) * log(1.0 + yc * xcinv)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 - xc) / (-1.0 - w3), 1.0 + xc) * power_of<2>(log(-(w3 / (-1.0 - xc)))) * my_sign(-imxc)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 - xc) / (-1.0 + w3), 1.0 + xc) * power_of<2>(log(w3 / (-1.0 - xc))) * my_sign(-imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 - w3, 1.0 + xc) * power_of<2>(log((-w3 - xc) / ((1.0 - w3) * (-1.0 - xc)))) * my_sign(-imxc)) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 + w3, 1.0 + xc) * power_of<2>(log((w3 - xc) / ((1.0 + w3) * (-1.0 - xc)))) * my_sign(-imxc)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 + xc) / (1.0 - w3), 1.0 + xc) * power_of<2>(log(-(w3 / (1.0 + xc)))) * my_sign(-imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 + xc) / (1.0 + w3), 1.0 + xc) * power_of<2>(log(w3 / (1.0 + xc))) * my_sign(-imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 - w3, 1.0 + xc) * power_of<2>(log((-w3 - xc) / ((-1.0 + w3) * (1.0 + xc)))) * my_sign(-imxc)) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 + w3, 1.0 + xc) * power_of<2>(log((w3 - xc) / ((-1.0 - w3) * (1.0 + xc)))) * my_sign(-imxc)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 - xc) / (1.0 - w3), 1.0 - xc) * power_of<2>(log(-(w3 / (1.0 - xc)))) * my_sign(imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 - xc) / (1.0 + w3), 1.0 - xc) * power_of<2>(log(w3 / (1.0 - xc))) * my_sign(imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 + xc) / (-1.0 - w3), 1.0 - xc) * power_of<2>(log(-(w3 / (-1.0 + xc)))) * my_sign(imxc)) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 + xc) / (-1.0 + w3), 1.0 - xc) * power_of<2>(log(w3 / (-1.0 + xc))) * my_sign(imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 - w3, 1.0 - xc) * power_of<2>(log((-w3 + xc) / ((-1.0 + w3) * (1.0 - xc)))) * my_sign(imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 - w3, 1.0 - xc) * power_of<2>(log((-w3 + xc) / ((1.0 - w3) * (-1.0 + xc)))) * my_sign(imxc)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 + w3, 1.0 - xc) * power_of<2>(log((w3 + xc) / ((-1.0 - w3) * (1.0 - xc)))) * my_sign(imxc)) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * H1(1.0 + w3, 1.0 - xc) * power_of<2>(log((w3 + xc) / ((1.0 + w3) * (-1.0 + xc)))) * my_sign(imxc)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 - w3) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 + w3) * (-1.0 - yc)) / (2.0 * (1.0 + w3)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom6 - ((144.0 * 1.0i) * num10 * M_PI * H1((-1.0 + xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 - xc) * (-1.0 - yc)) / (2.0 * (1.0 - xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom7
                + ((72.0 * 1.0i) * num9 * M_PI * H1((-1.0 + xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 - xc) * (-1.0 - yc)) / (2.0 * (1.0 - xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom6 - ((144.0 * 1.0i) * num10 * M_PI * H1((-1.0 - xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 + xc) * (-1.0 - yc)) / (2.0 * (1.0 + xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom7 + ((72.0 * 1.0i) * num9 * M_PI * H1((-1.0 - xc) / (-1.0 - yc), -2.0 / (-1.0 - yc)) * power_of<2>(log(((-1.0 + xc) * (-1.0 - yc)) / (2.0 * (1.0 + xc)))) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1(2.0 / (1.0 + w3), -2.0 / (-1.0 - yc)) * power_of<2>(log((w3 - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom6
                + ((144.0 * 1.0i) * num10 * M_PI * H1(2.0 / (1.0 - xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((-xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * H1(2.0 / (1.0 - xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((-xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * H1(2.0 / (1.0 + xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * H1(2.0 / (1.0 + xc), -2.0 / (-1.0 - yc)) * power_of<2>(log((xc - yc) / 2.0)) * my_sign(-2.0 * imag(1.0 / (-1.0 - yc)))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 + w3) / (-1.0 - yc), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log(-(((-w3 - xc) * (-1.0 - yc)) / ((1.0 - w3) * (-1.0 - xc))))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 - w3) / (-1.0 - yc), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log(-(((w3 - xc) * (-1.0 - yc)) / ((1.0 + w3) * (-1.0 - xc))))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 + xc) / (1.0 - w3), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log((-w3 - yc) / (1.0 + xc))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc)))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 + xc) / (1.0 + w3), (-1.0 - xc) / (-1.0 - yc)) * power_of<2>(log((w3 - yc) / (1.0 + xc))) * my_sign(imag((-1.0 - xc) / (-1.0 - yc)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 + w3) / (-1.0 - yc), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log(-(((-w3 + xc) * (-1.0 - yc)) / ((1.0 - w3) * (-1.0 + xc))))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 - w3) / (-1.0 - yc), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log(-(((w3 + xc) * (-1.0 - yc)) / ((1.0 + w3) * (-1.0 + xc))))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc)))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 - xc) / (1.0 - w3), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log((-w3 - yc) / (1.0 - xc))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 - xc) / (1.0 + w3), (-1.0 + xc) / (-1.0 - yc)) * power_of<2>(log((w3 - yc) / (1.0 - xc))) * my_sign(imag((-1.0 + xc) / (-1.0 - yc)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 + w3) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 - w3) * (1.0 - yc)) / (-1.0 - w3)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom6
                - ((144.0 * 1.0i) * num10 * M_PI * H1((1.0 + xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 - xc) * (1.0 - yc)) / (-1.0 - xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom7 + ((72.0 * 1.0i) * num9 * M_PI * H1((1.0 + xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 - xc) * (1.0 - yc)) / (-1.0 - xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom6 - ((144.0 * 1.0i) * num10 * M_PI * H1((1.0 - xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 + xc) * (1.0 - yc)) / (-1.0 + xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom7
                + ((72.0 * 1.0i) * num9 * M_PI * H1((1.0 - xc) / (1.0 - yc), 2.0 / (1.0 - yc)) * power_of<2>(log((-0.5) * (((1.0 + xc) * (1.0 - yc)) / (-1.0 + xc)))) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1(-2.0 / (-1.0 - w3), 2.0 / (1.0 - yc)) * power_of<2>(log((w3 + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * H1(-2.0 / (-1.0 + xc), 2.0 / (1.0 - yc)) * power_of<2>(log((-xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * H1(-2.0 / (-1.0 + xc), 2.0 / (1.0 - yc)) * power_of<2>(log((-xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom6
                + ((144.0 * 1.0i) * num10 * M_PI * H1(-2.0 / (-1.0 - xc), 2.0 / (1.0 - yc)) * power_of<2>(log((xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * H1(-2.0 / (-1.0 - xc), 2.0 / (1.0 - yc)) * power_of<2>(log((xc + yc) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yc)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 + w3) / (1.0 - yc), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log(-(((-w3 - xc) * (1.0 - yc)) / ((-1.0 - w3) * (1.0 - xc))))) * my_sign(imag((1.0 - xc) / (1.0 - yc)))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 - w3) / (1.0 - yc), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log(-(((w3 - xc) * (1.0 - yc)) / ((-1.0 + w3) * (1.0 - xc))))) * my_sign(imag((1.0 - xc) / (1.0 - yc)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 + xc) / (-1.0 - w3), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log((-w3 - yc) / (-1.0 + xc))) * my_sign(imag((1.0 - xc) / (1.0 - yc)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 + xc) / (-1.0 + w3), (1.0 - xc) / (1.0 - yc)) * power_of<2>(log((w3 - yc) / (-1.0 + xc))) * my_sign(imag((1.0 - xc) / (1.0 - yc)))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 + w3) / (1.0 - yc), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log(-(((-w3 + xc) * (1.0 - yc)) / ((-1.0 - w3) * (1.0 + xc))))) * my_sign(imag((1.0 + xc) / (1.0 - yc)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((1.0 - w3) / (1.0 - yc), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log(-(((w3 + xc) * (1.0 - yc)) / ((-1.0 + w3) * (1.0 + xc))))) * my_sign(imag((1.0 + xc) / (1.0 - yc)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 - xc) / (-1.0 - w3), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log((-w3 - yc) / (-1.0 - xc))) * my_sign(imag((1.0 + xc) / (1.0 - yc)))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * H1((-1.0 - xc) / (-1.0 + w3), (1.0 + xc) / (1.0 - yc)) * power_of<2>(log((w3 - yc) / (-1.0 - xc))) * my_sign(imag((1.0 + xc) / (1.0 - yc)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1(-w3inv, -ycinv) * power_of<2>(log(-w3 + yc)) * my_sign(-imycinv)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1(w3inv, -ycinv) * power_of<2>(log(w3 + yc)) * my_sign(-imycinv)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * H1(-w3inv, ycinv) * power_of<2>(log(-w3 - yc)) * my_sign(imycinv)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * H1(w3inv, ycinv) * power_of<2>(log(w3 - yc)) * my_sign(imycinv)) / denom6
                + ((36.0 * 1.0i) * num11 * M_PI * power_of<2>(log((1.0 - w3) / 2.0)) * my_sign(imw3 / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - w3) / 2.0)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(-(w3 / (-1.0 - w3)))) * my_sign(imag(1.0 / (-1.0 - w3))) * T(1.0, 1.0 / 2.0, -(w3 / (-1.0 - w3)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(w3 / (1.0 + w3))) * my_sign(-imag(1.0 / (1.0 + w3))) * T(1.0, 1.0 / 2.0, w3 / (1.0 + w3))) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * power_of<2>(log((1.0 - xc) / 2.0)) * my_sign(imxc / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - xc) / 2.0)) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * power_of<2>(log((1.0 - xc) / 2.0)) * my_sign(imxc / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - xc) / 2.0)) / denom6
                + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log(-(xc / (-1.0 - xc)))) * my_sign(imag(1.0 / (-1.0 - xc))) * T(1.0, 1.0 / 2.0, -(xc / (-1.0 - xc)))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log(-(xc / (-1.0 - xc)))) * my_sign(imag(1.0 / (-1.0 - xc))) * T(1.0, 1.0 / 2.0, -(xc / (-1.0 - xc)))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log(-(xc / (1.0 - xc)))) * my_sign(-imag(1.0 / (1.0 - xc))) * T(1.0, 1.0 / 2.0, -(xc / (1.0 - xc)))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log(-(xc / (1.0 - xc)))) * my_sign(-imag(1.0 / (1.0 - xc))) * T(1.0, 1.0 / 2.0, -(xc / (1.0 - xc)))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(lnr_xc2) * my_sign(imag(1.0 / (-1.0 + xc))) * T(1.0, 1.0 / 2.0, xc / (-1.0 + xc))) / denom7
                - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(lnr_xc2) * my_sign(imag(1.0 / (-1.0 + xc))) * T(1.0, 1.0 / 2.0, xc / (-1.0 + xc))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(lnr_xc1) * my_sign(-imag(1.0 / (1.0 + xc))) * T(1.0, 1.0 / 2.0, xc / (1.0 + xc))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(lnr_xc1) * my_sign(-imag(1.0 / (1.0 + xc))) * T(1.0, 1.0 / 2.0, xc / (1.0 + xc))) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * power_of<2>(log((1.0 + xc) / 2.0)) * my_sign((-0.5) * imxc) * T(1.0, 1.0 / 2.0, (1.0 + xc) / 2.0)) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * power_of<2>(log((1.0 + xc) / 2.0)) * my_sign((-0.5) * imxc) * T(1.0, 1.0 / 2.0, (1.0 + xc) / 2.0)) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(-(w3 / (-1.0 - w3)))) * my_sign(imag(1.0 / (-1.0 - w3))) * T(1.0, -(xc / (-1.0 - xc)), -(w3 / (-1.0 - w3)))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(w3 / (-1.0 + w3))) * my_sign(imag(1.0 / (-1.0 + w3))) * T(1.0, -(xc / (-1.0 - xc)), w3 / (-1.0 + w3))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 + w3) / (-1.0 - xc))) * T(1.0, -(xc / (-1.0 - xc)), (-w3 - xc) / (-1.0 - xc))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 - w3) / (-1.0 - xc))) * T(1.0, -(xc / (-1.0 - xc)), (w3 - xc) / (-1.0 - xc))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(-(w3 / (1.0 - w3)))) * my_sign(-imag(1.0 / (1.0 - w3))) * T(1.0, -(xc / (1.0 - xc)), -(w3 / (1.0 - w3)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(w3 / (1.0 + w3))) * my_sign(-imag(1.0 / (1.0 + w3))) * T(1.0, -(xc / (1.0 - xc)), w3 / (1.0 + w3))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 + w3) / (1.0 - xc))) * T(1.0, -(xc / (1.0 - xc)), (-w3 - xc) / (1.0 - xc))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 - w3) / (1.0 - xc))) * T(1.0, -(xc / (1.0 - xc)), (w3 - xc) / (1.0 - xc))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(-(w3 / (-1.0 - w3)))) * my_sign(imag(1.0 / (-1.0 - w3))) * T(1.0, xc / (-1.0 + xc), -(w3 / (-1.0 - w3)))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(w3 / (-1.0 + w3))) * my_sign(imag(1.0 / (-1.0 + w3))) * T(1.0, xc / (-1.0 + xc), w3 / (-1.0 + w3))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 + w3) / (-1.0 + xc))) * T(1.0, xc / (-1.0 + xc), (-w3 + xc) / (-1.0 + xc))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 - w3) / (-1.0 + xc))) * T(1.0, xc / (-1.0 + xc), (w3 + xc) / (-1.0 + xc))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(-(w3 / (1.0 - w3)))) * my_sign(-imag(1.0 / (1.0 - w3))) * T(1.0, xc / (1.0 + xc), -(w3 / (1.0 - w3)))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(w3 / (1.0 + w3))) * my_sign(-imag(1.0 / (1.0 + w3))) * T(1.0, xc / (1.0 + xc), w3 / (1.0 + w3))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 + w3) / (1.0 + xc))) * T(1.0, xc / (1.0 + xc), (-w3 + xc) / (1.0 + xc))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 - w3) / (1.0 + xc))) * T(1.0, xc / (1.0 + xc), (w3 + xc) / (1.0 + xc))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((1.0 - w3) / 2.0)) * my_sign(imw3 / 2.0) * T(1.0, (1.0 - yc) / 2.0, (1.0 - w3) / 2.0)) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log((1.0 - w3) / 2.0) * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 - w3) / 2.0)) / denom6 - ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((1.0 - xc) / 2.0)) * my_sign(imxc / 2.0) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0)) / denom7
                + ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((1.0 - xc) / 2.0)) * my_sign(imxc / 2.0) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0)) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * ln1mxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0)) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * ln1mxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 - xc) / 2.0)) / denom6 - ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((1.0 + xc) / 2.0)) * my_sign((-0.5) * imxc) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0)) / denom7 + ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((1.0 + xc) / 2.0)) * my_sign((-0.5) * imxc) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0)) / denom6
                + ((288.0 * 1.0i) * num10 * M_PI * ln1pxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0)) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * ln1pxchalf * ln1pyc * my_sign(-2.0 * imag(1.0 / (-1.0 - yc))) * T(1.0, (1.0 - yc) / 2.0, (1.0 + xc) / 2.0)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - yc) / (1.0 + w3))) * my_sign(imag((1.0 + yc) / (1.0 + w3))) * T(1.0, (1.0 - yc) / 2.0, (w3 - yc) / (1.0 + w3))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((-xc - yc) / (1.0 - xc))) * my_sign(imag((1.0 + yc) / (1.0 - xc))) * T(1.0, (1.0 - yc) / 2.0, (-xc - yc) / (1.0 - xc))) / denom7
                - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((-xc - yc) / (1.0 - xc))) * my_sign(imag((1.0 + yc) / (1.0 - xc))) * T(1.0, (1.0 - yc) / 2.0, (-xc - yc) / (1.0 - xc))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((xc - yc) / (1.0 + xc))) * my_sign(imag((1.0 + yc) / (1.0 + xc))) * T(1.0, (1.0 - yc) / 2.0, (xc - yc) / (1.0 + xc))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((xc - yc) / (1.0 + xc))) * my_sign(imag((1.0 + yc) / (1.0 + xc))) * T(1.0, (1.0 - yc) / 2.0, (xc - yc) / (1.0 + xc))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(1.0 - w3)) * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) / denom6
                - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3) * log((-w3 - xc) / (-1.0 - xc)) * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3) * log((-w3 + xc) / (-1.0 + xc)) * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) / denom6 + dilog((-1.0 + w3) / (-1.0 - xc)) * ((-18.0 * num11 * log(w3 / (-1.0 + w3))) / denom6 + (18.0 * num11 * log((w3 - yc) / (-1.0 + w3))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) / denom6);

            const complex<double> f29cPart6 = + dilog((-1.0 + w3) / (-1.0 + xc)) * ((-18.0 * num11 * log(w3 / (-1.0 + w3))) / denom6 + (18.0 * num11 * log((w3 - yc) / (-1.0 + w3))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) / denom6) - ((36.0 * 1.0i) * num11 * M_PI * log((1.0 - w3) / 2.0) * log(1.0 + w3) * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3) * log((w3 - xc) / (-1.0 - xc)) * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6
                + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3) * log((w3 + xc) / (-1.0 + xc)) * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6 + dilog(-1.0 / (-1.0 - xc)) * ((72.0 * num6) / denom5 - (144.0 * num10 * lnhalf) / denom7 + (72.0 * num9 * lnhalf) / denom6 - (18.0 * num11 * log(-(w3 / (-1.0 - w3)))) / denom6 + (18.0 * num11 * log(w3 / (-1.0 + w3))) / denom6 - (36.0 * num9 * ln1myc) / denom6 + (18.0 * num11 * log((-w3 - yc) / (-1.0 - w3))) / denom6 - (18.0 * num11 * log((w3 - yc) / (-1.0 + w3))) / denom6 + (36.0 * num9 * ln1pyc) / denom6
                  + ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6) + dilog(-1.0 / (-1.0 + xc)) * ((72.0 * num6) / denom5 - (144.0 * num10 * lnhalf) / denom7 + (72.0 * num9 * lnhalf) / denom6 - (18.0 * num11 * log(-(w3 / (-1.0 - w3)))) / denom6 + (18.0 * num11 * log(w3 / (-1.0 + w3))) / denom6 - (36.0 * num9 * ln1myc) / denom6 + (18.0 * num11 * log((-w3 - yc) / (-1.0 - w3))) / denom6 - (18.0 * num11 * log((w3 - yc) / (-1.0 + w3))) / denom6
                  + (36.0 * num9 * ln1pyc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 - yc, 1.0 - w3)) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6) + dilog((-1.0 - w3) / (-1.0 - xc)) * ((18.0 * num11 * log(-(w3 / (-1.0 - w3)))) / denom6 - (18.0 * num11 * log((-w3 - yc) / (-1.0 - w3))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6)
                + dilog((-1.0 - w3) / (-1.0 + xc)) * ((18.0 * num11 * log(-(w3 / (-1.0 - w3)))) / denom6 - (18.0 * num11 * log((-w3 - yc) / (-1.0 - w3))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6) + ((144.0 * 1.0i) * num6 * M_PI * ln1mxc * my_sign(imag(xcinv)) * t11myc1mxc) / denom5 + ((72.0 * 1.0i) * num9 * M_PI * ln1mxc * ln1pyc * my_sign(imag(xcinv)) * t11myc1mxc) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11myc1mxc) / denom7
                - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11myc1mxc) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((-w3 - xc) / (-1.0 - w3)) * my_sign(imxc) * t11myc1mxc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((w3 - xc) / (-1.0 + w3)) * my_sign(imxc) * t11myc1mxc) / denom6 - ((288.0 * 1.0i) * num10 * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11myc1mxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11myc1mxc) / denom6
                + ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((-w3 - yc) / (-1.0 - w3)) * my_sign(imxc) * t11myc1mxc) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((w3 - yc) / (-1.0 + w3)) * my_sign(imxc) * t11myc1mxc) / denom6 - ((72.0 * 1.0i) * num9 * M_PI * ln1mxc * ln1myc * my_sign(imycinv) * t11myc1mxc) / denom6 + ((144.0 * 1.0i) * num6 * M_PI * ln1pxc * my_sign(-imag(xcinv)) * t11myc1pxc) / denom5 + ((72.0 * 1.0i) * num9 * M_PI * ln1pxc * ln1pyc * my_sign(-imag(xcinv)) * t11myc1pxc) / denom6
                - ((288.0 * 1.0i) * num10 * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11myc1pxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11myc1pxc) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11myc1pxc) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11myc1pxc) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * ln1pxc * log((-w3 + xc) / (-1.0 - w3)) * my_sign(-imxc) * t11myc1pxc) / denom6
                + ((36.0 * 1.0i) * num11 * M_PI * ln1pxc * log((w3 + xc) / (-1.0 + w3)) * my_sign(-imxc) * t11myc1pxc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * ln1pxc * log((-w3 - yc) / (-1.0 - w3)) * my_sign(-imxc) * t11myc1pxc) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * ln1pxc * log((w3 - yc) / (-1.0 + w3)) * my_sign(-imxc) * t11myc1pxc) / denom6 - ((72.0 * 1.0i) * num9 * M_PI * ln1pxc * ln1myc * my_sign(imycinv) * t11myc1pxc) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(1.0 - yc / w3)) * my_sign(imag(yc / w3)) * T(1.0, 1.0 - yc, 1.0 - yc / w3)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(1.0 + yc / w3)) * my_sign(-imag(yc / w3)) * T(1.0, 1.0 - yc, 1.0 + yc / w3)) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 + w3) / (-1.0 - xc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (-w3 - xc) / (-1.0 - xc))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log((-w3 - xc) / (-1.0 - xc)) * ln1myc * my_sign(imag((1.0 + xc) / (1.0 - yc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (-w3 - xc) / (-1.0 - xc))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - xc) / (-1.0 - xc))) * my_sign(imag((-1.0 - w3) / (-1.0 - xc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (w3 - xc) / (-1.0 - xc))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * log((w3 - xc) / (-1.0 - xc)) * ln1myc * my_sign(imag((1.0 + xc) / (1.0 - yc))) * T(1.0, (-xc - yc) / (-1.0 - xc), (w3 - xc) / (-1.0 - xc))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - yc) / (-1.0 - w3))) * my_sign(imag((-1.0 + yc) / (-1.0 - w3))) * T(1.0, (-xc - yc) / (-1.0 - xc), (-w3 - yc) / (-1.0 - w3))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - yc) / (-1.0 + w3))) * my_sign(imag((-1.0 + yc) / (-1.0 + w3))) * T(1.0, (-xc - yc) / (-1.0 - xc), (w3 - yc) / (-1.0 + w3))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 + w3) / (1.0 - xc))) * T(1.0, (-xc - yc) / (1.0 - xc), (-w3 - xc) / (1.0 - xc))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * log((-w3 - xc) / (1.0 - xc)) * ln1pyc * my_sign(imag((-1.0 + xc) / (-1.0 - yc))) * T(1.0, (-xc - yc) / (1.0 - xc), (-w3 - xc) / (1.0 - xc))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - xc) / (1.0 - xc))) * my_sign(imag((1.0 - w3) / (1.0 - xc))) * T(1.0, (-xc - yc) / (1.0 - xc), (w3 - xc) / (1.0 - xc))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log((w3 - xc) / (1.0 - xc)) * ln1pyc * my_sign(imag((-1.0 + xc) / (-1.0 - yc))) * T(1.0, (-xc - yc) / (1.0 - xc), (w3 - xc) / (1.0 - xc))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - yc) / (1.0 - w3))) * my_sign(imag((1.0 + yc) / (1.0 - w3))) * T(1.0, (-xc - yc) / (1.0 - xc), (-w3 - yc) / (1.0 - w3))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - yc) / (1.0 + w3))) * my_sign(imag((1.0 + yc) / (1.0 + w3))) * T(1.0, (-xc - yc) / (1.0 - xc), (w3 - yc) / (1.0 + w3))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 + w3) / (-1.0 + xc))) * T(1.0, (xc - yc) / (-1.0 + xc), (-w3 + xc) / (-1.0 + xc))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log((-w3 + xc) / (-1.0 + xc)) * ln1myc * my_sign(imag((1.0 - xc) / (1.0 - yc))) * T(1.0, (xc - yc) / (-1.0 + xc), (-w3 + xc) / (-1.0 + xc))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 + xc) / (-1.0 + xc))) * my_sign(imag((-1.0 - w3) / (-1.0 + xc))) * T(1.0, (xc - yc) / (-1.0 + xc), (w3 + xc) / (-1.0 + xc))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * log((w3 + xc) / (-1.0 + xc)) * ln1myc * my_sign(imag((1.0 - xc) / (1.0 - yc))) * T(1.0, (xc - yc) / (-1.0 + xc), (w3 + xc) / (-1.0 + xc))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - yc) / (-1.0 - w3))) * my_sign(imag((-1.0 + yc) / (-1.0 - w3))) * T(1.0, (xc - yc) / (-1.0 + xc), (-w3 - yc) / (-1.0 - w3))) / denom6
                + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - yc) / (-1.0 + w3))) * my_sign(imag((-1.0 + yc) / (-1.0 + w3))) * T(1.0, (xc - yc) / (-1.0 + xc), (w3 - yc) / (-1.0 + w3))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 + w3) / (1.0 + xc))) * T(1.0, (xc - yc) / (1.0 + xc), (-w3 + xc) / (1.0 + xc))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * log((-w3 + xc) / (1.0 + xc)) * ln1pyc * my_sign(imag((-1.0 - xc) / (-1.0 - yc))) * T(1.0, (xc - yc) / (1.0 + xc), (-w3 + xc) / (1.0 + xc))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 + xc) / (1.0 + xc))) * my_sign(imag((1.0 - w3) / (1.0 + xc))) * T(1.0, (xc - yc) / (1.0 + xc), (w3 + xc) / (1.0 + xc))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log((w3 + xc) / (1.0 + xc)) * ln1pyc * my_sign(imag((-1.0 - xc) / (-1.0 - yc))) * T(1.0, (xc - yc) / (1.0 + xc), (w3 + xc) / (1.0 + xc))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - yc) / (1.0 - w3))) * my_sign(imag((1.0 + yc) / (1.0 - w3))) * T(1.0, (xc - yc) / (1.0 + xc), (-w3 - yc) / (1.0 - w3))) / denom6
                - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((w3 - yc) / (1.0 + w3))) * my_sign(imag((1.0 + yc) / (1.0 + w3))) * T(1.0, (xc - yc) / (1.0 + xc), (w3 - yc) / (1.0 + w3))) / denom6 - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((1.0 - w3) / 2.0)) * my_sign(imw3 / 2.0) * T(1.0, (1.0 + yc) / 2.0, (1.0 - w3) / 2.0)) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log((1.0 - w3) / 2.0) * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 - w3) / 2.0)) / denom6 - ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((1.0 - xc) / 2.0)) * my_sign(imxc / 2.0) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0)) / denom7
                + ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((1.0 - xc) / 2.0)) * my_sign(imxc / 2.0) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0)) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * ln1mxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0)) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * ln1mxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 - xc) / 2.0)) / denom6 - ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((1.0 + xc) / 2.0)) * my_sign((-0.5) * imxc) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0)) / denom7 + ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((1.0 + xc) / 2.0)) * my_sign((-0.5) * imxc) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0)) / denom6
                + ((288.0 * 1.0i) * num10 * M_PI * ln1pxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0)) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * ln1pxchalf * ln1myc * my_sign(2.0 * imag(1.0 / (1.0 - yc))) * T(1.0, (1.0 + yc) / 2.0, (1.0 + xc) / 2.0)) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log((-w3 - yc) / (-1.0 - w3))) * my_sign(imag((-1.0 + yc) / (-1.0 - w3))) * T(1.0, (1.0 + yc) / 2.0, (-w3 - yc) / (-1.0 - w3))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((-xc - yc) / (-1.0 - xc))) * my_sign(imag((-1.0 + yc) / (-1.0 - xc))) * T(1.0, (1.0 + yc) / 2.0, (-xc - yc) / (-1.0 - xc))) / denom7
                - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((-xc - yc) / (-1.0 - xc))) * my_sign(imag((-1.0 + yc) / (-1.0 - xc))) * T(1.0, (1.0 + yc) / 2.0, (-xc - yc) / (-1.0 - xc))) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(log((xc - yc) / (-1.0 + xc))) * my_sign(imag((-1.0 + yc) / (-1.0 + xc))) * T(1.0, (1.0 + yc) / 2.0, (xc - yc) / (-1.0 + xc))) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(log((xc - yc) / (-1.0 + xc))) * my_sign(imag((-1.0 + yc) / (-1.0 + xc))) * T(1.0, (1.0 + yc) / 2.0, (xc - yc) / (-1.0 + xc))) / denom6 + ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(1.0 - w3)) * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) / denom6
                - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3) * log((w3 - xc) / (1.0 - xc)) * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3) * log((w3 + xc) / (1.0 + xc)) * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) / denom6 + dilog((1.0 - w3) / (1.0 - xc)) * ((-18.0 * num11 * log(-(w3 / (1.0 - w3)))) / denom6 + (18.0 * num11 * log((-w3 - yc) / (1.0 - w3))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) / denom6);

            const complex<double> f29cPart7 = + dilog((1.0 - w3) / (1.0 + xc)) * ((-18.0 * num11 * log(-(w3 / (1.0 - w3)))) / denom6 + (18.0 * num11 * log((-w3 - yc) / (1.0 - w3))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) / denom6) - ((36.0 * 1.0i) * num11 * M_PI * log((1.0 - w3) / 2.0) * log(1.0 + w3) * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3) * log((-w3 - xc) / (1.0 - xc)) * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6
                + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3) * log((-w3 + xc) / (1.0 + xc)) * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6 + dilog((1.0 + w3) / 2.0) * ((-18.0 * num11 * log(-(w3 / (-1.0 - w3)))) / denom6 - (18.0 * num11 * log(w3 / (1.0 + w3))) / denom6 + (18.0 * num11 * log((-w3 - yc) / (-1.0 - w3))) / denom6 + (18.0 * num11 * log((w3 - yc) / (1.0 + w3))) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6)
                + dilog(1.0 / (1.0 - xc)) * ((-72.0 * num6) / denom5 - (144.0 * num10 * lnhalf) / denom7 + (72.0 * num9 * lnhalf) / denom6 + (18.0 * num11 * log(-(w3 / (1.0 - w3)))) / denom6 - (18.0 * num11 * log(w3 / (1.0 + w3))) / denom6 + (36.0 * num9 * ln1myc) / denom6 - (18.0 * num11 * log((-w3 - yc) / (1.0 - w3))) / denom6 + (18.0 * num11 * log((w3 - yc) / (1.0 + w3))) / denom6 - (36.0 * num9 * ln1pyc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) / denom6
                  - ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6) + dilog(1.0 / (1.0 + xc)) * ((-72.0 * num6) / denom5 - (144.0 * num10 * lnhalf) / denom7 + (72.0 * num9 * lnhalf) / denom6 + (18.0 * num11 * log(-(w3 / (1.0 - w3)))) / denom6 - (18.0 * num11 * log(w3 / (1.0 + w3))) / denom6 + (36.0 * num9 * ln1myc) / denom6 - (18.0 * num11 * log((-w3 - yc) / (1.0 - w3))) / denom6 + (18.0 * num11 * log((w3 - yc) / (1.0 + w3))) / denom6 - (36.0 * num9 * ln1pyc) / denom6
                  + ((36.0 * 1.0i) * num11 * M_PI * my_sign(imw3) * T(1.0, 1.0 + yc, 1.0 - w3)) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6) + dilog((1.0 + w3) / (1.0 - xc)) * ((18.0 * num11 * log(w3 / (1.0 + w3))) / denom6 - (18.0 * num11 * log((w3 - yc) / (1.0 + w3))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6)
                + dilog((1.0 + w3) / (1.0 + xc)) * ((18.0 * num11 * log(w3 / (1.0 + w3))) / denom6 - (18.0 * num11 * log((w3 - yc) / (1.0 + w3))) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6) - ((144.0 * 1.0i) * num6 * M_PI * ln1mxc * my_sign(imag(xcinv)) * t11pyc1mxc) / denom5 + ((72.0 * 1.0i) * num9 * M_PI * ln1mxc * ln1myc * my_sign(imag(xcinv)) * t11pyc1mxc) / denom6 + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11pyc1mxc) / denom7
                - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(ln1mxc) * my_sign(imxc) * t11pyc1mxc) / denom6 - ((288.0 * 1.0i) * num10 * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11pyc1mxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * ln1mxc * ln1pxchalf * my_sign(imxc) * t11pyc1mxc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((-w3 + xc) / (1.0 - w3)) * my_sign(imxc) * t11pyc1mxc) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((w3 + xc) / (1.0 + w3)) * my_sign(imxc) * t11pyc1mxc) / denom6
                - ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((-w3 - yc) / (1.0 - w3)) * my_sign(imxc) * t11pyc1mxc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * ln1mxc * log((w3 - yc) / (1.0 + w3)) * my_sign(imxc) * t11pyc1mxc) / denom6 - ((72.0 * 1.0i) * num9 * M_PI * ln1mxc * ln1pyc * my_sign(-imycinv) * t11pyc1mxc) / denom6
                + dilog((1.0 - xc) / 2.0) * ((36.0 * num3) / denom3 - (72.0 * num9 * ln1mxc) / denom6 - (144.0 * num10 * log(-(xc / (1.0 - xc)))) / denom7 + (72.0 * num9 * log(-(xc / (1.0 - xc)))) / denom6 - (144.0 * num10 * lnr_xc2) / denom7 + (72.0 * num9 * lnr_xc2) / denom6 - (36.0 * num9 * ln1myc) / denom6 + (144.0 * num10 * log((-xc - yc) / (1.0 - xc))) / denom7 - (72.0 * num9 * log((-xc - yc) / (1.0 - xc))) / denom6 + (144.0 * num10 * log((xc - yc) / (-1.0 + xc))) / denom7 - (72.0 * num9 * log((xc - yc) / (-1.0 + xc))) / denom6 - (36.0 * num9 * ln1pyc) / denom6
                  + (36.0 * num11 * log(1.0 - yc / w3)) / denom6 + (36.0 * num11 * log(1.0 + yc / w3)) / denom6 + (144.0 * num10 * log(1.0 - yc * xcinv)) / denom7 + (144.0 * num10 * log(1.0 + yc * xcinv)) / denom7 - ((288.0 * 1.0i) * num10 * M_PI * my_sign(imxc) * t11myc1mxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * my_sign(imxc) * t11myc1mxc) / denom6 - ((288.0 * 1.0i) * num10 * M_PI * my_sign(imxc) * t11pyc1mxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * my_sign(imxc) * t11pyc1mxc) / denom6) - ((144.0 * 1.0i) * num6 * M_PI * ln1pxc * my_sign(-imag(xcinv)) * t11pyc1pxc) / denom5
                + ((72.0 * 1.0i) * num9 * M_PI * ln1pxc * ln1myc * my_sign(-imag(xcinv)) * t11pyc1pxc) / denom6 - ((288.0 * 1.0i) * num10 * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11pyc1pxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * ln1mxchalf * ln1pxc * my_sign(-imxc) * t11pyc1pxc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * log((-w3 - xc) / (1.0 - w3)) * ln1pxc * my_sign(-imxc) * t11pyc1pxc) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * log((w3 - xc) / (1.0 + w3)) * ln1pxc * my_sign(-imxc) * t11pyc1pxc) / denom6
                + ((144.0 * 1.0i) * num10 * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11pyc1pxc) / denom7 - ((72.0 * 1.0i) * num9 * M_PI * power_of<2>(ln1pxc) * my_sign(-imxc) * t11pyc1pxc) / denom6 - ((36.0 * 1.0i) * num11 * M_PI * ln1pxc * log((-w3 - yc) / (1.0 - w3)) * my_sign(-imxc) * t11pyc1pxc) / denom6 + ((36.0 * 1.0i) * num11 * M_PI * ln1pxc * log((w3 - yc) / (1.0 + w3)) * my_sign(-imxc) * t11pyc1pxc) / denom6 - ((72.0 * 1.0i) * num9 * M_PI * ln1pxc * ln1pyc * my_sign(-imycinv) * t11pyc1pxc) / denom6
                + li2half * ((18.0 * num1) / denom1 - (72.0 * num3) / denom3 + (18.0 * num5) / denom4 + (18.0 * num11 * log(-(w3 / (-1.0 - w3)))) / denom6 + (18.0 * num11 * log(w3 / (1.0 + w3))) / denom6 + (144.0 * num10 * log(-(xc / (-1.0 - xc)))) / denom7 - (72.0 * num9 * log(-(xc / (-1.0 - xc)))) / denom6 + (144.0 * num10 * log(-(xc / (1.0 - xc)))) / denom7 - (72.0 * num9 * log(-(xc / (1.0 - xc)))) / denom6 + (144.0 * num10 * lnr_xc2) / denom7 - (72.0 * num9 * lnr_xc2) / denom6 + (144.0 * num10 * lnr_xc1) / denom7 - (72.0 * num9 * lnr_xc1) / denom6
                  + (72.0 * num9 * ln1myc) / denom6 - (18.0 * num11 * log((-w3 - yc) / (-1.0 - w3))) / denom6 - (18.0 * num11 * log((w3 - yc) / (1.0 + w3))) / denom6 - (144.0 * num10 * log((-xc - yc) / (-1.0 - xc))) / denom7 + (72.0 * num9 * log((-xc - yc) / (-1.0 - xc))) / denom6 - (144.0 * num10 * log((-xc - yc) / (1.0 - xc))) / denom7 + (72.0 * num9 * log((-xc - yc) / (1.0 - xc))) / denom6 - (144.0 * num10 * log((xc - yc) / (-1.0 + xc))) / denom7 + (72.0 * num9 * log((xc - yc) / (-1.0 + xc))) / denom6 - (144.0 * num10 * log((xc - yc) / (1.0 + xc))) / denom7 + (72.0 * num9 * log((xc - yc) / (1.0 + xc))) / denom6
                  + (72.0 * num9 * ln1pyc) / denom6 - (72.0 * num11 * log(1.0 - yc / w3)) / denom6 - (72.0 * num11 * log(1.0 + yc / w3)) / denom6 - (288.0 * num10 * log(1.0 - yc * xcinv)) / denom7 - (288.0 * num10 * log(1.0 + yc * xcinv)) / denom7 + ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 - yc, 1.0 + w3)) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * my_sign(imxc) * t11myc1mxc) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * my_sign(imxc) * t11myc1mxc) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * my_sign(-imxc) * t11myc1pxc) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * my_sign(-imxc) * t11myc1pxc) / denom6
                  + ((36.0 * 1.0i) * num11 * M_PI * my_sign(-imw3) * T(1.0, 1.0 + yc, 1.0 + w3)) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * my_sign(imxc) * t11pyc1mxc) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * my_sign(imxc) * t11pyc1mxc) / denom6 + ((288.0 * 1.0i) * num10 * M_PI * my_sign(-imxc) * t11pyc1pxc) / denom7 - ((144.0 * 1.0i) * num9 * M_PI * my_sign(-imxc) * t11pyc1pxc) / denom6)
                + dilog((1.0 + xc) / 2.0) * ((36.0 * num3) / denom3 - (144.0 * num10 * log(-(xc / (-1.0 - xc)))) / denom7 + (72.0 * num9 * log(-(xc / (-1.0 - xc)))) / denom6 - (144.0 * num10 * lnr_xc1) / denom7 + (72.0 * num9 * lnr_xc1) / denom6 - (72.0 * num9 * ln1pxc) / denom6 - (36.0 * num9 * ln1myc) / denom6 + (144.0 * num10 * log((-xc - yc) / (-1.0 - xc))) / denom7 - (72.0 * num9 * log((-xc - yc) / (-1.0 - xc))) / denom6 + (144.0 * num10 * log((xc - yc) / (1.0 + xc))) / denom7 - (72.0 * num9 * log((xc - yc) / (1.0 + xc))) / denom6 - (36.0 * num9 * ln1pyc) / denom6
                  + (36.0 * num11 * log(1.0 - yc / w3)) / denom6 + (36.0 * num11 * log(1.0 + yc / w3)) / denom6 + (144.0 * num10 * log(1.0 - yc * xcinv)) / denom7 + (144.0 * num10 * log(1.0 + yc * xcinv)) / denom7 - ((288.0 * 1.0i) * num10 * M_PI * my_sign(-imxc) * t11myc1pxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * my_sign(-imxc) * t11myc1pxc) / denom6 - ((288.0 * 1.0i) * num10 * M_PI * my_sign(-imxc) * t11pyc1pxc) / denom7 + ((144.0 * 1.0i) * num9 * M_PI * my_sign(-imxc) * t11pyc1pxc) / denom6) - ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(1.0 - yc / w3)) * my_sign(imag(yc / w3)) * T(1.0, 1.0 + yc, 1.0 - yc / w3)) / denom6;

            const complex<double> f29cPart8 = (+ ((18.0 * 1.0i) * num11 * M_PI * power_of<2>(log(1.0 + yc / w3)) * my_sign(-imag(yc / w3)) * T(1.0, 1.0 + yc, 1.0 + yc / w3)) + ((72.0 * 1.0i) * num9 * M_PI * log(1.0 - xcinv) * ln1mxc * my_sign(imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 - xcinv)) - ((72.0 * 1.0i) * num9 * M_PI * log(1.0 - xcinv) * ln1pxc * my_sign(imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 - xcinv)) - ((72.0 * 1.0i) * num9 * M_PI * log(1.0 + xcinv) * ln1mxc * my_sign(-imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 + xcinv)) + ((72.0 * 1.0i) * num9 * M_PI * log(1.0 + xcinv) * ln1pxc * my_sign(-imxc) * T(1.0, 1.0 - yc * xcinv, 1.0 + xcinv))
                - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3 * xcinv) * ln1mxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 - w3 * xcinv)) + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3 * xcinv) * ln1pxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 - w3 * xcinv)) + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3 * xcinv) * ln1mxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 + w3 * xcinv)) - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3 * xcinv) * ln1pxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 - yc * xcinv, 1.0 + w3 * xcinv))
                + ((72.0 * 1.0i) * num9 * M_PI * log(1.0 - xcinv) * ln1mxc * my_sign(imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 - xcinv)) - ((72.0 * 1.0i) * num9 * M_PI * log(1.0 - xcinv) * ln1pxc * my_sign(imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 - xcinv)) - ((72.0 * 1.0i) * num9 * M_PI * log(1.0 + xcinv) * ln1mxc * my_sign(-imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 + xcinv)) + ((72.0 * 1.0i) * num9 * M_PI * log(1.0 + xcinv) * ln1pxc * my_sign(-imxc) * T(1.0, 1.0 + yc * xcinv, 1.0 + xcinv)) - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3 * xcinv) * ln1mxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 - w3 * xcinv))
                + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 - w3 * xcinv) * ln1pxc * my_sign(imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 - w3 * xcinv)) + ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3 * xcinv) * ln1mxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 + w3 * xcinv)) - ((36.0 * 1.0i) * num11 * M_PI * log(1.0 + w3 * xcinv) * ln1pxc * my_sign(-imag(xc / w3)) * T(1.0, 1.0 + yc * xcinv, 1.0 + w3 * xcinv))
                - (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log((1.0 + xc) / w3) + log(((-1.0 + w3) * (1.0 + xc)) / (w3 + xc)) / 6.0 - log(((-1.0 + w3) * (1.0 + xc)) / ((w3 + xc) * (1.0 - yc))) / 6.0 + log((1.0 + xc) / (w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11myc1pxc * T(p(-xc, 1.0 - yc), 1.0 - yc, 1.0 - w3)))
                + (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log(-((1.0 + xc) / w3)) + log(((-1.0 - w3) * (1.0 + xc)) / (-w3 + xc)) / 6.0 - log(((-1.0 - w3) * (1.0 + xc)) / ((-w3 + xc) * (1.0 - yc))) / 6.0 + log((1.0 + xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11myc1pxc * T(p(-xc, 1.0 - yc), 1.0 - yc, 1.0 + w3)))
                - (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log(-((-1.0 - xc) / w3)) + log(((1.0 - w3) * (-1.0 - xc)) / (w3 + xc)) / 6.0 - log(((1.0 - w3) * (-1.0 - xc)) / ((-w3 - xc) * (-1.0 - yc))) / 6.0 + log((-1.0 - xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11pyc1pxc * T(p(-xc, 1.0 + yc), 1.0 + yc, 1.0 - w3)))
                + (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log((-1.0 - xc) / w3) + log(((1.0 + w3) * (-1.0 - xc)) / (-w3 + xc)) / 6.0 - log(((1.0 + w3) * (-1.0 - xc)) / ((w3 - xc) * (-1.0 - yc))) / 6.0 + log((-1.0 - xc) / (w3 - yc)) / 6.0 + 4.0 * ln1pxc * t11pyc1pxc * T(p(-xc, 1.0 + yc), 1.0 + yc, 1.0 + w3)))
                - (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log((1.0 - xc) / w3) + log(((-1.0 + w3) * (1.0 - xc)) / (w3 - xc)) / 6.0 - log(((-1.0 + w3) * (1.0 - xc)) / ((w3 - xc) * (1.0 - yc))) / 6.0 + log((1.0 - xc) / (w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11myc1mxc * T(p(xc, 1.0 - yc), 1.0 - yc, 1.0 - w3)))
                + (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log(-((1.0 - xc) / w3)) + log(((-1.0 - w3) * (1.0 - xc)) / (-w3 - xc)) / 6.0 - log(((-1.0 - w3) * (1.0 - xc)) / ((-w3 - xc) * (1.0 - yc))) / 6.0 + log((1.0 - xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11myc1mxc * T(p(xc, 1.0 - yc), 1.0 - yc, 1.0 + w3)))
                - (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log(-((-1.0 + xc) / w3)) + log(((1.0 - w3) * (-1.0 + xc)) / (w3 - xc)) / 6.0 - log(((1.0 - w3) * (-1.0 + xc)) / ((-w3 + xc) * (-1.0 - yc))) / 6.0 + log((-1.0 + xc) / (-w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11pyc1mxc * T(p(xc, 1.0 + yc), 1.0 + yc, 1.0 - w3)))
                + (18.0 * num11 * pisqu * ((-1.0 / 6.0) * log((-1.0 + xc) / w3) + log(((1.0 + w3) * (-1.0 + xc)) / (-w3 - xc)) / 6.0 - log(((1.0 + w3) * (-1.0 + xc)) / ((w3 + xc) * (-1.0 - yc))) / 6.0 + log((-1.0 + xc) / (w3 - yc)) / 6.0 + 4.0 * ln1mxc * t11pyc1mxc * T(p(xc, 1.0 + yc), 1.0 + yc, 1.0 + w3))) - (3.0 * num11 * (2.0 * pisqu * ln2 - 4.0 * power_of<3>(ln2) - 21.0 * zeta3)) / (2.0) - (6.0 * num9 * (2.0 * pisqu * ln2 - 4.0 * power_of<3>(ln2) - 21.0 * zeta3)) + (9.0 * num11 * zeta3) / (2.0) - (63.0 * num7 * zeta3) - (198.0 * num8 * zeta3)
                + (81.0 * num9 * zeta3)) / denom6;

        return (f29cPart1 + f29cPart2 + f29cPart3 + f29cPart4 + f29cPart5 + f29cPart6 + f29cPart7 + f29cPart8) / 9.0;

        }

        complex<double> f17e(const CharmLoopsParameters & )
        {
            return 0.0;
        }

        complex<double> f19e(const CharmLoopsParameters & clp)
        {
            const double C_F = 4.0 / 3.0; // SU(3) color factor

            // f19e = C_F f29e
            return C_F * f29e(clp);
        }

        complex<double> f27e(const CharmLoopsParameters & )
        {
            return 0.0;
        }

        complex<double> f29e(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xe = clp.xe;
            const complex<double> ye = clp.ye;

            const complex<double> ye2 = power_of<2>(ye);
            const complex<double> ye3 = power_of<3>(ye);
            const complex<double> ye4 = power_of<4>(ye);

            const complex<double> lnxe   = log(xe);
            const complex<double> lnye   = log(ye);
            const complex<double> lnmye  = log(-ye);
            const complex<double> ln1pxe = log(1.0 + xe);
            const complex<double> ln1mxe = log(1.0 - xe);
            const complex<double> ln1pye = log(1.0 + ye);
            const complex<double> ln1mye = log(1.0 - ye);

            const complex<double> result = -636.0 - (336.0 * 1.0i * M_PI) + 440.0 / ye2 + (288.0 * 1.0i * M_PI) / ye2 + (-264.0 + 56.0 / ye4 - 176.0 / ye2) * dilog(1.0 / 2.0) + (576.0 - 192.0 / ye4 + 384.0 / ye2) * trilog(1.0 / 2.0)
                + (696.0 - 232.0 / ye4 + 464.0 / ye2) * zeta3 + (-576.0 + 192.0 / ye4 - 384.0 / ye2) * trilog((1.0 - ye) / 2.0) + (192.0 - 64.0 / ye4 + 128.0 / ye2) * trilog(1.0 - ye)
                + (192.0 - 64.0 / ye4 + 128.0 / ye2) * trilog(-ye) + (192.0 - 64.0 / ye4 + 128.0 / ye2) * trilog(ye) + (192.0 - 64.0 / ye4 + 128.0 / ye2) * trilog(ye / (-1.0 + ye))
                + (-288.0 + 96.0 / ye4 - 192.0 / ye2) * trilog((2.0 * ye) / (-1.0 + ye)) + (192.0 - 64.0 / ye4 + 128.0 / ye2) * trilog(ye / (1.0 + ye)) + (-288.0 + 96.0 / ye4 - 192.0 / ye2) * trilog((2.0 * ye) / (1.0 + ye))
                + (-576.0 + 192.0 / ye4 - 384.0 / ye2) * trilog((1.0 + ye) / 2.0) + (192.0 - 64.0 / ye4 + 128.0 / ye2) * trilog(1.0 + ye) - 672.0 * ln2 - 80.0 * pisqu * ln2 + (80.0 * pisqu * ln2) / (3.0 * ye4)
                + (576.0 * ln2) / ye2 - (160.0 * pisqu * ln2) / (3.0 * ye2) + 96.0 * ln2cube - (32.0 * ln2cube) / ye4 + (64.0 * ln2cube) / ye2 + 32.0 * pisqu * ln2
                - (32.0 * pisqu * ln2) / (3.0 * ye4) + (64.0 * pisqu * ln2) / (3.0 * ye2) - 672.0 * lnmuhat + (576.0 * lnmuhat) / ye2 + (336.0 - 288.0 / ye2) * ln1mxe
                + (-672.0 + 576.0 / ye2) * lnxe + (336.0 - 288.0 / ye2) * ln1pxe + (-288.0 + 96.0 / ye4 - 192.0 / ye2) * dilog(1.0 - ye) * ln1mye
                + power_of<2>(ln1mye) * (66.0 - 14.0 / ye4 + 84.0 / ye3 + 44.0 / ye2 - 216.0 / ye + 36.0 * ye - 48.0 * ln2 + (16.0 * ln2) / ye4 - (32.0 * ln2) / ye2 - 48.0 * ln2
                    + (48.0 * ln2) / (3.0 * ye4) - (96.0 * ln2) / (3.0 * ye2) + (16.0 - 16.0 / (3.0 * ye4) + 32.0 / (3.0 * ye2)) * log((1.0 - ye) / 8.0) + (-192.0 + 64.0 / ye4 - 128.0 / ye2) * lnye
                    + (432.0 - 144.0 / ye4 + 288.0 / ye2) * log((1.0 + ye) / 2.0))
                + (-288.0 + 96.0 / ye4 - 192.0 / ye2) * dilog(1.0 + ye) * ln1pye + (-32.0 * pisqu + (32.0 * pisqu) / (3.0 * ye4) - 240.0 / ye3
                    - (144.0 * 1.0i * M_PI) / ye3 - (64.0 * pisqu) / (3.0 * ye2) + 312.0 / ye + (288.0 * 1.0i * M_PI) / ye - 24.0 * ye - (144.0 * 1.0i * M_PI * ye) - 132.0 * ln2 + (28.0 * ln2) / ye4 + (168.0 * ln2) / ye3
                    - (88.0 * ln2) / ye2 + (144.0 * ln2) / ye + 72.0 * ye * ln2 + 192.0 * ln2squ - (64.0 * ln2squ) / ye4 + (128.0 * ln2squ) / ye2 - 96.0 * (pisqu / 12.0 - ln2squ / 2.0)
                    + (32.0 * (pisqu / 12.0 - ln2squ / 2.0)) / ye4 - (64.0 * (pisqu / 12.0 - ln2squ / 2.0)) / ye2 + 8.0 * (pisqu + 6.0 * ln2squ) - (8.0 * (pisqu + 6.0 * ln2squ)) / (3.0 * ye4)
                    + (16.0 * (pisqu + 6.0 * ln2squ)) / (3.0 * ye2) - (288.0 * ln2) / ye3 - 288.0 * ye * ln2 - (288.0 * lnmuhat) / ye3 + (576.0 * lnmuhat) / ye - 288.0 * ye * lnmuhat
                    + (144.0 / ye3 - 288.0 / ye + 144.0 * ye) * ln1mxe + (-288.0 / ye3 + 576.0 / ye - 288.0 * ye) * lnxe + (144.0 / ye3 - 288.0 / ye + 144.0 * ye) * ln1pxe) * ln1pye
                + (66.0 - 14.0 / ye4 - 84.0 / ye3 + 44.0 / ye2 + 216.0 / ye - 36.0 * ye - 48.0 * ln2 + (16.0 * ln2) / ye4 - (32.0 * ln2) / ye2 - 48.0 * ln2 + (48.0 * ln2) / (3.0 * ye4)
                    - (96.0 * ln2) / (3.0 * ye2) + (432.0 - 144.0 / ye4 + 288.0 / ye2) * log((1.0 - ye) / 2.0) + (-192.0 + 64.0 / ye4 - 128.0 / ye2) * lnmye
                    + (16.0 - 16.0 / (3.0 * ye4) + 32.0 / (3.0 * ye2)) * log((1.0 + ye) / 8.0)) * power_of<2>(ln1pye) + dilog(ye) * (64.0 / ye3 - 192.0 / ye + (-96.0 + 32.0 / ye4 - 64.0 / ye2) * ln1mye
                    + (-192.0 + 64.0 / ye4 - 128.0 / ye2) * ln1pye) + dilog(-ye) * (-64.0 / ye3 + 192.0 / ye + (-192.0 + 64.0 / ye4 - 128.0 / ye2) * ln1mye + (-96.0 + 32.0 / ye4 - 64.0 / ye2) * ln1pye)
                + dilog((1.0 - ye) / 2.0) * (132.0 - 28.0 / ye4 + 168.0 / ye3 + 88.0 / ye2 - 432.0 / ye + 72.0 * ye + (864.0 - 288.0 / ye4 + 576.0 / ye2) * ln1mye + (288.0 - 96.0 / ye4 + 192.0 / ye2) * ln1pye)
                + dilog((1.0 + ye) / 2.0) * (132.0 - 28.0 / ye4 - 168.0 / ye3 + 88.0 / ye2 + 432.0 / ye - 72.0 * ye + (288.0 - 96.0 / ye4 + 192.0 / ye2) * ln1mye + (864.0 - 288.0 / ye4 + 576.0 / ye2) * ln1pye)
                + ln1mye * (-32.0 * pisqu + (32.0 * pisqu) / (3.0 * ye4) + 240.0 / ye3 + (144.0 * 1.0i * M_PI) / ye3 - (64.0 * pisqu) / (3.0 * ye2) - 312.0 / ye - (288.0 * 1.0i * M_PI) / ye + 24.0 * ye + (144.0 * 1.0i * M_PI * ye) - 132.0 * ln2
                    + (28.0 * ln2) / ye4 - (168.0 * ln2) / ye3 - (88.0 * ln2) / ye2 - (144.0 * ln2) / ye - 72.0 * ye * ln2 + 192.0 * ln2squ - (64.0 * ln2squ) / ye4 + (128.0 * ln2squ) / ye2
                    - 96.0 * (pisqu / 12.0 - ln2squ / 2.0) + (32.0 * (pisqu / 12.0 - ln2squ / 2.0)) / ye4 - (64.0 * (pisqu / 12.0 - ln2squ / 2.0)) / ye2 + 8.0 * (pisqu + 6.0 * ln2squ) - (8.0 * (pisqu + 6.0 * ln2squ)) / (3.0 * ye4)
                    + (16.0 * (pisqu + 6.0 * ln2squ)) / (3.0 * ye2) + (288.0 * ln2) / ye3 + 288.0 * ye * ln2 + (288.0 * lnmuhat) / ye3 - (576.0 * lnmuhat) / ye + 288.0 * ye * lnmuhat
                    + (-144.0 / ye3 + 288.0 / ye - 144.0 * ye) * ln1mxe + (288.0 / ye3 - 576.0 / ye + 288.0 * ye) * lnxe + (-144.0 / ye3 + 288.0 / ye - 144.0 * ye) * ln1pxe
                    + (-576.0 * ln2 + (192.0 * ln2) / ye4 - (384.0 * ln2) / ye2) * ln1pye)
                - 576.0 * zeta3 + (192.0 * zeta3) / ye4 - (384.0 * zeta3) / ye2;

            return result / 27.0;
        }

    }
}
