/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2022 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/polylog.hh>
#include <eos/maths/multiplepolylog-li22.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>
#include <eos/utils/stringify.hh>

#include <cmath>
#include <complex>

namespace eos
{
    using std::complex;
    using std::log;
    using std::real;

    namespace agv_2019a
    {
        // The following terms are the GPLs of weight 4 from f27d and f29d.
        // These are identical for the two functions, up to a global polynomial prefactor.
        complex<double> GPLweight4Parts(const CharmLoopsParameters & clp)
        {
            const complex<double> xd = clp.xd;
            const complex<double> yd = clp.yd;

            const complex<double> xdinv = 1.0 / xd;
            const complex<double> ydinv = 1.0 / yd;

            const complex<double> xd2 = power_of<2>(xd);
            const complex<double> xd4 = power_of<4>(xd);

            // weights appearing in the GPLs [AGV:2019A] p. 34

            const complex<double> w4 = (2.0 * xd) / power_of<2>(1.0 - xd);
            const complex<double> w5 = (2.0 * xd) / power_of<2>(1.0 + xd);
            const complex<double> w7 = (8.0 * xd2) / (1.0 - 6.0 * xd2 + xd4);

            const double imydinv = imag(ydinv);

            // often apperaing logs of xd and yd

            const complex<double> ln1pyd = log(1.0 + yd);
            const complex<double> ln1myd = log(1.0 - yd);

            // weight 4 GPLs

            // HPLs from F27(9)d that are free from problematic HeavisideTheta(0)
            const complex<double> w4Part1 = (-2.0 * power_of<4>(M_PI)) / 3.0 + 3.0 * power_of<2>(li2half) + (3.0 * power_of<2>(dilog((1.0 - yd) / 2.0))) / 2.0 + (9.0 * power_of<2>(dilog(-yd))) / 2.0 + (9.0 * power_of<2>(dilog(yd))) / 2.0 + (3.0 * power_of<2>(dilog((1.0 + yd) / 2.0))) / 2.0 - 6.0 * quadlog(-1.0) + 18.0 * quadlog(0.5) - 6.0 * quadlog(2.0)
                + 8.0 * quadlog(0.5 - (1.0i / 2.0) * xd) + 24.0 * quadlog(1.0 - 1.0i * xd) + 8.0 * quadlog(1.0 / (1.0 + 1.0i * xd)) + 8.0 * quadlog(2.0 / (1.0 + 1.0i * xd)) + 8.0 * quadlog((1.0 + 1.0i * xd) / 2.0) + 24.0 * quadlog(1.0 + 1.0i * xd) + 8.0 * quadlog((-1.0i) * xd) + 8.0 * quadlog(1.0i * xd)
                + 16.0 * quadlog(-1.0i / (-1.0i + xd)) + 24.0 * quadlog(xd / (-1.0i + xd)) + 24.0 * quadlog(1.0i / (1.0i + xd)) + 8.0 * quadlog((2.0 * 1.0i) / (1.0i + xd)) + 24.0 * quadlog(xd / (1.0i + xd)) + 8.0 * quadlog((-1.0i + xd) / (1.0i + xd)) + 8.0 * quadlog((1.0i + xd) / (-1.0i + xd))
                - 3.0 * quadlog(1.0 / (1.0 - yd)) - 17.0 * quadlog((1.0 - yd) / 2.0) - 15.0 * quadlog(1.0 - yd) - 5.0 * quadlog(-2.0 / (-1.0 + yd)) - 2.0 * quadlog(-yd) - 2.0 * quadlog(yd) - 3.0 * quadlog(yd / (-1.0 + yd)) - 3.0 * quadlog(1.0 / (1.0 + yd)) - 5.0 * quadlog(2.0 / (1.0 + yd))
                - 5.0 * quadlog((-1.0 + yd) / (1.0 + yd)) - 3.0 * quadlog(yd / (1.0 + yd)) - 17.0 * quadlog((1.0 + yd) / 2.0) - 15.0 * quadlog(1.0 + yd) - 5.0 * quadlog((1.0 + yd) / (-1.0 + yd)) - 4.0 * li22(-1.0, (-1.0i) * xd) - 4.0 * li22(-1.0, 1.0i * xd) - li22(-1.0, -yd) / 2.0
                - li22(-1.0, yd) / 2.0 - 16.0 * li22((-1.0i) * xd, -1.0) + 4.0 * li22((-1.0i) * xd, -1.0i / xd) - 16.0 * li22(1.0i * xd, -1.0) + 4.0 * li22(1.0i * xd, 1.0i / xd) - 2.0 * li22(-yd, -1.0) + li22(-yd, ydinv) / 2.0 - 2.0 * li22(yd, -1.0) + li22(yd, -ydinv) / 2.0
                + (pisqu * power_of<2>(lnhalf)) / 2.0 + lnmhalf * power_of<3>(lnhalf) - power_of<4>(lnhalf) / 2.0 - (4.0 * log(-0.5 - (1.0i / 2.0) * xd) * power_of<3>(log(0.5 - (1.0i / 2.0) * xd))) / 3.0 + (2.0 * power_of<4>(log(0.5 - (1.0i / 2.0) * xd))) / 3.0 + (7.0 * power_of<4>(log(1.0 - 1.0i * xd))) / 3.0
                - (2.0 * pisqu * power_of<2>(log((1.0 + 1.0i * xd) / 2.0))) / 3.0 + (2.0 * power_of<4>(log((1.0 + 1.0i * xd) / 2.0))) / 3.0 + (4.0 * log(1.0 - 1.0i * xd) * power_of<3>(log(1.0 + 1.0i * xd))) / 3.0 + (7.0 * power_of<4>(log(1.0 + 1.0i * xd))) / 3.0
                + power_of<2>(log(0.5 - (1.0i / 2.0) * xd)) * ((-2.0 * pisqu) / 3.0 - 4.0 * log((1.0 + 1.0i * xd) / 2.0) * log(1.0 + 1.0i * xd) - 2.0 * power_of<2>(log(1.0 + 1.0i * xd)))
                + dilog(0.5 - (1.0i / 2.0) * xd) * (-8.0 * log(0.5 - (1.0i / 2.0) * xd) * log(1.0 + 1.0i * xd) + log(1.0 - 1.0i * xd) * (-8.0 * log((1.0 + 1.0i * xd) / 2.0) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)))
                + power_of<3>(log(1.0 + 1.0i * xd)) * (-4.0 * log((-1.0i) * xd) - 4.0 * log(1.0i * xd) - (4.0 * log(xd)) / 3.0) + power_of<3>(log(1.0 - 1.0i * xd)) * ((4.0 * log(1.0 + 1.0i * xd)) / 3.0 - 4.0 * log((-1.0i) * xd) - 4.0 * log(1.0i * xd) - (4.0 * log(xd)) / 3.0) - 16.0 * trilog(-1.0) * log(xd)
                + 8.0 * trilog(0.5 + (1.0i / 2.0) * xd) * log(xd) + 8.0 * trilog(-1.0i / (-1.0i + xd)) * log(xd) - 8.0 * trilog((-1.0i + xd) / (2.0 * xd)) * log(xd) + 8.0 * trilog(1.0i / (1.0i + xd)) * log(xd) - 8.0 * trilog((1.0i - xd) / (1.0i + xd)) * log(xd)
                + 8.0 * trilog((-1.0i + xd) / (1.0i + xd)) * log(xd) + 8.0 * trilog(((-0.5) * 1.0i) * (1.0i + xd)) * log(xd) - 8.0 * trilog((1.0i + xd) / (1.0i - xd)) * log(xd) - 8.0 * trilog((1.0i + xd) / (2.0 * xd)) * log(xd)
                + 8.0 * trilog((1.0i + xd) / (-1.0i + xd)) * log(xd) + 8.0 * dilog((1.0i + xd) / (1.0i - xd)) * log(1.0 - 1.0i * xd) * log(xd) + 8.0 * dilog((1.0i + xd) / (2.0 * xd)) * log(1.0 - 1.0i * xd) * log(xd)
                - 8.0 * dilog((1.0i + xd) / (-1.0i + xd)) * log(1.0 - 1.0i * xd) * log(xd) + 8.0 * dilog((-1.0i + xd) / (2.0 * xd)) * log(1.0 + 1.0i * xd) * log(xd) + 8.0 * dilog((1.0i - xd) / (1.0i + xd)) * log(1.0 + 1.0i * xd) * log(xd)
                - 8.0 * dilog((-1.0i + xd) / (1.0i + xd)) * log(1.0 + 1.0i * xd) * log(xd) + trilog(1.0 - 1.0i * xd) * (-8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(xd)) + trilog(1.0 + 1.0i * xd) * (-8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                + trilog((-1.0i) * xd) * (16.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(1.0i * xd) * (-8.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                + dilog(1.0 - 1.0i * xd) * (-4.0 * power_of<2>(log(1.0 - 1.0i * xd)) - 8.0 * log(1.0 - 1.0i * xd) * log(xd)) + dilog(1.0i * xd) * (-4.0 * power_of<2>(log(1.0 - 1.0i * xd)) - 8.0 * log(1.0 - 1.0i * xd) * log(xd))
                + dilog(1.0 + 1.0i * xd) * (-4.0 * power_of<2>(log(1.0 + 1.0i * xd)) - 8.0 * log(1.0 + 1.0i * xd) * log(xd)) + dilog((-1.0i) * xd) * (16.0 * dilog(1.0i * xd) - 4.0 * power_of<2>(log(1.0 + 1.0i * xd)) - 8.0 * log(1.0 + 1.0i * xd) * log(xd))
                + dilog((1.0 + 1.0i * xd) / 2.0) * (-8.0 * log(0.5 - (1.0i / 2.0) * xd) * log(1.0 + 1.0i * xd) + log(1.0 - 1.0i * xd) * (-8.0 * log((1.0 + 1.0i * xd) / 2.0) + 8.0 * log(1.0 + 1.0i * xd)) - 8.0 * log(1.0 + 1.0i * xd) * log(xd))
                + dilog(-1.0) * (-4.0 * dilog((-1.0i) * xd) - 4.0 * dilog(1.0i * xd) - dilog(-yd) / 2.0 - dilog(yd) / 2.0 - 8.0 * log(1.0 - 1.0i * xd) * log(xd) - 8.0 * log(1.0 + 1.0i * xd) * log(xd))
                + power_of<2>(log(1.0 - 1.0i * xd)) * (-2.0 * pisqu - 2.0 * power_of<2>(log((1.0 + 1.0i * xd) / 2.0)) + log((1.0 + 1.0i * xd) / 2.0) * (4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) - 4.0 * log(1.0i * xd) * log(xd)
                + log(xd) * (-4.0 * log((-2.0 * 1.0i) / (-1.0i + xd)) + 4.0 * log((2.0 * xd) / (-1.0i + xd)) + 4.0 * log((-1.0i + xd) / (2.0 * xd)))) - (4.0 * power_of<3>(log((1.0 + 1.0i * xd) / 2.0)) * log((1.0i / 2.0) * (1.0i + xd))) / 3.0
                + power_of<2>(log(1.0 + 1.0i * xd)) * (-2.0 * pisqu - 4.0 * log((-1.0i) * xd) * log(xd) + log(xd) * (-4.0 * log((2.0 * 1.0i) / (1.0i + xd)) + 4.0 * log((2.0 * xd) / (1.0i + xd)) + 4.0 * log((1.0i + xd) / (2.0 * xd)))) + (5.0 * log((-1.0 - yd) / 2.0) * power_of<3>(log((1.0 - yd) / 2.0))) / 6.0
                - (5.0 * power_of<4>(log((1.0 - yd) / 2.0))) / 12.0 + 12.0 * trilog(1.0 - yd) * ln1myd - (9.0 * dilog(1.0 - yd) * power_of<2>(ln1myd)) / 2.0 - (dilog(yd) * power_of<2>(ln1myd)) / 2.0 - power_of<4>(ln1myd) / 4.0
                + power_of<3>(ln1myd) * (log(-yd) / 2.0 - log(yd) + log((1.0 + yd) / 2.0) / 2.0) + (5.0 * pisqu * power_of<2>(log((1.0 + yd) / 2.0))) / 12.0 + (5.0 * log((-1.0 + yd) / 2.0) * power_of<3>(log((1.0 + yd) / 2.0))) / 6.0 - (5.0 * power_of<4>(log((1.0 + yd) / 2.0))) / 12.0
                + trilog(yd) * (10.0 * ln1myd - 7.0 * ln1pyd) + trilog((1.0 - yd) / 2.0) * (2.0 * ln1myd - 2.0 * ln1pyd) + 12.0 * trilog(1.0 + yd) * ln1pyd + (pisqu * power_of<2>(ln1pyd)) / 4.0
                - (9.0 * dilog(1.0 + yd) * power_of<2>(ln1pyd)) / 2.0 - log(-yd) * power_of<3>(ln1pyd) + (log(yd) * power_of<3>(ln1pyd)) / 2.0 - power_of<4>(ln1pyd) / 4.0 + trilog((1.0 + yd) / 2.0) * (-2.0 * ln1myd + 2.0 * ln1pyd)
                + li3half * (6.0 * lnhalf - 8.0 * log(0.5 - (1.0i / 2.0) * xd) + 8.0 * log(1.0 - 1.0i * xd) - 8.0 * log((1.0 + 1.0i * xd) / 2.0) + 8.0 * log(1.0 + 1.0i * xd) - 16.0 * log(xd) + 5.0 * log((1.0 - yd) / 2.0) + 7.0 * ln1myd + 5.0 * log((1.0 + yd) / 2.0)
                + 7.0 * ln1pyd) + trilog(-yd) * (-7.0 * ln1myd + 10.0 * ln1pyd) + dilog((1.0 - yd) / 2.0) * ((3.0 * power_of<2>(ln1myd)) / 2.0 + ln1myd * (8.0 * log((1.0 + yd) / 2.0) - 3.0 * ln1pyd)
                + 5.0 * log((1.0 - yd) / 2.0) * ln1pyd) + power_of<2>(ln1myd) * (pisqu / 4.0 + (11.0 * power_of<2>(log((1.0 + yd) / 2.0))) / 4.0 - (3.0 * log((1.0 + yd) / 2.0) * ln1pyd) / 2.0) + dilog(-yd) * (-7.0 * dilog(yd) - power_of<2>(ln1pyd) / 2.0)
                + dilog((1.0 + yd) / 2.0) * (ln1myd * (5.0 * log((1.0 + yd) / 2.0) - 3.0 * ln1pyd) + 8.0 * log((1.0 - yd) / 2.0) * ln1pyd + (3.0 * power_of<2>(ln1pyd)) / 2.0)
                + li2half * (-3.0 * dilog((1.0 - yd) / 2.0) - 3.0 * dilog((1.0 + yd) / 2.0) + (5.0 * power_of<2>(ln1myd)) / 2.0 - 3.0 * ln1myd * log((1.0 + yd) / 2.0) - 3.0 * log((1.0 - yd) / 2.0) * ln1pyd + (5.0 * power_of<2>(ln1pyd)) / 2.0)
                + power_of<2>(log((1.0 - yd) / 2.0)) * ((5.0 * pisqu) / 12.0 + (5.0 * log((1.0 + yd) / 2.0) * ln1pyd) / 2.0 + (11.0 * power_of<2>(ln1pyd)) / 4.0) + log(xd) * ((-4.0 * pisqu * log(((-0.5) * 1.0i) * xdinv)) / 3.0 - (4.0 * power_of<3>(log(((-0.5) * 1.0i) * xdinv))) / 3.0
                - (4.0 * pisqu * log(1.0i / 2.0 * xdinv)) / 3.0 - (4.0 * power_of<3>(log(1.0i / 2.0 * xdinv))) / 3.0 + (4.0 * pisqu * log(-1.0i / (-1.0i + xd))) / 3.0 + (4.0 * power_of<3>(log(-1.0i / (-1.0i + xd)))) / 3.0 + (4.0 * pisqu * log(1.0i / (1.0i + xd))) / 3.0 + (4.0 * power_of<3>(log(1.0i / (1.0i + xd)))) / 3.0 - 16.0 * zeta3)
                + log((1.0 - yd) / 2.0) * (power_of<3>(ln1pyd) / 2.0 + ln1myd * ((5.0 * power_of<2>(log((1.0 + yd) / 2.0))) / 2.0 - (3.0 * power_of<2>(ln1pyd)) / 2.0) - 5.0 * zeta3) - 6.0 * lnhalf * zeta3 + 8.0 * log((1.0 + 1.0i * xd) / 2.0) * zeta3 + 5.0 * ln1myd * zeta3
                - 5.0 * log((1.0 + yd) / 2.0) * zeta3 + 5.0 * ln1pyd * zeta3 + log(0.5 - (1.0i / 2.0) * xd) * (log(1.0 - 1.0i * xd) * (-4.0 * power_of<2>(log((1.0 + 1.0i * xd) / 2.0)) + 4.0 * power_of<2>(log(1.0 + 1.0i * xd))) - 4.0 * power_of<2>(log(1.0 + 1.0i * xd)) * log(xd) + 8.0 * zeta3);

            //GPLs form F27(9)d with at least one weight being +-wx3
            const complex<double> w4Part2 = weight4_wx3_wx4(clp, wx3) + weight4_wx3_wx4(clp, -wx3);

            //GPLs form F27(9)d with at least one weight being +-wx4
            const complex<double> w4Part3 = weight4_wx3_wx4(clp, wx4) + weight4_wx3_wx4(clp, -wx4);

            //GPLs form F27(9)d with at least one weight being +-w4
            const complex<double> w4Part4 = weight4_w4_w5_w7(clp, w4);

            //GPLs form F27(9)d with at least one weight being +-w5
            const complex<double> w4Part5 = weight4_w4_w5_w7(clp, -w5);

            //GPLs form F27(9)d with at least one weight being +-w7
            const complex<double> w4Part6 = - 0.25 * weight4_w4_w5_w7(clp, w7);

            // Weight 4 HPLs where the weights are a permutation of {a, a, -a, 0}.
            // In order to make the expressions well defined, the replacement -a -> -a * (1 + i eta) was made
            const double eta = 1e-13;

            const complex<double> w4HPLs = (-4.0 * power_of<4>(M_PI)) / 3.0 + 3.0 * power_of<2>(dilog(2.0)) - (3.0 * power_of<2>(dilog(1.0 / (2.0 - 1.0i * eta)))) / 2.0 - 9.0 * power_of<2>(dilog(2.0 - 1.0i * eta)) - (51.0 * power_of<2>(dilog(1.0i / (2.0 * 1.0i + eta)))) / 2.0 - 18.0 * quadlog(2.0) + 18.0 * quadlog(1.0 + 1.0 / (1.0 - 1.0i * eta)) - 3.0 * quadlog(1.0 / (2.0 - 1.0i * eta)) + 90.0 * quadlog(2.0 - 1.0i * eta) + 18.0 * quadlog(1.0 / (-1.0 + 1.0i * eta)) + 18.0 * quadlog(-1.0i / (1.0i + eta)) + 93.0 * quadlog(1.0i / (2.0 * 1.0i + eta)) + 18.0 * quadlog((2.0 * 1.0i + eta) / (1.0i + eta)) + 24.0 * quadlog(1.0 - 1.0i * xd) + 24.0 * quadlog(1.0 + 1.0i * xd) + 2.0 * quadlog(1.0 - yd) + 8.0 * quadlog((-1.0 + yd) / (-2.0 + 1.0i * eta)) + 2.0 * quadlog(1.0 + yd) + 8.0 * quadlog((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))
                - 4.0 * li22(1.0 / (2.0 - 1.0i * eta), 1.0 + yd) - 12.0 * li22(2.0 - 1.0i * eta, (1.0i - xd) / (2.0 * 1.0i + eta)) - 12.0 * li22(2.0 - 1.0i * eta, (1.0i + xd) / (2.0 * 1.0i + eta)) - li22(2.0 - 1.0i * eta, (-1.0 + yd) / (-2.0 + 1.0i * eta)) - li22(2.0 - 1.0i * eta, (1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) - 4.0 * li22(1.0i / (2.0 * 1.0i + eta), 1.0 - yd) - 20.0 * li22((1.0i - xd) / (2.0 * 1.0i + eta), 2.0 - 1.0i * eta) + 8.0 * li22(0.5 - (1.0i / 2.0) * xd, 2.0) - 12.0 * li22(1.0 - 1.0i * xd, 1.0i / (2.0 * 1.0i + eta)) + 12.0 * li22(1.0 - 1.0i * xd, (2.0 * 1.0i + eta) / (1.0i + xd)) + 8.0 * li22((1.0 + 1.0i * xd) / 2.0, 2.0) - 12.0 * li22(1.0 + 1.0i * xd, 1.0i / (2.0 * 1.0i + eta)) + 12.0 * li22(1.0 + 1.0i * xd, (2.0 * 1.0i + eta) / (1.0i - xd))
                - 20.0 * li22((1.0i + xd) / (2.0 * 1.0i + eta), 2.0 - 1.0i * eta) - 5.0 * li22((1.0 - yd) / 2.0, 2.0) - 10.0 * li22(1.0 - yd, 1.0i / (2.0 * 1.0i + eta)) + li22(1.0 - yd, (1.0i * (2.0 * 1.0i + eta)) / (-1.0 + yd)) - 2.0 * li22((-1.0 + yd) / (-2.0 + 1.0i * eta), 2.0 - 1.0i * eta) + 4.0 * li22((-1.0 + yd) / (-2.0 + 1.0i * eta), 1.0 / (1.0 - yd)) - 5.0 * li22((1.0 + yd) / 2.0, 2.0) - 10.0 * li22(1.0 + yd, 1.0 / (2.0 - 1.0i * eta)) + li22(1.0 + yd, (2.0 - 1.0i * eta) / (1.0 + yd)) - 2.0 * li22((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta), 2.0 - 1.0i * eta) + 4.0 * li22((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta), 1.0 / (1.0 + yd)) + dilog(2.0) * (3.0 * pisqu + 3.0 * power_of<2>(lnm2)) + 3.0 * pisqu * power_of<2>(log(-1.0 + 1.0i * eta))
                + 6.0 * log(-2.0 + 1.0i * eta) * power_of<3>(log(-1.0 + 1.0i * eta)) - 3.0 * power_of<4>(log(-1.0 + 1.0i * eta)) - 6.0 * pisqu * power_of<2>(log((1.0i + eta) / (2.0 * 1.0i + eta))) - 2.0 * log(1.0 / (-2.0 + 1.0i * eta)) * power_of<3>(log((1.0i + eta) / (2.0 * 1.0i + eta))) - 10.0 * log(-1.0i / (2.0 * 1.0i + eta)) * power_of<3>(log((1.0i + eta) / (2.0 * 1.0i + eta))) + 6.0 * power_of<4>(log((1.0i + eta) / (2.0 * 1.0i + eta))) + trilog(1.0 - 1.0i * xd) * (-24.0 * log(-1.0 + 1.0i * eta) - 8.0 * log((1.0i + eta - xd) / (1.0i + eta)) + 24.0 * log((1.0i + eta - xd) / (2.0 * 1.0i + eta)) - 24.0 * log(1.0 - 1.0i * xd)) + 12.0 * trilog((1.0i + eta) / (1.0i + eta - xd)) * log(1.0 - 1.0i * xd) + 24.0 * trilog((2.0 * 1.0i + eta) / (1.0i + eta - xd)) * log(1.0 - 1.0i * xd)
                - 12.0 * trilog(1.0i * xd) * log(1.0 - 1.0i * xd) + 20.0 * trilog(xd / (1.0i + eta)) * log(1.0 - 1.0i * xd) - 24.0 * trilog((1.0i * (2.0 * 1.0i + eta) * xd) / (1.0i + eta - xd)) * log(1.0 - 1.0i * xd) + 8.0 * trilog((-1.0i + xd) / (2.0 * xd)) * log(1.0 - 1.0i * xd) + 4.0 * trilog(1.0i / (1.0i + xd)) * log(1.0 - 1.0i * xd) - 4.0 * trilog((1.0i * (1.0i + eta - xd)) / ((1.0i + eta) * (1.0i + xd))) * log(1.0 - 1.0i * xd) - 24.0 * trilog(((2.0 * 1.0i + eta) * xd) / ((1.0i + eta) * (1.0i + xd))) * log(1.0 - 1.0i * xd) + 8.0 * trilog((-1.0i + xd) / (1.0i + xd)) * log(1.0 - 1.0i * xd) + 24.0 * trilog(xd / (-1.0i - eta + xd)) * log(1.0 - 1.0i * xd) + 12.0 * trilog((1.0i + xd) / (-1.0i - eta + xd)) * log(1.0 - 1.0i * xd)
                + 4.0 * trilog((-1.0i - eta + xd) * xdinv) * log(1.0 - 1.0i * xd) - 4.0 * trilog((-1.0i - eta + xd) / (1.0i + xd)) * log(1.0 - 1.0i * xd) - 12.0 * trilog((1.0i + eta + xd - 1.0i * eta * xd) / (1.0i + eta - xd)) * log(1.0 - 1.0i * xd) - 8.0 * trilog((1.0 - 1.0i * eta + 1.0i * xd) / ((2.0 * 1.0i) * xd + eta * xd)) * log(1.0 - 1.0i * xd) + 4.0 * pisqu * log(-1.0i / (1.0i + eta - xd)) * log(1.0 - 1.0i * xd) + 4.0 * power_of<3>(log(-1.0i / (1.0i + eta - xd))) * log(1.0 - 1.0i * xd) + 4.0 * dilog(1.0i * xd) * log((1.0i + eta - xd) / (1.0i + eta)) * log(1.0 - 1.0i * xd) + 4.0 * dilog((1.0i * (1.0i + eta - xd)) / ((1.0i + eta) * (1.0i + xd))) * log((1.0i + eta - xd) / (1.0i + eta)) * log(1.0 - 1.0i * xd)
                + 8.0 * dilog((1.0 - 1.0i * eta + 1.0i * xd) / ((2.0 * 1.0i) * xd + eta * xd)) * log((1.0i + eta - xd) / (2.0 * 1.0i + eta)) * log(1.0 - 1.0i * xd) - 12.0 * dilog((1.0i + xd) / (-1.0i - eta + xd)) * power_of<2>(log(1.0 - 1.0i * xd)) + 12.0 * dilog((1.0i + eta + xd - 1.0i * eta * xd) / (1.0i + eta - xd)) * power_of<2>(log(1.0 - 1.0i * xd)) + dilog((-1.0i - eta + xd) * xdinv) * (4.0 * log((1.0i + eta - xd) / (1.0i + eta)) * log(1.0 - 1.0i * xd) - 8.0 * log((1.0i + eta - xd) / (2.0 * 1.0i + eta)) * log(1.0 - 1.0i * xd)) + dilog((-1.0i - eta + xd) / (1.0i + xd)) * (-4.0 * log((1.0i + eta - xd) / (1.0i + eta)) * log(1.0 - 1.0i * xd) + 8.0 * log((1.0i + eta - xd) / (2.0 * 1.0i + eta)) * log(1.0 - 1.0i * xd))
                - 8.0 * dilog((-1.0i + xd) / (2.0 * xd)) * log(1.0 - 1.0i * xd) * log((1.0 + 1.0i * xd) / 2.0) - 8.0 * dilog((-1.0i + xd) / (1.0i + xd)) * log(1.0 - 1.0i * xd) * log((1.0 + 1.0i * xd) / 2.0) + trilog((-1.0i + xd) * xdinv) * (-8.0 * log(1.0 - 1.0i * xd) - 12.0 * log(1.0 + 1.0i * xd)) + trilog((1.0i + xd) * xdinv) * (-12.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd)) - 12.0 * trilog((-1.0i) * xd) * log(1.0 + 1.0i * xd) + 20.0 * trilog(-(xd / (1.0i + eta))) * log(1.0 + 1.0i * xd) + 4.0 * trilog(-1.0i / (-1.0i + xd)) * log(1.0 + 1.0i * xd) - 24.0 * trilog(((2.0 * 1.0i + eta) * xd) / ((1.0i + eta) * (-1.0i + xd))) * log(1.0 + 1.0i * xd) + 8.0 * trilog((1.0i + xd) / (2.0 * xd)) * log(1.0 + 1.0i * xd)
                + 8.0 * trilog((1.0i + xd) / (-1.0i + xd)) * log(1.0 + 1.0i * xd) + 12.0 * trilog((1.0i + eta) / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) + 24.0 * trilog((2.0 * 1.0i + eta) / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) - 12.0 * trilog(((1.0i + eta) * (1.0 + 1.0i * xd)) / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) + 24.0 * trilog(xd / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) - 24.0 * trilog(((2.0 - 1.0i * eta) * xd) / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) + 12.0 * trilog((-1.0i + xd) / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) + 4.0 * trilog((1.0i + eta + xd) * xdinv) * log(1.0 + 1.0i * xd) - 8.0 * trilog((1.0i * (1.0i + eta + xd)) / ((2.0 * 1.0i + eta) * xd)) * log(1.0 + 1.0i * xd)
                - 4.0 * trilog((1.0i + eta + xd) / (-1.0i + xd)) * log(1.0 + 1.0i * xd) - 4.0 * trilog(((-1.0i) * (1.0i + eta + xd)) / ((1.0i + eta) * (-1.0i + xd))) * log(1.0 + 1.0i * xd) + (2.0 * pisqu * log((1.0i + eta) / (1.0i - xd)) * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * power_of<3>(log((1.0i + eta) / (1.0i - xd))) * log(1.0 + 1.0i * xd)) / 3.0 - 8.0 * dilog((1.0i + xd) / (2.0 * xd)) * log(0.5 - (1.0i / 2.0) * xd) * log(1.0 + 1.0i * xd) - 8.0 * dilog((1.0i + xd) / (-1.0i + xd)) * log(0.5 - (1.0i / 2.0) * xd) * log(1.0 + 1.0i * xd) + 12.0 * dilog(((1.0i + eta) * (1.0 + 1.0i * xd)) / (1.0i + eta + xd)) * power_of<2>(log(1.0 + 1.0i * xd)) - 12.0 * dilog((-1.0i + xd) / (1.0i + eta + xd)) * power_of<2>(log(1.0 + 1.0i * xd))
                + dilog((1.0i + xd) * xdinv) * (12.0 * power_of<2>(log(1.0 - 1.0i * xd)) + 8.0 * log(0.5 - (1.0i / 2.0) * xd) * log(1.0 + 1.0i * xd)) + dilog((-1.0i + xd) * xdinv) * (8.0 * log(1.0 - 1.0i * xd) * log((1.0 + 1.0i * xd) / 2.0) + 12.0 * power_of<2>(log(1.0 + 1.0i * xd))) + trilog((1.0 + 1.0i * xd) / 2.0) * (8.0 * log(1.0 + 1.0i * xd) - 16.0 * log((-1.0i) * xd)) + 24.0 * dilog(((2.0 * 1.0i + eta) * xd) / ((1.0i + eta) * (-1.0i + xd))) * log(1.0 + 1.0i * xd) * log((-1.0i) * xd) - 24.0 * dilog(xd / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) * log((-1.0i) * xd) + 24.0 * dilog(((2.0 - 1.0i * eta) * xd) / (1.0i + eta + xd)) * log(1.0 + 1.0i * xd) * log((-1.0i) * xd)
                + trilog((1.0i - xd) / (2.0 * 1.0i + eta)) * (24.0 * log(-1.0 + 1.0i * eta) + 16.0 * log(1.0 + 1.0i * xd) + 16.0 * log((-1.0i) * xd)) + dilog(-(xd / (1.0i + eta))) * (-12.0 * power_of<2>(log(1.0 + 1.0i * xd)) - 24.0 * log(1.0 + 1.0i * xd) * log((-1.0i) * xd)) + trilog(0.5 - (1.0i / 2.0) * xd) * (8.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0i * xd)) + 24.0 * dilog((1.0i * (2.0 * 1.0i + eta) * xd) / (1.0i + eta - xd)) * log(1.0 - 1.0i * xd) * log(1.0i * xd) + 24.0 * dilog(((2.0 * 1.0i + eta) * xd) / ((1.0i + eta) * (1.0i + xd))) * log(1.0 - 1.0i * xd) * log(1.0i * xd) - 24.0 * dilog(xd / (-1.0i - eta + xd)) * log(1.0 - 1.0i * xd) * log(1.0i * xd)
                + trilog((1.0i + xd) / (2.0 * 1.0i + eta)) * (24.0 * log(-1.0 + 1.0i * eta) + 16.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0i * xd)) + dilog(xd / (1.0i + eta)) * (-12.0 * power_of<2>(log(1.0 - 1.0i * xd)) - 24.0 * log(1.0 - 1.0i * xd) * log(1.0i * xd)) + 4.0 * dilog((-1.0i) * xd) * log(1.0 + 1.0i * xd) * log((1.0i + eta + xd) / (1.0i + eta)) + 4.0 * dilog(((-1.0i) * (1.0i + eta + xd)) / ((1.0i + eta) * (-1.0i + xd))) * log(1.0 + 1.0i * xd) * log((1.0i + eta + xd) / (1.0i + eta)) + dilog((1.0i + eta + xd) * xdinv) * log(1.0 + 1.0i * xd) * (4.0 * log((1.0i + eta + xd) / (1.0i + eta)) - 8.0 * log((1.0i + eta + xd) / (2.0 * 1.0i + eta)))
                + 8.0 * dilog((1.0i * (1.0i + eta + xd)) / ((2.0 * 1.0i + eta) * xd)) * log(1.0 + 1.0i * xd) * log((1.0i + eta + xd) / (2.0 * 1.0i + eta)) + dilog((1.0i + eta + xd) / (-1.0i + xd)) * log(1.0 + 1.0i * xd) * (-4.0 * log((1.0i + eta + xd) / (1.0i + eta)) + 8.0 * log((1.0i + eta + xd) / (2.0 * 1.0i + eta))) + trilog(1.0 + 1.0i * xd) * (-24.0 * log(-1.0 + 1.0i * eta) - 24.0 * log(1.0 + 1.0i * xd) - 8.0 * log((1.0i + eta + xd) / (1.0i + eta)) + 24.0 * log((1.0i + eta + xd) / (2.0 * 1.0i + eta))) - (3.0 * trilog(1.0 / (-1.0 + 1.0i * eta)) * ln1myd) / 2.0 - 8.0 * trilog((2.0 * 1.0i + eta) / (1.0i + eta)) * ln1myd - (trilog(1.0 / (1.0 - yd)) * ln1myd) / 2.0
                + 10.0 * trilog(ydinv) * ln1myd - 11.0 * trilog(((1.0i + eta) * (-1.0 + yd)) / ((2.0 * 1.0i + eta) * yd)) * ln1myd - (15.0 * trilog(yd) * ln1myd) / 2.0 + (27.0 * trilog(yd / (-1.0 + 1.0i * eta)) * ln1myd) / 2.0 - 5.0 * trilog(yd / (-1.0 + yd)) * ln1myd - 2.0 * trilog(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (-1.0 + yd))) * ln1myd - 5.0 * trilog((1.0 + yd) / (-1.0 + yd)) * ln1myd - 5.0 * trilog((1.0 + yd) / (2.0 * yd)) * ln1myd + (15.0 * trilog((-1.0 + yd) / (1.0 - 1.0i * eta + yd)) * ln1myd) / 2.0 + 7.0 * trilog(yd / (1.0 - 1.0i * eta + yd)) * ln1myd
                + (5.0 * trilog((1.0 - 1.0i * eta + yd) / (-1.0 + yd)) * ln1myd) / 2.0 + (11.0 * trilog((1.0 - 1.0i * eta + yd) / yd) * ln1myd) / 2.0 + 8.0 * trilog(1.0 + (1.0i * yd) / (1.0i + eta)) * ln1myd + (15.0 * trilog((1.0i + eta) / (eta + 1.0i * (1.0 + yd))) * ln1myd) / 2.0 + 7.0 * trilog((2.0 * 1.0i + eta) / (eta + 1.0i * (1.0 + yd))) * ln1myd - (15.0 * trilog(-(((1.0i + eta) * (-1.0 + yd)) / (eta + 1.0i * (1.0 + yd)))) * ln1myd) / 2.0 - 7.0 * trilog(((2.0 * 1.0i + eta) * yd) / (eta + 1.0i * (1.0 + yd))) * ln1myd - (15.0 * trilog(-((eta + 1.0i * (1.0 + yd)) / ((1.0i + eta) * (-1.0 + yd)))) * ln1myd) / 2.0
                - 5.0 * trilog((eta + 1.0i * (1.0 + yd)) / ((2.0 * 1.0i + eta) * yd)) * ln1myd - (5.0 * power_of<3>(log(1.0 / (1.0 - yd))) * ln1myd) / 6.0 + (3.0 * dilog(1.0 / (-1.0 + 1.0i * eta)) * power_of<2>(ln1myd)) / 2.0 + 4.0 * dilog((2.0 * 1.0i + eta) / (1.0i + eta)) * power_of<2>(ln1myd) + 11.0 * dilog(((1.0i + eta) * (-1.0 + yd)) / ((2.0 * 1.0i + eta) * yd)) * power_of<2>(ln1myd) - (15.0 * dilog((-1.0 + yd) / (1.0 - 1.0i * eta + yd)) * power_of<2>(ln1myd)) / 2.0 + (15.0 * dilog(-(((1.0i + eta) * (-1.0 + yd)) / (eta + 1.0i * (1.0 + yd)))) * power_of<2>(ln1myd)) / 2.0 + dilog((-1.0 + yd) / (-2.0 + 1.0i * eta)) * ((-5.0 * pisqu) / 6.0 + 11.0 * dilog(yd) - 11.0 * power_of<2>(ln1myd))
                + 5.0 * dilog(yd / (-1.0 + yd)) * ln1myd * log(yd) + 2.0 * dilog(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (-1.0 + yd))) * ln1myd * log(yd) - 7.0 * dilog(yd / (1.0 - 1.0i * eta + yd)) * ln1myd * log(yd) + 7.0 * dilog(((2.0 * 1.0i + eta) * yd) / (eta + 1.0i * (1.0 + yd))) * ln1myd * log(yd) + trilog((-1.0 + yd) / (-2.0 + 1.0i * eta)) * (2.0 * log(-1.0 + 1.0i * eta) - 8.0 * log((1.0i + eta) / (2.0 * 1.0i + eta)) + 8.0 * ln1myd + 2.0 * log(yd)) + trilog((1.0 - yd) / 2.0) * (-5.0 * ln1myd + 10.0 * log(yd))
                + dilog(1.0i / (2.0 * 1.0i + eta)) * ((-29.0 * pisqu) / 3.0 - 8.0 * dilog(1.0 - 1.0i * xd) - 8.0 * dilog(1.0 + 1.0i * xd) + 3.0 * dilog(1.0 - yd) - 4.0 * dilog((-1.0 + yd) / (-2.0 + 1.0i * eta)) - 11.0 * dilog(-yd) + (3.0 * power_of<2>(log(1.0 / (-2.0 + 1.0i * eta)))) / 2.0 - 12.0 * power_of<2>(log(-1.0i / (2.0 * 1.0i + eta))) + (8.0 * log(1.0) - 16.0 * log(1.0 + 1.0i * xd)) * log((-1.0i) * xd) + (8.0 * log(1.0) - 16.0 * log(1.0 - 1.0i * xd)) * log(1.0i * xd) + (5.0 * log(1.0) - 2.0 * ln1myd) * log(yd)) + dilog(yd / (-1.0 + 1.0i * eta)) * (-power_of<2>(ln1myd) - 2.0 * ln1myd * log(yd)) + log(1.0 / (1.0 - yd)) * ln1myd * ((-5.0 * pisqu) / 6.0 + (5.0 * power_of<2>(log(yd))) / 2.0)
                + 5.0 * dilog((1.0 + yd) / (-1.0 + yd)) * ln1myd * log((1.0 + yd) / 2.0) + 5.0 * dilog((1.0 + yd) / (2.0 * yd)) * ln1myd * log((1.0 + yd) / 2.0) + dilog(1.0 / (2.0 - 1.0i * eta)) * ((7.0 * pisqu) / 3.0 - 11.0 * dilog(yd) + 3.0 * dilog(1.0 + yd) - 4.0 * dilog((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) + (3.0 * power_of<2>(log(-1.0i / (2.0 * 1.0i + eta)))) / 2.0 + log(-yd) * (5.0 * log(1.0) - 2.0 * ln1pyd)) + trilog(1.0i / (2.0 * 1.0i + eta)) * (-42.0 * log(-1.0 + 1.0i * eta) - 60.0 * log((1.0i + eta) / (2.0 * 1.0i + eta)) - 8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) - 16.0 * log((-1.0i) * xd) - 16.0 * log(1.0i * xd) - 5.0 * ln1myd - 2.0 * log(yd) - 11.0 * ln1pyd)
                + trilog(-1.0 + 1.0i * eta) * (-12.0 * log(1.0 - 1.0i * xd) - 12.0 * log(1.0 + 1.0i * xd) - (15.0 * ln1myd) / 2.0 - (15.0 * ln1pyd) / 2.0) + trilog(1.0 / (2.0 - 1.0i * eta)) * (6.0 * log(-1.0 + 1.0i * eta) - 12.0 * log((1.0i + eta) / (2.0 * 1.0i + eta)) - 11.0 * ln1myd - 2.0 * log(-yd) - 5.0 * ln1pyd) + trilog((1.0 + yd) / 2.0) * (10.0 * log(-yd) - 5.0 * ln1pyd) + li3half * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) + 16.0 * log((-1.0i) * xd) + 16.0 * log(1.0i * xd) - 5.0 * ln1myd - 10.0 * log(-yd) - 10.0 * log(yd) - 5.0 * ln1pyd)
                + trilog(-1.0i / (1.0i + eta)) * (-28.0 * log(1.0 - 1.0i * xd) - 28.0 * log(1.0 + 1.0i * xd) - (3.0 * ln1pyd) / 2.0) - 8.0 * trilog(1.0 + 1.0 / (1.0 - 1.0i * eta)) * ln1pyd + (15.0 * trilog((1.0i + eta) / (1.0i + eta - 1.0i * yd)) * ln1pyd) / 2.0 + 7.0 * trilog((2.0 * 1.0i + eta) / (1.0i + eta - 1.0i * yd)) * ln1pyd + 10.0 * trilog(-ydinv) * ln1pyd - 5.0 * trilog((-1.0 + yd) / (2.0 * yd)) * ln1pyd - (15.0 * trilog(-yd) * ln1pyd) / 2.0 + (27.0 * trilog((1.0i * yd) / (1.0i + eta)) * ln1pyd) / 2.0 - (trilog(1.0 / (1.0 + yd)) * ln1pyd) / 2.0
                - (15.0 * trilog((eta - 1.0i * (-1.0 + yd)) / ((1.0i + eta) * (1.0 + yd))) * ln1pyd) / 2.0 - 5.0 * trilog((-1.0 + yd) / (1.0 + yd)) * ln1pyd - 5.0 * trilog(yd / (1.0 + yd)) * ln1pyd - 2.0 * trilog(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (1.0 + yd))) * ln1pyd + (15.0 * trilog(((-1.0i) * (1.0 + yd)) / (eta - 1.0i * (-1.0 + yd))) * ln1pyd) / 2.0 - (15.0 * trilog(((1.0i + eta) * (1.0 + yd)) / (eta - 1.0i * (-1.0 + yd))) * ln1pyd) / 2.0 - 11.0 * trilog(((1.0i + eta) * (1.0 + yd)) / ((2.0 * 1.0i + eta) * yd)) * ln1pyd + 7.0 * trilog(yd / (-1.0 + 1.0i * eta + yd)) * ln1pyd
                - 7.0 * trilog(((2.0 - 1.0i * eta) * yd) / (-1.0 + 1.0i * eta + yd)) * ln1pyd + (11.0 * trilog((-1.0 + 1.0i * eta + yd) / yd) * ln1pyd) / 2.0 - 5.0 * trilog((-1.0 + 1.0i * eta + yd) / ((2.0 - 1.0i * eta) * yd)) * ln1pyd + (5.0 * trilog((-1.0 + 1.0i * eta + yd) / (1.0 + yd)) * ln1pyd) / 2.0 + 8.0 * trilog(1.0 - (1.0i * yd) / (1.0i + eta)) * ln1pyd + 5.0 * dilog((-1.0 + yd) / (2.0 * yd)) * log((1.0 - yd) / 2.0) * ln1pyd + 5.0 * dilog((-1.0 + yd) / (1.0 + yd)) * log((1.0 - yd) / 2.0) * ln1pyd + (7.0 * pisqu * log(-1.0 / (1.0 - 1.0i * eta - yd)) * ln1pyd) / 6.0 + (7.0 * power_of<3>(log(-1.0 / (1.0 - 1.0i * eta - yd))) * ln1pyd) / 6.0
                + 5.0 * dilog((-1.0 + 1.0i * eta + yd) / ((2.0 - 1.0i * eta) * yd)) * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * ln1pyd + (7.0 * power_of<3>(log(-ydinv)) * ln1pyd) / 12.0 - (5.0 * power_of<2>(log((1.0 - yd) / 2.0)) * log(ydinv) * ln1pyd) / 2.0 + 5.0 * dilog(yd / (1.0 + yd)) * log(-yd) * ln1pyd + 2.0 * dilog(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (1.0 + yd))) * log(-yd) * ln1pyd - 7.0 * dilog(yd / (-1.0 + 1.0i * eta + yd)) * log(-yd) * ln1pyd + 7.0 * dilog(((2.0 - 1.0i * eta) * yd) / (-1.0 + 1.0i * eta + yd)) * log(-yd) * ln1pyd + 4.0 * dilog(1.0 + 1.0 / (1.0 - 1.0i * eta)) * power_of<2>(ln1pyd)
                - (15.0 * dilog(((-1.0i) * (1.0 + yd)) / (eta - 1.0i * (-1.0 + yd))) * power_of<2>(ln1pyd)) / 2.0 + (15.0 * dilog(((1.0i + eta) * (1.0 + yd)) / (eta - 1.0i * (-1.0 + yd))) * power_of<2>(ln1pyd)) / 2.0 + 11.0 * dilog(((1.0i + eta) * (1.0 + yd)) / ((2.0 * 1.0i + eta) * yd)) * power_of<2>(ln1pyd) + trilog(1.0 + ydinv) * (5.0 * ln1myd + (7.0 * ln1pyd) / 2.0) + trilog((-1.0 + yd) / yd) * ((7.0 * ln1myd) / 2.0 + 5.0 * ln1pyd) + trilog((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) * (2.0 * log(-1.0 + 1.0i * eta) - 8.0 * log((1.0i + eta) / (2.0 * 1.0i + eta)) + 2.0 * log(-yd) + 8.0 * ln1pyd)
                + dilog((-1.0 + yd) / yd) * ((-7.0 * power_of<2>(ln1myd)) / 2.0 - 5.0 * log((1.0 - yd) / 2.0) * ln1pyd) + li2half * (8.0 * dilog(1.0 - 1.0i * xd) + 8.0 * dilog(1.0 + 1.0i * xd) - 5.0 * dilog(1.0 - yd) - 5.0 * dilog(1.0 + yd) + 16.0 * log(1.0 + 1.0i * xd) * log((-1.0i) * xd) + 16.0 * log(1.0 - 1.0i * xd) * log(1.0i * xd) - 10.0 * ln1myd * log(yd) - 10.0 * log(-yd) * ln1pyd) + dilog((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) * ((-5.0 * pisqu) / 6.0 - 11.0 * power_of<2>(ln1pyd)) + dilog(1.0 - (1.0i * yd) / (1.0i + eta)) * (-8.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * ln1pyd - 4.0 * power_of<2>(ln1pyd))
                + dilog(1.0 + ydinv) * (-5.0 * ln1myd * log((1.0 + yd) / 2.0) - (7.0 * power_of<2>(ln1pyd)) / 2.0) + dilog((1.0i * yd) / (1.0i + eta)) * (-2.0 * log(-yd) * ln1pyd - power_of<2>(ln1pyd)) + dilog(-1.0i / (1.0i + eta)) * (8.0 * power_of<2>(log(1.0 - 1.0i * xd)) + 8.0 * power_of<2>(log(1.0 + 1.0i * xd)) + (3.0 * power_of<2>(ln1pyd)) / 2.0) + log(-ydinv) * ((7.0 * pisqu * ln1pyd) / 12.0 - (7.0 * power_of<3>(ln1pyd)) / 4.0) + trilog(1.0 + yd) * (-2.0 * log(-1.0 + 1.0i * eta) + 8.0 * log((1.0i + eta) / (2.0 * 1.0i + eta)) + 12.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) + ln1pyd - 12.0 * log(1.0 - (1.0i * yd) / (1.0i + eta)))
                + dilog((-1.0 + 1.0i * eta + yd) / yd) * (-5.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * ln1pyd - (ln1pyd * log(1.0 - (1.0i * yd) / (1.0i + eta))) / 2.0) + dilog((eta - 1.0i * (-1.0 + yd)) / ((1.0i + eta) * (1.0 + yd))) * (8.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * ln1pyd - (ln1pyd * log(1.0 - (1.0i * yd) / (1.0i + eta))) / 2.0) + dilog((-1.0 + 1.0i * eta + yd) / (1.0 + yd)) * (-3.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * ln1pyd + (ln1pyd * log(1.0 - (1.0i * yd) / (1.0i + eta))) / 2.0)
                + dilog(-1.0 + 1.0i * eta) * (-12.0 * log((1.0i + eta - xd) / (1.0i + eta)) * log(1.0 - 1.0i * xd) - 12.0 * log(1.0 + 1.0i * xd) * log((1.0i + eta + xd) / (1.0i + eta)) - (15.0 * ln1pyd * log(1.0 - (1.0i * yd) / (1.0i + eta))) / 2.0 - (15.0 * ln1myd * log(1.0 + (1.0i * yd) / (1.0i + eta))) / 2.0) + dilog((1.0 - 1.0i * eta + yd) / yd) * ln1myd * ((-0.5) * log(1.0 + (1.0i * yd) / (1.0i + eta)) - 5.0 * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) + dilog((1.0 - 1.0i * eta + yd) / (-1.0 + yd)) * ln1myd * (log(1.0 + (1.0i * yd) / (1.0i + eta)) / 2.0 - 3.0 * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)))
                + 5.0 * dilog((eta + 1.0i * (1.0 + yd)) / ((2.0 * 1.0i + eta) * yd)) * ln1myd * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) + dilog(-((eta + 1.0i * (1.0 + yd)) / ((1.0i + eta) * (-1.0 + yd)))) * ln1myd * ((-0.5) * log(1.0 + (1.0i * yd) / (1.0i + eta)) + 8.0 * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) + trilog(1.0 - yd) * (-2.0 * log(-1.0 + 1.0i * eta) + 8.0 * log((1.0i + eta) / (2.0 * 1.0i + eta)) + ln1myd - 12.0 * log(1.0 + (1.0i * yd) / (1.0i + eta)) + 12.0 * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) + dilog(1.0 + (1.0i * yd) / (1.0i + eta)) * (-4.0 * power_of<2>(ln1myd) - 8.0 * ln1myd * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)))
                + power_of<3>(log(1.0 - 1.0i * xd)) * (-6.0 * log((2.0 * 1.0i + eta) / (1.0i + eta - xd)) + 6.0 * log(-1.0i / xd) + 6.0 * log((1.0i * (2.0 * 1.0i + eta) * xd) / (1.0i + eta - xd)) - (4.0 * 1.0i) * M_PI * my_sign(real(xd)) * T(1.0, (1.0i + eta - xd) / (1.0i + eta), 1.0 - 1.0i * xd)) + (24.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * log(-1.0 + 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 - 1.0i * xd, 2.0 - 1.0i * eta) + dilog(1.0 - 1.0i * xd) * (-8.0 * dilog(0.5 - (1.0i / 2.0) * xd) + 20.0 * dilog((1.0i + xd) / (2.0 * 1.0i + eta)) + 8.0 * log((1.0i + eta - xd) / (1.0i + eta)) * log(1.0 - 1.0i * xd) - (16.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 - 1.0i * xd, 2.0 - 1.0i * eta))
                + log(1.0i * xd) * (4.0 * power_of<2>(log(1.0)) * log((1.0i + eta) / (2.0 * 1.0i + eta)) + (16.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * my_sign(real(eta)) * T(1.0, 1.0 - 1.0i * xd, 2.0 - 1.0i * eta)) + (24.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * log(-1.0 + 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 + 1.0i * xd, 2.0 - 1.0i * eta) + dilog(1.0 + 1.0i * xd) * (20.0 * dilog((1.0i - xd) / (2.0 * 1.0i + eta)) - 8.0 * dilog((1.0 + 1.0i * xd) / 2.0) + 8.0 * log(1.0 + 1.0i * xd) * log((1.0i + eta + xd) / (1.0i + eta)) - (16.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 + 1.0i * xd, 2.0 - 1.0i * eta))
                + log((-1.0i) * xd) * (4.0 * power_of<2>(log(1.0)) * log((1.0i + eta) / (2.0 * 1.0i + eta)) + (16.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * my_sign(real(eta)) * T(1.0, 1.0 + 1.0i * xd, 2.0 - 1.0i * eta)) + power_of<2>(log(1.0 + 1.0i * xd)) * (-8.0 * log((-1.0i) * xd) * log((1.0i + eta + xd) / (1.0i + eta)) - 4.0 * log(xd) * log((1.0i + eta + xd) / (1.0i + eta)) - (24.0 * 1.0i) * M_PI * log(-1.0 + 1.0i * eta) * my_sign(real(1.0 / (1.0i - xd))) * T(1.0, (-1.0i) * xd, -1.0 + 1.0i * eta)) + power_of<2>(log(1.0 - 1.0i * xd)) * (log((1.0i + eta - xd) / (1.0i + eta)) * (-8.0 * log(1.0i * xd) - 4.0 * log(xd)) - (24.0 * 1.0i) * M_PI * log(-1.0 + 1.0i * eta) * my_sign(real(1.0 / (1.0i + xd))) * T(1.0, 1.0i * xd, -1.0 + 1.0i * eta))
                + log(1.0 - 1.0i * xd) * (2.0 * pisqu * log(1.0i / (1.0i + eta - xd)) + 2.0 * power_of<3>(log(1.0i / (1.0i + eta - xd))) - 2.0 * pisqu * log(-1.0i / xd) - 2.0 * power_of<3>(log(-1.0i / xd)) - (4.0 * pisqu * log((2.0 * 1.0i) * xdinv)) / 3.0 - (4.0 * power_of<3>(log((2.0 * 1.0i) * xdinv))) / 3.0 + ((-2.0 * pisqu) / 3.0 + 2.0 * power_of<2>(log((1.0i + eta - xd) / (1.0i + eta))) - 4.0 * power_of<2>(log((1.0i + eta - xd) / (2.0 * 1.0i + eta)))) * log((1.0i + eta) * xdinv) - (2.0 * power_of<3>(log((1.0i + eta) * xdinv))) / 3.0 + (4.0 * pisqu * log((2.0 * 1.0i + eta) * xdinv)) / 3.0 + (4.0 * power_of<3>(log((2.0 * 1.0i + eta) * xdinv))) / 3.0 - 12.0 * log((1.0i + eta - xd) / (1.0i + eta)) * power_of<2>(log(1.0i * xd)) + (4.0 * pisqu * log((2.0 * 1.0i) / (1.0i + xd))) / 3.0 + (4.0 * power_of<3>(log((2.0 * 1.0i) / (1.0i + xd)))) / 3.0
                + (2.0 * pisqu * log((1.0i + eta) / (1.0i + xd))) / 3.0 + (2.0 * power_of<3>(log((1.0i + eta) / (1.0i + xd)))) / 3.0 - (4.0 * pisqu * log((2.0 * 1.0i + eta) / (1.0i + xd))) / 3.0 - (4.0 * power_of<3>(log((2.0 * 1.0i + eta) / (1.0i + xd)))) / 3.0 - 4.0 * pisqu * log((1.0i * (2.0 * 1.0i + eta)) / ((1.0i + eta) * (1.0i + xd))) - 4.0 * power_of<3>(log((1.0i * (2.0 * 1.0i + eta)) / ((1.0i + eta) * (1.0i + xd)))) + power_of<2>(log((1.0 + 1.0i * xd) / 2.0)) * (4.0 * log(1.0i / xd) - 4.0 * log((2.0 * 1.0i) / (1.0i + xd)) - 4.0 * log((1.0i + xd) / (2.0 * xd))) - (8.0 * 1.0i) * M_PI * H1(2.0 - 1.0i * eta, (2.0 * 1.0i + eta) / (1.0i + xd)) * power_of<2>(log(-(xd / (2.0 * 1.0i + eta)))) * my_sign(imag((2.0 * 1.0i + eta) / (1.0i + xd)))
                - (12.0 * 1.0i) * M_PI * H1(-1.0i / (1.0i + eta), -1.0i / xd) * power_of<2>(log(1.0i * (1.0i + eta - xd))) * my_sign(-real(xdinv)) - (4.0 * 1.0i) * M_PI * H1(-1.0 + 1.0i * eta, (1.0i + eta) * xdinv) * power_of<2>(log(-((1.0i + xd) / (1.0i + eta)))) * my_sign(real((1.0 - 1.0i * eta) * xdinv)) - (24.0 * 1.0i) * M_PI * H1(1.0i / (2.0 * 1.0i + eta), 1.0i / (1.0i + xd)) * power_of<2>(log(1.0 - 1.0i * eta + 1.0i * xd)) * my_sign(real(1.0 / (1.0i + xd))) + (24.0 * 1.0i) * M_PI * H1((2.0 * 1.0i + eta) / (1.0i + xd), 1.0i / (1.0i + xd)) * power_of<2>(log((1.0i * (1.0i + eta) * (1.0i + xd)) / (2.0 * 1.0i + eta))) * my_sign(real(1.0 / (1.0i + xd)))
                + power_of<2>(log(1.0i * xd)) * (-12.0 * log((1.0i + eta) / (1.0i + eta - xd)) + 12.0 * log((1.0i * (1.0i + eta - xd)) / ((1.0i + eta) * (1.0i + xd))) + 12.0 * log((1.0i + eta + xd - 1.0i * eta * xd) / (1.0i + eta - xd)) + (8.0 * 1.0i) * M_PI * my_sign(real(xd)) * T(1.0, (1.0i + eta - xd) / (2.0 * 1.0i + eta), 1.0i * xd)) - (16.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * log(1.0i * xd) * my_sign(real(eta)) * T(1.0, 1.0 - 1.0i * xd, 2.0 - 1.0i * eta) + power_of<2>(log((1.0i + eta - xd) / (1.0i + eta))) * (-2.0 * log((2.0 * 1.0i + eta) / (1.0i + xd)) + 2.0 * log(((2.0 * 1.0i + eta) * xd) / ((1.0i + eta) * (1.0i + xd))) - (12.0 * 1.0i) * M_PI * my_sign(imag(xd / (1.0i + eta))) * T(1.0, 1.0 - 1.0i * xd, (1.0i + eta - xd) / (1.0i + eta)))
                - (24.0 * 1.0i) * M_PI * power_of<2>(log(-1.0 + 1.0i * eta)) * my_sign(real(eta)) * T(1.0, 1.0i * xd, -1.0 + 1.0i * eta) + power_of<2>(log((1.0i + eta - xd) / (2.0 * 1.0i + eta))) * (4.0 * log((2.0 * 1.0i + eta) / (1.0i + xd)) + 4.0 * log(((1.0i + eta) * (1.0i + xd)) / ((2.0 * 1.0i + eta) * xd)) - (24.0 * 1.0i) * M_PI * my_sign(imag((1.0i + xd) / (2.0 * 1.0i + eta))) * T(1.0, 1.0i * xd, (1.0i + eta - xd) / (2.0 * 1.0i + eta)))) + power_of<3>(log(1.0 + 1.0i * xd)) * (6.0 * log(1.0i / xd) - 6.0 * log((2.0 * 1.0i + eta) / (1.0i + eta + xd)) + 6.0 * log(((2.0 - 1.0i * eta) * xd) / (1.0i + eta + xd)) - (4.0 * 1.0i) * M_PI * my_sign(-real(xd)) * T(1.0, (1.0i + eta + xd) / (1.0i + eta), 1.0 + 1.0i * xd))
                + log(1.0 + 1.0i * xd) * ((-4.0 * power_of<3>(log((2.0 * 1.0i + eta) / (1.0i - xd)))) / 3.0 + ((4.0 * pisqu) / 3.0 - 4.0 * power_of<2>(log(0.5 - (1.0i / 2.0) * xd))) * log(2.0 / (1.0 + 1.0i * xd)) + (4.0 * power_of<3>(log(2.0 / (1.0 + 1.0i * xd)))) / 3.0 - 2.0 * pisqu * log(1.0i / xd) - 2.0 * power_of<3>(log(1.0i / xd)) - (4.0 * pisqu * log((-2.0 * 1.0i) * xdinv)) / 3.0 - (4.0 * power_of<3>(log((-2.0 * 1.0i) * xdinv))) / 3.0 - (2.0 * power_of<3>(log(-((1.0i + eta) * xdinv)))) / 3.0 + (4.0 * pisqu * log(-((2.0 * 1.0i + eta) * xdinv))) / 3.0 + (4.0 * power_of<3>(log(-((2.0 * 1.0i + eta) * xdinv)))) / 3.0 - 4.0 * pisqu * log((2.0 - 1.0i * eta) / ((1.0i + eta) * (-1.0i + xd))) - 4.0 * power_of<3>(log((2.0 - 1.0i * eta) / ((1.0i + eta) * (-1.0i + xd)))) + power_of<2>(log(0.5 - (1.0i / 2.0) * xd)) * (4.0 * log(-1.0i / xd) - 4.0 * log((-1.0i + xd) / (2.0 * xd)))
                + 4.0 * pisqu * log(-1.0i / (1.0i + eta + xd)) + 4.0 * power_of<3>(log(-1.0i / (1.0i + eta + xd))) + 2.0 * pisqu * log(1.0i / (1.0i + eta + xd)) + 2.0 * power_of<3>(log(1.0i / (1.0i + eta + xd))) + log(-((1.0i + eta) * xdinv)) * ((-2.0 * pisqu) / 3.0 + 2.0 * power_of<2>(log((1.0i + eta + xd) / (1.0i + eta))) - 4.0 * power_of<2>(log((1.0i + eta + xd) / (2.0 * 1.0i + eta)))) + log((2.0 * 1.0i + eta) / (1.0i - xd)) * ((-4.0 * pisqu) / 3.0 - 2.0 * power_of<2>(log((1.0i + eta + xd) / (1.0i + eta))) + 4.0 * power_of<2>(log((1.0i + eta + xd) / (2.0 * 1.0i + eta)))) - (8.0 * 1.0i) * M_PI * H1(2.0 - 1.0i * eta, (2.0 * 1.0i + eta) / (1.0i - xd)) * power_of<2>(log(xd / (2.0 * 1.0i + eta))) * my_sign(imag((2.0 * 1.0i + eta) / (1.0i - xd)))
                + (24.0 * 1.0i) * M_PI * H1((2.0 * 1.0i + eta) / (1.0i - xd), -1.0i / (-1.0i + xd)) * power_of<2>(log(((1.0 - 1.0i * eta) * (-1.0i + xd)) / (2.0 * 1.0i + eta))) * my_sign(real(1.0 / (1.0i - xd))) - (24.0 * 1.0i) * M_PI * H1(1.0i / (2.0 * 1.0i + eta), -1.0i / (-1.0i + xd)) * power_of<2>(log((-1.0i) * (1.0i + eta + xd))) * my_sign(real(1.0 / (1.0i - xd))) - (12.0 * 1.0i) * M_PI * H1(-1.0i / (1.0i + eta), 1.0i / xd) * power_of<2>(log(1.0i * (1.0i + eta + xd))) * my_sign(real(xdinv)) - (4.0 * 1.0i) * M_PI * H1(-1.0 + 1.0i * eta, -((1.0i + eta) * xdinv)) * power_of<2>(log((-1.0i + xd) / (1.0i + eta))) * my_sign(-real((1.0 - 1.0i * eta) * xdinv)) - (16.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * log((-1.0i) * xd) * my_sign(real(eta)) * T(1.0, 1.0 + 1.0i * xd, 2.0 - 1.0i * eta)
                + power_of<2>(log((1.0i + eta + xd) / (1.0i + eta))) * (2.0 * log(((2.0 * 1.0i + eta) * xd) / ((1.0i + eta) * (-1.0i + xd))) - (12.0 * 1.0i) * M_PI * my_sign(-imag(xd / (1.0i + eta))) * T(1.0, 1.0 + 1.0i * xd, (1.0i + eta + xd) / (1.0i + eta))) - (24.0 * 1.0i) * M_PI * power_of<2>(log(-1.0 + 1.0i * eta)) * my_sign(real(eta)) * T(1.0, (-1.0i) * xd, -1.0 + 1.0i * eta) + power_of<2>(log((1.0i + eta + xd) / (2.0 * 1.0i + eta))) * (4.0 * log(((1.0i + eta) * (-1.0i + xd)) / ((2.0 * 1.0i + eta) * xd)) - (24.0 * 1.0i) * M_PI * my_sign(imag((1.0i - xd) / (2.0 * 1.0i + eta))) * T(1.0, (-1.0i) * xd, (1.0i + eta + xd) / (2.0 * 1.0i + eta)))
                + power_of<2>(log((-1.0i) * xd)) * (-12.0 * log((1.0i + eta) / (1.0i + eta + xd)) + 12.0 * log(((1.0i + eta) * (1.0 + 1.0i * xd)) / (1.0i + eta + xd)) - 12.0 * log((1.0i + eta + xd) / (1.0i + eta)) + 12.0 * log(((-1.0i) * (1.0i + eta + xd)) / ((1.0i + eta) * (-1.0i + xd))) + (8.0 * 1.0i) * M_PI * my_sign(-real(xd)) * T(1.0, (1.0i + eta + xd) / (2.0 * 1.0i + eta), (-1.0i) * xd))) + ((5.0 * 1.0i) / 3.0) * power_of<3>(M_PI) * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta) - (6.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * log(-1.0 + 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta)
                + dilog(yd) * (ln1myd * ((9.0 * log(1.0 + (1.0i * yd) / (1.0i + eta))) / 2.0 + 11.0 * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) + (22.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(imydinv) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta)) + dilog(1.0 - yd) * (5.0 * dilog((1.0 - yd) / 2.0) + 7.0 * dilog((-1.0 + yd) / (-2.0 + 1.0i * eta)) + 12.0 * ln1myd * log(1.0 + (1.0i * yd) / (1.0i + eta)) + (6.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta)) + log(yd) * ((5.0 * power_of<2>(log(1.0)) * log((1.0i + eta) / (2.0 * 1.0i + eta))) / 2.0 + (2.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta))
                + power_of<2>(log(-yd)) * ln1pyd * ((-7.0 * log((1.0i + eta) / (1.0i + eta - 1.0i * yd))) / 2.0 + (5.0 * log(1.0 / (1.0 + yd))) / 2.0 + log((eta - 1.0i * (-1.0 + yd)) / ((1.0i + eta) * (1.0 + yd))) + (7.0 * log(((1.0i + eta) * (1.0 + yd)) / (eta - 1.0i * (-1.0 + yd)))) / 2.0 - log(1.0 - (1.0i * yd) / (1.0i + eta)) + (5.0 * 1.0i) * M_PI * my_sign(-imag(yd)) * T(1.0, (1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta), -yd))
                + power_of<2>(ln1pyd) * (-4.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * log((1.0i * yd) / (1.0i + eta)) - (8.0 * 1.0i) * M_PI * log((1.0i + eta) / (2.0 * 1.0i + eta)) * my_sign(imag((2.0 - 1.0i * eta) / (1.0 + yd))) * T(1.0, (1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta), (1.0i + eta) / (2.0 * 1.0i + eta)) - (2.0 * 1.0i) * M_PI * log(-1.0 + 1.0i * eta) * my_sign(imag(1.0 / (1.0 + yd))) * T(1.0, -yd, -1.0 + 1.0i * eta)) + ((5.0 * 1.0i) / 3.0) * power_of<3>(M_PI) * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta) - (6.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * log(-1.0 + 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta)
                + dilog(-yd) * (11.0 * dilog((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) + 11.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * ln1pyd + (9.0 * ln1pyd * log(1.0 - (1.0i * yd) / (1.0i + eta))) / 2.0 + (22.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta)) + trilog(2.0 - 1.0i * eta) * (36.0 * log(-1.0 + 1.0i * eta) - 24.0 * log(1.0 - 1.0i * xd) - 24.0 * log(1.0 + 1.0i * xd) - 7.0 * ln1myd - 7.0 * ln1pyd - (48.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 - 1.0i * xd, 2.0 - 1.0i * eta) - (48.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 + 1.0i * xd, 2.0 - 1.0i * eta) - (24.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta)
                - (24.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta)) + dilog(1.0 + yd) * (5.0 * dilog((1.0 + yd) / 2.0) + 7.0 * dilog((1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) + 12.0 * ln1pyd * log(1.0 - (1.0i * yd) / (1.0i + eta)) + (6.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta)) + dilog(2.0 - 1.0i * eta) * (-18.0 * pisqu - 12.0 * dilog(1.0 - 1.0i * xd) - 12.0 * dilog(1.0 + 1.0i * xd) - dilog(1.0 - yd) - dilog(1.0 + yd) - 18.0 * power_of<2>(log(-2.0 + 1.0i * eta)) - 24.0 * log((1.0i + eta - xd) / (2.0 * 1.0i + eta)) * log(1.0 - 1.0i * xd) - 24.0 * log(1.0 + 1.0i * xd) * log((1.0i + eta + xd) / (2.0 * 1.0i + eta))
                - 7.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * ln1pyd - 7.0 * ln1myd * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) + (64.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 - 1.0i * xd, 2.0 - 1.0i * eta) + (64.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 + 1.0i * xd, 2.0 - 1.0i * eta) + (8.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta) + (8.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta))
                + log(-yd) * ((5.0 * power_of<2>(log(1.0)) * log((1.0i + eta) / (2.0 * 1.0i + eta))) / 2.0 + 5.0 * power_of<2>(ln1pyd) * log(1.0 - (1.0i * yd) / (1.0i + eta)) + (2.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta) + (6.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * ln1pyd * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta)) + ln1pyd * ((5.0 * pisqu * log(1.0 / (1.0 - 1.0i * eta - yd))) / 4.0 + (5.0 * power_of<3>(log(1.0 / (1.0 - 1.0i * eta - yd)))) / 4.0 - (5.0 * pisqu * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta))) / 6.0 + (5.0 * pisqu * log(2.0 / yd)) / 6.0 + (5.0 * power_of<3>(log(2.0 / yd))) / 6.0 + power_of<3>(log((1.0 - 1.0i * eta) / yd)) / 12.0 + (5.0 * pisqu * log((2.0 - 1.0i * eta) / yd)) / 6.0
                + (5.0 * power_of<3>(log((2.0 - 1.0i * eta) / yd))) / 6.0 - (5.0 * pisqu * log(1.0 / (1.0 + yd))) / 6.0 - (5.0 * power_of<3>(log(1.0 / (1.0 + yd)))) / 6.0 - (5.0 * pisqu * log(2.0 / (1.0 + yd))) / 6.0 - (5.0 * power_of<3>(log(2.0 / (1.0 + yd)))) / 6.0 - (pisqu * log((1.0 - 1.0i * eta) / (1.0 + yd))) / 12.0 - power_of<3>(log((1.0 - 1.0i * eta) / (1.0 + yd))) / 12.0 + power_of<3>(log((2.0 - 1.0i * eta) / (1.0 + yd))) / 2.0 - (4.0 * pisqu * log(-((2.0 * 1.0i + eta) / ((1.0i + eta) * (1.0 + yd))))) / 3.0 - (4.0 * power_of<3>(log(-((2.0 * 1.0i + eta) / ((1.0i + eta) * (1.0 + yd)))))) / 3.0 - (pisqu * log((2.0 * 1.0i + eta) / ((1.0i + eta) * (1.0 + yd)))) / 3.0 - power_of<3>(log((2.0 * 1.0i + eta) / ((1.0i + eta) * (1.0 + yd)))) / 3.0
                + power_of<2>(log((1.0 - yd) / 2.0)) * ((5.0 * log(2.0 / (1.0 + yd))) / 2.0 + (5.0 * log((1.0 + yd) / (2.0 * yd))) / 2.0) - (11.0 * pisqu * log(-((1.0i + eta) / ((2.0 * 1.0i) * yd + eta * yd)))) / 6.0 - (11.0 * power_of<3>(log(-((1.0i + eta) / ((2.0 * 1.0i) * yd + eta * yd))))) / 6.0 - (log(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (1.0 + yd))) * power_of<2>(log(1.0 - (1.0i * yd) / (1.0i + eta)))) / 4.0 + log((1.0 - 1.0i * eta) / yd) * (pisqu / 12.0 - power_of<2>(log(1.0 - (1.0i * yd) / (1.0i + eta))) / 4.0) + log((2.0 - 1.0i * eta) / (1.0 + yd)) * (pisqu / 2.0 + power_of<2>(log(1.0 - (1.0i * yd) / (1.0i + eta))) / 4.0) + (11.0 * 1.0i) * M_PI * H1((1.0 - 1.0i * eta) / yd, -ydinv) * power_of<2>(log(((2.0 * 1.0i + eta) * yd) / (1.0i + eta))) * my_sign(-imydinv)
                - ((15.0 * 1.0i) / 2.0) * M_PI * H1(-1.0i / (1.0i + eta), -ydinv) * power_of<2>(log(-1.0 + 1.0i * eta + yd)) * my_sign(-imydinv) + (1.0i / 2.0) * M_PI * H1(-1.0 + 1.0i * eta, (1.0 - 1.0i * eta) / yd) * power_of<2>(log(((-1.0i) * (1.0 + yd)) / (1.0i + eta))) * my_sign(imag((1.0 - 1.0i * eta) / yd)) - (7.0 * 1.0i) * M_PI * H1(1.0 / (2.0 - 1.0i * eta), 1.0 / (1.0 + yd)) * power_of<2>(log(1.0 - 1.0i * eta - yd)) * my_sign(imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1((2.0 - 1.0i * eta) / (1.0 + yd), 1.0 / (1.0 + yd)) * power_of<2>(log(-(((1.0i + eta) * (1.0 + yd)) / (2.0 * 1.0i + eta)))) * my_sign(imag(1.0 / (1.0 + yd))) - (5.0 * 1.0i) * M_PI * H1(2.0 - 1.0i * eta, (2.0 - 1.0i * eta) / (1.0 + yd)) * power_of<2>(log(((-1.0i) * yd) / (2.0 * 1.0i + eta))) * my_sign(imag((2.0 - 1.0i * eta) / (1.0 + yd)))
                + (8.0 * 1.0i) * M_PI * H1(1.0 / (1.0 + yd), (2.0 - 1.0i * eta) / (1.0 + yd)) * power_of<2>(log(((1.0i + eta) * (1.0 + yd)) / (2.0 * 1.0i + eta))) * my_sign(imag((2.0 - 1.0i * eta) / (1.0 + yd))) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0i + eta) / (2.0 * 1.0i + eta))) * my_sign(-real(1.0 / (2.0 * 1.0i + eta))) * T(1.0, (1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta), (1.0i + eta) / (2.0 * 1.0i + eta)) - (2.0 * 1.0i) * M_PI * power_of<2>(log(-1.0 + 1.0i * eta)) * my_sign(real(eta)) * T(1.0, -yd, -1.0 + 1.0i * eta)
                + power_of<2>(log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta))) * ((-5.0 * log((1.0 - 1.0i * eta) / yd)) / 2.0 - 4.0 * log((1.0i * yd) / (1.0i + eta)) - (3.0 * log((2.0 - 1.0i * eta) / (1.0 + yd))) / 2.0 + 4.0 * log(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (1.0 + yd))) + (5.0 * log(((1.0i + eta) * (1.0 + yd)) / ((2.0 * 1.0i + eta) * yd))) / 2.0 - (7.0 * 1.0i) * M_PI * my_sign(real((1.0 + yd) / (2.0 * 1.0i + eta))) * T(1.0, -yd, (1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta))) + (11.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta) - ((15.0 * 1.0i) / 2.0) * M_PI * power_of<2>(log(1.0 - (1.0i * yd) / (1.0i + eta))) * my_sign(real(yd / (1.0i + eta))) * T(1.0, 1.0 + yd, 1.0 - (1.0i * yd) / (1.0i + eta)))
                + power_of<3>(ln1pyd) * ((-15.0 * log((2.0 * 1.0i + eta) / (1.0i + eta - 1.0i * yd))) / 4.0 - (11.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta))) / 2.0 + (15.0 * log(((2.0 - 1.0i * eta) * yd) / (-1.0 + 1.0i * eta + yd))) / 4.0 + (11.0 * log((-1.0 + 1.0i * eta + yd) / ((2.0 - 1.0i * eta) * yd))) / 2.0 + (1.0i / 2.0) * M_PI * my_sign(-imag(yd)) * T(1.0, 1.0 - (1.0i * yd) / (1.0i + eta), 1.0 + yd))
                + power_of<3>(ln1myd) * ((-7.0 * log(ydinv)) / 4.0 - (15.0 * log((2.0 * 1.0i + eta) / (eta + 1.0i * (1.0 + yd)))) / 4.0 + (15.0 * log(((2.0 * 1.0i + eta) * yd) / (eta + 1.0i * (1.0 + yd)))) / 4.0 - (11.0 * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) / 2.0 + (11.0 * log((eta + 1.0i * (1.0 + yd)) / ((2.0 * 1.0i + eta) * yd))) / 2.0 + (1.0i / 2.0) * M_PI * my_sign(imag(yd)) * T(1.0, 1.0 + (1.0i * yd) / (1.0i + eta), 1.0 - yd))
                + power_of<2>(ln1myd) * (5.0 * log(yd) * log(1.0 + (1.0i * yd) / (1.0i + eta)) - 4.0 * log(yd / (-1.0 + 1.0i * eta)) * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) - (2.0 * 1.0i) * M_PI * log(-1.0 + 1.0i * eta) * my_sign(imag(1.0 / (1.0 - yd))) * T(1.0, yd, -1.0 + 1.0i * eta) - (8.0 * 1.0i) * M_PI * log((1.0i + eta) / (2.0 * 1.0i + eta)) * my_sign(real((2.0 * 1.0i + eta) / (-1.0 + yd))) * T(1.0, (eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta), (1.0i + eta) / (2.0 * 1.0i + eta)))
                + ln1myd * ((-5.0 * power_of<3>(log(-2.0 / (-1.0 + yd)))) / 6.0 - (pisqu * log((1.0i * (1.0i + eta)) / (-1.0 + yd))) / 12.0 - power_of<3>(log((1.0i * (1.0i + eta)) / (-1.0 + yd))) / 12.0 + power_of<3>(log((1.0i * (2.0 * 1.0i + eta)) / (-1.0 + yd))) / 2.0 - (pisqu * log(-((2.0 * 1.0i + eta) / ((1.0i + eta) * (-1.0 + yd))))) / 3.0 - power_of<3>(log(-((2.0 * 1.0i + eta) / ((1.0i + eta) * (-1.0 + yd))))) / 3.0 - (4.0 * pisqu * log((2.0 * 1.0i + eta) / ((1.0i + eta) * (-1.0 + yd)))) / 3.0 - (4.0 * power_of<3>(log((2.0 * 1.0i + eta) / ((1.0i + eta) * (-1.0 + yd))))) / 3.0 + (5.0 * pisqu * log(-2.0 / yd)) / 6.0 + (5.0 * power_of<3>(log(-2.0 / yd))) / 6.0 + (7.0 * pisqu * log(ydinv)) / 12.0 + (7.0 * power_of<3>(log(ydinv))) / 12.0 + power_of<3>(log((1.0i * (1.0i + eta)) / yd)) / 12.0
                + (5.0 * pisqu * log((1.0i * (2.0 * 1.0i + eta)) / yd)) / 6.0 + (5.0 * power_of<3>(log((1.0i * (2.0 * 1.0i + eta)) / yd))) / 6.0 + ((-5.0 * log(-ydinv)) / 2.0 + (5.0 * log((-1.0 + yd) / (2.0 * yd))) / 2.0) * power_of<2>(log((1.0 + yd) / 2.0)) + log(-2.0 / (-1.0 + yd)) * ((-5.0 * pisqu) / 6.0 + (5.0 * power_of<2>(log((1.0 + yd) / 2.0))) / 2.0) + (7.0 * pisqu * log(-1.0 / (1.0 - 1.0i * eta + yd))) / 6.0 + (7.0 * power_of<3>(log(-1.0 / (1.0 - 1.0i * eta + yd)))) / 6.0 - (11.0 * pisqu * log((1.0i + eta) / ((2.0 * 1.0i) * yd + eta * yd))) / 6.0 - (11.0 * power_of<3>(log((1.0i + eta) / ((2.0 * 1.0i) * yd + eta * yd)))) / 6.0 + (5.0 * pisqu * log(1.0i / (eta + 1.0i * (1.0 + yd)))) / 4.0 + (5.0 * power_of<3>(log(1.0i / (eta + 1.0i * (1.0 + yd))))) / 4.0
                - (5.0 * pisqu * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) / 6.0 + 4.0 * log(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (-1.0 + yd))) * power_of<2>(log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) + log((1.0i * (1.0i + eta)) / yd) * (pisqu / 12.0 - power_of<2>(log(1.0 + (1.0i * yd) / (1.0i + eta))) / 4.0 - (5.0 * power_of<2>(log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)))) / 2.0) + log((1.0i * (2.0 * 1.0i + eta)) / (-1.0 + yd)) * (pisqu / 2.0 + power_of<2>(log(1.0 + (1.0i * yd) / (1.0i + eta))) / 4.0 - (3.0 * power_of<2>(log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)))) / 2.0) + (2.0 * 1.0i) * M_PI * H1((1.0i * (2.0 * 1.0i + eta)) / (-1.0 + yd), 1.0 / (1.0 - yd)) * power_of<2>(log(((1.0i + eta) * (-1.0 + yd)) / (2.0 * 1.0i + eta))) * my_sign(imag(1.0 / (1.0 - yd)))
                - (7.0 * 1.0i) * M_PI * H1(1.0i / (2.0 * 1.0i + eta), 1.0 / (1.0 - yd)) * power_of<2>(log(1.0 - 1.0i * eta + yd)) * my_sign(imag(1.0 / (1.0 - yd))) - ((15.0 * 1.0i) / 2.0) * M_PI * H1(1.0 / (-1.0 + 1.0i * eta), ydinv) * power_of<2>(log(-1.0 + 1.0i * eta - yd)) * my_sign(imydinv) + (11.0 * 1.0i) * M_PI * H1((1.0i * (1.0i + eta)) / yd, ydinv) * power_of<2>(log((-1.0 + 1.0 / (-1.0 + 1.0i * eta)) * yd)) * my_sign(imydinv) + (8.0 * 1.0i) * M_PI * H1(1.0 / (1.0 - yd), (1.0i * (2.0 * 1.0i + eta)) / (-1.0 + yd)) * power_of<2>(log(-(((1.0i + eta) * (-1.0 + yd)) / (2.0 * 1.0i + eta)))) * my_sign(real((2.0 * 1.0i + eta) / (-1.0 + yd)))
                - (5.0 * 1.0i) * M_PI * H1(2.0 - 1.0i * eta, (1.0i * (2.0 * 1.0i + eta)) / (-1.0 + yd)) * power_of<2>(log((1.0i * yd) / (2.0 * 1.0i + eta))) * my_sign(real((2.0 * 1.0i + eta) / (-1.0 + yd))) + (1.0i / 2.0) * M_PI * H1(-1.0 + 1.0i * eta, (1.0i * (1.0i + eta)) / yd) * power_of<2>(log((1.0i * (-1.0 + yd)) / (1.0i + eta))) * my_sign(real((1.0i + eta) / yd)) + (11.0 * 1.0i) * M_PI * power_of<2>(log(2.0 - 1.0i * eta)) * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta) + (6.0 * 1.0i) * M_PI * log(2.0 - 1.0i * eta) * log(yd) * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta) + power_of<2>(log(1.0 + (1.0i * yd) / (1.0i + eta))) * ((-1.0 / 4.0) * log(((2.0 * 1.0i + eta) * yd) / ((1.0i + eta) * (-1.0 + yd))) - ((15.0 * 1.0i) / 2.0) * M_PI * my_sign(-real(yd / (1.0i + eta))) * T(1.0, 1.0 - yd, 1.0 + (1.0i * yd) / (1.0i + eta)))
                - (2.0 * 1.0i) * M_PI * power_of<2>(log(-1.0 + 1.0i * eta)) * my_sign(real(eta)) * T(1.0, yd, -1.0 + 1.0i * eta) + power_of<2>(log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) * ((5.0 * log(((1.0i + eta) * (-1.0 + yd)) / ((2.0 * 1.0i + eta) * yd))) / 2.0 - 4.0 * log(yd / (-1.0 + 1.0i * eta)) - (7.0 * 1.0i) * M_PI * my_sign(imag((-1.0 + yd) / (-2.0 + 1.0i * eta))) * T(1.0, yd, (eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta))) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0i + eta) / (2.0 * 1.0i + eta))) * my_sign(-real(1.0 / (2.0 * 1.0i + eta))) * T(1.0, (eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta), (1.0i + eta) / (2.0 * 1.0i + eta))
                + power_of<2>(log(yd)) * (-log(1.0 + (1.0i * yd) / (1.0i + eta)) - (7.0 * log((1.0i + eta) / (eta + 1.0i * (1.0 + yd)))) / 2.0 + (7.0 * log(-(((1.0i + eta) * (-1.0 + yd)) / (eta + 1.0i * (1.0 + yd))))) / 2.0 + log(-((eta + 1.0i * (1.0 + yd)) / ((1.0i + eta) * (-1.0 + yd)))) + (5.0 * 1.0i) * M_PI * my_sign(imag(yd)) * T(1.0, (eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta), yd))) + 72.0 * log((1.0i + eta) / (2.0 * 1.0i + eta)) * zeta3 + 8.0 * log((1.0i + eta - xd) / (1.0i + eta)) * zeta3 - 24.0 * log((1.0i + eta - xd) / (2.0 * 1.0i + eta)) * zeta3 + 8.0 * log((1.0i + eta + xd) / (1.0i + eta)) * zeta3 - 24.0 * log((1.0i + eta + xd) / (2.0 * 1.0i + eta)) * zeta3
                - 12.0 * log((1.0i + eta - 1.0i * yd) / (2.0 * 1.0i + eta)) * zeta3 + 12.0 * log(1.0 - (1.0i * yd) / (1.0i + eta)) * zeta3 + 12.0 * log(1.0 + (1.0i * yd) / (1.0i + eta)) * zeta3 - 12.0 * log((eta + 1.0i * (1.0 + yd)) / (2.0 * 1.0i + eta)) * zeta3 + (48.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 - 1.0i * xd, 2.0 - 1.0i * eta) * zeta3 + (48.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 + 1.0i * xd, 2.0 - 1.0i * eta) * zeta3 + (24.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 - yd, 2.0 - 1.0i * eta) * zeta3 + (24.0 * 1.0i) * M_PI * my_sign(real(eta)) * T(1.0, 1.0 + yd, 2.0 - 1.0i * eta) * zeta3;

            return w4Part1 + w4Part2 + w4Part3 + w4Part4 + w4Part5 + w4Part6 + w4HPLs;
        }
    }
}
