/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2023 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/polylog.hh>
#include <eos/maths/multiplepolylog-li22.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>
#include <eos/utils/stringify.hh>

#include <cmath>
#include <complex>

namespace eos
{
	using std::complex;
	using std::log;
	using std::real;
	using std::imag;

	namespace agv_2019a
	{
        complex<double> f29d_part6(const CharmLoopsParameters & clp)
        {
            const complex<double> xd = clp.xd;
            const complex<double> yd = clp.yd;

            const complex<double> ydinv = 1.0 / yd;

            const double imydinv = imag(ydinv);

            // Powers of xd and yd

            const complex<double> xd2 = power_of<2>(xd);
            const complex<double> xd4 = power_of<4>(xd);

            // weights appearing in the GPLs [AGV:2019A] p. 34

            const complex<double> w4 = (2.0 * xd) / power_of<2>(1.0 - xd);
            const complex<double> w5 = (2.0 * xd) / power_of<2>(1.0 + xd);
            const complex<double> w7 = (8.0 * xd2) / (1.0 - 6.0 * xd2 + xd4);

            const complex<double> w4inv = 1.0 / w4;
            const complex<double> w5inv = 1.0 / w5;
            const complex<double> w7inv = 1.0 / w7;

            // Logs of xd and yd

            const complex<double> ln1pyd = log(1.0 + yd);
            const complex<double> ln1myd = log(1.0 - yd);

            const complex<double> result = (trilog((2.0 * yd) / (-1.0 + yd)) * (16.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0 + 1.0i * xd) - 16.0 * log(xd)) + trilog((2.0 * yd) / (1.0 + yd)) * (16.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0 + 1.0i * xd) - 16.0 * log(xd))
                + trilog(1.0 + w4inv) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog(1.0 / (1.0 - w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                + trilog(-w4inv) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog(w4inv) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                + trilog((-1.0 + w4) / w4) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog(1.0 / (1.0 + w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                + trilog(1.0 + w5inv) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog(1.0 / (1.0 - w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                + trilog(-w5inv) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog(w5inv) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                + trilog((-1.0 + w5) / w5) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog(1.0 / (1.0 + w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                + trilog((1.0 - yd) / 2.0) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog((1.0 + yd) / 2.0) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                + trilog((-1.0 + w7) / (2.0 * w7)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 - w7) / (1.0 + w7)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog((1.0 + w7) / (1.0 - w7)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + w7) / (2.0 * w7)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog((-1.0 / 2.0) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 / 2.0) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) * 
                (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((w4 * (-1.0 + yd)) / ((1.0 + w4) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog((w5 * (-1.0 + yd)) / ((-1.0 + w5) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog(((1.0 + w4) * (-1.0 + yd)) / ((-1.0 + w4) * (1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(((-1.0 + w5) * (-1.0 + yd)) / ((1.0 + w5) * (1.0 + yd))) * 
                (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(((1.0 + w4) * (1.0 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) * 
                (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(((-1.0 + w5) * (1.0 + yd)) / ((1.0 + w5) * (-1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((w5 * (1.0 + yd)) / ((-1.0 + w5) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog((-1.0 / 2.0) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) * 
                (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 / 2.0) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((w4 + w4 * yd) / (yd + w4 * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                + trilog((1.0 - yd) / (1.0 + w7)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((-1.0 + w7) / (w7 - yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                + trilog((1.0 + w7) / (w7 - yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((-1.0 + yd) / (-1.0 + w7)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                + trilog((w7 * (-1.0 + yd)) / ((-1.0 + w7) * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((1.0 + yd) / (1.0 - w7)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                + trilog((1.0 + yd) / (1.0 + w7)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog(-((1.0 + yd) / (w7 - yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                + trilog((w7 * (1.0 + yd)) / ((-1.0 + w7) * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((-1.0 + yd) / (-w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                + trilog((-1.0 + w7) / (w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((1.0 + w7) / (w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                + trilog((-1.0 + yd) / (w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((1.0 + yd) / (w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                + trilog((-1.0 / 2.0) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((w7 * (-1.0 + yd)) / ((1.0 + w7) * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd))
                + trilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog(((1.0 + w7) * (-1.0 + yd)) / ((-1.0 + w7) * (1.0 + yd))) * 
                (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd))
                + trilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog(((1.0 + w7) * (1.0 + yd)) / ((-1.0 + w7) * (-1.0 + yd))) * 
                (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((-1.0 / 2.0) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd))
                + trilog(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((w7 + w7 * yd) / (yd + w7 * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd))
                + trilog(1.0 + w7inv) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(1.0 / (1.0 - w7)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog(-w7inv) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(w7inv) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + w7) / w7) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(1.0 / (1.0 + w7)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((1.0 - yd) / (1.0 + w4)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 - yd) / (1.0 + w5)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + w4) / (w4 - yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 + w4) / (w4 - yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + w5) / (w5 - yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 + w5) / (w5 - yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + yd) / (-1.0 + w4)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-1.0 + yd) / (-1.0 + w5)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((w4 * (-1.0 + yd)) / ((-1.0 + w4) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((w5 * (-1.0 + yd)) / ((1.0 + w5) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((1.0 + yd) / (1.0 - w4)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 + yd) / (1.0 + w4)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((1.0 + yd) / (1.0 - w5)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 + yd) / (1.0 + w5)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog(-((1.0 + yd) / (w4 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(-((1.0 + yd) / (w5 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((w4 * (1.0 + yd)) / ((-1.0 + w4) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-1.0 + yd) / (-w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + w4) / (w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 + w4) / (w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + yd) / (w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 + yd) / (w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + yd) / (-w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-1.0 + w5) / (w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((1.0 + w5) / (w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-1.0 + yd) / (w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((1.0 + yd) / (w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((w5 + w5 * yd) / (yd + w5 * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                + trilog((-1.0 + w4) / (2.0 * w4)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog((1.0 - w4) / (1.0 + w4)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                + trilog((1.0 + w4) / (1.0 - w4)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog((1.0 + w4) / (2.0 * w4)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                + trilog((-1.0 + w5) / (2.0 * w5)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog((1.0 - w5) / (1.0 + w5)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                + trilog((1.0 + w5) / (1.0 - w5)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog((1.0 + w5) / (2.0 * w5)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                + trilog(1.0 - yd) * (-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd)) + trilog(-yd) * (-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd))
                + trilog(yd) * (-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd)) + trilog(yd / (-1.0 + yd)) * (-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd))
                + trilog(yd / (1.0 + yd)) * (-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd)) + trilog(1.0 + yd) * (-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd))
                + li3half * (-32.0 * log(1.0 - 1.0i * xd) - 32.0 * log(1.0 + 1.0i * xd) + 32.0 * log(xd)) + dilog((-1.0 + w4) / (2.0 * w4)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog((1.0 - w4) / (1.0 + w4)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog((1.0 + w4) / (1.0 - w4)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog((1.0 + w4) / (2.0 * w4)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog((-1.0 + w5) / (2.0 * w5)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog((1.0 - w5) / (1.0 + w5)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog((1.0 + w5) / (1.0 - w5)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog((1.0 + w5) / (2.0 * w5)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                + dilog(-w7inv) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd) - 4.0 * lnhalf * log(xd)) + dilog(w7inv) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd)
                    - 4.0 * lnhalf * log(xd)) + dilog((-1.0 + w7) / (2.0 * w7)) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd))
                + dilog((1.0 - w7) / (1.0 + w7)) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd))
                + dilog((1.0 + w7) / (1.0 - w7)) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd))
                + dilog((1.0 + w7) / (2.0 * w7)) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd))
                + dilog(-w4inv) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog(w4inv) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd)
                    + 8.0 * lnhalf * log(xd)) + dilog(-w5inv) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd))
                + dilog(w5inv) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog((-1.0 + yd) / (-w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * 
                log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                + dilog((-1.0 + yd) / (-w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                + dilog((-1.0 / 2.0) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) * 
                (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((1.0 + w7) * (-1.0 + yd)) / ((-1.0 + w7) * (1.0 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0)
                + dilog((-1.0 / 2.0) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0)
                + dilog((-1.0 + yd) / (-w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * 
                log((1.0 - yd) / 2.0) + dilog((-1.0 / 2.0) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                + dilog((-1.0 / 2.0) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) * 
                (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((1.0 + w4) * (-1.0 + yd)) / ((-1.0 + w4) * (1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                + dilog(((-1.0 + w5) * (-1.0 + yd)) / ((1.0 + w5) * (1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd))) * 
                (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 / 2.0) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                + dilog((-1.0 / 2.0) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(1.0 - yd) * (16.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0 + 1.0i * xd) - 16.0 * log(xd)) * 
                ln1myd + dilog((w4 * (-1.0 + yd)) / ((-1.0 + w4) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + dilog((w5 * (-1.0 + yd)) / ((1.0 + w5) * yd)) * 
                (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + dilog((w7 * (-1.0 + yd)) / ((1.0 + w7) * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd
                + dilog((w7 * (-1.0 + yd)) / ((-1.0 + w7) * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd + dilog((w4 * (-1.0 + yd)) / ((1.0 + w4) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * 
                ln1myd + dilog((w5 * (-1.0 + yd)) / ((-1.0 + w5) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                + dilog((-1.0 + yd) / (-1.0 + w7)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd)
                + dilog((1.0 - yd) / (1.0 + w5)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd)
                + dilog((-1.0 + yd) / (-1.0 + w4)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd)
                + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w4 - yd)) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w4 - yd)))
                + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w4 - yd)) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w4 - yd)))
                + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w5 - yd)) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w5 - yd)))
                + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w5 - yd)) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w5 - yd)))
                + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(-2.0 / (w7 - yd)) + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(-2.0 / (w7 - yd)))
                + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(2.0 / (w7 - yd)) + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(2.0 / (w7 - yd)))
                + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd))) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * 
                power_of<3>(log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd)))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log((-2.0 * (1.0 + w4)) / ((-1.0 + w4) * (-1.0 + yd)))
                + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log((-2.0 * (1.0 + w4)) / ((-1.0 + w4) * (-1.0 + yd)))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * 
                log((-2.0 * (-1.0 + w5)) / ((1.0 + w5) * (-1.0 + yd))) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log((-2.0 * (-1.0 + w5)) / ((1.0 + w5) * (-1.0 + yd))))
                + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd))) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * 
                power_of<3>(log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd)))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd)))
                + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd)))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * 
                log((-2.0 * (1.0 + w7)) / ((-1.0 + w7) * (-1.0 + yd))) + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log((-2.0 * (1.0 + w7)) / ((-1.0 + w7) * (-1.0 + yd))))
                + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(w4 / ((-1.0 + w4) * yd)) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(w4 / ((-1.0 + w4) * yd)))
                + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(w5 / ((-1.0 + w5) * yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(w5 / ((-1.0 + w5) * yd)))
                + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(w7 / ((-1.0 + w7) * yd)) + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(w7 / ((-1.0 + w7) * yd)))
                + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd))) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * 
                power_of<3>(log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd)))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log((2.0 * (1.0 + w4)) / ((-1.0 + w4) * (1.0 + yd)))
                + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log((2.0 * (1.0 + w4)) / ((-1.0 + w4) * (1.0 + yd)))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * 
                log((2.0 * (-1.0 + w5)) / ((1.0 + w5) * (1.0 + yd))) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log((2.0 * (-1.0 + w5)) / ((1.0 + w5) * (1.0 + yd))))
                + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd))) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * 
                power_of<3>(log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd)))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd)))
                + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd)))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * 
                log((2.0 * (1.0 + w7)) / ((-1.0 + w7) * (1.0 + yd))) + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log((2.0 * (1.0 + w7)) / ((-1.0 + w7) * (1.0 + yd))))
                + dilog(-((1.0 + yd) / (w4 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(-((1.0 + yd) / (w5 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * 
                log((1.0 + yd) / 2.0) + dilog((1.0 + yd) / (w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog((1.0 + yd) / (w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * 
                log((1.0 + yd) / 2.0) + dilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0)
                + dilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w7) * (1.0 + yd)) / ((-1.0 + w7) * (-1.0 + yd))) * 
                (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0)
                + dilog(-((1.0 + yd) / (w7 - yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog((1.0 + yd) / (w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * 
                log((1.0 + yd) / 2.0) + dilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                + dilog(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd))) * 
                (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w4) * (1.0 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                + dilog(((-1.0 + w5) * (1.0 + yd)) / ((1.0 + w5) * (-1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) * 
                (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                + dilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                + dilog((1.0 + yd) / (1.0 + w4)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0))
                + dilog((1.0 + yd) / (1.0 - w5)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0))
                + dilog((1.0 + yd) / (1.0 + w7)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + yd) / 2.0))
                + dilog(1.0 + yd) * (16.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0 + 1.0i * xd) - 16.0 * log(xd)) * ln1pyd + dilog((w4 * (1.0 + yd)) / ((-1.0 + w4) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd
                + dilog((w5 + w5 * yd) / (yd + w5 * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd + dilog((w7 + w7 * yd) / (yd + w7 * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * 
                ln1pyd + dilog((w7 * (1.0 + yd)) / ((-1.0 + w7) * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd + dilog((w5 * (1.0 + yd)) / ((-1.0 + w5) * yd)) * 
                (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd + dilog((w4 + w4 * yd) / (yd + w4 * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd
                + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 - yd) / (1.0 + w4)) * power_of<2>(ln1pyd) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) * power_of<2>(ln1pyd)
                + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 - yd) / (1.0 + w7)) * power_of<2>(ln1pyd) + dilog(yd) * ((6.0 * log(1.0 - 1.0i * xd) + 6.0 * log(1.0 + 1.0i * xd) - 6.0 * log(xd)) * ln1myd
                    + (10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd)) * ln1pyd) + dilog(-yd) * ((10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd)) * ln1myd
                    + (6.0 * log(1.0 - 1.0i * xd) + 6.0 * log(1.0 + 1.0i * xd) - 6.0 * log(xd)) * ln1pyd) + dilog((1.0 - yd) / (1.0 + w4)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog((-1.0 + yd) / (-1.0 + w5)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog(1.0 / (1.0 - w4)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd)
                + dilog(1.0 / (1.0 + w5)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog(-2.0 / (-1.0 + w7)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1pyd) + dilog(2.0 / (1.0 + w7)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1pyd) + dilog(1.0 / (1.0 + w7)) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd) - 4.0 * lnhalf * log(xd)
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1pyd)
                + dilog((1.0 + yd) / (1.0 - w7)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1pyd)
                + dilog((1.0 - yd) / (1.0 + w7)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd)
                + dilog(1.0 / (1.0 - w7)) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd) - 4.0 * lnhalf * log(xd) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd) + dilog(-2.0 / (-1.0 + w4)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog(2.0 / (1.0 + w4)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog(-2.0 / (-1.0 + w5)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog(2.0 / (1.0 + w5)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog(1.0 / (1.0 + w4)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd)
                + dilog(1.0 / (1.0 - w5)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog((1.0 - yd) / 2.0) * ((-12.0 * log(1.0 - 1.0i * xd) - 12.0 * log(1.0 + 1.0i * xd) + 12.0 * log(xd)) * ln1myd
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog((1.0 + yd) / (1.0 - w4)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog((1.0 + yd) / (1.0 + w5)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog((1.0 + yd) / 2.0) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                    + (-12.0 * log(1.0 - 1.0i * xd) - 12.0 * log(1.0 + 1.0i * xd) + 12.0 * log(xd)) * ln1pyd) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w4 + yd))
                + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w4 + yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w4 + yd))
                + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w4 + yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w5 + yd))
                + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w5 + yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w5 + yd))
                + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w5 + yd))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(-2.0 / (w7 + yd))
                + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(-2.0 / (w7 + yd))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(2.0 / (w7 + yd))
                + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(2.0 / (w7 + yd))) + log((1.0 - yd) / 2.0) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) * ln1pyd
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) * ln1pyd + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) * ln1pyd
                    + (-6.0 * log(1.0 - 1.0i * xd) - 6.0 * log(1.0 + 1.0i * xd) + 6.0 * log(xd)) * power_of<2>(ln1pyd) + ln1pyd * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 + yd) / (1.0 + w4))
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 + yd) / (1.0 + w7))))
                + power_of<2>(log((1.0 + yd) / 2.0)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + w4) / (w4 - yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 - yd) / (1.0 + w4))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + w5) / (w5 - yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 - yd) / (1.0 + w5))
                    + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((1.0 + w7) / (w7 - yd)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 - yd) / (1.0 + w7))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-2.0 * (w4 - yd)) / ((1.0 + w4) * (-1.0 + yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-2.0 * (w5 - yd)) / ((1.0 + w5) * (-1.0 + yd)))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((-2.0 * (w7 - yd)) / ((1.0 + w7) * (-1.0 + yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-1.0 / 2.0) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd)))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-1.0 / 2.0) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * 
                    log((-1.0 / 2.0) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-1.0 + w4) / (w4 + yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * 
                    log((-1.0 / 2.0) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * 
                    log((-2.0 * (w4 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-1.0 + w5) / (w5 + yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * 
                    log((-1.0 / 2.0) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * 
                    log((-2.0 * (w5 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((-1.0 + w7) / (w7 + yd)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * 
                    log((-1.0 / 2.0) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 + yd) / (-1.0 + w7)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * 
                    log((-2.0 * (w7 + yd)) / ((-1.0 + w7) * (-1.0 + yd)))) + power_of<2>(log((1.0 - yd) / 2.0)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-1.0 + w4) / (w4 - yd))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-1.0 + w5) / (w5 - yd))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((-1.0 + w7) / (w7 - yd))
                    + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 - yd) / (-1.0 + w7)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((2.0 * (w4 - yd)) / ((-1.0 + w4) * (1.0 + yd)))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((2.0 * (w5 - yd)) / ((-1.0 + w5) * (1.0 + yd))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((2.0 * (w7 - yd)) / ((-1.0 + w7) * (1.0 + yd)))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd)))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + w4) / (w4 + yd))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 + yd) / (1.0 + w4))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((2.0 * (w4 + yd)) / ((1.0 + w4) * (1.0 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + w5) / (w5 + yd))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 + yd) / (1.0 + w5))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((2.0 * (w5 + yd)) / ((1.0 + w5) * (1.0 + yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((1.0 + w7) / (w7 + yd))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 + yd) / (1.0 + w7))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((2.0 * (w7 + yd)) / ((1.0 + w7) * (1.0 + yd)))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(w4 / (yd - w4 * yd))
                + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(w4 / (yd - w4 * yd))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(-(w4 / (yd + w4 * yd)))
                + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(-(w4 / (yd + w4 * yd)))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(w4 / (yd + w4 * yd))
                + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(w4 / (yd + w4 * yd))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(w5 / (yd - w5 * yd))
                + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(w5 / (yd - w5 * yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(-(w5 / (yd + w5 * yd)))
                + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(-(w5 / (yd + w5 * yd)))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(w5 / (yd + w5 * yd))
                + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(w5 / (yd + w5 * yd))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(w7 / (yd - w7 * yd))
                + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(w7 / (yd - w7 * yd))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(-(w7 / (yd + w7 * yd)))
                + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(-(w7 / (yd + w7 * yd)))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(w7 / (yd + w7 * yd))
                + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(w7 / (yd + w7 * yd))) + power_of<2>(ln1pyd) * ((2.0 * ln2 + ln64 / 2.0) * log(1.0 - 1.0i * xd) + (2.0 * ln2 + ln64 / 2.0) * log(1.0 + 1.0i * xd)
                    + (-2.0 * ln2 - ln64 / 2.0) * log(xd) + (11.0 * log(1.0 - 1.0i * xd) + 11.0 * log(1.0 + 1.0i * xd) - 11.0 * log(xd)) * log(-yd) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((1.0 + yd) / 8.0)
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 + yd) / (-1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 + yd) / (yd - w4 * yd))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log(-((w4 - yd) / (yd + w4 * yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 + yd) / (yd - w5 * yd))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log(-((w5 - yd) / (yd + w5 * yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 + yd) / (yd - w7 * yd))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log(-((w7 - yd) / (yd + w7 * yd)))) + power_of<2>(ln1myd) * ((2.0 * ln2 + ln64 / 2.0) * log(1.0 - 1.0i * xd) + (2.0 * ln2 + ln64 / 2.0) * log(1.0 + 1.0i * xd)
                    + (-2.0 * ln2 - ln64 / 2.0) * log(xd) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((1.0 - yd) / 8.0) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 - yd) / (-1.0 + w7))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 - yd) / ((-1.0 + w4) * yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 - yd) / ((-1.0 + w5) * yd))
                    + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 - yd) / ((-1.0 + w7) * yd)) + (11.0 * log(1.0 - 1.0i * xd) + 11.0 * log(1.0 + 1.0i * xd) - 11.0 * log(xd)) * log(yd)
                    + (-6.0 * log(1.0 - 1.0i * xd) - 6.0 * log(1.0 + 1.0i * xd) + 6.0 * log(xd)) * log((1.0 + yd) / 2.0) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 + yd) / (1.0 + w4))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 + yd) / (1.0 + w7))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 + yd) / (yd + w4 * yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 + yd) / (yd + w5 * yd))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 + yd) / (yd + w7 * yd))) + power_of<2>(log((-w4 + yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                + power_of<2>(log((w4 + yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log((-w5 + yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                + power_of<2>(log((w5 + yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log((-w7 + yd) / 2.0)) * ((-2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    - (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                + power_of<2>(log((w7 + yd) / 2.0)) * ((-2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    + (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log(((-1.0 + w4) * (-1.0 + yd)) / (2.0 * (1.0 + w4)))) * 
                ((-4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    + (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log(((-1.0 + w5) * (-1.0 + yd)) / (2.0 * (1.0 + w5)))) * 
                ((-4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    + (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log(((-1.0 + w7) * (-1.0 + yd)) / (2.0 * (1.0 + w7)))) * 
                ((2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    - (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log(((1.0 + w4) * (-1.0 + yd)) / (2.0 * (-1.0 + w4)))) * 
                ((-4.0 * 1.0i) * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    + (4.0 * 1.0i) * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log(((1.0 + w5) * (-1.0 + yd)) / (2.0 * (-1.0 + w5)))) * 
                ((-4.0 * 1.0i) * M_PI * H1((-1.0 + w5) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((-1.0 + w5) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    + (4.0 * 1.0i) * M_PI * H1((-1.0 + w5) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log(((1.0 + w7) * (-1.0 + yd)) / (2.0 * (-1.0 + w7)))) * 
                ((2.0 * 1.0i) * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (2.0 * 1.0i) * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                    - (2.0 * 1.0i) * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log(((-1.0 + w4) * yd) / w4)) * ((4.0 * 1.0i) * M_PI * H1(-(w4 / yd), -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv)
                    + (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) - (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), -ydinv) * log(xd) * my_sign(-imydinv))
                + power_of<2>(log((1.0 + w4inv) * yd)) * ((-4.0 * 1.0i) * M_PI * H1(w4 / yd, -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) - (4.0 * 1.0i) * M_PI * H1(w4 / yd, -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv)
                    + (4.0 * 1.0i) * M_PI * H1(w4 / yd, -ydinv) * log(xd) * my_sign(-imydinv)) + power_of<2>(log(((-1.0 + w5) * yd) / w5)) * ((-4.0 * 1.0i) * M_PI * H1(-(w5 / yd), -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv)
                    - (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) + (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), -ydinv) * log(xd) * my_sign(-imydinv))
                + power_of<2>(log((1.0 + w5inv) * yd)) * ((4.0 * 1.0i) * M_PI * H1(w5 / yd, -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) + (4.0 * 1.0i) * M_PI * H1(w5 / yd, -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv)
                    - (4.0 * 1.0i) * M_PI * H1(w5 / yd, -ydinv) * log(xd) * my_sign(-imydinv)) + power_of<2>(log(((-1.0 + w7) * yd) / w7)) * ((-2.0 * 1.0i) * M_PI * H1(-(w7 / yd), -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv)
                    - (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) + (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), -ydinv) * log(xd) * my_sign(-imydinv))
                + power_of<2>(log((1.0 + w7inv) * yd)) * ((2.0 * 1.0i) * M_PI * H1(w7 / yd, -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) + (2.0 * 1.0i) * M_PI * H1(w7 / yd, -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv)
                    - (2.0 * 1.0i) * M_PI * H1(w7 / yd, -ydinv) * log(xd) * my_sign(-imydinv)) + power_of<2>(log(-(((1.0 + w4) * yd) / w4))) * ((-4.0 * 1.0i) * M_PI * H1(-(w4 / yd), ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv)
                    - (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) + (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), ydinv) * log(xd) * my_sign(imydinv))
                + power_of<2>(log((-1.0 + w4inv) * yd)) * ((4.0 * 1.0i) * M_PI * H1(w4 / yd, ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) + (4.0 * 1.0i) * M_PI * H1(w4 / yd, ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv)
                    - (4.0 * 1.0i) * M_PI * H1(w4 / yd, ydinv) * log(xd) * my_sign(imydinv)) + power_of<2>(log(-(((1.0 + w5) * yd) / w5))) * ((4.0 * 1.0i) * M_PI * H1(-(w5 / yd), ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv)
                    + (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) - (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), ydinv) * log(xd) * my_sign(imydinv))
                + power_of<2>(log((-1.0 + w5inv) * yd)) * ((-4.0 * 1.0i) * M_PI * H1(w5 / yd, ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) - (4.0 * 1.0i) * M_PI * H1(w5 / yd, ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv)
                    + (4.0 * 1.0i) * M_PI * H1(w5 / yd, ydinv) * log(xd) * my_sign(imydinv)) + power_of<2>(log(-(((1.0 + w7) * yd) / w7))) * ((2.0 * 1.0i) * M_PI * H1(-(w7 / yd), ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv)
                    + (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) - (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), ydinv) * log(xd) * my_sign(imydinv))
                + power_of<2>(log((-1.0 + w7inv) * yd)) * ((-2.0 * 1.0i) * M_PI * H1(w7 / yd, ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) - (2.0 * 1.0i) * M_PI * H1(w7 / yd, ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv)
                    + (2.0 * 1.0i) * M_PI * H1(w7 / yd, ydinv) * log(xd) * my_sign(imydinv)) + power_of<2>(log((-w4 - yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                + power_of<2>(log((w4 - yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-w5 - yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                + power_of<2>(log((w5 - yd) / 2.0)) * ((4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-w7 - yd) / 2.0)) * ((-2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    - (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                + power_of<2>(log((w7 - yd) / 2.0)) * ((-2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    + (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log(((1.0 + w4) * (1.0 + yd)) / (2.0 - 2.0 * w4))) * 
                ((-4.0 * 1.0i) * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    + (4.0 * 1.0i) * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-1.0 / 2.0) * (((-1.0 + w4) * (1.0 + yd)) / (1.0 + w4)))) * 
                ((-4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    + (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log(((1.0 + w5) * (1.0 + yd)) / (2.0 - 2.0 * w5))) * 
                ((-4.0 * 1.0i) * M_PI * H1((1.0 - w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 - w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    + (4.0 * 1.0i) * M_PI * H1((1.0 - w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-1.0 / 2.0) * (((-1.0 + w5) * (1.0 + yd)) / (1.0 + w5)))) * 
                ((-4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    + (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log(((1.0 + w7) * (1.0 + yd)) / (2.0 - 2.0 * w7))) * 
                ((2.0 * 1.0i) * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    - (2.0 * 1.0i) * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-1.0 / 2.0) * (((-1.0 + w7) * (1.0 + yd)) / (1.0 + w7)))) * 
                ((2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                    - (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((w4 - yd) / (1.0 + w4))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 - yd) / (1.0 + w4))
                    + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 - yd) / (1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 + w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 - yd) / (1.0 + w4)))
                + power_of<2>(log((w5 - yd) / (1.0 + w5))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * 
                    T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)))
                + power_of<2>(log((w7 - yd) / (1.0 + w7))) * ((-2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 - yd) / (1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w7))) * 
                    T(1.0, (1.0 - yd) / 2.0, (w7 - yd) / (1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 + w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 - yd) / (1.0 + w7)))
                + power_of<2>(log((w4 + yd) / (-1.0 + w4))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * 
                    T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)))
                + power_of<2>(log((w5 + yd) / (-1.0 + w5))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 + yd) / (-1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w5))) * 
                    T(1.0, (1.0 - yd) / 2.0, (w5 + yd) / (-1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 - w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 + yd) / (-1.0 + w5)))
                + power_of<2>(log((w7 + yd) / (-1.0 + w7))) * ((-2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * 
                    T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7)))
                + power_of<2>(log((w4 - yd) / (-1.0 + w4))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4)) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * 
                    T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4)))
                + power_of<2>(log((w5 - yd) / (-1.0 + w5))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 - yd) / (-1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w5))) * 
                    T(1.0, (1.0 + yd) / 2.0, (w5 - yd) / (-1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 - yd) / (-1.0 + w5)))
                + power_of<2>(log((w7 - yd) / (-1.0 + w7))) * ((-2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * 
                    T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7)))
                + power_of<2>(log((w4 + yd) / (1.0 + w4))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 + yd) / (1.0 + w4)) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w4))) * 
                    T(1.0, (1.0 + yd) / 2.0, (w4 + yd) / (1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 - yd) / (1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 + yd) / (1.0 + w4)))
                + power_of<2>(log((w5 + yd) / (1.0 + w5))) * ((4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * 
                    T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)))
                + power_of<2>(log((w7 + yd) / (1.0 + w7))) * ((-2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 + yd) / (1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w7))) * 
                    T(1.0, (1.0 + yd) / 2.0, (w7 + yd) / (1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 - yd) / (1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 + yd) / (1.0 + w7)))
                + dilog((1.0 - w4) / 2.0) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4))
                    + log(1.0 - 1.0i * xd) * (-8.0 * log(w4 / (-1.0 + w4)) - (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4))
                    + log(1.0 + 1.0i * xd) * (-8.0 * log(w4 / (-1.0 + w4)) - (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4))
                    + log(xd) * (8.0 * log(w4 / (-1.0 + w4)) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4)))
                + dilog((1.0 + w4) / 2.0) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 - yd) / (1.0 + w4)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 + yd) / (1.0 + w4))
                    + log(1.0 - 1.0i * xd) * (-8.0 * log(w4 / (1.0 + w4)) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4))
                    + log(1.0 + 1.0i * xd) * (-8.0 * log(w4 / (1.0 + w4)) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4))
                    + log(xd) * (8.0 * log(w4 / (1.0 + w4)) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4)))
                + dilog((1.0 - w5) / 2.0) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5))
                    + log(1.0 - 1.0i * xd) * (-8.0 * log(w5 / (-1.0 + w5)) - (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) - (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5))
                    + log(1.0 + 1.0i * xd) * (-8.0 * log(w5 / (-1.0 + w5)) - (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) - (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5))
                    + log(xd) * (8.0 * log(w5 / (-1.0 + w5)) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)))
                + dilog((1.0 + w5) / 2.0) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5))
                    + log(1.0 - 1.0i * xd) * (-8.0 * log(w5 / (1.0 + w5)) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5))
                    + log(1.0 + 1.0i * xd) * (-8.0 * log(w5 / (1.0 + w5)) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5))
                    + log(xd) * (8.0 * log(w5 / (1.0 + w5)) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5)))
                + ln1myd * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) * ln1pyd + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) * 
                    ln1pyd + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) * ln1pyd + ln1pyd * (8.0 * ln2 * log(1.0 - 1.0i * xd) + 8.0 * ln2 * log(1.0 + 1.0i * xd)
                    - 8.0 * ln2 * log(xd) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7))) + log((1.0 + yd) / 2.0) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 - yd) / (1.0 + w4))
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 - yd) / (1.0 + w7))
                    + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5))
                    + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * 
                    ((2.0 * pisqu) / 3.0 - 10.0 * ln2squ + (8.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 - w5)
                    - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w7) + (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0)
                    + (8.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0)
                    + (8.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0) - (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0)
                    - (4.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0) - (8.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4)
                    - (8.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5) + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7)))
                + dilog((1.0 - w7) / 2.0) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7))
                    + log(xd) * (-4.0 * log(w7 / (-1.0 + w7)) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7))
                    + log(1.0 - 1.0i * xd) * (4.0 * log(w7 / (-1.0 + w7)) + (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) + (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7))
                    + log(1.0 + 1.0i * xd) * (4.0 * log(w7 / (-1.0 + w7)) + (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) + (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7)))
                + dilog((1.0 + w7) / 2.0) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 - yd) / (1.0 + w7)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 + yd) / (1.0 + w7))
                    + log(xd) * (-4.0 * log(w7 / (1.0 + w7)) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7))
                    + log(1.0 - 1.0i * xd) * (4.0 * log(w7 / (1.0 + w7)) + (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) + (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7))
                    + log(1.0 + 1.0i * xd) * (4.0 * log(w7 / (1.0 + w7)) + (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) + (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7)))
                + li2half * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 - yd) / (1.0 + w4))
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 - yd) / (1.0 + w7))
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 + yd) / (1.0 + w4))
                    + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5))
                    + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 + yd) / (1.0 + w7))
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * (8.0 * log(w4 / (-1.0 + w4)) + 8.0 * log(w4 / (1.0 + w4)) + 8.0 * log(w5 / (-1.0 + w5)) + 8.0 * log(w5 / (1.0 + w5)) - 4.0 * log(w7 / (-1.0 + w7)) - 4.0 * log(w7 / (1.0 + w7))
                    + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5)
                    - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4)
                    + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7)))
                + ln1pyd * ((log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * ((2.0 * pisqu) / 3.0 - 10.0 * ln2squ + (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w4) / 2.0)
                    + (8.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0)
                    + (8.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w5) / 2.0) - (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w7) / 2.0)
                    - (4.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0) - (8.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 - w4)
                    - (8.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 + w5) + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 - w7) + (8.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 + w4)
                    + (8.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 - w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 + w7)))
                + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * ((2.0 * pisqu * ln2) / 3.0 + (8.0 * power_of<3>(ln2)) / 3.0 - pisqu * ln4 - 4.0 * power_of<2>(lnhalf) * log(1.0 + w4inv) + (4.0 * pisqu * log(-2.0 / w4)) / 3.0 + (4.0 * power_of<3>(log(-2.0 / w4))) / 3.0
                    + (4.0 * pisqu * log(2.0 / w4)) / 3.0 + (4.0 * power_of<3>(log(2.0 / w4))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log((-1.0 + w4) / (2.0 * w4)) - 4.0 * power_of<2>(lnhalf) * log((-1.0 + w4) / w4) - 4.0 * power_of<2>(lnhalf) * log(w4 / (-1.0 + w4)) + 4.0 * power_of<2>(lnhalf) * log((2.0 * w4) / (-1.0 + w4))
                    - (4.0 * pisqu * log((2.0 * (-1.0 + w4)) / (1.0 + w4))) / 3.0 - (4.0 * power_of<3>(log((2.0 * (-1.0 + w4)) / (1.0 + w4)))) / 3.0 - 4.0 * power_of<2>(lnhalf) * log(w4 / (1.0 + w4)) + 4.0 * power_of<2>(lnhalf) * log((2.0 * w4) / (1.0 + w4)) - (4.0 * pisqu * log((2.0 * (1.0 + w4)) / (-1.0 + w4))) / 3.0
                    - (4.0 * power_of<3>(log((2.0 * (1.0 + w4)) / (-1.0 + w4)))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log((1.0 + w4) / (2.0 * w4)) - 4.0 * power_of<2>(lnhalf) * log(1.0 + w5inv) + (4.0 * pisqu * log(-2.0 / w5)) / 3.0 + (4.0 * power_of<3>(log(-2.0 / w5))) / 3.0 + (4.0 * pisqu * log(2.0 / w5)) / 3.0
                    + (4.0 * power_of<3>(log(2.0 / w5))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log((-1.0 + w5) / (2.0 * w5)) - 4.0 * power_of<2>(lnhalf) * log((-1.0 + w5) / w5) - 4.0 * power_of<2>(lnhalf) * log(w5 / (-1.0 + w5)) + 4.0 * power_of<2>(lnhalf) * log((2.0 * w5) / (-1.0 + w5))
                    - (4.0 * pisqu * log((2.0 * (-1.0 + w5)) / (1.0 + w5))) / 3.0 - (4.0 * power_of<3>(log((2.0 * (-1.0 + w5)) / (1.0 + w5)))) / 3.0 - 4.0 * power_of<2>(lnhalf) * log(w5 / (1.0 + w5)) + 4.0 * power_of<2>(lnhalf) * log((2.0 * w5) / (1.0 + w5)) - (4.0 * pisqu * log((2.0 * (1.0 + w5)) / (-1.0 + w5))) / 3.0
                    - (4.0 * power_of<3>(log((2.0 * (1.0 + w5)) / (-1.0 + w5)))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log((1.0 + w5) / (2.0 * w5)) + 2.0 * power_of<2>(lnhalf) * log(1.0 + w7inv) - (2.0 * pisqu * log(-2.0 / w7)) / 3.0 - (2.0 * power_of<3>(log(-2.0 / w7))) / 3.0 - (2.0 * pisqu * log(2.0 / w7)) / 3.0
                    - (2.0 * power_of<3>(log(2.0 / w7))) / 3.0 - 2.0 * power_of<2>(lnhalf) * log((-1.0 + w7) / (2.0 * w7)) + 2.0 * power_of<2>(lnhalf) * log((-1.0 + w7) / w7) + 2.0 * power_of<2>(lnhalf) * log(w7 / (-1.0 + w7)) - 2.0 * power_of<2>(lnhalf) * log((2.0 * w7) / (-1.0 + w7))
                    + (2.0 * pisqu * log((2.0 * (-1.0 + w7)) / (1.0 + w7))) / 3.0 + (2.0 * power_of<3>(log((2.0 * (-1.0 + w7)) / (1.0 + w7)))) / 3.0 + 2.0 * power_of<2>(lnhalf) * log(w7 / (1.0 + w7)) - 2.0 * power_of<2>(lnhalf) * log((2.0 * w7) / (1.0 + w7)) + (2.0 * pisqu * log((2.0 * (1.0 + w7)) / (-1.0 + w7))) / 3.0
                    + (2.0 * power_of<3>(log((2.0 * (1.0 + w7)) / (-1.0 + w7)))) / 3.0 - 2.0 * power_of<2>(lnhalf) * log((1.0 + w7) / (2.0 * w7)) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w4) / 2.0)) * my_sign(imag(w4) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w4) / 2.0)
                    - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w4)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0)
                    - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w5) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w5)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w5) / 2.0) + (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w7) / 2.0)) * my_sign(imag(w7) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w7) / 2.0)
                    + (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w7)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w4)) * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4)
                    - (8.0 * 1.0i) * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * log(1.0 + w4) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4)
                    + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) - (8.0 * 1.0i) * M_PI * log(1.0 - w5) * log((1.0 + w5) / 2.0) * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5)
                    - (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w5)) * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) - (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w7)) * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7)
                    + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) + (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * log(1.0 + w7) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7)
                    - (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w4) / 2.0)) * my_sign(imag(w4) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w4)) * 
                    T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w5) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w5)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0)
                    + (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w7) / 2.0)) * my_sign(imag(w7) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0) + (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w7)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0)
                    + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w4)) * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * log(1.0 + w4) * my_sign(-imag(w4)) * 
                    T(1.0, 1.0 + yd, 1.0 + w4) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)
                    - (8.0 * 1.0i) * M_PI * log(1.0 - w5) * log((1.0 + w5) / 2.0) * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5) - (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5)
                    + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w5)) * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) - (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w7)) * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7) + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7)
                    + (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * log(1.0 + w7) * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7) - (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7) + 34.0 * zeta3));

            return result;
        }
    }
}
