/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2023 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/polylog.hh>
#include <eos/maths/multiplepolylog-li22.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>
#include <eos/utils/stringify.hh>

#include <cmath>
#include <complex>

namespace eos
{
    using std::complex;
    using std::log;
    using std::sqrt;
    using std::real;

    namespace agv_2019a
    {
        complex<double> f19b(const CharmLoopsParameters & clp)
        {
            // f19b = - f29b / (2 * N_c)
            return - f29b(clp) / 6.0;
        }

        complex<double> f29b(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xb = clp.xb;
            const complex<double> yb = clp.yb;

            // Powers of xd and yd

            const complex<double> xb2 = power_of<2>(xb);
            const complex<double> xb3 = power_of<3>(xb);
            const complex<double> xb4 = power_of<4>(xb);
            const complex<double> xb5 = power_of<5>(xb);
            const complex<double> xb6 = power_of<6>(xb);
            const complex<double> xb7 = power_of<7>(xb);
            const complex<double> xb8 = power_of<8>(xb);
            const complex<double> xb9 = power_of<9>(xb);
            const complex<double> xb10 = power_of<10>(xb);
            const complex<double> xb11 = power_of<11>(xb);
            const complex<double> xb12 = power_of<12>(xb);
            const complex<double> xb13 = power_of<13>(xb);
            const complex<double> xb14 = power_of<14>(xb);
            const complex<double> xb15 = power_of<15>(xb);
            const complex<double> xb16 = power_of<16>(xb);

            const complex<double> yb2 = power_of<2>(yb);
            const complex<double> yb3 = power_of<3>(yb);
            const complex<double> yb4 = power_of<4>(yb);
            const complex<double> yb5 = power_of<5>(yb);
            const complex<double> yb6 = power_of<6>(yb);
            const complex<double> yb7 = power_of<7>(yb);
            const complex<double> yb8 = power_of<8>(yb);
            const complex<double> yb9 = power_of<9>(yb);
            const complex<double> yb10 = power_of<10>(yb);
            const complex<double> yb11 = power_of<11>(yb);

            // weights appearing in the GPLs [AGV:2019A] p. 34

            const double wx3 = 2.0 + sqrt(3);
            const double wx4 = 2.0 - sqrt(3);

            const complex<double> w4 = (2.0 * xb) / power_of<2>(1.0 - xb);
            const complex<double> w5 = (2.0 * xb) / power_of<2>(1.0 + xb);
            const complex<double> w7 = (8.0 * xb2) / (1.0 - 6.0 * xb2 + xb4);

            const double wx3inv = wx4;
            const double wx4inv = wx3;

            const complex<double> w4inv = 1.0 / w4;
            const complex<double> w5inv = 1.0 / w5;
            const complex<double> w7inv = 1.0 / w7;

            const complex<double> tb = (2.0 * (-2.0 * xb2 + 2.0 * xb2 * yb + sqrt(2) * xb2 * (1.0 + yb) * sqrt((2.0 * xb4 - xb2 * yb + 2.0 * xb4 * yb - xb6 * yb + xb2 * yb2 + 4.0 * xb4 * yb2 + xb6 * yb2) / (xb4 * power_of<2>(1.0 + yb))))) / (-1.0 + 6.0 * xb2 - xb4 + yb + 2.0 * xb2 * yb + xb4 * yb);
            const complex<double> vb = (2.0 * (-2.0 * xb2 - 2.0 * xb2 * yb + sqrt(2) * xb2 * (1.0 - yb) * sqrt((2.0 * xb4 + xb2 * yb - 2.0 * xb4 * yb + xb6 * yb + xb2 * yb2 + 4.0 * xb4 * yb2 + xb6 * yb2) / (xb4 * power_of<2>(1.0 - yb))))) / (1.0 - 6.0 * xb2 + xb4 + yb + 2.0 * xb2 * yb + xb4 * yb);

            const complex<double> tb2 = power_of<2>(tb);
            const complex<double> vb2 = power_of<2>(vb);

            // Logs of yb

            const complex<double> ln1pyb = log(1.0 + yb);
            const complex<double> ln1myb = log(1.0 - yb);

            const complex<double> num1 = 16.0 - 48.0 * yb2;
            const complex<double> num2 = 4.0 * tb * xb2 * (16.0 * vb * xb2 * power_of<2>(1.0i + xb) * yb * (6.0 * 1.0i * xb * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) + 6.0 * 1.0i * xb9 * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8)
                    + xb6 * (-68.0 - 222.0 * yb2 + 1242.0 * yb4 + 6334.0 * yb6 + 4578.0 * yb8 - 3672.0 * yb10) - 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + 3.0 * xb10 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    - 8.0 * 1.0i * xb3 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10) - 8.0 * 1.0i * xb7 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10)
                    - xb2 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10) + xb8 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10)
                    + 2.0 * xb4 * (34.0 + 111.0 * yb2 - 621.0 * yb4 - 3167.0 * yb6 - 2289.0 * yb8 + 1836.0 * yb10) - 4.0 * 1.0i * xb5 * (-36.0 - 543.0 * yb2 - 1857.0 * yb4 - 4851.0 * yb6 + 3529.0 * yb8 + 7854.0 * yb10))
                  - 4.0 * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10) - 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    + 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1623.0 * yb3 - 366.0 * yb4 - 6384.0 * yb5 + 36.0 * yb6 - 27874.0 * yb7 + 2646.0 * yb8 + 6654.0 * yb9 + 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8
                      + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 1599.0 * yb3 + 6430.0 * yb4 + 11580.0 * yb5 - 952.0 * yb6 + 128218.0 * yb7 - 47070.0 * yb8 + 26004.0 * yb9 - 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)))
                + tb2 * ((-vb2) * power_of<2>(1.0i + xb - 1.0i * xb2 - xb3) * yb * (xb10 * yb2 * (22.0 - 12477.0 * yb2 + 12675.0 * yb4 + 1001.0 * yb6 - 1221.0 * yb8) + 3.0 * yb2 * (2.0 + 3.0 * yb2 - 5.0 * yb4 + 9.0 * yb6 - 9.0 * yb8)
                    + 6.0 * 1.0i * xb * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) + 6.0 * 1.0i * xb13 * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) + 3.0 * xb14 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 3.0 * xb2 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) + 3.0 * xb12 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) - 4.0 * 1.0i * xb3 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8)
                    - 4.0 * 1.0i * xb11 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8) + xb4 * yb2 * (-22.0 + 12477.0 * yb2 - 12675.0 * yb4 - 1001.0 * yb6 + 1221.0 * yb8)
                    - 2.0 * 1.0i * xb5 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8) - 2.0 * 1.0i * xb9 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8)
                    + xb6 * (896.0 + 6022.0 * yb2 + 12885.0 * yb4 - 6451.0 * yb6 - 44481.0 * yb8 - 34407.0 * yb10) + 8.0 * 1.0i * xb7 * (224.0 + 1020.0 * yb2 + 3339.0 * yb4 - 11161.0 * yb6 - 10415.0 * yb8 + 609.0 * yb10)
                    + xb8 * (-896.0 - 6022.0 * yb2 - 12885.0 * yb4 + 6451.0 * yb6 + 44481.0 * yb8 + 34407.0 * yb10)) - 4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8
                      - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 1599.0 * yb3 - 6430.0 * yb4 + 11580.0 * yb5 + 952.0 * yb6 + 128218.0 * yb7 + 47070.0 * yb8 + 26004.0 * yb9 + 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11))
                  + 4.0 * xb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8
                      + 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 478.0 * yb3 + 2928.0 * yb4 - 4461.0 * yb5 - 288.0 * yb6 - 65749.0 * yb7 - 21168.0 * yb8 - 15767.0 * yb9 - 14904.0 * yb10 + 53223.0 * yb11)))
                + 4.0 * xb2 * (-4.0 * xb2 * power_of<2>(1.0i + xb) * yb * (-1.0 + yb2) * (6.0 * 1.0i * xb * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) + 6.0 * 1.0i * xb9 * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) - 3.0 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6) + 3.0 * xb10 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6)
                    - 8.0 * 1.0i * xb3 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) - 8.0 * 1.0i * xb7 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) + 3.0 * xb8 * yb2 * (-22.0 - 5.0 * yb2 + 252.0 * yb4 + 207.0 * yb6)
                    + xb2 * (66.0 * yb2 + 15.0 * yb4 - 756.0 * yb6 - 621.0 * yb8) - 2.0 * xb6 * (16.0 + 212.0 * yb2 + 1247.0 * yb4 + 3222.0 * yb6 + 2871.0 * yb8) + xb4 * (32.0 + 424.0 * yb2 + 2494.0 * yb4 + 6444.0 * yb6 + 5742.0 * yb8)
                    - 4.0 * 1.0i * xb5 * (-16.0 + 210.0 * yb2 + 717.0 * yb4 + 7232.0 * yb6 + 6465.0 * yb8)) + 4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11) - 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    + 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1623.0 * yb3 + 366.0 * yb4 - 6384.0 * yb5 - 36.0 * yb6 - 27874.0 * yb7 - 2646.0 * yb8 + 6654.0 * yb9 - 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8
                      - 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 478.0 * yb3 - 2928.0 * yb4 - 4461.0 * yb5 + 288.0 * yb6 - 65749.0 * yb7 + 21168.0 * yb8 - 15767.0 * yb9 + 14904.0 * yb10 + 53223.0 * yb11)));
            const complex<double> num3 = -4.0 * tb * xb2 * (16.0 * vb * xb2 * power_of<2>(1.0i + xb) * yb * (6.0 * 1.0i * xb * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) + 6.0 * 1.0i * xb9 * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8)
                    + xb6 * (-68.0 - 222.0 * yb2 + 1242.0 * yb4 + 6334.0 * yb6 + 4578.0 * yb8 - 3672.0 * yb10) - 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + 3.0 * xb10 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    - 8.0 * 1.0i * xb3 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10) - 8.0 * 1.0i * xb7 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10)
                    - xb2 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10) + xb8 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10)
                    + 2.0 * xb4 * (34.0 + 111.0 * yb2 - 621.0 * yb4 - 3167.0 * yb6 - 2289.0 * yb8 + 1836.0 * yb10) - 4.0 * 1.0i * xb5 * (-36.0 - 543.0 * yb2 - 1857.0 * yb4 - 4851.0 * yb6 + 3529.0 * yb8 + 7854.0 * yb10))
                  - 4.0 * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10) - 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    + 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1623.0 * yb3 - 366.0 * yb4 - 6384.0 * yb5 + 36.0 * yb6 - 27874.0 * yb7 + 2646.0 * yb8 + 6654.0 * yb9 + 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8
                      + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 1599.0 * yb3 + 6430.0 * yb4 + 11580.0 * yb5 - 952.0 * yb6 + 128218.0 * yb7 - 47070.0 * yb8 + 26004.0 * yb9 - 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)))
                + tb2 * (vb2 * power_of<2>(1.0i + xb - 1.0i * xb2 - xb3) * yb * (xb10 * yb2 * (22.0 - 12477.0 * yb2 + 12675.0 * yb4 + 1001.0 * yb6 - 1221.0 * yb8) + 3.0 * yb2 * (2.0 + 3.0 * yb2 - 5.0 * yb4 + 9.0 * yb6 - 9.0 * yb8)
                    + 6.0 * 1.0i * xb * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) + 6.0 * 1.0i * xb13 * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) + 3.0 * xb14 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 3.0 * xb2 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) + 3.0 * xb12 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) - 4.0 * 1.0i * xb3 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8)
                    - 4.0 * 1.0i * xb11 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8) + xb4 * yb2 * (-22.0 + 12477.0 * yb2 - 12675.0 * yb4 - 1001.0 * yb6 + 1221.0 * yb8)
                    - 2.0 * 1.0i * xb5 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8) - 2.0 * 1.0i * xb9 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8)
                    + xb6 * (896.0 + 6022.0 * yb2 + 12885.0 * yb4 - 6451.0 * yb6 - 44481.0 * yb8 - 34407.0 * yb10) + 8.0 * 1.0i * xb7 * (224.0 + 1020.0 * yb2 + 3339.0 * yb4 - 11161.0 * yb6 - 10415.0 * yb8 + 609.0 * yb10)
                    + xb8 * (-896.0 - 6022.0 * yb2 - 12885.0 * yb4 + 6451.0 * yb6 + 44481.0 * yb8 + 34407.0 * yb10)) + 4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8
                      - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 1599.0 * yb3 - 6430.0 * yb4 + 11580.0 * yb5 + 952.0 * yb6 + 128218.0 * yb7 + 47070.0 * yb8 + 26004.0 * yb9 + 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11))
                  - 4.0 * xb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8
                      + 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 478.0 * yb3 + 2928.0 * yb4 - 4461.0 * yb5 - 288.0 * yb6 - 65749.0 * yb7 - 21168.0 * yb8 - 15767.0 * yb9 - 14904.0 * yb10 + 53223.0 * yb11)))
                - 4.0 * xb2 * (-4.0 * xb2 * power_of<2>(1.0i + xb) * yb * (-1.0 + yb2) * (6.0 * 1.0i * xb * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) + 6.0 * 1.0i * xb9 * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) - 3.0 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6) + 3.0 * xb10 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6)
                    - 8.0 * 1.0i * xb3 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) - 8.0 * 1.0i * xb7 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) + 3.0 * xb8 * yb2 * (-22.0 - 5.0 * yb2 + 252.0 * yb4 + 207.0 * yb6)
                    + xb2 * (66.0 * yb2 + 15.0 * yb4 - 756.0 * yb6 - 621.0 * yb8) - 2.0 * xb6 * (16.0 + 212.0 * yb2 + 1247.0 * yb4 + 3222.0 * yb6 + 2871.0 * yb8) + xb4 * (32.0 + 424.0 * yb2 + 2494.0 * yb4 + 6444.0 * yb6 + 5742.0 * yb8)
                    - 4.0 * 1.0i * xb5 * (-16.0 + 210.0 * yb2 + 717.0 * yb4 + 7232.0 * yb6 + 6465.0 * yb8)) + 4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11) - 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    + 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1623.0 * yb3 + 366.0 * yb4 - 6384.0 * yb5 - 36.0 * yb6 - 27874.0 * yb7 - 2646.0 * yb8 + 6654.0 * yb9 - 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8
                      - 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 478.0 * yb3 - 2928.0 * yb4 - 4461.0 * yb5 + 288.0 * yb6 - 65749.0 * yb7 + 21168.0 * yb8 - 15767.0 * yb9 + 14904.0 * yb10 + 53223.0 * yb11)));
            const complex<double> num4 = 4.0 * tb * xb2 * (16.0 * vb * xb2 * power_of<2>(-1.0i + xb) * yb * (-6.0 * 1.0i * xb * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) - 6.0 * 1.0i * xb9 * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8)
                    + xb6 * (-68.0 - 222.0 * yb2 + 1242.0 * yb4 + 6334.0 * yb6 + 4578.0 * yb8 - 3672.0 * yb10) - 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + 3.0 * xb10 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    + 8.0 * 1.0i * xb3 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10) + 8.0 * 1.0i * xb7 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10)
                    - xb2 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10) + xb8 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10)
                    + 2.0 * xb4 * (34.0 + 111.0 * yb2 - 621.0 * yb4 - 3167.0 * yb6 - 2289.0 * yb8 + 1836.0 * yb10) + 4.0 * 1.0i * xb5 * (-36.0 - 543.0 * yb2 - 1857.0 * yb4 - 4851.0 * yb6 + 3529.0 * yb8 + 7854.0 * yb10))
                  - 4.0 * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10) + 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    - 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1623.0 * yb3 - 366.0 * yb4 - 6384.0 * yb5 + 36.0 * yb6 - 27874.0 * yb7 + 2646.0 * yb8 + 6654.0 * yb9 + 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8
                      + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 1599.0 * yb3 + 6430.0 * yb4 + 11580.0 * yb5 - 952.0 * yb6 + 128218.0 * yb7 - 47070.0 * yb8 + 26004.0 * yb9 - 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)))
                + tb2 * ((-vb2) * power_of<2>(1.0i - xb - 1.0i * xb2 + xb3) * yb * (xb10 * yb2 * (22.0 - 12477.0 * yb2 + 12675.0 * yb4 + 1001.0 * yb6 - 1221.0 * yb8) + 3.0 * yb2 * (2.0 + 3.0 * yb2 - 5.0 * yb4 + 9.0 * yb6 - 9.0 * yb8)
                    - 6.0 * 1.0i * xb * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) - 6.0 * 1.0i * xb13 * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) + 3.0 * xb14 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 3.0 * xb2 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) + 3.0 * xb12 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) + 4.0 * 1.0i * xb3 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8)
                    + 4.0 * 1.0i * xb11 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8) + xb4 * yb2 * (-22.0 + 12477.0 * yb2 - 12675.0 * yb4 - 1001.0 * yb6 + 1221.0 * yb8)
                    + 2.0 * 1.0i * xb5 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8) + 2.0 * 1.0i * xb9 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8)
                    + xb6 * (896.0 + 6022.0 * yb2 + 12885.0 * yb4 - 6451.0 * yb6 - 44481.0 * yb8 - 34407.0 * yb10) - 8.0 * 1.0i * xb7 * (224.0 + 1020.0 * yb2 + 3339.0 * yb4 - 11161.0 * yb6 - 10415.0 * yb8 + 609.0 * yb10)
                    + xb8 * (-896.0 - 6022.0 * yb2 - 12885.0 * yb4 + 6451.0 * yb6 + 44481.0 * yb8 + 34407.0 * yb10)) - 4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8
                      - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 1599.0 * yb3 - 6430.0 * yb4 + 11580.0 * yb5 + 952.0 * yb6 + 128218.0 * yb7 + 47070.0 * yb8 + 26004.0 * yb9 + 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11))
                  + 4.0 * xb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8
                      + 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 478.0 * yb3 + 2928.0 * yb4 - 4461.0 * yb5 - 288.0 * yb6 - 65749.0 * yb7 - 21168.0 * yb8 - 15767.0 * yb9 - 14904.0 * yb10 + 53223.0 * yb11)))
                + 4.0 * xb2 * (-4.0 * xb2 * power_of<2>(-1.0i + xb) * yb * (-1.0 + yb2) * (-6.0 * 1.0i * xb * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) - 6.0 * 1.0i * xb9 * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) - 3.0 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6)
                    + 3.0 * xb10 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6) + 8.0 * 1.0i * xb3 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) + 8.0 * 1.0i * xb7 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) + 3.0 * xb8 * yb2 * (-22.0 - 5.0 * yb2 + 252.0 * yb4 + 207.0 * yb6)
                    + xb2 * (66.0 * yb2 + 15.0 * yb4 - 756.0 * yb6 - 621.0 * yb8) - 2.0 * xb6 * (16.0 + 212.0 * yb2 + 1247.0 * yb4 + 3222.0 * yb6 + 2871.0 * yb8) + xb4 * (32.0 + 424.0 * yb2 + 2494.0 * yb4 + 6444.0 * yb6 + 5742.0 * yb8)
                    + 4.0 * 1.0i * xb5 * (-16.0 + 210.0 * yb2 + 717.0 * yb4 + 7232.0 * yb6 + 6465.0 * yb8)) + 4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11) + 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    - 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1623.0 * yb3 + 366.0 * yb4 - 6384.0 * yb5 - 36.0 * yb6 - 27874.0 * yb7 - 2646.0 * yb8 + 6654.0 * yb9 - 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8
                      - 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 478.0 * yb3 - 2928.0 * yb4 - 4461.0 * yb5 + 288.0 * yb6 - 65749.0 * yb7 + 21168.0 * yb8 - 15767.0 * yb9 + 14904.0 * yb10 + 53223.0 * yb11)));
            const complex<double> num5 = -4.0 * tb * xb2 * (16.0 * vb * xb2 * power_of<2>(-1.0i + xb) * yb * (-6.0 * 1.0i * xb * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) - 6.0 * 1.0i * xb9 * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8)
                    + xb6 * (-68.0 - 222.0 * yb2 + 1242.0 * yb4 + 6334.0 * yb6 + 4578.0 * yb8 - 3672.0 * yb10) - 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + 3.0 * xb10 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    + 8.0 * 1.0i * xb3 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10) + 8.0 * 1.0i * xb7 * (-1.0 + 6.0 * yb2 + 309.0 * yb4 - 769.0 * yb6 + 128.0 * yb8 + 327.0 * yb10)
                    - xb2 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10) + xb8 * (-4.0 + 37.0 * yb2 - 1881.0 * yb4 + 125.0 * yb6 + 385.0 * yb8 + 1338.0 * yb10)
                    + 2.0 * xb4 * (34.0 + 111.0 * yb2 - 621.0 * yb4 - 3167.0 * yb6 - 2289.0 * yb8 + 1836.0 * yb10) + 4.0 * 1.0i * xb5 * (-36.0 - 543.0 * yb2 - 1857.0 * yb4 - 4851.0 * yb6 + 3529.0 * yb8 + 7854.0 * yb10))
                  - 4.0 * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10) + 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    - 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1623.0 * yb3 - 366.0 * yb4 - 6384.0 * yb5 + 36.0 * yb6 - 27874.0 * yb7 + 2646.0 * yb8 + 6654.0 * yb9 + 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8
                      + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 1599.0 * yb3 + 6430.0 * yb4 + 11580.0 * yb5 - 952.0 * yb6 + 128218.0 * yb7 - 47070.0 * yb8 + 26004.0 * yb9 - 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3123.0 * yb3 + 1818.0 * yb4 - 31050.0 * yb5 + 534.0 * yb6 - 9010.0 * yb7 - 11826.0 * yb8 + 15753.0 * yb9 - 7236.0 * yb10 + 44037.0 * yb11)))
                + tb2 * (vb2 * power_of<2>(1.0i - xb - 1.0i * xb2 + xb3) * yb * (xb10 * yb2 * (22.0 - 12477.0 * yb2 + 12675.0 * yb4 + 1001.0 * yb6 - 1221.0 * yb8) + 3.0 * yb2 * (2.0 + 3.0 * yb2 - 5.0 * yb4 + 9.0 * yb6 - 9.0 * yb8)
                    - 6.0 * 1.0i * xb * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) - 6.0 * 1.0i * xb13 * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) + 3.0 * xb14 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 3.0 * xb2 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) + 3.0 * xb12 * yb2 * (50.0 - 123.0 * yb2 - 139.0 * yb4 + 47.0 * yb6 + 165.0 * yb8) + 4.0 * 1.0i * xb3 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8)
                    + 4.0 * 1.0i * xb11 * yb2 * (76.0 + 1323.0 * yb2 - 1689.0 * yb4 + 17.0 * yb6 + 273.0 * yb8) + xb4 * yb2 * (-22.0 + 12477.0 * yb2 - 12675.0 * yb4 - 1001.0 * yb6 + 1221.0 * yb8)
                    + 2.0 * 1.0i * xb5 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8) + 2.0 * 1.0i * xb9 * yb2 * (-718.0 - 22143.0 * yb2 + 2433.0 * yb4 + 3043.0 * yb6 + 17385.0 * yb8)
                    + xb6 * (896.0 + 6022.0 * yb2 + 12885.0 * yb4 - 6451.0 * yb6 - 44481.0 * yb8 - 34407.0 * yb10) - 8.0 * 1.0i * xb7 * (224.0 + 1020.0 * yb2 + 3339.0 * yb4 - 11161.0 * yb6 - 10415.0 * yb8 + 609.0 * yb10)
                    + xb8 * (-896.0 - 6022.0 * yb2 - 12885.0 * yb4 + 6451.0 * yb6 + 44481.0 * yb8 + 34407.0 * yb10)) + 4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8
                      - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10)
                    + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 1599.0 * yb3 - 6430.0 * yb4 + 11580.0 * yb5 + 952.0 * yb6 + 128218.0 * yb7 + 47070.0 * yb8 + 26004.0 * yb9 + 33633.0 * yb10 - 109233.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3123.0 * yb3 - 1818.0 * yb4 - 31050.0 * yb5 - 534.0 * yb6 - 9010.0 * yb7 + 11826.0 * yb8 + 15753.0 * yb9 + 7236.0 * yb10 + 44037.0 * yb11))
                  - 4.0 * xb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8
                      + 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1564.0 * yb3 - 890.0 * yb4 - 15759.0 * yb5 - 232.0 * yb6 - 4387.0 * yb7 + 5850.0 * yb8 + 7907.0 * yb9 + 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 478.0 * yb3 + 2928.0 * yb4 - 4461.0 * yb5 - 288.0 * yb6 - 65749.0 * yb7 - 21168.0 * yb8 - 15767.0 * yb9 - 14904.0 * yb10 + 53223.0 * yb11)))
                - 4.0 * xb2 * (-4.0 * xb2 * power_of<2>(-1.0i + xb) * yb * (-1.0 + yb2) * (-6.0 * 1.0i * xb * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) - 6.0 * 1.0i * xb9 * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) - 3.0 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6)
                    + 3.0 * xb10 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6) + 8.0 * 1.0i * xb3 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) + 8.0 * 1.0i * xb7 * yb2 * (-11.0 - 788.0 * yb2 + 217.0 * yb4 + 150.0 * yb6) + 3.0 * xb8 * yb2 * (-22.0 - 5.0 * yb2 + 252.0 * yb4 + 207.0 * yb6)
                    + xb2 * (66.0 * yb2 + 15.0 * yb4 - 756.0 * yb6 - 621.0 * yb8) - 2.0 * xb6 * (16.0 + 212.0 * yb2 + 1247.0 * yb4 + 3222.0 * yb6 + 2871.0 * yb8) + xb4 * (32.0 + 424.0 * yb2 + 2494.0 * yb4 + 6444.0 * yb6 + 5742.0 * yb8)
                    + 4.0 * 1.0i * xb5 * (-16.0 + 210.0 * yb2 + 717.0 * yb4 + 7232.0 * yb6 + 6465.0 * yb8)) + 4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11) + 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    - 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1623.0 * yb3 + 366.0 * yb4 - 6384.0 * yb5 - 36.0 * yb6 - 27874.0 * yb7 - 2646.0 * yb8 + 6654.0 * yb9 - 1863.0 * yb10 + 33345.0 * yb11))
                  + vb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9)
                    + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8
                      - 126.0 * yb9 + 573.0 * yb10) - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1564.0 * yb3 + 890.0 * yb4 - 15759.0 * yb5 + 232.0 * yb6 - 4387.0 * yb7 - 5850.0 * yb8 + 7907.0 * yb9 - 3627.0 * yb10 + 22107.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 478.0 * yb3 - 2928.0 * yb4 - 4461.0 * yb5 + 288.0 * yb6 - 65749.0 * yb7 + 21168.0 * yb8 - 15767.0 * yb9 + 14904.0 * yb10 + 53223.0 * yb11)));
            const complex<double> num6 = 8.0 * (-1.0 + 3.0 * yb2);
            const complex<double> num7 = 8.0 - 24.0 * yb2;
            const complex<double> num8 = 4.0 * xb2 * (-4.0 * xb2 * yb * (96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 6.0 * xb2 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6)
                    + 6.0 * xb10 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb12 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + xb4 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10) + xb8 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 776.0 * yb2 - 3585.0 * yb4 - 21917.0 * yb6 - 11939.0 * yb8 + 5433.0 * yb10)) + vb2 * (96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8
                      + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 990.0 * yb3 - 2928.0 * yb4 - 10605.0 * yb5 + 288.0 * yb6 - 93397.0 * yb7 + 21168.0 * yb8 - 71063.0 * yb9 + 14904.0 * yb10 + 11751.0 * yb11))
                  + 4.0 * vb * xb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1879.0 * yb3 + 366.0 * yb4 - 9456.0 * yb5 - 36.0 * yb6 - 41698.0 * yb7 - 2646.0 * yb8 - 20994.0 * yb9 - 1863.0 * yb10 + 12609.0 * yb11)))
                + 4.0 * tb * xb2 * (16.0 * vb * xb2 * yb * (96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    + 3.0 * xb12 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + xb4 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10) + xb8 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10)
                    + 2.0 * xb2 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10) + 2.0 * xb10 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10)
                    + 4.0 * xb6 * (-38.0 - 1103.0 * yb2 - 5871.0 * yb4 - 19781.0 * yb6 - 9055.0 * yb8 + 7176.0 * yb10)) - 4.0 * xb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10)
                    - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1879.0 * yb3 - 366.0 * yb4 - 9456.0 * yb5 + 36.0 * yb6 - 41698.0 * yb7 + 2646.0 * yb8 - 20994.0 * yb9 + 1863.0 * yb10 + 12609.0 * yb11))
                  + vb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7
                      + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 2623.0 * yb3 + 6430.0 * yb4 + 23868.0 * yb5 - 952.0 * yb6 + 183514.0 * yb7 - 47070.0 * yb8 + 136596.0 * yb9 - 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)))
                + tb2 * ((-vb2) * power_of<2>(-1.0 + xb2) * yb * (96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb11 * yb2 * power_of<4>(1.0 + 3.0 * yb2)
                    + 96.0 * xb13 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 12.0 * xb2 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8)
                    + 12.0 * xb14 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8) + 12.0 * xb4 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8) + 12.0 * xb12 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8)
                    + 2.0 * xb8 * (-896.0 - 1626.0 * yb2 - 7683.0 * yb4 + 110485.0 * yb6 + 94135.0 * yb8 + 2193.0 * yb10) + 4.0 * xb6 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)
                    + 4.0 * xb10 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)) + 4.0 * xb2 * (96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 990.0 * yb3 + 2928.0 * yb4 - 10605.0 * yb5 - 288.0 * yb6 - 93397.0 * yb7 - 21168.0 * yb8 - 71063.0 * yb9 - 14904.0 * yb10 + 11751.0 * yb11))
                  - 4.0 * vb * xb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7
                      + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 2623.0 * yb3 - 6430.0 * yb4 + 23868.0 * yb5 + 952.0 * yb6 + 183514.0 * yb7 + 47070.0 * yb8 + 136596.0 * yb9 + 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)));
            const complex<double> num9 = 4.0 * xb2 * (-4.0 * xb2 * yb * (-96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 6.0 * xb2 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6)
                    + 6.0 * xb10 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb12 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + xb4 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10) + xb8 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 776.0 * yb2 - 3585.0 * yb4 - 21917.0 * yb6 - 11939.0 * yb8 + 5433.0 * yb10)) + vb2 * (-96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8
                      + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 990.0 * yb3 - 2928.0 * yb4 - 10605.0 * yb5 + 288.0 * yb6 - 93397.0 * yb7 + 21168.0 * yb8 - 71063.0 * yb9 + 14904.0 * yb10 + 11751.0 * yb11))
                  + 4.0 * vb * xb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1879.0 * yb3 + 366.0 * yb4 - 9456.0 * yb5 - 36.0 * yb6 - 41698.0 * yb7 - 2646.0 * yb8 - 20994.0 * yb9 - 1863.0 * yb10 + 12609.0 * yb11)))
                + 4.0 * tb * xb2 * (16.0 * vb * xb2 * yb * (-96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    + 3.0 * xb12 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + xb4 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10) + xb8 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10)
                    + 2.0 * xb2 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10) + 2.0 * xb10 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10)
                    + 4.0 * xb6 * (-38.0 - 1103.0 * yb2 - 5871.0 * yb4 - 19781.0 * yb6 - 9055.0 * yb8 + 7176.0 * yb10)) - 4.0 * xb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10)
                    - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1879.0 * yb3 - 366.0 * yb4 - 9456.0 * yb5 + 36.0 * yb6 - 41698.0 * yb7 + 2646.0 * yb8 - 20994.0 * yb9 + 1863.0 * yb10 + 12609.0 * yb11))
                  + vb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7
                      + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 2623.0 * yb3 + 6430.0 * yb4 + 23868.0 * yb5 - 952.0 * yb6 + 183514.0 * yb7 - 47070.0 * yb8 + 136596.0 * yb9 - 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)))
                + tb2 * ((-vb2) * power_of<2>(-1.0 + xb2) * yb * (-96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb11 * yb2 * power_of<4>(1.0 + 3.0 * yb2)
                    - 96.0 * xb13 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 12.0 * xb2 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8)
                    + 12.0 * xb14 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8) + 12.0 * xb4 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8) + 12.0 * xb12 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8)
                    + 2.0 * xb8 * (-896.0 - 1626.0 * yb2 - 7683.0 * yb4 + 110485.0 * yb6 + 94135.0 * yb8 + 2193.0 * yb10) + 4.0 * xb6 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)
                    + 4.0 * xb10 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)) + 4.0 * xb2 * (-96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 990.0 * yb3 + 2928.0 * yb4 - 10605.0 * yb5 - 288.0 * yb6 - 93397.0 * yb7 - 21168.0 * yb8 - 71063.0 * yb9 - 14904.0 * yb10 + 11751.0 * yb11))
                  - 4.0 * vb * xb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7
                      + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 2623.0 * yb3 - 6430.0 * yb4 + 23868.0 * yb5 + 952.0 * yb6 + 183514.0 * yb7 + 47070.0 * yb8 + 136596.0 * yb9 + 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)));
            const complex<double> num10 = (-4.0 * xb2 - 4.0 * tb2 * xb2 + tb * (1.0 - 6.0 * xb2 + xb4)) * (1.0 + yb) * (3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<4>(1.0 + yb) * (-1.0 + 3.0 * yb2) + 3.0 * xb8 * power_of<2>(-1.0 + yb) * yb2 * power_of<4>(1.0 + yb) * (-1.0 + 3.0 * yb2)
                 - 4.0 * xb2 * yb * (-1.0 + yb - 6.0 * yb2 - 32.0 * yb3 - 436.0 * yb4 + 366.0 * yb5 - 130.0 * yb6 + 256.0 * yb7 - 195.0 * yb8 + 177.0 * yb9) - 4.0 * xb6 * yb * (-1.0 + yb - 6.0 * yb2 - 32.0 * yb3 - 436.0 * yb4 + 366.0 * yb5 - 130.0 * yb6 + 256.0 * yb7 - 195.0 * yb8
                   + 177.0 * yb9) + xb4 * (16.0 + 88.0 * yb + 62.0 * yb2 + 572.0 * yb3 + 8.0 * yb4 + 2692.0 * yb5 - 3044.0 * yb6 - 284.0 * yb7 - 4504.0 * yb8 - 1020.0 * yb9 - 2778.0 * yb10));
            const complex<double> num11 = (4.0 * xb2 + 4.0 * tb2 * xb2 - tb * (1.0 - 6.0 * xb2 + xb4)) * (1.0 + yb) * (3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<4>(1.0 + yb) * (-1.0 + 3.0 * yb2) + 3.0 * xb8 * power_of<2>(-1.0 + yb) * yb2 * power_of<4>(1.0 + yb) * (-1.0 + 3.0 * yb2)
                 - 4.0 * xb2 * yb * (-1.0 + yb - 6.0 * yb2 - 32.0 * yb3 - 436.0 * yb4 + 366.0 * yb5 - 130.0 * yb6 + 256.0 * yb7 - 195.0 * yb8 + 177.0 * yb9) - 4.0 * xb6 * yb * (-1.0 + yb - 6.0 * yb2 - 32.0 * yb3 - 436.0 * yb4 + 366.0 * yb5 - 130.0 * yb6 + 256.0 * yb7 - 195.0 * yb8
                   + 177.0 * yb9) + xb4 * (16.0 + 88.0 * yb + 62.0 * yb2 + 572.0 * yb3 + 8.0 * yb4 + 2692.0 * yb5 - 3044.0 * yb6 - 284.0 * yb7 - 4504.0 * yb8 - 1020.0 * yb9 - 2778.0 * yb10));
            const complex<double> num12 = (4.0 * xb2 + 4.0 * vb2 * xb2 + vb * (1.0 - 6.0 * xb2 + xb4)) * (-1.0 + yb) * (3.0 * power_of<4>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) + 3.0 * xb8 * power_of<4>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2)
                 - 4.0 * xb2 * yb * (1.0 + yb + 6.0 * yb2 - 32.0 * yb3 + 436.0 * yb4 + 366.0 * yb5 + 130.0 * yb6 + 256.0 * yb7 + 195.0 * yb8 + 177.0 * yb9) - 4.0 * xb6 * yb * (1.0 + yb + 6.0 * yb2 - 32.0 * yb3 + 436.0 * yb4 + 366.0 * yb5 + 130.0 * yb6 + 256.0 * yb7 + 195.0 * yb8
                   + 177.0 * yb9) - 2.0 * xb4 * (-8.0 + 44.0 * yb - 31.0 * yb2 + 286.0 * yb3 - 4.0 * yb4 + 1346.0 * yb5 + 1522.0 * yb6 - 142.0 * yb7 + 2252.0 * yb8 - 510.0 * yb9 + 1389.0 * yb10));
            const complex<double> num13 = -4.0 * xb2 * (-4.0 * xb2 * yb * (96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 6.0 * xb2 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6)
                    + 6.0 * xb10 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb12 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + xb4 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10) + xb8 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 776.0 * yb2 - 3585.0 * yb4 - 21917.0 * yb6 - 11939.0 * yb8 + 5433.0 * yb10)) + vb2 * (96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8
                      + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 990.0 * yb3 - 2928.0 * yb4 - 10605.0 * yb5 + 288.0 * yb6 - 93397.0 * yb7 + 21168.0 * yb8 - 71063.0 * yb9 + 14904.0 * yb10 + 11751.0 * yb11))
                  + 4.0 * vb * xb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1879.0 * yb3 + 366.0 * yb4 - 9456.0 * yb5 - 36.0 * yb6 - 41698.0 * yb7 - 2646.0 * yb8 - 20994.0 * yb9 - 1863.0 * yb10 + 12609.0 * yb11)))
                - 4.0 * tb * xb2 * (16.0 * vb * xb2 * yb * (96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    + 3.0 * xb12 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + xb4 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10) + xb8 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10)
                    + 2.0 * xb2 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10) + 2.0 * xb10 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10)
                    + 4.0 * xb6 * (-38.0 - 1103.0 * yb2 - 5871.0 * yb4 - 19781.0 * yb6 - 9055.0 * yb8 + 7176.0 * yb10)) - 4.0 * xb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10)
                    - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1879.0 * yb3 - 366.0 * yb4 - 9456.0 * yb5 + 36.0 * yb6 - 41698.0 * yb7 + 2646.0 * yb8 - 20994.0 * yb9 + 1863.0 * yb10 + 12609.0 * yb11))
                  + vb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7
                      + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 2623.0 * yb3 + 6430.0 * yb4 + 23868.0 * yb5 - 952.0 * yb6 + 183514.0 * yb7 - 47070.0 * yb8 + 136596.0 * yb9 - 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)))
                + tb2 * (vb2 * power_of<2>(-1.0 + xb2) * yb * (96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb11 * yb2 * power_of<4>(1.0 + 3.0 * yb2)
                    + 96.0 * xb13 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 12.0 * xb2 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8)
                    + 12.0 * xb14 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8) + 12.0 * xb4 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8) + 12.0 * xb12 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8)
                    + 2.0 * xb8 * (-896.0 - 1626.0 * yb2 - 7683.0 * yb4 + 110485.0 * yb6 + 94135.0 * yb8 + 2193.0 * yb10) + 4.0 * xb6 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)
                    + 4.0 * xb10 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)) - 4.0 * xb2 * (96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 990.0 * yb3 + 2928.0 * yb4 - 10605.0 * yb5 - 288.0 * yb6 - 93397.0 * yb7 - 21168.0 * yb8 - 71063.0 * yb9 - 14904.0 * yb10 + 11751.0 * yb11))
                  + 4.0 * vb * xb2 * (192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7
                      + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 2623.0 * yb3 - 6430.0 * yb4 + 23868.0 * yb5 + 952.0 * yb6 + 183514.0 * yb7 + 47070.0 * yb8 + 136596.0 * yb9 + 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)));
            const complex<double> num14 = -4.0 * xb2 * (-4.0 * xb2 * yb * (-96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 6.0 * xb2 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6)
                    + 6.0 * xb10 * yb4 * (35.0 - 217.0 * yb2 + 89.0 * yb4 + 93.0 * yb6) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb12 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + xb4 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10) + xb8 * (32.0 + 502.0 * yb2 + 14553.0 * yb4 - 11359.0 * yb6 + 235.0 * yb8 - 3963.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 776.0 * yb2 - 3585.0 * yb4 - 21917.0 * yb6 - 11939.0 * yb8 + 5433.0 * yb10)) + vb2 * (-96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (-1.0 + 4.0 * yb + 2.0 * yb2 + 5.0 * yb3 + 8.0 * yb4 - 95.0 * yb5 - 18.0 * yb6 + 47.0 * yb7 + 9.0 * yb8
                      + 39.0 * yb9) - 4.0 * xb4 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 + 6.0 * yb - 122.0 * yb2 - 164.0 * yb3 - 1575.0 * yb4 - 328.0 * yb5 + 1145.0 * yb6 + 612.0 * yb7 - 17.0 * yb8 - 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 - 1692.0 * yb3 + 890.0 * yb4 - 17295.0 * yb5 + 232.0 * yb6 - 11299.0 * yb7 - 5850.0 * yb8 - 5917.0 * yb9 - 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (-64.0 + 464.0 * yb - 600.0 * yb2 - 990.0 * yb3 - 2928.0 * yb4 - 10605.0 * yb5 + 288.0 * yb6 - 93397.0 * yb7 + 21168.0 * yb8 - 71063.0 * yb9 + 14904.0 * yb10 + 11751.0 * yb11))
                  + 4.0 * vb * xb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb12 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 2.0 * xb2 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7 + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 - 3.0 * yb - yb2 + 82.0 * yb3 - 768.0 * yb4 + 164.0 * yb5 - 782.0 * yb6 - 306.0 * yb7
                      + 650.0 * yb8 + 63.0 * yb9 + 903.0 * yb10) + xb4 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + xb8 * (16.0 - 48.0 * yb + 147.0 * yb2 + 339.0 * yb3 + 890.0 * yb4 + 22620.0 * yb5 + 232.0 * yb6 - 17454.0 * yb7 - 5850.0 * yb8 + 6476.0 * yb9 - 3627.0 * yb10 - 3741.0 * yb11)
                    + 4.0 * xb6 * (8.0 - 22.0 * yb + 75.0 * yb2 - 1879.0 * yb3 + 366.0 * yb4 - 9456.0 * yb5 - 36.0 * yb6 - 41698.0 * yb7 - 2646.0 * yb8 - 20994.0 * yb9 - 1863.0 * yb10 + 12609.0 * yb11)))
                - 4.0 * tb * xb2 * (16.0 * vb * xb2 * yb * (-96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 288.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10)
                    + 3.0 * xb12 * (yb2 - 17.0 * yb4 + 21.0 * yb6 - 11.0 * yb8 + 6.0 * yb10) + xb4 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10) + xb8 * (-80.0 - 163.0 * yb2 + 8067.0 * yb4 - 6095.0 * yb6 + 6241.0 * yb8 + 222.0 * yb10)
                    + 2.0 * xb2 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10) + 2.0 * xb10 * (-2.0 - yb2 - 873.0 * yb4 - 131.0 * yb6 + 383.0 * yb8 + 624.0 * yb10)
                    + 4.0 * xb6 * (-38.0 - 1103.0 * yb2 - 5871.0 * yb4 - 19781.0 * yb6 - 9055.0 * yb8 + 7176.0 * yb10)) - 4.0 * xb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 576.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    - 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 3.0 * xb12 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 2.0 * xb2 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8
                      - 63.0 * yb9 + 903.0 * yb10) + 2.0 * xb10 * yb * (-2.0 + 3.0 * yb - yb2 - 82.0 * yb3 - 768.0 * yb4 - 164.0 * yb5 - 782.0 * yb6 + 306.0 * yb7 + 650.0 * yb8 - 63.0 * yb9 + 903.0 * yb10)
                    - xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    - xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 - 339.0 * yb3 + 890.0 * yb4 - 22620.0 * yb5 + 232.0 * yb6 + 17454.0 * yb7 - 5850.0 * yb8 - 6476.0 * yb9 - 3627.0 * yb10 + 3741.0 * yb11)
                    + 4.0 * xb6 * (-8.0 - 22.0 * yb - 75.0 * yb2 - 1879.0 * yb3 - 366.0 * yb4 - 9456.0 * yb5 + 36.0 * yb6 - 41698.0 * yb7 + 2646.0 * yb8 - 20994.0 * yb9 + 1863.0 * yb10 + 12609.0 * yb11))
                  + vb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (1.0 - yb - 2.0 * yb2 - 20.0 * yb3 - 8.0 * yb4 + 26.0 * yb5 + 18.0 * yb6 - 20.0 * yb7 - 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7 + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 - 6.0 * yb + 19.0 * yb2 - 26.0 * yb3 - 438.0 * yb4 - 22.0 * yb5 - 334.0 * yb6 + 18.0 * yb7
                      + 343.0 * yb8 + 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 2623.0 * yb3 + 6430.0 * yb4 + 23868.0 * yb5 - 952.0 * yb6 + 183514.0 * yb7 - 47070.0 * yb8 + 136596.0 * yb9 - 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 81.0 * yb3 - 186.0 * yb4 - 2496.0 * yb5 - 756.0 * yb6 + 1482.0 * yb7 + 54.0 * yb8 - 1418.0 * yb9 - 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 - 3379.0 * yb3 + 1818.0 * yb4 - 34122.0 * yb5 + 534.0 * yb6 - 22834.0 * yb7 - 11826.0 * yb8 - 11895.0 * yb9 - 7236.0 * yb10 + 23301.0 * yb11)))
                + tb2 * (vb2 * power_of<2>(-1.0 + xb2) * yb * (-96.0 * xb3 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb5 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb9 * yb2 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb11 * yb2 * power_of<4>(1.0 + 3.0 * yb2)
                    - 96.0 * xb13 * yb2 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 12.0 * xb2 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8)
                    + 12.0 * xb14 * yb2 * (7.0 - 9.0 * yb2 - 79.0 * yb4 + 45.0 * yb6 + 36.0 * yb8) + 12.0 * xb4 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8) + 12.0 * xb12 * yb2 * (40.0 - 127.0 * yb2 - 35.0 * yb4 + 83.0 * yb6 + 39.0 * yb8)
                    + 2.0 * xb8 * (-896.0 - 1626.0 * yb2 - 7683.0 * yb4 + 110485.0 * yb6 + 94135.0 * yb8 + 2193.0 * yb10) + 4.0 * xb6 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)
                    + 4.0 * xb10 * (-224.0 - 2357.0 * yb2 - 23781.0 * yb4 - 6035.0 * yb6 + 89.0 * yb8 + 15924.0 * yb10)) - 4.0 * xb2 * (-96.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 96.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8)
                    + 12.0 * xb2 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9) + 12.0 * xb14 * yb2 * (1.0 + 4.0 * yb - 2.0 * yb2 + 5.0 * yb3 - 8.0 * yb4 - 95.0 * yb5 + 18.0 * yb6 + 47.0 * yb7 - 9.0 * yb8 + 39.0 * yb9)
                    - 4.0 * xb4 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    - 4.0 * xb12 * yb * (-4.0 - 6.0 * yb - 122.0 * yb2 + 164.0 * yb3 - 1575.0 * yb4 + 328.0 * yb5 + 1145.0 * yb6 - 612.0 * yb7 - 17.0 * yb8 + 126.0 * yb9 + 573.0 * yb10)
                    + 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    + 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 1692.0 * yb3 - 890.0 * yb4 - 17295.0 * yb5 - 232.0 * yb6 - 11299.0 * yb7 + 5850.0 * yb8 - 5917.0 * yb9 + 3627.0 * yb10 + 11739.0 * yb11)
                    - 2.0 * xb8 * (64.0 + 464.0 * yb + 600.0 * yb2 - 990.0 * yb3 + 2928.0 * yb4 - 10605.0 * yb5 - 288.0 * yb6 - 93397.0 * yb7 - 21168.0 * yb8 - 71063.0 * yb9 - 14904.0 * yb10 + 11751.0 * yb11))
                  + 4.0 * vb * xb2 * (-192.0 * xb3 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb7 * yb3 * power_of<4>(1.0 + 3.0 * yb2) + 384.0 * xb9 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb11 * yb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb13 * yb3 * power_of<4>(1.0 + 3.0 * yb2)
                    + 3.0 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9) + 3.0 * xb16 * yb2 * (-1.0 - yb + 2.0 * yb2 - 20.0 * yb3 + 8.0 * yb4 + 26.0 * yb5 - 18.0 * yb6 - 20.0 * yb7 + 9.0 * yb8 + 15.0 * yb9)
                    + 4.0 * xb2 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7 + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 4.0 * xb14 * yb * (-1.0 + 6.0 * yb + 19.0 * yb2 + 26.0 * yb3 - 438.0 * yb4 + 22.0 * yb5 - 334.0 * yb6 - 18.0 * yb7
                      + 343.0 * yb8 - 36.0 * yb9 + 411.0 * yb10) + 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 + 2623.0 * yb3 - 6430.0 * yb4 + 23868.0 * yb5 + 952.0 * yb6 + 183514.0 * yb7 + 47070.0 * yb8 + 136596.0 * yb9 + 33633.0 * yb10 - 26289.0 * yb11)
                    - 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    - 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 - 81.0 * yb3 + 186.0 * yb4 - 2496.0 * yb5 + 756.0 * yb6 + 1482.0 * yb7 - 54.0 * yb8 - 1418.0 * yb9 + 1215.0 * yb10 + 447.0 * yb11)
                    + 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)
                    + 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 3379.0 * yb3 - 1818.0 * yb4 - 34122.0 * yb5 - 534.0 * yb6 - 22834.0 * yb7 + 11826.0 * yb8 - 11895.0 * yb9 + 7236.0 * yb10 + 23301.0 * yb11)));
            const complex<double> num15 = power_of<4>(1.0 + xb2) * power_of<2>(-1.0 + yb2) * (1.0 - 4.0 * yb2 + 3.0 * yb4 + xb4 * (1.0 - 4.0 * yb2 + 3.0 * yb4) + 2.0 * xb2 * (-5.0 + 80.0 * yb2 + 21.0 * yb4));

            const complex<double> denom0 = 81.0 * yb3;
            const complex<double> denom1 = 648.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * yb3 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom2 = 1296.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * yb3 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom3 = 324.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * yb3 * power_of<4>(xb + 3.0 * xb * yb2);
            const complex<double> denom4 = 324.0 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * yb3 * power_of<4>(xb + 3.0 * xb * yb2);
            const complex<double> denom5 = 432.0 * xb6 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom6 = 324.0 * xb6 * yb3 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom7 = 81.0 * xb4 * power_of<2>(1.0 + 3.0 * yb2);
            const complex<double> denom8 = 81.0 * xb6 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom9 = 486.0 * xb4 * power_of<3>(yb + 3.0 * yb3);
            const complex<double> denom10 = 324.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * yb3 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom11 = 324.0 * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * yb3 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom12 = 324.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * yb3 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom13 = (243.0 * yb3) / 16.0;

            const complex<double> logs1 = 3.0 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 3.0 * xb12 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 18.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 24.0 * yb2 + 9.0 * yb4) * ln2
                + 18.0 * xb10 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 24.0 * yb2 + 9.0 * yb4) * ln2 + 9.0 * xb4 * yb3 * power_of<2>(-1.0 + yb2) * (-11.0 + 204.0 * yb2 + 63.0 * yb4) * ln2 + 9.0 * xb8 * yb3 * power_of<2>(-1.0 + yb2) * (-11.0 + 204.0 * yb2 + 63.0 * yb4) * ln2
                + 4.0 * xb6 * (16.0 + 144.0 * yb2 + 288.0 * yb4 - 864.0 * yb6 - 3888.0 * yb8 - 3888.0 * yb10 - 39.0 * yb3 * ln2 + 774.0 * yb5 * ln2 - 1224.0 * yb7 * ln2 + 282.0 * yb9 * ln2 + 207.0 * yb11 * ln2);
            const complex<double> logs2 = 3.0 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 3.0 * xb12 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 18.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 24.0 * yb2 + 9.0 * yb4) * ln2
                + 18.0 * xb10 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 24.0 * yb2 + 9.0 * yb4) * ln2 + 9.0 * xb4 * yb3 * power_of<2>(-1.0 + yb2) * (-11.0 + 204.0 * yb2 + 63.0 * yb4) * ln2 + 9.0 * xb8 * yb3 * power_of<2>(-1.0 + yb2) * (-11.0 + 204.0 * yb2 + 63.0 * yb4) * ln2
                + 4.0 * xb6 * (-16.0 - 144.0 * yb2 - 288.0 * yb4 + 864.0 * yb6 + 3888.0 * yb8 + 3888.0 * yb10 - 39.0 * yb3 * ln2 + 774.0 * yb5 * ln2 - 1224.0 * yb7 * ln2 + 282.0 * yb9 * ln2 + 207.0 * yb11 * ln2);
            const complex<double> logs3 = 3.0 * (-1.0 + yb2) + 3.0 * xb8 * (-1.0 + yb2) - 4.0 * xb2 * (4.0 - 7.0 * yb2 + 3.0 * yb4) - 4.0 * xb6 * (4.0 - 7.0 * yb2 + 3.0 * yb4) + 2.0 * xb4 * (-13.0 + 64.0 * ln2 + 12.0 * yb4 * (-1.0 + 48.0 * ln2) + yb2 * (25.0 + 384.0 * ln2));
            const complex<double> logs4 = -18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + 18.0 * xb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 24.0 * yb2 + 9.0 * yb4) * ln2 + 18.0 * xb10 * power_of<2>(-1.0 + yb2) * (-1.0 + 24.0 * yb2 + 9.0 * yb4) * ln2
                + 9.0 * xb4 * power_of<2>(-1.0 + yb2) * (-11.0 + 204.0 * yb2 + 63.0 * yb4) * ln2 + 9.0 * xb8 * power_of<2>(-1.0 + yb2) * (-11.0 + 204.0 * yb2 + 63.0 * yb4) * ln2
                + 4.0 * xb6 * (-16.0 - 39.0 * ln2 - 72.0 * yb4 * (12.0 + 17.0 * ln2) + 9.0 * yb8 * (-144.0 + 23.0 * ln2) + 6.0 * yb6 * (-288.0 + 47.0 * ln2) + 6.0 * yb2 * (-32.0 + 129.0 * ln2)) + log(8.0)
                + xb12 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0));
            const complex<double> logs5 = 9.0 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 9.0 * xb8 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 12.0 * xb2 * yb2 * (-3.0 - 163.0 * yb2 + 151.0 * yb4 + 15.0 * yb6) - 12.0 * xb6 * yb2 * (-3.0 - 163.0 * yb2 + 151.0 * yb4 + 15.0 * yb6)
                - 2.0 * xb4 * (-543.0 * yb4 - 3.0 * yb2 * (103.0 + 192.0 * ln2) + 9.0 * yb8 * (607.0 + 864.0 * ln2) + yb6 * (5181.0 + 5184.0 * ln2) - 32.0 * (2.0 + log(8.0)));
            const complex<double> logs6 = -9.0 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 9.0 * xb8 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 12.0 * xb2 * yb2 * (-3.0 - 163.0 * yb2 + 151.0 * yb4 + 15.0 * yb6) + 12.0 * xb6 * yb2 * (-3.0 - 163.0 * yb2 + 151.0 * yb4 + 15.0 * yb6)
                + 2.0 * xb4 * (-543.0 * yb4 - 3.0 * yb2 * (103.0 + 192.0 * ln2) + 9.0 * yb8 * (607.0 + 864.0 * ln2) + yb6 * (5181.0 + 5184.0 * ln2) - 32.0 * (2.0 + log(8.0)));
            const complex<double> logs7 = -4.0 * xb2 * (2.0 * xb6 * (8.0 + 52.0 * yb + 75.0 * yb2 + 290.0 * yb4 - 176.0 * yb6 - 2394.0 * yb8 - 1899.0 * yb10 + yb3 * (317.0 - 78.0 * ln2) + 18.0 * yb5 * (75.0 + 86.0 * ln2) + 3.0 * yb11 * (-463.0 + 138.0 * ln2) - 16.0 * yb7 * (104.0 + 153.0 * ln2)
                    + yb9 * (-2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (-4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (-4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + 3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2) - 3.0 * yb * (1.0 + ln4)) + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2)
                    - 3.0 * yb * (1.0 + ln4)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                + tb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (32.0 + 208.0 * yb + 300.0 * yb2 + 1160.0 * yb4 - 704.0 * yb6 - 9576.0 * yb8 - 7596.0 * yb10 + yb3 * (1268.0 - 57.0 * ln2) + 18.0 * yb5 * (300.0 + 59.0 * ln2) + 3.0 * yb11 * (-1852.0 + 87.0 * ln2) + 2.0 * yb9 * (-5524.0 + 213.0 * ln2)
                    - 4.0 * yb7 * (1664.0 + 423.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (-2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0)) + yb2 * (-12.0 + log(8.0)) - 12.0 * yb4 * (79.0 + log(32.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0)) + yb2 * (-12.0 + log(8.0)) - 12.0 * yb4 * (79.0 + log(32.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2) + 3.0 * yb5 * (-1.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2)
                    + 3.0 * yb5 * (-1.0 + log(128.0)))) + tb * xb2 * (-2.0 * xb2 * yb * (-1.0 + yb2) * (2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2)
                    + yb6 * (521.0 + 1080.0 * ln2)) - 2.0 * xb10 * yb * (-1.0 + yb2) * (2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2) + yb6 * (521.0 + 1080.0 * ln2))
                - 4.0 * xb6 * (24.0 + 154.0 * yb + 225.0 * yb2 + 794.0 * yb4 - 668.0 * yb6 - 6930.0 * yb8 - 5733.0 * yb10 + yb3 * (941.0 - 312.0 * ln2) + 18.0 * yb5 * (173.0 + 344.0 * ln2) + 3.0 * yb11 * (-1271.0 + 552.0 * ln2) + 4.0 * yb9 * (-2041.0 + 564.0 * ln2)
                    - 8.0 * yb7 * (565.0 + 1224.0 * ln2)) + xb4 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb11 * (1479.0 - 4536.0 * ln2) + yb3 * (505.0 + 792.0 * ln2) + 16.0 * yb7 * (143.0 + 1602.0 * ln2)
                    - 6.0 * yb5 * (1415.0 + 2712.0 * ln2) - 2.0 * yb9 * (2027.0 + 2808.0 * ln2)) + xb8 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb11 * (1479.0 - 4536.0 * ln2) + yb3 * (505.0 + 792.0 * ln2)
                    + 16.0 * yb7 * (143.0 + 1602.0 * ln2) - 6.0 * yb5 * (1415.0 + 2712.0 * ln2) - 2.0 * yb9 * (2027.0 + 2808.0 * ln2)) - 3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 - 2.0 * yb * (1.0 + log(16.0)) + yb2 * (-1.0 + ln256))
                - 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 - 2.0 * yb * (1.0 + log(16.0)) + yb2 * (-1.0 + ln256)));
            const complex<double> logs8 = 4.0 * xb2 * (2.0 * xb6 * (8.0 + 52.0 * yb + 75.0 * yb2 + 290.0 * yb4 - 176.0 * yb6 - 2394.0 * yb8 - 1899.0 * yb10 + yb3 * (317.0 - 78.0 * ln2) + 18.0 * yb5 * (75.0 + 86.0 * ln2) + 3.0 * yb11 * (-463.0 + 138.0 * ln2) - 16.0 * yb7 * (104.0 + 153.0 * ln2)
                    + yb9 * (-2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (-4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (-4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + 3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2) - 3.0 * yb * (1.0 + ln4)) + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2)
                    - 3.0 * yb * (1.0 + ln4)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                + tb2 * (-3.0 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 3.0 * xb16 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 2.0 * xb8 * (32.0 + 208.0 * yb + 300.0 * yb2 + 1160.0 * yb4 - 704.0 * yb6 - 9576.0 * yb8 - 7596.0 * yb10 + yb3 * (1268.0 - 57.0 * ln2) + 18.0 * yb5 * (300.0 + 59.0 * ln2)
                    + 3.0 * yb11 * (-1852.0 + 87.0 * ln2) + 2.0 * yb9 * (-5524.0 + 213.0 * ln2) - 4.0 * yb7 * (1664.0 + 423.0 * ln2)) + 8.0 * xb6 * yb * (-1.0 + yb2) * (-2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0))
                    + yb2 * (-12.0 + log(8.0)) - 12.0 * yb4 * (79.0 + log(32.0))) + 8.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0)) + yb2 * (-12.0 + log(8.0))
                    - 12.0 * yb4 * (79.0 + log(32.0))) - 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2) + 3.0 * yb5 * (-1.0 + log(128.0)))
                - 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2) + 3.0 * yb5 * (-1.0 + log(128.0))))
                + tb * xb2 * (2.0 * xb2 * yb * (-1.0 + yb2) * (2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2) + yb6 * (521.0 + 1080.0 * ln2))
                + 2.0 * xb10 * yb * (-1.0 + yb2) * (2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2) + yb6 * (521.0 + 1080.0 * ln2))
                + 4.0 * xb6 * (24.0 + 154.0 * yb + 225.0 * yb2 + 794.0 * yb4 - 668.0 * yb6 - 6930.0 * yb8 - 5733.0 * yb10 + yb3 * (941.0 - 312.0 * ln2) + 18.0 * yb5 * (173.0 + 344.0 * ln2) + 3.0 * yb11 * (-1271.0 + 552.0 * ln2) + 4.0 * yb9 * (-2041.0 + 564.0 * ln2)
                    - 8.0 * yb7 * (565.0 + 1224.0 * ln2)) + xb4 * (-16.0 - 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 - yb3 * (505.0 + 792.0 * ln2) + 3.0 * yb11 * (-493.0 + 1512.0 * ln2) - 16.0 * yb7 * (143.0 + 1602.0 * ln2)
                    + 6.0 * yb5 * (1415.0 + 2712.0 * ln2) + yb9 * (4054.0 + 5616.0 * ln2)) + xb8 * (-16.0 - 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 - yb3 * (505.0 + 792.0 * ln2) + 3.0 * yb11 * (-493.0 + 1512.0 * ln2)
                    - 16.0 * yb7 * (143.0 + 1602.0 * ln2) + 6.0 * yb5 * (1415.0 + 2712.0 * ln2) + yb9 * (4054.0 + 5616.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 - 2.0 * yb * (1.0 + log(16.0)) + yb2 * (-1.0 + ln256))
                + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 - 2.0 * yb * (1.0 + log(16.0)) + yb2 * (-1.0 + ln256)));
            const complex<double> logs9 = -4.0 * xb2 * (3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 2.0 * xb6 * (-8.0 - 52.0 * yb - 75.0 * yb2 - 290.0 * yb4 + 176.0 * yb6 + 2394.0 * yb8 + 1899.0 * yb10 + yb7 * (1664.0 - 2448.0 * ln2) - yb3 * (317.0 + 78.0 * ln2) + 18.0 * yb5 * (-75.0 + 86.0 * ln2) + 3.0 * yb11 * (463.0 + 138.0 * ln2)
                    + yb9 * (2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                + tb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (-32.0 - 208.0 * yb - 300.0 * yb2 - 1160.0 * yb4 + 704.0 * yb6 + 9576.0 * yb8 + 7596.0 * yb10 + yb7 * (6656.0 - 1692.0 * ln2) - yb3 * (1268.0 + 57.0 * ln2) + 18.0 * yb5 * (-300.0 + 59.0 * ln2) + 3.0 * yb11 * (1852.0 + 87.0 * ln2)
                    + 2.0 * yb9 * (5524.0 + 213.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2) + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2) + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0)) + 3.0 * yb5 * (1.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0))
                    + 3.0 * yb5 * (1.0 + log(128.0)))) - tb * xb2 * (4.0 * xb6 * (-24.0 - 154.0 * yb - 225.0 * yb2 - 794.0 * yb4 + 668.0 * yb6 + 6930.0 * yb8 + 5733.0 * yb10 + yb7 * (4520.0 - 9792.0 * ln2) - yb3 * (941.0 + 312.0 * ln2) + 18.0 * yb5 * (-173.0 + 344.0 * ln2)
                    + 3.0 * yb11 * (1271.0 + 552.0 * ln2) + 4.0 * yb9 * (2041.0 + 564.0 * ln2)) + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2)
                    + yb8 * (-381.0 + 648.0 * ln2) + yb6 * (-521.0 + 1080.0 * ln2)) + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2) + yb8 * (-381.0 + 648.0 * ln2)
                    + yb6 * (-521.0 + 1080.0 * ln2)) + xb4 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2) + 3.0 * yb11 * (493.0 + 1512.0 * ln2)
                    + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + xb8 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2)
                    + 3.0 * yb11 * (493.0 + 1512.0 * ln2) + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 8.0 * ln2) + yb2 * (1.0 + ln256))
                + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 8.0 * ln2) + yb2 * (1.0 + ln256)));
            const complex<double> logs10 = 4.0 * xb2 * (3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 2.0 * xb6 * (-8.0 - 52.0 * yb - 75.0 * yb2 - 290.0 * yb4 + 176.0 * yb6 + 2394.0 * yb8 + 1899.0 * yb10 + yb7 * (1664.0 - 2448.0 * ln2) - yb3 * (317.0 + 78.0 * ln2) + 18.0 * yb5 * (-75.0 + 86.0 * ln2) + 3.0 * yb11 * (463.0 + 138.0 * ln2)
                    + yb9 * (2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                - tb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (-32.0 - 208.0 * yb - 300.0 * yb2 - 1160.0 * yb4 + 704.0 * yb6 + 9576.0 * yb8 + 7596.0 * yb10 + yb7 * (6656.0 - 1692.0 * ln2) - yb3 * (1268.0 + 57.0 * ln2) + 18.0 * yb5 * (-300.0 + 59.0 * ln2) + 3.0 * yb11 * (1852.0 + 87.0 * ln2)
                    + 2.0 * yb9 * (5524.0 + 213.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2) + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2) + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0)) + 3.0 * yb5 * (1.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0))
                    + 3.0 * yb5 * (1.0 + log(128.0)))) + tb * xb2 * (4.0 * xb6 * (-24.0 - 154.0 * yb - 225.0 * yb2 - 794.0 * yb4 + 668.0 * yb6 + 6930.0 * yb8 + 5733.0 * yb10 + yb7 * (4520.0 - 9792.0 * ln2) - yb3 * (941.0 + 312.0 * ln2) + 18.0 * yb5 * (-173.0 + 344.0 * ln2)
                    + 3.0 * yb11 * (1271.0 + 552.0 * ln2) + 4.0 * yb9 * (2041.0 + 564.0 * ln2)) + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2)
                    + yb8 * (-381.0 + 648.0 * ln2) + yb6 * (-521.0 + 1080.0 * ln2)) + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2) + yb8 * (-381.0 + 648.0 * ln2)
                    + yb6 * (-521.0 + 1080.0 * ln2)) + xb4 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2) + 3.0 * yb11 * (493.0 + 1512.0 * ln2)
                    + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + xb8 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2)
                    + 3.0 * yb11 * (493.0 + 1512.0 * ln2) + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 8.0 * ln2) + yb2 * (1.0 + ln256))
                + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 8.0 * ln2) + yb2 * (1.0 + ln256)));
            const complex<double> logs11 = 4.0 * xb2 * (3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 2.0 * xb6 * (8.0 - 52.0 * yb + 75.0 * yb2 + 290.0 * yb4 - 176.0 * yb6 - 2394.0 * yb8 - 1899.0 * yb10 + yb7 * (1664.0 - 2448.0 * ln2) - yb3 * (317.0 + 78.0 * ln2) + 18.0 * yb5 * (-75.0 + 86.0 * ln2) + 3.0 * yb11 * (463.0 + 138.0 * ln2)
                    + yb9 * (2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                + vb2 * (-3.0 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 3.0 * xb16 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 2.0 * xb8 * (32.0 - 208.0 * yb + 300.0 * yb2 + 1160.0 * yb4 - 704.0 * yb6 - 9576.0 * yb8 - 7596.0 * yb10 + yb7 * (6656.0 - 1692.0 * ln2) - yb3 * (1268.0 + 57.0 * ln2)
                    + 18.0 * yb5 * (-300.0 + 59.0 * ln2) + 3.0 * yb11 * (1852.0 + 87.0 * ln2) + 2.0 * yb9 * (5524.0 + 213.0 * ln2)) + 8.0 * xb6 * yb * (-1.0 + yb2) * (2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2)
                    + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0))) + 8.0 * xb10 * yb * (-1.0 + yb2) * (2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2) + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0)))
                - 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0)) + 3.0 * yb5 * (1.0 + log(128.0))) - 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0))
                    + 3.0 * yb5 * (1.0 + log(128.0)))) - vb * xb2 * (4.0 * xb6 * (24.0 - 154.0 * yb + 225.0 * yb2 + 794.0 * yb4 - 668.0 * yb6 - 6930.0 * yb8 - 5733.0 * yb10 + yb7 * (4520.0 - 9792.0 * ln2) - yb3 * (941.0 + 312.0 * ln2) + 18.0 * yb5 * (-173.0 + 344.0 * ln2)
                    + 3.0 * yb11 * (1271.0 + 552.0 * ln2) + 4.0 * yb9 * (2041.0 + 564.0 * ln2)) + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2)
                    + yb8 * (-381.0 + 648.0 * ln2) + yb6 * (-521.0 + 1080.0 * ln2)) + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2) + yb8 * (-381.0 + 648.0 * ln2)
                    + yb6 * (-521.0 + 1080.0 * ln2)) + xb4 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2) + 3.0 * yb11 * (493.0 + 1512.0 * ln2)
                    + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + xb8 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2)
                    + 3.0 * yb11 * (493.0 + 1512.0 * ln2) + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + 3.0 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + ln256) + yb2 * (1.0 + ln256))
                + 3.0 * xb12 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + ln256) + yb2 * (1.0 + ln256)));
            const complex<double> logs12 = -4.0 * xb2 * (3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 6.0 * ln2) + 8.0 * yb3 * (-1.0 + 18.0 * ln2) + yb5 * (-3.0 + 54.0 * ln2))
                + 2.0 * xb6 * (8.0 - 52.0 * yb + 75.0 * yb2 + 290.0 * yb4 - 176.0 * yb6 - 2394.0 * yb8 - 1899.0 * yb10 + yb7 * (1664.0 - 2448.0 * ln2) - yb3 * (317.0 + 78.0 * ln2) + 18.0 * yb5 * (-75.0 + 86.0 * ln2) + 3.0 * yb11 * (463.0 + 138.0 * ln2)
                    + yb9 * (2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + 3.0 * yb2 * (8.0 + 33.0 * ln2) + yb8 * (708.0 + 567.0 * ln2) - 3.0 * yb4 * (-632.0 + 645.0 * ln2) + yb6 * (952.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                + vb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (32.0 - 208.0 * yb + 300.0 * yb2 + 1160.0 * yb4 - 704.0 * yb6 - 9576.0 * yb8 - 7596.0 * yb10 + yb7 * (6656.0 - 1692.0 * ln2) - yb3 * (1268.0 + 57.0 * ln2) + 18.0 * yb5 * (-300.0 + 59.0 * ln2) + 3.0 * yb11 * (1852.0 + 87.0 * ln2)
                    + 2.0 * yb9 * (5524.0 + 213.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2) + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (2.0 - 76.0 * yb3 - 216.0 * yb5 + 36.0 * yb7 + yb4 * (948.0 - 60.0 * ln2) + yb6 * (476.0 + 39.0 * ln2) + yb2 * (12.0 + log(8.0)) + 6.0 * yb8 * (59.0 + log(8.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0)) + 3.0 * yb5 * (1.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + yb3 * (8.0 + 80.0 * ln2) - yb * (3.0 + log(32.0))
                    + 3.0 * yb5 * (1.0 + log(128.0)))) + vb * xb2 * (4.0 * xb6 * (24.0 - 154.0 * yb + 225.0 * yb2 + 794.0 * yb4 - 668.0 * yb6 - 6930.0 * yb8 - 5733.0 * yb10 + yb7 * (4520.0 - 9792.0 * ln2) - yb3 * (941.0 + 312.0 * ln2) + 18.0 * yb5 * (-173.0 + 344.0 * ln2)
                    + 3.0 * yb11 * (1271.0 + 552.0 * ln2) + 4.0 * yb9 * (2041.0 + 564.0 * ln2)) + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2)
                    + yb8 * (-381.0 + 648.0 * ln2) + yb6 * (-521.0 + 1080.0 * ln2)) + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 72.0 * ln2) - 3.0 * yb4 * (283.0 + 600.0 * ln2) + yb8 * (-381.0 + 648.0 * ln2)
                    + yb6 * (-521.0 + 1080.0 * ln2)) + xb4 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2) + 3.0 * yb11 * (493.0 + 1512.0 * ln2)
                    + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + xb8 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb7 * (2288.0 - 25632.0 * ln2) + yb3 * (505.0 - 792.0 * ln2)
                    + 3.0 * yb11 * (493.0 + 1512.0 * ln2) + 6.0 * yb5 * (-1415.0 + 2712.0 * ln2) + yb9 * (-4054.0 + 5616.0 * ln2)) + 3.0 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + ln256) + yb2 * (1.0 + ln256))
                + 3.0 * xb12 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + ln256) + yb2 * (1.0 + ln256)));
            const complex<double> logs13 = 4.0 * xb2 * (2.0 * xb6 * (-8.0 + 52.0 * yb - 75.0 * yb2 - 290.0 * yb4 + 176.0 * yb6 + 2394.0 * yb8 + 1899.0 * yb10 + yb3 * (317.0 - 78.0 * ln2) + 18.0 * yb5 * (75.0 + 86.0 * ln2) + 3.0 * yb11 * (-463.0 + 138.0 * ln2) - 16.0 * yb7 * (104.0 + 153.0 * ln2)
                    + yb9 * (-2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (-4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (-4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + 3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2) - 3.0 * yb * (1.0 + ln4)) + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2)
                    - 3.0 * yb * (1.0 + ln4)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                - vb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (-32.0 + 208.0 * yb - 300.0 * yb2 - 1160.0 * yb4 + 704.0 * yb6 + 9576.0 * yb8 + 7596.0 * yb10 + yb3 * (1268.0 - 57.0 * ln2) + 18.0 * yb5 * (300.0 + 59.0 * ln2) + 3.0 * yb11 * (-1852.0 + 87.0 * ln2) + 2.0 * yb9 * (-5524.0 + 213.0 * ln2)
                    - 4.0 * yb7 * (1664.0 + 423.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (-2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0)) + yb2 * (-12.0 + log(8.0)) - 12.0 * yb4 * (79.0 + log(32.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0)) + yb2 * (-12.0 + log(8.0)) - 12.0 * yb4 * (79.0 + log(32.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2) + 3.0 * yb5 * (-1.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2)
                    + 3.0 * yb5 * (-1.0 + log(128.0)))) + vb * xb2 * (-2.0 * xb2 * yb * (-1.0 + yb2) * (2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2)
                    + yb6 * (521.0 + 1080.0 * ln2)) - 2.0 * xb10 * yb * (-1.0 + yb2) * (2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2) + yb6 * (521.0 + 1080.0 * ln2))
                - 4.0 * xb6 * (-24.0 + 154.0 * yb - 225.0 * yb2 - 794.0 * yb4 + 668.0 * yb6 + 6930.0 * yb8 + 5733.0 * yb10 + yb3 * (941.0 - 312.0 * ln2) + 18.0 * yb5 * (173.0 + 344.0 * ln2) + 3.0 * yb11 * (-1271.0 + 552.0 * ln2) + 4.0 * yb9 * (-2041.0 + 564.0 * ln2)
                    - 8.0 * yb7 * (565.0 + 1224.0 * ln2)) + xb4 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb11 * (1479.0 - 4536.0 * ln2) + yb3 * (505.0 + 792.0 * ln2) + 16.0 * yb7 * (143.0 + 1602.0 * ln2)
                    - 6.0 * yb5 * (1415.0 + 2712.0 * ln2) - 2.0 * yb9 * (2027.0 + 2808.0 * ln2)) + xb8 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb11 * (1479.0 - 4536.0 * ln2) + yb3 * (505.0 + 792.0 * ln2)
                    + 16.0 * yb7 * (143.0 + 1602.0 * ln2) - 6.0 * yb5 * (1415.0 + 2712.0 * ln2) - 2.0 * yb9 * (2027.0 + 2808.0 * ln2)) - 3.0 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 + yb2 * (-1.0 + ln256) + yb * (2.0 + ln256))
                - 3.0 * xb12 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 + yb2 * (-1.0 + ln256) + yb * (2.0 + ln256)));
            const complex<double> logs14 = -4.0 * xb2 * (2.0 * xb6 * (-8.0 + 52.0 * yb - 75.0 * yb2 - 290.0 * yb4 + 176.0 * yb6 + 2394.0 * yb8 + 1899.0 * yb10 + yb3 * (317.0 - 78.0 * ln2) + 18.0 * yb5 * (75.0 + 86.0 * ln2) + 3.0 * yb11 * (-463.0 + 138.0 * ln2) - 16.0 * yb7 * (104.0 + 153.0 * ln2)
                    + yb9 * (-2762.0 + 564.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (-4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + xb8 * yb * (-1.0 + yb2) * (-4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + 3.0 * yb2 * (-8.0 + 33.0 * ln2) + yb8 * (-708.0 + 567.0 * ln2) - 3.0 * yb4 * (632.0 + 645.0 * ln2) + yb6 * (-952.0 + 1269.0 * ln2))
                + 3.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2) - 3.0 * yb * (1.0 + ln4)) + 3.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 + 8.0 * yb3 * (1.0 + 18.0 * ln2) + yb5 * (3.0 + 54.0 * ln2)
                    - 3.0 * yb * (1.0 + ln4)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)))
                + vb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (-32.0 + 208.0 * yb - 300.0 * yb2 - 1160.0 * yb4 + 704.0 * yb6 + 9576.0 * yb8 + 7596.0 * yb10 + yb3 * (1268.0 - 57.0 * ln2) + 18.0 * yb5 * (300.0 + 59.0 * ln2) + 3.0 * yb11 * (-1852.0 + 87.0 * ln2) + 2.0 * yb9 * (-5524.0 + 213.0 * ln2)
                    - 4.0 * yb7 * (1664.0 + 423.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (-2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0)) + yb2 * (-12.0 + log(8.0)) - 12.0 * yb4 * (79.0 + log(32.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 76.0 * yb3 + 216.0 * yb5 - 36.0 * yb7 + yb6 * (-476.0 + 39.0 * ln2) + 6.0 * yb8 * (-59.0 + log(8.0)) + yb2 * (-12.0 + log(8.0)) - 12.0 * yb4 * (79.0 + log(32.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2) + 3.0 * yb5 * (-1.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 + yb * (3.0 - 5.0 * ln2) + yb3 * (-8.0 + 80.0 * ln2)
                    + 3.0 * yb5 * (-1.0 + log(128.0)))) + vb * xb2 * (2.0 * xb2 * yb * (-1.0 + yb2) * (2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2)
                    + yb6 * (521.0 + 1080.0 * ln2)) + 2.0 * xb10 * yb * (-1.0 + yb2) * (2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb4 * (849.0 - 1800.0 * ln2) + yb2 * (39.0 + 72.0 * ln2) + yb8 * (381.0 + 648.0 * ln2) + yb6 * (521.0 + 1080.0 * ln2))
                + 4.0 * xb6 * (-24.0 + 154.0 * yb - 225.0 * yb2 - 794.0 * yb4 + 668.0 * yb6 + 6930.0 * yb8 + 5733.0 * yb10 + yb3 * (941.0 - 312.0 * ln2) + 18.0 * yb5 * (173.0 + 344.0 * ln2) + 3.0 * yb11 * (-1271.0 + 552.0 * ln2) + 4.0 * yb9 * (-2041.0 + 564.0 * ln2)
                    - 8.0 * yb7 * (565.0 + 1224.0 * ln2)) + xb4 * (16.0 - 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 - yb3 * (505.0 + 792.0 * ln2) + 3.0 * yb11 * (-493.0 + 1512.0 * ln2) - 16.0 * yb7 * (143.0 + 1602.0 * ln2)
                    + 6.0 * yb5 * (1415.0 + 2712.0 * ln2) + yb9 * (4054.0 + 5616.0 * ln2)) + xb8 * (16.0 - 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 - yb3 * (505.0 + 792.0 * ln2) + 3.0 * yb11 * (-493.0 + 1512.0 * ln2)
                    - 16.0 * yb7 * (143.0 + 1602.0 * ln2) + 6.0 * yb5 * (1415.0 + 2712.0 * ln2) + yb9 * (4054.0 + 5616.0 * ln2)) + 3.0 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 + yb2 * (-1.0 + ln256) + yb * (2.0 + ln256))
                + 3.0 * xb12 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (-1.0 + yb2 * (-1.0 + ln256) + yb * (2.0 + ln256)));
            const complex<double> logs15 = 4.0 * tb * xb2 * (16.0 * vb * xb2 * power_of<2>(1.0i + xb) * yb * (6.0 * 1.0i * xb * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) * ln2 + 6.0 * 1.0i * xb9 * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) * ln2
                    - 8.0 * 1.0i * xb3 * (-1.0 + yb2) * (yb4 * (18.0 - 314.0 * ln2) + yb2 * (3.0 - 5.0 * ln2) + ln2 + 327.0 * yb8 * ln2 + yb6 * (27.0 + 455.0 * ln2))
                    - 8.0 * 1.0i * xb7 * (-1.0 + yb2) * (yb4 * (18.0 - 314.0 * ln2) + yb2 * (3.0 - 5.0 * ln2) + ln2 + 327.0 * yb8 * ln2 + yb6 * (27.0 + 455.0 * ln2))
                    + 2.0 * xb4 * (yb4 * (46.0 - 621.0 * ln2) + 34.0 * ln2 + 108.0 * yb10 * (2.0 + 17.0 * ln2) + yb2 * (14.0 + 111.0 * ln2) - 3.0 * yb8 * (66.0 + 763.0 * ln2) - yb6 * (78.0 + 3167.0 * ln2))
                    - 2.0 * xb6 * (yb4 * (46.0 - 621.0 * ln2) + 34.0 * ln2 + 108.0 * yb10 * (2.0 + 17.0 * ln2) + yb2 * (14.0 + 111.0 * ln2) - 3.0 * yb8 * (66.0 + 763.0 * ln2) - yb6 * (78.0 + 3167.0 * ln2))
                    - 4.0 * 1.0i * xb5 * (yb6 * (60.0 - 4851.0 * ln2) - 36.0 * ln2 - yb2 * (20.0 + 543.0 * ln2) + 6.0 * yb10 * (-36.0 + 1309.0 * ln2) - yb4 * (76.0 + 1857.0 * ln2) + yb8 * (252.0 + 3529.0 * ln2))
                    - yb2 * (-51.0 * yb2 * ln2 + 63.0 * yb4 * ln2 - 33.0 * yb6 * ln2 + 18.0 * yb8 * ln2 + log(8.0)) + xb10 * yb2 * (-51.0 * yb2 * ln2 + 63.0 * yb4 * ln2 - 33.0 * yb6 * ln2 + 18.0 * yb8 * ln2 + log(8.0))
                    + xb2 * (yb6 * (36.0 - 125.0 * ln2) + yb2 * (12.0 - 37.0 * ln2) - 1338.0 * yb10 * ln2 - yb8 * (108.0 + 385.0 * ln2) + 3.0 * yb4 * (20.0 + 627.0 * ln2) + log(16.0))
                    + xb8 * (-1.0 + yb2) * (yb2 * (12.0 - 33.0 * ln2) + 1338.0 * yb8 * ln2 + 24.0 * yb4 * (3.0 + 77.0 * ln2) + yb6 * (108.0 + 1723.0 * ln2) + log(16.0)))
                - 4.0 * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 16.0 * 1.0i * xb3 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2
                    + 16.0 * 1.0i * xb9 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2 + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * ln2
                    - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * ln2 - xb4 * (yb5 * (544.0 - 22620.0 * ln2) + yb3 * (128.0 - 339.0 * ln2) + 16.0 * ln2 + 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2
                    + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 - 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 6.0 * yb7 * (-16.0 + 2909.0 * ln2) + yb11 * (864.0 + 3741.0 * ln2))
                    - xb8 * (yb5 * (544.0 - 22620.0 * ln2) + yb3 * (128.0 - 339.0 * ln2) + 16.0 * ln2 + 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2
                    - 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 6.0 * yb7 * (-16.0 + 2909.0 * ln2) + yb11 * (864.0 + 3741.0 * ln2)) + 4.0 * xb6 * (yb7 * (84.0 - 27874.0 * ln2) - 8.0 * ln2 - 22.0 * yb * ln2 - 75.0 * yb2 * ln2 - 366.0 * yb4 * ln2 + 36.0 * yb6 * ln2
                    + 2646.0 * yb8 * ln2 + 1863.0 * yb10 * ln2 + 27.0 * yb11 * (-16.0 + 1235.0 * ln2) - 4.0 * yb5 * (53.0 + 1596.0 * ln2) - yb3 * (52.0 + 1623.0 * ln2) + yb9 * (612.0 + 6654.0 * ln2))
                    + xb2 * yb * (-4.0 * ln2 - 164.0 * yb3 * ln2 - 328.0 * yb5 * ln2 + 612.0 * yb7 * ln2 - 126.0 * yb9 * ln2 + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2) - 4.0 * yb6 * (18.0 + 391.0 * ln2)
                    - yb2 * (24.0 + ln4) + yb * ln64) + xb10 * yb * (-4.0 * ln2 - 164.0 * yb3 * ln2 - 328.0 * yb5 * ln2 + 612.0 * yb7 * ln2 - 126.0 * yb9 * ln2 + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2)
                    - 4.0 * yb6 * (18.0 + 391.0 * ln2) - yb2 * (24.0 + ln4) + yb * ln64) + (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0)
                    + yb2 * ln64) + xb12 * (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0) + yb2 * ln64))
                + vb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 + 4.0 * xb6 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2) + 32.0 * ln2 - 223.0 * yb * ln2
                    + 294.0 * yb2 * ln2 + 1818.0 * yb4 * ln2 + 534.0 * yb6 * ln2 - 11826.0 * yb8 * ln2 - 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2))
                    + 4.0 * xb10 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2) + 32.0 * ln2 - 223.0 * yb * ln2 + 294.0 * yb2 * ln2 + 1818.0 * yb4 * ln2 + 534.0 * yb6 * ln2 - 11826.0 * yb8 * ln2
                    - 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2)) - 2.0 * xb8 * (yb7 * (240.0 - 128218.0 * ln2) - 144.0 * ln2 + 824.0 * yb * ln2 - 1353.0 * yb2 * ln2 - 6430.0 * yb4 * ln2 + 952.0 * yb6 * ln2
                    + 47070.0 * yb8 * ln2 + 33633.0 * yb10 * ln2 - yb3 * (80.0 + 1599.0 * ln2) - 12.0 * yb9 * (-84.0 + 2167.0 * ln2) - 4.0 * yb5 * (76.0 + 2895.0 * ln2) + 9.0 * yb11 * (-96.0 + 12137.0 * ln2))
                    - 4.0 * xb4 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) + 18.0 * yb * ln2 + 51.0 * yb2 * ln2 - 186.0 * yb4 * ln2 - 756.0 * yb6 * ln2 + 54.0 * yb8 * ln2 - 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2)
                    + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2) + log(16.0)) - 4.0 * xb12 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) + 18.0 * yb * ln2 + 51.0 * yb2 * ln2 - 186.0 * yb4 * ln2 - 756.0 * yb6 * ln2
                    + 54.0 * yb8 * ln2 - 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2) + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2) + log(16.0))
                    + 4.0 * xb2 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 + 32.0 * yb3 * ln2 + 54.0 * yb5 * ln2 + 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2 + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2) + yb * ln64)
                    + 4.0 * xb14 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 + 32.0 * yb3 * ln2 + 54.0 * yb5 * ln2 + 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2 + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2) + yb * ln64)
                    + (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0) + yb2 * ln64)
                    + xb16 * (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0) + yb2 * ln64)))
                + tb2 * ((-vb2) * power_of<2>(1.0i + xb - 1.0i * xb2 - xb3) * yb * (6.0 * 1.0i * xb * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) * ln2 + 6.0 * 1.0i * xb13 * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) * ln2
                    + 3.0 * xb14 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 - 3.0 * xb2 * yb2 * (-1.0 + yb2) * (4.0 - 50.0 * ln2 + 165.0 * yb6 * ln2 + 4.0 * yb4 * (9.0 + 53.0 * ln2) + yb2 * (24.0 + 73.0 * ln2))
                    + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (4.0 - 50.0 * ln2 + 165.0 * yb6 * ln2 + 4.0 * yb4 * (9.0 + 53.0 * ln2) + yb2 * (24.0 + 73.0 * ln2)) - 4.0 * 1.0i * xb3 * yb2 * (-1.0 + yb2) * (6.0 + yb2 * (36.0 - 1399.0 * ln2) - 76.0 * ln2 + 273.0 * yb6 * ln2
                    + yb4 * (54.0 + 290.0 * ln2)) - 4.0 * 1.0i * xb11 * yb2 * (-1.0 + yb2) * (6.0 + yb2 * (36.0 - 1399.0 * ln2) - 76.0 * ln2 + 273.0 * yb6 * ln2 + yb4 * (54.0 + 290.0 * ln2))
                    + xb4 * yb2 * (-1.0 + yb2) * (-4.0 + yb2 * (24.0 - 12455.0 * ln2) + 22.0 * ln2 + 4.0 * yb4 * (63.0 + 55.0 * ln2) + 3.0 * yb6 * (144.0 + 407.0 * ln2))
                    - xb10 * yb2 * (-1.0 + yb2) * (-4.0 + yb2 * (24.0 - 12455.0 * ln2) + 22.0 * ln2 + 4.0 * yb4 * (63.0 + 55.0 * ln2) + 3.0 * yb6 * (144.0 + 407.0 * ln2))
                    + 8.0 * 1.0i * xb7 * (yb6 * (78.0 - 11161.0 * ln2) + yb8 * (198.0 - 10415.0 * ln2) + 224.0 * ln2 + 3.0 * yb10 * (-72.0 + 203.0 * ln2) + 2.0 * yb2 * (-7.0 + 510.0 * ln2) + yb4 * (-46.0 + 3339.0 * ln2))
                    + xb6 * (yb8 * (1080.0 - 44481.0 * ln2) + yb6 * (504.0 - 6451.0 * ln2) + 896.0 * ln2 - 9.0 * yb10 * (144.0 + 3823.0 * ln2) + 3.0 * yb4 * (-72.0 + 4295.0 * ln2) + yb2 * (-72.0 + 6022.0 * ln2))
                    + xb8 * (yb2 * (72.0 - 6022.0 * ln2) - 896.0 * ln2 + 9.0 * yb10 * (144.0 + 3823.0 * ln2) - 3.0 * yb4 * (-72.0 + 4295.0 * ln2) + yb6 * (-504.0 + 6451.0 * ln2) + 3.0 * yb8 * (-360.0 + 14827.0 * ln2))
                    - 2.0 * 1.0i * xb5 * yb2 * (-1.0 + yb2) * (16.0 + 718.0 * ln2 + 4.0 * yb4 * (-36.0 + 5107.0 * ln2) + 3.0 * yb6 * (-144.0 + 5795.0 * ln2) + yb2 * (48.0 + 22861.0 * ln2))
                    - 2.0 * 1.0i * xb9 * yb2 * (-1.0 + yb2) * (16.0 + 718.0 * ln2 + 4.0 * yb4 * (-36.0 + 5107.0 * ln2) + 3.0 * yb6 * (-144.0 + 5795.0 * ln2) + yb2 * (48.0 + 22861.0 * ln2))
                    + yb2 * (9.0 * yb2 * ln2 - 15.0 * yb4 * ln2 + 27.0 * yb6 * ln2 - 27.0 * yb8 * ln2 + ln64)) + 4.0 * xb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2
                    + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) - ln2 - 4.0 * yb * ln2 + yb2 * ln2 + 9.0 * yb4 * ln2 - 9.0 * yb6 * ln2
                    + 39.0 * yb7 * ln2 + yb5 * (9.0 + 86.0 * ln2)) + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) - ln2 - 4.0 * yb * ln2 + yb2 * ln2 + 9.0 * yb4 * ln2 - 9.0 * yb6 * ln2 + 39.0 * yb7 * ln2
                    + yb5 * (9.0 + 86.0 * ln2)) + 4.0 * xb6 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) - 16.0 * ln2 - 112.0 * yb * ln2 - 147.0 * yb2 * ln2 - 890.0 * yb4 * ln2 - 232.0 * yb6 * ln2 + 5850.0 * yb8 * ln2 + 3627.0 * yb10 * ln2
                    + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2)) + 4.0 * xb10 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) - 16.0 * ln2 - 112.0 * yb * ln2 - 147.0 * yb2 * ln2 - 890.0 * yb4 * ln2
                    - 232.0 * yb6 * ln2 + 5850.0 * yb8 * ln2 + 3627.0 * yb10 * ln2 + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2))
                    - 2.0 * xb8 * (yb7 * (120.0 - 65749.0 * ln2) + yb9 * (504.0 - 15767.0 * ln2) + 64.0 * ln2 + 464.0 * yb * ln2 + 600.0 * yb2 * ln2 + 2928.0 * yb4 * ln2 - 288.0 * yb6 * ln2 - 21168.0 * yb8 * ln2 - 14904.0 * yb10 * ln2
                    - 2.0 * yb3 * (20.0 + 239.0 * ln2) - yb5 * (152.0 + 4461.0 * ln2) + yb11 * (-432.0 + 53223.0 * ln2)) - 4.0 * xb4 * yb * (-1.0 + yb2) * (-158.0 * yb3 * ln2 - 486.0 * yb5 * ln2 + 126.0 * yb7 * ln2 + 2.0 * yb2 * (-5.0 + 63.0 * ln2)
                    + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0) + yb * ln64) - 4.0 * xb12 * yb * (-1.0 + yb2) * (-158.0 * yb3 * ln2 - 486.0 * yb5 * ln2 + 126.0 * yb7 * ln2
                    + 2.0 * yb2 * (-5.0 + 63.0 * ln2) + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0) + yb * ln64))
                - 4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 - 4.0 * xb4 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) - 4.0 * ln2 + 18.0 * yb * ln2 - 51.0 * yb2 * ln2 + 186.0 * yb4 * ln2
                    + 756.0 * yb6 * ln2 - 54.0 * yb8 * ln2 + 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2) + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2))
                    - 4.0 * xb12 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) - 4.0 * ln2 + 18.0 * yb * ln2 - 51.0 * yb2 * ln2 + 186.0 * yb4 * ln2 + 756.0 * yb6 * ln2 - 54.0 * yb8 * ln2 + 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2)
                    + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2)) + 4.0 * xb2 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 - 6.0 * yb * ln2 - 32.0 * yb3 * ln2 - 54.0 * yb5 * ln2 - 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2
                    + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2)) + 4.0 * xb14 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 - 6.0 * yb * ln2 - 32.0 * yb3 * ln2 - 54.0 * yb5 * ln2 - 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2
                    + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2)) + 4.0 * xb6 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2) - 32.0 * ln2 - 223.0 * yb * ln2 - 294.0 * yb2 * ln2 - 1818.0 * yb4 * ln2
                    - 534.0 * yb6 * ln2 + 11826.0 * yb8 * ln2 + 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2)) + 4.0 * xb10 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2)
                    - 32.0 * ln2 - 223.0 * yb * ln2 - 294.0 * yb2 * ln2 - 1818.0 * yb4 * ln2 - 534.0 * yb6 * ln2 + 11826.0 * yb8 * ln2 + 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2))
                    - 2.0 * xb8 * (yb7 * (240.0 - 128218.0 * ln2) + 144.0 * ln2 + 824.0 * yb * ln2 + 1353.0 * yb2 * ln2 + 6430.0 * yb4 * ln2 - 952.0 * yb6 * ln2 - 47070.0 * yb8 * ln2 - 33633.0 * yb10 * ln2 - yb3 * (80.0 + 1599.0 * ln2)
                    - 12.0 * yb9 * (-84.0 + 2167.0 * ln2) - 4.0 * yb5 * (76.0 + 2895.0 * ln2) + 9.0 * yb11 * (-96.0 + 12137.0 * ln2)) + yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2
                    + 45.0 * yb9 * ln2 - log(8.0) - yb * log(8.0) + yb2 * ln64) + xb16 * yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2 + 45.0 * yb9 * ln2 - log(8.0)
                    - yb * log(8.0) + yb2 * ln64))) + 4.0 * xb2 * (vb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2
                    + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) + ln2 - 4.0 * yb * ln2 - yb2 * ln2 - 9.0 * yb4 * ln2 + 9.0 * yb6 * ln2
                    + 39.0 * yb7 * ln2 + yb5 * (9.0 + 86.0 * ln2)) + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) + ln2 - 4.0 * yb * ln2 - yb2 * ln2 - 9.0 * yb4 * ln2 + 9.0 * yb6 * ln2 + 39.0 * yb7 * ln2
                    + yb5 * (9.0 + 86.0 * ln2)) + 4.0 * xb6 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) + 16.0 * ln2 - 112.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2
                    + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2)) + 4.0 * xb10 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) + 16.0 * ln2 - 112.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2
                    + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2))
                    - 2.0 * xb8 * (yb7 * (120.0 - 65749.0 * ln2) + yb9 * (504.0 - 15767.0 * ln2) - 64.0 * ln2 + 464.0 * yb * ln2 - 600.0 * yb2 * ln2 - 2928.0 * yb4 * ln2 + 288.0 * yb6 * ln2 + 21168.0 * yb8 * ln2 + 14904.0 * yb10 * ln2
                    - 2.0 * yb3 * (20.0 + 239.0 * ln2) - yb5 * (152.0 + 4461.0 * ln2) + yb11 * (-432.0 + 53223.0 * ln2)) - 4.0 * xb4 * yb * (-1.0 + yb2) * (-6.0 * yb * ln2 + 158.0 * yb3 * ln2 + 486.0 * yb5 * ln2 - 126.0 * yb7 * ln2 + 2.0 * yb2 * (-5.0 + 63.0 * ln2)
                    + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0)) - 4.0 * xb12 * yb * (-1.0 + yb2) * (-6.0 * yb * ln2 + 158.0 * yb3 * ln2 + 486.0 * yb5 * ln2 - 126.0 * yb7 * ln2
                    + 2.0 * yb2 * (-5.0 + 63.0 * ln2) + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0)))
                - 4.0 * xb2 * power_of<2>(1.0i + xb) * yb * (-1.0 + yb2) * (6.0 * 1.0i * xb * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) * ln2 + 6.0 * 1.0i * xb9 * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) * ln2 - 3.0 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6) * ln2
                    - 3.0 * xb2 * yb2 * (1.0 + 3.0 * yb2) * (4.0 - 22.0 * ln2 + 69.0 * yb4 * ln2 + yb2 * (12.0 + 61.0 * ln2)) + 3.0 * xb8 * yb2 * (1.0 + 3.0 * yb2) * (4.0 - 22.0 * ln2 + 69.0 * yb4 * ln2 + yb2 * (12.0 + 61.0 * ln2))
                    - 8.0 * 1.0i * xb3 * yb2 * (3.0 + yb2 * (18.0 - 788.0 * ln2) - 11.0 * ln2 + 150.0 * yb6 * ln2 + yb4 * (27.0 + 217.0 * ln2)) - 8.0 * 1.0i * xb7 * yb2 * (3.0 + yb2 * (18.0 - 788.0 * ln2) - 11.0 * ln2 + 150.0 * yb6 * ln2 + yb4 * (27.0 + 217.0 * ln2))
                    + 2.0 * xb4 * (1.0 + 3.0 * yb2) * (16.0 * ln2 + 2.0 * yb2 * (-7.0 + 82.0 * ln2) + yb4 * (-18.0 + 755.0 * ln2) + yb6 * (72.0 + 957.0 * ln2)) - 2.0 * xb6 * (1.0 + 3.0 * yb2) * (16.0 * ln2 + 2.0 * yb2 * (-7.0 + 82.0 * ln2) + yb4 * (-18.0 + 755.0 * ln2)
                    + yb6 * (72.0 + 957.0 * ln2)) - 4.0 * 1.0i * xb5 * (-16.0 * ln2 + 10.0 * yb2 * (2.0 + 21.0 * ln2) + yb4 * (96.0 + 717.0 * ln2) + 4.0 * yb6 * (9.0 + 1808.0 * ln2) + 3.0 * yb8 * (-72.0 + 2155.0 * ln2))
                    + xb10 * yb2 * (15.0 * yb2 * ln2 + 27.0 * yb6 * ln2 + ln64)) + 4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2
                    - 16.0 * 1.0i * xb3 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2 + 16.0 * 1.0i * xb9 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2 + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * 
                    ln2 - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * ln2 + xb4 * (yb7 * (96.0 - 17454.0 * ln2) + 16.0 * ln2 - 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2
                    - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 + yb3 * (-128.0 + 339.0 * ln2) - 3.0 * yb11 * (288.0 + 1247.0 * ln2) + 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 4.0 * yb5 * (-136.0 + 5655.0 * ln2))
                    + xb8 * (yb7 * (96.0 - 17454.0 * ln2) + 16.0 * ln2 - 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 + yb3 * (-128.0 + 339.0 * ln2)
                    - 3.0 * yb11 * (288.0 + 1247.0 * ln2) + 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 4.0 * yb5 * (-136.0 + 5655.0 * ln2)) + xb2 * yb * (-4.0 * ln2 - 6.0 * yb * ln2 + 164.0 * yb3 * ln2 + 328.0 * yb5 * ln2 - 612.0 * yb7 * ln2 + 126.0 * yb9 * ln2
                    + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2) - 4.0 * yb6 * (18.0 + 391.0 * ln2) - yb2 * (24.0 + ln4))
                    + xb10 * yb * (-4.0 * ln2 - 6.0 * yb * ln2 + 164.0 * yb3 * ln2 + 328.0 * yb5 * ln2 - 612.0 * yb7 * ln2 + 126.0 * yb9 * ln2 + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2)
                    - 4.0 * yb6 * (18.0 + 391.0 * ln2) - yb2 * (24.0 + ln4)) + yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2 + 45.0 * yb9 * ln2 - log(8.0) - yb * log(8.0)
                    + yb2 * ln64) + xb12 * yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2 + 45.0 * yb9 * ln2 - log(8.0) - yb * log(8.0) + yb2 * ln64)
                    + 4.0 * xb6 * (yb7 * (84.0 - 27874.0 * ln2) - 22.0 * yb * ln2 + 75.0 * yb2 * ln2 + 366.0 * yb4 * ln2 - 36.0 * yb6 * ln2 - 2646.0 * yb8 * ln2 - 1863.0 * yb10 * ln2 + 27.0 * yb11 * (-16.0 + 1235.0 * ln2) - 4.0 * yb5 * (53.0 + 1596.0 * ln2)
                    - yb3 * (52.0 + 1623.0 * ln2) + yb9 * (612.0 + 6654.0 * ln2) + ln256)));
            const complex<double> logs16 = 4.0 * tb * xb2 * (16.0 * vb * xb2 * power_of<2>(-1.0i + xb) * yb * (-6.0 * 1.0i * xb * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) * ln2 - 6.0 * 1.0i * xb9 * yb2 * (3.0 - 7.0 * yb2 + 27.0 * yb4 - 29.0 * yb6 + 6.0 * yb8) * ln2
                    + 8.0 * 1.0i * xb3 * (-1.0 + yb2) * (yb4 * (18.0 - 314.0 * ln2) + yb2 * (3.0 - 5.0 * ln2) + ln2 + 327.0 * yb8 * ln2 + yb6 * (27.0 + 455.0 * ln2))
                    + 8.0 * 1.0i * xb7 * (-1.0 + yb2) * (yb4 * (18.0 - 314.0 * ln2) + yb2 * (3.0 - 5.0 * ln2) + ln2 + 327.0 * yb8 * ln2 + yb6 * (27.0 + 455.0 * ln2))
                    + 2.0 * xb4 * (yb4 * (46.0 - 621.0 * ln2) + 34.0 * ln2 + 108.0 * yb10 * (2.0 + 17.0 * ln2) + yb2 * (14.0 + 111.0 * ln2) - 3.0 * yb8 * (66.0 + 763.0 * ln2) - yb6 * (78.0 + 3167.0 * ln2))
                    - 2.0 * xb6 * (yb4 * (46.0 - 621.0 * ln2) + 34.0 * ln2 + 108.0 * yb10 * (2.0 + 17.0 * ln2) + yb2 * (14.0 + 111.0 * ln2) - 3.0 * yb8 * (66.0 + 763.0 * ln2) - yb6 * (78.0 + 3167.0 * ln2))
                    + 4.0 * 1.0i * xb5 * (yb6 * (60.0 - 4851.0 * ln2) - 36.0 * ln2 - yb2 * (20.0 + 543.0 * ln2) + 6.0 * yb10 * (-36.0 + 1309.0 * ln2) - yb4 * (76.0 + 1857.0 * ln2) + yb8 * (252.0 + 3529.0 * ln2))
                    - yb2 * (-51.0 * yb2 * ln2 + 63.0 * yb4 * ln2 - 33.0 * yb6 * ln2 + 18.0 * yb8 * ln2 + log(8.0)) + xb10 * yb2 * (-51.0 * yb2 * ln2 + 63.0 * yb4 * ln2 - 33.0 * yb6 * ln2 + 18.0 * yb8 * ln2 + log(8.0))
                    + xb2 * (yb6 * (36.0 - 125.0 * ln2) + yb2 * (12.0 - 37.0 * ln2) - 1338.0 * yb10 * ln2 - yb8 * (108.0 + 385.0 * ln2) + 3.0 * yb4 * (20.0 + 627.0 * ln2) + log(16.0))
                    + xb8 * (-1.0 + yb2) * (yb2 * (12.0 - 33.0 * ln2) + 1338.0 * yb8 * ln2 + 24.0 * yb4 * (3.0 + 77.0 * ln2) + yb6 * (108.0 + 1723.0 * ln2) + log(16.0)))
                - 4.0 * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 + 16.0 * 1.0i * xb3 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2
                    - 16.0 * 1.0i * xb9 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2 - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * ln2
                    + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * ln2 - xb4 * (yb5 * (544.0 - 22620.0 * ln2) + yb3 * (128.0 - 339.0 * ln2) + 16.0 * ln2 + 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2
                    + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 - 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 6.0 * yb7 * (-16.0 + 2909.0 * ln2) + yb11 * (864.0 + 3741.0 * ln2))
                    - xb8 * (yb5 * (544.0 - 22620.0 * ln2) + yb3 * (128.0 - 339.0 * ln2) + 16.0 * ln2 + 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2
                    - 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 6.0 * yb7 * (-16.0 + 2909.0 * ln2) + yb11 * (864.0 + 3741.0 * ln2)) + 4.0 * xb6 * (yb7 * (84.0 - 27874.0 * ln2) - 8.0 * ln2 - 22.0 * yb * ln2 - 75.0 * yb2 * ln2 - 366.0 * yb4 * ln2 + 36.0 * yb6 * ln2
                    + 2646.0 * yb8 * ln2 + 1863.0 * yb10 * ln2 + 27.0 * yb11 * (-16.0 + 1235.0 * ln2) - 4.0 * yb5 * (53.0 + 1596.0 * ln2) - yb3 * (52.0 + 1623.0 * ln2) + yb9 * (612.0 + 6654.0 * ln2))
                    + xb2 * yb * (-4.0 * ln2 - 164.0 * yb3 * ln2 - 328.0 * yb5 * ln2 + 612.0 * yb7 * ln2 - 126.0 * yb9 * ln2 + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2) - 4.0 * yb6 * (18.0 + 391.0 * ln2)
                    - yb2 * (24.0 + ln4) + yb * ln64) + xb10 * yb * (-4.0 * ln2 - 164.0 * yb3 * ln2 - 328.0 * yb5 * ln2 + 612.0 * yb7 * ln2 - 126.0 * yb9 * ln2 + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2)
                    - 4.0 * yb6 * (18.0 + 391.0 * ln2) - yb2 * (24.0 + ln4) + yb * ln64) + (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0)
                    + yb2 * ln64) + xb12 * (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0) + yb2 * ln64))
                + vb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 + 4.0 * xb6 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2) + 32.0 * ln2 - 223.0 * yb * ln2
                    + 294.0 * yb2 * ln2 + 1818.0 * yb4 * ln2 + 534.0 * yb6 * ln2 - 11826.0 * yb8 * ln2 - 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2))
                    + 4.0 * xb10 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2) + 32.0 * ln2 - 223.0 * yb * ln2 + 294.0 * yb2 * ln2 + 1818.0 * yb4 * ln2 + 534.0 * yb6 * ln2 - 11826.0 * yb8 * ln2
                    - 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2)) - 2.0 * xb8 * (yb7 * (240.0 - 128218.0 * ln2) - 144.0 * ln2 + 824.0 * yb * ln2 - 1353.0 * yb2 * ln2 - 6430.0 * yb4 * ln2 + 952.0 * yb6 * ln2
                    + 47070.0 * yb8 * ln2 + 33633.0 * yb10 * ln2 - yb3 * (80.0 + 1599.0 * ln2) - 12.0 * yb9 * (-84.0 + 2167.0 * ln2) - 4.0 * yb5 * (76.0 + 2895.0 * ln2) + 9.0 * yb11 * (-96.0 + 12137.0 * ln2))
                    - 4.0 * xb4 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) + 18.0 * yb * ln2 + 51.0 * yb2 * ln2 - 186.0 * yb4 * ln2 - 756.0 * yb6 * ln2 + 54.0 * yb8 * ln2 - 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2)
                    + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2) + log(16.0)) - 4.0 * xb12 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) + 18.0 * yb * ln2 + 51.0 * yb2 * ln2 - 186.0 * yb4 * ln2 - 756.0 * yb6 * ln2
                    + 54.0 * yb8 * ln2 - 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2) + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2) + log(16.0))
                    + 4.0 * xb2 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 + 32.0 * yb3 * ln2 + 54.0 * yb5 * ln2 + 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2 + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2) + yb * ln64)
                    + 4.0 * xb14 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 + 32.0 * yb3 * ln2 + 54.0 * yb5 * ln2 + 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2 + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2) + yb * ln64)
                    + (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0) + yb2 * ln64)
                    + xb16 * (-1.0 + yb) * yb2 * (66.0 * yb3 * ln2 + 90.0 * yb4 * ln2 + 12.0 * yb5 * ln2 - 42.0 * yb6 * ln2 + 18.0 * yb7 * ln2 + 45.0 * yb8 * ln2 - log(8.0) + yb2 * ln64)))
                + tb2 * ((-vb2) * power_of<2>(1.0i - xb - 1.0i * xb2 + xb3) * yb * (-6.0 * 1.0i * xb * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) * ln2 - 6.0 * 1.0i * xb13 * yb2 * (6.0 - 21.0 * yb2 + 43.0 * yb4 - 31.0 * yb6 + 3.0 * yb8) * ln2
                    + 3.0 * xb14 * yb2 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 - 3.0 * xb2 * yb2 * (-1.0 + yb2) * (4.0 - 50.0 * ln2 + 165.0 * yb6 * ln2 + 4.0 * yb4 * (9.0 + 53.0 * ln2) + yb2 * (24.0 + 73.0 * ln2))
                    + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (4.0 - 50.0 * ln2 + 165.0 * yb6 * ln2 + 4.0 * yb4 * (9.0 + 53.0 * ln2) + yb2 * (24.0 + 73.0 * ln2)) + 4.0 * 1.0i * xb3 * yb2 * (-1.0 + yb2) * (6.0 + yb2 * (36.0 - 1399.0 * ln2) - 76.0 * ln2 + 273.0 * yb6 * ln2
                    + yb4 * (54.0 + 290.0 * ln2)) + 4.0 * 1.0i * xb11 * yb2 * (-1.0 + yb2) * (6.0 + yb2 * (36.0 - 1399.0 * ln2) - 76.0 * ln2 + 273.0 * yb6 * ln2 + yb4 * (54.0 + 290.0 * ln2))
                    + xb4 * yb2 * (-1.0 + yb2) * (-4.0 + yb2 * (24.0 - 12455.0 * ln2) + 22.0 * ln2 + 4.0 * yb4 * (63.0 + 55.0 * ln2) + 3.0 * yb6 * (144.0 + 407.0 * ln2))
                    - xb10 * yb2 * (-1.0 + yb2) * (-4.0 + yb2 * (24.0 - 12455.0 * ln2) + 22.0 * ln2 + 4.0 * yb4 * (63.0 + 55.0 * ln2) + 3.0 * yb6 * (144.0 + 407.0 * ln2))
                    - 8.0 * 1.0i * xb7 * (yb6 * (78.0 - 11161.0 * ln2) + yb8 * (198.0 - 10415.0 * ln2) + 224.0 * ln2 + 3.0 * yb10 * (-72.0 + 203.0 * ln2) + 2.0 * yb2 * (-7.0 + 510.0 * ln2) + yb4 * (-46.0 + 3339.0 * ln2))
                    + xb6 * (yb8 * (1080.0 - 44481.0 * ln2) + yb6 * (504.0 - 6451.0 * ln2) + 896.0 * ln2 - 9.0 * yb10 * (144.0 + 3823.0 * ln2) + 3.0 * yb4 * (-72.0 + 4295.0 * ln2) + yb2 * (-72.0 + 6022.0 * ln2))
                    + xb8 * (yb2 * (72.0 - 6022.0 * ln2) - 896.0 * ln2 + 9.0 * yb10 * (144.0 + 3823.0 * ln2) - 3.0 * yb4 * (-72.0 + 4295.0 * ln2) + yb6 * (-504.0 + 6451.0 * ln2) + 3.0 * yb8 * (-360.0 + 14827.0 * ln2))
                    + 2.0 * 1.0i * xb5 * yb2 * (-1.0 + yb2) * (16.0 + 718.0 * ln2 + 4.0 * yb4 * (-36.0 + 5107.0 * ln2) + 3.0 * yb6 * (-144.0 + 5795.0 * ln2) + yb2 * (48.0 + 22861.0 * ln2))
                    + 2.0 * 1.0i * xb9 * yb2 * (-1.0 + yb2) * (16.0 + 718.0 * ln2 + 4.0 * yb4 * (-36.0 + 5107.0 * ln2) + 3.0 * yb6 * (-144.0 + 5795.0 * ln2) + yb2 * (48.0 + 22861.0 * ln2))
                    + yb2 * (9.0 * yb2 * ln2 - 15.0 * yb4 * ln2 + 27.0 * yb6 * ln2 - 27.0 * yb8 * ln2 + ln64)) + 4.0 * xb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2
                    + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) - ln2 - 4.0 * yb * ln2 + yb2 * ln2 + 9.0 * yb4 * ln2 - 9.0 * yb6 * ln2
                    + 39.0 * yb7 * ln2 + yb5 * (9.0 + 86.0 * ln2)) + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) - ln2 - 4.0 * yb * ln2 + yb2 * ln2 + 9.0 * yb4 * ln2 - 9.0 * yb6 * ln2 + 39.0 * yb7 * ln2
                    + yb5 * (9.0 + 86.0 * ln2)) + 4.0 * xb6 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) - 16.0 * ln2 - 112.0 * yb * ln2 - 147.0 * yb2 * ln2 - 890.0 * yb4 * ln2 - 232.0 * yb6 * ln2 + 5850.0 * yb8 * ln2 + 3627.0 * yb10 * ln2
                    + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2)) + 4.0 * xb10 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) - 16.0 * ln2 - 112.0 * yb * ln2 - 147.0 * yb2 * ln2 - 890.0 * yb4 * ln2
                    - 232.0 * yb6 * ln2 + 5850.0 * yb8 * ln2 + 3627.0 * yb10 * ln2 + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2))
                    - 2.0 * xb8 * (yb7 * (120.0 - 65749.0 * ln2) + yb9 * (504.0 - 15767.0 * ln2) + 64.0 * ln2 + 464.0 * yb * ln2 + 600.0 * yb2 * ln2 + 2928.0 * yb4 * ln2 - 288.0 * yb6 * ln2 - 21168.0 * yb8 * ln2 - 14904.0 * yb10 * ln2
                    - 2.0 * yb3 * (20.0 + 239.0 * ln2) - yb5 * (152.0 + 4461.0 * ln2) + yb11 * (-432.0 + 53223.0 * ln2)) - 4.0 * xb4 * yb * (-1.0 + yb2) * (-158.0 * yb3 * ln2 - 486.0 * yb5 * ln2 + 126.0 * yb7 * ln2 + 2.0 * yb2 * (-5.0 + 63.0 * ln2)
                    + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0) + yb * ln64) - 4.0 * xb12 * yb * (-1.0 + yb2) * (-158.0 * yb3 * ln2 - 486.0 * yb5 * ln2 + 126.0 * yb7 * ln2
                    + 2.0 * yb2 * (-5.0 + 63.0 * ln2) + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0) + yb * ln64))
                - 4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 - 4.0 * xb4 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) - 4.0 * ln2 + 18.0 * yb * ln2 - 51.0 * yb2 * ln2 + 186.0 * yb4 * ln2
                    + 756.0 * yb6 * ln2 - 54.0 * yb8 * ln2 + 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2) + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2))
                    - 4.0 * xb12 * (yb5 * (76.0 - 2496.0 * ln2) + yb3 * (20.0 - 81.0 * ln2) - 4.0 * ln2 + 18.0 * yb * ln2 - 51.0 * yb2 * ln2 + 186.0 * yb4 * ln2 + 756.0 * yb6 * ln2 - 54.0 * yb8 * ln2 + 1215.0 * yb10 * ln2 + 3.0 * yb11 * (72.0 + 149.0 * ln2)
                    + 6.0 * yb7 * (-10.0 + 247.0 * ln2) - 2.0 * yb9 * (126.0 + 709.0 * ln2)) + 4.0 * xb2 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 - 6.0 * yb * ln2 - 32.0 * yb3 * ln2 - 54.0 * yb5 * ln2 - 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2
                    + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2)) + 4.0 * xb14 * yb * (-1.0 + yb2) * (yb2 * (6.0 - 18.0 * ln2) + ln2 - 6.0 * yb * ln2 - 32.0 * yb3 * ln2 - 54.0 * yb5 * ln2 - 36.0 * yb7 * ln2 + 411.0 * yb8 * ln2
                    + 12.0 * yb4 * (3.0 + 35.0 * ln2) + yb6 * (54.0 + 754.0 * ln2)) + 4.0 * xb6 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2) - 32.0 * ln2 - 223.0 * yb * ln2 - 294.0 * yb2 * ln2 - 1818.0 * yb4 * ln2
                    - 534.0 * yb6 * ln2 + 11826.0 * yb8 * ln2 + 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2)) + 4.0 * xb10 * (yb5 * (30.0 - 31050.0 * ln2) + yb7 * (18.0 - 9010.0 * ln2) + yb3 * (6.0 - 3123.0 * ln2)
                    - 32.0 * ln2 - 223.0 * yb * ln2 - 294.0 * yb2 * ln2 - 1818.0 * yb4 * ln2 - 534.0 * yb6 * ln2 + 11826.0 * yb8 * ln2 + 7236.0 * yb10 * ln2 + 44037.0 * yb11 * ln2 + 3.0 * yb9 * (-18.0 + 5251.0 * ln2))
                    - 2.0 * xb8 * (yb7 * (240.0 - 128218.0 * ln2) + 144.0 * ln2 + 824.0 * yb * ln2 + 1353.0 * yb2 * ln2 + 6430.0 * yb4 * ln2 - 952.0 * yb6 * ln2 - 47070.0 * yb8 * ln2 - 33633.0 * yb10 * ln2 - yb3 * (80.0 + 1599.0 * ln2)
                    - 12.0 * yb9 * (-84.0 + 2167.0 * ln2) - 4.0 * yb5 * (76.0 + 2895.0 * ln2) + 9.0 * yb11 * (-96.0 + 12137.0 * ln2)) + yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2
                    + 45.0 * yb9 * ln2 - log(8.0) - yb * log(8.0) + yb2 * ln64) + xb16 * yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2 + 45.0 * yb9 * ln2 - log(8.0)
                    - yb * log(8.0) + yb2 * ln64))) + 4.0 * xb2 * (vb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2
                    + 3.0 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 + 3.0 * xb16 * yb3 * (-2.0 - 3.0 * yb2 + 5.0 * yb4 - 9.0 * yb6 + 9.0 * yb8) * ln2 - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2
                    + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) * ln2 - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) * ln2 + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) * ln2 + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) + ln2 - 4.0 * yb * ln2 - yb2 * ln2 - 9.0 * yb4 * ln2 + 9.0 * yb6 * ln2
                    + 39.0 * yb7 * ln2 + yb5 * (9.0 + 86.0 * ln2)) + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (yb + yb3 * (6.0 - 9.0 * ln2) + ln2 - 4.0 * yb * ln2 - yb2 * ln2 - 9.0 * yb4 * ln2 + 9.0 * yb6 * ln2 + 39.0 * yb7 * ln2
                    + yb5 * (9.0 + 86.0 * ln2)) + 4.0 * xb6 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) + 16.0 * ln2 - 112.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2
                    + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2)) + 4.0 * xb10 * (yb7 * (9.0 - 4387.0 * ln2) + yb3 * (3.0 - 1564.0 * ln2) + 16.0 * ln2 - 112.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2
                    + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 + 22107.0 * yb11 * ln2 - 3.0 * yb5 * (-5.0 + 5253.0 * ln2) + yb9 * (-27.0 + 7907.0 * ln2))
                    - 2.0 * xb8 * (yb7 * (120.0 - 65749.0 * ln2) + yb9 * (504.0 - 15767.0 * ln2) - 64.0 * ln2 + 464.0 * yb * ln2 - 600.0 * yb2 * ln2 - 2928.0 * yb4 * ln2 + 288.0 * yb6 * ln2 + 21168.0 * yb8 * ln2 + 14904.0 * yb10 * ln2
                    - 2.0 * yb3 * (20.0 + 239.0 * ln2) - yb5 * (152.0 + 4461.0 * ln2) + yb11 * (-432.0 + 53223.0 * ln2)) - 4.0 * xb4 * yb * (-1.0 + yb2) * (-6.0 * yb * ln2 + 158.0 * yb3 * ln2 + 486.0 * yb5 * ln2 - 126.0 * yb7 * ln2 + 2.0 * yb2 * (-5.0 + 63.0 * ln2)
                    + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0)) - 4.0 * xb12 * yb * (-1.0 + yb2) * (-6.0 * yb * ln2 + 158.0 * yb3 * ln2 + 486.0 * yb5 * ln2 - 126.0 * yb7 * ln2
                    + 2.0 * yb2 * (-5.0 + 63.0 * ln2) + 3.0 * yb8 * (36.0 + 191.0 * ln2) + 2.0 * yb6 * (-9.0 + 278.0 * ln2) + 3.0 * yb4 * (-16.0 + 567.0 * ln2) + log(16.0)))
                - 4.0 * xb2 * power_of<2>(-1.0i + xb) * yb * (-1.0 + yb2) * (-6.0 * 1.0i * xb * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) * ln2 - 6.0 * 1.0i * xb9 * yb2 * (-6.0 + 15.0 * yb2 - 28.0 * yb4 + 3.0 * yb6) * ln2 - 3.0 * yb2 * (2.0 + 5.0 * yb2 + 9.0 * yb6) * ln2
                    - 3.0 * xb2 * yb2 * (1.0 + 3.0 * yb2) * (4.0 - 22.0 * ln2 + 69.0 * yb4 * ln2 + yb2 * (12.0 + 61.0 * ln2)) + 3.0 * xb8 * yb2 * (1.0 + 3.0 * yb2) * (4.0 - 22.0 * ln2 + 69.0 * yb4 * ln2 + yb2 * (12.0 + 61.0 * ln2))
                    + 8.0 * 1.0i * xb3 * yb2 * (3.0 + yb2 * (18.0 - 788.0 * ln2) - 11.0 * ln2 + 150.0 * yb6 * ln2 + yb4 * (27.0 + 217.0 * ln2)) + 8.0 * 1.0i * xb7 * yb2 * (3.0 + yb2 * (18.0 - 788.0 * ln2) - 11.0 * ln2 + 150.0 * yb6 * ln2 + yb4 * (27.0 + 217.0 * ln2))
                    + 2.0 * xb4 * (1.0 + 3.0 * yb2) * (16.0 * ln2 + 2.0 * yb2 * (-7.0 + 82.0 * ln2) + yb4 * (-18.0 + 755.0 * ln2) + yb6 * (72.0 + 957.0 * ln2)) - 2.0 * xb6 * (1.0 + 3.0 * yb2) * (16.0 * ln2 + 2.0 * yb2 * (-7.0 + 82.0 * ln2) + yb4 * (-18.0 + 755.0 * ln2)
                    + yb6 * (72.0 + 957.0 * ln2)) + 4.0 * 1.0i * xb5 * (-16.0 * ln2 + 10.0 * yb2 * (2.0 + 21.0 * ln2) + yb4 * (96.0 + 717.0 * ln2) + 4.0 * yb6 * (9.0 + 1808.0 * ln2) + 3.0 * yb8 * (-72.0 + 2155.0 * ln2))
                    + xb10 * yb2 * (15.0 * yb2 * ln2 + 27.0 * yb6 * ln2 + ln64)) + 4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2
                    + 16.0 * 1.0i * xb3 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2 - 16.0 * 1.0i * xb9 * yb3 * (1.0 - 774.0 * yb2 + 780.0 * yb4 - 10.0 * yb6 + 3.0 * yb8) * ln2 - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * 
                    ln2 + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) * ln2 + xb4 * (yb7 * (96.0 - 17454.0 * ln2) + 16.0 * ln2 - 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2
                    - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 + yb3 * (-128.0 + 339.0 * ln2) - 3.0 * yb11 * (288.0 + 1247.0 * ln2) + 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 4.0 * yb5 * (-136.0 + 5655.0 * ln2))
                    + xb8 * (yb7 * (96.0 - 17454.0 * ln2) + 16.0 * ln2 - 48.0 * yb * ln2 + 147.0 * yb2 * ln2 + 890.0 * yb4 * ln2 + 232.0 * yb6 * ln2 - 5850.0 * yb8 * ln2 - 3627.0 * yb10 * ln2 + yb3 * (-128.0 + 339.0 * ln2)
                    - 3.0 * yb11 * (288.0 + 1247.0 * ln2) + 4.0 * yb9 * (360.0 + 1619.0 * ln2) + 4.0 * yb5 * (-136.0 + 5655.0 * ln2)) + xb2 * yb * (-4.0 * ln2 - 6.0 * yb * ln2 + 164.0 * yb3 * ln2 + 328.0 * yb5 * ln2 - 612.0 * yb7 * ln2 + 126.0 * yb9 * ln2
                    + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2) - 4.0 * yb6 * (18.0 + 391.0 * ln2) - yb2 * (24.0 + ln4))
                    + xb10 * yb * (-4.0 * ln2 - 6.0 * yb * ln2 + 164.0 * yb3 * ln2 + 328.0 * yb5 * ln2 - 612.0 * yb7 * ln2 + 126.0 * yb9 * ln2 + 1806.0 * yb10 * ln2 - 24.0 * yb4 * (5.0 + 64.0 * ln2) + 4.0 * yb8 * (54.0 + 325.0 * ln2)
                    - 4.0 * yb6 * (18.0 + 391.0 * ln2) - yb2 * (24.0 + ln4)) + yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2 + 45.0 * yb9 * ln2 - log(8.0) - yb * log(8.0)
                    + yb2 * ln64) + xb12 * yb2 * (-60.0 * yb3 * ln2 + 24.0 * yb4 * ln2 + 78.0 * yb5 * ln2 - 54.0 * yb6 * ln2 - 60.0 * yb7 * ln2 + 27.0 * yb8 * ln2 + 45.0 * yb9 * ln2 - log(8.0) - yb * log(8.0) + yb2 * ln64)
                    + 4.0 * xb6 * (yb7 * (84.0 - 27874.0 * ln2) - 22.0 * yb * ln2 + 75.0 * yb2 * ln2 + 366.0 * yb4 * ln2 - 36.0 * yb6 * ln2 - 2646.0 * yb8 * ln2 - 1863.0 * yb10 * ln2 + 27.0 * yb11 * (-16.0 + 1235.0 * ln2) - 4.0 * yb5 * (53.0 + 1596.0 * ln2)
                    - yb3 * (52.0 + 1623.0 * ln2) + yb9 * (612.0 + 6654.0 * ln2) + ln256)));
            const complex<double> logs17 = 4.0 * tb * xb2 * (vb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2)
                    + yb3 * (-21.0 + 160.0 * ln2)) + 3.0 * xb16 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 4.0 * xb2 * yb * (-1.0 + yb2) * (-1.0 - 6.0 * yb - 32.0 * yb3 - 54.0 * yb5 - 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb14 * yb * (-1.0 + yb2) * (-1.0 - 6.0 * yb - 32.0 * yb3 - 54.0 * yb5 - 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 186.0 * yb4 - 756.0 * yb6 + 54.0 * yb8 - 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2) + 2.0 * yb9 * (-709.0 + 1824.0 * ln2)
                    - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) + 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 186.0 * yb4 - 756.0 * yb6 + 54.0 * yb8 - 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2)
                    + 2.0 * yb9 * (-709.0 + 1824.0 * ln2) - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) - 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 6430.0 * yb4 - 952.0 * yb6 - 47070.0 * yb8 - 33633.0 * yb10 + yb7 * (128218.0 - 54144.0 * ln2)
                    - 3.0 * yb3 * (-533.0 + 608.0 * ln2) + 9.0 * yb11 * (-12137.0 + 928.0 * ln2) + 12.0 * yb9 * (2167.0 + 1136.0 * ln2) + 12.0 * yb5 * (965.0 + 2832.0 * ln2))
                    - 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 + 1818.0 * yb4 + 534.0 * yb6 - 11826.0 * yb8 - 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2) + 18.0 * yb5 * (-1725.0 + 224.0 * ln2) + 3.0 * yb9 * (5251.0 + 448.0 * ln2)
                    - 2.0 * yb7 * (4505.0 + 3168.0 * ln2)) - 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 + 1818.0 * yb4 + 534.0 * yb6 - 11826.0 * yb8 - 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2) + 18.0 * yb5 * (-1725.0 + 224.0 * ln2)
                    + 3.0 * yb9 * (5251.0 + 448.0 * ln2) - 2.0 * yb7 * (4505.0 + 3168.0 * ln2))) - 4.0 * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    - 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) + 3.0 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2)
                    + yb3 * (-21.0 + 160.0 * ln2)) + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 + 3.0 * yb - 79.0 * yb3 - 243.0 * yb5 + 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 3.0 * yb - 79.0 * yb3 - 243.0 * yb5 + 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 4.0 * xb6 * (8.0 + 22.0 * yb + 75.0 * yb2 + 366.0 * yb4 - 36.0 * yb6 - 2646.0 * yb8 - 1863.0 * yb10 + yb7 * (27874.0 - 39168.0 * ln2) - 3.0 * yb3 * (-541.0 + 416.0 * ln2) + 48.0 * yb5 * (133.0 + 516.0 * ln2) + 9.0 * yb11 * (-3705.0 + 736.0 * ln2)
                    + yb9 * (-6654.0 + 9024.0 * ln2)) + xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2) + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2)
                    + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2)) + xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2)
                    + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2) + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2))) + 16.0 * vb * xb2 * yb * (24.0 * 1.0i * xb * yb2 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 24.0 * 1.0i * xb11 * yb2 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 48.0 * 1.0i * xb5 * yb2 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 48.0 * 1.0i * xb7 * yb2 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 8.0 * 1.0i * xb3 * (yb2 - 774.0 * yb4 + 780.0 * yb6 - 10.0 * yb8 + 3.0 * yb10)
                    - 8.0 * 1.0i * xb9 * (yb2 - 774.0 * yb4 + 780.0 * yb6 - 10.0 * yb8 + 3.0 * yb10) + 2.0 * xb2 * (-1.0 + yb2) * (-2.0 + 48.0 * yb8 * (-13.0 + 27.0 * ln2) + 3.0 * yb2 * (-1.0 + 48.0 * ln2) - 12.0 * yb4 * (73.0 + 300.0 * ln2) + yb6 * (-1007.0 + 2160.0 * ln2))
                    + 2.0 * xb10 * (-1.0 + yb2) * (-2.0 + 48.0 * yb8 * (-13.0 + 27.0 * ln2) + 3.0 * yb2 * (-1.0 + 48.0 * ln2) - 12.0 * yb4 * (73.0 + 300.0 * ln2) + yb6 * (-1007.0 + 2160.0 * ln2))
                    + 4.0 * xb6 * (38.0 + yb2 * (975.0 - 624.0 * ln2) + 24.0 * yb10 * (-731.0 + 138.0 * ln2) - 17.0 * yb6 * (-757.0 + 1152.0 * ln2) + 3.0 * yb4 * (1445.0 + 4128.0 * ln2) + yb8 * (-4769.0 + 4512.0 * ln2))
                    + xb4 * (80.0 + yb6 * (6095.0 - 51264.0 * ln2) + yb2 * (163.0 - 1584.0 * ln2) + 6.0 * yb10 * (-37.0 + 1512.0 * ln2) + yb8 * (-6241.0 + 11232.0 * ln2) + yb4 * (-8067.0 + 32544.0 * ln2))
                    + xb8 * (80.0 + yb6 * (6095.0 - 51264.0 * ln2) + yb2 * (163.0 - 1584.0 * ln2) + 6.0 * yb10 * (-37.0 + 1512.0 * ln2) + yb8 * (-6241.0 + 11232.0 * ln2) + yb4 * (-8067.0 + 32544.0 * ln2))
                    + 3.0 * yb2 * (-1.0 + yb2) * (1.0 + yb4 * (5.0 - 112.0 * ln2) - 16.0 * ln2 + 16.0 * yb2 * (-1.0 + log(32.0)) + 6.0 * yb6 * (-1.0 + ln256)) + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (1.0 + yb4 * (5.0 - 112.0 * ln2) - 16.0 * ln2 + 16.0 * yb2 * (-1.0 + log(32.0))
                    + 6.0 * yb6 * (-1.0 + ln256)))) + 4.0 * xb2 * (4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    + 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) + 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) - 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    + 3.0 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 - 3.0 * yb + 79.0 * yb3 + 243.0 * yb5 - 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 3.0 * yb + 79.0 * yb3 + 243.0 * yb5 - 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 4.0 * xb6 * (-8.0 + 22.0 * yb - 75.0 * yb2 - 366.0 * yb4 + 36.0 * yb6 + 2646.0 * yb8 + 1863.0 * yb10 + yb7 * (27874.0 - 39168.0 * ln2) - 3.0 * yb3 * (-541.0 + 416.0 * ln2) + 48.0 * yb5 * (133.0 + 516.0 * ln2) + 9.0 * yb11 * (-3705.0 + 736.0 * ln2)
                    + yb9 * (-6654.0 + 9024.0 * ln2)) + xb4 * (-16.0 + 48.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2) + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2)
                    + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2)) + xb8 * (-16.0 + 48.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2)
                    + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2) + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2))) + vb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 4.0 * xb4 * yb * (-1.0 + yb2) * (4.0 - 6.0 * yb + 158.0 * yb3 + 486.0 * yb5 - 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) + 4.0 * xb12 * yb * (-1.0 + yb2) * (4.0 - 6.0 * yb + 158.0 * yb3 + 486.0 * yb5 - 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) - 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2) + 3.0 * yb11 * (7369.0 + 192.0 * ln2)
                    + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2)
                    + 3.0 * yb11 * (7369.0 + 192.0 * ln2) + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 2.0 * xb8 * (64.0 - 464.0 * yb + 600.0 * yb2 + 2928.0 * yb4 - 288.0 * yb6 - 21168.0 * yb8 - 14904.0 * yb10
                    + yb7 * (65749.0 - 27072.0 * ln2) + yb3 * (478.0 - 912.0 * ln2) + yb11 * (-53223.0 + 4176.0 * ln2) + 3.0 * yb5 * (1487.0 + 5664.0 * ln2) + yb9 * (15767.0 + 6816.0 * ln2))
                    + 3.0 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) + 3.0 * xb16 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2)
                    - 2.0 * (1.0 + ln256)) + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256))
                    + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256)))
                - 4.0 * xb2 * power_of<2>(1.0i + xb) * yb * (-1.0 + yb2) * (3.0 * yb2 * (2.0 + yb2 * (5.0 - 80.0 * ln2) + yb6 * (9.0 - 48.0 * ln2) + 16.0 * ln2 + 112.0 * yb4 * ln2)
                    - 3.0 * xb2 * yb2 * (22.0 + yb2 * (5.0 - 2640.0 * ln2) + 144.0 * ln2 + 9.0 * yb6 * (-23.0 + 80.0 * ln2) + 12.0 * yb4 * (-21.0 + 148.0 * ln2)) + 3.0 * xb8 * yb2 * (22.0 + yb2 * (5.0 - 2640.0 * ln2) + 144.0 * ln2 + 9.0 * yb6 * (-23.0 + 80.0 * ln2)
                    + 12.0 * yb4 * (-21.0 + 148.0 * ln2)) - 2.0 * xb4 * (16.0 + yb4 * (1247.0 - 4080.0 * ln2) + 4.0 * yb2 * (53.0 + 60.0 * ln2) + 9.0 * yb8 * (319.0 + 112.0 * ln2) + 6.0 * yb6 * (537.0 + 472.0 * ln2))
                    + xb6 * (32.0 + yb4 * (2494.0 - 8160.0 * ln2) + 8.0 * yb2 * (53.0 + 60.0 * ln2) + 18.0 * yb8 * (319.0 + 112.0 * ln2) + 12.0 * yb6 * (537.0 + 472.0 * ln2))
                    - 8.0 * 1.0i * xb3 * yb2 * (11.0 + yb2 * (788.0 - 1920.0 * ln2) + 96.0 * ln2 + 6.0 * yb6 * (-25.0 + 96.0 * ln2) + yb4 * (-217.0 + 1248.0 * ln2)) - 8.0 * 1.0i * xb7 * yb2 * (11.0 + yb2 * (788.0 - 1920.0 * ln2) + 96.0 * ln2 + 6.0 * yb6 * (-25.0 + 96.0 * ln2)
                    + yb4 * (-217.0 + 1248.0 * ln2)) - 4.0 * 1.0i * xb5 * (16.0 + 6.0 * yb2 * (-35.0 + 72.0 * ln2) + 15.0 * yb8 * (-431.0 + 144.0 * ln2) + 16.0 * yb6 * (-452.0 + 333.0 * ln2) - 3.0 * yb4 * (239.0 + 2640.0 * ln2))
                    + 3.0 * xb10 * yb2 * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) - 6.0 * 1.0i * xb * yb2 * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0))
                    - 2.0 * (3.0 + ln256)) - 6.0 * 1.0i * xb9 * yb2 * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0)) - 2.0 * (3.0 + ln256))))
                - tb2 * (4.0 * vb * xb2 * (48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    + 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2)
                    + yb3 * (-21.0 + 160.0 * ln2)) + 3.0 * xb16 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 4.0 * xb2 * yb * (-1.0 + yb2) * (-1.0 + 6.0 * yb + 32.0 * yb3 + 54.0 * yb5 + 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb14 * yb * (-1.0 + yb2) * (-1.0 + 6.0 * yb + 32.0 * yb3 + 54.0 * yb5 + 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 + 186.0 * yb4 + 756.0 * yb6 - 54.0 * yb8 + 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2) + 2.0 * yb9 * (-709.0 + 1824.0 * ln2)
                    - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) + 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 + 186.0 * yb4 + 756.0 * yb6 - 54.0 * yb8 + 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2)
                    + 2.0 * yb9 * (-709.0 + 1824.0 * ln2) - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) - 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 - 6430.0 * yb4 + 952.0 * yb6 + 47070.0 * yb8 + 33633.0 * yb10 + yb7 * (128218.0 - 54144.0 * ln2)
                    - 3.0 * yb3 * (-533.0 + 608.0 * ln2) + 9.0 * yb11 * (-12137.0 + 928.0 * ln2) + 12.0 * yb9 * (2167.0 + 1136.0 * ln2) + 12.0 * yb5 * (965.0 + 2832.0 * ln2))
                    - 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 1818.0 * yb4 - 534.0 * yb6 + 11826.0 * yb8 + 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2) + 18.0 * yb5 * (-1725.0 + 224.0 * ln2) + 3.0 * yb9 * (5251.0 + 448.0 * ln2)
                    - 2.0 * yb7 * (4505.0 + 3168.0 * ln2)) - 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 1818.0 * yb4 - 534.0 * yb6 + 11826.0 * yb8 + 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2)
                    + 18.0 * yb5 * (-1725.0 + 224.0 * ln2) + 3.0 * yb9 * (5251.0 + 448.0 * ln2) - 2.0 * yb7 * (4505.0 + 3168.0 * ln2))) - 4.0 * xb2 * (24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) - 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    - 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    + 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) + 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    - 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 4.0 * xb4 * yb * (-1.0 + yb2) * (4.0 + 6.0 * yb - 158.0 * yb3 - 486.0 * yb5 + 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) + 4.0 * xb12 * yb * (-1.0 + yb2) * (4.0 + 6.0 * yb - 158.0 * yb3 - 486.0 * yb5 + 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) - 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2) + 3.0 * yb11 * (7369.0 + 192.0 * ln2)
                    + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2)
                    + 3.0 * yb11 * (7369.0 + 192.0 * ln2) + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 2.0 * xb8 * (-64.0 - 464.0 * yb - 600.0 * yb2 - 2928.0 * yb4 + 288.0 * yb6 + 21168.0 * yb8 + 14904.0 * yb10
                    + yb7 * (65749.0 - 27072.0 * ln2) + yb3 * (478.0 - 912.0 * ln2) + yb11 * (-53223.0 + 4176.0 * ln2) + 3.0 * yb5 * (1487.0 + 5664.0 * ln2) + yb9 * (15767.0 + 6816.0 * ln2))
                    + 3.0 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) + 3.0 * xb16 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2)
                    - 2.0 * (1.0 + ln256)) + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256))
                    + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256)))
                + vb2 * power_of<2>(1.0i + xb - 1.0i * xb2 - xb3) * yb * (2.0 * 1.0i * xb5 * yb2 * (-1.0 + yb2) * (718.0 - 48.0 * ln2 + 3.0 * yb6 * (5795.0 + 48.0 * ln2) - 4.0 * yb4 * (-5107.0 + 84.0 * ln2) + yb2 * (22861.0 + 240.0 * ln2))
                    + 2.0 * 1.0i * xb9 * yb2 * (-1.0 + yb2) * (718.0 - 48.0 * ln2 + 3.0 * yb6 * (5795.0 + 48.0 * ln2) - 4.0 * yb4 * (-5107.0 + 84.0 * ln2) + yb2 * (22861.0 + 240.0 * ln2))
                    + xb4 * yb2 * (-1.0 + yb2) * (-22.0 + yb2 * (12455.0 - 7920.0 * ln2) + 432.0 * ln2 + 3.0 * yb6 * (-407.0 + 720.0 * ln2) + 4.0 * yb4 * (-55.0 + 1332.0 * ln2))
                    - xb10 * yb2 * (-1.0 + yb2) * (-22.0 + yb2 * (12455.0 - 7920.0 * ln2) + 432.0 * ln2 + 3.0 * yb6 * (-407.0 + 720.0 * ln2) + 4.0 * yb4 * (-55.0 + 1332.0 * ln2))
                    - 3.0 * xb2 * yb2 * (-1.0 + yb2) * (50.0 + 176.0 * ln2 + 3.0 * yb6 * (-55.0 + 208.0 * ln2) + 4.0 * yb4 * (-53.0 + 500.0 * ln2) - yb2 * (73.0 + 2800.0 * ln2))
                    + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (50.0 + 176.0 * ln2 + 3.0 * yb6 * (-55.0 + 208.0 * ln2) + 4.0 * yb4 * (-53.0 + 500.0 * ln2) - yb2 * (73.0 + 2800.0 * ln2))
                    - 4.0 * 1.0i * xb3 * yb2 * (-1.0 + yb2) * (76.0 + yb2 * (1399.0 - 4080.0 * ln2) + 240.0 * ln2 + 21.0 * yb6 * (-13.0 + 48.0 * ln2) + yb4 * (-290.0 + 2832.0 * ln2))
                    - 4.0 * 1.0i * xb11 * yb2 * (-1.0 + yb2) * (76.0 + yb2 * (1399.0 - 4080.0 * ln2) + 240.0 * ln2 + 21.0 * yb6 * (-13.0 + 48.0 * ln2) + yb4 * (-290.0 + 2832.0 * ln2))
                    + xb6 * (-896.0 + yb6 * (6451.0 - 28224.0 * ln2) + 9.0 * yb10 * (3823.0 + 432.0 * ln2) - 2.0 * yb2 * (3011.0 + 504.0 * ln2) + 3.0 * yb4 * (-4295.0 + 5856.0 * ln2) + yb8 * (44481.0 + 7776.0 * ln2))
                    + xb8 * (896.0 - 9.0 * yb10 * (3823.0 + 432.0 * ln2) + 2.0 * yb2 * (3011.0 + 504.0 * ln2) - 3.0 * yb8 * (14827.0 + 2592.0 * ln2) - 3.0 * yb4 * (-4295.0 + 5856.0 * ln2) + yb6 * (-6451.0 + 28224.0 * ln2))
                    + 8.0 * 1.0i * xb7 * (-224.0 + yb6 * (11161.0 - 6912.0 * ln2) + 21.0 * yb10 * (-29.0 + 48.0 * ln2) + 9.0 * yb4 * (-371.0 + 480.0 * ln2) + yb8 * (10415.0 + 1824.0 * ln2) - 60.0 * yb2 * (17.0 + log(16.0)))
                    - 3.0 * yb2 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) + 3.0 * xb14 * yb2 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2)
                    - 2.0 * (1.0 + ln256)) - 6.0 * 1.0i * xb * yb2 * (-1.0 + yb2) * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0)) - 2.0 * (3.0 + ln256))
                    - 6.0 * 1.0i * xb13 * yb2 * (-1.0 + yb2) * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0)) - 2.0 * (3.0 + ln256))));
            const complex<double> logs18 = 4.0 * tb * xb2 * (vb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2)
                    + yb3 * (-21.0 + 160.0 * ln2)) + 3.0 * xb16 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 4.0 * xb2 * yb * (-1.0 + yb2) * (-1.0 - 6.0 * yb - 32.0 * yb3 - 54.0 * yb5 - 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb14 * yb * (-1.0 + yb2) * (-1.0 - 6.0 * yb - 32.0 * yb3 - 54.0 * yb5 - 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb4 * (4.0 + 18.0 * yb + 51.0 * yb2 - 186.0 * yb4 - 756.0 * yb6 + 54.0 * yb8 - 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2) + 2.0 * yb9 * (-709.0 + 1824.0 * ln2)
                    - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) + 4.0 * xb12 * (4.0 + 18.0 * yb + 51.0 * yb2 - 186.0 * yb4 - 756.0 * yb6 + 54.0 * yb8 - 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2)
                    + 2.0 * yb9 * (-709.0 + 1824.0 * ln2) - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) - 2.0 * xb8 * (144.0 - 824.0 * yb + 1353.0 * yb2 + 6430.0 * yb4 - 952.0 * yb6 - 47070.0 * yb8 - 33633.0 * yb10 + yb7 * (128218.0 - 54144.0 * ln2)
                    - 3.0 * yb3 * (-533.0 + 608.0 * ln2) + 9.0 * yb11 * (-12137.0 + 928.0 * ln2) + 12.0 * yb9 * (2167.0 + 1136.0 * ln2) + 12.0 * yb5 * (965.0 + 2832.0 * ln2))
                    - 4.0 * xb6 * (32.0 - 223.0 * yb + 294.0 * yb2 + 1818.0 * yb4 + 534.0 * yb6 - 11826.0 * yb8 - 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2) + 18.0 * yb5 * (-1725.0 + 224.0 * ln2) + 3.0 * yb9 * (5251.0 + 448.0 * ln2)
                    - 2.0 * yb7 * (4505.0 + 3168.0 * ln2)) - 4.0 * xb10 * (32.0 - 223.0 * yb + 294.0 * yb2 + 1818.0 * yb4 + 534.0 * yb6 - 11826.0 * yb8 - 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2) + 18.0 * yb5 * (-1725.0 + 224.0 * ln2)
                    + 3.0 * yb9 * (5251.0 + 448.0 * ln2) - 2.0 * yb7 * (4505.0 + 3168.0 * ln2))) - 4.0 * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) - 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    + 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) + 3.0 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2)
                    + yb3 * (-21.0 + 160.0 * ln2)) + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 + 3.0 * yb - 79.0 * yb3 - 243.0 * yb5 + 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 3.0 * yb - 79.0 * yb3 - 243.0 * yb5 + 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 4.0 * xb6 * (8.0 + 22.0 * yb + 75.0 * yb2 + 366.0 * yb4 - 36.0 * yb6 - 2646.0 * yb8 - 1863.0 * yb10 + yb7 * (27874.0 - 39168.0 * ln2) - 3.0 * yb3 * (-541.0 + 416.0 * ln2) + 48.0 * yb5 * (133.0 + 516.0 * ln2) + 9.0 * yb11 * (-3705.0 + 736.0 * ln2)
                    + yb9 * (-6654.0 + 9024.0 * ln2)) + xb4 * (16.0 + 48.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2) + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2)
                    + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2)) + xb8 * (16.0 + 48.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2)
                    + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2) + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2))) + 16.0 * vb * xb2 * yb * (-24.0 * 1.0i * xb * yb2 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 24.0 * 1.0i * xb11 * yb2 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 48.0 * 1.0i * xb5 * yb2 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) - 48.0 * 1.0i * xb7 * yb2 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) - 8.0 * 1.0i * xb3 * (yb2 - 774.0 * yb4 + 780.0 * yb6 - 10.0 * yb8 + 3.0 * yb10)
                    + 8.0 * 1.0i * xb9 * (yb2 - 774.0 * yb4 + 780.0 * yb6 - 10.0 * yb8 + 3.0 * yb10) + 2.0 * xb2 * (-1.0 + yb2) * (-2.0 + 48.0 * yb8 * (-13.0 + 27.0 * ln2) + 3.0 * yb2 * (-1.0 + 48.0 * ln2) - 12.0 * yb4 * (73.0 + 300.0 * ln2) + yb6 * (-1007.0 + 2160.0 * ln2))
                    + 2.0 * xb10 * (-1.0 + yb2) * (-2.0 + 48.0 * yb8 * (-13.0 + 27.0 * ln2) + 3.0 * yb2 * (-1.0 + 48.0 * ln2) - 12.0 * yb4 * (73.0 + 300.0 * ln2) + yb6 * (-1007.0 + 2160.0 * ln2))
                    + 4.0 * xb6 * (38.0 + yb2 * (975.0 - 624.0 * ln2) + 24.0 * yb10 * (-731.0 + 138.0 * ln2) - 17.0 * yb6 * (-757.0 + 1152.0 * ln2) + 3.0 * yb4 * (1445.0 + 4128.0 * ln2) + yb8 * (-4769.0 + 4512.0 * ln2))
                    + xb4 * (80.0 + yb6 * (6095.0 - 51264.0 * ln2) + yb2 * (163.0 - 1584.0 * ln2) + 6.0 * yb10 * (-37.0 + 1512.0 * ln2) + yb8 * (-6241.0 + 11232.0 * ln2) + yb4 * (-8067.0 + 32544.0 * ln2))
                    + xb8 * (80.0 + yb6 * (6095.0 - 51264.0 * ln2) + yb2 * (163.0 - 1584.0 * ln2) + 6.0 * yb10 * (-37.0 + 1512.0 * ln2) + yb8 * (-6241.0 + 11232.0 * ln2) + yb4 * (-8067.0 + 32544.0 * ln2))
                    + 3.0 * yb2 * (-1.0 + yb2) * (1.0 + yb4 * (5.0 - 112.0 * ln2) - 16.0 * ln2 + 16.0 * yb2 * (-1.0 + log(32.0)) + 6.0 * yb6 * (-1.0 + ln256)) + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (1.0 + yb4 * (5.0 - 112.0 * ln2) - 16.0 * ln2 + 16.0 * yb2 * (-1.0 + log(32.0))
                    + 6.0 * yb6 * (-1.0 + ln256)))) + 4.0 * xb2 * (4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb11 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 96.0 * 1.0i * xb5 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8)
                    - 96.0 * 1.0i * xb7 * yb3 * (-37.0 - 258.0 * yb2 - 540.0 * yb4 + 82.0 * yb6 + 753.0 * yb8) - 16.0 * 1.0i * xb3 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11) + 16.0 * 1.0i * xb9 * (yb3 - 774.0 * yb5 + 780.0 * yb7 - 10.0 * yb9 + 3.0 * yb11)
                    + 3.0 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 - 3.0 * yb + 79.0 * yb3 + 243.0 * yb5 - 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 3.0 * yb + 79.0 * yb3 + 243.0 * yb5 - 63.0 * yb7 + 3.0 * yb2 * (-1.0 + 96.0 * ln2) + 3.0 * yb8 * (-301.0 + 864.0 * ln2) - 3.0 * yb4 * (257.0 + 2400.0 * ln2) + yb6 * (-1553.0 + 4320.0 * ln2))
                    + 4.0 * xb6 * (-8.0 + 22.0 * yb - 75.0 * yb2 - 366.0 * yb4 + 36.0 * yb6 + 2646.0 * yb8 + 1863.0 * yb10 + yb7 * (27874.0 - 39168.0 * ln2) - 3.0 * yb3 * (-541.0 + 416.0 * ln2) + 48.0 * yb5 * (133.0 + 516.0 * ln2) + 9.0 * yb11 * (-3705.0 + 736.0 * ln2)
                    + yb9 * (-6654.0 + 9024.0 * ln2)) + xb4 * (-16.0 + 48.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2) + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2)
                    + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2)) + xb8 * (-16.0 + 48.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 3.0 * yb3 * (113.0 + 1056.0 * ln2) + 12.0 * yb5 * (-1885.0 + 5424.0 * ln2)
                    + 4.0 * yb9 * (-1619.0 + 5616.0 * ln2) + 3.0 * yb11 * (1247.0 + 6048.0 * ln2) - 6.0 * yb7 * (-2909.0 + 17088.0 * ln2))) + vb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 4.0 * xb4 * yb * (-1.0 + yb2) * (4.0 - 6.0 * yb + 158.0 * yb3 + 486.0 * yb5 - 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) + 4.0 * xb12 * yb * (-1.0 + yb2) * (4.0 - 6.0 * yb + 158.0 * yb3 + 486.0 * yb5 - 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) - 4.0 * xb6 * (16.0 - 112.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2) + 3.0 * yb11 * (7369.0 + 192.0 * ln2)
                    + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 4.0 * xb10 * (16.0 - 112.0 * yb + 147.0 * yb2 + 890.0 * yb4 + 232.0 * yb6 - 5850.0 * yb8 - 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2)
                    + 3.0 * yb11 * (7369.0 + 192.0 * ln2) + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 2.0 * xb8 * (64.0 - 464.0 * yb + 600.0 * yb2 + 2928.0 * yb4 - 288.0 * yb6 - 21168.0 * yb8 - 14904.0 * yb10
                    + yb7 * (65749.0 - 27072.0 * ln2) + yb3 * (478.0 - 912.0 * ln2) + yb11 * (-53223.0 + 4176.0 * ln2) + 3.0 * yb5 * (1487.0 + 5664.0 * ln2) + yb9 * (15767.0 + 6816.0 * ln2))
                    + 3.0 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) + 3.0 * xb16 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2)
                    - 2.0 * (1.0 + ln256)) + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256))
                    + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256)))
                - 4.0 * xb2 * power_of<2>(-1.0i + xb) * yb * (-1.0 + yb2) * (3.0 * yb2 * (2.0 + yb2 * (5.0 - 80.0 * ln2) + yb6 * (9.0 - 48.0 * ln2) + 16.0 * ln2 + 112.0 * yb4 * ln2)
                    - 3.0 * xb2 * yb2 * (22.0 + yb2 * (5.0 - 2640.0 * ln2) + 144.0 * ln2 + 9.0 * yb6 * (-23.0 + 80.0 * ln2) + 12.0 * yb4 * (-21.0 + 148.0 * ln2)) + 3.0 * xb8 * yb2 * (22.0 + yb2 * (5.0 - 2640.0 * ln2) + 144.0 * ln2 + 9.0 * yb6 * (-23.0 + 80.0 * ln2)
                    + 12.0 * yb4 * (-21.0 + 148.0 * ln2)) - 2.0 * xb4 * (16.0 + yb4 * (1247.0 - 4080.0 * ln2) + 4.0 * yb2 * (53.0 + 60.0 * ln2) + 9.0 * yb8 * (319.0 + 112.0 * ln2) + 6.0 * yb6 * (537.0 + 472.0 * ln2))
                    + xb6 * (32.0 + yb4 * (2494.0 - 8160.0 * ln2) + 8.0 * yb2 * (53.0 + 60.0 * ln2) + 18.0 * yb8 * (319.0 + 112.0 * ln2) + 12.0 * yb6 * (537.0 + 472.0 * ln2))
                    + 8.0 * 1.0i * xb3 * yb2 * (11.0 + yb2 * (788.0 - 1920.0 * ln2) + 96.0 * ln2 + 6.0 * yb6 * (-25.0 + 96.0 * ln2) + yb4 * (-217.0 + 1248.0 * ln2)) + 8.0 * 1.0i * xb7 * yb2 * (11.0 + yb2 * (788.0 - 1920.0 * ln2) + 96.0 * ln2 + 6.0 * yb6 * (-25.0 + 96.0 * ln2)
                    + yb4 * (-217.0 + 1248.0 * ln2)) + 4.0 * 1.0i * xb5 * (16.0 + 6.0 * yb2 * (-35.0 + 72.0 * ln2) + 15.0 * yb8 * (-431.0 + 144.0 * ln2) + 16.0 * yb6 * (-452.0 + 333.0 * ln2) - 3.0 * yb4 * (239.0 + 2640.0 * ln2))
                    + 3.0 * xb10 * yb2 * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) + 6.0 * 1.0i * xb * yb2 * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0))
                    - 2.0 * (3.0 + ln256)) + 6.0 * 1.0i * xb9 * yb2 * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0)) - 2.0 * (3.0 + ln256))))
                - tb2 * (4.0 * vb * xb2 * (-48.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 48.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 16.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8)
                    - 16.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 16.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 16.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 16.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 16.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 3.0 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2)
                    + yb3 * (-21.0 + 160.0 * ln2)) + 3.0 * xb16 * yb2 * (-1.0 + yb2) * (-1.0 + yb2 + 9.0 * yb4 - 9.0 * yb6 + yb5 * (5.0 - 224.0 * ln2) + 3.0 * yb7 * (-5.0 + 32.0 * ln2) - yb * (1.0 + 32.0 * ln2) + yb3 * (-21.0 + 160.0 * ln2))
                    + 4.0 * xb2 * yb * (-1.0 + yb2) * (-1.0 + 6.0 * yb + 32.0 * yb3 + 54.0 * yb5 + 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb14 * yb * (-1.0 + yb2) * (-1.0 + 6.0 * yb + 32.0 * yb3 + 54.0 * yb5 + 36.0 * yb7 + 6.0 * yb2 * (3.0 + 32.0 * ln2) - 60.0 * yb4 * (7.0 + 64.0 * ln2) + 26.0 * yb6 * (-29.0 + 96.0 * ln2) + 3.0 * yb8 * (-137.0 + 384.0 * ln2))
                    + 4.0 * xb4 * (-4.0 + 18.0 * yb - 51.0 * yb2 + 186.0 * yb4 + 756.0 * yb6 - 54.0 * yb8 + 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2) + 2.0 * yb9 * (-709.0 + 1824.0 * ln2)
                    - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) + 4.0 * xb12 * (-4.0 + 18.0 * yb - 51.0 * yb2 + 186.0 * yb4 + 756.0 * yb6 - 54.0 * yb8 + 1215.0 * yb10 + 192.0 * yb5 * (-13.0 + 45.0 * ln2) - 3.0 * yb3 * (27.0 + 160.0 * ln2) + 3.0 * yb11 * (149.0 + 672.0 * ln2)
                    + 2.0 * yb9 * (-709.0 + 1824.0 * ln2) - 6.0 * yb7 * (-247.0 + 2304.0 * ln2)) - 2.0 * xb8 * (-144.0 - 824.0 * yb - 1353.0 * yb2 - 6430.0 * yb4 + 952.0 * yb6 + 47070.0 * yb8 + 33633.0 * yb10 + yb7 * (128218.0 - 54144.0 * ln2)
                    - 3.0 * yb3 * (-533.0 + 608.0 * ln2) + 9.0 * yb11 * (-12137.0 + 928.0 * ln2) + 12.0 * yb9 * (2167.0 + 1136.0 * ln2) + 12.0 * yb5 * (965.0 + 2832.0 * ln2))
                    - 4.0 * xb6 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 1818.0 * yb4 - 534.0 * yb6 + 11826.0 * yb8 + 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2) + 18.0 * yb5 * (-1725.0 + 224.0 * ln2) + 3.0 * yb9 * (5251.0 + 448.0 * ln2)
                    - 2.0 * yb7 * (4505.0 + 3168.0 * ln2)) - 4.0 * xb10 * (-32.0 - 223.0 * yb - 294.0 * yb2 - 1818.0 * yb4 - 534.0 * yb6 + 11826.0 * yb8 + 7236.0 * yb10 - 3.0 * yb3 * (1041.0 + 64.0 * ln2) + 9.0 * yb11 * (4893.0 + 128.0 * ln2)
                    + 18.0 * yb5 * (-1725.0 + 224.0 * ln2) + 3.0 * yb9 * (5251.0 + 448.0 * ln2) - 2.0 * yb7 * (4505.0 + 3168.0 * ln2))) - 4.0 * xb2 * (-24.0 * 1.0i * xb * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) + 24.0 * 1.0i * xb15 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2)
                    + 8.0 * 1.0i * xb3 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) - 8.0 * 1.0i * xb13 * yb3 * (5.0 + 738.0 * yb2 - 708.0 * yb4 - 50.0 * yb6 + 15.0 * yb8) + 8.0 * 1.0i * xb5 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8)
                    - 8.0 * 1.0i * xb11 * yb3 * (-223.0 - 3078.0 * yb2 - 1716.0 * yb4 + 502.0 * yb6 + 4515.0 * yb8) - 8.0 * 1.0i * xb7 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8)
                    + 8.0 * 1.0i * xb9 * yb3 * (-665.0 - 5418.0 * yb2 - 8940.0 * yb4 + 1466.0 * yb6 + 13557.0 * yb8) + 4.0 * xb4 * yb * (-1.0 + yb2) * (4.0 + 6.0 * yb - 158.0 * yb3 - 486.0 * yb5 + 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) + 4.0 * xb12 * yb * (-1.0 + yb2) * (4.0 + 6.0 * yb - 158.0 * yb3 - 486.0 * yb5 + 126.0 * yb7 + yb4 * (1701.0 - 4080.0 * ln2) + 6.0 * yb2 * (21.0 + 40.0 * ln2)
                    + 3.0 * yb8 * (191.0 + 336.0 * ln2) + 4.0 * yb6 * (139.0 + 708.0 * ln2)) - 4.0 * xb6 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2) + 3.0 * yb11 * (7369.0 + 192.0 * ln2)
                    + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 4.0 * xb10 * (-16.0 - 112.0 * yb - 147.0 * yb2 - 890.0 * yb4 - 232.0 * yb6 + 5850.0 * yb8 + 3627.0 * yb10 - 4.0 * yb3 * (391.0 + 24.0 * ln2)
                    + 3.0 * yb11 * (7369.0 + 192.0 * ln2) + 9.0 * yb5 * (-1751.0 + 224.0 * ln2) + yb9 * (7907.0 + 672.0 * ln2) - yb7 * (4387.0 + 3168.0 * ln2)) - 2.0 * xb8 * (-64.0 - 464.0 * yb - 600.0 * yb2 - 2928.0 * yb4 + 288.0 * yb6 + 21168.0 * yb8 + 14904.0 * yb10
                    + yb7 * (65749.0 - 27072.0 * ln2) + yb3 * (478.0 - 912.0 * ln2) + yb11 * (-53223.0 + 4176.0 * ln2) + 3.0 * yb5 * (1487.0 + 5664.0 * ln2) + yb9 * (15767.0 + 6816.0 * ln2))
                    + 3.0 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) + 3.0 * xb16 * yb3 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2)
                    - 2.0 * (1.0 + ln256)) + 12.0 * xb2 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256))
                    + 12.0 * xb14 * yb2 * (-1.0 + yb2) * (1.0 - yb2 - 9.0 * yb4 + 9.0 * yb6 + yb3 * (9.0 - 640.0 * ln2) + 3.0 * yb7 * (-13.0 + 64.0 * ln2) + yb5 * (-86.0 + 416.0 * ln2) + 4.0 * yb * (1.0 + ln256)))
                + vb2 * power_of<2>(1.0i - xb - 1.0i * xb2 + xb3) * yb * (-2.0 * 1.0i * xb5 * yb2 * (-1.0 + yb2) * (718.0 - 48.0 * ln2 + 3.0 * yb6 * (5795.0 + 48.0 * ln2) - 4.0 * yb4 * (-5107.0 + 84.0 * ln2) + yb2 * (22861.0 + 240.0 * ln2))
                    - 2.0 * 1.0i * xb9 * yb2 * (-1.0 + yb2) * (718.0 - 48.0 * ln2 + 3.0 * yb6 * (5795.0 + 48.0 * ln2) - 4.0 * yb4 * (-5107.0 + 84.0 * ln2) + yb2 * (22861.0 + 240.0 * ln2))
                    + xb4 * yb2 * (-1.0 + yb2) * (-22.0 + yb2 * (12455.0 - 7920.0 * ln2) + 432.0 * ln2 + 3.0 * yb6 * (-407.0 + 720.0 * ln2) + 4.0 * yb4 * (-55.0 + 1332.0 * ln2))
                    - xb10 * yb2 * (-1.0 + yb2) * (-22.0 + yb2 * (12455.0 - 7920.0 * ln2) + 432.0 * ln2 + 3.0 * yb6 * (-407.0 + 720.0 * ln2) + 4.0 * yb4 * (-55.0 + 1332.0 * ln2))
                    - 3.0 * xb2 * yb2 * (-1.0 + yb2) * (50.0 + 176.0 * ln2 + 3.0 * yb6 * (-55.0 + 208.0 * ln2) + 4.0 * yb4 * (-53.0 + 500.0 * ln2) - yb2 * (73.0 + 2800.0 * ln2))
                    + 3.0 * xb12 * yb2 * (-1.0 + yb2) * (50.0 + 176.0 * ln2 + 3.0 * yb6 * (-55.0 + 208.0 * ln2) + 4.0 * yb4 * (-53.0 + 500.0 * ln2) - yb2 * (73.0 + 2800.0 * ln2))
                    + 4.0 * 1.0i * xb3 * yb2 * (-1.0 + yb2) * (76.0 + yb2 * (1399.0 - 4080.0 * ln2) + 240.0 * ln2 + 21.0 * yb6 * (-13.0 + 48.0 * ln2) + yb4 * (-290.0 + 2832.0 * ln2))
                    + 4.0 * 1.0i * xb11 * yb2 * (-1.0 + yb2) * (76.0 + yb2 * (1399.0 - 4080.0 * ln2) + 240.0 * ln2 + 21.0 * yb6 * (-13.0 + 48.0 * ln2) + yb4 * (-290.0 + 2832.0 * ln2))
                    + xb6 * (-896.0 + yb6 * (6451.0 - 28224.0 * ln2) + 9.0 * yb10 * (3823.0 + 432.0 * ln2) - 2.0 * yb2 * (3011.0 + 504.0 * ln2) + 3.0 * yb4 * (-4295.0 + 5856.0 * ln2) + yb8 * (44481.0 + 7776.0 * ln2))
                    + xb8 * (896.0 - 9.0 * yb10 * (3823.0 + 432.0 * ln2) + 2.0 * yb2 * (3011.0 + 504.0 * ln2) - 3.0 * yb8 * (14827.0 + 2592.0 * ln2) - 3.0 * yb4 * (-4295.0 + 5856.0 * ln2) + yb6 * (-6451.0 + 28224.0 * ln2))
                    - 8.0 * 1.0i * xb7 * (-224.0 + yb6 * (11161.0 - 6912.0 * ln2) + 21.0 * yb10 * (-29.0 + 48.0 * ln2) + 9.0 * yb4 * (-371.0 + 480.0 * ln2) + yb8 * (10415.0 + 1824.0 * ln2) - 60.0 * yb2 * (17.0 + log(16.0)))
                    - 3.0 * yb2 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2) - 2.0 * (1.0 + ln256)) + 3.0 * xb14 * yb2 * (-1.0 + yb2) * (-112.0 * yb4 * ln2 + yb6 * (-9.0 + 48.0 * ln2) + yb2 * (-5.0 + 80.0 * ln2)
                    - 2.0 * (1.0 + ln256)) + 6.0 * 1.0i * xb * yb2 * (-1.0 + yb2) * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0)) - 2.0 * (3.0 + ln256))
                    + 6.0 * 1.0i * xb13 * yb2 * (-1.0 + yb2) * (5.0 * yb2 * (3.0 + 16.0 * ln2) + yb6 * (3.0 + 48.0 * ln2) - 28.0 * yb4 * (1.0 + log(16.0)) - 2.0 * (3.0 + ln256))));
            const complex<double> logs19 = -2.0 * tb * xb2 * (2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 4.0 * xb6 * (-24.0 - 154.0 * yb - 225.0 * yb2 - 794.0 * yb4 + 668.0 * yb6 + 6930.0 * yb8 + 5733.0 * yb10 - yb3 * (941.0 + 156.0 * ln2) + 18.0 * yb5 * (-173.0 + 172.0 * ln2) + 4.0 * yb9 * (2041.0 + 282.0 * ln2) - 8.0 * yb7 * (-565.0 + 612.0 * ln2)
                    + yb11 * (3813.0 + 828.0 * ln2)) + xb4 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2) - 16.0 * yb7 * (-143.0 + 801.0 * ln2)
                    + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + xb8 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2)
                    - 16.0 * yb7 * (-143.0 + 801.0 * ln2) + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 4.0 * ln2) + yb2 * (1.0 + log(16.0)))
                + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 4.0 * ln2) + yb2 * (1.0 + log(16.0))))
                + tb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (-64.0 - 416.0 * yb - 600.0 * yb2 - 2320.0 * yb4 + 1408.0 * yb6 + 19152.0 * yb8 + 15192.0 * yb10 - yb3 * (2536.0 + 57.0 * ln2) + 18.0 * yb5 * (-600.0 + 59.0 * ln2) + 3.0 * yb11 * (3704.0 + 87.0 * ln2) - 4.0 * yb7 * (-3328.0 + 423.0 * ln2)
                    + yb9 * (22096.0 + 426.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (-2.0 + 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0)) + 3.0 * yb5 * (2.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (-2.0 + 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0))
                    + 3.0 * yb5 * (2.0 + log(128.0)))) - 4.0 * xb2 * (4.0 * xb6 * (-8.0 - 52.0 * yb - 75.0 * yb2 - 290.0 * yb4 + 176.0 * yb6 + 2394.0 * yb8 + 1899.0 * yb10 - yb3 * (317.0 + 39.0 * ln2) + 18.0 * yb5 * (-75.0 + 43.0 * ln2) + 3.0 * yb11 * (463.0 + 69.0 * ln2)
                    - 8.0 * yb7 * (-208.0 + 153.0 * ln2) + yb9 * (2762.0 + 282.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (8.0 + 304.0 * yb3 + 864.0 * yb5 - 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2)
                    + yb6 * (1904.0 + 1269.0 * ln2)) + xb8 * yb * (-1.0 + yb2) * (8.0 + 304.0 * yb3 + 864.0 * yb5 - 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2) + yb6 * (1904.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                + 6.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0)) + 3.0 * yb5 * (-1.0 + log(512.0))) + 6.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0))
                    + 3.0 * yb5 * (-1.0 + log(512.0))));
            const complex<double> logs20 = 2.0 * tb * xb2 * (2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 - 9.0 * yb - 67.0 * yb3 - 135.0 * yb5 - 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 4.0 * xb6 * (-24.0 - 154.0 * yb - 225.0 * yb2 - 794.0 * yb4 + 668.0 * yb6 + 6930.0 * yb8 + 5733.0 * yb10 - yb3 * (941.0 + 156.0 * ln2) + 18.0 * yb5 * (-173.0 + 172.0 * ln2) + 4.0 * yb9 * (2041.0 + 282.0 * ln2) - 8.0 * yb7 * (-565.0 + 612.0 * ln2)
                    + yb11 * (3813.0 + 828.0 * ln2)) + xb4 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2) - 16.0 * yb7 * (-143.0 + 801.0 * ln2)
                    + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + xb8 * (16.0 + 80.0 * yb + 147.0 * yb2 - 326.0 * yb4 - 2008.0 * yb6 - 1818.0 * yb8 - 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2)
                    - 16.0 * yb7 * (-143.0 + 801.0 * ln2) + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 4.0 * ln2) + yb2 * (1.0 + log(16.0)))
                + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb2 * power_of<3>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (2.0 - 4.0 * ln2) + yb2 * (1.0 + log(16.0))))
                - tb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (-64.0 - 416.0 * yb - 600.0 * yb2 - 2320.0 * yb4 + 1408.0 * yb6 + 19152.0 * yb8 + 15192.0 * yb10 - yb3 * (2536.0 + 57.0 * ln2) + 18.0 * yb5 * (-600.0 + 59.0 * ln2) + 3.0 * yb11 * (3704.0 + 87.0 * ln2) - 4.0 * yb7 * (-3328.0 + 423.0 * ln2)
                    + yb9 * (22096.0 + 426.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (4.0 + 152.0 * yb3 + 432.0 * yb5 - 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (-2.0 + 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0)) + 3.0 * yb5 * (2.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (-2.0 + 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0))
                    + 3.0 * yb5 * (2.0 + log(128.0)))) + 4.0 * xb2 * (4.0 * xb6 * (-8.0 - 52.0 * yb - 75.0 * yb2 - 290.0 * yb4 + 176.0 * yb6 + 2394.0 * yb8 + 1899.0 * yb10 - yb3 * (317.0 + 39.0 * ln2) + 18.0 * yb5 * (-75.0 + 43.0 * ln2) + 3.0 * yb11 * (463.0 + 69.0 * ln2)
                    - 8.0 * yb7 * (-208.0 + 153.0 * ln2) + yb9 * (2762.0 + 282.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (8.0 + 304.0 * yb3 + 864.0 * yb5 - 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2)
                    + yb6 * (1904.0 + 1269.0 * ln2)) + xb8 * yb * (-1.0 + yb2) * (8.0 + 304.0 * yb3 + 864.0 * yb5 - 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2) + yb6 * (1904.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                + 6.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0)) + 3.0 * yb5 * (-1.0 + log(512.0))) + 6.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (1.0 - 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0))
                    + 3.0 * yb5 * (-1.0 + log(512.0))));
            const complex<double> logs21 = 2.0 * vb * xb2 * (2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 4.0 * xb6 * (24.0 - 154.0 * yb + 225.0 * yb2 + 794.0 * yb4 - 668.0 * yb6 - 6930.0 * yb8 - 5733.0 * yb10 - yb3 * (941.0 + 156.0 * ln2) + 18.0 * yb5 * (-173.0 + 172.0 * ln2) + 4.0 * yb9 * (2041.0 + 282.0 * ln2) - 8.0 * yb7 * (-565.0 + 612.0 * ln2)
                    + yb11 * (3813.0 + 828.0 * ln2)) + xb4 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2) - 16.0 * yb7 * (-143.0 + 801.0 * ln2)
                    + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + xb8 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2)
                    - 16.0 * yb7 * (-143.0 + 801.0 * ln2) + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + 3.0 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + log(16.0)) + yb2 * (1.0 + log(16.0)))
                + 3.0 * xb12 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + log(16.0)) + yb2 * (1.0 + log(16.0))))
                + vb2 * (24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 + 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                - 2.0 * xb8 * (64.0 - 416.0 * yb + 600.0 * yb2 + 2320.0 * yb4 - 1408.0 * yb6 - 19152.0 * yb8 - 15192.0 * yb10 - yb3 * (2536.0 + 57.0 * ln2) + 18.0 * yb5 * (-600.0 + 59.0 * ln2) + 3.0 * yb11 * (3704.0 + 87.0 * ln2) - 4.0 * yb7 * (-3328.0 + 423.0 * ln2)
                    + yb9 * (22096.0 + 426.0 * ln2)) + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb16 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                - 8.0 * xb6 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                - 8.0 * xb10 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                + 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (2.0 - 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0)) + 3.0 * yb5 * (2.0 + log(128.0))) + 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (2.0 - 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0))
                    + 3.0 * yb5 * (2.0 + log(128.0)))) - 4.0 * xb2 * (4.0 * xb6 * (8.0 - 52.0 * yb + 75.0 * yb2 + 290.0 * yb4 - 176.0 * yb6 - 2394.0 * yb8 - 1899.0 * yb10 - yb3 * (317.0 + 39.0 * ln2) + 18.0 * yb5 * (-75.0 + 43.0 * ln2) + 3.0 * yb11 * (463.0 + 69.0 * ln2)
                    - 8.0 * yb7 * (-208.0 + 153.0 * ln2) + yb9 * (2762.0 + 282.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (8.0 - 304.0 * yb3 - 864.0 * yb5 + 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2)
                    + yb6 * (1904.0 + 1269.0 * ln2)) + xb8 * yb * (-1.0 + yb2) * (8.0 - 304.0 * yb3 - 864.0 * yb5 + 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2) + yb6 * (1904.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                + 6.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0)) + 3.0 * yb5 * (-1.0 + log(512.0))) + 6.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0))
                    + 3.0 * yb5 * (-1.0 + log(512.0))));
            const complex<double> logs22 = -2.0 * vb * xb2 * (2.0 * xb2 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 2.0 * xb10 * yb * (-1.0 + yb2) * (-2.0 + 9.0 * yb + 67.0 * yb3 + 135.0 * yb5 + 45.0 * yb7 + yb2 * (-39.0 + 36.0 * ln2) - 3.0 * yb4 * (283.0 + 300.0 * ln2) + yb8 * (-381.0 + 324.0 * ln2) + yb6 * (-521.0 + 540.0 * ln2))
                + 4.0 * xb6 * (24.0 - 154.0 * yb + 225.0 * yb2 + 794.0 * yb4 - 668.0 * yb6 - 6930.0 * yb8 - 5733.0 * yb10 - yb3 * (941.0 + 156.0 * ln2) + 18.0 * yb5 * (-173.0 + 172.0 * ln2) + 4.0 * yb9 * (2041.0 + 282.0 * ln2) - 8.0 * yb7 * (-565.0 + 612.0 * ln2)
                    + yb11 * (3813.0 + 828.0 * ln2)) + xb4 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2) - 16.0 * yb7 * (-143.0 + 801.0 * ln2)
                    + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + xb8 * (-16.0 + 80.0 * yb - 147.0 * yb2 + 326.0 * yb4 + 2008.0 * yb6 + 1818.0 * yb8 + 4203.0 * yb10 + yb3 * (505.0 - 396.0 * ln2) + 3.0 * yb11 * (493.0 + 756.0 * ln2)
                    - 16.0 * yb7 * (-143.0 + 801.0 * ln2) + yb9 * (-4054.0 + 2808.0 * ln2) + yb5 * (-8490.0 + 8136.0 * ln2)) + 3.0 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + log(16.0)) + yb2 * (1.0 + log(16.0)))
                + 3.0 * xb12 * power_of<3>(-1.0 + yb) * yb2 * power_of<2>(1.0 + yb) * (-1.0 + 3.0 * yb2) * (1.0 + yb * (-2.0 + log(16.0)) + yb2 * (1.0 + log(16.0)))) + vb2 * (-3.0 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2 - 3.0 * xb16 * yb3 * power_of<3>(-1.0 + yb2) * (-1.0 + 3.0 * yb2) * ln2
                - 24.0 * xb2 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2 - 24.0 * xb14 * yb3 * power_of<2>(-1.0 + yb2) * (-1.0 + 19.0 * yb2 + 6.0 * yb4) * ln2
                + 2.0 * xb8 * (64.0 - 416.0 * yb + 600.0 * yb2 + 2320.0 * yb4 - 1408.0 * yb6 - 19152.0 * yb8 - 15192.0 * yb10 - yb3 * (2536.0 + 57.0 * ln2) + 18.0 * yb5 * (-600.0 + 59.0 * ln2) + 3.0 * yb11 * (3704.0 + 87.0 * ln2) - 4.0 * yb7 * (-3328.0 + 423.0 * ln2)
                    + yb9 * (22096.0 + 426.0 * ln2)) + 8.0 * xb6 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                + 8.0 * xb10 * yb * (-1.0 + yb2) * (4.0 - 152.0 * yb3 - 432.0 * yb5 + 72.0 * yb7 + yb4 * (1896.0 - 60.0 * ln2) + yb6 * (952.0 + 39.0 * ln2) + yb2 * (24.0 + log(8.0)) + 6.0 * yb8 * (118.0 + log(8.0)))
                - 12.0 * xb4 * yb2 * power_of<2>(-1.0 + yb2) * (2.0 - 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0)) + 3.0 * yb5 * (2.0 + log(128.0))) - 12.0 * xb12 * yb2 * power_of<2>(-1.0 + yb2) * (2.0 - 18.0 * yb4 + 16.0 * yb3 * (1.0 + log(32.0)) - yb * (6.0 + log(32.0))
                    + 3.0 * yb5 * (2.0 + log(128.0)))) + 4.0 * xb2 * (4.0 * xb6 * (8.0 - 52.0 * yb + 75.0 * yb2 + 290.0 * yb4 - 176.0 * yb6 - 2394.0 * yb8 - 1899.0 * yb10 - yb3 * (317.0 + 39.0 * ln2) + 18.0 * yb5 * (-75.0 + 43.0 * ln2) + 3.0 * yb11 * (463.0 + 69.0 * ln2)
                    - 8.0 * yb7 * (-208.0 + 153.0 * ln2) + yb9 * (2762.0 + 282.0 * ln2)) + xb4 * yb * (-1.0 + yb2) * (8.0 - 304.0 * yb3 - 864.0 * yb5 + 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2)
                    + yb6 * (1904.0 + 1269.0 * ln2)) + xb8 * yb * (-1.0 + yb2) * (8.0 - 304.0 * yb3 - 864.0 * yb5 + 144.0 * yb7 + yb4 * (3792.0 - 1935.0 * ln2) + yb2 * (48.0 + 99.0 * ln2) + 3.0 * yb8 * (472.0 + 189.0 * ln2) + yb6 * (1904.0 + 1269.0 * ln2))
                + yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0)) + xb12 * yb3 * (-18.0 * yb2 * ln2 + 36.0 * yb4 * ln2 - 30.0 * yb6 * ln2 + 9.0 * yb8 * ln2 + log(8.0))
                + 6.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0)) + 3.0 * yb5 * (-1.0 + log(512.0))) + 6.0 * xb10 * yb2 * power_of<2>(-1.0 + yb2) * (-1.0 + 9.0 * yb4 - yb * (-3.0 + log(8.0)) + 8.0 * yb3 * (-1.0 + log(512.0))
                    + 3.0 * yb5 * (-1.0 + log(512.0))));

            const complex<double> term1 = -(64.0 * power_of<2>(lnmuhat)) / 81.0 + (1.0 / (1458.0 * xb4 * power_of<2>(yb + 3.0 * yb3))) * (27.0 * yb2 * (-1.0 + yb2) * (-1.0 + log(16.0)) + 27.0 * xb8 * yb2 * (-1.0 + yb2) * (-1.0 + log(16.0)) - 18.0 * xb2 * yb2 * (-1.0 + yb2) * (31.0 - 32.0 * ln2 + 3.0 * yb2 * (-7.0 + ln256))
                - 18.0 * xb6 * yb2 * (-1.0 + yb2) * (31.0 - 32.0 * ln2 + 3.0 * yb2 * (-7.0 + ln256)) - 2.0 * xb4 * (-528.0 + yb4 * (4449.0 + 72.0 * pisqu - 900.0 * ln2 - 6912.0 * ln2squ) + yb2 * (-2029.0 + 12.0 * pisqu + 468.0 * ln2 - 1152.0 * ln2squ) + 36.0 * yb6 * (359.0 + 3.0 * pisqu - 288.0 * ln2squ + log(4096.0))));

            const complex<double> f29bpart1 = (32.0 * power_of<2>(log(xb))) / 81.0 + (num4 * dilog((-1.0i) * xb) + num2 * dilog(1.0i * xb) + (num3 * power_of<2>(log(1.0 - 1.0i * xb))) / 2.0 + (num5 * power_of<2>(log(1.0 + 1.0i * xb))) / 2.0 + num2 * log(1.0 - 1.0i * xb) * log(xb) + num4 * log(1.0 + 1.0i * xb) * log(xb)) / denom1
                + ((num6 + num7) * li2half - num6 * dilog((1.0 - yb) / 2.0) - num7 * dilog((1.0 + yb) / 2.0) + (num6 * ln2 + num1 * log(xb)) * ln1myb + (num7 * power_of<2>(ln1myb)) / 2.0
                  + (num7 * ln2 + 2.0 * num6 * log(xb)) * ln1pyb + (num6 * power_of<2>(ln1pyb)) / 2.0) / denom0 + (-(num12 * dilog(-(vb / w4))) - num12 * dilog(w4 / (-1.0 + w4)) + num12 * dilog(w4 / (1.0 + w4))
                  + num12 * dilog((vb + w4) / (-1.0 + w4)) - num12 * dilog((vb + w4) / (1.0 + w4)) - num12 * dilog(vb / w5) + num12 * dilog(w5 / (-1.0 + w5)) - num12 * dilog(w5 / (1.0 + w5)) - num12 * dilog((-vb + w5) / (-1.0 + w5))
                  + num12 * dilog((-vb + w5) / (1.0 + w5)) + num12 * dilog(w4 / (w4 - w7)) - num12 * dilog((vb + w4) / (w4 - w7)) + num12 * dilog(w5 / (w5 + w7)) - num12 * dilog((-vb + w5) / (w5 + w7)) - 4.0 * num12 * ln2 * log(1.0 - vb)
                  + 4.0 * num12 * ln2 * log(1.0 + vb) + num12 * log((1.0 + vb) / (1.0 - w4)) * log((vb + w4) / w4) - num12 * log((1.0 - vb) / (1.0 + w4)) * log((vb + w4) / w4) - num12 * log(1.0 - vb / w5) * log((-1.0 + vb) / (-1.0 + w5))
                  + num12 * log(1.0 - vb / w5) * log((1.0 + vb) / (1.0 + w5)) - num12 * log((vb + w4) / w4) * log(-((vb + w7) / (w4 - w7))) - num12 * log(1.0 - vb / w5) * log((vb + w7) / (w5 + w7))
                  + (-2.0 * num12 * log(1.0 - vb) + 2.0 * num12 * log(1.0 + vb)) * log(xb) - (2.0 * 1.0i) * num12 * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (vb + w4) / w4, 1.0 + w4inv)
                  + (2.0 * 1.0i) * num12 * M_PI * log((-1.0 + w4) / w4) * my_sign(imag(w4)) * T(1.0, (vb + w4) / w4, (-1.0 + w4) / w4) - (2.0 * 1.0i) * num12 * M_PI * log(1.0 - w7 / w4) * my_sign(imag(w4 / w7)) * T(1.0, (vb + w4) / w4, 1.0 - w7 / w4)
                  + (2.0 * 1.0i) * num12 * M_PI * log(1.0 + w5inv) * my_sign(-imag(w5)) * T(1.0, 1.0 - vb / w5, 1.0 + w5inv) - (2.0 * 1.0i) * num12 * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - vb / w5, (-1.0 + w5) / w5)
                  - (2.0 * 1.0i) * num12 * M_PI * log((w5 + w7) / w5) * my_sign(-imag(w5 / w7)) * T(1.0, 1.0 - vb / w5, (w5 + w7) / w5)) / denom4 + (-(num11 * dilog(tb / w4)) - num11 * dilog(w4 / (-1.0 + w4)) - num10 * dilog(w4 / (1.0 + w4))
                  + num11 * dilog((-tb + w4) / (-1.0 + w4)) + num10 * dilog((-tb + w4) / (1.0 + w4)) - num11 * dilog(-(tb / w5)) - num10 * dilog(w5 / (-1.0 + w5)) - num11 * dilog(w5 / (1.0 + w5)) + num10 * dilog((tb + w5) / (-1.0 + w5))
                  + num11 * dilog((tb + w5) / (1.0 + w5)) - num10 * dilog(w4 / (w4 - w7)) + num10 * dilog((-tb + w4) / (w4 - w7)) - num10 * dilog(w5 / (w5 + w7)) + num10 * dilog((tb + w5) / (w5 + w7)) + 4.0 * num11 * ln2 * log(1.0 - tb)
                  + 4.0 * num10 * ln2 * log(1.0 + tb) + num11 * log(1.0 - tb / w4) * log((-1.0 + tb) / (-1.0 + w4)) + num10 * log(1.0 - tb / w4) * log((1.0 + tb) / (1.0 + w4)) + num10 * log((1.0 + tb) / (1.0 - w5)) * log((tb + w5) / w5)
                  + num11 * log((1.0 - tb) / (1.0 + w5)) * log((tb + w5) / w5) + num10 * log(1.0 - tb / w4) * log((tb - w7) / (w4 - w7)) + num10 * log((tb + w5) / w5) * log((-tb + w7) / (w5 + w7))
                  + (2.0 * num11 * log(1.0 - tb) + 2.0 * num10 * log(1.0 + tb)) * log(xb) + (2.0 * 1.0i) * num10 * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - tb / w4, 1.0 + w4inv)
                  + (2.0 * 1.0i) * num11 * M_PI * log((-1.0 + w4) / w4) * my_sign(imag(w4)) * T(1.0, 1.0 - tb / w4, (-1.0 + w4) / w4) + (2.0 * 1.0i) * num10 * M_PI * log(1.0 - w7 / w4) * my_sign(imag(w4 / w7)) * T(1.0, 1.0 - tb / w4, 1.0 - w7 / w4)
                  + (2.0 * 1.0i) * num11 * M_PI * log(1.0 + w5inv) * my_sign(-imag(w5)) * T(1.0, (tb + w5) / w5, 1.0 + w5inv) + (2.0 * 1.0i) * num10 * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (tb + w5) / w5, (-1.0 + w5) / w5)
                  + (2.0 * 1.0i) * num10 * M_PI * log((w5 + w7) / w5) * my_sign(-imag(w5 / w7)) * T(1.0, (tb + w5) / w5, (w5 + w7) / w5)) / denom3 + ((-num8 - num9) * dilog(-1.0i / (-1.0i + wx3)) + (-num8 - num9) * dilog(1.0i / (1.0i + wx3))
                  + (-num8 - num9) * dilog(-1.0i / (-1.0i + wx4)) + (-num8 - num9) * dilog(1.0i / (1.0i + wx4)) + num8 * dilog((1.0i - xb) / (1.0i + wx3)) + num8 * dilog((1.0i - xb) / (1.0i + wx4)) + num13 * dilog(-(xb / wx3)) + num14 * dilog(xb / wx3)
                  + num13 * dilog(-(xb / wx4)) + num14 * dilog(xb / wx4) + num9 * dilog((-1.0i + xb) / (-1.0i + wx3)) + num9 * dilog((-1.0i + xb) / (-1.0i + wx4)) + num8 * dilog((1.0i + xb) / (1.0i - wx3)) + num9 * dilog((1.0i + xb) / (1.0i + wx3))
                  + num8 * dilog((1.0i + xb) / (1.0i - wx4)) + num9 * dilog((1.0i + xb) / (1.0i + wx4)) + num9 * log((wx3 - xb) / (1.0i + wx3)) * log(1.0 - 1.0i * xb) + num9 * log((wx3 - xb) / (-1.0i + wx3)) * log(1.0 + 1.0i * xb)
                  + (2.0 * num13 * ln2 + num13 * log(xb)) * log((wx3 + xb) / wx3) + 2.0 * num13 * ln2 * log((wx4 + xb) / wx4) + log(1.0 - 1.0i * xb) * (num9 * log((wx4 - xb) / (1.0i + wx4)) + num8 * log((wx3 + xb) / (-1.0i + wx3))
                    + num8 * log((wx4 + xb) / (-1.0i + wx4))) + log(1.0 + 1.0i * xb) * (num9 * log((wx4 - xb) / (-1.0i + wx4)) + num8 * log((wx3 + xb) / (1.0i + wx3)) + num8 * log((wx4 + xb) / (1.0i + wx4))) + 2.0 * num14 * ln2 * log(1.0 - xb / wx3)
                  + 2.0 * num14 * ln2 * log(1.0 - xb / wx4) + log(xb) * (num13 * log((wx4 + xb) / wx4) + num14 * log(1.0 - xb / wx3) + num14 * log(1.0 - xb / wx4)) + (2.0 * 1.0i) * num9 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                  + (2.0 * 1.0i) * num8 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * num9 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                  + (2.0 * 1.0i) * num8 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * num8 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                  + (2.0 * 1.0i) * num9 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * num8 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                  + (2.0 * 1.0i) * num9 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)) / denom2;

            const complex<double> f29bpart2 = num15 * (((log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * power_of<2>(log(1.0 - 1.0i * xb))
                + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * power_of<2>(log(1.0 + 1.0i * xb))
                + dilog(1.0 / (1.0 - w7)) * (4.0 * log(1.0 - 1.0i * xb) + 4.0 * log(1.0 + 1.0i * xb) - 4.0 * log(xb)) + dilog((1.0 - tb) / 2.0) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb))
                + dilog(-tb) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb)) + dilog((1.0 + tb) / 2.0) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb))
                + dilog((1.0 - vb) / 2.0) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb)) + dilog(vb) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb))
                + dilog((1.0 + vb) / 2.0) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb)) + dilog((1.0 + tb) / (1.0 + w7)) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb))
                + dilog((1.0 - vb) / (1.0 + w7)) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb)) + dilog(tb) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb))
                + dilog(-vb) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb)) + dilog((1.0 + vb) / (1.0 - w7)) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb))
                + dilog((-1.0 + tb) / (-1.0 + w7)) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb)) + dilog(1.0 / (1.0 + w7)) * (-4.0 * log(1.0 - 1.0i * xb) - 4.0 * log(1.0 + 1.0i * xb) + 4.0 * log(xb))
                + dilog(1.0 / 2.0 - (1.0i / 2.0) * xb) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb)
                    - 2.0 * ln1pyb) + dilog((1.0 + 1.0i * xb) / 2.0) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb
                    + 4.0 * log(yb) - 2.0 * ln1pyb) + dilog((-1.0i) * xb) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7)
                    - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb) + dilog(1.0i * xb) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7)
                    + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb) + dilog((1.0i - xb) / (1.0i + wx3)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb)
                    + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb) + dilog((1.0i - xb) / (1.0i + wx4)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb)
                    + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb) + dilog((-1.0i + xb) / (-1.0i + wx3)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb)
                    - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb) + dilog((-1.0i + xb) / (-1.0i + wx4)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb)
                    - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog((1.0i + xb) / (1.0i - wx3)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog((1.0i + xb) / (1.0i + wx3)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog((1.0i + xb) / (1.0i - wx4)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog((1.0i + xb) / (1.0i + wx4)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog(-(xb / wx3)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog(xb / wx3) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog(-(xb / wx4)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog(xb / wx4) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog(-1.0i / (-1.0i + wx3)) * (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7) + 2.0 * ln1myb - 4.0 * log(yb)
                    + 2.0 * ln1pyb) + dilog(1.0i / (1.0i + wx3)) * (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7) + 2.0 * ln1myb
                    - 4.0 * log(yb) + 2.0 * ln1pyb) + dilog(-1.0i / (-1.0i + wx4)) * (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7)
                    + 2.0 * ln1myb - 4.0 * log(yb) + 2.0 * ln1pyb) + dilog(1.0i / (1.0i + wx4)) * (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7)
                    - 2.0 * log((vb + w7) / w7) + 2.0 * ln1myb - 4.0 * log(yb) + 2.0 * ln1pyb) + li2half * (4.0 * log(1.0 - tb) + 4.0 * log(tb) - 4.0 * log(1.0 + tb) - 4.0 * log(1.0 - vb) + 4.0 * log(vb) + 4.0 * log(1.0 + vb)
                    - 4.0 * log(1.0 - tb / w7) - 4.0 * log((vb + w7) / w7) - 8.0 * log(1.0 - 1.0i * xb) - 8.0 * log(1.0 + 1.0i * xb) + 8.0 * log(xb) + 4.0 * ln1myb - 8.0 * log(yb) + 4.0 * ln1pyb)
                + log(1.0 - 1.0i * xb) * (power_of<2>(log(1.0 - tb)) + 2.0 * ln2 * log(tb) - 4.0 * ln2 * log(1.0 + tb) + power_of<2>(log(1.0 + tb)) - 4.0 * ln2 * log(1.0 - vb) + power_of<2>(log(1.0 - vb)) + 2.0 * ln2 * log(vb) + power_of<2>(log(1.0 + vb))
                    - 2.0 * ln2 * log(1.0 - tb / w7) - 2.0 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) + 2.0 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7)) - 2.0 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7)) - 2.0 * ln2 * log((vb + w7) / w7)
                    + 2.0 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7)) + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx3 - xb) / (1.0i + wx3))
                    + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx4 - xb) / (1.0i + wx4))
                    + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7)) * log(xb)
                    + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx3 + xb) / (-1.0i + wx3))
                    + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx4 + xb) / (-1.0i + wx4))
                    - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7) + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7) + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7)
                    - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7)) + log(1.0 + 1.0i * xb) * (power_of<2>(log(1.0 - tb)) + 2.0 * ln2 * log(tb) - 4.0 * ln2 * log(1.0 + tb) + power_of<2>(log(1.0 + tb)) - 4.0 * ln2 * log(1.0 - vb)
                    + power_of<2>(log(1.0 - vb)) + 2.0 * ln2 * log(vb) + power_of<2>(log(1.0 + vb)) - 2.0 * ln2 * log(1.0 - tb / w7) - 2.0 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) + 2.0 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7))
                    - 2.0 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7)) - 2.0 * ln2 * log((vb + w7) / w7) + 2.0 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7))
                    + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx3 - xb) / (-1.0i + wx3))
                    + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx4 - xb) / (-1.0i + wx4))
                    + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7)) * log(xb)
                    + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx3 + xb) / (1.0i + wx3))
                    + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx4 + xb) / (1.0i + wx4))
                    - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7) + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7) + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7)
                    - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7)) + log(xb) * (2.0 * ln2 * log(1.0 - tb) - power_of<2>(log(1.0 - tb)) + 2.0 * ln2 * log(1.0 + tb) - power_of<2>(log(1.0 + tb)) + 2.0 * ln2 * log(1.0 - vb)
                    - power_of<2>(log(1.0 - vb)) + 2.0 * ln2 * log(1.0 + vb) - power_of<2>(log(1.0 + vb)) + 2.0 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) - 2.0 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7)) + 2.0 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7))
                    - 2.0 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7)) + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx3 + xb) / wx3)
                    + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx4 + xb) / wx4)
                    + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log(1.0 - xb / wx3)
                    + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log(1.0 - xb / wx4)
                    + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7)
                    + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7)) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) + ln1myb * (power_of<2>(log(1.0 - 1.0i * xb)) + power_of<2>(log(1.0 + 1.0i * xb))
                    + log(1.0 - 1.0i * xb) * (2.0 * ln2 - log((wx3 - xb) / (1.0i + wx3)) - log((wx4 - xb) / (1.0i + wx4)) - 2.0 * log(xb) - log((wx3 + xb) / (-1.0i + wx3)) - log((wx4 + xb) / (-1.0i + wx4)))
                    + log(1.0 + 1.0i * xb) * (2.0 * ln2 - log((wx3 - xb) / (-1.0i + wx3)) - log((wx4 - xb) / (-1.0i + wx4)) - 2.0 * log(xb) - log((wx3 + xb) / (1.0i + wx3)) - log((wx4 + xb) / (1.0i + wx4)))
                    + log(xb) * (log((wx3 + xb) / wx3) + log((wx4 + xb) / wx4) + log(1.0 - xb / wx3) + log(1.0 - xb / wx4)) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)) + ln1pyb * (power_of<2>(log(1.0 - 1.0i * xb)) + power_of<2>(log(1.0 + 1.0i * xb))
                    + log(1.0 - 1.0i * xb) * (2.0 * ln2 - log((wx3 - xb) / (1.0i + wx3)) - log((wx4 - xb) / (1.0i + wx4)) - 2.0 * log(xb) - log((wx3 + xb) / (-1.0i + wx3)) - log((wx4 + xb) / (-1.0i + wx4)))
                    + log(1.0 + 1.0i * xb) * (2.0 * ln2 - log((wx3 - xb) / (-1.0i + wx3)) - log((wx4 - xb) / (-1.0i + wx4)) - 2.0 * log(xb) - log((wx3 + xb) / (1.0i + wx3)) - log((wx4 + xb) / (1.0i + wx4)))
                    + log(xb) * (log((wx3 + xb) / wx3) + log((wx4 + xb) / wx4) + log(1.0 - xb / wx3) + log(1.0 - xb / wx4)) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                    - (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)) + log(yb) * (-2.0 * power_of<2>(log(1.0 - 1.0i * xb)) - 2.0 * power_of<2>(log(1.0 + 1.0i * xb))
                    + log(1.0 - 1.0i * xb) * (-4.0 * ln2 + 2.0 * log((wx3 - xb) / (1.0i + wx3)) + 2.0 * log((wx4 - xb) / (1.0i + wx4)) + 4.0 * log(xb) + 2.0 * log((wx3 + xb) / (-1.0i + wx3)) + 2.0 * log((wx4 + xb) / (-1.0i + wx4)))
                    + log(1.0 + 1.0i * xb) * (-4.0 * ln2 + 2.0 * log((wx3 - xb) / (-1.0i + wx3)) + 2.0 * log((wx4 - xb) / (-1.0i + wx4)) + 4.0 * log(xb) + 2.0 * log((wx3 + xb) / (1.0i + wx3)) + 2.0 * log((wx4 + xb) / (1.0i + wx4)))
                    + log(xb) * (-2.0 * log((wx3 + xb) / wx3) - 2.0 * log((wx4 + xb) / wx4) - 2.0 * log(1.0 - xb / wx3) - 2.0 * log(1.0 - xb / wx4)) + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                    + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                    + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                    + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                    + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4))) / denom5);

            const complex<double> f29bpart3 = - (num15 / denom5) * f279b_helper(clp);

            const complex<double> f29bpart4 = ln2 * num15 * ((-16.0 * dilog(wx3 / (-1.0i + wx3)) - 16.0 * dilog(wx3 / (1.0i + wx3)) - 16.0 * dilog(wx4 / (-1.0i + wx4)) - 16.0 * dilog(wx4 / (1.0i + wx4)) + 8.0 * dilog((wx3 - xb) / (-1.0i + wx3)) + 8.0 * dilog((wx3 - xb) / (1.0i + wx3))
                + 8.0 * dilog((wx4 - xb) / (-1.0i + wx4)) + 8.0 * dilog((wx4 - xb) / (1.0i + wx4)) + 8.0 * dilog(-(xb / wx3)) + 8.0 * dilog(xb / wx3) + 8.0 * dilog(-(xb / wx4)) + 8.0 * dilog(xb / wx4) + 8.0 * dilog((wx3 + xb) / (-1.0i + wx3))
                + 8.0 * dilog((wx3 + xb) / (1.0i + wx3)) + 8.0 * dilog((wx4 + xb) / (-1.0i + wx4)) + 8.0 * dilog((wx4 + xb) / (1.0i + wx4)) + (-4.0 * log(tb) - 4.0 * log(vb) + 4.0 * log(1.0 - tb / w7) + 4.0 * log((vb + w7) / w7)) * log(1.0 - 1.0i * xb)
                + (-4.0 * log(tb) - 4.0 * log(vb) + 4.0 * log(1.0 - tb / w7) + 4.0 * log((vb + w7) / w7)) * log(1.0 + 1.0i * xb) + (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb)
                    - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7) + 8.0 * log((1.0i - xb) / (1.0i + wx3)) + 8.0 * log((1.0i + xb) / (1.0i - wx3))) * log((wx3 + xb) / wx3)
                + (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7) + 8.0 * log((1.0i - xb) / (1.0i + wx4)) + 8.0 * log((1.0i + xb) / (1.0i - wx4))) * log((wx4 + xb) / wx4)
                + (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7) + 8.0 * log((-1.0i + xb) / (-1.0i + wx3))
                    + 8.0 * log((1.0i + xb) / (1.0i + wx3))) * log(1.0 - xb / wx3) + (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7)
                    + 8.0 * log((-1.0i + xb) / (-1.0i + wx4)) + 8.0 * log((1.0i + xb) / (1.0i + wx4))) * log(1.0 - xb / wx4) + (2.0 * log((wx3 + xb) / wx3) + 2.0 * log((wx4 + xb) / wx4) + 2.0 * log(1.0 - xb / wx3) + 2.0 * log(1.0 - xb / wx4)) * ln1myb
                + (8.0 * log(1.0 - 1.0i * xb) + 8.0 * log(1.0 + 1.0i * xb) - 4.0 * log((wx3 + xb) / wx3) - 4.0 * log((wx4 + xb) / wx4) - 4.0 * log(1.0 - xb / wx3) - 4.0 * log(1.0 - xb / wx4)) * log(yb)
                + (2.0 * log((wx3 + xb) / wx3) + 2.0 * log((wx4 + xb) / wx4) + 2.0 * log(1.0 - xb / wx3) + 2.0 * log(1.0 - xb / wx4)) * ln1pyb + (16.0 * 1.0i) * M_PI * log((-1.0i + wx3) / wx3) * my_sign(-real(wx3)) * T(1.0, (wx3 + xb) / wx3, (-1.0i + wx3) / wx3)
                + (16.0 * 1.0i) * M_PI * log((1.0i + wx3) / wx3) * my_sign(real(wx3)) * T(1.0, (wx3 + xb) / wx3, (1.0i + wx3) / wx3) + (16.0 * 1.0i) * M_PI * log((-1.0i + wx4) / wx4) * my_sign(-real(wx4)) * T(1.0, (wx4 + xb) / wx4, (-1.0i + wx4) / wx4)
                + (16.0 * 1.0i) * M_PI * log((1.0i + wx4) / wx4) * my_sign(real(wx4)) * T(1.0, (wx4 + xb) / wx4, (1.0i + wx4) / wx4) + (16.0 * 1.0i) * M_PI * log((-1.0i + wx3) / wx3) * my_sign(-real(wx3)) * T(1.0, 1.0 - xb / wx3, (-1.0i + wx3) / wx3)
                + (16.0 * 1.0i) * M_PI * log((1.0i + wx3) / wx3) * my_sign(real(wx3)) * T(1.0, 1.0 - xb / wx3, (1.0i + wx3) / wx3) + (16.0 * 1.0i) * M_PI * log((-1.0i + wx4) / wx4) * my_sign(-real(wx4)) * T(1.0, 1.0 - xb / wx4, (-1.0i + wx4) / wx4)
                + (16.0 * 1.0i) * M_PI * log((1.0i + wx4) / wx4) * my_sign(real(wx4)) * T(1.0, 1.0 - xb / wx4, (1.0i + wx4) / wx4)) / denom5);

            const complex<double> f29bpart5 = term1 + (-(logs4 * dilog((-1.0i) * xb)) - logs4 * dilog(1.0i * xb)) / denom8 + (logs15 * log(1.0 - 1.0i * xb) + logs16 * log(1.0 + 1.0i * xb)) / denom12
                + ((logs17 + logs18) * li2half - logs17 * dilog(1.0 / 2.0 - (1.0i / 2.0) * xb) - logs18 * dilog((1.0 + 1.0i * xb) / 2.0) + logs17 * ln2 * log(1.0 - 1.0i * xb) + logs18 * ln2 * log(1.0 + 1.0i * xb)) / denom1 + (logs3 * log(xb)) / denom7
                + (logs5 * ln1myb + logs6 * ln1pyb) / denom9 + (12.0 * yb * lnmuhat - 31.0 * yb3 * lnmuhat + (6.0 * lnmuhat - 18.0 * yb2 * lnmuhat) * ln1myb + (-6.0 * lnmuhat + 18.0 * yb2 * lnmuhat) * ln1pyb) / denom13
                + ((-(logs1 * log(1.0 - 1.0i * xb)) - logs1 * log(1.0 + 1.0i * xb)) * ln1myb + (-(logs2 * log(1.0 - 1.0i * xb)) - logs2 * log(1.0 + 1.0i * xb)) * ln1pyb) / denom6
                + ((logs7 + logs9) * li2half - logs7 * dilog((1.0 - tb) / 2.0) - logs7 * dilog(-tb) - logs10 * dilog(tb) - logs9 * dilog((1.0 + tb) / 2.0) - logs9 * dilog(1.0 / (1.0 - w7)) + logs9 * dilog((-1.0 + tb) / (-1.0 + w7))
                - logs8 * dilog(1.0 / (1.0 + w7)) + logs8 * dilog((1.0 + tb) / (1.0 + w7)) + logs7 * ln2 * log(1.0 - tb) + (logs10 * power_of<2>(log(1.0 - tb))) / 2.0 + logs9 * ln2 * log(1.0 + tb) + (logs8 * power_of<2>(log(1.0 + tb))) / 2.0
                + logs9 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) + logs8 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7)) + (logs20 * log(1.0 - tb) + logs19 * log(1.0 + tb)) * log(1.0 - 1.0i * xb)
                + (logs20 * log(1.0 - tb) + logs19 * log(1.0 + tb)) * log(1.0 + 1.0i * xb) + (2.0 * 1.0i) * logs9 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7) + (2.0 * 1.0i) * logs8 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7)) / denom10
                + ((logs12 + logs14) * li2half - logs12 * dilog((1.0 - vb) / 2.0) - logs11 * dilog(-vb) - logs14 * dilog(vb) - logs14 * dilog((1.0 + vb) / 2.0) - logs12 * dilog(1.0 / (1.0 - w7))
                + logs12 * dilog((1.0 + vb) / (1.0 - w7)) - logs13 * dilog(1.0 / (1.0 + w7)) + logs13 * dilog((1.0 - vb) / (1.0 + w7)) + logs12 * ln2 * log(1.0 - vb) + (logs13 * power_of<2>(log(1.0 - vb))) / 2.0 + logs14 * ln2 * log(1.0 + vb)
                + (logs11 * power_of<2>(log(1.0 + vb))) / 2.0 + logs12 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7)) + logs13 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7)) + (logs21 * log(1.0 - vb) + logs22 * log(1.0 + vb)) * log(1.0 - 1.0i * xb)
                + (logs21 * log(1.0 - vb) + logs22 * log(1.0 + vb)) * log(1.0 + 1.0i * xb) + (2.0 * 1.0i) * logs13 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7) + (2.0 * 1.0i) * logs12 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7)) / denom11;

            return f29bpart1 + f29bpart2 + f29bpart3 + f29bpart4 + f29bpart5;
        }

    }
}
