/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2022 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/polylog.hh>
#include <eos/maths/multiplepolylog-li22.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>
#include <eos/utils/stringify.hh>

#include <cmath>
#include <complex>

namespace eos
{
    using std::complex;
    using std::log;
    using std::sqrt;
    using std::real;

    namespace agv_2019a
    {
        complex<double> f17d(const CharmLoopsParameters & clp)
        {
            // f17d = - f27d / (2 * N_c)
            return - f27d(clp) / (2.0 * 3.0);
        }

        complex<double> f27d(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xd = clp.xd;
            const complex<double> yd = clp.yd;

            const complex<double> ydinv = 1.0 / yd;

            const double imydinv = imag(ydinv);

            // Powers of xd and yd

            const complex<double> xd2 = power_of<2>(xd);
            const complex<double> xd3 = power_of<3>(xd);
            const complex<double> xd4 = power_of<4>(xd);
            const complex<double> xd5 = power_of<5>(xd);
            const complex<double> xd6 = power_of<6>(xd);
            const complex<double> xd7 = power_of<7>(xd);
            const complex<double> xd8 = power_of<8>(xd);
            const complex<double> xd9 = power_of<9>(xd);
            const complex<double> xd10 = power_of<10>(xd);
            const complex<double> xd11 = power_of<11>(xd);
            const complex<double> xd12 = power_of<12>(xd);
            const complex<double> xd13 = power_of<13>(xd);
            const complex<double> xd14 = power_of<14>(xd);
            const complex<double> xd15 = power_of<15>(xd);
            const complex<double> xd16 = power_of<16>(xd);
            const complex<double> xd18 = power_of<18>(xd);
            const complex<double> xd20 = power_of<20>(xd);

            const complex<double> yd2 = power_of<2>(yd);
            const complex<double> yd3 = power_of<3>(yd);
            const complex<double> yd4 = power_of<4>(yd);
            const complex<double> yd5 = power_of<5>(yd);
            const complex<double> yd6 = power_of<6>(yd);
            const complex<double> yd7 = power_of<7>(yd);
            const complex<double> yd8 = power_of<8>(yd);

            // weights appearing in the GPLs [AGV:2019A] p. 34

            const double wx3 = 2.0 + sqrt(3);
            const double wx4 = 2.0 - sqrt(3);

            const complex<double> w4 = (2.0 * xd) / power_of<2>(1.0 - xd);
            const complex<double> w5 = (2.0 * xd) / power_of<2>(1.0 + xd);
            const complex<double> w7 = (8.0 * xd2) / (1.0 - 6.0 * xd2 + xd4);

            const double wx3inv = wx4;
            const double wx4inv = wx3;

            const complex<double> w4inv = 1.0 / w4;
            const complex<double> w5inv = 1.0 / w5;
            const complex<double> w7inv = 1.0 / w7;

            // Polylogs of xd and yd

            const complex<double> ln1pyd = log(1.0 + yd);
            const complex<double> ln1myd = log(1.0 - yd);

            // Appearing numerators and denominators as polynomials of xd and yd

            const complex<double> num1 = 8.0 * power_of<2>(1.0 + xd2) * (3.0 * yd4 + 3.0 * xd4 * yd4 + xd2 * (4.0 - 4.0 * yd2 + 6.0 * yd4));
            const complex<double> num2 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * yd * (1.0 + yd) * (yd4 + xd16 * yd4 + 4.0 * xd2 * (yd2 + yd4) + 4.0 * xd14 * (yd2 + yd4)
                    - 4.0 * xd4 * (-4.0 + 26.0 * yd2 + 35.0 * yd4) - 4.0 * xd12 * (-4.0 + 26.0 * yd2 + 35.0 * yd4) + 4.0 * xd6 * (-48.0 - 369.0 * yd2 + 175.0 * yd4)
                    + 4.0 * xd10 * (-48.0 - 369.0 * yd2 + 175.0 * yd4) - 2.0 * xd8 * (1232.0 - 680.0 * yd2 + 181.0 * yd4));
            const complex<double> num3 = 4.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * power_of<2>(1.0 + yd) * ((-1.0 + yd) * yd2 + xd8 * (-1.0 + yd) * yd2 + 4.0 * xd2 * yd2 * (5.0 + 11.0 * yd)
                    + 4.0 * xd6 * yd2 * (5.0 + 11.0 * yd) + 2.0 * xd4 * (16.0 - 16.0 * yd + 5.0 * yd2 + 59.0 * yd3));
            const complex<double> num4 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd6 + xd20 * yd6 + xd4 * yd2 * (16.0 - 288.0 * yd2 + 189.0 * yd4)
                    + xd16 * yd2 * (16.0 - 288.0 * yd2 + 189.0 * yd4) + 8.0 * xd6 * yd2 * (-20.0 + 212.0 * yd2 + 359.0 * yd4)
                    + 8.0 * xd14 * yd2 * (-20.0 + 212.0 * yd2 + 359.0 * yd4) + xd8 * (1024.0 + 240.0 * yd2 + 37152.0 * yd4 - 20670.0 * yd6)
                    + xd12 * (1024.0 + 240.0 * yd2 + 37152.0 * yd4 - 20670.0 * yd6) + xd2 * (8.0 * yd4 - 46.0 * yd6) + xd18 * (8.0 * yd4 - 46.0 * yd6)
                    + 128.0 * 1.0i * xd5 * (yd2 - 4.0 * yd4 + 3.0 * yd6) - 1664.0 * 1.0i * xd7 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    + 1664.0 * 1.0i * xd13 * (yd2 - 4.0 * yd4 + 3.0 * yd6) - 128.0 * 1.0i * xd15 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    + 256.0 * 1.0i * xd9 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6) - 256.0 * 1.0i * xd11 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6)
                    + 4.0 * xd10 * (-3584.0 + 28880.0 * yd2 - 27476.0 * yd4 + 8827.0 * yd6));
            const complex<double> num5 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd6 + xd20 * yd6 + xd4 * yd2 * (16.0 - 288.0 * yd2 + 189.0 * yd4)
                    + xd16 * yd2 * (16.0 - 288.0 * yd2 + 189.0 * yd4) + 8.0 * xd6 * yd2 * (-20.0 + 212.0 * yd2 + 359.0 * yd4)
                    + 8.0 * xd14 * yd2 * (-20.0 + 212.0 * yd2 + 359.0 * yd4) + xd8 * (1024.0 + 240.0 * yd2 + 37152.0 * yd4 - 20670.0 * yd6)
                    + xd12 * (1024.0 + 240.0 * yd2 + 37152.0 * yd4 - 20670.0 * yd6) + xd2 * (8.0 * yd4 - 46.0 * yd6) + xd18 * (8.0 * yd4 - 46.0 * yd6)
                    - 128.0 * 1.0i * xd5 * (yd2 - 4.0 * yd4 + 3.0 * yd6) + 1664.0 * 1.0i * xd7 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    - 1664.0 * 1.0i * xd13 * (yd2 - 4.0 * yd4 + 3.0 * yd6) + 128.0 * 1.0i * xd15 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    - 256.0 * 1.0i * xd9 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6) + 256.0 * 1.0i * xd11 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6)
                    + 4.0 * xd10 * (-3584.0 + 28880.0 * yd2 - 27476.0 * yd4 + 8827.0 * yd6));
            const complex<double> num6 = 4.0 * power_of<2>(1.0 + xd2) * power_of<2>(-1.0 + yd) * (1.0 + yd) * (yd2 * (1.0 + yd) + xd8 * yd2 * (1.0 + yd) + 4.0 * xd2 * yd2 * (-5.0 + 11.0 * yd)
                + 4.0 * xd6 * yd2 * (-5.0 + 11.0 * yd) + 2.0 * xd4 * (-16.0 - 16.0 * yd - 5.0 * yd2 + 59.0 * yd3));
            const complex<double> num7 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * power_of<2>(1.0 + yd) * (3.0 * (-1.0 + yd) * yd2 + 3.0 * xd8 * (-1.0 + yd) * yd2
                + 4.0 * xd2 * (1.0 - yd + 16.0 * yd2 + 32.0 * yd3) + 4.0 * xd6 * (1.0 - yd + 16.0 * yd2 + 32.0 * yd3)
                + xd4 * (72.0 - 72.0 * yd + 70.0 * yd2 + 314.0 * yd3));
            const complex<double> num8 = 128.0 * xd2 * power_of<4>(1.0 + xd2) * (-1.0 + yd) * yd3 * (1.0 + yd);
            const complex<double> num9 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 3.0 * yd4 + xd8 * (yd2 + 3.0 * yd4) - 4.0 * xd2 * (-3.0 + 5.0 * yd2 + 10.0 * yd4)
                - 4.0 * xd6 * (-3.0 + 5.0 * yd2 + 10.0 * yd4) - 2.0 * xd4 * (84.0 - 171.0 * yd2 + 139.0 * yd4));
            const complex<double> num10 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 3.0 * yd4 + xd8 * (yd2 + 3.0 * yd4) - 4.0 * xd2 * (1.0 + 5.0 * yd2 + 6.0 * yd4)
                - 4.0 * xd6 * (1.0 + 5.0 * yd2 + 6.0 * yd4) - 2.0 * xd4 * (36.0 - 43.0 * yd2 + 59.0 * yd4));
            const complex<double> num11 = power_of<2>(1.0 + xd2) * power_of<2>(-1.0 + yd) * (1.0 + yd) * (3.0 * yd2 * (1.0 + yd) + 3.0 * xd8 * yd2 * (1.0 + yd)
                + 4.0 * xd2 * (-1.0 - yd - 16.0 * yd2 + 32.0 * yd3) + 4.0 * xd6 * (-1.0 - yd - 16.0 * yd2 + 32.0 * yd3)
                + xd4 * (-72.0 - 72.0 * yd - 70.0 * yd2 + 314.0 * yd3));
            const complex<double> num12 = power_of<2>(1.0 + xd2) * (1.0 + 4.0 * xd + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd4 - 4.0 * xd * yd4 - 4.0 * xd13 * yd4 + xd14 * yd4
                    - 8.0 * xd3 * yd2 * (2.0 + yd2) - 8.0 * xd11 * yd2 * (2.0 + yd2) + xd4 * yd2 * (-44.0 + 161.0 * yd2) + xd10 * yd2 * (-44.0 + 161.0 * yd2)
                    + 4.0 * xd5 * yd2 * (48.0 + 193.0 * yd2) + 4.0 * xd9 * yd2 * (48.0 + 193.0 * yd2) + xd6 * (256.0 + 360.0 * yd2 - 133.0 * yd4)
                    + xd8 * (256.0 + 360.0 * yd2 - 133.0 * yd4) + xd2 * (4.0 * yd2 - 29.0 * yd4) + xd12 * (4.0 * yd2 - 29.0 * yd4)
                    - 16.0 * xd7 * (64.0 - 410.0 * yd2 + 223.0 * yd4));
            const complex<double> num13 = power_of<2>(1.0 + xd2) * (1.0 - 4.0 * xd + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd4 + 4.0 * xd * yd4 + 4.0 * xd13 * yd4 + xd14 * yd4
                    + 8.0 * xd3 * yd2 * (2.0 + yd2) + 8.0 * xd11 * yd2 * (2.0 + yd2) + xd4 * yd2 * (-44.0 + 161.0 * yd2) + xd10 * yd2 * (-44.0 + 161.0 * yd2)
                    - 4.0 * xd5 * yd2 * (48.0 + 193.0 * yd2) - 4.0 * xd9 * yd2 * (48.0 + 193.0 * yd2) + xd6 * (256.0 + 360.0 * yd2 - 133.0 * yd4)
                    + xd8 * (256.0 + 360.0 * yd2 - 133.0 * yd4) + xd2 * (4.0 * yd2 - 29.0 * yd4) + xd12 * (4.0 * yd2 - 29.0 * yd4)
                    + 16.0 * xd7 * (64.0 - 410.0 * yd2 + 223.0 * yd4));
            const complex<double> num21 = 64.0 * xd2 * power_of<4>(1.0 + xd2) * power_of<2>(-1.0 + yd) * yd2 * power_of<2>(1.0 + yd);

            const complex<double> denom1 = 9.0 * power_of<2>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));
            const complex<double> denom2 = 9.0 * (xd2 * (8.0 - 6.0 * yd) + yd + xd4 * yd) * (yd + xd4 * yd - 2.0 * xd2 * (4.0 + 3.0 * yd)) * power_of<3>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));
            const complex<double> denom3 = 9.0 * power_of<3>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));
            const complex<double> denom4 = 9.0 * xd2 * (xd2 * (8.0 - 6.0 * yd) + yd + xd4 * yd) * (yd + xd4 * yd - 2.0 * xd2 * (4.0 + 3.0 * yd)) * power_of<3>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));
            const complex<double> denom5 = 18.0 * xd2 * (xd2 * (8.0 - 6.0 * yd) + yd + xd4 * yd) * (yd + xd4 * yd - 2.0 * xd2 * (4.0 + 3.0 * yd)) * power_of<2>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));
            const complex<double> denom6 = 9.0 * yd * (xd2 * (8.0 - 6.0 * yd) + yd + xd4 * yd) * (yd + xd4 * yd - 2.0 * xd2 * (4.0 + 3.0 * yd)) * power_of<3>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));
            const complex<double> denom7 = 9.0 * yd2 * (xd2 * (8.0 - 6.0 * yd) + yd + xd4 * yd) * power_of<3>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));
            const complex<double> denom8 = 9.0 * yd2 * (yd + xd4 * yd - 2.0 * xd2 * (4.0 + 3.0 * yd)) * power_of<3>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));

            // GPL independent term

            const complex<double> term1 = (2.0 * (yd2 * (-1.0 + yd2 * (17.0 + 24.0 * ln2)) + xd8 * yd2 * (-1.0 + yd2 * (17.0 + 24.0 * ln2))
                    + 2.0 * xd4 * ((-yd2) * (99.0 + 32.0 * ln2) + yd4 * (39.0 + 72.0 * ln2) + 4.0 * (27.0 + ln256))
                    + 4.0 * xd2 * (-1.0 + yd2 * (31.0 - 8.0 * ln2) + ln256 + 2.0 * yd4 * (-7.0 + log(4096.0)))
                    + 4.0 * xd6 * (-1.0 + yd2 * (31.0 - 8.0 * ln2) + ln256 + 2.0 * yd4 * (-7.0 + log(4096.0))))) / denom1 + (32.0 * lnmuhat) / 9.0;

            // Polynomials in xd and yd that contain Log(2^n) terms

            const complex<double> logs1 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 3.0 * yd4 + xd8 * (yd2 + 3.0 * yd4) + 4.0 * xd2 * (3.0 + 2.0 * yd4 * (-5.0 + 16.0 * ln2) - yd2 * (5.0 + 32.0 * ln2)) + 4.0 * xd6 * (3.0 + 2.0 * yd4 * (-5.0 + 16.0 * ln2) - yd2 * (5.0 + 32.0 * ln2))
                    + 2.0 * xd4 * (-84.0 + yd2 * (171.0 - 128.0 * ln2) + yd4 * (-139.0 + 128.0 * ln2)));
            const complex<double> logs2 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 3.0 * yd4 + xd8 * (yd2 + 3.0 * yd4) + 4.0 * xd2 * (-1.0 + yd4 * (-6.0 + 32.0 * ln2) - yd2 * (5.0 + 32.0 * ln2)) + 4.0 * xd6 * (-1.0 + yd4 * (-6.0 + 32.0 * ln2) - yd2 * (5.0 + 32.0 * ln2))
                    + 2.0 * xd4 * (-36.0 + yd2 * (43.0 - 128.0 * ln2) + yd4 * (-59.0 + 128.0 * ln2)));
            const complex<double> logs3 = 4.0 * power_of<2>(1.0 + xd2) * power_of<2>(-1.0 + yd) * (1.0 + yd) * (yd2 * (1.0 + yd) + xd8 * yd2 * (1.0 + yd) - 4.0 * xd2 * yd2 * (5.0 + 32.0 * ln2 + yd * (-11.0 + 32.0 * ln2)) - 4.0 * xd6 * yd2 * (5.0 + 32.0 * ln2 + yd * (-11.0 + 32.0 * ln2))
                    - 2.0 * xd4 * (16.0 + 16.0 * yd + yd3 * (-59.0 + 128.0 * ln2) + yd2 * (5.0 + 128.0 * ln2)));
            const complex<double> logs4 = 4.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * power_of<2>(1.0 + yd) * ((-1.0 + yd) * yd2 + xd8 * (-1.0 + yd) * yd2 + 4.0 * xd2 * yd2 * (5.0 + yd * (11.0 - 32.0 * ln2) + 32.0 * ln2) + 4.0 * xd6 * yd2 * (5.0 + yd * (11.0 - 32.0 * ln2) + 32.0 * ln2)
                    + xd4 * (32.0 - 32.0 * yd + yd3 * (118.0 - 256.0 * ln2) + 2.0 * yd2 * (5.0 + 128.0 * ln2)));
            const complex<double> logs5 = 2.0 * power_of<4>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 * (-1.0 + yd2) + xd4 * yd2 * (-1.0 + yd2) + xd2 * (4.0 - 128.0 * yd3 + yd4 * (38.0 - 384.0 * ln2) + yd2 * (22.0 + 384.0 * ln2)));
            const complex<double> logs6 = 2.0 * power_of<4>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 * (-1.0 + yd2) + xd4 * yd2 * (-1.0 + yd2) + xd2 * (4.0 + 128.0 * yd3 + yd4 * (38.0 - 384.0 * ln2) + yd2 * (22.0 + 384.0 * ln2)));
            const complex<double> logs7 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 * (-1.0 + 17.0 * yd2) + xd8 * yd2 * (-1.0 + 17.0 * yd2) + 4.0 * xd2 * (-1.0 - 80.0 * yd3 + yd2 * (5.0 - 128.0 * ln2) + 4.0 * yd4 * (7.0 + 32.0 * ln2))
                    + 4.0 * xd6 * (-1.0 - 80.0 * yd3 + yd2 * (5.0 - 128.0 * ln2) + 4.0 * yd4 * (7.0 + 32.0 * ln2)) + 2.0 * xd4 * (-100.0 - 320.0 * yd3 + yd2 * (213.0 - 512.0 * ln2) + yd4 * (-1.0 + 512.0 * ln2)));
            const complex<double> logs8 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 * (-1.0 + 17.0 * yd2) + xd8 * yd2 * (-1.0 + 17.0 * yd2) + 4.0 * xd2 * (-1.0 + 80.0 * yd3 + yd2 * (5.0 - 128.0 * ln2) + 4.0 * yd4 * (7.0 + 32.0 * ln2))
                    + 4.0 * xd6 * (-1.0 + 80.0 * yd3 + yd2 * (5.0 - 128.0 * ln2) + 4.0 * yd4 * (7.0 + 32.0 * ln2)) + 2.0 * xd4 * (-100.0 + 320.0 * yd3 + yd2 * (213.0 - 512.0 * ln2) + yd4 * (-1.0 + 512.0 * ln2)));
            const complex<double> logs9 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 * (3.0 + 5.0 * yd2) + xd8 * yd2 * (3.0 + 5.0 * yd2) - 4.0 * xd2 * (1.0 - 176.0 * yd3 + yd2 * (15.0 - 128.0 * ln2) + 8.0 * yd4 * (3.0 + 16.0 * ln2))
                    - 4.0 * xd6 * (1.0 - 176.0 * yd3 + yd2 * (15.0 - 128.0 * ln2) + 8.0 * yd4 * (3.0 + 16.0 * ln2)) - 2.0 * xd4 * (100.0 - 704.0 * yd3 - yd2 * (129.0 + 512.0 * ln2) + yd4 * (197.0 + 512.0 * ln2)));
            const complex<double> logs10 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 * (3.0 + 5.0 * yd2) + xd8 * yd2 * (3.0 + 5.0 * yd2) - 4.0 * xd2 * (1.0 + 176.0 * yd3 + yd2 * (15.0 - 128.0 * ln2) + 8.0 * yd4 * (3.0 + 16.0 * ln2))
                    - 4.0 * xd6 * (1.0 + 176.0 * yd3 + yd2 * (15.0 - 128.0 * ln2) + 8.0 * yd4 * (3.0 + 16.0 * ln2)) - 2.0 * xd4 * (100.0 + 704.0 * yd3 - yd2 * (129.0 + 512.0 * ln2) + yd4 * (197.0 + 512.0 * ln2)));
            const complex<double> logs11 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd4 * (-yd - ln2 + yd2 * ln2)
                    + xd16 * yd4 * (-yd - ln2 + yd2 * ln2) + 4.0 * xd2 * yd2 * (-yd - ln2 + 8.0 * yd2 * ln2 + 9.0 * yd4 * ln2
                        - yd3 * (1.0 + 16.0 * ln2)) + 4.0 * xd14 * yd2 * (-yd - ln2 + 8.0 * yd2 * ln2 + 9.0 * yd4 * ln2 - yd3 * (1.0 + 16.0 * ln2))
                    - 4.0 * xd4 * yd * (4.0 - 26.0 * yd2 - 42.0 * yd * ln2 + 107.0 * yd3 * ln2 + 95.0 * yd5 * ln2 - 5.0 * yd4 * (7.0 + 32.0 * ln2))
                    - 4.0 * xd12 * yd * (4.0 - 26.0 * yd2 - 42.0 * yd * ln2 + 107.0 * yd3 * ln2 + 95.0 * yd5 * ln2 - 5.0 * yd4 * (7.0 + 32.0 * ln2))
                    + 2.0 * xd8 * (1232.0 * yd + yd5 * (181.0 - 1664.0 * ln2) - 1792.0 * ln2 + 3864.0 * yd2 * ln2 - 6931.0 * yd4 * ln2
                        + 2427.0 * yd6 * ln2 + 8.0 * yd3 * (-85.0 + 512.0 * ln2)) - 4.0 * xd6 * (-48.0 * yd - 64.0 * ln2 + 527.0 * yd2 * ln2
                        + 312.0 * yd4 * ln2 + 9.0 * yd6 * ln2 + 5.0 * yd5 * (35.0 + 48.0 * ln2) - yd3 * (369.0 + 1024.0 * ln2))
                    - 4.0 * xd10 * (-48.0 * yd - 64.0 * ln2 + 527.0 * yd2 * ln2 + 312.0 * yd4 * ln2 + 9.0 * yd6 * ln2 + 5.0 * yd5 * (35.0 + 48.0 * ln2)
                        - yd3 * (369.0 + 1024.0 * ln2)));
            const complex<double> logs12 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd4 * (yd - ln2 + yd2 * ln2)
                    + xd16 * yd4 * (yd - ln2 + yd2 * ln2) + 4.0 * xd2 * yd2 * (yd - ln2 + 8.0 * yd2 * ln2 + 9.0 * yd4 * ln2
                        + yd3 * (1.0 + 16.0 * ln2)) + 4.0 * xd14 * yd2 * (yd - ln2 + 8.0 * yd2 * ln2 + 9.0 * yd4 * ln2 + yd3 * (1.0 + 16.0 * ln2))
                    - 4.0 * xd4 * yd * (-4.0 + 26.0 * yd2 - 42.0 * yd * ln2 + 107.0 * yd3 * ln2 + 95.0 * yd5 * ln2 + 5.0 * yd4 * (7.0 + 32.0 * ln2))
                    - 4.0 * xd12 * yd * (-4.0 + 26.0 * yd2 - 42.0 * yd * ln2 + 107.0 * yd3 * ln2 + 95.0 * yd5 * ln2 + 5.0 * yd4 * (7.0 + 32.0 * ln2))
                    - 4.0 * xd6 * (48.0 * yd - 64.0 * ln2 + 527.0 * yd2 * ln2 + 312.0 * yd4 * ln2 + 9.0 * yd6 * ln2 - 5.0 * yd5 * (35.0 + 48.0 * ln2)
                        + yd3 * (369.0 + 1024.0 * ln2)) - 4.0 * xd10 * (48.0 * yd - 64.0 * ln2 + 527.0 * yd2 * ln2 + 312.0 * yd4 * ln2 + 9.0 * yd6 * ln2
                        - 5.0 * yd5 * (35.0 + 48.0 * ln2) + yd3 * (369.0 + 1024.0 * ln2)) + 2.0 * xd8 * (-1232.0 * yd - 1792.0 * ln2 + 3864.0 * yd2 * ln2
                        - 6931.0 * yd4 * ln2 + 2427.0 * yd6 * ln2 - 8.0 * yd3 * (-85.0 + 512.0 * ln2) + yd5 * (-181.0 + 1664.0 * ln2)));
            const complex<double> logs13 = 4.0 * power_of<2>(1.0 + xd2) * yd * (yd4 * (-1.0 + 5.0 * yd2) + xd16 * yd4 * (-1.0 + 5.0 * yd2)
                    + 8.0 * xd2 * yd4 * (-1.0 - 48.0 * ln2 + yd2 * (-3.0 + 48.0 * ln2)) + 8.0 * xd14 * yd4 * (-1.0 - 48.0 * ln2 + yd2 * (-3.0 + 48.0 * ln2))
                    - 4.0 * xd4 * (4.0 - 12.0 * yd2 + yd4 * (19.0 - 960.0 * ln2) + yd6 * (57.0 + 960.0 * ln2))
                    - 4.0 * xd12 * (4.0 - 12.0 * yd2 + yd4 * (19.0 - 960.0 * ln2) + yd6 * (57.0 + 960.0 * ln2))
                    + 8.0 * xd6 * (-8.0 + 8.0 * yd2 * (35.0 + 384.0 * ln2) + yd6 * (179.0 + 720.0 * ln2) - 3.0 * yd4 * (181.0 + 1264.0 * ln2))
                    + 8.0 * xd10 * (-8.0 + 8.0 * yd2 * (35.0 + 384.0 * ln2) + yd6 * (179.0 + 720.0 * ln2) - 3.0 * yd4 * (181.0 + 1264.0 * ln2))
                    + 2.0 * xd8 * (2000.0 + yd4 * (1869.0 - 34560.0 * ln2) + 16.0 * yd2 * (-247.0 + 1536.0 * ln2) + yd6 * (-417.0 + 9984.0 * ln2)));

            const complex<double> logs14 = 256.0 * xd2 * power_of<4>(1.0 + xd2) * (-1.0 + yd) * yd2 * (1.0 + yd) * (-3.0 * yd - ln4 + yd2 * ln4);
            const complex<double> logs15 = 256.0 * xd2 * power_of<4>(1.0 + xd2) * (-1.0 + yd) * yd2 * (1.0 + yd) * (3.0 * yd - ln4 + yd2 * ln4);
            const complex<double> logs16 = 8.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 3.0 * yd4 + xd8 * (yd2 + 3.0 * yd4)
                    + 2.0 * xd4 * (-60.0 + yd2 * (107.0 - 128.0 * ln2) + yd4 * (-99.0 + 128.0 * ln2))
                    + 4.0 * xd2 * (1.0 - yd2 * (5.0 + 32.0 * ln2) + 8.0 * yd4 * (-1.0 + log(16.0)))
                    + 4.0 * xd6 * (1.0 - yd2 * (5.0 + 32.0 * ln2) + 8.0 * yd4 * (-1.0 + log(16.0))));
            const complex<double> logs17 = 2.0 * power_of<2>(1.0 + xd2) * (yd6 * (-1.0 + yd2) * ln2 + xd20 * yd6 * (-1.0 + yd2) * ln2
                    + 128.0 * 1.0i * xd5 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2 - 1664.0 * 1.0i * xd7 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2
                    + 1664.0 * 1.0i * xd13 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2 - 128.0 * 1.0i * xd15 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2
                    + 256.0 * 1.0i * xd9 * power_of<2>(-1.0 + yd2) * (32.0 - 121.0 * yd2 + 75.0 * yd4) * ln2 - 256.0 * 1.0i * xd11 * power_of<2>(-1.0 + yd2) * (32.0 - 121.0 * yd2 + 75.0 * yd4) *
                        ln2 + 8.0 * xd6 * yd2 * (-8.0 + yd4 * (128.0 - 147.0 * ln2) + 20.0 * ln2 - 4.0 * yd2 * (-9.0 + 58.0 * ln2)
                        + yd6 * (-54.0 + 359.0 * ln2)) + 8.0 * xd14 * yd2 * (-8.0 + yd4 * (128.0 - 147.0 * ln2) + 20.0 * ln2 - 4.0 * yd2 * (-9.0 + 58.0 * ln2)
                        + yd6 * (-54.0 + 359.0 * ln2)) + xd4 * (-16.0 * yd2 * ln2 + 16.0 * yd4 * (-1.0 + 19.0 * ln2) + 9.0 * yd8 * (16.0 + 21.0 * ln2)
                        - yd6 * (32.0 + 477.0 * ln2)) + xd16 * (-16.0 * yd2 * ln2 + 16.0 * yd4 * (-1.0 + 19.0 * ln2) + 9.0 * yd8 * (16.0 + 21.0 * ln2)
                        - yd6 * (32.0 + 477.0 * ln2)) - 2.0 * xd2 * yd4 * (-27.0 * yd2 * ln2 + yd4 * (6.0 + 23.0 * ln2) + log(16.0))
                    - 2.0 * xd18 * yd4 * (-27.0 * yd2 * ln2 + yd4 * (6.0 + 23.0 * ln2) + log(16.0))
                    + 4.0 * xd10 * (-16.0 * yd2 * (134.0 + 2029.0 * ln2) + yd8 * (222.0 + 8827.0 * ln2) + 4.0 * yd4 * (764.0 + 14089.0 * ln2)
                        - yd6 * (1408.0 + 36303.0 * ln2) + 512.0 * (2.0 + log(128.0))) - 2.0 * xd8 * (512.0 * ln2 + 8.0 * yd4 * (-17.0 + 2307.0 * ln2)
                        + 3.0 * yd8 * (24.0 + 3445.0 * ln2) - 3.0 * yd6 * (48.0 + 9637.0 * ln2) - 56.0 * yd2 * (16.0 + log(128.0)))
                    - 2.0 * xd12 * (512.0 * ln2 + 8.0 * yd4 * (-17.0 + 2307.0 * ln2) + 3.0 * yd8 * (24.0 + 3445.0 * ln2) - 3.0 * yd6 * (48.0 + 9637.0 * ln2)
                        - 56.0 * yd2 * (16.0 + log(128.0))));
            const complex<double> logs18 = 2.0 * power_of<2>(1.0 + xd2) * (yd6 * (-1.0 + yd2) * ln2 + xd20 * yd6 * (-1.0 + yd2) * ln2
                    - 128.0 * 1.0i * xd5 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2 + 1664.0 * 1.0i * xd7 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2
                    - 1664.0 * 1.0i * xd13 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2 + 128.0 * 1.0i * xd15 * yd2 * power_of<2>(-1.0 + yd2) * (-1.0 + 3.0 * yd2) * ln2
                    - 256.0 * 1.0i * xd9 * power_of<2>(-1.0 + yd2) * (32.0 - 121.0 * yd2 + 75.0 * yd4) * ln2 + 256.0 * 1.0i * xd11 * power_of<2>(-1.0 + yd2) * (32.0 - 121.0 * yd2 + 75.0 * yd4) *
                     ln2 + 8.0 * xd6 * yd2 * (-8.0 + yd4 * (128.0 - 147.0 * ln2) + 20.0 * ln2 - 4.0 * yd2 * (-9.0 + 58.0 * ln2)
                      + yd6 * (-54.0 + 359.0 * ln2)) + 8.0 * xd14 * yd2 * (-8.0 + yd4 * (128.0 - 147.0 * ln2) + 20.0 * ln2 - 4.0 * yd2 * (-9.0 + 58.0 * ln2)
                      + yd6 * (-54.0 + 359.0 * ln2)) + xd4 * (-16.0 * yd2 * ln2 + 16.0 * yd4 * (-1.0 + 19.0 * ln2) + 9.0 * yd8 * (16.0 + 21.0 * ln2)
                      - yd6 * (32.0 + 477.0 * ln2)) + xd16 * (-16.0 * yd2 * ln2 + 16.0 * yd4 * (-1.0 + 19.0 * ln2) + 9.0 * yd8 * (16.0 + 21.0 * ln2)
                      - yd6 * (32.0 + 477.0 * ln2)) - 2.0 * xd2 * yd4 * (-27.0 * yd2 * ln2 + yd4 * (6.0 + 23.0 * ln2) + log(16.0))
                    - 2.0 * xd18 * yd4 * (-27.0 * yd2 * ln2 + yd4 * (6.0 + 23.0 * ln2) + log(16.0))
                    + 4.0 * xd10 * (-16.0 * yd2 * (134.0 + 2029.0 * ln2) + yd8 * (222.0 + 8827.0 * ln2) + 4.0 * yd4 * (764.0 + 14089.0 * ln2)
                      - yd6 * (1408.0 + 36303.0 * ln2) + 512.0 * (2.0 + log(128.0))) - 2.0 * xd8 * (512.0 * ln2 + 8.0 * yd4 * (-17.0 + 2307.0 * ln2)
                      + 3.0 * yd8 * (24.0 + 3445.0 * ln2) - 3.0 * yd6 * (48.0 + 9637.0 * ln2) - 56.0 * yd2 * (16.0 + log(128.0)))
                    - 2.0 * xd12 * (512.0 * ln2 + 8.0 * yd4 * (-17.0 + 2307.0 * ln2) + 3.0 * yd8 * (24.0 + 3445.0 * ln2) - 3.0 * yd6 * (48.0 + 9637.0 * ln2)
                      - 56.0 * yd2 * (16.0 + log(128.0))));
            const complex<double> logs19 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd6 + xd20 * yd6 + 128.0 * 1.0i * xd5 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    - 1664.0 * 1.0i * xd7 * (yd2 - 4.0 * yd4 + 3.0 * yd6) + 1664.0 * 1.0i * xd13 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    - 128.0 * 1.0i * xd15 * (yd2 - 4.0 * yd4 + 3.0 * yd6) + 256.0 * 1.0i * xd9 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6)
                    - 256.0 * 1.0i * xd11 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6) + xd4 * yd2 * (16.0 + yd4 * (189.0 - 1216.0 * ln2) + 192.0 * ln2
                        - 32.0 * yd2 * (9.0 + 16.0 * ln2)) + xd16 * yd2 * (16.0 + yd4 * (189.0 - 1216.0 * ln2) + 192.0 * ln2 - 32.0 * yd2 * (9.0 + 16.0 * ln2))
                    + 8.0 * xd6 * yd2 * (-4.0 * (5.0 + 188.0 * ln2) + 4.0 * yd2 * (53.0 + 214.0 * ln2) + yd4 * (359.0 + 632.0 * ln2))
                    + 8.0 * xd14 * yd2 * (-4.0 * (5.0 + 188.0 * ln2) + 4.0 * yd2 * (53.0 + 214.0 * ln2) + yd4 * (359.0 + 632.0 * ln2))
                    + 2.0 * xd8 * (512.0 + yd4 * (18576.0 - 18688.0 * ln2) - 6144.0 * ln2 + 15.0 * yd6 * (-689.0 + 928.0 * ln2)
                        + 8.0 * yd2 * (15.0 + 3764.0 * ln2)) + 2.0 * xd12 * (512.0 + yd4 * (18576.0 - 18688.0 * ln2) - 6144.0 * ln2
                        + 15.0 * yd6 * (-689.0 + 928.0 * ln2) + 8.0 * yd2 * (15.0 + 3764.0 * ln2))
                    - 4.0 * xd10 * (3584.0 * (1.0 - 12.0 * ln2) + 80.0 * yd2 * (-361.0 + 1428.0 * ln2) - 4.0 * yd4 * (-6869.0 + 31270.0 * ln2)
                        + yd6 * (-8827.0 + 38392.0 * ln2)) + 2.0 * xd2 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256)
                    + 2.0 * xd18 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256));
            const complex<double> logs20 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd6 + xd20 * yd6 - 128.0 * 1.0i * xd5 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    + 1664.0 * 1.0i * xd7 * (yd2 - 4.0 * yd4 + 3.0 * yd6) - 1664.0 * 1.0i * xd13 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    + 128.0 * 1.0i * xd15 * (yd2 - 4.0 * yd4 + 3.0 * yd6) - 256.0 * 1.0i * xd9 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6)
                    + 256.0 * 1.0i * xd11 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6) + xd4 * yd2 * (16.0 + yd4 * (189.0 - 1216.0 * ln2) + 192.0 * ln2
                        - 32.0 * yd2 * (9.0 + 16.0 * ln2)) + xd16 * yd2 * (16.0 + yd4 * (189.0 - 1216.0 * ln2) + 192.0 * ln2 - 32.0 * yd2 * (9.0 + 16.0 * ln2))
                    + 8.0 * xd6 * yd2 * (-4.0 * (5.0 + 188.0 * ln2) + 4.0 * yd2 * (53.0 + 214.0 * ln2) + yd4 * (359.0 + 632.0 * ln2))
                    + 8.0 * xd14 * yd2 * (-4.0 * (5.0 + 188.0 * ln2) + 4.0 * yd2 * (53.0 + 214.0 * ln2) + yd4 * (359.0 + 632.0 * ln2))
                    + 2.0 * xd8 * (512.0 + yd4 * (18576.0 - 18688.0 * ln2) - 6144.0 * ln2 + 15.0 * yd6 * (-689.0 + 928.0 * ln2)
                        + 8.0 * yd2 * (15.0 + 3764.0 * ln2)) + 2.0 * xd12 * (512.0 + yd4 * (18576.0 - 18688.0 * ln2) - 6144.0 * ln2
                        + 15.0 * yd6 * (-689.0 + 928.0 * ln2) + 8.0 * yd2 * (15.0 + 3764.0 * ln2))
                    - 4.0 * xd10 * (3584.0 * (1.0 - 12.0 * ln2) + 80.0 * yd2 * (-361.0 + 1428.0 * ln2) - 4.0 * yd4 * (-6869.0 + 31270.0 * ln2)
                        + yd6 * (-8827.0 + 38392.0 * ln2)) + 2.0 * xd2 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256)
                    + 2.0 * xd18 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256));
            const complex<double> logs21 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd6 + xd20 * yd6 + 128.0 * 1.0i * xd5 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    - 1664.0 * 1.0i * xd7 * (yd2 - 4.0 * yd4 + 3.0 * yd6) + 1664.0 * 1.0i * xd13 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    - 128.0 * 1.0i * xd15 * (yd2 - 4.0 * yd4 + 3.0 * yd6) + 256.0 * 1.0i * xd9 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6)
                    - 256.0 * 1.0i * xd11 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6) - xd4 * yd2 * (-16.0 + 64.0 * ln2 + 32.0 * yd2 * (9.0 + 16.0 * ln2)
                        + 3.0 * yd4 * (-63.0 + 320.0 * ln2)) - xd16 * yd2 * (-16.0 + 64.0 * ln2 + 32.0 * yd2 * (9.0 + 16.0 * ln2)
                        + 3.0 * yd4 * (-63.0 + 320.0 * ln2)) + 8.0 * xd6 * yd2 * (-4.0 * (5.0 + 44.0 * ln2) + 4.0 * yd2 * (53.0 + 86.0 * ln2)
                        + yd4 * (359.0 + 568.0 * ln2)) + 8.0 * xd14 * yd2 * (-4.0 * (5.0 + 44.0 * ln2) + 4.0 * yd2 * (53.0 + 86.0 * ln2)
                        + yd4 * (359.0 + 568.0 * ln2)) - 4.0 * xd10 * (yd4 * (27476.0 - 45208.0 * ln2) - 512.0 * (-7.0 + 36.0 * ln2)
                        + 16.0 * yd2 * (-1805.0 + 2036.0 * ln2) + yd6 * (-8827.0 + 15608.0 * ln2))
                    + 2.0 * xd8 * (-144.0 * yd4 * (-129.0 + 16.0 * ln2) + 3.0 * yd6 * (-3445.0 + 1184.0 * ln2) + 8.0 * yd2 * (15.0 + 1988.0 * ln2)
                        + 512.0 * (1.0 + log(16.0))) + 2.0 * xd12 * (-144.0 * yd4 * (-129.0 + 16.0 * ln2) + 3.0 * yd6 * (-3445.0 + 1184.0 * ln2)
                        + 8.0 * yd2 * (15.0 + 1988.0 * ln2) + 512.0 * (1.0 + log(16.0))) + 2.0 * xd2 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256)
                    + 2.0 * xd18 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256));
            const complex<double> logs22 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd6 + xd20 * yd6 - 128.0 * 1.0i * xd5 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    + 1664.0 * 1.0i * xd7 * (yd2 - 4.0 * yd4 + 3.0 * yd6) - 1664.0 * 1.0i * xd13 * (yd2 - 4.0 * yd4 + 3.0 * yd6)
                    + 128.0 * 1.0i * xd15 * (yd2 - 4.0 * yd4 + 3.0 * yd6) - 256.0 * 1.0i * xd9 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6)
                    + 256.0 * 1.0i * xd11 * (-32.0 + 153.0 * yd2 - 196.0 * yd4 + 75.0 * yd6) - xd4 * yd2 * (-16.0 + 64.0 * ln2 + 32.0 * yd2 * (9.0 + 16.0 * ln2)
                        + 3.0 * yd4 * (-63.0 + 320.0 * ln2)) - xd16 * yd2 * (-16.0 + 64.0 * ln2 + 32.0 * yd2 * (9.0 + 16.0 * ln2)
                        + 3.0 * yd4 * (-63.0 + 320.0 * ln2)) + 8.0 * xd6 * yd2 * (-4.0 * (5.0 + 44.0 * ln2) + 4.0 * yd2 * (53.0 + 86.0 * ln2)
                        + yd4 * (359.0 + 568.0 * ln2)) + 8.0 * xd14 * yd2 * (-4.0 * (5.0 + 44.0 * ln2) + 4.0 * yd2 * (53.0 + 86.0 * ln2)
                        + yd4 * (359.0 + 568.0 * ln2)) - 4.0 * xd10 * (yd4 * (27476.0 - 45208.0 * ln2) - 512.0 * (-7.0 + 36.0 * ln2)
                        + 16.0 * yd2 * (-1805.0 + 2036.0 * ln2) + yd6 * (-8827.0 + 15608.0 * ln2))
                    + 2.0 * xd8 * (-144.0 * yd4 * (-129.0 + 16.0 * ln2) + 3.0 * yd6 * (-3445.0 + 1184.0 * ln2) + 8.0 * yd2 * (15.0 + 1988.0 * ln2)
                        + 512.0 * (1.0 + log(16.0))) + 2.0 * xd12 * (-144.0 * yd4 * (-129.0 + 16.0 * ln2) + 3.0 * yd6 * (-3445.0 + 1184.0 * ln2)
                        + 8.0 * yd2 * (15.0 + 1988.0 * ln2) + 512.0 * (1.0 + log(16.0))) + 2.0 * xd2 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256)
                    + 2.0 * xd18 * yd4 * (4.0 + yd2 * (-23.0 + 24.0 * ln2) + ln256));
            const complex<double> logs23 = 2.0 * power_of<2>(1.0 + xd2) * (-4.0 * xd4 * yd2 * (-4.0 + yd4 * (195.0 - 72.0 * ln2) + 32.0 * ln2 - 12.0 * yd2 * (1.0 + 20.0 * ln2)
                        + 7.0 * yd6 * (-11.0 + 40.0 * ln2)) - 4.0 * xd12 * yd2 * (-4.0 + yd4 * (195.0 - 72.0 * ln2) + 32.0 * ln2 - 12.0 * yd2 * (1.0 + 20.0 * ln2)
                        + 7.0 * yd6 * (-11.0 + 40.0 * ln2)) - 2.0 * xd8 * (512.0 - 16.0 * yd2 * (51.0 + 616.0 * ln2) + 16.0 * yd4 * (103.0 + 956.0 * ln2)
                        + yd8 * (59.0 + 1448.0 * ln2) - yd6 * (653.0 + 6888.0 * ln2)) + 8.0 * xd2 * (5.0 * yd6 - 4.0 * yd4 * ln2 + yd8 * (-11.0 + log(16.0)))
                    + 8.0 * xd14 * (5.0 * yd6 - 4.0 * yd4 * ln2 + yd8 * (-11.0 + log(16.0)))
                    + 8.0 * xd6 * yd2 * (yd4 * (107.0 - 2176.0 * ln2) + 7.0 * yd6 * (-3.0 + 100.0 * ln2) + 4.0 * yd2 * (-50.0 + 321.0 * ln2)
                        + 24.0 * (-1.0 + ln256)) + 8.0 * xd10 * yd2 * (yd4 * (107.0 - 2176.0 * ln2) + 7.0 * yd6 * (-3.0 + 100.0 * ln2)
                        + 4.0 * yd2 * (-50.0 + 321.0 * ln2) + 24.0 * (-1.0 + ln256)) + yd6 * (-1.0 - 8.0 * ln2 + yd2 * (7.0 + ln256))
                    + xd16 * yd6 * (-1.0 - 8.0 * ln2 + yd2 * (7.0 + ln256)));
            const complex<double> logs24 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (-4.0 * xd6 * (32.0 + 8.0 * yd + yd7 * (345.0 - 1928.0 * ln2) + yd6 * (673.0 - 1464.0 * ln2)
                        - 128.0 * yd3 * (-6.0 + ln2) - 8.0 * yd2 * (5.0 + 64.0 * ln2) + 5.0 * yd8 * (-49.0 + 248.0 * ln2) + yd4 * (-76.0 + 736.0 * ln2)
                        + yd5 * (-1233.0 + 2056.0 * ln2)) - xd4 * yd * (16.0 + 176.0 * yd + yd2 * (16.0 - 256.0 * ln2) - 48.0 * yd3 * (-11.0 + 32.0 * ln2)
                        + 29.0 * yd7 * (17.0 + 40.0 * ln2) + yd5 * (-293.0 + 376.0 * ln2) - yd6 * (973.0 + 3704.0 * ln2) + yd4 * (669.0 + 3960.0 * ln2))
                    - xd8 * yd * (16.0 + 176.0 * yd + yd2 * (16.0 - 256.0 * ln2) - 48.0 * yd3 * (-11.0 + 32.0 * ln2) + 29.0 * yd7 * (17.0 + 40.0 * ln2)
                        + yd5 * (-293.0 + 376.0 * ln2) - yd6 * (973.0 + 3704.0 * ln2) + yd4 * (669.0 + 3960.0 * ln2))
                    + 2.0 * xd2 * yd3 * (-8.0 + yd4 * (57.0 - 72.0 * ln2) + yd2 * (-33.0 + 72.0 * ln2) - yd3 * (79.0 + 120.0 * ln2)
                        + yd5 * (-5.0 + 152.0 * ln2) - 4.0 * yd * (5.0 + ln256)) + 2.0 * xd10 * yd3 * (-8.0 + yd4 * (57.0 - 72.0 * ln2)
                        + yd2 * (-33.0 + 72.0 * ln2) - yd3 * (79.0 + 120.0 * ln2) + yd5 * (-5.0 + 152.0 * ln2) - 4.0 * yd * (5.0 + ln256))
                    + yd5 * (-3.0 - 8.0 * ln2 + yd2 * (-13.0 + ln256) - yd * (5.0 + ln256) + yd3 * (13.0 + ln256))
                    + xd12 * yd5 * (-3.0 - 8.0 * ln2 + yd2 * (-13.0 + ln256) - yd * (5.0 + ln256) + yd3 * (13.0 + ln256)));
            const complex<double> logs25 = power_of<2>(1.0 + xd2) * (1.0 + yd) * (-4.0 * xd6 * (32.0 - 8.0 * yd + yd5 * (1233.0 - 2056.0 * ln2) + yd6 * (673.0 - 1464.0 * ln2)
                        + 128.0 * yd3 * (-6.0 + ln2) - 8.0 * yd2 * (5.0 + 64.0 * ln2) + 5.0 * yd8 * (-49.0 + 248.0 * ln2) + yd4 * (-76.0 + 736.0 * ln2)
                        + yd7 * (-345.0 + 1928.0 * ln2)) - xd4 * yd * (-16.0 + 176.0 * yd + 16.0 * yd2 * (-1.0 + 16.0 * ln2) - 48.0 * yd3 * (-11.0 + 32.0 * ln2)
                        + 29.0 * yd7 * (17.0 + 40.0 * ln2) + yd5 * (-293.0 + 376.0 * ln2) - 3.0 * yd4 * (223.0 + 1320.0 * ln2) + yd6 * (973.0 + 3704.0 * ln2))
                    - xd8 * yd * (-16.0 + 176.0 * yd + 16.0 * yd2 * (-1.0 + 16.0 * ln2) - 48.0 * yd3 * (-11.0 + 32.0 * ln2) + 29.0 * yd7 * (17.0 + 40.0 * ln2)
                        + yd5 * (-293.0 + 376.0 * ln2) - 3.0 * yd4 * (223.0 + 1320.0 * ln2) + yd6 * (973.0 + 3704.0 * ln2))
                    + 2.0 * xd2 * yd3 * (8.0 + yd2 * (33.0 - 72.0 * ln2) + yd4 * (-57.0 + 72.0 * ln2) - yd3 * (79.0 + 120.0 * ln2)
                        + yd5 * (-5.0 + 152.0 * ln2) - 4.0 * yd * (5.0 + ln256)) + 2.0 * xd10 * yd3 * (8.0 + yd2 * (33.0 - 72.0 * ln2)
                        + yd4 * (-57.0 + 72.0 * ln2) - yd3 * (79.0 + 120.0 * ln2) + yd5 * (-5.0 + 152.0 * ln2) - 4.0 * yd * (5.0 + ln256))
                    + yd5 * (3.0 + yd2 * (13.0 - 8.0 * ln2) + ln256 - yd * (5.0 + ln256) + yd3 * (13.0 + ln256))
                    + xd12 * yd5 * (3.0 + yd2 * (13.0 - 8.0 * ln2) + ln256 - yd * (5.0 + ln256) + yd3 * (13.0 + ln256)));

            const complex<double> f27dPart1 = term1 + (logs19 / denom4 + logs20 / denom4 + logs24 / denom7 + logs25 / denom8) * li2half
                + (-(num12 / denom5) - num13 / denom5) * dilog(-1.0i / (-1.0i + wx3)) + (-(num12 / denom5) - num13 / denom5) * dilog(1.0i / (1.0i + wx3)) + (-(num12 / denom5) - num13 / denom5) * dilog(-1.0i / (-1.0i + wx4)) + (-(num12 / denom5) - num13 / denom5) * dilog(1.0i / (1.0i + wx4))
                + (num12 * dilog((1.0i - xd) / (1.0i + wx3))) / denom5 + (num12 * dilog((1.0i - xd) / (1.0i + wx4))) / denom5 - (logs19 * dilog(1.0 / 2.0 + (1.0i / 2.0) * xd)) / denom4 + (num5 * dilog((-1.0i) * xd)) / denom4 + (num4 * dilog(1.0i * xd)) / denom4
                - (num12 * dilog(-(xd / wx3))) / denom5 - (num13 * dilog(xd / wx3)) / denom5 - (num12 * dilog(-(xd / wx4))) / denom5 - (num13 * dilog(xd / wx4)) / denom5 + (num13 * dilog((-1.0i + xd) / (-1.0i + wx3))) / denom5
                + (num13 * dilog((-1.0i + xd) / (-1.0i + wx4))) / denom5 - (logs20 * dilog((-0.5 * 1.0i) * (1.0i + xd))) / denom4 + (num12 * dilog((1.0i + xd) / (1.0i - wx3))) / denom5 + (num13 * dilog((1.0i + xd) / (1.0i + wx3))) / denom5
                + (num12 * dilog((1.0i + xd) / (1.0i - wx4))) / denom5 + (num13 * dilog((1.0i + xd) / (1.0i + wx4))) / denom5 - (logs24 * dilog((1.0 - yd) / 2.0)) / denom7 - (logs13 * dilog(-yd)) / denom2 + (logs13 * dilog(yd)) / denom2
                - (logs25 * dilog((1.0 + yd) / 2.0)) / denom8 - (logs21 * power_of<2>(log(1.0 - 1.0i * xd))) / (2.0 * denom4) - (logs22 * power_of<2>(log(1.0 + 1.0i * xd))) / (2.0 * denom4) - (2.0 * num12 * ln2 * log((wx3 + xd) / wx3)) / denom5
                - (2.0 * num12 * ln2 * log((wx4 + xd) / wx4)) / denom5 + log(1.0 - 1.0i * xd) * (logs17 / denom4 + (num13 * log((wx3 - xd) / (1.0i + wx3))) / denom5 + (num13 * log((wx4 - xd) / (1.0i + wx4))) / denom5
                  - (logs20 * log(1.0 / 2.0 + (1.0i / 2.0) * xd)) / denom4 + (num4 * log(xd)) / denom4 + (num12 * log((wx3 + xd) / (-1.0i + wx3))) / denom5
                  + (num12 * log((wx4 + xd) / (-1.0i + wx4))) / denom5) + log(1.0 + 1.0i * xd) * (logs18 / denom4 + (num13 * log((wx3 - xd) / (-1.0i + wx3))) / denom5 + (num13 * log((wx4 - xd) / (-1.0i + wx4))) / denom5
                  - (logs19 * log(1.0 / 2.0 - (1.0i / 2.0) * xd)) / denom4 + (num5 * log(xd)) / denom4 + (num12 * log((wx3 + xd) / (1.0i + wx3))) / denom5 + (num12 * log((wx4 + xd) / (1.0i + wx4))) / denom5) - (2.0 * num13 * ln2 * log(1.0 - xd / wx3)) / denom5
                - (2.0 * num13 * ln2 * log(1.0 - xd / wx4)) / denom5 + log(xd) * (num1 / denom1 - (num12 * log((wx3 + xd) / wx3)) / denom5 - (num12 * log((wx4 + xd) / wx4)) / denom5 - (num13 * log(1.0 - xd / wx3)) / denom5 - (num13 * log(1.0 - xd / wx4)) / denom5)
                + (logs23 / denom6 + (logs24 * ln2) / denom7 + (logs11 * log(1.0 - 1.0i * xd)) / denom2 + (logs11 * log(1.0 + 1.0i * xd)) / denom2 + (num2 * log(xd)) / denom2) * ln1myd - (logs24 * power_of<2>(ln1myd)) / (2.0 * denom7)
                + (-(logs23 / denom6) + (logs25 * ln2) / denom8 + (logs12 * log(1.0 - 1.0i * xd)) / denom2 + (logs12 * log(1.0 + 1.0i * xd)) / denom2 - (num2 * log(xd)) / denom2) * ln1pyd - (logs25 * power_of<2>(ln1pyd)) / (2.0 * denom8)
                + ((2.0 * 1.0i) * num13 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)) / denom5 + ((2.0 * 1.0i) * num12 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)) / denom5
                + ((2.0 * 1.0i) * num13 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4)) / denom5 + ((2.0 * 1.0i) * num12 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4)) / denom5
                + ((2.0 * 1.0i) * num12 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)) / denom5 + ((2.0 * 1.0i) * num13 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)) / denom5
                + ((2.0 * 1.0i) * num12 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)) / denom5 + ((2.0 * 1.0i) * num13 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) / denom5;

            const complex<double> f27dPart21 = (-(logs2 * pisqu * ln2) / 6.0 - (num11 * pisqu * ln2) / 12.0 - (num7 * pisqu * ln2) / 12.0 + (2.0 * logs1 - logs7 - logs8) * li3half + (logs1 - logs16 - logs2 - num9) * trilog(1.0 / 2.0 - (1.0i / 2.0) * xd) + (logs16 + num10 + num9) * trilog(1.0 - 1.0i * xd) + (logs1 - logs16 - logs2 - num9) * trilog((1.0 + 1.0i * xd) / 2.0)
                + (logs16 + num10 + num9) * trilog(1.0 + 1.0i * xd) + (-logs16 + 2.0 * num9) * trilog((-1.0i) * xd) + (-logs16 + 2.0 * num9) * trilog(1.0i * xd) - (num9 * trilog(-xd2)) / 2.0 - logs16 * trilog(xd / (-1.0i + xd)) + (logs16 + num9) * trilog((2.0 * xd) / (-1.0i + xd)) - logs16 * trilog(xd / (1.0i + xd)) + (logs16 + num9) * trilog((2.0 * xd) / (1.0i + xd))
                + (logs10 + logs5 + logs6 + logs7 + 2.0 * logs8 - num11) * trilog((1.0 - yd) / 2.0) + (2.0 * logs5 - logs6 + logs9) * trilog(1.0 - yd) + (logs10 + logs6 + 7.0 * num8) * trilog(-yd) + (logs5 + logs9 - 7.0 * num8) * trilog(yd) + (logs10 + logs5) * trilog(yd / (-1.0 + yd)) + (-logs10 - logs5 - logs6) * trilog((2.0 * yd) / (-1.0 + yd))
                + (logs6 + logs9) * trilog(yd / (1.0 + yd)) + (-logs5 - logs6 - logs9) * trilog((2.0 * yd) / (1.0 + yd)) + (logs5 + logs6 + 2.0 * logs7 + logs8 + logs9 - num7) * trilog((1.0 + yd) / 2.0) + (logs10 - logs5 + 2.0 * logs6) * trilog(1.0 + yd) + (logs1 * pisqu * ln2) / 3.0 + (logs10 * pisqu * ln2) / 12.0 - (logs16 * pisqu * ln2) / 6.0
                + (logs7 * pisqu * ln2) / 6.0 + (logs8 * pisqu * ln2) / 6.0 + (logs9 * pisqu * ln2) / 12.0 - (2.0 * logs1 * power_of<3>(ln2)) / 3.0 - (logs10 * power_of<3>(ln2)) / 6.0 + (logs16 * power_of<3>(ln2)) / 3.0 + (logs2 * power_of<3>(ln2)) / 3.0 - (logs5 * power_of<3>(ln2)) / 3.0 - (logs6 * power_of<3>(ln2)) / 3.0 - (logs7 * power_of<3>(ln2)) / 3.0 - (logs8 * power_of<3>(ln2)) / 3.0 - (logs9 * power_of<3>(ln2)) / 6.0
                + (num11 * power_of<3>(ln2)) / 6.0 + (num7 * power_of<3>(ln2)) / 6.0 + (num9 * power_of<3>(ln2)) / 3.0 + (logs5 * pisqu * ln4) / 12.0 + (logs6 * pisqu * ln4) / 12.0 - (num9 * pisqu * ln4) / 12.0 + (-logs16 - num10) * dilog(1.0 - 1.0i * xd) * log(1.0 - 1.0i * xd) + (logs2 / 6.0 - num9 / 6.0) * power_of<3>(log(1.0 - 1.0i * xd)) + (-logs16 - num10) * dilog(1.0 + 1.0i * xd) * log(1.0 + 1.0i * xd)
                + (logs2 / 6.0 - num9 / 6.0) * power_of<3>(log(1.0 + 1.0i * xd)) + dilog(1.0i * xd) * (-(num10 * log(1.0 - 1.0i * xd)) + logs16 * log(1.0 + 1.0i * xd)) + dilog((-1.0i) * xd) * (logs16 * log(1.0 - 1.0i * xd) - num10 * log(1.0 + 1.0i * xd)) + ((num9 * pisqu) / 12.0 - (num9 * ln2squ) / 2.0) * log((-1.0i) * xd)
                + dilog(1.0 / 2.0 - (1.0i / 2.0) * xd) * (num9 * log(-1.0i) + (-logs1 + logs2) * log(1.0 - 1.0i * xd) + (-logs1 + logs2) * log(1.0 + 1.0i * xd) - num9 * log((-1.0i) * xd)) + ((num9 * pisqu) / 12.0 - (num9 * ln2squ) / 2.0) * log(1.0i * xd)
                + dilog((1.0 + 1.0i * xd) / 2.0) * (num9 * log(1.0i) + (-logs1 + logs2) * log(1.0 - 1.0i * xd) + (-logs1 + logs2) * log(1.0 + 1.0i * xd) - num9 * log(1.0i * xd)) + log(1.0 + 1.0i * xd) * ((logs1 * pisqu) / 12.0 + (logs16 * pisqu) / 12.0 - (logs2 * pisqu) / 12.0 - (num9 * pisqu) / 12.0 + (logs2 * ln2squ) / 2.0 - num9 * log(1.0i) * ln2 - (logs1 * ln2squ) / 2.0
                - (logs16 * ln2squ) / 2.0 - (num9 * ln2squ) / 2.0 + num9 * ln2 * log(1.0i * xd)) + log(1.0 - 1.0i * xd) * ((logs1 * pisqu) / 12.0 + (logs16 * pisqu) / 12.0 - (logs2 * pisqu) / 12.0 - (num9 * pisqu) / 12.0 + (logs2 * ln2squ) / 2.0 - num9 * log(-1.0i) * ln2 - (logs1 * ln2squ) / 2.0 - (logs16 * ln2squ) / 2.0 - (num9 * ln2squ) / 2.0
                + logs2 * log((1.0 + 1.0i * xd) / 2.0) * log(1.0 + 1.0i * xd) + num9 * ln2 * log((-1.0i) * xd) + log(1.0 + 1.0i * xd) * (2.0 * logs1 * ln2 + logs2 * log(1.0 / 2.0 - (1.0i / 2.0) * xd) - num9 * log((-1.0i) * xd) - num9 * log(1.0i * xd)))
                + power_of<2>(log(1.0 + 1.0i * xd)) * ((logs16 * ln2) / 2.0 - (logs2 * ln64) / 12.0 - (logs1 * log(1.0 / 2.0 - (1.0i / 2.0) * xd)) / 2.0 + ((-1.0 / 2.0) * logs16 - num10 / 2.0) * log((-1.0i) * xd) + (num9 * log((-2.0 * 1.0i) * xd)) / 2.0 - (num10 * log(xd)) / 2.0)
                + power_of<2>(log(1.0 - 1.0i * xd)) * ((logs16 * ln2) / 2.0 - (logs2 * ln64) / 12.0 - (logs1 * log((1.0 + 1.0i * xd) / 2.0)) / 2.0 + ((-1.0 / 2.0) * logs16 - num10 / 2.0) * log(1.0i * xd) + (num9 * log((2.0 * 1.0i) * xd)) / 2.0 - (num10 * log(xd)) / 2.0)
                + li2half * (-(num9 * log(-1.0i)) - num9 * log(1.0i) + (-logs3 - logs4) * log(1.0 - 1.0i * xd) + (-logs3 - logs4) * log(1.0 + 1.0i * xd) + (num3 + num6) * log(xd)) + (-2.0 * logs5 - logs9) * dilog(1.0 - yd) * ln1myd - (num11 * power_of<3>(ln1myd)) / 6.0
                + power_of<2>(ln1myd) * ((-1.0 / 2.0) * (logs10 * ln2) - (logs5 * ln64) / 12.0 - (num11 * ln64) / 12.0 + (num6 * log(1.0 - 1.0i * xd)) / 2.0 + (num6 * log(1.0 + 1.0i * xd)) / 2.0 - (num6 * log(xd)) / 2.0 + (logs6 * log((1.0 - yd) / 8.0)) / 6.0 + (-logs5 - logs6 / 2.0 - logs9 / 2.0) * log(yd) + ((-1.0 / 2.0) * logs7 - logs8) * log((1.0 + yd) / 2.0))
                + (-logs10 - 2.0 * logs6) * dilog(1.0 + yd) * ln1pyd + ((logs5 * pisqu) / 12.0 - (logs6 * pisqu) / 12.0 + (logs7 * pisqu) / 12.0 - (logs9 * pisqu) / 12.0 - (num7 * pisqu) / 12.0 + (num7 * ln2squ) / 2.0 + (logs5 * ln2squ) / 2.0 + (logs6 * ln2squ) / 2.0 - (logs7 * ln2squ) / 2.0 + (logs9 * ln2squ) / 2.0 - logs4 * ln2 * log(1.0 - 1.0i * xd)
                - logs4 * ln2 * log(1.0 + 1.0i * xd) + num3 * ln2 * log(xd)) * ln1pyd + ((-1.0 / 2.0) * (logs9 * ln2) - (logs6 * ln64) / 12.0 - (num7 * ln64) / 12.0 + (num3 * log(1.0 - 1.0i * xd)) / 2.0 + (num3 * log(1.0 + 1.0i * xd)) / 2.0 - (num3 * log(xd)) / 2.0 + (-logs7 - logs8 / 2.0) * log((1.0 - yd) / 2.0)
                + ((-1.0 / 2.0) * logs10 - logs5 / 2.0 - logs6) * log(-yd) + (logs5 * log((1.0 + yd) / 8.0)) / 6.0) * power_of<2>(ln1pyd) - (num7 * power_of<3>(ln1pyd)) / 6.0 + dilog(-yd) * (logs15 * log(1.0 - 1.0i * xd) + logs15 * log(1.0 + 1.0i * xd) - 6.0 * num8 * log(xd) + (-logs10 - logs6) * ln1myd - logs6 * ln1pyd)
                + dilog(yd) * (logs14 * log(1.0 - 1.0i * xd) + logs14 * log(1.0 + 1.0i * xd) + 6.0 * num8 * log(xd) - logs5 * ln1myd + (-logs5 - logs9) * ln1pyd) + dilog((1.0 - yd) / 2.0) * (logs3 * log(1.0 - 1.0i * xd) + logs3 * log(1.0 + 1.0i * xd) - num6 * log(xd) + (-logs7 - 2.0 * logs8 + num11) * ln1myd + (-logs7 + num7) * ln1pyd)
                + dilog((1.0 + yd) / 2.0) * (logs4 * log(1.0 - 1.0i * xd) + logs4 * log(1.0 + 1.0i * xd) - num3 * log(xd) + (-logs8 + num11) * ln1myd + (-2.0 * logs7 - logs8 + num7) * ln1pyd)
                + ln1myd * ((-1.0 / 12.0) * (logs10 * pisqu) - (logs5 * pisqu) / 12.0 + (logs6 * pisqu) / 12.0 + (logs8 * pisqu) / 12.0 - (num11 * pisqu) / 12.0 + (num11 * ln2squ) / 2.0 + (logs10 * ln2squ) / 2.0 + (logs5 * ln2squ) / 2.0 + (logs6 * ln2squ) / 2.0 - (logs8 * ln2squ) / 2.0 - logs3 * ln2 * log(1.0 - 1.0i * xd) - logs3 * ln2 * log(1.0 + 1.0i * xd)
                + num6 * ln2 * log(xd) + (logs7 * ln2 + logs8 * ln2 + num11 * log((1.0 - yd) / 2.0)) * ln1pyd + num7 * log((1.0 + yd) / 2.0) * ln1pyd) + (-((7.0 * logs1) / 2.0) - (15.0 * logs10) / 8.0 - logs16 / 4.0 + (7.0 * logs2) / 4.0 - (11.0 * logs5) / 4.0 - (11.0 * logs6) / 4.0 - (7.0 * logs7) / 4.0 - (7.0 * logs8) / 4.0 - (15.0 * logs9) / 8.0 - 2.0 * num10 + (7.0 * num11) / 8.0
                + (7.0 * num7) / 8.0 - num9 / 4.0) * zeta3) / denom3;

            const complex<double> f27dPart3 = num21 * (-16.0 * power_of<2>(li2half) + (dilog((w5 - yd) / (-1.0 + w5)) + dilog(-(yd / w5)) + dilog(yd / w5) + dilog((w5 + yd) / (-1.0 + w5)) + log((1.0 + yd) / (1.0 - w5)) * log((w5 + yd) / w5) + log((-1.0 + yd) / (-1.0 + w5)) * log(1.0 - yd / w5)) * (2.0 * power_of<2>(log(1.0 - 1.0i * xd)) + 2.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(xd) * (4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4)) + log(1.0 - 1.0i * xd) * (4.0 * ln2 - 4.0 * log(xd) - 4.0 * log((wx3 + xd) / (-1.0i + wx3)) - 4.0 * log((wx4 + xd) / (-1.0i + wx4)))
                      + log(1.0 + 1.0i * xd) * (4.0 * ln2 - 4.0 * log(xd) - 4.0 * log((wx3 + xd) / (1.0i + wx3)) - 4.0 * log((wx4 + xd) / (1.0i + wx4))) - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4))
                    + dilog(w5 / (-1.0 + w5)) * (-8.0 * dilog(-1.0i / (-1.0i + wx3)) - 8.0 * dilog(1.0i / (1.0i + wx3)) - 8.0 * dilog(-1.0i / (-1.0i + wx4)) - 8.0 * dilog(1.0i / (1.0i + wx4)) + 8.0 * dilog((1.0i - xd) / (1.0i + wx3)) + 8.0 * dilog((1.0i - xd) / (1.0i + wx4)) + 8.0 * dilog(1.0 / 2.0 + (1.0i / 2.0) * xd) + 8.0 * dilog((-1.0i) * xd) + 8.0 * dilog(1.0i * xd) - 8.0 * dilog(-(xd / wx3)) - 8.0 * dilog(-(xd / wx4)) + 8.0 * dilog(((-0.5) * 1.0i) * (1.0i + xd)) + 8.0 * dilog((1.0i + xd) / (1.0i - wx3)) + 8.0 * dilog((1.0i + xd) / (1.0i - wx4)) - 4.0 * power_of<2>(log(1.0 - 1.0i * xd)) - 4.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(xd) * (-8.0 * log((wx3 + xd) / wx3) - 8.0 * log((wx4 + xd) / wx4))
                      + log(1.0 - 1.0i * xd) * (-8.0 * ln2 + 8.0 * log(xd) + 8.0 * log((wx3 + xd) / (-1.0i + wx3)) + 8.0 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (-8.0 * ln2 + 8.0 * log(xd) + 8.0 * log((wx3 + xd) / (1.0i + wx3)) + 8.0 * log((wx4 + xd) / (1.0i + wx4))) + (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) + (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) + (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4))
                    + (dilog((1.0 - yd) / 2.0) + dilog((1.0 + yd) / 2.0)) * (2.0 * power_of<2>(log(1.0 - 1.0i * xd)) + 2.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(1.0 - 1.0i * xd) * (4.0 * ln2 - 2.0 * log((wx3 - xd) / (1.0i + wx3)) - 2.0 * log((wx4 - xd) / (1.0i + wx4)) - 4.0 * log(xd) - 2.0 * log((wx3 + xd) / (-1.0i + wx3)) - 2.0 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (4.0 * ln2 - 2.0 * log((wx3 - xd) / (-1.0i + wx3)) - 2.0 * log((wx4 - xd) / (-1.0i + wx4)) - 4.0 * log(xd) - 2.0 * log((wx3 + xd) / (1.0i + wx3)) - 2.0 * log((wx4 + xd) / (1.0i + wx4)))
                      + log(xd) * (2.0 * log((wx3 + xd) / wx3) + 2.0 * log((wx4 + xd) / wx4) + 2.0 * log(1.0 - xd / wx3) + 2.0 * log(1.0 - xd / wx4)) - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                      - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + (dilog((w7 - yd) / (1.0 + w7)) + dilog(-yd) + dilog(yd) + dilog(-(yd / w7)) + dilog(yd / w7) + dilog((w7 + yd) / (1.0 + w7)) + log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7) + log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7))  * 
                     (-2.0 * power_of<2>(log(1.0 - 1.0i * xd)) - 2.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(1.0 - 1.0i * xd) * (-4.0 * ln2 + 2.0 * log((wx3 - xd) / (1.0i + wx3)) + 2.0 * log((wx4 - xd) / (1.0i + wx4)) + 4.0 * log(xd) + 2.0 * log((wx3 + xd) / (-1.0i + wx3)) + 2.0 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (-4.0 * ln2 + 2.0 * log((wx3 - xd) / (-1.0i + wx3)) + 2.0 * log((wx4 - xd) / (-1.0i + wx4)) + 4.0 * log(xd) + 2.0 * log((wx3 + xd) / (1.0i + wx3)) + 2.0 * log((wx4 + xd) / (1.0i + wx4))) + log(xd) * (-2.0 * log((wx3 + xd) / wx3) - 2.0 * log((wx4 + xd) / wx4) - 2.0 * log(1.0 - xd / wx3) - 2.0 * log(1.0 - xd / wx4))
                      + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)
                      + (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + (dilog((w4 - yd) / (1.0 + w4)) + dilog(-(yd / w4)) + dilog(yd / w4) + dilog((w4 + yd) / (1.0 + w4)) + log((1.0 - yd) / (1.0 + w4)) * log((w4 + yd) / w4) + log((1.0 + yd) / (1.0 + w4)) * log(1.0 - yd / w4)) * (2.0 * power_of<2>(log(1.0 - 1.0i * xd)) + 2.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(1.0 + 1.0i * xd) * (4.0 * ln2 - 4.0 * log((wx3 - xd) / (-1.0i + wx3)) - 4.0 * log((wx4 - xd) / (-1.0i + wx4)) - 4.0 * log(xd))
                      + log(1.0 - 1.0i * xd) * (4.0 * ln2 - 4.0 * log((wx3 - xd) / (1.0i + wx3)) - 4.0 * log((wx4 - xd) / (1.0i + wx4)) - 4.0 * log(xd)) + log(xd) * (4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4)) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4))
                    + dilog(w7 / (1.0 + w7)) * (8.0 * dilog(-1.0i / (-1.0i + wx3)) + 8.0 * dilog(1.0i / (1.0i + wx3)) + 8.0 * dilog(-1.0i / (-1.0i + wx4)) + 8.0 * dilog(1.0i / (1.0i + wx4)) - 4.0 * dilog((1.0i - xd) / (1.0i + wx3)) - 4.0 * dilog((1.0i - xd) / (1.0i + wx4)) - 8.0 * dilog(1.0 / 2.0 + (1.0i / 2.0) * xd) - 8.0 * dilog((-1.0i) * xd) - 8.0 * dilog(1.0i * xd) + 4.0 * dilog(-(xd / wx3)) + 4.0 * dilog(xd / wx3) + 4.0 * dilog(-(xd / wx4)) + 4.0 * dilog(xd / wx4) - 4.0 * dilog((-1.0i + xd) / (-1.0i + wx3)) - 4.0 * dilog((-1.0i + xd) / (-1.0i + wx4)) - 8.0 * dilog(((-0.5) * 1.0i) * (1.0i + xd)) - 4.0 * dilog((1.0i + xd) / (1.0i - wx3)) - 4.0 * dilog((1.0i + xd) / (1.0i + wx3))
                      - 4.0 * dilog((1.0i + xd) / (1.0i - wx4)) - 4.0 * dilog((1.0i + xd) / (1.0i + wx4)) + 4.0 * power_of<2>(log(1.0 - 1.0i * xd)) + 4.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(1.0 - 1.0i * xd) * (8.0 * ln2 - 4.0 * log((wx3 - xd) / (1.0i + wx3)) - 4.0 * log((wx4 - xd) / (1.0i + wx4)) - 8.0 * log(xd) - 4.0 * log((wx3 + xd) / (-1.0i + wx3)) - 4.0 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (8.0 * ln2 - 4.0 * log((wx3 - xd) / (-1.0i + wx3)) - 4.0 * log((wx4 - xd) / (-1.0i + wx4)) - 8.0 * log(xd) - 4.0 * log((wx3 + xd) / (1.0i + wx3)) - 4.0 * log((wx4 + xd) / (1.0i + wx4)))
                      + log(xd) * (4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4)) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                      - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) - (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) - (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + dilog(w4 / (1.0 + w4)) * (-8.0 * dilog(-1.0i / (-1.0i + wx3)) - 8.0 * dilog(1.0i / (1.0i + wx3)) - 8.0 * dilog(-1.0i / (-1.0i + wx4)) - 8.0 * dilog(1.0i / (1.0i + wx4)) + 8.0 * dilog(1.0 / 2.0 + (1.0i / 2.0) * xd) + 8.0 * dilog((-1.0i) * xd) + 8.0 * dilog(1.0i * xd) - 8.0 * dilog(xd / wx3) - 8.0 * dilog(xd / wx4) + 8.0 * dilog((-1.0i + xd) / (-1.0i + wx3)) + 8.0 * dilog((-1.0i + xd) / (-1.0i + wx4))
                      + 8.0 * dilog(((-0.5) * 1.0i) * (1.0i + xd)) + 8.0 * dilog((1.0i + xd) / (1.0i + wx3)) + 8.0 * dilog((1.0i + xd) / (1.0i + wx4)) - 4.0 * power_of<2>(log(1.0 - 1.0i * xd)) - 4.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(1.0 + 1.0i * xd) * (-8.0 * ln2 + 8.0 * log((wx3 - xd) / (-1.0i + wx3)) + 8.0 * log((wx4 - xd) / (-1.0i + wx4)) + 8.0 * log(xd)) + log(1.0 - 1.0i * xd) * (-8.0 * ln2 + 8.0 * log((wx3 - xd) / (1.0i + wx3)) + 8.0 * log((wx4 - xd) / (1.0i + wx4)) + 8.0 * log(xd)) + log(xd) * (-8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4)) + (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)
                      + (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) + (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4))
                    + (ln1myd + ln1pyd) * (-2.0 * ln2 * power_of<2>(log(1.0 - 1.0i * xd)) - 2.0 * ln2 * power_of<2>(log(1.0 + 1.0i * xd)) + log(1.0 - 1.0i * xd) * (-4.0 * ln2squ + 2.0 * ln2 * log((wx3 - xd) / (1.0i + wx3)) + 2.0 * ln2 * log((wx4 - xd) / (1.0i + wx4)) + 4.0 * ln2 * log(xd) + 2.0 * ln2 * log((wx3 + xd) / (-1.0i + wx3)) + 2.0 * ln2 * log((wx4 + xd) / (-1.0i + wx4)))
                      + log(1.0 + 1.0i * xd) * (-4.0 * ln2squ + 2.0 * ln2 * log((wx3 - xd) / (-1.0i + wx3)) + 2.0 * ln2 * log((wx4 - xd) / (-1.0i + wx4)) + 4.0 * ln2 * log(xd) + 2.0 * ln2 * log((wx3 + xd) / (1.0i + wx3)) + 2.0 * ln2 * log((wx4 + xd) / (1.0i + wx4))) + log(xd) * (-2.0 * ln2 * log((wx3 + xd) / wx3) - 2.0 * ln2 * log((wx4 + xd) / wx4) - 2.0 * ln2 * log(1.0 - xd / wx3) - 2.0 * ln2 * log(1.0 - xd / wx4)) + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)
                      + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)
                      + (4.0 * 1.0i) * M_PI * ln2 * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 - 1.0i * wx3) * my_sign(-imag(w4)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 - 1.0i * wx4) * my_sign(-imag(w4)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 + 1.0i * wx3) * my_sign(-imag(w4)) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv)
                    + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 + 1.0i * wx4) * my_sign(-imag(w4)) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 + 1.0i * wx3) * my_sign(imag(w5)) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 + 1.0i * wx4) * my_sign(imag(w5)) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5)
                    + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 - 1.0i * wx3) * my_sign(imag(w5)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 - 1.0i * wx4) * my_sign(imag(w5)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                    - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                    - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                    - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 - 1.0i * wx3) * my_sign(-imag(w4)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 - 1.0i * wx4) * my_sign(-imag(w4)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv)
                    + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 + 1.0i * wx3) * my_sign(-imag(w4)) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + 16.0 * pisqu * log(1.0 + w4inv) * log(1.0 + 1.0i * wx4) * my_sign(-imag(w4)) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 + 1.0i * wx3) * my_sign(imag(w5)) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5)
                    + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 + 1.0i * wx4) * my_sign(imag(w5)) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 - 1.0i * wx3) * my_sign(imag(w5)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + 16.0 * pisqu * log((-1.0 + w5) / w5) * log(1.0 - 1.0i * wx4) * my_sign(imag(w5)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5)
                    - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)
                    - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx3) * my_sign(-imag(w7)) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)
                    - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 - 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv) - 8.0 * pisqu * log(1.0 + w7inv) * log(1.0 + 1.0i * wx4) * my_sign(-imag(w7)) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv) + (dilog(xd / wx3) + dilog(xd / wx4)) * (2.0 * dilog((1.0 - yd) / 2.0) + 4.0 * dilog((w4 - yd) / (1.0 + w4)) - 2.0 * dilog((w7 - yd) / (1.0 + w7)) - 2.0 * dilog(-yd) - 2.0 * dilog(yd) + 4.0 * dilog(-(yd / w4)) + 4.0 * dilog(yd / w4) - 2.0 * dilog(-(yd / w7)) - 2.0 * dilog(yd / w7)
                      + 2.0 * dilog((1.0 + yd) / 2.0) + 4.0 * dilog((w4 + yd) / (1.0 + w4)) - 2.0 * dilog((w7 + yd) / (1.0 + w7)) - 2.0 * ln2 * ln1myd - 2.0 * ln2 * ln1pyd + 4.0 * log((1.0 - yd) / (1.0 + w4)) * log((w4 + yd) / w4) - 2.0 * log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7) + 4.0 * log((1.0 + yd) / (1.0 + w4)) * log(1.0 - yd / w4) - 2.0 * log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7) + (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                      + (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)) + (power_of<2>(log(1.0 - 1.0i * xd)) + power_of<2>(log(1.0 + 1.0i * xd))) * ((4.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv)
                      + (4.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)) + (dilog(-(xd / wx3)) + dilog(-(xd / wx4))) * (2.0 * dilog((1.0 - yd) / 2.0) + 4.0 * dilog((w5 - yd) / (-1.0 + w5)) - 2.0 * dilog((w7 - yd) / (1.0 + w7)) - 2.0 * dilog(-yd) - 2.0 * dilog(yd) + 4.0 * dilog(-(yd / w5)) + 4.0 * dilog(yd / w5) - 2.0 * dilog(-(yd / w7)) - 2.0 * dilog(yd / w7) + 2.0 * dilog((1.0 + yd) / 2.0) + 4.0 * dilog((w5 + yd) / (-1.0 + w5)) - 2.0 * dilog((w7 + yd) / (1.0 + w7)) - 2.0 * ln2 * ln1myd
                      - 2.0 * ln2 * ln1pyd + 4.0 * log((1.0 + yd) / (1.0 - w5)) * log((w5 + yd) / w5) - 2.0 * log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7) + 4.0 * log((-1.0 + yd) / (-1.0 + w5)) * log(1.0 - yd / w5) - 2.0 * log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7) + (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                    + (dilog((-1.0i + xd) / (-1.0i + wx3)) + dilog((-1.0i + xd) / (-1.0i + wx4)) + dilog((1.0i + xd) / (1.0i + wx3)) + dilog((1.0i + xd) / (1.0i + wx4))) * (-2.0 * dilog((1.0 - yd) / 2.0) - 4.0 * dilog((w4 - yd) / (1.0 + w4)) + 2.0 * dilog((w7 - yd) / (1.0 + w7)) + 2.0 * dilog(-yd) + 2.0 * dilog(yd) - 4.0 * dilog(-(yd / w4)) - 4.0 * dilog(yd / w4) + 2.0 * dilog(-(yd / w7)) + 2.0 * dilog(yd / w7) - 2.0 * dilog((1.0 + yd) / 2.0) - 4.0 * dilog((w4 + yd) / (1.0 + w4)) + 2.0 * dilog((w7 + yd) / (1.0 + w7)) + 2.0 * ln2 * ln1myd + 2.0 * ln2 * ln1pyd - 4.0 * log((1.0 - yd) / (1.0 + w4)) * log((w4 + yd) / w4)
                      + 2.0 * log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7) - 4.0 * log((1.0 + yd) / (1.0 + w4)) * log(1.0 - yd / w4) + 2.0 * log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                    + (dilog((1.0i - xd) / (1.0i + wx3)) + dilog((1.0i - xd) / (1.0i + wx4)) + dilog((1.0i + xd) / (1.0i - wx3)) + dilog((1.0i + xd) / (1.0i - wx4))) * (-2.0 * dilog((1.0 - yd) / 2.0) - 4.0 * dilog((w5 - yd) / (-1.0 + w5)) + 2.0 * dilog((w7 - yd) / (1.0 + w7)) + 2.0 * dilog(-yd) + 2.0 * dilog(yd) - 4.0 * dilog(-(yd / w5)) - 4.0 * dilog(yd / w5) + 2.0 * dilog(-(yd / w7)) + 2.0 * dilog(yd / w7) - 2.0 * dilog((1.0 + yd) / 2.0) - 4.0 * dilog((w5 + yd) / (-1.0 + w5)) + 2.0 * dilog((w7 + yd) / (1.0 + w7)) + 2.0 * ln2 * ln1myd + 2.0 * ln2 * ln1pyd - 4.0 * log((1.0 + yd) / (1.0 - w5)) * log((w5 + yd) / w5)
                      + 2.0 * log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7) - 4.0 * log((-1.0 + yd) / (-1.0 + w5)) * log(1.0 - yd / w5) + 2.0 * log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                    + (dilog(-1.0i / (-1.0i + wx3)) + dilog(1.0i / (1.0i + wx3)) + dilog(-1.0i / (-1.0i + wx4)) + dilog(1.0i / (1.0i + wx4)) - dilog(1.0 / 2.0 + (1.0i / 2.0) * xd) - dilog((-1.0i) * xd) - dilog(1.0i * xd) - dilog(((-0.5) * 1.0i) * (1.0i + xd))) * (4.0 * dilog((1.0 - yd) / 2.0) + 4.0 * dilog((w4 - yd) / (1.0 + w4)) + 4.0 * dilog((w5 - yd) / (-1.0 + w5)) - 4.0 * dilog((w7 - yd) / (1.0 + w7)) - 4.0 * dilog(-yd) - 4.0 * dilog(yd) + 4.0 * dilog(-(yd / w4)) + 4.0 * dilog(yd / w4) + 4.0 * dilog(-(yd / w5)) + 4.0 * dilog(yd / w5) - 4.0 * dilog(-(yd / w7)) - 4.0 * dilog(yd / w7) + 4.0 * dilog((1.0 + yd) / 2.0) + 4.0 * dilog((w4 + yd) / (1.0 + w4))
                      + 4.0 * dilog((w5 + yd) / (-1.0 + w5)) - 4.0 * dilog((w7 + yd) / (1.0 + w7)) - 4.0 * ln2 * ln1myd - 4.0 * ln2 * ln1pyd + 4.0 * log((1.0 - yd) / (1.0 + w4)) * log((w4 + yd) / w4) + 4.0 * log((1.0 + yd) / (1.0 - w5)) * log((w5 + yd) / w5) - 4.0 * log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7) + 4.0 * log((1.0 + yd) / (1.0 + w4)) * log(1.0 - yd / w4) + 4.0 * log((-1.0 + yd) / (-1.0 + w5)) * log(1.0 - yd / w5) - 4.0 * log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7) + (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv)
                      + (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                    + li2half * (-16.0 * dilog(w4 / (1.0 + w4)) - 16.0 * dilog(w5 / (-1.0 + w5)) + 16.0 * dilog(w7 / (1.0 + w7)) - 8.0 * dilog(-1.0i / (-1.0i + wx3)) - 8.0 * dilog(1.0i / (1.0i + wx3)) - 8.0 * dilog(-1.0i / (-1.0i + wx4)) - 8.0 * dilog(1.0i / (1.0i + wx4)) + 4.0 * dilog((1.0i - xd) / (1.0i + wx3)) + 4.0 * dilog((1.0i - xd) / (1.0i + wx4)) + 8.0 * dilog(1.0 / 2.0 + (1.0i / 2.0) * xd) + 8.0 * dilog((-1.0i) * xd) + 8.0 * dilog(1.0i * xd) - 4.0 * dilog(-(xd / wx3)) - 4.0 * dilog(xd / wx3) - 4.0 * dilog(-(xd / wx4)) - 4.0 * dilog(xd / wx4) + 4.0 * dilog((-1.0i + xd) / (-1.0i + wx3)) + 4.0 * dilog((-1.0i + xd) / (-1.0i + wx4)) + 8.0 * dilog(((-0.5) * 1.0i) * (1.0i + xd))
                      + 4.0 * dilog((1.0i + xd) / (1.0i - wx3)) + 4.0 * dilog((1.0i + xd) / (1.0i + wx3)) + 4.0 * dilog((1.0i + xd) / (1.0i - wx4)) + 4.0 * dilog((1.0i + xd) / (1.0i + wx4)) + 8.0 * dilog((1.0 - yd) / 2.0) + 8.0 * dilog((w4 - yd) / (1.0 + w4)) + 8.0 * dilog((w5 - yd) / (-1.0 + w5)) - 8.0 * dilog((w7 - yd) / (1.0 + w7)) - 8.0 * dilog(-yd) - 8.0 * dilog(yd) + 8.0 * dilog(-(yd / w4)) + 8.0 * dilog(yd / w4) + 8.0 * dilog(-(yd / w5)) + 8.0 * dilog(yd / w5) - 8.0 * dilog(-(yd / w7)) - 8.0 * dilog(yd / w7) + 8.0 * dilog((1.0 + yd) / 2.0) + 8.0 * dilog((w4 + yd) / (1.0 + w4)) + 8.0 * dilog((w5 + yd) / (-1.0 + w5)) - 8.0 * dilog((w7 + yd) / (1.0 + w7))
                      - 4.0 * power_of<2>(log(1.0 - 1.0i * xd)) - 4.0 * power_of<2>(log(1.0 + 1.0i * xd)) + log(1.0 - 1.0i * xd) * (-8.0 * ln2 + 4.0 * log((wx3 - xd) / (1.0i + wx3)) + 4.0 * log((wx4 - xd) / (1.0i + wx4)) + 8.0 * log(xd) + 4.0 * log((wx3 + xd) / (-1.0i + wx3)) + 4.0 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (-8.0 * ln2 + 4.0 * log((wx3 - xd) / (-1.0i + wx3)) + 4.0 * log((wx4 - xd) / (-1.0i + wx4)) + 8.0 * log(xd) + 4.0 * log((wx3 + xd) / (1.0i + wx3)) + 4.0 * log((wx4 + xd) / (1.0i + wx4))) + log(xd) * (-4.0 * log((wx3 + xd) / wx3) - 4.0 * log((wx4 + xd) / wx4) - 4.0 * log(1.0 - xd / wx3) - 4.0 * log(1.0 - xd / wx4))
                      - 8.0 * ln2 * ln1myd - 8.0 * ln2 * ln1pyd + 8.0 * log((1.0 - yd) / (1.0 + w4)) * log((w4 + yd) / w4) + 8.0 * log((1.0 + yd) / (1.0 - w5)) * log((w5 + yd) / w5) - 8.0 * log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7) + 8.0 * log((1.0 + yd) / (1.0 + w4)) * log(1.0 - yd / w4) + 8.0 * log((-1.0 + yd) / (-1.0 + w5)) * log(1.0 - yd / w5) - 8.0 * log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7) + (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) + (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                      + (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) + (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) + (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (8.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) + (8.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) + (16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv)
                      + (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (16.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (16.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                    + log(xd) * (log(1.0 - xd / wx3) * ((8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log(1.0 - xd / wx4) * ((8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx3 + xd) / wx3) * ((8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx4 + xd) / wx4) * ((8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)))
                    + log(1.0 - 1.0i * xd) * ((8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * ln2 * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * ln2 * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)
                      + log((wx3 - xd) / (1.0i + wx3)) * ((-8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx4 - xd) / (1.0i + wx4)) * ((-8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx3 + xd) / (-1.0i + wx3)) * ((-8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx4 + xd) / (-1.0i + wx4)) * ((-8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log(xd) * ((-8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)))
                    + log(1.0 + 1.0i * xd) * ((8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * ln2 * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) - (8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * ln2 * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (8.0 * 1.0i) * M_PI * ln2 * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)
                      + log((wx3 - xd) / (-1.0i + wx3)) * ((-8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx4 - xd) / (-1.0i + wx4)) * ((-8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx3 + xd) / (1.0i + wx3)) * ((-8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log((wx4 + xd) / (1.0i + wx4)) * ((-8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                      + log(xd) * ((-8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) - (8.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) - (8.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)))) / denom3;

            const complex<double> f27dPart4 = (num21 * (li3half * (32.0 * log(1.0 - 1.0i * xd) + 32.0 * log(1.0 + 1.0i * xd) - 32.0 * log(xd)) + trilog(1.0 - yd) * (10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd)) + trilog(-yd) * (10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd)) + trilog(yd) * (10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd)) + trilog(yd / (-1.0 + yd)) * (10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd)) + trilog(yd / (1.0 + yd)) * (10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd))
                    + trilog(1.0 + yd) * (10.0 * log(1.0 - 1.0i * xd) + 10.0 * log(1.0 + 1.0i * xd) - 10.0 * log(xd)) + trilog((-1.0 + w4) / (2.0 * w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog((1.0 - w4) / (1.0 + w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog((1.0 + w4) / (1.0 - w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog((1.0 + w4) / (2.0 * w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog((-1.0 + w5) / (2.0 * w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd))
                    + trilog((1.0 - w5) / (1.0 + w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog((1.0 + w5) / (1.0 - w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog((1.0 + w5) / (2.0 * w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(xd)) + trilog(1.0 + w7inv) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(1.0 / (1.0 - w7)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(-w7inv) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                    + trilog(w7inv) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + w7) / w7) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(1.0 / (1.0 + w7)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 - yd) / (1.0 + w4)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 - yd) / (1.0 + w5)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + w4) / (w4 - yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                    + trilog((1.0 + w4) / (w4 - yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + w5) / (w5 - yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + w5) / (w5 - yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + yd) / (-1.0 + w4)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + yd) / (-1.0 + w5)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((w4 * (-1.0 + yd)) / ((-1.0 + w4) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                    + trilog((w5 * (-1.0 + yd)) / ((1.0 + w5) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + yd) / (1.0 - w4)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + yd) / (1.0 + w4)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + yd) / (1.0 - w5)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + yd) / (1.0 + w5)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog(-((1.0 + yd) / (w4 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                    + trilog(-((1.0 + yd) / (w5 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((w4 * (1.0 + yd)) / ((-1.0 + w4) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + yd) / (-w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + w4) / (w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + w4) / (w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + yd) / (w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                    + trilog((1.0 + yd) / (w4 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + yd) / (-w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + w5) / (w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + w5) / (w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-1.0 + yd) / (w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((1.0 + yd) / (w5 + yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd))
                    + trilog((w5 + w5 * yd) / (yd + w5 * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) + trilog((-0.5) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((w7 * (-1.0 + yd)) / ((1.0 + w7) * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog(((1.0 + w7) * (-1.0 + yd)) / ((-1.0 + w7) * (1.0 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                    + trilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog(((1.0 + w7) * (1.0 + yd)) / ((-1.0 + w7) * (-1.0 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((-0.5) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd))
                    + trilog((w7 + w7 * yd) / (yd + w7 * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) + trilog((1.0 - yd) / (1.0 + w7)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((-1.0 + w7) / (w7 - yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((1.0 + w7) / (w7 - yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((-1.0 + yd) / (-1.0 + w7)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((w7 * (-1.0 + yd)) / ((-1.0 + w7) * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd))
                    + trilog((1.0 + yd) / (1.0 - w7)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((1.0 + yd) / (1.0 + w7)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog(-((1.0 + yd) / (w7 - yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((w7 * (1.0 + yd)) / ((-1.0 + w7) * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((-1.0 + yd) / (-w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((-1.0 + w7) / (w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd))
                    + trilog((1.0 + w7) / (w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((-1.0 + yd) / (w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((1.0 + yd) / (w7 + yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) + trilog((-1.0 + w7) / (2.0 * w7)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 - w7) / (1.0 + w7)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((1.0 + w7) / (1.0 - w7)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                    + trilog((1.0 + w7) / (2.0 * w7)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-0.5) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-0.5) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((w4 * (-1.0 + yd)) / ((1.0 + w4) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((w5 * (-1.0 + yd)) / ((-1.0 + w5) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                    + trilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((1.0 + w4) * (-1.0 + yd)) / ((-1.0 + w4) * (1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((-1.0 + w5) * (-1.0 + yd)) / ((1.0 + w5) * (1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                    + trilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((1.0 + w4) * (1.0 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((-1.0 + w5) * (1.0 + yd)) / ((1.0 + w5) * (-1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                    + trilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((w5 * (1.0 + yd)) / ((-1.0 + w5) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-0.5) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((-0.5) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd))
                    + trilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog((w4 + w4 * yd) / (yd + w4 * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) + trilog(1.0 + w4inv) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(1.0 / (1.0 - w4)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(-w4inv) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(w4inv) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                    + trilog((-1.0 + w4) / w4) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(1.0 / (1.0 + w4)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(1.0 + w5inv) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(1.0 / (1.0 - w5)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(-w5inv) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(w5inv) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd))
                    + trilog((-1.0 + w5) / w5) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog(1.0 / (1.0 + w5)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog((1.0 - yd) / 2.0) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog((1.0 + yd) / 2.0) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 8.0 * log(xd)) + trilog((2.0 * yd) / (-1.0 + yd)) * (-16.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0 + 1.0i * xd) + 16.0 * log(xd)) + trilog((2.0 * yd) / (1.0 + yd)) * (-16.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0 + 1.0i * xd) + 16.0 * log(xd))
                    + dilog(-w4inv) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd)) + dilog(w4inv) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd)) + dilog(-w5inv) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd)) + dilog(w5inv) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd))
                    + dilog((-1.0 + w7) / (2.0 * w7)) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd) - 4.0 * lnhalf * log(xd)) + dilog((1.0 - w7) / (1.0 + w7)) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd) - 4.0 * lnhalf * log(xd)) + dilog((1.0 + w7) / (1.0 - w7)) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd) - 4.0 * lnhalf * log(xd)) + dilog((1.0 + w7) / (2.0 * w7)) * (4.0 * lnhalf * log(1.0 - 1.0i * xd) + 4.0 * lnhalf * log(1.0 + 1.0i * xd) - 4.0 * lnhalf * log(xd))
                    + dilog(-w7inv) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd)) + dilog(w7inv) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd)) + dilog((-1.0 + w4) / (2.0 * w4)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog((1.0 - w4) / (1.0 + w4)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd))
                    + dilog((1.0 + w4) / (1.0 - w4)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog((1.0 + w4) / (2.0 * w4)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog((-1.0 + w5) / (2.0 * w5)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog((1.0 - w5) / (1.0 + w5)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd))
                    + dilog((1.0 + w5) / (1.0 - w5)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog((1.0 + w5) / (2.0 * w5)) * (-8.0 * lnhalf * log(1.0 - 1.0i * xd) - 8.0 * lnhalf * log(1.0 + 1.0i * xd) + 8.0 * lnhalf * log(xd)) + dilog((-0.5) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-0.5) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                    + dilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((1.0 + w4) * (-1.0 + yd)) / ((-1.0 + w4) * (1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((-1.0 + w5) * (-1.0 + yd)) / ((1.0 + w5) * (1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0)
                    + dilog((-0.5) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-0.5) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (-w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0)
                    + dilog((-0.5) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog(((1.0 + w7) * (-1.0 + yd)) / ((-1.0 + w7) * (1.0 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-0.5) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 - yd) / 2.0)
                    + dilog((-1.0 + yd) / (-w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (-w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + dilog((w4 * (-1.0 + yd)) / ((1.0 + w4) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd
                    + dilog((w5 * (-1.0 + yd)) / ((-1.0 + w5) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + dilog((w7 * (-1.0 + yd)) / ((-1.0 + w7) * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd + dilog((w7 * (-1.0 + yd)) / ((1.0 + w7) * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd + dilog((w4 * (-1.0 + yd)) / ((-1.0 + w4) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd
                    + dilog((w5 * (-1.0 + yd)) / ((1.0 + w5) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd + dilog(1.0 - yd) * (-16.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0 + 1.0i * xd) + 16.0 * log(xd)) * ln1myd + dilog((1.0 - yd) / (1.0 + w5)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd)
                    + dilog((-1.0 + yd) / (-1.0 + w4)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd) + dilog((-1.0 + yd) / (-1.0 + w7)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w4 - yd))
                    + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w4 - yd))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w4 - yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w4 - yd))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w5 - yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w5 - yd)))
                    + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w5 - yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w5 - yd))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(-2.0 / (w7 - yd)) + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(-2.0 / (w7 - yd))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(2.0 / (w7 - yd))
                    + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(2.0 / (w7 - yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd))) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd)))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((-2.0 * (1.0 + w4)) / ((-1.0 + w4) * (-1.0 + yd)))
                    + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((-2.0 * (1.0 + w4)) / ((-1.0 + w4) * (-1.0 + yd)))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((-2.0 * (-1.0 + w5)) / ((1.0 + w5) * (-1.0 + yd))) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((-2.0 * (-1.0 + w5)) / ((1.0 + w5) * (-1.0 + yd)))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd)))
                    + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd)))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd))) + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd)))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log((-2.0 * (1.0 + w7)) / ((-1.0 + w7) * (-1.0 + yd)))
                    + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log((-2.0 * (1.0 + w7)) / ((-1.0 + w7) * (-1.0 + yd)))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(w4 / ((-1.0 + w4) * yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(w4 / ((-1.0 + w4) * yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(w5 / ((-1.0 + w5) * yd)) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(w5 / ((-1.0 + w5) * yd)))
                    + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(w7 / ((-1.0 + w7) * yd)) + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(w7 / ((-1.0 + w7) * yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd))) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd))))
                    + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((2.0 * (1.0 + w4)) / ((-1.0 + w4) * (1.0 + yd))) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((2.0 * (1.0 + w4)) / ((-1.0 + w4) * (1.0 + yd)))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((2.0 * (-1.0 + w5)) / ((1.0 + w5) * (1.0 + yd))) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((2.0 * (-1.0 + w5)) / ((1.0 + w5) * (1.0 + yd))))
                    + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd))) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd)))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd))) + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd))))
                    + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log((2.0 * (1.0 + w7)) / ((-1.0 + w7) * (1.0 + yd))) + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log((2.0 * (1.0 + w7)) / ((-1.0 + w7) * (1.0 + yd)))) + dilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                    + dilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w4) * (1.0 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((-1.0 + w5) * (1.0 + yd)) / ((1.0 + w5) * (-1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                    + dilog(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(-((1.0 + yd) / (w7 - yd))) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog((1.0 + yd) / (w7 + yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0)
                    + dilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w7) * (1.0 + yd)) / ((-1.0 + w7) * (-1.0 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + yd) / 2.0)
                    + dilog(-((1.0 + yd) / (w4 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog(-((1.0 + yd) / (w5 - yd))) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog((1.0 + yd) / (w4 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + dilog((1.0 + yd) / (w5 + yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0)
                    + dilog((1.0 + yd) / (1.0 + w7)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0)) + dilog((1.0 + yd) / (1.0 + w4)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0)) + dilog((1.0 + yd) / (1.0 - w5)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0))
                    + dilog((w5 * (1.0 + yd)) / ((-1.0 + w5) * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd + dilog((w4 + w4 * yd) / (yd + w4 * yd)) * (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd + dilog((w7 * (1.0 + yd)) / ((-1.0 + w7) * yd)) * (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1pyd + dilog((w7 + w7 * yd) / (yd + w7 * yd)) * (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd
                    + dilog((w4 * (1.0 + yd)) / ((-1.0 + w4) * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd + dilog((w5 + w5 * yd) / (yd + w5 * yd)) * (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd + dilog(1.0 + yd) * (-16.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0 + 1.0i * xd) + 16.0 * log(xd)) * ln1pyd + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 - yd) / (1.0 + w4)) * power_of<2>(ln1pyd) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) * power_of<2>(ln1pyd)
                    + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 - yd) / (1.0 + w7)) * power_of<2>(ln1pyd) + dilog((1.0 + yd) / 2.0) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (12.0 * log(1.0 - 1.0i * xd) + 12.0 * log(1.0 + 1.0i * xd) - 12.0 * log(xd)) * ln1pyd) + dilog((1.0 - yd) / 2.0) * ((12.0 * log(1.0 - 1.0i * xd) + 12.0 * log(1.0 + 1.0i * xd) - 12.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd)
                    + dilog(-2.0 / (-1.0 + w4)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog(2.0 / (1.0 + w4)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog(-2.0 / (-1.0 + w5)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd)
                    + dilog(2.0 / (1.0 + w5)) * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog(1.0 / (1.0 + w4)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd)
                    + dilog(1.0 / (1.0 - w5)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1myd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog((1.0 + yd) / (1.0 - w4)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd)
                    + dilog((1.0 + yd) / (1.0 + w5)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 + yd) / 2.0) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * ln1pyd) + dilog((1.0 - yd) / (1.0 + w7)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 - yd) / 2.0) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1pyd)
                    + dilog(1.0 / (1.0 - w7)) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1myd + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * ln1pyd) + dilog(-2.0 / (-1.0 + w7)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd)
                    + dilog(2.0 / (1.0 + w7)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd) + dilog(1.0 / (1.0 + w7)) * (-4.0 * lnhalf * log(1.0 - 1.0i * xd) - 4.0 * lnhalf * log(1.0 + 1.0i * xd) + 4.0 * lnhalf * log(xd) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1myd + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd)
                    + dilog((1.0 + yd) / (1.0 - w7)) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((1.0 + yd) / 2.0) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * ln1pyd) + dilog((1.0 - yd) / (1.0 + w4)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog((-1.0 + yd) / (-1.0 + w5)) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((1.0 - yd) / 2.0) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd)
                    + dilog(1.0 / (1.0 - w4)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd) + dilog(1.0 / (1.0 + w5)) * (8.0 * lnhalf * log(1.0 - 1.0i * xd) + 8.0 * lnhalf * log(1.0 + 1.0i * xd) - 8.0 * lnhalf * log(xd) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1myd + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * ln1pyd)
                    + dilog(-yd) * ((-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd)) * ln1myd + (-6.0 * log(1.0 - 1.0i * xd) - 6.0 * log(1.0 + 1.0i * xd) + 6.0 * log(xd)) * ln1pyd) + dilog(yd) * ((-6.0 * log(1.0 - 1.0i * xd) - 6.0 * log(1.0 + 1.0i * xd) + 6.0 * log(xd)) * ln1myd + (-10.0 * log(1.0 - 1.0i * xd) - 10.0 * log(1.0 + 1.0i * xd) + 10.0 * log(xd)) * ln1pyd) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w4 + yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w4 + yd)))
                    + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w4 + yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w4 + yd))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(-2.0 / (w5 + yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(-2.0 / (w5 + yd))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(2.0 / (w5 + yd))
                    + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(2.0 / (w5 + yd))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(-2.0 / (w7 + yd)) + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(-2.0 / (w7 + yd))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(2.0 / (w7 + yd)) + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(2.0 / (w7 + yd)))
                    + log((1.0 - yd) / 2.0) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) * ln1pyd + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) * ln1pyd + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) * ln1pyd + (6.0 * log(1.0 - 1.0i * xd) + 6.0 * log(1.0 + 1.0i * xd) - 6.0 * log(xd)) * power_of<2>(ln1pyd)
                      + ln1pyd * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 + yd) / (1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 + yd) / (1.0 + w7)))) + power_of<2>(log((1.0 + yd) / 2.0)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + w4) / (w4 - yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 - yd) / (1.0 + w4))
                      + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + w5) / (w5 - yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((1.0 + w7) / (w7 - yd)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 - yd) / (1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-2.0 * (w4 - yd)) / ((1.0 + w4) * (-1.0 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-2.0 * (w5 - yd)) / ((1.0 + w5) * (-1.0 + yd)))
                      + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((-2.0 * (w7 - yd)) / ((1.0 + w7) * (-1.0 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-0.5) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-0.5) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((-0.5) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-1.0 + w4) / (w4 + yd))
                      + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-0.5) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-2.0 * (w4 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-1.0 + w5) / (w5 + yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-0.5) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd)))
                      + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((-2.0 * (w5 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((-1.0 + w7) / (w7 + yd)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((-0.5) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 + yd) / (-1.0 + w7))
                      + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((-2.0 * (w7 + yd)) / ((-1.0 + w7) * (-1.0 + yd)))) + power_of<2>(log((1.0 - yd) / 2.0)) * ((-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-1.0 + w4) / (w4 - yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((-1.0 + w5) / (w5 - yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5))
                      + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((-1.0 + w7) / (w7 - yd)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 - yd) / (-1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((2.0 * (w4 - yd)) / ((-1.0 + w4) * (1.0 + yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((2.0 * (w5 - yd)) / ((-1.0 + w5) * (1.0 + yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((2.0 * (w7 - yd)) / ((-1.0 + w7) * (1.0 + yd)))
                      + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + w4) / (w4 + yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd)))
                      + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 + yd) / (1.0 + w4)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((2.0 * (w4 + yd)) / ((1.0 + w4) * (1.0 + yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((1.0 + w5) / (w5 + yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 + yd) / (1.0 + w5))
                      + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((2.0 * (w5 + yd)) / ((1.0 + w5) * (1.0 + yd))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((1.0 + w7) / (w7 + yd)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 + yd) / (1.0 + w7)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((2.0 * (w7 + yd)) / ((1.0 + w7) * (1.0 + yd))))
                    + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(w4 / (yd - w4 * yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(w4 / (yd - w4 * yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(-(w4 / (yd + w4 * yd))) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(-(w4 / (yd + w4 * yd)))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(w4 / (yd + w4 * yd))
                    + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(w4 / (yd + w4 * yd))) + ((-2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(xd)) / 3.0) * log(w5 / (yd - w5 * yd)) + ((-2.0 * log(1.0 - 1.0i * xd)) / 3.0 - (2.0 * log(1.0 + 1.0i * xd)) / 3.0 + (2.0 * log(xd)) / 3.0) * power_of<3>(log(w5 / (yd - w5 * yd))) + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(-(w5 / (yd + w5 * yd))) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(-(w5 / (yd + w5 * yd))))
                    + ((2.0 * pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * pisqu * log(xd)) / 3.0) * log(w5 / (yd + w5 * yd)) + ((2.0 * log(1.0 - 1.0i * xd)) / 3.0 + (2.0 * log(1.0 + 1.0i * xd)) / 3.0 - (2.0 * log(xd)) / 3.0) * power_of<3>(log(w5 / (yd + w5 * yd))) + ((-1.0 / 3.0) * (pisqu * log(1.0 - 1.0i * xd)) - (pisqu * log(1.0 + 1.0i * xd)) / 3.0 + (pisqu * log(xd)) / 3.0) * log(w7 / (yd - w7 * yd)) + ((-1.0 / 3.0) * log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) / 3.0 + log(xd) / 3.0) * power_of<3>(log(w7 / (yd - w7 * yd))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(-(w7 / (yd + w7 * yd)))
                    + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(-(w7 / (yd + w7 * yd)))) + ((pisqu * log(1.0 - 1.0i * xd)) / 3.0 + (pisqu * log(1.0 + 1.0i * xd)) / 3.0 - (pisqu * log(xd)) / 3.0) * log(w7 / (yd + w7 * yd)) + (log(1.0 - 1.0i * xd) / 3.0 + log(1.0 + 1.0i * xd) / 3.0 - log(xd) / 3.0) * power_of<3>(log(w7 / (yd + w7 * yd))) + power_of<2>(ln1pyd) * ((-2.0 * ln2 - ln64 / 2.0) * log(1.0 - 1.0i * xd) + (-2.0 * ln2 - ln64 / 2.0) * log(1.0 + 1.0i * xd) + (2.0 * ln2 + ln64 / 2.0) * log(xd) + (-11.0 * log(1.0 - 1.0i * xd) - 11.0 * log(1.0 + 1.0i * xd) + 11.0 * log(xd)) * log(-yd)
                      + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((1.0 + yd) / 8.0) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 + yd) / (-1.0 + w7)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 + yd) / (yd - w4 * yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log(-((w4 - yd) / (yd + w4 * yd)))
                      + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 + yd) / (yd - w5 * yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log(-((w5 - yd) / (yd + w5 * yd))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 + yd) / (yd - w7 * yd)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log(-((w7 - yd) / (yd + w7 * yd))))
                    + power_of<2>(ln1myd) * ((-2.0 * ln2 - ln64 / 2.0) * log(1.0 - 1.0i * xd) + (-2.0 * ln2 - ln64 / 2.0) * log(1.0 + 1.0i * xd) + (2.0 * ln2 + ln64 / 2.0) * log(xd) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((1.0 - yd) / 8.0) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 - yd) / (-1.0 + w7))
                      + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 - yd) / ((-1.0 + w4) * yd)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 - yd) / ((-1.0 + w5) * yd)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 - yd) / ((-1.0 + w7) * yd)) + (-11.0 * log(1.0 - 1.0i * xd) - 11.0 * log(1.0 + 1.0i * xd) + 11.0 * log(xd)) * log(yd) + (6.0 * log(1.0 - 1.0i * xd) + 6.0 * log(1.0 + 1.0i * xd) - 6.0 * log(xd)) * log((1.0 + yd) / 2.0) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w4 + yd) / (1.0 + w4))
                      + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * log((w7 + yd) / (1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w4 + yd) / (yd + w4 * yd)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w5 + yd) / (yd + w5 * yd)) + (-log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd) + log(xd)) * log((w7 + yd) / (yd + w7 * yd)))
                    + power_of<2>(log((-w4 + yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log((w4 + yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                      + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log((-w5 + yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log((w5 + yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log((-w7 + yd) / 2.0)) * ((2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                      - (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))) + power_of<2>(log((w7 + yd) / 2.0)) * ((2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log(((-1.0 + w4) * (-1.0 + yd)) / (2.0 * (1.0 + w4)))) * ((4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log(((-1.0 + w5) * (-1.0 + yd)) / (2.0 * (1.0 + w5)))) * ((4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log(((-1.0 + w7) * (-1.0 + yd)) / (2.0 * (1.0 + w7)))) * ((-2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 - yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log(((1.0 + w4) * (-1.0 + yd)) / (2.0 * (-1.0 + w4)))) * ((4.0 * 1.0i) * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log(((1.0 + w5) * (-1.0 + yd)) / (2.0 * (-1.0 + w5)))) * ((4.0 * 1.0i) * M_PI * H1((-1.0 + w5) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1((-1.0 + w5) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((-1.0 + w5) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log(((1.0 + w7) * (-1.0 + yd)) / (2.0 * (-1.0 + w7)))) * ((-2.0 * 1.0i) * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (2.0 * 1.0i) * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (2.0 * 1.0i) * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 - yd))))
                    + power_of<2>(log(((-1.0 + w4) * yd) / w4)) * ((-4.0 * 1.0i) * M_PI * H1(-(w4 / yd), -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) - (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) + (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), -ydinv) * log(xd) * my_sign(-imydinv)) + power_of<2>(log((1.0 + w4inv) * yd)) * ((4.0 * 1.0i) * M_PI * H1(w4 / yd, -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) + (4.0 * 1.0i) * M_PI * H1(w4 / yd, -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) - (4.0 * 1.0i) * M_PI * H1(w4 / yd, -ydinv) * log(xd) * my_sign(-imydinv))
                    + power_of<2>(log(((-1.0 + w5) * yd) / w5)) * ((4.0 * 1.0i) * M_PI * H1(-(w5 / yd), -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) + (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) - (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), -ydinv) * log(xd) * my_sign(-imydinv)) + power_of<2>(log((1.0 + w5inv) * yd)) * ((-4.0 * 1.0i) * M_PI * H1(w5 / yd, -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) - (4.0 * 1.0i) * M_PI * H1(w5 / yd, -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) + (4.0 * 1.0i) * M_PI * H1(w5 / yd, -ydinv) * log(xd) * my_sign(-imydinv))
                    + power_of<2>(log(((-1.0 + w7) * yd) / w7)) * ((2.0 * 1.0i) * M_PI * H1(-(w7 / yd), -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) + (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) - (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), -ydinv) * log(xd) * my_sign(-imydinv)) + power_of<2>(log((1.0 + w7inv) * yd)) * ((-2.0 * 1.0i) * M_PI * H1(w7 / yd, -ydinv) * log(1.0 - 1.0i * xd) * my_sign(-imydinv) - (2.0 * 1.0i) * M_PI * H1(w7 / yd, -ydinv) * log(1.0 + 1.0i * xd) * my_sign(-imydinv) + (2.0 * 1.0i) * M_PI * H1(w7 / yd, -ydinv) * log(xd) * my_sign(-imydinv))
                    + power_of<2>(log(-(((1.0 + w4) * yd) / w4))) * ((4.0 * 1.0i) * M_PI * H1(-(w4 / yd), ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) + (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) - (4.0 * 1.0i) * M_PI * H1(-(w4 / yd), ydinv) * log(xd) * my_sign(imydinv)) + power_of<2>(log((-1.0 + w4inv) * yd)) * ((-4.0 * 1.0i) * M_PI * H1(w4 / yd, ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) - (4.0 * 1.0i) * M_PI * H1(w4 / yd, ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) + (4.0 * 1.0i) * M_PI * H1(w4 / yd, ydinv) * log(xd) * my_sign(imydinv))
                    + power_of<2>(log(-(((1.0 + w5) * yd) / w5))) * ((-4.0 * 1.0i) * M_PI * H1(-(w5 / yd), ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) - (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) + (4.0 * 1.0i) * M_PI * H1(-(w5 / yd), ydinv) * log(xd) * my_sign(imydinv)) + power_of<2>(log((-1.0 + w5inv) * yd)) * ((4.0 * 1.0i) * M_PI * H1(w5 / yd, ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) + (4.0 * 1.0i) * M_PI * H1(w5 / yd, ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) - (4.0 * 1.0i) * M_PI * H1(w5 / yd, ydinv) * log(xd) * my_sign(imydinv))
                    + power_of<2>(log(-(((1.0 + w7) * yd) / w7))) * ((-2.0 * 1.0i) * M_PI * H1(-(w7 / yd), ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) - (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) + (2.0 * 1.0i) * M_PI * H1(-(w7 / yd), ydinv) * log(xd) * my_sign(imydinv)) + power_of<2>(log((-1.0 + w7inv) * yd)) * ((2.0 * 1.0i) * M_PI * H1(w7 / yd, ydinv) * log(1.0 - 1.0i * xd) * my_sign(imydinv) + (2.0 * 1.0i) * M_PI * H1(w7 / yd, ydinv) * log(1.0 + 1.0i * xd) * my_sign(imydinv) - (2.0 * 1.0i) * M_PI * H1(w7 / yd, ydinv) * log(xd) * my_sign(imydinv))
                    + power_of<2>(log((-w4 - yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((w4 - yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                      + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-w5 - yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                    + power_of<2>(log((w5 - yd) / 2.0)) * ((-4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-w7 - yd) / 2.0)) * ((2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (2.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                    + power_of<2>(log((w7 - yd) / 2.0)) * ((2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (2.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log(((1.0 + w4) * (1.0 + yd)) / (2.0 - 2.0 * w4))) * ((4.0 * 1.0i) * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                      - (4.0 * 1.0i) * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))) + power_of<2>(log((-0.5) * (((-1.0 + w4) * (1.0 + yd)) / (1.0 + w4)))) * ((4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                    + power_of<2>(log(((1.0 + w5) * (1.0 + yd)) / (2.0 - 2.0 * w5))) * ((4.0 * 1.0i) * M_PI * H1((1.0 - w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1((1.0 - w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 - w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                    + power_of<2>(log((-0.5) * (((-1.0 + w5) * (1.0 + yd)) / (1.0 + w5)))) * ((4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                    + power_of<2>(log(((1.0 + w7) * (1.0 + yd)) / (2.0 - 2.0 * w7))) * ((-2.0 * 1.0i) * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (2.0 * 1.0i) * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                    + power_of<2>(log((-0.5) * (((-1.0 + w7) * (1.0 + yd)) / (1.0 + w7)))) * ((-2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 - 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(1.0 + 1.0i * xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (2.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * log(xd) * my_sign(2.0 * imag(1.0 / (1.0 + yd))))
                    + power_of<2>(log((w4 - yd) / (1.0 + w4))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 - yd) / (1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 - yd) / (1.0 + w4)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 + w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 - yd) / (1.0 + w4)))
                    + power_of<2>(log((w5 - yd) / (1.0 + w5))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)))
                    + power_of<2>(log((w7 - yd) / (1.0 + w7))) * ((2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 - yd) / (1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 + w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 - yd) / (1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 + w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 - yd) / (1.0 + w7)))
                    + power_of<2>(log((w4 + yd) / (-1.0 + w4))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)))
                    + power_of<2>(log((w5 + yd) / (-1.0 + w5))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 + yd) / (-1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 + yd) / (-1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 - w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 + yd) / (-1.0 + w5)))
                    + power_of<2>(log((w7 + yd) / (-1.0 + w7))) * ((2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7)))
                    + power_of<2>(log((w4 - yd) / (-1.0 + w4))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4)))
                    + power_of<2>(log((w5 - yd) / (-1.0 + w5))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 - yd) / (-1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 - yd) / (-1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 - yd) / (-1.0 + w5)))
                    + power_of<2>(log((w7 - yd) / (-1.0 + w7))) * ((2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7)))
                    + power_of<2>(log((w4 + yd) / (1.0 + w4))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 + yd) / (1.0 + w4)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 + yd) / (1.0 + w4)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 - yd) / (1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 + yd) / (1.0 + w4)))
                    + power_of<2>(log((w5 + yd) / (1.0 + w5))) * ((-4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)) + (4.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)))
                    + power_of<2>(log((w7 + yd) / (1.0 + w7))) * ((2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 + yd) / (1.0 + w7)) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * xd) * my_sign(imag((1.0 - yd) / (1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 + yd) / (1.0 + w7)) - (2.0 * 1.0i) * M_PI * log(xd) * my_sign(imag((1.0 - yd) / (1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 + yd) / (1.0 + w7)))
                    + dilog((1.0 - w4) / 2.0) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + log(xd) * (-8.0 * log(w4 / (-1.0 + w4)) - (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4)) + log(1.0 - 1.0i * xd) * (8.0 * log(w4 / (-1.0 + w4)) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4))
                      + log(1.0 + 1.0i * xd) * (8.0 * log(w4 / (-1.0 + w4)) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4))) + dilog((1.0 + w4) / 2.0) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 - yd) / (1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 + yd) / (1.0 + w4)) + log(xd) * (-8.0 * log(w4 / (1.0 + w4)) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4))
                      + log(1.0 - 1.0i * xd) * (8.0 * log(w4 / (1.0 + w4)) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4)) + log(1.0 + 1.0i * xd) * (8.0 * log(w4 / (1.0 + w4)) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4))) + dilog((1.0 - w5) / 2.0) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5))
                      + log(xd) * (-8.0 * log(w5 / (-1.0 + w5)) - (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) - (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)) + log(1.0 - 1.0i * xd) * (8.0 * log(w5 / (-1.0 + w5)) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)) + log(1.0 + 1.0i * xd) * (8.0 * log(w5 / (-1.0 + w5)) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)))
                    + dilog((1.0 + w5) / 2.0) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + log(xd) * (-8.0 * log(w5 / (1.0 + w5)) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5)) + log(1.0 - 1.0i * xd) * (8.0 * log(w5 / (1.0 + w5)) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5))
                      + log(1.0 + 1.0i * xd) * (8.0 * log(w5 / (1.0 + w5)) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5))) + ln1myd * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) * ln1pyd + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) * ln1pyd + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) * ln1pyd
                      + log((1.0 + yd) / 2.0) * ((-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 - yd) / (1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 - yd) / (1.0 + w7)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (-4.0 * log(1.0 - 1.0i * xd) - 4.0 * log(1.0 + 1.0i * xd) + 4.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7)))
                      + ln1pyd * (-8.0 * ln2 * log(1.0 - 1.0i * xd) - 8.0 * ln2 * log(1.0 + 1.0i * xd) + 8.0 * ln2 * log(xd) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7)))
                      + log(1.0 - 1.0i * xd) * ((-2.0 * pisqu) / 3.0 + 10.0 * ln2squ - (8.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 - w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w7) - (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0) - (8.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) - (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0)
                        - (8.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0) + (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0) + (4.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0) + (8.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5) - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7))
                      + log(1.0 + 1.0i * xd) * ((-2.0 * pisqu) / 3.0 + 10.0 * ln2squ - (8.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 - w5) + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w7) - (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0) - (8.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) - (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0)
                        - (8.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0) + (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0) + (4.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0) + (8.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5) - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7))
                      + log(xd) * ((2.0 * pisqu) / 3.0 - 10.0 * ln2squ + (8.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 - w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w7) + (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0)
                        + (8.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0) - (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0) - (4.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0) - (8.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5) + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7)))
                    + dilog((1.0 - w7) / 2.0) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7)) + log(1.0 - 1.0i * xd) * (-4.0 * log(w7 / (-1.0 + w7)) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7)) + log(1.0 + 1.0i * xd) * (-4.0 * log(w7 / (-1.0 + w7)) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7))
                      + log(xd) * (4.0 * log(w7 / (-1.0 + w7)) + (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) + (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7))) + dilog((1.0 + w7) / 2.0) * ((2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 - yd) / (1.0 + w7)) + (2.0 * log(1.0 - 1.0i * xd) + 2.0 * log(1.0 + 1.0i * xd) - 2.0 * log(xd)) * log((w7 + yd) / (1.0 + w7)) + log(1.0 - 1.0i * xd) * (-4.0 * log(w7 / (1.0 + w7)) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7))
                      + log(1.0 + 1.0i * xd) * (-4.0 * log(w7 / (1.0 + w7)) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7)) + log(xd) * (4.0 * log(w7 / (1.0 + w7)) + (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) + (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7))) + li2half * ((4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 - yd) / (1.0 + w4))
                      + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 - yd) / (-1.0 + w5)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 - yd) / (1.0 + w5)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 - yd) / (1.0 + w7)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 + yd) / (-1.0 + w4)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w4 + yd) / (1.0 + w4))
                      + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 + yd) / (-1.0 + w5)) + (4.0 * log(1.0 - 1.0i * xd) + 4.0 * log(1.0 + 1.0i * xd) - 4.0 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 + yd) / (-1.0 + w7)) + (-2.0 * log(1.0 - 1.0i * xd) - 2.0 * log(1.0 + 1.0i * xd) + 2.0 * log(xd)) * log((w7 + yd) / (1.0 + w7))
                      + (log(xd) - log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd)) * (8.0 * log(w4 / (-1.0 + w4)) + 8.0 * log(w4 / (1.0 + w4)) + 8.0 * log(w5 / (-1.0 + w5)) + 8.0 * log(w5 / (1.0 + w5)) - 4.0 * log(w7 / (-1.0 + w7)) - 4.0 * log(w7 / (1.0 + w7)) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) + (8.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4)
                        + (8.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) - (4.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7))) + ln1pyd * ((log(xd) - log(1.0 - 1.0i * xd) - log(1.0 + 1.0i * xd)) * ((2.0 * pisqu) / 3.0 - 10.0 * ln2squ + (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w4) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0)
                       + (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w5) / 2.0) - (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w7) / 2.0) - (4.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0) - (8.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 - w4) - (8.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 + w5)
                       + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 - w7) + (8.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 + w4) + (8.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 - w5) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 + w7))) + (log(1.0 - 1.0i * xd) + log(1.0 + 1.0i * xd) - log(xd)) * ((-2.0 * pisqu * ln2) / 3.0 - (8.0 * power_of<3>(ln2)) / 3.0 + pisqu * ln4 + 4.0 * power_of<2>(lnhalf) * log(1.0 + w4inv) - (4.0 * pisqu * log(-2.0 / w4)) / 3.0 - (4.0 * power_of<3>(log(-2.0 / w4))) / 3.0 - (4.0 * pisqu * log(2.0 / w4)) / 3.0 - (4.0 * power_of<3>(log(2.0 / w4))) / 3.0
                      - 4.0 * power_of<2>(lnhalf) * log((-1.0 + w4) / (2.0 * w4)) + 4.0 * power_of<2>(lnhalf) * log((-1.0 + w4) / w4) + 4.0 * power_of<2>(lnhalf) * log(w4 / (-1.0 + w4)) - 4.0 * power_of<2>(lnhalf) * log((2.0 * w4) / (-1.0 + w4)) + (4.0 * pisqu * log((2.0 * (-1.0 + w4)) / (1.0 + w4))) / 3.0 + (4.0 * power_of<3>(log((2.0 * (-1.0 + w4)) / (1.0 + w4)))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log(w4 / (1.0 + w4)) - 4.0 * power_of<2>(lnhalf) * log((2.0 * w4) / (1.0 + w4)) + (4.0 * pisqu * log((2.0 * (1.0 + w4)) / (-1.0 + w4))) / 3.0 + (4.0 * power_of<3>(log((2.0 * (1.0 + w4)) / (-1.0 + w4)))) / 3.0 - 4.0 * power_of<2>(lnhalf) * log((1.0 + w4) / (2.0 * w4)) + 4.0 * power_of<2>(lnhalf) * log(1.0 + w5inv) - (4.0 * pisqu * log(-2.0 / w5)) / 3.0 - (4.0 * power_of<3>(log(-2.0 / w5))) / 3.0
                      - (4.0 * pisqu * log(2.0 / w5)) / 3.0 - (4.0 * power_of<3>(log(2.0 / w5))) / 3.0 - 4.0 * power_of<2>(lnhalf) * log((-1.0 + w5) / (2.0 * w5)) + 4.0 * power_of<2>(lnhalf) * log((-1.0 + w5) / w5) + 4.0 * power_of<2>(lnhalf) * log(w5 / (-1.0 + w5)) - 4.0 * power_of<2>(lnhalf) * log((2.0 * w5) / (-1.0 + w5)) + (4.0 * pisqu * log((2.0 * (-1.0 + w5)) / (1.0 + w5))) / 3.0 + (4.0 * power_of<3>(log((2.0 * (-1.0 + w5)) / (1.0 + w5)))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log(w5 / (1.0 + w5)) - 4.0 * power_of<2>(lnhalf) * log((2.0 * w5) / (1.0 + w5)) + (4.0 * pisqu * log((2.0 * (1.0 + w5)) / (-1.0 + w5))) / 3.0 + (4.0 * power_of<3>(log((2.0 * (1.0 + w5)) / (-1.0 + w5)))) / 3.0 - 4.0 * power_of<2>(lnhalf) * log((1.0 + w5) / (2.0 * w5)) - 2.0 * power_of<2>(lnhalf) * log(1.0 + w7inv)
                      + (2.0 * pisqu * log(-2.0 / w7)) / 3.0 + (2.0 * power_of<3>(log(-2.0 / w7))) / 3.0 + (2.0 * pisqu * log(2.0 / w7)) / 3.0 + (2.0 * power_of<3>(log(2.0 / w7))) / 3.0 + 2.0 * power_of<2>(lnhalf) * log((-1.0 + w7) / (2.0 * w7)) - 2.0 * power_of<2>(lnhalf) * log((-1.0 + w7) / w7) - 2.0 * power_of<2>(lnhalf) * log(w7 / (-1.0 + w7)) + 2.0 * power_of<2>(lnhalf) * log((2.0 * w7) / (-1.0 + w7)) - (2.0 * pisqu * log((2.0 * (-1.0 + w7)) / (1.0 + w7))) / 3.0 - (2.0 * power_of<3>(log((2.0 * (-1.0 + w7)) / (1.0 + w7)))) / 3.0 - 2.0 * power_of<2>(lnhalf) * log(w7 / (1.0 + w7)) + 2.0 * power_of<2>(lnhalf) * log((2.0 * w7) / (1.0 + w7)) - (2.0 * pisqu * log((2.0 * (1.0 + w7)) / (-1.0 + w7))) / 3.0 - (2.0 * power_of<3>(log((2.0 * (1.0 + w7)) / (-1.0 + w7)))) / 3.0
                      + 2.0 * power_of<2>(lnhalf) * log((1.0 + w7) / (2.0 * w7)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w4) / 2.0)) * my_sign(imag(w4) / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - w4) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log(w4 / (-1.0 + w4))) * my_sign(-imag(1.0 / (1.0 - w4))) * T(1.0, 1.0 / 2.0, w4 / (-1.0 + w4)) - (4.0 * 1.0i) * M_PI * power_of<2>(log(w4 / (-1.0 + w4))) * my_sign(imag(1.0 / (-1.0 + w4))) * T(1.0, 1.0 / 2.0, w4 / (-1.0 + w4)) - (8.0 * 1.0i) * M_PI * power_of<2>(log(w4 / (1.0 + w4))) * my_sign(-imag(1.0 / (1.0 + w4))) * T(1.0, 1.0 / 2.0, w4 / (1.0 + w4)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-0.5) * imag(w4)) * T(1.0, 1.0 / 2.0, (1.0 + w4) / 2.0) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - w5) / 2.0)
                      - (4.0 * 1.0i) * M_PI * power_of<2>(log(w5 / (-1.0 + w5))) * my_sign(-imag(1.0 / (1.0 - w5))) * T(1.0, 1.0 / 2.0, w5 / (-1.0 + w5)) - (4.0 * 1.0i) * M_PI * power_of<2>(log(w5 / (-1.0 + w5))) * my_sign(imag(1.0 / (-1.0 + w5))) * T(1.0, 1.0 / 2.0, w5 / (-1.0 + w5)) - (8.0 * 1.0i) * M_PI * power_of<2>(log(w5 / (1.0 + w5))) * my_sign(-imag(1.0 / (1.0 + w5))) * T(1.0, 1.0 / 2.0, w5 / (1.0 + w5)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w5) / 2.0)) * my_sign((-0.5) * imag(w5)) * T(1.0, 1.0 / 2.0, (1.0 + w5) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w7) / 2.0)) * my_sign(imag(w7) / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - w7) / 2.0) + (2.0 * 1.0i) * M_PI * power_of<2>(log(w7 / (-1.0 + w7))) * my_sign(-imag(1.0 / (1.0 - w7))) * T(1.0, 1.0 / 2.0, w7 / (-1.0 + w7))
                      + (2.0 * 1.0i) * M_PI * power_of<2>(log(w7 / (-1.0 + w7))) * my_sign(imag(1.0 / (-1.0 + w7))) * T(1.0, 1.0 / 2.0, w7 / (-1.0 + w7)) + (4.0 * 1.0i) * M_PI * power_of<2>(log(w7 / (1.0 + w7))) * my_sign(-imag(1.0 / (1.0 + w7))) * T(1.0, 1.0 / 2.0, w7 / (1.0 + w7)) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-0.5) * imag(w7)) * T(1.0, 1.0 / 2.0, (1.0 + w7) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w4) / 2.0)) * my_sign(imag(w4) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w4) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-0.5) * imag(w4)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0)
                      + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w5) / 2.0)) * my_sign((-0.5) * imag(w5)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w5) / 2.0) - (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w7) / 2.0)) * my_sign(imag(w7) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w7) / 2.0) - (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-0.5) * imag(w7)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w4)) * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * log(1.0 + w4) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4)
                      - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * log(1.0 - w5) * log((1.0 + w5) / 2.0) * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w5)) * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w7)) * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * log(1.0 + w7) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7)
                      + (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w4) / 2.0)) * my_sign(imag(w4) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-0.5) * imag(w4)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0) + (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w5) / 2.0)) * my_sign((-0.5) * imag(w5)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0) - (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w7) / 2.0)) * my_sign(imag(w7) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0)
                      - (2.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-0.5) * imag(w7)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w4)) * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4) + (8.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * log(1.0 + w4) * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4) - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4) - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * log(1.0 - w5) * log((1.0 + w5) / 2.0) * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)
                      + (8.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) - (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w5)) * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) + (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w7)) * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7) - (4.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * log(1.0 + w7) * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7) + (2.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7) - 34.0 * zeta3))) / denom3;

            const complex<double> f27dPart22 = f27d_part22(clp);

            const complex<double> f27dPart5 = - (num21 / denom3) * GPLweight4Parts(clp);

            const complex<double> f27dPart6 = - (num21 / denom3) * f279d_log2_terms(clp);

            return f27dPart1 + f27dPart21 + f27dPart22 + f27dPart3 + f27dPart4 + f27dPart5 + f27dPart6;
        }
    }
}
