/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2023 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/polylog.hh>
#include <eos/maths/multiplepolylog-li22.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>
#include <eos/utils/stringify.hh>

#include <cmath>
#include <complex>

namespace eos
{
	using std::complex;
	using std::log;
	using std::real;
	using std::imag;

	namespace agv_2019a
	{
        complex<double> f27d_part22(const CharmLoopsParameters & clp)
        {
            const complex<double> xd = clp.xd;
            const complex<double> yd = clp.yd;

            const complex<double> xdinv = 1.0 / xd;
            const complex<double> ydinv = 1.0 / yd;

            const double imydinv = imag(ydinv);

            // Powers of xd and yd

            const complex<double> xd2 = power_of<2>(xd);
            const complex<double> xd4 = power_of<4>(xd);
            const complex<double> xd6 = power_of<6>(xd);
            const complex<double> xd8 = power_of<8>(xd);

            const complex<double> yd2 = power_of<2>(yd);
            const complex<double> yd3 = power_of<3>(yd);
            const complex<double> yd4 = power_of<4>(yd);

            // weights appearing in the GPLs [AGV:2019A] p. 34

            const complex<double> w4 = (2.0 * xd) / power_of<2>(1.0 - xd);
            const complex<double> w5 = (2.0 * xd) / power_of<2>(1.0 + xd);
            const complex<double> w7 = (8.0 * xd2) / (1.0 - 6.0 * xd2 + xd4);

            const complex<double> w4inv = 1.0 / w4;
            const complex<double> w5inv = 1.0 / w5;
            const complex<double> w7inv = 1.0 / w7;

            // Logs of xd and yd

            const complex<double> ln1pyd = log(1.0 + yd);
            const complex<double> ln1myd = log(1.0 - yd);

            const complex<double> num14 = 2.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * power_of<2>(1.0 + yd) * ((-1.0 + yd) * yd2 + xd8 * (-1.0 + yd) * yd2
                    + 4.0 * xd2 * (-1.0 + yd + 4.0 * yd2 + 12.0 * yd3) + 4.0 * xd6 * (-1.0 + yd + 4.0 * yd2 + 12.0 * yd3)
                    + 2.0 * xd4 * (28.0 - 28.0 * yd - 15.0 * yd2 + 79.0 * yd3));
            const complex<double> num15 = 2.0 * power_of<2>(1.0 + xd2) * power_of<2>(-1.0 + yd) * (1.0 + yd) * (yd2 * (1.0 + yd) + xd8 * yd2 * (1.0 + yd) + 4.0 * xd2 * (1.0 + yd - 4.0 * yd2 + 12.0 * yd3)
                    + 4.0 * xd6 * (1.0 + yd - 4.0 * yd2 + 12.0 * yd3) + 2.0 * xd4 * (-28.0 - 28.0 * yd + 15.0 * yd2 + 79.0 * yd3));
            const complex<double> num16 = 4.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 3.0 * yd4 + xd8 * (yd2 + 3.0 * yd4) - 4.0 * xd2 * (-1.0 + 5.0 * yd2 + 8.0 * yd4)
                    - 4.0 * xd6 * (-1.0 + 5.0 * yd2 + 8.0 * yd4) - 2.0 * xd4 * (60.0 - 107.0 * yd2 + 99.0 * yd4));
            const complex<double> num17 = 16.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (-2.0 * 1.0i * xd * (-1.0 + yd) + yd + xd2 * yd) * (2.0 * 1.0i * xd * (-1.0 + yd) + yd + xd2 * yd) * (yd + xd2 * yd - 2.0 * xd * (1.0 + yd)) * (yd + xd2 * yd + 2.0 * xd * (1.0 + yd));
            const complex<double> num18 = 16.0 * power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (-2.0 * xd * (-1.0 + yd) + yd + xd2 * yd) * (2.0 * xd * (-1.0 + yd) + yd + xd2 * yd) * (yd + xd2 * yd - 2.0 * 1.0i * xd * (1.0 + yd)) * (yd + xd2 * yd + 2.0 * 1.0i * xd * (1.0 + yd));
            const complex<double> num19 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 7.0 * yd4 - 4.0 * xd2 * (-1.0 + 5.0 * yd2 - 16.0 * yd3 + 4.0 * yd4)
                    - 4.0 * xd6 * (-1.0 + 5.0 * yd2 - 16.0 * yd3 + 4.0 * yd4) + xd8 * (yd2 + 7.0 * yd4) - 2.0 * xd4 * (92.0 - 171.0 * yd2 - 64.0 * yd3 + 119.0 * yd4));
            const complex<double> num20 = power_of<2>(1.0 + xd2) * (-1.0 + yd) * (1.0 + yd) * (yd2 + 7.0 * yd4 - 4.0 * xd2 * (-1.0 + 5.0 * yd2 + 16.0 * yd3 + 4.0 * yd4)
                    - 4.0 * xd6 * (-1.0 + 5.0 * yd2 + 16.0 * yd3 + 4.0 * yd4) + xd8 * (yd2 + 7.0 * yd4) - 2.0 * xd4 * (92.0 - 171.0 * yd2 + 64.0 * yd3 + 119.0 * yd4));

            const complex<double> denom3 = 9.0 * power_of<3>(yd2 + xd4 * yd2 - 2.0 * xd2 * (-2.0 + yd2));

            const complex<double> result = ((-16.0 * num16 + 2.0 * num17 + 2.0 * num18 - num19 - num20) * li3half + (-num17 - num18) * trilog(1.0 / (1.0 - w4)) + (-num17 - num18) * trilog(-w4inv) + (num17 + num18) * trilog((-1.0 + w4) / (2.0 * w4))
                + (-num17 - num18) * trilog((-1.0 + w4) / w4) + (num17 + num18) * trilog(-w4) + (-num17 - num18) * trilog(w4) + (num17 + num18) * trilog((1.0 - w4) / (1.0 + w4)) + (-num17 - num18) * trilog(1.0 + w5inv)
                + (-num17 - num18) * trilog(w5inv) + (-num17 - num18) * trilog(-w5) + (num17 + num18) * trilog(w5) + (-num17 - num18) * trilog(1.0 / (1.0 + w5)) + (num17 + num18) * trilog((1.0 + w5) / (1.0 - w5))
                + (num17 + num18) * trilog((1.0 + w5) / (2.0 * w5)) + (num19 + num20) * trilog(1.0 / (1.0 - w7)) + (num19 + num20) * trilog(-w7inv) + (-num19 - num20) * trilog((-1.0 + w7) / (2.0 * w7)) + (num19 + num20) * trilog((-1.0 + w7) / w7)
                + (-num19 - num20) * trilog(-w7) + (num19 + num20) * trilog(w7) + (-num19 - num20) * trilog((1.0 - w7) / (1.0 + w7)) + 6.0 * num16 * trilog(-1.0i / wx3) + 6.0 * num16 * trilog(1.0i / wx3) + 10.0 * num16 * trilog(-1.0i / (-1.0i + wx3))
                + 2.0 * num16 * trilog(wx3 / (-1.0i + wx3)) - 4.0 * num16 * trilog((-1.0i + wx3) / (2.0 * wx3)) + 4.0 * num16 * trilog((-1.0i + wx3) / wx3) + 10.0 * num16 * trilog(1.0i / (1.0i + wx3)) - 4.0 * num16 * trilog((1.0i - wx3) / (1.0i + wx3))
                + 2.0 * num16 * trilog(wx3 / (1.0i + wx3)) - 4.0 * num16 * trilog((1.0i + wx3) / (1.0i - wx3)) - 4.0 * num16 * trilog((1.0i + wx3) / (2.0 * wx3)) + 4.0 * num16 * trilog((1.0i + wx3) / wx3) + 6.0 * num16 * trilog(-1.0i / wx4) + 6.0 * num16 * trilog(1.0i / wx4)
                + 10.0 * num16 * trilog(-1.0i / (-1.0i + wx4)) + 2.0 * num16 * trilog(wx4 / (-1.0i + wx4)) - 4.0 * num16 * trilog((-1.0i + wx4) / (2.0 * wx4)) + 4.0 * num16 * trilog((-1.0i + wx4) / wx4) + 10.0 * num16 * trilog(1.0i / (1.0i + wx4))
                - 4.0 * num16 * trilog((1.0i - wx4) / (1.0i + wx4)) + 2.0 * num16 * trilog(wx4 / (1.0i + wx4)) - 4.0 * num16 * trilog((1.0i + wx4) / (1.0i - wx4)) - 4.0 * num16 * trilog((1.0i + wx4) / (2.0 * wx4)) + 4.0 * num16 * trilog((1.0i + wx4) / wx4)
                - 5.0 * num16 * trilog((1.0i - xd) / (1.0i + wx3)) - 5.0 * num16 * trilog((1.0i - xd) / (1.0i + wx4)) - 2.0 * num16 * trilog((-1.0i + wx3) / (wx3 - xd)) - 2.0 * num16 * trilog((1.0i + wx3) / (wx3 - xd)) - num16 * trilog((wx3 - xd) / (-1.0i + wx3))
                - num16 * trilog((wx3 - xd) / (1.0i + wx3)) - 2.0 * num16 * trilog((-1.0i + wx4) / (wx4 - xd)) - 2.0 * num16 * trilog((1.0i + wx4) / (wx4 - xd)) - num16 * trilog((wx4 - xd) / (-1.0i + wx4)) - num16 * trilog((wx4 - xd) / (1.0i + wx4))
                + 8.0 * num16 * trilog(1.0 / 2.0 + (1.0i / 2.0) * xd) + 2.0 * num16 * trilog(((-1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 - xd))) + 2.0 * num16 * trilog(((-1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 - xd))) + 2.0 * num16 * trilog(((1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 - xd)))
                + 2.0 * num16 * trilog(((1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 - xd))) + num16 * trilog((1.0i * (wx3 - xd)) / ((-1.0i + wx3) * xd)) + num16 * trilog(((-1.0i) * (wx3 - xd)) / ((1.0i + wx3) * xd)) + num16 * trilog((1.0i * (wx4 - xd)) / ((-1.0i + wx4) * xd))
                + num16 * trilog(((-1.0i) * (wx4 - xd)) / ((1.0i + wx4) * xd)) + 2.0 * num16 * trilog(-(xd / wx3)) + 2.0 * num16 * trilog(xd / wx3) + 2.0 * num16 * trilog(-(xd / wx4)) + 2.0 * num16 * trilog(xd / wx4) - 4.0 * num16 * trilog(-1.0i / (-1.0i + xd))
                + num16 * trilog(((-1.0i) * (wx3 - xd)) / (wx3 * (-1.0i + xd))) + num16 * trilog(((-1.0i) * (wx4 - xd)) / (wx4 * (-1.0i + xd))) - 5.0 * num16 * trilog((-1.0i + xd) / (-1.0i + wx3)) - 5.0 * num16 * trilog((-1.0i + xd) / (-1.0i + wx4))
                + 4.0 * num16 * trilog((-1.0i + xd) * xdinv) - num16 * trilog((wx3 * (-1.0i + xd)) / ((-1.0i + wx3) * xd)) - num16 * trilog((wx3 * (-1.0i + xd)) / ((1.0i + wx3) * xd)) - num16 * trilog((wx4 * (-1.0i + xd)) / ((-1.0i + wx4) * xd))
                - num16 * trilog((wx4 * (-1.0i + xd)) / ((1.0i + wx4) * xd)) - 4.0 * num16 * trilog(1.0i / (1.0i + xd)) + num16 * trilog((1.0i * (wx3 - xd)) / (wx3 * (1.0i + xd))) + num16 * trilog((1.0i * (wx4 - xd)) / (wx4 * (1.0i + xd)))
                + 2.0 * num16 * trilog(((-1.0i + wx3) * (-1.0i + xd)) / ((1.0i + wx3) * (1.0i + xd))) + 2.0 * num16 * trilog(((1.0i + wx3) * (-1.0i + xd)) / ((-1.0i + wx3) * (1.0i + xd))) + 2.0 * num16 * trilog(((-1.0i + wx4) * (-1.0i + xd)) / ((1.0i + wx4) * (1.0i + xd)))
                + 2.0 * num16 * trilog(((1.0i + wx4) * (-1.0i + xd)) / ((-1.0i + wx4) * (1.0i + xd))) + 8.0 * num16 * trilog(((-0.5) * 1.0i) * (1.0i + xd)) - 5.0 * num16 * trilog((1.0i + xd) / (1.0i - wx3)) - 5.0 * num16 * trilog((1.0i + xd) / (1.0i + wx3))
                - 5.0 * num16 * trilog((1.0i + xd) / (1.0i - wx4)) - 5.0 * num16 * trilog((1.0i + xd) / (1.0i + wx4)) - 2.0 * num16 * trilog(-((1.0i + xd) / (wx3 - xd))) - 2.0 * num16 * trilog(-((1.0i + xd) / (wx4 - xd))) + 4.0 * num16 * trilog((1.0i + xd) * xdinv)
                - num16 * trilog((wx3 * (1.0i + xd)) / ((-1.0i + wx3) * xd)) - num16 * trilog((wx3 * (1.0i + xd)) / ((1.0i + wx3) * xd)) - num16 * trilog((wx4 * (1.0i + xd)) / ((-1.0i + wx4) * xd)) - num16 * trilog((wx4 * (1.0i + xd)) / ((1.0i + wx4) * xd))
                + 2.0 * num16 * trilog(((-1.0i + wx3) * (1.0i + xd)) / ((1.0i + wx3) * (-1.0i + xd))) + 2.0 * num16 * trilog(((1.0i + wx3) * (1.0i + xd)) / ((-1.0i + wx3) * (-1.0i + xd))) + 2.0 * num16 * trilog(((-1.0i + wx4) * (1.0i + xd)) / ((1.0i + wx4) * (-1.0i + xd)))
                + 2.0 * num16 * trilog(((1.0i + wx4) * (1.0i + xd)) / ((-1.0i + wx4) * (-1.0i + xd))) - 2.0 * num16 * trilog((-1.0i + xd) / (-wx3 + xd)) - num16 * trilog((-wx3 + xd) / (-1.0i + xd)) - num16 * trilog((-wx3 + xd) / (1.0i + xd))
                - 2.0 * num16 * trilog((-1.0i + wx3) / (wx3 + xd)) - 2.0 * num16 * trilog((1.0i + wx3) / (wx3 + xd)) + 2.0 * num16 * trilog(((1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 + xd))) + 2.0 * num16 * trilog(((-1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 + xd)))
                - 2.0 * num16 * trilog((-1.0i + xd) / (wx3 + xd)) - 2.0 * num16 * trilog((1.0i + xd) / (wx3 + xd)) - num16 * trilog((wx3 + xd) / (-1.0i + wx3)) - num16 * trilog((wx3 + xd) / (1.0i + wx3)) + num16 * trilog(((-1.0i) * (wx3 + xd)) / ((-1.0i + wx3) * xd))
                + num16 * trilog((1.0i * (wx3 + xd)) / ((1.0i + wx3) * xd)) - num16 * trilog((wx3 + xd) / (-1.0i + xd)) + num16 * trilog(((-1.0i) * (wx3 + xd)) / (wx3 * (-1.0i + xd))) - num16 * trilog((wx3 + xd) / (1.0i + xd))
                + num16 * trilog((1.0i * (wx3 + xd)) / (wx3 * (1.0i + xd))) - 2.0 * num16 * trilog((-1.0i + xd) / (-wx4 + xd)) - num16 * trilog((-wx4 + xd) / (-1.0i + xd)) - num16 * trilog((-wx4 + xd) / (1.0i + xd)) - 2.0 * num16 * trilog((-1.0i + wx4) / (wx4 + xd))
                - 2.0 * num16 * trilog((1.0i + wx4) / (wx4 + xd)) + 2.0 * num16 * trilog(((1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 + xd))) + 2.0 * num16 * trilog(((-1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 + xd))) - 2.0 * num16 * trilog((-1.0i + xd) / (wx4 + xd))
                - 2.0 * num16 * trilog((1.0i + xd) / (wx4 + xd)) - num16 * trilog((wx4 + xd) / (-1.0i + wx4)) - num16 * trilog((wx4 + xd) / (1.0i + wx4)) + num16 * trilog(((-1.0i) * (wx4 + xd)) / ((-1.0i + wx4) * xd))
                + num16 * trilog((1.0i * (wx4 + xd)) / ((1.0i + wx4) * xd)) - num16 * trilog((wx4 + xd) / (-1.0i + xd)) + num16 * trilog(((-1.0i) * (wx4 + xd)) / (wx4 * (-1.0i + xd))) - num16 * trilog((wx4 + xd) / (1.0i + xd))
                + num16 * trilog((1.0i * (wx4 + xd)) / (wx4 * (1.0i + xd))) + (2.0 * num17 - 2.0 * num18 + num19 - num20) * trilog(1.0 + ydinv) + (-2.0 * num17 + num20) * trilog((1.0 - yd) / 2.0) + num17 * trilog((1.0 - yd) / (1.0 + w5))
                + num17 * trilog((-1.0 + w4) / (w4 - yd)) + num18 * trilog((1.0 + w5) / (w5 - yd)) - num20 * trilog((-1.0 + w7) / (w7 - yd)) + num17 * trilog((-1.0 + yd) / (-1.0 + w4)) - num20 * trilog((-1.0 + yd) / (-1.0 + w7))
                - num18 * trilog((-0.5) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) + (-2.0 * num17 + 2.0 * num18 - num19 + num20) * trilog((-1.0 + yd) / yd) + (2.0 * num17 - 2.0 * num18 + num19 - num20) * trilog(-yd)
                + (-2.0 * num17 + 2.0 * num18 - num19 + num20) * trilog(yd) - num18 * trilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) - num18 * trilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd)))
                + num19 * trilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) + (-2.0 * num18 + num19) * trilog((1.0 + yd) / 2.0) + num18 * trilog((1.0 + yd) / (1.0 - w4)) + num18 * trilog((1.0 + yd) / (1.0 + w5)) - num19 * trilog((1.0 + yd) / (1.0 - w7))
                - num17 * trilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) + num17 * trilog((w4 * (1.0 + yd)) / (w4 - yd)) + num18 * trilog(-((1.0 + yd) / (w5 - yd))) - num18 * trilog((w5 * (1.0 + yd)) / (w5 - yd))
                + num20 * trilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) - num20 * trilog((w7 * (1.0 + yd)) / (w7 - yd)) - num17 * trilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd)))
                - num17 * trilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) + num20 * trilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd))) + num17 * trilog((-1.0 + yd) / (-w4 + yd)) + num18 * trilog((-1.0 + w4) / (w4 + yd))
                - num18 * trilog((-0.5) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) + num18 * trilog((1.0 + yd) / (w4 + yd)) - num18 * trilog((w4 * (1.0 + yd)) / (w4 + yd)) + num17 * trilog((1.0 + w5) / (w5 + yd))
                + num17 * trilog((-1.0 + yd) / (w5 + yd)) + num17 * trilog((w5 * (1.0 + yd)) / (w5 + yd)) - num17 * trilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) - num20 * trilog((-1.0 + yd) / (-w7 + yd)) - num19 * trilog((-1.0 + w7) / (w7 + yd))
                + num19 * trilog((-0.5) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) - num19 * trilog((1.0 + yd) / (w7 + yd)) + num19 * trilog((w7 * (1.0 + yd)) / (w7 + yd)) - num17 * trilog((w4 - w4 * yd) / (w4 - yd))
                + num18 * trilog((w4 - w4 * yd) / (w4 + yd)) + num18 * trilog((w5 - w5 * yd) / (w5 - yd)) - num17 * trilog((w5 - w5 * yd) / (w5 + yd)) + num20 * trilog((w7 - w7 * yd) / (w7 - yd)) - num19 * trilog((w7 - w7 * yd) / (w7 + yd))
                + 4.0 * num16 * dilog((-1.0i + wx3) / (2.0 * wx3)) * lnhalf + 4.0 * num16 * dilog((1.0i - wx3) / (1.0i + wx3)) * lnhalf + 4.0 * num16 * dilog((1.0i + wx3) / (1.0i - wx3)) * lnhalf + 4.0 * num16 * dilog((1.0i + wx3) / (2.0 * wx3)) * lnhalf
                + 4.0 * num16 * dilog((-1.0i + wx4) / (2.0 * wx4)) * lnhalf + 4.0 * num16 * dilog((1.0i - wx4) / (1.0i + wx4)) * lnhalf + 4.0 * num16 * dilog((1.0i + wx4) / (1.0i - wx4)) * lnhalf + 4.0 * num16 * dilog((1.0i + wx4) / (2.0 * wx4)) * lnhalf
                + dilog((-1.0 + w4) / (2.0 * w4)) * (-(num17 * lnhalf) - num18 * lnhalf) + dilog((1.0 - w4) / (1.0 + w4)) * (-(num17 * lnhalf) - num18 * lnhalf)
                + dilog((1.0 + w5) / (1.0 - w5)) * (-(num17 * lnhalf) - num18 * lnhalf) + dilog((1.0 + w5) / (2.0 * w5)) * (-(num17 * lnhalf) - num18 * lnhalf) + dilog(-w4inv) * (num17 * lnhalf + num18 * lnhalf)
                + dilog(w5inv) * (num17 * lnhalf + num18 * lnhalf) + dilog(-w7inv) * (-(num19 * lnhalf) - num20 * lnhalf) + dilog((-1.0 + w7) / (2.0 * w7)) * (num19 * lnhalf + num20 * lnhalf)
                + dilog((1.0 - w7) / (1.0 + w7)) * (num19 * lnhalf + num20 * lnhalf) + (num17 * power_of<2>(lnhalf) * log(1.0 + w4inv)) / 2.0 + (num18 * power_of<2>(lnhalf) * log(1.0 + w4inv)) / 2.0 - (num17 * pisqu * log(2.0 / w4)) / 6.0
                - (num18 * pisqu * log(2.0 / w4)) / 6.0 - (num17 * power_of<3>(log(2.0 / w4))) / 6.0 - (num18 * power_of<3>(log(2.0 / w4))) / 6.0 + (num17 * pisqu * log((2.0 * (-1.0 + w4)) / (1.0 + w4))) / 6.0 + (num18 * pisqu * log((2.0 * (-1.0 + w4)) / (1.0 + w4))) / 6.0
                + (num17 * power_of<3>(log((2.0 * (-1.0 + w4)) / (1.0 + w4)))) / 6.0 + (num18 * power_of<3>(log((2.0 * (-1.0 + w4)) / (1.0 + w4)))) / 6.0 + (num17 * power_of<2>(lnhalf) * log(w4 / (1.0 + w4))) / 2.0 + (num18 * power_of<2>(lnhalf) * log(w4 / (1.0 + w4))) / 2.0
                - (num17 * power_of<2>(lnhalf) * log((2.0 * w4) / (1.0 + w4))) / 2.0 - (num18 * power_of<2>(lnhalf) * log((2.0 * w4) / (1.0 + w4))) / 2.0 - (num17 * power_of<2>(lnhalf) * log((1.0 + w4) / (2.0 * w4))) / 2.0 - (num18 * power_of<2>(lnhalf) * log((1.0 + w4) / (2.0 * w4))) / 2.0
                - (num17 * pisqu * log(-2.0 / w5)) / 6.0 - (num18 * pisqu * log(-2.0 / w5)) / 6.0 - (num17 * power_of<3>(log(-2.0 / w5))) / 6.0 - (num18 * power_of<3>(log(-2.0 / w5))) / 6.0 - (num17 * power_of<2>(lnhalf) * log((-1.0 + w5) / (2.0 * w5))) / 2.0
                - (num18 * power_of<2>(lnhalf) * log((-1.0 + w5) / (2.0 * w5))) / 2.0 + (num17 * power_of<2>(lnhalf) * log((-1.0 + w5) / w5)) / 2.0 + (num18 * power_of<2>(lnhalf) * log((-1.0 + w5) / w5)) / 2.0 + (num17 * power_of<2>(lnhalf) * log(w5 / (-1.0 + w5))) / 2.0
                + (num18 * power_of<2>(lnhalf) * log(w5 / (-1.0 + w5))) / 2.0 - (num17 * power_of<2>(lnhalf) * log((2.0 * w5) / (-1.0 + w5))) / 2.0 - (num18 * power_of<2>(lnhalf) * log((2.0 * w5) / (-1.0 + w5))) / 2.0 + (num17 * pisqu * log((2.0 * (1.0 + w5)) / (-1.0 + w5))) / 6.0
                + (num18 * pisqu * log((2.0 * (1.0 + w5)) / (-1.0 + w5))) / 6.0 + (num17 * power_of<3>(log((2.0 * (1.0 + w5)) / (-1.0 + w5)))) / 6.0 + (num18 * power_of<3>(log((2.0 * (1.0 + w5)) / (-1.0 + w5)))) / 6.0 - (num19 * power_of<2>(lnhalf) * log(1.0 + w7inv)) / 2.0
                - (num20 * power_of<2>(lnhalf) * log(1.0 + w7inv)) / 2.0 + (num19 * pisqu * log(2.0 / w7)) / 6.0 + (num20 * pisqu * log(2.0 / w7)) / 6.0 + (num19 * power_of<3>(log(2.0 / w7))) / 6.0 + (num20 * power_of<3>(log(2.0 / w7))) / 6.0 - (num19 * pisqu * log((2.0 * (-1.0 + w7)) / (1.0 + w7))) / 6.0
                - (num20 * pisqu * log((2.0 * (-1.0 + w7)) / (1.0 + w7))) / 6.0 - (num19 * power_of<3>(log((2.0 * (-1.0 + w7)) / (1.0 + w7)))) / 6.0 - (num20 * power_of<3>(log((2.0 * (-1.0 + w7)) / (1.0 + w7)))) / 6.0 - (num19 * power_of<2>(lnhalf) * log(w7 / (1.0 + w7))) / 2.0
                - (num20 * power_of<2>(lnhalf) * log(w7 / (1.0 + w7))) / 2.0 + (num19 * power_of<2>(lnhalf) * log((2.0 * w7) / (1.0 + w7))) / 2.0 + (num20 * power_of<2>(lnhalf) * log((2.0 * w7) / (1.0 + w7))) / 2.0 + (num19 * power_of<2>(lnhalf) * log((1.0 + w7) / (2.0 * w7))) / 2.0
                + (num20 * power_of<2>(lnhalf) * log((1.0 + w7) / (2.0 * w7))) / 2.0 + (2.0 * num16 * pisqu * log((-2.0 * 1.0i) / wx3)) / 3.0 + (2.0 * num16 * power_of<3>(log((-2.0 * 1.0i) / wx3))) / 3.0 + (2.0 * num16 * pisqu * log((2.0 * 1.0i) / wx3)) / 3.0 + (2.0 * num16 * power_of<3>(log((2.0 * 1.0i) / wx3))) / 3.0
                - 2.0 * num16 * power_of<2>(lnhalf) * log(wx3 / (-1.0i + wx3)) + 2.0 * num16 * power_of<2>(lnhalf) * log((2.0 * wx3) / (-1.0i + wx3)) + 2.0 * num16 * power_of<2>(lnhalf) * log((-1.0i + wx3) / (2.0 * wx3)) - 2.0 * num16 * power_of<2>(lnhalf) * log((-1.0i + wx3) / wx3)
                - 2.0 * num16 * power_of<2>(lnhalf) * log(wx3 / (1.0i + wx3)) + 2.0 * num16 * power_of<2>(lnhalf) * log((2.0 * wx3) / (1.0i + wx3)) - (2.0 * num16 * pisqu * log((2.0 * (-1.0i + wx3)) / (1.0i + wx3))) / 3.0 - (2.0 * num16 * power_of<3>(log((2.0 * (-1.0i + wx3)) / (1.0i + wx3)))) / 3.0
                + 2.0 * num16 * power_of<2>(lnhalf) * log((1.0i + wx3) / (2.0 * wx3)) - 2.0 * num16 * power_of<2>(lnhalf) * log((1.0i + wx3) / wx3) - (2.0 * num16 * pisqu * log((2.0 * (1.0i + wx3)) / (-1.0i + wx3))) / 3.0 - (2.0 * num16 * power_of<3>(log((2.0 * (1.0i + wx3)) / (-1.0i + wx3)))) / 3.0
                + (2.0 * num16 * pisqu * log((-2.0 * 1.0i) / wx4)) / 3.0 + (2.0 * num16 * power_of<3>(log((-2.0 * 1.0i) / wx4))) / 3.0 + (2.0 * num16 * pisqu * log((2.0 * 1.0i) / wx4)) / 3.0 + (2.0 * num16 * power_of<3>(log((2.0 * 1.0i) / wx4))) / 3.0 - 2.0 * num16 * power_of<2>(lnhalf) * log(wx4 / (-1.0i + wx4))
                + 2.0 * num16 * power_of<2>(lnhalf) * log((2.0 * wx4) / (-1.0i + wx4)) + 2.0 * num16 * power_of<2>(lnhalf) * log((-1.0i + wx4) / (2.0 * wx4)) - 2.0 * num16 * power_of<2>(lnhalf) * log((-1.0i + wx4) / wx4) - 2.0 * num16 * power_of<2>(lnhalf) * log(wx4 / (1.0i + wx4))
                + 2.0 * num16 * power_of<2>(lnhalf) * log((2.0 * wx4) / (1.0i + wx4)) - (2.0 * num16 * pisqu * log((2.0 * (-1.0i + wx4)) / (1.0i + wx4))) / 3.0 - (2.0 * num16 * power_of<3>(log((2.0 * (-1.0i + wx4)) / (1.0i + wx4)))) / 3.0 + 2.0 * num16 * power_of<2>(lnhalf) * log((1.0i + wx4) / (2.0 * wx4))
                - 2.0 * num16 * power_of<2>(lnhalf) * log((1.0i + wx4) / wx4) - (2.0 * num16 * pisqu * log((2.0 * (1.0i + wx4)) / (-1.0i + wx4))) / 3.0 - (2.0 * num16 * power_of<3>(log((2.0 * (1.0i + wx4)) / (-1.0i + wx4)))) / 3.0 - (num16 * pisqu * log(wx3 / (1.0i - xd))) / 6.0
                - (num16 * power_of<3>(log(wx3 / (1.0i - xd)))) / 6.0 - (num16 * pisqu * log(wx4 / (1.0i - xd))) / 6.0 - (num16 * power_of<3>(log(wx4 / (1.0i - xd)))) / 6.0 - (num16 * pisqu * log((-2.0 * 1.0i) / (wx3 - xd))) / 3.0 - (num16 * power_of<3>(log((-2.0 * 1.0i) / (wx3 - xd)))) / 3.0
                - (num16 * pisqu * log((2.0 * 1.0i) / (wx3 - xd))) / 3.0 - (num16 * power_of<3>(log((2.0 * 1.0i) / (wx3 - xd)))) / 3.0 - (num16 * pisqu * log((-2.0 * 1.0i) / (wx4 - xd))) / 3.0 - (num16 * power_of<3>(log((-2.0 * 1.0i) / (wx4 - xd)))) / 3.0 - (num16 * pisqu * log((2.0 * 1.0i) / (wx4 - xd))) / 3.0
                - (num16 * power_of<3>(log((2.0 * 1.0i) / (wx4 - xd)))) / 3.0 - 2.0 * num16 * dilog(((-1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 - xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((-1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 - xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd)
                + 2.0 * num16 * dilog(-((1.0i + xd) / (wx3 - xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd) + 2.0 * num16 * dilog(-((1.0i + xd) / (wx4 - xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((-1.0i + wx3) * (1.0i + xd)) / ((1.0i + wx3) * (-1.0i + xd))) * 
                log(1.0 / 2.0 - (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((1.0i + wx3) * (1.0i + xd)) / ((-1.0i + wx3) * (-1.0i + xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((-1.0i + wx4) * (1.0i + xd)) / ((1.0i + wx4) * (-1.0i + xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd)
                - 2.0 * num16 * dilog(((1.0i + wx4) * (1.0i + xd)) / ((-1.0i + wx4) * (-1.0i + xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 + xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd)
                + 2.0 * num16 * dilog((1.0i + xd) / (wx3 + xd)) * log(1.0 / 2.0 - (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 + xd))) * log(1.0 / 2.0 - (1.0i / 2.0) * xd) + 2.0 * num16 * dilog((1.0i + xd) / (wx4 + xd)) * log(1.0 / 2.0 - (1.0i / 2.0) * xd)
                + num16 * log((1.0i + wx3) / (wx3 - xd)) * power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * xd)) - 2.0 * num16 * dilog(((1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 - xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd)
                - 2.0 * num16 * dilog(((1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 - xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((-1.0i + wx3) * (-1.0i + xd)) / ((1.0i + wx3) * (1.0i + xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd)
                - 2.0 * num16 * dilog(((1.0i + wx3) * (-1.0i + xd)) / ((-1.0i + wx3) * (1.0i + xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd) - 2.0 * num16 * dilog(((-1.0i + wx4) * (-1.0i + xd)) / ((1.0i + wx4) * (1.0i + xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd)
                - 2.0 * num16 * dilog(((1.0i + wx4) * (-1.0i + xd)) / ((-1.0i + wx4) * (1.0i + xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + 2.0 * num16 * dilog((-1.0i + xd) / (-wx3 + xd)) * log(1.0 / 2.0 + (1.0i / 2.0) * xd)
                - 2.0 * num16 * dilog(((-1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 + xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + 2.0 * num16 * dilog((-1.0i + xd) / (wx3 + xd)) * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + 2.0 * num16 * dilog((-1.0i + xd) / (-wx4 + xd)) * log(1.0 / 2.0 + (1.0i / 2.0) * xd)
                - 2.0 * num16 * dilog(((-1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 + xd))) * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + 2.0 * num16 * dilog((-1.0i + xd) / (wx4 + xd)) * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + num16 * log((-1.0i + wx3) / (wx3 - xd)) * power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * xd))
                - 4.0 * num16 * dilog((1.0i + xd) * xdinv) * log(1.0 - 1.0i * xd) + num16 * dilog((wx3 * (1.0i + xd)) / ((-1.0i + wx3) * xd)) * log(1.0 - 1.0i * xd) + num16 * dilog((wx3 * (1.0i + xd)) / ((1.0i + wx3) * xd)) * log(1.0 - 1.0i * xd)
                + num16 * dilog((wx4 * (1.0i + xd)) / ((-1.0i + wx4) * xd)) * log(1.0 - 1.0i * xd) + num16 * dilog((wx4 * (1.0i + xd)) / ((1.0i + wx4) * xd)) * log(1.0 - 1.0i * xd) - 4.0 * num16 * dilog((-1.0i + xd) * xdinv) * log(1.0 + 1.0i * xd)
                + num16 * dilog((wx3 * (-1.0i + xd)) / ((-1.0i + wx3) * xd)) * log(1.0 + 1.0i * xd) + num16 * dilog((wx3 * (-1.0i + xd)) / ((1.0i + wx3) * xd)) * log(1.0 + 1.0i * xd) + num16 * dilog((wx4 * (-1.0i + xd)) / ((-1.0i + wx4) * xd)) * log(1.0 + 1.0i * xd)
                + num16 * dilog((wx4 * (-1.0i + xd)) / ((1.0i + wx4) * xd)) * log(1.0 + 1.0i * xd) + dilog((-2.0 * 1.0i) / (-1.0i + wx3)) * (-2.0 * num16 * log(1.0 - 1.0i * xd) - 2.0 * num16 * log(1.0 + 1.0i * xd))
                + dilog((2.0 * 1.0i) / (1.0i + wx3)) * (-2.0 * num16 * log(1.0 - 1.0i * xd) - 2.0 * num16 * log(1.0 + 1.0i * xd)) + dilog((-2.0 * 1.0i) / (-1.0i + wx4)) * (-2.0 * num16 * log(1.0 - 1.0i * xd) - 2.0 * num16 * log(1.0 + 1.0i * xd))
                + dilog((2.0 * 1.0i) / (1.0i + wx4)) * (-2.0 * num16 * log(1.0 - 1.0i * xd) - 2.0 * num16 * log(1.0 + 1.0i * xd)) + dilog(-1.0i / wx3) * (-4.0 * num16 * lnhalf + num16 * log(1.0 - 1.0i * xd) + num16 * log(1.0 + 1.0i * xd))
                + dilog(1.0i / wx3) * (-4.0 * num16 * lnhalf + num16 * log(1.0 - 1.0i * xd) + num16 * log(1.0 + 1.0i * xd)) + dilog(-1.0i / wx4) * (-4.0 * num16 * lnhalf + num16 * log(1.0 - 1.0i * xd) + num16 * log(1.0 + 1.0i * xd))
                + dilog(1.0i / wx4) * (-4.0 * num16 * lnhalf + num16 * log(1.0 - 1.0i * xd) + num16 * log(1.0 + 1.0i * xd)) + log((wx3 - xd) / (1.0i + wx3)) * ((-0.5) * (num16 * power_of<2>(log(1.0 - 1.0i * xd))) - num16 * log(1.0 - 1.0i * xd) * log(1.0 + 1.0i * xd))
                + log((wx4 - xd) / (1.0i + wx4)) * ((-0.5) * (num16 * power_of<2>(log(1.0 - 1.0i * xd))) - num16 * log(1.0 - 1.0i * xd) * log(1.0 + 1.0i * xd))
                + log((wx3 - xd) / (-1.0i + wx3)) * (-(num16 * log(1.0 - 1.0i * xd) * log(1.0 + 1.0i * xd)) - (num16 * power_of<2>(log(1.0 + 1.0i * xd))) / 2.0) + log((wx4 - xd) / (-1.0i + wx4)) * (-(num16 * log(1.0 - 1.0i * xd) * log(1.0 + 1.0i * xd)) - (num16 * power_of<2>(log(1.0 + 1.0i * xd))) / 2.0)
                + (2.0 * num16 * pisqu * log(-1.0i / xd)) / 3.0 + (2.0 * num16 * power_of<3>(log(-1.0i / xd))) / 3.0 + (2.0 * num16 * pisqu * log(1.0i / xd)) / 3.0 + (2.0 * num16 * power_of<3>(log(1.0i / xd))) / 3.0 + dilog(wx3 / (-1.0i + wx3)) * (-4.0 * num16 * ln2 - 2.0 * num16 * log(xd))
                + dilog(wx3 / (1.0i + wx3)) * (-4.0 * num16 * ln2 - 2.0 * num16 * log(xd)) + dilog(wx4 / (-1.0i + wx4)) * (-4.0 * num16 * ln2 - 2.0 * num16 * log(xd)) + dilog(wx4 / (1.0i + wx4)) * (-4.0 * num16 * ln2 - 2.0 * num16 * log(xd))
                + dilog((1.0 - yd) / (1.0 + w5)) * (-2.0 * num17 * ln2 + num17 * log(1.0 - 1.0i * xd) + num17 * log(1.0 + 1.0i * xd) - num17 * log(xd))
                + dilog((-1.0 + yd) / (-1.0 + w4)) * (-2.0 * num17 * ln2 + num17 * log(1.0 - 1.0i * xd) + num17 * log(1.0 + 1.0i * xd) - num17 * log(xd))
                + dilog(1.0 / (1.0 + w4)) * (num17 * lnhalf + num18 * lnhalf - 2.0 * num17 * ln2 - 2.0 * num18 * ln2 + (num17 + num18) * log(1.0 - 1.0i * xd) + (num17 + num18) * log(1.0 + 1.0i * xd) + (-num17 - num18) * log(xd))
                + dilog(1.0 / (1.0 - w5)) * (num17 * lnhalf + num18 * lnhalf - 2.0 * num17 * ln2 - 2.0 * num18 * ln2 + (num17 + num18) * log(1.0 - 1.0i * xd) + (num17 + num18) * log(1.0 + 1.0i * xd) + (-num17 - num18) * log(xd))
                + dilog((1.0 + yd) / (1.0 - w4)) * (-2.0 * num18 * ln2 + num18 * log(1.0 - 1.0i * xd) + num18 * log(1.0 + 1.0i * xd) - num18 * log(xd))
                + dilog((1.0 + yd) / (1.0 + w5)) * (-2.0 * num18 * ln2 + num18 * log(1.0 - 1.0i * xd) + num18 * log(1.0 + 1.0i * xd) - num18 * log(xd))
                + dilog(1.0 / (1.0 - w4)) * (2.0 * num17 * ln2 + 2.0 * num18 * ln2 + (-num17 - num18) * log(1.0 - 1.0i * xd) + (-num17 - num18) * log(1.0 + 1.0i * xd) + (num17 + num18) * log(xd))
                + dilog(1.0 / (1.0 + w5)) * (2.0 * num17 * ln2 + 2.0 * num18 * ln2 + (-num17 - num18) * log(1.0 - 1.0i * xd) + (-num17 - num18) * log(1.0 + 1.0i * xd) + (num17 + num18) * log(xd))
                + dilog((1.0 + yd) / (1.0 - w7)) * (4.0 * num19 * ln2 - 2.0 * num19 * log(1.0 - 1.0i * xd) - 2.0 * num19 * log(1.0 + 1.0i * xd) + 2.0 * num19 * log(xd))
                + dilog(1.0 / (1.0 - w7)) * (-4.0 * num19 * ln2 - 4.0 * num20 * ln2 + (2.0 * num19 + 2.0 * num20) * log(1.0 - 1.0i * xd) + (2.0 * num19 + 2.0 * num20) * log(1.0 + 1.0i * xd) + (-2.0 * num19 - 2.0 * num20) * log(xd))
                + dilog((-1.0 + yd) / (-1.0 + w7)) * (4.0 * num20 * ln2 - 2.0 * num20 * log(1.0 - 1.0i * xd) - 2.0 * num20 * log(1.0 + 1.0i * xd) + 2.0 * num20 * log(xd))
                + dilog(1.0 / (1.0 + w7)) * (-(num19 * lnhalf) - num20 * lnhalf + 4.0 * num19 * ln2 + 4.0 * num20 * ln2 + (-2.0 * num19 - 2.0 * num20) * log(1.0 - 1.0i * xd) + (-2.0 * num19 - 2.0 * num20) * log(1.0 + 1.0i * xd)
                + (2.0 * num19 + 2.0 * num20) * log(xd)) - (num16 * pisqu * log(wx3 / (-1.0i + xd))) / 6.0 - (num16 * power_of<3>(log(wx3 / (-1.0i + xd)))) / 6.0 + (num16 * pisqu * log((2.0 - (2.0 * 1.0i) * wx3) / ((-1.0i + wx3) * (-1.0i + xd)))) / 3.0
                + (num16 * power_of<3>(log((2.0 - (2.0 * 1.0i) * wx3) / ((-1.0i + wx3) * (-1.0i + xd))))) / 3.0 + (num16 * pisqu * log((-2.0 - (2.0 * 1.0i) * wx3) / ((1.0i + wx3) * (-1.0i + xd)))) / 3.0 + (num16 * power_of<3>(log((-2.0 - (2.0 * 1.0i) * wx3) / ((1.0i + wx3) * (-1.0i + xd))))) / 3.0
                - (num16 * pisqu * log(wx4 / (-1.0i + xd))) / 6.0 - (num16 * power_of<3>(log(wx4 / (-1.0i + xd)))) / 6.0 + (num16 * pisqu * log((2.0 - (2.0 * 1.0i) * wx4) / ((-1.0i + wx4) * (-1.0i + xd)))) / 3.0 + (num16 * power_of<3>(log((2.0 - (2.0 * 1.0i) * wx4) / ((-1.0i + wx4) * (-1.0i + xd))))) / 3.0
                + (num16 * pisqu * log((-2.0 - (2.0 * 1.0i) * wx4) / ((1.0i + wx4) * (-1.0i + xd)))) / 3.0 + (num16 * power_of<3>(log((-2.0 - (2.0 * 1.0i) * wx4) / ((1.0i + wx4) * (-1.0i + xd))))) / 3.0 - (num16 * pisqu * log(-(wx3 / (1.0i + xd)))) / 6.0 - (num16 * power_of<3>(log(-(wx3 / (1.0i + xd))))) / 6.0
                - (num16 * pisqu * log(wx3 / (1.0i + xd))) / 6.0 - (num16 * power_of<3>(log(wx3 / (1.0i + xd)))) / 6.0 + (num16 * pisqu * log((2.0 + (2.0 * 1.0i) * wx3) / ((1.0i + wx3) * (1.0i + xd)))) / 3.0 + (num16 * power_of<3>(log((2.0 + (2.0 * 1.0i) * wx3) / ((1.0i + wx3) * (1.0i + xd))))) / 3.0
                + (num16 * pisqu * log(((2.0 * 1.0i) * (1.0i + wx3)) / ((-1.0i + wx3) * (1.0i + xd)))) / 3.0 + (num16 * power_of<3>(log(((2.0 * 1.0i) * (1.0i + wx3)) / ((-1.0i + wx3) * (1.0i + xd))))) / 3.0 - (num16 * pisqu * log(-(wx4 / (1.0i + xd)))) / 6.0 - (num16 * power_of<3>(log(-(wx4 / (1.0i + xd))))) / 6.0
                - (num16 * pisqu * log(wx4 / (1.0i + xd))) / 6.0 - (num16 * power_of<3>(log(wx4 / (1.0i + xd)))) / 6.0 + (num16 * pisqu * log((2.0 + (2.0 * 1.0i) * wx4) / ((1.0i + wx4) * (1.0i + xd)))) / 3.0 + (num16 * power_of<3>(log((2.0 + (2.0 * 1.0i) * wx4) / ((1.0i + wx4) * (1.0i + xd))))) / 3.0
                + (num16 * pisqu * log(((2.0 * 1.0i) * (1.0i + wx4)) / ((-1.0i + wx4) * (1.0i + xd)))) / 3.0 + (num16 * power_of<3>(log(((2.0 * 1.0i) * (1.0i + wx4)) / ((-1.0i + wx4) * (1.0i + xd))))) / 3.0 - (num16 * pisqu * log((-2.0 * 1.0i) / (wx3 + xd))) / 3.0 - (num16 * power_of<3>(log((-2.0 * 1.0i) / (wx3 + xd)))) / 3.0
                - (num16 * pisqu * log((2.0 * 1.0i) / (wx3 + xd))) / 3.0 - (num16 * power_of<3>(log((2.0 * 1.0i) / (wx3 + xd)))) / 3.0 - num16 * dilog(((-1.0i) * (wx3 + xd)) / ((-1.0i + wx3) * xd)) * log((wx3 + xd) / wx3)
                - num16 * dilog((1.0i * (wx3 + xd)) / ((1.0i + wx3) * xd)) * log((wx3 + xd) / wx3) + num16 * dilog((wx3 + xd) / (-1.0i + xd)) * log((wx3 + xd) / wx3) - num16 * dilog(((-1.0i) * (wx3 + xd)) / (wx3 * (-1.0i + xd))) * log((wx3 + xd) / wx3)
                + num16 * dilog((wx3 + xd) / (1.0i + xd)) * log((wx3 + xd) / wx3) - num16 * dilog((1.0i * (wx3 + xd)) / (wx3 * (1.0i + xd))) * log((wx3 + xd) / wx3)
                + (2.0 * num16 * ln2 * log((1.0i + xd) / (1.0i - wx3)) + num16 * log(xd) * log((1.0i + xd) / (1.0i - wx3))) * log((wx3 + xd) / wx3)
                + ((num16 * log((1.0i + wx3) / (1.0i - xd))) / 2.0 - (num16 * log(((1.0i + wx3) * xd) / (wx3 * (-1.0i + xd)))) / 2.0 - (num16 * log((wx3 * (-1.0i + xd)) / ((1.0i + wx3) * xd))) / 2.0 + (num16 * log((1.0i - wx3) / (1.0i + xd))) / 2.0
                - (num16 * log(((-1.0i + wx3) * xd) / (wx3 * (1.0i + xd)))) / 2.0 + (num16 * log((1.0i + xd) / (1.0i - wx3))) / 2.0 - (num16 * log((wx3 * (1.0i + xd)) / ((-1.0i + wx3) * xd))) / 2.0) * power_of<2>(log((wx3 + xd) / wx3))
                + dilog((wx3 + xd) / (-1.0i + wx3)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log((wx3 + xd) / wx3)) + dilog((wx3 + xd) / (1.0i + wx3)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log((wx3 + xd) / wx3))
                + log((1.0i - xd) / (1.0i + wx3)) * ((2.0 * num16 * ln2 + num16 * log(xd)) * log((wx3 + xd) / wx3) + (num16 * power_of<2>(log((wx3 + xd) / wx3))) / 2.0) - (num16 * pisqu * log((-2.0 * 1.0i) / (wx4 + xd))) / 3.0 - (num16 * power_of<3>(log((-2.0 * 1.0i) / (wx4 + xd)))) / 3.0
                - (num16 * pisqu * log((2.0 * 1.0i) / (wx4 + xd))) / 3.0 - (num16 * power_of<3>(log((2.0 * 1.0i) / (wx4 + xd)))) / 3.0 - num16 * dilog(((-1.0i) * (wx4 + xd)) / ((-1.0i + wx4) * xd)) * log((wx4 + xd) / wx4)
                - num16 * dilog((1.0i * (wx4 + xd)) / ((1.0i + wx4) * xd)) * log((wx4 + xd) / wx4) + num16 * dilog((wx4 + xd) / (-1.0i + xd)) * log((wx4 + xd) / wx4) - num16 * dilog(((-1.0i) * (wx4 + xd)) / (wx4 * (-1.0i + xd))) * log((wx4 + xd) / wx4)
                + num16 * dilog((wx4 + xd) / (1.0i + xd)) * log((wx4 + xd) / wx4) - num16 * dilog((1.0i * (wx4 + xd)) / (wx4 * (1.0i + xd))) * log((wx4 + xd) / wx4) + 2.0 * num16 * ln2 * log((1.0i + xd) / (1.0i - wx4)) * log((wx4 + xd) / wx4)
                + ((num16 * log((1.0i + wx4) / (1.0i - xd))) / 2.0 - (num16 * log(((1.0i + wx4) * xd) / (wx4 * (-1.0i + xd)))) / 2.0 - (num16 * log((wx4 * (-1.0i + xd)) / ((1.0i + wx4) * xd))) / 2.0 + (num16 * log((1.0i - wx4) / (1.0i + xd))) / 2.0
                - (num16 * log(((-1.0i + wx4) * xd) / (wx4 * (1.0i + xd)))) / 2.0 + (num16 * log((1.0i + xd) / (1.0i - wx4))) / 2.0 - (num16 * log((wx4 * (1.0i + xd)) / ((-1.0i + wx4) * xd))) / 2.0) * power_of<2>(log((wx4 + xd) / wx4))
                + dilog((wx4 + xd) / (-1.0i + wx4)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log((wx4 + xd) / wx4)) + dilog((wx4 + xd) / (1.0i + wx4)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log((wx4 + xd) / wx4))
                + log((1.0i - xd) / (1.0i + wx4)) * (2.0 * num16 * ln2 * log((wx4 + xd) / wx4) + (num16 * power_of<2>(log((wx4 + xd) / wx4))) / 2.0) + log(1.0 / 2.0 - (1.0i / 2.0) * xd) * (2.0 * num16 * log((wx3 - xd) / (1.0i + wx3)) * log(1.0 + 1.0i * xd)
                + 2.0 * num16 * log((wx4 - xd) / (1.0i + wx4)) * log(1.0 + 1.0i * xd) + log(1.0 + 1.0i * xd) * (2.0 * num16 * log((wx3 + xd) / (-1.0i + wx3)) + 2.0 * num16 * log((wx4 + xd) / (-1.0i + wx4))))
                + log(1.0 / 2.0 + (1.0i / 2.0) * xd) * (2.0 * num16 * log((wx3 - xd) / (-1.0i + wx3)) * log(1.0 - 1.0i * xd) + 2.0 * num16 * log((wx4 - xd) / (-1.0i + wx4)) * log(1.0 - 1.0i * xd)
                + log(1.0 - 1.0i * xd) * (2.0 * num16 * log((wx3 + xd) / (1.0i + wx3)) + 2.0 * num16 * log((wx4 + xd) / (1.0i + wx4)))) + power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * xd)) * (num16 * log((wx3 - xd) / (1.0i + wx3)) + num16 * log((1.0i + wx4) / (wx4 - xd))
                + num16 * log((wx4 - xd) / (1.0i + wx4)) - num16 * log(((1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 - xd))) - num16 * log(((1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 - xd))) - num16 * log(((-2.0 * 1.0i) * (wx3 - xd)) / ((1.0i + wx3) * (-1.0i + xd)))
                - num16 * log(((-2.0 * 1.0i) * (wx4 - xd)) / ((1.0i + wx4) * (-1.0i + xd))) + num16 * log((-1.0i + wx3) / (wx3 + xd)) - num16 * log(((-1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 + xd))) + num16 * log((wx3 + xd) / (-1.0i + wx3))
                - num16 * log(((-2.0 * 1.0i) * (wx3 + xd)) / ((-1.0i + wx3) * (-1.0i + xd))) + num16 * log((-1.0i + wx4) / (wx4 + xd)) - num16 * log(((-1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 + xd))) + num16 * log((wx4 + xd) / (-1.0i + wx4))
                - num16 * log(((-2.0 * 1.0i) * (wx4 + xd)) / ((-1.0i + wx4) * (-1.0i + xd)))) + power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * xd)) * (num16 * log((wx3 - xd) / (-1.0i + wx3)) + num16 * log((-1.0i + wx4) / (wx4 - xd)) + num16 * log((wx4 - xd) / (-1.0i + wx4))
                - num16 * log(((-1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 - xd))) - num16 * log(((-1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 - xd))) - num16 * log(((2.0 * 1.0i) * (wx3 - xd)) / ((-1.0i + wx3) * (1.0i + xd)))
                - num16 * log(((2.0 * 1.0i) * (wx4 - xd)) / ((-1.0i + wx4) * (1.0i + xd))) + num16 * log((1.0i + wx3) / (wx3 + xd)) - num16 * log(((1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 + xd))) + num16 * log((wx3 + xd) / (1.0i + wx3))
                - num16 * log(((2.0 * 1.0i) * (wx3 + xd)) / ((1.0i + wx3) * (1.0i + xd))) + num16 * log((1.0i + wx4) / (wx4 + xd)) - num16 * log(((1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 + xd))) + num16 * log((wx4 + xd) / (1.0i + wx4))
                - num16 * log(((2.0 * 1.0i) * (wx4 + xd)) / ((1.0i + wx4) * (1.0i + xd)))) - num16 * dilog((1.0i * (wx3 - xd)) / ((-1.0i + wx3) * xd)) * log(1.0 - xd / wx3) - num16 * dilog(((-1.0i) * (wx3 - xd)) / ((1.0i + wx3) * xd)) * log(1.0 - xd / wx3)
                - num16 * dilog(((-1.0i) * (wx3 - xd)) / (wx3 * (-1.0i + xd))) * log(1.0 - xd / wx3) - num16 * dilog((1.0i * (wx3 - xd)) / (wx3 * (1.0i + xd))) * log(1.0 - xd / wx3) + num16 * dilog((-wx3 + xd) / (-1.0i + xd)) * log(1.0 - xd / wx3)
                + num16 * dilog((-wx3 + xd) / (1.0i + xd)) * log(1.0 - xd / wx3) + 2.0 * num16 * ln2 * log((1.0i + xd) / (1.0i + wx3)) * log(1.0 - xd / wx3)
                + ((num16 * log((-1.0i + wx3) / (-1.0i + xd))) / 2.0 - (num16 * log(((-1.0i + wx3) * xd) / (wx3 * (-1.0i + xd)))) / 2.0 - (num16 * log((wx3 * (-1.0i + xd)) / ((-1.0i + wx3) * xd))) / 2.0 + (num16 * log((1.0i + wx3) / (1.0i + xd))) / 2.0
                - (num16 * log(((1.0i + wx3) * xd) / (wx3 * (1.0i + xd)))) / 2.0 + (num16 * log((1.0i + xd) / (1.0i + wx3))) / 2.0 - (num16 * log((wx3 * (1.0i + xd)) / ((1.0i + wx3) * xd))) / 2.0) * power_of<2>(log(1.0 - xd / wx3))
                + dilog((wx3 - xd) / (-1.0i + wx3)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log(1.0 - xd / wx3)) + dilog((wx3 - xd) / (1.0i + wx3)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log(1.0 - xd / wx3))
                + log((-1.0i + xd) / (-1.0i + wx3)) * (2.0 * num16 * ln2 * log(1.0 - xd / wx3) + (num16 * power_of<2>(log(1.0 - xd / wx3))) / 2.0) - num16 * dilog((1.0i * (wx4 - xd)) / ((-1.0i + wx4) * xd)) * log(1.0 - xd / wx4)
                - num16 * dilog(((-1.0i) * (wx4 - xd)) / ((1.0i + wx4) * xd)) * log(1.0 - xd / wx4) - num16 * dilog(((-1.0i) * (wx4 - xd)) / (wx4 * (-1.0i + xd))) * log(1.0 - xd / wx4) - num16 * dilog((1.0i * (wx4 - xd)) / (wx4 * (1.0i + xd))) * log(1.0 - xd / wx4)
                + num16 * dilog((-wx4 + xd) / (-1.0i + xd)) * log(1.0 - xd / wx4) + num16 * dilog((-wx4 + xd) / (1.0i + xd)) * log(1.0 - xd / wx4) + 2.0 * num16 * ln2 * log((1.0i + xd) / (1.0i + wx4)) * log(1.0 - xd / wx4)
                + ((num16 * log((-1.0i + wx4) / (-1.0i + xd))) / 2.0 - (num16 * log(((-1.0i + wx4) * xd) / (wx4 * (-1.0i + xd)))) / 2.0 - (num16 * log((wx4 * (-1.0i + xd)) / ((-1.0i + wx4) * xd))) / 2.0 + (num16 * log((1.0i + wx4) / (1.0i + xd))) / 2.0
                - (num16 * log(((1.0i + wx4) * xd) / (wx4 * (1.0i + xd)))) / 2.0 + (num16 * log((1.0i + xd) / (1.0i + wx4))) / 2.0 - (num16 * log((wx4 * (1.0i + xd)) / ((1.0i + wx4) * xd))) / 2.0) * power_of<2>(log(1.0 - xd / wx4))
                + dilog((wx4 - xd) / (-1.0i + wx4)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log(1.0 - xd / wx4)) + dilog((wx4 - xd) / (1.0i + wx4)) * (2.0 * num16 * ln2 + num16 * log(xd) + num16 * log(1.0 - xd / wx4))
                + log((-1.0i + xd) / (-1.0i + wx4)) * (2.0 * num16 * ln2 * log(1.0 - xd / wx4) + (num16 * power_of<2>(log(1.0 - xd / wx4))) / 2.0) + num18 * dilog((-0.5) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) * log((1.0 - yd) / 2.0)
                + num18 * dilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) * log((1.0 - yd) / 2.0) + num18 * dilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd))) * log((1.0 - yd) / 2.0)
                - num19 * dilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) * log((1.0 - yd) / 2.0) + num18 * dilog((-0.5) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) * log((1.0 - yd) / 2.0)
                - num19 * dilog((-0.5) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) * log((1.0 - yd) / 2.0) + (2.0 * num17 - 2.0 * num18 + num19 - num20) * dilog((-1.0 + yd) / yd) * ln1myd - num17 * dilog((-1.0 + yd) / (-w4 + yd)) * ln1myd
                - num17 * dilog((-1.0 + yd) / (w5 + yd)) * ln1myd + num20 * dilog((-1.0 + yd) / (-w7 + yd)) * ln1myd + num17 * dilog((w4 - w4 * yd) / (w4 - yd)) * ln1myd
                - num18 * dilog((w4 - w4 * yd) / (w4 + yd)) * ln1myd - num18 * dilog((w5 - w5 * yd) / (w5 - yd)) * ln1myd + num17 * dilog((w5 - w5 * yd) / (w5 + yd)) * ln1myd
                - num20 * dilog((w7 - w7 * yd) / (w7 - yd)) * ln1myd + num19 * dilog((w7 - w7 * yd) / (w7 + yd)) * ln1myd + dilog((-1.0 + yd) / (w4 + yd)) * (-(num18 * log((1.0 - yd) / 2.0)) + num18 * ln1myd)
                + dilog((-1.0 + yd) / (-w5 + yd)) * (-(num18 * log((1.0 - yd) / 2.0)) + num18 * ln1myd) + dilog((-1.0 + yd) / (w7 + yd)) * (num19 * log((1.0 - yd) / 2.0) - num19 * ln1myd) - (num17 * pisqu * log(1.0 / (w4 - yd))) / 6.0
                - (num17 * power_of<3>(log(1.0 / (w4 - yd)))) / 6.0 + (num17 * pisqu * log(2.0 / (w4 - yd))) / 6.0 + (num17 * power_of<3>(log(2.0 / (w4 - yd)))) / 6.0 + (num18 * pisqu * log(-2.0 / (w5 - yd))) / 6.0 + (num18 * power_of<3>(log(-2.0 / (w5 - yd)))) / 6.0
                + (num18 * pisqu * log(1.0 / (w5 - yd))) / 6.0 + (num18 * power_of<3>(log(1.0 / (w5 - yd)))) / 6.0 + (num20 * pisqu * log(1.0 / (w7 - yd))) / 6.0 + (num20 * power_of<3>(log(1.0 / (w7 - yd)))) / 6.0 - (num20 * pisqu * log(2.0 / (w7 - yd))) / 6.0
                - (num20 * power_of<3>(log(2.0 / (w7 - yd)))) / 6.0 - (num17 * pisqu * log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd)))) / 6.0 - (num17 * power_of<3>(log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd))))) / 6.0
                - (num17 * pisqu * log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd)))) / 6.0 - (num17 * power_of<3>(log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd))))) / 6.0 + (num20 * pisqu * log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd)))) / 6.0
                + (num20 * power_of<3>(log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd))))) / 6.0 + (num17 / 3.0 - num18 / 3.0 + num19 / 6.0 - num20 / 6.0) * power_of<3>(log(-ydinv)) + ((-1.0 / 3.0) * (num17 * pisqu) + (num18 * pisqu) / 3.0 - (num19 * pisqu) / 6.0 + (num20 * pisqu) / 6.0) * 
                log(ydinv) + ((-1.0 / 3.0) * num17 + num18 / 3.0 - num19 / 6.0 + num20 / 6.0) * power_of<3>(log(ydinv)) - (num18 * pisqu * log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd)))) / 6.0 - (num18 * power_of<3>(log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd))))) / 6.0
                - (num18 * pisqu * log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd)))) / 6.0 - (num18 * power_of<3>(log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd))))) / 6.0 + (num19 * pisqu * log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd)))) / 6.0
                + (num19 * power_of<3>(log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd))))) / 6.0 + num17 * dilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) * log((1.0 + yd) / 2.0) - num20 * dilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) * log((1.0 + yd) / 2.0)
                + num17 * dilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd))) * log((1.0 + yd) / 2.0) + num17 * dilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) * log((1.0 + yd) / 2.0)
                - num20 * dilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd))) * log((1.0 + yd) / 2.0) + num17 * dilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) * log((1.0 + yd) / 2.0)
                + (-2.0 * num17 + 2.0 * num18 - num19 + num20) * dilog(1.0 + ydinv) * ln1pyd - num17 * dilog((w4 * (1.0 + yd)) / (w4 - yd)) * ln1pyd - num18 * dilog(-((1.0 + yd) / (w5 - yd))) * ln1pyd
                + num18 * dilog((w5 * (1.0 + yd)) / (w5 - yd)) * ln1pyd + num20 * dilog((w7 * (1.0 + yd)) / (w7 - yd)) * ln1pyd - num18 * dilog((1.0 + yd) / (w4 + yd)) * ln1pyd
                + num18 * dilog((w4 * (1.0 + yd)) / (w4 + yd)) * ln1pyd - num17 * dilog((w5 * (1.0 + yd)) / (w5 + yd)) * ln1pyd + num19 * dilog((1.0 + yd) / (w7 + yd)) * ln1pyd
                - num19 * dilog((w7 * (1.0 + yd)) / (w7 + yd)) * ln1pyd + (num17 * log((1.0 + w4) / (w4 - yd)) * power_of<2>(ln1pyd)) / 2.0 - (num20 * log((1.0 + w7) / (w7 - yd)) * power_of<2>(ln1pyd)) / 2.0
                + dilog(-((1.0 + yd) / (w4 - yd))) * (-(num17 * log((1.0 + yd) / 2.0)) + num17 * ln1pyd) + dilog((1.0 + yd) / (w5 + yd)) * (-(num17 * log((1.0 + yd) / 2.0)) + num17 * ln1pyd)
                + dilog((1.0 + yd) / (1.0 + w4)) * (2.0 * num17 * ln2 - num17 * log(1.0 - 1.0i * xd) - num17 * log(1.0 + 1.0i * xd) + num17 * log(xd) - num17 * ln1myd - num17 * log((1.0 + yd) / 2.0) + num17 * ln1pyd)
                + dilog((1.0 + yd) / (1.0 - w5)) * (2.0 * num17 * ln2 - num17 * log(1.0 - 1.0i * xd) - num17 * log(1.0 + 1.0i * xd) + num17 * log(xd) - num17 * ln1myd - num17 * log((1.0 + yd) / 2.0) + num17 * ln1pyd)
                + dilog((1.0 - yd) / (1.0 + w4)) * (2.0 * num18 * ln2 - num18 * log(1.0 - 1.0i * xd) - num18 * log(1.0 + 1.0i * xd) + num18 * log(xd) - num18 * log((1.0 - yd) / 2.0) + num18 * ln1myd - num18 * ln1pyd)
                + dilog((-1.0 + yd) / (-1.0 + w5)) * (2.0 * num18 * ln2 - num18 * log(1.0 - 1.0i * xd) - num18 * log(1.0 + 1.0i * xd) + num18 * log(xd) - num18 * log((1.0 - yd) / 2.0) + num18 * ln1myd - num18 * ln1pyd)
                + dilog(2.0 / (1.0 + w4)) * (num17 * ln1myd + num18 * ln1pyd) + dilog(-2.0 / (-1.0 + w5)) * (num17 * ln1myd + num18 * ln1pyd) + dilog(2.0 / (1.0 + w7)) * (-(num20 * ln1myd) - num19 * ln1pyd)
                + dilog((1.0 - yd) / (1.0 + w7)) * (-4.0 * num19 * ln2 + 2.0 * num19 * log(1.0 - 1.0i * xd) + 2.0 * num19 * log(1.0 + 1.0i * xd) - 2.0 * num19 * log(xd) + num19 * log((1.0 - yd) / 2.0) - num19 * ln1myd + num19 * ln1pyd)
                + dilog(-((1.0 + yd) / (w7 - yd))) * (num20 * log((1.0 + yd) / 2.0) - num20 * ln1pyd) + dilog((1.0 + yd) / (1.0 + w7)) * (-4.0 * num20 * ln2 + 2.0 * num20 * log(1.0 - 1.0i * xd) + 2.0 * num20 * log(1.0 + 1.0i * xd) - 2.0 * num20 * log(xd)
                + num20 * ln1myd + num20 * log((1.0 + yd) / 2.0) - num20 * ln1pyd) + log((w4 - yd) / (1.0 + w4)) * ((-0.5) * (num17 * power_of<2>(log((1.0 + yd) / 2.0)))
                + (2.0 * num17 * ln2 - num17 * log(1.0 - 1.0i * xd) - num17 * log(1.0 + 1.0i * xd) + num17 * log(xd)) * ln1pyd + (num17 * power_of<2>(ln1pyd)) / 2.0)
                + log(-ydinv) * ((num17 * pisqu) / 3.0 - (num18 * pisqu) / 3.0 + (num19 * pisqu) / 6.0 - (num20 * pisqu) / 6.0 + (-num17 + num18 - num19 / 2.0 + num20 / 2.0) * power_of<2>(ln1pyd))
                + log((w7 - yd) / (1.0 + w7)) * ((-4.0 * num20 * ln2 + 2.0 * num20 * log(1.0 - 1.0i * xd) + 2.0 * num20 * log(1.0 + 1.0i * xd) - 2.0 * num20 * log(xd)) * ln1pyd - (num20 * power_of<2>(ln1pyd)) / 2.0) + (num17 * pisqu * log(1.0 / (-w4 + yd))) / 6.0
                + (num17 * power_of<3>(log(1.0 / (-w4 + yd)))) / 6.0 + (num18 * pisqu * log(-1.0 / (w4 + yd))) / 6.0 + (num18 * power_of<3>(log(-1.0 / (w4 + yd)))) / 6.0 - (num18 * pisqu * log(1.0 / (w4 + yd))) / 6.0 - (num18 * power_of<3>(log(1.0 / (w4 + yd)))) / 6.0
                + (num18 * pisqu * log(2.0 / (w4 + yd))) / 6.0 + (num18 * power_of<3>(log(2.0 / (w4 + yd)))) / 6.0 - (num18 * pisqu * log(1.0 / (-w5 + yd))) / 6.0 - (num18 * power_of<3>(log(1.0 / (-w5 + yd)))) / 6.0 + (num17 * pisqu * log(-2.0 / (w5 + yd))) / 6.0
                + (num17 * power_of<3>(log(-2.0 / (w5 + yd)))) / 6.0 - (num17 * pisqu * log(-1.0 / (w5 + yd))) / 6.0 - (num17 * power_of<3>(log(-1.0 / (w5 + yd)))) / 6.0 + (num17 * pisqu * log(1.0 / (w5 + yd))) / 6.0 + (num17 * power_of<3>(log(1.0 / (w5 + yd)))) / 6.0
                + power_of<2>(log((1.0 + yd) / 2.0)) * ((-0.5) * (num17 * log((1.0 + w4) / (w4 - yd))) + (num20 * log((1.0 + w7) / (w7 - yd))) / 2.0 + (num20 * log((w7 - yd) / (1.0 + w7))) / 2.0 + (num17 * log((-2.0 * (w4 - yd)) / ((1.0 + w4) * (-1.0 + yd)))) / 2.0
                - (num20 * log((-2.0 * (w7 - yd)) / ((1.0 + w7) * (-1.0 + yd)))) / 2.0 + (num17 * log((-0.5) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd)))) / 2.0 - (num20 * log((-0.5) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd)))) / 2.0
                - (num17 * log((-1.0 + w5) / (w5 + yd))) / 2.0 + (num17 * log((-0.5) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd)))) / 2.0 - (num17 * log((w5 + yd) / (-1.0 + w5))) / 2.0 + (num17 * log((-2.0 * (w5 + yd)) / ((-1.0 + w5) * (-1.0 + yd)))) / 2.0)
                - (num20 * pisqu * log(1.0 / (-w7 + yd))) / 6.0 - (num20 * power_of<3>(log(1.0 / (-w7 + yd)))) / 6.0 - (num19 * pisqu * log(-1.0 / (w7 + yd))) / 6.0 - (num19 * power_of<3>(log(-1.0 / (w7 + yd)))) / 6.0 + (num19 * pisqu * log(1.0 / (w7 + yd))) / 6.0
                + (num19 * power_of<3>(log(1.0 / (w7 + yd)))) / 6.0 - (num19 * pisqu * log(2.0 / (w7 + yd))) / 6.0 - (num19 * power_of<3>(log(2.0 / (w7 + yd)))) / 6.0 + power_of<2>(ln1myd) * ((-0.5) * (num17 * log((-1.0 + w4) / (w4 - yd))) + (num18 * log((-1.0 + w5) / (w5 - yd))) / 2.0
                + (num18 * log((w5 - yd) / (-1.0 + w5))) / 2.0 + (num20 * log((-1.0 + w7) / (w7 - yd))) / 2.0 + (num17 - num18 + num19 / 2.0 - num20 / 2.0) * log(ydinv) + (num17 * log(((-1.0 + w4) * yd) / (w4 - yd))) / 2.0
                - (num18 * log(((-1.0 + w5) * yd) / (w5 - yd))) / 2.0 - (num20 * log(((-1.0 + w7) * yd) / (w7 - yd))) / 2.0 + (num18 * log((1.0 + w4) / (w4 + yd))) / 2.0 - (num18 * log(((1.0 + w4) * yd) / (w4 + yd))) / 2.0
                + (num18 * log((w4 + yd) / (1.0 + w4))) / 2.0 - (num17 * log((1.0 + w5) / (w5 + yd))) / 2.0 + (num17 * log(((1.0 + w5) * yd) / (w5 + yd))) / 2.0 - (num19 * log((1.0 + w7) / (w7 + yd))) / 2.0 + (num19 * log(((1.0 + w7) * yd) / (w7 + yd))) / 2.0
                - (num19 * log((w7 + yd) / (1.0 + w7))) / 2.0) + log((1.0 - yd) / 2.0) * (-(num18 * log((w5 - yd) / (-1.0 + w5)) * ln1pyd) + ln1pyd * (-(num18 * log((w4 + yd) / (1.0 + w4))) + num19 * log((w7 + yd) / (1.0 + w7))))
                + power_of<2>(log((1.0 - yd) / 2.0)) * ((-0.5) * (num18 * log((-1.0 + w5) / (w5 - yd))) - (num18 * log((w5 - yd) / (-1.0 + w5))) / 2.0 + (num18 * log((2.0 * (w5 - yd)) / ((-1.0 + w5) * (1.0 + yd)))) / 2.0
                + (num18 * log(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd)))) / 2.0 - (num18 * log((1.0 + w4) / (w4 + yd))) / 2.0 + (num18 * log(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd)))) / 2.0 - (num18 * log((w4 + yd) / (1.0 + w4))) / 2.0
                + (num18 * log((2.0 * (w4 + yd)) / ((1.0 + w4) * (1.0 + yd)))) / 2.0 + (num19 * log((1.0 + w7) / (w7 + yd))) / 2.0 - (num19 * log(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd)))) / 2.0 + (num19 * log((w7 + yd) / (1.0 + w7))) / 2.0
                - (num19 * log((2.0 * (w7 + yd)) / ((1.0 + w7) * (1.0 + yd)))) / 2.0) + dilog(w4) * (-(num18 * log((w4 + yd) / w4)) - num17 * log(1.0 - yd / w4)) + dilog(-w4) * (num18 * log((w4 + yd) / w4) + num17 * log(1.0 - yd / w4))
                + dilog(-w5) * (-(num17 * log((w5 + yd) / w5)) - num18 * log(1.0 - yd / w5)) + dilog(w5) * (num17 * log((w5 + yd) / w5) + num18 * log(1.0 - yd / w5))
                + dilog(-1.0i / (-1.0i + wx3)) * (-4.0 * num16 * lnhalf + 2.0 * num16 * log(1.0 - 1.0i * xd) + 2.0 * num16 * log(1.0 + 1.0i * xd) - 2.0 * num15 * ln1myd - 2.0 * num14 * ln1pyd + num18 * log((w4 + yd) / w4) + num17 * log((w5 + yd) / w5)
                - 4.0 * num19 * log((w7 + yd) / w7) + num17 * log(1.0 - yd / w4) + num18 * log(1.0 - yd / w5) - 4.0 * num20 * log(1.0 - yd / w7))
                + dilog(1.0i / (1.0i + wx3)) * (-4.0 * num16 * lnhalf + 2.0 * num16 * log(1.0 - 1.0i * xd) + 2.0 * num16 * log(1.0 + 1.0i * xd) - 2.0 * num15 * ln1myd - 2.0 * num14 * ln1pyd + num18 * log((w4 + yd) / w4) + num17 * log((w5 + yd) / w5)
                - 4.0 * num19 * log((w7 + yd) / w7) + num17 * log(1.0 - yd / w4) + num18 * log(1.0 - yd / w5) - 4.0 * num20 * log(1.0 - yd / w7))
                + dilog(-1.0i / (-1.0i + wx4)) * (-4.0 * num16 * lnhalf + 2.0 * num16 * log(1.0 - 1.0i * xd) + 2.0 * num16 * log(1.0 + 1.0i * xd) - 2.0 * num15 * ln1myd - 2.0 * num14 * ln1pyd + num18 * log((w4 + yd) / w4) + num17 * log((w5 + yd) / w5)
                - 4.0 * num19 * log((w7 + yd) / w7) + num17 * log(1.0 - yd / w4) + num18 * log(1.0 - yd / w5) - 4.0 * num20 * log(1.0 - yd / w7))
                + dilog(1.0i / (1.0i + wx4)) * (-4.0 * num16 * lnhalf + 2.0 * num16 * log(1.0 - 1.0i * xd) + 2.0 * num16 * log(1.0 + 1.0i * xd) - 2.0 * num15 * ln1myd - 2.0 * num14 * ln1pyd + num18 * log((w4 + yd) / w4) + num17 * log((w5 + yd) / w5)
                - 4.0 * num19 * log((w7 + yd) / w7) + num17 * log(1.0 - yd / w4) + num18 * log(1.0 - yd / w5) - 4.0 * num20 * log(1.0 - yd / w7))
                + dilog(xd / wx3) * (2.0 * num16 * ln2 + num16 * log(xd) - num15 * ln1myd - num14 * ln1pyd + num18 * log((w4 + yd) / w4) - 2.0 * num19 * log((w7 + yd) / w7) + num17 * log(1.0 - yd / w4)
                - 2.0 * num20 * log(1.0 - yd / w7)) + dilog(xd / wx4) * (2.0 * num16 * ln2 + num16 * log(xd) - num15 * ln1myd - num14 * ln1pyd + num18 * log((w4 + yd) / w4) - 2.0 * num19 * log((w7 + yd) / w7)
                + num17 * log(1.0 - yd / w4) - 2.0 * num20 * log(1.0 - yd / w7)) + dilog(-(xd / wx3)) * (2.0 * num16 * ln2 + num16 * log(xd) - num15 * ln1myd - num14 * ln1pyd + num17 * log((w5 + yd) / w5)
                - 2.0 * num19 * log((w7 + yd) / w7) + num18 * log(1.0 - yd / w5) - 2.0 * num20 * log(1.0 - yd / w7)) + dilog(-(xd / wx4)) * (2.0 * num16 * ln2 + num16 * log(xd) - num15 * ln1myd - num14 * ln1pyd
                + num17 * log((w5 + yd) / w5) - 2.0 * num19 * log((w7 + yd) / w7) + num18 * log(1.0 - yd / w5) - 2.0 * num20 * log(1.0 - yd / w7)) + dilog(-w7) * (-(num19 * log((w7 + yd) / w7)) - num20 * log(1.0 - yd / w7))
                + dilog(w7) * (num19 * log((w7 + yd) / w7) + num20 * log(1.0 - yd / w7)) + dilog((-1.0i + xd) / (-1.0i + wx3)) * (2.0 * num16 * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + num16 * log(1.0 - 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd
                - num18 * log((w4 + yd) / w4) + 2.0 * num19 * log((w7 + yd) / w7) - num17 * log(1.0 - yd / w4) + 2.0 * num20 * log(1.0 - yd / w7))
                + dilog((-1.0i + xd) / (-1.0i + wx4)) * (2.0 * num16 * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + num16 * log(1.0 - 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd - num18 * log((w4 + yd) / w4) + 2.0 * num19 * log((w7 + yd) / w7)
                - num17 * log(1.0 - yd / w4) + 2.0 * num20 * log(1.0 - yd / w7)) + dilog((1.0i + xd) / (1.0i + wx3)) * (2.0 * num16 * log(1.0 / 2.0 - (1.0i / 2.0) * xd) + num16 * log(1.0 + 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd
                - num18 * log((w4 + yd) / w4) + 2.0 * num19 * log((w7 + yd) / w7) - num17 * log(1.0 - yd / w4) + 2.0 * num20 * log(1.0 - yd / w7))
                + dilog((1.0i + xd) / (1.0i + wx4)) * (2.0 * num16 * log(1.0 / 2.0 - (1.0i / 2.0) * xd) + num16 * log(1.0 + 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd - num18 * log((w4 + yd) / w4) + 2.0 * num19 * log((w7 + yd) / w7)
                - num17 * log(1.0 - yd / w4) + 2.0 * num20 * log(1.0 - yd / w7)) + dilog((1.0i - xd) / (1.0i + wx3)) * (2.0 * num16 * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + num16 * log(1.0 - 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd
                - num17 * log((w5 + yd) / w5) + 2.0 * num19 * log((w7 + yd) / w7) - num18 * log(1.0 - yd / w5) + 2.0 * num20 * log(1.0 - yd / w7))
                + dilog((1.0i - xd) / (1.0i + wx4)) * (2.0 * num16 * log(1.0 / 2.0 + (1.0i / 2.0) * xd) + num16 * log(1.0 - 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd - num17 * log((w5 + yd) / w5) + 2.0 * num19 * log((w7 + yd) / w7)
                - num18 * log(1.0 - yd / w5) + 2.0 * num20 * log(1.0 - yd / w7)) + dilog((1.0i + xd) / (1.0i - wx3)) * (2.0 * num16 * log(1.0 / 2.0 - (1.0i / 2.0) * xd) + num16 * log(1.0 + 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd
                - num17 * log((w5 + yd) / w5) + 2.0 * num19 * log((w7 + yd) / w7) - num18 * log(1.0 - yd / w5) + 2.0 * num20 * log(1.0 - yd / w7))
                + dilog((1.0i + xd) / (1.0i - wx4)) * (2.0 * num16 * log(1.0 / 2.0 - (1.0i / 2.0) * xd) + num16 * log(1.0 + 1.0i * xd) + num15 * ln1myd + num14 * ln1pyd - num17 * log((w5 + yd) / w5) + 2.0 * num19 * log((w7 + yd) / w7)
                - num18 * log(1.0 - yd / w5) + 2.0 * num20 * log(1.0 - yd / w7)) + dilog(1.0 / 2.0 - (1.0i / 2.0) * xd) * (2.0 * num15 * ln1myd + 2.0 * num14 * ln1pyd - num18 * log((w4 + yd) / w4) - num17 * log((w5 + yd) / w5)
                + 4.0 * num19 * log((w7 + yd) / w7) - num17 * log(1.0 - yd / w4) - num18 * log(1.0 - yd / w5) + 4.0 * num20 * log(1.0 - yd / w7))
                + dilog((1.0 + 1.0i * xd) / 2.0) * (2.0 * num15 * ln1myd + 2.0 * num14 * ln1pyd - num18 * log((w4 + yd) / w4) - num17 * log((w5 + yd) / w5) + 4.0 * num19 * log((w7 + yd) / w7) - num17 * log(1.0 - yd / w4) - num18 * log(1.0 - yd / w5)
                + 4.0 * num20 * log(1.0 - yd / w7)) + dilog((-1.0i) * xd) * (-4.0 * num16 * ln2 - 2.0 * num16 * log(xd) + 2.0 * num15 * ln1myd + 2.0 * num14 * ln1pyd - num18 * log((w4 + yd) / w4) - num17 * log((w5 + yd) / w5)
                + 4.0 * num19 * log((w7 + yd) / w7) - num17 * log(1.0 - yd / w4) - num18 * log(1.0 - yd / w5) + 4.0 * num20 * log(1.0 - yd / w7))
                + dilog(1.0i * xd) * (-4.0 * num16 * ln2 - 2.0 * num16 * log(xd) + 2.0 * num15 * ln1myd + 2.0 * num14 * ln1pyd - num18 * log((w4 + yd) / w4) - num17 * log((w5 + yd) / w5) + 4.0 * num19 * log((w7 + yd) / w7)
                - num17 * log(1.0 - yd / w4) - num18 * log(1.0 - yd / w5) + 4.0 * num20 * log(1.0 - yd / w7)) + power_of<2>(ln1pyd) * ((-0.5) * (num18 * log((1.0 + w5) / (w5 - yd))) - (num17 * log(((1.0 + w4) * yd) / (-w4 + yd))) / 2.0
                - (num18 * log((-1.0 + w4) / (w4 + yd))) / 2.0 + (num18 * log(((1.0 + w5) * yd) / (-w5 + yd))) / 2.0 + (num17 * log((-1.0 + w5) / (w5 + yd))) / 2.0 + (num17 * log((w5 + yd) / (-1.0 + w5))) / 2.0 + (num20 * log(((1.0 + w7) * yd) / (-w7 + yd))) / 2.0
                + (num19 * log((-1.0 + w7) / (w7 + yd))) / 2.0 + (num18 * log((yd - w4 * yd) / (w4 + yd))) / 2.0 - (num17 * log((yd - w5 * yd) / (w5 + yd))) / 2.0 - (num19 * log((yd - w7 * yd) / (w7 + yd))) / 2.0)
                + 1.0i * num16 * M_PI * H1(1.0i * wx3, -(wx3 * xdinv)) * power_of<2>(log((-1.0i + xd) / wx3)) * my_sign(-imag(wx3 * xdinv)) + 1.0i * num16 * M_PI * H1((-1.0i) * wx3, -(wx3 * xdinv)) * power_of<2>(log((1.0i + xd) / wx3)) * my_sign(-imag(wx3 * xdinv))
                + 1.0i * num16 * M_PI * H1((-1.0i) * wx3, wx3 * xdinv) * power_of<2>(log((1.0i - xd) / wx3)) * my_sign(imag(wx3 * xdinv)) + 1.0i * num16 * M_PI * H1(1.0i * wx3, wx3 * xdinv) * power_of<2>(log(-((1.0i + xd) / wx3))) * my_sign(imag(wx3 * xdinv))
                + 1.0i * num16 * M_PI * H1(1.0i * wx4, -(wx4 * xdinv)) * power_of<2>(log((-1.0i + xd) / wx4)) * my_sign(-imag(wx4 * xdinv)) + 1.0i * num16 * M_PI * H1((-1.0i) * wx4, -(wx4 * xdinv)) * power_of<2>(log((1.0i + xd) / wx4)) * my_sign(-imag(wx4 * xdinv))
                + 1.0i * num16 * M_PI * H1((-1.0i) * wx4, wx4 * xdinv) * power_of<2>(log((1.0i - xd) / wx4)) * my_sign(imag(wx4 * xdinv)) + 1.0i * num16 * M_PI * H1(1.0i * wx4, wx4 * xdinv) * power_of<2>(log(-((1.0i + xd) / wx4))) * my_sign(imag(wx4 * xdinv))
                + 1.0i * num17 * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * power_of<2>(log(((1.0 + w4) * (-1.0 + yd)) / (2.0 * (-1.0 + w4)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + 1.0i * num17 * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * 
                power_of<2>(log(((-1.0 + w5) * (-1.0 + yd)) / (2.0 * (1.0 + w5)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - 1.0i * num20 * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * power_of<2>(log(((1.0 + w7) * (-1.0 + yd)) / (2.0 * (-1.0 + w7)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                - 1.0i * num17 * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * power_of<2>(log((-w4 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - 1.0i * num17 * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * power_of<2>(log((w5 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                + 1.0i * num20 * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * power_of<2>(log((-w7 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + 1.0i * num17 * M_PI * H1(-w4inv, -ydinv) * power_of<2>(log(-w4 + yd)) * my_sign(-imydinv)
                - 1.0i * num18 * M_PI * H1(w4inv, -ydinv) * power_of<2>(log(w4 + yd)) * my_sign(-imydinv) - 1.0i * num18 * M_PI * H1(-w5inv, -ydinv) * power_of<2>(log(-w5 + yd)) * my_sign(-imydinv)
                + 1.0i * num17 * M_PI * H1(w5inv, -ydinv) * power_of<2>(log(w5 + yd)) * my_sign(-imydinv) - 1.0i * num20 * M_PI * H1(-w7inv, -ydinv) * power_of<2>(log(-w7 + yd)) * my_sign(-imydinv)
                + 1.0i * num19 * M_PI * H1(w7inv, -ydinv) * power_of<2>(log(w7 + yd)) * my_sign(-imydinv) + 1.0i * num18 * M_PI * H1(-w4inv, ydinv) * power_of<2>(log(-w4 - yd)) * my_sign(imydinv)
                - 1.0i * num17 * M_PI * H1(w4inv, ydinv) * power_of<2>(log(w4 - yd)) * my_sign(imydinv) - 1.0i * num17 * M_PI * H1(-w5inv, ydinv) * power_of<2>(log(-w5 - yd)) * my_sign(imydinv)
                + 1.0i * num18 * M_PI * H1(w5inv, ydinv) * power_of<2>(log(w5 - yd)) * my_sign(imydinv) - 1.0i * num19 * M_PI * H1(-w7inv, ydinv) * power_of<2>(log(-w7 - yd)) * my_sign(imydinv)
                + 1.0i * num20 * M_PI * H1(w7inv, ydinv) * power_of<2>(log(w7 - yd)) * my_sign(imydinv) - 1.0i * num18 * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * power_of<2>(log((-w4 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                - 1.0i * num18 * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * power_of<2>(log((w5 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + 1.0i * num19 * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * power_of<2>(log((-w7 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                + 1.0i * num18 * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * power_of<2>(log(((1.0 + w4) * (1.0 + yd)) / (2.0 - 2.0 * w4))) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + 1.0i * num18 * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * 
                power_of<2>(log((-0.5) * (((-1.0 + w5) * (1.0 + yd)) / (1.0 + w5)))) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - 1.0i * num19 * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * power_of<2>(log(((1.0 + w7) * (1.0 + yd)) / (2.0 - 2.0 * w7))) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                + (2.0 * 1.0i) * num16 * M_PI * H1((-2.0 * 1.0i) / (-1.0i + wx3), (-2.0 * 1.0i) / (-1.0i + xd)) * power_of<2>(log((1.0i / 2.0) * (wx3 - xd))) * my_sign(2.0 * real(1.0 / (1.0i - xd))) + (2.0 * 1.0i) * num16 * M_PI * H1((-2.0 * 1.0i) / (-1.0i + wx4), (-2.0 * 1.0i) / (-1.0i + xd)) * power_of<2>(log((1.0i / 2.0) * (wx4 - xd))) * 
                my_sign(2.0 * real(1.0 / (1.0i - xd))) - (2.0 * 1.0i) * num16 * M_PI * H1((-1.0i + wx3) / (-1.0i + xd), (-2.0 * 1.0i) / (-1.0i + xd)) * power_of<2>(log(((1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * 1.0i - 2.0 * wx3))) * my_sign(2.0 * real(1.0 / (1.0i - xd)))
                - (2.0 * 1.0i) * num16 * M_PI * H1((-1.0i + wx4) / (-1.0i + xd), (-2.0 * 1.0i) / (-1.0i + xd)) * power_of<2>(log(((1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * 1.0i - 2.0 * wx4))) * my_sign(2.0 * real(1.0 / (1.0i - xd))) - (2.0 * 1.0i) * num16 * M_PI * H1((1.0i + wx3) / (1.0i - xd), (-2.0 * 1.0i) / (-1.0i + xd)) * 
                power_of<2>(log((((-0.5) * 1.0i) * (-1.0i + wx3) * (-1.0i + xd)) / (1.0i + wx3))) * my_sign(2.0 * real(1.0 / (1.0i - xd))) - (2.0 * 1.0i) * num16 * M_PI * H1((1.0i + wx4) / (1.0i - xd), (-2.0 * 1.0i) / (-1.0i + xd)) * power_of<2>(log((((-0.5) * 1.0i) * (-1.0i + wx4) * (-1.0i + xd)) / (1.0i + wx4))) * 
                my_sign(2.0 * real(1.0 / (1.0i - xd))) + (2.0 * 1.0i) * num16 * M_PI * H1((2.0 * 1.0i) / (1.0i + wx3), (-2.0 * 1.0i) / (-1.0i + xd)) * power_of<2>(log(((-0.5) * 1.0i) * (wx3 + xd))) * my_sign(2.0 * real(1.0 / (1.0i - xd)))
                + (2.0 * 1.0i) * num16 * M_PI * H1((2.0 * 1.0i) / (1.0i + wx4), (-2.0 * 1.0i) / (-1.0i + xd)) * power_of<2>(log(((-0.5) * 1.0i) * (wx4 + xd))) * my_sign(2.0 * real(1.0 / (1.0i - xd)))
                + power_of<2>(log((-1.0i - wx3inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(1.0i / xd, wx3 * xdinv) * my_sign(imag(wx3 * xdinv)) + 1.0i * num16 * M_PI * H1(-(wx3 * xdinv), -1.0i / xd) * my_sign(-real(xdinv)))
                + power_of<2>(log((-1.0i + wx3inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(1.0i / xd, -(wx3 * xdinv)) * my_sign(-imag(wx3 * xdinv)) + 1.0i * num16 * M_PI * H1(wx3 * xdinv, -1.0i / xd) * my_sign(-real(xdinv)))
                + power_of<2>(log((-1.0i - wx4inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(1.0i / xd, wx4 * xdinv) * my_sign(imag(wx4 * xdinv)) + 1.0i * num16 * M_PI * H1(-(wx4 * xdinv), -1.0i / xd) * my_sign(-real(xdinv)))
                + power_of<2>(log((-1.0i + wx4inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(1.0i / xd, -(wx4 * xdinv)) * my_sign(-imag(wx4 * xdinv)) + 1.0i * num16 * M_PI * H1(wx4 * xdinv, -1.0i / xd) * my_sign(-real(xdinv)))
                + power_of<2>(log((1.0i - wx3inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(-1.0i / xd, wx3 * xdinv) * my_sign(imag(wx3 * xdinv)) + 1.0i * num16 * M_PI * H1(-(wx3 * xdinv), 1.0i / xd) * my_sign(real(xdinv)))
                + power_of<2>(log((1.0i + wx3inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(-1.0i / xd, -(wx3 * xdinv)) * my_sign(-imag(wx3 * xdinv)) + 1.0i * num16 * M_PI * H1(wx3 * xdinv, 1.0i / xd) * my_sign(real(xdinv)))
                + power_of<2>(log((1.0i - wx4inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(-1.0i / xd, wx4 * xdinv) * my_sign(imag(wx4 * xdinv)) + 1.0i * num16 * M_PI * H1(-(wx4 * xdinv), 1.0i / xd) * my_sign(real(xdinv)))
                + power_of<2>(log((1.0i + wx4inv) * xd)) * ((-1.0i) * num16 * M_PI * H1(-1.0i / xd, -(wx4 * xdinv)) * my_sign(-imag(wx4 * xdinv)) + 1.0i * num16 * M_PI * H1(wx4 * xdinv, 1.0i / xd) * my_sign(real(xdinv)))
                + (2.0 * 1.0i) * num16 * M_PI * H1((2.0 * 1.0i) / (1.0i + wx3), (2.0 * 1.0i) / (1.0i + xd)) * power_of<2>(log(((-0.5) * 1.0i) * (wx3 - xd))) * my_sign(2.0 * real(1.0 / (1.0i + xd))) + (2.0 * 1.0i) * num16 * M_PI * H1((2.0 * 1.0i) / (1.0i + wx4), (2.0 * 1.0i) / (1.0i + xd)) * power_of<2>(log(((-0.5) * 1.0i) * (wx4 - xd))) * 
                my_sign(2.0 * real(1.0 / (1.0i + xd))) - (2.0 * 1.0i) * num16 * M_PI * H1((1.0i + wx3) / (1.0i + xd), (2.0 * 1.0i) / (1.0i + xd)) * power_of<2>(log(((1.0 + 1.0i * wx3) * (1.0i + xd)) / (2.0 * (1.0i + wx3)))) * my_sign(2.0 * real(1.0 / (1.0i + xd)))
                - (2.0 * 1.0i) * num16 * M_PI * H1((1.0i + wx4) / (1.0i + xd), (2.0 * 1.0i) / (1.0i + xd)) * power_of<2>(log(((1.0 + 1.0i * wx4) * (1.0i + xd)) / (2.0 * (1.0i + wx4)))) * my_sign(2.0 * real(1.0 / (1.0i + xd))) + (2.0 * 1.0i) * num16 * M_PI * H1((-2.0 * 1.0i) / (-1.0i + wx3), (2.0 * 1.0i) / (1.0i + xd)) * 
                power_of<2>(log((1.0i / 2.0) * (wx3 + xd))) * my_sign(2.0 * real(1.0 / (1.0i + xd))) + (2.0 * 1.0i) * num16 * M_PI * H1((-2.0 * 1.0i) / (-1.0i + wx4), (2.0 * 1.0i) / (1.0i + xd)) * power_of<2>(log((1.0i / 2.0) * (wx4 + xd))) * my_sign(2.0 * real(1.0 / (1.0i + xd)))
                - (2.0 * 1.0i) * num16 * M_PI * H1((1.0i - wx3) / (1.0i + xd), (2.0 * 1.0i) / (1.0i + xd)) * power_of<2>(log((1.0i + wx3 + xd - 1.0i * wx3 * xd) / (2.0 * 1.0i - 2.0 * wx3))) * my_sign(2.0 * real(1.0 / (1.0i + xd))) - (2.0 * 1.0i) * num16 * M_PI * H1((1.0i - wx4) / (1.0i + xd), (2.0 * 1.0i) / (1.0i + xd)) * 
                power_of<2>(log((1.0i + wx4 + xd - 1.0i * wx4 * xd) / (2.0 * 1.0i - 2.0 * wx4))) * my_sign(2.0 * real(1.0 / (1.0i + xd))) - 1.0i * num18 * M_PI * power_of<2>(log(w4 / (-1.0 + w4))) * my_sign(-imag(1.0 / (1.0 - w4))) * T(1.0, 1.0 / 2.0, w4 / (-1.0 + w4))
                - 1.0i * num17 * M_PI * power_of<2>(log(w4 / (-1.0 + w4))) * my_sign(imag(1.0 / (-1.0 + w4))) * T(1.0, 1.0 / 2.0, w4 / (-1.0 + w4)) - 1.0i * num17 * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-0.5) * imag(w4)) * T(1.0, 1.0 / 2.0, (1.0 + w4) / 2.0)
                - 1.0i * num18 * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-0.5) * imag(w4)) * T(1.0, 1.0 / 2.0, (1.0 + w4) / 2.0) - 1.0i * num17 * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - w5) / 2.0)
                - 1.0i * num18 * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, 1.0 / 2.0, (1.0 - w5) / 2.0) - 1.0i * num17 * M_PI * power_of<2>(log(w5 / (1.0 + w5))) * my_sign(-imag(1.0 / (1.0 + w5))) * T(1.0, 1.0 / 2.0, w5 / (1.0 + w5))
                - 1.0i * num18 * M_PI * power_of<2>(log(w5 / (1.0 + w5))) * my_sign(-imag(1.0 / (1.0 + w5))) * T(1.0, 1.0 / 2.0, w5 / (1.0 + w5)) + 1.0i * num19 * M_PI * power_of<2>(log(w7 / (-1.0 + w7))) * my_sign(-imag(1.0 / (1.0 - w7))) * T(1.0, 1.0 / 2.0, w7 / (-1.0 + w7))
                + 1.0i * num20 * M_PI * power_of<2>(log(w7 / (-1.0 + w7))) * my_sign(imag(1.0 / (-1.0 + w7))) * T(1.0, 1.0 / 2.0, w7 / (-1.0 + w7)) + 1.0i * num19 * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-0.5) * imag(w7)) * T(1.0, 1.0 / 2.0, (1.0 + w7) / 2.0)
                + 1.0i * num20 * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-0.5) * imag(w7)) * T(1.0, 1.0 / 2.0, (1.0 + w7) / 2.0) + (4.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * wx3)) * my_sign(real(wx3) / 2.0) * T(1.0, 1.0 / 2.0, 1.0 / 2.0 - (1.0i / 2.0) * wx3)
                + (4.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * wx3)) * my_sign((-0.5) * real(wx3)) * T(1.0, 1.0 / 2.0, 1.0 / 2.0 + (1.0i / 2.0) * wx3) + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(wx3 / (-1.0i + wx3))) * my_sign(-real(1.0 / (1.0i - wx3))) * T(1.0, 1.0 / 2.0, wx3 / (-1.0i + wx3))
                + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(wx3 / (-1.0i + wx3))) * my_sign(real(1.0 / (-1.0i + wx3))) * T(1.0, 1.0 / 2.0, wx3 / (-1.0i + wx3)) + (4.0 * 1.0i) * num16 * M_PI * power_of<2>(log(wx3 / (1.0i + wx3))) * my_sign(-real(1.0 / (1.0i + wx3))) * T(1.0, 1.0 / 2.0, wx3 / (1.0i + wx3))
                + (4.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * wx4)) * my_sign(real(wx4) / 2.0) * T(1.0, 1.0 / 2.0, 1.0 / 2.0 - (1.0i / 2.0) * wx4) + (4.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * wx4)) * my_sign((-0.5) * real(wx4)) * T(1.0, 1.0 / 2.0, 1.0 / 2.0 + (1.0i / 2.0) * wx4)
                + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(wx4 / (-1.0i + wx4))) * my_sign(-real(1.0 / (1.0i - wx4))) * T(1.0, 1.0 / 2.0, wx4 / (-1.0i + wx4)) + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(wx4 / (-1.0i + wx4))) * my_sign(real(1.0 / (-1.0i + wx4))) * T(1.0, 1.0 / 2.0, wx4 / (-1.0i + wx4))
                + (4.0 * 1.0i) * num16 * M_PI * power_of<2>(log(wx4 / (1.0i + wx4))) * my_sign(-real(1.0 / (1.0i + wx4))) * T(1.0, 1.0 / 2.0, wx4 / (1.0i + wx4)) - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * wx3)) * my_sign(real(wx3) / 2.0) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx3)
                - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * wx3)) * my_sign((-0.5) * real(wx3)) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx3) - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * wx4)) * my_sign(real(wx4) / 2.0) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx4)
                - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * wx4)) * my_sign((-0.5) * real(wx4)) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx4) + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx3 + xd) / (1.0i + wx3))) * my_sign(imag((1.0i - xd) / (1.0i + wx3))) * 
                T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, (wx3 + xd) / (1.0i + wx3)) + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx4 + xd) / (1.0i + wx4))) * my_sign(imag((1.0i - xd) / (1.0i + wx4))) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, (wx4 + xd) / (1.0i + wx4))
                - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * wx3)) * my_sign(real(wx3) / 2.0) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx3) - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * wx3)) * my_sign((-0.5) * real(wx3)) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx3)
                - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 - (1.0i / 2.0) * wx4)) * my_sign(real(wx4) / 2.0) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx4) - (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log(1.0 / 2.0 + (1.0i / 2.0) * wx4)) * my_sign((-0.5) * real(wx4)) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx4)
                + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx3 - xd) / (1.0i + wx3))) * my_sign(imag((1.0i + xd) / (1.0i + wx3))) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, (wx3 - xd) / (1.0i + wx3)) + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx4 - xd) / (1.0i + wx4))) * my_sign(imag((1.0i + xd) / (1.0i + wx4))) * 
                T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, (wx4 - xd) / (1.0i + wx4)) - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx3) * log(1.0 - 1.0i * wx3) * my_sign(real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)
                - 1.0i * num16 * M_PI * power_of<2>(log(1.0 - 1.0i * wx3)) * my_sign(real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx3) * log(1.0 + 1.0i * wx3) * my_sign(-real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                - 1.0i * num16 * M_PI * power_of<2>(log(1.0 + 1.0i * wx3)) * my_sign(-real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx4) * log(1.0 - 1.0i * wx4) * my_sign(real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4)
                - 1.0i * num16 * M_PI * power_of<2>(log(1.0 - 1.0i * wx4)) * my_sign(real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx4) * log(1.0 + 1.0i * wx4) * my_sign(-real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4)
                - 1.0i * num16 * M_PI * power_of<2>(log(1.0 + 1.0i * wx4)) * my_sign(-real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx3 + xd) / (-1.0i + wx3))) * my_sign(imag((1.0i + xd) / (1.0i - wx3))) * T(1.0, (1.0 + 1.0i * xd) / 2.0, (wx3 + xd) / (-1.0i + wx3))
                + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx4 + xd) / (-1.0i + wx4))) * my_sign(imag((1.0i + xd) / (1.0i - wx4))) * T(1.0, (1.0 + 1.0i * xd) / 2.0, (wx4 + xd) / (-1.0i + wx4)) - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx3) * log(1.0 - 1.0i * wx3) * my_sign(real(wx3)) * 
                T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) - 1.0i * num16 * M_PI * power_of<2>(log(1.0 - 1.0i * wx3)) * my_sign(real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) + dilog(1.0 / 2.0 - (1.0i / 2.0) * wx3) * (-4.0 * num16 * log(wx3 / (1.0i + wx3)) + 2.0 * num16 * log((wx3 - xd) / (1.0i + wx3))
                + 2.0 * num16 * log((wx3 + xd) / (1.0i + wx3)) - (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3))
                - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx3) * log(1.0 + 1.0i * wx3) * my_sign(-real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) - 1.0i * num16 * M_PI * power_of<2>(log(1.0 + 1.0i * wx3)) * my_sign(-real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)
                + dilog(1.0 / 2.0 + (1.0i / 2.0) * wx3) * (-4.0 * num16 * log(wx3 / (-1.0i + wx3)) + 2.0 * num16 * log((wx3 - xd) / (-1.0i + wx3)) + 2.0 * num16 * log((wx3 + xd) / (-1.0i + wx3)) - (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                - (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)) - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx4) * log(1.0 - 1.0i * wx4) * my_sign(real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)
                - 1.0i * num16 * M_PI * power_of<2>(log(1.0 - 1.0i * wx4)) * my_sign(real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) + log((w5 + yd) / w5) * ((num17 * power_of<2>(log(1.0 - 1.0i * xd))) / 2.0 + (num17 * power_of<2>(log(1.0 + 1.0i * xd))) / 2.0 + 2.0 * num17 * ln2 * log((wx3 + xd) / wx3)
                + 2.0 * num17 * ln2 * log((wx4 + xd) / wx4) + log(xd) * (num17 * log((wx3 + xd) / wx3) + num17 * log((wx4 + xd) / wx4)) + log(1.0 - 1.0i * xd) * (-(num17 * ln2) - num17 * log(xd) - num17 * log((wx3 + xd) / (-1.0i + wx3))
                    - num17 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (-(num17 * ln2) - num17 * log(xd) - num17 * log((wx3 + xd) / (1.0i + wx3)) - num17 * log((wx4 + xd) / (1.0i + wx4)))
                - (2.0 * 1.0i) * num17 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * num17 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * num17 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * num17 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4))
                + log(1.0 - yd / w5) * ((num18 * power_of<2>(log(1.0 - 1.0i * xd))) / 2.0 + (num18 * power_of<2>(log(1.0 + 1.0i * xd))) / 2.0 + 2.0 * num18 * ln2 * log((wx3 + xd) / wx3) + 2.0 * num18 * ln2 * log((wx4 + xd) / wx4)
                + log(xd) * (num18 * log((wx3 + xd) / wx3) + num18 * log((wx4 + xd) / wx4)) + log(1.0 - 1.0i * xd) * (-(num18 * ln2) - num18 * log(xd) - num18 * log((wx3 + xd) / (-1.0i + wx3)) - num18 * log((wx4 + xd) / (-1.0i + wx4)))
                + log(1.0 + 1.0i * xd) * (-(num18 * ln2) - num18 * log(xd) - num18 * log((wx3 + xd) / (1.0i + wx3)) - num18 * log((wx4 + xd) / (1.0i + wx4))) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * 
                    T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * num18 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)) + dilog(1.0 / 2.0 - (1.0i / 2.0) * wx4) * (-4.0 * num16 * log(wx4 / (1.0i + wx4)) + 2.0 * num16 * log((wx4 - xd) / (1.0i + wx4))
                + 2.0 * num16 * log((wx4 + xd) / (1.0i + wx4)) - (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4))
                - (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx4) * log(1.0 + 1.0i * wx4) * my_sign(-real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) - 1.0i * num16 * M_PI * power_of<2>(log(1.0 + 1.0i * wx4)) * my_sign(-real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)
                + log(1.0 - yd / w4) * ((num17 * power_of<2>(log(1.0 - 1.0i * xd))) / 2.0 + (num17 * power_of<2>(log(1.0 + 1.0i * xd))) / 2.0 + log(1.0 + 1.0i * xd) * (-(num17 * ln2) - num17 * log((wx3 - xd) / (-1.0i + wx3)) - num17 * log((wx4 - xd) / (-1.0i + wx4)) - num17 * log(xd))
                + log(1.0 - 1.0i * xd) * (-(num17 * ln2) - num17 * log((wx3 - xd) / (1.0i + wx3)) - num17 * log((wx4 - xd) / (1.0i + wx4)) - num17 * log(xd)) + 2.0 * num17 * ln2 * log(1.0 - xd / wx3) + 2.0 * num17 * ln2 * log(1.0 - xd / wx4)
                + log(xd) * (num17 * log(1.0 - xd / wx3) + num17 * log(1.0 - xd / wx4)) - (2.0 * 1.0i) * num17 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * num17 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * num17 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * num17 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + log((w4 + yd) / w4) * ((num18 * power_of<2>(log(1.0 - 1.0i * xd))) / 2.0 + (num18 * power_of<2>(log(1.0 + 1.0i * xd))) / 2.0
                + log(1.0 + 1.0i * xd) * (-(num18 * ln2) - num18 * log((wx3 - xd) / (-1.0i + wx3)) - num18 * log((wx4 - xd) / (-1.0i + wx4)) - num18 * log(xd))
                + log(1.0 - 1.0i * xd) * (-(num18 * ln2) - num18 * log((wx3 - xd) / (1.0i + wx3)) - num18 * log((wx4 - xd) / (1.0i + wx4)) - num18 * log(xd)) + 2.0 * num18 * ln2 * log(1.0 - xd / wx3) + 2.0 * num18 * ln2 * log(1.0 - xd / wx4)
                + log(xd) * (num18 * log(1.0 - xd / wx3) + num18 * log(1.0 - xd / wx4)) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * num18 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * num18 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + log((w7 + yd) / w7) * (-2.0 * num19 * power_of<2>(log(1.0 - 1.0i * xd)) - 2.0 * num19 * power_of<2>(log(1.0 + 1.0i * xd)) - 4.0 * num19 * ln2 * log((wx3 + xd) / wx3)
                - 4.0 * num19 * ln2 * log((wx4 + xd) / wx4) + log(1.0 - 1.0i * xd) * (4.0 * num19 * ln2 + 2.0 * num19 * log((wx3 - xd) / (1.0i + wx3)) + 2.0 * num19 * log((wx4 - xd) / (1.0i + wx4)) + 4.0 * num19 * log(xd)
                    + 2.0 * num19 * log((wx3 + xd) / (-1.0i + wx3)) + 2.0 * num19 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (4.0 * num19 * ln2 + 2.0 * num19 * log((wx3 - xd) / (-1.0i + wx3)) + 2.0 * num19 * log((wx4 - xd) / (-1.0i + wx4))
                    + 4.0 * num19 * log(xd) + 2.0 * num19 * log((wx3 + xd) / (1.0i + wx3)) + 2.0 * num19 * log((wx4 + xd) / (1.0i + wx4))) - 4.0 * num19 * ln2 * log(1.0 - xd / wx3) - 4.0 * num19 * ln2 * log(1.0 - xd / wx4)
                + log(xd) * (-2.0 * num19 * log((wx3 + xd) / wx3) - 2.0 * num19 * log((wx4 + xd) / wx4) - 2.0 * num19 * log(1.0 - xd / wx3) - 2.0 * num19 * log(1.0 - xd / wx4)) + (4.0 * 1.0i) * num19 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * 
                    T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) + (4.0 * 1.0i) * num19 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * num19 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4)
                + (4.0 * 1.0i) * num19 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (4.0 * 1.0i) * num19 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                + (4.0 * 1.0i) * num19 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * num19 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)
                + (4.0 * 1.0i) * num19 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + log(1.0 - yd / w7) * (-2.0 * num20 * power_of<2>(log(1.0 - 1.0i * xd)) - 2.0 * num20 * power_of<2>(log(1.0 + 1.0i * xd)) - 4.0 * num20 * ln2 * log((wx3 + xd) / wx3)
                - 4.0 * num20 * ln2 * log((wx4 + xd) / wx4) + log(1.0 - 1.0i * xd) * (4.0 * num20 * ln2 + 2.0 * num20 * log((wx3 - xd) / (1.0i + wx3)) + 2.0 * num20 * log((wx4 - xd) / (1.0i + wx4)) + 4.0 * num20 * log(xd)
                    + 2.0 * num20 * log((wx3 + xd) / (-1.0i + wx3)) + 2.0 * num20 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (4.0 * num20 * ln2 + 2.0 * num20 * log((wx3 - xd) / (-1.0i + wx3)) + 2.0 * num20 * log((wx4 - xd) / (-1.0i + wx4))
                    + 4.0 * num20 * log(xd) + 2.0 * num20 * log((wx3 + xd) / (1.0i + wx3)) + 2.0 * num20 * log((wx4 + xd) / (1.0i + wx4))) - 4.0 * num20 * ln2 * log(1.0 - xd / wx3) - 4.0 * num20 * ln2 * log(1.0 - xd / wx4)
                + log(xd) * (-2.0 * num20 * log((wx3 + xd) / wx3) - 2.0 * num20 * log((wx4 + xd) / wx4) - 2.0 * num20 * log(1.0 - xd / wx3) - 2.0 * num20 * log(1.0 - xd / wx4)) + (4.0 * 1.0i) * num20 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * 
                    T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) + (4.0 * 1.0i) * num20 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * num20 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4)
                + (4.0 * 1.0i) * num20 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (4.0 * 1.0i) * num20 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                + (4.0 * 1.0i) * num20 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * num20 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)
                + (4.0 * 1.0i) * num20 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)) + dilog(1.0 / 2.0 + (1.0i / 2.0) * wx4) * (-4.0 * num16 * log(wx4 / (-1.0i + wx4)) + 2.0 * num16 * log((wx4 - xd) / (-1.0i + wx4))
                + 2.0 * num16 * log((wx4 + xd) / (-1.0i + wx4)) - (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) - (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4))
                + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx3 - xd) / (-1.0i + wx3))) * my_sign(imag((-1.0i + xd) / (-1.0i + wx3))) * T(1.0, ((-0.5) * 1.0i) * (1.0i + xd), (wx3 - xd) / (-1.0i + wx3)) + (2.0 * 1.0i) * num16 * M_PI * power_of<2>(log((wx4 - xd) / (-1.0i + wx4))) * my_sign(imag((-1.0i + xd) / (-1.0i + wx4))) * 
                T(1.0, ((-0.5) * 1.0i) * (1.0i + xd), (wx4 - xd) / (-1.0i + wx4)) - 1.0i * num16 * M_PI * power_of<2>(log((-1.0i + wx3) / wx3)) * my_sign(real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((-1.0i + wx3) / wx3) * my_sign(-real(wx3)) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3) - 1.0i * num16 * M_PI * power_of<2>(log((1.0i + wx3) / wx3)) * my_sign(-real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((1.0i + wx3) / wx3) * my_sign(real(wx3)) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3) - 1.0i * num16 * M_PI * power_of<2>(log((-1.0i + wx4) / wx4)) * my_sign(real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((-1.0i + wx4) / wx4) * my_sign(-real(wx4)) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4) - 1.0i * num16 * M_PI * power_of<2>(log((1.0i + wx4) / wx4)) * my_sign(-real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((1.0i + wx4) / wx4) * my_sign(real(wx4)) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4) - 1.0i * num16 * M_PI * power_of<2>(log((-1.0i + wx3) / wx3)) * my_sign(real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((-1.0i + wx3) / wx3) * my_sign(-real(wx3)) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3) - 1.0i * num16 * M_PI * power_of<2>(log((1.0i + wx3) / wx3)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((1.0i + wx3) / wx3) * my_sign(real(wx3)) * T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3) - 1.0i * num16 * M_PI * power_of<2>(log((-1.0i + wx4) / wx4)) * my_sign(real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((-1.0i + wx4) / wx4) * my_sign(-real(wx4)) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4) - 1.0i * num16 * M_PI * power_of<2>(log((1.0i + wx4) / wx4)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4)
                + (4.0 * 1.0i) * num16 * M_PI * ln2 * log((1.0i + wx4) / wx4) * my_sign(real(wx4)) * T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4) + power_of<2>(log(1.0 - 1.0i * xd)) * (-2.0 * num16 * log(-1.0i / xd) + (num16 * log(((-1.0i) * (wx3 - xd)) / ((1.0i + wx3) * xd))) / 2.0
                + (num16 * log(((-1.0i) * (wx4 - xd)) / ((1.0i + wx4) * xd))) / 2.0 - (num16 * log((wx3 + xd) / (-1.0i + wx3))) / 2.0 + (num16 * log(((-1.0i) * (wx3 + xd)) / ((-1.0i + wx3) * xd))) / 2.0 - (num16 * log((wx4 + xd) / (-1.0i + wx4))) / 2.0
                + (num16 * log(((-1.0i) * (wx4 + xd)) / ((-1.0i + wx4) * xd))) / 2.0 + 1.0i * num16 * M_PI * my_sign(real(xd)) * T(1.0, (wx3 + xd) / wx3, 1.0 - 1.0i * xd) + 1.0i * num16 * M_PI * my_sign(real(xd)) * T(1.0, (wx4 + xd) / wx4, 1.0 - 1.0i * xd)
                + 1.0i * num16 * M_PI * my_sign(real(xd)) * T(1.0, 1.0 - xd / wx3, 1.0 - 1.0i * xd) + 1.0i * num16 * M_PI * my_sign(real(xd)) * T(1.0, 1.0 - xd / wx4, 1.0 - 1.0i * xd))
                + power_of<2>(log(1.0 + 1.0i * xd)) * (-2.0 * num16 * log(1.0i / xd) + (num16 * log((1.0i * (wx3 - xd)) / ((-1.0i + wx3) * xd))) / 2.0 + (num16 * log((1.0i * (wx4 - xd)) / ((-1.0i + wx4) * xd))) / 2.0 - (num16 * log((wx3 + xd) / (1.0i + wx3))) / 2.0
                + (num16 * log((1.0i * (wx3 + xd)) / ((1.0i + wx3) * xd))) / 2.0 - (num16 * log((wx4 + xd) / (1.0i + wx4))) / 2.0 + (num16 * log((1.0i * (wx4 + xd)) / ((1.0i + wx4) * xd))) / 2.0 + 1.0i * num16 * M_PI * my_sign(-real(xd)) * T(1.0, (wx3 + xd) / wx3, 1.0 + 1.0i * xd)
                + 1.0i * num16 * M_PI * my_sign(-real(xd)) * T(1.0, (wx4 + xd) / wx4, 1.0 + 1.0i * xd) + 1.0i * num16 * M_PI * my_sign(-real(xd)) * T(1.0, 1.0 - xd / wx3, 1.0 + 1.0i * xd) + 1.0i * num16 * M_PI * my_sign(-real(xd)) * T(1.0, 1.0 - xd / wx4, 1.0 + 1.0i * xd))
                + 1.0i * num18 * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-0.5) * imag(w4)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0) + 1.0i * num18 * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0)
                - 1.0i * num19 * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-0.5) * imag(w7)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0) + ln1pyd * (-(num14 * power_of<2>(log(1.0 - 1.0i * xd))) - num14 * power_of<2>(log(1.0 + 1.0i * xd)) - 2.0 * num14 * ln2 * log((wx3 + xd) / wx3)
                - 2.0 * num14 * ln2 * log((wx4 + xd) / wx4) + log(1.0 - 1.0i * xd) * (-2.0 * num14 * ln2 + num14 * log((wx3 - xd) / (1.0i + wx3)) + num14 * log((wx4 - xd) / (1.0i + wx4)) + 2.0 * num14 * log(xd)
                    + num14 * log((wx3 + xd) / (-1.0i + wx3)) + num14 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (-2.0 * num14 * ln2 + num14 * log((wx3 - xd) / (-1.0i + wx3)) + num14 * log((wx4 - xd) / (-1.0i + wx4))
                    + 2.0 * num14 * log(xd) + num14 * log((wx3 + xd) / (1.0i + wx3)) + num14 * log((wx4 + xd) / (1.0i + wx4))) - 2.0 * num14 * ln2 * log(1.0 - xd / wx3) - 2.0 * num14 * ln2 * log(1.0 - xd / wx4)
                + log(xd) * (-(num14 * log((wx3 + xd) / wx3)) - num14 * log((wx4 + xd) / wx4) - num14 * log(1.0 - xd / wx3) - num14 * log(1.0 - xd / wx4))
                + (-2.0 * num18 * ln2 + num18 * log(1.0 - 1.0i * xd) + num18 * log(1.0 + 1.0i * xd) - num18 * log(xd)) * log((w5 - yd) / (1.0 + w5)) + (-2.0 * num18 * ln2 + num18 * log(1.0 - 1.0i * xd) + num18 * log(1.0 + 1.0i * xd) - num18 * log(xd)) * 
                    log((w4 + yd) / (-1.0 + w4)) + (2.0 * num17 * ln2 - num17 * log(1.0 - 1.0i * xd) - num17 * log(1.0 + 1.0i * xd) + num17 * log(xd)) * log((w5 + yd) / (-1.0 + w5))
                + (4.0 * num19 * ln2 - 2.0 * num19 * log(1.0 - 1.0i * xd) - 2.0 * num19 * log(1.0 + 1.0i * xd) + 2.0 * num19 * log(xd)) * log((w7 + yd) / (-1.0 + w7)) + (2.0 * 1.0i) * num14 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * num14 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * num14 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * num14 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * num14 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * num14 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * num14 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * num14 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * num18 * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0)
                - (2.0 * 1.0i) * num18 * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0) + (2.0 * 1.0i) * num19 * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0))
                - 1.0i * num18 * M_PI * power_of<2>(log((w5 - yd) / (1.0 + w5))) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)) - 1.0i * num18 * M_PI * power_of<2>(log((w4 + yd) / (-1.0 + w4))) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * 
                T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)) + 1.0i * num19 * M_PI * power_of<2>(log((w7 + yd) / (-1.0 + w7))) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7))
                - (4.0 * 1.0i) * num17 * M_PI * ln2 * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 - w4) + (2.0 * 1.0i) * num17 * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4)
                + (4.0 * 1.0i) * num18 * M_PI * ln2 * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 + w4) - 1.0i * num18 * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4)
                + (4.0 * 1.0i) * num18 * M_PI * ln2 * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 - w5) - 1.0i * num18 * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5)
                - (4.0 * 1.0i) * num17 * M_PI * ln2 * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 + w5) + (2.0 * 1.0i) * num17 * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5)
                + (8.0 * 1.0i) * num20 * M_PI * ln2 * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 - w7) - (2.0 * 1.0i) * num20 * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7)
                - (8.0 * 1.0i) * num19 * M_PI * ln2 * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 + w7) + 1.0i * num19 * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7)
                + 1.0i * num17 * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-0.5) * imag(w4)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) + 1.0i * num17 * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0)
                - 1.0i * num20 * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-0.5) * imag(w7)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0) + ln1myd * (-(num15 * power_of<2>(log(1.0 - 1.0i * xd))) - num15 * power_of<2>(log(1.0 + 1.0i * xd)) - 2.0 * num15 * ln2 * log((wx3 + xd) / wx3)
                - 2.0 * num15 * ln2 * log((wx4 + xd) / wx4) + log(1.0 - 1.0i * xd) * (-2.0 * num15 * ln2 + num15 * log((wx3 - xd) / (1.0i + wx3)) + num15 * log((wx4 - xd) / (1.0i + wx4)) + 2.0 * num15 * log(xd)
                    + num15 * log((wx3 + xd) / (-1.0i + wx3)) + num15 * log((wx4 + xd) / (-1.0i + wx4))) + log(1.0 + 1.0i * xd) * (-2.0 * num15 * ln2 + num15 * log((wx3 - xd) / (-1.0i + wx3)) + num15 * log((wx4 - xd) / (-1.0i + wx4))
                    + 2.0 * num15 * log(xd) + num15 * log((wx3 + xd) / (1.0i + wx3)) + num15 * log((wx4 + xd) / (1.0i + wx4))) - 2.0 * num15 * ln2 * log(1.0 - xd / wx3) - 2.0 * num15 * ln2 * log(1.0 - xd / wx4)
                + log(xd) * (-(num15 * log((wx3 + xd) / wx3)) - num15 * log((wx4 + xd) / wx4) - num15 * log(1.0 - xd / wx3) - num15 * log(1.0 - xd / wx4))
                + (-2.0 * num17 * ln2 + num17 * log(1.0 - 1.0i * xd) + num17 * log(1.0 + 1.0i * xd) - num17 * log(xd)) * log((w4 - yd) / (-1.0 + w4)) + (2.0 * num18 * ln2 - num18 * log(1.0 - 1.0i * xd) - num18 * log(1.0 + 1.0i * xd) + num18 * log(xd)) * 
                    log((w5 - yd) / (-1.0 + w5)) + (4.0 * num20 * ln2 - 2.0 * num20 * log(1.0 - 1.0i * xd) - 2.0 * num20 * log(1.0 + 1.0i * xd) + 2.0 * num20 * log(xd)) * log((w7 - yd) / (-1.0 + w7)) - num17 * log((w4 - yd) / (1.0 + w4)) * log((1.0 + yd) / 2.0)
                + (2.0 * num18 * ln2 - num18 * log(1.0 - 1.0i * xd) - num18 * log(1.0 + 1.0i * xd) + num18 * log(xd)) * log((w4 + yd) / (1.0 + w4)) + log((1.0 + yd) / 2.0) * (num20 * log((w7 - yd) / (1.0 + w7)) - num17 * log((w5 + yd) / (-1.0 + w5)))
                + (-2.0 * num17 * ln2 + num17 * log(1.0 - 1.0i * xd) + num17 * log(1.0 + 1.0i * xd) - num17 * log(xd)) * log((w5 + yd) / (1.0 + w5)) + (-4.0 * num19 * ln2 + 2.0 * num19 * log(1.0 - 1.0i * xd) + 2.0 * num19 * log(1.0 + 1.0i * xd) - 2.0 * num19 * log(xd)) * 
                    log((w7 + yd) / (1.0 + w7)) + (2.0 * 1.0i) * num15 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * num15 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * num15 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * num15 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * num15 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * num15 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * num15 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * num15 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * num17 * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) - (2.0 * 1.0i) * num17 * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0)
                + (2.0 * 1.0i) * num20 * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0)) - 1.0i * num17 * M_PI * power_of<2>(log((w4 - yd) / (-1.0 + w4))) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4))
                + 1.0i * num20 * M_PI * power_of<2>(log((w7 - yd) / (-1.0 + w7))) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7)) - 1.0i * num17 * M_PI * power_of<2>(log((w5 + yd) / (1.0 + w5))) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * 
                T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)) - (4.0 * 1.0i) * num18 * M_PI * ln2 * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4) + (2.0 * 1.0i) * num18 * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4)
                + dilog((1.0 - w4) / 2.0) * (num17 * log(w4 / (-1.0 + w4)) + num18 * log(w4 / (-1.0 + w4)) - num17 * log((w4 - yd) / (-1.0 + w4)) - num18 * log((w4 + yd) / (-1.0 + w4)) + (2.0 * 1.0i) * num17 * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4)
                + (2.0 * 1.0i) * num18 * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4)) + (4.0 * 1.0i) * num17 * M_PI * ln2 * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 + w4) - 1.0i * num17 * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4)
                + (4.0 * 1.0i) * num17 * M_PI * ln2 * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 - w5) - 1.0i * num17 * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)
                - (4.0 * 1.0i) * num18 * M_PI * ln2 * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5) + (2.0 * 1.0i) * num18 * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5)
                + dilog((1.0 + w5) / 2.0) * (num17 * log(w5 / (1.0 + w5)) + num18 * log(w5 / (1.0 + w5)) - num18 * log((w5 - yd) / (1.0 + w5)) - num17 * log((w5 + yd) / (1.0 + w5)) + (2.0 * 1.0i) * num17 * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5)
                + (2.0 * 1.0i) * num18 * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5)) + (8.0 * 1.0i) * num19 * M_PI * ln2 * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7)
                - (2.0 * 1.0i) * num19 * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7) + dilog((1.0 - w7) / 2.0) * (-(num19 * log(w7 / (-1.0 + w7))) - num20 * log(w7 / (-1.0 + w7)) + num20 * log((w7 - yd) / (-1.0 + w7))
                + num19 * log((w7 + yd) / (-1.0 + w7)) - (2.0 * 1.0i) * num20 * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (2.0 * 1.0i) * num19 * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7))
                + li2half * (-(num17 * log(w4 / (-1.0 + w4))) - num18 * log(w4 / (-1.0 + w4)) - num17 * log(w5 / (1.0 + w5)) - num18 * log(w5 / (1.0 + w5)) + num19 * log(w7 / (-1.0 + w7)) + num20 * log(w7 / (-1.0 + w7))
                + 4.0 * num16 * log(wx3 / (-1.0i + wx3)) + 4.0 * num16 * log(wx3 / (1.0i + wx3)) + 4.0 * num16 * log(wx4 / (-1.0i + wx4)) + 4.0 * num16 * log(wx4 / (1.0i + wx4)) - 2.0 * num16 * log((wx3 - xd) / (-1.0i + wx3)) - 2.0 * num16 * log((wx3 - xd) / (1.0i + wx3))
                - 2.0 * num16 * log((wx4 - xd) / (-1.0i + wx4)) - 2.0 * num16 * log((wx4 - xd) / (1.0i + wx4)) - 2.0 * num16 * log((wx3 + xd) / (-1.0i + wx3)) - 2.0 * num16 * log((wx3 + xd) / (1.0i + wx3)) - 2.0 * num16 * log((wx4 + xd) / (-1.0i + wx4))
                - 2.0 * num16 * log((wx4 + xd) / (1.0i + wx4)) - 4.0 * num15 * ln1myd + num17 * log((w4 - yd) / (-1.0 + w4)) + num18 * log((w5 - yd) / (1.0 + w5)) - num20 * log((w7 - yd) / (-1.0 + w7)) - 4.0 * num14 * ln1pyd
                + num18 * log((w4 + yd) / (-1.0 + w4)) + 2.0 * num18 * log((w4 + yd) / w4) + 2.0 * num17 * log((w5 + yd) / w5) + num17 * log((w5 + yd) / (1.0 + w5)) - num19 * log((w7 + yd) / (-1.0 + w7)) - 8.0 * num19 * log((w7 + yd) / w7)
                + 2.0 * num17 * log(1.0 - yd / w4) + 2.0 * num18 * log(1.0 - yd / w5) - 8.0 * num20 * log(1.0 - yd / w7) + (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) + (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                + (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) + (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) + (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                + (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) + (4.0 * 1.0i) * num16 * M_PI * my_sign(real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) + (4.0 * 1.0i) * num16 * M_PI * my_sign(-real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * num17 * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) - (2.0 * 1.0i) * num17 * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (2.0 * 1.0i) * num20 * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7)
                - (2.0 * 1.0i) * num18 * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4) - (2.0 * 1.0i) * num18 * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) + (2.0 * 1.0i) * num19 * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7))
                - (8.0 * 1.0i) * num20 * M_PI * ln2 * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 + w7) + 1.0i * num20 * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7)
                + log(xd) * (num16 * log((1.0i - xd) / (1.0i + wx4)) * log((wx4 + xd) / wx4) + num16 * log((1.0i + xd) / (1.0i - wx4)) * log((wx4 + xd) / wx4) + num16 * log((-1.0i + xd) / (-1.0i + wx3)) * log(1.0 - xd / wx3)
                + num16 * log((1.0i + xd) / (1.0i + wx3)) * log(1.0 - xd / wx3) + num16 * log((-1.0i + xd) / (-1.0i + wx4)) * log(1.0 - xd / wx4) + num16 * log((1.0i + xd) / (1.0i + wx4)) * log(1.0 - xd / wx4)
                + (2.0 * 1.0i) * num16 * M_PI * log((-1.0i + wx3) / wx3) * my_sign(-imag(wx3 * xdinv)) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3) + (2.0 * 1.0i) * num16 * M_PI * log((1.0i + wx3) / wx3) * my_sign(-imag(wx3 * xdinv)) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3)
                + (2.0 * 1.0i) * num16 * M_PI * log((-1.0i + wx4) / wx4) * my_sign(-imag(wx4 * xdinv)) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4) + (2.0 * 1.0i) * num16 * M_PI * log((1.0i + wx4) / wx4) * my_sign(-imag(wx4 * xdinv)) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4)
                + (2.0 * 1.0i) * num16 * M_PI * log((-1.0i + wx3) / wx3) * my_sign(imag(wx3 * xdinv)) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3) + (2.0 * 1.0i) * num16 * M_PI * log((1.0i + wx3) / wx3) * my_sign(imag(wx3 * xdinv)) * T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3)
                + (2.0 * 1.0i) * num16 * M_PI * log((-1.0i + wx4) / wx4) * my_sign(imag(wx4 * xdinv)) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4) + (2.0 * 1.0i) * num16 * M_PI * log((1.0i + wx4) / wx4) * my_sign(imag(wx4 * xdinv)) * T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4)
                - (2.0 * 1.0i) * num17 * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 - w4) + (2.0 * 1.0i) * num18 * M_PI * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 + w4)
                + (2.0 * 1.0i) * num18 * M_PI * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 - w5) - (2.0 * 1.0i) * num17 * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 + w5)
                + (4.0 * 1.0i) * num20 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 - w7) - (4.0 * 1.0i) * num19 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 + w7)
                - (2.0 * 1.0i) * num18 * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4) + (2.0 * 1.0i) * num17 * M_PI * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 + w4)
                + (2.0 * 1.0i) * num17 * M_PI * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 - w5) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5)
                + (4.0 * 1.0i) * num19 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7) - (4.0 * 1.0i) * num20 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 + w7))
                + log(1.0 - 1.0i * xd) * (log(1.0 + 1.0i * xd) * (-(num16 * log((wx3 + xd) / (-1.0i + wx3))) - num16 * log((wx3 + xd) / (1.0i + wx3)) - num16 * log((wx4 + xd) / (-1.0i + wx4)) - num16 * log((wx4 + xd) / (1.0i + wx4)))
                + (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx3) * my_sign(2.0 * real(1.0 / (1.0i + xd))) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx3) + (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx3) * my_sign(2.0 * real(1.0 / (1.0i + xd))) * 
                    T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx3) + (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx4) * my_sign(2.0 * real(1.0 / (1.0i + xd))) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx4)
                + (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx4) * my_sign(2.0 * real(1.0 / (1.0i + xd))) * T(1.0, 1.0 / 2.0 + (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx4) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * num17 * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 - w4)
                - (2.0 * 1.0i) * num18 * M_PI * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 + w4) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 - w5)
                + (2.0 * 1.0i) * num17 * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 + w5) - (4.0 * 1.0i) * num20 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 - w7)
                + (4.0 * 1.0i) * num19 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 + w7) + (2.0 * 1.0i) * num18 * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4)
                - (2.0 * 1.0i) * num17 * M_PI * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 + w4) - (2.0 * 1.0i) * num17 * M_PI * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 - w5)
                + (2.0 * 1.0i) * num18 * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5) - (4.0 * 1.0i) * num19 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7)
                + (4.0 * 1.0i) * num20 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 + w7)) + log(1.0 + 1.0i * xd) * ((4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx3) * my_sign(2.0 * real(1.0 / (1.0i - xd))) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx3)
                + (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx3) * my_sign(2.0 * real(1.0 / (1.0i - xd))) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx3) + (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 - (1.0i / 2.0) * wx4) * my_sign(2.0 * real(1.0 / (1.0i - xd))) * 
                    T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 - (1.0i / 2.0) * wx4) + (4.0 * 1.0i) * num16 * M_PI * log(1.0 / 2.0 + (1.0i / 2.0) * wx4) * my_sign(2.0 * real(1.0 / (1.0i - xd))) * T(1.0, 1.0 / 2.0 - (1.0i / 2.0) * xd, 1.0 / 2.0 + (1.0i / 2.0) * wx4)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * num16 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * num16 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * num17 * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 - w4) - (2.0 * 1.0i) * num18 * M_PI * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 + w4)
                - (2.0 * 1.0i) * num18 * M_PI * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 - w5) + (2.0 * 1.0i) * num17 * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 + w5)
                - (4.0 * 1.0i) * num20 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 - w7) + (4.0 * 1.0i) * num19 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 + w7)
                + (2.0 * 1.0i) * num18 * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4) - (2.0 * 1.0i) * num17 * M_PI * log(1.0 + w4) * my_sign(-imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 + w4)
                - (2.0 * 1.0i) * num17 * M_PI * log(1.0 - w5) * my_sign(imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 - w5) + (2.0 * 1.0i) * num18 * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5)
                - (4.0 * 1.0i) * num19 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7) + (4.0 * 1.0i) * num20 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 + w7))
                + power_of<2>(log((w4 + yd) / w4)) * (1.0i * num18 * M_PI * my_sign(-imag(yd / w4)) * T(1.0, 1.0 - yd, (w4 + yd) / w4) - 1.0i * num18 * M_PI * my_sign(-imag(yd / w4)) * T(1.0, 1.0 + yd, (w4 + yd) / w4))
                + power_of<2>(log((w5 + yd) / w5)) * ((-1.0i) * num17 * M_PI * my_sign(-imag(yd / w5)) * T(1.0, 1.0 - yd, (w5 + yd) / w5) + 1.0i * num17 * M_PI * my_sign(-imag(yd / w5)) * T(1.0, 1.0 + yd, (w5 + yd) / w5))
                + power_of<2>(log((w7 + yd) / w7)) * ((-1.0i) * num19 * M_PI * my_sign(-imag(yd / w7)) * T(1.0, 1.0 - yd, (w7 + yd) / w7) + 1.0i * num19 * M_PI * my_sign(-imag(yd / w7)) * T(1.0, 1.0 + yd, (w7 + yd) / w7))
                + power_of<2>(log(1.0 - yd / w4)) * ((-1.0i) * num17 * M_PI * my_sign(imag(yd / w4)) * T(1.0, 1.0 - yd, 1.0 - yd / w4) + 1.0i * num17 * M_PI * my_sign(imag(yd / w4)) * T(1.0, 1.0 + yd, 1.0 - yd / w4))
                + power_of<2>(log(1.0 - yd / w5)) * (1.0i * num18 * M_PI * my_sign(imag(yd / w5)) * T(1.0, 1.0 - yd, 1.0 - yd / w5) - 1.0i * num18 * M_PI * my_sign(imag(yd / w5)) * T(1.0, 1.0 + yd, 1.0 - yd / w5))
                + power_of<2>(log(1.0 - yd / w7)) * (1.0i * num20 * M_PI * my_sign(imag(yd / w7)) * T(1.0, 1.0 - yd, 1.0 - yd / w7) - 1.0i * num20 * M_PI * my_sign(imag(yd / w7)) * T(1.0, 1.0 + yd, 1.0 - yd / w7))) / denom3;

            return result;
        }
    }
}