/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2023 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/polylog.hh>
#include <eos/maths/multiplepolylog-li22.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>
#include <eos/utils/stringify.hh>

#include <cmath>
#include <complex>

namespace eos
{
    using std::complex;
    using std::log;
    using std::sqrt;
    using std::real;

    namespace agv_2019a
    {
        complex<double> f17b(const CharmLoopsParameters & clp)
        {
            // f17b = - f27b / (2 * N_c)
            return - f27b(clp) / 6.0;
        }

        complex<double> f27b(const CharmLoopsParameters & clp)
        {
            const double lnmuhat = log(clp.muhat);
            const complex<double> xb = clp.xb;
            const complex<double> yb = clp.yb;

            // Powers of xd and yd

            const complex<double> xb2 = power_of<2>(xb);
            const complex<double> xb3 = power_of<3>(xb);
            const complex<double> xb4 = power_of<4>(xb);
            const complex<double> xb5 = power_of<5>(xb);
            const complex<double> xb6 = power_of<6>(xb);
            const complex<double> xb7 = power_of<7>(xb);
            const complex<double> xb8 = power_of<8>(xb);
            const complex<double> xb9 = power_of<9>(xb);
            const complex<double> xb10 = power_of<10>(xb);
            const complex<double> xb11 = power_of<11>(xb);
            const complex<double> xb12 = power_of<12>(xb);
            const complex<double> xb13 = power_of<13>(xb);
            const complex<double> xb14 = power_of<14>(xb);
            const complex<double> xb16 = power_of<16>(xb);

            const complex<double> yb2 = power_of<2>(yb);
            const complex<double> yb3 = power_of<3>(yb);
            const complex<double> yb4 = power_of<4>(yb);
            const complex<double> yb5 = power_of<5>(yb);
            const complex<double> yb6 = power_of<6>(yb);
            const complex<double> yb7 = power_of<7>(yb);
            const complex<double> yb8 = power_of<8>(yb);

            // weights appearing in the GPLs [AGV:2019A] p. 34

            const double wx3 = 2.0 + sqrt(3);
            const double wx4 = 2.0 - sqrt(3);

            const complex<double> w4 = (2.0 * xb) / power_of<2>(1.0 - xb);
            const complex<double> w5 = (2.0 * xb) / power_of<2>(1.0 + xb);
            const complex<double> w7 = (8.0 * xb2) / (1.0 - 6.0 * xb2 + xb4);

            const double wx3inv = wx4;
            const double wx4inv = wx3;

            const complex<double> w4inv = 1.0 / w4;
            const complex<double> w5inv = 1.0 / w5;
            const complex<double> w7inv = 1.0 / w7;

            const complex<double> tb = (2.0 * (-2.0 * xb2 + 2.0 * xb2 * yb + sqrt(2) * xb2 * (1.0 + yb) * sqrt((2.0 * xb4 - xb2 * yb + 2.0 * xb4 * yb - xb6 * yb + xb2 * yb2 + 4.0 * xb4 * yb2 + xb6 * yb2) / (xb4 * power_of<2>(1.0 + yb))))) / (-1.0 + 6.0 * xb2 - xb4 + yb + 2.0 * xb2 * yb + xb4 * yb);
            const complex<double> vb = (2.0 * (-2.0 * xb2 - 2.0 * xb2 * yb + sqrt(2) * xb2 * (1.0 - yb) * sqrt((2.0 * xb4 + xb2 * yb - 2.0 * xb4 * yb + xb6 * yb + xb2 * yb2 + 4.0 * xb4 * yb2 + xb6 * yb2) / (xb4 * power_of<2>(1.0 - yb))))) / (1.0 - 6.0 * xb2 + xb4 + yb + 2.0 * xb2 * yb + xb4 * yb);

            // Logs of yb

            const complex<double> ln1pyb = log(1.0 + yb);
            const complex<double> ln1myb = log(1.0 - yb);

            // Polynomials in xb and yb

            const complex<double> num1 = power_of<2>(1.0 + xb2) * (-1.0 + yb) * (1.0 + yb) * (-1.0 + 3.0 * yb2 + xb4 * (-1.0 + 3.0 * yb2) + xb2 * (-2.0 + 22.0 * yb2));
            const complex<double> num2 = 9.0 * yb2 * power_of<2>(-1.0 + yb2) + 9.0 * xb8 * yb2 * power_of<2>(-1.0 + yb2) + 48.0 * xb2 * yb2 * (3.0 - 13.0 * yb2 + 10.0 * yb4) + 48.0 * xb6 * yb2 * (3.0 - 13.0 * yb2 + 10.0 * yb4)
                + 2.0 * xb4 * (40.0 + 447.0 * yb2 + 762.0 * yb4 + 1311.0 * yb6);
            const complex<double> num3 = power_of<2>(1.0i + xb) * (-1.0 + yb) * (1.0 + yb) * (-8.0 * tb * xb2 * (-4.0 * xb2 * (-1.0 - 6.0 * yb - 9.0 * yb2 - 12.0 * yb3 + 45.0 * yb4 + 18.0 * yb5 - 3.0 * yb6
                     + xb10 * (1.0 + 6.0 * yb + 9.0 * yb2 + 12.0 * yb3 - 45.0 * yb4 - 18.0 * yb5 + 3.0 * yb6) + 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6)
                     + 2.0 * 1.0i * xb9 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) + xb2 * (45.0 - 18.0 * yb - 219.0 * yb2 + 28.0 * yb3 + 567.0 * yb4 + 246.0 * yb5 + 215.0 * yb6)
                     - xb8 * (45.0 - 18.0 * yb - 219.0 * yb2 + 28.0 * yb3 + 567.0 * yb4 + 246.0 * yb5 + 215.0 * yb6) - 8.0 * 1.0i * xb3 * (13.0 + 6.0 * yb - 31.0 * yb2 - 4.0 * yb3 + 1103.0 * yb4 - 66.0 * yb5 + 291.0 * yb6)
                     - 8.0 * 1.0i * xb7 * (13.0 + 6.0 * yb - 31.0 * yb2 - 4.0 * yb3 + 1103.0 * yb4 - 66.0 * yb5 + 291.0 * yb6) - 4.0 * 1.0i * xb5 * (51.0 + 18.0 * yb + 43.0 * yb2 - 28.0 * yb3 + 2569.0 * yb4 - 246.0 * yb5 + 825.0 * yb6)
                     + 2.0 * xb4 * (23.0 - 6.0 * yb + 279.0 * yb2 + 20.0 * yb3 + 1029.0 * yb4 + 114.0 * yb5 + 1005.0 * yb6) - 2.0 * xb6 * (23.0 - 6.0 * yb + 279.0 * yb2 + 20.0 * yb3 + 1029.0 * yb4 + 114.0 * yb5 + 1005.0 * yb6))
                   - 8.0 * vb * xb2 * (1.0 + 27.0 * yb2 - 57.0 * yb4 - 3.0 * yb6 + xb10 * (-1.0 - 27.0 * yb2 + 57.0 * yb4 + 3.0 * yb6) - 2.0 * 1.0i * xb * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6)
                     - 2.0 * 1.0i * xb9 * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6) + 8.0 * 1.0i * xb3 * (7.0 + 5.0 * yb2 + 833.0 * yb4 + 211.0 * yb6) + 8.0 * 1.0i * xb7 * (7.0 + 5.0 * yb2 + 833.0 * yb4 + 211.0 * yb6)
                     + 4.0 * 1.0i * xb5 * (27.0 + 41.0 * yb2 + 1757.0 * yb4 + 383.0 * yb6) - xb2 * (69.0 - 345.0 * yb2 + 1635.0 * yb4 + 529.0 * yb6) + xb8 * (69.0 - 345.0 * yb2 + 1635.0 * yb4 + 529.0 * yb6)
                     - 2.0 * xb4 * (35.0 + 353.0 * yb2 + 1301.0 * yb4 + 1287.0 * yb6) + xb6 * (70.0 + 706.0 * yb2 + 2602.0 * yb4 + 2574.0 * yb6))
                   + power_of<2>(vb) * (-1.0 - 6.0 * yb - 9.0 * yb2 - 12.0 * yb3 + 45.0 * yb4 + 18.0 * yb5 - 3.0 * yb6 + xb14 * (1.0 + 6.0 * yb + 9.0 * yb2 + 12.0 * yb3 - 45.0 * yb4 - 18.0 * yb5 + 3.0 * yb6)
                     + 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) + 2.0 * 1.0i * xb13 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6)
                     + xb2 * (47.0 + 42.0 * yb - 345.0 * yb2 + 148.0 * yb3 + 573.0 * yb4 + 66.0 * yb5 + 269.0 * yb6) - xb12 * (47.0 + 42.0 * yb - 345.0 * yb2 + 148.0 * yb3 + 573.0 * yb4 + 66.0 * yb5 + 269.0 * yb6)
                     - 4.0 * 1.0i * xb3 * (25.0 - 18.0 * yb + 49.0 * yb2 - 68.0 * yb3 + 2107.0 * yb4 - 42.0 * yb5 + 603.0 * yb6) - 4.0 * 1.0i * xb11 * (25.0 - 18.0 * yb + 49.0 * yb2 - 68.0 * yb3 + 2107.0 * yb4 - 42.0 * yb5 + 603.0 * yb6)
                     - xb6 * (-99.0 - 114.0 * yb + 709.0 * yb2 + 412.0 * yb3 + 1255.0 * yb4 + 2262.0 * yb5 + 1303.0 * yb6) + xb8 * (-99.0 - 114.0 * yb + 709.0 * yb2 + 412.0 * yb3 + 1255.0 * yb4 + 2262.0 * yb5 + 1303.0 * yb6)
                     + 3.0 * xb4 * (49.0 + 54.0 * yb - 7.0 * yb2 - 84.0 * yb3 + 3171.0 * yb4 - 738.0 * yb5 + 1363.0 * yb6) - 3.0 * xb10 * (49.0 + 54.0 * yb - 7.0 * yb2 - 84.0 * yb3 + 3171.0 * yb4 - 738.0 * yb5 + 1363.0 * yb6)
                     + 8.0 * 1.0i * xb7 * (121.0 + 78.0 * yb + 113.0 * yb2 - 132.0 * yb3 + 3611.0 * yb4 - 1098.0 * yb5 + 2011.0 * yb6) + 2.0 * 1.0i * xb5 * (193.0 + 198.0 * yb - 1447.0 * yb2 - 116.0 * yb3 + 12275.0 * yb4 - 2130.0 * yb5
                       + 3283.0 * yb6) + 2.0 * 1.0i * xb9 * (193.0 + 198.0 * yb - 1447.0 * yb2 - 116.0 * yb3 + 12275.0 * yb4 - 2130.0 * yb5 + 3283.0 * yb6)))
                 - 4.0 * xb2 * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 9.0 * yb2 + 12.0 * yb3 + 45.0 * yb4 - 18.0 * yb5 - 3.0 * yb6 + xb8 * (-45.0 - 18.0 * yb + 219.0 * yb2 + 28.0 * yb3 - 567.0 * yb4 + 246.0 * yb5 - 215.0 * yb6)
                     + xb10 * (1.0 - 6.0 * yb + 9.0 * yb2 - 12.0 * yb3 - 45.0 * yb4 + 18.0 * yb5 + 3.0 * yb6) + 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6)
                     + 2.0 * 1.0i * xb9 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) + xb2 * (45.0 + 18.0 * yb - 219.0 * yb2 - 28.0 * yb3 + 567.0 * yb4 - 246.0 * yb5 + 215.0 * yb6)
                     - 8.0 * 1.0i * xb3 * (13.0 - 6.0 * yb - 31.0 * yb2 + 4.0 * yb3 + 1103.0 * yb4 + 66.0 * yb5 + 291.0 * yb6) - 8.0 * 1.0i * xb7 * (13.0 - 6.0 * yb - 31.0 * yb2 + 4.0 * yb3 + 1103.0 * yb4 + 66.0 * yb5 + 291.0 * yb6)
                     - 4.0 * 1.0i * xb5 * (51.0 - 18.0 * yb + 43.0 * yb2 + 28.0 * yb3 + 2569.0 * yb4 + 246.0 * yb5 + 825.0 * yb6) + 2.0 * xb4 * (23.0 + 6.0 * yb + 279.0 * yb2 - 20.0 * yb3 + 1029.0 * yb4 - 114.0 * yb5 + 1005.0 * yb6)
                     - 2.0 * xb6 * (23.0 + 6.0 * yb + 279.0 * yb2 - 20.0 * yb3 + 1029.0 * yb4 - 114.0 * yb5 + 1005.0 * yb6)) - 4.0 * xb2 * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb10 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6)
                     - 3.0 * xb2 * (-7.0 + 31.0 * yb2 + 167.0 * yb4 + 33.0 * yb6) + 3.0 * xb8 * (-7.0 + 31.0 * yb2 + 167.0 * yb4 + 33.0 * yb6) + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6)
                     + 2.0 * 1.0i * xb9 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) - 8.0 * 1.0i * xb3 * (19.0 - 67.0 * yb2 + 1373.0 * yb4 + 371.0 * yb6) - 8.0 * 1.0i * xb7 * (19.0 - 67.0 * yb2 + 1373.0 * yb4 + 371.0 * yb6)
                     + 2.0 * xb4 * (11.0 + 205.0 * yb2 + 757.0 * yb4 + 723.0 * yb6) - 2.0 * xb6 * (11.0 + 205.0 * yb2 + 757.0 * yb4 + 723.0 * yb6) - 4.0 * 1.0i * xb5 * (75.0 + 45.0 * yb2 + 3381.0 * yb4 + 1267.0 * yb6))
                   + power_of<2>(vb) * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb10 * (-171.0 - 144.0 * yb + 403.0 * yb2 + 224.0 * yb3 - 11093.0 * yb4 + 1968.0 * yb5 - 4147.0 * yb6)
                     + xb8 * (-147.0 - 96.0 * yb + 139.0 * yb2 + 320.0 * yb3 + 691.0 * yb4 + 1824.0 * yb5 - 75.0 * yb6) + xb2 * (23.0 + 48.0 * yb - 255.0 * yb2 + 96.0 * yb3 - 471.0 * yb4 - 144.0 * yb5 - 33.0 * yb6)
                     + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) + xb12 * (-23.0 - 48.0 * yb + 255.0 * yb2 - 96.0 * yb3 + 471.0 * yb4 + 144.0 * yb5 + 33.0 * yb6) + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6)
                     + 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) + xb6 * (147.0 + 96.0 * yb - 139.0 * yb2 - 320.0 * yb3 - 691.0 * yb4 - 1824.0 * yb5 + 75.0 * yb6)
                     - 4.0 * 1.0i * xb3 * (37.0 - 24.0 * yb - 5.0 * yb2 - 48.0 * yb3 + 2635.0 * yb4 + 72.0 * yb5 + 757.0 * yb6) - 4.0 * 1.0i * xb11 * (37.0 - 24.0 * yb - 5.0 * yb2 - 48.0 * yb3 + 2635.0 * yb4 + 72.0 * yb5 + 757.0 * yb6)
                     + 8.0 * 1.0i * xb7 * (133.0 + 72.0 * yb + 187.0 * yb2 - 112.0 * yb3 + 3883.0 * yb4 - 984.0 * yb5 + 2293.0 * yb6) + 2.0 * 1.0i * xb5 * (193.0 + 192.0 * yb - 1721.0 * yb2 - 128.0 * yb3 + 12799.0 * yb4 - 2112.0 * yb5
                       + 3033.0 * yb6) + 2.0 * 1.0i * xb9 * (193.0 + 192.0 * yb - 1721.0 * yb2 - 128.0 * yb3 + 12799.0 * yb4 - 2112.0 * yb5 + 3033.0 * yb6)
                     + xb4 * (171.0 + 144.0 * yb - 403.0 * yb2 - 224.0 * yb3 + 11093.0 * yb4 - 1968.0 * yb5 + 4147.0 * yb6)))
                 + power_of<2>(tb) * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 9.0 * yb2 + 12.0 * yb3 + 45.0 * yb4 - 18.0 * yb5 - 3.0 * yb6 + xb6 * (99.0 - 114.0 * yb - 709.0 * yb2 + 412.0 * yb3 - 1255.0 * yb4 + 2262.0 * yb5 - 1303.0 * yb6)
                     + xb12 * (-47.0 + 42.0 * yb + 345.0 * yb2 + 148.0 * yb3 - 573.0 * yb4 + 66.0 * yb5 - 269.0 * yb6) + xb14 * (1.0 - 6.0 * yb + 9.0 * yb2 - 12.0 * yb3 - 45.0 * yb4 + 18.0 * yb5 + 3.0 * yb6)
                     + 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) + 2.0 * 1.0i * xb13 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6)
                     + xb2 * (47.0 - 42.0 * yb - 345.0 * yb2 - 148.0 * yb3 + 573.0 * yb4 - 66.0 * yb5 + 269.0 * yb6) - 4.0 * 1.0i * xb3 * (25.0 + 18.0 * yb + 49.0 * yb2 + 68.0 * yb3 + 2107.0 * yb4 + 42.0 * yb5 + 603.0 * yb6)
                     - 4.0 * 1.0i * xb11 * (25.0 + 18.0 * yb + 49.0 * yb2 + 68.0 * yb3 + 2107.0 * yb4 + 42.0 * yb5 + 603.0 * yb6) + xb8 * (-99.0 + 114.0 * yb + 709.0 * yb2 - 412.0 * yb3 + 1255.0 * yb4 - 2262.0 * yb5 + 1303.0 * yb6)
                     + 3.0 * xb4 * (49.0 - 54.0 * yb - 7.0 * yb2 + 84.0 * yb3 + 3171.0 * yb4 + 738.0 * yb5 + 1363.0 * yb6) - 3.0 * xb10 * (49.0 - 54.0 * yb - 7.0 * yb2 + 84.0 * yb3 + 3171.0 * yb4 + 738.0 * yb5 + 1363.0 * yb6)
                     + 8.0 * 1.0i * xb7 * (121.0 - 78.0 * yb + 113.0 * yb2 + 132.0 * yb3 + 3611.0 * yb4 + 1098.0 * yb5 + 2011.0 * yb6) + 2.0 * 1.0i * xb5 * (193.0 - 198.0 * yb - 1447.0 * yb2 + 116.0 * yb3 + 12275.0 * yb4 + 2130.0 * yb5
                       + 3283.0 * yb6) + 2.0 * 1.0i * xb9 * (193.0 - 198.0 * yb - 1447.0 * yb2 + 116.0 * yb3 + 12275.0 * yb4 + 2130.0 * yb5 + 3283.0 * yb6))
                   - 4.0 * xb2 * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb8 * (-147.0 + 96.0 * yb + 139.0 * yb2 - 320.0 * yb3 + 691.0 * yb4 - 1824.0 * yb5 - 75.0 * yb6) + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6)
                     - xb2 * (-23.0 + 48.0 * yb + 255.0 * yb2 + 96.0 * yb3 + 471.0 * yb4 - 144.0 * yb5 + 33.0 * yb6) + xb12 * (-23.0 + 48.0 * yb + 255.0 * yb2 + 96.0 * yb3 + 471.0 * yb4 - 144.0 * yb5 + 33.0 * yb6)
                     + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) + 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) + xb6 * (147.0 - 96.0 * yb - 139.0 * yb2 + 320.0 * yb3 - 691.0 * yb4 + 1824.0 * yb5
                       + 75.0 * yb6) - 4.0 * 1.0i * xb3 * (37.0 + 24.0 * yb - 5.0 * yb2 + 48.0 * yb3 + 2635.0 * yb4 - 72.0 * yb5 + 757.0 * yb6) - 4.0 * 1.0i * xb11 * (37.0 + 24.0 * yb - 5.0 * yb2 + 48.0 * yb3 + 2635.0 * yb4 - 72.0 * yb5
                       + 757.0 * yb6) + 8.0 * 1.0i * xb7 * (133.0 - 72.0 * yb + 187.0 * yb2 + 112.0 * yb3 + 3883.0 * yb4 + 984.0 * yb5 + 2293.0 * yb6) + 2.0 * 1.0i * xb5 * (193.0 - 192.0 * yb - 1721.0 * yb2 + 128.0 * yb3 + 12799.0 * yb4
                       + 2112.0 * yb5 + 3033.0 * yb6) + 2.0 * 1.0i * xb9 * (193.0 - 192.0 * yb - 1721.0 * yb2 + 128.0 * yb3 + 12799.0 * yb4 + 2112.0 * yb5 + 3033.0 * yb6)
                     + xb4 * (171.0 - 144.0 * yb - 403.0 * yb2 + 224.0 * yb3 + 11093.0 * yb4 + 1968.0 * yb5 + 4147.0 * yb6) - xb10 * (171.0 - 144.0 * yb - 403.0 * yb2 + 224.0 * yb3 + 11093.0 * yb4 + 1968.0 * yb5 + 4147.0 * yb6))
                   + power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb12 * (-23.0 + 399.0 * yb2 + 375.0 * yb4 - 15.0 * yb6) + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6)
                     + xb2 * (23.0 - 399.0 * yb2 - 375.0 * yb4 + 15.0 * yb6) + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) + 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6)
                     - 4.0 * 1.0i * xb3 * (37.0 + 67.0 * yb2 + 2587.0 * yb4 + 733.0 * yb6) - 4.0 * 1.0i * xb11 * (37.0 + 67.0 * yb2 + 2587.0 * yb4 + 733.0 * yb6) + 8.0 * 1.0i * xb7 * (229.0 + 195.0 * yb2 + 7131.0 * yb4 + 4061.0 * yb6)
                     + xb6 * (339.0 + 1045.0 * yb2 + 3661.0 * yb4 + 4587.0 * yb6) - xb8 * (339.0 + 1045.0 * yb2 + 3661.0 * yb4 + 4587.0 * yb6) + 2.0 * 1.0i * xb5 * (385.0 - 2873.0 * yb2 + 21439.0 * yb4 + 5593.0 * yb6)
                     + 2.0 * 1.0i * xb9 * (385.0 - 2873.0 * yb2 + 21439.0 * yb4 + 5593.0 * yb6) + xb4 * (363.0 - 1411.0 * yb2 + 19637.0 * yb4 + 6659.0 * yb6) - xb10 * (363.0 - 1411.0 * yb2 + 19637.0 * yb4 + 6659.0 * yb6))));
            const complex<double> num4 = power_of<2>(-1.0i + xb) * (-1.0 + yb) * (1.0 + yb) *
                (-8.0 * tb * xb2 * (-4.0 * xb2 * (-1.0 - 6.0 * yb - 9.0 * yb2 - 12.0 * yb3 + 45.0 * yb4 + 18.0 * yb5 - 3.0 * yb6 + xb10 * (1.0 + 6.0 * yb + 9.0 * yb2 + 12.0 * yb3 - 45.0 * yb4 - 18.0 * yb5 + 3.0 * yb6)
                     - 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) - 2.0 * 1.0i * xb9 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6)
                     + xb2 * (45.0 - 18.0 * yb - 219.0 * yb2 + 28.0 * yb3 + 567.0 * yb4 + 246.0 * yb5 + 215.0 * yb6) - xb8 * (45.0 - 18.0 * yb - 219.0 * yb2 + 28.0 * yb3 + 567.0 * yb4 + 246.0 * yb5 + 215.0 * yb6)
                     + 8.0 * 1.0i * xb3 * (13.0 + 6.0 * yb - 31.0 * yb2 - 4.0 * yb3 + 1103.0 * yb4 - 66.0 * yb5 + 291.0 * yb6) + 8.0 * 1.0i * xb7 * (13.0 + 6.0 * yb - 31.0 * yb2 - 4.0 * yb3 + 1103.0 * yb4 - 66.0 * yb5 + 291.0 * yb6)
                     + 4.0 * 1.0i * xb5 * (51.0 + 18.0 * yb + 43.0 * yb2 - 28.0 * yb3 + 2569.0 * yb4 - 246.0 * yb5 + 825.0 * yb6) + 2.0 * xb4 * (23.0 - 6.0 * yb + 279.0 * yb2 + 20.0 * yb3 + 1029.0 * yb4 + 114.0 * yb5 + 1005.0 * yb6)
                     - 2.0 * xb6 * (23.0 - 6.0 * yb + 279.0 * yb2 + 20.0 * yb3 + 1029.0 * yb4 + 114.0 * yb5 + 1005.0 * yb6)) - 8.0 * vb * xb2 * (1.0 + 27.0 * yb2 - 57.0 * yb4 - 3.0 * yb6 + xb10 * (-1.0 - 27.0 * yb2 + 57.0 * yb4 + 3.0 * yb6)
                     + 2.0 * 1.0i * xb * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6) + 2.0 * 1.0i * xb9 * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6) - 8.0 * 1.0i * xb3 * (7.0 + 5.0 * yb2 + 833.0 * yb4 + 211.0 * yb6)
                     - 8.0 * 1.0i * xb7 * (7.0 + 5.0 * yb2 + 833.0 * yb4 + 211.0 * yb6) - 4.0 * 1.0i * xb5 * (27.0 + 41.0 * yb2 + 1757.0 * yb4 + 383.0 * yb6) - xb2 * (69.0 - 345.0 * yb2 + 1635.0 * yb4 + 529.0 * yb6)
                     + xb8 * (69.0 - 345.0 * yb2 + 1635.0 * yb4 + 529.0 * yb6) - 2.0 * xb4 * (35.0 + 353.0 * yb2 + 1301.0 * yb4 + 1287.0 * yb6) + xb6 * (70.0 + 706.0 * yb2 + 2602.0 * yb4 + 2574.0 * yb6))
                   + power_of<2>(vb) * (-1.0 - 6.0 * yb - 9.0 * yb2 - 12.0 * yb3 + 45.0 * yb4 + 18.0 * yb5 - 3.0 * yb6 + xb14 * (1.0 + 6.0 * yb + 9.0 * yb2 + 12.0 * yb3 - 45.0 * yb4 - 18.0 * yb5 + 3.0 * yb6)
                     - 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) - 2.0 * 1.0i * xb13 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6)
                     + xb2 * (47.0 + 42.0 * yb - 345.0 * yb2 + 148.0 * yb3 + 573.0 * yb4 + 66.0 * yb5 + 269.0 * yb6) - xb12 * (47.0 + 42.0 * yb - 345.0 * yb2 + 148.0 * yb3 + 573.0 * yb4 + 66.0 * yb5 + 269.0 * yb6)
                     + 4.0 * 1.0i * xb3 * (25.0 - 18.0 * yb + 49.0 * yb2 - 68.0 * yb3 + 2107.0 * yb4 - 42.0 * yb5 + 603.0 * yb6) + 4.0 * 1.0i * xb11 * (25.0 - 18.0 * yb + 49.0 * yb2 - 68.0 * yb3 + 2107.0 * yb4 - 42.0 * yb5 + 603.0 * yb6)
                     - xb6 * (-99.0 - 114.0 * yb + 709.0 * yb2 + 412.0 * yb3 + 1255.0 * yb4 + 2262.0 * yb5 + 1303.0 * yb6) + xb8 * (-99.0 - 114.0 * yb + 709.0 * yb2 + 412.0 * yb3 + 1255.0 * yb4 + 2262.0 * yb5 + 1303.0 * yb6)
                     + 3.0 * xb4 * (49.0 + 54.0 * yb - 7.0 * yb2 - 84.0 * yb3 + 3171.0 * yb4 - 738.0 * yb5 + 1363.0 * yb6) - 3.0 * xb10 * (49.0 + 54.0 * yb - 7.0 * yb2 - 84.0 * yb3 + 3171.0 * yb4 - 738.0 * yb5 + 1363.0 * yb6)
                     - 8.0 * 1.0i * xb7 * (121.0 + 78.0 * yb + 113.0 * yb2 - 132.0 * yb3 + 3611.0 * yb4 - 1098.0 * yb5 + 2011.0 * yb6) - 2.0 * 1.0i * xb5 * (193.0 + 198.0 * yb - 1447.0 * yb2 - 116.0 * yb3 + 12275.0 * yb4 - 2130.0 * yb5
                       + 3283.0 * yb6) - 2.0 * 1.0i * xb9 * (193.0 + 198.0 * yb - 1447.0 * yb2 - 116.0 * yb3 + 12275.0 * yb4 - 2130.0 * yb5 + 3283.0 * yb6)))
                 - 4.0 * xb2 * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 9.0 * yb2 + 12.0 * yb3 + 45.0 * yb4 - 18.0 * yb5 - 3.0 * yb6 + xb8 * (-45.0 - 18.0 * yb + 219.0 * yb2 + 28.0 * yb3 - 567.0 * yb4 + 246.0 * yb5 - 215.0 * yb6)
                     + xb10 * (1.0 - 6.0 * yb + 9.0 * yb2 - 12.0 * yb3 - 45.0 * yb4 + 18.0 * yb5 + 3.0 * yb6) - 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6)
                     - 2.0 * 1.0i * xb9 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) + xb2 * (45.0 + 18.0 * yb - 219.0 * yb2 - 28.0 * yb3 + 567.0 * yb4 - 246.0 * yb5 + 215.0 * yb6)
                     + 8.0 * 1.0i * xb3 * (13.0 - 6.0 * yb - 31.0 * yb2 + 4.0 * yb3 + 1103.0 * yb4 + 66.0 * yb5 + 291.0 * yb6) + 8.0 * 1.0i * xb7 * (13.0 - 6.0 * yb - 31.0 * yb2 + 4.0 * yb3 + 1103.0 * yb4 + 66.0 * yb5 + 291.0 * yb6)
                     + 4.0 * 1.0i * xb5 * (51.0 - 18.0 * yb + 43.0 * yb2 + 28.0 * yb3 + 2569.0 * yb4 + 246.0 * yb5 + 825.0 * yb6) + 2.0 * xb4 * (23.0 + 6.0 * yb + 279.0 * yb2 - 20.0 * yb3 + 1029.0 * yb4 - 114.0 * yb5 + 1005.0 * yb6)
                     - 2.0 * xb6 * (23.0 + 6.0 * yb + 279.0 * yb2 - 20.0 * yb3 + 1029.0 * yb4 - 114.0 * yb5 + 1005.0 * yb6)) - 4.0 * xb2 * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb10 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6)
                     - 3.0 * xb2 * (-7.0 + 31.0 * yb2 + 167.0 * yb4 + 33.0 * yb6) + 3.0 * xb8 * (-7.0 + 31.0 * yb2 + 167.0 * yb4 + 33.0 * yb6) - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6)
                     - 2.0 * 1.0i * xb9 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) + 8.0 * 1.0i * xb3 * (19.0 - 67.0 * yb2 + 1373.0 * yb4 + 371.0 * yb6) + 8.0 * 1.0i * xb7 * (19.0 - 67.0 * yb2 + 1373.0 * yb4 + 371.0 * yb6)
                     + 2.0 * xb4 * (11.0 + 205.0 * yb2 + 757.0 * yb4 + 723.0 * yb6) - 2.0 * xb6 * (11.0 + 205.0 * yb2 + 757.0 * yb4 + 723.0 * yb6) + 4.0 * 1.0i * xb5 * (75.0 + 45.0 * yb2 + 3381.0 * yb4 + 1267.0 * yb6))
                   + power_of<2>(vb) * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb10 * (-171.0 - 144.0 * yb + 403.0 * yb2 + 224.0 * yb3 - 11093.0 * yb4 + 1968.0 * yb5 - 4147.0 * yb6)
                     + xb8 * (-147.0 - 96.0 * yb + 139.0 * yb2 + 320.0 * yb3 + 691.0 * yb4 + 1824.0 * yb5 - 75.0 * yb6) + xb2 * (23.0 + 48.0 * yb - 255.0 * yb2 + 96.0 * yb3 - 471.0 * yb4 - 144.0 * yb5 - 33.0 * yb6)
                     + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) + xb12 * (-23.0 - 48.0 * yb + 255.0 * yb2 - 96.0 * yb3 + 471.0 * yb4 + 144.0 * yb5 + 33.0 * yb6) - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6)
                     - 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) + xb6 * (147.0 + 96.0 * yb - 139.0 * yb2 - 320.0 * yb3 - 691.0 * yb4 - 1824.0 * yb5 + 75.0 * yb6)
                     + 4.0 * 1.0i * xb3 * (37.0 - 24.0 * yb - 5.0 * yb2 - 48.0 * yb3 + 2635.0 * yb4 + 72.0 * yb5 + 757.0 * yb6) + 4.0 * 1.0i * xb11 * (37.0 - 24.0 * yb - 5.0 * yb2 - 48.0 * yb3 + 2635.0 * yb4 + 72.0 * yb5 + 757.0 * yb6)
                     - 8.0 * 1.0i * xb7 * (133.0 + 72.0 * yb + 187.0 * yb2 - 112.0 * yb3 + 3883.0 * yb4 - 984.0 * yb5 + 2293.0 * yb6) - 2.0 * 1.0i * xb5 * (193.0 + 192.0 * yb - 1721.0 * yb2 - 128.0 * yb3 + 12799.0 * yb4 - 2112.0 * yb5
                       + 3033.0 * yb6) - 2.0 * 1.0i * xb9 * (193.0 + 192.0 * yb - 1721.0 * yb2 - 128.0 * yb3 + 12799.0 * yb4 - 2112.0 * yb5 + 3033.0 * yb6)
                     + xb4 * (171.0 + 144.0 * yb - 403.0 * yb2 - 224.0 * yb3 + 11093.0 * yb4 - 1968.0 * yb5 + 4147.0 * yb6)))
                 + power_of<2>(tb) * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 9.0 * yb2 + 12.0 * yb3 + 45.0 * yb4 - 18.0 * yb5 - 3.0 * yb6 + xb6 * (99.0 - 114.0 * yb - 709.0 * yb2 + 412.0 * yb3 - 1255.0 * yb4 + 2262.0 * yb5 - 1303.0 * yb6)
                     + xb12 * (-47.0 + 42.0 * yb + 345.0 * yb2 + 148.0 * yb3 - 573.0 * yb4 + 66.0 * yb5 - 269.0 * yb6) + xb14 * (1.0 - 6.0 * yb + 9.0 * yb2 - 12.0 * yb3 - 45.0 * yb4 + 18.0 * yb5 + 3.0 * yb6)
                     - 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) - 2.0 * 1.0i * xb13 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6)
                     + xb2 * (47.0 - 42.0 * yb - 345.0 * yb2 - 148.0 * yb3 + 573.0 * yb4 - 66.0 * yb5 + 269.0 * yb6) + 4.0 * 1.0i * xb3 * (25.0 + 18.0 * yb + 49.0 * yb2 + 68.0 * yb3 + 2107.0 * yb4 + 42.0 * yb5 + 603.0 * yb6)
                     + 4.0 * 1.0i * xb11 * (25.0 + 18.0 * yb + 49.0 * yb2 + 68.0 * yb3 + 2107.0 * yb4 + 42.0 * yb5 + 603.0 * yb6) + xb8 * (-99.0 + 114.0 * yb + 709.0 * yb2 - 412.0 * yb3 + 1255.0 * yb4 - 2262.0 * yb5 + 1303.0 * yb6)
                     + 3.0 * xb4 * (49.0 - 54.0 * yb - 7.0 * yb2 + 84.0 * yb3 + 3171.0 * yb4 + 738.0 * yb5 + 1363.0 * yb6) - 3.0 * xb10 * (49.0 - 54.0 * yb - 7.0 * yb2 + 84.0 * yb3 + 3171.0 * yb4 + 738.0 * yb5 + 1363.0 * yb6)
                     - 8.0 * 1.0i * xb7 * (121.0 - 78.0 * yb + 113.0 * yb2 + 132.0 * yb3 + 3611.0 * yb4 + 1098.0 * yb5 + 2011.0 * yb6) - 2.0 * 1.0i * xb5 * (193.0 - 198.0 * yb - 1447.0 * yb2 + 116.0 * yb3 + 12275.0 * yb4 + 2130.0 * yb5
                       + 3283.0 * yb6) - 2.0 * 1.0i * xb9 * (193.0 - 198.0 * yb - 1447.0 * yb2 + 116.0 * yb3 + 12275.0 * yb4 + 2130.0 * yb5 + 3283.0 * yb6))
                   - 4.0 * xb2 * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb8 * (-147.0 + 96.0 * yb + 139.0 * yb2 - 320.0 * yb3 + 691.0 * yb4 - 1824.0 * yb5 - 75.0 * yb6) + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6)
                     - xb2 * (-23.0 + 48.0 * yb + 255.0 * yb2 + 96.0 * yb3 + 471.0 * yb4 - 144.0 * yb5 + 33.0 * yb6) + xb12 * (-23.0 + 48.0 * yb + 255.0 * yb2 + 96.0 * yb3 + 471.0 * yb4 - 144.0 * yb5 + 33.0 * yb6)
                     - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) - 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) + xb6 * (147.0 - 96.0 * yb - 139.0 * yb2 + 320.0 * yb3 - 691.0 * yb4 + 1824.0 * yb5
                       + 75.0 * yb6) + 4.0 * 1.0i * xb3 * (37.0 + 24.0 * yb - 5.0 * yb2 + 48.0 * yb3 + 2635.0 * yb4 - 72.0 * yb5 + 757.0 * yb6) + 4.0 * 1.0i * xb11 * (37.0 + 24.0 * yb - 5.0 * yb2 + 48.0 * yb3 + 2635.0 * yb4 - 72.0 * yb5
                       + 757.0 * yb6) - 8.0 * 1.0i * xb7 * (133.0 - 72.0 * yb + 187.0 * yb2 + 112.0 * yb3 + 3883.0 * yb4 + 984.0 * yb5 + 2293.0 * yb6) - 2.0 * 1.0i * xb5 * (193.0 - 192.0 * yb - 1721.0 * yb2 + 128.0 * yb3 + 12799.0 * yb4
                       + 2112.0 * yb5 + 3033.0 * yb6) - 2.0 * 1.0i * xb9 * (193.0 - 192.0 * yb - 1721.0 * yb2 + 128.0 * yb3 + 12799.0 * yb4 + 2112.0 * yb5 + 3033.0 * yb6)
                     + xb4 * (171.0 - 144.0 * yb - 403.0 * yb2 + 224.0 * yb3 + 11093.0 * yb4 + 1968.0 * yb5 + 4147.0 * yb6) - xb10 * (171.0 - 144.0 * yb - 403.0 * yb2 + 224.0 * yb3 + 11093.0 * yb4 + 1968.0 * yb5 + 4147.0 * yb6))
                   + power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6 + xb12 * (-23.0 + 399.0 * yb2 + 375.0 * yb4 - 15.0 * yb6) + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6)
                     + xb2 * (23.0 - 399.0 * yb2 - 375.0 * yb4 + 15.0 * yb6) - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) - 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6)
                     + 4.0 * 1.0i * xb3 * (37.0 + 67.0 * yb2 + 2587.0 * yb4 + 733.0 * yb6) + 4.0 * 1.0i * xb11 * (37.0 + 67.0 * yb2 + 2587.0 * yb4 + 733.0 * yb6) - 8.0 * 1.0i * xb7 * (229.0 + 195.0 * yb2 + 7131.0 * yb4 + 4061.0 * yb6)
                     + xb6 * (339.0 + 1045.0 * yb2 + 3661.0 * yb4 + 4587.0 * yb6) - xb8 * (339.0 + 1045.0 * yb2 + 3661.0 * yb4 + 4587.0 * yb6) - 2.0 * 1.0i * xb5 * (385.0 - 2873.0 * yb2 + 21439.0 * yb4 + 5593.0 * yb6)
                     - 2.0 * 1.0i * xb9 * (385.0 - 2873.0 * yb2 + 21439.0 * yb4 + 5593.0 * yb6) + xb4 * (363.0 - 1411.0 * yb2 + 19637.0 * yb4 + 6659.0 * yb6) - xb10 * (363.0 - 1411.0 * yb2 + 19637.0 * yb4 + 6659.0 * yb6))));
            const complex<double> num5 = 8.0 * tb * xb2 * (-8.0 * vb * xb2 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8) + xb8 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8)
                    + xb12 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8) + 2.0 * xb2 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8)
                    + 2.0 * xb10 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8) - 4.0 * xb6 * (-89.0 - 346.0 * yb2 - 4380.0 * yb4 + 2762.0 * yb6 + 2053.0 * yb8))
                  - 4.0 * xb2 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6
                      - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 - 30.0 * yb - 240.0 * yb2 + 66.0 * yb3 - 5802.0 * yb4 + 342.0 * yb5 + 3512.0 * yb6 - 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8))
                  + power_of<2>(vb) * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + 4.0 * xb2 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7
                      - 113.0 * yb8) + 4.0 * xb14 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7 - 113.0 * yb8)
                    + xb16 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb4 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6
                      + 486.0 * yb7 + 251.0 * yb8) + 4.0 * xb12 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6 + 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 - 510.0 * yb - 744.0 * yb2 + 1154.0 * yb3 - 28530.0 * yb4 + 5878.0 * yb5 + 12752.0 * yb6 - 6522.0 * yb7 + 17391.0 * yb8)))
                + 4.0 * xb2 * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6
                      + 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 + 30.0 * yb - 240.0 * yb2 - 66.0 * yb3 - 5802.0 * yb4 - 342.0 * yb5 + 3512.0 * yb6 + 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8))
                  - 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb2 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb10 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb12 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    + 4.0 * xb6 * (-161.0 - 134.0 * yb2 - 7224.0 * yb4 + 4262.0 * yb6 + 3257.0 * yb8) + xb4 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)
                    + xb8 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)) + power_of<2>(vb) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5 - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5
                      - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 - 480.0 * yb - 718.0 * yb2 + 1056.0 * yb3 - 30120.0 * yb4 + 5472.0 * yb5 + 13486.0 * yb6 - 6048.0 * yb7 + 18269.0 * yb8)))
                + power_of<2>(tb) * (-8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb16 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8) - 4.0 * xb2 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6
                      - 30.0 * yb7 + 113.0 * yb8) - 4.0 * xb14 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6 - 30.0 * yb7 + 113.0 * yb8)
                    + 4.0 * xb4 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    + 4.0 * xb12 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 + 510.0 * yb - 744.0 * yb2 - 1154.0 * yb3 - 28530.0 * yb4 - 5878.0 * yb5 + 12752.0 * yb6 + 6522.0 * yb7 + 17391.0 * yb8))
                  + 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5 - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5
                      - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 + 480.0 * yb - 718.0 * yb2 - 1056.0 * yb3 - 30120.0 * yb4 - 5472.0 * yb5 + 13486.0 * yb6 + 6048.0 * yb7 + 18269.0 * yb8))
                  - power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8)
                    + xb12 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8) - 20.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 20.0 * xb14 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb6 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8) - 4.0 * xb10 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8)
                    - 2.0 * xb8 * (-1493.0 + 978.0 * yb2 - 52872.0 * yb4 + 25486.0 * yb6 + 27901.0 * yb8)));
            const complex<double> num6 = 8.0 * tb * xb2 * (-8.0 * vb * xb2 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8) + xb8 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8)
                    + xb12 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8) + 2.0 * xb2 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8)
                    + 2.0 * xb10 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8) - 4.0 * xb6 * (-89.0 - 346.0 * yb2 - 4380.0 * yb4 + 2762.0 * yb6 + 2053.0 * yb8))
                  - 4.0 * xb2 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6
                      - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 - 30.0 * yb - 240.0 * yb2 + 66.0 * yb3 - 5802.0 * yb4 + 342.0 * yb5 + 3512.0 * yb6 - 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8))
                  + power_of<2>(vb) * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + 4.0 * xb2 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7
                      - 113.0 * yb8) + 4.0 * xb14 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7 - 113.0 * yb8)
                    + xb16 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb4 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6
                      + 486.0 * yb7 + 251.0 * yb8) + 4.0 * xb12 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6 + 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 - 510.0 * yb - 744.0 * yb2 + 1154.0 * yb3 - 28530.0 * yb4 + 5878.0 * yb5 + 12752.0 * yb6 - 6522.0 * yb7 + 17391.0 * yb8)))
                + 4.0 * xb2 * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6
                      + 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 + 30.0 * yb - 240.0 * yb2 - 66.0 * yb3 - 5802.0 * yb4 - 342.0 * yb5 + 3512.0 * yb6 + 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8))
                  - 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb2 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb10 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb12 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    + 4.0 * xb6 * (-161.0 - 134.0 * yb2 - 7224.0 * yb4 + 4262.0 * yb6 + 3257.0 * yb8) + xb4 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)
                    + xb8 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)) + power_of<2>(vb) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5 - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5
                      - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 - 480.0 * yb - 718.0 * yb2 + 1056.0 * yb3 - 30120.0 * yb4 + 5472.0 * yb5 + 13486.0 * yb6 - 6048.0 * yb7 + 18269.0 * yb8)))
                + power_of<2>(tb) * (-8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb16 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8) - 4.0 * xb2 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6
                      - 30.0 * yb7 + 113.0 * yb8) - 4.0 * xb14 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6 - 30.0 * yb7 + 113.0 * yb8)
                    + 4.0 * xb4 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    + 4.0 * xb12 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 + 510.0 * yb - 744.0 * yb2 - 1154.0 * yb3 - 28530.0 * yb4 - 5878.0 * yb5 + 12752.0 * yb6 + 6522.0 * yb7 + 17391.0 * yb8))
                  + 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5 - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5
                      - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 + 480.0 * yb - 718.0 * yb2 - 1056.0 * yb3 - 30120.0 * yb4 - 5472.0 * yb5 + 13486.0 * yb6 + 6048.0 * yb7 + 18269.0 * yb8))
                  - power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8)
                    + xb12 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8) - 20.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 20.0 * xb14 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb6 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8) - 4.0 * xb10 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8)
                    - 2.0 * xb8 * (-1493.0 + 978.0 * yb2 - 52872.0 * yb4 + 25486.0 * yb6 + 27901.0 * yb8)));
            const complex<double> num7 = (4.0 * xb2 + 4.0 * power_of<2>(tb) * xb2 - tb * (1.0 - 6.0 * xb2 + xb4)) * (-1.0 + yb) * (1.0 + yb) * (3.0 * (-1.0 + yb) * yb * power_of<4>(1.0 + yb) + 3.0 * xb8 * (-1.0 + yb) * yb * power_of<4>(1.0 + yb)
                 + 4.0 * xb2 * (3.0 - 3.0 * yb - 18.0 * yb2 + 2.0 * yb3 + 135.0 * yb4 + 33.0 * yb5 + 40.0 * yb6) + 4.0 * xb6 * (3.0 - 3.0 * yb - 18.0 * yb2 + 2.0 * yb3 + 135.0 * yb4 + 33.0 * yb5 + 40.0 * yb6)
                 + 2.0 * xb4 * (12.0 - 9.0 * yb + yb2 + 14.0 * yb3 + 406.0 * yb4 + 123.0 * yb5 + 221.0 * yb6));
            const complex<double> num8 = (-4.0 * xb2 - 4.0 * power_of<2>(tb) * xb2 + tb * (1.0 - 6.0 * xb2 + xb4)) * (-1.0 + yb) * (1.0 + yb) * (3.0 * (-1.0 + yb) * yb * power_of<4>(1.0 + yb) + 3.0 * xb8 * (-1.0 + yb) * yb * power_of<4>(1.0 + yb)
                 + 4.0 * xb2 * (3.0 - 3.0 * yb - 18.0 * yb2 + 2.0 * yb3 + 135.0 * yb4 + 33.0 * yb5 + 40.0 * yb6) + 4.0 * xb6 * (3.0 - 3.0 * yb - 18.0 * yb2 + 2.0 * yb3 + 135.0 * yb4 + 33.0 * yb5 + 40.0 * yb6)
                 + 2.0 * xb4 * (12.0 - 9.0 * yb + yb2 + 14.0 * yb3 + 406.0 * yb4 + 123.0 * yb5 + 221.0 * yb6));
            const complex<double> num9 = (4.0 * xb2 + 4.0 * power_of<2>(vb) * xb2 + vb * (1.0 - 6.0 * xb2 + xb4)) * (-1.0 + yb) * (1.0 + yb) * (3.0 * power_of<4>(-1.0 + yb) * yb * (1.0 + yb) + 3.0 * xb8 * power_of<4>(-1.0 + yb) * yb * (1.0 + yb)
                 + 4.0 * xb2 * (3.0 + 3.0 * yb - 18.0 * yb2 - 2.0 * yb3 + 135.0 * yb4 - 33.0 * yb5 + 40.0 * yb6) + 4.0 * xb6 * (3.0 + 3.0 * yb - 18.0 * yb2 - 2.0 * yb3 + 135.0 * yb4 - 33.0 * yb5 + 40.0 * yb6)
                 + 2.0 * xb4 * (12.0 + 9.0 * yb + yb2 - 14.0 * yb3 + 406.0 * yb4 - 123.0 * yb5 + 221.0 * yb6));
            const complex<double> num10 = -8.0 * tb * xb2 * (-8.0 * vb * xb2 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8) + xb8 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8)
                    + xb12 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8) + 2.0 * xb2 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8)
                    + 2.0 * xb10 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8) - 4.0 * xb6 * (-89.0 - 346.0 * yb2 - 4380.0 * yb4 + 2762.0 * yb6 + 2053.0 * yb8))
                  - 4.0 * xb2 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6
                      - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 - 30.0 * yb - 240.0 * yb2 + 66.0 * yb3 - 5802.0 * yb4 + 342.0 * yb5 + 3512.0 * yb6 - 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8))
                  + power_of<2>(vb) * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + 4.0 * xb2 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7
                      - 113.0 * yb8) + 4.0 * xb14 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7 - 113.0 * yb8)
                    + xb16 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb4 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6
                      + 486.0 * yb7 + 251.0 * yb8) + 4.0 * xb12 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6 + 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 - 510.0 * yb - 744.0 * yb2 + 1154.0 * yb3 - 28530.0 * yb4 + 5878.0 * yb5 + 12752.0 * yb6 - 6522.0 * yb7 + 17391.0 * yb8)))
                - 4.0 * xb2 * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6
                      + 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 + 30.0 * yb - 240.0 * yb2 - 66.0 * yb3 - 5802.0 * yb4 - 342.0 * yb5 + 3512.0 * yb6 + 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8))
                  - 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb2 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb10 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb12 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    + 4.0 * xb6 * (-161.0 - 134.0 * yb2 - 7224.0 * yb4 + 4262.0 * yb6 + 3257.0 * yb8) + xb4 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)
                    + xb8 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)) + power_of<2>(vb) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5 - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5
                      - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 - 480.0 * yb - 718.0 * yb2 + 1056.0 * yb3 - 30120.0 * yb4 + 5472.0 * yb5 + 13486.0 * yb6 - 6048.0 * yb7 + 18269.0 * yb8)))
                + power_of<2>(tb) * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb16 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8) - 4.0 * xb2 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6
                      - 30.0 * yb7 + 113.0 * yb8) - 4.0 * xb14 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6 - 30.0 * yb7 + 113.0 * yb8)
                    + 4.0 * xb4 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    + 4.0 * xb12 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 + 510.0 * yb - 744.0 * yb2 - 1154.0 * yb3 - 28530.0 * yb4 - 5878.0 * yb5 + 12752.0 * yb6 + 6522.0 * yb7 + 17391.0 * yb8))
                  - 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5 - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5
                      - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 + 480.0 * yb - 718.0 * yb2 - 1056.0 * yb3 - 30120.0 * yb4 - 5472.0 * yb5 + 13486.0 * yb6 + 6048.0 * yb7 + 18269.0 * yb8))
                  + power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) - 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8)
                    + xb12 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8) - 20.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 20.0 * xb14 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb6 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8) - 4.0 * xb10 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8)
                    - 2.0 * xb8 * (-1493.0 + 978.0 * yb2 - 52872.0 * yb4 + 25486.0 * yb6 + 27901.0 * yb8)));
            const complex<double> num11 = -8.0 * tb * xb2 * (-8.0 * vb * xb2 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8 - 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    - 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8) + xb8 * (111.0 - 1082.0 * yb2 + 13332.0 * yb4 - 11030.0 * yb6 - 1331.0 * yb8)
                    + xb12 * (1.0 + 26.0 * yb2 - 84.0 * yb4 + 54.0 * yb6 + 3.0 * yb8) + 2.0 * xb2 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8)
                    + 2.0 * xb10 * (-33.0 + 150.0 * yb2 - 828.0 * yb4 + 346.0 * yb6 + 365.0 * yb8) - 4.0 * xb6 * (-89.0 - 346.0 * yb2 - 4380.0 * yb4 + 2762.0 * yb6 + 2053.0 * yb8))
                  - 4.0 * xb2 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 - 36.0 * yb - 96.0 * yb2 + 28.0 * yb3 + 372.0 * yb4 + 308.0 * yb5 + 80.0 * yb6 - 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6
                      - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 - 30.0 * yb - 240.0 * yb2 + 66.0 * yb3 - 5802.0 * yb4 + 342.0 * yb5 + 3512.0 * yb6 - 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 - 90.0 * yb + 1480.0 * yb2 + 166.0 * yb3 - 17430.0 * yb4 + 962.0 * yb5 + 13296.0 * yb6 - 1038.0 * yb7 + 2861.0 * yb8))
                  + power_of<2>(vb) * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2)
                    - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + 4.0 * xb2 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7
                      - 113.0 * yb8) + 4.0 * xb14 * (11.0 + 6.0 * yb - 56.0 * yb2 + 22.0 * yb3 + 126.0 * yb4 + 2.0 * yb5 + 32.0 * yb6 - 30.0 * yb7 - 113.0 * yb8)
                    + xb16 * (-1.0 - 6.0 * yb - 8.0 * yb2 - 6.0 * yb3 + 54.0 * yb4 + 30.0 * yb5 - 48.0 * yb6 - 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb4 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6
                      + 486.0 * yb7 + 251.0 * yb8) + 4.0 * xb12 * (-25.0 + 66.0 * yb + 8.0 * yb2 - 30.0 * yb3 - 1962.0 * yb4 - 522.0 * yb5 + 1728.0 * yb6 + 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 - 186.0 * yb + 1800.0 * yb2 + 342.0 * yb3 - 11202.0 * yb4 + 1986.0 * yb5 + 7648.0 * yb6 - 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 - 510.0 * yb - 744.0 * yb2 + 1154.0 * yb3 - 28530.0 * yb4 + 5878.0 * yb5 + 12752.0 * yb6 - 6522.0 * yb7 + 17391.0 * yb8)))
                - 4.0 * xb2 * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + xb2 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8)
                    + xb10 * (42.0 + 36.0 * yb - 96.0 * yb2 - 28.0 * yb3 + 372.0 * yb4 - 308.0 * yb5 + 80.0 * yb6 + 300.0 * yb7 - 398.0 * yb8) + xb12 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6
                      + 18.0 * yb7 + 3.0 * yb8) + 4.0 * xb6 * (-125.0 + 30.0 * yb - 240.0 * yb2 - 66.0 * yb3 - 5802.0 * yb4 - 342.0 * yb5 + 3512.0 * yb6 + 378.0 * yb7 + 2655.0 * yb8)
                    + xb4 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8)
                    + xb8 * (-207.0 + 90.0 * yb + 1480.0 * yb2 - 166.0 * yb3 - 17430.0 * yb4 - 962.0 * yb5 + 13296.0 * yb6 + 1038.0 * yb7 + 2861.0 * yb8))
                  - 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) + 192.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb2 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb10 * (18.0 + 108.0 * yb2 - 912.0 * yb4 + 852.0 * yb6 - 66.0 * yb8) + xb12 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    + 4.0 * xb6 * (-161.0 - 134.0 * yb2 - 7224.0 * yb4 + 4262.0 * yb6 + 3257.0 * yb8) + xb4 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)
                    + xb8 * (-303.0 + 1878.0 * yb2 - 21528.0 * yb4 + 15562.0 * yb6 + 4391.0 * yb8)) + power_of<2>(vb) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5 - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 - 12.0 * yb + 14.0 * yb2 - 12.0 * yb3 + 180.0 * yb4 + 60.0 * yb5
                      - 222.0 * yb6 - 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 + 72.0 * yb + 10.0 * yb2 - 56.0 * yb3 - 2352.0 * yb4 - 616.0 * yb5 + 1910.0 * yb6 + 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 - 180.0 * yb + 1842.0 * yb2 + 332.0 * yb3 - 11508.0 * yb4 + 1924.0 * yb5 + 7838.0 * yb6 - 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 - 480.0 * yb - 718.0 * yb2 + 1056.0 * yb3 - 30120.0 * yb4 + 5472.0 * yb5 + 13486.0 * yb6 - 6048.0 * yb7 + 18269.0 * yb8)))
                + power_of<2>(tb) * (8.0 * vb * xb2 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2)
                    - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2)
                    + xb16 * (-1.0 + 6.0 * yb - 8.0 * yb2 + 6.0 * yb3 + 54.0 * yb4 - 30.0 * yb5 - 48.0 * yb6 + 18.0 * yb7 + 3.0 * yb8) - 4.0 * xb2 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6
                      - 30.0 * yb7 + 113.0 * yb8) - 4.0 * xb14 * (-11.0 + 6.0 * yb + 56.0 * yb2 + 22.0 * yb3 - 126.0 * yb4 + 2.0 * yb5 - 32.0 * yb6 - 30.0 * yb7 + 113.0 * yb8)
                    + 4.0 * xb4 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    + 4.0 * xb12 * (-25.0 - 66.0 * yb + 8.0 * yb2 + 30.0 * yb3 - 1962.0 * yb4 + 522.0 * yb5 + 1728.0 * yb6 - 486.0 * yb7 + 251.0 * yb8)
                    - 4.0 * xb6 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 4.0 * xb10 * (-181.0 + 186.0 * yb + 1800.0 * yb2 - 342.0 * yb3 - 11202.0 * yb4 - 1986.0 * yb5 + 7648.0 * yb6 + 2142.0 * yb7 + 1935.0 * yb8)
                    - 2.0 * xb8 * (-869.0 + 510.0 * yb - 744.0 * yb2 - 1154.0 * yb3 - 28530.0 * yb4 - 5878.0 * yb5 + 12752.0 * yb6 + 6522.0 * yb7 + 17391.0 * yb8))
                  - 4.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb2 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5 - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) - 4.0 * xb14 * (-5.0 + 12.0 * yb + 14.0 * yb2 + 12.0 * yb3 + 180.0 * yb4 - 60.0 * yb5
                      - 222.0 * yb6 + 36.0 * yb7 + 33.0 * yb8) + 4.0 * xb4 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    + 4.0 * xb12 * (-37.0 - 72.0 * yb + 10.0 * yb2 + 56.0 * yb3 - 2352.0 * yb4 + 616.0 * yb5 + 1910.0 * yb6 - 600.0 * yb7 + 469.0 * yb8)
                    - 4.0 * xb6 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 4.0 * xb10 * (-187.0 + 180.0 * yb + 1842.0 * yb2 - 332.0 * yb3 - 11508.0 * yb4 - 1924.0 * yb5 + 7838.0 * yb6 + 2076.0 * yb7 + 2015.0 * yb8)
                    - 2.0 * xb8 * (-917.0 + 480.0 * yb - 718.0 * yb2 - 1056.0 * yb3 - 30120.0 * yb4 - 5472.0 * yb5 + 13486.0 * yb6 + 6048.0 * yb7 + 18269.0 * yb8))
                  + power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8 + 64.0 * xb3 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb5 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb7 * power_of<4>(1.0 + 3.0 * yb2) - 128.0 * xb9 * power_of<4>(1.0 + 3.0 * yb2)
                    + 64.0 * xb11 * power_of<4>(1.0 + 3.0 * yb2) + 64.0 * xb13 * power_of<4>(1.0 + 3.0 * yb2) + xb4 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8)
                    + xb12 * (44.0 - 1592.0 * yb2 + 864.0 * yb4 + 1464.0 * yb6 - 780.0 * yb8) - 20.0 * xb2 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 20.0 * xb14 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8) + xb16 * (-1.0 + 10.0 * yb2 + 24.0 * yb4 - 42.0 * yb6 + 9.0 * yb8)
                    - 4.0 * xb6 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8) - 4.0 * xb10 * (-379.0 + 2638.0 * yb2 - 19704.0 * yb4 + 12370.0 * yb6 + 5075.0 * yb8)
                    - 2.0 * xb8 * (-1493.0 + 978.0 * yb2 - 52872.0 * yb4 + 25486.0 * yb6 + 27901.0 * yb8)));
            const complex<double> num12 = power_of<4>(1.0 + xb2) * (1.0 - 22.0 * xb2 + xb4) * yb2 * power_of<3>(-1.0 + yb2);

            const complex<double> denom1 = 108.0 * xb4 * power_of<2>(1.0 + 3.0 * yb2);
            const complex<double> denom2 = 162.0 * xb4 * yb * power_of<3>(1.0 + 3.0 * yb2);
            const complex<double> denom3 = 864.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom4 = 1728.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom5 = 108.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * power_of<4>(xb + 3.0 * xb * yb2);
            const complex<double> denom6 = 108.0 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * power_of<4>(xb + 3.0 * xb * yb2);
            const complex<double> denom7 = 144.0 * xb6 * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom8 = 108.0 * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom9 = 108.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * power_of<4>(1.0 + 3.0 * yb2);
            const complex<double> denom10 = 432.0 * (tb * power_of<2>(-1.0 + xb) - 2.0 * xb) * xb6 * (vb * power_of<2>(-1.0 + xb) + 2.0 * xb) * (2.0 * xb + tb * power_of<2>(1.0 + xb)) * (-2.0 * xb + vb * power_of<2>(1.0 + xb)) * power_of<4>(1.0 + 3.0 * yb2);

            const complex<double> logs1 = power_of<2>(1.0i + xb) * (-1.0 + yb) * (1.0 + yb) * (-8.0 * tb * xb2 * (-4.0 * xb2 * (1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb10 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb9 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) + 8.0 * 1.0i * xb3 * (13.0 + 6.0 * yb - 4.0 * yb3 - 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2))
                     + 8.0 * 1.0i * xb7 * (13.0 + 6.0 * yb - 4.0 * yb3 - 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2)) + xb8 * (45.0 - 18.0 * yb + 28.0 * yb3 + 246.0 * yb5 + yb6 * (215.0 - 4032.0 * ln2) + 63.0 * yb4 * (9.0 + 128.0 * ln2) - 3.0 * yb2 * (73.0 + 1344.0 * ln2)) + 2.0 * xb4 * (-23.0 + 6.0 * yb - 20.0 * yb3 - 114.0 * yb5 + 3.0 * yb6 * (-335.0 + 704.0 * ln2) + 3.0 * yb2 * (-93.0 + 704.0 * ln2) - 3.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb6 * (46.0 - 12.0 * yb + 40.0 * yb3 + 228.0 * yb5 - 6.0 * yb6 * (-335.0 + 704.0 * ln2) - 6.0 * yb2 * (-93.0 + 704.0 * ln2) + 6.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb2 * (-45.0 + 18.0 * yb - 28.0 * yb3 - 246.0 * yb5 - 63.0 * yb4 * (9.0 + 128.0 * ln2) + 3.0 * yb2 * (73.0 + 1344.0 * ln2) + yb6 * (-215.0 + 4032.0 * ln2)) + 4.0 * 1.0i * xb5 * (51.0 + 18.0 * yb - 28.0 * yb3 - 246.0 * yb5 + yb4 * (2569.0 - 8064.0 * ln2) + yb2 * (43.0 + 4032.0 * ln2) + yb6 * (825.0 + 4032.0 * ln2)))
                   + 8.0 * vb * xb2 * (1.0 - 3.0 * yb2 * (-9.0 + 64.0 * ln2) - 3.0 * yb6 * (1.0 + 64.0 * ln2) + yb4 * (-57.0 + 384.0 * ln2) + xb10 * (-1.0 + yb4 * (57.0 - 384.0 * ln2) + 3.0 * yb2 * (-9.0 + 64.0 * ln2) + 3.0 * yb6 * (1.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb2 * (7.0 + 64.0 * ln2) + 3.0 * yb6 * (17.0 + 64.0 * ln2) - 3.0 * yb4 * (13.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb9 * (-1.0 + 3.0 * yb2 * (7.0 + 64.0 * ln2) + 3.0 * yb6 * (17.0 + 64.0 * ln2) - 3.0 * yb4 * (13.0 + 128.0 * ln2)) + 8.0 * 1.0i * xb3 * (7.0 + yb4 * (833.0 - 1920.0 * ln2) + yb2 * (5.0 + 960.0 * ln2) + yb6 * (211.0 + 960.0 * ln2)) + 8.0 * 1.0i * xb7 * (7.0 + yb4 * (833.0 - 1920.0 * ln2) + yb2 * (5.0 + 960.0 * ln2) + yb6 * (211.0 + 960.0 * ln2)) + xb8 * (69.0 + yb6 * (529.0 - 4032.0 * ln2) - 3.0 * yb2 * (115.0 + 1344.0 * ln2) + 3.0 * yb4 * (545.0 + 2688.0 * ln2)) + xb2 * (-69.0 - 3.0 * yb4 * (545.0 + 2688.0 * ln2) + yb6 * (-529.0 + 4032.0 * ln2) + yb2 * (345.0 + 4032.0 * ln2))
                     + 4.0 * 1.0i * xb5 * (27.0 + yb4 * (1757.0 - 8064.0 * ln2) + yb2 * (41.0 + 4032.0 * ln2) + yb6 * (383.0 + 4032.0 * ln2)) + xb4 * (-70.0 + 66.0 * yb6 * (-39.0 + 64.0 * ln2) + yb2 * (-706.0 + 4224.0 * ln2) - 2.0 * yb4 * (1301.0 + 4224.0 * ln2)) + xb6 * (70.0 + yb2 * (706.0 - 4224.0 * ln2) - 66.0 * yb6 * (-39.0 + 64.0 * ln2) + yb4 * (2602.0 + 8448.0 * ln2))) + power_of<2>(vb) * (1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb14 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2))
                     - 2.0 * 1.0i * xb13 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb5 * (-193.0 - 198.0 * yb + 116.0 * yb3 + 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) + 2.0 * 1.0i * xb9 * (-193.0 - 198.0 * yb + 116.0 * yb3 + 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) - 3.0 * xb4 * (49.0 + 54.0 * yb - 84.0 * yb3 - 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2)) + 3.0 * xb10 * (49.0 + 54.0 * yb - 84.0 * yb3 - 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2))
                     + 4.0 * 1.0i * xb3 * (25.0 - 18.0 * yb - 68.0 * yb3 - 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2)) + 4.0 * 1.0i * xb11 * (25.0 - 18.0 * yb - 68.0 * yb3 - 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2)) - 8.0 * 1.0i * xb7 * (121.0 + 78.0 * yb - 132.0 * yb3 - 1098.0 * yb5 + yb4 * (3611.0 - 4224.0 * ln2) + yb2 * (113.0 + 2112.0 * ln2) + yb6 * (2011.0 + 2112.0 * ln2)) + xb6 * (-99.0 - 114.0 * yb + 412.0 * yb3 + 2262.0 * yb5 + yb2 * (709.0 - 8640.0 * ln2) + yb6 * (1303.0 - 8640.0 * ln2) + 5.0 * yb4 * (251.0 + 3456.0 * ln2)) + xb2 * (-47.0 - 42.0 * yb - 148.0 * yb3 - 66.0 * yb5 + 69.0 * yb2 * (5.0 + 64.0 * ln2) - 3.0 * yb4 * (191.0 + 2944.0 * ln2) + yb6 * (-269.0 + 4416.0 * ln2))
                     + xb8 * (99.0 + 114.0 * yb - 412.0 * yb3 - 2262.0 * yb5 - 5.0 * yb4 * (251.0 + 3456.0 * ln2) + yb6 * (-1303.0 + 8640.0 * ln2) + yb2 * (-709.0 + 8640.0 * ln2)) + xb12 * (47.0 + 42.0 * yb + 148.0 * yb3 + 66.0 * yb5 + yb6 * (269.0 - 4416.0 * ln2) - 69.0 * yb2 * (5.0 + 64.0 * ln2) + yb4 * (573.0 + 8832.0 * ln2)))) + power_of<2>(tb) * (8.0 * vb * xb2 * (1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb14 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb13 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2))
                     + 2.0 * 1.0i * xb5 * (-193.0 + 198.0 * yb - 116.0 * yb3 - 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) + 2.0 * 1.0i * xb9 * (-193.0 + 198.0 * yb - 116.0 * yb3 - 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) - 3.0 * xb4 * (49.0 - 54.0 * yb + 84.0 * yb3 + 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2)) + 3.0 * xb10 * (49.0 - 54.0 * yb + 84.0 * yb3 + 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2)) + 4.0 * 1.0i * xb3 * (25.0 + 18.0 * yb + 68.0 * yb3 + 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2))
                     + 4.0 * 1.0i * xb11 * (25.0 + 18.0 * yb + 68.0 * yb3 + 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2)) - 8.0 * 1.0i * xb7 * (121.0 - 78.0 * yb + 132.0 * yb3 + 1098.0 * yb5 + yb4 * (3611.0 - 4224.0 * ln2) + yb2 * (113.0 + 2112.0 * ln2) + yb6 * (2011.0 + 2112.0 * ln2)) + xb6 * (-99.0 + 114.0 * yb - 412.0 * yb3 - 2262.0 * yb5 + yb2 * (709.0 - 8640.0 * ln2) + yb6 * (1303.0 - 8640.0 * ln2) + 5.0 * yb4 * (251.0 + 3456.0 * ln2)) + xb2 * (-47.0 + 42.0 * yb + 148.0 * yb3 + 66.0 * yb5 + 69.0 * yb2 * (5.0 + 64.0 * ln2) - 3.0 * yb4 * (191.0 + 2944.0 * ln2) + yb6 * (-269.0 + 4416.0 * ln2)) - xb12 * (-47.0 + 42.0 * yb + 148.0 * yb3 + 66.0 * yb5 + 69.0 * yb2 * (5.0 + 64.0 * ln2) - 3.0 * yb4 * (191.0 + 2944.0 * ln2) + yb6 * (-269.0 + 4416.0 * ln2))
                     + xb8 * (99.0 - 114.0 * yb + 412.0 * yb3 + 2262.0 * yb5 - 5.0 * yb4 * (251.0 + 3456.0 * ln2) + yb6 * (-1303.0 + 8640.0 * ln2) + yb2 * (-709.0 + 8640.0 * ln2))) + power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb14 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb13 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb5 * (-385.0 + yb6 * (-5593.0 + 192.0 * ln2) + yb2 * (2873.0 + 192.0 * ln2) - yb4 * (21439.0 + 384.0 * ln2)) + 2.0 * 1.0i * xb9 * (-385.0 + yb6 * (-5593.0 + 192.0 * ln2) + yb2 * (2873.0 + 192.0 * ln2) - yb4 * (21439.0 + 384.0 * ln2))
                     + 4.0 * 1.0i * xb3 * (37.0 + yb4 * (2587.0 - 4224.0 * ln2) + yb2 * (67.0 + 2112.0 * ln2) + yb6 * (733.0 + 2112.0 * ln2)) + 4.0 * 1.0i * xb11 * (37.0 + yb4 * (2587.0 - 4224.0 * ln2) + yb2 * (67.0 + 2112.0 * ln2) + yb6 * (733.0 + 2112.0 * ln2)) - 8.0 * 1.0i * xb7 * (229.0 + yb4 * (7131.0 - 4224.0 * ln2) + 3.0 * yb2 * (65.0 + 704.0 * ln2) + yb6 * (4061.0 + 2112.0 * ln2)) - xb4 * (363.0 + yb4 * (19637.0 - 8064.0 * ln2) + yb2 * (-1411.0 + 4032.0 * ln2) + yb6 * (6659.0 + 4032.0 * ln2)) + xb10 * (363.0 + yb4 * (19637.0 - 8064.0 * ln2) + yb2 * (-1411.0 + 4032.0 * ln2) + yb6 * (6659.0 + 4032.0 * ln2)) + xb2 * (-23.0 + yb4 * (375.0 - 8832.0 * ln2) + 3.0 * yb6 * (-5.0 + 1472.0 * ln2) + yb2 * (399.0 + 4416.0 * ln2)) - xb6 * (339.0 + yb4 * (3661.0 - 17280.0 * ln2) + 5.0 * yb2 * (209.0 + 1728.0 * ln2) + yb6 * (4587.0 + 8640.0 * ln2))
                     + xb8 * (339.0 + yb4 * (3661.0 - 17280.0 * ln2) + 5.0 * yb2 * (209.0 + 1728.0 * ln2) + yb6 * (4587.0 + 8640.0 * ln2)) + xb12 * (23.0 + yb6 * (15.0 - 4416.0 * ln2) - 3.0 * yb2 * (133.0 + 1472.0 * ln2) + yb4 * (-375.0 + 8832.0 * ln2))) - 4.0 * xb2 * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb14 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb13 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb5 * (-193.0 + 192.0 * yb - 128.0 * yb3 - 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2))
                     + 2.0 * 1.0i * xb9 * (-193.0 + 192.0 * yb - 128.0 * yb3 - 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2)) + 4.0 * 1.0i * xb3 * (37.0 + 24.0 * yb + 48.0 * yb3 - 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) + 4.0 * 1.0i * xb11 * (37.0 + 24.0 * yb + 48.0 * yb3 - 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) - 8.0 * 1.0i * xb7 * (133.0 - 72.0 * yb + 112.0 * yb3 + 984.0 * yb5 + yb4 * (3883.0 - 4224.0 * ln2) + 11.0 * yb2 * (17.0 + 192.0 * ln2) + yb6 * (2293.0 + 2112.0 * ln2)) - xb4 * (171.0 - 144.0 * yb + 224.0 * yb3 + 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2))
                     + xb10 * (171.0 - 144.0 * yb + 224.0 * yb3 + 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2)) + xb2 * (-23.0 + 48.0 * yb + 96.0 * yb3 - 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) - xb12 * (-23.0 + 48.0 * yb + 96.0 * yb3 - 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) - xb6 * (147.0 - 96.0 * yb + 320.0 * yb3 + 1824.0 * yb5 + 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb2 * (-139.0 + 8640.0 * ln2) - yb4 * (691.0 + 17280.0 * ln2)) + xb8 * (147.0 - 96.0 * yb + 320.0 * yb3 + 1824.0 * yb5 + 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb2 * (-139.0 + 8640.0 * ln2) - yb4 * (691.0 + 17280.0 * ln2))))
                 - 4.0 * xb2 * (8.0 * vb * xb2 * (1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb10 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb9 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) + 8.0 * 1.0i * xb3 * (13.0 - 6.0 * yb + 4.0 * yb3 + 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2)) + 8.0 * 1.0i * xb7 * (13.0 - 6.0 * yb + 4.0 * yb3 + 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2))
                     + xb8 * (45.0 + 18.0 * yb - 28.0 * yb3 - 246.0 * yb5 + yb6 * (215.0 - 4032.0 * ln2) + 63.0 * yb4 * (9.0 + 128.0 * ln2) - 3.0 * yb2 * (73.0 + 1344.0 * ln2)) + 2.0 * xb4 * (-23.0 - 6.0 * yb + 20.0 * yb3 + 114.0 * yb5 + 3.0 * yb6 * (-335.0 + 704.0 * ln2) + 3.0 * yb2 * (-93.0 + 704.0 * ln2) - 3.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb6 * (46.0 + 12.0 * yb - 40.0 * yb3 - 228.0 * yb5 - 6.0 * yb6 * (-335.0 + 704.0 * ln2) - 6.0 * yb2 * (-93.0 + 704.0 * ln2) + 6.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb2 * (-45.0 - 18.0 * yb + 28.0 * yb3 + 246.0 * yb5 - 63.0 * yb4 * (9.0 + 128.0 * ln2) + 3.0 * yb2 * (73.0 + 1344.0 * ln2) + yb6 * (-215.0 + 4032.0 * ln2)) + 4.0 * 1.0i * xb5 * (51.0 - 18.0 * yb + 28.0 * yb3 + 246.0 * yb5 + yb4 * (2569.0 - 8064.0 * ln2) + yb2 * (43.0 + 4032.0 * ln2) + yb6 * (825.0 + 4032.0 * ln2)))
                   - 4.0 * xb2 * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb10 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb9 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) + 4.0 * 1.0i * xb5 * (75.0 + yb4 * (3381.0 - 8064.0 * ln2) + 9.0 * yb2 * (5.0 + 448.0 * ln2) + 7.0 * yb6 * (181.0 + 576.0 * ln2)) + 8.0 * 1.0i * xb3 * (19.0 + yb4 * (1373.0 - 1920.0 * ln2) + yb2 * (-67.0 + 960.0 * ln2) + yb6 * (371.0 + 960.0 * ln2)) + 8.0 * 1.0i * xb7 * (19.0 + yb4 * (1373.0 - 1920.0 * ln2) + yb2 * (-67.0 + 960.0 * ln2) + yb6 * (371.0 + 960.0 * ln2)) + 3.0 * xb2 * (-7.0 + yb4 * (167.0 - 2688.0 * ln2) + 3.0 * yb6 * (11.0 + 448.0 * ln2) + yb2 * (31.0 + 1344.0 * ln2))
                     - 3.0 * xb8 * (-7.0 + yb4 * (167.0 - 2688.0 * ln2) + 3.0 * yb6 * (11.0 + 448.0 * ln2) + yb2 * (31.0 + 1344.0 * ln2)) + 2.0 * xb4 * (-11.0 + 3.0 * yb6 * (-241.0 + 704.0 * ln2) + yb2 * (-205.0 + 2112.0 * ln2) - yb4 * (757.0 + 4224.0 * ln2)) + xb6 * (22.0 + yb2 * (410.0 - 4224.0 * ln2) - 6.0 * yb6 * (-241.0 + 704.0 * ln2) + 2.0 * yb4 * (757.0 + 4224.0 * ln2))) + power_of<2>(vb) * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb14 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) - 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb13 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2))
                     + 2.0 * 1.0i * xb5 * (-193.0 - 192.0 * yb + 128.0 * yb3 + 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2)) + 2.0 * 1.0i * xb9 * (-193.0 - 192.0 * yb + 128.0 * yb3 + 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2)) + 4.0 * 1.0i * xb3 * (37.0 - 24.0 * yb - 48.0 * yb3 + 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) + 4.0 * 1.0i * xb11 * (37.0 - 24.0 * yb - 48.0 * yb3 + 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) - 8.0 * 1.0i * xb7 * (133.0 + 72.0 * yb - 112.0 * yb3 - 984.0 * yb5 + yb4 * (3883.0 - 4224.0 * ln2) + 11.0 * yb2 * (17.0 + 192.0 * ln2) + yb6 * (2293.0 + 2112.0 * ln2))
                     - xb4 * (171.0 + 144.0 * yb - 224.0 * yb3 - 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2)) + xb10 * (171.0 + 144.0 * yb - 224.0 * yb3 - 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2)) + xb2 * (-23.0 - 48.0 * yb - 96.0 * yb3 + 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) - xb12 * (-23.0 - 48.0 * yb - 96.0 * yb3 + 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) + xb8 * (147.0 + 96.0 * yb - 320.0 * yb3 - 1824.0 * yb5 + 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb2 * (-139.0 + 8640.0 * ln2) - yb4 * (691.0 + 17280.0 * ln2))
                     + xb6 * (-147.0 - 96.0 * yb + 320.0 * yb3 + 1824.0 * yb5 + yb2 * (139.0 - 8640.0 * ln2) - 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb4 * (691.0 + 17280.0 * ln2)))));
            const complex<double> logs2 = power_of<2>(-1.0i + xb) * (-1.0 + yb) * (1.0 + yb) * (-8.0 * tb * xb2 * (-4.0 * xb2 * (1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb10 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb9 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) - 8.0 * 1.0i * xb3 * (13.0 + 6.0 * yb - 4.0 * yb3 - 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2))
                     - 8.0 * 1.0i * xb7 * (13.0 + 6.0 * yb - 4.0 * yb3 - 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2)) + xb8 * (45.0 - 18.0 * yb + 28.0 * yb3 + 246.0 * yb5 + yb6 * (215.0 - 4032.0 * ln2) + 63.0 * yb4 * (9.0 + 128.0 * ln2) - 3.0 * yb2 * (73.0 + 1344.0 * ln2)) + 2.0 * xb4 * (-23.0 + 6.0 * yb - 20.0 * yb3 - 114.0 * yb5 + 3.0 * yb6 * (-335.0 + 704.0 * ln2) + 3.0 * yb2 * (-93.0 + 704.0 * ln2) - 3.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb6 * (46.0 - 12.0 * yb + 40.0 * yb3 + 228.0 * yb5 - 6.0 * yb6 * (-335.0 + 704.0 * ln2) - 6.0 * yb2 * (-93.0 + 704.0 * ln2) + 6.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb2 * (-45.0 + 18.0 * yb - 28.0 * yb3 - 246.0 * yb5 - 63.0 * yb4 * (9.0 + 128.0 * ln2) + 3.0 * yb2 * (73.0 + 1344.0 * ln2) + yb6 * (-215.0 + 4032.0 * ln2)) - 4.0 * 1.0i * xb5 * (51.0 + 18.0 * yb - 28.0 * yb3 - 246.0 * yb5 + yb4 * (2569.0 - 8064.0 * ln2) + yb2 * (43.0 + 4032.0 * ln2) + yb6 * (825.0 + 4032.0 * ln2)))
                   + 8.0 * vb * xb2 * (1.0 - 3.0 * yb2 * (-9.0 + 64.0 * ln2) - 3.0 * yb6 * (1.0 + 64.0 * ln2) + yb4 * (-57.0 + 384.0 * ln2) + xb10 * (-1.0 + yb4 * (57.0 - 384.0 * ln2) + 3.0 * yb2 * (-9.0 + 64.0 * ln2) + 3.0 * yb6 * (1.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb2 * (7.0 + 64.0 * ln2) + 3.0 * yb6 * (17.0 + 64.0 * ln2) - 3.0 * yb4 * (13.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb9 * (-1.0 + 3.0 * yb2 * (7.0 + 64.0 * ln2) + 3.0 * yb6 * (17.0 + 64.0 * ln2) - 3.0 * yb4 * (13.0 + 128.0 * ln2)) - 8.0 * 1.0i * xb3 * (7.0 + yb4 * (833.0 - 1920.0 * ln2) + yb2 * (5.0 + 960.0 * ln2) + yb6 * (211.0 + 960.0 * ln2)) - 8.0 * 1.0i * xb7 * (7.0 + yb4 * (833.0 - 1920.0 * ln2) + yb2 * (5.0 + 960.0 * ln2) + yb6 * (211.0 + 960.0 * ln2)) + xb8 * (69.0 + yb6 * (529.0 - 4032.0 * ln2) - 3.0 * yb2 * (115.0 + 1344.0 * ln2) + 3.0 * yb4 * (545.0 + 2688.0 * ln2)) + xb2 * (-69.0 - 3.0 * yb4 * (545.0 + 2688.0 * ln2) + yb6 * (-529.0 + 4032.0 * ln2) + yb2 * (345.0 + 4032.0 * ln2))
                     - 4.0 * 1.0i * xb5 * (27.0 + yb4 * (1757.0 - 8064.0 * ln2) + yb2 * (41.0 + 4032.0 * ln2) + yb6 * (383.0 + 4032.0 * ln2)) + xb4 * (-70.0 + 66.0 * yb6 * (-39.0 + 64.0 * ln2) + yb2 * (-706.0 + 4224.0 * ln2) - 2.0 * yb4 * (1301.0 + 4224.0 * ln2)) + xb6 * (70.0 + yb2 * (706.0 - 4224.0 * ln2) - 66.0 * yb6 * (-39.0 + 64.0 * ln2) + yb4 * (2602.0 + 8448.0 * ln2))) + power_of<2>(vb) * (1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb14 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2))
                     + 2.0 * 1.0i * xb13 * (-1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb5 * (-193.0 - 198.0 * yb + 116.0 * yb3 + 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) - 2.0 * 1.0i * xb9 * (-193.0 - 198.0 * yb + 116.0 * yb3 + 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) - 3.0 * xb4 * (49.0 + 54.0 * yb - 84.0 * yb3 - 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2)) + 3.0 * xb10 * (49.0 + 54.0 * yb - 84.0 * yb3 - 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2))
                     - 4.0 * 1.0i * xb3 * (25.0 - 18.0 * yb - 68.0 * yb3 - 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2)) - 4.0 * 1.0i * xb11 * (25.0 - 18.0 * yb - 68.0 * yb3 - 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2)) + 8.0 * 1.0i * xb7 * (121.0 + 78.0 * yb - 132.0 * yb3 - 1098.0 * yb5 + yb4 * (3611.0 - 4224.0 * ln2) + yb2 * (113.0 + 2112.0 * ln2) + yb6 * (2011.0 + 2112.0 * ln2)) + xb6 * (-99.0 - 114.0 * yb + 412.0 * yb3 + 2262.0 * yb5 + yb2 * (709.0 - 8640.0 * ln2) + yb6 * (1303.0 - 8640.0 * ln2) + 5.0 * yb4 * (251.0 + 3456.0 * ln2)) + xb2 * (-47.0 - 42.0 * yb - 148.0 * yb3 - 66.0 * yb5 + 69.0 * yb2 * (5.0 + 64.0 * ln2) - 3.0 * yb4 * (191.0 + 2944.0 * ln2) + yb6 * (-269.0 + 4416.0 * ln2))
                     + xb8 * (99.0 + 114.0 * yb - 412.0 * yb3 - 2262.0 * yb5 - 5.0 * yb4 * (251.0 + 3456.0 * ln2) + yb6 * (-1303.0 + 8640.0 * ln2) + yb2 * (-709.0 + 8640.0 * ln2)) + xb12 * (47.0 + 42.0 * yb + 148.0 * yb3 + 66.0 * yb5 + yb6 * (269.0 - 4416.0 * ln2) - 69.0 * yb2 * (5.0 + 64.0 * ln2) + yb4 * (573.0 + 8832.0 * ln2)))) + power_of<2>(tb) * (8.0 * vb * xb2 * (1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb14 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb13 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2))
                     - 2.0 * 1.0i * xb5 * (-193.0 + 198.0 * yb - 116.0 * yb3 - 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) - 2.0 * 1.0i * xb9 * (-193.0 + 198.0 * yb - 116.0 * yb3 - 2130.0 * yb5 + yb6 * (-3283.0 + 192.0 * ln2) + yb2 * (1447.0 + 192.0 * ln2) - yb4 * (12275.0 + 384.0 * ln2)) - 3.0 * xb4 * (49.0 - 54.0 * yb + 84.0 * yb3 + 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2)) + 3.0 * xb10 * (49.0 - 54.0 * yb + 84.0 * yb3 + 738.0 * yb5 + yb4 * (3171.0 - 2688.0 * ln2) + 7.0 * yb2 * (-1.0 + 192.0 * ln2) + yb6 * (1363.0 + 1344.0 * ln2)) - 4.0 * 1.0i * xb3 * (25.0 + 18.0 * yb + 68.0 * yb3 + 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2))
                     - 4.0 * 1.0i * xb11 * (25.0 + 18.0 * yb + 68.0 * yb3 + 42.0 * yb5 + yb4 * (2107.0 - 4224.0 * ln2) + 3.0 * yb6 * (201.0 + 704.0 * ln2) + yb2 * (49.0 + 2112.0 * ln2)) + 8.0 * 1.0i * xb7 * (121.0 - 78.0 * yb + 132.0 * yb3 + 1098.0 * yb5 + yb4 * (3611.0 - 4224.0 * ln2) + yb2 * (113.0 + 2112.0 * ln2) + yb6 * (2011.0 + 2112.0 * ln2)) + xb6 * (-99.0 + 114.0 * yb - 412.0 * yb3 - 2262.0 * yb5 + yb2 * (709.0 - 8640.0 * ln2) + yb6 * (1303.0 - 8640.0 * ln2) + 5.0 * yb4 * (251.0 + 3456.0 * ln2)) + xb2 * (-47.0 + 42.0 * yb + 148.0 * yb3 + 66.0 * yb5 + 69.0 * yb2 * (5.0 + 64.0 * ln2) - 3.0 * yb4 * (191.0 + 2944.0 * ln2) + yb6 * (-269.0 + 4416.0 * ln2)) - xb12 * (-47.0 + 42.0 * yb + 148.0 * yb3 + 66.0 * yb5 + 69.0 * yb2 * (5.0 + 64.0 * ln2) - 3.0 * yb4 * (191.0 + 2944.0 * ln2) + yb6 * (-269.0 + 4416.0 * ln2))
                     + xb8 * (99.0 - 114.0 * yb + 412.0 * yb3 + 2262.0 * yb5 - 5.0 * yb4 * (251.0 + 3456.0 * ln2) + yb6 * (-1303.0 + 8640.0 * ln2) + yb2 * (-709.0 + 8640.0 * ln2))) + power_of<2>(vb) * power_of<2>(-1.0 + xb2) * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb14 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb13 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb5 * (-385.0 + yb6 * (-5593.0 + 192.0 * ln2) + yb2 * (2873.0 + 192.0 * ln2) - yb4 * (21439.0 + 384.0 * ln2)) - 2.0 * 1.0i * xb9 * (-385.0 + yb6 * (-5593.0 + 192.0 * ln2) + yb2 * (2873.0 + 192.0 * ln2) - yb4 * (21439.0 + 384.0 * ln2))
                     - 4.0 * 1.0i * xb3 * (37.0 + yb4 * (2587.0 - 4224.0 * ln2) + yb2 * (67.0 + 2112.0 * ln2) + yb6 * (733.0 + 2112.0 * ln2)) - 4.0 * 1.0i * xb11 * (37.0 + yb4 * (2587.0 - 4224.0 * ln2) + yb2 * (67.0 + 2112.0 * ln2) + yb6 * (733.0 + 2112.0 * ln2)) + 8.0 * 1.0i * xb7 * (229.0 + yb4 * (7131.0 - 4224.0 * ln2) + 3.0 * yb2 * (65.0 + 704.0 * ln2) + yb6 * (4061.0 + 2112.0 * ln2)) - xb4 * (363.0 + yb4 * (19637.0 - 8064.0 * ln2) + yb2 * (-1411.0 + 4032.0 * ln2) + yb6 * (6659.0 + 4032.0 * ln2)) + xb10 * (363.0 + yb4 * (19637.0 - 8064.0 * ln2) + yb2 * (-1411.0 + 4032.0 * ln2) + yb6 * (6659.0 + 4032.0 * ln2)) + xb2 * (-23.0 + yb4 * (375.0 - 8832.0 * ln2) + 3.0 * yb6 * (-5.0 + 1472.0 * ln2) + yb2 * (399.0 + 4416.0 * ln2)) - xb6 * (339.0 + yb4 * (3661.0 - 17280.0 * ln2) + 5.0 * yb2 * (209.0 + 1728.0 * ln2) + yb6 * (4587.0 + 8640.0 * ln2))
                     + xb8 * (339.0 + yb4 * (3661.0 - 17280.0 * ln2) + 5.0 * yb2 * (209.0 + 1728.0 * ln2) + yb6 * (4587.0 + 8640.0 * ln2)) + xb12 * (23.0 + yb6 * (15.0 - 4416.0 * ln2) - 3.0 * yb2 * (133.0 + 1472.0 * ln2) + yb4 * (-375.0 + 8832.0 * ln2))) - 4.0 * xb2 * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb14 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb13 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) - 2.0 * 1.0i * xb5 * (-193.0 + 192.0 * yb - 128.0 * yb3 - 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2))
                     - 2.0 * 1.0i * xb9 * (-193.0 + 192.0 * yb - 128.0 * yb3 - 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2)) - 4.0 * 1.0i * xb3 * (37.0 + 24.0 * yb + 48.0 * yb3 - 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) - 4.0 * 1.0i * xb11 * (37.0 + 24.0 * yb + 48.0 * yb3 - 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) + 8.0 * 1.0i * xb7 * (133.0 - 72.0 * yb + 112.0 * yb3 + 984.0 * yb5 + yb4 * (3883.0 - 4224.0 * ln2) + 11.0 * yb2 * (17.0 + 192.0 * ln2) + yb6 * (2293.0 + 2112.0 * ln2)) - xb4 * (171.0 - 144.0 * yb + 224.0 * yb3 + 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2))
                     + xb10 * (171.0 - 144.0 * yb + 224.0 * yb3 + 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2)) + xb2 * (-23.0 + 48.0 * yb + 96.0 * yb3 - 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) - xb12 * (-23.0 + 48.0 * yb + 96.0 * yb3 - 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) - xb6 * (147.0 - 96.0 * yb + 320.0 * yb3 + 1824.0 * yb5 + 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb2 * (-139.0 + 8640.0 * ln2) - yb4 * (691.0 + 17280.0 * ln2)) + xb8 * (147.0 - 96.0 * yb + 320.0 * yb3 + 1824.0 * yb5 + 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb2 * (-139.0 + 8640.0 * ln2) - yb4 * (691.0 + 17280.0 * ln2))))
                 - 4.0 * xb2 * (8.0 * vb * xb2 * (1.0 - 6.0 * yb - 12.0 * yb3 + 18.0 * yb5 + yb6 * (3.0 - 192.0 * ln2) + yb2 * (9.0 - 192.0 * ln2) + yb4 * (-45.0 + 384.0 * ln2) + xb10 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + yb4 * (45.0 - 384.0 * ln2) + 3.0 * yb2 * (-3.0 + 64.0 * ln2) + 3.0 * yb6 * (-1.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb9 * (-1.0 + 6.0 * yb + 12.0 * yb3 - 18.0 * yb5 + 3.0 * yb2 * (13.0 + 64.0 * ln2) + 3.0 * yb6 * (15.0 + 64.0 * ln2) - 3.0 * yb4 * (17.0 + 128.0 * ln2)) - 8.0 * 1.0i * xb3 * (13.0 - 6.0 * yb + 4.0 * yb3 + 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2)) - 8.0 * 1.0i * xb7 * (13.0 - 6.0 * yb + 4.0 * yb3 + 66.0 * yb5 + yb4 * (1103.0 - 1920.0 * ln2) + 3.0 * yb6 * (97.0 + 320.0 * ln2) + yb2 * (-31.0 + 960.0 * ln2))
                     + xb8 * (45.0 + 18.0 * yb - 28.0 * yb3 - 246.0 * yb5 + yb6 * (215.0 - 4032.0 * ln2) + 63.0 * yb4 * (9.0 + 128.0 * ln2) - 3.0 * yb2 * (73.0 + 1344.0 * ln2)) + 2.0 * xb4 * (-23.0 - 6.0 * yb + 20.0 * yb3 + 114.0 * yb5 + 3.0 * yb6 * (-335.0 + 704.0 * ln2) + 3.0 * yb2 * (-93.0 + 704.0 * ln2) - 3.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb6 * (46.0 + 12.0 * yb - 40.0 * yb3 - 228.0 * yb5 - 6.0 * yb6 * (-335.0 + 704.0 * ln2) - 6.0 * yb2 * (-93.0 + 704.0 * ln2) + 6.0 * yb4 * (343.0 + 1408.0 * ln2)) + xb2 * (-45.0 - 18.0 * yb + 28.0 * yb3 + 246.0 * yb5 - 63.0 * yb4 * (9.0 + 128.0 * ln2) + 3.0 * yb2 * (73.0 + 1344.0 * ln2) + yb6 * (-215.0 + 4032.0 * ln2)) - 4.0 * 1.0i * xb5 * (51.0 - 18.0 * yb + 28.0 * yb3 + 246.0 * yb5 + yb4 * (2569.0 - 8064.0 * ln2) + yb2 * (43.0 + 4032.0 * ln2) + yb6 * (825.0 + 4032.0 * ln2)))
                   - 4.0 * xb2 * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb10 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb9 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) - 4.0 * 1.0i * xb5 * (75.0 + yb4 * (3381.0 - 8064.0 * ln2) + 9.0 * yb2 * (5.0 + 448.0 * ln2) + 7.0 * yb6 * (181.0 + 576.0 * ln2)) - 8.0 * 1.0i * xb3 * (19.0 + yb4 * (1373.0 - 1920.0 * ln2) + yb2 * (-67.0 + 960.0 * ln2) + yb6 * (371.0 + 960.0 * ln2)) - 8.0 * 1.0i * xb7 * (19.0 + yb4 * (1373.0 - 1920.0 * ln2) + yb2 * (-67.0 + 960.0 * ln2) + yb6 * (371.0 + 960.0 * ln2)) + 3.0 * xb2 * (-7.0 + yb4 * (167.0 - 2688.0 * ln2) + 3.0 * yb6 * (11.0 + 448.0 * ln2) + yb2 * (31.0 + 1344.0 * ln2))
                     - 3.0 * xb8 * (-7.0 + yb4 * (167.0 - 2688.0 * ln2) + 3.0 * yb6 * (11.0 + 448.0 * ln2) + yb2 * (31.0 + 1344.0 * ln2)) + 2.0 * xb4 * (-11.0 + 3.0 * yb6 * (-241.0 + 704.0 * ln2) + yb2 * (-205.0 + 2112.0 * ln2) - yb4 * (757.0 + 4224.0 * ln2)) + xb6 * (22.0 + yb2 * (410.0 - 4224.0 * ln2) - 6.0 * yb6 * (-241.0 + 704.0 * ln2) + 2.0 * yb4 * (757.0 + 4224.0 * ln2))) + power_of<2>(vb) * (1.0 + yb6 * (9.0 - 192.0 * ln2) - 3.0 * yb2 * (3.0 + 64.0 * ln2) + yb4 * (-33.0 + 384.0 * ln2) + xb14 * (-1.0 + yb4 * (33.0 - 384.0 * ln2) + 3.0 * yb6 * (-3.0 + 64.0 * ln2) + 3.0 * yb2 * (3.0 + 64.0 * ln2)) + 2.0 * 1.0i * xb * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2)) + 2.0 * 1.0i * xb13 * (-1.0 + 3.0 * yb6 * (13.0 + 64.0 * ln2) + 3.0 * yb2 * (19.0 + 64.0 * ln2) - 3.0 * yb4 * (21.0 + 128.0 * ln2))
                     - 2.0 * 1.0i * xb5 * (-193.0 - 192.0 * yb + 128.0 * yb3 + 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2)) - 2.0 * 1.0i * xb9 * (-193.0 - 192.0 * yb + 128.0 * yb3 + 2112.0 * yb5 + 3.0 * yb6 * (-1011.0 + 64.0 * ln2) + yb2 * (1721.0 + 192.0 * ln2) - yb4 * (12799.0 + 384.0 * ln2)) - 4.0 * 1.0i * xb3 * (37.0 - 24.0 * yb - 48.0 * yb3 + 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) - 4.0 * 1.0i * xb11 * (37.0 - 24.0 * yb - 48.0 * yb3 + 72.0 * yb5 + yb4 * (2635.0 - 4224.0 * ln2) + yb2 * (-5.0 + 2112.0 * ln2) + yb6 * (757.0 + 2112.0 * ln2)) + 8.0 * 1.0i * xb7 * (133.0 + 72.0 * yb - 112.0 * yb3 - 984.0 * yb5 + yb4 * (3883.0 - 4224.0 * ln2) + 11.0 * yb2 * (17.0 + 192.0 * ln2) + yb6 * (2293.0 + 2112.0 * ln2))
                     - xb4 * (171.0 + 144.0 * yb - 224.0 * yb3 - 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2)) + xb10 * (171.0 + 144.0 * yb - 224.0 * yb3 - 1968.0 * yb5 + yb4 * (11093.0 - 8064.0 * ln2) + yb2 * (-403.0 + 4032.0 * ln2) + yb6 * (4147.0 + 4032.0 * ln2)) + xb2 * (-23.0 - 48.0 * yb - 96.0 * yb3 + 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) - xb12 * (-23.0 - 48.0 * yb - 96.0 * yb3 + 144.0 * yb5 + yb4 * (471.0 - 8832.0 * ln2) + 3.0 * yb2 * (85.0 + 1472.0 * ln2) + yb6 * (33.0 + 4416.0 * ln2)) + xb8 * (147.0 + 96.0 * yb - 320.0 * yb3 - 1824.0 * yb5 + 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb2 * (-139.0 + 8640.0 * ln2) - yb4 * (691.0 + 17280.0 * ln2))
                     + xb6 * (-147.0 - 96.0 * yb + 320.0 * yb3 + 1824.0 * yb5 + yb2 * (139.0 - 8640.0 * ln2) - 15.0 * yb6 * (5.0 + 576.0 * ln2) + yb4 * (691.0 + 17280.0 * ln2)))));
            const complex<double> logs3 = (-1.0 + yb) * (1.0 + yb) * (-4.0 * xb2 * (2.0 * xb6 * (12.0 + 9.0 * yb - 14.0 * yb3 - 123.0 * yb5 + yb2 * (1.0 - 186.0 * ln2) + yb6 * (221.0 - 186.0 * ln2) + yb4 * (406.0 + 372.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb12 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) - 3.0 * xb2 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-1.0 + yb2 * (-1.0 + 18.0 * ln2) + 2.0 * yb * (1.0 + log(512.0))) - 3.0 * xb10 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-1.0 + yb2 * (-1.0 + 18.0 * ln2) + 2.0 * yb * (1.0 + log(512.0))) + xb4 * (12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 + yb6 * (160.0 - 243.0 * ln2) - 9.0 * yb2 * (8.0 + 27.0 * ln2) + 54.0 * yb4 * (10.0 + log(512.0))) + xb8 * (12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 + yb6 * (160.0 - 243.0 * ln2) - 9.0 * yb2 * (8.0 + 27.0 * ln2) + 54.0 * yb4 * (10.0 + log(512.0))))
                 + power_of<2>(vb) * (-60.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * ln2 - 60.0 * xb14 * yb2 * power_of<2>(-1.0 + yb2) * ln2 + 2.0 * xb8 * (-48.0 - 36.0 * yb + 56.0 * yb3 + 492.0 * yb5 + yb6 * (-884.0 + 129.0 * ln2) + yb2 * (-4.0 + 129.0 * ln2) - 2.0 * yb4 * (812.0 + 129.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb16 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + 4.0 * xb6 * (-12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 - 30.0 * yb4 * (18.0 + ln2) + 5.0 * yb6 * (-32.0 + log(8.0)) + 3.0 * yb2 * (24.0 + log(32.0))) + 4.0 * xb10 * (-12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 - 30.0 * yb4 * (18.0 + ln2) + 5.0 * yb6 * (-32.0 + log(8.0)) + 3.0 * yb2 * (24.0 + log(32.0))) - 12.0 * xb4 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + log(2048.0)) + yb2 * (1.0 + log(2048.0))) - 12.0 * xb12 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + log(2048.0)) + yb2 * (1.0 + log(2048.0))))
                 + vb * xb2 * (-4.0 * xb6 * (-30.0 - 21.0 * yb + 38.0 * yb3 + 303.0 * yb5 - 12.0 * yb4 * (79.0 + 124.0 * ln2) + yb6 * (-583.0 + 744.0 * ln2) + yb2 * (-39.0 + 744.0 * ln2)) + xb4 * (48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb6 * (515.0 - 1944.0 * ln2) - yb2 * (425.0 + 1944.0 * ln2) + yb4 * (2422.0 + 3888.0 * ln2)) + xb8 * (48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb6 * (515.0 - 1944.0 * ln2) - yb2 * (425.0 + 1944.0 * ln2) + yb4 * (2422.0 + 3888.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-1.0 + yb2 * (-1.0 + ln256) + yb * (2.0 + ln256)) + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-1.0 + yb2 * (-1.0 + ln256) + yb * (2.0 + ln256)) - 2.0 * xb2 * (6.0 - 3.0 * yb - 22.0 * yb3 - 39.0 * yb5 + 9.0 * yb2 * (-1.0 + 24.0 * ln2) + yb6 * (71.0 + 216.0 * ln2) - 36.0 * yb4 * (-7.0 + log(4096.0)))
                   - 2.0 * xb10 * (6.0 - 3.0 * yb - 22.0 * yb3 - 39.0 * yb5 + 9.0 * yb2 * (-1.0 + 24.0 * ln2) + yb6 * (71.0 + 216.0 * ln2) - 36.0 * yb4 * (-7.0 + log(4096.0)))));
            const complex<double> logs4 = (-1.0 + yb) * (1.0 + yb) * (-4.0 * xb2 * (2.0 * xb6 * (12.0 - 9.0 * yb + 14.0 * yb3 + 123.0 * yb5 + yb2 * (1.0 - 186.0 * ln2) + yb6 * (221.0 - 186.0 * ln2) + yb4 * (406.0 + 372.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb12 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) - 3.0 * xb2 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-1.0 + yb2 * (-1.0 + 18.0 * ln2) - 2.0 * yb * (1.0 + log(512.0))) - 3.0 * xb10 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-1.0 + yb2 * (-1.0 + 18.0 * ln2) - 2.0 * yb * (1.0 + log(512.0))) + xb4 * (12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 + yb6 * (160.0 - 243.0 * ln2) - 9.0 * yb2 * (8.0 + 27.0 * ln2) + 54.0 * yb4 * (10.0 + log(512.0))) + xb8 * (12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 + yb6 * (160.0 - 243.0 * ln2) - 9.0 * yb2 * (8.0 + 27.0 * ln2) + 54.0 * yb4 * (10.0 + log(512.0))))
                 + power_of<2>(tb) * (-60.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * ln2 - 60.0 * xb14 * yb2 * power_of<2>(-1.0 + yb2) * ln2 + 2.0 * xb8 * (-48.0 + 36.0 * yb - 56.0 * yb3 - 492.0 * yb5 + yb6 * (-884.0 + 129.0 * ln2) + yb2 * (-4.0 + 129.0 * ln2) - 2.0 * yb4 * (812.0 + 129.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb16 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + 4.0 * xb6 * (-12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 - 30.0 * yb4 * (18.0 + ln2) + 5.0 * yb6 * (-32.0 + log(8.0)) + 3.0 * yb2 * (24.0 + log(32.0))) + 4.0 * xb10 * (-12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 - 30.0 * yb4 * (18.0 + ln2) + 5.0 * yb6 * (-32.0 + log(8.0)) + 3.0 * yb2 * (24.0 + log(32.0))) - 12.0 * xb4 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 11.0 * ln2) + yb2 * (1.0 + log(2048.0))) - 12.0 * xb12 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 11.0 * ln2) + yb2 * (1.0 + log(2048.0))))
                 + tb * xb2 * (4.0 * xb6 * (-30.0 + 21.0 * yb - 38.0 * yb3 - 303.0 * yb5 - 12.0 * yb4 * (79.0 + 124.0 * ln2) + yb6 * (-583.0 + 744.0 * ln2) + yb2 * (-39.0 + 744.0 * ln2)) + xb4 * (-48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb6 * (-515.0 + 1944.0 * ln2) + yb2 * (425.0 + 1944.0 * ln2) - 2.0 * yb4 * (1211.0 + 1944.0 * ln2)) + xb8 * (-48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb6 * (-515.0 + 1944.0 * ln2) + yb2 * (425.0 + 1944.0 * ln2) - 2.0 * yb4 * (1211.0 + 1944.0 * ln2)) - 3.0 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-1.0 - 2.0 * yb * (1.0 + log(16.0)) + yb2 * (-1.0 + ln256)) - 3.0 * xb12 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-1.0 - 2.0 * yb * (1.0 + log(16.0)) + yb2 * (-1.0 + ln256)) + 2.0 * xb2 * (6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + 9.0 * yb2 * (-1.0 + 24.0 * ln2) + yb6 * (71.0 + 216.0 * ln2) - 36.0 * yb4 * (-7.0 + log(4096.0)))
                   + 2.0 * xb10 * (6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + 9.0 * yb2 * (-1.0 + 24.0 * ln2) + yb6 * (71.0 + 216.0 * ln2) - 36.0 * yb4 * (-7.0 + log(4096.0)))));
            const complex<double> logs5 = (-1.0 + yb) * (1.0 + yb) * (-4.0 * xb2 * (-4.0 * xb6 * (12.0 - 9.0 * yb + 14.0 * yb3 + 123.0 * yb5 + yb4 * (406.0 - 186.0 * ln2) + yb2 * (1.0 + 93.0 * ln2) + yb6 * (221.0 + 93.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb12 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) - xb4 * (24.0 - 24.0 * yb + 16.0 * yb3 + 264.0 * yb5 + 9.0 * yb2 * (-16.0 + 27.0 * ln2) + yb6 * (320.0 + 243.0 * ln2) - 54.0 * yb4 * (-20.0 + log(512.0))) - xb8 * (24.0 - 24.0 * yb + 16.0 * yb3 + 264.0 * yb5 + 9.0 * yb2 * (-16.0 + 27.0 * ln2) + yb6 * (320.0 + 243.0 * ln2) - 54.0 * yb4 * (-20.0 + log(512.0))) - 6.0 * xb2 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 9.0 * ln2) + yb2 * (1.0 + log(512.0))) - 6.0 * xb10 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 9.0 * ln2) + yb2 * (1.0 + log(512.0))))
                 + power_of<2>(tb) * (-60.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * ln2 - 60.0 * xb14 * yb2 * power_of<2>(-1.0 + yb2) * ln2 + 2.0 * xb8 * (96.0 - 72.0 * yb + 112.0 * yb3 + 984.0 * yb5 + yb4 * (3248.0 - 258.0 * ln2) + yb2 * (8.0 + 129.0 * ln2) + yb6 * (1768.0 + 129.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb16 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + 4.0 * xb6 * (24.0 - 24.0 * yb + 16.0 * yb3 + 264.0 * yb5 - 30.0 * yb4 * (-36.0 + ln2) + 5.0 * yb6 * (64.0 + log(8.0)) + 3.0 * yb2 * (-48.0 + log(32.0))) + 4.0 * xb10 * (24.0 - 24.0 * yb + 16.0 * yb3 + 264.0 * yb5 - 30.0 * yb4 * (-36.0 + ln2) + 5.0 * yb6 * (64.0 + log(8.0)) + 3.0 * yb2 * (-48.0 + log(32.0))) - 12.0 * xb4 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-2.0 + yb2 * (-2.0 + log(2048.0)) - yb * (4.0 + log(2048.0))) - 12.0 * xb12 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-2.0 + yb2 * (-2.0 + log(2048.0)) - yb * (4.0 + log(2048.0))))
                 - 2.0 * tb * xb2 * (-4.0 * xb6 * (30.0 - 21.0 * yb + 38.0 * yb3 + 303.0 * yb5 + yb4 * (948.0 - 744.0 * ln2) + yb2 * (39.0 + 372.0 * ln2) + yb6 * (583.0 + 372.0 * ln2)) - xb4 * (48.0 - 51.0 * yb + 26.0 * yb3 + 537.0 * yb5 + yb4 * (2422.0 - 1944.0 * ln2) + yb2 * (-425.0 + 972.0 * ln2) + yb6 * (515.0 + 972.0 * ln2)) - xb8 * (48.0 - 51.0 * yb + 26.0 * yb3 + 537.0 * yb5 + yb4 * (2422.0 - 1944.0 * ln2) + yb2 * (-425.0 + 972.0 * ln2) + yb6 * (515.0 + 972.0 * ln2)) + 3.0 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 4.0 * ln2) + yb2 * (1.0 + log(16.0))) + 3.0 * xb12 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 4.0 * ln2) + yb2 * (1.0 + log(16.0))) + 2.0 * xb2 * (6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + yb6 * (71.0 - 108.0 * ln2) + 36.0 * yb4 * (7.0 + ln64) - 9.0 * yb2 * (1.0 + log(4096.0))) + 2.0 * xb10 * (6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + yb6 * (71.0 - 108.0 * ln2) + 36.0 * yb4 * (7.0 + ln64) - 9.0 * yb2 * (1.0 + log(4096.0)))));
            const complex<double> logs6 = (-1.0 + yb) * (1.0 + yb) * (-4.0 * xb2 * (-4.0 * xb6 * (12.0 + 9.0 * yb - 14.0 * yb3 - 123.0 * yb5 + yb4 * (406.0 - 186.0 * ln2) + yb2 * (1.0 + 93.0 * ln2) + yb6 * (221.0 + 93.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb12 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) - xb4 * (24.0 + 24.0 * yb - 16.0 * yb3 - 264.0 * yb5 + 9.0 * yb2 * (-16.0 + 27.0 * ln2) + yb6 * (320.0 + 243.0 * ln2) - 54.0 * yb4 * (-20.0 + log(512.0))) - xb8 * (24.0 + 24.0 * yb - 16.0 * yb3 - 264.0 * yb5 + 9.0 * yb2 * (-16.0 + 27.0 * ln2) + yb6 * (320.0 + 243.0 * ln2) - 54.0 * yb4 * (-20.0 + log(512.0))) - 6.0 * xb2 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + log(512.0)) + yb2 * (1.0 + log(512.0))) - 6.0 * xb10 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + log(512.0)) + yb2 * (1.0 + log(512.0))))
                 + power_of<2>(vb) * (-60.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * ln2 - 60.0 * xb14 * yb2 * power_of<2>(-1.0 + yb2) * ln2 + 2.0 * xb8 * (96.0 + 72.0 * yb - 112.0 * yb3 - 984.0 * yb5 + yb4 * (3248.0 - 258.0 * ln2) + yb2 * (8.0 + 129.0 * ln2) + yb6 * (1768.0 + 129.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb16 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + 4.0 * xb6 * (24.0 + 24.0 * yb - 16.0 * yb3 - 264.0 * yb5 - 30.0 * yb4 * (-36.0 + ln2) + 5.0 * yb6 * (64.0 + log(8.0)) + 3.0 * yb2 * (-48.0 + log(32.0))) + 4.0 * xb10 * (24.0 + 24.0 * yb - 16.0 * yb3 - 264.0 * yb5 - 30.0 * yb4 * (-36.0 + ln2) + 5.0 * yb6 * (64.0 + log(8.0)) + 3.0 * yb2 * (-48.0 + log(32.0))) - 12.0 * xb4 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-2.0 + yb2 * (-2.0 + log(2048.0)) + yb * (4.0 + log(2048.0))) - 12.0 * xb12 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-2.0 + yb2 * (-2.0 + log(2048.0)) + yb * (4.0 + log(2048.0))))
                 + 2.0 * vb * xb2 * (-4.0 * xb6 * (30.0 + 21.0 * yb - 38.0 * yb3 - 303.0 * yb5 + yb4 * (948.0 - 744.0 * ln2) + yb2 * (39.0 + 372.0 * ln2) + yb6 * (583.0 + 372.0 * ln2)) - xb4 * (48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb4 * (2422.0 - 1944.0 * ln2) + yb2 * (-425.0 + 972.0 * ln2) + yb6 * (515.0 + 972.0 * ln2)) - xb8 * (48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb4 * (2422.0 - 1944.0 * ln2) + yb2 * (-425.0 + 972.0 * ln2) + yb6 * (515.0 + 972.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + log(16.0)) + yb2 * (1.0 + log(16.0))) + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + log(16.0)) + yb2 * (1.0 + log(16.0))) - 2.0 * xb2 * (-6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + yb6 * (-71.0 + 108.0 * ln2) - 36.0 * yb4 * (7.0 + ln64) + 9.0 * yb2 * (1.0 + log(4096.0)))
                   - 2.0 * xb10 * (-6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + yb6 * (-71.0 + 108.0 * ln2) - 36.0 * yb4 * (7.0 + ln64) + 9.0 * yb2 * (1.0 + log(4096.0)))));
            const complex<double> logs7 = (-1.0 + yb) * (1.0 + yb) * (-4.0 * xb2 * (-3.0 * xb2 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 18.0 * ln2) + yb2 * (1.0 + 18.0 * ln2)) - 3.0 * xb10 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 18.0 * ln2) + yb2 * (1.0 + 18.0 * ln2)) - 2.0 * xb6 * (12.0 - 9.0 * yb + 14.0 * yb3 + 123.0 * yb5 + yb4 * (406.0 - 372.0 * ln2) + yb2 * (1.0 + 186.0 * ln2) + yb6 * (221.0 + 186.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb12 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) - xb4 * (12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 + 9.0 * yb2 * (-8.0 + 27.0 * ln2) + yb6 * (160.0 + 243.0 * ln2) - 54.0 * yb4 * (-10.0 + log(512.0))) - xb8 * (12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 + 9.0 * yb2 * (-8.0 + 27.0 * ln2) + yb6 * (160.0 + 243.0 * ln2) - 54.0 * yb4 * (-10.0 + log(512.0))))
                 + power_of<2>(tb) * (-60.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * ln2 - 60.0 * xb14 * yb2 * power_of<2>(-1.0 + yb2) * ln2 + 2.0 * xb8 * (48.0 - 36.0 * yb + 56.0 * yb3 + 492.0 * yb5 + yb4 * (1624.0 - 258.0 * ln2) + yb2 * (4.0 + 129.0 * ln2) + yb6 * (884.0 + 129.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb16 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + 4.0 * xb6 * (12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 - 30.0 * yb4 * (-18.0 + ln2) + 5.0 * yb6 * (32.0 + log(8.0)) + 3.0 * yb2 * (-24.0 + log(32.0))) + 4.0 * xb10 * (12.0 - 12.0 * yb + 8.0 * yb3 + 132.0 * yb5 - 30.0 * yb4 * (-18.0 + ln2) + 5.0 * yb6 * (32.0 + log(8.0)) + 3.0 * yb2 * (-24.0 + log(32.0))) - 12.0 * xb4 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-1.0 + yb2 * (-1.0 + log(2048.0)) - yb * (2.0 + log(2048.0))) - 12.0 * xb12 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (-1.0 + yb2 * (-1.0 + log(2048.0)) - yb * (2.0 + log(2048.0))))
                 + tb * xb2 * (4.0 * xb6 * (30.0 - 21.0 * yb + 38.0 * yb3 + 303.0 * yb5 + yb4 * (948.0 - 1488.0 * ln2) + yb2 * (39.0 + 744.0 * ln2) + yb6 * (583.0 + 744.0 * ln2)) + xb4 * (48.0 - 51.0 * yb + 26.0 * yb3 + 537.0 * yb5 + yb4 * (2422.0 - 3888.0 * ln2) + yb2 * (-425.0 + 1944.0 * ln2) + yb6 * (515.0 + 1944.0 * ln2)) + xb8 * (48.0 - 51.0 * yb + 26.0 * yb3 + 537.0 * yb5 + yb4 * (2422.0 - 3888.0 * ln2) + yb2 * (-425.0 + 1944.0 * ln2) + yb6 * (515.0 + 1944.0 * ln2)) - 3.0 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 8.0 * ln2) + yb2 * (1.0 + ln256)) - 3.0 * xb12 * (-1.0 + yb) * yb * power_of<2>(1.0 + yb) * (1.0 + yb * (2.0 - 8.0 * ln2) + yb2 * (1.0 + ln256)) + 2.0 * xb2 * (-6.0 - 3.0 * yb - 22.0 * yb3 - 39.0 * yb5 + 9.0 * yb2 * (1.0 + 24.0 * ln2) + yb6 * (-71.0 + 216.0 * ln2) - 36.0 * yb4 * (7.0 + log(4096.0)))
                   + 2.0 * xb10 * (-6.0 - 3.0 * yb - 22.0 * yb3 - 39.0 * yb5 + 9.0 * yb2 * (1.0 + 24.0 * ln2) + yb6 * (-71.0 + 216.0 * ln2) - 36.0 * yb4 * (7.0 + log(4096.0)))));
            const complex<double> logs8 = (-1.0 + yb) * (1.0 + yb) * (-4.0 * xb2 * (-2.0 * xb6 * (12.0 + 9.0 * yb - 14.0 * yb3 - 123.0 * yb5 + yb4 * (406.0 - 372.0 * ln2) + yb2 * (1.0 + 186.0 * ln2) + yb6 * (221.0 + 186.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb12 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) - xb4 * (12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 + 9.0 * yb2 * (-8.0 + 27.0 * ln2) + yb6 * (160.0 + 243.0 * ln2) - 54.0 * yb4 * (-10.0 + log(512.0))) - xb8 * (12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 + 9.0 * yb2 * (-8.0 + 27.0 * ln2) + yb6 * (160.0 + 243.0 * ln2) - 54.0 * yb4 * (-10.0 + log(512.0))) - 3.0 * xb2 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb2 * (1.0 + 18.0 * ln2) + 2.0 * yb * (-1.0 + log(512.0))) - 3.0 * xb10 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb2 * (1.0 + 18.0 * ln2) + 2.0 * yb * (-1.0 + log(512.0))))
                 + power_of<2>(vb) * (-60.0 * xb2 * yb2 * power_of<2>(-1.0 + yb2) * ln2 - 60.0 * xb14 * yb2 * power_of<2>(-1.0 + yb2) * ln2 + 2.0 * xb8 * (48.0 + 36.0 * yb - 56.0 * yb3 - 492.0 * yb5 + yb4 * (1624.0 - 258.0 * ln2) + yb2 * (4.0 + 129.0 * ln2) + yb6 * (884.0 + 129.0 * ln2)) + yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + xb16 * yb2 * (-6.0 * yb2 * ln2 + log(8.0) + yb4 * log(8.0)) + 4.0 * xb6 * (12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 - 30.0 * yb4 * (-18.0 + ln2) + 5.0 * yb6 * (32.0 + log(8.0)) + 3.0 * yb2 * (-24.0 + log(32.0))) + 4.0 * xb10 * (12.0 + 12.0 * yb - 8.0 * yb3 - 132.0 * yb5 - 30.0 * yb4 * (-18.0 + ln2) + 5.0 * yb6 * (32.0 + log(8.0)) + 3.0 * yb2 * (-24.0 + log(32.0))) - 12.0 * xb4 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-1.0 + yb2 * (-1.0 + log(2048.0)) + yb * (2.0 + log(2048.0))) - 12.0 * xb12 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (-1.0 + yb2 * (-1.0 + log(2048.0)) + yb * (2.0 + log(2048.0))))
                 + vb * xb2 * (-4.0 * xb6 * (30.0 + 21.0 * yb - 38.0 * yb3 - 303.0 * yb5 + yb4 * (948.0 - 1488.0 * ln2) + yb2 * (39.0 + 744.0 * ln2) + yb6 * (583.0 + 744.0 * ln2)) - xb4 * (48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb4 * (2422.0 - 3888.0 * ln2) + yb2 * (-425.0 + 1944.0 * ln2) + yb6 * (515.0 + 1944.0 * ln2)) - xb8 * (48.0 + 51.0 * yb - 26.0 * yb3 - 537.0 * yb5 + yb4 * (2422.0 - 3888.0 * ln2) + yb2 * (-425.0 + 1944.0 * ln2) + yb6 * (515.0 + 1944.0 * ln2)) + 3.0 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + ln256) + yb2 * (1.0 + ln256)) + 3.0 * xb12 * power_of<2>(-1.0 + yb) * yb * (1.0 + yb) * (1.0 + yb * (-2.0 + ln256) + yb2 * (1.0 + ln256)) - 2.0 * xb2 * (-6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + 9.0 * yb2 * (1.0 + 24.0 * ln2) + yb6 * (-71.0 + 216.0 * ln2) - 36.0 * yb4 * (7.0 + log(4096.0)))
                   - 2.0 * xb10 * (-6.0 + 3.0 * yb + 22.0 * yb3 + 39.0 * yb5 + 9.0 * yb2 * (1.0 + 24.0 * ln2) + yb6 * (-71.0 + 216.0 * ln2) - 36.0 * yb4 * (7.0 + log(4096.0)))));
            const complex<double> logs9 = power_of<2>(1.0i + xb) * (-1.0 + yb) * (1.0 + yb) * (power_of<2>(tb) * (power_of<2>(vb) * power_of<2>(-1.0 + xb2) * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 - 4.0 * 1.0i * xb3 * (2.0 + 37.0 * ln2 + yb2 * (6.0 + 67.0 * ln2) + yb6 * (-54.0 + 733.0 * ln2) + yb4 * (-18.0 + 2587.0 * ln2)) - 4.0 * 1.0i * xb11 * (2.0 + 37.0 * ln2 + yb2 * (6.0 + 67.0 * ln2) + yb6 * (-54.0 + 733.0 * ln2) + yb4 * (-18.0 + 2587.0 * ln2)) + xb6 * (8.0 + 339.0 * ln2 + yb2 * (-168.0 + 1045.0 * ln2) + 3.0 * yb6 * (-648.0 + 1529.0 * ln2) + yb4 * (-1224.0 + 3661.0 * ln2)) - xb8 * (8.0 + 339.0 * ln2 + yb2 * (-168.0 + 1045.0 * ln2) + 3.0 * yb6 * (-648.0 + 1529.0 * ln2) + yb4 * (-1224.0 + 3661.0 * ln2))
                     + 8.0 * 1.0i * xb7 * (2.0 + 229.0 * ln2 + 13.0 * yb2 * (-2.0 + 15.0 * ln2) + 3.0 * yb4 * (-70.0 + 2377.0 * ln2) + yb6 * (-342.0 + 4061.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 1411.0 * ln2) + 363.0 * ln2 + yb6 * (468.0 + 6659.0 * ln2) + yb4 * (348.0 + 19637.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 1411.0 * ln2) + 363.0 * ln2 + yb6 * (468.0 + 6659.0 * ln2) + yb4 * (348.0 + 19637.0 * ln2)) + 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 2873.0 * ln2) + 385.0 * ln2 + yb6 * (576.0 + 5593.0 * ln2) + yb4 * (384.0 + 21439.0 * ln2)) + 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 2873.0 * ln2) + 385.0 * ln2 + yb6 * (576.0 + 5593.0 * ln2) + yb4 * (384.0 + 21439.0 * ln2)) + xb12 * (4.0 - 23.0 * ln2 + 3.0 * yb2 * (4.0 + 133.0 * ln2) + yb4 * (-36.0 + 375.0 * ln2) - 3.0 * yb6 * (36.0 + log(32.0))) + xb2 * (-4.0 + yb4 * (36.0 - 375.0 * ln2) + 23.0 * ln2 - 3.0 * yb2 * (4.0 + 133.0 * ln2) + 3.0 * yb6 * (36.0 + log(32.0))))
                   + 8.0 * vb * xb2 * (-ln2 - 9.0 * yb2 * ln2 + 12.0 * yb3 * ln2 + 45.0 * yb4 * ln2 - 18.0 * yb5 * ln2 + 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 + 2.0 * 1.0i * xb13 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 + xb12 * (4.0 - 47.0 * ln2 + 42.0 * yb * ln2 + 148.0 * yb3 * ln2 + 66.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 115.0 * ln2) - 3.0 * yb4 * (12.0 + 191.0 * ln2) - yb6 * (108.0 + 269.0 * ln2)) + xb2 * (-4.0 + 47.0 * ln2 - 42.0 * yb * ln2 - 148.0 * yb3 * ln2 - 66.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 269.0 * ln2) + yb4 * (36.0 + 573.0 * ln2)) - xb6 * (-8.0 - 99.0 * ln2 + 114.0 * yb * ln2 - 412.0 * yb3 * ln2 - 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2))
                     + xb8 * (-8.0 - 99.0 * ln2 + 114.0 * yb * ln2 - 412.0 * yb3 * ln2 - 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2)) - 4.0 * 1.0i * xb3 * (2.0 + 25.0 * ln2 + 18.0 * yb * ln2 + 68.0 * yb3 * ln2 + 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2)) - 4.0 * 1.0i * xb11 * (2.0 + 25.0 * ln2 + 18.0 * yb * ln2 + 68.0 * yb3 * ln2 + 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2)) + 8.0 * 1.0i * xb7 * (2.0 + 121.0 * ln2 - 78.0 * yb * ln2 + 132.0 * yb3 * ln2 + 1098.0 * yb5 * ln2 + yb2 * (-26.0 + 113.0 * ln2) + yb6 * (-342.0 + 2011.0 * ln2) + yb4 * (-210.0 + 3611.0 * ln2))
                     + xb4 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 - 162.0 * yb * ln2 + 252.0 * yb3 * ln2 + 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 - 162.0 * yb * ln2 + 252.0 * yb3 * ln2 + 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) + 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 - 198.0 * yb * ln2 + 116.0 * yb3 * ln2 + 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2)) + 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 - 198.0 * yb * ln2 + 116.0 * yb3 * ln2 + 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2)) - yb6 * log(8.0) + xb14 * (ln2 - 6.0 * yb * ln2 + 9.0 * yb2 * ln2 - 12.0 * yb3 * ln2 - 45.0 * yb4 * ln2 + 18.0 * yb5 * ln2 + yb6 * log(8.0))
                     + yb * ln64) - 4.0 * xb2 * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + xb6 * (8.0 + 147.0 * ln2 - 96.0 * yb * ln2 + 320.0 * yb3 * ln2 + 1824.0 * yb5 * ln2 + 3.0 * yb6 * (-648.0 + 25.0 * ln2) - yb2 * (168.0 + 139.0 * ln2) - yb4 * (1224.0 + 691.0 * ln2)) + xb8 * (-8.0 + yb6 * (1944.0 - 75.0 * ln2) - 147.0 * ln2 + 96.0 * yb * ln2 - 320.0 * yb3 * ln2 - 1824.0 * yb5 * ln2 + yb2 * (168.0 + 139.0 * ln2) + yb4 * (1224.0 + 691.0 * ln2)) - 4.0 * 1.0i * xb3 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 + 24.0 * yb * ln2 + 48.0 * yb3 * ln2 - 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2))
                     - 4.0 * 1.0i * xb11 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 + 24.0 * yb * ln2 + 48.0 * yb3 * ln2 - 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2)) + 8.0 * 1.0i * xb7 * (2.0 + 133.0 * ln2 - 72.0 * yb * ln2 + 112.0 * yb3 * ln2 + 984.0 * yb5 * ln2 + yb2 * (-26.0 + 187.0 * ln2) + yb6 * (-342.0 + 2293.0 * ln2) + yb4 * (-210.0 + 3883.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 - 144.0 * yb * ln2 + 224.0 * yb3 * ln2 + 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 - 144.0 * yb * ln2 + 224.0 * yb3 * ln2 + 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2))
                     + 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 - 192.0 * yb * ln2 + 128.0 * yb3 * ln2 + 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) + 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 - 192.0 * yb * ln2 + 128.0 * yb3 * ln2 + 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) - xb2 * (4.0 - 23.0 * ln2 + 48.0 * yb * ln2 + 96.0 * yb3 * ln2 - 144.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 85.0 * ln2) + yb4 * (-36.0 + 471.0 * ln2) + 3.0 * yb6 * (-36.0 + log(2048.0))) + xb12 * (4.0 - 23.0 * ln2 + 48.0 * yb * ln2 + 96.0 * yb3 * ln2 - 144.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 85.0 * ln2) + yb4 * (-36.0 + 471.0 * ln2) + 3.0 * yb6 * (-36.0 + log(2048.0)))))
                 - 4.0 * xb2 * (power_of<2>(vb) * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + xb2 * (-4.0 + yb4 * (36.0 - 471.0 * ln2) + yb6 * (108.0 - 33.0 * ln2) + 23.0 * ln2 + 48.0 * yb * ln2 + 96.0 * yb3 * ln2 - 144.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 85.0 * ln2)) + xb6 * (8.0 + 147.0 * ln2 + 96.0 * yb * ln2 - 320.0 * yb3 * ln2 - 1824.0 * yb5 * ln2 + 3.0 * yb6 * (-648.0 + 25.0 * ln2) - yb2 * (168.0 + 139.0 * ln2) - yb4 * (1224.0 + 691.0 * ln2)) + xb8 * (-8.0 + yb6 * (1944.0 - 75.0 * ln2) - 147.0 * ln2 - 96.0 * yb * ln2 + 320.0 * yb3 * ln2 + 1824.0 * yb5 * ln2 + yb2 * (168.0 + 139.0 * ln2) + yb4 * (1224.0 + 691.0 * ln2))
                     - 4.0 * 1.0i * xb3 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 - 24.0 * yb * ln2 - 48.0 * yb3 * ln2 + 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2)) - 4.0 * 1.0i * xb11 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 - 24.0 * yb * ln2 - 48.0 * yb3 * ln2 + 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2)) + 8.0 * 1.0i * xb7 * (2.0 + 133.0 * ln2 + 72.0 * yb * ln2 - 112.0 * yb3 * ln2 - 984.0 * yb5 * ln2 + yb2 * (-26.0 + 187.0 * ln2) + yb6 * (-342.0 + 2293.0 * ln2) + yb4 * (-210.0 + 3883.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 + 144.0 * yb * ln2 - 224.0 * yb3 * ln2 - 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2))
                     - xb10 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 + 144.0 * yb * ln2 - 224.0 * yb3 * ln2 - 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2)) + 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 + 192.0 * yb * ln2 - 128.0 * yb3 * ln2 - 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) + 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 + 192.0 * yb * ln2 - 128.0 * yb3 * ln2 - 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) + xb12 * (4.0 - 23.0 * ln2 - 48.0 * yb * ln2 - 96.0 * yb3 * ln2 + 144.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 85.0 * ln2) + yb4 * (-36.0 + 471.0 * ln2) + 3.0 * yb6 * (-36.0 + log(2048.0))))
                   - 4.0 * xb2 * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb10 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 + 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + 2.0 * 1.0i * xb9 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 - xb2 * (1.0 + 3.0 * yb2) * (4.0 - 21.0 * ln2 + 156.0 * yb2 * ln2 + yb4 * (-36.0 + 33.0 * ln2)) + xb8 * (1.0 + 3.0 * yb2) * (4.0 - 21.0 * ln2 + 156.0 * yb2 * ln2 + yb4 * (-36.0 + 33.0 * ln2)) - 8.0 * 1.0i * xb3 * (1.0 + yb2 * (3.0 - 67.0 * ln2) + 19.0 * ln2 + yb6 * (-27.0 + 371.0 * ln2) + yb4 * (-9.0 + 1373.0 * ln2)) - 8.0 * 1.0i * xb7 * (1.0 + yb2 * (3.0 - 67.0 * ln2) + 19.0 * ln2 + yb6 * (-27.0 + 371.0 * ln2) + yb4 * (-9.0 + 1373.0 * ln2)) - 4.0 * 1.0i * xb5 * (4.0 + 75.0 * ln2 + 3.0 * yb4 * (-76.0 + 1127.0 * ln2) + yb6 * (-396.0 + 1267.0 * ln2) + 5.0 * yb2 * (-4.0 + log(512.0)))
                     + 2.0 * xb4 * (1.0 + 3.0 * yb2) * (-2.0 + 4.0 * yb2 * (8.0 + 43.0 * ln2) + yb4 * (114.0 + 241.0 * ln2) + log(2048.0)) - 2.0 * xb6 * (1.0 + 3.0 * yb2) * (-2.0 + 4.0 * yb2 * (8.0 + 43.0 * ln2) + yb4 * (114.0 + 241.0 * ln2) + log(2048.0))) + 8.0 * vb * xb2 * (-ln2 - 9.0 * yb2 * ln2 + 12.0 * yb3 * ln2 + 45.0 * yb4 * ln2 - 18.0 * yb5 * ln2 + 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 + 2.0 * 1.0i * xb9 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 + xb8 * (4.0 - 45.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 - 9.0 * yb4 * (4.0 + 63.0 * ln2) + 3.0 * yb2 * (4.0 + 73.0 * ln2) - yb6 * (108.0 + 215.0 * ln2)) + xb2 * (-4.0 + 45.0 * ln2 + 18.0 * yb * ln2 - 28.0 * yb3 * ln2 - 246.0 * yb5 * ln2 + 9.0 * yb4 * (4.0 + 63.0 * ln2) - 3.0 * yb2 * (4.0 + 73.0 * ln2) + yb6 * (108.0 + 215.0 * ln2))
                     - 4.0 * 1.0i * xb5 * (4.0 + 51.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 + 33.0 * yb6 * (-12.0 + 25.0 * ln2) + yb2 * (-20.0 + 43.0 * ln2) + yb4 * (-228.0 + 2569.0 * ln2)) - yb6 * log(8.0) + xb10 * (ln2 - 6.0 * yb * ln2 + 9.0 * yb2 * ln2 - 12.0 * yb3 * ln2 - 45.0 * yb4 * ln2 + 18.0 * yb5 * ln2 + yb6 * log(8.0)) + yb * ln64 + 2.0 * xb4 * (-2.0 + 23.0 * ln2 - 20.0 * yb3 * ln2 - 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2) + yb * ln64) - 2.0 * xb6 * (-2.0 + 23.0 * ln2 - 20.0 * yb3 * ln2 - 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2) + yb * ln64) - 8.0 * 1.0i * xb3 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 6.0 * yb * ln2 + 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb3 * log(16.0) + log(8192.0))
                     - 8.0 * 1.0i * xb7 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 6.0 * yb * ln2 + 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb3 * log(16.0) + log(8192.0)))) - 8.0 * tb * xb2 * (power_of<2>(vb) * (-ln2 - 6.0 * yb * ln2 - 9.0 * yb2 * ln2 - 12.0 * yb3 * ln2 + 45.0 * yb4 * ln2 + 18.0 * yb5 * ln2 + 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 + 2.0 * 1.0i * xb13 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 + xb2 * (-4.0 + 47.0 * ln2 + 42.0 * yb * ln2 + 148.0 * yb3 * ln2 + 66.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 269.0 * ln2) + yb4 * (36.0 + 573.0 * ln2)) - xb12 * (-4.0 + 47.0 * ln2 + 42.0 * yb * ln2 + 148.0 * yb3 * ln2 + 66.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 269.0 * ln2) + yb4 * (36.0 + 573.0 * ln2))
                     - xb6 * (-8.0 - 99.0 * ln2 - 114.0 * yb * ln2 + 412.0 * yb3 * ln2 + 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2)) + xb8 * (-8.0 - 99.0 * ln2 - 114.0 * yb * ln2 + 412.0 * yb3 * ln2 + 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2)) - 4.0 * 1.0i * xb3 * (2.0 + 25.0 * ln2 - 18.0 * yb * ln2 - 68.0 * yb3 * ln2 - 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2)) - 4.0 * 1.0i * xb11 * (2.0 + 25.0 * ln2 - 18.0 * yb * ln2 - 68.0 * yb3 * ln2 - 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2))
                     + 8.0 * 1.0i * xb7 * (2.0 + 121.0 * ln2 + 78.0 * yb * ln2 - 132.0 * yb3 * ln2 - 1098.0 * yb5 * ln2 + yb2 * (-26.0 + 113.0 * ln2) + yb6 * (-342.0 + 2011.0 * ln2) + yb4 * (-210.0 + 3611.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 + 162.0 * yb * ln2 - 252.0 * yb3 * ln2 - 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 + 162.0 * yb * ln2 - 252.0 * yb3 * ln2 - 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) + 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 + 198.0 * yb * ln2 - 116.0 * yb3 * ln2 - 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2))
                     + 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 + 198.0 * yb * ln2 - 116.0 * yb3 * ln2 - 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2)) - yb6 * log(8.0) + xb14 * (ln2 + 9.0 * yb2 * ln2 + 12.0 * yb3 * ln2 - 45.0 * yb4 * ln2 - 18.0 * yb5 * ln2 + yb6 * log(8.0) + yb * ln64)) - 8.0 * vb * xb2 * (ln2 + 27.0 * yb2 * ln2 - 57.0 * yb4 * ln2 - 2.0 * 1.0i * xb * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6) * ln2 - 2.0 * 1.0i * xb9 * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6) * ln2 - xb2 * (-4.0 + 69.0 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 529.0 * ln2) + 3.0 * yb4 * (12.0 + 545.0 * ln2)) + xb8 * (-4.0 + 69.0 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 529.0 * ln2) + 3.0 * yb4 * (12.0 + 545.0 * ln2)) - 2.0 * xb4 * (-2.0 + 35.0 * ln2 + 9.0 * yb6 * (38.0 + 143.0 * ln2) + yb2 * (26.0 + 353.0 * ln2) + yb4 * (210.0 + 1301.0 * ln2))
                     + 4.0 * 1.0i * xb5 * (4.0 + 27.0 * ln2 + yb2 * (-20.0 + 41.0 * ln2) + yb6 * (-396.0 + 383.0 * ln2) + yb4 * (-228.0 + 1757.0 * ln2)) + xb6 * (-4.0 + 70.0 * ln2 + 18.0 * yb6 * (38.0 + 143.0 * ln2) + yb2 * (52.0 + 706.0 * ln2) + yb4 * (420.0 + 2602.0 * ln2)) - yb6 * log(8.0) + xb10 * (-ln2 - 27.0 * yb2 * ln2 + 57.0 * yb4 * ln2 + yb6 * log(8.0)) + 8.0 * 1.0i * xb3 * (1.0 + yb6 * (-27.0 + 211.0 * ln2) + yb4 * (-9.0 + 833.0 * ln2) + yb2 * (3.0 + log(32.0)) + log(128.0)) + 8.0 * 1.0i * xb7 * (1.0 + yb6 * (-27.0 + 211.0 * ln2) + yb4 * (-9.0 + 833.0 * ln2) + yb2 * (3.0 + log(32.0)) + log(128.0)))
                   + xb2 * (24.0 * yb * ln2 + 36.0 * yb2 * ln2 + 48.0 * yb3 * ln2 - 180.0 * yb4 * ln2 - 72.0 * yb5 * ln2 + 12.0 * yb6 * ln2 - 4.0 * xb10 * (1.0 + 6.0 * yb + 9.0 * yb2 + 12.0 * yb3 - 45.0 * yb4 - 18.0 * yb5 + 3.0 * yb6) * ln2 - 8.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 - 8.0 * 1.0i * xb9 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 - 4.0 * xb2 * (-4.0 + 45.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 + 9.0 * yb4 * (4.0 + 63.0 * ln2) - 3.0 * yb2 * (4.0 + 73.0 * ln2) + yb6 * (108.0 + 215.0 * ln2)) + 4.0 * xb8 * (-4.0 + 45.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 + 9.0 * yb4 * (4.0 + 63.0 * ln2) - 3.0 * yb2 * (4.0 + 73.0 * ln2) + yb6 * (108.0 + 215.0 * ln2))
                     - 8.0 * xb4 * (-2.0 + 23.0 * ln2 - 6.0 * yb * ln2 + 20.0 * yb3 * ln2 + 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2)) + 8.0 * xb6 * (-2.0 + 23.0 * ln2 - 6.0 * yb * ln2 + 20.0 * yb3 * ln2 + 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2)) + 16.0 * 1.0i * xb5 * (4.0 + 51.0 * ln2 + 18.0 * yb * ln2 - 28.0 * yb3 * ln2 - 246.0 * yb5 * ln2 + 33.0 * yb6 * (-12.0 + 25.0 * ln2) + yb2 * (-20.0 + 43.0 * ln2) + yb4 * (-228.0 + 2569.0 * ln2)) + log(16.0) + 32.0 * 1.0i * xb3 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 4.0 * yb3 * ln2 - 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb * ln64 + log(8192.0))
                     + 32.0 * 1.0i * xb7 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 4.0 * yb3 * ln2 - 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb * ln64 + log(8192.0)))));
            const complex<double> logs10 = power_of<2>(-1.0i + xb) * (-1.0 + yb) * (1.0 + yb) * (power_of<2>(tb) * (power_of<2>(vb) * power_of<2>(-1.0 + xb2) * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 - 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + 4.0 * 1.0i * xb3 * (2.0 + 37.0 * ln2 + yb2 * (6.0 + 67.0 * ln2) + yb6 * (-54.0 + 733.0 * ln2) + yb4 * (-18.0 + 2587.0 * ln2)) + 4.0 * 1.0i * xb11 * (2.0 + 37.0 * ln2 + yb2 * (6.0 + 67.0 * ln2) + yb6 * (-54.0 + 733.0 * ln2) + yb4 * (-18.0 + 2587.0 * ln2)) + xb6 * (8.0 + 339.0 * ln2 + yb2 * (-168.0 + 1045.0 * ln2) + 3.0 * yb6 * (-648.0 + 1529.0 * ln2) + yb4 * (-1224.0 + 3661.0 * ln2)) - xb8 * (8.0 + 339.0 * ln2 + yb2 * (-168.0 + 1045.0 * ln2) + 3.0 * yb6 * (-648.0 + 1529.0 * ln2) + yb4 * (-1224.0 + 3661.0 * ln2))
                     - 8.0 * 1.0i * xb7 * (2.0 + 229.0 * ln2 + 13.0 * yb2 * (-2.0 + 15.0 * ln2) + 3.0 * yb4 * (-70.0 + 2377.0 * ln2) + yb6 * (-342.0 + 4061.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 1411.0 * ln2) + 363.0 * ln2 + yb6 * (468.0 + 6659.0 * ln2) + yb4 * (348.0 + 19637.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 1411.0 * ln2) + 363.0 * ln2 + yb6 * (468.0 + 6659.0 * ln2) + yb4 * (348.0 + 19637.0 * ln2)) - 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 2873.0 * ln2) + 385.0 * ln2 + yb6 * (576.0 + 5593.0 * ln2) + yb4 * (384.0 + 21439.0 * ln2)) - 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 2873.0 * ln2) + 385.0 * ln2 + yb6 * (576.0 + 5593.0 * ln2) + yb4 * (384.0 + 21439.0 * ln2)) + xb12 * (4.0 - 23.0 * ln2 + 3.0 * yb2 * (4.0 + 133.0 * ln2) + yb4 * (-36.0 + 375.0 * ln2) - 3.0 * yb6 * (36.0 + log(32.0))) + xb2 * (-4.0 + yb4 * (36.0 - 375.0 * ln2) + 23.0 * ln2 - 3.0 * yb2 * (4.0 + 133.0 * ln2) + 3.0 * yb6 * (36.0 + log(32.0))))
                   + 8.0 * vb * xb2 * (-ln2 - 9.0 * yb2 * ln2 + 12.0 * yb3 * ln2 + 45.0 * yb4 * ln2 - 18.0 * yb5 * ln2 - 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 - 2.0 * 1.0i * xb13 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 + xb12 * (4.0 - 47.0 * ln2 + 42.0 * yb * ln2 + 148.0 * yb3 * ln2 + 66.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 115.0 * ln2) - 3.0 * yb4 * (12.0 + 191.0 * ln2) - yb6 * (108.0 + 269.0 * ln2)) + xb2 * (-4.0 + 47.0 * ln2 - 42.0 * yb * ln2 - 148.0 * yb3 * ln2 - 66.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 269.0 * ln2) + yb4 * (36.0 + 573.0 * ln2)) - xb6 * (-8.0 - 99.0 * ln2 + 114.0 * yb * ln2 - 412.0 * yb3 * ln2 - 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2))
                     + xb8 * (-8.0 - 99.0 * ln2 + 114.0 * yb * ln2 - 412.0 * yb3 * ln2 - 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2)) + 4.0 * 1.0i * xb3 * (2.0 + 25.0 * ln2 + 18.0 * yb * ln2 + 68.0 * yb3 * ln2 + 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2)) + 4.0 * 1.0i * xb11 * (2.0 + 25.0 * ln2 + 18.0 * yb * ln2 + 68.0 * yb3 * ln2 + 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2)) - 8.0 * 1.0i * xb7 * (2.0 + 121.0 * ln2 - 78.0 * yb * ln2 + 132.0 * yb3 * ln2 + 1098.0 * yb5 * ln2 + yb2 * (-26.0 + 113.0 * ln2) + yb6 * (-342.0 + 2011.0 * ln2) + yb4 * (-210.0 + 3611.0 * ln2))
                     + xb4 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 - 162.0 * yb * ln2 + 252.0 * yb3 * ln2 + 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 - 162.0 * yb * ln2 + 252.0 * yb3 * ln2 + 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) - 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 - 198.0 * yb * ln2 + 116.0 * yb3 * ln2 + 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2)) - 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 - 198.0 * yb * ln2 + 116.0 * yb3 * ln2 + 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2)) - yb6 * log(8.0) + xb14 * (ln2 - 6.0 * yb * ln2 + 9.0 * yb2 * ln2 - 12.0 * yb3 * ln2 - 45.0 * yb4 * ln2 + 18.0 * yb5 * ln2 + yb6 * log(8.0))
                     + yb * ln64) - 4.0 * xb2 * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 - 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + xb6 * (8.0 + 147.0 * ln2 - 96.0 * yb * ln2 + 320.0 * yb3 * ln2 + 1824.0 * yb5 * ln2 + 3.0 * yb6 * (-648.0 + 25.0 * ln2) - yb2 * (168.0 + 139.0 * ln2) - yb4 * (1224.0 + 691.0 * ln2)) + xb8 * (-8.0 + yb6 * (1944.0 - 75.0 * ln2) - 147.0 * ln2 + 96.0 * yb * ln2 - 320.0 * yb3 * ln2 - 1824.0 * yb5 * ln2 + yb2 * (168.0 + 139.0 * ln2) + yb4 * (1224.0 + 691.0 * ln2)) + 4.0 * 1.0i * xb3 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 + 24.0 * yb * ln2 + 48.0 * yb3 * ln2 - 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2))
                     + 4.0 * 1.0i * xb11 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 + 24.0 * yb * ln2 + 48.0 * yb3 * ln2 - 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2)) - 8.0 * 1.0i * xb7 * (2.0 + 133.0 * ln2 - 72.0 * yb * ln2 + 112.0 * yb3 * ln2 + 984.0 * yb5 * ln2 + yb2 * (-26.0 + 187.0 * ln2) + yb6 * (-342.0 + 2293.0 * ln2) + yb4 * (-210.0 + 3883.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 - 144.0 * yb * ln2 + 224.0 * yb3 * ln2 + 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 - 144.0 * yb * ln2 + 224.0 * yb3 * ln2 + 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2))
                     - 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 - 192.0 * yb * ln2 + 128.0 * yb3 * ln2 + 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) - 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 - 192.0 * yb * ln2 + 128.0 * yb3 * ln2 + 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) - xb2 * (4.0 - 23.0 * ln2 + 48.0 * yb * ln2 + 96.0 * yb3 * ln2 - 144.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 85.0 * ln2) + yb4 * (-36.0 + 471.0 * ln2) + 3.0 * yb6 * (-36.0 + log(2048.0))) + xb12 * (4.0 - 23.0 * ln2 + 48.0 * yb * ln2 + 96.0 * yb3 * ln2 - 144.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 85.0 * ln2) + yb4 * (-36.0 + 471.0 * ln2) + 3.0 * yb6 * (-36.0 + log(2048.0)))))
                 - 4.0 * xb2 * (power_of<2>(vb) * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb14 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 - 2.0 * 1.0i * xb13 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 + xb2 * (-4.0 + yb4 * (36.0 - 471.0 * ln2) + yb6 * (108.0 - 33.0 * ln2) + 23.0 * ln2 + 48.0 * yb * ln2 + 96.0 * yb3 * ln2 - 144.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 85.0 * ln2)) + xb6 * (8.0 + 147.0 * ln2 + 96.0 * yb * ln2 - 320.0 * yb3 * ln2 - 1824.0 * yb5 * ln2 + 3.0 * yb6 * (-648.0 + 25.0 * ln2) - yb2 * (168.0 + 139.0 * ln2) - yb4 * (1224.0 + 691.0 * ln2)) + xb8 * (-8.0 + yb6 * (1944.0 - 75.0 * ln2) - 147.0 * ln2 - 96.0 * yb * ln2 + 320.0 * yb3 * ln2 + 1824.0 * yb5 * ln2 + yb2 * (168.0 + 139.0 * ln2) + yb4 * (1224.0 + 691.0 * ln2))
                     + 4.0 * 1.0i * xb3 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 - 24.0 * yb * ln2 - 48.0 * yb3 * ln2 + 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2)) + 4.0 * 1.0i * xb11 * (2.0 + yb2 * (6.0 - 5.0 * ln2) + 37.0 * ln2 - 24.0 * yb * ln2 - 48.0 * yb3 * ln2 + 72.0 * yb5 * ln2 + yb6 * (-54.0 + 757.0 * ln2) + yb4 * (-18.0 + 2635.0 * ln2)) - 8.0 * 1.0i * xb7 * (2.0 + 133.0 * ln2 + 72.0 * yb * ln2 - 112.0 * yb3 * ln2 - 984.0 * yb5 * ln2 + yb2 * (-26.0 + 187.0 * ln2) + yb6 * (-342.0 + 2293.0 * ln2) + yb4 * (-210.0 + 3883.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 + 144.0 * yb * ln2 - 224.0 * yb3 * ln2 - 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2))
                     - xb10 * (4.0 + yb2 * (76.0 - 403.0 * ln2) + 171.0 * ln2 + 144.0 * yb * ln2 - 224.0 * yb3 * ln2 - 1968.0 * yb5 * ln2 + 13.0 * yb6 * (36.0 + 319.0 * ln2) + yb4 * (348.0 + 11093.0 * ln2)) - 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 + 192.0 * yb * ln2 - 128.0 * yb3 * ln2 - 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) - 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1721.0 * ln2) + 193.0 * ln2 + 192.0 * yb * ln2 - 128.0 * yb3 * ln2 - 2112.0 * yb5 * ln2 + 9.0 * yb6 * (64.0 + 337.0 * ln2) + yb4 * (384.0 + 12799.0 * ln2)) + xb12 * (4.0 - 23.0 * ln2 - 48.0 * yb * ln2 - 96.0 * yb3 * ln2 + 144.0 * yb5 * ln2 + 3.0 * yb2 * (4.0 + 85.0 * ln2) + yb4 * (-36.0 + 471.0 * ln2) + 3.0 * yb6 * (-36.0 + log(2048.0))))
                   - 4.0 * xb2 * ((-1.0 + 9.0 * yb2 + 33.0 * yb4 - 9.0 * yb6) * ln2 + xb10 * (1.0 - 9.0 * yb2 - 33.0 * yb4 + 9.0 * yb6) * ln2 - 2.0 * 1.0i * xb * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 - 2.0 * 1.0i * xb9 * (-1.0 + 57.0 * yb2 - 63.0 * yb4 + 39.0 * yb6) * ln2 - xb2 * (1.0 + 3.0 * yb2) * (4.0 - 21.0 * ln2 + 156.0 * yb2 * ln2 + yb4 * (-36.0 + 33.0 * ln2)) + xb8 * (1.0 + 3.0 * yb2) * (4.0 - 21.0 * ln2 + 156.0 * yb2 * ln2 + yb4 * (-36.0 + 33.0 * ln2)) + 8.0 * 1.0i * xb3 * (1.0 + yb2 * (3.0 - 67.0 * ln2) + 19.0 * ln2 + yb6 * (-27.0 + 371.0 * ln2) + yb4 * (-9.0 + 1373.0 * ln2)) + 8.0 * 1.0i * xb7 * (1.0 + yb2 * (3.0 - 67.0 * ln2) + 19.0 * ln2 + yb6 * (-27.0 + 371.0 * ln2) + yb4 * (-9.0 + 1373.0 * ln2)) + 4.0 * 1.0i * xb5 * (4.0 + 75.0 * ln2 + 3.0 * yb4 * (-76.0 + 1127.0 * ln2) + yb6 * (-396.0 + 1267.0 * ln2) + 5.0 * yb2 * (-4.0 + log(512.0)))
                     + 2.0 * xb4 * (1.0 + 3.0 * yb2) * (-2.0 + 4.0 * yb2 * (8.0 + 43.0 * ln2) + yb4 * (114.0 + 241.0 * ln2) + log(2048.0)) - 2.0 * xb6 * (1.0 + 3.0 * yb2) * (-2.0 + 4.0 * yb2 * (8.0 + 43.0 * ln2) + yb4 * (114.0 + 241.0 * ln2) + log(2048.0))) + 8.0 * vb * xb2 * (-ln2 - 9.0 * yb2 * ln2 + 12.0 * yb3 * ln2 + 45.0 * yb4 * ln2 - 18.0 * yb5 * ln2 - 2.0 * 1.0i * xb * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 - 2.0 * 1.0i * xb9 * (-1.0 + 6.0 * yb + 39.0 * yb2 + 12.0 * yb3 - 51.0 * yb4 - 18.0 * yb5 + 45.0 * yb6) * ln2 + xb8 * (4.0 - 45.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 - 9.0 * yb4 * (4.0 + 63.0 * ln2) + 3.0 * yb2 * (4.0 + 73.0 * ln2) - yb6 * (108.0 + 215.0 * ln2)) + xb2 * (-4.0 + 45.0 * ln2 + 18.0 * yb * ln2 - 28.0 * yb3 * ln2 - 246.0 * yb5 * ln2 + 9.0 * yb4 * (4.0 + 63.0 * ln2) - 3.0 * yb2 * (4.0 + 73.0 * ln2) + yb6 * (108.0 + 215.0 * ln2))
                     + 4.0 * 1.0i * xb5 * (4.0 + 51.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 + 33.0 * yb6 * (-12.0 + 25.0 * ln2) + yb2 * (-20.0 + 43.0 * ln2) + yb4 * (-228.0 + 2569.0 * ln2)) - yb6 * log(8.0) + xb10 * (ln2 - 6.0 * yb * ln2 + 9.0 * yb2 * ln2 - 12.0 * yb3 * ln2 - 45.0 * yb4 * ln2 + 18.0 * yb5 * ln2 + yb6 * log(8.0)) + yb * ln64 + 2.0 * xb4 * (-2.0 + 23.0 * ln2 - 20.0 * yb3 * ln2 - 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2) + yb * ln64) - 2.0 * xb6 * (-2.0 + 23.0 * ln2 - 20.0 * yb3 * ln2 - 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2) + yb * ln64) + 8.0 * 1.0i * xb3 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 6.0 * yb * ln2 + 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb3 * log(16.0) + log(8192.0))
                     + 8.0 * 1.0i * xb7 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 6.0 * yb * ln2 + 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb3 * log(16.0) + log(8192.0)))) - 8.0 * tb * xb2 * (power_of<2>(vb) * (-ln2 - 6.0 * yb * ln2 - 9.0 * yb2 * ln2 - 12.0 * yb3 * ln2 + 45.0 * yb4 * ln2 + 18.0 * yb5 * ln2 - 2.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 - 2.0 * 1.0i * xb13 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 + xb2 * (-4.0 + 47.0 * ln2 + 42.0 * yb * ln2 + 148.0 * yb3 * ln2 + 66.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 269.0 * ln2) + yb4 * (36.0 + 573.0 * ln2)) - xb12 * (-4.0 + 47.0 * ln2 + 42.0 * yb * ln2 + 148.0 * yb3 * ln2 + 66.0 * yb5 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 269.0 * ln2) + yb4 * (36.0 + 573.0 * ln2))
                     - xb6 * (-8.0 - 99.0 * ln2 - 114.0 * yb * ln2 + 412.0 * yb3 * ln2 + 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2)) + xb8 * (-8.0 - 99.0 * ln2 - 114.0 * yb * ln2 + 412.0 * yb3 * ln2 + 2262.0 * yb5 * ln2 + yb2 * (168.0 + 709.0 * ln2) + yb4 * (1224.0 + 1255.0 * ln2) + yb6 * (1944.0 + 1303.0 * ln2)) + 4.0 * 1.0i * xb3 * (2.0 + 25.0 * ln2 - 18.0 * yb * ln2 - 68.0 * yb3 * ln2 - 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2)) + 4.0 * 1.0i * xb11 * (2.0 + 25.0 * ln2 - 18.0 * yb * ln2 - 68.0 * yb3 * ln2 - 42.0 * yb5 * ln2 + yb2 * (6.0 + 49.0 * ln2) + 9.0 * yb6 * (-6.0 + 67.0 * ln2) + yb4 * (-18.0 + 2107.0 * ln2))
                     - 8.0 * 1.0i * xb7 * (2.0 + 121.0 * ln2 + 78.0 * yb * ln2 - 132.0 * yb3 * ln2 - 1098.0 * yb5 * ln2 + yb2 * (-26.0 + 113.0 * ln2) + yb6 * (-342.0 + 2011.0 * ln2) + yb4 * (-210.0 + 3611.0 * ln2)) + xb4 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 + 162.0 * yb * ln2 - 252.0 * yb3 * ln2 - 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) - xb10 * (4.0 + yb2 * (76.0 - 21.0 * ln2) + 147.0 * ln2 + 162.0 * yb * ln2 - 252.0 * yb3 * ln2 - 2214.0 * yb5 * ln2 + yb6 * (468.0 + 4089.0 * ln2) + yb4 * (348.0 + 9513.0 * ln2)) - 2.0 * 1.0i * xb5 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 + 198.0 * yb * ln2 - 116.0 * yb3 * ln2 - 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2))
                     - 2.0 * 1.0i * xb9 * (yb2 * (64.0 - 1447.0 * ln2) + 193.0 * ln2 + 198.0 * yb * ln2 - 116.0 * yb3 * ln2 - 2130.0 * yb5 * ln2 + yb6 * (576.0 + 3283.0 * ln2) + yb4 * (384.0 + 12275.0 * ln2)) - yb6 * log(8.0) + xb14 * (ln2 + 9.0 * yb2 * ln2 + 12.0 * yb3 * ln2 - 45.0 * yb4 * ln2 - 18.0 * yb5 * ln2 + yb6 * log(8.0) + yb * ln64)) - 8.0 * vb * xb2 * (ln2 + 27.0 * yb2 * ln2 - 57.0 * yb4 * ln2 + 2.0 * 1.0i * xb * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6) * ln2 + 2.0 * 1.0i * xb9 * (-1.0 + 21.0 * yb2 - 39.0 * yb4 + 51.0 * yb6) * ln2 - xb2 * (-4.0 + 69.0 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 529.0 * ln2) + 3.0 * yb4 * (12.0 + 545.0 * ln2)) + xb8 * (-4.0 + 69.0 * ln2 - 3.0 * yb2 * (4.0 + 115.0 * ln2) + yb6 * (108.0 + 529.0 * ln2) + 3.0 * yb4 * (12.0 + 545.0 * ln2)) - 2.0 * xb4 * (-2.0 + 35.0 * ln2 + 9.0 * yb6 * (38.0 + 143.0 * ln2) + yb2 * (26.0 + 353.0 * ln2) + yb4 * (210.0 + 1301.0 * ln2))
                     - 4.0 * 1.0i * xb5 * (4.0 + 27.0 * ln2 + yb2 * (-20.0 + 41.0 * ln2) + yb6 * (-396.0 + 383.0 * ln2) + yb4 * (-228.0 + 1757.0 * ln2)) + xb6 * (-4.0 + 70.0 * ln2 + 18.0 * yb6 * (38.0 + 143.0 * ln2) + yb2 * (52.0 + 706.0 * ln2) + yb4 * (420.0 + 2602.0 * ln2)) - yb6 * log(8.0) + xb10 * (-ln2 - 27.0 * yb2 * ln2 + 57.0 * yb4 * ln2 + yb6 * log(8.0)) - 8.0 * 1.0i * xb3 * (1.0 + yb6 * (-27.0 + 211.0 * ln2) + yb4 * (-9.0 + 833.0 * ln2) + yb2 * (3.0 + log(32.0)) + log(128.0)) - 8.0 * 1.0i * xb7 * (1.0 + yb6 * (-27.0 + 211.0 * ln2) + yb4 * (-9.0 + 833.0 * ln2) + yb2 * (3.0 + log(32.0)) + log(128.0)))
                   - xb2 * (-24.0 * yb * ln2 - 36.0 * yb2 * ln2 - 48.0 * yb3 * ln2 + 180.0 * yb4 * ln2 + 72.0 * yb5 * ln2 - 12.0 * yb6 * ln2 - 8.0 * 1.0i * xb * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 - 8.0 * 1.0i * xb9 * (-1.0 - 6.0 * yb + 39.0 * yb2 - 12.0 * yb3 - 51.0 * yb4 + 18.0 * yb5 + 45.0 * yb6) * ln2 + 4.0 * xb2 * (-4.0 + 45.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 + 9.0 * yb4 * (4.0 + 63.0 * ln2) - 3.0 * yb2 * (4.0 + 73.0 * ln2) + yb6 * (108.0 + 215.0 * ln2)) - 4.0 * xb8 * (-4.0 + 45.0 * ln2 - 18.0 * yb * ln2 + 28.0 * yb3 * ln2 + 246.0 * yb5 * ln2 + 9.0 * yb4 * (4.0 + 63.0 * ln2) - 3.0 * yb2 * (4.0 + 73.0 * ln2) + yb6 * (108.0 + 215.0 * ln2)) + 8.0 * xb4 * (-2.0 + 23.0 * ln2 - 6.0 * yb * ln2 + 20.0 * yb3 * ln2 + 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2))
                     - 8.0 * xb6 * (-2.0 + 23.0 * ln2 - 6.0 * yb * ln2 + 20.0 * yb3 * ln2 + 114.0 * yb5 * ln2 + 21.0 * yb4 * (10.0 + 49.0 * ln2) + yb2 * (26.0 + 279.0 * ln2) + 3.0 * yb6 * (114.0 + 335.0 * ln2)) + 16.0 * 1.0i * xb5 * (4.0 + 51.0 * ln2 + 18.0 * yb * ln2 - 28.0 * yb3 * ln2 - 246.0 * yb5 * ln2 + 33.0 * yb6 * (-12.0 + 25.0 * ln2) + yb2 * (-20.0 + 43.0 * ln2) + yb4 * (-228.0 + 2569.0 * ln2)) - log(16.0) + xb10 * (24.0 * yb * ln2 + 36.0 * yb2 * ln2 + 48.0 * yb3 * ln2 - 180.0 * yb4 * ln2 - 72.0 * yb5 * ln2 + 12.0 * yb6 * ln2 + log(16.0)) + 32.0 * 1.0i * xb3 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 4.0 * yb3 * ln2 - 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb * ln64 + log(8192.0)) + 32.0 * 1.0i * xb7 * (1.0 + yb2 * (3.0 - 31.0 * ln2) - 4.0 * yb3 * ln2 - 66.0 * yb5 * ln2 + 3.0 * yb6 * (-9.0 + 97.0 * ln2) + yb4 * (-9.0 + 1103.0 * ln2) + yb * ln64 + log(8192.0)))));

            const complex<double> term1 = -(80.0 * lnmuhat) / 81.0 + (2.0 * xb4 * (-991.0 + 108.0 * ln2 - 36.0 * yb2 * (135.0 + 28.0 * ln2) + 9.0 * yb4 * (-957.0 + 100.0 * ln2))
                  + 72.0 * xb2 * (-1.0 + yb2) * (2.0 + yb2 * (-13.0 + 14.0 * ln2) - ln4) + 72.0 * xb6 * (-1.0 + yb2) * (2.0 + yb2 * (-13.0 + 14.0 * ln2) - ln4) + 9.0 * (1.0 - 4.0 * yb2 + 3.0 * yb4) * (-1.0 + log(16.0))
                  + 9.0 * xb8 * (1.0 - 4.0 * yb2 + 3.0 * yb4) * (-1.0 + log(16.0))) / (1944.0 * xb4 * power_of<2>(1.0 + 3.0 * yb2));

            const complex<double> f27bpart1 = (num1 * log(xb)) / denom1 + (num4 * dilog((-1.0i) * xb) + num3 * dilog(1.0i * xb) - (num3 * power_of<2>(log(1.0 - 1.0i * xb))) / 2.0 - (num4 * power_of<2>(log(1.0 + 1.0i * xb))) / 2.0 + num3 * log(1.0 - 1.0i * xb) * log(xb)
                    + num4 * log(1.0 + 1.0i * xb) * log(xb)) / denom3 + (-(num2 * ln1myb) + num2 * ln1pyb) / denom2
                + (num9 * dilog(-(vb / w4)) + num9 * dilog(w4 / (-1.0 + w4)) - num9 * dilog(w4 / (1.0 + w4)) - num9 * dilog((vb + w4) / (-1.0 + w4)) + num9 * dilog((vb + w4) / (1.0 + w4))
                    + num9 * dilog(vb / w5) - num9 * dilog(w5 / (-1.0 + w5)) + num9 * dilog(w5 / (1.0 + w5)) + num9 * dilog((-vb + w5) / (-1.0 + w5)) - num9 * dilog((-vb + w5) / (1.0 + w5))
                    - num9 * dilog(w4 / (w4 - w7)) + num9 * dilog((vb + w4) / (w4 - w7)) - num9 * dilog(w5 / (w5 + w7)) + num9 * dilog((-vb + w5) / (w5 + w7)) + 4.0 * num9 * ln2 * log(1.0 - vb)
                    - 4.0 * num9 * ln2 * log(1.0 + vb) - num9 * log((1.0 + vb) / (1.0 - w4)) * log((vb + w4) / w4) + num9 * log((1.0 - vb) / (1.0 + w4)) * log((vb + w4) / w4)
                    + num9 * log(1.0 - vb / w5) * log((-1.0 + vb) / (-1.0 + w5)) - num9 * log(1.0 - vb / w5) * log((1.0 + vb) / (1.0 + w5)) + num9 * log((vb + w4) / w4) * log(-((vb + w7) / (w4 - w7)))
                    + num9 * log(1.0 - vb / w5) * log((vb + w7) / (w5 + w7)) + (2.0 * num9 * log(1.0 - vb) - 2.0 * num9 * log(1.0 + vb)) * log(xb) + (2.0 * 1.0i) * num9 * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (vb + w4) / w4, 1.0 + w4inv)
                    - (2.0 * 1.0i) * num9 * M_PI * log((-1.0 + w4) / w4) * my_sign(imag(w4)) * T(1.0, (vb + w4) / w4, (-1.0 + w4) / w4)
                    + (2.0 * 1.0i) * num9 * M_PI * log(1.0 - w7 / w4) * my_sign(imag(w4 / w7)) * T(1.0, (vb + w4) / w4, 1.0 - w7 / w4) - (2.0 * 1.0i) * num9 * M_PI * log(1.0 + w5inv) * my_sign(-imag(w5)) * T(1.0, 1.0 - vb / w5, 1.0 + w5inv)
                    + (2.0 * 1.0i) * num9 * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - vb / w5, (-1.0 + w5) / w5) + (2.0 * 1.0i) * num9 * M_PI * log((w5 + w7) / w5) * my_sign(-imag(w5 / w7)) * T(1.0, 1.0 - vb / w5, (w5 + w7) / w5)) / denom6
                + (-(num8 * dilog(tb / w4)) - num8 * dilog(w4 / (-1.0 + w4)) - num7 * dilog(w4 / (1.0 + w4)) + num8 * dilog((-tb + w4) / (-1.0 + w4)) + num7 * dilog((-tb + w4) / (1.0 + w4))
                    - num8 * dilog(-(tb / w5)) - num7 * dilog(w5 / (-1.0 + w5)) - num8 * dilog(w5 / (1.0 + w5)) + num7 * dilog((tb + w5) / (-1.0 + w5)) + num8 * dilog((tb + w5) / (1.0 + w5))
                    - num7 * dilog(w4 / (w4 - w7)) + num7 * dilog((-tb + w4) / (w4 - w7)) - num7 * dilog(w5 / (w5 + w7)) + num7 * dilog((tb + w5) / (w5 + w7)) + 4.0 * num8 * ln2 * log(1.0 - tb)
                    + 4.0 * num7 * ln2 * log(1.0 + tb) + num8 * log(1.0 - tb / w4) * log((-1.0 + tb) / (-1.0 + w4)) + num7 * log(1.0 - tb / w4) * log((1.0 + tb) / (1.0 + w4))
                    + num7 * log((1.0 + tb) / (1.0 - w5)) * log((tb + w5) / w5) + num8 * log((1.0 - tb) / (1.0 + w5)) * log((tb + w5) / w5) + num7 * log(1.0 - tb / w4) * log((tb - w7) / (w4 - w7))
                    + num7 * log((tb + w5) / w5) * log((-tb + w7) / (w5 + w7)) + (2.0 * num8 * log(1.0 - tb) + 2.0 * num7 * log(1.0 + tb)) * log(xb)
                    + (2.0 * 1.0i) * num7 * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - tb / w4, 1.0 + w4inv) + (2.0 * 1.0i) * num8 * M_PI * log((-1.0 + w4) / w4) * my_sign(imag(w4)) * T(1.0, 1.0 - tb / w4, (-1.0 + w4) / w4)
                    + (2.0 * 1.0i) * num7 * M_PI * log(1.0 - w7 / w4) * my_sign(imag(w4 / w7)) * T(1.0, 1.0 - tb / w4, 1.0 - w7 / w4) + (2.0 * 1.0i) * num8 * M_PI * log(1.0 + w5inv) * my_sign(-imag(w5)) * T(1.0, (tb + w5) / w5, 1.0 + w5inv)
                    + (2.0 * 1.0i) * num7 * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (tb + w5) / w5, (-1.0 + w5) / w5) + (2.0 * 1.0i) * num7 * M_PI * log((w5 + w7) / w5) * my_sign(-imag(w5 / w7)) * T(1.0, (tb + w5) / w5, (w5 + w7) / w5)) / denom5
                + ((-num10 - num11) * dilog(-1.0i / (-1.0i + wx3)) + (-num10 - num11) * dilog(1.0i / (1.0i + wx3)) + (-num10 - num11) * dilog(-1.0i / (-1.0i + wx4))
                    + (-num10 - num11) * dilog(1.0i / (1.0i + wx4)) + num10 * dilog((1.0i - xb) / (1.0i + wx3)) + num10 * dilog((1.0i - xb) / (1.0i + wx4)) + num5 * dilog(-(xb / wx3)) + num6 * dilog(xb / wx3)
                    + num5 * dilog(-(xb / wx4)) + num6 * dilog(xb / wx4) + num11 * dilog((-1.0i + xb) / (-1.0i + wx3)) + num11 * dilog((-1.0i + xb) / (-1.0i + wx4)) + num10 * dilog((1.0i + xb) / (1.0i - wx3))
                    + num11 * dilog((1.0i + xb) / (1.0i + wx3)) + num10 * dilog((1.0i + xb) / (1.0i - wx4)) + num11 * dilog((1.0i + xb) / (1.0i + wx4)) + num11 * log((wx3 - xb) / (1.0i + wx3)) * log(1.0 - 1.0i * xb)
                    + num11 * log((wx3 - xb) / (-1.0i + wx3)) * log(1.0 + 1.0i * xb) + (-2.0 * num10 * ln2 + num5 * log(xb)) * log((wx3 + xb) / wx3) - 2.0 * num10 * ln2 * log((wx4 + xb) / wx4)
                    + log(1.0 - 1.0i * xb) * (num11 * log((wx4 - xb) / (1.0i + wx4)) + num10 * log((wx3 + xb) / (-1.0i + wx3)) + num10 * log((wx4 + xb) / (-1.0i + wx4)))
                    + log(1.0 + 1.0i * xb) * (num11 * log((wx4 - xb) / (-1.0i + wx4)) + num10 * log((wx3 + xb) / (1.0i + wx3)) + num10 * log((wx4 + xb) / (1.0i + wx4))) - 2.0 * num11 * ln2 * log(1.0 - xb / wx3)
                    - 2.0 * num11 * ln2 * log(1.0 - xb / wx4) + log(xb) * (num5 * log((wx4 + xb) / wx4) + num6 * log(1.0 - xb / wx3) + num6 * log(1.0 - xb / wx4))
                    + (2.0 * 1.0i) * num11 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * num10 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                    + (2.0 * 1.0i) * num11 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * num10 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                    + (2.0 * 1.0i) * num10 * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * num11 * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                    + (2.0 * 1.0i) * num10 * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * num11 * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)) / denom4;

            const complex<double> f27bpart2 = (num12 * ((-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * power_of<2>(log(1.0 - 1.0i * xb)) + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * power_of<2>(log(1.0 + 1.0i * xb))
                + dilog(1.0 / (1.0 + w7)) * (4.0 * log(1.0 - 1.0i * xb) + 4.0 * log(1.0 + 1.0i * xb) - 4.0 * log(xb)) + dilog(tb) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb)) + dilog(-vb) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb))
                + dilog((1.0 + vb) / (1.0 - w7)) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb)) + dilog((-1.0 + tb) / (-1.0 + w7)) * (2.0 * log(1.0 - 1.0i * xb) + 2.0 * log(1.0 + 1.0i * xb) - 2.0 * log(xb)) + dilog((1.0 - tb) / 2.0) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb))
                + dilog(-tb) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb)) + dilog((1.0 + tb) / 2.0) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb)) + dilog((1.0 - vb) / 2.0) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb)) + dilog(vb) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb))
                + dilog((1.0 + vb) / 2.0) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb)) + dilog((1.0 + tb) / (1.0 + w7)) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb)) + dilog((1.0 - vb) / (1.0 + w7)) * (-2.0 * log(1.0 - 1.0i * xb) - 2.0 * log(1.0 + 1.0i * xb) + 2.0 * log(xb))
                + dilog(1.0 / (1.0 - w7)) * (-4.0 * log(1.0 - 1.0i * xb) - 4.0 * log(1.0 + 1.0i * xb) + 4.0 * log(xb)) + dilog(-1.0i / (-1.0i + wx3)) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb)
                + dilog(1.0i / (1.0i + wx3)) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb)
                + dilog(-1.0i / (-1.0i + wx4)) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb)
                + dilog(1.0i / (1.0i + wx4)) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb)
                + dilog(1.0 / 2.0 + (1.0i / 2.0) * xb) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb)
                + dilog(((-0.5) * 1.0i) * (1.0i + xb)) * (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 2.0 * ln1myb + 4.0 * log(yb) - 2.0 * ln1pyb)
                + dilog(-(xb / wx3)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog(xb / wx3) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog(-(xb / wx4)) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog(xb / wx4) * (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7) - ln1myb + 2.0 * log(yb) - ln1pyb)
                + dilog((1.0i - xb) / (1.0i + wx3)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((1.0i - xb) / (1.0i + wx4)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((-1.0i + xb) / (-1.0i + wx3)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((-1.0i + xb) / (-1.0i + wx4)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((1.0i + xb) / (1.0i - wx3)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((1.0i + xb) / (1.0i + wx3)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((1.0i + xb) / (1.0i - wx4)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((1.0i + xb) / (1.0i + wx4)) * (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7) + ln1myb - 2.0 * log(yb) + ln1pyb)
                + dilog((-1.0i) * xb) * (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7) + 2.0 * ln1myb - 4.0 * log(yb) + 2.0 * ln1pyb)
                + dilog(1.0i * xb) * (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7) + 2.0 * ln1myb - 4.0 * log(yb) + 2.0 * ln1pyb)
                + li2half * (4.0 * log(1.0 - tb) + 4.0 * log(tb) - 4.0 * log(1.0 + tb) - 4.0 * log(1.0 - vb) + 4.0 * log(vb) + 4.0 * log(1.0 + vb) - 4.0 * log(1.0 - tb / w7) - 4.0 * log((vb + w7) / w7) + 8.0 * log(1.0 - 1.0i * xb) + 8.0 * log(1.0 + 1.0i * xb) - 8.0 * log(xb) + 4.0 * ln1myb - 8.0 * log(yb) + 4.0 * ln1pyb)
                + log(xb) * (-2.0 * ln2 * log(1.0 - tb) + power_of<2>(log(1.0 - tb)) - 2.0 * ln2 * log(1.0 + tb) + power_of<2>(log(1.0 + tb)) - 2.0 * ln2 * log(1.0 - vb) + power_of<2>(log(1.0 - vb)) - 2.0 * ln2 * log(1.0 + vb) + power_of<2>(log(1.0 + vb)) - 2.0 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) + 2.0 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7))
                  - 2.0 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7)) + 2.0 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7)) + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx3 + xb) / wx3)
                  + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log((wx4 + xb) / wx4) + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log(1.0 - xb / wx3)
                  + (-log(1.0 - tb) - log(tb) + log(1.0 + tb) + log(1.0 - vb) - log(vb) - log(1.0 + vb) + log(1.0 - tb / w7) + log((vb + w7) / w7)) * log(1.0 - xb / wx4) - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7) + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7)
                  + (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7) - (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7)) + log(1.0 - 1.0i * xb) * (2.0 * ln2 * log(1.0 - tb) - power_of<2>(log(1.0 - tb)) + 2.0 * ln2 * log(1.0 + tb) - power_of<2>(log(1.0 + tb)) + 2.0 * ln2 * log(1.0 - vb) - power_of<2>(log(1.0 - vb))
                  + 2.0 * ln2 * log(1.0 + vb) - power_of<2>(log(1.0 + vb)) + 2.0 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) - 2.0 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7)) + 2.0 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7)) - 2.0 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7))
                  + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx3 - xb) / (1.0i + wx3)) + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx4 - xb) / (1.0i + wx4))
                   + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7)) * log(1.0 / 2.0 + (1.0i / 2.0) * xb)
                  + (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7)) * log(xb) + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx3 + xb) / (-1.0i + wx3))
                   + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx4 + xb) / (-1.0i + wx4)) + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7)
                  - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7) + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7))
                + log(1.0 + 1.0i * xb) * (2.0 * ln2 * log(1.0 - tb) - power_of<2>(log(1.0 - tb)) + 2.0 * ln2 * log(1.0 + tb) - power_of<2>(log(1.0 + tb)) + 2.0 * ln2 * log(1.0 - vb) - power_of<2>(log(1.0 - vb)) + 2.0 * ln2 * log(1.0 + vb) - power_of<2>(log(1.0 + vb)) + 2.0 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) - 2.0 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7))
                  + 2.0 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7)) - 2.0 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7)) + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx3 - xb) / (-1.0i + wx3))
                  + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx4 - xb) / (-1.0i + wx4)) + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7)
                    + 2.0 * log((vb + w7) / w7)) * log(1.0 / 2.0 - (1.0i / 2.0) * xb) + (2.0 * log(1.0 - tb) + 2.0 * log(tb) - 2.0 * log(1.0 + tb) - 2.0 * log(1.0 - vb) + 2.0 * log(vb) + 2.0 * log(1.0 + vb) - 2.0 * log(1.0 - tb / w7) - 2.0 * log((vb + w7) / w7)) * log(xb)
                  + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx3 + xb) / (1.0i + wx3)) + (log(1.0 - tb) + log(tb) - log(1.0 + tb) - log(1.0 - vb) + log(vb) + log(1.0 + vb) - log(1.0 - tb / w7) - log((vb + w7) / w7)) * log((wx4 + xb) / (1.0i + wx4))
                  + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7) - (4.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7) + (4.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7))
                + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 + tb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)
                + (2.0 * 1.0i) * M_PI * log(vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + vb) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) - (2.0 * 1.0i) * M_PI * log(1.0 - tb / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)
                - (2.0 * 1.0i) * M_PI * log((vb + w7) / w7) * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4) + ln1myb * (log((wx3 - xb) / (1.0i + wx3)) * log(1.0 - 1.0i * xb) - power_of<2>(log(1.0 - 1.0i * xb)) + log((wx3 - xb) / (-1.0i + wx3)) * log(1.0 + 1.0i * xb) - power_of<2>(log(1.0 + 1.0i * xb))
                  + log(1.0 - 1.0i * xb) * (log((wx4 - xb) / (1.0i + wx4)) - 2.0 * log(1.0 / 2.0 + (1.0i / 2.0) * xb) + 2.0 * log(xb) + log((wx3 + xb) / (-1.0i + wx3)) + log((wx4 + xb) / (-1.0i + wx4))) + log(1.0 + 1.0i * xb) * (log((wx4 - xb) / (-1.0i + wx4)) - 2.0 * log(1.0 / 2.0 - (1.0i / 2.0) * xb) + 2.0 * log(xb) + log((wx3 + xb) / (1.0i + wx3)) + log((wx4 + xb) / (1.0i + wx4)))
                  + log(xb) * (-log((wx3 + xb) / wx3) - log((wx4 + xb) / wx4) - log(1.0 - xb / wx3) - log(1.0 - xb / wx4)) + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                  + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                  + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)) + ln1pyb * (log((wx3 - xb) / (1.0i + wx3)) * log(1.0 - 1.0i * xb) - power_of<2>(log(1.0 - 1.0i * xb)) + log((wx3 - xb) / (-1.0i + wx3)) * log(1.0 + 1.0i * xb) - power_of<2>(log(1.0 + 1.0i * xb))
                  + log(1.0 - 1.0i * xb) * (log((wx4 - xb) / (1.0i + wx4)) - 2.0 * log(1.0 / 2.0 + (1.0i / 2.0) * xb) + 2.0 * log(xb) + log((wx3 + xb) / (-1.0i + wx3)) + log((wx4 + xb) / (-1.0i + wx4))) + log(1.0 + 1.0i * xb) * (log((wx4 - xb) / (-1.0i + wx4)) - 2.0 * log(1.0 / 2.0 - (1.0i / 2.0) * xb) + 2.0 * log(xb) + log((wx3 + xb) / (1.0i + wx3)) + log((wx4 + xb) / (1.0i + wx4)))
                  + log(xb) * (-log((wx3 + xb) / wx3) - log((wx4 + xb) / wx4) - log(1.0 - xb / wx3) - log(1.0 - xb / wx4)) + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3)
                  + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3)
                  + (2.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) + (2.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)) + log(yb) * (-2.0 * log((wx3 - xb) / (1.0i + wx3)) * log(1.0 - 1.0i * xb) + 2.0 * power_of<2>(log(1.0 - 1.0i * xb)) - 2.0 * log((wx3 - xb) / (-1.0i + wx3)) * log(1.0 + 1.0i * xb)
                  + 2.0 * power_of<2>(log(1.0 + 1.0i * xb)) + log(1.0 - 1.0i * xb) * (-2.0 * log((wx4 - xb) / (1.0i + wx4)) + 4.0 * log(1.0 / 2.0 + (1.0i / 2.0) * xb) - 4.0 * log(xb) - 2.0 * log((wx3 + xb) / (-1.0i + wx3)) - 2.0 * log((wx4 + xb) / (-1.0i + wx4)))
                  + log(1.0 + 1.0i * xb) * (-2.0 * log((wx4 - xb) / (-1.0i + wx4)) + 4.0 * log(1.0 / 2.0 - (1.0i / 2.0) * xb) - 4.0 * log(xb) - 2.0 * log((wx3 + xb) / (1.0i + wx3)) - 2.0 * log((wx4 + xb) / (1.0i + wx4))) + log(xb) * (2.0 * log((wx3 + xb) / wx3) + 2.0 * log((wx4 + xb) / wx4) + 2.0 * log(1.0 - xb / wx3) + 2.0 * log(1.0 - xb / wx4))
                  - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx3) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx3) - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 - 1.0i * wx4) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - 1.0i * xb, 1.0 + 1.0i * wx4)
                  - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx3) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx3) - (4.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 - 1.0i * wx4) - (4.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 + 1.0i * xb, 1.0 + 1.0i * wx4)))) / denom7;

            const complex<double> f27bpart3 = (num12 / denom7) * f279b_helper(clp);

            const complex<double> f27bpart4 = (num12 * ln2 * (16.0 * dilog(wx3 / (-1.0i + wx3)) + 16.0 * dilog(wx3 / (1.0i + wx3)) + 16.0 * dilog(wx4 / (-1.0i + wx4)) + 16.0 * dilog(wx4 / (1.0i + wx4))
                    - 8.0 * dilog((wx3 - xb) / (-1.0i + wx3)) - 8.0 * dilog((wx3 - xb) / (1.0i + wx3)) - 8.0 * dilog((wx4 - xb) / (-1.0i + wx4)) - 8.0 * dilog((wx4 - xb) / (1.0i + wx4)) + 16.0 * dilog((-1.0i) * xb)
                    + 16.0 * dilog(1.0i * xb) - 8.0 * dilog(-(xb / wx3)) - 8.0 * dilog(xb / wx3) - 8.0 * dilog(-(xb / wx4)) - 8.0 * dilog(xb / wx4) - 8.0 * dilog((wx3 + xb) / (-1.0i + wx3))
                    - 8.0 * dilog((wx3 + xb) / (1.0i + wx3)) - 8.0 * dilog((wx4 + xb) / (-1.0i + wx4)) - 8.0 * dilog((wx4 + xb) / (1.0i + wx4))
                    + (4.0 * log(tb) + 4.0 * log(vb) - 4.0 * log(1.0 - tb / w7) - 4.0 * log((vb + w7) / w7)) * log(1.0 - 1.0i * xb) + (4.0 * log(tb) + 4.0 * log(vb) - 4.0 * log(1.0 - tb / w7) - 4.0 * log((vb + w7) / w7)) * log(1.0 + 1.0i * xb)
                    + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7)
                        - 8.0 * log((1.0i - xb) / (1.0i + wx3)) - 8.0 * log((1.0i + xb) / (1.0i - wx3))) * log((wx3 + xb) / wx3) + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb)
                        - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 8.0 * log((1.0i - xb) / (1.0i + wx4)) - 8.0 * log((1.0i + xb) / (1.0i - wx4))) * log((wx4 + xb) / wx4)
                    + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb) - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7)
                        - 8.0 * log((-1.0i + xb) / (-1.0i + wx3)) - 8.0 * log((1.0i + xb) / (1.0i + wx3))) * log(1.0 - xb / wx3) + (-2.0 * log(1.0 - tb) - 2.0 * log(tb) + 2.0 * log(1.0 + tb) + 2.0 * log(1.0 - vb) - 2.0 * log(vb)
                        - 2.0 * log(1.0 + vb) + 2.0 * log(1.0 - tb / w7) + 2.0 * log((vb + w7) / w7) - 8.0 * log((-1.0i + xb) / (-1.0i + wx4)) - 8.0 * log((1.0i + xb) / (1.0i + wx4))) * log(1.0 - xb / wx4)
                    + (4.0 * log(1.0 - 1.0i * xb) + 4.0 * log(1.0 + 1.0i * xb) - 2.0 * log((wx3 + xb) / wx3) - 2.0 * log((wx4 + xb) / wx4) - 2.0 * log(1.0 - xb / wx3) - 2.0 * log(1.0 - xb / wx4)) * ln1myb
                    + (-8.0 * log(1.0 - 1.0i * xb) - 8.0 * log(1.0 + 1.0i * xb) + 4.0 * log((wx3 + xb) / wx3) + 4.0 * log((wx4 + xb) / wx4) + 4.0 * log(1.0 - xb / wx3) + 4.0 * log(1.0 - xb / wx4)) * log(yb)
                    + (4.0 * log(1.0 - 1.0i * xb) + 4.0 * log(1.0 + 1.0i * xb) - 2.0 * log((wx3 + xb) / wx3) - 2.0 * log((wx4 + xb) / wx4) - 2.0 * log(1.0 - xb / wx3) - 2.0 * log(1.0 - xb / wx4)) * ln1pyb
                    - (16.0 * 1.0i) * M_PI * log((-1.0i + wx3) / wx3) * my_sign(-real(wx3)) * T(1.0, (wx3 + xb) / wx3, (-1.0i + wx3) / wx3) - (16.0 * 1.0i) * M_PI * log((1.0i + wx3) / wx3) * my_sign(real(wx3)) * T(1.0, (wx3 + xb) / wx3, (1.0i + wx3) / wx3)
                    - (16.0 * 1.0i) * M_PI * log((-1.0i + wx4) / wx4) * my_sign(-real(wx4)) * T(1.0, (wx4 + xb) / wx4, (-1.0i + wx4) / wx4)
                    - (16.0 * 1.0i) * M_PI * log((1.0i + wx4) / wx4) * my_sign(real(wx4)) * T(1.0, (wx4 + xb) / wx4, (1.0i + wx4) / wx4) - (16.0 * 1.0i) * M_PI * log((-1.0i + wx3) / wx3) * my_sign(-real(wx3)) * T(1.0, 1.0 - xb / wx3, (-1.0i + wx3) / wx3)
                    - (16.0 * 1.0i) * M_PI * log((1.0i + wx3) / wx3) * my_sign(real(wx3)) * T(1.0, 1.0 - xb / wx3, (1.0i + wx3) / wx3) - (16.0 * 1.0i) * M_PI * log((-1.0i + wx4) / wx4) * my_sign(-real(wx4)) * T(1.0, 1.0 - xb / wx4, (-1.0i + wx4) / wx4)
                    - (16.0 * 1.0i) * M_PI * log((1.0i + wx4) / wx4) * my_sign(real(wx4)) * T(1.0, 1.0 - xb / wx4, (1.0i + wx4) / wx4))) / denom7;

            const complex<double> f27bpart5 = term1 + (logs9 * log(1.0 - 1.0i * xb) + logs10 * log(1.0 + 1.0i * xb)) / denom10 + ((-logs1 - logs2) * li2half + logs1 * dilog(1.0 / 2.0 + (1.0i / 2.0) * xb)
                    + logs2 * dilog(((-0.5) * 1.0i) * (1.0i + xb)) + logs2 * log(1.0 / 2.0 + (1.0i / 2.0) * xb) * log(1.0 - 1.0i * xb) + logs1 * log(1.0 / 2.0 - (1.0i / 2.0) * xb) * log(1.0 + 1.0i * xb)) / denom3
                + ((-logs4 - logs7) * li2half + logs4 * dilog((1.0 - tb) / 2.0) + logs4 * dilog(-tb) - logs7 * dilog(tb) + logs7 * dilog((1.0 + tb) / 2.0) + logs7 * dilog(1.0 / (1.0 - w7))
                    - logs7 * dilog((-1.0 + tb) / (-1.0 + w7)) - logs4 * dilog(1.0 / (1.0 + w7)) + logs4 * dilog((1.0 + tb) / (1.0 + w7)) - logs4 * ln2 * log(1.0 - tb) + (logs7 * power_of<2>(log(1.0 - tb))) / 2.0
                    - logs7 * ln2 * log(1.0 + tb) + (logs4 * power_of<2>(log(1.0 + tb))) / 2.0 - logs7 * log(1.0 - tb) * log((-tb + w7) / (-1.0 + w7)) + logs4 * log(1.0 + tb) * log((-tb + w7) / (1.0 + w7))
                    + (logs5 * log(1.0 - tb) - logs5 * log(1.0 + tb)) * log(1.0 - 1.0i * xb) + (logs5 * log(1.0 - tb) - logs5 * log(1.0 + tb)) * log(1.0 + 1.0i * xb)
                    - (2.0 * 1.0i) * logs7 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - tb, 1.0 - w7) + (2.0 * 1.0i) * logs4 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 + tb, 1.0 + w7)) / denom9
                + ((-logs3 - logs8) * li2half + logs8 * dilog((1.0 - vb) / 2.0) - logs8 * dilog(-vb) + logs3 * dilog(vb) + logs3 * dilog((1.0 + vb) / 2.0) + logs8 * dilog(1.0 / (1.0 - w7))
                    - logs8 * dilog((1.0 + vb) / (1.0 - w7)) - logs3 * dilog(1.0 / (1.0 + w7)) + logs3 * dilog((1.0 - vb) / (1.0 + w7)) - logs8 * ln2 * log(1.0 - vb) + (logs3 * power_of<2>(log(1.0 - vb))) / 2.0
                    - logs3 * ln2 * log(1.0 + vb) + (logs8 * power_of<2>(log(1.0 + vb))) / 2.0 - logs8 * log(1.0 + vb) * log((vb + w7) / (-1.0 + w7)) + logs3 * log(1.0 - vb) * log((vb + w7) / (1.0 + w7))
                    + (-(logs6 * log(1.0 - vb)) + logs6 * log(1.0 + vb)) * log(1.0 - 1.0i * xb) + (-(logs6 * log(1.0 - vb)) + logs6 * log(1.0 + vb)) * log(1.0 + 1.0i * xb)
                    + (2.0 * 1.0i) * logs3 * M_PI * log(1.0 + w7) * my_sign(-imag(w7inv)) * T(1.0, 1.0 - vb, 1.0 + w7) - (2.0 * 1.0i) * logs8 * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + vb, 1.0 - w7)) / denom8;

            return f27bpart1 + f27bpart2 +f27bpart3 + f27bpart4 + f27bpart5;
        }
    }
}
