/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2023 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/maths/power-of.hh>
#include <eos/maths/polylog.hh>
#include <eos/maths/multiplepolylog-li22.hh>

#include <eos/rare-b-decays/charm-loops-impl.hh>

#include <eos/utils/exception.hh>
#include <eos/utils/log.hh>
#include <eos/utils/stringify.hh>

#include <cmath>
#include <complex>

namespace eos
{
	using std::complex;
	using std::log;
	using std::real;
	using std::imag;

	namespace agv_2019a
	{
        /*
            blocks in F27d and F29d:

            f29dpart7 + f29dpart8 = num20 * f279d_log2_terms / denom2

            f27dpart6 = - num21 * f279d_log2_terms / denom3
        */

        complex<double> f279d_log2_terms(const CharmLoopsParameters & clp)
        {
            const complex<double> xd = clp.xd;
            const complex<double> yd = clp.yd;

            const complex<double> xdinv = 1.0 / xd;
            const complex<double> ydinv = 1.0 / yd;

            const double imydinv = imag(ydinv);

            // Powers of xd and yd

            const complex<double> xd2 = power_of<2>(xd);
            const complex<double> xd4 = power_of<4>(xd);

            // weights appearing in the GPLs [AGV:2019A] p. 34

            const complex<double> w4 = (2.0 * xd) / power_of<2>(1.0 - xd);
            const complex<double> w5 = (2.0 * xd) / power_of<2>(1.0 + xd);
            const complex<double> w7 = (8.0 * xd2) / (1.0 - 6.0 * xd2 + xd4);

            const complex<double> w4inv = 1.0 / w4;
            const complex<double> w5inv = 1.0 / w5;
            const complex<double> w7inv = 1.0 / w7;

            // Logs of xd and yd

            const complex<double> ln1pyd = log(1.0 + yd);
            const complex<double> ln1myd = log(1.0 - yd);


            const complex<double> resultpart1 = ln2 * (dilog((w5 - yd) / (-1.0 + w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log((wx3 + xd) / wx3) - 8.0 * log((wx4 + xd) / wx4))
                + dilog(-(yd / w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log((wx3 + xd) / wx3) - 8.0 * log((wx4 + xd) / wx4))
                + dilog(yd / w5) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log((wx3 + xd) / wx3) - 8.0 * log((wx4 + xd) / wx4)) + dilog((w5 + yd) / (-1.0 + w5)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log((wx3 + xd) / wx3)
                    - 8.0 * log((wx4 + xd) / wx4)) + dilog(w5 / (-1.0 + w5)) * (-16.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0 + 1.0i * xd) + 16.0 * log((wx3 + xd) / wx3) + 16.0 * log((wx4 + xd) / wx4))
                + dilog((w4 - yd) / (1.0 + w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4))
                + dilog(-(yd / w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4)) + dilog(yd / w4) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4))
                + dilog((w4 + yd) / (1.0 + w4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4))
                + dilog(w7 / (1.0 + w7)) * (16.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0 + 1.0i * xd) - 8.0 * log((wx3 + xd) / wx3) - 8.0 * log((wx4 + xd) / wx4) - 8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4))
                + dilog((1.0 - yd) / 2.0) * (-4.0 * log((wx3 + xd) / wx3) - 4.0 * log((wx4 + xd) / wx4) - 4.0 * log(1.0 - xd / wx3) - 4.0 * log(1.0 - xd / wx4))
                + dilog((1.0 + yd) / 2.0) * (-4.0 * log((wx3 + xd) / wx3) - 4.0 * log((wx4 + xd) / wx4) - 4.0 * log(1.0 - xd / wx3) - 4.0 * log(1.0 - xd / wx4))
                + dilog(-yd) * (4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4)) + dilog(yd) * (4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3)
                    + 4.0 * log(1.0 - xd / wx4)) + dilog((w7 - yd) / (1.0 + w7)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4))
                + dilog(-(yd / w7)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4))
                + dilog(yd / w7) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4))
                + dilog((w7 + yd) / (1.0 + w7)) * (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4))
                + li2half * (8.0 * log((wx3 + xd) / wx3) + 8.0 * log((wx4 + xd) / wx4) + 8.0 * log(1.0 - xd / wx3) + 8.0 * log(1.0 - xd / wx4)) + dilog(w4 / (1.0 + w4)) * (-16.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0 + 1.0i * xd) + 16.0 * log(1.0 - xd / wx3)
                    + 16.0 * log(1.0 - xd / wx4)) + (4.0 * ln2 * log((wx3 + xd) / wx3) + 4.0 * ln2 * log((wx4 + xd) / wx4) + 4.0 * ln2 * log(1.0 - xd / wx3) + 4.0 * ln2 * log(1.0 - xd / wx4)) * ln1myd
                + (4.0 * ln2 * log((wx3 + xd) / wx3) + 4.0 * ln2 * log((wx4 + xd) / wx4) + 4.0 * ln2 * log(1.0 - xd / wx3) + 4.0 * ln2 * log(1.0 - xd / wx4)) * ln1pyd
                + (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4)) * log((1.0 - yd) / (1.0 + w4)) * log((w4 + yd) / w4)
                + (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log((wx3 + xd) / wx3) - 8.0 * log((wx4 + xd) / wx4)) * log((1.0 + yd) / (1.0 - w5)) * log((w5 + yd) / w5)
                + (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4)) * log((1.0 - yd) / (1.0 + w7)) * log((w7 + yd) / w7)
                + (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log(1.0 - xd / wx3) - 8.0 * log(1.0 - xd / wx4)) * log((1.0 + yd) / (1.0 + w4)) * log(1.0 - yd / w4)
                + (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd) - 8.0 * log((wx3 + xd) / wx3) - 8.0 * log((wx4 + xd) / wx4)) * log((-1.0 + yd) / (-1.0 + w5)) * log(1.0 - yd / w5)
                + (-8.0 * log(1.0 - 1.0i * xd) - 8.0 * log(1.0 + 1.0i * xd) + 4.0 * log((wx3 + xd) / wx3) + 4.0 * log((wx4 + xd) / wx4) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4)) * log((1.0 + yd) / (1.0 + w7)) * log(1.0 - yd / w7)
                + log(1.0 - xd / wx3) * ((-16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                    - (16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                + log(1.0 - xd / wx4) * ((-16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                    - (16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                + log((wx3 + xd) / wx3) * ((-16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                    - (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                + log((wx4 + xd) / wx4) * ((-16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv)
                    - (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                + log(1.0 - 1.0i * xd) * ((16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5)
                    - (16.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv)
                    + (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (16.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv))
                + log(1.0 + 1.0i * xd) * ((16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, (w4 + yd) / w4, 1.0 + w4inv) + (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, (w5 + yd) / w5, (-1.0 + w5) / w5)
                    - (16.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, (w7 + yd) / w7, 1.0 + w7inv) + (16.0 * 1.0i) * M_PI * log(1.0 + w4inv) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd / w4, 1.0 + w4inv)
                    + (16.0 * 1.0i) * M_PI * log((-1.0 + w5) / w5) * my_sign(imag(w5)) * T(1.0, 1.0 - yd / w5, (-1.0 + w5) / w5) - (16.0 * 1.0i) * M_PI * log(1.0 + w7inv) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd / w7, 1.0 + w7inv)));

            const complex<double> resultpart2 = ln2 * (64.0 * trilog(-1.0) + 112.0 * li3half - 16.0 * trilog(1.0 + w4inv) - 16.0 * trilog(1.0 / (1.0 - w4)) - 16.0 * trilog(-w4inv) - 16.0 * trilog(w4inv) + 16.0 * trilog((-1.0 + w4) / (2.0 * w4))
                - 16.0 * trilog((-1.0 + w4) / w4) - 16.0 * trilog(1.0 / (1.0 + w4)) + 16.0 * trilog((1.0 - w4) / (1.0 + w4)) + 16.0 * trilog((1.0 + w4) / (1.0 - w4)) + 16.0 * trilog((1.0 + w4) / (2.0 * w4)) - 16.0 * trilog(1.0 + w5inv) - 16.0 * trilog(1.0 / (1.0 - w5))
                - 16.0 * trilog(-w5inv) - 16.0 * trilog(w5inv) + 16.0 * trilog((-1.0 + w5) / (2.0 * w5)) - 16.0 * trilog((-1.0 + w5) / w5) - 16.0 * trilog(1.0 / (1.0 + w5)) + 16.0 * trilog((1.0 - w5) / (1.0 + w5)) + 16.0 * trilog((1.0 + w5) / (1.0 - w5))
                + 16.0 * trilog((1.0 + w5) / (2.0 * w5)) + 8.0 * trilog(1.0 + w7inv) + 8.0 * trilog(1.0 / (1.0 - w7)) + 8.0 * trilog(-w7inv) + 8.0 * trilog(w7inv) - 8.0 * trilog((-1.0 + w7) / (2.0 * w7)) + 8.0 * trilog((-1.0 + w7) / w7)
                + 8.0 * trilog(1.0 / (1.0 + w7)) - 8.0 * trilog((1.0 - w7) / (1.0 + w7)) - 8.0 * trilog((1.0 + w7) / (1.0 - w7)) - 8.0 * trilog((1.0 + w7) / (2.0 * w7)) + 16.0 * trilog(1.0 - 1.0i * wx3) + 16.0 * trilog(1.0 + 1.0i * wx3) + 16.0 * trilog(-1.0i / wx3) + 16.0 * trilog(1.0i / wx3)
                + 16.0 * trilog(-1.0i / (-1.0i + wx3)) + 16.0 * trilog(wx3 / (-1.0i + wx3)) - 16.0 * trilog((-1.0i + wx3) / (2.0 * wx3)) + 16.0 * trilog(1.0i / (1.0i + wx3)) - 16.0 * trilog((1.0i - wx3) / (1.0i + wx3)) + 16.0 * trilog(wx3 / (1.0i + wx3))
                - 16.0 * trilog((1.0i + wx3) / (1.0i - wx3)) - 16.0 * trilog((1.0i + wx3) / (2.0 * wx3)) + 16.0 * trilog(1.0 - 1.0i * wx4) + 16.0 * trilog(1.0 + 1.0i * wx4) + 16.0 * trilog(-1.0i / wx4) + 16.0 * trilog(1.0i / wx4) + 16.0 * trilog(-1.0i / (-1.0i + wx4))
                + 16.0 * trilog(wx4 / (-1.0i + wx4)) - 16.0 * trilog((-1.0i + wx4) / (2.0 * wx4)) + 16.0 * trilog(1.0i / (1.0i + wx4)) - 16.0 * trilog((1.0i - wx4) / (1.0i + wx4)) + 16.0 * trilog(wx4 / (1.0i + wx4)) - 16.0 * trilog((1.0i + wx4) / (1.0i - wx4))
                - 16.0 * trilog((1.0i + wx4) / (2.0 * wx4)) - 8.0 * trilog((1.0i + wx3) / (1.0i - xd)) - 8.0 * trilog((1.0i + wx4) / (1.0i - xd)) - 8.0 * trilog((1.0i - xd) / (1.0i + wx3)) - 8.0 * trilog((1.0i - xd) / (1.0i + wx4)) - 8.0 * trilog((wx3 - xd) / (-1.0i + wx3))
                - 8.0 * trilog((wx3 - xd) / (1.0i + wx3)) - 8.0 * trilog((wx4 - xd) / (-1.0i + wx4)) - 8.0 * trilog((wx4 - xd) / (1.0i + wx4)) - 64.0 * trilog(1.0 / 2.0 - (1.0i / 2.0) * xd) + 48.0 * trilog(1.0 - 1.0i * xd) + 8.0 * trilog(((-1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 - xd)))
                + 8.0 * trilog(((-1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 - xd))) - 64.0 * trilog((1.0 + 1.0i * xd) / 2.0) + 48.0 * trilog(1.0 + 1.0i * xd) + 8.0 * trilog(((1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 - xd))) + 8.0 * trilog(((1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 - xd)))
                + 8.0 * trilog((1.0i * (wx3 - xd)) / ((-1.0i + wx3) * xd)) + 8.0 * trilog(((-1.0i) * (wx3 - xd)) / ((1.0i + wx3) * xd)) + 8.0 * trilog((1.0i * (wx4 - xd)) / ((-1.0i + wx4) * xd)) + 8.0 * trilog(((-1.0i) * (wx4 - xd)) / ((1.0i + wx4) * xd)) - 32.0 * trilog((-1.0i) * xd)
                - 32.0 * trilog(1.0i * xd) + 8.0 * trilog(-(xd / wx3)) + 8.0 * trilog(xd / wx3) + 8.0 * trilog(-(xd / wx4)) + 8.0 * trilog(xd / wx4) - 32.0 * trilog(-1.0i / (-1.0i + xd)) - 8.0 * trilog((-1.0i + wx3) / (-1.0i + xd)) - 8.0 * trilog((-1.0i + wx4) / (-1.0i + xd))
                + 8.0 * trilog(((-1.0i) * (wx3 - xd)) / (wx3 * (-1.0i + xd))) + 8.0 * trilog(((-1.0i) * (wx4 - xd)) / (wx4 * (-1.0i + xd))) - 16.0 * trilog(xd / (-1.0i + xd)) + 32.0 * trilog((2.0 * xd) / (-1.0i + xd)) - 8.0 * trilog((-1.0i + xd) / (-1.0i + wx3))
                - 8.0 * trilog((-1.0i + xd) / (-1.0i + wx4)) - 32.0 * trilog(1.0i / (1.0i + xd)) - 8.0 * trilog((1.0i - wx3) / (1.0i + xd)) - 8.0 * trilog((1.0i + wx3) / (1.0i + xd)) - 8.0 * trilog((1.0i - wx4) / (1.0i + xd)) - 8.0 * trilog((1.0i + wx4) / (1.0i + xd))
                + 8.0 * trilog((1.0i * (wx3 - xd)) / (wx3 * (1.0i + xd))) + 8.0 * trilog((1.0i * (wx4 - xd)) / (wx4 * (1.0i + xd))) - 16.0 * trilog(xd / (1.0i + xd)) + 32.0 * trilog((2.0 * xd) / (1.0i + xd)) - 32.0 * trilog((-1.0i + xd) / (1.0i + xd))
                + 8.0 * trilog(((-1.0i + wx3) * (-1.0i + xd)) / ((1.0i + wx3) * (1.0i + xd))) + 8.0 * trilog(((1.0i + wx3) * (-1.0i + xd)) / ((-1.0i + wx3) * (1.0i + xd))) + 8.0 * trilog(((-1.0i + wx4) * (-1.0i + xd)) / ((1.0i + wx4) * (1.0i + xd)))
                + 8.0 * trilog(((1.0i + wx4) * (-1.0i + xd)) / ((-1.0i + wx4) * (1.0i + xd))) - 8.0 * trilog((1.0i + xd) / (1.0i - wx3)) - 8.0 * trilog((1.0i + xd) / (1.0i + wx3)) - 8.0 * trilog((1.0i + xd) / (1.0i - wx4)) - 8.0 * trilog((1.0i + xd) / (1.0i + wx4))
                - 32.0 * trilog((1.0i + xd) / (-1.0i + xd)) + 8.0 * trilog(((-1.0i + wx3) * (1.0i + xd)) / ((1.0i + wx3) * (-1.0i + xd))) + 8.0 * trilog(((1.0i + wx3) * (1.0i + xd)) / ((-1.0i + wx3) * (-1.0i + xd))) + 8.0 * trilog(((-1.0i + wx4) * (1.0i + xd)) / ((1.0i + wx4) * (-1.0i + xd)))
                + 8.0 * trilog(((1.0i + wx4) * (1.0i + xd)) / ((-1.0i + wx4) * (-1.0i + xd))) - 8.0 * trilog((-wx3 + xd) / (-1.0i + xd)) - 8.0 * trilog((-wx3 + xd) / (1.0i + xd)) + 8.0 * trilog(((1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 + xd)))
                + 8.0 * trilog(((-1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 + xd))) - 8.0 * trilog((wx3 + xd) / (-1.0i + wx3)) - 8.0 * trilog((wx3 + xd) / (1.0i + wx3)) + 8.0 * trilog(((-1.0i) * (wx3 + xd)) / ((-1.0i + wx3) * xd))
                + 8.0 * trilog((1.0i * (wx3 + xd)) / ((1.0i + wx3) * xd)) - 8.0 * trilog((wx3 + xd) / (-1.0i + xd)) + 8.0 * trilog(((-1.0i) * (wx3 + xd)) / (wx3 * (-1.0i + xd))) - 8.0 * trilog((wx3 + xd) / (1.0i + xd)) + 8.0 * trilog((1.0i * (wx3 + xd)) / (wx3 * (1.0i + xd)))
                - 8.0 * trilog((-wx4 + xd) / (-1.0i + xd)) - 8.0 * trilog((-wx4 + xd) / (1.0i + xd)) + 8.0 * trilog(((1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 + xd))) + 8.0 * trilog(((-1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 + xd))) - 8.0 * trilog((wx4 + xd) / (-1.0i + wx4))
                - 8.0 * trilog((wx4 + xd) / (1.0i + wx4)) + 8.0 * trilog(((-1.0i) * (wx4 + xd)) / ((-1.0i + wx4) * xd)) + 8.0 * trilog((1.0i * (wx4 + xd)) / ((1.0i + wx4) * xd)) - 8.0 * trilog((wx4 + xd) / (-1.0i + xd)) + 8.0 * trilog(((-1.0i) * (wx4 + xd)) / (wx4 * (-1.0i + xd)))
                - 8.0 * trilog((wx4 + xd) / (1.0i + xd)) + 8.0 * trilog((1.0i * (wx4 + xd)) / (wx4 * (1.0i + xd))) - 24.0 * trilog((1.0 - yd) / 2.0) + 8.0 * trilog((1.0 - yd) / (1.0 + w4)) + 8.0 * trilog((1.0 - yd) / (1.0 + w5)) - 4.0 * trilog((1.0 - yd) / (1.0 + w7))
                + 8.0 * trilog((-1.0 + w4) / (w4 - yd)) + 8.0 * trilog((1.0 + w4) / (w4 - yd)) + 8.0 * trilog((-1.0 + w5) / (w5 - yd)) + 8.0 * trilog((1.0 + w5) / (w5 - yd)) - 4.0 * trilog((-1.0 + w7) / (w7 - yd)) - 4.0 * trilog((1.0 + w7) / (w7 - yd))
                + 8.0 * trilog((-1.0 + yd) / (-1.0 + w4)) + 8.0 * trilog((-1.0 + yd) / (-1.0 + w5)) - 4.0 * trilog((-1.0 + yd) / (-1.0 + w7)) - 8.0 * trilog((-1.0 / 2.0) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd)))
                - 8.0 * trilog((-1.0 / 2.0) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) + 4.0 * trilog((-1.0 / 2.0) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) + 8.0 * trilog((w4 * (-1.0 + yd)) / ((-1.0 + w4) * yd)) - 8.0 * trilog((w4 * (-1.0 + yd)) / ((1.0 + w4) * yd))
                - 8.0 * trilog((w5 * (-1.0 + yd)) / ((-1.0 + w5) * yd)) + 8.0 * trilog((w5 * (-1.0 + yd)) / ((1.0 + w5) * yd)) - 4.0 * trilog((w7 * (-1.0 + yd)) / ((-1.0 + w7) * yd)) + 4.0 * trilog((w7 * (-1.0 + yd)) / ((1.0 + w7) * yd))
                - 8.0 * trilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) - 8.0 * trilog(((1.0 + w4) * (-1.0 + yd)) / ((-1.0 + w4) * (1.0 + yd))) - 8.0 * trilog(((-1.0 + w5) * (-1.0 + yd)) / ((1.0 + w5) * (1.0 + yd)))
                - 8.0 * trilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd))) + 4.0 * trilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) + 4.0 * trilog(((1.0 + w7) * (-1.0 + yd)) / ((-1.0 + w7) * (1.0 + yd))) - 24.0 * trilog((1.0 + yd) / 2.0)
                + 8.0 * trilog((1.0 + yd) / (1.0 - w4)) + 8.0 * trilog((1.0 + yd) / (1.0 + w4)) + 8.0 * trilog((1.0 + yd) / (1.0 - w5)) + 8.0 * trilog((1.0 + yd) / (1.0 + w5)) - 4.0 * trilog((1.0 + yd) / (1.0 - w7)) - 4.0 * trilog((1.0 + yd) / (1.0 + w7))
                + 8.0 * trilog(-((1.0 + yd) / (w4 - yd))) - 8.0 * trilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) + 8.0 * trilog(-((1.0 + yd) / (w5 - yd))) - 8.0 * trilog(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) - 4.0 * trilog(-((1.0 + yd) / (w7 - yd)))
                + 4.0 * trilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) - 8.0 * trilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd))) - 8.0 * trilog(((1.0 + w4) * (1.0 + yd)) / ((-1.0 + w4) * (-1.0 + yd)))
                - 8.0 * trilog(((-1.0 + w5) * (1.0 + yd)) / ((1.0 + w5) * (-1.0 + yd))) - 8.0 * trilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) + 4.0 * trilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd)))
                + 4.0 * trilog(((1.0 + w7) * (1.0 + yd)) / ((-1.0 + w7) * (-1.0 + yd))) + 8.0 * trilog((w4 * (1.0 + yd)) / ((-1.0 + w4) * yd)) - 8.0 * trilog((w5 * (1.0 + yd)) / ((-1.0 + w5) * yd)) - 4.0 * trilog((w7 * (1.0 + yd)) / ((-1.0 + w7) * yd))
                + 8.0 * trilog((-1.0 + yd) / (-w4 + yd)) + 8.0 * trilog((-1.0 + w4) / (w4 + yd)) + 8.0 * trilog((1.0 + w4) / (w4 + yd)) + 8.0 * trilog((-1.0 + yd) / (w4 + yd)) - 8.0 * trilog((-1.0 / 2.0) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd)))
                + 8.0 * trilog((1.0 + yd) / (w4 + yd)) - 8.0 * trilog(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) + 8.0 * trilog((-1.0 + yd) / (-w5 + yd)) + 8.0 * trilog((-1.0 + w5) / (w5 + yd)) + 8.0 * trilog((1.0 + w5) / (w5 + yd))
                + 8.0 * trilog((-1.0 + yd) / (w5 + yd)) - 8.0 * trilog((-1.0 / 2.0) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) + 8.0 * trilog((1.0 + yd) / (w5 + yd)) - 8.0 * trilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) - 4.0 * trilog((-1.0 + yd) / (-w7 + yd))
                - 4.0 * trilog((-1.0 + w7) / (w7 + yd)) - 4.0 * trilog((1.0 + w7) / (w7 + yd)) - 4.0 * trilog((-1.0 + yd) / (w7 + yd)) + 4.0 * trilog((-1.0 / 2.0) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) - 4.0 * trilog((1.0 + yd) / (w7 + yd))
                + 4.0 * trilog(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) - 8.0 * trilog((w4 + w4 * yd) / (yd + w4 * yd)) + 8.0 * trilog((w5 + w5 * yd) / (yd + w5 * yd)) + 4.0 * trilog((w7 + w7 * yd) / (yd + w7 * yd)) + 16.0 * dilog(-w4inv) * lnhalf
                + 16.0 * dilog(w4inv) * lnhalf - 16.0 * dilog((-1.0 + w4) / (2.0 * w4)) * lnhalf - 16.0 * dilog((1.0 - w4) / (1.0 + w4)) * lnhalf - 16.0 * dilog((1.0 + w4) / (1.0 - w4)) * lnhalf - 16.0 * dilog((1.0 + w4) / (2.0 * w4)) * lnhalf
                + 16.0 * dilog(-w5inv) * lnhalf + 16.0 * dilog(w5inv) * lnhalf - 16.0 * dilog((-1.0 + w5) / (2.0 * w5)) * lnhalf - 16.0 * dilog((1.0 - w5) / (1.0 + w5)) * lnhalf - 16.0 * dilog((1.0 + w5) / (1.0 - w5)) * lnhalf
                - 16.0 * dilog((1.0 + w5) / (2.0 * w5)) * lnhalf - 8.0 * dilog(-w7inv) * lnhalf - 8.0 * dilog(w7inv) * lnhalf + 8.0 * dilog((-1.0 + w7) / (2.0 * w7)) * lnhalf + 8.0 * dilog((1.0 - w7) / (1.0 + w7)) * lnhalf
                + 8.0 * dilog((1.0 + w7) / (1.0 - w7)) * lnhalf + 8.0 * dilog((1.0 + w7) / (2.0 * w7)) * lnhalf + (32.0 * power_of<3>(ln2)) / 3.0 - (8.0 * pisqu * ln4) / 3.0 + 8.0 * power_of<2>(lnhalf) * log(1.0 + w4inv) - (8.0 * pisqu * log(-2.0 / w4)) / 3.0
                - (8.0 * power_of<3>(log(-2.0 / w4))) / 3.0 - (8.0 * pisqu * log(2.0 / w4)) / 3.0 - (8.0 * power_of<3>(log(2.0 / w4))) / 3.0 - 8.0 * power_of<2>(lnhalf) * log((-1.0 + w4) / (2.0 * w4)) + 8.0 * power_of<2>(lnhalf) * log((-1.0 + w4) / w4) + 8.0 * power_of<2>(lnhalf) * log(w4 / (-1.0 + w4))
                - 8.0 * power_of<2>(lnhalf) * log((2.0 * w4) / (-1.0 + w4)) + (8.0 * pisqu * log((2.0 * (-1.0 + w4)) / (1.0 + w4))) / 3.0 + (8.0 * power_of<3>(log((2.0 * (-1.0 + w4)) / (1.0 + w4)))) / 3.0 + 8.0 * power_of<2>(lnhalf) * log(w4 / (1.0 + w4)) - 8.0 * power_of<2>(lnhalf) * log((2.0 * w4) / (1.0 + w4))
                + (8.0 * pisqu * log((2.0 * (1.0 + w4)) / (-1.0 + w4))) / 3.0 + (8.0 * power_of<3>(log((2.0 * (1.0 + w4)) / (-1.0 + w4)))) / 3.0 - 8.0 * power_of<2>(lnhalf) * log((1.0 + w4) / (2.0 * w4)) + 8.0 * power_of<2>(lnhalf) * log(1.0 + w5inv) - (8.0 * pisqu * log(-2.0 / w5)) / 3.0
                - (8.0 * power_of<3>(log(-2.0 / w5))) / 3.0 - (8.0 * pisqu * log(2.0 / w5)) / 3.0 - (8.0 * power_of<3>(log(2.0 / w5))) / 3.0 - 8.0 * power_of<2>(lnhalf) * log((-1.0 + w5) / (2.0 * w5)) + 8.0 * power_of<2>(lnhalf) * log((-1.0 + w5) / w5) + 8.0 * power_of<2>(lnhalf) * log(w5 / (-1.0 + w5))
                - 8.0 * power_of<2>(lnhalf) * log((2.0 * w5) / (-1.0 + w5)) + (8.0 * pisqu * log((2.0 * (-1.0 + w5)) / (1.0 + w5))) / 3.0 + (8.0 * power_of<3>(log((2.0 * (-1.0 + w5)) / (1.0 + w5)))) / 3.0 + 8.0 * power_of<2>(lnhalf) * log(w5 / (1.0 + w5)) - 8.0 * power_of<2>(lnhalf) * log((2.0 * w5) / (1.0 + w5))
                + (8.0 * pisqu * log((2.0 * (1.0 + w5)) / (-1.0 + w5))) / 3.0 + (8.0 * power_of<3>(log((2.0 * (1.0 + w5)) / (-1.0 + w5)))) / 3.0 - 8.0 * power_of<2>(lnhalf) * log((1.0 + w5) / (2.0 * w5)) - 4.0 * power_of<2>(lnhalf) * log(1.0 + w7inv) + (4.0 * pisqu * log(-2.0 / w7)) / 3.0
                + (4.0 * power_of<3>(log(-2.0 / w7))) / 3.0 + (4.0 * pisqu * log(2.0 / w7)) / 3.0 + (4.0 * power_of<3>(log(2.0 / w7))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log((-1.0 + w7) / (2.0 * w7)) - 4.0 * power_of<2>(lnhalf) * log((-1.0 + w7) / w7) - 4.0 * power_of<2>(lnhalf) * log(w7 / (-1.0 + w7))
                + 4.0 * power_of<2>(lnhalf) * log((2.0 * w7) / (-1.0 + w7)) - (4.0 * pisqu * log((2.0 * (-1.0 + w7)) / (1.0 + w7))) / 3.0 - (4.0 * power_of<3>(log((2.0 * (-1.0 + w7)) / (1.0 + w7)))) / 3.0 - 4.0 * power_of<2>(lnhalf) * log(w7 / (1.0 + w7)) + 4.0 * power_of<2>(lnhalf) * log((2.0 * w7) / (1.0 + w7))
                - (4.0 * pisqu * log((2.0 * (1.0 + w7)) / (-1.0 + w7))) / 3.0 - (4.0 * power_of<3>(log((2.0 * (1.0 + w7)) / (-1.0 + w7)))) / 3.0 + 4.0 * power_of<2>(lnhalf) * log((1.0 + w7) / (2.0 * w7)) + (8.0 * pisqu * log(1.0 - 1.0i * wx3)) / 3.0 + (8.0 * power_of<3>(log(1.0 - 1.0i * wx3))) / 3.0
                + (8.0 * pisqu * log(1.0 + 1.0i * wx3)) / 3.0 + (8.0 * power_of<3>(log(1.0 + 1.0i * wx3))) / 3.0 + 16.0 * dilog((1.0i + wx3) / (1.0i - wx3)) * log(-1.0i / (-1.0i + wx3)) + 16.0 * dilog((1.0i + wx3) / (2.0 * wx3)) * log(-1.0i / (-1.0i + wx3)) - 8.0 * lnhalf * power_of<2>(log(-1.0i / (-1.0i + wx3)))
                - 8.0 * ln2 * power_of<2>(log(-1.0i / (-1.0i + wx3))) + 8.0 * power_of<2>(log(-1.0i / (-1.0i + wx3))) * log((2.0 * wx3) / (-1.0i + wx3)) + 8.0 * power_of<2>(log(-1.0i / (-1.0i + wx3))) * log((-1.0i + wx3) / (2.0 * wx3)) + 16.0 * dilog((-1.0i + wx3) / (2.0 * wx3)) * log(1.0i / (1.0i + wx3))
                + 16.0 * dilog((1.0i - wx3) / (1.0i + wx3)) * log(1.0i / (1.0i + wx3)) - 8.0 * lnhalf * power_of<2>(log(1.0i / (1.0i + wx3))) - 8.0 * ln2 * power_of<2>(log(1.0i / (1.0i + wx3))) + 8.0 * power_of<2>(log(1.0i / (1.0i + wx3))) * log((2.0 * wx3) / (1.0i + wx3))
                + 8.0 * power_of<2>(log(1.0i / (1.0i + wx3))) * log((1.0i + wx3) / (2.0 * wx3)) - (8.0 * pisqu * log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx3))) / wx3)) / 3.0 - (8.0 * power_of<3>(log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx3))) / wx3))) / 3.0 - (8.0 * pisqu * log(((1.0i / 2.0) * (1.0 + power_of<2>(wx3))) / wx3)) / 3.0
                - (8.0 * power_of<3>(log(((1.0i / 2.0) * (1.0 + power_of<2>(wx3))) / wx3))) / 3.0 + (8.0 * pisqu * log(1.0 - 1.0i * wx4)) / 3.0 + (8.0 * power_of<3>(log(1.0 - 1.0i * wx4))) / 3.0 + (8.0 * pisqu * log(1.0 + 1.0i * wx4)) / 3.0 + (8.0 * power_of<3>(log(1.0 + 1.0i * wx4))) / 3.0
                + 16.0 * dilog((1.0i + wx4) / (1.0i - wx4)) * log(-1.0i / (-1.0i + wx4)) + 16.0 * dilog((1.0i + wx4) / (2.0 * wx4)) * log(-1.0i / (-1.0i + wx4)) - 8.0 * lnhalf * power_of<2>(log(-1.0i / (-1.0i + wx4))) - 8.0 * ln2 * power_of<2>(log(-1.0i / (-1.0i + wx4)))
                + 8.0 * power_of<2>(log(-1.0i / (-1.0i + wx4))) * log((2.0 * wx4) / (-1.0i + wx4)) + 8.0 * power_of<2>(log(-1.0i / (-1.0i + wx4))) * log((-1.0i + wx4) / (2.0 * wx4)) + dilog(-1.0) * (-16.0 * log(-1.0i / (-1.0i + wx3)) - 16.0 * log(1.0i / (1.0i + wx3)) - 16.0 * log(-1.0i / (-1.0i + wx4))
                    - 16.0 * log(1.0i / (1.0i + wx4))) + 16.0 * dilog((-1.0i + wx4) / (2.0 * wx4)) * log(1.0i / (1.0i + wx4)) + 16.0 * dilog((1.0i - wx4) / (1.0i + wx4)) * log(1.0i / (1.0i + wx4)) - 8.0 * lnhalf * power_of<2>(log(1.0i / (1.0i + wx4))) - 8.0 * ln2 * power_of<2>(log(1.0i / (1.0i + wx4)))
                + 8.0 * power_of<2>(log(1.0i / (1.0i + wx4))) * log((2.0 * wx4) / (1.0i + wx4)) + 8.0 * power_of<2>(log(1.0i / (1.0i + wx4))) * log((1.0i + wx4) / (2.0 * wx4)) - (8.0 * pisqu * log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx4))) / wx4)) / 3.0 - (8.0 * power_of<3>(log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx4))) / wx4))) / 3.0
                - (8.0 * pisqu * log(((1.0i / 2.0) * (1.0 + power_of<2>(wx4))) / wx4)) / 3.0 - (8.0 * power_of<3>(log(((1.0i / 2.0) * (1.0 + power_of<2>(wx4))) / wx4))) / 3.0 - (4.0 * pisqu * log(wx3 / (1.0i - xd))) / 3.0 - (4.0 * power_of<3>(log(wx3 / (1.0i - xd)))) / 3.0 - (4.0 * power_of<3>(log((1.0i + wx3) / (1.0i - xd)))) / 3.0
                - (4.0 * pisqu * log(wx4 / (1.0i - xd))) / 3.0 - (4.0 * power_of<3>(log(wx4 / (1.0i - xd)))) / 3.0 - (4.0 * power_of<3>(log((1.0i + wx4) / (1.0i - xd)))) / 3.0 - 8.0 * dilog(((-1.0i + wx3) * (-1.0i + xd)) / ((1.0i + wx3) * (1.0i + xd))) * log((1.0i - xd) / (1.0i + wx3))
                - 8.0 * dilog(((-1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 + xd))) * log((1.0i - xd) / (1.0i + wx3)) - 8.0 * dilog(((-1.0i + wx4) * (-1.0i + xd)) / ((1.0i + wx4) * (1.0i + xd))) * log((1.0i - xd) / (1.0i + wx4))
                - 8.0 * dilog(((-1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 + xd))) * log((1.0i - xd) / (1.0i + wx4)) + (4.0 * pisqu * log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx3))) / (wx3 - xd))) / 3.0 + (4.0 * power_of<3>(log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx3))) / (wx3 - xd)))) / 3.0
                + (4.0 * pisqu * log(((1.0i / 2.0) * (1.0 + power_of<2>(wx3))) / (wx3 - xd))) / 3.0 + (4.0 * power_of<3>(log(((1.0i / 2.0) * (1.0 + power_of<2>(wx3))) / (wx3 - xd)))) / 3.0 + (4.0 * pisqu * log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx4))) / (wx4 - xd))) / 3.0
                + (4.0 * power_of<3>(log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx4))) / (wx4 - xd)))) / 3.0 + (4.0 * pisqu * log(((1.0i / 2.0) * (1.0 + power_of<2>(wx4))) / (wx4 - xd))) / 3.0 + (4.0 * power_of<3>(log(((1.0i / 2.0) * (1.0 + power_of<2>(wx4))) / (wx4 - xd)))) / 3.0 - 32.0 * dilog(1.0 - 1.0i * xd) * log(1.0 - 1.0i * xd)
                + dilog((-1.0i) * xd) * (16.0 * log(1.0 - 1.0i * xd) - 16.0 * log(1.0 + 1.0i * xd)) - 32.0 * dilog(1.0 + 1.0i * xd) * log(1.0 + 1.0i * xd) + dilog(-1.0i / wx3) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd))
                + dilog(1.0i / wx3) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd)) + dilog(-1.0i / (-1.0i + wx3)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd)) + dilog(1.0i / (1.0i + wx3)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd))
                + dilog(-1.0i / wx4) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd)) + dilog(1.0i / wx4) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd)) + dilog(-1.0i / (-1.0i + wx4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd))
                + dilog(1.0i / (1.0i + wx4)) * (8.0 * log(1.0 - 1.0i * xd) + 8.0 * log(1.0 + 1.0i * xd)) + dilog(1.0i * xd) * (-16.0 * log(1.0 - 1.0i * xd) + 16.0 * log(1.0 + 1.0i * xd)) + (4.0 * pisqu * log((1.0i * wx3) / ((-1.0i + wx3) * xd))) / 3.0 + (4.0 * power_of<3>(log((1.0i * wx3) / ((-1.0i + wx3) * xd)))) / 3.0
                + (4.0 * pisqu * log((1.0i * wx4) / ((-1.0i + wx4) * xd))) / 3.0 + (4.0 * power_of<3>(log((1.0i * wx4) / ((-1.0i + wx4) * xd)))) / 3.0 - (4.0 * pisqu * log(wx3 / (-1.0i + xd))) / 3.0 - (4.0 * power_of<3>(log(wx3 / (-1.0i + xd)))) / 3.0 - (4.0 * power_of<3>(log((-1.0i + wx3) / (-1.0i + xd)))) / 3.0
                - (4.0 * pisqu * log(wx4 / (-1.0i + xd))) / 3.0 - (4.0 * power_of<3>(log(wx4 / (-1.0i + xd)))) / 3.0 - (4.0 * power_of<3>(log((-1.0i + wx4) / (-1.0i + xd)))) / 3.0 - 8.0 * dilog(((1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 - xd))) * log((-1.0i + xd) / (-1.0i + wx3))
                - 8.0 * dilog(((1.0i + wx3) * (-1.0i + xd)) / ((-1.0i + wx3) * (1.0i + xd))) * log((-1.0i + xd) / (-1.0i + wx3)) - 8.0 * dilog(((1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 - xd))) * log((-1.0i + xd) / (-1.0i + wx4))
                - 8.0 * dilog(((1.0i + wx4) * (-1.0i + xd)) / ((-1.0i + wx4) * (1.0i + xd))) * log((-1.0i + xd) / (-1.0i + wx4)) + dilog((-1.0i + xd) / (1.0i + xd)) * (8.0 * log((1.0i - xd) / (1.0i + wx3)) + 8.0 * log((1.0i - xd) / (1.0i + wx4)) + 8.0 * log((-1.0i + xd) / (-1.0i + wx3))
                    + 8.0 * log((-1.0i + xd) / (-1.0i + wx4))) - (4.0 * power_of<3>(log((1.0i - wx3) / (1.0i + xd)))) / 3.0 - (4.0 * pisqu * log(-(wx3 / (1.0i + xd)))) / 3.0 - (4.0 * power_of<3>(log(-(wx3 / (1.0i + xd))))) / 3.0 - (4.0 * pisqu * log(wx3 / (1.0i + xd))) / 3.0 - (4.0 * power_of<3>(log(wx3 / (1.0i + xd)))) / 3.0
                - (4.0 * power_of<3>(log((1.0i + wx3) / (1.0i + xd)))) / 3.0 - (4.0 * power_of<3>(log((1.0i - wx4) / (1.0i + xd)))) / 3.0 - (4.0 * pisqu * log(-(wx4 / (1.0i + xd)))) / 3.0 - (4.0 * power_of<3>(log(-(wx4 / (1.0i + xd))))) / 3.0 - (4.0 * pisqu * log(wx4 / (1.0i + xd))) / 3.0 - (4.0 * power_of<3>(log(wx4 / (1.0i + xd)))) / 3.0
                - (4.0 * power_of<3>(log((1.0i + wx4) / (1.0i + xd)))) / 3.0 - 8.0 * dilog(((1.0i + wx3) * (1.0i + xd)) / ((-1.0i + wx3) * (-1.0i + xd))) * log((1.0i + xd) / (1.0i - wx3)) - 8.0 * dilog(((1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 + xd))) * log((1.0i + xd) / (1.0i - wx3))
                - 8.0 * dilog(((-1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 - xd))) * log((1.0i + xd) / (1.0i + wx3)) - 8.0 * dilog(((-1.0i + wx3) * (1.0i + xd)) / ((1.0i + wx3) * (-1.0i + xd))) * log((1.0i + xd) / (1.0i + wx3))
                - 8.0 * dilog(((1.0i + wx4) * (1.0i + xd)) / ((-1.0i + wx4) * (-1.0i + xd))) * log((1.0i + xd) / (1.0i - wx4)) - 8.0 * dilog(((1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 + xd))) * log((1.0i + xd) / (1.0i - wx4))
                - 8.0 * dilog(((-1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 - xd))) * log((1.0i + xd) / (1.0i + wx4)) - 8.0 * dilog(((-1.0i + wx4) * (1.0i + xd)) / ((1.0i + wx4) * (-1.0i + xd))) * log((1.0i + xd) / (1.0i + wx4))
                + dilog((1.0i + xd) / (-1.0i + xd)) * (8.0 * log((1.0i + xd) / (1.0i - wx3)) + 8.0 * log((1.0i + xd) / (1.0i + wx3)) + 8.0 * log((1.0i + xd) / (1.0i - wx4)) + 8.0 * log((1.0i + xd) / (1.0i + wx4)))
                + log((-2.0 * 1.0i) / (-1.0i + xd)) * (4.0 * power_of<2>(log((1.0i + xd) / (1.0i + wx3))) + 4.0 * power_of<2>(log((1.0i + xd) / (1.0i - wx4))) + 4.0 * power_of<2>(log((1.0i + xd) / (1.0i + wx4)))) + (4.0 * pisqu * log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx3))) / (wx3 + xd))) / 3.0
                + (4.0 * power_of<3>(log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx3))) / (wx3 + xd)))) / 3.0 + (4.0 * pisqu * log(((1.0i / 2.0) * (1.0 + power_of<2>(wx3))) / (wx3 + xd))) / 3.0 + (4.0 * power_of<3>(log(((1.0i / 2.0) * (1.0 + power_of<2>(wx3))) / (wx3 + xd)))) / 3.0
                + 8.0 * dilog((wx3 + xd) / (-1.0i + wx3)) * log((wx3 + xd) / wx3) + 8.0 * dilog((wx3 + xd) / (1.0i + wx3)) * log((wx3 + xd) / wx3) - 8.0 * dilog(((-1.0i) * (wx3 + xd)) / ((-1.0i + wx3) * xd)) * log((wx3 + xd) / wx3)
                - 8.0 * dilog((1.0i * (wx3 + xd)) / ((1.0i + wx3) * xd)) * log((wx3 + xd) / wx3) + 8.0 * dilog((wx3 + xd) / (-1.0i + xd)) * log((wx3 + xd) / wx3) - 8.0 * dilog(((-1.0i) * (wx3 + xd)) / (wx3 * (-1.0i + xd))) * log((wx3 + xd) / wx3)
                + 8.0 * dilog((wx3 + xd) / (1.0i + xd)) * log((wx3 + xd) / wx3) - 8.0 * dilog((1.0i * (wx3 + xd)) / (wx3 * (1.0i + xd))) * log((wx3 + xd) / wx3)
                + (-4.0 * log(((1.0i + wx3) * xd) / (wx3 * (-1.0i + xd))) - 4.0 * log((wx3 * (-1.0i + xd)) / ((1.0i + wx3) * xd)) - 4.0 * log(((-1.0i + wx3) * xd) / (wx3 * (1.0i + xd))) - 4.0 * log((wx3 * (1.0i + xd)) / ((-1.0i + wx3) * xd))) * power_of<2>(log((wx3 + xd) / wx3))
                + log((1.0i + wx3) / (1.0i - xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log((wx3 + xd) / wx3))) + log((1.0i - wx3) / (1.0i + xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log((wx3 + xd) / wx3))) + (4.0 * pisqu * log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx4))) / (wx4 + xd))) / 3.0
                + (4.0 * power_of<3>(log((((-1.0 / 2.0) * 1.0i) * (1.0 + power_of<2>(wx4))) / (wx4 + xd)))) / 3.0 + (4.0 * pisqu * log(((1.0i / 2.0) * (1.0 + power_of<2>(wx4))) / (wx4 + xd))) / 3.0 + (4.0 * power_of<3>(log(((1.0i / 2.0) * (1.0 + power_of<2>(wx4))) / (wx4 + xd)))) / 3.0
                + 8.0 * dilog((wx4 + xd) / (-1.0i + wx4)) * log((wx4 + xd) / wx4) + 8.0 * dilog((wx4 + xd) / (1.0i + wx4)) * log((wx4 + xd) / wx4) - 8.0 * dilog(((-1.0i) * (wx4 + xd)) / ((-1.0i + wx4) * xd)) * log((wx4 + xd) / wx4)
                - 8.0 * dilog((1.0i * (wx4 + xd)) / ((1.0i + wx4) * xd)) * log((wx4 + xd) / wx4) + 8.0 * dilog((wx4 + xd) / (-1.0i + xd)) * log((wx4 + xd) / wx4) - 8.0 * dilog(((-1.0i) * (wx4 + xd)) / (wx4 * (-1.0i + xd))) * log((wx4 + xd) / wx4)
                + 8.0 * dilog((wx4 + xd) / (1.0i + xd)) * log((wx4 + xd) / wx4) - 8.0 * dilog((1.0i * (wx4 + xd)) / (wx4 * (1.0i + xd))) * log((wx4 + xd) / wx4)
                + (-4.0 * log(((1.0i + wx4) * xd) / (wx4 * (-1.0i + xd))) - 4.0 * log((wx4 * (-1.0i + xd)) / ((1.0i + wx4) * xd)) - 4.0 * log(((-1.0i + wx4) * xd) / (wx4 * (1.0i + xd))) - 4.0 * log((wx4 * (1.0i + xd)) / ((-1.0i + wx4) * xd))) * power_of<2>(log((wx4 + xd) / wx4))
                + log((1.0i + wx4) / (1.0i - xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log((wx4 + xd) / wx4))) + log((1.0i - wx4) / (1.0i + xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log((wx4 + xd) / wx4)))
                + dilog(1.0 / 2.0 + (1.0i / 2.0) * wx3) * (-8.0 * log((wx3 + xd) / wx3) - 8.0 * log(1.0 - xd / wx3)) + dilog(((-1.0 / 2.0) * 1.0i) * (1.0i + wx3)) * (-8.0 * log((wx3 + xd) / wx3) - 8.0 * log(1.0 - xd / wx3))
                + 8.0 * dilog((wx3 - xd) / (-1.0i + wx3)) * log(1.0 - xd / wx3) + 8.0 * dilog((wx3 - xd) / (1.0i + wx3)) * log(1.0 - xd / wx3) - 8.0 * dilog((1.0i * (wx3 - xd)) / ((-1.0i + wx3) * xd)) * log(1.0 - xd / wx3)
                - 8.0 * dilog(((-1.0i) * (wx3 - xd)) / ((1.0i + wx3) * xd)) * log(1.0 - xd / wx3) - 8.0 * dilog(((-1.0i) * (wx3 - xd)) / (wx3 * (-1.0i + xd))) * log(1.0 - xd / wx3) - 8.0 * dilog((1.0i * (wx3 - xd)) / (wx3 * (1.0i + xd))) * log(1.0 - xd / wx3)
                + 8.0 * dilog((-wx3 + xd) / (-1.0i + xd)) * log(1.0 - xd / wx3) + 8.0 * dilog((-wx3 + xd) / (1.0i + xd)) * log(1.0 - xd / wx3) + (-4.0 * log(((-1.0i + wx3) * xd) / (wx3 * (-1.0i + xd))) - 4.0 * log((wx3 * (-1.0i + xd)) / ((-1.0i + wx3) * xd))
                    - 4.0 * log(((1.0i + wx3) * xd) / (wx3 * (1.0i + xd))) - 4.0 * log((wx3 * (1.0i + xd)) / ((1.0i + wx3) * xd))) * power_of<2>(log(1.0 - xd / wx3)) + log((-1.0i + wx3) / (-1.0i + xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log(1.0 - xd / wx3)))
                + log((1.0i + wx3) / (1.0i + xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log(1.0 - xd / wx3))) + (4.0 * pisqu * log((1.0i * wx3) / (1.0i * xd - wx3 * xd))) / 3.0 + (4.0 * power_of<3>(log((1.0i * wx3) / (1.0i * xd - wx3 * xd)))) / 3.0 + (4.0 * pisqu * log(((-1.0i) * wx3) / (1.0i * xd + wx3 * xd))) / 3.0
                + (4.0 * power_of<3>(log(((-1.0i) * wx3) / (1.0i * xd + wx3 * xd)))) / 3.0 + (4.0 * pisqu * log((1.0i * wx3) / (1.0i * xd + wx3 * xd))) / 3.0 + (4.0 * power_of<3>(log((1.0i * wx3) / (1.0i * xd + wx3 * xd)))) / 3.0 + dilog(1.0 / 2.0 + (1.0i / 2.0) * wx4) * (-8.0 * log((wx4 + xd) / wx4) - 8.0 * log(1.0 - xd / wx4))
                + dilog(((-1.0 / 2.0) * 1.0i) * (1.0i + wx4)) * (-8.0 * log((wx4 + xd) / wx4) - 8.0 * log(1.0 - xd / wx4)) + 8.0 * dilog((wx4 - xd) / (-1.0i + wx4)) * log(1.0 - xd / wx4) + 8.0 * dilog((wx4 - xd) / (1.0i + wx4)) * log(1.0 - xd / wx4)
                - 8.0 * dilog((1.0i * (wx4 - xd)) / ((-1.0i + wx4) * xd)) * log(1.0 - xd / wx4) - 8.0 * dilog(((-1.0i) * (wx4 - xd)) / ((1.0i + wx4) * xd)) * log(1.0 - xd / wx4) - 8.0 * dilog(((-1.0i) * (wx4 - xd)) / (wx4 * (-1.0i + xd))) * log(1.0 - xd / wx4)
                - 8.0 * dilog((1.0i * (wx4 - xd)) / (wx4 * (1.0i + xd))) * log(1.0 - xd / wx4) + 8.0 * dilog((-wx4 + xd) / (-1.0i + xd)) * log(1.0 - xd / wx4) + 8.0 * dilog((-wx4 + xd) / (1.0i + xd)) * log(1.0 - xd / wx4)
                + (-4.0 * log(((-1.0i + wx4) * xd) / (wx4 * (-1.0i + xd))) - 4.0 * log((wx4 * (-1.0i + xd)) / ((-1.0i + wx4) * xd)) - 4.0 * log(((1.0i + wx4) * xd) / (wx4 * (1.0i + xd))) - 4.0 * log((wx4 * (1.0i + xd)) / ((1.0i + wx4) * xd))) * power_of<2>(log(1.0 - xd / wx4))
                + dilog(1.0 / 2.0 + (1.0i / 2.0) * xd) * (8.0 * log((1.0i - xd) / (1.0i + wx3)) + 8.0 * log((1.0i - xd) / (1.0i + wx4)) + 8.0 * log((-1.0i + xd) / (-1.0i + wx3)) + 8.0 * log((-1.0i + xd) / (-1.0i + wx4)) + 8.0 * log((wx3 + xd) / wx3) + 8.0 * log((wx4 + xd) / wx4)
                    + 8.0 * log(1.0 - xd / wx3) + 8.0 * log(1.0 - xd / wx4)) + dilog(((-1.0 / 2.0) * 1.0i) * (1.0i + xd)) * (8.0 * log((1.0i + xd) / (1.0i - wx3)) + 8.0 * log((1.0i + xd) / (1.0i + wx3)) + 8.0 * log((1.0i + xd) / (1.0i - wx4)) + 8.0 * log((1.0i + xd) / (1.0i + wx4))
                    + 8.0 * log((wx3 + xd) / wx3) + 8.0 * log((wx4 + xd) / wx4) + 8.0 * log(1.0 - xd / wx3) + 8.0 * log(1.0 - xd / wx4)) + log(1.0 / 2.0 - (1.0i / 2.0) * xd) * (4.0 * power_of<2>(log((1.0i - xd) / (1.0i + wx4))) + 4.0 * power_of<2>(log((-1.0i + xd) / (-1.0i + wx3)))
                    + 4.0 * power_of<2>(log((-1.0i + xd) / (-1.0i + wx4))) + 8.0 * log((1.0i - xd) / (1.0i + wx4)) * log((wx4 + xd) / wx4) + 8.0 * log((-1.0i + xd) / (-1.0i + wx3)) * log(1.0 - xd / wx3) + 8.0 * log((-1.0i + xd) / (-1.0i + wx4)) * log(1.0 - xd / wx4))
                + log(1.0 / 2.0 + (1.0i / 2.0) * xd) * (4.0 * power_of<2>(log((1.0i + xd) / (1.0i - wx3))) + 4.0 * power_of<2>(log((1.0i + xd) / (1.0i + wx3))) + 4.0 * power_of<2>(log((1.0i + xd) / (1.0i - wx4))) + 4.0 * power_of<2>(log((1.0i + xd) / (1.0i + wx4))) + 8.0 * log((1.0i + xd) / (1.0i - wx3)) * log((wx3 + xd) / wx3)
                    + 8.0 * log((1.0i + xd) / (1.0i - wx4)) * log((wx4 + xd) / wx4) + 8.0 * log((1.0i + xd) / (1.0i + wx3)) * log(1.0 - xd / wx3) + 8.0 * log((1.0i + xd) / (1.0i + wx4)) * log(1.0 - xd / wx4))
                + log((-1.0i + wx4) / (-1.0i + xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log(1.0 - xd / wx4))) + log((1.0i + wx4) / (1.0i + xd)) * ((-4.0 * pisqu) / 3.0 + 4.0 * power_of<2>(log(1.0 - xd / wx4))) + (4.0 * pisqu * log((1.0i * wx4) / (1.0i * xd - wx4 * xd))) / 3.0
                + (4.0 * power_of<3>(log((1.0i * wx4) / (1.0i * xd - wx4 * xd)))) / 3.0 + (4.0 * pisqu * log(((-1.0i) * wx4) / (1.0i * xd + wx4 * xd))) / 3.0 + (4.0 * power_of<3>(log(((-1.0i) * wx4) / (1.0i * xd + wx4 * xd)))) / 3.0 + (4.0 * pisqu * log((1.0i * wx4) / (1.0i * xd + wx4 * xd))) / 3.0
                + (4.0 * power_of<3>(log((1.0i * wx4) / (1.0i * xd + wx4 * xd)))) / 3.0 + 8.0 * dilog((-1.0 / 2.0) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd))) * log((1.0 - yd) / 2.0) + 8.0 * dilog((-1.0 / 2.0) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd))) * log((1.0 - yd) / 2.0)
                - 4.0 * dilog((-1.0 / 2.0) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) * log((1.0 - yd) / 2.0) + 8.0 * dilog(((-1.0 + w4) * (-1.0 + yd)) / ((1.0 + w4) * (1.0 + yd))) * log((1.0 - yd) / 2.0)
                + 8.0 * dilog(((1.0 + w4) * (-1.0 + yd)) / ((-1.0 + w4) * (1.0 + yd))) * log((1.0 - yd) / 2.0) + 8.0 * dilog(((-1.0 + w5) * (-1.0 + yd)) / ((1.0 + w5) * (1.0 + yd))) * log((1.0 - yd) / 2.0)
                + 8.0 * dilog(((1.0 + w5) * (-1.0 + yd)) / ((-1.0 + w5) * (1.0 + yd))) * log((1.0 - yd) / 2.0) - 4.0 * dilog(((-1.0 + w7) * (-1.0 + yd)) / ((1.0 + w7) * (1.0 + yd))) * log((1.0 - yd) / 2.0)
                - 4.0 * dilog(((1.0 + w7) * (-1.0 + yd)) / ((-1.0 + w7) * (1.0 + yd))) * log((1.0 - yd) / 2.0) - 8.0 * dilog((-1.0 + yd) / (-w4 + yd)) * log((1.0 - yd) / 2.0) - 8.0 * dilog((-1.0 + yd) / (w4 + yd)) * log((1.0 - yd) / 2.0)
                + 8.0 * dilog((-1.0 / 2.0) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) * log((1.0 - yd) / 2.0) - 8.0 * dilog((-1.0 + yd) / (-w5 + yd)) * log((1.0 - yd) / 2.0) - 8.0 * dilog((-1.0 + yd) / (w5 + yd)) * log((1.0 - yd) / 2.0)
                + 8.0 * dilog((-1.0 / 2.0) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) * log((1.0 - yd) / 2.0) + 4.0 * dilog((-1.0 + yd) / (-w7 + yd)) * log((1.0 - yd) / 2.0) + 4.0 * dilog((-1.0 + yd) / (w7 + yd)) * log((1.0 - yd) / 2.0)
                - 4.0 * dilog((-1.0 / 2.0) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) * log((1.0 - yd) / 2.0) + dilog((-1.0 + yd) / (-1.0 + w7)) * (4.0 * log((1.0 - yd) / 2.0) - 4.0 * ln1myd) - 8.0 * dilog((w4 * (-1.0 + yd)) / ((-1.0 + w4) * yd)) * ln1myd
                + 8.0 * dilog((w4 * (-1.0 + yd)) / ((1.0 + w4) * yd)) * ln1myd + 8.0 * dilog((w5 * (-1.0 + yd)) / ((-1.0 + w5) * yd)) * ln1myd - 8.0 * dilog((w5 * (-1.0 + yd)) / ((1.0 + w5) * yd)) * ln1myd
                + 4.0 * dilog((w7 * (-1.0 + yd)) / ((-1.0 + w7) * yd)) * ln1myd - 4.0 * dilog((w7 * (-1.0 + yd)) / ((1.0 + w7) * yd)) * ln1myd + dilog((1.0 - yd) / (1.0 + w5)) * (-8.0 * log((1.0 - yd) / 2.0) + 8.0 * ln1myd)
                + dilog((-1.0 + yd) / (-1.0 + w4)) * (-8.0 * log((1.0 - yd) / 2.0) + 8.0 * ln1myd) + (4.0 * pisqu * log(-2.0 / (w4 - yd))) / 3.0 + (4.0 * power_of<3>(log(-2.0 / (w4 - yd)))) / 3.0 + (4.0 * pisqu * log(2.0 / (w4 - yd))) / 3.0 + (4.0 * power_of<3>(log(2.0 / (w4 - yd)))) / 3.0
                + (4.0 * pisqu * log(-2.0 / (w5 - yd))) / 3.0 + (4.0 * power_of<3>(log(-2.0 / (w5 - yd)))) / 3.0 + (4.0 * pisqu * log(2.0 / (w5 - yd))) / 3.0 + (4.0 * power_of<3>(log(2.0 / (w5 - yd)))) / 3.0 - (2.0 * pisqu * log(-2.0 / (w7 - yd))) / 3.0 - (2.0 * power_of<3>(log(-2.0 / (w7 - yd)))) / 3.0
                - (2.0 * pisqu * log(2.0 / (w7 - yd))) / 3.0 - (2.0 * power_of<3>(log(2.0 / (w7 - yd)))) / 3.0 - (4.0 * pisqu * log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd)))) / 3.0 - (4.0 * power_of<3>(log((-2.0 * (-1.0 + w4)) / ((1.0 + w4) * (-1.0 + yd))))) / 3.0
                - (4.0 * pisqu * log((-2.0 * (1.0 + w4)) / ((-1.0 + w4) * (-1.0 + yd)))) / 3.0 - (4.0 * power_of<3>(log((-2.0 * (1.0 + w4)) / ((-1.0 + w4) * (-1.0 + yd))))) / 3.0 - (4.0 * pisqu * log((-2.0 * (-1.0 + w5)) / ((1.0 + w5) * (-1.0 + yd)))) / 3.0
                - (4.0 * power_of<3>(log((-2.0 * (-1.0 + w5)) / ((1.0 + w5) * (-1.0 + yd))))) / 3.0 - (4.0 * pisqu * log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd)))) / 3.0 - (4.0 * power_of<3>(log((-2.0 * (1.0 + w5)) / ((-1.0 + w5) * (-1.0 + yd))))) / 3.0
                + (2.0 * pisqu * log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd)))) / 3.0 + (2.0 * power_of<3>(log((-2.0 * (-1.0 + w7)) / ((1.0 + w7) * (-1.0 + yd))))) / 3.0 + (2.0 * pisqu * log((-2.0 * (1.0 + w7)) / ((-1.0 + w7) * (-1.0 + yd)))) / 3.0
                + (2.0 * power_of<3>(log((-2.0 * (1.0 + w7)) / ((-1.0 + w7) * (-1.0 + yd))))) / 3.0 + (4.0 * pisqu * log(w4 / ((-1.0 + w4) * yd))) / 3.0 + (4.0 * power_of<3>(log(w4 / ((-1.0 + w4) * yd)))) / 3.0 - (4.0 * pisqu * log(w5 / ((-1.0 + w5) * yd))) / 3.0 - (4.0 * power_of<3>(log(w5 / ((-1.0 + w5) * yd)))) / 3.0
                - (2.0 * pisqu * log(w7 / ((-1.0 + w7) * yd))) / 3.0 - (2.0 * power_of<3>(log(w7 / ((-1.0 + w7) * yd)))) / 3.0 - (4.0 * pisqu * log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd)))) / 3.0 - (4.0 * power_of<3>(log((2.0 * (-1.0 + w4)) / ((1.0 + w4) * (1.0 + yd))))) / 3.0
                - (4.0 * pisqu * log((2.0 * (1.0 + w4)) / ((-1.0 + w4) * (1.0 + yd)))) / 3.0 - (4.0 * power_of<3>(log((2.0 * (1.0 + w4)) / ((-1.0 + w4) * (1.0 + yd))))) / 3.0 - (4.0 * pisqu * log((2.0 * (-1.0 + w5)) / ((1.0 + w5) * (1.0 + yd)))) / 3.0
                - (4.0 * power_of<3>(log((2.0 * (-1.0 + w5)) / ((1.0 + w5) * (1.0 + yd))))) / 3.0 - (4.0 * pisqu * log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd)))) / 3.0 - (4.0 * power_of<3>(log((2.0 * (1.0 + w5)) / ((-1.0 + w5) * (1.0 + yd))))) / 3.0
                + (2.0 * pisqu * log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd)))) / 3.0 + (2.0 * power_of<3>(log((2.0 * (-1.0 + w7)) / ((1.0 + w7) * (1.0 + yd))))) / 3.0 + (2.0 * pisqu * log((2.0 * (1.0 + w7)) / ((-1.0 + w7) * (1.0 + yd)))) / 3.0
                + (2.0 * power_of<3>(log((2.0 * (1.0 + w7)) / ((-1.0 + w7) * (1.0 + yd))))) / 3.0 + dilog((1.0 + yd) / (1.0 + w4)) * (-8.0 * ln1myd - 8.0 * log((1.0 + yd) / 2.0)) + dilog((1.0 + yd) / (1.0 - w5)) * (-8.0 * ln1myd - 8.0 * log((1.0 + yd) / 2.0))
                - 8.0 * dilog(-((1.0 + yd) / (w4 - yd))) * log((1.0 + yd) / 2.0) + 8.0 * dilog(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) * log((1.0 + yd) / 2.0) - 8.0 * dilog(-((1.0 + yd) / (w5 - yd))) * log((1.0 + yd) / 2.0)
                + 8.0 * dilog(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) * log((1.0 + yd) / 2.0) + 4.0 * dilog(-((1.0 + yd) / (w7 - yd))) * log((1.0 + yd) / 2.0) - 4.0 * dilog(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) * log((1.0 + yd) / 2.0)
                + 8.0 * dilog(((-1.0 + w4) * (1.0 + yd)) / ((1.0 + w4) * (-1.0 + yd))) * log((1.0 + yd) / 2.0) + 8.0 * dilog(((1.0 + w4) * (1.0 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) * log((1.0 + yd) / 2.0)
                + 8.0 * dilog(((-1.0 + w5) * (1.0 + yd)) / ((1.0 + w5) * (-1.0 + yd))) * log((1.0 + yd) / 2.0) + 8.0 * dilog(((1.0 + w5) * (1.0 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) * log((1.0 + yd) / 2.0)
                - 4.0 * dilog(((-1.0 + w7) * (1.0 + yd)) / ((1.0 + w7) * (-1.0 + yd))) * log((1.0 + yd) / 2.0) - 4.0 * dilog(((1.0 + w7) * (1.0 + yd)) / ((-1.0 + w7) * (-1.0 + yd))) * log((1.0 + yd) / 2.0) - 8.0 * dilog((1.0 + yd) / (w4 + yd)) * log((1.0 + yd) / 2.0)
                + 8.0 * dilog(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) * log((1.0 + yd) / 2.0) - 8.0 * dilog((1.0 + yd) / (w5 + yd)) * log((1.0 + yd) / 2.0) + 8.0 * dilog(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd))) * log((1.0 + yd) / 2.0)
                + 4.0 * dilog((1.0 + yd) / (w7 + yd)) * log((1.0 + yd) / 2.0) - 4.0 * dilog(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) * log((1.0 + yd) / 2.0) + dilog((1.0 + yd) / (1.0 + w7)) * (4.0 * ln1myd + 4.0 * log((1.0 + yd) / 2.0))
                + dilog((1.0 - yd) / (1.0 + w4)) * (-8.0 * log((1.0 - yd) / 2.0) - 8.0 * ln1pyd) + dilog((-1.0 + yd) / (-1.0 + w5)) * (-8.0 * log((1.0 - yd) / 2.0) - 8.0 * ln1pyd)
                + dilog(1.0 / (1.0 - w4)) * (16.0 * lnhalf - 8.0 * ln1myd - 8.0 * ln1pyd) + dilog(1.0 / (1.0 + w5)) * (16.0 * lnhalf - 8.0 * ln1myd - 8.0 * ln1pyd)
                + dilog(-2.0 / (-1.0 + w7)) * (-4.0 * ln1myd - 4.0 * ln1pyd) + dilog(2.0 / (1.0 + w7)) * (-4.0 * ln1myd - 4.0 * ln1pyd) + dilog(1.0 / (1.0 + w7)) * (-8.0 * lnhalf - 4.0 * ln1myd - 4.0 * ln1pyd)
                + dilog((1.0 + yd) / (1.0 - w7)) * (4.0 * log((1.0 + yd) / 2.0) - 4.0 * ln1pyd) - 8.0 * dilog((w4 * (1.0 + yd)) / ((-1.0 + w4) * yd)) * ln1pyd + 8.0 * dilog((w5 * (1.0 + yd)) / ((-1.0 + w5) * yd)) * ln1pyd
                + 4.0 * dilog((w7 * (1.0 + yd)) / ((-1.0 + w7) * yd)) * ln1pyd + 8.0 * dilog((w4 + w4 * yd) / (yd + w4 * yd)) * ln1pyd - 8.0 * dilog((w5 + w5 * yd) / (yd + w5 * yd)) * ln1pyd
                - 4.0 * dilog((w7 + w7 * yd) / (yd + w7 * yd)) * ln1pyd + 4.0 * log((w5 - yd) / (1.0 + w5)) * power_of<2>(ln1pyd) + 2.0 * log((w7 - yd) / (1.0 + w7)) * power_of<2>(ln1pyd)
                + dilog((1.0 - yd) / (1.0 + w7)) * (4.0 * log((1.0 - yd) / 2.0) + 4.0 * ln1pyd) + dilog(1.0 / (1.0 - w7)) * (-8.0 * lnhalf + 4.0 * ln1myd + 4.0 * ln1pyd) + dilog(-2.0 / (-1.0 + w4)) * (8.0 * ln1myd + 8.0 * ln1pyd)
                + dilog(2.0 / (1.0 + w4)) * (8.0 * ln1myd + 8.0 * ln1pyd) + dilog(-2.0 / (-1.0 + w5)) * (8.0 * ln1myd + 8.0 * ln1pyd) + dilog(2.0 / (1.0 + w5)) * (8.0 * ln1myd + 8.0 * ln1pyd)
                + dilog(1.0 / (1.0 + w4)) * (16.0 * lnhalf + 8.0 * ln1myd + 8.0 * ln1pyd) + dilog(1.0 / (1.0 - w5)) * (16.0 * lnhalf + 8.0 * ln1myd + 8.0 * ln1pyd)
                + dilog((1.0 + yd) / (1.0 - w4)) * (-8.0 * log((1.0 + yd) / 2.0) + 8.0 * ln1pyd) + dilog((1.0 + yd) / (1.0 + w5)) * (-8.0 * log((1.0 + yd) / 2.0) + 8.0 * ln1pyd)
                + log((w4 - yd) / (-1.0 + w4)) * (4.0 * power_of<2>(ln1myd) + 8.0 * ln1myd * ln1pyd) + log((w4 - yd) / (1.0 + w4)) * (-4.0 * power_of<2>(log((1.0 + yd) / 2.0)) - 4.0 * power_of<2>(ln1pyd)) + (4.0 * pisqu * log(-2.0 / (w4 + yd))) / 3.0
                + (4.0 * power_of<3>(log(-2.0 / (w4 + yd)))) / 3.0 + (4.0 * pisqu * log(2.0 / (w4 + yd))) / 3.0 + (4.0 * power_of<3>(log(2.0 / (w4 + yd)))) / 3.0 + (4.0 * pisqu * log(-2.0 / (w5 + yd))) / 3.0 + (4.0 * power_of<3>(log(-2.0 / (w5 + yd)))) / 3.0 + (4.0 * pisqu * log(2.0 / (w5 + yd))) / 3.0
                + (4.0 * power_of<3>(log(2.0 / (w5 + yd)))) / 3.0 - (2.0 * pisqu * log(-2.0 / (w7 + yd))) / 3.0 - (2.0 * power_of<3>(log(-2.0 / (w7 + yd)))) / 3.0 - (2.0 * pisqu * log(2.0 / (w7 + yd))) / 3.0 - (2.0 * power_of<3>(log(2.0 / (w7 + yd)))) / 3.0
                + log((1.0 - yd) / 2.0) * (-8.0 * log((w4 - yd) / (-1.0 + w4)) * ln1pyd - 8.0 * log((w5 - yd) / (-1.0 + w5)) * ln1pyd + 4.0 * log((w7 - yd) / (-1.0 + w7)) * ln1pyd
                    + ln1pyd * (-8.0 * log((w4 + yd) / (1.0 + w4)) - 8.0 * log((w5 + yd) / (1.0 + w5)) + 4.0 * log((w7 + yd) / (1.0 + w7)))) + power_of<2>(log((1.0 + yd) / 2.0)) * (-4.0 * log((1.0 + w4) / (w4 - yd)) - 4.0 * log((1.0 + w5) / (w5 - yd))
                    - 4.0 * log((w5 - yd) / (1.0 + w5)) + 2.0 * log((1.0 + w7) / (w7 - yd)) + 2.0 * log((w7 - yd) / (1.0 + w7)) + 4.0 * log((-2.0 * (w4 - yd)) / ((1.0 + w4) * (-1.0 + yd))) + 4.0 * log((-2.0 * (w5 - yd)) / ((1.0 + w5) * (-1.0 + yd)))
                    - 2.0 * log((-2.0 * (w7 - yd)) / ((1.0 + w7) * (-1.0 + yd))) + 4.0 * log((-1.0 / 2.0) * (((1.0 + w4) * (-1.0 + yd)) / (w4 - yd))) + 4.0 * log((-1.0 / 2.0) * (((1.0 + w5) * (-1.0 + yd)) / (w5 - yd)))
                    - 2.0 * log((-1.0 / 2.0) * (((1.0 + w7) * (-1.0 + yd)) / (w7 - yd))) - 4.0 * log((-1.0 + w4) / (w4 + yd)) + 4.0 * log((-1.0 / 2.0) * (((-1.0 + w4) * (-1.0 + yd)) / (w4 + yd))) - 4.0 * log((w4 + yd) / (-1.0 + w4))
                    + 4.0 * log((-2.0 * (w4 + yd)) / ((-1.0 + w4) * (-1.0 + yd))) - 4.0 * log((-1.0 + w5) / (w5 + yd)) + 4.0 * log((-1.0 / 2.0) * (((-1.0 + w5) * (-1.0 + yd)) / (w5 + yd))) - 4.0 * log((w5 + yd) / (-1.0 + w5))
                    + 4.0 * log((-2.0 * (w5 + yd)) / ((-1.0 + w5) * (-1.0 + yd))) + 2.0 * log((-1.0 + w7) / (w7 + yd)) - 2.0 * log((-1.0 / 2.0) * (((-1.0 + w7) * (-1.0 + yd)) / (w7 + yd))) + 2.0 * log((w7 + yd) / (-1.0 + w7))
                    - 2.0 * log((-2.0 * (w7 + yd)) / ((-1.0 + w7) * (-1.0 + yd)))) + power_of<2>(log((1.0 - yd) / 2.0)) * (-4.0 * log((-1.0 + w4) / (w4 - yd)) - 4.0 * log((w4 - yd) / (-1.0 + w4)) - 4.0 * log((-1.0 + w5) / (w5 - yd)) - 4.0 * log((w5 - yd) / (-1.0 + w5))
                    + 2.0 * log((-1.0 + w7) / (w7 - yd)) + 2.0 * log((w7 - yd) / (-1.0 + w7)) + 4.0 * log((2.0 * (w4 - yd)) / ((-1.0 + w4) * (1.0 + yd))) + 4.0 * log((2.0 * (w5 - yd)) / ((-1.0 + w5) * (1.0 + yd))) - 2.0 * log((2.0 * (w7 - yd)) / ((-1.0 + w7) * (1.0 + yd)))
                    + 4.0 * log(((-1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 - yd))) + 4.0 * log(((-1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 - yd))) - 2.0 * log(((-1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 - yd))) - 4.0 * log((1.0 + w4) / (w4 + yd))
                    + 4.0 * log(((1.0 + w4) * (1.0 + yd)) / (2.0 * (w4 + yd))) - 4.0 * log((w4 + yd) / (1.0 + w4)) + 4.0 * log((2.0 * (w4 + yd)) / ((1.0 + w4) * (1.0 + yd))) - 4.0 * log((1.0 + w5) / (w5 + yd)) + 4.0 * log(((1.0 + w5) * (1.0 + yd)) / (2.0 * (w5 + yd)))
                    - 4.0 * log((w5 + yd) / (1.0 + w5)) + 4.0 * log((2.0 * (w5 + yd)) / ((1.0 + w5) * (1.0 + yd))) + 2.0 * log((1.0 + w7) / (w7 + yd)) - 2.0 * log(((1.0 + w7) * (1.0 + yd)) / (2.0 * (w7 + yd))) + 2.0 * log((w7 + yd) / (1.0 + w7))
                    - 2.0 * log((2.0 * (w7 + yd)) / ((1.0 + w7) * (1.0 + yd)))) + (4.0 * pisqu * log(w4 / (yd - w4 * yd))) / 3.0 + (4.0 * power_of<3>(log(w4 / (yd - w4 * yd)))) / 3.0 - (4.0 * pisqu * log(-(w4 / (yd + w4 * yd)))) / 3.0 - (4.0 * power_of<3>(log(-(w4 / (yd + w4 * yd))))) / 3.0
                - (4.0 * pisqu * log(w4 / (yd + w4 * yd))) / 3.0 - (4.0 * power_of<3>(log(w4 / (yd + w4 * yd)))) / 3.0 - (4.0 * pisqu * log(w5 / (yd - w5 * yd))) / 3.0 - (4.0 * power_of<3>(log(w5 / (yd - w5 * yd)))) / 3.0 + (4.0 * pisqu * log(-(w5 / (yd + w5 * yd)))) / 3.0
                + (4.0 * power_of<3>(log(-(w5 / (yd + w5 * yd))))) / 3.0 + (4.0 * pisqu * log(w5 / (yd + w5 * yd))) / 3.0 + (4.0 * power_of<3>(log(w5 / (yd + w5 * yd)))) / 3.0 - (2.0 * pisqu * log(w7 / (yd - w7 * yd))) / 3.0 - (2.0 * power_of<3>(log(w7 / (yd - w7 * yd)))) / 3.0
                + (2.0 * pisqu * log(-(w7 / (yd + w7 * yd)))) / 3.0 + (2.0 * power_of<3>(log(-(w7 / (yd + w7 * yd))))) / 3.0 + (2.0 * pisqu * log(w7 / (yd + w7 * yd))) / 3.0 + (2.0 * power_of<3>(log(w7 / (yd + w7 * yd)))) / 3.0
                + power_of<2>(ln1pyd) * (4.0 * log((w4 + yd) / (-1.0 + w4)) - 4.0 * log((w5 + yd) / (-1.0 + w5)) - 2.0 * log((w7 + yd) / (-1.0 + w7)) - 4.0 * log((w4 + yd) / (yd - w4 * yd)) + 4.0 * log(-((w4 - yd) / (yd + w4 * yd)))
                    + 4.0 * log((w5 + yd) / (yd - w5 * yd)) - 4.0 * log(-((w5 - yd) / (yd + w5 * yd))) + 2.0 * log((w7 + yd) / (yd - w7 * yd)) - 2.0 * log(-((w7 - yd) / (yd + w7 * yd))))
                + power_of<2>(ln1myd) * (-4.0 * log((w5 - yd) / (-1.0 + w5)) - 2.0 * log((w7 - yd) / (-1.0 + w7)) - 4.0 * log((w4 - yd) / ((-1.0 + w4) * yd)) + 4.0 * log((w5 - yd) / ((-1.0 + w5) * yd)) + 2.0 * log((w7 - yd) / ((-1.0 + w7) * yd))
                    - 4.0 * log((w4 + yd) / (1.0 + w4)) + 4.0 * log((w5 + yd) / (1.0 + w5)) + 2.0 * log((w7 + yd) / (1.0 + w7)) + 4.0 * log((w4 + yd) / (yd + w4 * yd)) - 4.0 * log((w5 + yd) / (yd + w5 * yd)) - 2.0 * log((w7 + yd) / (yd + w7 * yd)))
                - (8.0 * 1.0i) * M_PI * H1((1.0i + wx3) / (wx3 - xd), (-1.0i + wx3) / (wx3 - xd)) * power_of<2>(log(((-2.0 * 1.0i) * (wx3 - xd)) / (1.0 + power_of<2>(wx3)))) * my_sign(imag((-1.0i + wx3) / (wx3 - xd))) + (8.0 * 1.0i) * M_PI * H1((-1.0i + wx3) / (1.0i + wx3), (-1.0i + wx3) / (wx3 - xd)) * 
                power_of<2>(log((1.0i + xd) / (-1.0i + wx3))) * my_sign(imag((-1.0i + wx3) / (wx3 - xd))) - (8.0 * 1.0i) * M_PI * H1((-1.0i + wx3) / (wx3 - xd), (1.0i + wx3) / (wx3 - xd)) * power_of<2>(log(((2.0 * 1.0i) * (wx3 - xd)) / (1.0 + power_of<2>(wx3)))) * my_sign(imag((1.0i + wx3) / (wx3 - xd)))
                + (8.0 * 1.0i) * M_PI * H1((1.0i + wx3) / (-1.0i + wx3), (1.0i + wx3) / (wx3 - xd)) * power_of<2>(log((-1.0i + xd) / (1.0i + wx3))) * my_sign(imag((1.0i + wx3) / (wx3 - xd))) - (8.0 * 1.0i) * M_PI * H1((1.0i + wx4) / (wx4 - xd), (-1.0i + wx4) / (wx4 - xd)) * 
                power_of<2>(log(((-2.0 * 1.0i) * (wx4 - xd)) / (1.0 + power_of<2>(wx4)))) * my_sign(imag((-1.0i + wx4) / (wx4 - xd))) + (8.0 * 1.0i) * M_PI * H1((-1.0i + wx4) / (1.0i + wx4), (-1.0i + wx4) / (wx4 - xd)) * power_of<2>(log((1.0i + xd) / (-1.0i + wx4))) * my_sign(imag((-1.0i + wx4) / (wx4 - xd)))
                - (8.0 * 1.0i) * M_PI * H1((-1.0i + wx4) / (wx4 - xd), (1.0i + wx4) / (wx4 - xd)) * power_of<2>(log(((2.0 * 1.0i) * (wx4 - xd)) / (1.0 + power_of<2>(wx4)))) * my_sign(imag((1.0i + wx4) / (wx4 - xd))) + (8.0 * 1.0i) * M_PI * H1((1.0i + wx4) / (-1.0i + wx4), (1.0i + wx4) / (wx4 - xd)) * 
                power_of<2>(log((-1.0i + xd) / (1.0i + wx4))) * my_sign(imag((1.0i + wx4) / (wx4 - xd))) - (8.0 * 1.0i) * M_PI * H1(1.0i / xd, -(wx3 * xdinv)) * power_of<2>(log((-1.0i + wx3inv) * xd)) * my_sign(-imag(wx3 * xdinv))
                - (8.0 * 1.0i) * M_PI * H1(-1.0i / xd, -(wx3 * xdinv)) * power_of<2>(log((1.0i + wx3inv) * xd)) * my_sign(-imag(wx3 * xdinv)) + (8.0 * 1.0i) * M_PI * H1(1.0i * wx3, -(wx3 * xdinv)) * power_of<2>(log((-1.0i + xd) / wx3)) * my_sign(-imag(wx3 * xdinv))
                + (8.0 * 1.0i) * M_PI * H1((-1.0i) * wx3, -(wx3 * xdinv)) * power_of<2>(log((1.0i + xd) / wx3)) * my_sign(-imag(wx3 * xdinv)) + (8.0 * 1.0i) * M_PI * H1((-1.0i) * wx3, wx3 * xdinv) * power_of<2>(log((1.0i - xd) / wx3)) * my_sign(imag(wx3 * xdinv))
                - (8.0 * 1.0i) * M_PI * H1(1.0i / xd, wx3 * xdinv) * power_of<2>(log((-1.0i - wx3inv) * xd)) * my_sign(imag(wx3 * xdinv)) - (8.0 * 1.0i) * M_PI * H1(-1.0i / xd, wx3 * xdinv) * power_of<2>(log((1.0i - wx3inv) * xd)) * my_sign(imag(wx3 * xdinv))
                + (8.0 * 1.0i) * M_PI * H1(1.0i * wx3, wx3 * xdinv) * power_of<2>(log(-((1.0i + xd) / wx3))) * my_sign(imag(wx3 * xdinv)) - (8.0 * 1.0i) * M_PI * H1(1.0i / xd, -(wx4 * xdinv)) * power_of<2>(log((-1.0i + wx4inv) * xd)) * my_sign(-imag(wx4 * xdinv))
                - (8.0 * 1.0i) * M_PI * H1(-1.0i / xd, -(wx4 * xdinv)) * power_of<2>(log((1.0i + wx4inv) * xd)) * my_sign(-imag(wx4 * xdinv)) + (8.0 * 1.0i) * M_PI * H1(1.0i * wx4, -(wx4 * xdinv)) * power_of<2>(log((-1.0i + xd) / wx4)) * my_sign(-imag(wx4 * xdinv))
                + (8.0 * 1.0i) * M_PI * H1((-1.0i) * wx4, -(wx4 * xdinv)) * power_of<2>(log((1.0i + xd) / wx4)) * my_sign(-imag(wx4 * xdinv)) + (8.0 * 1.0i) * M_PI * H1((-1.0i) * wx4, wx4 * xdinv) * power_of<2>(log((1.0i - xd) / wx4)) * my_sign(imag(wx4 * xdinv))
                - (8.0 * 1.0i) * M_PI * H1(1.0i / xd, wx4 * xdinv) * power_of<2>(log((-1.0i - wx4inv) * xd)) * my_sign(imag(wx4 * xdinv)) - (8.0 * 1.0i) * M_PI * H1(-1.0i / xd, wx4 * xdinv) * power_of<2>(log((1.0i - wx4inv) * xd)) * my_sign(imag(wx4 * xdinv))
                + (8.0 * 1.0i) * M_PI * H1(1.0i * wx4, wx4 * xdinv) * power_of<2>(log(-((1.0i + xd) / wx4))) * my_sign(imag(wx4 * xdinv)) + (8.0 * 1.0i) * M_PI * H1((-1.0i + wx3) / (1.0i + wx3), (-1.0i + wx3) / (wx3 + xd)) * power_of<2>(log((1.0i - xd) / (-1.0i + wx3))) * my_sign(imag((-1.0i + wx3) / (wx3 + xd)))
                - (8.0 * 1.0i) * M_PI * H1((1.0i + wx3) / (wx3 + xd), (-1.0i + wx3) / (wx3 + xd)) * power_of<2>(log(((-2.0 * 1.0i) * (wx3 + xd)) / (1.0 + power_of<2>(wx3)))) * my_sign(imag((-1.0i + wx3) / (wx3 + xd))) + (8.0 * 1.0i) * M_PI * H1((1.0i + wx3) / (-1.0i + wx3), (1.0i + wx3) / (wx3 + xd)) * 
                power_of<2>(log(-((1.0i + xd) / (1.0i + wx3)))) * my_sign(imag((1.0i + wx3) / (wx3 + xd))) - (8.0 * 1.0i) * M_PI * H1((-1.0i + wx3) / (wx3 + xd), (1.0i + wx3) / (wx3 + xd)) * power_of<2>(log(((2.0 * 1.0i) * (wx3 + xd)) / (1.0 + power_of<2>(wx3)))) * my_sign(imag((1.0i + wx3) / (wx3 + xd)))
                + (8.0 * 1.0i) * M_PI * H1((-1.0i + wx4) / (1.0i + wx4), (-1.0i + wx4) / (wx4 + xd)) * power_of<2>(log((1.0i - xd) / (-1.0i + wx4))) * my_sign(imag((-1.0i + wx4) / (wx4 + xd))) - (8.0 * 1.0i) * M_PI * H1((1.0i + wx4) / (wx4 + xd), (-1.0i + wx4) / (wx4 + xd)) * 
                power_of<2>(log(((-2.0 * 1.0i) * (wx4 + xd)) / (1.0 + power_of<2>(wx4)))) * my_sign(imag((-1.0i + wx4) / (wx4 + xd))) + (8.0 * 1.0i) * M_PI * H1((1.0i + wx4) / (-1.0i + wx4), (1.0i + wx4) / (wx4 + xd)) * power_of<2>(log(-((1.0i + xd) / (1.0i + wx4)))) * my_sign(imag((1.0i + wx4) / (wx4 + xd)))
                - (8.0 * 1.0i) * M_PI * H1((-1.0i + wx4) / (wx4 + xd), (1.0i + wx4) / (wx4 + xd)) * power_of<2>(log(((2.0 * 1.0i) * (wx4 + xd)) / (1.0 + power_of<2>(wx4)))) * my_sign(imag((1.0i + wx4) / (wx4 + xd))) + (8.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 - yd), -2.0 / (-1.0 + yd)) * 
                power_of<2>(log(((-1.0 + w4) * (-1.0 + yd)) / (2.0 * (1.0 + w4)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (8.0 * 1.0i) * M_PI * H1((-1.0 + w4) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * power_of<2>(log(((1.0 + w4) * (-1.0 + yd)) / (2.0 * (-1.0 + w4)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                + (8.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 - yd), -2.0 / (-1.0 + yd)) * power_of<2>(log(((-1.0 + w5) * (-1.0 + yd)) / (2.0 * (1.0 + w5)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (8.0 * 1.0i) * M_PI * H1((-1.0 + w5) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * 
                power_of<2>(log(((1.0 + w5) * (-1.0 + yd)) / (2.0 * (-1.0 + w5)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 - yd), -2.0 / (-1.0 + yd)) * power_of<2>(log(((-1.0 + w7) * (-1.0 + yd)) / (2.0 * (1.0 + w7)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                - (4.0 * 1.0i) * M_PI * H1((-1.0 + w7) / (-1.0 + yd), -2.0 / (-1.0 + yd)) * power_of<2>(log(((1.0 + w7) * (-1.0 + yd)) / (2.0 * (-1.0 + w7)))) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (8.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), -2.0 / (-1.0 + yd)) * power_of<2>(log((-w4 + yd) / 2.0)) * 
                my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (8.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), -2.0 / (-1.0 + yd)) * power_of<2>(log((w4 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) - (8.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), -2.0 / (-1.0 + yd)) * power_of<2>(log((-w5 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                - (8.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), -2.0 / (-1.0 + yd)) * power_of<2>(log((w5 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), -2.0 / (-1.0 + yd)) * power_of<2>(log((-w7 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd)))
                + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), -2.0 / (-1.0 + yd)) * power_of<2>(log((w7 + yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) + (8.0 * 1.0i) * M_PI * H1(w4 / yd, -ydinv) * power_of<2>(log((1.0 + w4inv) * yd)) * my_sign(-imydinv)
                - (8.0 * 1.0i) * M_PI * H1(-(w4 / yd), -ydinv) * power_of<2>(log(((-1.0 + w4) * yd) / w4)) * my_sign(-imydinv) - (8.0 * 1.0i) * M_PI * H1(w5 / yd, -ydinv) * power_of<2>(log((1.0 + w5inv) * yd)) * my_sign(-imydinv)
                + (8.0 * 1.0i) * M_PI * H1(-(w5 / yd), -ydinv) * power_of<2>(log(((-1.0 + w5) * yd) / w5)) * my_sign(-imydinv) - (4.0 * 1.0i) * M_PI * H1(w7 / yd, -ydinv) * power_of<2>(log((1.0 + w7inv) * yd)) * my_sign(-imydinv)
                + (4.0 * 1.0i) * M_PI * H1(-(w7 / yd), -ydinv) * power_of<2>(log(((-1.0 + w7) * yd) / w7)) * my_sign(-imydinv) - (8.0 * 1.0i) * M_PI * H1(w4 / yd, ydinv) * power_of<2>(log((-1.0 + w4inv) * yd)) * my_sign(imydinv)
                + (8.0 * 1.0i) * M_PI * H1(-(w4 / yd), ydinv) * power_of<2>(log(-(((1.0 + w4) * yd) / w4))) * my_sign(imydinv) + (8.0 * 1.0i) * M_PI * H1(w5 / yd, ydinv) * power_of<2>(log((-1.0 + w5inv) * yd)) * my_sign(imydinv)
                - (8.0 * 1.0i) * M_PI * H1(-(w5 / yd), ydinv) * power_of<2>(log(-(((1.0 + w5) * yd) / w5))) * my_sign(imydinv) + (4.0 * 1.0i) * M_PI * H1(w7 / yd, ydinv) * power_of<2>(log((-1.0 + w7inv) * yd)) * my_sign(imydinv)
                - (4.0 * 1.0i) * M_PI * H1(-(w7 / yd), ydinv) * power_of<2>(log(-(((1.0 + w7) * yd) / w7))) * my_sign(imydinv) - (8.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w4), 2.0 / (1.0 + yd)) * power_of<2>(log((-w4 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                - (8.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w4), 2.0 / (1.0 + yd)) * power_of<2>(log((w4 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (8.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w5), 2.0 / (1.0 + yd)) * power_of<2>(log((-w5 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                - (8.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w5), 2.0 / (1.0 + yd)) * power_of<2>(log((w5 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (4.0 * 1.0i) * M_PI * H1(-2.0 / (-1.0 + w7), 2.0 / (1.0 + yd)) * power_of<2>(log((-w7 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                + (4.0 * 1.0i) * M_PI * H1(2.0 / (1.0 + w7), 2.0 / (1.0 + yd)) * power_of<2>(log((w7 - yd) / 2.0)) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (8.0 * 1.0i) * M_PI * H1((1.0 + w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * power_of<2>(log((-1.0 / 2.0) * (((-1.0 + w4) * (1.0 + yd)) / (1.0 + w4)))) * 
                my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (8.0 * 1.0i) * M_PI * H1((1.0 - w4) / (1.0 + yd), 2.0 / (1.0 + yd)) * power_of<2>(log(((1.0 + w4) * (1.0 + yd)) / (2.0 - 2.0 * w4))) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                + (8.0 * 1.0i) * M_PI * H1((1.0 + w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * power_of<2>(log((-1.0 / 2.0) * (((-1.0 + w5) * (1.0 + yd)) / (1.0 + w5)))) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (8.0 * 1.0i) * M_PI * H1((1.0 - w5) / (1.0 + yd), 2.0 / (1.0 + yd)) * 
                power_of<2>(log(((1.0 + w5) * (1.0 + yd)) / (2.0 - 2.0 * w5))) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) - (4.0 * 1.0i) * M_PI * H1((1.0 + w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * power_of<2>(log((-1.0 / 2.0) * (((-1.0 + w7) * (1.0 + yd)) / (1.0 + w7)))) * my_sign(2.0 * imag(1.0 / (1.0 + yd)))
                - (4.0 * 1.0i) * M_PI * H1((1.0 - w7) / (1.0 + yd), 2.0 / (1.0 + yd)) * power_of<2>(log(((1.0 + w7) * (1.0 + yd)) / (2.0 - 2.0 * w7))) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) + (16.0 * 1.0i) * M_PI * H1((1.0i + wx3) / (-1.0i + wx3), (1.0i + wx3) / wx3) * power_of<2>(log(-1.0i / (1.0i + wx3))) * my_sign(-real(wx3inv))
                - (16.0 * 1.0i) * M_PI * H1((-1.0i + wx3) / wx3, (1.0i + wx3) / wx3) * power_of<2>(log(((2.0 * 1.0i) * wx3) / (1.0 + power_of<2>(wx3)))) * my_sign(-real(wx3inv)) + (16.0 * 1.0i) * M_PI * H1((-1.0i + wx3) / (1.0i + wx3), (-1.0i + wx3) / wx3) * power_of<2>(log(1.0i / (-1.0i + wx3))) * my_sign(real(wx3inv))
                - (16.0 * 1.0i) * M_PI * H1((1.0i + wx3) / wx3, (-1.0i + wx3) / wx3) * power_of<2>(log(((-2.0 * 1.0i) * wx3) / (1.0 + power_of<2>(wx3)))) * my_sign(real(wx3inv)) + (16.0 * 1.0i) * M_PI * H1((1.0i + wx4) / (-1.0i + wx4), (1.0i + wx4) / wx4) * power_of<2>(log(-1.0i / (1.0i + wx4))) * my_sign(-real(wx4inv))
                - (16.0 * 1.0i) * M_PI * H1((-1.0i + wx4) / wx4, (1.0i + wx4) / wx4) * power_of<2>(log(((2.0 * 1.0i) * wx4) / (1.0 + power_of<2>(wx4)))) * my_sign(-real(wx4inv)) + (16.0 * 1.0i) * M_PI * H1((-1.0i + wx4) / (1.0i + wx4), (-1.0i + wx4) / wx4) * power_of<2>(log(1.0i / (-1.0i + wx4))) * my_sign(real(wx4inv))
                - (16.0 * 1.0i) * M_PI * H1((1.0i + wx4) / wx4, (-1.0i + wx4) / wx4) * power_of<2>(log(((-2.0 * 1.0i) * wx4) / (1.0 + power_of<2>(wx4)))) * my_sign(real(wx4inv)) + (16.0 * 1.0i) * M_PI * power_of<2>(log((-2.0 * 1.0i) / (-1.0i + wx3))) * my_sign(imag((1.0i + wx3) / (-1.0i + wx3))) * 
                T(1.0, -1.0i / (-1.0i + wx3), (-2.0 * 1.0i) / (-1.0i + wx3)) + (16.0 * 1.0i) * M_PI * power_of<2>(log(1.0i / (1.0i + wx3))) * my_sign(-imag(wx3 / (1.0i + wx3))) * T(1.0, -1.0i / (-1.0i + wx3), 1.0i / (1.0i + wx3)) + (16.0 * 1.0i) * M_PI * power_of<2>(log(-1.0i / (-1.0i + wx3))) * my_sign(imag(wx3 / (1.0i - wx3))) * 
                T(1.0, 1.0i / (1.0i + wx3), -1.0i / (-1.0i + wx3)) + (16.0 * 1.0i) * M_PI * power_of<2>(log((2.0 * 1.0i) / (1.0i + wx3))) * my_sign(imag((-1.0i + wx3) / (1.0i + wx3))) * T(1.0, 1.0i / (1.0i + wx3), (2.0 * 1.0i) / (1.0i + wx3))
                + (16.0 * 1.0i) * M_PI * power_of<2>(log((-2.0 * 1.0i) / (-1.0i + wx4))) * my_sign(imag((1.0i + wx4) / (-1.0i + wx4))) * T(1.0, -1.0i / (-1.0i + wx4), (-2.0 * 1.0i) / (-1.0i + wx4)) + (16.0 * 1.0i) * M_PI * power_of<2>(log(1.0i / (1.0i + wx4))) * my_sign(-imag(wx4 / (1.0i + wx4))) * T(1.0, -1.0i / (-1.0i + wx4), 1.0i / (1.0i + wx4))
                + (16.0 * 1.0i) * M_PI * power_of<2>(log(-1.0i / (-1.0i + wx4))) * my_sign(imag(wx4 / (1.0i - wx4))) * T(1.0, 1.0i / (1.0i + wx4), -1.0i / (-1.0i + wx4)) + (16.0 * 1.0i) * M_PI * power_of<2>(log((2.0 * 1.0i) / (1.0i + wx4))) * my_sign(imag((-1.0i + wx4) / (1.0i + wx4))) * T(1.0, 1.0i / (1.0i + wx4), (2.0 * 1.0i) / (1.0i + wx4))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((2.0 * 1.0i) / (1.0i + wx3))) * my_sign(imag((-1.0i + wx3) / (1.0i + wx3))) * T(1.0, (1.0i - xd) / (1.0i + wx3), (2.0 * 1.0i) / (1.0i + wx3))
                + power_of<2>(log((1.0i + xd) / (1.0i - wx3))) * (4.0 * log((-2.0 * 1.0i) / (-1.0i + xd)) - 4.0 * log(((-1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 + xd))) - 4.0 * log(((-2.0 * 1.0i) * (wx3 + xd)) / ((-1.0i + wx3) * (-1.0i + xd)))
                    + (8.0 * 1.0i) * M_PI * my_sign(imag((wx3 + xd) / (-1.0i + wx3))) * T(1.0, (1.0i - xd) / (1.0i + wx3), (1.0i + xd) / (1.0i - wx3))) - (8.0 * 1.0i) * M_PI * power_of<2>(log((2.0 * 1.0i) / (1.0i + wx4))) * my_sign(imag((-1.0i + wx4) / (1.0i + wx4))) * T(1.0, (1.0i - xd) / (1.0i + wx4), (2.0 * 1.0i) / (1.0i + wx4))
                + power_of<2>(log((1.0i + xd) / (1.0i - wx4))) * (-4.0 * log(((-1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 + xd))) - 4.0 * log(((-2.0 * 1.0i) * (wx4 + xd)) / ((-1.0i + wx4) * (-1.0i + xd))) + (8.0 * 1.0i) * M_PI * my_sign(imag((wx4 + xd) / (-1.0i + wx4))) * 
                    T(1.0, (1.0i - xd) / (1.0i + wx4), (1.0i + xd) / (1.0i - wx4))) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - 1.0i * wx3)) * my_sign(real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + 1.0i * wx3)) * my_sign(-real(wx3)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - 1.0i * wx4)) * my_sign(real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + 1.0i * wx4)) * my_sign(-real(wx4)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4)
                + log(1.0 - 1.0i * xd) * (-16.0 * ln2squ - (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx3) - (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx3)
                    - (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 - 1.0i * wx4) - (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(-real(xdinv)) * T(1.0, 1.0 - 1.0i * xd, 1.0 + 1.0i * wx4))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - 1.0i * wx3)) * my_sign(real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + 1.0i * wx3)) * my_sign(-real(wx3)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)
                - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - 1.0i * wx4)) * my_sign(real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + 1.0i * wx4)) * my_sign(-real(wx4)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4)
                + log(1.0 + 1.0i * xd) * (-16.0 * ln2squ - (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx3) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx3) - (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx3) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx3)
                    - (16.0 * 1.0i) * M_PI * log(1.0 - 1.0i * wx4) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 - 1.0i * wx4) - (16.0 * 1.0i) * M_PI * log(1.0 + 1.0i * wx4) * my_sign(real(xdinv)) * T(1.0, 1.0 + 1.0i * xd, 1.0 + 1.0i * wx4))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((-2.0 * 1.0i) / (-1.0i + wx3))) * my_sign(imag((1.0i + wx3) / (-1.0i + wx3))) * T(1.0, (-1.0i + xd) / (-1.0i + wx3), (-2.0 * 1.0i) / (-1.0i + wx3))
                + power_of<2>(log((1.0i + xd) / (1.0i + wx3))) * (-4.0 * log(((1.0i + wx3) * (1.0 + 1.0i * xd)) / (2.0 * (wx3 - xd))) - 4.0 * log(((-2.0 * 1.0i) * (wx3 - xd)) / ((1.0i + wx3) * (-1.0i + xd))) + (8.0 * 1.0i) * M_PI * my_sign(imag((wx3 - xd) / (1.0i + wx3))) * 
                    T(1.0, (-1.0i + xd) / (-1.0i + wx3), (1.0i + xd) / (1.0i + wx3))) - (8.0 * 1.0i) * M_PI * power_of<2>(log((-2.0 * 1.0i) / (-1.0i + wx4))) * my_sign(imag((1.0i + wx4) / (-1.0i + wx4))) * T(1.0, (-1.0i + xd) / (-1.0i + wx4), (-2.0 * 1.0i) / (-1.0i + wx4))
                + power_of<2>(log((1.0i + xd) / (1.0i + wx4))) * (-4.0 * log(((1.0i + wx4) * (1.0 + 1.0i * xd)) / (2.0 * (wx4 - xd))) - 4.0 * log(((-2.0 * 1.0i) * (wx4 - xd)) / ((1.0i + wx4) * (-1.0i + xd))) + (8.0 * 1.0i) * M_PI * my_sign(imag((wx4 - xd) / (1.0i + wx4))) * 
                    T(1.0, (-1.0i + xd) / (-1.0i + wx4), (1.0i + xd) / (1.0i + wx4))) - (8.0 * 1.0i) * M_PI * power_of<2>(log((-2.0 * 1.0i) / (-1.0i + wx3))) * my_sign(imag((1.0i + wx3) / (-1.0i + wx3))) * T(1.0, (1.0i + xd) / (1.0i - wx3), (-2.0 * 1.0i) / (-1.0i + wx3))
                + log((wx3 + xd) / wx3) * ((16.0 * 1.0i) * M_PI * log((2.0 * 1.0i) / (1.0i + wx3)) * my_sign(imag((1.0i + wx3) / (wx3 + xd))) * T(1.0, (1.0i - xd) / (1.0i + wx3), (2.0 * 1.0i) / (1.0i + wx3)) + (16.0 * 1.0i) * M_PI * log((-2.0 * 1.0i) / (-1.0i + wx3)) * my_sign(imag((-1.0i + wx3) / (wx3 + xd))) * 
                    T(1.0, (1.0i + xd) / (1.0i - wx3), (-2.0 * 1.0i) / (-1.0i + wx3))) + power_of<2>(log((1.0i - xd) / (1.0i + wx3))) * (4.0 * log(1.0 / 2.0 - (1.0i / 2.0) * xd) + 4.0 * log((2.0 * 1.0i) / (1.0i + xd)) - 4.0 * log(((1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 + xd)))
                    - 4.0 * log(((2.0 * 1.0i) * (wx3 + xd)) / ((1.0i + wx3) * (1.0i + xd))) + (8.0 * 1.0i) * M_PI * my_sign(imag((wx3 + xd) / (1.0i + wx3))) * T(1.0, (1.0i + xd) / (1.0i - wx3), (1.0i - xd) / (1.0i + wx3)))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((2.0 * 1.0i) / (1.0i + wx3))) * my_sign(imag((-1.0i + wx3) / (1.0i + wx3))) * T(1.0, (1.0i + xd) / (1.0i + wx3), (2.0 * 1.0i) / (1.0i + wx3))
                + log(1.0 - xd / wx3) * ((16.0 * 1.0i) * M_PI * log((-2.0 * 1.0i) / (-1.0i + wx3)) * my_sign(imag((-1.0i + wx3) / (wx3 - xd))) * T(1.0, (-1.0i + xd) / (-1.0i + wx3), (-2.0 * 1.0i) / (-1.0i + wx3)) + (16.0 * 1.0i) * M_PI * log((2.0 * 1.0i) / (1.0i + wx3)) * my_sign(imag((1.0i + wx3) / (wx3 - xd))) * 
                    T(1.0, (1.0i + xd) / (1.0i + wx3), (2.0 * 1.0i) / (1.0i + wx3))) + power_of<2>(log((-1.0i + xd) / (-1.0i + wx3))) * (-4.0 * log(((-1.0i + wx3) * (1.0 - 1.0i * xd)) / (2.0 * (wx3 - xd))) + 4.0 * log((2.0 * 1.0i) / (1.0i + xd))
                    - 4.0 * log(((2.0 * 1.0i) * (wx3 - xd)) / ((-1.0i + wx3) * (1.0i + xd))) + (8.0 * 1.0i) * M_PI * my_sign(imag((wx3 - xd) / (-1.0i + wx3))) * T(1.0, (1.0i + xd) / (1.0i + wx3), (-1.0i + xd) / (-1.0i + wx3)))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((-2.0 * 1.0i) / (-1.0i + wx4))) * my_sign(imag((1.0i + wx4) / (-1.0i + wx4))) * T(1.0, (1.0i + xd) / (1.0i - wx4), (-2.0 * 1.0i) / (-1.0i + wx4))
                + log((wx4 + xd) / wx4) * ((16.0 * 1.0i) * M_PI * log((2.0 * 1.0i) / (1.0i + wx4)) * my_sign(imag((1.0i + wx4) / (wx4 + xd))) * T(1.0, (1.0i - xd) / (1.0i + wx4), (2.0 * 1.0i) / (1.0i + wx4)) + (16.0 * 1.0i) * M_PI * log((-2.0 * 1.0i) / (-1.0i + wx4)) * my_sign(imag((-1.0i + wx4) / (wx4 + xd))) * 
                    T(1.0, (1.0i + xd) / (1.0i - wx4), (-2.0 * 1.0i) / (-1.0i + wx4))) + power_of<2>(log((1.0i - xd) / (1.0i + wx4))) * (4.0 * log((2.0 * 1.0i) / (1.0i + xd)) - 4.0 * log(((1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 + xd))) - 4.0 * log(((2.0 * 1.0i) * (wx4 + xd)) / ((1.0i + wx4) * (1.0i + xd)))
                    + (8.0 * 1.0i) * M_PI * my_sign(imag((wx4 + xd) / (1.0i + wx4))) * T(1.0, (1.0i + xd) / (1.0i - wx4), (1.0i - xd) / (1.0i + wx4))) - (8.0 * 1.0i) * M_PI * power_of<2>(log((2.0 * 1.0i) / (1.0i + wx4))) * my_sign(imag((-1.0i + wx4) / (1.0i + wx4))) * T(1.0, (1.0i + xd) / (1.0i + wx4), (2.0 * 1.0i) / (1.0i + wx4))
                + log(1.0 - xd / wx4) * ((16.0 * 1.0i) * M_PI * log((-2.0 * 1.0i) / (-1.0i + wx4)) * my_sign(imag((-1.0i + wx4) / (wx4 - xd))) * T(1.0, (-1.0i + xd) / (-1.0i + wx4), (-2.0 * 1.0i) / (-1.0i + wx4)) + (16.0 * 1.0i) * M_PI * log((2.0 * 1.0i) / (1.0i + wx4)) * my_sign(imag((1.0i + wx4) / (wx4 - xd))) * 
                    T(1.0, (1.0i + xd) / (1.0i + wx4), (2.0 * 1.0i) / (1.0i + wx4))) + power_of<2>(log((-1.0i + xd) / (-1.0i + wx4))) * (-4.0 * log(((-1.0i + wx4) * (1.0 - 1.0i * xd)) / (2.0 * (wx4 - xd))) + 4.0 * log((2.0 * 1.0i) / (1.0i + xd))
                    - 4.0 * log(((2.0 * 1.0i) * (wx4 - xd)) / ((-1.0i + wx4) * (1.0i + xd))) + (8.0 * 1.0i) * M_PI * my_sign(imag((wx4 - xd) / (-1.0i + wx4))) * T(1.0, (1.0i + xd) / (1.0i + wx4), (-1.0i + xd) / (-1.0i + wx4)))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((-1.0i + wx3) / wx3)) * my_sign(real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3) + log((1.0i - xd) / (1.0i + wx3)) * (8.0 * log(1.0 / 2.0 - (1.0i / 2.0) * xd) * log((wx3 + xd) / wx3) + 4.0 * power_of<2>(log((wx3 + xd) / wx3))
                    - (16.0 * 1.0i) * M_PI * log((-1.0i + wx3) / wx3) * my_sign(real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0i + wx3) / wx3)) * my_sign(-real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3)
                + log((1.0i + xd) / (1.0i - wx3)) * (4.0 * power_of<2>(log((wx3 + xd) / wx3)) - (16.0 * 1.0i) * M_PI * log((1.0i + wx3) / wx3) * my_sign(-real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((-1.0i + wx4) / wx4)) * my_sign(real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4) + log((1.0i - xd) / (1.0i + wx4)) * (4.0 * power_of<2>(log((wx4 + xd) / wx4))
                    - (16.0 * 1.0i) * M_PI * log((-1.0i + wx4) / wx4) * my_sign(real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0i + wx4) / wx4)) * my_sign(-real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4)
                + log((1.0i + xd) / (1.0i - wx4)) * (4.0 * power_of<2>(log((wx4 + xd) / wx4)) - (16.0 * 1.0i) * M_PI * log((1.0i + wx4) / wx4) * my_sign(-real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((-1.0i + wx3) / wx3)) * my_sign(real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3) + dilog((-1.0i + wx3) / (1.0i + wx3)) * (-16.0 * log(-1.0i / (-1.0i + wx3)) + 8.0 * log((-1.0i + xd) / (-1.0i + wx3))
                    + 8.0 * log((1.0i + xd) / (1.0i - wx3)) - (16.0 * 1.0i) * M_PI * my_sign(real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3) - (16.0 * 1.0i) * M_PI * my_sign(real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3))
                + dilog(wx3 / (1.0i + wx3)) * (16.0 * log(-1.0i / (-1.0i + wx3)) - 8.0 * log((-1.0i + xd) / (-1.0i + wx3)) - 8.0 * log((1.0i + xd) / (1.0i - wx3)) + (16.0 * 1.0i) * M_PI * my_sign(real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3)
                    + (16.0 * 1.0i) * M_PI * my_sign(real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3)) + log((1.0i + xd) / (1.0i + wx3)) * (4.0 * power_of<2>(log(1.0 - xd / wx3)) - (16.0 * 1.0i) * M_PI * log((-1.0i + wx3) / wx3) * my_sign(real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0i + wx3) / wx3)) * my_sign(-real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3) + dilog((1.0i + wx3) / (-1.0i + wx3)) * (-16.0 * log(1.0i / (1.0i + wx3)) + 8.0 * log((1.0i - xd) / (1.0i + wx3)) + 8.0 * log((1.0i + xd) / (1.0i + wx3))
                    - (16.0 * 1.0i) * M_PI * my_sign(-real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3) - (16.0 * 1.0i) * M_PI * my_sign(-real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3))
                + dilog(wx3 / (-1.0i + wx3)) * (16.0 * log(1.0i / (1.0i + wx3)) - 8.0 * log((1.0i - xd) / (1.0i + wx3)) - 8.0 * log((1.0i + xd) / (1.0i + wx3)) + (16.0 * 1.0i) * M_PI * my_sign(-real(wx3inv)) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3)
                    + (16.0 * 1.0i) * M_PI * my_sign(-real(wx3inv)) * T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3)) + log((-1.0i + xd) / (-1.0i + wx3)) * (4.0 * power_of<2>(log(1.0 - xd / wx3)) - (16.0 * 1.0i) * M_PI * log((1.0i + wx3) / wx3) * my_sign(-real(wx3inv)) * 
                    T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((-1.0i + wx4) / wx4)) * my_sign(real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4)
                + dilog((-1.0i + wx4) / (1.0i + wx4)) * (-16.0 * log(-1.0i / (-1.0i + wx4)) + 8.0 * log((-1.0i + xd) / (-1.0i + wx4)) + 8.0 * log((1.0i + xd) / (1.0i - wx4)) - (16.0 * 1.0i) * M_PI * my_sign(real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4)
                    - (16.0 * 1.0i) * M_PI * my_sign(real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4)) + dilog(wx4 / (1.0i + wx4)) * (16.0 * log(-1.0i / (-1.0i + wx4)) - 8.0 * log((-1.0i + xd) / (-1.0i + wx4)) - 8.0 * log((1.0i + xd) / (1.0i - wx4))
                    + (16.0 * 1.0i) * M_PI * my_sign(real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4) + (16.0 * 1.0i) * M_PI * my_sign(real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4))
                + log((1.0i + xd) / (1.0i + wx4)) * (4.0 * power_of<2>(log(1.0 - xd / wx4)) - (16.0 * 1.0i) * M_PI * log((-1.0i + wx4) / wx4) * my_sign(real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0i + wx4) / wx4)) * my_sign(-real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4) + dilog((1.0i + wx4) / (-1.0i + wx4)) * (-16.0 * log(1.0i / (1.0i + wx4)) + 8.0 * log((1.0i - xd) / (1.0i + wx4)) + 8.0 * log((1.0i + xd) / (1.0i + wx4))
                    - (16.0 * 1.0i) * M_PI * my_sign(-real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4) - (16.0 * 1.0i) * M_PI * my_sign(-real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4))
                + dilog(wx4 / (-1.0i + wx4)) * (16.0 * log(1.0i / (1.0i + wx4)) - 8.0 * log((1.0i - xd) / (1.0i + wx4)) - 8.0 * log((1.0i + xd) / (1.0i + wx4)) + (16.0 * 1.0i) * M_PI * my_sign(-real(wx4inv)) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4)
                    + (16.0 * 1.0i) * M_PI * my_sign(-real(wx4inv)) * T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4)) + log((-1.0i + xd) / (-1.0i + wx4)) * (4.0 * power_of<2>(log(1.0 - xd / wx4)) - (16.0 * 1.0i) * M_PI * log((1.0i + wx4) / wx4) * my_sign(-real(wx4inv)) * 
                    T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4)) + power_of<2>(log(1.0 - 1.0i * xd)) * ((4.0 * ln64) / 3.0 - 4.0 * log((wx3 - xd) / (1.0i + wx3)) - 4.0 * log((wx4 - xd) / (1.0i + wx4)) - (8.0 * log(1.0 / 8.0 - (1.0i / 8.0) * xd)) / 3.0 - 8.0 * log(1.0i * xd)
                    + 4.0 * log((wx3 + xd) / wx3) - 4.0 * log((wx3 + xd) / (-1.0i + wx3)) + 4.0 * log((wx4 + xd) / wx4) - 4.0 * log((wx4 + xd) / (-1.0i + wx4)) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4)
                    + (8.0 * 1.0i) * M_PI * my_sign(real(xd)) * T(1.0, (wx3 + xd) / wx3, 1.0 - 1.0i * xd) + (8.0 * 1.0i) * M_PI * my_sign(real(xd)) * T(1.0, (wx4 + xd) / wx4, 1.0 - 1.0i * xd) + (8.0 * 1.0i) * M_PI * my_sign(real(xd)) * T(1.0, 1.0 - xd / wx3, 1.0 - 1.0i * xd)
                    + (8.0 * 1.0i) * M_PI * my_sign(real(xd)) * T(1.0, 1.0 - xd / wx4, 1.0 - 1.0i * xd)) + power_of<2>(log(1.0 + 1.0i * xd)) * ((4.0 * ln64) / 3.0 - 4.0 * log((wx3 - xd) / (-1.0i + wx3)) - 4.0 * log((wx4 - xd) / (-1.0i + wx4)) - (8.0 * log((1.0 + 1.0i * xd) / 8.0)) / 3.0 - 8.0 * log((-1.0i) * xd)
                    + 4.0 * log((wx3 + xd) / wx3) - 4.0 * log((wx3 + xd) / (1.0i + wx3)) + 4.0 * log((wx4 + xd) / wx4) - 4.0 * log((wx4 + xd) / (1.0i + wx4)) + 4.0 * log(1.0 - xd / wx3) + 4.0 * log(1.0 - xd / wx4)
                    + (8.0 * 1.0i) * M_PI * my_sign(-real(xd)) * T(1.0, (wx3 + xd) / wx3, 1.0 + 1.0i * xd) + (8.0 * 1.0i) * M_PI * my_sign(-real(xd)) * T(1.0, (wx4 + xd) / wx4, 1.0 + 1.0i * xd) + (8.0 * 1.0i) * M_PI * my_sign(-real(xd)) * T(1.0, 1.0 - xd / wx3, 1.0 + 1.0i * xd)
                    + (8.0 * 1.0i) * M_PI * my_sign(-real(xd)) * T(1.0, 1.0 - xd / wx4, 1.0 + 1.0i * xd)) + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w4) / 2.0)) * my_sign(imag(w4) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w4) / 2.0) + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w4)) * 
                T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0) + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0) + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w5) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w5)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w5) / 2.0)
                - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w7) / 2.0)) * my_sign(imag(w7) / 2.0) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w7) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w7)) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0)
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((w4 - yd) / (1.0 + w4))) * my_sign(imag((1.0 + yd) / (1.0 + w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 - yd) / (1.0 + w4)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((w5 - yd) / (1.0 + w5))) * my_sign(imag((1.0 + yd) / (1.0 + w5))) * 
                T(1.0, (1.0 - yd) / 2.0, (w5 - yd) / (1.0 + w5)) + (4.0 * 1.0i) * M_PI * power_of<2>(log((w7 - yd) / (1.0 + w7))) * my_sign(imag((1.0 + yd) / (1.0 + w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 - yd) / (1.0 + w7))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((w4 + yd) / (-1.0 + w4))) * my_sign(imag((1.0 + yd) / (1.0 - w4))) * T(1.0, (1.0 - yd) / 2.0, (w4 + yd) / (-1.0 + w4)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((w5 + yd) / (-1.0 + w5))) * my_sign(imag((1.0 + yd) / (1.0 - w5))) * 
                T(1.0, (1.0 - yd) / 2.0, (w5 + yd) / (-1.0 + w5)) + (4.0 * 1.0i) * M_PI * power_of<2>(log((w7 + yd) / (-1.0 + w7))) * my_sign(imag((1.0 + yd) / (1.0 - w7))) * T(1.0, (1.0 - yd) / 2.0, (w7 + yd) / (-1.0 + w7))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w4)) * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (16.0 * 1.0i) * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4)
                + (16.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * log(1.0 + w4) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5)
                + (16.0 * 1.0i) * M_PI * log(1.0 - w5) * log((1.0 + w5) / 2.0) * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (16.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5)
                - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w5)) * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w7)) * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (8.0 * 1.0i) * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7)
                - (8.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * log(1.0 + w7) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7)
                + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w4) / 2.0)) * my_sign(imag(w4) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0) + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w4) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w4)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0)
                + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w5) / 2.0)) * my_sign(imag(w5) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0) + (8.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w5) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w5)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0)
                - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 - w7) / 2.0)) * my_sign(imag(w7) / 2.0) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0) - (4.0 * 1.0i) * M_PI * power_of<2>(log((1.0 + w7) / 2.0)) * my_sign((-1.0 / 2.0) * imag(w7)) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0)
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((w4 - yd) / (-1.0 + w4))) * my_sign(imag((-1.0 + yd) / (-1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 - yd) / (-1.0 + w4)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((w5 - yd) / (-1.0 + w5))) * my_sign(imag((-1.0 + yd) / (-1.0 + w5))) * 
                T(1.0, (1.0 + yd) / 2.0, (w5 - yd) / (-1.0 + w5)) + (4.0 * 1.0i) * M_PI * power_of<2>(log((w7 - yd) / (-1.0 + w7))) * my_sign(imag((-1.0 + yd) / (-1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 - yd) / (-1.0 + w7))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log((w4 + yd) / (1.0 + w4))) * my_sign(imag((1.0 - yd) / (1.0 + w4))) * T(1.0, (1.0 + yd) / 2.0, (w4 + yd) / (1.0 + w4)) - (8.0 * 1.0i) * M_PI * power_of<2>(log((w5 + yd) / (1.0 + w5))) * my_sign(imag((1.0 - yd) / (1.0 + w5))) * 
                T(1.0, (1.0 + yd) / 2.0, (w5 + yd) / (1.0 + w5)) + (4.0 * 1.0i) * M_PI * power_of<2>(log((w7 + yd) / (1.0 + w7))) * my_sign(imag((1.0 - yd) / (1.0 + w7))) * T(1.0, (1.0 + yd) / 2.0, (w7 + yd) / (1.0 + w7))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w4)) * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4) + (16.0 * 1.0i) * M_PI * log(1.0 - w4) * log((1.0 + w4) / 2.0) * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4)
                + dilog((1.0 - w4) / 2.0) * (16.0 * log(w4 / (-1.0 + w4)) - 8.0 * log((w4 - yd) / (-1.0 + w4)) - 8.0 * log((w4 + yd) / (-1.0 + w4)) + (16.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) + (16.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4))
                + (16.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * log(1.0 + w4) * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w4)) * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4)
                + dilog((1.0 + w4) / 2.0) * (16.0 * log(w4 / (1.0 + w4)) - 8.0 * log((w4 - yd) / (1.0 + w4)) - 8.0 * log((w4 + yd) / (1.0 + w4)) + (16.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) + (16.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4))
                - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w5)) * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5) + (16.0 * 1.0i) * M_PI * log(1.0 - w5) * log((1.0 + w5) / 2.0) * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5)
                + dilog((1.0 - w5) / 2.0) * (16.0 * log(w5 / (-1.0 + w5)) - 8.0 * log((w5 - yd) / (-1.0 + w5)) - 8.0 * log((w5 + yd) / (-1.0 + w5)) + (16.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5) + (16.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5))
                + (16.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * log(1.0 + w5) * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) - (8.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w5)) * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5)
                + dilog((1.0 + w5) / 2.0) * (16.0 * log(w5 / (1.0 + w5)) - 8.0 * log((w5 - yd) / (1.0 + w5)) - 8.0 * log((w5 + yd) / (1.0 + w5)) + (16.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (16.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5))
                + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 - w7)) * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7) - (8.0 * 1.0i) * M_PI * log(1.0 - w7) * log((1.0 + w7) / 2.0) * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7)
                + ln1myd * (-8.0 * log((w4 - yd) / (1.0 + w4)) * log((1.0 + yd) / 2.0) + 8.0 * log((w5 - yd) / (1.0 + w5)) * ln1pyd - 4.0 * log((w7 - yd) / (-1.0 + w7)) * ln1pyd
                    + ln1pyd * (8.0 * log((w4 + yd) / (-1.0 + w4)) + 8.0 * log((w5 + yd) / (1.0 + w5)) - 4.0 * log((w7 + yd) / (-1.0 + w7))) + log((1.0 + yd) / 2.0) * (-8.0 * log((w5 - yd) / (1.0 + w5)) + 4.0 * log((w7 - yd) / (1.0 + w7))
                    - 8.0 * log((w4 + yd) / (-1.0 + w4)) - 8.0 * log((w5 + yd) / (-1.0 + w5)) + 4.0 * log((w7 + yd) / (-1.0 + w7))) - (16.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w4)
                    - (16.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(imydinv) * T(1.0, 1.0 - yd, 1.0 + w7) - (16.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * 
                    T(1.0, (1.0 + yd) / 2.0, (1.0 - w4) / 2.0) - (16.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w4) / 2.0) - (16.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * 
                    T(1.0, (1.0 + yd) / 2.0, (1.0 - w5) / 2.0) - (16.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w5) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 - w7) / 2.0)
                    + (8.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 - yd))) * T(1.0, (1.0 + yd) / 2.0, (1.0 + w7) / 2.0) + (16.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 + yd, 1.0 - w4)
                    + (16.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 + yd, 1.0 + w5) - (8.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 + yd, 1.0 - w7))
                + dilog((1.0 - w7) / 2.0) * (-8.0 * log(w7 / (-1.0 + w7)) + 4.0 * log((w7 - yd) / (-1.0 + w7)) + 4.0 * log((w7 + yd) / (-1.0 + w7)) - (8.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) - (8.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7))
                - (8.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * log(1.0 + w7) * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7) + (4.0 * 1.0i) * M_PI * power_of<2>(log(1.0 + w7)) * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7)
                + dilog((1.0 + w7) / 2.0) * (-8.0 * log(w7 / (1.0 + w7)) + 4.0 * log((w7 - yd) / (1.0 + w7)) + 4.0 * log((w7 + yd) / (1.0 + w7)) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) - (8.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7))
                + li2half * (-16.0 * log(w4 / (-1.0 + w4)) - 16.0 * log(w4 / (1.0 + w4)) - 16.0 * log(w5 / (-1.0 + w5)) - 16.0 * log(w5 / (1.0 + w5)) + 8.0 * log(w7 / (-1.0 + w7)) + 8.0 * log(w7 / (1.0 + w7)) - 16.0 * log(-1.0i / (-1.0i + wx3))
                    - 16.0 * log(1.0i / (1.0i + wx3)) - 16.0 * log(-1.0i / (-1.0i + wx4)) - 16.0 * log(1.0i / (1.0i + wx4)) + 8.0 * log((w4 - yd) / (-1.0 + w4)) + 8.0 * log((w4 - yd) / (1.0 + w4)) + 8.0 * log((w5 - yd) / (-1.0 + w5)) + 8.0 * log((w5 - yd) / (1.0 + w5))
                    - 4.0 * log((w7 - yd) / (-1.0 + w7)) - 4.0 * log((w7 - yd) / (1.0 + w7)) + 8.0 * log((w4 + yd) / (-1.0 + w4)) + 8.0 * log((w4 + yd) / (1.0 + w4)) + 8.0 * log((w5 + yd) / (-1.0 + w5)) + 8.0 * log((w5 + yd) / (1.0 + w5))
                    - 4.0 * log((w7 + yd) / (-1.0 + w7)) - 4.0 * log((w7 + yd) / (1.0 + w7)) - (16.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 - yd, 1.0 - w4) - (16.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 - yd, 1.0 + w4) - (16.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 - yd, 1.0 - w5)
                    - (16.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 - yd, 1.0 + w5) + (8.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 - yd, 1.0 - w7) + (8.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 - yd, 1.0 + w7) - (16.0 * 1.0i) * M_PI * my_sign(imag(w4)) * T(1.0, 1.0 + yd, 1.0 - w4)
                    - (16.0 * 1.0i) * M_PI * my_sign(-imag(w4)) * T(1.0, 1.0 + yd, 1.0 + w4) - (16.0 * 1.0i) * M_PI * my_sign(imag(w5)) * T(1.0, 1.0 + yd, 1.0 - w5) - (16.0 * 1.0i) * M_PI * my_sign(-imag(w5)) * T(1.0, 1.0 + yd, 1.0 + w5) + (8.0 * 1.0i) * M_PI * my_sign(imag(w7)) * T(1.0, 1.0 + yd, 1.0 - w7)
                    + (8.0 * 1.0i) * M_PI * my_sign(-imag(w7)) * T(1.0, 1.0 + yd, 1.0 + w7)) + ln1pyd * ((-16.0 * 1.0i) * M_PI * log((1.0 - w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w4) / 2.0)
                    - (16.0 * 1.0i) * M_PI * log((1.0 + w4) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w4) / 2.0) - (16.0 * 1.0i) * M_PI * log((1.0 - w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w5) / 2.0)
                    - (16.0 * 1.0i) * M_PI * log((1.0 + w5) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w5) / 2.0) + (8.0 * 1.0i) * M_PI * log((1.0 - w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 - w7) / 2.0)
                    + (8.0 * 1.0i) * M_PI * log((1.0 + w7) / 2.0) * my_sign(2.0 * imag(1.0 / (1.0 + yd))) * T(1.0, (1.0 - yd) / 2.0, (1.0 + w7) / 2.0) + (16.0 * 1.0i) * M_PI * log(1.0 - w4) * my_sign(imag(w4inv)) * T(1.0, 1.0 - yd, 1.0 - w4)
                    + (16.0 * 1.0i) * M_PI * log(1.0 + w5) * my_sign(-imag(w5inv)) * T(1.0, 1.0 - yd, 1.0 + w5) - (8.0 * 1.0i) * M_PI * log(1.0 - w7) * my_sign(imag(w7inv)) * T(1.0, 1.0 - yd, 1.0 - w7) - (16.0 * 1.0i) * M_PI * log(1.0 + w4) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 + w4)
                    - (16.0 * 1.0i) * M_PI * log(1.0 - w5) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 - w5) + (8.0 * 1.0i) * M_PI * log(1.0 + w7) * my_sign(-imydinv) * T(1.0, 1.0 + yd, 1.0 + w7)) - 32.0 * pisqu * log((1.0i + wx3) / wx3) * T(1.0, (wx3 + xd) / wx3, (1.0i + wx3) / wx3) * 
                T(p(-1.0i / wx3, (wx3 + xd) / wx3), (wx3 + xd) / wx3, (-1.0i + wx3) / wx3) - 32.0 * pisqu * log((1.0i + wx3) / wx3) * T(1.0, 1.0 - xd / wx3, (1.0i + wx3) / wx3) * T(p(-1.0i / wx3, 1.0 - xd / wx3), 1.0 - xd / wx3, (-1.0i + wx3) / wx3)
                - 32.0 * pisqu * log((-1.0i + wx3) / wx3) * T(1.0, (wx3 + xd) / wx3, (-1.0i + wx3) / wx3) * T(p(1.0i / wx3, (wx3 + xd) / wx3), (wx3 + xd) / wx3, (1.0i + wx3) / wx3) - 32.0 * pisqu * log((-1.0i + wx3) / wx3) * T(1.0, 1.0 - xd / wx3, (-1.0i + wx3) / wx3) * 
                T(p(1.0i / wx3, 1.0 - xd / wx3), 1.0 - xd / wx3, (1.0i + wx3) / wx3) - 32.0 * pisqu * log((1.0i + wx4) / wx4) * T(1.0, (wx4 + xd) / wx4, (1.0i + wx4) / wx4) * T(p(-1.0i / wx4, (wx4 + xd) / wx4), (wx4 + xd) / wx4, (-1.0i + wx4) / wx4)
                - 32.0 * pisqu * log((1.0i + wx4) / wx4) * T(1.0, 1.0 - xd / wx4, (1.0i + wx4) / wx4) * T(p(-1.0i / wx4, 1.0 - xd / wx4), 1.0 - xd / wx4, (-1.0i + wx4) / wx4) - 32.0 * pisqu * log((-1.0i + wx4) / wx4) * T(1.0, (wx4 + xd) / wx4, (-1.0i + wx4) / wx4) * 
                T(p(1.0i / wx4, (wx4 + xd) / wx4), (wx4 + xd) / wx4, (1.0i + wx4) / wx4) - 32.0 * pisqu * log((-1.0i + wx4) / wx4) * T(1.0, 1.0 - xd / wx4, (-1.0i + wx4) / wx4) * T(p(1.0i / wx4, 1.0 - xd / wx4), 1.0 - xd / wx4, (1.0i + wx4) / wx4) - 40.0 * zeta3);

            return resultpart1 + resultpart2;
        }
	}
}