/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2022 Viktor Kuschke
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <vector>
#include <test/test.hh>
#include <eos/maths/multiplepolylog-li22.hh>

using namespace test;
using namespace eos;

class MultiplePolylogarithmTest :
    public TestCase
{
    public:
        MultiplePolylogarithmTest() :
            TestCase("multiplepolylogarithm_test")
        {
        }

        void test_region(const std::vector<std::vector<complex<double>>> & input, const std::vector<complex<double>> & reference, const double & eps) const
        {
            // check that the C implementation uses the principal branch of the (complex) logarithm
            TEST_CHECK_RELATIVE_ERROR(+M_PI, imag(std::log(complex<double>(-1.0, 0.0))), eps);

            if ( input.size() != reference.size())
                TEST_CHECK_FAILED("Number of reference values does not match number of complex points");

            if ( input[0].size() != 2)
                TEST_CHECK_FAILED("The array of complex points does not have the correct number of columns");

            for (unsigned int i = 0; i < reference.size(); i++)
            {
                complex<double> x = input[i][0];
                complex<double> y = input[i][1];
                complex<double> li22_reference = reference[i];
                complex<double> li22_value = li22(x, y);

                TEST_CHECK_NEARLY_EQUAL(real(li22_reference),   real(li22_value),     eps);
                TEST_CHECK_NEARLY_EQUAL(imag(li22_reference),   imag(li22_value),     eps);
            }
        }

        virtual void run() const
        {
            // Testing region A
            {
                const std::vector<std::vector<complex<double>>> x_y_points = {
                    {{0.0, 0.0}, {0.0, 0.0}},                   {{0.0, 0.0}, {0.3, 0.0}},
                    {{0.0, 0.0}, {1.5, 0.0}},                   {{0.0, 0.0}, {5.0, 0.0}},
                    {{0.0, 0.0}, {50, 0.0}},                    {{0.0, 0.0}, {800, 0.0}},
                    {{0.16, 0.0}, {0.0, 0.0}},                  {{0.16, 0.0}, {0.3, 0.0}},
                    {{0.16, 0.0}, {1.5, 0.0}},                  {{0.16, 0.0}, {4.0, 0.0}},
                    {{0.35, 0.0}, {1.2, 0.0}},                  {{0.35, 0.0}, {1.5, 0.0}},
                    {{0.35, 0.0}, {1.7, 0.0}},                  {{0.35, 0.0}, {1.8, 0.0}},
                    {{0.5, 0.0}, {1.2, 0.0}},                   {{0.5, 0.0}, {1.8, 0.0}},
                    {{0.0003, 0.0}, {1500, 0.0}},               {{0.0, 0.0}, {-0.28935, 0.07924}},
                    {{0.0, 0.0}, {0.029333,-1.4997}},           {{0.0, 0.0}, {-4.7215, 1.6454}},
                    {{0.0, 0.0}, {-35.449,35.261}},             {{0.0, 0.0}, {-328.08, -729.63}},
                    {{0.16, 0.0}, {0.0,0.0}},                   {{0.16, 0.0}, {-0.28947, -0.07879}},
                    {{0.16, 0.0}, {-0.060793,-1.4988}},         {{0.16, 0.0}, {3.3019, 2.2577}},
                    {{0.35, 0.0}, {0.28478,1.1657}},            {{0.35, 0.0}, {1.4939, 0.13548}},
                    {{0.35, 0.0}, {0.2133,-1.6866}},            {{0.35, 0.0}, {-0.29534, 1.7756}},
                    {{0.5, 0.0}, {-0.32722,1.1545}},            {{0.5, 0.0}, {-1.7027, 0.5838}},
                    {{0.0003, 0.0}, {-1187.14, - 916.893}},     {{0.000175, 0.000244}, {1251.04, - 827.592}},
                    {{-0.15834, 0.022961}, {0.0, 0.0}},         {{0.020353, 0.1587}, {0.28868, -0.081619}},
                    {{0.074773, 0.14145}, {1.2915, -0.76287}},  {{0.1553, -0.0385}, {3.1237, 2.4985}},
                    {{0.2491, -0.24586}, {-0.50285, -1.0896}},  {{0.33194, -0.11099}, {1.1014, 1.0183}},
                    {{0.22294, 0.26981}, {0.41431, 1.6487}},    {{0.031092, 0.34862}, {-0.24654, 1.783}},
                    {{-0.33074, 0.37498}, {1.1671, -0.27924}},  {{-0.020469, -0.49958}, {-1.5476, -0.91925}}
                };

                const std::vector<complex<double>> li22_values = {
                    {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0.002081645530457917, 0.0}, {0.010651696273675876, 0.0}, {0.030039829442010797, 0.0}, {0.04635897385686277, 0.0}, {0.05884713061848893, 0.0}, {0.06741927429845772, 0.0}, {0.07178642142151906, 0.0}, {0.10727377899617292, 0.0}, {0.1701767184161538, 0.0}, {3.567098698155471E-5, 0.0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {-0.0019879538929080446, -5.380986738117047E-4}, {-6.968004429581845E-4, -0.010303480373845512}, {0.023354855690842665, 0.01781048747466586}, {0.008386863823227531, 0.043462760293781426}, {0.05854320883808645, 0.005731913423855528}, {0.0035178299296248893, -0.06216402156165925}, {-0.014918156554264827, 0.06286811835616828}, {-0.03208004722066546, 0.09067923334907338}, {-0.12917467990172696, 0.04050101707796992}, {-2.6235017138206976E-5, -1.9159369420803295E-5}, {8.327568547787913E-6, 3.4570077614650976E-5}, {0, 0}, {-0.0017375127489347817, 8.488891936263332E-4}, {-0.0014781071218621642, 0.0100226570597535}, {0.028950758006117468, 0.006141292171766552}, {-0.03317905494549053, 0.020919980537159315}, {0.057409210598555145, 0.003844704121116567}, {-0.05422570788903269, -0.0070197437565258}, {0.005619024913061165, -0.051239352427759365}, {-0.010948576857822117, -0.06164866503987189}, {0.09746578054527927, 0.0322617807345509}
                };

                const double epsilon = 1e-12;
                test_region(x_y_points, li22_values, epsilon);
            }

            // Testing region B
            {
                const std::vector<std::vector<complex<double>>> x_y_points = {
                    {{0.3, 0.0}, {0.0, 0.0}},       {{1.0, 0.0}, {0.0, 0.0}},
                    {{7.3, 0.0}, {0.0, 0.0}},       {{89.0, 0.0}, {0.0, 0.0}},
                    {{929.0, 0.0}, {0.0, 0.0}},
                    {{0.3, 0.0}, {0.2, 0.0}},       {{0.67, 0.0}, {0.2, 0.0}},
                    {{1.05, 0.0}, {0.2, 0.0}},      {{1.3, 0.0}, {0.2, 0.0}},
                    {{2.8, 0.0}, {0.2, 0.0}},       {{3.3, 0.0}, {0.2, 0.0}},
                    {{0.3, 0.0}, {0.45, 0.0}},      {{0.67, 0.0}, {0.45, 0.0}},
                    {{1.05, 0.0}, {0.45, 0.0}},     {{1.3, 0.0}, {0.45, 0.0}},
                    {{0.28, 0.0}, {0.73, 0.0}},     {{0.48, 0.0}, {0.73, 0.0}},
                    {{0.73, 0.0}, {0.73, 0.0}},     {{0.94, 0.0}, {0.73, 0.0}},
                    {{0.31, 0.0}, {1.08, 0.0}},     {{0.42, 0.0}, {1.08, 0.0}},
                    {{0.56, 0.0}, {1.08, 0.0}},     {{0.61, 0.0}, {1.08, 0.0}},

                    {{0.3,0.},{-0.141,-0.142}},{{0.67,0.},{-0.152,-0.13}},{{1.05,0.},{0.199,-0.0154}},
                    {{1.3,0.},{0.0593,0.191}},{{2.8,0.},{0.0372,0.197}},{{3.3,0.},{0.000449,-0.2}},{{0.3,0.},{0.336,-0.299}},{{0.67,0.},{0.203,-0.402}},{{1.05,0.},{0.45,-0.00929}},{{1.3,0.},{0.45,0.00725}},
                    {{0.28,0.},{-0.655,-0.321}},{{0.48,0.},{0.275,0.676}},{{0.73,0.},{0.401,0.61}},{{0.94,0.},{-0.404,-0.608}},{{0.31,0.},{-0.15,1.07}},{{0.42,0.},{0.757,-0.771}},{{0.56,0.},{-0.469,0.973}},{{0.61,0.},{-1.07,0.117}},

                    {{-0.081,-0.289},{0.,0.}},{{-0.399,-0.917},{0.,0.}},{{-7.08,-1.76},{0.,0.}},{{61.8,-64.},{0.,0.}},{{899.,-233.},{0.,0.}},{{-0.0979,0.284},{0.0406,0.196}},{{-0.204,0.638},{-0.161,0.118}},{{-0.0793,-1.05},{-0.108,0.168}},
                    {{1.19,-0.521},{0.0728,0.186}},{{2.76,-0.465},{0.0872,0.18}},{{3.29,0.273},{0.183,0.08}},{{-0.0266,-0.299},{0.445,0.0642}},{{0.605,-0.287},{0.0391,0.448}},{{-0.0645,1.05},{-0.438,-0.102}},{{1.13,-0.637},{-0.27,-0.36}},
                    {{0.249,-0.129},{0.206,0.7}},{{0.468,0.106},{0.00499,-0.73}},{{-0.168,-0.71},{0.665,-0.3}},{{0.463,-0.818},{-0.461,0.566}},{{-0.0226,0.309},{0.873,0.636}},{{-0.36,-0.216},{-1.06,0.199}},{{-0.43,-0.359},{0.549,-0.93}},{{0.513,0.329},{-0.721,0.804}}

                };

                const std::vector<complex<double>> li22_values = {
                    {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0.00526276935063225, 0.0}, {0.03434245237309184, 0.0}, {0.16362368523104376, -0.032342409993916854}, {0.19361371318058962, -0.17391809914171896}, {-0.11651570511012121, -0.6825222643754504}, {-0.2572715751715332, -0.7914367707841685}, {0.011948733860527645, 0.0}, {0.07907798665721592, 0.0}, {0.3867526110902215, -0.07884647074685154}, {0.4525379945735788, -0.42398906942632786}, {0.016841121219089022, 0.0}, {0.057100914992508425, 0.0}, {0.1672733602085725, 0.0}, {0.38600998075669657, 0.0}, {0.0315959003992087, 0.0}, {0.06306554790669047, 0.0}, {0.12674432993356427, 0.0}, {0.15803099838966947, 0.0},
                    {-0.0036841293098482, -0.0036744474654907275}, {-0.025556443992496857, -0.0213993562040376}, {0.16010611954555218, -0.04521080728605774}, {0.21290350565946398, 0.1409352250065981}, {0.6541702440649971, -0.1792857755726549}, {-0.7111332498233617, 0.23960785589682787}, {0.008796860175557003, -0.008000316859635093}, {0.03232330587500785, -0.07008806328569803}, {0.3848413874097143, -0.08759620171999077}, {0.4604680478351079, -0.41616145187705256}, {-0.014510729596664959, -0.006938050849528761}, {0.018734882667306395, 0.05201733301462775}, {0.07918169860051713, 0.1389541489910755}, {-0.19736926221647497, -0.2572508787579554},
                    {-0.005290352158253035, 0.029614344630722867}, {0.04124809518865999, -0.04585277283273988}, {-0.054638102658826335, 0.09808341056096191}, {-0.13252659274946565, 0.013552140075921811}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0, 0}, {0.0023177426318715952, -0.003571212427864303}, {0.019198731167252975, 0.004110192013473406}, {0.0020310081843759598, -0.048574834972416325}, {0.13680238397850988, -0.03966850177990916}, {0.4394709560941469, -0.4008789186548645}, {-0.5623869962281389, 0.5032974453942372}, {-0.009728183693082813, 0.0017472043885830818}, {0.06409183779350557, 0.028149842222190687}, {0.08388852172586203, 0.06971139685691502},
                    {-0.18944945257890394, 0.18645968278371117}, {0.015778735810255705, 0.004488300664933672}, {0.02513315446103679, -0.04810034717899023}, {-0.030427966474235783, 0.07671618746393245}, {0.17535580657503685, -0.038854310932935655}, {-0.01468625167649329, -0.01985318638178056}, {-0.02883251274475478, -0.031995071458599283}, {0.06496097639663402, 0.01660737003391211}, {-0.1025876271596167, -0.06049582452368328}
                };

                const double epsilon = 1e-12;
                test_region(x_y_points, li22_values, epsilon);
            }

            // Testing region C
            {
                const std::vector<std::vector<complex<double>>> x_y_points= {
                    {{1.9, 0.0}, {0.8, 0.0}},           {{2.4, 0.0}, {0.8, 0.0}},
                    {{2.9, 0.0}, {0.8, 0.0}},           {{3.5, 0.0}, {0.8, 0.0}},
                    {{2.1, 0.0}, {0.7, 0.0}},           {{2.6, 0.0}, {0.7, 0.0}},
                    {{3.1, 0.0}, {0.7, 0.0}},           {{3.5, 0.0}, {0.7, 0.0}},
                    {{15.3, 0.0}, {0.46, 0.0}},         {{92.0, 0.0}, {0.46, 0.0}},
                    {{485.0, 0.0}, {0.46, 0.0}},        {{2934.0, 0.0}, {0.46, 0.0}},
                    {{2.1, 0.0}, {0.7, 0.0}},           {{2.6, 0.0}, {0.7, 0.0}},
                    {{3.1, 0.0}, {0.7, 0.0}},           {{3.5, 0.0}, {0.7, 0.0}},
                    {{66.8191, 0.0}, {1714.38, 0.0}},   {{854.501, 0.0}, {2284.28, 0.0}},
                    {{3968.69, 0.0}, {4464.67, 0.0}},   {{4893.33, 0.0}, {3772.2, 0.0}},
                    {{1862.48, 0.0}, {768.447, 0.0}},   {{3751.05, 0.0}, {1422.75, 0.0}},
                    {{3528.37, 0.0}, {1204.62, 0.0}},   {{3644.34, 0.0}, {1397.47, 0.0}},
                    {{3376.79, 0.0}, {4309.79, 0.0}},   {{2185.94, 0.0}, {4821.93, 0.0}},
                    {{39.7149, 0.0}, {495.591, 0.0}},   {{111.063, 0.0}, {1696.5, 0.0}},
                    {{179.057, 0.0}, {2575.51, 0.0}},   {{331.342, 0.0}, {1602.03, 0.0}},
                    {{141.788, 0.0}, {1215.02, 0.0}},   {{289.046, 0.0}, {3751.41, 0.0}},
                    {{261.518, 0.0}, {2605.56, 0.0}},   {{229.026, 0.0}, {2187.99, 0.0}},
                    {{106.095, 0.0}, {3946.89, 0.0}},   {{199.371, 0.0}, {3706.25, 0.0}},
                    {{1657.03, 0.0}, {290.394, 0.0}},   {{1241.87, 0.0}, {341.499, 0.0}},
                    {{3503.81, 0.0}, {25.8186, 0.0}},   {{583.105, 0.0}, {327.216, 0.0}},
                    {{3546.98, 0.0}, {320.056, 0.0}},   {{4888.59, 0.0}, {55.0363, 0.0}},
                    {{2149.24, 0.0}, {390.111, 0.0}},   {{741.874, 0.0}, {94.1922, 0.0}},
                    {{4857.6, 0.0}, {357.294, 0.0}},    {{3701.55, 0.0}, {193.039, 0.0}},

                    {{1.90,0.0},{0.77,-0.21}},{{2.40,0.0},{0.58,-0.55}},{{2.90,0.0},{0.08,-0.80}},{{3.50,0.0},{0.45,-0.66}},{{2.10,0.0},{-0.15,-0.68}},{{2.60,0.0},{-0.14,0.69}},{{3.10,0.0},{0.63,-0.30}},{{3.50,0.0},{0.65,-0.26}},{{15.30,0.0},{0.16,0.43}},{{92.00,0.0},{-0.01,0.46}},{{485.00,0.0},{0.37,-0.28}},{{2934.00,0.0},{0.27,0.37}},{{2.10,0.0},{0.04,0.70}},{{2.60,0.0},{0.62,0.32}},{{3.10,0.0},{-0.58,-0.39}},{{3.50,0.0},{0.48,-0.51}},{{66.82,0.0},{1425.60,-952.23}},{{854.50,0.0},{1755.40,1461.60}},{{3968.70,0.0},{-1548.20,-4187.60}},{{4893.30,0.0},{-2259.80,3020.40}},{{1862.50,0.0},{-694.93,328.01}},{{3751.10,0.0},{529.09,1320.70}},{{3528.40,0.0},{-794.43,905.54}},
                    {{3644.30,0.0},{716.84,-1199.60}},{{3376.80,0.0},{-4232.30,-813.43}},{{2185.90,0.0},{-4256.90,-2264.90}},{{39.71,0.0},{-14.04,-495.39}},{{111.06,0.0},{584.93,1592.50}},{{179.06,0.0},{2433.50,-843.51}},{{331.34,0.0},{-1533.40,-463.78}},{{141.79,0.0},{932.45,778.98}},{{289.05,0.0},{3166.70,2011.30}},{{261.52,0.0},{932.77,-2432.90}},{{229.03,0.0},{2041.70,-786.50}},{{106.10,0.0},{1161.40,-3772.10}},{{199.37,0.0},{3188.80,1888.90}},{{1657.00,0.0},{71.45,-281.47}},{{1241.90,0.0},{-12.61,341.27}},{{3503.80,0.0},{-3.31,-25.61}},{{583.11,0.0},{-303.35,-122.68}},{{3547.00,0.0},{279.96,155.10}},{{4888.60,0.0},{35.28,42.24}},{{2149.20,0.0},{-4.79,390.08}},{{741.87,0.0},{-93.78,8.79}},{{4857.60,0.0},{-218.33,-282.83}},{{3701.60,0.0},{-49.18,186.67}},

                    {{-1.11,-1.54},{0.33,0.73}},{{-1.83,1.55},{-0.62,0.50}},{{-1.98,-2.12},{-0.22,0.77}},{{-1.82,2.99},{-0.39,0.70}},{{0.53,2.03},{0.68,0.16}},{{-2.57,0.36},{-0.53,0.45}},{{0.30,-3.09},{-0.69,-0.08}},{{-2.75,2.16},{-0.25,0.65}},{{-14.64,4.46},{0.25,0.39}},{{64.95,65.16},{0.21,0.41}},{{134.15,466.08},{-0.30,-0.35}},{{1375.30,2591.70},{-0.46,-0.05}},{{-1.44,1.53},{-0.66,0.24}},{{-2.58,-0.31},{-0.45,-0.53}},{{-2.21,-2.17},{0.69,-0.14}},{{-0.11,3.50},{-0.49,-0.50}},{{-66.48,-6.68},{796.92,-1517.90}},{{-854.49,-3.77},{-1412.30,-1795.40}},{{-2480.60,-3098.00},{3887.20,-2196.20}},{{4506.50,-1906.80},{683.17,3709.80}},{{-1135.20,1476.50},{580.36,-503.68}},{{-749.68,-3675.40},{1358.80,-421.65}},{{-597.68,3477.40},{-398.84,-1136.70}},{{-3537.80,874.71},{-470.27,1316.00}},
                    {{1694.50,-2920.90},{-4308.60,100.98}},{{2008.40,-863.01},{-2330.10,-4221.60}},{{2.88,-39.61},{92.61,486.86}},{{47.73,-100.28},{1662.30,338.78}},{{179.06,-0.73},{-972.48,2384.90}},{{315.95,-99.82},{1567.20,-332.23}},{{106.23,-93.91},{-950.09,-757.36}},{{-75.80,278.93},{-902.76,3641.20}},{{-50.13,-256.67},{-2026.00,1638.40}},{{-163.02,160.87},{-555.32,2116.30}},{{-38.01,99.05},{-3476.30,1869.00}},{{173.58,98.08},{235.67,3698.70}},{{-456.38,-1592.90},{201.37,-209.23}},{{1140.10,-492.33},{236.84,246.02}},{{-3480.10,407.12},{-23.94,9.68}},{{-538.35,-224.03},{272.41,-181.28}},{{-3003.70,1886.50},{-170.85,-270.64}},{{-2902.60,3933.60},{-36.88,40.86}},{{-2117.70,-366.56},{254.60,-295.57}},{{272.56,689.99},{-60.81,71.93}},{{4335.50,-2190.80},{-279.84,-222.15}},{{2806.50,2413.60},{185.04,-54.98}}

                };

                const std::vector<complex<double>> li22_values = {
                    {0.1523595924359218, -2.0531470793335234}, {-0.6560667548170098, -2.590009465543745}, {-1.4823623486645983, -2.8863113904296807}, {-2.436151109172794, -3.037705433119426}, {-0.04807422208794687, -2.000761542352614}, {-0.7480281710993575, -2.428199378702683}, {-1.4624263752167321, -2.6771330233078285}, {-2.0215492800774206, -2.785088627993278}, {-8.767087991009326, 0.3394674288518812}, {-20.80916429571243, 23.33133958387653}, {-20.841567260205114, 78.47312024750467}, {16.07745122088063, 191.68865150306652}, {-0.04807422208794687, -2.0007615423526146}, {-0.748028171099357, -2.4281993787026823}, {-1.4624263752167321, -2.6771330233078285}, {-2.0215492800774197, -2.7850886279932783}, {29.35236520998945, 535.0747054064823}, {652.0120319364809, 1268.997341242691}, {1522.6681688929118, 2036.697764399904}, {1596.1493397566317, 2074.700882646159}, {741.6285202761171, 1268.7357796132176}, {1171.1906908190379, 1670.6648626728984}, {1099.8757129652745, 1601.3102703190561}, {1152.7373548981977, 1654.7494045322962}, {1418.5305386494538, 1955.8572743634224}, {1211.4263404896014, 1807.0203273653804}, {-40.270252332593564, 317.74079026132364}, {103.95886662593819, 646.5153636198077}, {219.02715872807454, 826.8280284773725}, {323.9156020998913, 912.8736070458535}, {128.66186269125342, 656.9784552641069}, {369.9022076258442, 1028.3577921116173}, {309.27105898046864, 932.9573032855935}, {262.5264135590478, 866.2625259390578}, {135.34500185844797, 753.3618388217735}, {270.85590302273954, 916.6829460580045}, {536.8432599819224, 1018.6200138408511}, {481.9445339584843, 974.9017871882245}, {356.03615329247145, 726.1045786607297}, {297.5122660834518, 779.7068008575457}, {796.0232298595641, 1258.8915461528768}, {550.1437637356299, 947.8340769954975}, {665.0860851985274, 1155.6243661583658}, {224.0665219409688, 630.382174099152}, {938.3911240663311, 1385.503404328354}, {706.7164976519506, 1151.5354306916277},
                    {-0.3854555139230902, -1.8498939657689542}, {-1.8002559048184628, -1.2224557029007808}, {-2.3193691328010333, 0.5396948003864614}, {-3.037280010400048, -0.10409986960521467}, {-1.4438266223376655, 0.23720407921641856}, {2.0084193131133574, 0.6876644670110017}, {-2.1266906806230583, -1.6086077158310892}, {-2.548757620887706, -1.6999574298252016}, {4.194953814591382, -7.568174452108512}, {19.39343952865623, -21.62724203550637}, {5.314060490879275, 73.57027278174405}, {102.64233610914356, -165.161682522784}, {1.6635988679399238, 0.4726615617744052}, {1.3087434495451111, -2.4994747964727115}, {-0.7628539092927369, 1.9951072161499415}, {-2.7313855508884872, -0.5391674791059078}, {96.57101412118573, 517.3572574840713}, {1606.5105978353272, 56.81264794039403}, {2074.030435308936, 1408.7146719466475}, {2638.5348110476343, 565.8803269024672}, {1426.22583061779, 443.06581641982143}, {2220.03484825821, 7.3524087292834395}, {1964.3738140218925, 397.0762602431369}, {1427.5885197394853, 1374.983304139224}, {2195.1996435180863, 1004.5732063763055}, {1881.4807539330782, 1064.0310882733452}, {87.8049267106073, 293.1966296494144}, {685.3883426554253, 231.0890531230407}, {271.842005979277, 802.8728204996557}, {773.4427181001327, 570.065606746879}, {763.8376556901992, 126.01076281315581}, {1228.2188791615338, 176.26881137030148}, {512.3464140972504, 814.0344630443113}, {323.29297129499486, 835.9407374749193}, {313.9519841665801, 678.908374484112}, {1073.200660944101, 192.60459884970186}, {791.0236066228243, 791.405059110945}, {1175.8098472610077, 65.50289086923932}, {616.9582390685748, 458.8002940348848}, {700.0132852622874, 429.130796966624}, {1695.9494715613698, -388.0480408121699}, {1207.4564812861126, -401.4023803682603}, {1438.6947381544385, 39.85339403824487}, {622.3109063723322, 233.97573949695425}, {1436.0081000097243, 787.6488625311655}, {1436.7860939025534, 10.62045494656553},
                    {-0.3773509199925824, 0.2709495823603243}, {0.23129755160017096, 0.5770930065587319}, {-0.9549211847907926, 0.26939939364079324}, {0.9666425217984851, 0.4044214649186202}, {-0.4800843553473433, -0.32092834946870585}, {-0.2920848338661657, 0.5650815611120219}, {0.9207389543209072, -0.33097622839822627}, {0.6287838147824258, 0.6258939923739071}, {3.104152828092248, 1.6114339412561325}, {10.210000372313942, -18.779198537064808}, {34.3779518941172, 65.3227263773317}, {145.7033223250194, 77.9277956303733}, {0.10305418087198337, 0.45982786375732054}, {-0.20951497593125845, -0.5840574072275881}, {0.43814966188987015, 0.5694266473830076}, {0.193206733990833, 1.3469614969281372}, {363.12492688539635, -71.66178496469905}, {1113.0542325617853, -510.76280882496144}, {2199.862122867428, 408.4637064417182}, {2738.3304158604215, -95.60699801636154}, {1140.8750740248893, -567.7510253928972}, {1689.5279730477278, 659.0141531684283}, {1784.9403052903085, 726.9410151220814}, {1719.5646896033675, 508.12632688600314}, {2283.972638834786, 207.79519038389225}, {1830.165768453327, 991.1306944819166}, {65.2886283909296, 238.2124177367126}, {300.91404029275725, 454.8957954535365}, {815.9254122722369, 374.29996435580756}, {436.72758763127905, 812.8381663004362}, {509.7004965563895, 301.89019929775236}, {882.5641298501748, -159.34153979193576}, {854.7270170939247, -163.09334743192701}, {715.3600018413871, 32.10296991775602}, {600.3125633109518, -17.64346811030598}, {609.4754401863956, -620.6283103223301}, {951.7043444035942, 269.8219271436765}, {1192.7742776115088, -293.771637654363}, {570.5165037000295, 502.491019424176}, {669.26332434349, 12.039946091204001}, {1090.6409962021116, -819.3701093670722}, {955.0231254518608, 261.8105098816078}, {1126.0617544853906, -146.8843595977826}, {579.383893267656, -99.80260526795213}, {1524.3649301279431, 489.5707668817098}, {833.5700270021003, -977.20479885937}
                };

                const double epsilon = 5e-11;
                test_region(x_y_points, li22_values, epsilon);
            }

            // Testing region D
            {
                const std::vector<std::vector<complex<double>>> x_y_points= {
                    {{1.84,0.0},{4.63,0.0}},{{2.69,0.0},{9.8,0.0}},{{1.74,0.0},{2.11,0.0}},{{2.53,0.0},{3.28,0.0}},{{2.23,0.0},{4.97,0.0}},{{3.11,0.0},{8.32,0.0}},{{2.21,0.0},{5.99,0.0}},{{3.14,0.0},{1.53,0.0}},{{1.91,0.0},{0.874,0.0}},{{1.98,0.0},{2.13,0.0}},{{2.74,0.0},{2.18,0.0}},{{2.2,0.0},{4.18,0.0}},{{1.97,0.0},{4.45,0.0}},{{1.85,0.0},{8.15,0.0}},{{3.31,0.0},{1.76,0.0}},{{3.47,0.0},{1.73,0.0}},{{2.34,0.0},{1.66,0.0}},{{2.32,0.0},{9.14,0.0}},{{2.37,0.0},{8.66,0.0}},{{3.12,0.0},{4.18,0.0}},{{3.37,0.0},{1448.60,0.0}},{{1.99,0.0},{2655.90,0.0}},{{3.32,0.0},{1830.00,0.0}},{{3.21,0.0},{67.03,0.0}},{{3.12,0.0},{969.08,0.0}},{{3.02,0.0},{357.84,0.0}},
                    {{2.79,0.0},{4898.00,0.0}},{{2.23,0.0},{2809.20,0.0}},{{2.26,0.0},{493.86,0.0}},{{1.84,0.0},{3100.40,0.0}},{{1.69,0.0},{1786.20,0.0}},{{1.73,0.0},{3169.30,0.0}},{{2.12,0.0},{1550.60,0.0}},{{2.03,0.0},{817.70,0.0}},{{2.94,0.0},{2205.90,0.0}},{{1.73,0.0},{4076.30,0.0}},{{3.34,0.0},{888.26,0.0}},{{3.03,0.0},{1944.90,0.0}},{{2.97,0.0},{1715.00,0.0}},{{1.78,0.0},{431.59,0.0}},
                    {{1.84,0.0},{-3.87,2.55}},{{2.69,0.0},{4.11,-8.89}},{{1.74,0.0},{-1.35,-1.62}},{{2.53,0.0},{2.11,-2.51}},{{2.23,0.0},{-4.83,1.15}},{{3.11,0.0},{-3.42,7.59}},{{2.21,0.0},{-5.92,-0.94}},{{3.14,0.0},{0.82,-1.29}},{{1.91,0.0},{-0.01,0.87}},{{1.98,0.0},{1.25,1.72}},{{2.74,0.0},{-1.77,1.27}},{{2.20,0.0},{2.26,3.51}},{{1.97,0.0},{2.18,3.88}},{{1.85,0.0},{7.77,2.45}},{{3.31,0.0},{1.74,0.23}},{{3.47,0.0},{-1.73,0.0}},{{2.34,0.0},{0.14,-1.65}},{{2.32,0.0},{2.90,8.67}},{{2.37,0.0},{-4.80,-7.20}},{{3.12,0.0},{-0.16,-4.18}},{{3.37,0.0},{-45.45,1447.90}},{{1.99,0.0},{343.47,-2633.60}},{{3.32,0.0},{1818.80,-202.56}},{{3.21,0.0},{65.65,-13.55}},
                    {{3.12,0.0},{839.18,484.67}},{{3.02,0.0},{-308.79,180.83}},{{2.79,0.0},{-3041.00,-3839.70}},{{2.23,0.0},{1050.30,2605.50}},{{2.26,0.0},{-297.71,394.04}},{{1.84,0.0},{-384.77,3076.40}},{{1.69,0.0},{666.48,1657.20}},{{1.73,0.0},{1839.10,-2581.10}},{{2.12,0.0},{188.02,1539.20}},{{2.03,0.0},{690.67,-437.73}},{{2.94,0.0},{554.28,2135.10}},{{1.73,0.0},{-2864.00,2900.70}},{{3.34,0.0},{883.82,88.67}},{{3.03,0.0},{1923.60,286.78}},{{2.97,0.0},{1655.50,447.99}},{{1.78,0.0},{401.55,158.21}},
                    {{0.54,-1.76},{-1.30,4.44}},{{0.51,2.64},{5.13,8.35}},{{1.74,0.0},{-1.20,-1.74}},{{2.25,-1.16},{-0.47,3.25}},{{-1.02,1.98},{-4.49,2.14}},{{3.10,-0.19},{0.29,8.31}},{{-1.01,1.97},{-1.11,5.89}},{{-2.56,-1.82},{-1.16,1.00}},{{-0.63,1.80},{-0.65,-0.58}},{{-1.83,-0.76},{0.18,-2.12}},{{2.73,-0.18},{-1.72,1.34}},{{2.07,-0.74},{-2.93,-2.98}},{{-1.28,1.49},{2.24,3.85}},{{1.85,-0.03},{8.11,-0.79}},{{3.30,0.25},{1.71,-0.42}},{{2.90,-1.90},{0.91,1.47}},{{2.02,-1.18},{0.98,1.34}},{{-2.22,-0.67},{6.52,-6.41}},{{2.35,-0.33},{8.66,0.12}},{{1.63,2.66},{-4.02,-1.15}},{{-2.39,2.38},{940.53,1101.80}},{{1.65,-1.11},{-1554.50,-2153.40}},
                    {{0.37,3.30},{-985.17,-1542.20}},{{-2.84,1.50},{10.36,-66.22}},{{3.10,0.32},{124.08,961.10}},{{-2.51,1.68},{275.35,228.55}},{{-2.48,-1.28},{-2884.10,3958.80}},{{1.85,-1.25},{-1716.20,-2224.00}},{{1.07,1.99},{-493.06,28.05}},{{-0.71,-1.70},{967.88,2945.50}},{{1.69,0.09},{-1698.60,552.63}},{{-1.45,0.94},{1445.70,-2820.30}},{{1.14,-1.79},{-212.17,1536.00}},{{-2.03,0.02},{562.43,593.55}},{{-2.39,1.71},{2190.00,-264.23}},{{-0.54,1.64},{-1851.20,-3631.70}},{{0.17,3.34},{-390.94,-797.60}},{{-2.99,-0.51},{1864.10,554.74}},{{2.37,1.79},{1476.50,-872.44}},{{-0.72,-1.63},{413.17,-124.74}},

                    {{0.164215,0.0},{11.0561,0.0}},{{0.164215,0.0},{12.5675,0.0}},{{0.164215,0.0},{12.9002,0.0}},{{0.164215,0.0},{25.7391,0.0}},{{0.164215,0.0},{39.2696,0.0}},{{0.164215,0.0},{39.9814,0.0}},{{0.164215,0.0},{60.1693,0.0}},{{0.164215,0.0},{70.5265,0.0}},{{0.164215,0.0},{83.8777,0.0}},{{0.164215,0.0},{85.5156,0.0}},{{0.164215,0.0},{98.4431,0.0}},{{0.164215,0.0},{406.29,0.0}},{{0.164215,0.0},{655.463,0.0}},{{0.164215,0.0},{1742.86,0.0}},{{0.164215,0.0},{1756.87,0.0}},{{0.164215,0.0},{1933.57,0.0}},{{0.164215,0.0},{2070.8,0.0}},{{0.164215,0.0},{3610.01,0.0}},{{0.164215,0.0},{4355.04,0.0}},{{0.164215,0.0},{4939.46,0.0}},
                    {{0.492645,0.0},{9.9692,0.0}},{{0.492645,0.0},{41.2165,0.0}},{{0.492645,0.0},{43.5295,0.0}},{{0.492645,0.0},{52.2695,0.0}},{{0.492645,0.0},{58.1329,0.0}},{{0.492645,0.0},{63.1639,0.0}},{{0.492645,0.0},{65.2156,0.0}},{{0.492645,0.0},{88.542,0.0}},{{0.492645,0.0},{91.1837,0.0}},{{0.492645,0.0},{97.4534,0.0}},{{0.492645,0.0},{1012.05,0.0}},{{0.492645,0.0},{1022.33,0.0}},{{0.492645,0.0},{1161.04,0.0}},{{0.492645,0.0},{1907.48,0.0}},{{0.492645,0.0},{2544.2,0.0}},{{0.492645,0.0},{2668.51,0.0}},{{0.492645,0.0},{3354.94,0.0}},{{0.492645,0.0},{3634.28,0.0}},{{0.492645,0.0},{4497.56,0.0}},{{0.492645,0.0},{4867.53,0.0}},
                    {{0.821075,0.0},{28.983,0.0}},{{0.821075,0.0},{49.7849,0.0}},{{0.821075,0.0},{58.1025,0.0}},{{0.821075,0.0},{59.6563,0.0}},{{0.821075,0.0},{61.1824,0.0}},{{0.821075,0.0},{64.9382,0.0}},{{0.821075,0.0},{74.2389,0.0}},{{0.821075,0.0},{85.6865,0.0}},{{0.821075,0.0},{85.8905,0.0}},{{0.821075,0.0},{85.8973,0.0}},{{0.821075,0.0},{128.905,0.0}},{{0.821075,0.0},{311.534,0.0}},{{0.821075,0.0},{437.078,0.0}},{{0.821075,0.0},{464.413,0.0}},{{0.821075,0.0},{726.516,0.0}},{{0.821075,0.0},{1361.19,0.0}},{{0.821075,0.0},{2141.25,0.0}},{{0.821075,0.0},{3520.59,0.0}},{{0.821075,0.0},{3940.32,0.0}},{{0.821075,0.0},{4768.36,0.0}},
                    {{1.14951,0.0},{3.30421,0.0}},{{1.14951,0.0},{7.46497,0.0}},{{1.14951,0.0},{14.5266,0.0}},{{1.14951,0.0},{25.3756,0.0}},{{1.14951,0.0},{42.8173,0.0}},{{1.14951,0.0},{51.4839,0.0}},{{1.14951,0.0},{69.1304,0.0}},{{1.14951,0.0},{86.5328,0.0}},{{1.14951,0.0},{90.4354,0.0}},{{1.14951,0.0},{96.2415,0.0}},{{1.14951,0.0},{287.868,0.0}},{{1.14951,0.0},{469.682,0.0}},{{1.14951,0.0},{595.145,0.0}},{{1.14951,0.0},{1385.55,0.0}},{{1.14951,0.0},{1738.4,0.0}},{{1.14951,0.0},{2120.01,0.0}},{{1.14951,0.0},{2384.44,0.0}},{{1.14951,0.0},{2551.88,0.0}},{{1.14951,0.0},{2650.28,0.0}},{{1.14951,0.0},{3651.51,0.0}},
                    {{1.47794,0.0},{9.54098,0.0}},{{1.47794,0.0},{14.3083,0.0}},{{1.47794,0.0},{17.6018,0.0}},{{1.47794,0.0},{26.5127,0.0}},{{1.47794,0.0},{27.4424,0.0}},{{1.47794,0.0},{53.0386,0.0}},{{1.47794,0.0},{58.2122,0.0}},{{1.47794,0.0},{59.6236,0.0}},{{1.47794,0.0},{71.1118,0.0}},{{1.47794,0.0},{82.1192,0.0}},{{1.47794,0.0},{1031.06,0.0}},{{1.47794,0.0},{1080.9,0.0}},{{1.47794,0.0},{1588.37,0.0}},{{1.47794,0.0},{2314.36,0.0}},{{1.47794,0.0},{2391.17,0.0}},{{1.47794,0.0},{2970.34,0.0}},{{1.47794,0.0},{3895.74,0.0}},{{1.47794,0.0},{3979.36,0.0}},{{1.47794,0.0},{4086.82,0.0}},{{1.47794,0.0},{4217.14,0.0}},

                    {{0.16,0.00},{-11.04,-0.57}},{{0.16,0.00},{-12.51,1.17}},{{0.16,0.00},{-12.89,0.54}},{{0.16,0.00},{-24.30,8.48}},{{0.16,0.00},{35.72,-16.31}},{{0.16,0.00},{-31.49,24.63}},{{0.16,0.00},{58.10,-15.63}},{{0.16,0.00},{-45.54,-53.86}},{{0.16,0.00},{-62.91,55.47}},{{0.16,0.00},{75.77,-39.64}},{{0.16,0.00},{-72.81,-66.25}},{{0.16,0.00},{-116.16,389.33}},{{0.16,0.00},{648.85,92.85}},{{0.16,0.00},{1614.90,-655.39}},{{0.16,0.00},{-386.78,-1713.80}},{{0.16,0.00},{-915.53,1703.10}},{{0.16,0.00},{1125.20,-1738.40}},{{0.16,0.00},{-73.55,-3609.30}},{{0.16,0.00},{-3883.60,1970.80}},
                    {{0.16,0.00},{4848.10,-945.71}},{{0.49,0.00},{-8.30,-5.53}},{{0.49,0.00},{-39.79,10.73}},{{0.49,0.00},{3.29,43.40}},{{0.49,0.00},{6.61,51.85}},{{0.49,0.00},{58.13,-0.65}},{{0.49,0.00},{-62.79,-6.91}},{{0.49,0.00},{-29.32,58.25}},{{0.49,0.00},{55.95,-68.62}},{{0.49,0.00},{46.80,78.26}},{{0.49,0.00},{-31.32,92.28}},{{0.49,0.00},{-33.25,-1011.50}},{{0.49,0.00},{452.97,916.50}},{{0.49,0.00},{554.81,-1019.90}},{{0.49,0.00},{-1530.50,1138.50}},{{0.49,0.00},{-2264.90,-1159.00}},{{0.49,0.00},{-2663.70,-160.62}},{{0.49,0.00},{1112.00,3165.30}},{{0.49,0.00},{-1965.00,-3057.20}},
                    {{0.49,0.00},{4364.30,-1086.80}},{{0.49,0.00},{-3115.40,-3740.00}},{{0.82,0.00},{-26.59,11.54}},{{0.82,0.00},{47.95,-13.38}},{{0.82,0.00},{-35.84,45.73}},{{0.82,0.00},{-53.77,25.83}},{{0.82,0.00},{6.25,60.86}},{{0.82,0.00},{56.75,31.57}},{{0.82,0.00},{-37.27,-64.20}},{{0.82,0.00},{7.45,-85.36}},{{0.82,0.00},{-24.04,-82.46}},{{0.82,0.00},{85.86,2.52}},{{0.82,0.00},{87.47,94.68}},{{0.82,0.00},{259.18,172.86}},{{0.82,0.00},{193.25,392.04}},{{0.82,0.00},{-463.39,30.80}},{{0.82,0.00},{716.14,122.37}},{{0.82,0.00},{-1169.50,-696.57}},{{0.82,0.00},{-740.45,-2009.10}},
                    {{0.82,0.00},{-3438.40,756.47}},{{0.82,0.00},{1595.00,-3603.10}},{{0.82,0.00},{-3096.70,3626.00}},{{1.15,0.00},{-2.32,-2.35}},{{1.15,0.00},{7.38,-1.12}},{{1.15,0.00},{0.53,-14.52}},{{1.15,0.00},{1.76,25.31}},{{1.15,0.00},{42.67,3.56}},{{1.15,0.00},{49.94,-12.50}},{{1.15,0.00},{-68.21,-11.27}},{{1.15,0.00},{9.80,85.98}},{{1.15,0.00},{85.76,-28.71}},{{1.15,0.00},{52.08,80.93}},{{1.15,0.00},{-63.90,280.69}},{{1.15,0.00},{441.57,160.05}},{{1.15,0.00},{-320.36,501.56}},{{1.15,0.00},{-1342.90,340.98}},{{1.15,0.00},{659.45,-1608.50}},{{1.15,0.00},{401.93,2081.60}},
                    {{1.15,0.00},{-2110.70,-1109.20}},{{1.15,0.00},{2471.20,636.60}},{{1.15,0.00},{2002.30,-1736.30}},{{1.15,0.00},{-2959.50,2138.90}},{{1.48,0.00},{-4.59,8.36}},{{1.48,0.00},{-13.06,5.85}},{{1.48,0.00},{13.17,11.68}},{{1.48,0.00},{8.56,-25.09}},{{1.48,0.00},{-14.75,-23.14}},{{1.48,0.00},{18.17,-49.83}},{{1.48,0.00},{-58.21,-0.58}},{{1.48,0.00},{39.49,-44.67}},{{1.48,0.00},{70.03,12.35}},{{1.48,0.00},{71.29,40.76}},{{1.48,0.00},{1028.20,77.41}},{{1.48,0.00},{90.30,1077.10}},{{1.48,0.00},{-1054.30,1188.00}},{{1.48,0.00},{-1036.00,-2069.50}},
                    {{1.48,0.00},{323.80,-2369.10}},{{1.48,0.00},{579.19,2913.30}},{{1.48,0.00},{-2889.10,2613.40}},{{1.48,0.00},{3969.30,282.86}},{{1.48,0.00},{3824.30,1441.00}},{{1.48,0.00},{1084.60,4075.30}},

                    {{-0.09,-0.14},{-4.50,-10.10}},{{0.09,-0.14},{12.44,1.77}},{{0.16,-0.05},{9.86,-8.32}},{{-0.16,-0.02},{9.44,-23.94}},{{0.07,0.15},{-30.67,24.53}},{{0.15,-0.08},{-30.14,-26.27}},{{0.13,0.10},{-53.45,27.64}},{{-0.16,0.03},{59.23,-38.28}},{{0.03,-0.16},{-66.34,-51.33}},{{0.16,0.04},{63.18,-57.63}},{{0.11,-0.13},{-78.42,-59.51}},{{0.15,-0.07},{-331.82,-234.46}},{{-0.06,-0.15},{255.07,603.80}},{{0.04,-0.16},{1239.40,-1225.40}},{{-0.16,-0.03},{1739.40,-247.15}},{{-0.04,-0.16},{-188.34,-1924.40}},{{-0.01,0.16},{281.28,-2051.60}},{{-0.15,0.08},{-3608.30,112.43}},
                    {{0.09,-0.14},{3411.60,2706.90}},{{-0.10,0.13},{2599.20,4200.30}},{{-0.37,-0.32},{9.04,-4.20}},{{0.26,0.42},{39.43,-11.99}},{{-0.39,0.31},{40.15,16.81}},{{-0.01,-0.49},{-52.21,2.44}},{{-0.22,-0.44},{22.78,53.48}},{{0.39,0.30},{44.10,45.22}},{{0.42,-0.25},{-41.88,-49.99}},{{-0.04,0.49},{-3.10,-88.49}},{{-0.48,0.09},{-87.02,27.23}},{{-0.45,-0.20},{-82.28,-52.22}},{{0.10,-0.48},{154.00,-1000.30}},{{0.22,0.44},{-927.16,-430.74}},{{0.37,0.32},{-739.53,895.04}},{{0.32,-0.38},{-1842.60,493.15}},{{0.37,0.32},{1781.80,1816.10}},{{-0.20,0.45},{1897.50,-1876.30}},
                    {{-0.48,-0.10},{3344.30,-266.84}},{{-0.49,-0.08},{-3552.00,-768.77}},{{0.14,-0.47},{-3007.60,-3344.00}},{{-0.49,-0.05},{-4771.70,961.22}},{{-0.51,0.64},{11.87,-26.44}},{{0.70,-0.42},{49.09,-8.29}},{{-0.37,-0.73},{36.41,-45.28}},{{0.61,0.54},{59.64,-1.24}},{{-0.67,-0.48},{-4.81,-60.99}},{{0.03,0.82},{-60.31,-24.07}},{{-0.53,0.62},{-54.10,50.84}},{{-0.82,0.03},{14.00,84.53}},{{-0.38,-0.73},{-85.79,4.19}},{{0.09,0.82},{26.50,81.71}},{{0.35,-0.74},{6.09,128.76}},{{-0.78,-0.26},{-303.77,69.12}},{{0.81,0.13},{-259.62,351.62}},{{-0.82,0.07},{-127.59,446.54}},
                    {{0.79,0.23},{721.78,82.80}},{{-0.54,0.62},{1261.40,-511.48}},{{0.20,0.80},{-1270.40,-1723.60}},{{-0.82,-0.03},{228.60,-3513.20}},{{0.80,-0.20},{779.53,3862.40}},{{-0.74,-0.36},{2626.80,-3979.60}},{{-0.30,1.11},{3.12,-1.10}},{{0.86,-0.77},{-5.12,5.43}},{{0.36,-1.09},{12.34,-7.67}},{{-1.15,0.07},{24.42,-6.89}},{{-0.61,-0.98},{36.15,-22.94}},{{0.36,1.09},{46.39,22.33}},{{0.09,1.15},{69.13,0.16}},{{0.86,-0.77},{-78.15,-37.15}},{{-0.81,0.81},{5.57,90.26}},{{-0.96,-0.64},{56.11,-78.20}},{{-0.70,-0.91},{279.73,-67.98}},{{-0.58,0.99},{-453.02,123.99}},
                    {{-0.97,0.61},{-383.82,454.84}},{{0.85,-0.77},{-1362.90,249.59}},{{-1.11,0.31},{715.46,-1584.30}},{{0.62,0.97},{955.17,-1892.60}},{{-1.01,-0.54},{-1815.30,1546.00}},{{0.81,0.82},{-438.20,-2514.00}},{{1.09,0.36},{-1764.00,-1977.90}},{{0.75,0.87},{-1463.60,-3345.40}},{{0.78,1.26},{-0.88,-9.50}},{{-0.33,-1.44},{-7.78,-12.01}},{{-0.42,1.42},{-11.27,13.52}},{{-1.05,-1.04},{10.60,24.30}},{{-0.76,1.27},{-3.46,-27.22}},{{1.03,1.06},{27.74,-45.20}},{{-0.39,-1.42},{44.33,-37.73}},{{1.37,-0.56},{-50.45,31.78}},{{1.09,-0.99},{5.31,70.91}},{{0.94,1.14},{-6.95,-81.82}},
                    {{1.48,-0.09},{35.14,-1030.50}},{{1.45,0.27},{-808.33,717.60}},{{1.42,0.41},{617.29,1463.50}},{{-1.33,-0.64},{112.38,2311.60}},{{1.40,-0.48},{2370.20,315.64}},{{1.47,-0.12},{1923.60,2263.30}},{{-1.48,0.09},{-3696.80,1229.10}},{{-0.15,1.47},{3739.80,-1359.70}},{{-1.31,0.68},{3071.40,2696.00}},{{0.84,1.22},{-3169.10,2782.30}}
                };

                const std::vector<complex<double>> li22_values = {
                    {-4.39131242273384, -7.0460060813405345}, {-15.921737311470512, -6.165351646756168}, {-0.8413729332902768, -4.503826313565111}, {-6.39312506441334, -5.858044289012671}, {-7.416286984771917, -6.937647505110364}, {-16.7754953503346, -4.732982867715663}, {-8.559404967951394, -7.301710180629588}, {-4.341275437868532, -4.224628055637825}, {0.07637790662384303, -2.275161160594363}, {-1.9086227148457802, -4.840895287671575}, {-4.925826637892774, -4.99071578515751}, {-6.149931105106237, -6.6212392074085535}, {-5.078824020098429, -6.914806860729641}, {-7.826187320301152, -8.554306519617576}, {-5.5109863533012255, -4.363275683613498}, {-5.8342081130360075, -4.247966268635963}, {-2.37935983721359, -4.385779817244646}, {-12.672190433798335, -7.5296760503259215}, {-12.579942305281236, -7.331465075531936}, {-10.675942508175979, -5.212413906209573}, {-104.9270655109705, 54.203319482383804}, {-96.21468404276203, 10.898430829900647}, {-109.90255860463637, 57.54241004233515}, {-44.23920093550241, 6.234000005688237}, {-93.59365708797228, 39.10061532207993}, {-71.95626633081585, 21.329421070458764}, {-127.98596600168423, 57.38801067140453}, {-103.8983967505794, 22.71887588803756}, {-67.68710023103895, 5.954426384534102}, {-94.72183768447545, 4.805472529069789}, {-77.38741987723884, -5.822456950563804}, {-90.94805972395511, -0.7482142850428772}, {-87.91693963434545, 11.487845757405665}, {-72.52399967556028, 2.7876104919430986}, {-110.34724583172388, 48.10752738228463}, {-96.47131634375828, 0.8686043398782459}, {-93.74990511537499, 43.67769218681559}, {-108.43027644818312, 48.959015057035515}, {-104.80849486116907, 44.52751922659073}, {-54.17255152796927, -8.32925548248491},
                    {-0.3604259222409443, 5.919556850536017}, {-13.381366330962791, 3.387230387487893}, {-2.6936586683371484, 1.141111537639845}, {-6.8317005796135355, -1.066125785718329}, {-0.8288021546220445, 7.252473969158046}, {9.446734500130384, 12.761702880334957}, {-3.064796835126341, 7.343512266169116}, {-4.660625489153745, 0.0395615532993735}, {1.6622620762326905, 0.877772184548355}, {5.455725625881401, 0.7404712491493166}, {2.242083014871362, 4.6951162028049325}, {9.306086264883472, 3.6360735130404462}, {7.763337431619673, 4.549331518679748}, {14.716620154312762, 6.3161388436767645}, {7.919787229067856, -11.817782235888846}, {0.8667480827652483, 5.019911325432698}, {-3.4828273016103495, 0.3886485651300019}, {11.292678603925685, 10.4729794891078}, {-7.978722898575782, 6.994113577795119}, {-7.400848452799954, 4.755400540767405}, {16.802282990313984, 119.70643705352262}, {-80.23535091724409, 40.98686362485682}, {-107.52796988578712, 60.11588551791219}, {-42.454476753013275, 9.340442777904222}, {44.538950495932255, 106.21256906941602}, {-7.8655647623089, 69.88364008111093}, {-77.31383173966091, 106.46899420419362}, {-9.362420792218607, 108.9052131451894}, {-14.002337303730194, 62.69218813423468}, {-33.60659747503907, 86.45137040795606}, {-21.849978462457244, 74.78217028458349}, {-83.92799114201256, 19.068367347216764}, {-12.675838897692579, 88.17433429516353}, {-68.06072320154814, 13.532076963509077}, {11.590350515003976, 124.2630203965096}, {-51.43969609848049, 75.14741196604606}, {64.60807926787815, 106.22186689928117}, {52.33192073670676, 127.58521780980334}, {46.63577868833905, 122.08231674709191}, {8.067234155699646, 60.876999878532715}, {2.755323236086398, -2.1067012011120667},
                    {0.4050980510571769, -5.5766947417951656}, {-2.7849085369307893, 0.9527861077559425}, {5.608376598257417, 0.710619584002111}, {2.4260881124883653, 1.2369540085514465}, {13.842142475181497, 9.91726335048887}, {2.6237831245968195, -0.5864824595827116}, {-2.206916205693794, 0.6601734457482191}, {-0.10500758099708385, 0.6140336742869821}, {0.7826659524499249, -0.594938298014404}, {2.558621659645386, 4.257816736809687}, {-1.9325572445232648, 4.025128757927172}, {1.5729639961599915, -0.6854289641701783}, {-7.989684531476003, -7.633325831369396}, {7.020619585494879, 10.412662607434003}, {3.7800934258654406, -5.262230951963997}, {2.625301197799965, -2.8969459864315183}, {3.308975426637268, -0.44377799595913814}, {-11.5561192286495, -5.176045735418143}, {5.019593093876384, -1.2564213339828054}, {41.529117566564175, -20.827918441295026}, {-20.59376579150807, 53.77180620220761}, {61.829150463018, -26.130907472550348}, {6.870612435127946, -14.409186405340279}, {-57.1795868432412, -66.5692711840838}, {24.863097598508382, -8.59977286775461}, {32.29570199154518, 47.8191698252171}, {-18.668340573382487, 62.56844903877206}, {15.800539835827989, -28.646838389921356}, {2.3795686044236177, 37.01076359057592}, {-49.5168660729929, -44.428483566178905}, {19.78731479143992, -23.942293749841635}, {33.808605032291794, 36.7089875594635}, {22.32119226327321, 5.815837667240203}, {37.3715720315953, -25.94378850323171}, {40.6200796872563, -6.318376799003829}, {54.400416297435086, -13.489781978970086}, {42.64629965383495, 10.29736940960757}, {-57.81197680035529, -53.0938000288339}, {7.604161403051499, 16.213465890776085},

                    {0.11220289527456373, -0.015617135646444284}, {0.1293378625795505, -0.026188044535815625}, {0.13290157121067425, -0.028735445028317298}, {0.22160059872176768, -0.14551938107690712}, {0.2564525475632421, -0.26072422050320915}, {0.2574290618162469, -0.2662676952215147}, {0.2665980019711754, -0.40478398700647156}, {0.2629252076137877, -0.46436181324194464}, {0.2539887260713328, -0.5325750452166231}, {0.2526661039969409, -0.5403753911599369}, {0.24103501647987308, -0.5982375730999682}, {-0.0793994482082494, -1.257034112403625}, {-0.27176250352410924, -1.498771635668149}, {-0.7951526559426288, -2.0074502581035905}, {-0.800149434363328, -2.01166817773819}, {-0.8608489482785598, -2.0622048131972184}, {-0.9052840502280268, -2.0984157014429554}, {-1.2962709046541698, -2.3932759790110882}, {-1.440585370462422, -2.4932508533513107}, {-1.5409117325765187, -2.5604437926140435}, {0.8292010298031058, -0.6918068255104239}, {0.6796805828855084, -2.473380725873371}, {0.650250831358832, -2.5549841077554722}, {0.5381390372521386, -2.833229479677101}, {0.4634246628920342, -2.9980260252904283}, {0.4001974036190614, -3.128137753751779}, {0.37470066206035724, -3.1785701879271473}, {0.09859177526157104, -3.669208274453081}, {0.06897424279315345, -3.71709671880887}, {-2.2324729929135145E-6, -3.8258347736189195}, {-4.1475807579975745, -7.867823266216009}, {-4.172601568205713, -7.885761388901273}, {-4.492732731873451, -8.11175136207637}, {-5.832633592343824, -8.995992988673319}, {-6.675924960083873, -9.510365123925467}, {-6.820247572440794, -9.595631072136808}, {-7.531172861704716, -10.00504488982147}, {-7.786714504245174, -10.148174257327248}, {-8.485764116356993, -10.529782021144454},
                    {-8.751732554656286, -10.67139203176159}, {0.9736912441984953, -5.796771120241236}, {0.02698982144106621, -7.494588728012842}, {-0.30877441991406807, -7.993659174095059}, {-0.3690231780538289, -8.07943452038079}, {-0.4274790089238465, -8.161669093108866}, {-0.5684062933340268, -8.356135575305203}, {-0.9006605320361896, -8.7955319662341}, {-1.280565117056339, -9.269793364432529}, {-1.287072766317658, -9.277685833962797}, {-1.2872895383250103, -9.27794860865771}, {-2.497231974368479, -10.637009005572507}, {-5.799013244256429, -13.650045060492467}, {-7.306907388531219, -14.819940367675596}, {-7.591039911861503, -15.030089143428304}, {-9.817616897571622, -16.584412856204793}, {-13.327328478700345, -18.774264608375837}, {-16.137767453854895, -20.358607235941093}, {-19.489443370794657, -22.100149575714024}, {-20.287376128799163, -22.494914686450496}, {-21.671214819168497, -23.1636176249727}, {1.7137168172491908, -3.7531201324470658}, {0.5975741199355671, -6.955733788714474}, {-1.4083242642226494, -9.861710419692006}, {-3.859213119246668, -12.367510615513652}, {-6.790597388944715, -14.714115638789089}, {-7.96776226008847, -15.532530965223991}, {-10.004740842113783, -16.82709326744452}, {-11.683524963623896, -17.79943747436693}, {-12.026187419092746, -17.988894921418822}, {-12.516695974813647, -18.255217043793916}, {-22.51096971165566, -22.741016277755676}, {-27.79792686035608, -24.604974168140167}, {-30.53491784491242, -25.472588060753218}, {-41.25672013665092, -28.38259061460711}, {-44.38759810933081, -29.113037538366434}, {-47.2137374678434, -29.734106745032683}, {-48.926041673321, -30.094047095995144}, {-49.92767734021635, -30.2991771825845}, {-50.490215134103984, -30.412680287662276},
                    {-55.373665901374, -31.34945476397575}, {-4.746471349624331, -9.269026911086383}, {-7.14511869208183, -10.66304895433844}, {-8.539552338561943, -11.33969145067011}, {-11.628247048538839, -12.588169926588613}, {-11.908055693047363, -12.687216629349681}, {-17.844618748972028, -14.378018239397928}, {-18.772258390739566, -14.583658913759017}, {-19.014559994420424, -14.635186717829033}, {-20.841057939594048, -14.996252151097647}, {-22.39063998475655, -15.267337343581332}, {-57.92105475642891, -16.221547974278295}, {-58.73071260873853, -16.16764962284966}, {-65.53097045276216, -15.627646801736581}, {-72.5226186672572, -14.92580938306078}, {-73.14493338509601, -14.856816955976086}, {-77.34333599825469, -14.365545470347044}, {-82.75029511407517, -13.670574861611186}, {-83.18106802236971, -13.61236519408034}, {-83.72306559556668, -13.538551711244494}, {-84.3637171042341, -13.450485043624226},

                    {-0.0652501994623047, -0.0029476965977368597}, {-0.07279291243890629, 0.005880601547522568}, {-0.07464550000486048, 0.002695470865847538}, {-0.12852881108049585, 0.035203972261058425}, {0.13314649524084626, -0.25759722762304366}, {-0.1675980775928938, 0.09103906213256632}, {0.17222551882408854, -0.37891868482430147}, {-0.24138836792230733, -0.1639527721795373}, {-0.28688249691227663, 0.15030021068743504}, {0.07811150248416851, -0.47383575797056815}, {-0.3195394440178523, -0.1664262655220518}, {-0.6794273400498799, 0.5179770757798183}, {-0.3216587969422102, 1.3837923812249346}, {-0.9421090845991117, -1.70682783711494}, {-1.4090675246416722, -0.8446587229834028}, {-1.521583009510551, 0.6918589820202408}, {-1.291039667067691, -1.3937943003808568}, {-1.8518539457559173, -1.1489545487445334}, {-2.170151357739173, 0.3629189837494846}, {-1.5752398090136128, -2.327029673752612}, {-0.5081297487818519, -0.2474808886362262}, {-1.468874182520839, 0.24066066960030574}, {-0.9624738045939079, 1.482792196344841}, {-1.0838175693110492, 1.660658772958996}, {0.44771389277307644, -2.960934430428147}, {-1.9095223442078986, -0.12021688942828357}, {-1.6741993710697698, 1.2116918344057983}, {-1.0962450235100272, -2.70681798534891}, {-1.283931233547454, 2.5756017081720493}, {-2.0638726665532032, 1.5921129223378303}, {-6.193251183995499, -3.8439510844996363}, {-5.733654581675973, 5.076901831252382}, {-6.009666487842715, -5.320231154737696}, {-8.427421606003685, 1.8252808082877578}, {-9.323847228731148, -1.4258159057919082}, {-9.529679101775045, -0.1831752665516415}, {-9.21974000950047, 6.042108335987811}, {-10.2167608061667, -3.21161950474829}, {-8.827908094341446, -9.641764249885934},
                    {-11.244077247965254, -2.959473127553327}, {-3.450585180767021, 0.8442900225645653}, {-0.7967782030401191, -6.899753228385797}, {-4.785331859400971, 2.427724418506517}, {-5.150281000957744, 1.2158785982058662}, {-3.969987747692036, 4.511023253021411}, {-2.066906215766643, 7.091684525301659}, {-5.332253238463815, -3.041614170441717}, {-4.931613587920529, -5.022306179651406}, {-5.48292478287363, -3.921385313835513}, {-1.3758536207097032, 9.172222927258463}, {-4.868812477052664, 7.9168932135782795}, {-7.6029520486194535, 11.1074428226277}, {-10.424656043280843, 9.593360577044422}, {-12.928675680369441, 0.31950272981527306}, {-10.35872221216441, 15.660132106636357}, {-18.57502364171859, -3.2125940559142254}, {-20.737774531118284, -7.886080171218303}, {-24.88730607956896, 1.5218923210081137}, {-23.5224880594105, -14.207456812765294}, {-26.68249493099178, 6.361291994019595}, {-2.322838177591363, -0.47677049234189184}, {-0.02086874414438178, -6.612296814479793}, {-5.833995713191145, -3.905591722029296}, {-3.574404807336146, 9.711262070988358}, {2.921439009567222, 17.75724074984146}, {-9.003754629022039, -13.921985433639751}, {-13.950024063551584, 3.613403402565204}, {-9.128963577174225, 15.754423867545928}, {-13.32027379392732, -15.476178593889479}, {-7.1446580237782165, 19.035174439074904}, {-19.489190204188688, 19.58158908556508}, {-13.900367463810056, 35.32132761031395}, {-27.944243941252452, 20.020478427539935}, {-40.54268521543353, 15.242000131703723}, {-46.73099949142665, -12.79459943391685}, {-37.69876761524011, 36.01082479971424}, {-49.758077885385056, 7.486587940989558}, {-30.763344521811504, 51.35086350396192}, {-52.14513274370098, -19.846029090766237},
                    {-51.567064731853336, 24.309809459244835}, {-0.8947073806508303, 7.863139787230969}, {-4.84265065094897, 7.890632220058462}, {6.944998371226937, 13.521808086392555}, {-13.332350605011293, -3.161723974813622}, {-12.559662443577217, 2.899141294107789}, {-18.801433294627607, -3.04190912310532}, {-14.01498757840277, 12.108424612759933}, {-19.97827933792649, -6.370817176394002}, {9.975691153888565, 28.780006928677974}, {5.444088863693125, 29.234968722993944}, {-5.518384784887322, 66.6269296252829}, {-25.947131511269294, 52.58641342254013}, {-40.21756239326466, 47.142260393298564}, {-63.40447101123522, 21.40084833083985}, {-67.70997332219952, 11.42477001885446}, {-37.074354980800855, 67.38280787637402}, {-54.39829096520638, 55.86991651333784}, {-20.408463036883585, 89.11223629479449}, {-25.707231439965483, 85.84110053965503}, {-41.06070640807801, 73.28908219739438},

                    {0.065391356191208, 0.004639425589311519}, {-0.05426456856554899, -0.07991524154861217}, {-8.448253391318966E-4, -0.10106704665020123}, {0.09180680528163482, -0.08444917339516189}, {-0.005282585440735121, -0.18902555985521508}, {-0.19327290795670837, 0.07223965363902085}, {-0.17513011292581504, -0.18958892499541335}, {0.20014233650869362, -0.16823692078876112}, {0.11257663199631505, 0.2944274058307509}, {0.20000125649324588, -0.46198266651074604}, {-0.18396537778250988, 0.3321867480766292}, {-0.7771660472815123, 0.30500942659304986}, {-0.7090682393254588, 0.9187067191870318}, {-0.9048757438868962, 1.3060382460967315}, {1.3967404840788906, 0.27439908598427576}, {0.6615662303001173, 1.4229969040232984}, {-1.6743170774295866, -1.0892035519922523}, {2.1529319137466247, 1.040844357356269}, {-2.6964823426340527, 0.2682596889213529}, {1.444110232668221, -1.608395951299769}, {0.2534817972966552, 0.27483890247156495}, {-1.66255434849367, 0.4599524437491751}, {0.7964463113369336, -0.7400595894235695}, {1.4220889836305637, 0.5837801731014666}, {-1.1225254703245575, 1.2767622861408725}, {-2.0007667580472055, 0.31381125584522374}, {-1.637610306015548, 0.8410051229294806}, {-2.670774599943345, -0.5350606671715736}, {0.6763649125121787, 2.1277841549526695}, {1.4725080836926048, -1.5080326446406098}, {-0.9698001387996529, 5.567509685180234}, {1.6799336926016428, -6.179883121191476}, {-4.664745604895048, -4.7208753945694495}, {-1.3603610106913164, 8.017161364016127}, {-8.659831666882837, -2.9243717535710854}, {-0.595055378082856, -7.856051609518381}, {7.1878195707969645, 1.6108077563673984}, {6.871336936646418, -5.757210959285388}, {0.039768001596785396, 9.498849037720264},
                    {5.553491948286289, 8.176947566442061}, {-0.8454363517867131, -2.0121186621630307}, {-4.424486805559848, -2.2664530375311163}, {1.4789003526937887, 2.3311922478859373}, {-4.873509830657729, 1.1711119921315947}, {2.8491713001659655, -0.18263783952036716}, {3.743529562560205, -1.2469473380866172}, {3.3876644914055394, 0.3479764235887952}, {2.651014509559805, 2.128691845447272}, {4.160275328263268, -0.7136333013340461}, {-0.14808027658541612, -3.871823212144008}, {6.3113052472995825, 3.265302201447725}, {4.557951804610472, -6.375800056042078}, {-12.480139178825347, 0.2774604741732567}, {6.075842810051109, 4.316193329942662}, {-15.008290017157831, 7.689518544924886}, {4.71437756804217, -9.76178764458519}, {10.567817269087252, -12.418183019913382}, {12.995705148007161, -5.552035168151505}, {-15.612960015248595, 21.534216239339013}, {14.773220072956416, 3.3254747461046286}, {-0.5125153753616489, -0.563764927380169}, {1.6285646559787068, 1.68095291822757}, {-1.7287617537523345, 1.8814133800080546}, {2.4219125391290888, -0.6142748555853075}, {1.9513642388688623, 3.120053644539894}, {-3.2667255811959013, -4.233674067420111}, {-2.974555033163888, -5.127628650919794}, {-2.085116786067239, 8.019313303611334}, {5.163007660847988, -1.284205946752195}, {5.1203151292667854, 1.4208505406725749}, {4.9707565998018595, 7.420625200010541}, {12.031679804863984, -1.636900050915301}, {12.031474511791743, 2.177716622216508}, {-3.5621669665396922, 23.898507928037134}, {13.20127390900538, -10.13198667377522}, {18.303634513614696, -22.885317767357797}, {6.79965947876903, 19.49811996409946}, {7.038822493719508, -29.992338745529203}, {-20.13065962061968, -31.8590212157873}, {7.121132433354134, -31.487252410351445},
                    {4.5392273674282615, 0.473463123620915}, {2.7093762217090998, 1.2812104309250714}, {3.2739740914539155, -1.0316353936087808}, {-0.37059975667064293, 4.368622556100561}, {-3.404402411993715, -4.285534195902528}, {12.411011012666917, -5.402034515912856}, {2.0181208932398107, 5.943549368461252}, {-1.299969771151586, 13.137836904916147}, {10.110898698977733, 9.474451694389739}, {10.756710316669418, -8.290842946874818}, {-49.31207059883768, 11.191873269843214}, {-35.883488926863585, -25.31514770736709}, {-43.2723932259287, -16.900810298068947}, {15.102166839560596, 19.31434307749369}, {-53.73975735223602, 5.621604660433292}, {-21.522269029578766, 69.9174713303928}, {22.984440040080017, 19.088770619981858}, {-1.259601684652921, -32.30921896823236}, {25.918698346389608, -7.830834994949328}, {-2.238632800252198, -39.67898292387497}
                };

                const double epsilon = 7.5e-12;
                test_region(x_y_points, li22_values, epsilon);
            }

            // Testing regions Ea, Eb, Ec and Ed
            {
                const std::vector<std::vector<complex<double>>> x_y_points = {
                    // Ea
                    {{0.505655, 0.0}, {2.59658, 0.0}},      {{0.255197, 0.0}, {5.14494, 0.0}},
                    {{0.695389, 0.0}, {1.88811, 0.0}},      {{0.308169, 0.0}, {4.26056, 0.0}},
                    {{0.156575, 0.0}, {8.3856, 0.0}},       {{0.522395, 0.0}, {2.51337, 0.0}},
                    {{0.496703, 0.0}, {2.64338, 0.0}},      {{0.606517, 0.0}, {2.16477, 0.0}},
                    {{0.586386, 0.0}, {2.23909, 0.0}},      {{0.403183, 0.0}, {3.25652, 0.0}},
                    {{0.505655, 0.0}, {1.7441, 0.0}},       {{0.255197, 0.0}, {3.45581, 0.0}},
                    {{0.695389, 0.0}, {1.26823, 0.0}},      {{0.308169, 0.0}, {2.86178, 0.0}},
                    {{0.156575, 0.0}, {5.63254, 0.0}},      {{0.522395, 0.0}, {1.68821, 0.0}},
                    {{0.496703, 0.0}, {1.77553, 0.0}},      {{0.606517, 0.0}, {1.45406, 0.0}},
                    {{0.586386, 0.0}, {1.50398, 0.0}},      {{0.403183, 0.0}, {2.18738, 0.0}},
                    {{0.505655, 0.0}, {2.33937, 0.0}},      {{0.255197, 0.0}, {4.63529, 0.0}},
                    {{0.695389, 0.0}, {1.70108, 0.0}},      {{0.308169, 0.0}, {3.83852, 0.0}},
                    {{0.156575, 0.0}, {7.55494, 0.0}},      {{0.522395, 0.0}, {2.2644, 0.0}},
                    {{0.496703, 0.0}, {2.38153, 0.0}},      {{0.606517, 0.0}, {1.95034, 0.0}},
                    {{0.586386, 0.0}, {2.01729, 0.0}},      {{0.403183, 0.0}, {2.93394, 0.0}},

                    {{0.506, 0.0}, {-1.23, 2.28}},          {{0.255, 0.0}, {5.12, -0.458}},
                    {{0.695, 0.0}, {1.73, 0.76}},           {{0.308, 0.0}, {-3.4, 2.56}},
                    {{0.157, 0.0}, {5.17, -6.6}},           {{0.522, 0.0}, {2.48, 0.381}},
                    {{0.497, 0.0}, {1.38, -2.25}},          {{0.607, 0.0}, {-0.428, -2.12}},
                    {{0.586, 0.0}, {1.55, -1.61}},          {{0.403, 0.0}, {-3.23, -0.431}},
                    {{0.506, 0.0}, {1.64, 0.601}},          {{0.255, 0.0}, {-1.28, -3.21}},
                    {{0.695, 0.0}, {0.223, -1.25}},         {{0.308, 0.0}, {-2.3, 1.7}},
                    {{0.157, 0.0}, {1.3, 5.48}},            {{0.522, 0.0}, {0.97, 1.38}},
                    {{0.497, 0.0}, {1.4, -1.09}},           {{0.607, 0.0}, {1.45, -0.0161}},
                    {{0.586, 0.0}, {-0.969, 1.15}},         {{0.403, 0.0}, {-2.01, -0.854}},
                    {{0.506, 0.0}, {-1.35, 1.91}},          {{0.255, 0.0}, {3.61, -2.9}},
                    {{0.695, 0.0}, {-0.918, -1.43}},        {{0.308, 0.0}, {-3.68, -1.09}},
                    {{0.157, 0.0}, {7.53, 0.549}},          {{0.522, 0.0}, {-1.92, -1.2}},
                    {{0.497, 0.0}, {-1.82, 1.54}},          {{0.607, 0.0}, {-1.88, 0.503}},
                    {{0.586, 0.0}, {1.66, -1.15}},          {{0.403, 0.0}, {-2.17, 1.97}},

                    {{-0.0259, -0.505}, {-1.06, -2.37}},    {{-0.01, -0.255}, {-4.72, 2.04}},
                    {{-0.688, -0.0979}, {0.841, -1.69}},    {{0.193, -0.24}, {4., 1.46}},
                    {{-0.113, 0.109}, {7.75, 3.19}},        {{0.473, -0.223}, {0.0368, 2.51}},
                    {{0.285, 0.407}, {-2.62, -0.318}},      {{-0.599, 0.0977}, {1.38, 1.66}},
                    {{-0.539, 0.231}, {-1.87, 1.23}},       {{-0.249, 0.317}, {-0.199, 3.25}},
                    {{0.145, -0.485}, {1.44, 0.982}},       {{-0.247, 0.0642}, {-0.689, 3.39}},
                    {{-0.474, 0.509}, {1.08, -0.669}},      {{-0.111, 0.287}, {0.508, -2.82}},
                    {{0.136, -0.078}, {4.2, -3.75}},        {{0.427, -0.302}, {1.4, 0.942}},
                    {{0.417, -0.27}, {0.077, 1.77}},        {{0.533, -0.29}, {0.869, -1.17}},
                    {{0.563, 0.165}, {-1.25, -0.83}},       {{-0.273, 0.297}, {-1.9, -1.09}},
                    {{-0.184, -0.471}, {0.729, -2.22}},     {{0.194, 0.166}, {-3.74, 2.73}},
                    {{-0.424, 0.551}, {1.7, -0.00635}},     {{-0.0216, 0.307}, {-3.01, -2.39}},
                    {{-0.145, 0.0592}, {3.8, 6.53}},        {{-0.242, -0.463}, {1.67, 1.53}},
                    {{0.221, -0.445}, {1.02, -2.15}},       {{-0.382, 0.471}, {-0.395, -1.91}},
                    {{0.163, 0.563}, {0.351, 1.99}},        {{-0.328, -0.234}, {1.54, 2.5}},

                    // Eb
                    {{18.6101, 0.0}, {0.0564046, 0.0}},     {{2.08132, 0.0}, {0.50434, 0.0}},
                    {{18.6641, 0.0}, {0.0562415, 0.0}},     {{6.76049, 0.0}, {0.155269, 0.0}},
                    {{1.50772, 0.0}, {0.696211, 0.0}},      {{4.36459, 0.0}, {0.240503, 0.0}},
                    {{4.29293, 0.0}, {0.244517, 0.0}},      {{8.32422, 0.0}, {0.126101, 0.0}},
                    {{3.60256, 0.0}, {0.291375, 0.0}},      {{2.5693, 0.0}, {0.408553, 0.0}},
                    {{24.0677, 0.0}, {0.0564046, 0.0}},     {{2.69169, 0.0}, {0.50434, 0.0}},
                    {{24.1375, 0.0}, {0.0562415, 0.0}},     {{8.74308, 0.0}, {0.155269, 0.0}},
                    {{1.94988, 0.0}, {0.696211, 0.0}},      {{5.64455, 0.0}, {0.240503, 0.0}},
                    {{5.55187, 0.0}, {0.244517, 0.0}},      {{10.7654, 0.0}, {0.126101, 0.0}},
                    {{4.65905, 0.0}, {0.291375, 0.0}},      {{3.32277, 0.0}, {0.408553, 0.0}},
                    {{12.5658, 0.0}, {0.0564046, 0.0}},     {{1.40534, 0.0}, {0.50434, 0.0}},
                    {{12.6023, 0.0}, {0.0562415, 0.0}},     {{4.56479, 0.0}, {0.155269, 0.0}},
                    {{1.01804, 0.0}, {0.696211, 0.0}},      {{2.94704, 0.0}, {0.240503, 0.0}},
                    {{2.89865, 0.0}, {0.244517, 0.0}},      {{5.62065, 0.0}, {0.126101, 0.0}},
                    {{2.4325, 0.0},  {0.291375, 0.0}},      {{1.73483, 0.0}, {0.408553, 0.0}},

                    {{18.6, 0.0}, {0.0246,0.0508}},         {{2.08, 0.0}, {0.36,0.353}},
                    {{18.7, 0.0}, {-0.056,-0.00472}},       {{6.76, 0.0}, {0.0694,0.139}},
                    {{1.51, 0.0}, {0.553,-0.423}},          {{4.36, 0.0}, {-0.0212,-0.24}},
                    {{4.29, 0.0}, {0.226,-0.0939}},         {{8.32, 0.0}, {-0.0718,-0.104}},
                    {{3.6, 0.0}, {0.227,-0.182}},           {{2.57, 0.0}, {0.0732,-0.402}},
                    {{24.1, 0.0}, {0.0354,0.0439}},         {{2.69, 0.0}, {0.504,-0.0216}},
                    {{24.1, 0.0}, {-0.0523,-0.0207}},       {{8.74, 0.0}, {-0.0874,0.128}},
                    {{1.95, 0.0}, {0.672,0.183}},           {{5.64, 0.0}, {-0.24,0.0072}},
                    {{5.55, 0.0}, {0.0312,0.243}},          {{10.8, 0.0}, {0.0685,0.106}},
                    {{4.66, 0.0}, {0.032,0.29}},            {{3.32, 0.0}, {-0.406,0.048}},
                    {{12.6, 0.0}, {-0.0563,0.00335}},       {{1.41, 0.0}, {-0.464,0.197}},
                    {{12.6, 0.0}, {0.0195,0.0528}},         {{4.56, 0.0}, {0.126,-0.0906}},
                    {{1.02, 0.0}, {-0.604,0.346}},          {{2.95, 0.0}, {0.175,0.164}},
                    {{2.9, 0.0}, {-0.111,0.218}},           {{5.62, 0.0}, {-0.0156,-0.125}},
                    {{2.43, 0.0}, {-0.0149,0.291}},         {{1.73, 0.0}, {-0.394,-0.108}},

                    {{7.55, 17.0}, {-0.0375, -0.0422}},     {{-2.06, 0.323}, {0.0287, 0.504}},
                    {{-6.24, -17.6}, {-0.0131, -0.0547}},   {{-0.183, 6.76}, {-0.142, -0.0618}},
                    {{-1.46, -0.381}, {-0.639, 0.276}},     {{-4.36, 0.133}, {0.011, -0.24}},
                    {{2.51, 3.48}, {0.0894, 0.228}},        {{2.9, -7.8}, {-0.126, 0.00289}},
                    {{3.48, -0.943}, {-0.243, -0.161}},     {{-2.42, 0.855}, {0.396, -0.101}},
                    {{14.7, -19.0}, {0.0456, 0.0332}},      {{-1.04, -2.48}, {-0.45, -0.228}},
                    {{-19.8, -13.8}, {0.0444, -0.0345}},    {{6.8, -5.49}, {0.0778, -0.134}},
                    {{1.91, -0.392}, {-0.695, -0.0387}},    {{-4.9, 2.79}, {0.163, 0.177}},
                    {{5.01, -2.4}, {0.239, -0.0528}},       {{-3.52, 10.2}, {-0.0965, 0.0812}},
                    {{4.32, 1.75}, {0.29, 0.0286}},         {{2.67, 1.98}, {-0.401, -0.0778}},
                    {{-7.99, -9.7}, {-0.0498, -0.0266}},    {{0.217, 1.39}, {0.0284, 0.504}},
                    {{-5.26, -11.4}, {-0.027, 0.0493}},     {{3.96, 2.27}, {-0.152, 0.0299}},
                    {{-0.0833, -1.01}, {0.41, 0.563}},      {{2.49, -1.57}, {-0.0703, 0.23}},
                    {{2.13, -1.97}, {0.24, -0.0464}},       {{1.59, 5.39}, {0.0998, 0.077}},
                    {{2.21, -1.01}, {0.263, 0.125}},        {{0.0128, 1.73}, {-0.303, 0.274}},

                    // Ec
                    {{1.82885, 0.0}, {0.64269, 0.0}},       {{1.60388, 0.0}, {0.732836, 0.0}},
                    {{1.90091, 0.0}, {0.618327, 0.0}},      {{1.82885, 0.0}, {0.70907, 0.0}},
                    {{1.60388, 0.0}, {0.808526, 0.0}},      {{1.90091, 0.0}, {0.68219, 0.0}},

                    {{1.83, 0.0}, {-0.59, 0.256}},          {{1.6, 0.0},  {0.00683, -0.733}},
                    {{1.9, 0.0},  {-0.494, -0.372}},        {{1.83, 0.0}, {-0.442, -0.554}},
                    {{1.6, 0.0},  {0.478, -0.652}},         {{1.9, 0.0},  {-0.658, 0.181}},

                    {{0.656, 1.71},  {0.3, 0.568}},         {{-1.19, 1.08},  {0.666, -0.306}},
                    {{-0.675, 1.78}, {0.29, -0.546}},       {{1.7, -0.672},  {-0.14, -0.695}},
                    {{1.54, -0.443}, {0.808, -0.0274}},     {{1.62, -0.993}, {-0.226, 0.644}},

                    // Ed
                    {{0.593606, 0.0}, {1.93261, 0.0}},      {{0.65611, 0.0},  {1.7485, 0.0}},
                    {{0.764874, 0.0}, {1.49986, 0.0}},      {{0.685393, 0.0}, {1.93261, 0.0}},
                    {{0.757562, 0.0}, {1.7485, 0.0}},       {{0.883144, 0.0}, {1.49986, 0.0}},

                    {{0.594, 0.0}, {-0.043, -1.93}},        {{0.656, 0.0}, {-1.44, 0.985}},
                    {{0.765, 0.0}, {-0.84, -1.24}},         {{0.685, 0.0}, {1.11, -1.58}},
                    {{0.758, 0.0}, {-0.144, -1.74}},        {{0.883, 0.0}, {-0.255, -1.48}},

                    {{0.314, 0.504},   {-1.85, -0.553}},    {{0.636, -0.16},   {-0.447, -1.69}},
                    {{-0.764, 0.0428}, {1.38, -0.581}},     {{0.684, 0.0462},  {-0.3, 1.91}},
                    {{-0.445, -0.613}, {0.371, 1.71}},      {{-0.875, -0.122}, {0.342, 1.46}}
                };

                const std::vector<complex<double>> li22_values = {
                    {0.28430906801447176, -0.008421755457385799}, {0.1211701267401113, -0.0030402737517393574}, {0.4637316172512538, -0.01664330974338289}, {0.15109917230063807, -0.0039060474306414996}, {0.07030478034857879, -0.001677856023727695}, {0.29761348220806205, -0.008939134452426332}, {0.2773494774550338, -0.008156323078269169}, {0.370996577676471, -0.012042604160393234}, {0.35234426509832373, -0.011212030242349898}, {0.21034749248280216, -0.005773305141739412}, {0.1686890851453892, 0.0}, {0.07250655025895895, 0.0}, {0.2724553887314359, 0.0}, {0.09027818273134139, 0.0}, {0.04217689088655046, 0.0}, {0.17645752288754155, 0.0}, {0.16462027452348652, 0.0}, {0.21909408928814697, 0.0}, {0.2082909346398506, 0.0}, {0.1252939956470777, 0.0}, {0.2459232669771465, -0.0021642651934197952}, {0.10505079394864336, -7.619819675261071E-4}, {0.4002053752913035, -0.004439758487501313}, {0.13094242031022163, -9.829657628014836E-4}, {0.06099686119240033, -4.1784356175588865E-4}, {0.2573845442234788, -0.002302636175067873}, {0.2399255142514507, -0.0020934408896459036}, {0.3205397389568958, -0.003146593124554218}, {0.3044959188953915, -0.002918326930068194}, {0.18213414073753542, -0.0014655980914849683},
                    {-0.11405342652927392, 0.16599692350579964}, {0.11724705072956979, -0.01693898370683647}, {0.3546050578360589, 0.21767263724350555}, {-0.08834939650465495, 0.057897257676353675}, {0.029101421250432802, -0.051574982806298866}, {0.27783717243536843, 0.06523987457001898}, {0.08503177284132962, -0.2071144375939194}, {-0.08742740718829395, -0.25323792067969597}, {0.16664603868025143, -0.24053529428675174}, {-0.14307925519413267, -0.017068652163503728}, {0.15227645835597692, 0.06607475979022848}, {-0.027023553097685817, -0.05444509796700258}, {0.015621885557611936, -0.23525820178724557}, {-0.060976412820337195, 0.04072855855715542}, {0.004968881736976424, 0.037259810828906656}, {0.07847942840232208, 0.14067718299010193}, {0.11426971805090046, -0.10654680258581319}, {0.21882701473722121, -0.002898205641153952}, {-0.11715157501592373, 0.12164041955939225}, {-0.09374596149667835, -0.03643958950372976}, {-0.11774844016599208, 0.13844914344393766}, {0.06534187193557853, -0.06756880627653464}, {-0.17944081734026307, -0.22769377988091133}, {-0.09185393873348768, -0.024475561760136877}, {0.06047735623493673, 0.006263403750500275}, {-0.16225129782229386, -0.08951654542421572},
                    {-0.1408885937847178, 0.10374999269493745}, {-0.22103393204678512, 0.052961835496649456}, {0.20217362866147304, -0.1831830947652388}, {-0.10478532932061979, 0.08262521869210282}, {0.08985987907847637, 0.11036203495726704}, {0.07676257254143817, -0.037147827915619464}, {0.11738262820700962, -0.11363798336307762}, {-0.015157109154841648, -0.12271691384141903}, {0.02130841150321705, -0.03833355140011463}, {0.1745419156785799, 0.14295701114647769}, {0.09153617581243778, -0.14044847079472067}, {0.12537261156966895, 0.07149038904119509}, {-0.018904445738567324, 0.16308589906082746}, {0.10644219235685992, -0.013934045546690643}, {-0.0797316429342043, -0.09345053528560512}, {0.018221851983295644, 0.046485528784523465}, {-0.044488873134278926, -0.11287984899939829}, {-0.062305158874294074, 0.031140863251059026}, {-0.01238726489035058, -0.03521720687967823}, {0.10713636378730987, -0.11884176478799162}, {0.1333992130670516, 0.04285878755002549}, {-0.10599294402597423, -0.13626884130730985}, {-0.057873200415352884, -0.15361866951010594}, {-0.04264914407265297, 0.07703668245043588}, {0.0684576882979897, 0.10018973995245739}, {-0.048046865947073414, -0.055811450726961365},
                    {-4.1101392367393075E-4, -0.15668292192847}, {0.06554807251435392, 0.07004085212698813}, {0.03860454969856286, 0.009178040800023712}, {-0.1186210088514973, 0.06800334277946851}, {-0.1259966874541565, 0.0695080293026098}, {-0.16798106115364664, 0.02469676039971146}, {-0.06787636087591853, -0.14444907260790196}, {-0.047262865243554525, 0.09026358045992611},
                    {-1.2199221924765637, -0.5255127337596961}, {0.09950503550217582, -1.3545442497121163}, {-1.2201714255747824, -0.5244901353940187}, {-0.9823955657228464, -0.9710721910148715}, {0.6170423928968147, -1.1386663146730935}, {-0.7230742521340536, -1.1885044551510593}, {-0.7104473994097161, -1.1961929243103044}, {-1.0639949677853306, -0.8675218112165057}, {-0.5620258240822349, -1.271876815909998}, {-0.19167159815889145, -1.364292245643684}, {-1.6695598685250124, -0.5560888938280396}, {-0.4335543373491906, -1.803445699788755}, {-1.669700141267774, -0.5549326909636257}, {-1.480823152032849, -1.0845639940520926}, {0.15037912065514059, -1.814461341507382}, {-1.2467749450729624, -1.3771587039939295}, {-1.235042706936044, -1.3884788499888527}, {-1.5506846311152633, -0.9561007218717168}, {-1.0953414184138173, -1.5072176699532478}, {-0.7330379127631472, -1.7131356449452382}, {-0.7569375289097257, -0.4549781373386546}, {0.48964658629273927, -0.6288702673930138}, {-0.757314185312226, -0.4541632008510234}, {-0.4704690237974807, -0.7715764673833464}, {0.5738414761918524, -0.0495904989906628}, {-0.1999475837284327, -0.8718020777158744}, {-0.18739320948033777, -0.873817869124017},
                    {-0.5618541131649821, -0.7068091535952747}, {-0.043750606455733365, -0.882202650570478}, {0.28069339706696295, -0.7965552366833291}, {0.056429140784334475, -1.276010176583563}, {1.14764551698589, -0.7161575792720367}, {0.9927884601256878, 0.590229591498064}, {0.565747867211138, -1.2204222637308977}, {-0.24547134298414974, -1.1145809494272498}, {-0.9511520550215611, 0.699017993870147}, {-1.0865926191131523, -0.7591366049658326}, {-0.11661906516539489, 1.1701452976455207}, {-1.1482338660740488, -0.5436425730791238}, {-1.1510768112692646, -0.030047793427222996}, {-0.40210143478917115, -1.6236181839787993}, {-0.5121135718707046, -1.760238296870693}, {1.0677703504813536, 0.9824766343115441}, {1.5558577700122864, -0.26953024425261596}, {0.8981981273559994, -1.7407345919974735}, {0.8660670538806503, 1.211844400711397}, {1.3669511318893384, -1.0366971168239396}, {0.21207272398827515, -1.7362071929256713}, {1.4902965871628142, -0.8467806880526483}, {0.4735690369871506, 1.3719141420734169}, {0.6985744744654758, 0.4039583879020092}, {-0.23263838304093487, 0.6214530157514817}, {0.2142787256336575, -0.8387891630123243}, {-0.8063918507876541, -0.3156048609831699},
                    {-0.392462425513334, 0.24569890326278276}, {0.507434787820312, -0.741691776228006}, {0.7660674897072892, 0.30609799815634814}, {-0.5668528343728876, 0.5885815074351399}, {0.8264175544860809, 0.10911571211240778}, {-0.43725801670907294, 0.5485951381442222}, {0.028511509987715566, 0.9365459972458373}, {0.1129509420132857, 0.2721372135439821}, {0.7158118755494574, -0.19926463533556638}, {0.13048390684131217, 0.6665060256938883}, {-0.28452057068954817, 0.005347737542558018}, {0.03954784927745969, -0.43516542252865487}, {-0.19547436730603795, -0.6407713534869809}, {0.6383080355404829, -0.41000491624204455}, {0.03851762029624872, 0.9380520804851255}, {0.23670523663048115, -0.21455500027035296}, {-1.3906744260423276, 0.012359257097820125}, {0.3747781154404164, -0.3448077740518958}, {0.9168415138312209, 0.06871763141861553}, {-0.4841771843855103, 1.0386534825265905}, {-0.09793416390761855, 1.0519434368477105}, {0.5965188948167253, 0.041900669788367484}, {-1.3134397956095145, -0.1493964486228912}, {0.6314061796719006, 0.5975398857036494}, {-1.2595814178068883, 0.5285947120110825}, {0.8645342072905597, -0.39507614798516855}, {0.04539333726311884, -0.4668608726202505},
                    {0.0361804352768693, -0.1993941756940265}, {-0.4767479171332401, -0.1845156096376652}, {0.44438681882878067, -0.3819152629005412}, {-0.15346516540647426, -0.053499918224759746}, {0.481702329047619, -0.26712861920086195}, {-0.4685722732640545, -0.09137137427575762}, {-0.12430521438214265, -0.42494613507912743}, {-0.07833494430808674, -0.5912359017333737}, {0.21980412655563142, -0.05120942546436702},
                    {0.30481842380191804, -1.5010261170360355}, {0.550966830632813, -1.3988713402966406}, {0.2350884590619784, -1.520111169095683}, {0.29466674343516086, -1.69184949090345}, {0.5703852935885241, -1.5883233467828601}, {0.21699866587164474, -1.7104632159789743}, {0.0027101247523830807, 1.1668826170844695}, {-1.0561823354604152, -0.43493439732039985}, {-0.8952266393072676, 0.7118545568266659}, {-1.143780270203135, 0.48931017704957225}, {-0.7802670337654798, -1.1068757013796167}, {-0.10740177317991417, 1.2712586770873942}, {-0.18386255313687982, -0.3902493696622933}, {0.003981665030649673, -0.28714385129377445}, {-0.3812378578883693, -0.08841099080763892}, {-0.733443778849642, 0.3815176362818885}, {-0.12074799446149864, -1.0712301317913901}, {0.8374412378507479, -0.02977551249085142},
                    {0.29719243281141505, -0.0016922760453696416}, {0.34771731219875246, -0.0021551800434253337}, {0.4554604122714423, -0.0034214081054655975}, {0.4579115618625673, -0.017465496251575632}, {0.5485225088151857, -0.02306130282590044}, {0.7673626298880896, -0.041190359917272046}, {-0.035087401580323974, -0.23036984468998667}, {-0.21512461161522878, 0.12938124344595037}, {-0.21107689644871908, -0.2559268692194183}, {0.15650424461776868, -0.33029663716254465}, {-0.08938797326118042, -0.3810621734972792}, {-0.16557057221693006, -0.5011660875722455}, {0.14105350475410372, -0.11137097501368975}, {-0.20675380665097642, -0.13930514484484946}, {0.13585529850810196, -0.06723454983004751}, {-0.14568260982107928, 0.2891517061512161}, {-0.2115638017290158, 0.052635487067505426}, {0.022276626205105732, 0.20567256160082037}
                };

                const double epsilon = 1e-12;
                test_region(x_y_points, li22_values, epsilon);
            }

            // Testing regions F, G and H
            {
                const std::vector<std::vector<complex<double>>> x_y_points = {
                    {{0.913148, 0.0}, {0.906048, 0.0}},     {{0.881196, 0.0}, {0.938901, 0.0}},
                    {{1.189, 0.0}, {0.70132, 0.0}},         {{0.768063, 0.0}, {1.0772, 0.0}},
                    {{1.12417, 0.0}, {0.735973, 0.0}},      {{0.913148, 0.0}, {1.00895, 0.0}},
                    {{0.881196, 0.0}, {1.04553, 0.0}},      {{1.189, 0.0}, {0.774868, 0.0}},
                    {{0.768063, 0.0}, {1.19954, 0.0}},      {{1.12417, 0.0}, {0.819559, 0.0}},
                    {{0.913148, 0.0}, {1.16872, 0.0}},      {{0.881196, 0.0}, {1.2111, 0.0}},
                    {{1.189, 0.0}, {0.897573, 0.0}},        {{0.768063, 0.0}, {1.38949, 0.0}},
                    {{1.12417, 0.0}, {0.949342, 0.0}},      {{0.913148, 0.0}, {1.40294, 0.0}},
                    {{0.881196, 0.0}, {1.2839, 0.0}},       {{1.189, 0.0}, {1.14163, 0.0}},
                    {{0.768063, 0.0}, {1.0357, 0.0}},       {{1.12417, 0.0}, {1.20748, 0.0}},

                    // F
                    {{-1.181,-0.142},{0.654,0.254}},{{-1.181,-0.142},{0.722,0.280}},{{-1.181,-0.142},{0.837,0.325}},{{-1.181,-0.142},{1.064,0.413}},{{-1.116,-0.134},{0.686,0.266}},{{-1.116,-0.134},{0.764,0.297}},{{-1.116,-0.134},{0.885,0.344}},{{-1.116,-0.134},{1.126,0.437}},{{-0.907,-0.109},{0.845,0.328}},{{-0.907,-0.109},{0.941,0.365}},{{-0.907,-0.109},{1.089,0.423}},{{-0.907,-0.109},{1.308,0.508}},{{-0.875,-0.105},{0.875,0.340}},{{-0.875,-0.105},{0.975,0.378}},{{-0.875,-0.105},{1.129,0.438}},{{-0.875,-0.105},{1.197,0.465}},{{-0.763,-0.091},{0.965,0.375}},{{-0.763,-0.091},{1.004,0.390}},{{-0.763,-0.091},{1.118,0.434}},{{-0.763,-0.091},{1.295,0.503}},{{0.250,-0.726},{1.026,0.139}},{{0.250,-0.726},{1.068,0.144}},{{0.250,-0.726},{1.189,0.161}},{{0.250,-0.726},{1.377,0.186}},{{0.287,-0.833},{0.930,0.126}},{{0.287,-0.833},{1.036,0.140}},{{0.287,-0.833},{1.200,0.162}},{{0.287,-0.833},{1.272,0.172}},{{0.297,-0.863},{0.898,0.121}},{{0.297,-0.863},{1.000,0.135}},{{0.297,-0.863},{1.158,0.156}},{{0.297,-0.863},{1.390,0.188}},{{0.366,-1.063},{0.729,0.098}},{{0.366,-1.063},{0.812,0.110}},{{0.366,-1.063},{0.941,0.127}},{{0.366,-1.063},{1.197,0.162}},{{0.387,-1.124},{0.695,0.094}},{{0.387,-1.124},{0.768,0.104}},{{0.387,-1.124},{0.890,0.120}},{{0.387,-1.124},{1.131,0.153}},
                    {{-1.187,-0.062},{0.648,-0.268}},{{-1.187,-0.062},{0.716,-0.297}},{{-1.187,-0.062},{0.829,-0.343}},{{-1.187,-0.062},{1.055,-0.437}},{{-1.123,-0.059},{0.680,-0.282}},{{-1.123,-0.059},{0.757,-0.314}},{{-1.123,-0.059},{0.877,-0.363}},{{-1.123,-0.059},{1.116,-0.462}},{{-0.912,-0.048},{0.837,-0.347}},{{-0.912,-0.048},{0.932,-0.386}},{{-0.912,-0.048},{1.080,-0.447}},{{-0.912,-0.048},{1.296,-0.537}},{{-0.880,-0.046},{0.867,-0.359}},{{-0.880,-0.046},{0.966,-0.400}},{{-0.880,-0.046},{1.119,-0.463}},{{-0.880,-0.046},{1.186,-0.491}},{{-0.767,-0.040},{0.957,-0.396}},{{-0.767,-0.040},{0.995,-0.412}},{{-0.767,-0.040},{1.108,-0.459}},{{-0.767,-0.040},{1.284,-0.532}},{{0.033,-0.767},{-1.379,0.174}},{{0.033,-0.767},{-1.190,0.150}},{{0.033,-0.767},{-1.069,0.135}},{{0.033,-0.767},{-1.028,0.130}},{{0.037,-0.880},{-1.274,0.161}},{{0.037,-0.880},{-1.202,0.152}},{{0.037,-0.880},{-1.037,0.131}},{{0.037,-0.880},{-0.931,0.118}},{{0.039,-0.912},{-1.392,0.176}},{{0.039,-0.912},{-1.160,0.146}},{{0.039,-0.912},{-1.001,0.126}},{{0.039,-0.912},{-0.899,0.114}},{{0.048,-1.123},{-1.198,0.151}},{{0.048,-1.123},{-0.942,0.119}},{{0.048,-1.123},{-0.813,0.103}},{{0.048,-1.123},{-0.730,0.092}},{{0.050,-1.188},{-1.133,0.143}},{{0.050,-1.188},{-0.890,0.112}},{{0.050,-1.188},{-0.769,0.097}},{{0.050,-1.188},{-0.696,0.088}},
                    {{-0.860,0.821},{0.693,-0.106}},{{-0.860,0.821},{0.766,-0.117}},{{-0.860,0.821},{0.887,-0.135}},{{-0.860,0.821},{1.129,-0.172}},{{-0.813,0.776},{0.728,-0.111}},{{-0.813,0.776},{0.810,-0.123}},{{-0.813,0.776},{0.939,-0.143}},{{-0.813,0.776},{1.194,-0.182}},{{-0.661,0.630},{0.896,-0.136}},{{-0.661,0.630},{0.997,-0.152}},{{-0.661,0.630},{1.155,-0.176}},{{-0.661,0.630},{1.387,-0.211}},{{-0.638,0.608},{0.928,-0.141}},{{-0.638,0.608},{1.034,-0.157}},{{-0.638,0.608},{1.197,-0.182}},{{-0.638,0.608},{1.269,-0.193}},{{-0.556,0.530},{1.024,-0.156}},{{-0.556,0.530},{1.065,-0.162}},{{-0.556,0.530},{1.186,-0.180}},{{-0.556,0.530},{1.374,-0.209}},{{-0.422,1.112},{0.526,0.464}},{{-0.422,1.112},{0.581,0.513}},{{-0.422,1.112},{0.673,0.594}},{{-0.422,1.112},{0.856,0.756}},{{-0.399,1.051},{0.552,0.487}},{{-0.399,1.051},{0.614,0.542}},{{-0.399,1.051},{0.712,0.628}},{{-0.399,1.051},{0.905,0.799}},{{-0.324,0.854},{0.679,0.600}},{{-0.324,0.854},{0.756,0.668}},{{-0.324,0.854},{0.876,0.773}},{{-0.324,0.854},{1.052,0.929}},{{-0.312,0.824},{0.704,0.621}},{{-0.312,0.824},{0.784,0.692}},{{-0.312,0.824},{0.908,0.802}},{{-0.312,0.824},{0.962,0.850}},{{-0.272,0.718},{0.776,0.685}},{{-0.272,0.718},{0.808,0.713}},{{-0.272,0.718},{0.899,0.794}},{{-0.272,0.718},{1.042,0.920}},
                    {{-1.049,0.560},{-1.055,-0.435}},{{-1.049,0.560},{-0.830,-0.342}},{{-1.049,0.560},{-0.716,-0.295}},{{-1.049,0.560},{-0.648,-0.267}},{{-0.992,0.530},{-1.116,-0.460}},{{-0.992,0.530},{-0.878,-0.362}},{{-0.992,0.530},{-0.758,-0.313}},{{-0.992,0.530},{-0.680,-0.281}},{{-0.805,0.430},{-1.297,-0.535}},{{-0.805,0.430},{-1.080,-0.446}},{{-0.805,0.430},{-0.933,-0.385}},{{-0.805,0.430},{-0.838,-0.345}},{{-0.777,0.415},{-1.187,-0.490}},{{-0.777,0.415},{-1.120,-0.462}},{{-0.777,0.415},{-0.967,-0.399}},{{-0.777,0.415},{-0.868,-0.358}},{{-0.677,0.362},{-1.285,-0.530}},{{-0.677,0.362},{-1.109,-0.457}},{{-0.677,0.362},{-0.996,-0.411}},{{-0.677,0.362},{-0.957,-0.395}},{{-0.588,1.034},{-1.097,-0.317}},{{-0.588,1.034},{-0.862,-0.249}},{{-0.588,1.034},{-0.744,-0.215}},{{-0.588,1.034},{-0.674,-0.195}},{{-0.556,0.977},{-1.160,-0.336}},{{-0.556,0.977},{-0.912,-0.264}},{{-0.556,0.977},{-0.787,-0.228}},{{-0.556,0.977},{-0.707,-0.205}},{{-0.451,0.794},{-1.348,-0.390}},{{-0.451,0.794},{-1.123,-0.325}},{{-0.451,0.794},{-0.969,-0.280}},{{-0.451,0.794},{-0.870,-0.252}},{{-0.435,0.766},{-1.233,-0.357}},{{-0.435,0.766},{-1.163,-0.337}},{{-0.435,0.766},{-1.004,-0.291}},{{-0.435,0.766},{-0.902,-0.261}},{{-0.380,0.668},{-1.335,-0.386}},{{-0.380,0.668},{-1.152,-0.333}},{{-0.380,0.668},{-1.035,-0.299}},{{-0.380,0.668},{-0.995,-0.288}},
                    // F_B
                    {{0.461,0.614},{-0.071,1.388}},{{0.461,0.614},{-0.062,1.198}},{{0.461,0.614},{-0.055,1.076}},{{0.461,0.614},{-0.053,1.034}},{{0.529,0.705},{-0.066,1.282}},{{0.529,0.705},{-0.062,1.209}},{{0.529,0.705},{-0.054,1.044}},{{0.529,0.705},{-0.048,0.938}},{{0.548,0.730},{-0.072,1.401}},{{0.548,0.730},{-0.060,1.167}},{{0.548,0.730},{-0.052,1.008}},{{0.548,0.730},{-0.047,0.905}},{{0.675,0.899},{-0.062,1.206}},{{0.675,0.899},{-0.049,0.948}},{{0.675,0.899},{-0.042,0.818}},{{0.675,0.899},{-0.038,0.735}},{{0.714,0.951},{-0.059,1.140}},{{0.714,0.951},{-0.046,0.896}},{{0.714,0.951},{-0.040,0.774}},{{0.714,0.951},{-0.036,0.700}},{{0.768,0.012},{-0.897,1.061}},{{0.768,0.012},{-0.775,0.916}},{{0.768,0.012},{-0.696,0.822}},{{0.768,0.012},{-0.669,0.791}},{{0.881,0.014},{-0.829,0.980}},{{0.881,0.014},{-0.782,0.925}},{{0.881,0.014},{-0.675,0.798}},{{0.881,0.014},{-0.606,0.717}},{{0.913,0.014},{-0.906,1.071}},{{0.913,0.014},{-0.755,0.892}},{{0.913,0.014},{-0.652,0.770}},{{0.913,0.014},{-0.585,0.692}},{{1.124,0.018},{-0.780,0.922}},{{1.124,0.018},{-0.613,0.725}},{{1.124,0.018},{-0.529,0.626}},{{1.124,0.018},{-0.475,0.562}},{{1.189,0.019},{-0.737,0.872}},{{1.189,0.019},{-0.580,0.685}},{{1.189,0.019},{-0.500,0.592}},{{1.189,0.019},{-0.453,0.535}},
                    {{0.596,-0.484},{-0.691,-1.206}},{{0.596,-0.484},{-0.596,-1.041}},{{0.596,-0.484},{-0.535,-0.935}},{{0.596,-0.484},{-0.515,-0.899}},{{0.681,-0.356},{-0.327,-1.350}},{{0.681,-0.356},{-0.283,-1.166}},{{0.681,-0.356},{-0.254,-1.047}},{{0.681,-0.356},{-0.244,-1.007}},{{0.684,-0.555},{-0.638,-1.114}},{{0.684,-0.555},{-0.602,-1.051}},{{0.684,-0.555},{-0.520,-0.907}},{{0.684,-0.555},{-0.467,-0.815}},{{0.709,-0.575},{-0.697,-1.217}},{{0.709,-0.575},{-0.581,-1.014}},{{0.709,-0.575},{-0.501,-0.876}},{{0.709,-0.575},{-0.450,-0.786}},{{0.781,-0.408},{-0.302,-1.248}},{{0.781,-0.408},{-0.285,-1.177}},{{0.781,-0.408},{-0.246,-1.016}},{{0.781,-0.408},{-0.221,-0.912}},{{0.809,-0.423},{-0.330,-1.363}},{{0.809,-0.423},{-0.275,-1.136}},{{0.809,-0.423},{-0.238,-0.981}},{{0.809,-0.423},{-0.213,-0.881}},{{0.873,-0.708},{-0.600,-1.048}},{{0.873,-0.708},{-0.472,-0.824}},{{0.873,-0.708},{-0.407,-0.711}},{{0.873,-0.708},{-0.366,-0.639}},{{0.923,-0.749},{-0.567,-0.991}},{{0.923,-0.749},{-0.446,-0.779}},{{0.923,-0.749},{-0.385,-0.672}},{{0.923,-0.749},{-0.349,-0.609}},{{0.996,-0.521},{-0.284,-1.174}},{{0.996,-0.521},{-0.224,-0.923}},{{0.996,-0.521},{-0.193,-0.796}},{{0.996,-0.521},{-0.173,-0.715}},{{1.054,-0.551},{-0.269,-1.110}},{{1.054,-0.551},{-0.211,-0.872}},{{1.054,-0.551},{-0.183,-0.753}},{{1.054,-0.551},{-0.165,-0.682}},

                    // G
                    {{-1.090,-0.474},{-0.034,-1.141}},{{-1.090,-0.474},{-0.027,-0.897}},{{-1.090,-0.474},{-0.023,-0.775}},{{-1.090,-0.474},{-0.021,-0.701}},{{-1.031,-0.449},{-0.036,-1.207}},{{-1.031,-0.449},{-0.028,-0.949}},{{-1.031,-0.449},{-0.025,-0.819}},{{-1.031,-0.449},{-0.022,-0.736}},{{-0.837,-0.364},{-0.042,-1.402}},{{-0.837,-0.364},{-0.035,-1.168}},{{-0.837,-0.364},{-0.030,-1.008}},{{-0.837,-0.364},{-0.027,-0.906}},{{-0.808,-0.352},{-0.039,-1.283}},{{-0.808,-0.352},{-0.036,-1.211}},{{-0.808,-0.352},{-0.031,-1.045}},{{-0.808,-0.352},{-0.028,-0.938}},{{-0.704,-0.307},{-0.042,-1.389}},{{-0.704,-0.307},{-0.036,-1.199}},{{-0.704,-0.307},{-0.032,-1.077}},{{-0.704,-0.307},{-0.031,-1.035}},{{0.107,-0.761},{-0.013,-1.389}},{{0.107,-0.761},{-0.011,-1.199}},{{0.107,-0.761},{-0.010,-1.077}},{{0.107,-0.761},{-0.010,-1.036}},{{0.123,-0.873},{-0.012,-1.284}},{{0.123,-0.873},{-0.011,-1.211}},{{0.123,-0.873},{-0.010,-1.045}},{{0.123,-0.873},{-0.009,-0.939}},{{0.127,-0.904},{-0.013,-1.403}},{{0.127,-0.904},{-0.011,-1.169}},{{0.127,-0.904},{-0.010,-1.009}},{{0.127,-0.904},{-0.009,-0.906}},{{0.157,-1.113},{-0.011,-1.207}},{{0.157,-1.113},{-0.009,-0.949}},{{0.157,-1.113},{-0.008,-0.820}},{{0.157,-1.113},{-0.007,-0.736}},{{0.166,-1.177},{-0.011,-1.142}},{{0.166,-1.177},{-0.009,-0.898}},{{0.166,-1.177},{-0.007,-0.775}},{{0.166,-1.177},{-0.007,-0.701}},
                    {{-1.182,0.132},{0.030,-0.701}},{{-1.182,0.132},{0.033,-0.774}},{{-1.182,0.132},{0.038,-0.897}},{{-1.182,0.132},{0.049,-1.141}},{{-1.117,0.124},{0.031,-0.735}},{{-1.117,0.124},{0.035,-0.819}},{{-1.117,0.124},{0.040,-0.948}},{{-1.117,0.124},{0.051,-1.206}},{{-0.908,0.101},{0.039,-0.905}},{{-0.908,0.101},{0.043,-1.008}},{{-0.908,0.101},{0.050,-1.168}},{{-0.908,0.101},{0.060,-1.402}},{{-0.876,0.097},{0.040,-0.938}},{{-0.876,0.097},{0.044,-1.045}},{{-0.876,0.097},{0.052,-1.210}},{{-0.876,0.097},{0.055,-1.283}},{{-0.763,0.085},{0.044,-1.035}},{{-0.763,0.085},{0.046,-1.076}},{{-0.763,0.085},{0.051,-1.198}},{{-0.763,0.085},{0.059,-1.388}},{{-0.142,1.180},{0.007,0.701}},{{-0.142,1.180},{0.008,0.775}},{{-0.142,1.180},{0.009,0.898}},{{-0.142,1.180},{0.011,1.142}},{{-0.134,1.116},{0.007,0.736}},{{-0.134,1.116},{0.008,0.820}},{{-0.134,1.116},{0.010,0.949}},{{-0.134,1.116},{0.012,1.207}},{{-0.109,0.907},{0.009,0.906}},{{-0.109,0.907},{0.010,1.009}},{{-0.109,0.907},{0.012,1.169}},{{-0.109,0.907},{0.014,1.403}},{{-0.105,0.875},{0.009,0.939}},{{-0.105,0.875},{0.010,1.045}},{{-0.105,0.875},{0.012,1.211}},{{-0.105,0.875},{0.013,1.284}},{{-0.092,0.763},{0.010,1.036}},{{-0.092,0.763},{0.011,1.077}},{{-0.092,0.763},{0.012,1.199}},{{-0.092,0.763},{0.014,1.389}},
                    {{-1.034,0.586},{-0.017,1.141}},{{-1.034,0.586},{-0.014,0.897}},{{-1.034,0.586},{-0.012,0.775}},{{-1.034,0.586},{-0.011,0.701}},{{-0.978,0.554},{-0.018,1.207}},{{-0.978,0.554},{-0.014,0.949}},{{-0.978,0.554},{-0.012,0.819}},{{-0.978,0.554},{-0.011,0.736}},{{-0.794,0.450},{-0.021,1.403}},{{-0.794,0.450},{-0.018,1.169}},{{-0.794,0.450},{-0.015,1.009}},{{-0.794,0.450},{-0.014,0.906}},{{-0.767,0.434},{-0.020,1.284}},{{-0.767,0.434},{-0.018,1.211}},{{-0.767,0.434},{-0.016,1.045}},{{-0.767,0.434},{-0.014,0.939}},{{-0.668,0.379},{-0.021,1.389}},{{-0.668,0.379},{-0.018,1.199}},{{-0.668,0.379},{-0.016,1.077}},{{-0.668,0.379},{-0.016,1.036}},{{0.262,0.722},{-1.376,-0.190}},{{0.262,0.722},{-1.188,-0.164}},{{0.262,0.722},{-1.067,-0.147}},{{0.262,0.722},{-1.026,-0.142}},{{0.301,0.828},{-1.272,-0.175}},{{0.301,0.828},{-1.200,-0.166}},{{0.301,0.828},{-1.036,-0.143}},{{0.301,0.828},{-0.930,-0.128}},{{0.312,0.858},{-1.390,-0.192}},{{0.312,0.858},{-1.158,-0.160}},{{0.312,0.858},{-0.999,-0.138}},{{0.312,0.858},{-0.898,-0.124}},{{0.384,1.057},{-1.196,-0.165}},{{0.384,1.057},{-0.940,-0.130}},{{0.384,1.057},{-0.812,-0.112}},{{0.384,1.057},{-0.729,-0.101}},{{0.406,1.118},{-1.131,-0.156}},{{0.406,1.118},{-0.889,-0.123}},{{0.406,1.118},{-0.768,-0.106}},{{0.406,1.118},{-0.695,-0.096}},
                    {{-1.142,-0.329},{0.622,0.324}},{{-1.142,-0.329},{0.687,0.358}},{{-1.142,-0.329},{0.796,0.414}},{{-1.142,-0.329},{1.013,0.527}},{{-1.080,-0.311},{0.653,0.340}},{{-1.080,-0.311},{0.727,0.378}},{{-1.080,-0.311},{0.842,0.438}},{{-1.080,-0.311},{1.071,0.557}},{{-0.877,-0.253},{0.804,0.418}},{{-0.877,-0.253},{0.895,0.466}},{{-0.877,-0.253},{1.037,0.539}},{{-0.877,-0.253},{1.245,0.648}},{{-0.847,-0.244},{0.833,0.433}},{{-0.847,-0.244},{0.927,0.483}},{{-0.847,-0.244},{1.074,0.559}},{{-0.847,-0.244},{1.139,0.593}},{{-0.810,-0.870},{0.636,0.295}},{{-0.810,-0.870},{0.703,0.325}},{{-0.810,-0.870},{0.815,0.377}},{{-0.810,-0.870},{1.036,0.479}},{{-0.766,-0.823},{0.668,0.309}},{{-0.766,-0.823},{0.744,0.344}},{{-0.766,-0.823},{0.862,0.399}},{{-0.766,-0.823},{1.096,0.507}},{{-0.738,-0.213},{0.919,0.478}},{{-0.738,-0.213},{0.956,0.497}},{{-0.738,-0.213},{1.064,0.554}},{{-0.738,-0.213},{1.233,0.641}},{{-0.622,-0.668},{0.822,0.381}},{{-0.622,-0.668},{0.916,0.424}},{{-0.622,-0.668},{1.061,0.491}},{{-0.622,-0.668},{1.273,0.589}},{{-0.601,-0.645},{0.852,0.394}},{{-0.601,-0.645},{0.949,0.439}},{{-0.601,-0.645},{1.099,0.509}},{{-0.601,-0.645},{1.165,0.539}},{{-0.524,-0.562},{0.940,0.435}},{{-0.524,-0.562},{0.978,0.452}},{{-0.524,-0.562},{1.089,0.504}},{{-0.524,-0.562},{1.261,0.584}},
                    {{0.489,-0.593},{-1.267,0.570}},{{0.489,-0.593},{-1.094,0.492}},{{0.489,-0.593},{-0.982,0.442}},{{0.489,-0.593},{-0.945,0.425}},{{0.561,-0.680},{-1.171,0.526}},{{0.561,-0.680},{-1.105,0.497}},{{0.561,-0.680},{-0.954,0.429}},{{0.561,-0.680},{-0.856,0.385}},{{0.581,-0.503},{-0.964,1.001}},{{0.581,-0.503},{-0.832,0.864}},{{0.581,-0.503},{-0.747,0.776}},{{0.581,-0.503},{-0.719,0.746}},{{0.581,-0.704},{-1.280,0.575}},{{0.581,-0.704},{-1.066,0.479}},{{0.581,-0.704},{-0.920,0.414}},{{0.581,-0.704},{-0.826,0.372}},{{0.666,-0.577},{-0.891,0.925}},{{0.666,-0.577},{-0.840,0.872}},{{0.666,-0.577},{-0.725,0.753}},{{0.666,-0.577},{-0.651,0.676}},{{0.691,-0.598},{-0.973,1.010}},{{0.691,-0.598},{-0.811,0.842}},{{0.691,-0.598},{-0.700,0.727}},{{0.691,-0.598},{-0.629,0.653}},{{0.715,-0.867},{-1.101,0.495}},{{0.715,-0.867},{-0.866,0.389}},{{0.715,-0.867},{-0.747,0.336}},{{0.715,-0.867},{-0.671,0.302}},{{0.757,-0.917},{-1.041,0.468}},{{0.757,-0.917},{-0.819,0.368}},{{0.757,-0.917},{-0.707,0.318}},{{0.757,-0.917},{-0.640,0.288}},{{0.850,-0.736},{-0.838,0.870}},{{0.850,-0.736},{-0.659,0.684}},{{0.850,-0.736},{-0.569,0.590}},{{0.850,-0.736},{-0.511,0.530}},{{0.899,-0.778},{-0.792,0.822}},{{0.899,-0.778},{-0.623,0.646}},{{0.899,-0.778},{-0.538,0.558}},{{0.899,-0.778},{-0.487,0.505}},
                    {{0.587,0.495},{0.319,0.985}},{{0.587,0.495},{0.332,1.025}},{{0.587,0.495},{0.370,1.141}},{{0.587,0.495},{0.428,1.322}},{{0.673,0.568},{0.290,0.893}},{{0.673,0.568},{0.322,0.995}},{{0.673,0.568},{0.373,1.152}},{{0.673,0.568},{0.396,1.221}},{{0.693,0.331},{0.603,0.842}},{{0.693,0.331},{0.627,0.876}},{{0.693,0.331},{0.698,0.976}},{{0.693,0.331},{0.809,1.130}},{{0.698,0.589},{0.279,0.862}},{{0.698,0.589},{0.311,0.960}},{{0.698,0.589},{0.360,1.112}},{{0.698,0.589},{0.433,1.335}},{{0.795,0.379},{0.546,0.764}},{{0.795,0.379},{0.608,0.850}},{{0.795,0.379},{0.705,0.985}},{{0.795,0.379},{0.747,1.044}},{{0.824,0.393},{0.527,0.737}},{{0.824,0.393},{0.587,0.821}},{{0.824,0.393},{0.680,0.950}},{{0.824,0.393},{0.816,1.141}},{{0.859,0.725},{0.227,0.700}},{{0.859,0.725},{0.253,0.780}},{{0.859,0.725},{0.293,0.903}},{{0.859,0.725},{0.372,1.149}},{{0.909,0.767},{0.216,0.667}},{{0.909,0.767},{0.239,0.737}},{{0.909,0.767},{0.277,0.854}},{{0.909,0.767},{0.352,1.086}},{{1.015,0.484},{0.428,0.599}},{{1.015,0.484},{0.477,0.667}},{{1.015,0.484},{0.552,0.772}},{{1.015,0.484},{0.703,0.982}},{{1.073,0.512},{0.408,0.570}},{{1.073,0.512},{0.451,0.630}},{{1.073,0.512},{0.522,0.730}},{{1.073,0.512},{0.664,0.928}},
                    {{0.630,0.440},{-1.135,-0.802}},{{0.630,0.440},{-0.980,-0.692}},{{0.630,0.440},{-0.880,-0.622}},{{0.630,0.440},{-0.846,-0.598}},{{0.664,0.385},{-0.719,-1.189}},{{0.664,0.385},{-0.620,-1.027}},{{0.664,0.385},{-0.557,-0.922}},{{0.664,0.385},{-0.536,-0.886}},{{0.722,0.505},{-1.049,-0.741}},{{0.722,0.505},{-0.989,-0.699}},{{0.722,0.505},{-0.854,-0.603}},{{0.722,0.505},{-0.767,-0.542}},{{0.749,0.523},{-1.146,-0.809}},{{0.749,0.523},{-0.955,-0.674}},{{0.749,0.523},{-0.824,-0.582}},{{0.749,0.523},{-0.740,-0.523}},{{0.762,0.442},{-0.664,-1.099}},{{0.762,0.442},{-0.626,-1.037}},{{0.762,0.442},{-0.541,-0.895}},{{0.762,0.442},{-0.486,-0.804}},{{0.790,0.458},{-0.726,-1.201}},{{0.790,0.458},{-0.604,-1.000}},{{0.790,0.458},{-0.522,-0.864}},{{0.790,0.458},{-0.469,-0.775}},{{0.922,0.644},{-0.986,-0.697}},{{0.922,0.644},{-0.775,-0.548}},{{0.922,0.644},{-0.669,-0.473}},{{0.922,0.644},{-0.601,-0.425}},{{0.973,0.564},{-0.624,-1.033}},{{0.973,0.564},{-0.491,-0.813}},{{0.973,0.564},{-0.424,-0.701}},{{0.973,0.564},{-0.381,-0.630}},{{0.975,0.681},{-0.932,-0.659}},{{0.975,0.681},{-0.733,-0.518}},{{0.975,0.681},{-0.633,-0.447}},{{0.975,0.681},{-0.573,-0.405}},{{1.029,0.596},{-0.590,-0.977}},{{1.029,0.596},{-0.464,-0.768}},{{1.029,0.596},{-0.401,-0.663}},{{1.029,0.596},{-0.363,-0.600}},
                    {{0.509,-0.575},{0.434,-0.940}},{{0.509,-0.575},{0.452,-0.978}},{{0.509,-0.575},{0.503,-1.089}},{{0.509,-0.575},{0.583,-1.261}},{{0.584,-0.660},{0.394,-0.852}},{{0.584,-0.660},{0.438,-0.949}},{{0.584,-0.660},{0.508,-1.100}},{{0.584,-0.660},{0.538,-1.166}},{{0.605,-0.684},{0.380,-0.823}},{{0.605,-0.684},{0.423,-0.916}},{{0.605,-0.684},{0.490,-1.061}},{{0.605,-0.684},{0.588,-1.274}},{{0.731,-0.234},{0.191,-1.018}},{{0.731,-0.234},{0.199,-1.059}},{{0.731,-0.234},{0.221,-1.179}},{{0.731,-0.234},{0.256,-1.366}},{{0.745,-0.842},{0.309,-0.668}},{{0.745,-0.842},{0.344,-0.744}},{{0.745,-0.842},{0.398,-0.862}},{{0.745,-0.842},{0.506,-1.096}},{{0.787,-0.891},{0.294,-0.637}},{{0.787,-0.891},{0.325,-0.703}},{{0.787,-0.891},{0.376,-0.815}},{{0.787,-0.891},{0.479,-1.036}},{{0.839,-0.269},{0.173,-0.923}},{{0.839,-0.269},{0.193,-1.028}},{{0.839,-0.269},{0.223,-1.190}},{{0.839,-0.269},{0.237,-1.262}},{{0.870,-0.278},{0.167,-0.891}},{{0.870,-0.278},{0.186,-0.992}},{{0.870,-0.278},{0.216,-1.149}},{{0.870,-0.278},{0.259,-1.379}},{{1.071,-0.343},{0.136,-0.723}},{{1.071,-0.343},{0.151,-0.805}},{{1.071,-0.343},{0.175,-0.933}},{{1.071,-0.343},{0.223,-1.187}},{{1.132,-0.363},{0.129,-0.689}},{{1.132,-0.363},{0.143,-0.762}},{{1.132,-0.363},{0.166,-0.882}},{{1.132,-0.363},{0.211,-1.122}},

                    // H
                    {{-1.013,-0.623},{-1.023,0.508}},{{-1.013,-0.623},{-0.804,0.399}},{{-1.013,-0.623},{-0.694,0.345}},{{-1.013,-0.623},{-0.628,0.312}},{{-0.957,-0.589},{-1.081,0.537}},{{-0.957,-0.589},{-0.850,0.422}},{{-0.957,-0.589},{-0.734,0.364}},{{-0.957,-0.589},{-0.659,0.327}},{{-0.778,-0.479},{-1.257,0.624}},{{-0.778,-0.479},{-1.047,0.520}},{{-0.778,-0.479},{-0.904,0.449}},{{-0.778,-0.479},{-0.812,0.403}},{{-0.750,-0.462},{-1.150,0.571}},{{-0.750,-0.462},{-1.085,0.539}},{{-0.750,-0.462},{-0.936,0.465}},{{-0.750,-0.462},{-0.841,0.418}},{{-0.654,-0.403},{-1.245,0.618}},{{-0.654,-0.403},{-1.074,0.533}},{{-0.654,-0.403},{-0.965,0.479}},{{-0.654,-0.403},{-0.928,0.461}},{{-0.622,-1.013},{-0.625,0.955}},{{-0.622,-1.013},{-0.491,0.751}},{{-0.622,-1.013},{-0.424,0.648}},{{-0.622,-1.013},{-0.384,0.587}},{{-0.588,-0.958},{-0.661,1.010}},{{-0.588,-0.958},{-0.520,0.794}},{{-0.588,-0.958},{-0.449,0.686}},{{-0.588,-0.958},{-0.403,0.616}},{{-0.478,-0.778},{-0.768,1.174}},{{-0.478,-0.778},{-0.640,0.978}},{{-0.478,-0.778},{-0.552,0.844}},{{-0.478,-0.778},{-0.496,0.758}},{{-0.461,-0.751},{-0.703,1.074}},{{-0.461,-0.751},{-0.663,1.013}},{{-0.461,-0.751},{-0.572,0.875}},{{-0.461,-0.751},{-0.514,0.786}},{{-0.402,-0.654},{-0.761,1.163}},{{-0.402,-0.654},{-0.657,1.004}},{{-0.402,-0.654},{-0.590,0.901}},{{-0.402,-0.654},{-0.567,0.867}},
                    {{-1.073,0.512},{-1.106,-0.282}},{{-1.073,0.512},{-0.870,-0.222}},{{-1.073,0.512},{-0.751,-0.192}},{{-1.073,0.512},{-0.680,-0.173}},{{-1.014,0.484},{-1.170,-0.299}},{{-1.014,0.484},{-0.920,-0.235}},{{-1.014,0.484},{-0.794,-0.203}},{{-1.014,0.484},{-0.713,-0.182}},{{-0.824,0.394},{-1.359,-0.347}},{{-0.824,0.394},{-1.132,-0.289}},{{-0.824,0.394},{-0.978,-0.250}},{{-0.824,0.394},{-0.878,-0.224}},{{-0.795,0.380},{-1.244,-0.318}},{{-0.795,0.380},{-1.173,-0.300}},{{-0.795,0.380},{-1.013,-0.259}},{{-0.795,0.380},{-0.910,-0.232}},{{-0.693,0.331},{-1.346,-0.344}},{{-0.693,0.331},{-1.162,-0.297}},{{-0.693,0.331},{-1.044,-0.266}},{{-0.693,0.331},{-1.004,-0.256}},{{0.231,0.733},{0.377,-0.965}},{{0.231,0.733},{0.392,-1.003}},{{0.231,0.733},{0.436,-1.117}},{{0.231,0.733},{0.505,-1.294}},{{0.265,0.841},{0.342,-0.875}},{{0.265,0.841},{0.380,-0.974}},{{0.265,0.841},{0.441,-1.128}},{{0.265,0.841},{0.467,-1.196}},{{0.274,0.871},{0.330,-0.844}},{{0.274,0.871},{0.367,-0.940}},{{0.274,0.871},{0.425,-1.089}},{{0.274,0.871},{0.510,-1.307}},{{0.338,1.072},{0.268,-0.686}},{{0.338,1.072},{0.298,-0.763}},{{0.338,1.072},{0.345,-0.884}},{{0.338,1.072},{0.439,-1.125}},{{0.357,1.134},{0.255,-0.653}},{{0.357,1.134},{0.282,-0.722}},{{0.357,1.134},{0.327,-0.836}},{{0.357,1.134},{0.415,-1.063}},
                    {{0.402,-0.655},{0.314,0.987}},{{0.402,-0.655},{0.326,1.027}},{{0.402,-0.655},{0.363,1.143}},{{0.402,-0.655},{0.421,1.324}},{{0.461,-0.751},{0.284,0.895}},{{0.461,-0.751},{0.317,0.996}},{{0.461,-0.751},{0.367,1.154}},{{0.461,-0.751},{0.389,1.224}},{{0.478,-0.778},{0.274,0.864}},{{0.478,-0.778},{0.306,0.962}},{{0.478,-0.778},{0.354,1.114}},{{0.478,-0.778},{0.425,1.337}},{{0.587,0.496},{-0.359,-1.342}},{{0.587,0.496},{-0.310,-1.159}},{{0.587,0.496},{-0.279,-1.041}},{{0.587,0.496},{-0.268,-1.000}},{{0.588,-0.494},{1.004,0.255}},{{0.588,-0.494},{1.044,0.265}},{{0.588,-0.494},{1.163,0.295}},{{0.588,-0.494},{1.347,0.342}},{{0.588,-0.958},{0.223,0.701}},{{0.588,-0.958},{0.248,0.781}},{{0.588,-0.958},{0.287,0.905}},{{0.588,-0.958},{0.366,1.151}},{{0.622,-1.013},{0.212,0.668}},{{0.622,-1.013},{0.235,0.738}},{{0.622,-1.013},{0.272,0.855}},{{0.622,-1.013},{0.346,1.088}},{{0.673,0.569},{-0.332,-1.240}},{{0.673,0.569},{-0.313,-1.170}},{{0.673,0.569},{-0.270,-1.010}},{{0.673,0.569},{-0.243,-0.907}},{{0.674,-0.567},{0.910,0.231}},{{0.674,-0.567},{1.013,0.257}},{{0.674,-0.567},{1.174,0.298}},{{0.674,-0.567},{1.244,0.316}},{{0.676,0.365},{0.985,0.320}},{{0.676,0.365},{1.024,0.333}},{{0.676,0.365},{1.141,0.371}},{{0.676,0.365},{1.321,0.430}},{{0.698,0.589},{-0.363,-1.355}},
                    {{0.698,0.589},{-0.302,-1.129}},{{0.698,0.589},{-0.261,-0.975}},{{0.698,0.589},{-0.234,-0.875}},{{0.699,-0.588},{0.878,0.223}},{{0.699,-0.588},{0.978,0.248}},{{0.699,-0.588},{1.133,0.288}},{{0.699,-0.588},{1.360,0.345}},{{0.776,0.418},{0.893,0.290}},{{0.776,0.418},{0.994,0.323}},{{0.776,0.418},{1.152,0.375}},{{0.776,0.418},{1.221,0.397}},{{0.804,0.433},{0.862,0.280}},{{0.804,0.433},{0.959,0.312}},{{0.804,0.433},{1.111,0.362}},{{0.804,0.433},{1.334,0.434}},{{0.859,0.725},{-0.312,-1.166}},{{0.859,0.725},{-0.246,-0.917}},{{0.859,0.725},{-0.212,-0.792}},{{0.859,0.725},{-0.190,-0.711}},{{0.860,-0.723},{0.713,0.181}},{{0.860,-0.723},{0.794,0.202}},{{0.860,-0.723},{0.920,0.234}},{{0.860,-0.723},{1.170,0.297}},{{0.908,0.767},{-0.295,-1.103}},{{0.908,0.767},{-0.232,-0.867}},{{0.908,0.767},{-0.200,-0.749}},{{0.908,0.767},{-0.181,-0.677}},{{0.910,-0.765},{0.680,0.173}},{{0.910,-0.765},{0.751,0.191}},{{0.910,-0.765},{0.870,0.221}},{{0.910,-0.765},{1.106,0.281}},{{0.989,0.534},{0.700,0.228}},{{0.989,0.534},{0.779,0.254}},{{0.989,0.534},{0.903,0.294}},{{0.989,0.534},{1.148,0.374}},{{1.047,0.564},{0.667,0.217}},{{1.047,0.564},{0.737,0.240}},{{1.047,0.564},{0.854,0.278}},{{1.047,0.564},{1.086,0.353}}

                };

                const std::vector<complex<double>> li22_values = {
                    {0.4419958572037096, 0.0}, {0.40310586453967534, 0.0}, {0.7389277874689575, -0.48492324493010136}, {0.3013035215086935, 0.0}, {0.7561067400674584, -0.35035852312356913}, {0.5045165399680003, 0.0}, {0.45975365575848176, 0.0}, {0.8299504630169592, -0.5576409766147369}, {0.34300324804620463, 0.0}, {0.8613316133568842, -0.40995236316805067}, {0.6143785369243933, -0.0011143627827525596}, {0.5586461062108299, -8.441583062031283E-4}, {0.9844796563156657, -0.7023749302111062}, {0.4147674045806786, -4.1896848466970035E-4}, {1.033058157602941, -0.527137366217685}, {0.802233113938307, -0.03613118512220782}, {0.6078776559155328, -0.0048728312088201714}, {1.2391387406613363, -1.063559747492915}, {0.28769637316218544, 0.0}, {1.3303883308111262, -0.846486729405542}, {0.13434056410260542, 0.08101838245139015}, {0.1476033100528671, 0.08841580521417325}, {0.16972242970594031, 0.10081516556251546}, {0.21242049954839182, 0.1240374918624001}, {0.12792574033518367, 0.07724284768543876}, {0.1417012854096824, 0.08518073502662649}, {0.1628254540176215, 0.09697084831230257}, {0.2039592748883834, 0.11929593440634711}, {0.10978837848886264, 0.06693935491826823}, {0.12161762360672905, 0.07364882376854591}, {0.13960005929084585, 0.08384192219440623}, {0.16571984679058974, 0.09825484242111449},
                    {0.10672821740883616, 0.06517616758356486}, {0.1183060667599666, 0.07167575975986278}, {0.13587633934467347, 0.08159716297024588}, {0.14353936164244818, 0.08593936569489302}, {0.09252025777543714, 0.056771007772329865}, {0.09608427707312708, 0.058810403499514774}, {0.1064265456615928, 0.06469983131394248}, {0.12226403608494002, 0.07366451690153397}, {-0.14893685171073834, -0.0564860706381633}, {-0.15529786149096456, -0.05811443460401682}, {-0.17361824689545122, -0.06293242362937071}, {-0.20218852729715442, -0.06940366371904894}, {-0.17877248890587275, -0.05750868401117868}, {-0.19986533620289593, -0.06182903389989096}, {-0.23253238543964283, -0.06767937523240164}, {-0.24686310181417337, -0.06997693990049836}, {-0.18527234352422892, -0.05667486566561475}, {-0.20700395256516632, -0.060937637613056476}, {-0.2406987530441418, -0.06646909132719481}, {-0.2900241518123896, -0.07287739205277699}, {-0.2260358772792168, -0.049339084685578904}, {-0.2523288620351982, -0.052524311044114924}, {-0.2931056256698028, -0.05585995389881315}, {-0.37334761470961475, -0.059153686138062156}, {-0.2395564279825253, -0.0464890994740762}, {-0.26517917608172015, -0.04878890679431158}, {-0.30787598396488747, -0.05128111371950972}, {-0.3914327533822781, -0.05286346591378477},
                    {0.1498664712106308, -0.04329721602695069}, {0.16461363146064567, -0.04740294708838766}, {0.18869707379652212, -0.05357733173040635}, {0.2356442355626882, -0.0656490608549829}, {0.14319202588863078, -0.04135891397259295}, {0.1583691483173708, -0.045415859789127655}, {0.18163698266712736, -0.051378925048532365}, {0.22678114350782153, -0.06286109275558993}, {0.12303271669622141, -0.03522062507349637}, {0.1360838129579093, -0.038615178101284835}, {0.15610367242007017, -0.04376315362527805}, {0.18468592708679069, -0.05105245191559127}, {0.11970092990169759, -0.03424941994231134}, {0.13248040569158095, -0.03761883073814939}, {0.15191632503202945, -0.04261396236406023}, {0.1603211687124788, -0.04478975962295828}, {0.10393464156876019, -0.02966941995289303}, {0.10781878009846654, -0.030722734728802065}, {0.11926424198403816, -0.03374415378984602}, {0.13680834993204666, -0.038277231699794134}, {0.1856792722251348, -0.04626970975178768}, {0.1600813974123559, -0.0423826465704365}, {0.1436433108485977, -0.039550965489584984}, {0.13806745713426, -0.038531268170834874}, {0.21935497882047672, -0.06299003443246631}, {0.20685768100962765, -0.0608302017859752},

                    {0.178147765355177, -0.05522514487582307}, {0.15966572228348333, -0.051263339020177406}, {0.2555613173176443, -0.07206228811094872}, {0.21271329809814743, -0.06462650978262285}, {0.1832015813432548, -0.058613843012578104}, {0.16422641055666032, -0.05445432420495054}, {0.31481893371874037, -0.10940751973056573}, {0.24660122243100613, -0.09351164748885266}, {0.21211981127403678, -0.08407884653392078}, {0.18998145799007027, -0.0772749587243711}, {0.3268001435348822, -0.12110671272867268}, {0.2555465227147269, -0.1027583349142338}, {0.21998532853715425, -0.09224183596333313}, {0.19855298486072104, -0.0854038213092291}, {0.029047330642818725, -0.16444887423347665}, {0.03278504048386921, -0.18058019274598303}, {0.039205771954900825, -0.20685016602803485}, {0.052403720622251626, -0.2578833930968447}, {0.02655856928706468, -0.15706366548772488}, {0.0303134715342801, -0.17351768165760495}, {0.03619197518418815, -0.1990392138338246}, {0.0484660973415032, -0.24792823771386163}, {0.019368317341291023, -0.13507081998232287}, {0.022071992354961687, -0.14930427748714672}, {0.026607946312599218, -0.17114223359066338}, {0.033618474263653164, -0.2023945929274804}, {0.018256124431277127, -0.13148736255651106},
                    {0.02096185025561114, -0.14549205103260252}, {0.02523895924445685, -0.16667725464415062}, {0.027191491023300207, -0.17589542157639126}, {0.01365293161495866, -0.11412007391601985}, {0.014402710484406826, -0.1184054945070855}, {0.01664318522489782, -0.13094706131667364}, {0.02017984836228104, -0.15013555701565462}, {0.04507431022722663, -0.17298525340155635}, {0.049960672445986454, -0.1897387140786133}, {0.05792488358586034, -0.2171735463225659}, {0.07390441603182296, -0.2701326560912305}, {0.04079768936906536, -0.16516414362126186}, {0.045531316368400354, -0.18232634291864214}, {0.052838805358138785, -0.20887667863698}, {0.06751730151312074, -0.2596838964266994}, {0.028416837717721886, -0.14214209282902318}, {0.03178025329992475, -0.15703001172530848}, {0.03692318416380236, -0.17974604410794937}, {0.04475885340760327, -0.2123347795039635}, {0.026296682050037955, -0.1383198469327983}, {0.029494119000842322, -0.15287128154721835}, {0.03447091123962576, -0.1750146375447387}, {0.03665758125304347, -0.18451557316631778}, {0.01933229657018153, -0.11971425600833559}, {0.020164575438224915, -0.12429505557768258}, {0.02267446667404157, -0.1372708989990062}, {0.02653616853179941, -0.15722438540249964},
                    {-0.3015999706223056, 0.15890941785317136}, {-0.22677680972182346, 0.11215406898452074}, {-0.1909718237060427, 0.0931974603363348}, {-0.1705623879646445, 0.08280119216703419}, {-0.28936502221924343, 0.15481696132357703}, {-0.21770093274196567, 0.10918344397240265}, {-0.1835908820172496, 0.09070294963474679}, {-0.16238501807072114, 0.0797183430396301}, {-0.2300955211369251, 0.1259617083215001}, {-0.18516358925437437, 0.09702318200768789}, {-0.1561051064083818, 0.08075830445796263}, {-0.13808852221191645, 0.07119338575658088}, {-0.19311396696429206, 0.10293958994888719}, {-0.18018243991096927, 0.09521673352316373}, {-0.151862341075561, 0.07913567281209723}, {-0.13433296009662254, 0.06962933916736558}, {-0.16122728864467328, 0.08779075991700847}, {-0.13575175502428946, 0.07295628016630176}, {-0.12023334671743678, 0.06415567415698611}, {-0.11498497386292662, 0.06123720129550352}, {0.025655837837760487, 0.3305475692837295}, {0.01105358679033408, 0.2566545316223432}, {0.005825159911502897, 0.21972720491368258}, {0.0033352414173200007, 0.19799427535782188}, {0.027996982365962708, 0.3167501658127887}, {0.01323606368024155, 0.24617426783631036}, {0.007791924233075953, 0.21075960249591918},
                    {0.005006195254009693, 0.1882546062989443}, {0.03132274717100314, 0.2536231410302836}, {0.020405229314535145, 0.20968930353528403}, {0.01450011079221387, 0.17960701964483797}, {0.011170149283858332, 0.16044722309979828}, {0.02421780014881625, 0.2163547057617463}, {0.02116734221098593, 0.20356578144732707}, {0.01516640002357767, 0.17451382265988452}, {0.011997492741678205, 0.15593192364079578}, {0.023500153974088445, 0.1822646800935379}, {0.017446100299706824, 0.15625201369914202}, {0.014137961445581871, 0.13967982739276882}, {0.01302572399721761, 0.13407233454369918}, {-0.15942359627334624, -0.14240986921147436}, {-0.1397201838248928, -0.12382079787074722}, {-0.126874029307098, -0.11161275508761458}, {-0.12235456636561959, -0.10743013980728369}, {-0.18255406146179165, -0.1840887141368437}, {-0.1733858388922922, -0.17414526945400327}, {-0.15208830863183886, -0.1515662087613232}, {-0.1381796352212113, -0.13676051128048466}, {-0.20728733527826826, -0.21725465748462824}, {-0.1766223573748662, -0.18299279723125156}, {-0.1550148214829207, -0.15929438784990668}, {-0.14064033551064975, -0.14377421250242026}, {-0.23545083080107254, -0.3048098972250517}, {-0.19096579362886956, -0.24372638734269936},
                    {-0.1676489151546906, -0.21208656805524484}, {-0.15226368860921685, -0.19169172667303958}, {-0.2376535419805581, -0.32769070102808207}, {-0.19295436951216516, -0.2619759454617405}, {-0.16945904174223314, -0.22837687919016325}, {-0.15494128707994986, -0.20763461158346486}, {-0.22803929582435, 0.21061370127318768}, {-0.1977203274734887, 0.18562199942586285}, {-0.17792323810036909, 0.16887294335648328}, {-0.1711683288830534, 0.1632831556631567}, {-0.3218199067303713, 0.2878915752901592}, {-0.3042070420441596, 0.27429302475875805}, {-0.2636608201434986, 0.24185110203969182}, {-0.23732654673070502, 0.22047320380709523}, {-0.3966796204353993, 0.3464806750972851}, {-0.3328712553794686, 0.2977067406896192}, {-0.2887007308874585, 0.2627429522388327}, {-0.2597849804324281, 0.23966398098042196}, {-0.9022592552112103, 0.3312748867965447}, {-0.7254688050555869, 0.27773628804901457}, {-0.6335288986734566, 0.24822562117905878},

                    {-0.5731408135665682, 0.22795393038294476}, {-0.9951476222959155, 0.2293028918553035}, {-0.8020921058974375, 0.19462596049430525}, {-0.7013022477723863, 0.17615204818586316}, {-0.6396247685271642, 0.16291005617931242}, {-0.18477399764225189, 0.1459440094156923}, {-0.1618254393636121, 0.1270855380379181}, {-0.14675531859616406, 0.11479292219653157}, {-0.14155780190556203, 0.11069872865706894}, {-0.26268002633527865, -0.004138953239839982}, {-0.22947488526734047, -0.005193431459769924}, {-0.20758256773328565, -0.00575405796729869}, {-0.20014136462884666, -0.0059459949871930795}, {-0.21520297650368436, 0.19448159944561083}, {-0.20429476758849496, 0.1842916707344988}, {-0.17887652473842072, 0.16074288889216015}, {-0.16230404829320824, 0.14533260365681414}, {-0.2456781176054964, 0.23144833912743512}, {-0.2089046698307141, 0.19570350881749665}, {-0.183170542032366, 0.17058440277031497}, {-0.16594568627862147, 0.15419857715285806}, {-0.32978506914805994, 0.019683022242442803}, {-0.3126632126104123, 0.01776149410590072}, {-0.2731710361259017, 0.013560040938458516}, {-0.2471668326774869, 0.011106691553395304}, {-0.38475910443160233, 0.03435473495835717}, {-0.3263278606510973, 0.025992606606840944}, {-0.2852918342458344, 0.020837021587735375},
                    {-0.2582562355980972, 0.01742264347254215}, {-0.28114713702523453, 0.3415570286978875}, {-0.22722609206862643, 0.27434719051667766}, {-0.19902293956843486, 0.2392758093669638}, {-0.18059561414794145, 0.21667843180459534}, {-0.2829243624279876, 0.37140035375052105}, {-0.22860004170161297, 0.29837110400789957}, {-0.200106070307992, 0.26040478973752035}, {-0.1829762389808005, 0.2376398297732638}, {-0.5046500959416687, 0.12632193565604943}, {-0.40714620849517225, 0.09709840475089448}, {-0.35595086049133295, 0.08233673074190406}, {-0.32259137109535285, 0.07295580908462496}, {-0.5291127857035445, 0.15931191742108575}, {-0.4268633884261188, 0.12300530890185857}, {-0.3737195701225491, 0.10548695466209868}, {-0.3413829014946886, 0.09452346175438953}, {0.18148636366333903, -0.1862818915995943}, {0.14190091712971764, -0.15130342898494487}, {0.12219593191013728, -0.13283874861294587}, {0.11020470951751443, -0.1214020803624809}, {0.1755410983975878, -0.17827390048484215}, {0.13738000133056616, -0.14477178039955385}, {0.11802428369391282, -0.12717781570791553}, {0.10583388483201656, -0.1154560494442185}, {0.14393781650892645, -0.14328258290225993}, {0.11950486720820219, -0.12236909745811109}, {0.1028175319710744, -0.10741976038018779}, {0.0921770502506731, -0.09762942889520934},
                    {0.12400415010051535, -0.12506265466796473}, {0.11697950014576675, -0.11884692781426587}, {0.10063148559883132, -0.10434628674683502}, {0.09008748313266197, -0.09474343765274469}, {0.10621243197139396, -0.1058241895045764}, {0.09143280207666159, -0.0929550319392861}, {0.08196186952415126, -0.08443029073350874}, {0.07867891930285567, -0.08147900899437072}, {0.011243074905694872, 0.17867176132139767}, {0.009746808049971785, 0.15607377134908765}, {0.008808417235383528, 0.1412967518733417}, {0.008532328296257467, 0.13627992620386165}, {0.022037540180913195, 0.2120718877623961}, {0.0208256854836526, 0.2010375250470921}, {0.01822785265410702, 0.17553843899088856}, {0.016486584609886515, 0.1589654311053366}, {0.028264954415106347, 0.24410453148173605}, {0.023926647288489256, 0.20678152189602872}, {0.020949735669008703, 0.18058883271458137}, {0.018942733851051833, 0.16342731573789776}, {0.05617714977618604, 0.30264907068996094}, {0.04524093702514698, 0.24324084112166844}, {0.039599205823481975, 0.21261455652122407}, {0.03577387249399344, 0.19232969149108037}, {0.06514865397686265, 0.31533104035945647}, {0.05248640604350285, 0.2534377828413348}, {0.045634611313424944, 0.2213216610001889}, {0.04175822838196919, 0.20160015783323315}, {-0.010615878108580501, -0.16628898552145976},
                    {-0.010254876683434777, -0.18332858375784555}, {-0.009102205407828023, -0.21180724739750764}, {-0.0046827327146043984, -0.26730398899965135}, {-0.01027148442374759, -0.15832541911621512}, {-0.00977951826064577, -0.17613280962208888}, {-0.008718790748275751, -0.20324784831111906}, {-0.004576924590203646, -0.2565264726518395}, {-0.009152593488950865, -0.13629725662771439}, {-0.008895856365753494, -0.1515588268879296}, {-0.00789939439987819, -0.17506644149275052}, {-0.005503810851163138, -0.20894620947982467}, {-0.008918518975194458, -0.13266518633550045}, {-0.008681203600399168, -0.1475531869166175}, {-0.007596162610420176, -0.1703171340289451}, {-0.007019710336459539, -0.18030516389553997}, {-0.008432041554618686, -0.11482584824523868}, {-0.008370611366629152, -0.11931879701499981}, {-0.008143176541118813, -0.13263080002302277}, {-0.007341670448997029, -0.1531863943581242}, {0.111403200959104, -0.1533374537645078}, {0.12204455393486996, -0.16853225560930074}, {0.13950037388394776, -0.19332001748645755}, {0.172877899672784, -0.24121238507972542}, {0.1037119354055322, -0.14763894428250213}, {0.11442975546378636, -0.16342966154933025}, {0.13045644785178823, -0.18736678165854714}, {0.16170814542039125, -0.23373317439498376}, {0.0806969136306363, -0.129993854665725},
                    {0.08901628838790565, -0.14378288115025545}, {0.10160089544903372, -0.16489890451890918}, {0.11955615669137365, -0.19500743553255678}, {0.07715865812027693, -0.1269079111005564}, {0.0850537764318468, -0.14027095732420336}, {0.09709582060906663, -0.16089731621168513}, {0.10227875312321455, -0.1698528840373622}, {0.06273177069116935, -0.11118498893479933}, {0.06493758021918507, -0.11534402347946685}, {0.0716185980137678, -0.12750175131441482}, {0.08175125430746544, -0.1461540852863434}, {0.2127507778754989, 0.14929177697261942}, {0.16776114701711187, 0.12221349875072063}, {0.14511192957036745, 0.10774118126529864}, {0.13129217041732727, 0.09870829005159587}, {0.2052382206550476, 0.14245311965603058}, {0.16197047015934807, 0.11653421582146972}, {0.13995961505018728, 0.10266001712039197}, {0.12581516728899828, 0.09354001313279037}, {0.16852734736185893, 0.11314656016568558}, {0.14081304374400094, 0.09726688153462315}, {0.12176729787009671, 0.08570974228437746}, {0.10937947534200082, 0.07811276224958039}, {0.1454896383519304, 0.09926014198757804}, {0.1373884548104476, 0.09439940598423058}, {0.11871458434286372, 0.08329110562987434}, {0.10677933952564068, 0.07585504796680036}, {0.12462387763282612, 0.08315652564828957}, {0.10778257311911255, 0.07337268855127982},
                    {0.09691496178262354, 0.06684828020006629}, {0.09321778784771907, 0.06467316882399819}, {0.19394221924353072, -0.06379985173859559}, {0.16913363051163588, -0.053247425441833536}, {0.15284480063625194, -0.04672953310822764}, {0.14732546981627775, -0.04447151460790041}, {0.23609327210249687, -0.06693831153537078}, {0.2237389585299827, -0.061942364987159326}, {0.19492684857427936, -0.051328604970812826}, {0.17596461654185194, -0.04478742388136225}, {0.27459468680017385, -0.07748318165409662}, {0.23193691016827622, -0.06096185736518267}, {0.20190774422103838, -0.05028711246621597}, {0.18250027600663368, -0.04381651554618496}, {0.3549743359864834, -0.07072150810282762}, {0.2835595952604031, -0.049286479105854486}, {0.24676553765240028, -0.039710951949743695}, {0.22262265883821675, -0.033726417763643074}, {0.37374882106483653, -0.06504101323995393}, {0.2983598176214901, -0.04437538968243615}, {0.2595608998314085, -0.03527847119133365}, {0.23585666112411507, -0.030109300052662324}, {0.09740620514686515, 0.12538803710344482}, {0.10747794966275803, 0.13728797281202465}, {0.12436653350903448, 0.1566063112145761}, {0.15754099298181057, 0.19384462294074378}, {0.09267954134745571, 0.11992720355931241}, {0.10312400537713096, 0.13214316943569715},
                    {0.11920868011091912, 0.15086192544553922}, {0.15095864306179863, 0.18659565882650542}, {0.07857311953136685, 0.10372304014642482}, {0.08735873821799599, 0.11443663811698161}, {0.10110230695281985, 0.13054044424456768}, {0.12097587559704762, 0.1536008530320272}, {0.07648903155989044, 0.10113797723560952}, {0.08499564367204389, 0.11160122255403468}, {0.09834479759181497, 0.1273385489078338}, {0.1042029710073237, 0.1342052381951344}, {-0.03271866579718852, 0.1682215201334314}, {-0.03460260166645224, 0.18525908384293377}, {-0.03771194449017656, 0.21356577535198773}, {-0.04222405788057504, 0.2682312829021116},

                    {-0.03247275908908964, 0.1602662181849286}, {-0.03475931242829189, 0.17787461374903923}, {-0.03799192541179597, 0.20496016155251628}, {-0.042740977967962834, 0.25751036457250365}, {0.06560041541674841, 0.08827113959820339}, {0.06823737367566574, 0.09149419284594926}, {0.07585765439299703, 0.10092540471191083}, {0.08781305500169873, 0.11514785908261027}, {-0.0318327743134226, 0.13624013614629688}, {-0.0341761197303164, 0.1513021373837882}, {-0.037491017166137675, 0.17431799775527512}, {-0.04155436915579114, 0.2074225001382164}, {-0.031436448026547444, 0.13263512375794667}, {-0.033850792374873075, 0.1472636500399359}, {-0.03725091964001348, 0.1696660808812882}, {-0.038503185688226194, 0.17939893087705538}, {-0.029475443043984022, 0.11416929918330446}, {-0.030266637584849935, 0.11864750611167169}, {-0.032698969048289124, 0.131723225296535}, {-0.0360531895726246, 0.15176402030268976}, {0.17481617060276133, 0.15101902708232195}, {0.15359468673837348, 0.12932107829586412}, {0.1394963954280848, 0.11534711415286843}, {0.13468939572278527, 0.11078106518785936}, {0.2227740112229336, 0.17252074874167933}, {0.2118457724439169, 0.16206117827047706}, {0.18587254832218, 0.13843564468500752}, {0.16854569833994007, 0.1232473072006238}, {0.1949068066213305, 0.1618846879347804},
                    {0.1710918023353465, 0.13785328563684354}, {0.15529979160329746, 0.12258674664880864}, {0.1498295800552226, 0.11760132141076962}, {0.260268828320712, 0.19997859246075889}, {0.22191979170382936, 0.16420670859467545}, {0.19471897178365988, 0.14005501881547042}, {0.1766687793566537, 0.12468741631510485}, {0.2523161928037281, 0.18503781606123848}, {0.23949808054634686, 0.17327118246196882}, {0.20995592077712422, 0.14707976519086374}, {0.19028466955249, 0.13052456510645602}, {0.29681537775340316, 0.2160057160559372}, {0.2529963643393698, 0.17608177777284606}, {0.22166849908988348, 0.14932704745589112}, {0.20094270803911612, 0.1325395324090112}, {0.36161205658574147, 0.2224567879039796}, {0.29219914500655264, 0.17017421902433005}, {0.25563328558753895, 0.14424182911881145}, {0.23165170964305215, 0.12799535591114486}, {0.3882664335214443, 0.22244079241706127}, {0.31347818281993867, 0.16961062893273882}, {0.27422920047794586, 0.14362113419154499}, {0.2501757320595628, 0.12838892178402816}, {0.42141142931581044, 0.236030646485891}, {0.3396148998824673, 0.1776480151288422}, {0.2964474379706139, 0.14953093354314695}, {0.26820466013518196, 0.1318784097501282}, {0.4537611002811026, 0.23261939888947003}, {0.3649984130647175, 0.17423042697334923}, {0.31868615079997653, 0.1461725847408848},
                    {0.29021681282721534, 0.12987260424207392}, {-0.18317594623137098, 0.014338157340459642}, {-0.19025179209789622, 0.014434524095352683}, {-0.21061955338875032, 0.014592200287061406}, {-0.24183800711114145, 0.014169111455869937}, {-0.22217892052223265, 0.0020292078903739754}, {-0.24592397600055943, 3.395425860819345E-4}, {-0.2819985028782299, -0.0023873343585000123}, {-0.2976703873098949, -0.0036422253263017484}, {-0.1289988618188396, 0.17346268932787096}, {-0.13476821275193138, 0.17975056088882843}, {-0.15187995453977435, 0.19800937330273616}, {-0.17858477053429933, 0.2255321995725511}, {-0.23131516413308195, -0.002591066209828421}, {-0.25599877484332706, -0.004532416871649631}, {-0.2935695992832801, -0.008178306883299798}, {-0.3473936224818561, -0.014248069601681512}, {-0.17719302034613696, 0.20266069771104198}, {-0.19888059289283225, 0.2228895580237924}, {-0.23333159786415258, 0.25339306510182347}, {-0.24851618479296356, 0.2661790143491057}, {-0.190599842434216, 0.20787387369763527}, {-0.2141032716320406, 0.22858525453126333}, {-0.2505440347183878, 0.259436100385258}, {-0.3050785029979446, 0.30199491817688773}, {-0.2836845003078568, -0.03884534090236347}, {-0.3137878341505006, -0.0450364803506087}, {-0.3591947117725285, -0.05502601307932986}, {-0.44676361025506317, -0.07674133524034299},

                    {-0.2998304413508692, -0.052767504283186994}, {-0.32902618028746355, -0.059801798966951036}, {-0.3768006834250688, -0.07216444007175582}, {-0.46818156258492927, -0.09816169694948934}, {-0.2857036328147937, 0.22378470266116013}, {-0.3195673818599343, 0.24521665839030948}, {-0.3719736355272989, 0.27639057706028125}, {-0.4767794384015993, 0.33369931054624447}, {-0.3167647503762867, 0.22327151676880597}, {-0.3512409825749252, 0.2426170282332154}, {-0.40868738055301035, 0.27284234315443556}, {-0.5221131282024749, 0.3276725276333141}, {0.11811743352382081, -0.2292873732980052}, {0.10516343533688283, -0.19818903027630203}, {0.09649004270827764, -0.1780524433356996}, {0.0934156878228622, -0.1711818427370287}, {0.17418694075485108, -0.2143811853241142}, {0.1540672680943679, -0.18350111966270832}, {0.14033678373987085, -0.16382030320162894}, {0.1354741136553175, -0.15721160654181368}, {0.16521339530521095, -0.2751602259416015}, {0.1577948293526043, -0.25946259470183897}, {0.1399885141049478, -0.22395015602861693}, {0.128135819198707, -0.2010164729154653}, {0.19583204807039878, -0.32146241525739905}, {0.16973031131219385, -0.2679555021781135}, {0.15063403711416326, -0.23104239000800517}, {0.13777373195145048, -0.20730189373911093}, {0.23723490088793242, -0.25534440993880025},
                    {0.22597878934743965, -0.23980330269237715}, {0.19905835290901264, -0.20494574671220467}, {0.18116162638631153, -0.18269752303585224}, {0.281463093377359, -0.29951915062321977}, {0.24161121236247274, -0.24578429345376396}, {0.21294049232729476, -0.20998862090523066}, {0.19338170948385322, -0.18700234717608513}, {0.31446448872257804, -0.3902089987663122}, {0.2583527926511864, -0.3049891459475469}, {0.22801892911948185, -0.2621546070500713}, {0.207823164830087, -0.23472563303131921}, {0.4342110300514612, -0.3473669016341725}, {0.35297689073383154, -0.26518155897433715}, {0.30909153524592514, -0.2247775353552932}, {0.2804567685993482, -0.19942851220850977}, {0.34975411548279567, -0.3990309046832383}, {0.28658227492333443, -0.3114016347945032}, {0.25256747847801764, -0.2674675531028924}, {0.2316841376470673, -0.24117882566334203}, {0.47747126798989115, -0.3478156185476484}, {0.38676365233819787, -0.2641221157818805}, {0.3386422435716978, -0.2235620571134811}, {0.3089796906370641, -0.19965732680970105}, {-0.17137823889703832, 0.01828556039348803}, {-0.1779282095616584, 0.019339035330522336}, {-0.19673747402207362, 0.022742869005947727}, {-0.2254431393623748, 0.028222216911648064}, {-0.2035591832782847, 0.034878207907638674}, {-0.22495679581423902, 0.04021436926392831},
                    {-0.2578185984784569, 0.04861066926406965}, {-0.271866167931746, 0.05248714947338781}, {-0.2104455181350052, 0.040000912311477343}, {-0.23249179051004937, 0.045733496666110034}, {-0.26619485631532536, 0.05500718615580644}, {-0.31423498408000405, 0.06930988821226536}, {-0.15125763509751974, -0.1657547413775337}, {-0.1576013029550005, -0.17170709486619473}, {-0.17637713443681874, -0.1885514920906666}, {-0.20566970037071242, -0.21398139289099638}, {-0.25028014851748626, 0.07673017296367535}, {-0.2764120739000515, 0.08668691404829953}, {-0.31605128381228553, 0.10250880539302942}, {-0.39197668572061595, 0.13448190613347583}, {-0.262000117709394, 0.09048745329726149}, {-0.2870345025661887, 0.10077320828981184}, {-0.3281988068716579, 0.1188825015677546}, {-0.4070032468358544, 0.15477999022362027}, {-0.20972496134675953, -0.19533918178638984}, {-0.2342777273558467, -0.21439992296059318}, {-0.2722865747441965, -0.24241560590127176}, {-0.28908481004475073, -0.25460067538528663}, {-0.22633493969849108, -0.20128807101880608}, {-0.2526536246596073, -0.2206751129557422}, {-0.29348761994528694, -0.24969700558483654}, {-0.3532174240843644, -0.28937639539695387}, {-0.34721772025704767, -0.20501734171614805}, {-0.38636388862938686, -0.2233759606503738}, {-0.4469560053311692, -0.25072648394953556},
                    {-0.5653110189420114, -0.2999388291175358}, {-0.3859485137677585, -0.19607494724838848}, {-0.42613378882057384, -0.2124048226151305}, {-0.4916375151653043, -0.2376403943695089}, {-0.62039308018573, -0.2821802477345905}, {-0.2992866662753215, -0.16974963331884196}, {-0.2245734129736019, -0.12005192995046499}, {-0.18933084174517978, -0.09988505881561471}, {-0.16898345955158067, -0.0888352930280224}, {-0.286161331892373, -0.1647459958934432}, {-0.2148823310524408, -0.11662007925216189},

                    {-0.18099496554035888, -0.09724266195969411}, {-0.16021680237414127, -0.08563179728641424}, {-0.22773086794587272, -0.13518794708409687}, {-0.1831886052599474, -0.10446784031510647}, {-0.15432758806554595, -0.08697581979986199}, {-0.13656768757287935, -0.07666536298475674}, {-0.19039985451227714, -0.11054492020770446}, {-0.17770446984410615, -0.10224587034382093}, {-0.14956059411063843, -0.08503764100951199}, {-0.132476097334985, -0.07493277590035133}, {-0.15896888414449206, -0.09436015136987175}, {-0.13376126883316786, -0.07845809277158622}, {-0.11845939263236388, -0.06913172273008711}, {-0.11342417885709843, -0.06604768660782281}, {-0.27043159063465794, -0.2578309892248383}, {-0.2014045938009561, -0.18504168778930263}, {-0.16874471073350472, -0.1553221440945709}, {-0.15055963832985508, -0.13870543624872145}, {-0.2569698181423661, -0.25088497184232406}, {-0.19135132063296956, -0.1802238916409733}, {-0.16059338925771005, -0.15125839420219678}, {-0.14185604726612544, -0.13369533928112953}, {-0.19779956918225583, -0.20557841570893132}, {-0.15798484068936203, -0.1606222302467435}, {-0.13236666087438015, -0.13451425680312615}, {-0.11690517691093409, -0.11896479134879108}, {-0.1639500533708964, -0.16965497797444584}, {-0.15251350513849715, -0.15736140482449468},
                    {-0.12793914799065817, -0.13183337559771036}, {-0.11302015717877854, -0.11659488311616073}, {-0.13359432538849836, -0.1443235405712544}, {-0.11194850220783464, -0.12090173533208536}, {-0.09876881955819013, -0.10680769076612799}, {-0.09457088867452397, -0.10215205628556037}, {-0.2803922319768617, 0.18086294470551412}, {-0.2143783666721215, 0.12918599013259413}, {-0.18184571457535892, 0.10714056459486004}, {-0.1627119217015649, 0.09522095914214274}, {-0.2687753334870706, 0.17514975906712582}, {-0.20537133972651134, 0.12508978058260034}, {-0.17412679288306654, 0.10375439693266147}, {-0.15450903001052918, 0.0911940951665044}, {-0.21457443653696728, 0.142849873443486}, {-0.17483854575705848, 0.11100671208604718}, {-0.1484476488639831, 0.09216164097552329}, {-0.13167194359696327, 0.08097014745500068}, {-0.1815779240423394, 0.11728222928688736}, {-0.1699728533558975, 0.10841384866855575}, {-0.14418919289472754, 0.09000963907053904}, {-0.12795603655891585, 0.07916453542420337}, {-0.15214559389294724, 0.09925550906027114}, {-0.12903673135082777, 0.08234621695743877}, {-0.11451814249481508, 0.07243539322350508}, {-0.10972846714581325, 0.06915230053741245}, {-0.02179545254019302, 0.17110027752882087}, {-0.02303190035839076, 0.17875940000419227}, {-0.02686310940319992, 0.20230912702659948},
                    {-0.034366211363354715, 0.24100182280128557}, {-0.03596121549004623, 0.20111548201938637}, {-0.04156395655306219, 0.2273846473153659}, {-0.0525148854248807, 0.27053719009254973}, {-0.05816277523923108, 0.29038461735791987}, {-0.03994555221049472, 0.2065536276223286}, {-0.046176346253965474, 0.23364490693813353}, {-0.05783076308244817, 0.2779690765002547}, {-0.0819810405770901, 0.3465368719280022}, {-0.06750307800681732, 0.24174984659158394}, {-0.07758482673653039, 0.27281850854051726}, {-0.0957443965011367, 0.3239957516204717}, {-0.14642896548997797, 0.4306062603342622}, {-0.07739680010749994, 0.2529277131926815}, {-0.08823534582858555, 0.2834634069227455}, {-0.10883469323358447, 0.3362034976941133},

                    {-0.163945303282377, 0.44557521439402814}, {0.08365516211928938, -0.1688512708001217}, {0.08789657088062247, -0.17639693216587177}, {0.10043930882867559, -0.1990323122166915}, {0.12201318087722157, -0.23559358815263565}, {0.08874110051161584, -0.2069646259963847}, {0.10131547078272451, -0.23378207415704566}, {0.12352679300941016, -0.2769838047203539}, {0.13447285196167336, -0.29660361727385887}, {0.08863456186113629, -0.2155650122624102}, {0.1012808979670427, -0.24366813743049834}, {0.123501418558007, -0.28858075378080184}, {0.16263935307204136, -0.3572161595361445}, {0.2637017307897801, -0.05911757831259479}, {0.22811130698682527, -0.046774344222104434}, {0.20493566786307207, -0.03962994808648557}, {0.19685349850359132, -0.03723394727472847}, {0.004452515931746162, -0.21060015592068912}, {0.0037843917926855125, -0.21975435660971476}, {0.0012891910226453822, -0.2472970452408996}, {-0.004247819513067519, -0.2906232031490452}, {0.07954330910312316, -0.26981441599478684}, {0.09152453855736867, -0.3050715828345041}, {0.11238979561039364, -0.362142335973191}, {0.16589459499684622, -0.48255917150163397}, {0.07615124186897787, -0.28772588346315864}, {0.08629875876958765, -0.3226291678568156}, {0.10614781966498088, -0.3830732908813195}, {0.15854628464372666, -0.5111317949879146},
                    {0.3309728165798478, -0.053575881539296155}, {0.31242676945846637, -0.04793410531795761}, {0.26968568548828203, -0.036262599034181414}, {0.24198631748072208, -0.029777143991783182}, {-0.013711882831378268, -0.2555537984469301}, {-0.018878784682208095, -0.28702084204530365}, {-0.02901485201912468, -0.3368435368216196}, {-0.03424176338625973, -0.35864770914610933}, {-0.025721684700593255, 0.22255685383741242}, {-0.02783638235128538, 0.23159635214748325}, {-0.03447032530299288, 0.2586579140395001}, {-0.04628668133402947, 0.30005719613082577}, {0.3906607901684852, -0.06376501279860403}, {0.3261194700443292, -0.044176383058922575}, {0.281502582516369, -0.03294018249153609}, {0.25234761985978593, -0.026428676945591263}, {-0.019656470289065173, -0.2657007639608612}, {-0.025829092416593313, -0.2984850048004005}, {-0.037285040760557386, -0.3500312187631014}, {-0.05974657742995593, -0.42546102248369033}, {-0.05350364191668105, 0.2729020017082033}, {-0.06376542023551597, 0.30406665592246607}, {-0.08167875170066896, 0.35234314088531793}, {-0.08992261545049018, 0.37319863558167543}, {-0.06238490941820885, 0.28387842185389817}, {-0.07388776431684278, 0.3159971532084954}, {-0.09365460182752461, 0.3657801990461153}, {-0.12586876056571228, 0.43716151687143}, {0.5230751563368514, -0.021664354848166312},
                    {0.41019411510934206, -0.0023670444876211505}, {0.3529760065358698, 0.004625690830156515}, {0.3158611541105051, 0.007998824454486186}, {-0.06637347951221884, -0.3221027819980533}, {-0.07880574321080047, -0.36111260910947696}, {-0.10120230785941864, -0.42182543515812165}, {-0.1559240643905957, -0.5404917555814215}, {0.5533156879843588, -9.970570199681651E-4}, {0.43283087924538055, 0.014896758500088314}, {0.37218102017422694, 0.01980781431078027}, {0.3352521153405269, 0.02144075818951796}, {-0.08411913927473202, -0.34002428394307993}, {-0.09798295966742376, -0.3774893265153003}, {-0.12412089062678433, -0.44027437444570405}, {-0.18614218575334096, -0.5627538826141034}, {-0.13589914898601974, 0.34321500526160587}, {-0.15673134199058358, 0.381000115437379}, {-0.19106719769096658, 0.4392757208962812}, {-0.26580735522772514, 0.5493317996070499}, {-0.16248035759862045, 0.3599393380720395}, {-0.18490048462306152, 0.3964476428104585}, {-0.22414765951909432, 0.4562589174513185}, {-0.3082302376456223, 0.5693217397730302}
                };

                const double epsilon = 1e-12;
                test_region(x_y_points, li22_values, epsilon);
            }

        }

} multiplepolylogarithm_test;
