/* vim: set sw=4 sts=4 et foldmethod=syntax : */

/*
 * Copyright (c) 2022 Danny van Dyk
 * Copyright (c) 2022 Philip Lüghausen
 *
 * This file is part of the EOS project. EOS is free software;
 * you can redistribute it and/or modify it under the terms of the GNU General
 * Public License version 2, as published by the Free Software Foundation.
 *
 * EOS is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 59 Temple
 * Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <eos/form-factors/analytic-b-to-gamma-qcdf.hh>
#include <eos/maths/integrate-impl.hh>
#include <eos/maths/interpolation.hh>
#include <eos/maths/matrix.hh>
#include <eos/maths/polylog.hh>
#include <eos/utils/exception.hh>

#include <functional>
#include <numeric>

#include <gsl/gsl_sf_expint.h>

namespace eos
{
    static const constexpr double pi = M_PI;
    static const constexpr double pi2 = pi * pi;
    static const constexpr double eulergamma = 0.577215664901533;
    static const constexpr double eulergamma2 = eulergamma * eulergamma;
    static const constexpr double zeta_3 = 1.20205690315959;

    std::string
    AnalyticFormFactorBToGammaQCDF::par_qname(const std::string & _name) const
    {
        return QualifiedName(qnp::Prefix("B_u"), _name, qnp::Suffix("FLvD2022")).str();
    }

    AnalyticFormFactorBToGammaQCDF::AnalyticFormFactorBToGammaQCDF(const Parameters & p, const Options & o):
        blcdas(BMesonLCDAs::make("FLvD2022", p, o)),
        model(Model::make("SM", p, o)),
        mu(p["B->gamma::mu@FLvD2022QCDF"], *this),
        omega_0(p[par_qname("omega_0")], *this),
        f_B(p["decay-constant::B_u"], *this),
        m_B(p["mass::B_u"], *this),
        m_rho(p["mass::rho^+"], *this),
        lambda_bar(p["B::LambdaBar"], *this),
        lambda_E2(p["B::lambda_E^2"], *this),
        lambda_H2(p["B::lambda_H^2"], *this),
        M2(p["B->gamma::M^2@FLvD2022QCDF"], *this),
        s_0(p["B->gamma::s_0@FLvD2022QCDF"], *this),
        mu_h1(p["B->gamma::mu_h1@FLvD2022QCDF"], *this),
        mu_h2(p["B->gamma::mu_h2@FLvD2022QCDF"], *this)
    {
        // Verify once that BMesonLCDAs is compatible with this implementation
        Weights weights;
        auto [coeff_begin, coeff_end] = blcdas->coefficient_range(mu);

        if (weights.size() - std::distance(coeff_begin, coeff_end) > 0)
        {
            throw InternalError("The number of weights implemented is smaller than the number of coefficients of phi_+");
        }
    }

    FormFactors<PToGamma> *
    AnalyticFormFactorBToGammaQCDF::make(const Parameters & p, const Options & o)
    {
        return new AnalyticFormFactorBToGammaQCDF(p, o);
    }

    std::tuple<double, double, double>
    AnalyticFormFactorBToGammaQCDF::C_K_inv_U(const double & Egamma) const
    {
        // cf. the factor in [BR:2011A], Eq. (2.10)

        const double alpha_s_mu = model->alpha_s(mu);
        const double m_bquark = model->m_b_pole(1);

        const double x = 2.0 * Egamma / m_bquark;
        const double log_x = std::log(x);
        const double log_x_prime = std::log(2.0 * Egamma / mu);
        const double li2_1_m_x = dilog(1.0 - x).real(); // result is always real as x > 0

        const double C = 1.0 + alpha_s_mu * C_F / (4.0 * pi) * (
                    - 2.0 * log_x_prime * log_x_prime
                    + 5.0 * log_x_prime
                    - (3.0 - 2.0 * x) / (1.0 - x) * log_x
                    - 2.0 * li2_1_m_x
                    - 6.0 - pi2 / 12.0
                );


        const double alpha_s_mu_h1 = model->alpha_s(mu_h1);
        const double alpha_s_mu_h2 = model->alpha_s(mu_h2);

        const double Gamma_0 = 4.0 * C_F;
        const double Gamma_1 = C_F * (268.0 / 3.0 - 4.0 * pi2 - 40.0 / 9.0 * n_l);
        const double Gamma_2 = C_F * (
                1470.0 - 536.0 * pi2 / 3.0 + 44.0 * pi2 * pi2 / 5.0
                + 264.0 * zeta_3 + n_l * (-1276.0 / 9.0 + 80.0 * pi2 /  9.0 - 208.0 / 3.0 * zeta_3)
                - 16.0 / 27.0 * n_l * n_l
                );

        const double beta_0 = 11.0 - 2.0 * n_l / 3.0;
        const double beta_1 = 102.0 - 38.0 * n_l / 3.0;
        const double beta_2 = 2857.0 / 2.0 - 5033.0 / 18.0 * n_l + 325.0 / 54.0 * n_l * n_l;

        const double gamma_0 = -5.0 * C_F;
        const double gamma_1 = C_F * (
                -1585.0 / 18.0 - 5.0 * pi2 / 6.0 + 34.0 * zeta_3 + n_l * (125.0 / 27.0 + pi2 / 3.0)
            );

        const double r1 = alpha_s_mu / alpha_s_mu_h1;
        const double log_r1 = std::log(r1);

        // cf. [BR:2011A], Eq. (A.3)
        const double U1 =
            // first line
            std::exp(
                - Gamma_0 / (4.0 * beta_0 * beta_0) * (
                    4.0 * pi / alpha_s_mu_h1 * (log_r1 - 1.0 + 1.0 / r1)
                    - beta_1 / (2.0 * beta_0) * log_r1 * log_r1
                    + (Gamma_1 / Gamma_0 - beta_1 / beta_0) * (r1 - 1.0 - log_r1)
                )
            )
            // second line
            * std::pow(2.0 * Egamma / mu_h1, - Gamma_0 / (2.0 * beta_0) * log_r1)
            * std::pow(r1, - gamma_0 / (2.0 * beta_0))
            * (
                1.0 - alpha_s_mu_h1 / (4.0 * pi) * Gamma_0 / (4.0 * beta_0 * beta_0) * (
                    Gamma_2 / (2.0 * Gamma_0) * (1.0 - r1) * (1.0 - r1)
                    + beta_2 / (2.0 * beta_0) * (1.0 - r1 * r1 + 2.0 * log_r1)
            // third line
                    - Gamma_1 * beta_1 / (2.0 * Gamma_0 * beta_0)
                        * (3.0 - 4.0 * r1 + r1 * r1 + 2.0 * r1 * log_r1)
                    + beta_1 * beta_1 / (2.0 * beta_0 * beta_0) * (1.0 - r1)
                        * (1.0 - r1 - 2.0 * log_r1)
                )
            // fourth line
                + alpha_s_mu_h1 / (4.0 * pi) * (
                    std::log(2.0 * Egamma / mu_h1) * (Gamma_1 / (2.0 * beta_0) - Gamma_0 * beta_1 / (2.0 * beta_0 * beta_0))
                    + gamma_1 / (2.0 * beta_0) - gamma_0 * beta_1 / (2.0 * beta_0 * beta_0)
                ) * (1.0 - r1)
            );


        const double r2 = alpha_s_mu / alpha_s_mu_h2;
        const double U2 = std::pow(r2, - gamma_0 / (2.0 * beta_0)) * (
                1.0 + alpha_s_mu_h2 / (4.0 * pi) * (
                    gamma_1 / (2.0 * beta_0) - gamma_0 * beta_1 / (2.0 * beta_0 * beta_0)
                ) * (1.0 - r2)
            );

        const double K = 1 + alpha_s_mu * C_F / (4.0 * pi) * (3.0 * std::log(m_bquark / mu) - 2.0);

        return { C, 1.0 / K, U1 / U2 };
    }

    double
    AnalyticFormFactorBToGammaQCDF::F_leading_power(const double & Egamma) const
    {
        // cf. [BR:2011A], Eq. (2.10)
        const auto [C, K_inv, U] = C_K_inv_U(Egamma);

        // cf. [BBJW:2018A], Eq. (2.7), first term,
        // cf. [BBJW:2018A], Eq. (4.12)
        const double C_NLO = model->alpha_s(mu) * C_F / (4.0 * pi);
        return e_u * f_B * m_B / (2.0 * Egamma) * C * K_inv * U *
            (L0() + C_NLO * L0_effective(Egamma));
    }

    double
    AnalyticFormFactorBToGammaQCDF::xi(const double & Egamma) const
    {
        const double prefactor = e_u * f_B * m_B / (2.0 * Egamma);
        const double m_bquark = model->m_b_pole(1);

        // Higher twist contribution
        // cf. [BBJW:2018A], Eq. (5.9)
        // Use EOM constraints to eliminate higher-twist LCDAs (twist 3+4)
        const double term_ht =
            - prefactor / Egamma * (
                + 2.0 * (lambda_E2 + 2.0 * lambda_H2)
                    / (6.0 * lambda_bar * lambda_bar + 2.0 * lambda_E2 + lambda_H2)
                + 1.0 / 2.0
            )
            + prefactor / (2.0 * m_bquark) * (
                + lambda_bar * L0()
                - 2.0
                + 4.0 * (lambda_E2 - lambda_H2)
                    / (6.0 * lambda_bar * lambda_bar + 2.0 * lambda_E2 + lambda_H2)
            );

        // Soft contribution
        // cf. [BBJW:2018A], Eq. (4.10)
        const double C_NLO = model->alpha_s(mu) * C_F / (4.0 * pi);
        const double omega_cut = s_0 / (2.0 * Egamma);
        const double sigma = 2.0 * Egamma / M2;
        const auto [C, K_inv, U] = C_K_inv_U(Egamma);
        const double term_soft_nlo =
            prefactor * C * K_inv * U * (
                    (2.0 * Egamma / (m_rho * m_rho) * std::exp(m_rho * m_rho / M2))
                        * (lapltr_incomplete(omega_cut, sigma) + C_NLO * lapltr_incomplete_effective(Egamma, omega_cut, sigma))
                    - (L0_incomplete(omega_cut) + C_NLO * L0_incomplete_effective(Egamma, omega_cut))
            );

        return term_ht + term_soft_nlo;
    }

    double
    AnalyticFormFactorBToGammaQCDF::delta_xi(const double & Egamma) const
    {
        // See second line of Eq. (5.1), p. 14 in Ref. [BBJW:2018A]

        // cf. [BBJW:2018A], Eq. (3.11)
        const double m_bquark = model->m_b_pole(1);
        const double term_ht = f_B * m_B / (2.0 * Egamma) * (e_b / m_bquark + e_u / (2.0 * Egamma));

        // cf. [BBJW:2018A], Eq. (4.10)
        const double term_soft_nlo = 0.0;

        // cf. [BBJW:2018A], Eq. (4.14)
        const double omega_cut = s_0 / (2.0 * Egamma);
        const double sigma = 2.0 * Egamma / M2;

        const double term_soft_tw_3_4 = e_u * m_B * f_B / (4.0 * Egamma * Egamma) * (
                2.0 * Egamma / (m_rho * m_rho) * std::exp(m_rho * m_rho / M2)
                    * lapltr_incomplete_dsigma(omega_cut, sigma) / (-sigma)
                - norm_incomplete(omega_cut)
            );

        return term_ht + term_soft_nlo + term_soft_tw_3_4;
    }

    double
    AnalyticFormFactorBToGammaQCDF::L0() const
    {
        const Weights c = { 1.0, 0.0, 1.0 / 3.0, 0.0, 1.0 / 5.0, 0.0, 1.0 / 7.0, 0.0, 1.0 / 9.0 };

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);
        return 1.0 / omega_0 * std::inner_product(a_begin, a_end, c.begin(), 0.0);
    }

    double
    AnalyticFormFactorBToGammaQCDF::L0_incomplete(const double & omega_cut) const
    {
        const double xom = omega_cut / omega_0;
        const double xom2 = xom  * xom;
        const double xom3 = xom2 * xom;
        const double xom4 = xom3 * xom;
        const double xom5 = xom4 * xom;
        const double xom6 = xom5 * xom;
        const double xom7 = xom6 * xom;
        const double xom8 = xom7 * xom;

        const double exp_xom = std::exp(-xom);

        const Weights c = {
            1 - exp_xom,
            xom * exp_xom,
            0.3333333333333333 + (exp_xom * (-1 + 2 * xom - 2 * xom2)) / 3.,
            (xom * exp_xom * (3 - 3 * xom + xom2)) / 3.,
            0.2 + (exp_xom * (-3 + 12 * xom - 24 * xom2 + 12 * xom3 - 2 * xom4)) / 15.,
            (xom * exp_xom * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4)) / 45.,
            0.14285714285714285 + (exp_xom * (-45 + 270 * xom - 810 * xom2 + 780 * xom3 - 330 * xom4 + 60 * xom5 - 4 * xom6)) / 315.,
            (xom * exp_xom * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6)) / 315.,
            0.1111111111111111 + (exp_xom * (-315 + 2520 * xom - 10080 * xom2 + 14280 * xom3 - 9660 * xom4 + 3360 * xom5 - 616 * xom6 + 56 * xom7 - 2 * xom8)) / 2835.
        };

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);
        return 1.0 / omega_0 * std::inner_product(a_begin, a_end, c.begin(), 0.0);
    }

    double
    AnalyticFormFactorBToGammaQCDF::norm_incomplete(const double & omega_cut) const
    {
        const double xom = omega_cut / omega_0;
        const double xom2 = xom  * xom;
        const double xom3 = xom2 * xom;
        const double xom4 = xom3 * xom;
        const double xom5 = xom4 * xom;
        const double xom6 = xom5 * xom;
        const double xom7 = xom6 * xom;
        const double xom8 = xom7 * xom;
        const double xom9 = xom8 * xom;

        const double exp_xom = std::exp(-xom);

        const Weights c = {
            1 + (-1 - xom) * exp_xom,
            -1 + exp_xom * (1 + xom + xom2),
            1 + (exp_xom * (-3 - 3 * xom - 2 * xom3)) / 3.,
            -1 + (exp_xom * (3 + 3 * xom + 3 * xom2 - 2 * xom3 + xom4)) / 3.,
            1 + (exp_xom * (-15 - 15 * xom - 20 * xom3 + 10 * xom4 - 2 * xom5)) / 15.,
            -1 + (exp_xom * (45 + 45 * xom + 45 * xom2 - 60 * xom3 + 60 * xom4 - 18 * xom5 + 2 * xom6)) / 45.,
            1 + (exp_xom * (-315 - 315 * xom - 630 * xom3 + 630 * xom4 - 294 * xom5 + 56 * xom6 - 4 * xom7)) / 315.,
            -1 + (exp_xom * (315 + 315 * xom + 315 * xom2 - 630 * xom3 + 945 * xom4 - 546 * xom5 + 154 * xom6 - 20 * xom7 + xom8)) / 315.,
            1 + (exp_xom * (-2835 - 2835 * xom - 7560 * xom3 + 11340 * xom4 - 8316 * xom5 + 3024 * xom6 - 576 * xom7 + 54 * xom8 - 2 * xom9)) / 2835.
        };

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);
        return std::inner_product(a_begin, a_end, c.begin(), 0.0);
    }

    double
    AnalyticFormFactorBToGammaQCDF::lapltr_incomplete(const double & omega_cut, const double & sigma) const
    {
        const double xom = omega_cut / omega_0;
        const double xom2 = xom  * xom;
        const double xom3 = xom2 * xom;
        const double xom4 = xom3 * xom;
        const double xom5 = xom4 * xom;
        const double xom6 = xom5 * xom;
        const double xom7 = xom6 * xom;
        const double xom8 = xom7 * xom;
        const double xom9 = xom8 * xom;

        const double xsg = sigma * omega_0;
        const double xsg2 = xsg * xsg;
        const double xsg3 = xsg * xsg2;
        const double xsg4 = xsg * xsg3;
        const double xsg5 = xsg * xsg4;
        const double xsg6 = xsg * xsg5;
        const double xsg7 = xsg * xsg6;
        const double xsg8 = xsg * xsg7;
        const double xsg9 = xsg * xsg8;

        const double xsgplus = 1.0 + xsg;
        const double xsgplus_m1  = 1.0 / xsgplus;
        const double xsgplus_m2  = xsgplus_m1 * xsgplus_m1;
        const double xsgplus_m3  = xsgplus_m2 * xsgplus_m1;
        const double xsgplus_m4  = xsgplus_m3 * xsgplus_m1;
        const double xsgplus_m5  = xsgplus_m4 * xsgplus_m1;
        const double xsgplus_m6  = xsgplus_m5 * xsgplus_m1;
        const double xsgplus_m7  = xsgplus_m6 * xsgplus_m1;
        const double xsgplus_m8  = xsgplus_m7 * xsgplus_m1;
        const double xsgplus_m9  = xsgplus_m8 * xsgplus_m1;
        const double xsgplus_m10 = xsgplus_m9 * xsgplus_m1;

        const double exp_xom_xsgplus = std::exp(- xom * xsgplus);

        const Weights c = {
            (1 + (-1 - xom - xom * xsg) * exp_xom_xsgplus) * xsgplus_m2,
            (-1 + xsg + exp_xom_xsgplus * (1 + xom - xsg + xom2 + 2 * xsg * xom2 - xom * xsg2 + xom2 * xsg2)) * xsgplus_m3,
            ((3 - 6 * xsg + 3 * xsg2) / 3. + (exp_xom_xsgplus * (-3 - 3 * xom + 6 * xsg + 3 * xom * xsg + 6 * xsg * xom2 - 2 * xom3 - 6 * xsg * xom3 - 3 * xsg2 + 3 * xom * xsg2 + 12 * xom2 * xsg2 - 6 * xom3 * xsg2 - 3 * xom * xsg3 + 6 * xom2 * xsg3 - 2 * xom3 * xsg3)) / 3.) * xsgplus_m4,
            ((-3 + 9 * xsg - 9 * xsg2 + 3 * xsg3) / 3. + (exp_xom_xsgplus * (3 + 3 * xom - 9 * xsg - 6 * xom * xsg + 3 * xom2 + 6 * xsg * xom2 - 2 * xom3 - 12 * xsg * xom3 + xom4 + 4 * xsg * xom4 + 9 * xsg2 + 12 * xom2 * xsg2 - 24 * xom3 * xsg2 + 6 * xom4 * xsg2 - 3 * xsg3 + 6 * xom * xsg3 + 18 * xom2 * xsg3 - 20 * xom3 * xsg3 + 4 * xom4 * xsg3 - 3 * xom * xsg4 + 9 * xom2 * xsg4 - 6 * xom3 * xsg4 + xom4 * xsg4)) / 3.) * xsgplus_m5,
            ((15 - 60 * xsg + 90 * xsg2 - 60 * xsg3 + 15 * xsg4) / 15. + (exp_xom_xsgplus * (-15 - 15 * xom + 60 * xsg + 45 * xom * xsg + 60 * xsg * xom2 - 20 * xom3 - 100 * xsg * xom3 + 10 * xom4 + 60 * xsg * xom4 - 2 * xom5 - 10 * xsg * xom5 - 90 * xsg2 - 30 * xom * xsg2 + 120 * xom2 * xsg2 - 240 * xom3 * xsg2 + 140 * xom4 * xsg2 - 20 * xom5 * xsg2 + 60 * xsg3 - 30 * xom * xsg3 + 120 * xom2 * xsg3 - 320 * xom3 * xsg3 + 160 * xom4 * xsg3 - 20 * xom5 * xsg3 - 15 * xsg4 + 45 * xom * xsg4 + 120 * xom2 * xsg4 - 220 * xom3 * xsg4 + 90 * xom4 * xsg4 - 10 * xom5 * xsg4 - 15 * xom * xsg5 + 60 * xom2 * xsg5 - 60 * xom3 * xsg5 + 20 * xom4 * xsg5 - 2 * xom5 * xsg5)) / 15.) * xsgplus_m6,
            ((-45 + 225 * xsg - 450 * xsg2 + 450 * xsg3 - 225 * xsg4 + 45 * xsg5) / 45. + (exp_xom_xsgplus * (45 + 45 * xom - 225 * xsg - 180 * xom * xsg + 45 * xom2 + 90 * xsg * xom2 - 60 * xom3 - 480 * xsg * xom3 + 60 * xom4 + 390 * xsg * xom4 - 18 * xom5 - 120 * xsg * xom5 + 2 * xom6 + 12 * xsg * xom6 + 450 * xsg2 + 225 * xom * xsg2 + 495 * xom2 * xsg2 - 1380 * xom3 * xsg2 + 1110 * xom4 * xsg2 - 330 * xom5 * xsg2 + 30 * xom6 * xsg2 - 450 * xsg3 + 900 * xom2 * xsg3 - 2160 * xom3 * xsg3 + 1740 * xom4 * xsg3 - 480 * xom5 * xsg3 + 40 * xom6 * xsg3 + 225 * xsg4 - 225 * xom * xsg4 + 675 * xom2 * xsg4 - 2100 * xom3 * xsg4 + 1560 * xom4 * xsg4 - 390 * xom5 * xsg4 + 30 * xom6 * xsg4 - 45 * xsg5 + 180 * xom * xsg5 + 450 * xom2 * xsg5 - 1200 * xom3 * xsg5 + 750 * xom4 * xsg5 - 168 * xom5 * xsg5 + 12 * xom6 * xsg5 - 45 * xom * xsg6 + 225 * xom2 * xsg6 - 300 * xom3 * xsg6 + 150 * xom4 * xsg6 - 30 * xom5 * xsg6 + 2 * xom6 * xsg6)) / 45.) * xsgplus_m7,
            ((315 - 1890 * xsg + 4725 * xsg2 - 6300 * xsg3 + 4725 * xsg4 - 1890 * xsg5 + 315 * xsg6) / 315. + (exp_xom_xsgplus * (-315 - 315 * xom + 1890 * xsg + 1575 * xom * xsg + 1890 * xsg * xom2 - 630 * xom3 - 4410 * xsg * xom3 + 630 * xom4 + 5040 * xsg * xom4 - 294 * xom5 - 2226 * xsg * xom5 + 56 * xom6 + 420 * xsg * xom6 - 4 * xom7 - 28 * xsg * xom7 - 4725 * xsg2 - 2835 * xom * xsg2 + 3780 * xom2 * xsg2 - 15750 * xom3 * xsg2 + 17010 * xom4 * xsg2 - 7350 * xom5 * xsg2 + 1344 * xom6 * xsg2 - 84 * xom7 * xsg2 + 6300 * xsg3 + 1575 * xom * xsg3 + 8190 * xom2 * xsg3 - 31290 * xom3 * xsg3 + 32340 * xom4 * xsg3 - 13650 * xom5 * xsg3 + 2380 * xom6 * xsg3 - 140 * xom7 * xsg3 - 4725 * xsg4 + 1575 * xom * xsg4 + 12600 * xom2 * xsg4 - 37170 * xom3 * xsg4 + 38010 * xom4 * xsg4 - 15330 * xom5 * xsg4 + 2520 * xom6 * xsg4 - 140 * xom7 * xsg4 + 1890 * xsg5 - 2835 * xom * xsg5 + 8190 * xom2 * xsg5 - 28350 * xom3 * xsg5 + 27720 * xom4 * xsg5 - 10374 * xom5 * xsg5 + 1596 * xom6 * xsg5 - 84 * xom7 * xsg5 - 315 * xsg6 + 1575 * xom * xsg6 + 3780 * xom2 * xsg6 - 13650 * xom3 * xsg6 + 11550 * xom4 * xsg6 - 3906 * xom5 * xsg6 + 560 * xom6 * xsg6 - 28 * xom7 * xsg6 - 315 * xom * xsg7 + 1890 * xom2 * xsg7 - 3150 * xom3 * xsg7 + 2100 * xom4 * xsg7 - 630 * xom5 * xsg7 + 84 * xom6 * xsg7 - 4 * xom7 * xsg7)) / 315.) * xsgplus_m8,
            ((-315 + 2205 * xsg - 6615 * xsg2 + 11025 * xsg3 - 11025 * xsg4 + 6615 * xsg5 - 2205 * xsg6 + 315 * xsg7) / 315. + (exp_xom_xsgplus * (315 + 315 * xom - 2205 * xsg - 1890 * xom * xsg + 315 * xom2 + 630 * xsg * xom2 - 630 * xom3 - 6300 * xsg * xom3 + 945 * xom4 + 8190 * xsg * xom4 - 546 * xom5 - 4788 * xsg * xom5 + 154 * xom6 + 1316 * xsg * xom6 - 20 * xom7 - 168 * xsg * xom7 + xom8 + 8 * xsg * xom8 + 6615 * xsg2 + 4410 * xom * xsg2 + 6930 * xom2 * xsg2 - 23940 * xom3 * xsg2 + 32130 * xom4 * xsg2 - 18396 * xom5 * xsg2 + 4956 * xom6 * xsg2 - 616 * xom7 * xsg2 + 28 * xom8 * xsg2 - 11025 * xsg3 - 4410 * xom * xsg3 + 13230 * xom2 * xsg3 - 55020 * xom3 * xsg3 + 72870 * xom4 * xsg3 - 40740 * xom5 * xsg3 + 10724 * xom6 * xsg3 - 1288 * xom7 * xsg3 + 56 * xom8 * xsg3 + 11025 * xsg4 + 17640 * xom2 * xsg4 - 82320 * xom3 * xsg4 + 104580 * xom4 * xsg4 - 57120 * xom5 * xsg4 + 14560 * xom6 * xsg4 - 1680 * xom7 * xsg4 + 70 * xom8 * xsg4 - 6615 * xsg5 + 4410 * xom * xsg5 + 22050 * xom2 * xsg5 - 79380 * xom3 * xsg5 + 98490 * xom4 * xsg5 - 51996 * xom5 * xsg5 + 12684 * xom6 * xsg5 - 1400 * xom7 * xsg5 + 56 * xom8 * xsg5 + 2205 * xsg6 - 4410 * xom * xsg6 + 13230 * xom2 * xsg6 - 49980 * xom3 * xsg6 + 60270 * xom4 * xsg6 - 29988 * xom5 * xsg6 + 6916 * xom6 * xsg6 - 728 * xom7 * xsg6 + 28 * xom8 * xsg6 - 315 * xsg7 + 1890 * xom * xsg7 + 4410 * xom2 * xsg7 - 20580 * xom3 * xsg7 + 22050 * xom4 * xsg7 - 9996 * xom5 * xsg7 + 2156 * xom6 * xsg7 - 216 * xom7 * xsg7 + 8 * xom8 * xsg7 - 315 * xom * xsg8 + 2205 * xom2 * xsg8 - 4410 * xom3 * xsg8 + 3675 * xom4 * xsg8 - 1470 * xom5 * xsg8 + 294 * xom6 * xsg8 - 28 * xom7 * xsg8 + xom8 * xsg8)) / 315.) * xsgplus_m9,
            ((2835 - 22680 * xsg + 79380 * xsg2 - 158760 * xsg3 + 198450 * xsg4 - 158760 * xsg5 + 79380 * xsg6 - 22680 * xsg7 + 2835 * xsg8) / 2835. + (exp_xom_xsgplus * (-2835 - 2835 * xom + 22680 * xsg + 19845 * xom * xsg + 22680 * xsg * xom2 - 7560 * xom3 - 68040 * xsg * xom3 + 11340 * xom4 + 113400 * xsg * xom4 - 8316 * xom5 - 80892 * xsg * xom5 + 3024 * xom6 + 29232 * xsg * xom6 - 576 * xom7 - 5472 * xsg * xom7 + 54 * xom8 + 504 * xsg * xom8 - 2 * xom9 - 18 * xsg * xom9 - 79380 * xsg2 - 56700 * xom * xsg2 + 45360 * xom2 * xsg2 - 317520 * xom3 * xsg2 + 498960 * xom4 * xsg2 - 353808 * xom5 * xsg2 + 126000 * xom6 * xsg2 - 23184 * xom7 * xsg2 + 2088 * xom8 * xsg2 - 72 * xom9 * xsg2 + 158760 * xsg3 + 79380 * xom * xsg3 + 181440 * xom2 * xsg3 - 831600 * xom3 * xsg3 + 1300320 * xom4 * xsg3 - 910224 * xom5 * xsg3 + 318528 * xom6 * xsg3 - 57456 * xom7 * xsg3 + 5040 * xom8 * xsg3 - 168 * xom9 * xsg3 - 198450 * xsg4 - 39690 * xom * xsg4 + 317520 * xom2 * xsg4 - 1421280 * xom3 * xsg4 + 2215080 * xom4 * xsg4 - 1519560 * xom5 * xsg4 + 521136 * xom6 * xsg4 - 91728 * xom7 * xsg4 + 7812 * xom8 * xsg4 - 252 * xom9 * xsg4 + 158760 * xsg5 - 39690 * xom * xsg5 + 317520 * xom2 * xsg5 - 1693440 * xom3 * xsg5 + 2555280 * xom4 * xsg5 - 1713096 * xom5 * xsg5 + 572544 * xom6 * xsg5 - 97776 * xom7 * xsg5 + 8064 * xom8 * xsg5 - 252 * xom9 * xsg5 - 79380 * xsg6 + 79380 * xom * xsg6 + 317520 * xom2 * xsg6 - 1375920 * xom3 * xsg6 + 2010960 * xom4 * xsg6 - 1309392 * xom5 * xsg6 + 422352 * xom6 * xsg6 - 69552 * xom7 * xsg6 + 5544 * xom8 * xsg6 - 168 * xom9 * xsg6 + 22680 * xsg7 - 56700 * xom * xsg7 + 181440 * xom2 * xsg7 - 740880 * xom3 * xsg7 + 1058400 * xom4 * xsg7 - 656208 * xom5 * xsg7 + 201600 * xom6 * xsg7 - 31824 * xom7 * xsg7 + 2448 * xom8 * xsg7 - 72 * xom9 * xsg7 - 2835 * xsg8 + 19845 * xom * xsg8 + 45360 * xom2 * xsg8 - 264600 * xom3 * xsg8 + 343980 * xom4 * xsg8 - 195804 * xom5 * xsg8 + 56448 * xom6 * xsg8 - 8496 * xom7 * xsg8 + 630 * xom8 * xsg8 - 18 * xom9 * xsg8 - 2835 * xom * xsg9 + 22680 * xom2 * xsg9 - 52920 * xom3 * xsg9 + 52920 * xom4 * xsg9 - 26460 * xom5 * xsg9 + 7056 * xom6 * xsg9 - 1008 * xom7 * xsg9 + 72 * xom8 * xsg9 - 2 * xom9 * xsg9)) / 2835.) * xsgplus_m10
        };

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);
        return std::inner_product(a_begin, a_end, c.begin(), 0.0);
    }

    double
    AnalyticFormFactorBToGammaQCDF::lapltr_incomplete_dsigma(const double & omega_cut, const double & sigma) const
    {
        const double xom = omega_cut / omega_0;
        const double xom2 = xom  * xom;
        const double xom3 = xom2 * xom;
        const double xom4 = xom3 * xom;
        const double xom5 = xom4 * xom;
        const double xom6 = xom5 * xom;
        const double xom7 = xom6 * xom;
        const double xom8 = xom7 * xom;
        const double xom9 = xom8 * xom;
        const double xom10 = xom9 * xom;

        const double xsg = sigma * omega_0;
        const double xsg2 = xsg * xsg;
        const double xsg3 = xsg * xsg2;
        const double xsg4 = xsg * xsg3;
        const double xsg5 = xsg * xsg4;
        const double xsg6 = xsg * xsg5;
        const double xsg7 = xsg * xsg6;
        const double xsg8 = xsg * xsg7;
        const double xsg9 = xsg * xsg8;
        const double xsg10 = xsg * xsg9;

        const double xsgplus = 1.0 + xsg;
        const double xsgplus_m1  = 1.0 / xsgplus;
        const double xsgplus_m2  = xsgplus_m1  * xsgplus_m1;
        const double xsgplus_m3  = xsgplus_m2  * xsgplus_m1;
        const double xsgplus_m4  = xsgplus_m3  * xsgplus_m1;
        const double xsgplus_m5  = xsgplus_m4  * xsgplus_m1;
        const double xsgplus_m6  = xsgplus_m5  * xsgplus_m1;
        const double xsgplus_m7  = xsgplus_m6  * xsgplus_m1;
        const double xsgplus_m8  = xsgplus_m7  * xsgplus_m1;
        const double xsgplus_m9  = xsgplus_m8  * xsgplus_m1;
        const double xsgplus_m10 = xsgplus_m9  * xsgplus_m1;
        const double xsgplus_m11 = xsgplus_m10 * xsgplus_m1;

        const double exp_xom_xsgplus = std::exp(-xom * xsgplus);

        const Weights c = {
            (-2 * xsg + xsg * exp_xom_xsgplus * (2 + 2 * xom + 2 * xom * xsg + xom2 + 2 * xsg * xom2 + xom2 * xsg2)) * xsgplus_m3,
            ((4 - 2 * xsg) * xsg + xsg * exp_xom_xsgplus * (-4 - 4 * xom + 2 * xsg - 2 * xom * xsg - 2 * xom2 - 3 * xsg * xom2 - xom3 - 3 * xsg * xom3 + 2 * xom * xsg2 - 3 * xom3 * xsg2 + xom2 * xsg3 - xom3 * xsg3)) * xsgplus_m4,
            ((xsg * (-18 + 24 * xsg - 6 * xsg2)) / 3. + (xsg * exp_xom_xsgplus * (18 + 18 * xom - 24 * xsg - 6 * xom * xsg + 9 * xom2 + 6 * xsg * xom2 + 2 * xom3 + 2 * xom4 + 8 * xsg * xom4 + 6 * xsg2 - 18 * xom * xsg2 - 12 * xom2 * xsg2 - 12 * xom3 * xsg2 + 12 * xom4 * xsg2 + 6 * xom * xsg3 - 6 * xom2 * xsg3 - 16 * xom3 * xsg3 + 8 * xom4 * xsg3 + 3 * xom2 * xsg4 - 6 * xom3 * xsg4 + 2 * xom4 * xsg4)) / 3.) * xsgplus_m5,
            ((xsg * (24 - 54 * xsg + 36 * xsg2 - 6 * xsg3)) / 3. + (xsg * exp_xom_xsgplus * (-24 - 24 * xom + 54 * xsg + 30 * xom * xsg - 12 * xom2 + 3 * xsg * xom2 - 5 * xom3 - 9 * xsg * xom3 + xom4 + 10 * xsg * xom4 - xom5 - 5 * xsg * xom5 - 36 * xsg2 + 18 * xom * xsg2 + 24 * xom2 * xsg2 - 6 * xom3 * xsg2 + 30 * xom4 * xsg2 - 10 * xom5 * xsg2 + 6 * xsg3 - 30 * xom * xsg3 - 6 * xom2 * xsg3 - 14 * xom3 * xsg3 + 40 * xom4 * xsg3 - 10 * xom5 * xsg3 + 6 * xom * xsg4 - 12 * xom2 * xsg4 - 21 * xom3 * xsg4 + 25 * xom4 * xsg4 - 5 * xom5 * xsg4 + 3 * xom2 * xsg5 - 9 * xom3 * xsg5 + 6 * xom4 * xsg5 - xom5 * xsg5)) / 3.) * xsgplus_m6,
            ((xsg * (-150 + 480 * xsg - 540 * xsg2 + 240 * xsg3 - 30 * xsg4)) / 15. + (xsg * exp_xom_xsgplus * (150 + 150 * xom - 480 * xsg - 330 * xom * xsg + 75 * xom2 - 90 * xsg * xom2 + 20 * xom3 - 40 * xsg * xom3 + 20 * xom4 + 100 * xsg * xom4 - 8 * xom5 - 60 * xsg * xom5 + 2 * xom6 + 12 * xsg * xom6 + 540 * xsg2 + 60 * xom * xsg2 - 135 * xom2 * xsg2 - 180 * xom3 * xsg2 + 260 * xom4 * xsg2 - 180 * xom5 * xsg2 + 30 * xom6 * xsg2 - 240 * xsg3 + 300 * xom * xsg3 + 180 * xom2 * xsg3 - 160 * xom3 * xsg3 + 440 * xom4 * xsg3 - 280 * xom5 * xsg3 + 40 * xom6 * xsg3 + 30 * xsg4 - 210 * xom * xsg4 + 45 * xom2 * xsg4 - 100 * xom3 * xsg4 + 460 * xom4 * xsg4 - 240 * xom5 * xsg4 + 30 * xom6 * xsg4 + 30 * xom * xsg5 - 90 * xom2 * xsg5 - 120 * xom3 * xsg5 + 260 * xom4 * xsg5 - 108 * xom5 * xsg5 + 12 * xom6 * xsg5 + 15 * xom2 * xsg6 - 60 * xom3 * xsg6 + 60 * xom4 * xsg6 - 20 * xom5 * xsg6 + 2 * xom6 * xsg6)) / 15.) * xsgplus_m7,
            ((xsg * (540 - 2250 * xsg + 3600 * xsg2 - 2700 * xsg3 + 900 * xsg4 - 90 * xsg5)) / 45. + (xsg * exp_xom_xsgplus * (-540 - 540 * xom + 2250 * xsg + 1710 * xom * xsg - 270 * xom2 + 585 * xsg * xom2 - 105 * xom3 - 15 * xsg * xom3 + 30 * xom4 + 420 * xsg * xom4 - 54 * xom5 - 390 * xsg * xom5 + 16 * xom6 + 126 * xsg * xom6 - 2 * xom7 - 14 * xsg * xom7 - 3600 * xsg2 - 1350 * xom * xsg2 + 180 * xom2 * xsg2 - 165 * xom3 * xsg2 + 1530 * xom4 * xsg2 - 1290 * xom5 * xsg2 + 420 * xom6 * xsg2 - 42 * xom7 * xsg2 + 2700 * xsg3 - 900 * xom * xsg3 - 1125 * xom2 * xsg3 - 1155 * xom3 * xsg3 + 2820 * xom4 * xsg3 - 2490 * xom5 * xsg3 + 770 * xom6 * xsg3 - 70 * xom7 * xsg3 - 900 * xsg4 + 1800 * xom * xsg4 + 450 * xom2 * xsg4 - 1275 * xom3 * xsg4 + 3330 * xom4 * xsg4 - 2970 * xom5 * xsg4 + 840 * xom6 * xsg4 - 70 * xom7 * xsg4 + 90 * xsg5 - 810 * xom * xsg5 + 495 * xom2 * xsg5 - 525 * xom3 * xsg5 + 2700 * xom4 * xsg5 - 2154 * xom5 * xsg5 + 546 * xom6 * xsg5 - 42 * xom7 * xsg5 + 90 * xom * xsg6 - 360 * xom2 * xsg6 - 375 * xom3 * xsg6 + 1350 * xom4 * xsg6 - 870 * xom5 * xsg6 + 196 * xom6 * xsg6 - 14 * xom7 * xsg6 + 45 * xom2 * xsg7 - 225 * xom3 * xsg7 + 300 * xom4 * xsg7 - 150 * xom5 * xsg7 + 30 * xom6 * xsg7 - 2 * xom7 * xsg7)) / 45.) * xsgplus_m8,
            ((xsg * (-4410 + 22680 * xsg - 47250 * xsg2 + 50400 * xsg3 - 28350 * xsg4 + 7560 * xsg5 - 630 * xsg6)) / 315. + (xsg * exp_xom_xsgplus * (4410 + 4410 * xom - 22680 * xsg - 18270 * xom * xsg + 2205 * xom2 - 6930 * xsg * xom2 + 630 * xom3 - 2520 * xsg * xom3 + 630 * xom4 + 3780 * xsg * xom4 - 504 * xom5 - 4788 * xsg * xom5 + 266 * xom6 + 2268 * xsg * xom6 - 52 * xom7 - 448 * xsg * xom7 + 4 * xom8 + 32 * xsg * xom8 + 47250 * xsg2 + 24570 * xom * xsg2 + 3150 * xom2 * xsg2 - 5040 * xom3 * xsg2 + 15120 * xom4 * xsg2 - 18900 * xom5 * xsg2 + 8652 * xom6 * xsg2 - 1680 * xom7 * xsg2 + 112 * xom8 * xsg2 - 50400 * xsg3 - 3150 * xom * xsg3 + 10710 * xom2 * xsg3 - 4200 * xom3 * xsg3 + 37380 * xom4 * xsg3 - 42420 * xom5 * xsg3 + 19180 * xom6 * xsg3 - 3584 * xom7 * xsg3 + 224 * xom8 * xsg3 + 28350 * xsg4 - 22050 * xom * xsg4 - 12600 * xom2 * xsg4 - 13860 * xom3 * xsg4 + 55020 * xom4 * xsg4 - 60900 * xom5 * xsg4 + 26880 * xom6 * xsg4 - 4760 * xom7 * xsg4 + 280 * xom8 * xsg4 - 7560 * xsg5 + 20790 * xom * xsg5 - 630 * xom2 * xsg5 - 17640 * xom3 * xsg5 + 51660 * xom4 * xsg5 - 58044 * xom5 * xsg5 + 24276 * xom6 * xsg5 - 4032 * xom7 * xsg5 + 224 * xom8 * xsg5 + 630 * xsg6 - 6930 * xom * xsg6 + 6930 * xom2 * xsg6 - 6720 * xom3 * xsg6 + 33600 * xom4 * xsg6 - 35868 * xom5 * xsg6 + 13748 * xom6 * xsg6 - 2128 * xom7 * xsg6 + 112 * xom8 * xsg6 + 630 * xom * xsg7 - 3150 * xom2 * xsg7 - 2520 * xom3 * xsg7 + 14700 * xom4 * xsg7 - 13020 * xom5 * xsg7 + 4452 * xom6 * xsg7 - 640 * xom7 * xsg7 + 32 * xom8 * xsg7 + 315 * xom2 * xsg8 - 1890 * xom3 * xsg8 + 3150 * xom4 * xsg8 - 2100 * xom5 * xsg8 + 630 * xom6 * xsg8 - 84 * xom7 * xsg8 + 4 * xom8 * xsg8)) / 315.) * xsgplus_m9,
            ((xsg * (5040 - 30870 * xsg + 79380 * xsg2 - 110250 * xsg3 + 88200 * xsg4 - 39690 * xsg5 + 8820 * xsg6 - 630 * xsg7)) / 315. + (xsg * exp_xom_xsgplus * (-5040 - 5040 * xom + 30870 * xsg + 25830 * xom * xsg - 2520 * xom2 + 10395 * xsg * xom2 - 945 * xom3 + 1575 * xsg * xom3 + 315 * xom4 + 5670 * xsg * xom4 - 819 * xom5 - 7623 * xsg * xom5 + 476 * xom6 + 4718 * xsg * xom6 - 142 * xom7 - 1358 * xsg * xom7 + 19 * xom8 + 180 * xsg * xom8 - xom9 - 9 * xsg * xom9 - 79380 * xsg2 - 48510 * xom * xsg2 - 11340 * xom2 * xsg2 - 5040 * xom3 * xsg2 + 23940 * xom4 * xsg2 - 33768 * xom5 * xsg2 + 20664 * xom6 * xsg2 - 5824 * xom7 * xsg2 + 756 * xom8 * xsg2 - 36 * xom9 * xsg2 + 110250 * xsg3 + 30870 * xom * xsg3 - 8820 * xom2 * xsg3 - 19320 * xom3 * xsg3 + 60060 * xom4 * xsg3 - 89040 * xom5 * xsg3 + 53032 * xom6 * xsg3 - 14672 * xom7 * xsg3 + 1848 * xom8 * xsg3 - 84 * xom9 * xsg3 - 88200 * xsg4 + 22050 * xom * xsg4 + 26460 * xom2 * xsg4 - 16170 * xom3 * xsg4 + 106890 * xom4 * xsg4 - 151830 * xom5 * xsg4 + 88480 * xom6 * xsg4 - 23884 * xom7 * xsg4 + 2898 * xom8 * xsg4 - 126 * xom9 * xsg4 + 39690 * xsg5 - 48510 * xom * xsg5 - 13230 * xom2 * xsg5 - 22050 * xom3 * xsg5 + 129360 * xom4 * xsg5 - 175014 * xom5 * xsg5 + 99876 * xom6 * xsg5 - 26012 * xom7 * xsg5 + 3024 * xom8 * xsg5 - 126 * xom9 * xsg5 - 8820 * xsg6 + 30870 * xom * xsg6 - 8820 * xom2 * xsg6 - 29400 * xom3 * xsg6 + 102900 * xom4 * xsg6 - 138768 * xom5 * xsg6 + 76328 * xom6 * xsg6 - 18928 * xom7 * xsg6 + 2100 * xom8 * xsg6 - 84 * xom9 * xsg6 + 630 * xsg7 - 8190 * xom * xsg7 + 11340 * xom2 * xsg7 - 11760 * xom3 * xsg7 + 55860 * xom4 * xsg7 - 74088 * xom5 * xsg7 + 38024 * xom6 * xsg7 - 8864 * xom7 * xsg7 + 936 * xom8 * xsg7 - 36 * xom9 * xsg7 + 630 * xom * xsg8 - 3780 * xom2 * xsg8 - 2205 * xom3 * xsg8 + 21315 * xom4 * xsg8 - 24255 * xom5 * xsg8 + 11172 * xom6 * xsg8 - 2422 * xom7 * xsg8 + 243 * xom8 * xsg8 - 9 * xom9 * xsg8 + 315 * xom2 * xsg9 - 2205 * xom3 * xsg9 + 4410 * xom4 * xsg9 - 3675 * xom5 * xsg9 + 1470 * xom6 * xsg9 - 294 * xom7 * xsg9 + 28 * xom8 * xsg9 - xom9 * xsg9)) / 315.) * xsgplus_m10,
            ((xsg * (-51030 + 362880 * xsg - 1111320 * xsg2 + 1905120 * xsg3 - 1984500 * xsg4 + 1270080 * xsg5 - 476280 * xsg6 + 90720 * xsg7 - 5670 * xsg8)) / 2835. + (xsg * exp_xom_xsgplus * (51030 + 51030 * xom - 362880 * xsg - 311850 * xom * xsg + 2 * xom10 + 20 * xsg * xom10 + 25515 * xom2 - 130410 * xsg * xom2 + 7560 * xom3 - 45360 * xsg * xom3 + 7560 * xom4 + 52920 * xsg * xom4 - 9072 * xom5 - 104328 * xsg * xom5 + 7308 * xom6 + 78120 * xsg * xom6 - 2736 * xom7 - 29376 * xsg * xom7 + 540 * xom8 + 5688 * xsg * xom8 - 52 * xom9 - 540 * xsg * xom9 + 2 * xom10 * xsg10 + 2835 * xom2 * xsg10 - 22680 * xom3 * xsg10 + 52920 * xom4 * xsg10 - 52920 * xom5 * xsg10 + 26460 * xom6 * xsg10 - 7056 * xom7 * xsg10 + 1008 * xom8 * xsg10 - 72 * xom9 * xsg10 + 1111320 * xsg2 + 748440 * xom * xsg2 + 90 * xom10 * xsg2 + 218295 * xom2 * xsg2 - 22680 * xom3 * xsg2 + 294840 * xom4 * xsg2 - 512568 * xom5 * xsg2 + 382284 * xom6 * xsg2 - 142128 * xom7 * xsg2 + 27072 * xom8 * xsg2 - 2520 * xom9 * xsg2 - 1905120 * xsg3 - 793800 * xom * xsg3 + 240 * xom10 * xsg3 - 22680 * xom2 * xsg3 - 90720 * xom3 * xsg3 + 907200 * xom4 * xsg3 - 1505952 * xom5 * xsg3 + 1118880 * xom6 * xsg3 - 409248 * xom7 * xsg3 + 76608 * xom8 * xsg3 - 6960 * xom9 * xsg3 + 1984500 * xsg4 + 79380 * xom * xsg4 + 420 * xom10 * xsg4 - 357210 * xom2 * xsg4 - 438480 * xom3 * xsg4 + 1738800 * xom4 * xsg4 - 2963520 * xom5 * xsg4 + 2165688 * xom6 * xsg4 - 778176 * xom7 * xsg4 + 142632 * xom8 * xsg4 - 12600 * xom9 * xsg4 - 1270080 * xsg5 + 714420 * xom * xsg5 + 504 * xom10 * xsg5 + 396900 * xom2 * xsg5 - 423360 * xom3 * xsg5 + 2404080 * xom4 * xsg5 - 4061232 * xom5 * xsg5 + 2904048 * xom6 * xsg5 - 1022112 * xom7 * xsg5 + 182448 * xom8 * xsg5 - 15624 * xom9 * xsg5 + 476280 * xsg6 - 793800 * xom * xsg6 + 420 * xom10 * xsg6 - 39690 * xom2 * xsg6 - 317520 * xom3 * xsg6 + 2434320 * xom4 * xsg6 - 3922128 * xom5 * xsg6 + 2744280 * xom6 * xsg6 - 939456 * xom7 * xsg6 + 162288 * xom8 * xsg6 - 13440 * xom9 * xsg6 - 90720 * xsg7 + 385560 * xom * xsg7 + 240 * xom10 * xsg7 - 204120 * xom2 * xsg7 - 393120 * xom3 * xsg7 + 1693440 * xom4 * xsg7 - 2667168 * xom5 * xsg7 + 1812384 * xom6 * xsg7 - 596448 * xom7 * xsg7 + 99072 * xom8 * xsg7 - 7920 * xom9 * xsg7 + 5670 * xsg8 - 85050 * xom * xsg8 + 90 * xom10 * xsg8 + 150255 * xom2 * xsg8 - 173880 * xom3 * xsg8 + 793800 * xom4 * xsg8 - 1248912 * xom5 * xsg8 + 802620 * xom6 * xsg8 - 250128 * xom7 * xsg8 + 39708 * xom8 * xsg8 - 3060 * xom9 * xsg8 + 5670 * xom * xsg9 + 20 * xom10 * xsg9 - 39690 * xom2 * xsg9 - 15120 * xom3 * xsg9 + 264600 * xom4 * xsg9 - 370440 * xom5 * xsg9 + 215208 * xom6 * xsg9 - 62496 * xom7 * xsg9 + 9432 * xom8 * xsg9 - 700 * xom9 * xsg9)) / 2835.) * xsgplus_m11
        };

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);
        return std::inner_product(a_begin, a_end, c.begin(), 0.0);
    }

    double
    AnalyticFormFactorBToGammaQCDF::L0_effective(const double & Egamma) const
    {
        const double L_hat = eulergamma + std::log(mu*mu/(2.0 * Egamma*omega_0));
        const double L_hat2 = L_hat * L_hat;

        const Weights c = {
            -1 + L_hat2,
            2 * L_hat,
            (3 + L_hat2) / 3.,
            (4 * L_hat) / 3.,
            (5.666666666666667 + L_hat2) / 5.,
            (46 * L_hat) / 45.,
            (7.711111111111111 + L_hat2) / 7.,
            (88 * L_hat) / 105.,
            (9.387301587301588 + L_hat2) / 9.
        };

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);
        return 1.0 / omega_0 * std::inner_product(a_begin, a_end, c.begin(), 0.0);
    }

    double
    AnalyticFormFactorBToGammaQCDF::L0_incomplete_effective(const double & Egamma, const double & omega_cut) const
    {
        const double L = std::log(mu * mu / (2.0 * Egamma * omega_0));
        const double L2 = L * L;

        const double xom = omega_cut / omega_0;
        const double xom2 = xom  * xom;
        const double xom3 = xom2 * xom;
        const double xom4 = xom3 * xom;
        const double xom5 = xom4 * xom;
        const double xom6 = xom5 * xom;
        const double xom7 = xom6 * xom;
        const double xom8 = xom7 * xom;
        const double xom9 = xom8 * xom;

        const double log_xom = std::log(xom);
        const double log_xom2 = log_xom * log_xom;
        const double exp_xom = std::exp(-xom);
        const double expint_ei_xom = gsl_sf_expint_Ei(-xom);
        const double expint_ei_xom_plus = gsl_sf_expint_Ei(xom);

        const static thread_local CSplineInterpolation hypg_a_1_1_1_b_2_2_2_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.2031247968750000, 0.2109372890625000, 0.2187497812500000, 0.2265622734375000, 0.2343747656250000, 0.2421872578125000, 0.2499997500000000, 0.2578122421875000, 0.2656247343750000, 0.2734372265625000, 0.2812497187500000, 0.2890622109375000, 0.2968747031250000, 0.3046871953125000, 0.3124996875000000, 0.3203121796875000, 0.3281246718750000, 0.3359371640625000, 0.3437496562500000, 0.3515621484375000, 0.3593746406250000, 0.3671871328125000, 0.3749996250000000, 0.3828121171875000, 0.3906246093750000, 0.3984371015625000, 0.4062495937500000, 0.4140620859375000, 0.4218745781250000, 0.4296870703125000, 0.4374995625000000, 0.4453120546875000, 0.4531245468750000, 0.4609370390624999, 0.4687495312500000, 0.4765620234375000, 0.4843745156250000, 0.4921870078125000, 0.4999995000000000, 0.5078119921874999, 0.5156244843750000, 0.5234369765625000, 0.5312494687500000, 0.5390619609375000, 0.5468744531250000, 0.5546869453125001, 0.5624994375000000, 0.5703119296874999, 0.5781244218749999, 0.5859369140624999, 0.5937494062499999, 0.6015618984375000, 0.6093743906250000, 0.6171868828125000, 0.6249993750000000, 0.6328118671875000, 0.6406243593750001, 0.6562493437500000, 0.6718743281249999, 0.6874993125000000, 0.6953118046875000, 0.7031242968750000, 0.7109367890625000, 0.7187492812500000, 0.7265617734375001, 0.7343742656250001, 0.7421867578125001, 0.7499992500000000, 0.7578117421874999, 0.7656242343749999, 0.7734367265625000, 0.7812492187500000, 0.7890617109375000, 0.7968742031250000, 0.8046866953125000, 0.8124991875000001, 0.8203116796875001, 0.8281241718750001, 0.8359366640625001, 0.8437491562500000, 0.8515616484374999, 0.8593741406249999, 0.8671866328125000, 0.8749991250000000, 0.8789053710937500, 0.8828116171875000, 0.8906241093750000, 0.8945303554687500, 0.8984366015625000, 0.9062490937500000, 0.9101553398437499, 0.9140615859374999, 0.9218740781249999, 0.9257803242187499, 0.9296865703124999, 0.9335928164062499, 0.9374990624999999, 0.9414053085937499, 0.9453115546874999, 0.9492178007812500, 0.9531240468750000, 0.9570302929687500, 0.9609365390625000, 0.9648427851562500, 0.9687490312500000, 0.9726552773437500, 0.9765615234375000, 0.9785146464843750, 0.9804677695312500, 0.9843740156250000, 0.9863271386718750, 0.9882802617187501, 0.9902333847656251, 0.9921865078125001, 0.9941396308593751, 0.9960927539062501, 0.9980458769531251, 0.9990224384765625, 0.9999990000000000},
            {1.0000000000000000, 0.9941583929877398, 0.9883521510465745, 0.9768448054284812, 0.9711432221891033, 0.9654760449433876, 0.9542439450347797, 0.9486785398233764, 0.9431465755566899, 0.9321820015427964, 0.9267489070299220, 0.9213482839788568, 0.9106434805773999, 0.9053388140186672, 0.9000656465535604, 0.8896128354009065, 0.8844327048379499, 0.8792830996660734, 0.8690744917209817, 0.8640150021772723, 0.8589850645313900, 0.8490128724204984, 0.8440701320532680, 0.8391559718262925, 0.8294124220512941, 0.8245825482188180, 0.8197802860033563, 0.8102576309261300, 0.8055367561278833, 0.8008425291177979, 0.7915330586366719, 0.7869173362824501, 0.7823273039922284, 0.7732233568201523, 0.7687089667831926, 0.7642193165399559, 0.7553132909937376, 0.7508964449022166, 0.7465033970646480, 0.7377877612736387, 0.7291645203530777, 0.7206318217958313, 0.7121878243784299, 0.7038306986569567, 0.6955586274343231, 0.6873698061976266, 0.6792624435243149, 0.6712347614559194, 0.6632849958381590, 0.6554113966262541, 0.6476122281543372, 0.6398857693678888, 0.6322303140181778, 0.6246441708177264, 0.6171256635558746, 0.6096731311735598, 0.6022849277964757, 0.5949594227258184, 0.5876950003858655, 0.5804900602276731, 0.5733430165882032, 0.5662522985042238, 0.5592163494803361, 0.5522336272104963, 0.5453026032523957, 0.5384217626540494, 0.5315896035319203, 0.5248046365998611, 0.5180653846480998, 0.5113703819714212, 0.5047181737455944, 0.4981073153509825, 0.4915363716421214, 0.4850039161618859, 0.4785085302986579, 0.4720488023846726, 0.4656233267334515, 0.4592307026139130, 0.4528695331583941, 0.4465384242014142, 0.4402359830455434, 0.4339608171502190, 0.4277115327387633, 0.4214867333181869, 0.4152850181056122, 0.4091049803542997, 0.4029452055713061, 0.3968042696177170, 0.3906807366811726, 0.3845731571090184, 0.3784800650888335, 0.3723999761612959, 0.3663313845482948, 0.3602727602768579, 0.3542225460767691, 0.3481791540266612, 0.3361063093165388, 0.3240407635128744, 0.3119682948323468, 0.3059247705619786, 0.2998737489207574, 0.2938131554252623, 0.2877408285031435, 0.2816545100914322, 0.2755518351207863, 0.2694303197118364, 0.2632873478767649, 0.2571201564787214, 0.2509258181516748, 0.2447012218212089, 0.2384430503891427, 0.2321477550471277, 0.2258115255603932, 0.2194302557042400, 0.2129995028313228, 0.2065144402813779, 0.1999698009946798, 0.1933598102246399, 0.1866781046181862, 0.1799176340784925, 0.1730705416444632, 0.1661280149655309, 0.1590801005875836, 0.1555131439463915, 0.1519154688310793, 0.1446211119445585, 0.1409206683676983, 0.1371819504727896, 0.1295803107026743, 0.1257121599092061, 0.1217952166481772, 0.1138014077773262, 0.1097168370871317, 0.1055679378727146, 0.1013496160676611, 0.0970561092213285, 0.0926808517420605, 0.0882163022620624, 0.0836537190049423, 0.0789828622727168, 0.0741915923206841, 0.0692653128846872, 0.0641861795352822, 0.0589319357745224, 0.0534741321275083, 0.0477752628698260, 0.0448200951501434, 0.0417838625672159, 0.0354253761121182, 0.0320745745955113, 0.0285830500427352, 0.0249220635580111, 0.0210501476667170, 0.0169026546168545, 0.0123657102415211, 0.0071870900768604, 0.0041373732987571, 0.0000000000000000}
        };

        // precision target for cubic interpolation is 1e-6
        // data range is mapped from (-1e6, 0) -> (0, 1) by x -> x/(-4 + x)
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_0_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0117187382812500, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0273437226562500, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0429687070312500, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0585936914062500, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0742186757812500, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0898436601562500, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1054686445312500, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1210936289062500, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1367186132812500, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1523435976562500, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1679685820312500, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1835935664062500, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.1992185507812500, 0.2031247968750000, 0.2070310429687500, 0.2109372890625000, 0.2148435351562500, 0.2187497812500000, 0.2226560273437500, 0.2265622734375000, 0.2304685195312500, 0.2343747656250000, 0.2382810117187500, 0.2421872578125000, 0.2460935039062500, 0.2499997500000000, 0.2539059960937500, 0.2578122421875000, 0.2617184882812500, 0.2656247343750000, 0.2695309804687500, 0.2734372265625000, 0.2773434726562500, 0.2812497187500000, 0.2851559648437499, 0.2890622109375000, 0.2929684570312500, 0.2968747031250000, 0.3007809492187500, 0.3046871953125000, 0.3085934414062500, 0.3124996875000000, 0.3164059335937500, 0.3203121796875000, 0.3242184257812500, 0.3281246718750000, 0.3320309179687500, 0.3359371640625000, 0.3398434101562500, 0.3437496562500000, 0.3476559023437500, 0.3515621484375000, 0.3554683945312500, 0.3593746406250000, 0.3632808867187500, 0.3671871328125000, 0.3710933789062500, 0.3749996250000000, 0.3789058710937500, 0.3828121171875000, 0.3867183632812500, 0.3906246093750000, 0.3945308554687500, 0.3984371015625000, 0.4023433476562500, 0.4062495937500000, 0.4101558398437500, 0.4140620859375000, 0.4179683320312501, 0.4218745781250000, 0.4257808242187500, 0.4296870703125000, 0.4335933164062500, 0.4374995625000000, 0.4414058085937500, 0.4453120546875000, 0.4492183007812500, 0.4531245468750000, 0.4570307929687499, 0.4609370390624999, 0.4648432851562500, 0.4687495312500000, 0.4726557773437500, 0.4765620234375000, 0.4804682695312500, 0.4843745156250000, 0.4882807617187500, 0.4921870078125000, 0.4960932539062500, 0.4999995000000000, 0.5039057460937499, 0.5078119921874999, 0.5117182382812500, 0.5156244843750000, 0.5195307304687500, 0.5234369765625000, 0.5273432226562500, 0.5312494687500000, 0.5351557148437500, 0.5390619609375000, 0.5429682070312500, 0.5468744531250000, 0.5507806992187501, 0.5546869453125001, 0.5585931914062501, 0.5624994375000000, 0.5664056835937499, 0.5703119296874999, 0.5742181757812499, 0.5781244218749999, 0.5820306679687499, 0.5859369140624999, 0.5898431601562499, 0.5937494062499999, 0.5976556523437500, 0.6015618984375000, 0.6054681445312500, 0.6093743906250000, 0.6132806367187500, 0.6171868828125000, 0.6210931289062500, 0.6249993750000000, 0.6289056210937500, 0.6328118671875000, 0.6367181132812501, 0.6406243593750001, 0.6445306054687501, 0.6484368515625001, 0.6523430976562501, 0.6562493437500000, 0.6601555898437499, 0.6640618359374999, 0.6679680820312499, 0.6718743281249999, 0.6757805742187499, 0.6796868203124999, 0.6835930664062499, 0.6874993125000000, 0.6914055585937500, 0.6953118046875000, 0.6992180507812500, 0.7031242968750000, 0.7070305429687500, 0.7109367890625000, 0.7148430351562500, 0.7187492812500000, 0.7226555273437500, 0.7265617734375001, 0.7304680195312501, 0.7343742656250001, 0.7382805117187501, 0.7421867578125001, 0.7460930039062501, 0.7499992500000000, 0.7539054960937499, 0.7578117421874999, 0.7617179882812499, 0.7656242343749999, 0.7695304804687499, 0.7734367265625000, 0.7773429726562500, 0.7812492187500000, 0.7851554648437500, 0.7890617109375000, 0.7929679570312500, 0.7968742031250000, 0.8007804492187500, 0.8046866953125000, 0.8085929414062500, 0.8124991875000001, 0.8164054335937501, 0.8203116796875001, 0.8242179257812501, 0.8281241718750001, 0.8320304179687501, 0.8359366640625001, 0.8398429101562501, 0.8437491562500000, 0.8476554023437499, 0.8515616484374999, 0.8554678945312499, 0.8593741406249999, 0.8632803867187500, 0.8671866328125000, 0.8710928789062500, 0.8749991250000000, 0.8789053710937500, 0.8828116171875000, 0.8867178632812500, 0.8906241093750000, 0.8945303554687500, 0.8984366015625000, 0.9023428476562501, 0.9062490937500000, 0.9082022167968749, 0.9101553398437499, 0.9140615859374999, 0.9160147089843749, 0.9179678320312499, 0.9218740781249999, 0.9238272011718749, 0.9257803242187499, 0.9296865703124999, 0.9316396933593749, 0.9335928164062499, 0.9374990624999999, 0.9394521855468749, 0.9414053085937499, 0.9453115546874999, 0.9472646777343749, 0.9492178007812500, 0.9531240468750000, 0.9550771699218750, 0.9570302929687500, 0.9589834160156250, 0.9609365390625000, 0.9628896621093750, 0.9648427851562500, 0.9667959082031250, 0.9687490312500000, 0.9707021542968750, 0.9726552773437500, 0.9746084003906250, 0.9765615234375000, 0.9785146464843750, 0.9804677695312500, 0.9824208925781250, 0.9843740156250000, 0.9863271386718750, 0.9882802617187501, 0.9892568232421876, 0.9902333847656251, 0.9921865078125001, 0.9931630693359376, 0.9941396308593751, 0.9951161923828126, 0.9960927539062501, 0.9970693154296876, 0.9980458769531251, 0.9990224384765625, 0.9995107192382813, 0.9999990000000000},
            {0.0000000000000000, 0.0019494341650450, 0.0038914284669273, 0.0058258893267273, 0.0077527225949886, 0.0096718335548274, 0.0115831269252528, 0.0134865068647041, 0.0153818769748094, 0.0172691403043711, 0.0191481993535844, 0.0210189560784937, 0.0228813118956928, 0.0247351676872748, 0.0265804238060372, 0.0284169800809481, 0.0302447358228797, 0.0320635898306150, 0.0338734403971336, 0.0356741853161833, 0.0374657218891426, 0.0392479469321821, 0.0410207567837296, 0.0427840473122461, 0.0445377139243186, 0.0462816515730776, 0.0480157547669436, 0.0497399175787116, 0.0514540336549784, 0.0531579962259209, 0.0548516981154304, 0.0565350317516116, 0.0582078891776518, 0.0598701620630676, 0.0615217417153349, 0.0631625190919116, 0.0647923848126566, 0.0664112291726538, 0.0680189421554483, 0.0696154134466999, 0.0712005324482620, 0.0727741882926921, 0.0743362698582007, 0.0758866657840449, 0.0774252644863743, 0.0789519541745333, 0.0804666228678295, 0.0819691584127713, 0.0834594485007828, 0.0849373806864012, 0.0864028424059631, 0.0878557209967850, 0.0892959037168429, 0.0907232777649586, 0.0921377303014948, 0.0935391484695665, 0.0949274194167717, 0.0963024303174471, 0.0976640683954516, 0.0990122209474823, 0.1003467753669271, 0.1016676191682549, 0.1029746400119490, 0.1042677257299834, 0.1055467643518462, 0.1068116441311102, 0.1080622535725525, 0.1092984814598226, 0.1105202168836610, 0.1117273492706651, 0.1129197684126036, 0.1140973644962763, 0.1152600281339166, 0.1164076503941355, 0.1175401228334000, 0.1186573375280440, 0.1197591871068052, 0.1208455647838809, 0.1219163643924969, 0.1229714804189801, 0.1240108080373256, 0.1250342431442494, 0.1260416823947134, 0.1270330232379118, 0.1280081639537048, 0.1289670036894845, 0.1299094424974568, 0.1308353813723220, 0.1317447222893360, 0.1326373682427301, 0.1335132232844689, 0.1343721925633224, 0.1352141823642280, 0.1360391001479141, 0.1368468545907590, 0.1376373556248537, 0.1384105144782360, 0.1391662437152641, 0.1399044572770909, 0.1406250705222041, 0.1413280002669892, 0.1420131648262753, 0.1426804840538186, 0.1433298793826765, 0.1439612738654240, 0.1445745922141606, 0.1451697608402541, 0.1457467078937649, 0.1463053633024922, 0.1468456588105802, 0.1473675280166205, 0.1478709064111845, 0.1483557314137151, 0.1488219424087069, 0.1492694807810994, 0.1496982899508042, 0.1501083154062881, 0.1504995047371265, 0.1508718076654417, 0.1512251760761368, 0.1515595640458333, 0.1518749278704179, 0.1521712260910998, 0.1524484195188791, 0.1527064712573235, 0.1529453467235455, 0.1531650136672746, 0.1533654421879110, 0.1535466047494480, 0.1537084761931488, 0.1538510337478570, 0.1539742570378234, 0.1540781280879244, 0.1541626313261503, 0.1542277535832356, 0.1542734840893066, 0.1542998144674155, 0.1543067387238334, 0.1542942532349713, 0.1542623567307983, 0.1542110502746250, 0.1541403372391231, 0.1540502232784486, 0.1539407162963385, 0.1538118264100527, 0.1536635659100322, 0.1534959492151468, 0.1533089928234097, 0.1531027152580340, 0.1528771370087147, 0.1526322804680176, 0.1523681698627627, 0.1520848311802946, 0.1517822920895328, 0.1514605818567025, 0.1511197312556511, 0.1507597724726587, 0.1503807390056589, 0.1499826655577897, 0.1495655879252015, 0.1491295428790549, 0.1486745680416462, 0.1482007017566062, 0.1477079829531217, 0.1471964510041366, 0.1466661455784944, 0.1461171064869883, 0.1455493735222899, 0.1449629862927327, 0.1443579840499256, 0.1437344055101784, 0.1430922886697173, 0.1424316706136705, 0.1417525873187986, 0.1410550734499433, 0.1403391621501558, 0.1396048848244619, 0.1388522709172047, 0.1380813476828908, 0.1372921399504474, 0.1364846698807726, 0.1356589567174344, 0.1348150165303403, 0.1339528619521614, 0.1330725019072503, 0.1321739413327411, 0.1312571808914632, 0.1303222166762340, 0.1293690399050214, 0.1283976366063853, 0.1274079872945139, 0.1264000666330665, 0.1253738430869195, 0.1243292785607840, 0.1232663280235225, 0.1221849391168315, 0.1210850517467852, 0.1199665976565406, 0.1188294999782913, 0.1176736727623170, 0.1164990204807169, 0.1153054375031157, 0.1140928075413114, 0.1128610030594627, 0.1116098846460112, 0.1103393003430711, 0.1090490849285034, 0.1077390591453093, 0.1064090288723108, 0.1050587842293334, 0.1036880986092418, 0.1022967276281851, 0.1008844079842677, 0.0994508562135340, 0.0979957673306222, 0.0965188133396424, 0.0950196415987369, 0.0934978730193059, 0.0919531000779621, 0.0903848846158183, 0.0887927553955834, 0.0871762053820123, 0.0855346887053220, 0.0838676172600224, 0.0821743568829103, 0.0804542230433597, 0.0787064759659973, 0.0769303150897513, 0.0751248727472315, 0.0732892069233272, 0.0714222929202839, 0.0695230137163175, 0.0675901487532783, 0.0666107086950009, 0.0656223608221823, 0.0636181806282873, 0.0626019437292555, 0.0615759885023123, 0.0594939925706736, 0.0584374547751791, 0.0573702024896724, 0.0552023975634305, 0.0541012231425472, 0.0529880876684684, 0.0507244648461554, 0.0495731820709986, 0.0484083426148083, 0.0460360788626732, 0.0448276072815311, 0.0436034763916744, 0.0411056524162698, 0.0398305272527474, 0.0385368639255316, 0.0372237710323088, 0.0358902685894090, 0.0345352745625388, 0.0331575885362366, 0.0317558717216962, 0.0303286222183275, 0.0288741440349857, 0.0273905077749643, 0.0258754999838590, 0.0243265567631168, 0.0227406750338798, 0.0211142911914191, 0.0194431106649468, 0.0177218607627844, 0.0159439181422156, 0.0141007197255096, 0.0131512131407799, 0.0121807724340095, 0.0101678480765908, 0.0091193232381536, 0.0080372985562515, 0.0069159622225416, 0.0057471225171468, 0.0045183600390773, 0.0032085515710022, 0.0017731461793306, 0.0009717935716340, 0.0000035335249754}
        };
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_1_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0117187382812500, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0273437226562500, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0429687070312500, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0585936914062500, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0742186757812500, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0898436601562500, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1054686445312500, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1210936289062500, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1367186132812500, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1523435976562500, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1679685820312500, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1835935664062500, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.1992185507812500, 0.2031247968750000, 0.2070310429687500, 0.2109372890625000, 0.2148435351562500, 0.2187497812500000, 0.2226560273437500, 0.2265622734375000, 0.2304685195312500, 0.2343747656250000, 0.2382810117187500, 0.2421872578125000, 0.2460935039062500, 0.2499997500000000, 0.2539059960937500, 0.2578122421875000, 0.2617184882812500, 0.2656247343750000, 0.2695309804687500, 0.2734372265625000, 0.2773434726562500, 0.2812497187500000, 0.2851559648437499, 0.2890622109375000, 0.2929684570312500, 0.2968747031250000, 0.3007809492187500, 0.3046871953125000, 0.3085934414062500, 0.3124996875000000, 0.3164059335937500, 0.3203121796875000, 0.3242184257812500, 0.3281246718750000, 0.3320309179687500, 0.3359371640625000, 0.3398434101562500, 0.3437496562500000, 0.3476559023437500, 0.3515621484375000, 0.3554683945312500, 0.3593746406250000, 0.3632808867187500, 0.3671871328125000, 0.3710933789062500, 0.3749996250000000, 0.3789058710937500, 0.3828121171875000, 0.3867183632812500, 0.3906246093750000, 0.3945308554687500, 0.3984371015625000, 0.4023433476562500, 0.4062495937500000, 0.4101558398437500, 0.4140620859375000, 0.4179683320312501, 0.4218745781250000, 0.4257808242187500, 0.4296870703125000, 0.4335933164062500, 0.4374995625000000, 0.4414058085937500, 0.4453120546875000, 0.4492183007812500, 0.4531245468750000, 0.4570307929687499, 0.4609370390624999, 0.4648432851562500, 0.4687495312500000, 0.4726557773437500, 0.4765620234375000, 0.4804682695312500, 0.4843745156250000, 0.4882807617187500, 0.4921870078125000, 0.4960932539062500, 0.4999995000000000, 0.5039057460937499, 0.5078119921874999, 0.5117182382812500, 0.5156244843750000, 0.5234369765625000, 0.5273432226562500, 0.5312494687500000, 0.5390619609375000, 0.5429682070312500, 0.5468744531250000, 0.5546869453125001, 0.5585931914062501, 0.5624994375000000, 0.5703119296874999, 0.5742181757812499, 0.5781244218749999, 0.5859369140624999, 0.5898431601562499, 0.5937494062499999, 0.6015618984375000, 0.6054681445312500, 0.6093743906250000, 0.6171868828125000, 0.6210931289062500, 0.6249993750000000, 0.6328118671875000, 0.6367181132812501, 0.6406243593750001, 0.6484368515625001, 0.6523430976562501, 0.6562493437500000, 0.6640618359374999, 0.6679680820312499, 0.6718743281249999, 0.6796868203124999, 0.6835930664062499, 0.6874993125000000, 0.6953118046875000, 0.6992180507812500, 0.7031242968750000, 0.7109367890625000, 0.7187492812500000, 0.7265617734375001, 0.7343742656250001, 0.7421867578125001, 0.7499992500000000, 0.7578117421874999, 0.7656242343749999, 0.7734367265625000, 0.7812492187500000, 0.7890617109375000, 0.7968742031250000, 0.8046866953125000, 0.8124991875000001, 0.8203116796875001, 0.8281241718750001, 0.8437491562500000, 0.8593741406249999, 0.8671866328125000, 0.8749991250000000, 0.8828116171875000, 0.8906241093750000, 0.8984366015625000, 0.9062490937500000, 0.9140615859374999, 0.9218740781249999, 0.9296865703124999, 0.9374990624999999, 0.9453115546874999, 0.9531240468750000, 0.9609365390625000, 0.9687490312500000, 0.9765615234375000, 0.9804677695312500, 0.9843740156250000, 0.9921865078125001, 0.9960927539062501, 0.9999990000000000},
            {-0.1250000000000000, -0.1235558658249852, -0.1221170184759929, -0.1206835174490849, -0.1192554224618543, -0.1178327934476991, -0.1164156905498895, -0.1150041741154252, -0.1135983046886775, -0.1121981430048129, -0.1108037499829920, -0.1094151867193419, -0.1080325144796940, -0.1066557946920858, -0.1052850889390198, -0.1039204589494766, -0.1025619665906752, -0.1012096738595783, -0.0998636428741358, -0.0985239358642626, -0.0971906151625452, -0.0958637431946735, -0.0945433824695901, -0.0932295955693563, -0.0919224451387255, -0.0906219938744220, -0.0893283045141188, -0.0880414398251100, -0.0867614625926725, -0.0854884356081116, -0.0842224216564864, -0.0829634835040097, -0.0817116838851167, -0.0804670854891982, -0.0792297509469939, -0.0779997428166394, -0.0767771235693633, -0.0755619555748296, -0.0743543010861199, -0.0731542222243507, -0.0719617809629227, -0.0707770391113955, -0.0696000582989843, -0.0684308999576750, -0.0672696253049518, -0.0661162953261341, -0.0649709707563196, -0.0638337120619274, -0.0627045794218401, -0.0615836327081395, -0.0604709314664332, -0.0593665348957691, -0.0582705018281345, -0.0571828907075376, -0.0561037595686682, -0.0550331660151357, -0.0539711671972829, -0.0529178197895731, -0.0518731799675497, -0.0508373033843666, -0.0498102451468900, -0.0487920597913693, -0.0477828012586789, -0.0467825228691305, -0.0457912772968563, -0.0448091165437657, -0.0438360919130761, -0.0428722539824202, -0.0419176525765332, -0.0409723367395230, -0.0400363547067263, -0.0391097538761570, -0.0381925807795499, -0.0372848810530075, -0.0363866994072545, -0.0354980795975082, -0.0346190643929736, -0.0337496955459689, -0.0328900137606948, -0.0320400586616546, -0.0311998687617389, -0.0303694814299845, -0.0295489328590236, -0.0287382580322345, -0.0279374906906107, -0.0271466632993639, -0.0263658070142781, -0.0255949516478339, -0.0248341256351212, -0.0240833559995623, -0.0233426683184665, -0.0226120866884408, -0.0218916336906791, -0.0211813303561582, -0.0204811961307660, -0.0197912488403924, -0.0191115046560115, -0.0184419780587880, -0.0177826818052408, -0.0171336268924977, -0.0164948225236798, -0.0158662760734507, -0.0152479930537732, -0.0146399770799126, -0.0140422298367307, -0.0134547510453156, -0.0128775384299931, -0.0123105876857687, -0.0117538924462492, -0.0112074442520975, -0.0106712325200726, -0.0101452445127109, -0.0096294653087060, -0.0091238777740458, -0.0086284625339674, -0.0081431979457920, -0.0076680600727051, -0.0072030226585454, -0.0067480571036712, -0.0063031324419736, -0.0058682153191040, -0.0054432699719912, -0.0050282582097174, -0.0046231393958297, -0.0042278704321611, -0.0038424057442372, -0.0034666972683455, -0.0031006944403453, -0.0027443441862957, -0.0023975909149808, -0.0020603765124100, -0.0017326403383721, -0.0014143192251221, -0.0008056568799992, -0.0005151766945408, -0.0002338336762269, 0.0003017463257751, 0.0005561422430556, 0.0008017188262207, 0.0012667587384433, 0.0014864004320544, 0.0016975794638405, 0.0020949319820289, 0.0022813023529515, 0.0024596037500883, 0.0027924174061432, 0.0029471437725396, 0.0030942292741455, 0.0033659276686001, 0.0034907701967745, 0.0036084310001183, 0.0038226856850915, 0.0039195226406826, 0.0040096638654414, 0.0041703609473082, 0.0042411708579454, 0.0043057929706487, 0.0044169938250063, 0.0044638348190246, 0.0045050123301091, 0.0045709092034097, 0.0045958959891502, 0.0046157539400851, 0.0046406216342974, 0.0046459007994905, 0.0046465897735420, 0.0046347350405608, 0.0046224595626193, 0.0046061301594390, 0.0045618408239977, 0.0045029227472987, 0.0044304188708822, 0.0043453567711508, 0.0042487462980402, 0.0041415774777552, 0.0040248187069510, 0.0038994152604154, 0.0037662881282566, 0.0036263331921027, 0.0034804207432199, 0.0033293953391779, 0.0031740759901812, 0.0030152566619044, 0.0028537070790492, 0.0026901738132483, 0.0023600351587877, 0.0020304082606532, 0.0018674543255909, 0.0017066041878403, 0.0015484952542815, 0.0013937605595234, 0.0012430328955793, 0.0010969496896764, 0.0009561588808497, 0.0008213261636451, 0.0006931441534068, 0.0005723443347200, 0.0004597131862118, 0.0003561148496028, 0.0002625246293297, 0.0001800817448329, 0.0001101798358275, 0.0000804766552739, 0.0000546430401231, 0.0000161496571136, 0.0000046751592665, 0.0000000000008209}
        };
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_2_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0117187382812500, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0273437226562500, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0429687070312500, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0585936914062500, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0742186757812500, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0898436601562500, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1054686445312500, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1210936289062500, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1367186132812500, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1523435976562500, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1679685820312500, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1835935664062500, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.1992185507812500, 0.2031247968750000, 0.2070310429687500, 0.2109372890625000, 0.2148435351562500, 0.2187497812500000, 0.2226560273437500, 0.2265622734375000, 0.2304685195312500, 0.2343747656250000, 0.2382810117187500, 0.2421872578125000, 0.2460935039062500, 0.2499997500000000, 0.2539059960937500, 0.2578122421875000, 0.2617184882812500, 0.2656247343750000, 0.2695309804687500, 0.2734372265625000, 0.2773434726562500, 0.2812497187500000, 0.2851559648437499, 0.2890622109375000, 0.2929684570312500, 0.2968747031250000, 0.3007809492187500, 0.3046871953125000, 0.3085934414062500, 0.3124996875000000, 0.3164059335937500, 0.3203121796875000, 0.3242184257812500, 0.3281246718750000, 0.3320309179687500, 0.3359371640625000, 0.3398434101562500, 0.3437496562500000, 0.3476559023437500, 0.3515621484375000, 0.3554683945312500, 0.3593746406250000, 0.3632808867187500, 0.3671871328125000, 0.3710933789062500, 0.3749996250000000, 0.3789058710937500, 0.3828121171875000, 0.3867183632812500, 0.3906246093750000, 0.3945308554687500, 0.3984371015625000, 0.4023433476562500, 0.4062495937500000, 0.4101558398437500, 0.4140620859375000, 0.4179683320312501, 0.4218745781250000, 0.4257808242187500, 0.4296870703125000, 0.4335933164062500, 0.4374995625000000, 0.4414058085937500, 0.4453120546875000, 0.4492183007812500, 0.4531245468750000, 0.4570307929687499, 0.4609370390624999, 0.4648432851562500, 0.4687495312500000, 0.4726557773437500, 0.4765620234375000, 0.4804682695312500, 0.4843745156250000, 0.4921870078125000, 0.4960932539062500, 0.4999995000000000, 0.5078119921874999, 0.5117182382812500, 0.5156244843750000, 0.5234369765625000, 0.5273432226562500, 0.5312494687500000, 0.5390619609375000, 0.5429682070312500, 0.5468744531250000, 0.5546869453125001, 0.5585931914062501, 0.5624994375000000, 0.5703119296874999, 0.5742181757812499, 0.5781244218749999, 0.5859369140624999, 0.5898431601562499, 0.5937494062499999, 0.6015618984375000, 0.6054681445312500, 0.6093743906250000, 0.6171868828125000, 0.6210931289062500, 0.6249993750000000, 0.6328118671875000, 0.6367181132812501, 0.6406243593750001, 0.6484368515625001, 0.6523430976562501, 0.6562493437500000, 0.6640618359374999, 0.6718743281249999, 0.6796868203124999, 0.6874993125000000, 0.6953118046875000, 0.7031242968750000, 0.7109367890625000, 0.7187492812500000, 0.7265617734375001, 0.7343742656250001, 0.7421867578125001, 0.7499992500000000, 0.7578117421874999, 0.7656242343749999, 0.7734367265625000, 0.7812492187500000, 0.7890617109375000, 0.7968742031250000, 0.8046866953125000, 0.8124991875000001, 0.8281241718750001, 0.8437491562500000, 0.8593741406249999, 0.8749991250000000, 0.8828116171875000, 0.8906241093750000, 0.9062490937500000, 0.9140615859374999, 0.9218740781249999, 0.9374990624999999, 0.9453115546874999, 0.9531240468750000, 0.9687490312500000, 0.9765615234375000, 0.9843740156250000, 0.9999990000000000},
            {-0.0925925925925926, -0.0915367918340707, -0.0904853017288529, -0.0894381656561913, -0.0883954270907969, -0.0873571295971853, -0.0863233168238511, -0.0852940324972690, -0.0842693204157169, -0.0832492244429193, -0.0822337885015060, -0.0812230565662849, -0.0802170726573235, -0.0792158808328377, -0.0782195251818828, -0.0772280498168453, -0.0762414988657305, -0.0752599164642431, -0.0742833467476582, -0.0733118338424780, -0.0723454218578722, -0.0713841548768980, -0.0704280769474960, -0.0694772320732600, -0.0685316642039759, -0.0675914172259275, -0.0666565349519657, -0.0657270611113370, -0.0648030393392695, -0.0638845131663125, -0.0629715260074260, -0.0620641211508183, -0.0611623417465279, -0.0602662307947465, -0.0593758311338808, -0.0584911854283502, -0.0576123361561170, -0.0567393255959473, -0.0558721958143991, -0.0550109886525367, -0.0541557457123663, -0.0533065083429938, -0.0524633176265001, -0.0516262143635331, -0.0507952390586144, -0.0499704319051588, -0.0491518327702051, -0.0483394811788570, -0.0475334162984326, -0.0467336769223218, -0.0459403014535505, -0.0451533278880517, -0.0443727937976421, -0.0435987363127049, -0.0428311921045792, -0.0420701973676549, -0.0413157878011762, -0.0405679985907527, -0.0398268643895802, -0.0390924192993732, -0.0383646968510106, -0.0376437299848973, -0.0369295510310445, -0.0362221916888714, -0.0355216830067331, -0.0348280553611772, -0.0341413384359349, -0.0334615612006512, -0.0327887518893593, -0.0321229379787059, -0.0314641461659340, -0.0308124023466290, -0.0301677315922386, -0.0295301581273717, -0.0288997053068877, -0.0282763955927855, -0.0276602505309005, -0.0270512907274245, -0.0264495358252568, -0.0258550044802007, -0.0252677143370197, -0.0246876820053653, -0.0241149230355937, -0.0235494518944866, -0.0229912819408922, -0.0224404254013057, -0.0218968933454069, -0.0213606956615753, -0.0208318410324031, -0.0203103369102283, -0.0197961894927108, -0.0192894036984746, -0.0187899831428427, -0.0182979301136897, -0.0178132455474405, -0.0173359290052423, -0.0168659786493418, -0.0164033912196969, -0.0159481620108554, -0.0155002848491359, -0.0150597520701433, -0.0146265544966574, -0.0142006814169300, -0.0137821205634315, -0.0133708580920847, -0.0129668785620295, -0.0125701649159592, -0.0121806984610747, -0.0117984588506994, -0.0114234240666041, -0.0110555704020878, -0.0106948724458646, -0.0103413030668076, -0.0099948333995998, -0.0096554328313465, -0.0093230689892024, -0.0089977077290681, -0.0086793131254125, -0.0083678474622775, -0.0080632712255228, -0.0077655430963701, -0.0074746199463044, -0.0071904568333930, -0.0069130070000824, -0.0066422218725337, -0.0061204423651992, -0.0058693417730695, -0.0056246934724205, -0.0051545215323254, -0.0049288773365876, -0.0047094443453420, -0.0042889515846130, -0.0040877573266531, -0.0038925053388075, -0.0035195407940982, -0.0033416804997141, -0.0031694670615045, -0.0028416681550848, -0.0026859226793646, -0.0025355041255668, -0.0022503123529280, -0.0021153681495894, -0.0019854090001676, -0.0017400906290268, -0.0016245510613711, -0.0015136359767293, -0.0013053078909022, -0.0012077071163347, -0.0011143554177251, -0.0009400160360374, -0.0008588353844074, -0.0007815180276282, -0.0006380829712837, -0.0005717695991922, -0.0005089283453810, -0.0003932702209779, -0.0003402576630535, -0.0002903260244856, -0.0001993173462271, -0.0001194727685444, -0.0000500306847399, 0.0000097579218754, 0.0000606267710789, 0.0001032916841431, 0.0001384481208921, 0.0001667688880756, 0.0001889020501067, 0.0002054690705757, 0.0002170632092729, 0.0002242481947470, 0.0002275571867863, 0.0002274920367640, 0.0002245228467877, 0.0002190878212949, 0.0002115933975105, 0.0002024146344009, 0.0001918958338106, 0.0001803513627194, 0.0001552992677260, 0.0001292151570925, 0.0001036507073424, 0.0000797938362560, 0.0000687874508492, 0.0000585088694674, 0.0000403731919531, 0.0000325961907488, 0.0000257087541639, 0.0000146069702374, 0.0000103618276218, 0.0000069449346932, 0.0000023886647772, 0.0000011088029502, 0.0000003717014763, 0.0000000000000000}
        };
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_3_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0117187382812500, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0273437226562500, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0429687070312500, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0585936914062500, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0742186757812500, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0898436601562500, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1054686445312500, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1210936289062500, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1367186132812500, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1523435976562500, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1679685820312500, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1835935664062500, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.1992185507812500, 0.2031247968750000, 0.2070310429687500, 0.2109372890625000, 0.2148435351562500, 0.2187497812500000, 0.2226560273437500, 0.2265622734375000, 0.2304685195312500, 0.2343747656250000, 0.2382810117187500, 0.2421872578125000, 0.2460935039062500, 0.2499997500000000, 0.2539059960937500, 0.2578122421875000, 0.2617184882812500, 0.2656247343750000, 0.2695309804687500, 0.2734372265625000, 0.2773434726562500, 0.2812497187500000, 0.2851559648437499, 0.2890622109375000, 0.2929684570312500, 0.2968747031250000, 0.3007809492187500, 0.3046871953125000, 0.3085934414062500, 0.3124996875000000, 0.3164059335937500, 0.3203121796875000, 0.3242184257812500, 0.3281246718750000, 0.3320309179687500, 0.3359371640625000, 0.3398434101562500, 0.3437496562500000, 0.3476559023437500, 0.3515621484375000, 0.3554683945312500, 0.3593746406250000, 0.3632808867187500, 0.3671871328125000, 0.3710933789062500, 0.3749996250000000, 0.3789058710937500, 0.3828121171875000, 0.3867183632812500, 0.3906246093750000, 0.3945308554687500, 0.3984371015625000, 0.4023433476562500, 0.4062495937500000, 0.4101558398437500, 0.4140620859375000, 0.4179683320312501, 0.4218745781250000, 0.4257808242187500, 0.4296870703125000, 0.4335933164062500, 0.4374995625000000, 0.4414058085937500, 0.4453120546875000, 0.4492183007812500, 0.4531245468750000, 0.4570307929687499, 0.4609370390624999, 0.4648432851562500, 0.4687495312500000, 0.4726557773437500, 0.4765620234375000, 0.4843745156250000, 0.4882807617187500, 0.4921870078125000, 0.4999995000000000, 0.5039057460937499, 0.5078119921874999, 0.5156244843750000, 0.5195307304687500, 0.5234369765625000, 0.5312494687500000, 0.5351557148437500, 0.5390619609375000, 0.5468744531250000, 0.5507806992187501, 0.5546869453125001, 0.5624994375000000, 0.5664056835937499, 0.5703119296874999, 0.5781244218749999, 0.5820306679687499, 0.5859369140624999, 0.5937494062499999, 0.5976556523437500, 0.6015618984375000, 0.6093743906250000, 0.6132806367187500, 0.6171868828125000, 0.6249993750000000, 0.6289056210937500, 0.6328118671875000, 0.6406243593750001, 0.6445306054687501, 0.6484368515625001, 0.6562493437500000, 0.6640618359374999, 0.6718743281249999, 0.6796868203124999, 0.6874993125000000, 0.6953118046875000, 0.7031242968750000, 0.7109367890625000, 0.7187492812500000, 0.7265617734375001, 0.7343742656250001, 0.7421867578125001, 0.7499992500000000, 0.7578117421874999, 0.7656242343749999, 0.7734367265625000, 0.7812492187500000, 0.7890617109375000, 0.7968742031250000, 0.8046866953125000, 0.8124991875000001, 0.8281241718750001, 0.8437491562500000, 0.8593741406249999, 0.8749991250000000, 0.8906241093750000, 0.9062490937500000, 0.9218740781249999, 0.9374990624999999, 0.9531240468750000, 0.9687490312500000, 0.9843740156250000, 0.9999990000000000},
            {-0.0677083333333333, -0.0669080398885679, -0.0661113466582041, -0.0653182867310031, -0.0645288932271553, -0.0637431992930979, -0.0629612380961919, -0.0621830428192554, -0.0614086466549526, -0.0606380828000343, -0.0598713844494290, -0.0591085847901805, -0.0583497169952325, -0.0575948142170540, -0.0568439095811064, -0.0560970361791489, -0.0553542270623786, -0.0546155152344047, -0.0538809336440539, -0.0531505151780042, -0.0524242926532453, -0.0517022988093634, -0.0509845663006474, -0.0502711276880156, -0.0495620154307594, -0.0488572618781026, -0.0481568992605739, -0.0474609596811907, -0.0467694751064522, -0.0460824773571397, -0.0453999980989218, -0.0447220688327638, -0.0440487208851387, -0.0433799853980373, -0.0427158933187780, -0.0420564753896126, -0.0414017621371277, -0.0407517838614402, -0.0401065706251856, -0.0394661522422980, -0.0388305582665802, -0.0381998179800646, -0.0375739603811621, -0.0369530141725996, -0.0363370077491460, -0.0357259691851245, -0.0351199262217138, -0.0345189062540355, -0.0339229363180302, -0.0333320430771206, -0.0327462528086642, -0.0321655913901953, -0.0315900842854570, -0.0310197565302262, -0.0304546327179312, -0.0298947369850646, -0.0293400929963941, -0.0287907239299728, -0.0282466524619518, -0.0277079007511983, -0.0271744904237230, -0.0266464425569191, -0.0261237776636186, -0.0256065156759691, -0.0250946759291369, -0.0245882771448402, -0.0240873374147199, -0.0235918741835521, -0.0231019042323112, -0.0226174436610879, -0.0221385078718716, -0.0216651115512051, -0.0211972686527185, -0.0207349923795538, -0.0202782951666880, -0.0198271886631655, -0.0193816837142514, -0.0189417903435158, -0.0185075177348627, -0.0180788742145153, -0.0176558672329715, -0.0172385033469440, -0.0168267882012996, -0.0164207265110137, -0.0160203220431560, -0.0156255775989252, -0.0152364949957503, -0.0148530750494774, -0.0144753175566616, -0.0141032212769855, -0.0137367839158238, -0.0133760021069785, -0.0130208713956060, -0.0126713862213622, -0.0123275399017886, -0.0119893246159667, -0.0116567313884682, -0.0113297500736274, -0.0110083693401655, -0.0106925766561973, -0.0103823582746505, -0.0100776992191290, -0.0097785832702546, -0.0094849929525195, -0.0091969095216846, -0.0089143129527606, -0.0086371819286065, -0.0083654938291857, -0.0080992247215155, -0.0078383493503520, -0.0075828411296487, -0.0073326721348309, -0.0070878130959278, -0.0068482333916044, -0.0066139010441368, -0.0063847827153747, -0.0061608437037358, -0.0059420479422764, -0.0057283579978844, -0.0055197350716394, -0.0053161390003871, -0.0051175282595720, -0.0049238599673767, -0.0045511724496034, -0.0043720607305222, -0.0041977064912006, -0.0038630709207312, -0.0037026865823962, -0.0035468537401799, -0.0032486226173337, -0.0031061113091740, -0.0029679254866718, -0.0027042912634598, -0.0025787205297505, -0.0024572306751953, -0.0022262372138749, -0.0021166029214064, -0.0020107882117372, -0.0018103460421619, -0.0017155807448533, -0.0016243594462793, -0.0014522649845454, -0.0013712482834459, -0.0012934886110731, -0.0011474473668695, -0.0010790182487408, -0.0010135511863271, -0.0008912048093551, -0.0008341758504503, -0.0007798097894315, -0.0006787665856433, -0.0006319397898483, -0.0005874767244697, -0.0005053450147240, -0.0004675289200443, -0.0004317817967271, -0.0003662052045256, -0.0003080434178493, -0.0002567368786252, -0.0002117404453101, -0.0001725254571836, -0.0001385816801595, -0.0001094191057354, -0.0000845695755001, -0.0000635882052528, -0.0000460545853083, -0.0000315737370093, -0.0000197768098218, -0.0000103215085916, -0.0000028922464714, 0.0000027999745098, 0.0000070179465074, 0.0000099988870967, 0.0000119552968722, 0.0000130760159885, 0.0000135274477195, 0.0000134549140211, 0.0000122225975083, 0.0000101763704848, 0.0000078724140361, 0.0000056725639486, 0.0000037869071180, 0.0000023110501723, 0.0000012578550199, 0.0000005840187013, 0.0000002121922798, 0.0000000495396625, 0.0000000039351887, 0.0000000000000000}
        };
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_4_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0117187382812500, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0273437226562500, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0429687070312500, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0585936914062500, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0742186757812500, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0898436601562500, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1054686445312500, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1210936289062500, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1367186132812500, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1523435976562500, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1679685820312500, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1835935664062500, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.1992185507812500, 0.2031247968750000, 0.2070310429687500, 0.2109372890625000, 0.2148435351562500, 0.2187497812500000, 0.2226560273437500, 0.2265622734375000, 0.2304685195312500, 0.2343747656250000, 0.2382810117187500, 0.2421872578125000, 0.2460935039062500, 0.2499997500000000, 0.2539059960937500, 0.2578122421875000, 0.2617184882812500, 0.2656247343750000, 0.2695309804687500, 0.2734372265625000, 0.2773434726562500, 0.2812497187500000, 0.2851559648437499, 0.2890622109375000, 0.2929684570312500, 0.2968747031250000, 0.3007809492187500, 0.3046871953125000, 0.3085934414062500, 0.3124996875000000, 0.3164059335937500, 0.3203121796875000, 0.3242184257812500, 0.3281246718750000, 0.3320309179687500, 0.3359371640625000, 0.3398434101562500, 0.3437496562500000, 0.3476559023437500, 0.3515621484375000, 0.3554683945312500, 0.3593746406250000, 0.3632808867187500, 0.3671871328125000, 0.3710933789062500, 0.3749996250000000, 0.3789058710937500, 0.3828121171875000, 0.3867183632812500, 0.3906246093750000, 0.3945308554687500, 0.3984371015625000, 0.4023433476562500, 0.4062495937500000, 0.4101558398437500, 0.4140620859375000, 0.4179683320312501, 0.4218745781250000, 0.4257808242187500, 0.4296870703125000, 0.4335933164062500, 0.4374995625000000, 0.4414058085937500, 0.4453120546875000, 0.4492183007812500, 0.4531245468750000, 0.4570307929687499, 0.4609370390624999, 0.4648432851562500, 0.4687495312500000, 0.4726557773437500, 0.4765620234375000, 0.4843745156250000, 0.4882807617187500, 0.4921870078125000, 0.4999995000000000, 0.5039057460937499, 0.5078119921874999, 0.5156244843750000, 0.5195307304687500, 0.5234369765625000, 0.5312494687500000, 0.5351557148437500, 0.5390619609375000, 0.5468744531250000, 0.5507806992187501, 0.5546869453125001, 0.5624994375000000, 0.5664056835937499, 0.5703119296874999, 0.5781244218749999, 0.5820306679687499, 0.5859369140624999, 0.5937494062499999, 0.5976556523437500, 0.6015618984375000, 0.6093743906250000, 0.6132806367187500, 0.6171868828125000, 0.6249993750000000, 0.6289056210937500, 0.6328118671875000, 0.6406243593750001, 0.6484368515625001, 0.6562493437500000, 0.6640618359374999, 0.6718743281249999, 0.6796868203124999, 0.6874993125000000, 0.6953118046875000, 0.7031242968750000, 0.7109367890625000, 0.7187492812500000, 0.7265617734375001, 0.7343742656250001, 0.7421867578125001, 0.7499992500000000, 0.7578117421874999, 0.7656242343749999, 0.7734367265625000, 0.7812492187500000, 0.7968742031250000, 0.8046866953125000, 0.8124991875000001, 0.8281241718750001, 0.8437491562500000, 0.8593741406249999, 0.8749991250000000, 0.8906241093750000, 0.9062490937500000, 0.9218740781249999, 0.9374990624999999, 0.9531240468750000, 0.9687490312500000, 0.9843740156250000, 0.9999990000000000},
            {-0.0513333333333333, -0.0507055078294963, -0.0500807279486127, -0.0494590196914553, -0.0488404090553522, -0.0482249220295445, -0.0476125845904280, -0.0470034226966763, -0.0463974622842444, -0.0457947292612504, -0.0451952495027330, -0.0445990488452853, -0.0440061530815602, -0.0434165879546478, -0.0428303791523221, -0.0422475523011565, -0.0416681329605037, -0.0410921466163425, -0.0405196186749852, -0.0399505744566482, -0.0393850391888815, -0.0388230379998559, -0.0382645959115079, -0.0377097378325394, -0.0371584885512703, -0.0366108727283449, -0.0360669148892880, -0.0355266394169115, -0.0349900705435687, -0.0344572323432569, -0.0339281487235646, -0.0334028434174652, -0.0328813399749542, -0.0323636617545291, -0.0318498319145125, -0.0313398734042161, -0.0308338089549461, -0.0303316610708490, -0.0298334520195969, -0.0293392038229130, -0.0288489382469364, -0.0283626767924259, -0.0278804406848038, -0.0274022508640387, -0.0269281279743688, -0.0264580923538647, -0.0259921640238343, -0.0255303626780686, -0.0250727076719311, -0.0246192180112904, -0.0241699123412988, -0.0237248089350179, -0.0232839256818918, -0.0228472800760728, -0.0224148892045982, -0.0219867697354243, -0.0215629379053170, -0.0211434095076049, -0.0207281998797961, -0.0203173238910629, -0.0199107959295995, -0.0195086298898542, -0.0191108391596440, -0.0187174366071527, -0.0183284345678213, -0.0179438448311338, -0.0175636786273050, -0.0171879466138770, -0.0168166588622312, -0.0164498248440214, -0.0160874534175387, -0.0157295528140123, -0.0153761306238579, -0.0150271937828812, -0.0146827485584459, -0.0143428005356171, -0.0140073546032887, -0.0136764149403085, -0.0133499850016100, -0.0130280675043646, -0.0127106644141661, -0.0123977769312616, -0.0120894054768419, -0.0117855496794059, -0.0114862083612152, -0.0111913795248539, -0.0109010603399095, -0.0106152471297940, -0.0103339353587203, -0.0100571196188556, -0.0097847936176682, -0.0095169501654889, -0.0092535811633088, -0.0089946775908321, -0.0087402294948093, -0.0084902259776712, -0.0082446551864883, -0.0080035043022807, -0.0077667595297024, -0.0075344060871262, -0.0073064281971568, -0.0070828090775982, -0.0068635309329040, -0.0066485749461400, -0.0064379212714869, -0.0062315490273158, -0.0060294362898648, -0.0058315600875497, -0.0056378963959409, -0.0054484201334376, -0.0052631051576748, -0.0050819242626951, -0.0049048491769192, -0.0047318505619518, -0.0045628980122545, -0.0043979600557239, -0.0042370041552087, -0.0040799967110005, -0.0039269030643361, -0.0037776875019452, -0.0036323132616783, -0.0034907425392518, -0.0033529364961440, -0.0030884579783118, -0.0029617027432112, -0.0028385466910661, -0.0026028557803306, -0.0024902303589125, -0.0023810230299396, -0.0021726714243864, -0.0020734293459044, -0.0019774098028011, -0.0017948336070221, -0.0017081726618696, -0.0016245257260342, -0.0014660574383377, -0.0013911262459333, -0.0013189894529875, -0.0011828730490785, -0.0011187791887464, -0.0010572513123664, -0.0009416604664850, -0.0008874801692755, -0.0008356312948575, -0.0007386906479090, -0.0006934799752298, -0.0006503630285676, -0.0005701722473187, -0.0005329795932193, -0.0004976431362633, -0.0004323033097541, -0.0004021829592739, -0.0003736849572818, -0.0003213266523992, -0.0002747751596518, -0.0002335872198037, -0.0001973312921594, -0.0001655892315844, -0.0001379578769906, -0.0001140505276710, -0.0000934982840348, -0.0000759512300857, -0.0000610794364878, -0.0000485737653214, -0.0000381464606854, -0.0000295315131491, -0.0000224847906436, -0.0000167839336240, -0.0000122280180717, -0.0000086369959297, -0.0000058509286191, -0.0000037290350460, -0.0000010036383005, -0.0000002037545133, 0.0000003274436236, 0.0000008273704830, 0.0000008789522863, 0.0000007271833720, 0.0000005167775013, 0.0000003232578495, 0.0000001773575211, 0.0000000832823803, 0.0000000317559943, 0.0000000088355218, 0.0000000014001484, 0.0000000000566561, 0.0000000000000000}
        };
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_5_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0117187382812500, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0273437226562500, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0429687070312500, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0585936914062500, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0742186757812500, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0898436601562500, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1054686445312500, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1210936289062500, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1367186132812500, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1523435976562500, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1679685820312500, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1835935664062500, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.1992185507812500, 0.2031247968750000, 0.2070310429687500, 0.2109372890625000, 0.2148435351562500, 0.2187497812500000, 0.2226560273437500, 0.2265622734375000, 0.2304685195312500, 0.2343747656250000, 0.2382810117187500, 0.2421872578125000, 0.2460935039062500, 0.2499997500000000, 0.2539059960937500, 0.2578122421875000, 0.2617184882812500, 0.2656247343750000, 0.2695309804687500, 0.2734372265625000, 0.2773434726562500, 0.2812497187500000, 0.2851559648437499, 0.2890622109375000, 0.2929684570312500, 0.2968747031250000, 0.3007809492187500, 0.3046871953125000, 0.3085934414062500, 0.3124996875000000, 0.3164059335937500, 0.3203121796875000, 0.3242184257812500, 0.3281246718750000, 0.3320309179687500, 0.3359371640625000, 0.3398434101562500, 0.3437496562500000, 0.3476559023437500, 0.3515621484375000, 0.3554683945312500, 0.3593746406250000, 0.3632808867187500, 0.3671871328125000, 0.3710933789062500, 0.3749996250000000, 0.3789058710937500, 0.3828121171875000, 0.3867183632812500, 0.3906246093750000, 0.3945308554687500, 0.3984371015625000, 0.4023433476562500, 0.4062495937500000, 0.4101558398437500, 0.4140620859375000, 0.4179683320312501, 0.4218745781250000, 0.4257808242187500, 0.4296870703125000, 0.4335933164062500, 0.4374995625000000, 0.4414058085937500, 0.4453120546875000, 0.4492183007812500, 0.4531245468750000, 0.4570307929687499, 0.4609370390624999, 0.4648432851562500, 0.4687495312500000, 0.4765620234375000, 0.4804682695312500, 0.4843745156250000, 0.4921870078125000, 0.4960932539062500, 0.4999995000000000, 0.5078119921874999, 0.5117182382812500, 0.5156244843750000, 0.5234369765625000, 0.5273432226562500, 0.5312494687500000, 0.5390619609375000, 0.5429682070312500, 0.5468744531250000, 0.5546869453125001, 0.5585931914062501, 0.5624994375000000, 0.5703119296874999, 0.5742181757812499, 0.5781244218749999, 0.5859369140624999, 0.5898431601562499, 0.5937494062499999, 0.6015618984375000, 0.6054681445312500, 0.6093743906250000, 0.6171868828125000, 0.6210931289062500, 0.6249993750000000, 0.6328118671875000, 0.6406243593750001, 0.6484368515625001, 0.6562493437500000, 0.6640618359374999, 0.6718743281249999, 0.6796868203124999, 0.6874993125000000, 0.6953118046875000, 0.7031242968750000, 0.7109367890625000, 0.7187492812500000, 0.7265617734375001, 0.7343742656250001, 0.7421867578125001, 0.7499992500000000, 0.7578117421874999, 0.7656242343749999, 0.7812492187500000, 0.7890617109375000, 0.7968742031250000, 0.8124991875000001, 0.8281241718750001, 0.8437491562500000, 0.8593741406249999, 0.8749991250000000, 0.8906241093750000, 0.9062490937500000, 0.9218740781249999, 0.9374990624999999, 0.9531240468750000, 0.9687490312500000, 0.9843740156250000, 0.9999990000000000},
            {-0.0402777777777778, -0.0397711478544953, -0.0392671237975230, -0.0387657265308463, -0.0382669769557638, -0.0377708959467606, -0.0372775043472840, -0.0367868229654209, -0.0362988725694747, -0.0358136738834416, -0.0353312475823831, -0.0348516142876957, -0.0343747945622745, -0.0339008089055709, -0.0334296777485418, -0.0329614214484907, -0.0324960602837976, -0.0320336144485383, -0.0315741040469904, -0.0311175490880266, -0.0306639694793919, -0.0302133850218658, -0.0297658154033077, -0.0293212801925836, -0.0288797988333748, -0.0284413906378667, -0.0280060747803161, -0.0275738702904990, -0.0271447960470339, -0.0267188707705847, -0.0262961130169383, -0.0258765411699590, -0.0254601734344191, -0.0250470278287030, -0.0246371221773882, -0.0242304741036990, -0.0238271010218359, -0.0234270201291793, -0.0230302483983669, -0.0226368025692468, -0.0222466991407046, -0.0218599543623667, -0.0214765842261785, -0.0210966044578597, -0.0207200305082368, -0.0203468775444535, -0.0199771604410605, -0.0196108937709859, -0.0192480917963871, -0.0188887684593868, -0.0185329373726934, -0.0181806118101091, -0.0178318046969269, -0.0174865286002193, -0.0171447957190211, -0.0168066178744093, -0.0164720064994822, -0.0161409726292420, -0.0158135268903845, -0.0154896794909978, -0.0151694402101767, -0.0148528183875544, -0.0145398229127584, -0.0142304622147937, -0.0139247442513593, -0.0136226764981038, -0.0133242659378249, -0.0130295190496196, -0.0127384417979923, -0.0124510396219258, -0.0121673174239243, -0.0118872795590356, -0.0116109298238592, -0.0113382714455508, -0.0110693070708307, -0.0108040387550057, -0.0105424679510145, -0.0102845954985064, -0.0100304216129643, -0.0097799458748829, -0.0095331672190140, -0.0092900839236908, -0.0090506936002438, -0.0088149931825216, -0.0085829789165303, -0.0083546463502055, -0.0081299903233320, -0.0079090049576262, -0.0076916836469971, -0.0074780190480029, -0.0072680030705191, -0.0070616268686367, -0.0068588808318087, -0.0066597545762624, -0.0064642369366984, -0.0062723159582956, -0.0060839788890417, -0.0058992121724128, -0.0057180014404202, -0.0055403315070504, -0.0053661863621180, -0.0051955491655567, -0.0050284022421712, -0.0048647270768751, -0.0047045043104387, -0.0045477137357738, -0.0043943342947785, -0.0042443440757711, -0.0040977203115377, -0.0039544393780215, -0.0038144767936799, -0.0036778072195384, -0.0035444044599683, -0.0034142414642151, -0.0032872903287078, -0.0031635223001753, -0.0030429077795986, -0.0029254163270267, -0.0028110166672848, -0.0026996766966004, -0.0025913634901768, -0.0023836816180781, -0.0022842430796238, -0.0021876915820617, -0.0020031014298994, -0.0019149867647007, -0.0018296071501518, -0.0016668931677606, -0.0015894771475650, -0.0015146329136574, -0.0013724894704825, -0.0013051036131277, -0.0012401162960173, -0.0011171580253344, -0.0010590962301518, -0.0010032513519884, -0.0008980259877704, -0.0008485514260813, -0.0008011056996546, -0.0007121094215547, -0.0006704626713238, -0.0006306524373907, -0.0005563476307464, -0.0005217559850797, -0.0004888067950766, -0.0004276419970189, -0.0003993298032501, -0.0003724669838868, -0.0003228986248966, -0.0003000984257750, -0.0002785583741358, -0.0002390737417488, -0.0002040797583458, -0.0001732204162719, -0.0001461500701773, -0.0001225347783560, -0.0001020535656511, -0.0000843995884289, -0.0000692811822700, -0.0000564227737045, -0.0000455656385676, -0.0000364684914525, -0.0000289078932977, -0.0000226784673855, -0.0000175929179154, -0.0000134818497741, -0.0000101933930471, -0.0000075926410239, -0.0000055609157358, -0.0000028055208557, -0.0000019170285768, -0.0000012656072639, -0.0000004714654620, -0.0000001062481128, 0.0000000318050493, 0.0000000628129988, 0.0000000525288517, 0.0000000329429128, 0.0000000167639303, 0.0000000068979595, 0.0000000021771236, 0.0000000004655570, 0.0000000000501290, 0.0000000000010323, 0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 1}, 6][HypergeometricPFQ][{1., 1.}, {2., 2.}, (-4*#1)/(1. - 1.*#1)]&
        // 2E-16 < total abs error < 3E-09
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_6_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4647972656250000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5819730468749998, 0.5858789062499998, 0.5936906249999999, 0.5975964843749999, 0.6015023437500000, 0.6093140625000000, 0.6132199218750000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7264898437500000, 0.7343015625000000, 0.7421132812500000, 0.7499250000000000, 0.7577367187499999, 0.7655484374999999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {-0.0325072886297376, -0.0320890525514971, -0.0316730693338442, -0.0312593561497609, -0.0308479301391100, -0.0304388084049708, -0.0300320080098936, -0.0296275459720706, -0.0292254392614226, -0.0288257047956005, -0.0284283594359001, -0.0280334199830904, -0.0276409031731534, -0.0272508256729342, -0.0268632040757021, -0.0264780548966193, -0.0260953945681196, -0.0257152394351923, -0.0253376057505743, -0.0249625096698467, -0.0245899672464371, -0.0242199944265253, -0.0238526070438533, -0.0234878208144375, -0.0231256513311832, -0.0227661140584014, -0.0224092243262259, -0.0220549973249315, -0.0217034480991529, -0.0213545915420026, -0.0210084423890893, -0.0206650152124357, -0.0203243244142951, -0.0199863842208674, -0.0196512086759152, -0.0193188116342777, -0.0189892067552856, -0.0186624074960745, -0.0183384271047997, -0.0180172786137503, -0.0176989748323652, -0.0173835283401510, -0.0170709514795020, -0.0167612563484238, -0.0164544547931618, -0.0161505584007350, -0.0158495784913769, -0.0155515261108847, -0.0152564120228790, -0.0149642467009742, -0.0146750403208639, -0.0143888027523209, -0.0141055435511159, -0.0138252719508561, -0.0135479968547473, -0.0132737268272813, -0.0130024700858532, -0.0127342344923093, -0.0124690275444320, -0.0122068563673633, -0.0119477277049714, -0.0116916479111649, -0.0114386229411591, -0.0111886583426979, -0.0109417592472379, -0.0106979303610989, -0.0104571759565865, -0.0102194998630927, -0.0099849054581810, -0.0097533956586616, -0.0095249729116640, -0.0092996391857145, -0.0090773959618249, -0.0088582442246014, -0.0086421844533811, -0.0084292166134051, -0.0082193401470362, -0.0080125539650323, -0.0078088564378825, -0.0076082453872191, -0.0074107180773133, -0.0072162712066669, -0.0070249008997115, -0.0068366026986259, -0.0066513715552851, -0.0064692018233526, -0.0062900872505300, -0.0061140209709765, -0.0059409954979129, -0.0057710027164253, -0.0056040338764812, -0.0054400795861760, -0.0052791298052245, -0.0051211738387132, -0.0049662003311319, -0.0048141972606998, -0.0046651519340056, -0.0045190509809787, -0.0043758803502105, -0.0042356253046447, -0.0040982704176563, -0.0039637995695383, -0.0038321959444181, -0.0037034420276218, -0.0035775196035094, -0.0034544097538015, -0.0033340928564182, -0.0032165485848535, -0.0031017559081066, -0.0029896930911914, -0.0028803376962487, -0.0027736665842821, -0.0026696559175401, -0.0025682811625683, -0.0024695170939532, -0.0023733377987806, -0.0022797166818303, -0.0021886264715305, -0.0021000392266926, -0.0020139263440475, -0.0019302585666065, -0.0018490059928642, -0.0017701380868658, -0.0016194310286318, -0.0015475277353041, -0.0014778808540055, -0.0013452216697881, -0.0012821406673114, -0.0012211787119093, -0.0011054688921612, -0.0010506483167568, -0.0009978014084681, -0.0008978784194357, -0.0008507262907166, -0.0008053957833592, -0.0007200438612249, -0.0006799438643321, -0.0006415083825019, -0.0005694713716997, -0.0005357896524798, -0.0005036121327591, -0.0004436082873440, -0.0004157012006240, -0.0003891368620299, -0.0003398754182354, -0.0003170981097837, -0.0002955032175387, -0.0002557024137743, -0.0002374180470232, -0.0002201592620359, -0.0001885653334582, -0.0001606187234309, -0.0001360252397669, -0.0001144995563432, -0.0000957663169062, -0.0000795611731506, -0.0000656317408918, -0.0000537384582231, -0.0000436553300481, -0.0000351705443500, -0.0000280869470583, -0.0000222223644362, -0.0000174097645396, -0.0000134972524865, -0.0000103478979760, -0.0000078393976173, -0.0000058635790496, -0.0000043257583782, -0.0000031439669116, -0.0000015787827343, -0.0000007310322855, -0.0000003039177211, -0.0000001073921908, -0.0000000275250444, -0.0000000009520070, 0.0000000045629851, 0.0000000036354568, 0.0000000018408826, 0.0000000006831773, 0.0000000001810454, 0.0000000000300224, 0.0000000000022164, 0.0000000000000237, 0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 1}, 7][HypergeometricPFQ][{1., 1.}, {2., 2.}, (-4*#1)/(1. - 1.*#1)]&
        // 8E-17 < total abs error < 3E-09
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_7_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4647972656250000, 0.4687031250000000, 0.4765148437500000, 0.4804207031250000, 0.4843265625000000, 0.4921382812500000, 0.4960441406250000, 0.4999500000000000, 0.5077617187500001, 0.5116675781250001, 0.5155734375000001, 0.5233851562500000, 0.5272910156250000, 0.5311968750000000, 0.5390085937500000, 0.5429144531250000, 0.5468203125000000, 0.5546320312499999, 0.5585378906249999, 0.5624437499999999, 0.5702554687499999, 0.5741613281249999, 0.5780671874999999, 0.5858789062499998, 0.5897847656249999, 0.5936906249999999, 0.6015023437500000, 0.6054082031250000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7264898437500000, 0.7343015625000000, 0.7421132812500000, 0.7499250000000000, 0.7655484374999999, 0.7733601562499999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {-0.0268415178571429, -0.0264897216385731, -0.0261398924669529, -0.0257920446820215, -0.0254461925850390, -0.0251023504355250, -0.0247605324479275, -0.0244207527882194, -0.0240830255704249, -0.0237473648530726, -0.0234137846355748, -0.0230822988545341, -0.0227529213799739, -0.0224256660114940, -0.0221005464743502, -0.0217775764154566, -0.0214567693993107, -0.0211381389038404, -0.0208216983161720, -0.0205074609283193, -0.0201954399327926, -0.0198856484181280, -0.0195780993643351, -0.0192728056382642, -0.0189697799888922, -0.0186690350425257, -0.0183705832979229, -0.0180744371213332, -0.0177806087414531, -0.0174891102443012, -0.0171999535680091, -0.0169131504975303, -0.0166287126592663, -0.0163466515156100, -0.0160669783594076, -0.0157897043083373, -0.0155148402992073, -0.0152423970821725, -0.0149723852148699, -0.0147048150564752, -0.0144396967616786, -0.0141770402745833, -0.0139168553225260, -0.0136591514098211, -0.0134039378114290, -0.0131512235665511, -0.0129010174721517, -0.0126533280764086, -0.0124081636720946, -0.0121655322898911, -0.0119254416916360, -0.0116878993635079, -0.0114529125091492, -0.0112204880427297, -0.0109906325819547, -0.0107633524410185, -0.0105386536235083, -0.0103165418152597, -0.0100970223771689, -0.0098801003379635, -0.0096657803869369, -0.0094540668666502, -0.0092449637656043, -0.0090384747108888, -0.0088346029608110, -0.0086333513975099, -0.0084347225195609, -0.0082387184345765, -0.0080453408518081, -0.0078545910747564, -0.0076664699937939, -0.0074809780788093, -0.0072981153718783, -0.0071178814799684, -0.0069402755676861, -0.0067652963500732, -0.0065929420854602, -0.0064232105683869, -0.0062560991225961, -0.0060916045941127, -0.0059297233444146, -0.0057704512437077, -0.0056137836643138, -0.0054597154741817, -0.0053082410305337, -0.0051593541736576, -0.0050130482208558, -0.0048693159605646, -0.0047281496466547, -0.0045895409929273, -0.0044534811678166, -0.0043199607893148, -0.0041889699201318, -0.0040604980631044, -0.0039345341568698, -0.0038110665718178, -0.0036900831063378, -0.0035715709833759, -0.0034555168473184, -0.0033419067612173, -0.0032307262043758, -0.0031219600703091, -0.0030155926651002, -0.0029116077061647, -0.0028099883214462, -0.0027107170490575, -0.0026137758373869, -0.0025191460456876, -0.0024268084451692, -0.0023367432206090, -0.0022489299725025, -0.0021633477197707, -0.0020799749030446, -0.0019987893885426, -0.0019197684725624, -0.0018428888866029, -0.0017681268031355, -0.0016954578420421, -0.0016248570777366, -0.0015562990469876, -0.0014897577574577, -0.0013626188435559, -0.0013019666761798, -0.0012432221863525, -0.0011313418428463, -0.0010781476498958, -0.0010267444846717, -0.0009291898577592, -0.0008829767203843, -0.0008384312947201, -0.0007542162805831, -0.0007144822457783, -0.0006762870720536, -0.0006043813699358, -0.0005706042968085, -0.0005382330437622, -0.0004775729070047, -0.0004492161549240, -0.0004221295404954, -0.0003716301608755, -0.0003481490713665, -0.0003258015300064, -0.0002843709003380, -0.0002652199760071, -0.0002470669902345, -0.0002136209822329, -0.0001982616090196, -0.0001837675361525, -0.0001572458079315, -0.0001338004640411, -0.0001131826803696, -0.0000951511136028, -0.0000794728324029, -0.0000659241951444, -0.0000542916606636, -0.0000443725184557, -0.0000359755250748, -0.0000289214342011, -0.0000230434089758, -0.0000181873068147, -0.0000142118289918, -0.0000109885298558, -0.0000084016835606, -0.0000063480096040, -0.0000047362621793, -0.0000034866922279, -0.0000018085592738, -0.0000012716388200, -0.0000008784669191, -0.0000003950899800, -0.0000001618396866, -0.0000000587541516, -0.0000000179107471, -0.0000000039599239, -0.0000000002071630, 0.0000000003514606, 0.0000000002178890, 0.0000000000767335, 0.0000000000173590, 0.0000000000022417, 0.0000000000001131, 0.0000000000000006, 0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 1}, 8][HypergeometricPFQ][{1., 1.}, {2., 2.}, (-4*#1)/(1. - 1.*#1)]&
        // 6E-18 < total abs error < 2E-09
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_2_d_0_0_0_1_8_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4647972656250000, 0.4687031250000000, 0.4765148437500000, 0.4804207031250000, 0.4843265625000000, 0.4921382812500000, 0.4960441406250000, 0.4999500000000000, 0.5077617187500001, 0.5116675781250001, 0.5155734375000001, 0.5233851562500000, 0.5272910156250000, 0.5311968750000000, 0.5390085937500000, 0.5429144531250000, 0.5468203125000000, 0.5546320312499999, 0.5585378906249999, 0.5624437499999999, 0.5702554687499999, 0.5741613281249999, 0.5780671874999999, 0.5858789062499998, 0.5897847656249999, 0.5936906249999999, 0.6015023437500000, 0.6054082031250000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7264898437500000, 0.7343015625000000, 0.7421132812500000, 0.7499250000000000, 0.7655484374999999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {-0.0225798549872624, -0.0222793460938868, -0.0219805698315775, -0.0216835383494822, -0.0213882637558992, -0.0210947581153639, -0.0208030334456737, -0.0205131017148522, -0.0202249748380520, -0.0199386646743949, -0.0196541830237489, -0.0193715416234424, -0.0190907521449138, -0.0188118261902964, -0.0185347752889392, -0.0182596108938605, -0.0179863443781370, -0.0177149870312251, -0.0174455500552163, -0.0171780445610243, -0.0169124815645054, -0.0166488719825101, -0.0163872266288669, -0.0161275562102971, -0.0158698713222617, -0.0156141824447381, -0.0153604999379293, -0.0151088340379026, -0.0148591948521603, -0.0146115923551404, -0.0143660363836488, -0.0141225366322230, -0.0138811026484260, -0.0136417438280732, -0.0134044694103901, -0.0131692884731029, -0.0129362099274622, -0.0127052425131993, -0.0124763947934175, -0.0122496751494173, -0.0120250917754577, -0.0118026526734537, -0.0115823656476117, -0.0113642382990030, -0.0111482780200771, -0.0109344919891166, -0.0107228871646338, -0.0105134702797126, -0.0103062478362946, -0.0101012260994140, -0.0098984110913813, -0.0096978085859187, -0.0094994241022491, -0.0093032628991416, -0.0091093299689150, -0.0089176300314027, -0.0087281675278819, -0.0085409466149701, -0.0083559711584907, -0.0081732447273135, -0.0079927705871716, -0.0078145516944594, -0.0076385906900152, -0.0074648898928927, -0.0072934512941260, -0.0071242765504923, -0.0069573669782767, -0.0067927235470455, -0.0066303468734316, -0.0064702372149384, -0.0063123944637678, -0.0061568181406780, -0.0060035073888781, -0.0058524609679647, -0.0057036772479082, -0.0055571542030964, -0.0054128894064407, -0.0052708800235548, -0.0051311228070123, -0.0049936140906921, -0.0048583497842191, -0.0047253253675112, -0.0045945358854385, -0.0044659759426069, -0.0043396396982745, -0.0042155208614109, -0.0040936126859097, -0.0039739079659656, -0.0038563990316256, -0.0037410777445275, -0.0036279354938349, -0.0035169631923836, -0.0034081512730483, -0.0033014896853445, -0.0031969678922781, -0.0030945748674539, -0.0029942990924595, -0.0028961285545351, -0.0028000507445453, -0.0027060526552661, -0.0026141207800015, -0.0025242411115449, -0.0024363991414990, -0.0023505798599705, -0.0022667677556536, -0.0021849468163191, -0.0021051005297218, -0.0020272118849452, -0.0019512633741961, -0.0018772369950660, -0.0018051142532753, -0.0017348761659140, -0.0016665032651961, -0.0015999756027424, -0.0015352727544062, -0.0014723738256572, -0.0014112574575378, -0.0013519018332070, -0.0012942846850837, -0.0012383833026045, -0.0011841745406080, -0.0010807401792116, -0.0010314662009613, -0.0009837881068296, -0.0008931185217614, -0.0008500755930190, -0.0008085256996062, -0.0007297984225105, -0.0006925669714236, -0.0006567204533144, -0.0005890710548127, -0.0005572119966400, -0.0005266255539164, -0.0004691559500149, -0.0004422151088091, -0.0004164315662928, -0.0003682197421466, -0.0003457329701859, -0.0003242865641795, -0.0002843976325281, -0.0002658965718224, -0.0002483188585002, -0.0002158172886768, -0.0002008356754184, -0.0001866619497782, -0.0001606247077128, -0.0001487050704829, -0.0001374811333422, -0.0001170113666271, -0.0000990009699924, -0.0000832417609867, -0.0000695326136595, -0.0000576802175644, -0.0000474997829635, -0.0000388156807014, -0.0000314620053813, -0.0000252830509495, -0.0000201336886214, -0.0000158796382975, -0.0000123976262294, -0.0000095754237348, -0.0000073117641828, -0.0000055161382704, -0.0000041084706978, -0.0000030186846414, -0.0000021861637873, -0.0000010939173655, -0.0000005104868929, -0.0000002199129934, -0.0000000862878305, -0.0000000302413624, -0.0000000091648365, -0.0000000022505192, -0.0000000003721001, -0.0000000000011188, 0.0000000000250232, 0.0000000000094649, 0.0000000000018844, 0.0000000000001914, 0.0000000000000066, 0.0000000000000000, 0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 2}, 0][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 4E-13 < total abs error < 1E-07
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_0_mapped = {
            {0.0000000000000000, 0.0019529296875000, 0.0039058593750000, 0.0078117187500000, 0.0097646484375000, 0.0117175781250000, 0.0156234375000000, 0.0175763671875000, 0.0195292968750000, 0.0234351562500000, 0.0253880859375000, 0.0273410156250000, 0.0312468750000000, 0.0331998046875000, 0.0351527343750000, 0.0390585937500000, 0.0410115234375000, 0.0429644531250000, 0.0468703125000000, 0.0488232421875000, 0.0507761718750000, 0.0546820312500000, 0.0566349609375000, 0.0585878906250000, 0.0624937500000000, 0.0644466796875000, 0.0663996093750000, 0.0703054687500000, 0.0722583984375000, 0.0742113281250000, 0.0781171875000000, 0.0800701171875000, 0.0820230468750000, 0.0859289062500000, 0.0878818359375000, 0.0898347656250000, 0.0937406250000000, 0.0956935546875000, 0.0976464843750000, 0.1015523437500000, 0.1035052734375000, 0.1054582031250000, 0.1093640625000000, 0.1113169921875000, 0.1132699218750000, 0.1171757812500000, 0.1191287109375000, 0.1210816406250000, 0.1249875000000000, 0.1269404296875000, 0.1288933593750000, 0.1327992187500000, 0.1347521484375000, 0.1367050781250000, 0.1406109375000000, 0.1425638671875000, 0.1445167968750000, 0.1484226562500000, 0.1503755859375000, 0.1523285156250000, 0.1562343750000000, 0.1581873046875000, 0.1601402343750000, 0.1640460937500000, 0.1659990234375000, 0.1679519531250000, 0.1718578125000000, 0.1738107421875000, 0.1757636718750000, 0.1796695312500000, 0.1816224609375000, 0.1835753906250000, 0.1874812500000000, 0.1894341796875000, 0.1913871093750000, 0.1952929687500000, 0.1972458984375000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4647972656250000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4804207031250000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4960441406250000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5116675781250001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5272910156250000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5429144531250000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5585378906249999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5819730468749998, 0.5858789062499998, 0.5936906249999999, 0.5975964843749999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6600902343749999, 0.6639960937500000, 0.6718078125000000, 0.6757136718750000, 0.6796195312500000, 0.6874312499999999, 0.6913371093749999, 0.6952429687499999, 0.7030546874999999, 0.7069605468749999, 0.7108664062499999, 0.7186781249999999, 0.7225839843749999, 0.7264898437500000, 0.7343015625000000, 0.7382074218750000, 0.7421132812500000, 0.7499250000000000, 0.7538308593749999, 0.7577367187499999, 0.7655484374999999, 0.7694542968749999, 0.7733601562499999, 0.7811718750000000, 0.7850777343749999, 0.7889835937500000, 0.7967953125000000, 0.8007011718750000, 0.8046070312500000, 0.8124187500000000, 0.8163246093750000, 0.8202304687499999, 0.8280421874999999, 0.8319480468749999, 0.8358539062499999, 0.8436656250000000, 0.8475714843750000, 0.8514773437500001, 0.8592890625000000, 0.8631949218750000, 0.8671007812500000, 0.8749125000000000, 0.8788183593750000, 0.8827242187500000, 0.8905359374999999, 0.8944417968749999, 0.8983476562499999, 0.9061593750000000, 0.9100652343750000, 0.9139710937500001, 0.9217828125000000, 0.9256886718750000, 0.9295945312500000, 0.9374062500000000, 0.9413121093750000, 0.9452179687500000, 0.9530296874999999, 0.9569355468749999, 0.9608414062499999, 0.9686531250000000, 0.9725589843750000, 0.9764648437500001, 0.9842765625000001, 0.9881824218750000, 0.9920882812500000, 0.9999000000000000},
            {0.0000000000000000, -0.0077913623399249, -0.0155419503266351, -0.0309204361594837, -0.0385481518104983, -0.0461347287357296, -0.0611841081297833, -0.0686467328937605, -0.0760678635404863, -0.0907852935826661, -0.0980814200551099, -0.1053357065830491, -0.1197184208805362, -0.1268466808091842, -0.1339327651308409, -0.1479780786154913, -0.1549371453286963, -0.1618537115551892, -0.1755590254275207, -0.1823476163323897, -0.1890933932890062, -0.2024562001014768, -0.2090730792515536, -0.2156468430627591, -0.2286647319416459, -0.2351087126737084, -0.2415092894171287, -0.2541799514218614, -0.2604498990611714, -0.2666761674898230, -0.2789974011081374, -0.2850922357412267, -0.2911431300730118, -0.3031128468497435, -0.3090315461635422, -0.3149060589367406, -0.3265222892335024, -0.3322638914193113, -0.3379610764124923, -0.3492219752948354, -0.3547855820153197, -0.3603045572297658, -0.3712084104777410, -0.3765931898947671, -0.3819331405971363, -0.3924783708344255, -0.3976835606951787, -0.4028437425177449, -0.4130289154537331, -0.4180538262321430, -0.4230335683275320, -0.4328573991058220, -0.4377014171958775, -0.4425001254427655, -0.4519614850887124, -0.4566240760457717, -0.4612412363015271, -0.4703391582603731, -0.4748198700860650, -0.4792550514817945, -0.4879887382379173, -0.4922872047036794, -0.4965400629765859, -0.5049088927432407, -0.5090248367469641, -0.5130951176043299, -0.5210986510720442, -0.5250318880216653, -0.5289194305301440, -0.5365574176606428, -0.5403078588778706, -0.5440125988711003, -0.5512849857222605, -0.5548526418589209, -0.5583746153557646, -0.5652815509216536, -0.5686665353803971, -0.5720058820053588, -0.5785477250538749, -0.5849072238007845, -0.5910845496898014, -0.5970799022964748, -0.6028935097476918, -0.6085256291385943, -0.6139765469465127, -0.6192465794415070, -0.6243360730930848, -0.6292454049726547, -0.6339749831512532, -0.6385252470920703, -0.6428966680372797, -0.6470897493886647, -0.6511050270815094, -0.6549430699512157, -0.6586044800920771, -0.6620898932076356, -0.6653999789520182, -0.6685354412616403, -0.6714970186766398, -0.6742854846513897, -0.6769016478534192, -0.6793463524500526, -0.6816204783820590, -0.6837249416235891, -0.6856606944276531, -0.6874287255563796, -0.6890300604952767, -0.6904657616506983, -0.6917369285297031, -0.6928446979014763, -0.6937902439394671, -0.6945747783433823, -0.6951995504401584, -0.6956658472630207, -0.6959749936077281, -0.6961283520650831, -0.6961273230287824, -0.6959733446776663, -0.6956678929314196, -0.6952124813787691, -0.6946086611772121, -0.6938580209233096, -0.6929621864925715, -0.6919228208479610, -0.6907416238160445, -0.6894203318298159, -0.6879607176372288, -0.6863645899744777, -0.6846337932030764, -0.6827702069097963, -0.6807757454685412, -0.6786523575632528, -0.6764020256709632, -0.6740267655041344, -0.6715286254114546, -0.6689096857362899, -0.6661720581320305, -0.6633178848336069, -0.6603493378844971, -0.6572686183185981, -0.6540779552963820, -0.6507796051948244, -0.6473758506506496, -0.6438689995565122, -0.6402613840098038, -0.6365553592138605, -0.6327533023314298, -0.6288576112903475, -0.6248707035414766, -0.6207950147690643, -0.6166329975537839, -0.6123871199888481, -0.6080598642497032, -0.6036537251179471, -0.5991712084602506, -0.5946148296632054, -0.5899871120251770, -0.5852905851063923, -0.5805277830386648, -0.5757012427963176, -0.5708135024300569, -0.5658670992657135, -0.5608645680699709, -0.5558084391853810, -0.5507012366371701, -0.5455454762145340, -0.5403436635293284, -0.5350982920552625, -0.5298118411509104, -0.5244867740700693, -0.5191255359631897, -0.5137305518738213, -0.5083042247342087, -0.4973670304792734, -0.4918608407085786, -0.4863326586342647, -0.4752193340532519, -0.4696386131588383, -0.4640447394670575, -0.4528259560787740, -0.4415794070037422, -0.4303208042830836, -0.4190651002091760, -0.4078264436961884, -0.3966181424065984, -0.3854526313033318, -0.3798895632073978, -0.3743414482349106, -0.3632952170763168, -0.3577995261535984, -0.3523236388404219, -0.3414354910438240, -0.3360251957513433, -0.3306386354579989, -0.3199400342042242, -0.3146295144142568, -0.3093457739627632, -0.2988610978688272, -0.2936612711545010, -0.2884904444671255, -0.2782374929020516, -0.2731561097122893, -0.2681052133418517, -0.2580959214832514, -0.2531379547094257, -0.2482113358689646, -0.2384526366843880, -0.2336207306252854, -0.2288205246898142, -0.2193152789957453, -0.2146102166466108, -0.2099368125021840, -0.2006847239936889, -0.1961058721102612, -0.1915583461004677, -0.1825567885863510, -0.1781024878700985, -0.1736789766873950, -0.1649236844781413, -0.1605915659511057, -0.1562895634648422, -0.1477751667787321, -0.1435623910750138, -0.1393789694568707, -0.1310993853510213, -0.1270028143662200, -0.1229347812030107, -0.1148834883815479, -0.1108998050499256, -0.1069438127075458, -0.0991140427190419, -0.0952398345405299, -0.0913924566553311, -0.0837773283798776, -0.0800091465534269, -0.0762669324147651, -0.0688595482244021, -0.0651939501372839, -0.0615534638601125, -0.0543469790575251, -0.0507805590091997, -0.0472384078633690, -0.0402260808561321, -0.0367554922152125, -0.0333083470152150, -0.0264835752851685, -0.0231055462907161, -0.0197501558749369, -0.0131065013264736, -0.0098178461186722, -0.0065510473824140, -0.0000822536785384}
        };
        // Derivative[{0, 0}, {0, 2}, 1][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 9E-12 < total abs error < 9E-07
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_1_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4647972656250000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6405609374999999, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7264898437500000, 0.7343015625000000, 0.7421132812500000, 0.7499250000000000, 0.7577367187499999, 0.7655484374999999, 0.7733601562499999, 0.7811718750000000, 0.7889835937500000, 0.7967953125000000, 0.8046070312500000, 0.8124187500000000, 0.8202304687499999, 0.8280421874999999, 0.8358539062499999, 0.8436656250000000, 0.8514773437500001, 0.8592890625000000, 0.8671007812500000, 0.8749125000000000, 0.8827242187500000, 0.8905359374999999, 0.8983476562499999, 0.9061593750000000, 0.9139710937500001, 0.9217828125000000, 0.9295945312500000, 0.9374062500000000, 0.9452179687500000, 0.9530296874999999, 0.9608414062499999, 0.9686531250000000, 0.9764648437500001, 0.9842765625000001, 0.9920882812500000, 0.9999000000000000},
            {1.0000000000000000, 0.9818456592164198, 0.9638377795702662, 0.9459770379760093, 0.9282641032839433, 0.9106996360173950, 0.8932842881060176, 0.8760187026151737, 0.8589035134713973, 0.8419393451839461, 0.8251268125624488, 0.8084665204306470, 0.7919590633362560, 0.7756050252569501, 0.7594049793024904, 0.7433594874130224, 0.7274691000535610, 0.7117343559046985, 0.6961557815495601, 0.6807338911570524, 0.6654691861614399, 0.6503621549382950, 0.6354132724768756, 0.6206230000489824, 0.6059917848743527, 0.5915200597826634, 0.5772082428722072, 0.5630567371653200, 0.5490659302606420, 0.5352361939823006, 0.5215678840261084, 0.5080613396028771, 0.4947168830789558, 0.4815348196141068, 0.4685154367968427, 0.4556590042773500, 0.4429657733981399, 0.4304359768225697, 0.4180698281613863, 0.4058675215974565, 0.3938292315088502, 0.3819551120904590, 0.3702452969743378, 0.3586998988489669, 0.3473190090776430, 0.3361026973162177, 0.3250510111304107, 0.3141639756129396, 0.3034415930007122, 0.2928838422923488, 0.2824906788663012, 0.2722620340998607, 0.2621978149893440, 0.2522979037717747, 0.2425621575483784, 0.2329904079102269, 0.2235824605663838, 0.2143380949749085, 0.2052570639771009, 0.1963390934353734, 0.1875838818751558, 0.1789911001312534, 0.1705603909990924, 0.1622913688913022, 0.1541836195000982, 0.1462366994659480, 0.1384501360530154, 0.1308234268318922, 0.1233560393701518, 0.1160474109312612, 0.1088969481824175, 0.1019040269118828, 0.0950679917564110, 0.0883881559393769, 0.0818638010202314, 0.0754941766559270, 0.0692785003749721, 0.0632159573647844, 0.0573057002730408, 0.0515468490237235, 0.0459384906485865, 0.0404796791347813, 0.0351694352893879, 0.0300067466216219, 0.0249905672434947, 0.0201198177897195, 0.0153933853576686, 0.0108101234681998, 0.0063688520481786, 0.0020683574355356, -0.0020926075922947, -0.0061153237656967, -0.0100011052450989, -0.0137512995172295, -0.0173672872645341, -0.0208504822068792, -0.0242023309146666, -0.0274243125924821, -0.0305179388324091, -0.0334847533361367, -0.0363263316050042, -0.0390442805971272, -0.0416402383507694, -0.0441158735731304, -0.0464728851937442, -0.0487130018816971, -0.0508379815259012, -0.0528496106776839, -0.0547497039549839, -0.0565401034074772, -0.0582226778419973, -0.0597993221076485, -0.0612719563400641, -0.0626425251643059, -0.0639129968559581, -0.0650853624600272, -0.0661616348673244, -0.0671438478480731, -0.0680340550425639, -0.0688343289087528, -0.0695467596267881, -0.0701734539605404, -0.0707165340763066, -0.0715604099459452, -0.0718655158195614, -0.0720956250582418, -0.0723395810113394, -0.0723578085455306, -0.0723097981132370, -0.0720238678640015, -0.0717903520787843, -0.0714994031445099, -0.0707539863101745, -0.0703038937926838, -0.0698051154306545, -0.0686701489614639, -0.0680382523216958, -0.0673662491801671, -0.0659103247679202, -0.0643189729656850, -0.0626084217825899, -0.0607944727804815, -0.0588924450023936, -0.0569171206519088, -0.0548826931634825, -0.0528027183288347, -0.0506900691594366, -0.0464145867382351, -0.0421441594087372, -0.0400347887314231, -0.0379537539311595, -0.0359083352244436, -0.0339049783427218, -0.0319493084324288, -0.0300461517000043, -0.0281995644956805, -0.0264128693398702, -0.0246886972058331, -0.0230290351906553, -0.0214352785432937, -0.0199082858839847, -0.0184484363541367, -0.0170556873893271, -0.0157296318175905, -0.0144695530549052, -0.0132744772994658, -0.0121432218105412, -0.0110744385855493, -0.0100666530043021, -0.0091182972721460, -0.0082277387420757, -0.0073933034090276, -0.0066132950308781, -0.0058860104306264, -0.0052097515720043, -0.0045828349841786, -0.0040035990548251, -0.0034704096322887, -0.0029816642934329, -0.0025357955564537, -0.0021312732539485, -0.0017666062322204, -0.0014403435062370, -0.0011510749727290, -0.0008974317637712, -0.0006780863077471, -0.0004917521525009, -0.0003371835958452, -0.0002131751608164, -0.0001185609467416, -0.0000522138819885, -0.0000130448999821, -0.0000000020565164}
        };
        // Derivative[{0, 0}, {0, 2}, 2][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 8E-12 < total abs error < 5E-07
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_2_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6718078125000000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7264898437500000, 0.7343015625000000, 0.7421132812500000, 0.7499250000000000, 0.7577367187499999, 0.7655484374999999, 0.7733601562499999, 0.7811718750000000, 0.7889835937500000, 0.7967953125000000, 0.8046070312500000, 0.8124187500000000, 0.8202304687499999, 0.8280421874999999, 0.8436656250000000, 0.8514773437500001, 0.8592890625000000, 0.8749125000000000, 0.8827242187500000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {1.1666666666666665, 1.1482910915864102, 1.1300534760039718, 1.1119544711508360, 1.0939947208068426, 1.0761748610551389, 1.0584955200335213, 1.0409573176821565, 1.0235608654876946, 1.0063067662237730, 0.9891956136879196, 0.9722279924348650, 0.9554044775062779, 0.9387256341569387, 0.9221920175773698, 0.9058041726129470, 0.8895626334795158, 0.8734679234755435, 0.8575205546908395, 0.8417210277118814, 0.8260698313237885, 0.8105674422089864, 0.7952143246426143, 0.7800109301847301, 0.7649576973693692, 0.7500550513905242, 0.7353034037851138, 0.7207031521130155, 0.7062546796342416, 0.6919583549833456, 0.6778145318411495, 0.6638235486038910, 0.6499857280498939, 0.6363013770038725, 0.6227707859989906, 0.6093942289367948, 0.5961719627451579, 0.5831042270343705, 0.5701912437515289, 0.5574332168333729, 0.5448303318577393, 0.5323827556938002, 0.5200906361512698, 0.5079541016287672, 0.4959732607615356, 0.4841482020687250, 0.4724789936004586, 0.4609656825849096, 0.4496082950756274, 0.4384068355993634, 0.4273612868046555, 0.4164716091114428, 0.4057377403619929, 0.3951595954734378, 0.3847370660922201, 0.3744700202507747, 0.3643583020267695, 0.3544017312052525, 0.3446001029440626, 0.3349531874428696, 0.3254607296162289, 0.3161224487710476, 0.3069380382888689, 0.2979071653134042, 0.2890294704437446, 0.2803045674337107, 0.2717320428978032, 0.2633114560242373, 0.2550423382955592, 0.2469241932173520, 0.2389564960555615, 0.2311386935829788, 0.2234702038354381, 0.2159504158783006, 0.2085786895838077, 0.2013543554199083, 0.1942767142511710, 0.1873450371524138, 0.1805585652356946, 0.1739165094913203, 0.1674180506435466, 0.1610623390216551, 0.1548484944471033, 0.1487756061374624, 0.1428427326278615, 0.1370489017106781, 0.1313931103942137, 0.1258743248811162, 0.1204914805673101, 0.1152434820622071, 0.1101292032309812, 0.1051474872596899, 0.1002971467440375, 0.0955769638025742, 0.0909856902151287, 0.0865220475872748, 0.0821847275416290, 0.0779723919367771, 0.0738836731146195, 0.0699171741769231, 0.0660714692918553, 0.0623451040312695, 0.0587365957394941, 0.0552444339343647, 0.0518670807412208, 0.0486029713605650, 0.0454505145700623, 0.0424080932615269, 0.0394740650135155, 0.0366467627001113, 0.0339244951364458, 0.0313055477614643, 0.0287881833583950, 0.0263706428133327, 0.0240511459122957, 0.0218278921770527, 0.0196990617399588, 0.0176628162579675, 0.0157172998659153, 0.0120909492750854, 0.0104063248645859, 0.0088048513011808, 0.0058436345115656, 0.0044800039503309, 0.0031917520504129, 0.0008335213753083, -0.0002404021392744, -0.0012468337070686, -0.0030651388328953, -0.0038809703569646, -0.0046372228625948, -0.0059788698511495, -0.0065681875419611, -0.0071057694468691, -0.0080334660001880, -0.0084274182680910, -0.0087773067076363, -0.0093523891606292, -0.0095812828316492, -0.0097735087133685, -0.0100551061280254, -0.0102112159973675, -0.0102554164837966, -0.0102007809479151, -0.0100598320675637, -0.0098444990679625, -0.0095660786931985, -0.0092352005468082, -0.0088617974125773, -0.0084550811339691, -0.0080235245800329, -0.0071160252252214, -0.0061920409488730, -0.0057371009241234, -0.0052924904556651, -0.0048615852168407, -0.0044471279029286, -0.0040512708769145, -0.0036756234006366, -0.0033213024182760, -0.0029889857420997, -0.0026789664133251, -0.0023912069826517, -0.0021253924818917, -0.0018809809433856, -0.0016572504663462, -0.0014533420226304, -0.0012682974270580, -0.0011010921527262, -0.0009506629296692, -0.0008159303041847, -0.0005892593570139, -0.0004952221947651, -0.0004127014917150, -0.0002783885542702, -0.0002247908342906, -0.0001791013360718, -0.0001083182066183, -0.0000602131015610, -0.0000296261960842, -0.0000120204315448, -0.0000034315264080, -0.0000004160121197, -0.0000000000001028}
        };
        // Derivative[{0, 0}, {0, 2}, 3][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 7E-12 < total abs error < 2E-07
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_3_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.7030546874999999, 0.7186781249999999, 0.7343015625000000, 0.7421132812500000, 0.7499250000000000, 0.7655484374999999, 0.7733601562499999, 0.7811718750000000, 0.7967953125000000, 0.8046070312500000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {1.1805555555555556, 1.1626107848811367, 1.1447975958322465, 1.1271166130347312, 1.1095684540668866, 1.0921537292275423, 1.0748730413007506, 1.0577269853170828, 1.0407161483115352, 1.0238411090780548, 1.0071024379206874, 0.9905006964013632, 0.9740364370843306, 0.9577102032772581, 0.9415225287690207, 0.9254739375641967, 0.9095649436142973, 0.8937960505457628, 0.8781677513847550, 0.8626805282787854, 0.8473348522152176, 0.8321311827366913, 0.8170699676535133, 0.8021516427530732, 0.7873766315063395, 0.7727453447714970, 0.7582581804947983, 0.7439155234086943, 0.7297177447273306, 0.7156652018394862, 0.7017582379990450, 0.6879971820130988, 0.6743823479277761, 0.6609140347119113, 0.6475925259386612, 0.6344180894651934, 0.6213909771105721, 0.6085114243319761, 0.5957796498993906, 0.5831958555689234, 0.5707602257548984, 0.5584729272008967, 0.5463341086499163, 0.5343439005138316, 0.5225024145423456, 0.5108097434916309, 0.4992659607928737, 0.4878711202209340, 0.4766252555633546, 0.4655283802899524, 0.4545804872232457, 0.4437815482099730, 0.4331315137939719, 0.4226303128907037, 0.4122778524637096, 0.4020740172033058, 0.3920186692078312, 0.3821116476677725, 0.3723527685531100, 0.3627418243042305, 0.3532785835267744, 0.3439627906907919, 0.3347941658345976, 0.3257724042737269, 0.3168971763154085, 0.3081681269789830, 0.2995848757227104, 0.2911470161774196, 0.2828541158874747, 0.2747057160595374, 0.2667013313196253, 0.2588404494789741, 0.2511225313092347, 0.2435470103275368, 0.2361132925919772, 0.2288207565080954, 0.2216687526469194, 0.2146566035751709, 0.2077836036982398, 0.2010490191165432, 0.1944520874959046, 0.1879920179525933, 0.1816679909536816, 0.1754791582333853, 0.1694246427260665, 0.1635035385165854, 0.1577149108086977, 0.1520577959122060, 0.1465312012495770, 0.1411341053827460, 0.1358654580608381, 0.1307241802895351, 0.1257091644228272, 0.1208192742778876, 0.1160533452738098, 0.1114101845949484, 0.1068885713796025, 0.1024872569347757, 0.0982049649777447, 0.0940403919051574, 0.0899922070903769, 0.0860590532097730, 0.0822395465986506, 0.0785322776374891, 0.0749358111691474, 0.0714486869476685, 0.0680694201192926, 0.0647965017362625, 0.0616283993039725, 0.0585635573619803, 0.0556003980993622, 0.0527373220048554, 0.0499727085521803, 0.0473049169208944, 0.0447322867530717, 0.0422531389460471, 0.0398657764814046, 0.0375684852903216, 0.0353595351553121, 0.0311996621051664, 0.0292452066358089, 0.0273720291707019, 0.0238623136015925, 0.0222221543688929, 0.0206560332183344, 0.0177385837647397, 0.0163835830928794, 0.0150952783551614, 0.0127113888888090, 0.0116121219353831, 0.0105721892702259, 0.0086630002393876, 0.0077900965791380, 0.0069692355771721, 0.0054764494125143, 0.0048009591336908, 0.0041703843168679, 0.0030370198302505, 0.0025307956743691, 0.0020626210219138, 0.0012337866484542, 0.0005374964340542, -0.0000388430372332, -0.0005073543572509, -0.0008796478212995, -0.0011667830267969, -0.0013792341279867, -0.0015268593254713, -0.0016188751494774, -0.0016638360604070, -0.0016696198378039, -0.0016434191581526, -0.0015917396721683, -0.0015204047833870, -0.0014345672027038, -0.0012367573959248, -0.0010269707389536, -0.0008249379746282, -0.0007308849437849, -0.0006428247767010, -0.0004867777083427, -0.0004191955505033, -0.0003585091973409, -0.0002567773115946, -0.0002149813745218, -0.0001786403392135, -0.0001204137667432, -0.0000783165229210, -0.0000488465689554, -0.0000289570004783, -0.0000161072425775, -0.0000082475136959, -0.0000037728906837, -0.0000014663863355, -0.0000004406421399, -0.0000000828493255, -0.0000000049717343, -0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 2}, 4][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 4E-12 < total abs error < 2E-07
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_4_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7186781249999999, 0.7343015625000000, 0.7499250000000000, 0.7655484374999999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {1.1527777777777775, 1.1354537642850187, 1.1182558466150514, 1.1011846220127950, 1.0842406809665754, 1.0674246069868638, 1.0507369763818044, 1.0341783580295318, 1.0177493131472870, 1.0014503950573326, 0.9852821489496800, 0.9692451116416416, 0.9533398113342167, 0.9375667673653341, 0.9219264899599683, 0.9064194799771531, 0.8910462286539186, 0.8758072173461824, 0.8607029172666252, 0.8457337892195912, 0.8309002833330503, 0.8162028387876683, 0.8016418835430320, 0.7872178340610837, 0.7729310950268193, 0.7587820590663131, 0.7447711064621346, 0.7308986048662294, 0.7171649090103376, 0.7035703604140336, 0.6901152870904725, 0.6768000032499345, 0.6636248090012677, 0.6505899900513306, 0.6376958174025458, 0.6249425470486804, 0.6123304196689792, 0.5998596603207758, 0.5875304781307217, 0.5753430659847779, 0.5632976002171186, 0.5513942402981078, 0.5396331285215140, 0.5280143896911433, 0.5165381308070658, 0.5052044407516371, 0.4940133899755094, 0.4829650301838447, 0.4720593940229489, 0.4612964947675561, 0.4506763260090008, 0.4401988613445279, 0.4298640540679964, 0.4196718368622501, 0.4096221214934302, 0.3997147985075244, 0.3899497369294513, 0.3803267839649944, 0.3708457647059094, 0.3615064818385406, 0.3523087153562948, 0.3432522222763318, 0.3343367363608433, 0.3255619678433058, 0.3169276031601016, 0.3084333046879210, 0.3000787104873636, 0.2918634340531759, 0.2837870640715734, 0.2758491641851057, 0.2680492727655395, 0.2603869026952459, 0.2528615411575909, 0.2454726494368427, 0.2382196627281183, 0.2311019899579109, 0.2241190136157452, 0.2172700895975248, 0.2105545470611478, 0.2039716882949741, 0.1975207885997464, 0.1912010961845722, 0.1850118320775879, 0.1789521900519355, 0.1730213365676930, 0.1672184107304069, 0.1615425242668869, 0.1559927615189267, 0.1505681794556268, 0.1452678077049953, 0.1400906486055154, 0.1350356772783634, 0.1301018417209746, 0.1252880629226454, 0.1205932350028712, 0.1160162253731076, 0.1115558749226525, 0.1072109982293311, 0.1029803837956682, 0.0988627943112200, 0.0948569669417311, 0.0909616136457695, 0.0871754215194767, 0.0834970531700590, 0.0799251471186222, 0.0764583182329327, 0.0730951581906657, 0.0698342359736716, 0.0666740983937633, 0.0636132706504941, 0.0606502569213594, 0.0577835409848152, 0.0550115868764633, 0.0523328395787065, 0.0497457257441255, 0.0472486544527741, 0.0448400180035307, 0.0425181927395809, 0.0402815399080373, 0.0360571264463358, 0.0340660210426925, 0.0321534004805381, 0.0285568040373458, 0.0268694012498535, 0.0252536317063946, 0.0222300660753851, 0.0208187963653128, 0.0194722151046765, 0.0169661513146800, 0.0158031876044569, 0.0146979526277388, 0.0126537446724035, 0.0117113252161665, 0.0108197443223526, 0.0091823042050373, 0.0084330776589691, 0.0077279578769128, 0.0064434653234502, 0.0058608498132502, 0.0053158584037537, 0.0043324858224090, 0.0034810571762182, 0.0027496855919415, 0.0021269284444247, 0.0016018256240665, 0.0011639348101124, 0.0008033632034804, 0.0005107951815083, 0.0002775153589192, 0.0000954265760644, -0.0000429376114504, -0.0001444062972500, -0.0002151707099575, -0.0002607899812594, -0.0002862032342253, -0.0002957478918282, -0.0002931839557445, -0.0002640672503277, -0.0002186377083730, -0.0001698017889929, -0.0001250312868351, -0.0000878231163331, -0.0000590408836227, -0.0000380297039537, -0.0000234433085797, -0.0000137797464948, -0.0000076715564742, -0.0000040030843990, -0.0000019273416037, -0.0000008361640127, -0.0000003149445326, -0.0000000967283428, -0.0000000215321325, -0.0000000026667837, -0.0000000000792208, -0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 2}, 5][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 7E-14 < total abs error < 2E-07
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_5_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7186781249999999, 0.7343015625000000, 0.7499250000000000, 0.7655484374999999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {1.1128703703703704, 1.0962050070088420, 1.0796608161038608, 1.0632383682187057, 1.0469382273908456, 1.0307609509198505, 1.0147070891522467, 0.9987771852633263, 0.9829717750359110, 0.9672913866360812, 0.9517365403858776, 0.9363077485329899, 0.9210055150174442, 0.9058303352353102, 0.8907826957994469, 0.8758630742973091, 0.8610719390458429, 0.8464097488434980, 0.8318769527193921, 0.8174739896796617, 0.8032012884510383, 0.7890592672216979, 0.7750483333794264, 0.7611688832471564, 0.7474213018159285, 0.7338059624753390, 0.7203232267415388, 0.7069734439828504, 0.6937569511430809, 0.6806740724626075, 0.6677251191973208, 0.6549103893355159, 0.6422301673128270, 0.6296847237253048, 0.6172743150407471, 0.6049991833083922, 0.5928595558670973, 0.5808556450521259, 0.5689876479006796, 0.5572557458563107, 0.5456601044723656, 0.5342008731146110, 0.5228781846632052, 0.5116921552141862, 0.5006428837806495, 0.4897304519938058, 0.4789549238041110, 0.4683163451826713, 0.4578147438231341, 0.4474501288442876, 0.4372224904935971, 0.4271317998519191, 0.4171780085396407, 0.4073610484245046, 0.3976808313313889, 0.3881372487543215, 0.3787301715710175, 0.3694594497602424, 0.3603249121223126, 0.3513263660030546, 0.3424635970215581, 0.3337363688020686, 0.3251444227103753, 0.3166874775950650, 0.3083652295340197, 0.3001773515865536, 0.2921234935515916, 0.2842032817323064, 0.2764163187076440, 0.2687621831111768, 0.2612404294177404, 0.2538505877383153, 0.2465921636236350, 0.2394646378770087, 0.2324674663768597, 0.2256000799094944, 0.2188618840126273, 0.2122522588301980, 0.2057705589790301, 0.1994161134278900, 0.1931882253895175, 0.1870861722262074, 0.1811092053695343, 0.1752565502548186, 0.1695274062709459, 0.1639209467261545, 0.1584363188304184, 0.1530726436950576, 0.1478290163502148, 0.1427045057808413, 0.1376981549818445, 0.1328089810330443, 0.1280359751945972, 0.1233781030235425, 0.1188343045121263, 0.1144034942485577, 0.1100845616008500, 0.1058763709243933, 0.1017777617938996, 0.0977875492603543, 0.0939045241335963, 0.0901274532911410, 0.0864550800138405, 0.0828861243489658, 0.0794192835012720, 0.0760532322525899, 0.0727866234104612, 0.0696180882863102, 0.0665462372036130, 0.0635696600364966, 0.0606869267791595, 0.0578965881464722, 0.0551971762060689, 0.0525872050421983, 0.0500651714515534, 0.0476295556712402, 0.0452788221389980, 0.0430114202857131, 0.0408257853602079, 0.0366934915513778, 0.0347436401280227, 0.0328691724253979, 0.0293398909341338, 0.0276818081501836, 0.0260925732130923, 0.0231140424373544, 0.0217214349858907, 0.0203910544918356, 0.0179103353308615, 0.0167566797641045, 0.0156586198718294, 0.0136226922355549, 0.0126815425035622, 0.0117894271120147, 0.0101458319949763, 0.0093911473731026, 0.0086790899928953, 0.0073766028870599, 0.0067830883463576, 0.0062260340719892, 0.0052153509784052, 0.0043328665467141, 0.0035672792900157, 0.0029077114615163, 0.0023437448233089, 0.0018654535033563, 0.0014634334268190, 0.0011288278182772, 0.0008533482945468, 0.0006292911051932, 0.0004495481307978, 0.0003076123184648, 0.0001975773204061, 0.0001141312045094, 0.0000525442245266, 0.0000086507699232, -0.0000211742416347, -0.0000505937240527, -0.0000551057367578, -0.0000477030487627, -0.0000363359783271, -0.0000252708759017, -0.0000163251820497, -0.0000098803730099, -0.0000056209545480, -0.0000030024390233, -0.0000014970897463, -0.0000006890944381, -0.0000002875699083, -0.0000001058054803, -0.0000000328328787, -0.0000000079716697, -0.0000000013148901, -0.0000000001072887, -0.0000000000015779, -0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 2}, 6][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 2E-15 < total abs error < 9E-08
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_6_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7343015625000000, 0.7499250000000000, 0.7655484374999999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {1.0705555555555555, 1.0545319161768467, 1.0386249652820581, 1.0228352482580627, 1.0071633041623560, 0.9916096655191100, 0.9761748581123197, 0.9608594007760408, 0.9456638051817311, 0.9305885756226956, 0.9156342087956539, 0.9008011935794367, 0.8860900108108291, 0.8715011330575781, 0.8570350243885868, 0.8426921401413150, 0.8284729266864147, 0.8143778211896331, 0.8004072513710089, 0.7865616352614044, 0.7728413809564112, 0.7592468863676696, 0.7457785389716535, 0.7324367155559682, 0.7192217819632188, 0.7061340928325041, 0.6931739913386044, 0.6803418089289290, 0.6676378650582956, 0.6550624669216186, 0.6426159091845925, 0.6302984737124540, 0.6181104292969170, 0.6060520313813808, 0.5941235217845154, 0.5823251284223311, 0.5706570650288535, 0.5591195308755232, 0.5477127104894502, 0.5364367733706600, 0.5252918737084711, 0.5142781500971574, 0.5033957252510495, 0.4926447057192407, 0.4820251816000699, 0.4715372262555599, 0.4611808960260020, 0.4509562299448803, 0.4408632494543420, 0.4309019581214283, 0.4210723413552849, 0.4113743661255888, 0.4018079806824263, 0.3923731142778787, 0.3830696768895696, 0.3738975589464493, 0.3648566310570934, 0.3559467437408035, 0.3471677271618182, 0.3385193908669356, 0.3300015235268790, 0.3216138926817307, 0.3133562444907834, 0.3052283034871602, 0.2972297723375736, 0.2893603316075961, 0.2816196395328373, 0.2740073317964212, 0.2665230213131840, 0.2591662980210083, 0.2519367286797353, 0.2448338566780969, 0.2378572018491307, 0.2310062602945446, 0.2242805042185136, 0.2176793817714018, 0.2112023169039124, 0.2048487092321810, 0.1986179339143379, 0.1925093415390729, 0.1865222580267514, 0.1806559845436359, 0.1749097974297764, 0.1692829481411462, 0.1637746632066022, 0.1583841442002620, 0.1531105677298926, 0.1479530854419152, 0.1429108240436352, 0.1379828853433102, 0.1331683463086754, 0.1284662591445449, 0.1238756513901140, 0.1193955260365838, 0.1150248616657320, 0.1107626126100507, 0.1066077091350685, 0.1025590576444706, 0.0986155409086234, 0.0947760183171012, 0.0910393261558039, 0.0874042779092440, 0.0838696645885620, 0.0804342550858200, 0.0770967965550998, 0.0738560148209134, 0.0707106148144093, 0.0676592810378321, 0.0647006780576650, 0.0618334510268518, 0.0590562262364606, 0.0563676116971134, 0.0537661977504650, 0.0512505577109689, 0.0488192485381209, 0.0464708115393176, 0.0442037731034132, 0.0420166454649974, 0.0399079274993535, 0.0359196542745590, 0.0340370375509303, 0.0322267093730424, 0.0288166909589776, 0.0272138679860488, 0.0256770701177250, 0.0227952233617299, 0.0214470029529077, 0.0201584668450780, 0.0177540921635463, 0.0166350790558378, 0.0155694035880346, 0.0135917565484997, 0.0126766458734972, 0.0118085971531794, 0.0102075024943578, 0.0094713931360172, 0.0087762214576408, 0.0075027157172486, 0.0069214348482204, 0.0063752005659028, 0.0053821851953207, 0.0045125120131101, 0.0037553945551093, 0.0031004535301971, 0.0025377505186976, 0.0020578188372135, 0.0016516910836017, 0.0013109228874324, 0.0010276124151951, 0.0007944152171196, 0.0006045540549702, 0.0004518234183454, 0.0003305885212296, 0.0002357786704752, 0.0001628750124538, 0.0001078927912381, 0.0000673583882515, 0.0000382815550521, 0.0000181233923387, -0.0000035520544084, -0.0000103661233227, -0.0000102777931817, -0.0000077845246152, -0.0000050750455431, -0.0000029666938014, -0.0000015824670915, -0.0000007751061906, -0.0000003479272423, -0.0000001416866114, -0.0000000513423356, -0.0000000160384452, -0.0000000041032966, -0.0000000007879361, -0.0000000000963295, -0.0000000000051791, -0.0000000000000377, -0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 2}, 7][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 2E-16 < total abs error < 9E-08
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_7_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7343015625000000, 0.7499250000000000, 0.7655484374999999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {1.0293381519274376, 1.0139198243776921, 0.9986140749763865, 0.9834214257409344, 0.9683423925341904, 0.9533774848678706, 0.9385272057031870, 0.9237920512487094, 0.9091725107554519, 0.8946690663092021, 0.8802821926200948, 0.8660123568094528, 0.8518600181939020, 0.8378256280667834, 0.8239096294768840, 0.8101124570045054, 0.7964345365349024, 0.7828762850291161, 0.7694381102922377, 0.7561204107391369, 0.7429235751576955, 0.7298479824695868, 0.7168940014886509, 0.7040619906769111, 0.6913522978982912, 0.6787652601700870, 0.6663012034122570, 0.6539604421945984, 0.6417432794818805, 0.6296500063770103, 0.6176809018623107, 0.6058362325389998, 0.5941162523649589, 0.5825212023908893, 0.5710513104949563, 0.5597067911160283, 0.5484878449856291, 0.5373946588587170, 0.5264274052434182, 0.5155862421298507, 0.5048713127181711, 0.4942827451459941, 0.4838206522153376, 0.4734851311192504, 0.4632762631682926, 0.4531941135170404, 0.4432387308908020, 0.4334101473127314, 0.4237083778315403, 0.4141334202500163, 0.4046852548545598, 0.3953638441459689, 0.3861691325717004, 0.3771010462598525, 0.3681594927551207, 0.3593443607569870, 0.3506555198604149, 0.3420928202993274, 0.3336560926931645, 0.3253451477968151, 0.3171597762542380, 0.3090997483560931, 0.3011648138017125, 0.2933547014657580, 0.2856691191699151, 0.2781077534599899, 0.2706702693887841, 0.2633563103051316, 0.2561654976494992, 0.2490974307565542, 0.2421516866651230, 0.2353278199359667, 0.2286253624778194, 0.2220438233821379, 0.2155826887670281, 0.2092414216308216, 0.2030194617157856, 0.1969162253824617, 0.1909311054951385, 0.1850634713189712, 0.1793126684292735, 0.1736780186335145, 0.1681588199065615, 0.1627543463397202, 0.1574638481041290, 0.1522865514290730, 0.1472216585957888, 0.1422683479473375, 0.1374257739151306, 0.1326930670626907, 0.1280693341472436, 0.1235536581997297, 0.1191450986238316, 0.1148426913146109, 0.1106454487973494, 0.1065523603871840, 0.1025623923701259, 0.0986744882060446, 0.0948875687541937, 0.0912005325218473, 0.0876122559366030, 0.0841215936428996, 0.0807273788232784, 0.0774284235449077, 0.0742235191318642, 0.0711114365636515, 0.0680909269004065, 0.0651607217352247, 0.0623195336740016, 0.0595660568431610, 0.0568989674256032, 0.0543169242251711, 0.0518185692598910, 0.0494025283842017, 0.0470674119403392, 0.0448118154389908, 0.0426343202692824, 0.0405334944381028, 0.0385078933387067, 0.0346765286556327, 0.0328678201146823, 0.0311284481301685, 0.0278517258965415, 0.0263113641486339, 0.0248343179176040, 0.0220640933014226, 0.0207678681746861, 0.0195288672346216, 0.0172164354112140, 0.0161399569443860, 0.0151146098078357, 0.0132112556267798, 0.0123302370606407, 0.0114943291972742, 0.0099519165933946, 0.0092424749595839, 0.0085722726943246, 0.0073438603799067, 0.0067828271977248, 0.0062553895222740, 0.0052958554036170, 0.0044545761617756, 0.0037212288404366, 0.0030858836004810, 0.0025390356347873, 0.0020716343219933, 0.0016751091568326, 0.0013413920071871, 0.0010629352723950, 0.0008327255549229, 0.0006442925093460, 0.0004917125994038, 0.0003696075759678, 0.0002731375856849, 0.0001979889307000, 0.0001403566221428, 0.0000969220008765, 0.0000648258341514, 0.0000416374309299, 0.0000141961503405, 0.0000023677768872, -0.0000015943413919, -0.0000021645303828, -0.0000016314716941, -0.0000009763106638, -0.0000005027384689, -0.0000002292848813, -0.0000000933991836, -0.0000000337945194, -0.0000000106587791, -0.0000000028308838, -0.0000000005976255, -0.0000000000908088, -0.0000000000082311, -0.0000000000002916, -0.0000000000000011, -0.0000000000000000}
        };
        // Derivative[{0, 0}, {0, 2}, 8][HypergeometricPFQ][{1., 1.}, {2., 1.}, (-4.*#1)/(1. - 1.*#1)]&
        // 2E-18 < total abs error < 8E-08
        const static thread_local CSplineInterpolation hypg_a_1_1_b_2_1_d_0_0_0_2_8_mapped = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5858789062499998, 0.5936906249999999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7186781249999999, 0.7264898437500000, 0.7343015625000000, 0.7499250000000000, 0.7655484374999999, 0.7811718750000000, 0.7967953125000000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {0.9904632779037538, 0.9756083278447334, 0.9608621730948642, 0.9462253141448034, 0.9316982454904394, 0.9172814554430347, 0.9029754259367014, 0.8887806323332185, 0.8746975432241955, 0.8607266202305921, 0.8468683177996049, 0.8331230829989364, 0.8194913553084594, 0.8059735664092984, 0.7925701399703464, 0.7792814914322432, 0.7661080277888394, 0.7530501473661778, 0.7401082395990221, 0.7272826848049719, 0.7145738539562000, 0.7019821084488544, 0.6895077998701742, 0.6771512697633641, 0.6649128493902859, 0.6527928594920183, 0.6407916100473524, 0.6289094000292854, 0.6171465171595827, 0.6055032376614811, 0.5939798260106185, 0.5825765346842663, 0.5712936039089591, 0.5601312614066135, 0.5490897221392372, 0.5381691880523307, 0.5273698478170961, 0.5166918765715670, 0.5061354356607826, 0.4957006723761366, 0.4853877196940332, 0.4751966960139943, 0.4651277048963674, 0.4551808347997860, 0.4453561588185498, 0.4356537344200904, 0.4260736031827035, 0.4166157905337317, 0.4072803054883891, 0.3980671403894321, 0.3889762706478828, 0.3800076544850257, 0.3711612326759008, 0.3624369282945312, 0.3538346464611278, 0.3453542740915254, 0.3369956796491107, 0.3287587128995168, 0.3206432046683645, 0.3126489666023411, 0.3047757909339193, 0.2970234502500260, 0.2893916972649839, 0.2818802645980556, 0.2744888645559336, 0.2672171889205273, 0.2600649087424118, 0.2530316741403074, 0.2461171141069788, 0.2393208363219453, 0.2326424269714069, 0.2260814505758030, 0.2196374498254281, 0.2133099454245426, 0.2070984359444233, 0.2010023976858116, 0.1950212845512256, 0.1891545279276129, 0.1834015365798294, 0.1777616965554403, 0.1722343711013465, 0.1668189005927506, 0.1615146024749824, 0.1563207712187127, 0.1512366782890918, 0.1462615721293562, 0.1413946781594504, 0.1366351987902191, 0.1319823134537292, 0.1274351786502828, 0.1229929280126883, 0.1186546723883568, 0.1144194999397923, 0.1102864762640443, 0.1062546445316926, 0.1023230256459253, 0.0984906184222743, 0.0947563997895642, 0.0911193250126216, 0.0875783279372875, 0.0841323212582622, 0.0807801968103024, 0.0775208258832731, 0.0743530595615453, 0.0712757290882085, 0.0682876462545488, 0.0653876038152186, 0.0625743759295031, 0.0598467186290535, 0.0572033703124349, 0.0546430522667960, 0.0521644692169364, 0.0497663099020045, 0.0474472476800202, 0.0452059411603655, 0.0430410348643434, 0.0409511599138458, 0.0389349347481222, 0.0369909658685738, 0.0333141679481783, 0.0315784993132486, 0.0299094094589793, 0.0267651950061794, 0.0252871685651659, 0.0238699191110121, 0.0212118964576277, 0.0199681893967254, 0.0187793936753735, 0.0165606625371003, 0.0155277944102319, 0.0145439744273318, 0.0127176559057437, 0.0118722613340180, 0.0110701251616741, 0.0095899291246712, 0.0089090469645093, 0.0082657809648259, 0.0070865976853723, 0.0065479693787036, 0.0060415374794517, 0.0051200364938069, 0.0043118446154240, 0.0036070598738260, 0.0029961610330117, 0.0024700379342027, 0.0020200191449969, 0.0016378964736276, 0.0013159459202376, 0.0010469446617605, 0.0008241837043884, 0.0006414758887002, 0.0004931589979752, 0.0003740938002073, 0.0002796569484253, 0.0002057287709303, 0.0001486761009029, 0.0001053304204499, 0.0000492486288597, 0.0000322453974292, 0.0000203466073300, 0.0000069207046648, 0.0000015267345488, -0.0000001752282594, -0.0000004521991594, -0.0000003275422112, -0.0000001729652178, -0.0000000754808244, -0.0000000282693676, -0.0000000091461674, -0.0000000025189148, -0.0000000005697920, -0.0000000000993560, -0.0000000000119533, -0.0000000000008036, -0.0000000000000188, -0.0000000000000000, -0.0000000000000000}
        };

        const double hyp = hypg_a_1_1_1_b_2_2_2_mapped(-xom / (-12.0 - xom));

        const double hyp_c_0 = hypg_a_1_1_b_2_2_d_0_0_0_1_0_mapped(-xom / (-4.0 - xom));
        const double hyp_c_1 = hypg_a_1_1_b_2_2_d_0_0_0_1_1_mapped(-xom / (-4.0 - xom));
        const double hyp_c_2 = hypg_a_1_1_b_2_2_d_0_0_0_1_2_mapped(-xom / (-4.0 - xom));
        const double hyp_c_3 = hypg_a_1_1_b_2_2_d_0_0_0_1_3_mapped(-xom / (-4.0 - xom));
        const double hyp_c_4 = hypg_a_1_1_b_2_2_d_0_0_0_1_4_mapped(-xom / (-4.0 - xom));
        const double hyp_c_5 = hypg_a_1_1_b_2_2_d_0_0_0_1_5_mapped(-xom / (-4.0 - xom));
        const double hyp_c_6 = hypg_a_1_1_b_2_2_d_0_0_0_1_6_mapped(-xom / (-4.0 - xom));
        const double hyp_c_7 = hypg_a_1_1_b_2_2_d_0_0_0_1_7_mapped(-xom / (-4.0 - xom));
        const double hyp_c_8 = hypg_a_1_1_b_2_2_d_0_0_0_1_8_mapped(-xom / (-4.0 - xom));

        const double hyp_d_0 = hypg_a_1_1_b_2_1_d_0_0_0_2_0_mapped(-xom / (-4.0 - xom));
        const double hyp_d_1 = hypg_a_1_1_b_2_1_d_0_0_0_2_1_mapped(-xom / (-4.0 - xom));
        const double hyp_d_2 = hypg_a_1_1_b_2_1_d_0_0_0_2_2_mapped(-xom / (-4.0 - xom));
        const double hyp_d_3 = hypg_a_1_1_b_2_1_d_0_0_0_2_3_mapped(-xom / (-4.0 - xom));
        const double hyp_d_4 = hypg_a_1_1_b_2_1_d_0_0_0_2_4_mapped(-xom / (-4.0 - xom));
        const double hyp_d_5 = hypg_a_1_1_b_2_1_d_0_0_0_2_5_mapped(-xom / (-4.0 - xom));
        const double hyp_d_6 = hypg_a_1_1_b_2_1_d_0_0_0_2_6_mapped(-xom / (-4.0 - xom));
        const double hyp_d_7 = hypg_a_1_1_b_2_1_d_0_0_0_2_7_mapped(-xom / (-4.0 - xom));
        const double hyp_d_8 = hypg_a_1_1_b_2_1_d_0_0_0_2_8_mapped(-xom / (-4.0 - xom));

        // calculate weights term by term
        Weights c = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };

        const Weights c_a = {
            -1 + pi2 / 6. + 2 * hyp * xom - 2 * expint_ei_xom * L + L2 + 2 * expint_ei_xom * log_xom - 2 * eulergamma * (-L + log_xom) - log_xom2 - (exp_xom * (-6 + pi2 + 6 * L2 - 12 * L * log_xom + 6 * log_xom2)) / 6.,
            2 * eulergamma - 2 * expint_ei_xom + 2 * L + exp_xom * (((-6 + pi2) * xom) / 6. + xom * L2 + 2 * log_xom - 2 * L * (1 + xom * log_xom) + xom * log_xom2),
            (18 + pi2 + 12 * hyp * xom - 12 * expint_ei_xom * L + 6 * L2 + 12 * expint_ei_xom * log_xom - 12 * eulergamma * (-L + log_xom) - 6 * log_xom2 - exp_xom * (18 + pi2 + 12 * xom - 2 * pi2 * xom + 24 * xom * log_xom - 12 * xom2 + 2 * pi2 * xom2 + 6 * L2 * (1 - 2 * xom + 2 * xom2) + 6 * log_xom2 * (1 - 2 * xom + 2 * xom2) - 12 * L * (2 * xom + log_xom * (1 - 2 * xom + 2 * xom2)))) / 18.,
            (24 * eulergamma + 24 * (-expint_ei_xom + L) + exp_xom * (-6 * xom * (1 - 3 * xom + xom2) + pi2 * xom * (3 - 3 * xom + xom2) + 6 * xom * L2 * (3 - 3 * xom + xom2) + 6 * xom * log_xom2 * (3 - 3 * xom + xom2) + 12 * log_xom * (2 - xom + xom2) - 12 * L * (2 - xom + xom2 + xom * log_xom * (3 - 3 * xom + xom2)))) / 18.,
            (-36 * eulergamma * (-L + log_xom) + 3 * (34 + pi2 + 12 * hyp * xom - 12 * expint_ei_xom * L + 6 * L2 + 12 * expint_ei_xom * log_xom - 6 * log_xom2) - exp_xom * (102 + 3 * pi2 + 48 * xom - 12 * pi2 * xom - 120 * xom2 + 24 * pi2 * xom2 + 24 * xom * log_xom * (6 - 3 * xom + xom2) + 72 * xom3 - 12 * pi2 * xom3 - 12 * xom4 + 2 * pi2 * xom4 + 6 * L2 * (3 - 12 * xom + 24 * xom2 - 12 * xom3 + 2 * xom4) + 6 * log_xom2 * (3 - 12 * xom + 24 * xom2 - 12 * xom3 + 2 * xom4) - 12 * L * (2 * xom * (6 - 3 * xom + xom2) + log_xom * (3 - 12 * xom + 24 * xom2 - 12 * xom3 + 2 * xom4)))) / 90.,
            (276 * eulergamma - 276 * (expint_ei_xom - L) + exp_xom * (-6 * xom * (1 - 78 * xom + 66 * xom2 - 20 * xom3 + 2 * xom4) + pi2 * xom * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4) + 6 * xom * L2 * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4) + 6 * xom * log_xom2 * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4) + 12 * log_xom * (23 - 22 * xom + 34 * xom2 - 12 * xom3 + 2 * xom4) - 12 * L * (23 - 22 * xom + 34 * xom2 - 12 * xom3 + 2 * xom4 + xom * log_xom * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4)))) / 270.,
            (-540 * eulergamma * (-L + log_xom) + 3 * (694 + 15 * pi2 + 180 * hyp * xom - 180 * expint_ei_xom * L + 90 * L2 + 180 * expint_ei_xom * log_xom - 90 * log_xom2) - exp_xom * (2082 + 45 * pi2 + 732 * xom - 270 * pi2 * xom - 3684 * xom2 + 810 * pi2 * xom2 + 4392 * xom3 - 780 * pi2 * xom3 - 1932 * xom4 + 330 * pi2 * xom4 + 24 * xom * log_xom * (135 - 135 * xom + 85 * xom2 - 20 * xom3 + 2 * xom4) + 360 * xom5 - 60 * pi2 * xom5 - 24 * xom6 + 4 * pi2 * xom6 + 6 * L2 * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6) + 6 * log_xom2 * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6) - 12 * L * (2 * xom * (135 - 135 * xom + 85 * xom2 - 20 * xom3 + 2 * xom4) + log_xom * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6)))) / 1890.,
            (1584 * eulergamma - 1584 * (expint_ei_xom - L) + exp_xom * (-6 * xom * (-51 - 747 * xom + 1049 * xom2 - 610 * xom3 + 166 * xom4 - 21 * xom5 + xom6) + pi2 * xom * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6) + 6 * xom * L2 * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6) + 6 * xom * log_xom2 * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6) + 12 * log_xom * (132 - 183 * xom + 381 * xom2 - 258 * xom3 + 93 * xom4 - 15 * xom5 + xom6) - 12 * L * (132 - 183 * xom + 381 * xom2 - 258 * xom3 + 93 * xom4 - 15 * xom5 + xom6 + xom * log_xom * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6)))) / 1890.,
            (-3780 * eulergamma * (-L + log_xom) + 3 * (5914 + 105 * pi2 + 1260 * hyp * xom - 1260 * expint_ei_xom * L + 630 * L2 + 1260 * expint_ei_xom * log_xom - 630 * log_xom2) - exp_xom * (17742 + 315 * pi2 + 4512 * xom - 2520 * pi2 * xom - 43104 * xom2 + 10080 * pi2 * xom2 + 77472 * xom3 - 14280 * pi2 * xom3 - 55392 * xom4 + 9660 * pi2 * xom4 + 19800 * xom5 - 3360 * pi2 * xom5 - 3672 * xom6 + 616 * pi2 * xom6 + 24 * xom * log_xom * (1260 - 1890 * xom + 1750 * xom2 - 770 * xom3 + 182 * xom4 - 21 * xom5 + xom6) + 336 * xom7 - 56 * pi2 * xom7 - 12 * xom8 + 2 * pi2 * xom8 + 6 * L2 * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8) + 6 * log_xom2 * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8) - 12 * L * (2 * xom * (1260 - 1890 * xom + 1750 * xom2 - 770 * xom3 + 182 * xom4 - 21 * xom5 + xom6) + log_xom * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8)))) / 17010.
        };
        c = c + c_a;

        const Weights c_b = {
            -4 * hyp * xom + 2 * eulergamma2 + 3 * expint_ei_xom - 2 * eulergamma * expint_ei_xom + 2 * expint_ei_xom * L + 4 * eulergamma * log_xom - 4 * expint_ei_xom * log_xom + exp_xom * (eulergamma + log_xom) * (-3 - 2 * L + 2 * log_xom) + 2 * log_xom2,
            2 * expint_ei_xom - exp_xom * (eulergamma + log_xom) * (2 - 3 * xom - 2 * xom * L + 2 * xom * log_xom),
            (-4 * hyp * xom + 2 * eulergamma2 + 3 * expint_ei_xom + 6 * xom * exp_xom + 2 * expint_ei_xom * L + 4 * xom * exp_xom * L - 4 * expint_ei_xom * log_xom - 3 * exp_xom * log_xom + 6 * xom * exp_xom * log_xom - 2 * exp_xom * L * log_xom + 4 * xom * exp_xom * L * log_xom + 2 * log_xom2 + 2 * exp_xom * log_xom2 - 4 * xom * exp_xom * log_xom2 - 6 * exp_xom * log_xom * xom2 - 4 * exp_xom * L * log_xom * xom2 + 4 * exp_xom * log_xom2 * xom2 + eulergamma * (-2 * expint_ei_xom + 4 * log_xom + exp_xom * (-3 + 10 * xom + L * (-2 + 4 * xom - 4 * xom2) - 6 * xom2 + log_xom * (2 - 4 * xom + 4 * xom2)))) / 3.,
            (4 * expint_ei_xom + exp_xom * (8 * xom - 4 * log_xom + 7 * xom * log_xom - 6 * xom * log_xom2 - 6 * xom2 - 7 * log_xom * xom2 + 6 * log_xom2 * xom2 + 2 * xom * L * (2 - 2 * xom + log_xom * (3 - 3 * xom + xom2)) + 3 * log_xom * xom3 - 2 * log_xom2 * xom3 + eulergamma * (-4 + 11 * xom - 11 * xom2 + 2 * xom * L * (3 - 3 * xom + xom2) - 2 * xom * log_xom * (3 - 3 * xom + xom2) + 3 * xom3))) / 3.,
            (-12 * hyp * xom + 6 * eulergamma2 + 9 * expint_ei_xom + 56 * xom * exp_xom + 6 * expint_ei_xom * L + 32 * xom * exp_xom * L - 12 * expint_ei_xom * log_xom - 9 * exp_xom * log_xom + 28 * xom * exp_xom * log_xom - 6 * exp_xom * L * log_xom + 24 * xom * exp_xom * L * log_xom + 6 * log_xom2 + 6 * exp_xom * log_xom2 - 24 * xom * exp_xom * log_xom2 - 62 * exp_xom * xom2 - 36 * exp_xom * L * xom2 - 48 * exp_xom * log_xom * xom2 - 48 * exp_xom * L * log_xom * xom2 + 48 * exp_xom * log_xom2 * xom2 + 18 * exp_xom * xom3 + 12 * exp_xom * L * xom3 + 28 * exp_xom * log_xom * xom3 + 24 * exp_xom * L * log_xom * xom3 - 24 * exp_xom * log_xom2 * xom3 - 6 * exp_xom * log_xom * xom4 - 4 * exp_xom * L * log_xom * xom4 + 4 * exp_xom * log_xom2 * xom4 + eulergamma * (-6 * (expint_ei_xom - 2 * log_xom) + exp_xom * (-9 + 60 * xom - 84 * xom2 + 40 * xom3 + L * (-6 + 24 * xom - 48 * xom2 + 24 * xom3 - 4 * xom4) - 6 * xom4 + log_xom * (6 - 24 * xom + 48 * xom2 - 24 * xom3 + 4 * xom4)))) / 15.,
            (46 * expint_ei_xom + exp_xom * (196 * xom - 46 * log_xom + 83 * xom * log_xom - 90 * xom * log_xom2 - 324 * xom2 - 146 * log_xom * xom2 + 180 * log_xom2 * xom2 + 156 * xom3 + 138 * log_xom * xom3 - 140 * log_xom2 * xom3 - 24 * xom4 - 48 * log_xom * xom4 + 40 * log_xom2 * xom4 + 2 * xom * L * (-8 * (-6 + 12 * xom - 6 * xom2 + xom3) + log_xom * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4)) + 6 * log_xom * xom5 - 4 * log_xom2 * xom5 + eulergamma * (-46 + 179 * xom - 338 * xom2 + 234 * xom3 - 64 * xom4 + 2 * xom * L * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4) - 2 * xom * log_xom * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4) + 6 * xom5))) / 45.,
            (-180 * hyp * xom + 90 * eulergamma2 + 135 * expint_ei_xom + 1590 * xom * exp_xom + 90 * expint_ei_xom * L + 852 * xom * exp_xom * L - 180 * expint_ei_xom * log_xom - 135 * exp_xom * log_xom + 498 * xom * exp_xom * log_xom - 90 * exp_xom * L * log_xom + 540 * xom * exp_xom * L * log_xom + 90 * log_xom2 + 90 * exp_xom * log_xom2 - 540 * xom * exp_xom * log_xom2 - 3294 * exp_xom * xom2 - 1860 * exp_xom * L * xom2 - 1110 * exp_xom * log_xom * xom2 - 1620 * exp_xom * L * log_xom * xom2 + 1620 * exp_xom * log_xom2 * xom2 + 2322 * exp_xom * xom3 + 1420 * exp_xom * L * xom3 + 1260 * exp_xom * log_xom * xom3 + 1560 * exp_xom * L * log_xom * xom3 - 1560 * exp_xom * log_xom2 * xom3 - 632 * exp_xom * xom4 - 400 * exp_xom * L * xom4 - 670 * exp_xom * log_xom * xom4 - 660 * exp_xom * L * log_xom * xom4 + 660 * exp_xom * log_xom2 * xom4 + 60 * exp_xom * xom5 + 40 * exp_xom * L * xom5 + 148 * exp_xom * log_xom * xom5 + 120 * exp_xom * L * log_xom * xom5 - 120 * exp_xom * log_xom2 * xom5 - 12 * exp_xom * log_xom * xom6 - 8 * exp_xom * L * log_xom * xom6 + 8 * exp_xom * log_xom2 * xom6 + eulergamma * (-90 * (expint_ei_xom - 2 * log_xom) + exp_xom * (-135 + 1350 * xom - 2970 * xom2 + 2680 * xom3 - 1070 * xom4 + 188 * xom5 - 12 * xom6 - 2 * L * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6) + 2 * log_xom * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6)))) / 315.,
            (264 * expint_ei_xom + exp_xom * (1764 * xom - 264 * log_xom + 459 * xom * log_xom - 630 * xom * log_xom2 - 4488 * xom2 - 1041 * log_xom * xom2 + 1890 * log_xom2 * xom2 + 4054 * xom3 + 1557 * log_xom * xom3 - 2310 * log_xom2 * xom3 - 1606 * xom4 - 1072 * log_xom * xom4 + 1260 * log_xom2 * xom4 + 280 * xom5 + 354 * log_xom * xom5 - 336 * log_xom2 * xom5 - 18 * xom6 - 53 * log_xom * xom6 + 42 * log_xom2 * xom6 + 2 * xom * L * (426 - 1278 * xom + 1212 * xom2 - 502 * xom3 + 90 * xom4 - 6 * xom5 + log_xom * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6)) + 3 * log_xom * xom7 - 2 * log_xom2 * xom7 + eulergamma * (-264 + 1311 * xom - 3597 * xom2 + 3981 * xom3 - 2076 * xom4 + 534 * xom5 - 65 * xom6 + 2 * xom * L * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6) - 2 * xom * log_xom * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6) + 3 * xom7))) / 315.,
            (-1260 * hyp * xom + 630 * eulergamma2 + 945 * expint_ei_xom + 17280 * xom * exp_xom + 630 * expint_ei_xom * L + 8928 * xom * exp_xom * L - 1260 * expint_ei_xom * log_xom - 945 * exp_xom * log_xom + 3672 * xom * exp_xom * log_xom - 630 * exp_xom * L * log_xom + 5040 * xom * exp_xom * L * log_xom + 630 * log_xom2 + 630 * exp_xom * log_xom2 - 5040 * xom * exp_xom * log_xom2 - 51732 * exp_xom * xom2 - 28728 * exp_xom * L * xom2 - 9072 * exp_xom * log_xom * xom2 - 20160 * exp_xom * L * log_xom * xom2 + 20160 * exp_xom * log_xom2 * xom2 + 57516 * exp_xom * xom3 + 34216 * exp_xom * L * xom3 + 15624 * exp_xom * log_xom * xom3 + 28560 * exp_xom * L * log_xom * xom3 - 28560 * exp_xom * log_xom2 * xom3 - 29532 * exp_xom * xom4 - 18200 * exp_xom * L * xom4 - 13860 * exp_xom * log_xom * xom4 - 19320 * exp_xom * L * log_xom * xom4 + 19320 * exp_xom * log_xom2 * xom4 + 7500 * exp_xom * xom5 + 4760 * exp_xom * L * xom5 + 6048 * exp_xom * log_xom * xom5 + 6720 * exp_xom * L * log_xom * xom5 - 6720 * exp_xom * log_xom2 * xom5 - 906 * exp_xom * xom6 - 588 * exp_xom * L * xom6 - 1344 * exp_xom * log_xom * xom6 - 1232 * exp_xom * L * log_xom * xom6 + 1232 * exp_xom * log_xom2 * xom6 + 42 * exp_xom * xom7 + 28 * exp_xom * L * xom7 + 144 * exp_xom * log_xom * xom7 + 112 * exp_xom * L * log_xom * xom7 - 112 * exp_xom * log_xom2 * xom7 - 6 * exp_xom * log_xom * xom8 - 4 * exp_xom * L * log_xom * xom8 + 4 * exp_xom * log_xom2 * xom8 + eulergamma * (-630 * (expint_ei_xom - 2 * log_xom) + exp_xom * (-945 + 12600 * xom - 37800 * xom2 + 49840 * xom3 - 32060 * xom4 + 10808 * xom5 - 1932 * xom6 + 172 * xom7 - 6 * xom8 - 2 * L * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8) + 2 * log_xom * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8)))) / 2835.
        };
        c = c + c_b;

        Weights c_c = {
            2 * (eulergamma - expint_ei_xom - xom * hyp_c_0 + log_xom - exp_xom * (-eulergamma + expint_ei_xom_plus - log_xom) * (-L + log_xom)),
            2 * (1 - xom * hyp_c_0 + L - log_xom + exp_xom * (-1 + xom * eulergamma * (L - log_xom) + log_xom + xom * expint_ei_xom_plus * log_xom + L * (-1 - xom * expint_ei_xom_plus + xom * log_xom) - xom * log_xom2) + hyp_c_1 * xom2),
            (-2 * (expint_ei_xom + 3 * xom * hyp_c_0 + 2 * xom * L - log_xom - 2 * xom * log_xom - 6 * hyp_c_1 * xom2 + eulergamma * (-1 - exp_xom * (-L + log_xom) * (1 - 2 * xom + 2 * xom2)) - exp_xom * (2 * xom - 4 * xom * log_xom + log_xom2 - 2 * xom * log_xom2 + 2 * log_xom2 * xom2 - expint_ei_xom_plus * log_xom * (1 - 2 * xom + 2 * xom2) + L * (4 * xom + log_xom * (-1 + 2 * xom - 2 * xom2) + expint_ei_xom_plus * (1 - 2 * xom + 2 * xom2))) + 2 * hyp_c_2 * xom3)) / 3.,
            (-2 * (-2 + 3 * xom * hyp_c_0 - 2 * L + 2 * xom * L + 2 * log_xom - 2 * xom * log_xom - 9 * hyp_c_1 * xom2 - L * xom2 + log_xom * xom2 + 6 * hyp_c_2 * xom3 + exp_xom * (2 - xom - 2 * log_xom + 3 * xom * log_xom - 3 * xom * expint_ei_xom_plus * log_xom + 3 * xom * log_xom2 + xom2 - 3 * log_xom * xom2 + 3 * expint_ei_xom_plus * log_xom * xom2 - 3 * log_xom2 * xom2 + xom * eulergamma * (-L + log_xom) * (3 - 3 * xom + xom2) + L * (2 - 3 * xom + 3 * xom2 + xom * expint_ei_xom_plus * (3 - 3 * xom + xom2) - xom * log_xom * (3 - 3 * xom + xom2)) - expint_ei_xom_plus * log_xom * xom3 + log_xom2 * xom3) - hyp_c_3 * xom4)) / 3.,
            (-2 * (3 * expint_ei_xom + 15 * xom * hyp_c_0 + 16 * xom * L - 3 * log_xom - 16 * xom * log_xom - 60 * hyp_c_1 * xom2 - 10 * L * xom2 + 10 * log_xom * xom2 + 60 * hyp_c_2 * xom3 + 2 * L * xom3 - 2 * log_xom * xom3 - 20 * hyp_c_3 * xom4 + eulergamma * (-3 - exp_xom * (-L + log_xom) * (3 - 12 * xom + 24 * xom2 - 12 * xom3 + 2 * xom4)) + exp_xom * (-12 * xom + 28 * xom * log_xom - 3 * log_xom2 + 12 * xom * log_xom2 + 6 * xom2 - 24 * log_xom * xom2 - 24 * log_xom2 * xom2 - 2 * xom3 + 8 * log_xom * xom3 + 12 * log_xom2 * xom3 - 2 * log_xom2 * xom4 + expint_ei_xom_plus * log_xom * (3 - 12 * xom + 24 * xom2 - 12 * xom3 + 2 * xom4) + L * (-4 * xom * (7 - 6 * xom + 2 * xom2) + expint_ei_xom_plus * (-3 + 12 * xom - 24 * xom2 + 12 * xom3 - 2 * xom4) + log_xom * (3 - 12 * xom + 24 * xom2 - 12 * xom3 + 2 * xom4))) + 2 * hyp_c_4 * xom5)) / 15.,
            (-2 * (-23 + 45 * xom * hyp_c_0 - 23 * L + 48 * xom * L + 23 * log_xom - 48 * xom * log_xom - 225 * hyp_c_1 * xom2 - 54 * L * xom2 + 54 * log_xom * xom2 + 300 * hyp_c_2 * xom3 + 18 * L * xom3 - 18 * log_xom * xom3 - 150 * hyp_c_3 * xom4 - 2 * L * xom4 + 2 * log_xom * xom4 + 30 * hyp_c_4 * xom5 + exp_xom * (23 - 22 * xom - 23 * log_xom + 70 * xom * log_xom - 45 * xom * expint_ei_xom_plus * log_xom + 45 * xom * log_xom2 + 34 * xom2 - 130 * log_xom * xom2 + 90 * expint_ei_xom_plus * log_xom * xom2 - 90 * log_xom2 * xom2 - 12 * xom3 + 60 * log_xom * xom3 - 70 * expint_ei_xom_plus * log_xom * xom3 + 70 * log_xom2 * xom3 + 2 * xom4 - 10 * log_xom * xom4 + 20 * expint_ei_xom_plus * log_xom * xom4 - 20 * log_xom2 * xom4 + xom * eulergamma * (-L + log_xom) * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4) + L * (23 - 70 * xom + 130 * xom2 - 60 * xom3 + xom * log_xom * (-45 + 90 * xom - 70 * xom2 + 20 * xom3 - 2 * xom4) + 10 * xom4 + xom * expint_ei_xom_plus * (45 - 90 * xom + 70 * xom2 - 20 * xom3 + 2 * xom4)) - 2 * expint_ei_xom_plus * log_xom * xom5 + 2 * log_xom2 * xom5) - 2 * hyp_c_5 * xom6)) / 45.,
            (-2 * (45 * expint_ei_xom + 315 * xom * hyp_c_0 + 426 * xom * L - 45 * log_xom - 426 * xom * log_xom - 1890 * hyp_c_1 * xom2 - 546 * L * xom2 + 546 * log_xom * xom2 + 3150 * hyp_c_2 * xom3 + 278 * L * xom3 - 278 * log_xom * xom3 - 2100 * hyp_c_3 * xom4 - 56 * L * xom4 + 56 * log_xom * xom4 + 630 * hyp_c_4 * xom5 + 4 * L * xom5 - 4 * log_xom * xom5 - 84 * hyp_c_5 * xom6 + eulergamma * (-45 - exp_xom * (-L + log_xom) * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6)) + exp_xom * (-270 * xom + 696 * xom * log_xom - 45 * log_xom2 + 270 * xom * log_xom2 + 270 * xom2 - 1200 * log_xom * xom2 - 810 * log_xom2 * xom2 - 170 * xom3 + 880 * log_xom * xom3 + 780 * log_xom2 * xom3 + 40 * xom4 - 240 * log_xom * xom4 - 330 * log_xom2 * xom4 - 4 * xom5 + 24 * log_xom * xom5 + 60 * log_xom2 * xom5 - 4 * log_xom2 * xom6 + expint_ei_xom_plus * log_xom * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6) + L * (-8 * xom * (87 - 150 * xom + 110 * xom2 - 30 * xom3 + 3 * xom4) + expint_ei_xom_plus * (-45 + 270 * xom - 810 * xom2 + 780 * xom3 - 330 * xom4 + 60 * xom5 - 4 * xom6) + log_xom * (45 - 270 * xom + 810 * xom2 - 780 * xom3 + 330 * xom4 - 60 * xom5 + 4 * xom6))) + 4 * hyp_c_6 * xom7)) / 315.,
            (-2 * (-132 + 315 * xom * hyp_c_0 - 132 * L + 426 * xom * L + 132 * log_xom - 426 * xom * log_xom - 2205 * hyp_c_1 * xom2 - 759 * L * xom2 + 759 * log_xom * xom2 + 4410 * hyp_c_2 * xom3 + 498 * L * xom3 - 498 * log_xom * xom3 - 3675 * hyp_c_3 * xom4 - 149 * L * xom4 + 149 * log_xom * xom4 + 1470 * hyp_c_4 * xom5 + 20 * L * xom5 - 20 * log_xom * xom5 - 294 * hyp_c_5 * xom6 - L * xom6 + log_xom * xom6 + 28 * hyp_c_6 * xom7 + exp_xom * (132 - 183 * xom - 132 * log_xom + 609 * xom * log_xom - 315 * xom * expint_ei_xom_plus * log_xom + 315 * xom * log_xom2 + 381 * xom2 - 1659 * log_xom * xom2 + 945 * expint_ei_xom_plus * log_xom * xom2 - 945 * log_xom2 * xom2 - 258 * xom3 + 1470 * log_xom * xom3 - 1155 * expint_ei_xom_plus * log_xom * xom3 + 1155 * log_xom2 * xom3 + 93 * xom4 - 595 * log_xom * xom4 + 630 * expint_ei_xom_plus * log_xom * xom4 - 630 * log_xom2 * xom4 - 15 * xom5 + 105 * log_xom * xom5 - 168 * expint_ei_xom_plus * log_xom * xom5 + 168 * log_xom2 * xom5 + xom6 - 7 * log_xom * xom6 + 21 * expint_ei_xom_plus * log_xom * xom6 - 21 * log_xom2 * xom6 + xom * eulergamma * (-L + log_xom) * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6) + L * (132 - 609 * xom + 1659 * xom2 - 1470 * xom3 + 595 * xom4 - 105 * xom5 + 7 * xom6 + xom * expint_ei_xom_plus * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6) - xom * log_xom * (315 - 945 * xom + 1155 * xom2 - 630 * xom3 + 168 * xom4 - 21 * xom5 + xom6)) - expint_ei_xom_plus * log_xom * xom7 + log_xom2 * xom7) - hyp_c_7 * xom8)) / 315.,
            (-2 * (315 * expint_ei_xom + 2835 * xom * hyp_c_0 + 4464 * xom * L - 315 * log_xom - 4464 * xom * log_xom - 22680 * hyp_c_1 * xom2 - 8748 * L * xom2 + 8748 * log_xom * xom2 + 52920 * hyp_c_2 * xom3 + 7244 * L * xom3 - 7244 * log_xom * xom3 - 52920 * hyp_c_3 * xom4 - 2844 * L * xom4 + 2844 * log_xom * xom4 + 26460 * hyp_c_4 * xom5 + 564 * L * xom5 - 564 * log_xom * xom5 - 7056 * hyp_c_5 * xom6 - 54 * L * xom6 + 54 * log_xom * xom6 + 1008 * hyp_c_6 * xom7 + 2 * L * xom7 - 2 * log_xom * xom7 - 72 * hyp_c_7 * xom8 + eulergamma * (-315 - exp_xom * (-L + log_xom) * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8)) + exp_xom * (-2520 * xom + 6984 * xom * log_xom - 315 * log_xom2 + 2520 * xom * log_xom2 + 3780 * xom2 - 18144 * log_xom * xom2 - 10080 * log_xom2 * xom2 - 3500 * xom3 + 20608 * log_xom * xom3 + 14280 * log_xom2 * xom3 + 1540 * xom4 - 10640 * log_xom * xom4 - 9660 * log_xom2 * xom4 - 364 * xom5 + 2744 * log_xom * xom5 + 3360 * log_xom2 * xom5 + 42 * xom6 - 336 * log_xom * xom6 - 616 * log_xom2 * xom6 - 2 * xom7 + 16 * log_xom * xom7 + 56 * log_xom2 * xom7 - 2 * log_xom2 * xom8 + expint_ei_xom_plus * log_xom * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8) + L * (-8 * xom * (873 - 2268 * xom + 2576 * xom2 - 1330 * xom3 + 343 * xom4 - 42 * xom5 + 2 * xom6) + expint_ei_xom_plus * (-315 + 2520 * xom - 10080 * xom2 + 14280 * xom3 - 9660 * xom4 + 3360 * xom5 - 616 * xom6 + 56 * xom7 - 2 * xom8) + log_xom * (315 - 2520 * xom + 10080 * xom2 - 14280 * xom3 + 9660 * xom4 - 3360 * xom5 + 616 * xom6 - 56 * xom7 + 2 * xom8))) + 2 * hyp_c_8 * xom9)) / 2835.
        };
        c = c + c_c;

        Weights c_d = {
            2 * (eulergamma - expint_ei_xom - xom * hyp_c_0 + xom * hyp_d_0 + log_xom),
            -2 * (-1 + exp_xom + xom * hyp_c_0 - xom * hyp_d_0 - hyp_c_1 * xom2 + hyp_d_1 * xom2),
            (2 * (eulergamma - expint_ei_xom + 2 * xom * exp_xom - 3 * xom * hyp_c_0 + 3 * xom * hyp_d_0 + log_xom + 6 * hyp_c_1 * xom2 - 6 * hyp_d_1 * xom2 - 2 * hyp_c_2 * xom3 + 2 * hyp_d_2 * xom3)) / 3.,
            (2 * (2 - 3 * xom * hyp_c_0 + 3 * xom * hyp_d_0 + exp_xom * (-2 + xom - xom2) + 9 * hyp_c_1 * xom2 - 9 * hyp_d_1 * xom2 - 6 * hyp_c_2 * xom3 + 6 * hyp_d_2 * xom3 + hyp_c_3 * xom4 - hyp_d_3 * xom4)) / 3.,
            (2 * (3 * eulergamma - 3 * expint_ei_xom - 15 * xom * hyp_c_0 + 15 * xom * hyp_d_0 + 3 * log_xom + 60 * hyp_c_1 * xom2 - 60 * hyp_d_1 * xom2 + 2 * xom * exp_xom * (6 - 3 * xom + xom2) - 60 * hyp_c_2 * xom3 + 60 * hyp_d_2 * xom3 + 20 * hyp_c_3 * xom4 - 20 * hyp_d_3 * xom4 - 2 * hyp_c_4 * xom5 + 2 * hyp_d_4 * xom5)) / 15.,
            (2 * (23 - 45 * xom * hyp_c_0 + 45 * xom * hyp_d_0 + 225 * hyp_c_1 * xom2 - 225 * hyp_d_1 * xom2 - 300 * hyp_c_2 * xom3 + 300 * hyp_d_2 * xom3 + exp_xom * (-23 + 22 * xom - 34 * xom2 + 12 * xom3 - 2 * xom4) + 150 * hyp_c_3 * xom4 - 150 * hyp_d_3 * xom4 - 30 * hyp_c_4 * xom5 + 30 * hyp_d_4 * xom5 + 2 * hyp_c_5 * xom6 - 2 * hyp_d_5 * xom6)) / 45.,
            (2 * eulergamma) / 7. - (2 * expint_ei_xom) / 7. - 2 * xom * hyp_c_0 + 2 * xom * hyp_d_0 + (2 * log_xom) / 7. + 12 * hyp_c_1 * xom2 - 12 * hyp_d_1 * xom2 - 20 * hyp_c_2 * xom3 + 20 * hyp_d_2 * xom3 + (40 * hyp_c_3 * xom4) / 3. - (40 * hyp_d_3 * xom4) / 3. + (4 * xom * exp_xom * (135 - 135 * xom + 85 * xom2 - 20 * xom3 + 2 * xom4)) / 315. - 4 * hyp_c_4 * xom5 + 4 * hyp_d_4 * xom5 + (8 * hyp_c_5 * xom6) / 15. - (8 * hyp_d_5 * xom6) / 15. - (8 * hyp_c_6 * xom7) / 315. + (8 * hyp_d_6 * xom7) / 315.,
            (2 * (132 - 315 * xom * hyp_c_0 + 315 * xom * hyp_d_0 + 2205 * hyp_c_1 * xom2 - 2205 * hyp_d_1 * xom2 - 4410 * hyp_c_2 * xom3 + 4410 * hyp_d_2 * xom3 + 3675 * hyp_c_3 * xom4 - 3675 * hyp_d_3 * xom4 - 1470 * hyp_c_4 * xom5 + 1470 * hyp_d_4 * xom5 + 294 * hyp_c_5 * xom6 - 294 * hyp_d_5 * xom6 - exp_xom * (132 - 183 * xom + 381 * xom2 - 258 * xom3 + 93 * xom4 - 15 * xom5 + xom6) - 28 * hyp_c_6 * xom7 + 28 * hyp_d_6 * xom7 + hyp_c_7 * xom8 - hyp_d_7 * xom8)) / 315.,
            (-2 * (-315 * eulergamma + 315 * expint_ei_xom + 2835 * xom * hyp_c_0 - 2835 * xom * hyp_d_0 - 315 * log_xom - 22680 * hyp_c_1 * xom2 + 22680 * hyp_d_1 * xom2 + 52920 * hyp_c_2 * xom3 - 52920 * hyp_d_2 * xom3 - 52920 * hyp_c_3 * xom4 + 52920 * hyp_d_3 * xom4 + 26460 * hyp_c_4 * xom5 - 26460 * hyp_d_4 * xom5 - 7056 * hyp_c_5 * xom6 + 7056 * hyp_d_5 * xom6 - 2 * xom * exp_xom * (1260 - 1890 * xom + 1750 * xom2 - 770 * xom3 + 182 * xom4 - 21 * xom5 + xom6) + 1008 * hyp_c_6 * xom7 - 1008 * hyp_d_6 * xom7 - 72 * hyp_c_7 * xom8 + 72 * hyp_d_7 * xom8 + 2 * hyp_c_8 * xom9 - 2 * hyp_d_8 * xom9)) / 2835.,
        };
        c = c + c_d;

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);
        return 1.0 / omega_0 * std::inner_product(a_begin, a_end, c.begin(), 0.0);
    }

    double
    AnalyticFormFactorBToGammaQCDF::lapltr_incomplete_effective(const double & Egamma, const double & omega_cut, const double & sigma) const
    {
        const double L = std::log(mu*mu / (2.*Egamma * omega_0));
        const double L2 = L * L;

        const double xom = omega_cut / omega_0;
        const double xom2 = xom  * xom;
        const double xom3 = xom2 * xom;
        const double xom4 = xom3 * xom;
        const double xom5 = xom4 * xom;
        const double xom6 = xom5 * xom;
        const double xom7 = xom6 * xom;
        const double xom8 = xom7 * xom;
        const double xom9 = xom8 * xom;
        const double xom10 = xom9 * xom;

        const double log_xom = std::log(xom);
        const double log2_xom = log_xom * log_xom;

        const double xsg = sigma * omega_0;
        const double xsg2 = xsg * xsg;
        const double xsg3 = xsg * xsg2;
        const double xsg4 = xsg * xsg3;
        const double xsg5 = xsg * xsg4;
        const double xsg6 = xsg * xsg5;
        const double xsg7 = xsg * xsg6;
        const double xsg8 = xsg * xsg7;
        const double xsg9 = xsg * xsg8;
        const double xsg10 = xsg * xsg9;

        const double xsgplus = 1.0 + xsg;
        const double xsgplus_m1  = 1.0 / xsgplus;
        const double xsgplus_m2  = xsgplus_m1 * xsgplus_m1;
        const double xsgplus_m3  = xsgplus_m2 * xsgplus_m1;
        const double xsgplus_m4  = xsgplus_m3 * xsgplus_m1;
        const double xsgplus_m5  = xsgplus_m4 * xsgplus_m1;
        const double xsgplus_m6  = xsgplus_m5 * xsgplus_m1;
        const double xsgplus_m7  = xsgplus_m6 * xsgplus_m1;
        const double xsgplus_m8  = xsgplus_m7 * xsgplus_m1;
        const double xsgplus_m9  = xsgplus_m8 * xsgplus_m1;
        const double xsgplus_m10 = xsgplus_m9 * xsgplus_m1;

        const double log_xsgplus = std::log(xsgplus);

        const double log_xsg = std::log(xsg);
        const double expint_ei_xom_xsg = gsl_sf_expint_Ei(-xom * xsg);
        const double expint_ei_plus_xom = gsl_sf_expint_Ei(xom);
        const double exp_xom_xsg = std::exp(-xom * xsg);
        const double exp_xom_xsgplus = std::exp(-xom * xsgplus);
        const double expint_ei_xom_xsgplus = gsl_sf_expint_Ei(-xom * xsgplus);

        const static thread_local CSplineInterpolation hypg_a_1_1_1_b_2_2_2_mapped = {
            {0.0000000000000000, 0.0039062460937500, 0.0078124921875000, 0.0156249843750000, 0.0195312304687500, 0.0234374765625000, 0.0312499687500000, 0.0351562148437500, 0.0390624609375000, 0.0468749531250000, 0.0507811992187500, 0.0546874453125000, 0.0624999375000000, 0.0664061835937500, 0.0703124296875000, 0.0781249218750000, 0.0820311679687500, 0.0859374140625000, 0.0937499062500000, 0.0976561523437500, 0.1015623984375000, 0.1093748906250000, 0.1132811367187500, 0.1171873828125000, 0.1249998750000000, 0.1289061210937500, 0.1328123671875000, 0.1406248593750000, 0.1445311054687500, 0.1484373515625000, 0.1562498437500000, 0.1601560898437500, 0.1640623359375000, 0.1718748281250000, 0.1757810742187500, 0.1796873203125000, 0.1874998125000000, 0.1914060585937500, 0.1953123046875000, 0.2031247968750000, 0.2109372890625000, 0.2187497812500000, 0.2265622734375000, 0.2343747656250000, 0.2421872578125000, 0.2499997500000000, 0.2578122421875000, 0.2656247343750000, 0.2734372265625000, 0.2812497187500000, 0.2890622109375000, 0.2968747031250000, 0.3046871953125000, 0.3124996875000000, 0.3203121796875000, 0.3281246718750000, 0.3359371640625000, 0.3437496562500000, 0.3515621484375000, 0.3593746406250000, 0.3671871328125000, 0.3749996250000000, 0.3828121171875000, 0.3906246093750000, 0.3984371015625000, 0.4062495937500000, 0.4140620859375000, 0.4218745781250000, 0.4296870703125000, 0.4374995625000000, 0.4453120546875000, 0.4531245468750000, 0.4609370390624999, 0.4687495312500000, 0.4765620234375000, 0.4843745156250000, 0.4921870078125000, 0.4999995000000000, 0.5078119921874999, 0.5156244843750000, 0.5234369765625000, 0.5312494687500000, 0.5390619609375000, 0.5468744531250000, 0.5546869453125001, 0.5624994375000000, 0.5703119296874999, 0.5781244218749999, 0.5859369140624999, 0.5937494062499999, 0.6015618984375000, 0.6093743906250000, 0.6171868828125000, 0.6249993750000000, 0.6328118671875000, 0.6406243593750001, 0.6562493437500000, 0.6718743281249999, 0.6874993125000000, 0.6953118046875000, 0.7031242968750000, 0.7109367890625000, 0.7187492812500000, 0.7265617734375001, 0.7343742656250001, 0.7421867578125001, 0.7499992500000000, 0.7578117421874999, 0.7656242343749999, 0.7734367265625000, 0.7812492187500000, 0.7890617109375000, 0.7968742031250000, 0.8046866953125000, 0.8124991875000001, 0.8203116796875001, 0.8281241718750001, 0.8359366640625001, 0.8437491562500000, 0.8515616484374999, 0.8593741406249999, 0.8671866328125000, 0.8749991250000000, 0.8789053710937500, 0.8828116171875000, 0.8906241093750000, 0.8945303554687500, 0.8984366015625000, 0.9062490937500000, 0.9101553398437499, 0.9140615859374999, 0.9218740781249999, 0.9257803242187499, 0.9296865703124999, 0.9335928164062499, 0.9374990624999999, 0.9414053085937499, 0.9453115546874999, 0.9492178007812500, 0.9531240468750000, 0.9570302929687500, 0.9609365390625000, 0.9648427851562500, 0.9687490312500000, 0.9726552773437500, 0.9765615234375000, 0.9785146464843750, 0.9804677695312500, 0.9843740156250000, 0.9863271386718750, 0.9882802617187501, 0.9902333847656251, 0.9921865078125001, 0.9941396308593751, 0.9960927539062501, 0.9980458769531251, 0.9990224384765625, 0.9999990000000000},
            {1.0000000000000000, 0.9941583929877398, 0.9883521510465745, 0.9768448054284812, 0.9711432221891033, 0.9654760449433876, 0.9542439450347797, 0.9486785398233764, 0.9431465755566899, 0.9321820015427964, 0.9267489070299220, 0.9213482839788568, 0.9106434805773999, 0.9053388140186672, 0.9000656465535604, 0.8896128354009065, 0.8844327048379499, 0.8792830996660734, 0.8690744917209817, 0.8640150021772723, 0.8589850645313900, 0.8490128724204984, 0.8440701320532680, 0.8391559718262925, 0.8294124220512941, 0.8245825482188180, 0.8197802860033563, 0.8102576309261300, 0.8055367561278833, 0.8008425291177979, 0.7915330586366719, 0.7869173362824501, 0.7823273039922284, 0.7732233568201523, 0.7687089667831926, 0.7642193165399559, 0.7553132909937376, 0.7508964449022166, 0.7465033970646480, 0.7377877612736387, 0.7291645203530777, 0.7206318217958313, 0.7121878243784299, 0.7038306986569567, 0.6955586274343231, 0.6873698061976266, 0.6792624435243149, 0.6712347614559194, 0.6632849958381590, 0.6554113966262541, 0.6476122281543372, 0.6398857693678888, 0.6322303140181778, 0.6246441708177264, 0.6171256635558746, 0.6096731311735598, 0.6022849277964757, 0.5949594227258184, 0.5876950003858655, 0.5804900602276731, 0.5733430165882032, 0.5662522985042238, 0.5592163494803361, 0.5522336272104963, 0.5453026032523957, 0.5384217626540494, 0.5315896035319203, 0.5248046365998611, 0.5180653846480998, 0.5113703819714212, 0.5047181737455944, 0.4981073153509825, 0.4915363716421214, 0.4850039161618859, 0.4785085302986579, 0.4720488023846726, 0.4656233267334515, 0.4592307026139130, 0.4528695331583941, 0.4465384242014142, 0.4402359830455434, 0.4339608171502190, 0.4277115327387633, 0.4214867333181869, 0.4152850181056122, 0.4091049803542997, 0.4029452055713061, 0.3968042696177170, 0.3906807366811726, 0.3845731571090184, 0.3784800650888335, 0.3723999761612959, 0.3663313845482948, 0.3602727602768579, 0.3542225460767691, 0.3481791540266612, 0.3361063093165388, 0.3240407635128744, 0.3119682948323468, 0.3059247705619786, 0.2998737489207574, 0.2938131554252623, 0.2877408285031435, 0.2816545100914322, 0.2755518351207863, 0.2694303197118364, 0.2632873478767649, 0.2571201564787214, 0.2509258181516748, 0.2447012218212089, 0.2384430503891427, 0.2321477550471277, 0.2258115255603932, 0.2194302557042400, 0.2129995028313228, 0.2065144402813779, 0.1999698009946798, 0.1933598102246399, 0.1866781046181862, 0.1799176340784925, 0.1730705416444632, 0.1661280149655309, 0.1590801005875836, 0.1555131439463915, 0.1519154688310793, 0.1446211119445585, 0.1409206683676983, 0.1371819504727896, 0.1295803107026743, 0.1257121599092061, 0.1217952166481772, 0.1138014077773262, 0.1097168370871317, 0.1055679378727146, 0.1013496160676611, 0.0970561092213285, 0.0926808517420605, 0.0882163022620624, 0.0836537190049423, 0.0789828622727168, 0.0741915923206841, 0.0692653128846872, 0.0641861795352822, 0.0589319357745224, 0.0534741321275083, 0.0477752628698260, 0.0448200951501434, 0.0417838625672159, 0.0354253761121182, 0.0320745745955113, 0.0285830500427352, 0.0249220635580111, 0.0210501476667170, 0.0169026546168545, 0.0123657102415211, 0.0071870900768604, 0.0041373732987571, 0.0000000000000000}
        };
        const double hyp_xom_xsgplus = hypg_a_1_1_1_b_2_2_2_mapped(-xom * xsgplus / (-12.0 - xom * xsgplus));

        // N[HypergeometricPFQ[{1, 1, 1}, {2, 2, 2}, (4.*#1)/(1. - 1.*#1)]/Exp[(4.*#1)/(1. - 1.*#1)]] &
        // 6E-13 < total abs error < 7E-08
        const static thread_local CSplineInterpolation hypg_a_1_1_1_b_2_2_2_mapped_scaled = {
            {0.0000000000000000, 0.0039058593750000, 0.0078117187500000, 0.0117175781250000, 0.0156234375000000, 0.0195292968750000, 0.0234351562500000, 0.0273410156250000, 0.0312468750000000, 0.0351527343750000, 0.0390585937500000, 0.0429644531250000, 0.0468703125000000, 0.0507761718750000, 0.0546820312500000, 0.0585878906250000, 0.0624937500000000, 0.0663996093750000, 0.0703054687500000, 0.0742113281250000, 0.0781171875000000, 0.0820230468750000, 0.0859289062500000, 0.0898347656250000, 0.0937406250000000, 0.0976464843750000, 0.1015523437500000, 0.1054582031250000, 0.1093640625000000, 0.1132699218750000, 0.1171757812500000, 0.1210816406250000, 0.1249875000000000, 0.1288933593750000, 0.1327992187500000, 0.1367050781250000, 0.1406109375000000, 0.1445167968750000, 0.1484226562500000, 0.1523285156250000, 0.1562343750000000, 0.1601402343750000, 0.1640460937500000, 0.1679519531250000, 0.1718578125000000, 0.1757636718750000, 0.1796695312500000, 0.1835753906250000, 0.1874812500000000, 0.1913871093750000, 0.1952929687500000, 0.1991988281250000, 0.2031046875000000, 0.2070105468750000, 0.2109164062500000, 0.2148222656250000, 0.2187281250000000, 0.2226339843750000, 0.2265398437500000, 0.2304457031250000, 0.2343515625000000, 0.2382574218750000, 0.2421632812500000, 0.2460691406250000, 0.2499750000000000, 0.2538808593750000, 0.2577867187500000, 0.2616925781250000, 0.2655984375000000, 0.2695042968750000, 0.2734101562500000, 0.2773160156250000, 0.2812218750000000, 0.2851277343749999, 0.2890335937499999, 0.2929394531249999, 0.2968453125000000, 0.3007511718750000, 0.3046570312500000, 0.3085628906250000, 0.3124687500000000, 0.3163746093749999, 0.3202804687499999, 0.3241863281249999, 0.3280921875000000, 0.3319980468750000, 0.3359039062500000, 0.3398097656250000, 0.3437156250000000, 0.3476214843750000, 0.3515273437499999, 0.3554332031249999, 0.3593390625000000, 0.3632449218750000, 0.3671507812500000, 0.3710566406250000, 0.3749625000000000, 0.3788683593750000, 0.3827742187500000, 0.3866800781249999, 0.3905859375000000, 0.3944917968750000, 0.3983976562500000, 0.4023035156250000, 0.4062093750000000, 0.4101152343750000, 0.4140210937500000, 0.4179269531249999, 0.4218328125000000, 0.4257386718750000, 0.4296445312500000, 0.4335503906250000, 0.4374562500000000, 0.4413621093750000, 0.4452679687500000, 0.4491738281250000, 0.4530796875000000, 0.4569855468750000, 0.4608914062500000, 0.4687031250000000, 0.4726089843750000, 0.4765148437500000, 0.4843265625000000, 0.4882324218750000, 0.4921382812500000, 0.4999500000000000, 0.5038558593750000, 0.5077617187500001, 0.5155734375000001, 0.5194792968750001, 0.5233851562500000, 0.5311968750000000, 0.5351027343750000, 0.5390085937500000, 0.5468203125000000, 0.5507261718749999, 0.5546320312499999, 0.5624437499999999, 0.5663496093749999, 0.5702554687499999, 0.5780671874999999, 0.5819730468749998, 0.5858789062499998, 0.5936906249999999, 0.5975964843749999, 0.6015023437500000, 0.6093140625000000, 0.6171257812499999, 0.6249374999999999, 0.6327492187499999, 0.6405609374999999, 0.6483726562499998, 0.6561843749999999, 0.6639960937500000, 0.6718078125000000, 0.6796195312500000, 0.6874312499999999, 0.6952429687499999, 0.7030546874999999, 0.7108664062499999, 0.7186781249999999, 0.7264898437500000, 0.7343015625000000, 0.7421132812500000, 0.7499250000000000, 0.7577367187499999, 0.7655484374999999, 0.7733601562499999, 0.7811718750000000, 0.7967953125000000, 0.8046070312500000, 0.8124187500000000, 0.8280421874999999, 0.8436656250000000, 0.8592890625000000, 0.8749125000000000, 0.8905359374999999, 0.9061593750000000, 0.9217828125000000, 0.9374062500000000, 0.9530296874999999, 0.9686531250000000, 0.9842765625000001, 0.9999000000000000},
            {1.0000000000000000, 0.9863722354846718, 0.9728302914625573, 0.9593746663760025, 0.9460058553457373, 0.9327243500267448, 0.9195306384616720, 0.9064252049317680, 0.8934085298053444, 0.8804810893837401, 0.8676433557447891, 0.8548957965837793, 0.8422388750518993, 0.8296730495921694, 0.8171987737728497, 0.8048164961183321, 0.7925266599375076, 0.7803297031496172, 0.7682260581075937, 0.7562161514188885, 0.7443004037638048, 0.7324792297113419, 0.7207530375325616, 0.7091222290114993, 0.6975871992536220, 0.6861483364918756, 0.6748060218903217, 0.6635606293454070, 0.6524125252848821, 0.6413620684644084, 0.6304096097618819, 0.6195554919695199, 0.6088000495837416, 0.5981436085928983, 0.5875864862628897, 0.5771289909207324, 0.5667714217361259, 0.5565140685010805, 0.5463572114076717, 0.5363011208239941, 0.5263460570683784, 0.5164922701819609, 0.5067399996996803, 0.4970894744197970, 0.4875409121720225, 0.4780945195843618, 0.4687504918487690, 0.4595090124857318, 0.4503702531078961, 0.4413343731828539, 0.4324015197952250, 0.4235718274081641, 0.4148454176244367, 0.4062223989472100, 0.3977028665407162, 0.3892869019909489, 0.3809745730665618, 0.3727659334801464, 0.3646610226500773, 0.3566598654631112, 0.3487624720379444, 0.3409688374899389, 0.3332789416972278, 0.3256927490684353, 0.3182102083122357, 0.3108312522090023, 0.3035557973847949, 0.2963837440879469, 0.2893149759685262, 0.2823493598609451, 0.2754867455700140, 0.2687269656607331, 0.2620698352521363, 0.2555151518155016, 0.2490626949772592, 0.2427122263269350, 0.2364634892304797, 0.2303162086493385, 0.2242700909656334, 0.2183248238138348, 0.2124800759193101, 0.2067354969441493, 0.2010907173406732, 0.1955453482130422, 0.1900989811873930, 0.1847511882909380, 0.1795015218404725, 0.1743495143407423, 0.1692946783931360, 0.1643365066151675, 0.1594744715712300, 0.1547080257151031, 0.1500366013447057, 0.1454596105695904, 0.1409764452916832, 0.1365864771997744, 0.1322890577782739, 0.1280835183307423, 0.1239691700187191, 0.1199453039163616, 0.1160111910814163, 0.1121660826430409, 0.1084092099069897, 0.1047397844786803, 0.1011569984046453, 0.0976600243328760, 0.0942480156925495, 0.0909201068936277, 0.0876754135468033, 0.0845130327042539, 0.0814320431216554, 0.0784315055418833, 0.0755104630008171, 0.0726679411556396, 0.0699029486360000, 0.0672144774183838, 0.0646015032240054, 0.0620629859405075, 0.0595978700677187, 0.0548835464591316, 0.0526321551365505, 0.0504497991138933, 0.0462876811768008, 0.0443056334868979, 0.0423880508059735, 0.0387415913301510, 0.0370103467245532, 0.0353388329585336, 0.0321701760058212, 0.0306706061877744, 0.0292259155765864, 0.0264962676328686, 0.0252088508975590, 0.0239713963469736, 0.0216414426180239, 0.0205464797545958, 0.0194965536060355, 0.0175269131114324, 0.0166047611403072, 0.0157227726013228, 0.0140744826569557, 0.0133058010866269, 0.0125725245901170, 0.0112075419504258, 0.0105735445530476, 0.0099703716213212, 0.0088520741808416, 0.0078439631850624, 0.0069376330183263, 0.0061249849463173, 0.0053982510067575, 0.0047500153792389, 0.0041732328915481, 0.0036612443458231, 0.0032077883847811, 0.0028070096665912, 0.0024534631768201, 0.0021421145768412, 0.0018683365691125, 0.0016279013490432, 0.0014169693082048, 0.0012320742510395, 0.0010701054828087, 0.0009282872154859, 0.0008041558153707, 0.0006955354759978, 0.0006005129373914, 0.0005174118836798, 0.0004447676328020, 0.0003259036109122, 0.0002775996933430, 0.0002355435539335, 0.0001673007855705, 0.0001162634184505, 0.0000786096415829, 0.0000513326933705, 0.0000320465645687, 0.0000188465601028, 0.0000102063958331, 0.0000049003091178, 0.0000019430123421, 0.0000005427873405, 0.0000000644703628, 0.0000000000000156}
        };

        const double hyp_plus_xom = hypg_a_1_1_1_b_2_2_2_mapped_scaled(+xom / (+4.0 + xom)) * std::exp(+xom);
        const double hyp_xom_xsg = hypg_a_1_1_1_b_2_2_2_mapped(-xom * xsg / (-12 - xom * xsg));

        // calculate weights term by term
        Weights c = { 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0 };

        Weights c_a_star = {
            -0.16666666666666666 * ((-1 + (1 + xom + xom * xsg) * exp_xom_xsgplus) * (-6 + pi2) * xsgplus_m2),
            ((-6 + pi2) * (-1 + xsg + exp_xom_xsgplus * (1 + xom - xsg - xom * xsg2 + xom2 * (1 + 2 * xsg + xsg2))) * xsgplus_m3) / 6.,
            ((-6 + pi2) * (3 - 6 * xsg + 3 * xsg2 + exp_xom_xsgplus * (-3 - 3 * xom + 6 * xsg + 3 * xom * xsg + 3 * (-1 + xom) * xsg2 - 3 * xom * xsg3 + 6 * xom2 * (xsg + 2 * xsg2 + xsg3) - 2 * xom3 * (1 + 3 * xsg + 3 * xsg2 + xsg3))) * xsgplus_m4) / 18.,
            -0.05555555555555555 * ((-6 + pi2) * (3 - 9 * xsg + 9 * xsg2 - 3 * xsg3 + exp_xom_xsgplus * (-3 - 3 * xom + 9 * xsg + 6 * xom * xsg - xom4 - 4 * xsg * xom4 - 9 * xsg2 - 6 * xom4 * xsg2 + 3 * xsg3 - 6 * xom * xsg3 - 4 * xom4 * xsg3 + 3 * xom * xsg4 - xom4 * xsg4 - 3 * xom2 * (1 + 2 * xsg + 4 * xsg2 + 6 * xsg3 + 3 * xsg4) + 2 * xom3 * (1 + 6 * xsg + 12 * xsg2 + 10 * xsg3 + 3 * xsg4))) * xsgplus_m5),
            ((-6 + pi2) * (15 * (1 - 4 * xsg + 6 * xsg2 - 4 * xsg3 + xsg4) + exp_xom_xsgplus * (-15 - 15 * xom + 60 * xsg + 45 * xom * xsg + 10 * xom4 + 60 * xsg * xom4 - 2 * xom5 - 10 * xsg * xom5 - 90 * xsg2 - 30 * xom * xsg2 + 140 * xom4 * xsg2 - 20 * xom5 * xsg2 + 60 * xsg3 - 30 * xom * xsg3 + 160 * xom4 * xsg3 - 20 * xom5 * xsg3 - 15 * xsg4 + 45 * xom * xsg4 + 90 * xom4 * xsg4 - 10 * xom5 * xsg4 - 15 * xom * xsg5 + 20 * xom4 * xsg5 - 2 * xom5 * xsg5 + 60 * xom2 * (xsg + 2 * xsg2 + 2 * xsg3 + 2 * xsg4 + xsg5) - 20 * xom3 * (1 + 5 * xsg + 12 * xsg2 + 16 * xsg3 + 11 * xsg4 + 3 * xsg5))) * xsgplus_m6) / 90.,
            -0.003703703703703704 * ((-6 + pi2) * (-45 * (-1 + 5 * xsg - 10 * xsg2 + 10 * xsg3 - 5 * xsg4 + xsg5) + exp_xom_xsgplus * (-45 - 45 * xom + 225 * xsg + 180 * xom * xsg - 60 * xom4 - 390 * xsg * xom4 + 18 * xom5 + 120 * xsg * xom5 - 2 * xom6 - 12 * xsg * xom6 - 450 * xsg2 - 225 * xom * xsg2 - 1110 * xom4 * xsg2 + 330 * xom5 * xsg2 - 30 * xom6 * xsg2 + 450 * xsg3 - 1740 * xom4 * xsg3 + 480 * xom5 * xsg3 - 40 * xom6 * xsg3 - 225 * xsg4 + 225 * xom * xsg4 - 1560 * xom4 * xsg4 + 390 * xom5 * xsg4 - 30 * xom6 * xsg4 + 45 * xsg5 - 180 * xom * xsg5 - 750 * xom4 * xsg5 + 168 * xom5 * xsg5 - 12 * xom6 * xsg5 + 45 * xom * xsg6 - 150 * xom4 * xsg6 + 30 * xom5 * xsg6 - 2 * xom6 * xsg6 - 45 * xom2 * (1 + 2 * xsg + 11 * xsg2 + 20 * xsg3 + 15 * xsg4 + 10 * xsg5 + 5 * xsg6) + 60 * xom3 * (1 + 8 * xsg + 23 * xsg2 + 36 * xsg3 + 35 * xsg4 + 20 * xsg5 + 5 * xsg6))) * xsgplus_m7),
            ((-6 + pi2) * (-315 * (-1 + 6 * xsg - 15 * xsg2 + 20 * xsg3 - 15 * xsg4 + 6 * xsg5 - xsg6) + exp_xom_xsgplus * (-315 - 315 * xom + 1890 * xsg + 1575 * xom * xsg + 630 * xom4 + 5040 * xsg * xom4 - 294 * xom5 - 2226 * xsg * xom5 + 56 * xom6 + 420 * xsg * xom6 - 4 * xom7 - 28 * xsg * xom7 - 4725 * xsg2 - 2835 * xom * xsg2 + 17010 * xom4 * xsg2 - 7350 * xom5 * xsg2 + 1344 * xom6 * xsg2 - 84 * xom7 * xsg2 + 6300 * xsg3 + 1575 * xom * xsg3 + 32340 * xom4 * xsg3 - 13650 * xom5 * xsg3 + 2380 * xom6 * xsg3 - 140 * xom7 * xsg3 - 4725 * xsg4 + 1575 * xom * xsg4 + 38010 * xom4 * xsg4 - 15330 * xom5 * xsg4 + 2520 * xom6 * xsg4 - 140 * xom7 * xsg4 + 1890 * xsg5 - 2835 * xom * xsg5 + 27720 * xom4 * xsg5 - 10374 * xom5 * xsg5 + 1596 * xom6 * xsg5 - 84 * xom7 * xsg5 - 315 * xsg6 + 1575 * xom * xsg6 + 11550 * xom4 * xsg6 - 3906 * xom5 * xsg6 + 560 * xom6 * xsg6 - 28 * xom7 * xsg6 - 315 * xom * xsg7 + 2100 * xom4 * xsg7 - 630 * xom5 * xsg7 + 84 * xom6 * xsg7 - 4 * xom7 * xsg7 + 630 * xom2 * (3 * xsg + 6 * xsg2 + 13 * xsg3 + 20 * xsg4 + 13 * xsg5 + 6 * xsg6 + 3 * xsg7) - 210 * xom3 * (3 + 21 * xsg + 75 * xsg2 + 149 * xsg3 + 177 * xsg4 + 135 * xsg5 + 65 * xsg6 + 15 * xsg7))) * xsgplus_m8) / 1890.,
            -0.0005291005291005291 * ((-6 + pi2) * (-315 * (-1 + 7 * xsg - 21 * xsg2 + 35 * xsg3 - 35 * xsg4 + 21 * xsg5 - 7 * xsg6 + xsg7) + exp_xom_xsgplus * (-315 - 315 * xom + 2205 * xsg + 1890 * xom * xsg - 945 * xom4 - 8190 * xsg * xom4 + 546 * xom5 + 4788 * xsg * xom5 - 154 * xom6 - 1316 * xsg * xom6 + 20 * xom7 + 168 * xsg * xom7 - xom8 - 8 * xsg * xom8 - 6615 * xsg2 - 4410 * xom * xsg2 - 32130 * xom4 * xsg2 + 18396 * xom5 * xsg2 - 4956 * xom6 * xsg2 + 616 * xom7 * xsg2 - 28 * xom8 * xsg2 + 11025 * xsg3 + 4410 * xom * xsg3 - 72870 * xom4 * xsg3 + 40740 * xom5 * xsg3 - 10724 * xom6 * xsg3 + 1288 * xom7 * xsg3 - 56 * xom8 * xsg3 - 11025 * xsg4 - 104580 * xom4 * xsg4 + 57120 * xom5 * xsg4 - 14560 * xom6 * xsg4 + 1680 * xom7 * xsg4 - 70 * xom8 * xsg4 + 6615 * xsg5 - 4410 * xom * xsg5 - 98490 * xom4 * xsg5 + 51996 * xom5 * xsg5 - 12684 * xom6 * xsg5 + 1400 * xom7 * xsg5 - 56 * xom8 * xsg5 - 2205 * xsg6 + 4410 * xom * xsg6 - 60270 * xom4 * xsg6 + 29988 * xom5 * xsg6 - 6916 * xom6 * xsg6 + 728 * xom7 * xsg6 - 28 * xom8 * xsg6 + 315 * xsg7 - 1890 * xom * xsg7 - 22050 * xom4 * xsg7 + 9996 * xom5 * xsg7 - 2156 * xom6 * xsg7 + 216 * xom7 * xsg7 - 8 * xom8 * xsg7 + 315 * xom * xsg8 - 3675 * xom4 * xsg8 + 1470 * xom5 * xsg8 - 294 * xom6 * xsg8 + 28 * xom7 * xsg8 - xom8 * xsg8 - 315 * xom2 * (1 + 2 * xsg + 22 * xsg2 + 42 * xsg3 + 56 * xsg4 + 70 * xsg5 + 42 * xsg6 + 14 * xsg7 + 7 * xsg8) + 210 * xom3 * (3 + 30 * xsg + 114 * xsg2 + 262 * xsg3 + 392 * xsg4 + 378 * xsg5 + 238 * xsg6 + 98 * xsg7 + 21 * xsg8))) * xsgplus_m9),
            ((-6 + pi2) * (-2835 * (-1 + 8 * xsg - 28 * xsg2 + 56 * xsg3 - 70 * xsg4 + 56 * xsg5 - 28 * xsg6 + 8 * xsg7 - xsg8) + exp_xom_xsgplus * (-2835 - 2835 * xom + 22680 * xsg + 19845 * xom * xsg + 11340 * xom4 + 113400 * xsg * xom4 - 8316 * xom5 - 80892 * xsg * xom5 + 3024 * xom6 + 29232 * xsg * xom6 - 576 * xom7 - 5472 * xsg * xom7 + 54 * xom8 + 504 * xsg * xom8 - 2 * xom9 - 18 * xsg * xom9 - 79380 * xsg2 - 56700 * xom * xsg2 + 498960 * xom4 * xsg2 - 353808 * xom5 * xsg2 + 126000 * xom6 * xsg2 - 23184 * xom7 * xsg2 + 2088 * xom8 * xsg2 - 72 * xom9 * xsg2 + 158760 * xsg3 + 79380 * xom * xsg3 + 1300320 * xom4 * xsg3 - 910224 * xom5 * xsg3 + 318528 * xom6 * xsg3 - 57456 * xom7 * xsg3 + 5040 * xom8 * xsg3 - 168 * xom9 * xsg3 - 198450 * xsg4 - 39690 * xom * xsg4 + 2215080 * xom4 * xsg4 - 1519560 * xom5 * xsg4 + 521136 * xom6 * xsg4 - 91728 * xom7 * xsg4 + 7812 * xom8 * xsg4 - 252 * xom9 * xsg4 + 158760 * xsg5 - 39690 * xom * xsg5 + 2555280 * xom4 * xsg5 - 1713096 * xom5 * xsg5 + 572544 * xom6 * xsg5 - 97776 * xom7 * xsg5 + 8064 * xom8 * xsg5 - 252 * xom9 * xsg5 - 79380 * xsg6 + 79380 * xom * xsg6 + 2010960 * xom4 * xsg6 - 1309392 * xom5 * xsg6 + 422352 * xom6 * xsg6 - 69552 * xom7 * xsg6 + 5544 * xom8 * xsg6 - 168 * xom9 * xsg6 + 22680 * xsg7 - 56700 * xom * xsg7 + 1058400 * xom4 * xsg7 - 656208 * xom5 * xsg7 + 201600 * xom6 * xsg7 - 31824 * xom7 * xsg7 + 2448 * xom8 * xsg7 - 72 * xom9 * xsg7 - 2835 * xsg8 + 19845 * xom * xsg8 + 343980 * xom4 * xsg8 - 195804 * xom5 * xsg8 + 56448 * xom6 * xsg8 - 8496 * xom7 * xsg8 + 630 * xom8 * xsg8 - 18 * xom9 * xsg8 - 2835 * xom * xsg9 + 52920 * xom4 * xsg9 - 26460 * xom5 * xsg9 + 7056 * xom6 * xsg9 - 1008 * xom7 * xsg9 + 72 * xom8 * xsg9 - 2 * xom9 * xsg9 + 22680 * xom2 * (xsg + 2 * xsg2 + 8 * xsg3 + 14 * xsg4 + 14 * xsg5 + 14 * xsg6 + 8 * xsg7 + 2 * xsg8 + xsg9) - 7560 * xom3 * (1 + 9 * xsg + 42 * xsg2 + 110 * xsg3 + 188 * xsg4 + 224 * xsg5 + 182 * xsg6 + 98 * xsg7 + 35 * xsg8 + 7 * xsg9))) * xsgplus_m10) / 17010.
        };
        c = c + c_a_star;

        const Weights c_b = {
            (3 + 2 * eulergamma + 2 * eulergamma2 - expint_ei_xom_xsgplus - 2 * eulergamma * expint_ei_xom_xsgplus - 4 * xom * hyp_xom_xsgplus - 4 * xom * xsg * hyp_xom_xsgplus + 2 * L + 2 * expint_ei_xom_xsgplus * L - exp_xom_xsgplus * (3 + 2 * L + eulergamma * (1 + 3 * xom * (1 + xsg) + 2 * (1 + xom + xom * xsg) * L)) + 2 * log2_xom + 2 * (1 + xom + xom * xsg) * exp_xom_xsgplus * log2_xom + 4 * (eulergamma - expint_ei_xom_xsgplus) * log_xom + exp_xom_xsgplus * (1 - 3 * xom - 3 * xom * xsg + 2 * (1 + xom + xom * xsg) * eulergamma - 2 * (1 + xom + xom * xsg) * L) * log_xom + (1 + 2 * eulergamma - 2 * L) * log_xsgplus + 4 * log_xom * log_xsgplus) * xsgplus_m2,
            (-1 + 4 * xsg - 2 * eulergamma + 4 * xsg * eulergamma - 2 * eulergamma2 + 2 * xsg * eulergamma2 + 3 * expint_ei_xom_xsgplus - 3 * xsg * expint_ei_xom_xsgplus + 2 * eulergamma * expint_ei_xom_xsgplus - 2 * xsg * eulergamma * expint_ei_xom_xsgplus + 4 * xom * hyp_xom_xsgplus - 2 * L + 4 * xsg * L - 2 * expint_ei_xom_xsgplus * L + 2 * xsg * expint_ei_xom_xsgplus * L + 2 * (-1 + xsg) * log2_xom + 4 * (-1 + xsg) * (eulergamma - expint_ei_xom_xsgplus) * log_xom + (-1 + xsg) * (3 + 2 * eulergamma - 2 * L) * log_xsgplus + 4 * (-1 + xsg) * log_xom * log_xsgplus - 4 * xom * hyp_xom_xsgplus * xsg2 + 2 * exp_xom_xsgplus * log2_xom * (-1 - xom + xsg + xom * xsg2 - xom2 * (1 + 2 * xsg + xsg2)) + exp_xom_xsgplus * log_xom * (-3 + xom + 3 * xsg + 3 * xom2 + 6 * xsg * xom2 - xom * xsg2 + 3 * xom2 * xsg2 + 2 * eulergamma * (-1 - xom + xsg + xom * xsg2 - xom2 * (1 + 2 * xsg + xsg2)) + 2 * L * (1 + xom - xsg - xom * xsg2 + xom2 * (1 + 2 * xsg + xsg2))) - exp_xom_xsgplus * (-1 + 4 * xsg + 3 * xom * xsg + 3 * xom * xsg2 + 2 * L * (-1 + (2 + xom) * xsg + xom * xsg2) + eulergamma * (1 - xom + xsg + 2 * xom * xsg + 3 * xom * xsg2 - 3 * xom2 * (1 + 2 * xsg + xsg2) + 2 * L * (-1 - xom + xsg + xom * xsg2 - xom2 * (1 + 2 * xsg + xsg2))))) * xsgplus_m3,
            ((1 - 4 * xsg + 6 * eulergamma - 24 * xsg * eulergamma + 6 * eulergamma2 - 12 * xsg * eulergamma2 - 11 * expint_ei_xom_xsgplus + 30 * xsg * expint_ei_xom_xsgplus - 6 * eulergamma * expint_ei_xom_xsgplus + 12 * xsg * eulergamma * expint_ei_xom_xsgplus - 12 * xom * hyp_xom_xsgplus + 12 * xom * xsg * hyp_xom_xsgplus + 6 * L - 24 * xsg * L + 6 * expint_ei_xom_xsgplus * L - 12 * xsg * expint_ei_xom_xsgplus * L - 12 * (eulergamma - expint_ei_xom_xsgplus) * log_xom * (-1 + 2 * xsg - xsg2) + 13 * xsg2 + 14 * eulergamma * xsg2 + 6 * eulergamma2 * xsg2 - 11 * expint_ei_xom_xsgplus * xsg2 - 6 * eulergamma * expint_ei_xom_xsgplus * xsg2 + 12 * xom * hyp_xom_xsgplus * xsg2 + 14 * L * xsg2 + 6 * expint_ei_xom_xsgplus * L * xsg2 + 6 * log2_xom * (1 - 2 * xsg + xsg2) + 12 * log_xom * log_xsgplus * (1 - 2 * xsg + xsg2) + log_xsgplus * (11 - 30 * xsg + 6 * L * (-1 + 2 * xsg - xsg2) + 11 * xsg2 + 6 * eulergamma * (1 - 2 * xsg + xsg2)) - 12 * xom * hyp_xom_xsgplus * xsg3 - 2 * exp_xom_xsgplus * log2_xom * (-3 - 3 * xom + 6 * xsg + 3 * xom * xsg + 3 * (-1 + xom) * xsg2 - 3 * xom * xsg3 + 6 * xom2 * (xsg + 2 * xsg2 + xsg3) - 2 * xom3 * (1 + 3 * xsg + 3 * xsg2 + xsg3)) - exp_xom_xsgplus * log_xom * (-11 + xom + 30 * xsg + 7 * xom * xsg - 10 * xsg * xom2 + 6 * xom3 + 18 * xsg * xom3 - 11 * xsg2 + 7 * xom * xsg2 - 20 * xom2 * xsg2 + 18 * xom3 * xsg2 + xom * xsg3 - 10 * xom2 * xsg3 + 6 * xom3 * xsg3 + 2 * eulergamma * (-3 - 3 * xom + 6 * xsg + 3 * xom * xsg + 3 * (-1 + xom) * xsg2 - 3 * xom * xsg3 + 6 * xom2 * (xsg + 2 * xsg2 + xsg3) - 2 * xom3 * (1 + 3 * xsg + 3 * xsg2 + xsg3)) - 2 * L * (-3 - 3 * xom + 6 * xsg + 3 * xom * xsg + 3 * (-1 + xom) * xsg2 - 3 * xom * xsg3 + 6 * xom2 * (xsg + 2 * xsg2 + xsg3) - 2 * xom3 * (1 + 3 * xsg + 3 * xsg2 + xsg3))) + exp_xom_xsgplus * (-1 + 4 * xsg + 10 * xom * xsg + 6 * xom2 + 24 * xsg * xom2 - 13 * xsg2 - 2 * xom * xsg2 + 30 * xom2 * xsg2 - 12 * xom * xsg3 + 12 * xom2 * xsg3 + 2 * L * (-3 + 12 * xsg + 6 * xom * xsg + (-7 + 2 * xom) * xsg2 - 4 * xom * xsg3 + 2 * xom2 * (1 + 4 * xsg + 5 * xsg2 + 2 * xsg3)) + eulergamma * (5 - xom - 6 * xsg + 5 * xom * xsg + 4 * xom2 + 26 * xsg * xom2 - 6 * xom3 - 18 * xsg * xom3 - 3 * xsg2 - 3 * xom * xsg2 + 40 * xom2 * xsg2 - 18 * xom3 * xsg2 - 9 * xom * xsg3 + 18 * xom2 * xsg3 - 6 * xom3 * xsg3 + 2 * L * (-3 - 3 * xom + 6 * xsg + 3 * xom * xsg + 3 * (-1 + xom) * xsg2 - 3 * xom * xsg3 + 6 * xom2 * (xsg + 2 * xsg2 + xsg3) - 2 * xom3 * (1 + 3 * xsg + 3 * xsg2 + xsg3))))) * xsgplus_m4) / 3.,
            ((1 - 6 * xsg - 6 * eulergamma + 36 * xsg * eulergamma - 6 * eulergamma2 + 18 * xsg * eulergamma2 + 13 * expint_ei_xom_xsgplus - 51 * xsg * expint_ei_xom_xsgplus + 6 * eulergamma * expint_ei_xom_xsgplus - 18 * xsg * eulergamma * expint_ei_xom_xsgplus + 12 * xom * hyp_xom_xsgplus - 24 * xom * xsg * hyp_xom_xsgplus - 6 * L + 36 * xsg * L - 6 * expint_ei_xom_xsgplus * L + 18 * xsg * expint_ei_xom_xsgplus * L - 3 * xsg2 - 42 * eulergamma * xsg2 - 18 * eulergamma2 * xsg2 + 51 * expint_ei_xom_xsgplus * xsg2 + 18 * eulergamma * expint_ei_xom_xsgplus * xsg2 - 42 * L * xsg2 - 18 * expint_ei_xom_xsgplus * L * xsg2 + 14 * xsg3 + 16 * eulergamma * xsg3 + 6 * eulergamma2 * xsg3 - 13 * expint_ei_xom_xsgplus * xsg3 - 6 * eulergamma * expint_ei_xom_xsgplus * xsg3 + 24 * xom * hyp_xom_xsgplus * xsg3 + 16 * L * xsg3 + 6 * expint_ei_xom_xsgplus * L * xsg3 + 6 * log2_xom * (-1 + 3 * xsg - 3 * xsg2 + xsg3) + 12 * (eulergamma - expint_ei_xom_xsgplus) * log_xom * (-1 + 3 * xsg - 3 * xsg2 + xsg3) + 12 * log_xom * log_xsgplus * (-1 + 3 * xsg - 3 * xsg2 + xsg3) + log_xsgplus * (-13 + 51 * xsg - 51 * xsg2 + 13 * xsg3 + 6 * eulergamma * (-1 + 3 * xsg - 3 * xsg2 + xsg3) - 6 * L * (-1 + 3 * xsg - 3 * xsg2 + xsg3)) - 12 * xom * hyp_xom_xsgplus * xsg4 + 2 * exp_xom_xsgplus * log2_xom * (-3 - 3 * xom + 9 * xsg + 6 * xom * xsg - xom4 - 4 * xsg * xom4 - 9 * xsg2 - 6 * xom4 * xsg2 + 3 * xsg3 - 6 * xom * xsg3 - 4 * xom4 * xsg3 + 3 * xom * xsg4 - xom4 * xsg4 - 3 * xom2 * (1 + 2 * xsg + 4 * xsg2 + 6 * xsg3 + 3 * xsg4) + 2 * xom3 * (1 + 6 * xsg + 12 * xsg2 + 10 * xsg3 + 3 * xsg4)) + exp_xom_xsgplus * log_xom * (-13 - xom + 51 * xsg + 14 * xom * xsg + 5 * xom2 + 16 * xsg * xom2 - 4 * xom3 - 24 * xsg * xom3 + 3 * xom4 + 12 * xsg * xom4 - 51 * xsg2 + 26 * xom2 * xsg2 - 48 * xom3 * xsg2 + 18 * xom4 * xsg2 + 13 * xsg3 - 14 * xom * xsg3 + 24 * xom2 * xsg3 - 40 * xom3 * xsg3 + 12 * xom4 * xsg3 + xom * xsg4 + 9 * xom2 * xsg4 - 12 * xom3 * xsg4 + 3 * xom4 * xsg4 + 2 * eulergamma * (-3 - 3 * xom + 9 * xsg + 6 * xom * xsg - xom4 - 4 * xsg * xom4 - 9 * xsg2 - 6 * xom4 * xsg2 + 3 * xsg3 - 6 * xom * xsg3 - 4 * xom4 * xsg3 + 3 * xom * xsg4 - xom4 * xsg4 - 3 * xom2 * (1 + 2 * xsg + 4 * xsg2 + 6 * xsg3 + 3 * xsg4) + 2 * xom3 * (1 + 6 * xsg + 12 * xsg2 + 10 * xsg3 + 3 * xsg4)) - 2 * L * (-3 - 3 * xom + 9 * xsg + 6 * xom * xsg - xom4 - 4 * xsg * xom4 - 9 * xsg2 - 6 * xom4 * xsg2 + 3 * xsg3 - 6 * xom * xsg3 - 4 * xom4 * xsg3 + 3 * xom * xsg4 - xom4 * xsg4 - 3 * xom2 * (1 + 2 * xsg + 4 * xsg2 + 6 * xsg3 + 3 * xsg4) + 2 * xom3 * (1 + 6 * xsg + 12 * xsg2 + 10 * xsg3 + 3 * xsg4))) + exp_xom_xsgplus * (-1 + 6 * xsg - 9 * xom * xsg + 5 * xom2 + 25 * xsg * xom2 - 6 * xom3 - 27 * xsg * xom3 + 3 * xsg2 + 9 * xom * xsg2 + 62 * xom2 * xsg2 - 45 * xom3 * xsg2 - 14 * xsg3 + 3 * xom * xsg3 + 69 * xom2 * xsg3 - 33 * xom3 * xsg3 - 15 * xom * xsg4 + 27 * xom2 * xsg4 - 9 * xom3 * xsg4 - 2 * L * (-3 + 18 * xsg + 9 * xom * xsg - 21 * xsg2 - 3 * xom * xsg2 + 8 * xsg3 - 7 * xom * xsg3 + 5 * xom * xsg4 + xom3 * (2 + 9 * xsg + 15 * xsg2 + 11 * xsg3 + 3 * xsg4) - xom2 * (1 + 5 * xsg + 16 * xsg2 + 21 * xsg3 + 9 * xsg4)) + eulergamma * (-7 - xom + 15 * xsg - 4 * xom * xsg - 8 * xom3 - 42 * xsg * xom3 + 3 * xom4 + 12 * xsg * xom4 - 9 * xsg2 + 6 * xom * xsg2 - 78 * xom3 * xsg2 + 18 * xom4 * xsg2 - 3 * xsg3 - 62 * xom3 * xsg3 + 12 * xom4 * xsg3 - 9 * xom * xsg4 - 18 * xom3 * xsg4 + 3 * xom4 * xsg4 + xom2 * (7 + 26 * xsg + 58 * xsg2 + 66 * xsg3 + 27 * xsg4) - 2 * L * (-3 - 3 * xom + 9 * xsg + 6 * xom * xsg - xom4 - 4 * xsg * xom4 - 9 * xsg2 - 6 * xom4 * xsg2 + 3 * xsg3 - 6 * xom * xsg3 - 4 * xom4 * xsg3 + 3 * xom * xsg4 - xom4 * xsg4 - 3 * xom2 * (1 + 2 * xsg + 4 * xsg2 + 6 * xsg3 + 3 * xsg4) + 2 * xom3 * (1 + 6 * xsg + 12 * xsg2 + 10 * xsg3 + 3 * xsg4))))) * xsgplus_m5) / 3.,
            ((-11 + 112 * xsg + 30 * eulergamma - 240 * xsg * eulergamma + 30 * eulergamma2 - 120 * xsg * eulergamma2 - 71 * expint_ei_xom_xsgplus + 380 * xsg * expint_ei_xom_xsgplus - 30 * eulergamma * expint_ei_xom_xsgplus + 120 * xsg * eulergamma * expint_ei_xom_xsgplus - 60 * xom * hyp_xom_xsgplus + 180 * xom * xsg * hyp_xom_xsgplus + 30 * L - 240 * xsg * L + 30 * expint_ei_xom_xsgplus * L - 120 * xsg * expint_ei_xom_xsgplus * L - 114 * xsg2 + 420 * eulergamma * xsg2 + 180 * eulergamma2 * xsg2 - 570 * expint_ei_xom_xsgplus * xsg2 - 180 * eulergamma * expint_ei_xom_xsgplus * xsg2 - 120 * xom * hyp_xom_xsgplus * xsg2 + 420 * L * xsg2 + 180 * expint_ei_xom_xsgplus * L * xsg2 - 8 * xsg3 - 320 * eulergamma * xsg3 - 120 * eulergamma2 * xsg3 + 380 * expint_ei_xom_xsgplus * xsg3 + 120 * eulergamma * expint_ei_xom_xsgplus * xsg3 - 120 * xom * hyp_xom_xsgplus * xsg3 - 320 * L * xsg3 - 120 * expint_ei_xom_xsgplus * L * xsg3 - 30 * log2_xom * (-1 + 4 * xsg - 6 * xsg2 + 4 * xsg3 - xsg4) - 60 * (eulergamma - expint_ei_xom_xsgplus) * log_xom * (-1 + 4 * xsg - 6 * xsg2 + 4 * xsg3 - xsg4) - 60 * log_xom * log_xsgplus * (-1 + 4 * xsg - 6 * xsg2 + 4 * xsg3 - xsg4) + 73 * xsg4 + 86 * eulergamma * xsg4 + 30 * eulergamma2 * xsg4 - 71 * expint_ei_xom_xsgplus * xsg4 - 30 * eulergamma * expint_ei_xom_xsgplus * xsg4 + 180 * xom * hyp_xom_xsgplus * xsg4 + 86 * L * xsg4 + 30 * expint_ei_xom_xsgplus * L * xsg4 + log_xsgplus * (71 - 380 * xsg + 570 * xsg2 - 380 * xsg3 - 30 * eulergamma * (-1 + 4 * xsg - 6 * xsg2 + 4 * xsg3 - xsg4) + 30 * L * (-1 + 4 * xsg - 6 * xsg2 + 4 * xsg3 - xsg4) + 71 * xsg4) - 60 * xom * hyp_xom_xsgplus * xsg5 - 2 * exp_xom_xsgplus * log2_xom * (-15 - 15 * xom + 60 * xsg + 45 * xom * xsg + 10 * xom4 + 60 * xsg * xom4 - 2 * xom5 - 10 * xsg * xom5 - 90 * xsg2 - 30 * xom * xsg2 + 140 * xom4 * xsg2 - 20 * xom5 * xsg2 + 60 * xsg3 - 30 * xom * xsg3 + 160 * xom4 * xsg3 - 20 * xom5 * xsg3 - 15 * xsg4 + 45 * xom * xsg4 + 90 * xom4 * xsg4 - 10 * xom5 * xsg4 - 15 * xom * xsg5 + 20 * xom4 * xsg5 - 2 * xom5 * xsg5 + 60 * xom2 * (xsg + 2 * xsg2 + 2 * xsg3 + 2 * xsg4 + xsg5) - 20 * xom3 * (1 + 5 * xsg + 12 * xsg2 + 16 * xsg3 + 11 * xsg4 + 3 * xsg5)) + exp_xom_xsgplus * log_xom * (71 + 11 * xom - 380 * xsg - 129 * xom * xsg + 28 * xsg * xom2 - 36 * xom3 - 180 * xsg * xom3 + 22 * xom4 + 132 * xsg * xom4 - 6 * xom5 - 30 * xsg * xom5 + 570 * xsg2 + 70 * xom * xsg2 + 128 * xom2 * xsg2 - 408 * xom3 * xsg2 + 308 * xom4 * xsg2 - 60 * xom5 * xsg2 - 380 * xsg3 + 70 * xom * xsg3 + 200 * xom2 * xsg3 - 504 * xom3 * xsg3 + 352 * xom4 * xsg3 - 60 * xom5 * xsg3 + 71 * xsg4 - 129 * xom * xsg4 + 128 * xom2 * xsg4 - 324 * xom3 * xsg4 + 198 * xom4 * xsg4 - 30 * xom5 * xsg4 + 11 * xom * xsg5 + 28 * xom2 * xsg5 - 84 * xom3 * xsg5 + 44 * xom4 * xsg5 - 6 * xom5 * xsg5 - 2 * eulergamma * (-15 - 15 * xom + 60 * xsg + 45 * xom * xsg + 10 * xom4 + 60 * xsg * xom4 - 2 * xom5 - 10 * xsg * xom5 - 90 * xsg2 - 30 * xom * xsg2 + 140 * xom4 * xsg2 - 20 * xom5 * xsg2 + 60 * xsg3 - 30 * xom * xsg3 + 160 * xom4 * xsg3 - 20 * xom5 * xsg3 - 15 * xsg4 + 45 * xom * xsg4 + 90 * xom4 * xsg4 - 10 * xom5 * xsg4 - 15 * xom * xsg5 + 20 * xom4 * xsg5 - 2 * xom5 * xsg5 + 60 * xom2 * (xsg + 2 * xsg2 + 2 * xsg3 + 2 * xsg4 + xsg5) - 20 * xom3 * (1 + 5 * xsg + 12 * xsg2 + 16 * xsg3 + 11 * xsg4 + 3 * xsg5)) + 2 * L * (-15 - 15 * xom + 60 * xsg + 45 * xom * xsg + 10 * xom4 + 60 * xsg * xom4 - 2 * xom5 - 10 * xsg * xom5 - 90 * xsg2 - 30 * xom * xsg2 + 140 * xom4 * xsg2 - 20 * xom5 * xsg2 + 60 * xsg3 - 30 * xom * xsg3 + 160 * xom4 * xsg3 - 20 * xom5 * xsg3 - 15 * xsg4 + 45 * xom * xsg4 + 90 * xom4 * xsg4 - 10 * xom5 * xsg4 - 15 * xom * xsg5 + 20 * xom4 * xsg5 - 2 * xom5 * xsg5 + 60 * xom2 * (xsg + 2 * xsg2 + 2 * xsg3 + 2 * xsg4 + xsg5) - 20 * xom3 * (1 + 5 * xsg + 12 * xsg2 + 16 * xsg3 + 11 * xsg4 + 3 * xsg5))) + exp_xom_xsgplus * (11 - 112 * xsg + 28 * xom * xsg + 42 * xom2 + 252 * xsg * xom2 - 50 * xom3 - 298 * xsg * xom3 + 18 * xom4 + 96 * xsg * xom4 + 114 * xsg2 - 68 * xom * xsg2 + 546 * xom2 * xsg2 - 738 * xom3 * xsg2 + 204 * xom4 * xsg2 + 8 * xsg3 + 52 * xom * xsg3 + 732 * xom2 * xsg3 - 926 * xom3 * xsg3 + 216 * xom4 * xsg3 - 73 * xsg4 + 64 * xom * xsg4 + 624 * xom2 * xsg4 - 580 * xom3 * xsg4 + 114 * xom4 * xsg4 - 84 * xom * xsg5 + 228 * xom2 * xsg5 - 144 * xom3 * xsg5 + 24 * xom4 * xsg5 + 2 * L * (-15 + 120 * xsg + 60 * xom * xsg + 6 * xom4 + 32 * xsg * xom4 - 210 * xsg2 - 60 * xom * xsg2 + 68 * xom4 * xsg2 + 160 * xsg3 - 20 * xom * xsg3 + 72 * xom4 * xsg3 - 43 * xsg4 + 72 * xom * xsg4 + 38 * xom4 * xsg4 - 28 * xom * xsg5 + 8 * xom4 * xsg5 - 2 * xom3 * (7 + 43 * xsg + 111 * xsg2 + 145 * xsg3 + 94 * xsg4 + 24 * xsg5) + 2 * xom2 * (7 + 42 * xsg + 75 * xsg2 + 90 * xsg3 + 88 * xsg4 + 38 * xsg5)) + eulergamma * (41 + 11 * xom - 140 * xsg - 9 * xom * xsg + 28 * xom2 + 196 * xsg * xom2 - 64 * xom3 - 352 * xsg * xom3 + 34 * xom4 + 196 * xsg * xom4 - 6 * xom5 - 30 * xsg * xom5 + 150 * xsg2 - 50 * xom * xsg2 + 428 * xom2 * xsg2 - 852 * xom3 * xsg2 + 444 * xom4 * xsg2 - 60 * xom5 * xsg2 - 60 * xsg3 + 30 * xom * xsg3 + 560 * xom2 * xsg3 - 1084 * xom3 * xsg3 + 496 * xom4 * xsg3 - 60 * xom5 * xsg3 - 15 * xsg4 + 15 * xom * xsg4 + 480 * xom2 * xsg4 - 700 * xom3 * xsg4 + 274 * xom4 * xsg4 - 30 * xom5 * xsg4 - 45 * xom * xsg5 + 180 * xom2 * xsg5 - 180 * xom3 * xsg5 + 60 * xom4 * xsg5 - 6 * xom5 * xsg5 + 2 * L * (-15 - 15 * xom + 60 * xsg + 45 * xom * xsg + 10 * xom4 + 60 * xsg * xom4 - 2 * xom5 - 10 * xsg * xom5 - 90 * xsg2 - 30 * xom * xsg2 + 140 * xom4 * xsg2 - 20 * xom5 * xsg2 + 60 * xsg3 - 30 * xom * xsg3 + 160 * xom4 * xsg3 - 20 * xom5 * xsg3 - 15 * xsg4 + 45 * xom * xsg4 + 90 * xom4 * xsg4 - 10 * xom5 * xsg4 - 15 * xom * xsg5 + 20 * xom4 * xsg5 - 2 * xom5 * xsg5 + 60 * xom2 * (xsg + 2 * xsg2 + 2 * xsg3 + 2 * xsg4 + xsg5) - 20 * xom3 * (1 + 5 * xsg + 12 * xsg2 + 16 * xsg3 + 11 * xsg4 + 3 * xsg5))))) * xsgplus_m6) / 15.,
            (1.1333333333333333 - (40 * xsg) / 3. - 2 * eulergamma + 20 * xsg * eulergamma - 2 * eulergamma2 + 10 * xsg * eulergamma2 + (77 * expint_ei_xom_xsgplus) / 15. - (101 * xsg * expint_ei_xom_xsgplus) / 3. + 2 * eulergamma * expint_ei_xom_xsgplus - 10 * xsg * eulergamma * expint_ei_xom_xsgplus + 4 * xom * hyp_xom_xsgplus - 16 * xom * xsg * hyp_xom_xsgplus - 2 * L + 20 * xsg * L - 2 * expint_ei_xom_xsgplus * L + 10 * xsg * expint_ei_xom_xsgplus * L + (242 * xsg2) / 9. - (140 * eulergamma * xsg2) / 3. - 20 * eulergamma2 * xsg2 + 70 * expint_ei_xom_xsgplus * xsg2 + 20 * eulergamma * expint_ei_xom_xsgplus * xsg2 + 20 * xom * hyp_xom_xsgplus * xsg2 - (140 * L * xsg2) / 3. - 20 * expint_ei_xom_xsgplus * L * xsg2 - (152 * xsg3) / 9. + (160 * eulergamma * xsg3) / 3. + 20 * eulergamma2 * xsg3 - 70 * expint_ei_xom_xsgplus * xsg3 - 20 * eulergamma * expint_ei_xom_xsgplus * xsg3 + (160 * L * xsg3) / 3. + 20 * expint_ei_xom_xsgplus * L * xsg3 + xsg4 / 3. - (86 * eulergamma * xsg4) / 3. - 10 * eulergamma2 * xsg4 + (101 * expint_ei_xom_xsgplus * xsg4) / 3. + 10 * eulergamma * expint_ei_xom_xsgplus * xsg4 - 20 * xom * hyp_xom_xsgplus * xsg4 - (86 * L * xsg4) / 3. - 10 * expint_ei_xom_xsgplus * L * xsg4 + (76 * xsg5) / 15. + (92 * eulergamma * xsg5) / 15. + 2 * eulergamma2 * xsg5 - (77 * expint_ei_xom_xsgplus * xsg5) / 15. - 2 * eulergamma * expint_ei_xom_xsgplus * xsg5 + 16 * xom * hyp_xom_xsgplus * xsg5 + (92 * L * xsg5) / 15. + 2 * expint_ei_xom_xsgplus * L * xsg5 + 2 * log2_xom * (-1 + 5 * xsg - 10 * xsg2 + 10 * xsg3 - 5 * xsg4 + xsg5) + 4 * (eulergamma - expint_ei_xom_xsgplus) * log_xom * (-1 + 5 * xsg - 10 * xsg2 + 10 * xsg3 - 5 * xsg4 + xsg5) + 4 * log_xom * log_xsgplus * (-1 + 5 * xsg - 10 * xsg2 + 10 * xsg3 - 5 * xsg4 + xsg5) + (log_xsgplus * (-77 + 505 * xsg - 1050 * xsg2 + 1050 * xsg3 - 505 * xsg4 + 77 * xsg5 + 30 * eulergamma * (-1 + 5 * xsg - 10 * xsg2 + 10 * xsg3 - 5 * xsg4 + xsg5) - 30 * L * (-1 + 5 * xsg - 10 * xsg2 + 10 * xsg3 - 5 * xsg4 + xsg5))) / 15. - 4 * xom * hyp_xom_xsgplus * xsg6 + (2 * exp_xom_xsgplus * log2_xom * (-45 - 45 * xom + 225 * xsg + 180 * xom * xsg - 60 * xom4 - 390 * xsg * xom4 + 18 * xom5 + 120 * xsg * xom5 - 2 * xom6 - 12 * xsg * xom6 - 450 * xsg2 - 225 * xom * xsg2 - 1110 * xom4 * xsg2 + 330 * xom5 * xsg2 - 30 * xom6 * xsg2 + 450 * xsg3 - 1740 * xom4 * xsg3 + 480 * xom5 * xsg3 - 40 * xom6 * xsg3 - 225 * xsg4 + 225 * xom * xsg4 - 1560 * xom4 * xsg4 + 390 * xom5 * xsg4 - 30 * xom6 * xsg4 + 45 * xsg5 - 180 * xom * xsg5 - 750 * xom4 * xsg5 + 168 * xom5 * xsg5 - 12 * xom6 * xsg5 + 45 * xom * xsg6 - 150 * xom4 * xsg6 + 30 * xom5 * xsg6 - 2 * xom6 * xsg6 - 45 * xom2 * (1 + 2 * xsg + 11 * xsg2 + 20 * xsg3 + 15 * xsg4 + 10 * xsg5 + 5 * xsg6) + 60 * xom3 * (1 + 8 * xsg + 23 * xsg2 + 36 * xsg3 + 35 * xsg4 + 20 * xsg5 + 5 * xsg6))) / 45. + (exp_xom_xsgplus * log_xom * (-231 - 51 * xom + 1515 * xsg + 564 * xom * xsg + 39 * xom2 + 258 * xsg * xom2 - 72 * xom3 - 576 * xsg * xom3 + 112 * xom4 + 718 * xsg * xom4 - 42 * xom5 - 280 * xsg * xom5 + 6 * xom6 + 36 * xsg * xom6 - 3150 * xsg2 - 735 * xom * xsg2 + 369 * xom2 * xsg2 - 1736 * xom3 * xsg2 + 2002 * xom4 * xsg2 - 770 * xom5 * xsg2 + 90 * xom6 * xsg2 + 3150 * xsg3 + 540 * xom2 * xsg3 - 2752 * xom3 * xsg3 + 3068 * xom4 * xsg3 - 1120 * xom5 * xsg3 + 120 * xom6 * xsg3 - 1515 * xsg4 + 735 * xom * xsg4 + 825 * xom2 * xsg4 - 2520 * xom3 * xsg4 + 2692 * xom4 * xsg4 - 910 * xom5 * xsg4 + 90 * xom6 * xsg4 + 231 * xsg5 - 564 * xom * xsg5 + 450 * xom2 * xsg5 - 1280 * xom3 * xsg5 + 1270 * xom4 * xsg5 - 392 * xom5 * xsg5 + 36 * xom6 * xsg5 + 51 * xom * xsg6 + 15 * xom2 * xsg6 - 280 * xom3 * xsg6 + 250 * xom4 * xsg6 - 70 * xom5 * xsg6 + 6 * xom6 * xsg6 + 2 * L * (45 + 45 * xom - 225 * xsg - 180 * xom * xsg + 60 * xom4 + 390 * xsg * xom4 - 18 * xom5 - 120 * xsg * xom5 + 2 * xom6 + 12 * xsg * xom6 + 450 * xsg2 + 225 * xom * xsg2 + 1110 * xom4 * xsg2 - 330 * xom5 * xsg2 + 30 * xom6 * xsg2 - 450 * xsg3 + 1740 * xom4 * xsg3 - 480 * xom5 * xsg3 + 40 * xom6 * xsg3 + 225 * xsg4 - 225 * xom * xsg4 + 1560 * xom4 * xsg4 - 390 * xom5 * xsg4 + 30 * xom6 * xsg4 - 45 * xsg5 + 180 * xom * xsg5 + 750 * xom4 * xsg5 - 168 * xom5 * xsg5 + 12 * xom6 * xsg5 - 45 * xom * xsg6 + 150 * xom4 * xsg6 - 30 * xom5 * xsg6 + 2 * xom6 * xsg6 + 45 * xom2 * (1 + 2 * xsg + 11 * xsg2 + 20 * xsg3 + 15 * xsg4 + 10 * xsg5 + 5 * xsg6) - 60 * xom3 * (1 + 8 * xsg + 23 * xsg2 + 36 * xsg3 + 35 * xsg4 + 20 * xsg5 + 5 * xsg6)) + 2 * eulergamma * (-45 - 45 * xom + 225 * xsg + 180 * xom * xsg - 60 * xom4 - 390 * xsg * xom4 + 18 * xom5 + 120 * xsg * xom5 - 2 * xom6 - 12 * xsg * xom6 - 450 * xsg2 - 225 * xom * xsg2 - 1110 * xom4 * xsg2 + 330 * xom5 * xsg2 - 30 * xom6 * xsg2 + 450 * xsg3 - 1740 * xom4 * xsg3 + 480 * xom5 * xsg3 - 40 * xom6 * xsg3 - 225 * xsg4 + 225 * xom * xsg4 - 1560 * xom4 * xsg4 + 390 * xom5 * xsg4 - 30 * xom6 * xsg4 + 45 * xsg5 - 180 * xom * xsg5 - 750 * xom4 * xsg5 + 168 * xom5 * xsg5 - 12 * xom6 * xsg5 + 45 * xom * xsg6 - 150 * xom4 * xsg6 + 30 * xom5 * xsg6 - 2 * xom6 * xsg6 - 45 * xom2 * (1 + 2 * xsg + 11 * xsg2 + 20 * xsg3 + 15 * xsg4 + 10 * xsg5 + 5 * xsg6) + 60 * xom3 * (1 + 8 * xsg + 23 * xsg2 + 36 * xsg3 + 35 * xsg4 + 20 * xsg5 + 5 * xsg6)))) / 45. + (exp_xom_xsgplus * (-51 + 600 * xsg - 15 * xom * xsg + 120 * xom2 + 840 * xsg * xom2 - 266 * xom3 - 1798 * xsg * xom3 + 138 * xom4 + 932 * xsg * xom4 - 24 * xom5 - 150 * xsg * xom5 - 1210 * xsg2 + 125 * xom * xsg2 + 2800 * xom2 * xsg2 - 5138 * xom3 * xsg2 + 2648 * xom4 * xsg2 - 390 * xom5 * xsg2 + 760 * xsg3 - 450 * xom * xsg3 + 4450 * xom2 * xsg3 - 8216 * xom3 * xsg3 + 4032 * xom4 * xsg3 - 540 * xom5 * xsg3 - 15 * xsg4 + 10 * xom * xsg4 + 4250 * xom2 * xsg4 - 7810 * xom3 * xsg4 + 3458 * xom4 * xsg4 - 420 * xom5 * xsg4 - 228 * xsg5 + 321 * xom * xsg5 + 2870 * xom2 * xsg5 - 4130 * xom3 * xsg5 + 1580 * xom4 * xsg5 - 174 * xom5 * xsg5 - 279 * xom * xsg6 + 990 * xom2 * xsg6 - 930 * xom3 * xsg6 + 300 * xom4 * xsg6 - 30 * xom5 * xsg6 - 2 * L * (-45 + 450 * xsg + 225 * xom * xsg - 42 * xom4 - 288 * xsg * xom4 + 8 * xom5 + 50 * xsg * xom5 - 1050 * xsg2 - 375 * xom * xsg2 - 832 * xom4 * xsg2 + 130 * xom5 * xsg2 + 1200 * xsg3 + 150 * xom * xsg3 - 1288 * xom4 * xsg3 + 180 * xom5 * xsg3 - 645 * xsg4 + 330 * xom * xsg4 - 1122 * xom4 * xsg4 + 140 * xom5 * xsg4 + 138 * xsg5 - 327 * xom * xsg5 - 520 * xom4 * xsg5 + 58 * xom5 * xsg5 + 93 * xom * xsg6 - 100 * xom4 * xsg6 + 10 * xom5 * xsg6 - 6 * xom2 * (4 + 28 * xsg + 134 * xsg2 + 215 * xsg3 + 175 * xsg4 + 125 * xsg5 + 55 * xsg6) + 2 * xom3 * (41 + 273 * xsg + 763 * xsg2 + 1216 * xsg3 + 1185 * xsg4 + 655 * xsg5 + 155 * xsg6)) + eulergamma * (-141 - 51 * xom + 615 * xsg + 114 * xom * xsg + 87 * xom2 + 594 * xsg * xom2 - 236 * xom3 - 1668 * xsg * xom3 + 196 * xom4 + 1294 * xsg * xom4 - 58 * xom5 - 380 * xsg * xom5 + 6 * xom6 + 36 * xsg * xom6 - 1050 * xsg2 + 15 * xom * xsg2 + 1977 * xom2 * xsg2 - 4788 * xom3 * xsg2 + 3666 * xom4 * xsg2 - 1030 * xom5 * xsg2 + 90 * xom6 * xsg2 + 750 * xsg3 - 300 * xom * xsg3 + 3120 * xom2 * xsg3 - 7616 * xom3 * xsg3 + 5644 * xom4 * xsg3 - 1480 * xom5 * xsg3 + 120 * xom6 * xsg3 - 225 * xsg4 + 75 * xom * xsg4 + 2925 * xom2 * xsg4 - 7260 * xom3 * xsg4 + 4936 * xom4 * xsg4 - 1190 * xom5 * xsg4 + 90 * xom6 * xsg4 - 45 * xsg5 + 90 * xom * xsg5 + 1950 * xom2 * xsg5 - 3900 * xom3 * xsg5 + 2310 * xom4 * xsg5 - 508 * xom5 * xsg5 + 36 * xom6 * xsg5 - 135 * xom * xsg6 + 675 * xom2 * xsg6 - 900 * xom3 * xsg6 + 450 * xom4 * xsg6 - 90 * xom5 * xsg6 + 6 * xom6 * xsg6 + 2 * L * (45 + 45 * xom - 225 * xsg - 180 * xom * xsg + 60 * xom4 + 390 * xsg * xom4 - 18 * xom5 - 120 * xsg * xom5 + 2 * xom6 + 12 * xsg * xom6 + 450 * xsg2 + 225 * xom * xsg2 + 1110 * xom4 * xsg2 - 330 * xom5 * xsg2 + 30 * xom6 * xsg2 - 450 * xsg3 + 1740 * xom4 * xsg3 - 480 * xom5 * xsg3 + 40 * xom6 * xsg3 + 225 * xsg4 - 225 * xom * xsg4 + 1560 * xom4 * xsg4 - 390 * xom5 * xsg4 + 30 * xom6 * xsg4 - 45 * xsg5 + 180 * xom * xsg5 + 750 * xom4 * xsg5 - 168 * xom5 * xsg5 + 12 * xom6 * xsg5 - 45 * xom * xsg6 + 150 * xom4 * xsg6 - 30 * xom5 * xsg6 + 2 * xom6 * xsg6 + 45 * xom2 * (1 + 2 * xsg + 11 * xsg2 + 20 * xsg3 + 15 * xsg4 + 10 * xsg5 + 5 * xsg6) - 60 * xom3 * (1 + 8 * xsg + 23 * xsg2 + 36 * xsg3 + 35 * xsg4 + 20 * xsg5 + 5 * xsg6))))) / 45.) * xsgplus_m7,
            (-1.4190476190476191 + (716 * xsg) / 35. + 2 * eulergamma - 24 * xsg * eulergamma + 2 * eulergamma2 - 12 * xsg * eulergamma2 - (569 * expint_ei_xom_xsgplus) / 105. + (214 * xsg * expint_ei_xom_xsgplus) / 5. - 2 * eulergamma * expint_ei_xom_xsgplus + 12 * xsg * eulergamma * expint_ei_xom_xsgplus - 4 * xom * hyp_xom_xsgplus + 20 * xom * xsg * hyp_xom_xsgplus + 2 * L - 24 * xsg * L + 2 * expint_ei_xom_xsgplus * L - 12 * xsg * expint_ei_xom_xsgplus * L - (1147 * xsg2) / 21. + 70 * eulergamma * xsg2 + 30 * eulergamma2 * xsg2 - 111 * expint_ei_xom_xsgplus * xsg2 - 30 * eulergamma * expint_ei_xom_xsgplus * xsg2 - 36 * xom * hyp_xom_xsgplus * xsg2 + 70 * L * xsg2 + 30 * expint_ei_xom_xsgplus * L * xsg2 + (4160 * xsg3) / 63. - (320 * eulergamma * xsg3) / 3. - 40 * eulergamma2 * xsg3 + (460 * expint_ei_xom_xsgplus * xsg3) / 3. + 40 * eulergamma * expint_ei_xom_xsgplus * xsg3 + 20 * xom * hyp_xom_xsgplus * xsg3 - (320 * L * xsg3) / 3. - 40 * expint_ei_xom_xsgplus * L * xsg3 - (643 * xsg4) / 21. + 86 * eulergamma * xsg4 + 30 * eulergamma2 * xsg4 - 111 * expint_ei_xom_xsgplus * xsg4 - 30 * eulergamma * expint_ei_xom_xsgplus * xsg4 + 20 * xom * hyp_xom_xsgplus * xsg4 + 86 * L * xsg4 + 30 * expint_ei_xom_xsgplus * L * xsg4 + (44 * xsg5) / 35. - (184 * eulergamma * xsg5) / 5. - 12 * eulergamma2 * xsg5 + (214 * expint_ei_xom_xsgplus * xsg5) / 5. + 12 * eulergamma * expint_ei_xom_xsgplus * xsg5 - 36 * xom * hyp_xom_xsgplus * xsg5 - (184 * L * xsg5) / 5. - 12 * expint_ei_xom_xsgplus * L * xsg5 - (log_xsgplus * (-569 + 4494 * xsg - 11655 * xsg2 + 16100 * xsg3 - 11655 * xsg4 + 4494 * xsg5 + 210 * eulergamma * (-1 + 6 * xsg - 15 * xsg2 + 20 * xsg3 - 15 * xsg4 + 6 * xsg5 - xsg6) - 210 * L * (-1 + 6 * xsg - 15 * xsg2 + 20 * xsg3 - 15 * xsg4 + 6 * xsg5 - xsg6) - 569 * xsg6)) / 105. - 2 * log2_xom * (-1 + 6 * xsg - 15 * xsg2 + 20 * xsg3 - 15 * xsg4 + 6 * xsg5 - xsg6) - 4 * (eulergamma - expint_ei_xom_xsgplus) * log_xom * (-1 + 6 * xsg - 15 * xsg2 + 20 * xsg3 - 15 * xsg4 + 6 * xsg5 - xsg6) - 4 * log_xom * log_xsgplus * (-1 + 6 * xsg - 15 * xsg2 + 20 * xsg3 - 15 * xsg4 + 6 * xsg5 - xsg6) + (547 * xsg6) / 105. + (674 * eulergamma * xsg6) / 105. + 2 * eulergamma2 * xsg6 - (569 * expint_ei_xom_xsgplus * xsg6) / 105. - 2 * eulergamma * expint_ei_xom_xsgplus * xsg6 + 20 * xom * hyp_xom_xsgplus * xsg6 + (674 * L * xsg6) / 105. + 2 * expint_ei_xom_xsgplus * L * xsg6 - 4 * xom * hyp_xom_xsgplus * xsg7 - (2 * exp_xom_xsgplus * log2_xom * (-315 - 315 * xom + 1890 * xsg + 1575 * xom * xsg + 630 * xom4 + 5040 * xsg * xom4 - 294 * xom5 - 2226 * xsg * xom5 + 56 * xom6 + 420 * xsg * xom6 - 4 * xom7 - 28 * xsg * xom7 - 4725 * xsg2 - 2835 * xom * xsg2 + 17010 * xom4 * xsg2 - 7350 * xom5 * xsg2 + 1344 * xom6 * xsg2 - 84 * xom7 * xsg2 + 6300 * xsg3 + 1575 * xom * xsg3 + 32340 * xom4 * xsg3 - 13650 * xom5 * xsg3 + 2380 * xom6 * xsg3 - 140 * xom7 * xsg3 - 4725 * xsg4 + 1575 * xom * xsg4 + 38010 * xom4 * xsg4 - 15330 * xom5 * xsg4 + 2520 * xom6 * xsg4 - 140 * xom7 * xsg4 + 1890 * xsg5 - 2835 * xom * xsg5 + 27720 * xom4 * xsg5 - 10374 * xom5 * xsg5 + 1596 * xom6 * xsg5 - 84 * xom7 * xsg5 - 315 * xsg6 + 1575 * xom * xsg6 + 11550 * xom4 * xsg6 - 3906 * xom5 * xsg6 + 560 * xom6 * xsg6 - 28 * xom7 * xsg6 - 315 * xom * xsg7 + 2100 * xom4 * xsg7 - 630 * xom5 * xsg7 + 84 * xom6 * xsg7 - 4 * xom7 * xsg7 + 630 * xom2 * (3 * xsg + 6 * xsg2 + 13 * xsg3 + 20 * xsg4 + 13 * xsg5 + 6 * xsg6 + 3 * xsg7) - 210 * xom3 * (3 + 21 * xsg + 75 * xsg2 + 149 * xsg3 + 177 * xsg4 + 135 * xsg5 + 65 * xsg6 + 15 * xsg7))) / 315. - (exp_xom_xsgplus * log_xom * (-1707 - 447 * xom + 13482 * xsg + 5475 * xom * xsg + 522 * xsg * xom2 + 666 * xom3 + 4662 * xsg * xom3 - 906 * xom4 - 7248 * xsg * xom4 + 578 * xom5 + 4342 * xsg * xom5 - 136 * xom6 - 1020 * xsg * xom6 + 12 * xom7 + 84 * xsg * xom7 - 34965 * xsg2 - 10143 * xom * xsg2 - 3636 * xom2 * xsg2 + 15090 * xom3 * xsg2 - 24462 * xom4 * xsg2 + 14210 * xom5 * xsg2 - 3264 * xom6 * xsg2 + 252 * xom7 * xsg2 + 48300 * xsg3 + 7035 * xom * xsg3 - 5418 * xom2 * xsg3 + 28958 * xom3 * xsg3 - 46108 * xom4 * xsg3 + 26150 * xom5 * xsg3 - 5780 * xom6 * xsg3 + 420 * xom7 * xsg3 - 34965 * xsg4 + 7035 * xom * xsg4 - 2520 * xom2 * xsg4 + 34734 * xom3 * xsg4 - 53062 * xom4 * xsg4 + 29110 * xom5 * xsg4 - 6120 * xom6 * xsg4 + 420 * xom7 * xsg4 + 13482 * xsg5 - 10143 * xom * xsg5 - 5418 * xom2 * xsg5 + 25410 * xom3 * xsg5 - 37464 * xom4 * xsg5 + 19538 * xom5 * xsg5 - 3876 * xom6 * xsg5 + 252 * xom7 * xsg5 - 1707 * xsg6 + 5475 * xom * xsg6 - 3636 * xom2 * xsg6 + 10310 * xom3 * xsg6 - 15010 * xom4 * xsg6 + 7302 * xom5 * xsg6 - 1360 * xom6 * xsg6 + 84 * xom7 * xsg6 - 447 * xom * xsg7 + 522 * xom2 * xsg7 + 1770 * xom3 * xsg7 - 2620 * xom4 * xsg7 + 1170 * xom5 * xsg7 - 204 * xom6 * xsg7 + 12 * xom7 * xsg7 + 2 * eulergamma * (-315 - 315 * xom + 1890 * xsg + 1575 * xom * xsg + 630 * xom4 + 5040 * xsg * xom4 - 294 * xom5 - 2226 * xsg * xom5 + 56 * xom6 + 420 * xsg * xom6 - 4 * xom7 - 28 * xsg * xom7 - 4725 * xsg2 - 2835 * xom * xsg2 + 17010 * xom4 * xsg2 - 7350 * xom5 * xsg2 + 1344 * xom6 * xsg2 - 84 * xom7 * xsg2 + 6300 * xsg3 + 1575 * xom * xsg3 + 32340 * xom4 * xsg3 - 13650 * xom5 * xsg3 + 2380 * xom6 * xsg3 - 140 * xom7 * xsg3 - 4725 * xsg4 + 1575 * xom * xsg4 + 38010 * xom4 * xsg4 - 15330 * xom5 * xsg4 + 2520 * xom6 * xsg4 - 140 * xom7 * xsg4 + 1890 * xsg5 - 2835 * xom * xsg5 + 27720 * xom4 * xsg5 - 10374 * xom5 * xsg5 + 1596 * xom6 * xsg5 - 84 * xom7 * xsg5 - 315 * xsg6 + 1575 * xom * xsg6 + 11550 * xom4 * xsg6 - 3906 * xom5 * xsg6 + 560 * xom6 * xsg6 - 28 * xom7 * xsg6 - 315 * xom * xsg7 + 2100 * xom4 * xsg7 - 630 * xom5 * xsg7 + 84 * xom6 * xsg7 - 4 * xom7 * xsg7 + 630 * xom2 * (3 * xsg + 6 * xsg2 + 13 * xsg3 + 20 * xsg4 + 13 * xsg5 + 6 * xsg6 + 3 * xsg7) - 210 * xom3 * (3 + 21 * xsg + 75 * xsg2 + 149 * xsg3 + 177 * xsg4 + 135 * xsg5 + 65 * xsg6 + 15 * xsg7)) - 2 * L * (-315 - 315 * xom + 1890 * xsg + 1575 * xom * xsg + 630 * xom4 + 5040 * xsg * xom4 - 294 * xom5 - 2226 * xsg * xom5 + 56 * xom6 + 420 * xsg * xom6 - 4 * xom7 - 28 * xsg * xom7 - 4725 * xsg2 - 2835 * xom * xsg2 + 17010 * xom4 * xsg2 - 7350 * xom5 * xsg2 + 1344 * xom6 * xsg2 - 84 * xom7 * xsg2 + 6300 * xsg3 + 1575 * xom * xsg3 + 32340 * xom4 * xsg3 - 13650 * xom5 * xsg3 + 2380 * xom6 * xsg3 - 140 * xom7 * xsg3 - 4725 * xsg4 + 1575 * xom * xsg4 + 38010 * xom4 * xsg4 - 15330 * xom5 * xsg4 + 2520 * xom6 * xsg4 - 140 * xom7 * xsg4 + 1890 * xsg5 - 2835 * xom * xsg5 + 27720 * xom4 * xsg5 - 10374 * xom5 * xsg5 + 1596 * xom6 * xsg5 - 84 * xom7 * xsg5 - 315 * xsg6 + 1575 * xom * xsg6 + 11550 * xom4 * xsg6 - 3906 * xom5 * xsg6 + 560 * xom6 * xsg6 - 28 * xom7 * xsg6 - 315 * xom * xsg7 + 2100 * xom4 * xsg7 - 630 * xom5 * xsg7 + 84 * xom6 * xsg7 - 4 * xom7 * xsg7 + 630 * xom2 * (3 * xsg + 6 * xsg2 + 13 * xsg3 + 20 * xsg4 + 13 * xsg5 + 6 * xsg6 + 3 * xsg7) - 210 * xom3 * (3 + 21 * xsg + 75 * xsg2 + 149 * xsg3 + 177 * xsg4 + 135 * xsg5 + 65 * xsg6 + 15 * xsg7)))) / 315. + (exp_xom_xsgplus * (447 - 6444 * xsg - 522 * xom * xsg + 1044 * xom2 + 8352 * xsg * xom2 - 2526 * xom3 - 20082 * xsg * xom3 + 2022 * xom4 + 15576 * xsg * xom4 - 584 * xom5 - 4444 * xsg * xom5 + 60 * xom6 + 432 * xsg * xom6 + 17205 * xsg2 + 618 * xom * xsg2 + 27462 * xom2 * xsg2 - 69720 * xom3 * xsg2 + 52074 * xom4 * xsg2 - 14540 * xom5 * xsg2 + 1332 * xom6 * xsg2 - 20800 * xsg3 + 3440 * xom * xsg3 + 57784 * xom2 * xsg3 - 135308 * xom3 * xsg3 + 98496 * xom4 * xsg3 - 26480 * xom5 * xsg3 + 2280 * xom6 * xsg3 + 9645 * xsg4 - 4120 * xom * xsg4 + 71480 * xom2 * xsg4 - 162054 * xom3 * xsg4 + 113994 * xom4 * xsg4 - 28960 * xom5 * xsg4 + 2340 * xom6 * xsg4 - 396 * xsg5 - 894 * xom * xsg5 + 53248 * xom2 * xsg5 - 123330 * xom3 * xsg5 + 80568 * xom4 * xsg5 - 19004 * xom5 * xsg5 + 1440 * xom6 * xsg5 - 1641 * xsg6 + 3438 * xom * xsg6 + 28686 * xom2 * xsg6 - 55940 * xom3 * xsg6 + 32070 * xom4 * xsg6 - 6924 * xom5 * xsg6 + 492 * xom6 * xsg6 - 2088 * xom * xsg7 + 9288 * xom2 * xsg7 - 11520 * xom3 * xsg7 + 5520 * xom4 * xsg7 - 1080 * xom5 * xsg7 + 72 * xom6 * xsg7 + 2 * L * (-315 + 3780 * xsg + 1890 * xom * xsg + 626 * xom4 + 4808 * xsg * xom4 - 184 * xom5 - 1412 * xsg * xom5 + 20 * xom6 + 144 * xsg * xom6 - 11025 * xsg2 - 4410 * xom * xsg2 + 16062 * xom4 * xsg2 - 4660 * xom5 * xsg2 + 444 * xom6 * xsg2 + 16800 * xsg3 + 4200 * xom * xsg3 + 30528 * xom4 * xsg3 - 8560 * xom5 * xsg3 + 760 * xom6 * xsg3 - 13545 * xsg4 + 1680 * xom * xsg4 + 35742 * xom4 * xsg4 - 9440 * xom5 * xsg4 + 780 * xom6 * xsg4 + 5796 * xsg5 - 4914 * xom * xsg5 + 25704 * xom4 * xsg5 - 6244 * xom5 * xsg5 + 480 * xom6 * xsg5 - 1011 * xsg6 + 3210 * xom * xsg6 + 10450 * xom4 * xsg6 - 2292 * xom5 * xsg6 + 164 * xom6 * xsg6 - 696 * xom * xsg7 + 1840 * xom4 * xsg7 - 360 * xom5 * xsg7 + 24 * xom6 * xsg7 + 6 * xom2 * (58 + 464 * xsg + 1099 * xsg2 + 2548 * xsg3 + 3500 * xsg4 + 2296 * xsg5 + 1167 * xsg6 + 516 * xsg7) - 2 * xom3 * (353 + 2871 * xsg + 10260 * xsg2 + 20034 * xsg3 + 23877 * xsg4 + 18375 * xsg5 + 8710 * xsg6 + 1920 * xsg7)) + eulergamma * (1077 + 447 * xom - 5922 * xsg - 1695 * xom * xsg + 696 * xom2 + 5046 * xsg * xom2 - 2078 * xom3 - 16146 * xsg * xom3 + 2158 * xom4 + 16864 * xsg * xom4 - 946 * xom5 - 7166 * xsg * xom5 + 176 * xom6 + 1308 * xsg * xom6 - 12 * xom7 - 84 * xsg * xom7 + 12915 * xsg2 + 1323 * xom * xsg2 + 16824 * xom2 * xsg2 - 56130 * xom3 * xsg2 + 56586 * xom4 * xsg2 - 23530 * xom5 * xsg2 + 4152 * xom6 * xsg2 - 252 * xom7 * xsg2 - 14700 * xsg3 + 1365 * xom * xsg3 + 35994 * xom2 * xsg3 - 109094 * xom3 * xsg3 + 107164 * xom4 * xsg3 - 43270 * xom5 * xsg3 + 7300 * xom6 * xsg3 - 420 * xom7 * xsg3 + 7875 * xsg4 - 3675 * xom * xsg4 + 44520 * xom2 * xsg4 - 130242 * xom3 * xsg4 + 124546 * xom4 * xsg4 - 47990 * xom5 * xsg4 + 7680 * xom6 * xsg4 - 420 * xom7 * xsg4 - 1890 * xsg5 + 315 * xom * xsg5 + 32970 * xom2 * xsg5 - 98910 * xom3 * xsg5 + 88872 * xom4 * xsg5 - 32026 * xom5 * xsg5 + 4836 * xom6 * xsg5 - 252 * xom7 * xsg5 - 315 * xsg6 + 945 * xom * xsg6 + 17640 * xom2 * xsg6 - 45150 * xom3 * xsg6 + 35910 * xom4 * xsg6 - 11886 * xom5 * xsg6 + 1688 * xom6 * xsg6 - 84 * xom7 * xsg6 - 945 * xom * xsg7 + 5670 * xom2 * xsg7 - 9450 * xom3 * xsg7 + 6300 * xom4 * xsg7 - 1890 * xom5 * xsg7 + 252 * xom6 * xsg7 - 12 * xom7 * xsg7 + 2 * L * (-315 - 315 * xom + 1890 * xsg + 1575 * xom * xsg + 630 * xom4 + 5040 * xsg * xom4 - 294 * xom5 - 2226 * xsg * xom5 + 56 * xom6 + 420 * xsg * xom6 - 4 * xom7 - 28 * xsg * xom7 - 4725 * xsg2 - 2835 * xom * xsg2 + 17010 * xom4 * xsg2 - 7350 * xom5 * xsg2 + 1344 * xom6 * xsg2 - 84 * xom7 * xsg2 + 6300 * xsg3 + 1575 * xom * xsg3 + 32340 * xom4 * xsg3 - 13650 * xom5 * xsg3 + 2380 * xom6 * xsg3 - 140 * xom7 * xsg3 - 4725 * xsg4 + 1575 * xom * xsg4 + 38010 * xom4 * xsg4 - 15330 * xom5 * xsg4 + 2520 * xom6 * xsg4 - 140 * xom7 * xsg4 + 1890 * xsg5 - 2835 * xom * xsg5 + 27720 * xom4 * xsg5 - 10374 * xom5 * xsg5 + 1596 * xom6 * xsg5 - 84 * xom7 * xsg5 - 315 * xsg6 + 1575 * xom * xsg6 + 11550 * xom4 * xsg6 - 3906 * xom5 * xsg6 + 560 * xom6 * xsg6 - 28 * xom7 * xsg6 - 315 * xom * xsg7 + 2100 * xom4 * xsg7 - 630 * xom5 * xsg7 + 84 * xom6 * xsg7 - 4 * xom7 * xsg7 + 630 * xom2 * (3 * xsg + 6 * xsg2 + 13 * xsg3 + 20 * xsg4 + 13 * xsg5 + 6 * xsg6 + 3 * xsg7) - 210 * xom3 * (3 + 21 * xsg + 75 * xsg2 + 149 * xsg3 + 177 * xsg4 + 135 * xsg5 + 65 * xsg6 + 15 * xsg7))))) / 315.) * xsgplus_m8,
            (1.7047619047619047 - (418 * xsg) / 15. - 2 * eulergamma + 28 * xsg * eulergamma - 2 * eulergamma2 + 14 * xsg * eulergamma2 + (599 * expint_ei_xom_xsgplus) / 105. - (779 * xsg * expint_ei_xom_xsgplus) / 15. + 2 * eulergamma * expint_ei_xom_xsgplus - 14 * xsg * eulergamma * expint_ei_xom_xsgplus + 4 * xom * hyp_xom_xsgplus - 24 * xom * xsg * hyp_xom_xsgplus - 2 * L + 28 * xsg * L - 2 * expint_ei_xom_xsgplus * L + 14 * xsg * expint_ei_xom_xsgplus * L + (1423 * xsg2) / 15. - 98 * eulergamma * xsg2 - 42 * eulergamma2 * xsg2 + (819 * expint_ei_xom_xsgplus * xsg2) / 5. + 42 * eulergamma * expint_ei_xom_xsgplus * xsg2 + 56 * xom * hyp_xom_xsgplus * xsg2 - 98 * L * xsg2 - 42 * expint_ei_xom_xsgplus * L * xsg2 - (1382 * xsg3) / 9. + (560 * eulergamma * xsg3) / 3. + 70 * eulergamma2 * xsg3 - (847 * expint_ei_xom_xsgplus * xsg3) / 3. - 70 * eulergamma * expint_ei_xom_xsgplus * xsg3 - 56 * xom * hyp_xom_xsgplus * xsg3 + (560 * L * xsg3) / 3. + 70 * expint_ei_xom_xsgplus * L * xsg3 + (1193 * xsg4) / 9. - (602 * eulergamma * xsg4) / 3. - 70 * eulergamma2 * xsg4 + (847 * expint_ei_xom_xsgplus * xsg4) / 3. + 70 * eulergamma * expint_ei_xom_xsgplus * xsg4 - (602 * L * xsg4) / 3. - 70 * expint_ei_xom_xsgplus * L * xsg4 - (146 * xsg5) / 3. + (644 * eulergamma * xsg5) / 5. + 42 * eulergamma2 * xsg5 - (819 * expint_ei_xom_xsgplus * xsg5) / 5. - 42 * eulergamma * expint_ei_xom_xsgplus * xsg5 + 56 * xom * hyp_xom_xsgplus * xsg5 + (644 * L * xsg5) / 5. + 42 * expint_ei_xom_xsgplus * L * xsg5 + (37 * xsg6) / 15. - (674 * eulergamma * xsg6) / 15. - 14 * eulergamma2 * xsg6 + (779 * expint_ei_xom_xsgplus * xsg6) / 15. + 14 * eulergamma * expint_ei_xom_xsgplus * xsg6 - 56 * xom * hyp_xom_xsgplus * xsg6 - (674 * L * xsg6) / 15. - 14 * expint_ei_xom_xsgplus * L * xsg6 + (562 * xsg7) / 105. + (704 * eulergamma * xsg7) / 105. + 2 * eulergamma2 * xsg7 - (599 * expint_ei_xom_xsgplus * xsg7) / 105. - 2 * eulergamma * expint_ei_xom_xsgplus * xsg7 + 24 * xom * hyp_xom_xsgplus * xsg7 + (704 * L * xsg7) / 105. + 2 * expint_ei_xom_xsgplus * L * xsg7 + 2 * log2_xom * (-1 + 7 * xsg - 21 * xsg2 + 35 * xsg3 - 35 * xsg4 + 21 * xsg5 - 7 * xsg6 + xsg7) + 4 * (eulergamma - expint_ei_xom_xsgplus) * log_xom * (-1 + 7 * xsg - 21 * xsg2 + 35 * xsg3 - 35 * xsg4 + 21 * xsg5 - 7 * xsg6 + xsg7) + 4 * log_xom * log_xsgplus * (-1 + 7 * xsg - 21 * xsg2 + 35 * xsg3 - 35 * xsg4 + 21 * xsg5 - 7 * xsg6 + xsg7) + (log_xsgplus * (-599 + 5453 * xsg - 17199 * xsg2 + 29645 * xsg3 - 29645 * xsg4 + 17199 * xsg5 - 5453 * xsg6 + 599 * xsg7 + 210 * eulergamma * (-1 + 7 * xsg - 21 * xsg2 + 35 * xsg3 - 35 * xsg4 + 21 * xsg5 - 7 * xsg6 + xsg7) - 210 * L * (-1 + 7 * xsg - 21 * xsg2 + 35 * xsg3 - 35 * xsg4 + 21 * xsg5 - 7 * xsg6 + xsg7))) / 105. - 4 * xom * hyp_xom_xsgplus * xsg8 + (2 * exp_xom_xsgplus * log2_xom * (-315 - 315 * xom + 2205 * xsg + 1890 * xom * xsg - 945 * xom4 - 8190 * xsg * xom4 + 546 * xom5 + 4788 * xsg * xom5 - 154 * xom6 - 1316 * xsg * xom6 + 20 * xom7 + 168 * xsg * xom7 - xom8 - 8 * xsg * xom8 - 6615 * xsg2 - 4410 * xom * xsg2 - 32130 * xom4 * xsg2 + 18396 * xom5 * xsg2 - 4956 * xom6 * xsg2 + 616 * xom7 * xsg2 - 28 * xom8 * xsg2 + 11025 * xsg3 + 4410 * xom * xsg3 - 72870 * xom4 * xsg3 + 40740 * xom5 * xsg3 - 10724 * xom6 * xsg3 + 1288 * xom7 * xsg3 - 56 * xom8 * xsg3 - 11025 * xsg4 - 104580 * xom4 * xsg4 + 57120 * xom5 * xsg4 - 14560 * xom6 * xsg4 + 1680 * xom7 * xsg4 - 70 * xom8 * xsg4 + 6615 * xsg5 - 4410 * xom * xsg5 - 98490 * xom4 * xsg5 + 51996 * xom5 * xsg5 - 12684 * xom6 * xsg5 + 1400 * xom7 * xsg5 - 56 * xom8 * xsg5 - 2205 * xsg6 + 4410 * xom * xsg6 - 60270 * xom4 * xsg6 + 29988 * xom5 * xsg6 - 6916 * xom6 * xsg6 + 728 * xom7 * xsg6 - 28 * xom8 * xsg6 + 315 * xsg7 - 1890 * xom * xsg7 - 22050 * xom4 * xsg7 + 9996 * xom5 * xsg7 - 2156 * xom6 * xsg7 + 216 * xom7 * xsg7 - 8 * xom8 * xsg7 + 315 * xom * xsg8 - 3675 * xom4 * xsg8 + 1470 * xom5 * xsg8 - 294 * xom6 * xsg8 + 28 * xom7 * xsg8 - xom8 * xsg8 - 315 * xom2 * (1 + 2 * xsg + 22 * xsg2 + 42 * xsg3 + 56 * xsg4 + 70 * xsg5 + 42 * xsg6 + 14 * xsg7 + 7 * xsg8) + 210 * xom3 * (3 + 30 * xsg + 114 * xsg2 + 262 * xsg3 + 392 * xsg4 + 378 * xsg5 + 238 * xsg6 + 98 * xsg7 + 21 * xsg8))) / 315. + (exp_xom_xsgplus * log_xom * (-1797 - 537 * xom + 16359 * xsg + 7002 * xom * xsg + 93 * xom2 + 2076 * xsg * xom2 - 396 * xom3 - 3960 * xsg * xom3 + 1119 * xom4 + 9558 * xsg * xom4 - 878 * xom5 - 7680 * xsg * xom5 + 318 * xom6 + 2706 * xsg * xom6 - 50 * xom7 - 420 * xsg * xom7 + 3 * xom8 + 24 * xsg * xom8 - 51597 * xsg2 - 17598 * xom * xsg2 - 1104 * xom2 * xsg2 - 16728 * xom3 * xsg2 + 36786 * xom4 * xsg2 - 29388 * xom5 * xsg2 + 10146 * xom6 * xsg2 - 1540 * xom7 * xsg2 + 84 * xom8 * xsg2 + 88935 * xsg3 + 19698 * xom * xsg3 + 4956 * xom2 * xsg3 - 37384 * xom3 * xsg3 + 82274 * xom4 * xsg3 - 64640 * xom5 * xsg3 + 21858 * xom6 * xsg3 - 3220 * xom7 * xsg3 + 168 * xom8 * xsg3 - 88935 * xsg4 + 11298 * xom2 * xsg4 - 52304 * xom3 * xsg4 + 116556 * xom4 * xsg4 - 89720 * xom5 * xsg4 + 29550 * xom6 * xsg4 - 4200 * xom7 * xsg4 + 210 * xom8 * xsg4 + 51597 * xsg5 - 19698 * xom * xsg5 + 420 * xom2 * xsg5 - 49896 * xom3 * xsg5 + 107338 * xom4 * xsg5 - 80608 * xom5 * xsg5 + 25638 * xom6 * xsg5 - 3500 * xom7 * xsg5 + 168 * xom8 * xsg5 - 16359 * xsg6 + 17598 * xom * xsg6 + 2856 * xom2 * xsg6 - 30856 * xom3 * xsg6 + 63014 * xom4 * xsg6 - 45780 * xom5 * xsg6 + 13926 * xom6 * xsg6 - 1820 * xom7 * xsg6 + 84 * xom8 * xsg6 + 1797 * xsg7 - 7002 * xom * xsg7 + 4452 * xom2 * xsg7 - 10136 * xom3 * xsg7 + 21630 * xom4 * xsg7 - 15008 * xom5 * xsg7 + 4326 * xom6 * xsg7 - 540 * xom7 * xsg7 + 24 * xom8 * xsg7 + 537 * xom * xsg8 - 1239 * xom2 * xsg8 - 1092 * xom3 * xsg8 + 3325 * xom4 * xsg8 - 2170 * xom5 * xsg8 + 588 * xom6 * xsg8 - 70 * xom7 * xsg8 + 3 * xom8 * xsg8 + 2 * eulergamma * (-315 - 315 * xom + 2205 * xsg + 1890 * xom * xsg - 945 * xom4 - 8190 * xsg * xom4 + 546 * xom5 + 4788 * xsg * xom5 - 154 * xom6 - 1316 * xsg * xom6 + 20 * xom7 + 168 * xsg * xom7 - xom8 - 8 * xsg * xom8 - 6615 * xsg2 - 4410 * xom * xsg2 - 32130 * xom4 * xsg2 + 18396 * xom5 * xsg2 - 4956 * xom6 * xsg2 + 616 * xom7 * xsg2 - 28 * xom8 * xsg2 + 11025 * xsg3 + 4410 * xom * xsg3 - 72870 * xom4 * xsg3 + 40740 * xom5 * xsg3 - 10724 * xom6 * xsg3 + 1288 * xom7 * xsg3 - 56 * xom8 * xsg3 - 11025 * xsg4 - 104580 * xom4 * xsg4 + 57120 * xom5 * xsg4 - 14560 * xom6 * xsg4 + 1680 * xom7 * xsg4 - 70 * xom8 * xsg4 + 6615 * xsg5 - 4410 * xom * xsg5 - 98490 * xom4 * xsg5 + 51996 * xom5 * xsg5 - 12684 * xom6 * xsg5 + 1400 * xom7 * xsg5 - 56 * xom8 * xsg5 - 2205 * xsg6 + 4410 * xom * xsg6 - 60270 * xom4 * xsg6 + 29988 * xom5 * xsg6 - 6916 * xom6 * xsg6 + 728 * xom7 * xsg6 - 28 * xom8 * xsg6 + 315 * xsg7 - 1890 * xom * xsg7 - 22050 * xom4 * xsg7 + 9996 * xom5 * xsg7 - 2156 * xom6 * xsg7 + 216 * xom7 * xsg7 - 8 * xom8 * xsg7 + 315 * xom * xsg8 - 3675 * xom4 * xsg8 + 1470 * xom5 * xsg8 - 294 * xom6 * xsg8 + 28 * xom7 * xsg8 - xom8 * xsg8 - 315 * xom2 * (1 + 2 * xsg + 22 * xsg2 + 42 * xsg3 + 56 * xsg4 + 70 * xsg5 + 42 * xsg6 + 14 * xsg7 + 7 * xsg8) + 210 * xom3 * (3 + 30 * xsg + 114 * xsg2 + 262 * xsg3 + 392 * xsg4 + 378 * xsg5 + 238 * xsg6 + 98 * xsg7 + 21 * xsg8)) - 2 * L * (-315 - 315 * xom + 2205 * xsg + 1890 * xom * xsg - 945 * xom4 - 8190 * xsg * xom4 + 546 * xom5 + 4788 * xsg * xom5 - 154 * xom6 - 1316 * xsg * xom6 + 20 * xom7 + 168 * xsg * xom7 - xom8 - 8 * xsg * xom8 - 6615 * xsg2 - 4410 * xom * xsg2 - 32130 * xom4 * xsg2 + 18396 * xom5 * xsg2 - 4956 * xom6 * xsg2 + 616 * xom7 * xsg2 - 28 * xom8 * xsg2 + 11025 * xsg3 + 4410 * xom * xsg3 - 72870 * xom4 * xsg3 + 40740 * xom5 * xsg3 - 10724 * xom6 * xsg3 + 1288 * xom7 * xsg3 - 56 * xom8 * xsg3 - 11025 * xsg4 - 104580 * xom4 * xsg4 + 57120 * xom5 * xsg4 - 14560 * xom6 * xsg4 + 1680 * xom7 * xsg4 - 70 * xom8 * xsg4 + 6615 * xsg5 - 4410 * xom * xsg5 - 98490 * xom4 * xsg5 + 51996 * xom5 * xsg5 - 12684 * xom6 * xsg5 + 1400 * xom7 * xsg5 - 56 * xom8 * xsg5 - 2205 * xsg6 + 4410 * xom * xsg6 - 60270 * xom4 * xsg6 + 29988 * xom5 * xsg6 - 6916 * xom6 * xsg6 + 728 * xom7 * xsg6 - 28 * xom8 * xsg6 + 315 * xsg7 - 1890 * xom * xsg7 - 22050 * xom4 * xsg7 + 9996 * xom5 * xsg7 - 2156 * xom6 * xsg7 + 216 * xom7 * xsg7 - 8 * xom8 * xsg7 + 315 * xom * xsg8 - 3675 * xom4 * xsg8 + 1470 * xom5 * xsg8 - 294 * xom6 * xsg8 + 28 * xom7 * xsg8 - xom8 * xsg8 - 315 * xom2 * (1 + 2 * xsg + 22 * xsg2 + 42 * xsg3 + 56 * xsg4 + 70 * xsg5 + 42 * xsg6 + 14 * xsg7 + 7 * xsg8) + 210 * xom3 * (3 + 30 * xsg + 114 * xsg2 + 262 * xsg3 + 392 * xsg4 + 378 * xsg5 + 238 * xsg6 + 98 * xsg7 + 21 * xsg8)))) / 315. - (exp_xom_xsgplus * (537 - 8778 * xsg - 1239 * xom * xsg - 1065 * xom2 - 9585 * xsg * xom2 + 3440 * xom3 + 30473 * xsg * xom3 - 3412 * xom4 - 30098 * xsg * xom4 + 1452 * xom5 + 12552 * xsg * xom5 - 265 * xom6 - 2255 * xsg * xom6 + 18 * xom7 + 147 * xsg * xom7 + 29883 * xsg2 + 3507 * xom * xsg2 - 39432 * xom2 * xsg2 + 119457 * xom3 * xsg2 - 117184 * xom4 * xsg2 + 47826 * xom5 * xsg2 - 8406 * xom6 * xsg2 + 525 * xom7 * xsg2 - 48370 * xsg3 + 1211 * xom * xsg3 - 88529 * xom2 * xsg3 + 274753 * xom3 * xsg3 - 263226 * xom4 * xsg3 + 105030 * xom5 * xsg3 - 17921 * xom6 * xsg3 + 1071 * xom7 * xsg3 + 41755 * xsg4 - 6615 * xom * xsg4 - 137102 * xom2 * xsg4 + 402157 * xom3 * xsg4 - 375154 * xom4 * xsg4 + 145440 * xom5 * xsg4 - 23890 * xom6 * xsg4 + 1365 * xom7 * xsg4 - 15330 * xsg5 + 6727 * xom * xsg5 - 139783 * xom2 * xsg5 + 385875 * xom3 * xsg5 - 349678 * xom4 * xsg5 + 129972 * xom5 * xsg5 - 20385 * xom6 * xsg5 + 1113 * xom7 * xsg5 + 777 * xsg6 + 3045 * xom * xsg6 - 87052 * xom2 * xsg6 + 243971 * xom3 * xsg6 - 209132 * xom4 * xsg6 + 73122 * xom5 * xsg6 - 10870 * xom6 * xsg6 + 567 * xom7 * xsg6 + 1686 * xsg7 - 4539 * xom * xsg7 - 38535 * xom2 * xsg7 + 96131 * xom3 * xsg7 - 73430 * xom4 * xsg7 + 23646 * xom5 * xsg7 - 3311 * xom6 * xsg7 + 165 * xom7 * xsg7 + 2223 * xom * xsg8 - 11781 * xom2 * xsg8 + 18207 * xom3 * xsg8 - 11550 * xom4 * xsg8 + 3360 * xom5 * xsg8 - 441 * xom6 * xsg8 + 21 * xom7 * xsg8 + 2 * L * (-315 + 4410 * xsg + 2205 * xom * xsg - 1024 * xom4 - 9108 * xsg * xom4 + 456 * xom5 + 3946 * xsg * xom5 - 85 * xom6 - 727 * xsg * xom6 + 6 * xom7 + 49 * xsg * xom7 - 15435 * xsg2 - 6615 * xom * xsg2 - 35726 * xom4 * xsg2 + 15074 * xom5 * xsg2 - 2724 * xom6 * xsg2 + 175 * xom7 * xsg2 + 29400 * xsg3 + 9555 * xom * xsg3 - 80634 * xom4 * xsg3 + 33260 * xom5 * xsg3 - 5837 * xom6 * xsg3 + 357 * xom7 * xsg3 - 31605 * xsg4 - 2205 * xom * xsg4 - 115446 * xom4 * xsg4 + 46380 * xom5 * xsg4 - 7820 * xom6 * xsg4 + 455 * xom7 * xsg4 + 20286 * xsg5 - 6909 * xom * xsg5 - 108528 * xom4 * xsg5 + 41826 * xom5 * xsg5 - 6705 * xom6 * xsg5 + 371 * xom7 * xsg5 - 7077 * xsg6 + 8799 * xom * xsg6 - 65954 * xom4 * xsg6 + 23786 * xom5 * xsg6 - 3592 * xom6 * xsg6 + 189 * xom7 * xsg6 + 1056 * xsg7 - 4131 * xom * xsg7 - 23730 * xom4 * xsg7 + 7784 * xom5 * xsg7 - 1099 * xom6 * xsg7 + 55 * xom7 * xsg7 + 741 * xom * xsg8 - 3850 * xom4 * xsg8 + 1120 * xom5 * xsg8 - 147 * xom6 * xsg8 + 7 * xom7 * xsg8 - 3 * xom2 * (71 + 639 * xsg + 4026 * xsg2 + 7679 * xsg3 + 11396 * xsg4 + 13405 * xsg5 + 7854 * xsg6 + 2933 * xsg7 + 1309 * xsg8) + xom3 * (1028 + 8971 * xsg + 34479 * xsg2 + 79911 * xsg3 + 118419 * xsg4 + 113673 * xsg5 + 72037 * xsg6 + 29477 * xsg7 + 6069 * xsg8)) + eulergamma * (1167 + 537 * xom - 7539 * xsg - 2592 * xom * xsg - 519 * xom2 - 5910 * xsg * xom2 + 2452 * xom3 + 21902 * xsg * xom3 - 3167 * xom4 - 27774 * xsg * xom4 + 1790 * xom5 + 15572 * xsg * xom5 - 488 * xom6 - 4160 * xsg * xom6 + 62 * xom7 + 518 * xsg * xom7 - 3 * xom8 - 24 * xsg * xom8 + 20727 * xsg2 + 4368 * xom * xsg2 - 23052 * xom2 * xsg2 + 85686 * xom3 * xsg2 - 108238 * xom4 * xsg2 + 59536 * xom5 * xsg2 - 15594 * xom6 * xsg2 + 1890 * xom7 * xsg2 - 84 * xom8 * xsg2 - 30135 * xsg3 - 588 * xom * xsg3 - 51030 * xom2 * xsg3 + 197206 * xom3 * xsg3 - 243542 * xom4 * xsg3 + 131160 * xom5 * xsg3 - 33532 * xom6 * xsg3 + 3934 * xom7 * xsg3 - 168 * xom8 * xsg3 + 25725 * xsg4 - 4410 * xom * xsg4 - 79674 * xom2 * xsg4 + 289142 * xom3 * xsg4 - 347448 * xom4 * xsg4 + 182480 * xom5 * xsg4 - 45190 * xom6 * xsg4 + 5110 * xom7 * xsg4 - 210 * xom8 * xsg4 - 11025 * xsg5 + 5880 * xom * xsg5 - 80850 * xom2 * xsg5 + 277242 * xom3 * xsg5 - 324394 * xom4 * xsg5 + 164260 * xom5 * xsg5 - 39048 * xom6 * xsg5 + 4242 * xom7 * xsg5 - 168 * xom8 * xsg5 + 2205 * xsg6 - 49980 * xom2 * xsg6 + 174930 * xom3 * xsg6 - 194922 * xom4 * xsg6 + 93352 * xom5 * xsg6 - 21110 * xom6 * xsg6 + 2198 * xom7 * xsg6 - 84 * xom8 * xsg6 + 315 * xsg7 - 1260 * xom * xsg7 - 22050 * xom2 * xsg7 + 69090 * xom3 * xsg7 - 69090 * xom4 * xsg7 + 30576 * xom5 * xsg7 - 6524 * xom6 * xsg7 + 650 * xom7 * xsg7 - 24 * xom8 * xsg7 + 945 * xom * xsg8 - 6615 * xom2 * xsg8 + 13230 * xom3 * xsg8 - 11025 * xom4 * xsg8 + 4410 * xom5 * xsg8 - 882 * xom6 * xsg8 + 84 * xom7 * xsg8 - 3 * xom8 * xsg8 + 2 * L * (-315 - 315 * xom + 2205 * xsg + 1890 * xom * xsg - 945 * xom4 - 8190 * xsg * xom4 + 546 * xom5 + 4788 * xsg * xom5 - 154 * xom6 - 1316 * xsg * xom6 + 20 * xom7 + 168 * xsg * xom7 - xom8 - 8 * xsg * xom8 - 6615 * xsg2 - 4410 * xom * xsg2 - 32130 * xom4 * xsg2 + 18396 * xom5 * xsg2 - 4956 * xom6 * xsg2 + 616 * xom7 * xsg2 - 28 * xom8 * xsg2 + 11025 * xsg3 + 4410 * xom * xsg3 - 72870 * xom4 * xsg3 + 40740 * xom5 * xsg3 - 10724 * xom6 * xsg3 + 1288 * xom7 * xsg3 - 56 * xom8 * xsg3 - 11025 * xsg4 - 104580 * xom4 * xsg4 + 57120 * xom5 * xsg4 - 14560 * xom6 * xsg4 + 1680 * xom7 * xsg4 - 70 * xom8 * xsg4 + 6615 * xsg5 - 4410 * xom * xsg5 - 98490 * xom4 * xsg5 + 51996 * xom5 * xsg5 - 12684 * xom6 * xsg5 + 1400 * xom7 * xsg5 - 56 * xom8 * xsg5 - 2205 * xsg6 + 4410 * xom * xsg6 - 60270 * xom4 * xsg6 + 29988 * xom5 * xsg6 - 6916 * xom6 * xsg6 + 728 * xom7 * xsg6 - 28 * xom8 * xsg6 + 315 * xsg7 - 1890 * xom * xsg7 - 22050 * xom4 * xsg7 + 9996 * xom5 * xsg7 - 2156 * xom6 * xsg7 + 216 * xom7 * xsg7 - 8 * xom8 * xsg7 + 315 * xom * xsg8 - 3675 * xom4 * xsg8 + 1470 * xom5 * xsg8 - 294 * xom6 * xsg8 + 28 * xom7 * xsg8 - xom8 * xsg8 - 315 * xom2 * (1 + 2 * xsg + 22 * xsg2 + 42 * xsg3 + 56 * xsg4 + 70 * xsg5 + 42 * xsg6 + 14 * xsg7 + 7 * xsg8) + 210 * xom3 * (3 + 30 * xsg + 114 * xsg2 + 262 * xsg3 + 392 * xsg4 + 378 * xsg5 + 238 * xsg6 + 98 * xsg7 + 21 * xsg8))))) / 315.) * xsgplus_m9,
            (-1.926984126984127 + (11392 * xsg) / 315. + 2 * eulergamma - 32 * xsg * eulergamma + 2 * eulergamma2 - 16 * xsg * eulergamma2 - (1867 * expint_ei_xom_xsgplus) / 315. + (6472 * xsg * expint_ei_xom_xsgplus) / 105. - 2 * eulergamma * expint_ei_xom_xsgplus + 16 * xsg * eulergamma * expint_ei_xom_xsgplus - 4 * xom * hyp_xom_xsgplus + 28 * xom * xsg * hyp_xom_xsgplus + 2 * L - 32 * xsg * L + 2 * expint_ei_xom_xsgplus * L - 16 * xsg * expint_ei_xom_xsgplus * L - (19636 * xsg2) / 135. + (392 * eulergamma * xsg2) / 3. + 56 * eulergamma2 * xsg2 - (1132 * expint_ei_xom_xsgplus * xsg2) / 5. - 56 * eulergamma * expint_ei_xom_xsgplus * xsg2 - 80 * xom * hyp_xom_xsgplus * xsg2 + (392 * L * xsg2) / 3. + 56 * expint_ei_xom_xsgplus * L * xsg2 + (40688 * xsg3) / 135. - (896 * eulergamma * xsg3) / 3. - 112 * eulergamma2 * xsg3 + (7112 * expint_ei_xom_xsgplus * xsg3) / 15. + 112 * eulergamma * expint_ei_xom_xsgplus * xsg3 + 112 * xom * hyp_xom_xsgplus * xsg3 - (896 * L * xsg3) / 3. - 112 * expint_ei_xom_xsgplus * L * xsg3 - (1042 * xsg4) / 3. + (1204 * eulergamma * xsg4) / 3. + 140 * eulergamma2 * xsg4 - (1778 * expint_ei_xom_xsgplus * xsg4) / 3. - 140 * eulergamma * expint_ei_xom_xsgplus * xsg4 - 56 * xom * hyp_xom_xsgplus * xsg4 + (1204 * L * xsg4) / 3. + 140 * expint_ei_xom_xsgplus * L * xsg4 + (31616 * xsg5) / 135. - (5152 * eulergamma * xsg5) / 15. - 112 * eulergamma2 * xsg5 + (7112 * expint_ei_xom_xsgplus * xsg5) / 15. + 112 * eulergamma * expint_ei_xom_xsgplus * xsg5 - 56 * xom * hyp_xom_xsgplus * xsg5 - (5152 * L * xsg5) / 15. - 112 * expint_ei_xom_xsgplus * L * xsg5 - (1940 * xsg6) / 27. + (2696 * eulergamma * xsg6) / 15. + 56 * eulergamma2 * xsg6 - (1132 * expint_ei_xom_xsgplus * xsg6) / 5. - 56 * eulergamma * expint_ei_xom_xsgplus * xsg6 + 112 * xom * hyp_xom_xsgplus * xsg6 + (2696 * L * xsg6) / 15. + 56 * expint_ei_xom_xsgplus * L * xsg6 + (1168 * xsg7) / 315. - (5632 * eulergamma * xsg7) / 105. - 16 * eulergamma2 * xsg7 + (6472 * expint_ei_xom_xsgplus * xsg7) / 105. + 16 * eulergamma * expint_ei_xom_xsgplus * xsg7 - 80 * xom * hyp_xom_xsgplus * xsg7 - (5632 * L * xsg7) / 105. - 16 * expint_ei_xom_xsgplus * L * xsg7 - (log_xsgplus * (-1867 + 19416 * xsg - 71316 * xsg2 + 149352 * xsg3 - 186690 * xsg4 + 149352 * xsg5 - 71316 * xsg6 + 19416 * xsg7 + 630 * eulergamma * (-1 + 8 * xsg - 28 * xsg2 + 56 * xsg3 - 70 * xsg4 + 56 * xsg5 - 28 * xsg6 + 8 * xsg7 - xsg8) - 630 * L * (-1 + 8 * xsg - 28 * xsg2 + 56 * xsg3 - 70 * xsg4 + 56 * xsg5 - 28 * xsg6 + 8 * xsg7 - xsg8) - 1867 * xsg8)) / 315. - 2 * log2_xom * (-1 + 8 * xsg - 28 * xsg2 + 56 * xsg3 - 70 * xsg4 + 56 * xsg5 - 28 * xsg6 + 8 * xsg7 - xsg8) - 4 * (eulergamma - expint_ei_xom_xsgplus) * log_xom * (-1 + 8 * xsg - 28 * xsg2 + 56 * xsg3 - 70 * xsg4 + 56 * xsg5 - 28 * xsg6 + 8 * xsg7 - xsg8) - 4 * log_xom * log_xsgplus * (-1 + 8 * xsg - 28 * xsg2 + 56 * xsg3 - 70 * xsg4 + 56 * xsg5 - 28 * xsg6 + 8 * xsg7 - xsg8) + (1721 * xsg8) / 315. + (2182 * eulergamma * xsg8) / 315. + 2 * eulergamma2 * xsg8 - (1867 * expint_ei_xom_xsgplus * xsg8) / 315. - 2 * eulergamma * expint_ei_xom_xsgplus * xsg8 + 28 * xom * hyp_xom_xsgplus * xsg8 + (2182 * L * xsg8) / 315. + 2 * expint_ei_xom_xsgplus * L * xsg8 - 4 * xom * hyp_xom_xsgplus * xsg9 - (2 * exp_xom_xsgplus * log2_xom * (-2835 - 2835 * xom + 22680 * xsg + 19845 * xom * xsg + 11340 * xom4 + 113400 * xsg * xom4 - 8316 * xom5 - 80892 * xsg * xom5 + 3024 * xom6 + 29232 * xsg * xom6 - 576 * xom7 - 5472 * xsg * xom7 + 54 * xom8 + 504 * xsg * xom8 - 2 * xom9 - 18 * xsg * xom9 - 79380 * xsg2 - 56700 * xom * xsg2 + 498960 * xom4 * xsg2 - 353808 * xom5 * xsg2 + 126000 * xom6 * xsg2 - 23184 * xom7 * xsg2 + 2088 * xom8 * xsg2 - 72 * xom9 * xsg2 + 158760 * xsg3 + 79380 * xom * xsg3 + 1300320 * xom4 * xsg3 - 910224 * xom5 * xsg3 + 318528 * xom6 * xsg3 - 57456 * xom7 * xsg3 + 5040 * xom8 * xsg3 - 168 * xom9 * xsg3 - 198450 * xsg4 - 39690 * xom * xsg4 + 2215080 * xom4 * xsg4 - 1519560 * xom5 * xsg4 + 521136 * xom6 * xsg4 - 91728 * xom7 * xsg4 + 7812 * xom8 * xsg4 - 252 * xom9 * xsg4 + 158760 * xsg5 - 39690 * xom * xsg5 + 2555280 * xom4 * xsg5 - 1713096 * xom5 * xsg5 + 572544 * xom6 * xsg5 - 97776 * xom7 * xsg5 + 8064 * xom8 * xsg5 - 252 * xom9 * xsg5 - 79380 * xsg6 + 79380 * xom * xsg6 + 2010960 * xom4 * xsg6 - 1309392 * xom5 * xsg6 + 422352 * xom6 * xsg6 - 69552 * xom7 * xsg6 + 5544 * xom8 * xsg6 - 168 * xom9 * xsg6 + 22680 * xsg7 - 56700 * xom * xsg7 + 1058400 * xom4 * xsg7 - 656208 * xom5 * xsg7 + 201600 * xom6 * xsg7 - 31824 * xom7 * xsg7 + 2448 * xom8 * xsg7 - 72 * xom9 * xsg7 - 2835 * xsg8 + 19845 * xom * xsg8 + 343980 * xom4 * xsg8 - 195804 * xom5 * xsg8 + 56448 * xom6 * xsg8 - 8496 * xom7 * xsg8 + 630 * xom8 * xsg8 - 18 * xom9 * xsg8 - 2835 * xom * xsg9 + 52920 * xom4 * xsg9 - 26460 * xom5 * xsg9 + 7056 * xom6 * xsg9 - 1008 * xom7 * xsg9 + 72 * xom8 * xsg9 - 2 * xom9 * xsg9 + 22680 * xom2 * (xsg + 2 * xsg2 + 8 * xsg3 + 14 * xsg4 + 14 * xsg5 + 14 * xsg6 + 8 * xsg7 + 2 * xsg8 + xsg9) - 7560 * xom3 * (1 + 9 * xsg + 42 * xsg2 + 110 * xsg3 + 188 * xsg4 + 224 * xsg5 + 182 * xsg6 + 98 * xsg7 + 35 * xsg8 + 7 * xsg9))) / 2835. - (exp_xom_xsgplus * log_xom * (-16803 - 5463 * xom + 174744 * xsg + 78561 * xom * xsg + 18504 * xsg * xom2 + 3912 * xom3 + 35208 * xsg * xom3 - 10068 * xom4 - 100680 * xsg * xom4 + 11124 * xom5 + 107412 * xsg * xom5 - 5272 * xom6 - 50832 * xsg * xom6 + 1240 * xom7 + 11752 * xsg * xom7 - 138 * xom8 - 1288 * xsg * xom8 + 6 * xom9 + 54 * xsg * xom9 - 641844 * xsg2 - 240300 * xom * xsg2 - 48672 * xom2 * xsg2 + 135744 * xom3 * xsg2 - 442992 * xom4 * xsg2 + 466128 * xom5 * xsg2 - 218360 * xom6 * xsg2 + 49672 * xom7 * xsg2 - 5336 * xom8 * xsg2 + 216 * xom9 * xsg2 + 1344168 * xsg3 + 384804 * xom * xsg3 + 37152 * xom2 * xsg3 + 364800 * xom3 * xsg3 - 1139904 * xom4 * xsg3 + 1189776 * xom5 * xsg3 - 549568 * xom6 * xsg3 + 122808 * xom7 * xsg3 - 12880 * xom8 * xsg3 + 504 * xom9 * xsg3 - 1680210 * xsg4 - 177282 * xom * xsg4 + 42336 * xom2 * xsg4 + 626256 * xom3 * xsg4 - 1905576 * xom4 * xsg4 + 1968360 * xom5 * xsg4 - 894168 * xom6 * xsg4 + 195608 * xom7 * xsg4 - 19964 * xom8 * xsg4 + 756 * xom9 * xsg4 + 1344168 * xsg5 - 177282 * xom * xsg5 - 123984 * xom2 * xsg5 + 676368 * xom3 * xsg5 - 2155536 * xom4 * xsg5 + 2192424 * xom5 * xsg5 - 975952 * xom6 * xsg5 + 208040 * xom7 * xsg5 - 20608 * xom8 * xsg5 + 756 * xom9 * xsg5 - 641844 * xsg6 + 384804 * xom * xsg6 + 42336 * xom2 * xsg6 + 512064 * xom3 * xsg6 - 1652112 * xom4 * xsg6 + 1647792 * xom5 * xsg6 - 714632 * xom6 * xsg6 + 147672 * xom7 * xsg6 - 14168 * xom8 * xsg6 + 504 * xom9 * xsg6 + 174744 * xsg7 - 240300 * xom * xsg7 + 37152 * xom2 * xsg7 + 274176 * xom3 * xsg7 - 826560 * xom4 * xsg7 + 807408 * xom5 * xsg7 - 338400 * xom6 * xsg7 + 67432 * xom7 * xsg7 - 6256 * xom8 * xsg7 + 216 * xom9 * xsg7 - 16803 * xsg8 + 78561 * xom * xsg8 - 48672 * xom2 * xsg8 + 71400 * xom3 * xsg8 - 244356 * xom4 * xsg8 + 234276 * xom5 * xsg8 - 93968 * xom6 * xsg8 + 17968 * xom7 * xsg8 - 1610 * xom8 * xsg8 + 54 * xom9 * xsg8 - 5463 * xom * xsg9 + 18504 * xom2 * xsg9 - 1176 * xom3 * xsg9 - 32424 * xom4 * xsg9 + 30660 * xom5 * xsg9 - 11648 * xom6 * xsg9 + 2128 * xom7 * xsg9 - 184 * xom8 * xsg9 + 6 * xom9 * xsg9 + 2 * eulergamma * (-2835 - 2835 * xom + 22680 * xsg + 19845 * xom * xsg + 11340 * xom4 + 113400 * xsg * xom4 - 8316 * xom5 - 80892 * xsg * xom5 + 3024 * xom6 + 29232 * xsg * xom6 - 576 * xom7 - 5472 * xsg * xom7 + 54 * xom8 + 504 * xsg * xom8 - 2 * xom9 - 18 * xsg * xom9 - 79380 * xsg2 - 56700 * xom * xsg2 + 498960 * xom4 * xsg2 - 353808 * xom5 * xsg2 + 126000 * xom6 * xsg2 - 23184 * xom7 * xsg2 + 2088 * xom8 * xsg2 - 72 * xom9 * xsg2 + 158760 * xsg3 + 79380 * xom * xsg3 + 1300320 * xom4 * xsg3 - 910224 * xom5 * xsg3 + 318528 * xom6 * xsg3 - 57456 * xom7 * xsg3 + 5040 * xom8 * xsg3 - 168 * xom9 * xsg3 - 198450 * xsg4 - 39690 * xom * xsg4 + 2215080 * xom4 * xsg4 - 1519560 * xom5 * xsg4 + 521136 * xom6 * xsg4 - 91728 * xom7 * xsg4 + 7812 * xom8 * xsg4 - 252 * xom9 * xsg4 + 158760 * xsg5 - 39690 * xom * xsg5 + 2555280 * xom4 * xsg5 - 1713096 * xom5 * xsg5 + 572544 * xom6 * xsg5 - 97776 * xom7 * xsg5 + 8064 * xom8 * xsg5 - 252 * xom9 * xsg5 - 79380 * xsg6 + 79380 * xom * xsg6 + 2010960 * xom4 * xsg6 - 1309392 * xom5 * xsg6 + 422352 * xom6 * xsg6 - 69552 * xom7 * xsg6 + 5544 * xom8 * xsg6 - 168 * xom9 * xsg6 + 22680 * xsg7 - 56700 * xom * xsg7 + 1058400 * xom4 * xsg7 - 656208 * xom5 * xsg7 + 201600 * xom6 * xsg7 - 31824 * xom7 * xsg7 + 2448 * xom8 * xsg7 - 72 * xom9 * xsg7 - 2835 * xsg8 + 19845 * xom * xsg8 + 343980 * xom4 * xsg8 - 195804 * xom5 * xsg8 + 56448 * xom6 * xsg8 - 8496 * xom7 * xsg8 + 630 * xom8 * xsg8 - 18 * xom9 * xsg8 - 2835 * xom * xsg9 + 52920 * xom4 * xsg9 - 26460 * xom5 * xsg9 + 7056 * xom6 * xsg9 - 1008 * xom7 * xsg9 + 72 * xom8 * xsg9 - 2 * xom9 * xsg9 + 22680 * xom2 * (xsg + 2 * xsg2 + 8 * xsg3 + 14 * xsg4 + 14 * xsg5 + 14 * xsg6 + 8 * xsg7 + 2 * xsg8 + xsg9) - 7560 * xom3 * (1 + 9 * xsg + 42 * xsg2 + 110 * xsg3 + 188 * xsg4 + 224 * xsg5 + 182 * xsg6 + 98 * xsg7 + 35 * xsg8 + 7 * xsg9)) - 2 * L * (-2835 - 2835 * xom + 22680 * xsg + 19845 * xom * xsg + 11340 * xom4 + 113400 * xsg * xom4 - 8316 * xom5 - 80892 * xsg * xom5 + 3024 * xom6 + 29232 * xsg * xom6 - 576 * xom7 - 5472 * xsg * xom7 + 54 * xom8 + 504 * xsg * xom8 - 2 * xom9 - 18 * xsg * xom9 - 79380 * xsg2 - 56700 * xom * xsg2 + 498960 * xom4 * xsg2 - 353808 * xom5 * xsg2 + 126000 * xom6 * xsg2 - 23184 * xom7 * xsg2 + 2088 * xom8 * xsg2 - 72 * xom9 * xsg2 + 158760 * xsg3 + 79380 * xom * xsg3 + 1300320 * xom4 * xsg3 - 910224 * xom5 * xsg3 + 318528 * xom6 * xsg3 - 57456 * xom7 * xsg3 + 5040 * xom8 * xsg3 - 168 * xom9 * xsg3 - 198450 * xsg4 - 39690 * xom * xsg4 + 2215080 * xom4 * xsg4 - 1519560 * xom5 * xsg4 + 521136 * xom6 * xsg4 - 91728 * xom7 * xsg4 + 7812 * xom8 * xsg4 - 252 * xom9 * xsg4 + 158760 * xsg5 - 39690 * xom * xsg5 + 2555280 * xom4 * xsg5 - 1713096 * xom5 * xsg5 + 572544 * xom6 * xsg5 - 97776 * xom7 * xsg5 + 8064 * xom8 * xsg5 - 252 * xom9 * xsg5 - 79380 * xsg6 + 79380 * xom * xsg6 + 2010960 * xom4 * xsg6 - 1309392 * xom5 * xsg6 + 422352 * xom6 * xsg6 - 69552 * xom7 * xsg6 + 5544 * xom8 * xsg6 - 168 * xom9 * xsg6 + 22680 * xsg7 - 56700 * xom * xsg7 + 1058400 * xom4 * xsg7 - 656208 * xom5 * xsg7 + 201600 * xom6 * xsg7 - 31824 * xom7 * xsg7 + 2448 * xom8 * xsg7 - 72 * xom9 * xsg7 - 2835 * xsg8 + 19845 * xom * xsg8 + 343980 * xom4 * xsg8 - 195804 * xom5 * xsg8 + 56448 * xom6 * xsg8 - 8496 * xom7 * xsg8 + 630 * xom8 * xsg8 - 18 * xom9 * xsg8 - 2835 * xom * xsg9 + 52920 * xom4 * xsg9 - 26460 * xom5 * xsg9 + 7056 * xom6 * xsg9 - 1008 * xom7 * xsg9 + 72 * xom8 * xsg9 - 2 * xom9 * xsg9 + 22680 * xom2 * (xsg + 2 * xsg2 + 8 * xsg3 + 14 * xsg4 + 14 * xsg5 + 14 * xsg6 + 8 * xsg7 + 2 * xsg8 + xsg9) - 7560 * xom3 * (1 + 9 * xsg + 42 * xsg2 + 110 * xsg3 + 188 * xsg4 + 224 * xsg5 + 182 * xsg6 + 98 * xsg7 + 35 * xsg8 + 7 * xsg9)))) / 2835. + (exp_xom_xsgplus * (5463 - 102528 * xsg - 18504 * xom * xsg + 10476 * xom2 + 104760 * xsg * xom2 - 38476 * xom3 - 382572 * xsg * xom3 + 47840 * xom4 + 469328 * xsg * xom4 - 26108 * xom5 - 254092 * xsg * xom5 + 6950 * xom6 + 66628 * xsg * xom6 - 870 * xom7 - 8210 * xsg * xom7 + 42 * xom8 + 384 * xsg * xom8 + 412356 * xsg2 + 69528 * xom * xsg2 + 472596 * xom2 * xsg2 - 1709540 * xom3 * xsg2 + 2065144 * xom4 * xsg2 - 1105416 * xom5 * xsg2 + 285122 * xom6 * xsg2 - 34454 * xom7 * xsg2 + 1560 * xom8 * xsg2 - 854448 * xsg3 - 57288 * xom * xsg3 + 1281816 * xom2 * xsg3 - 4486236 * xom3 * xsg3 + 5363200 * xom4 * xsg3 - 2824472 * xom5 * xsg3 + 715024 * xom6 * xsg3 - 84378 * xom7 * xsg3 + 3696 * xom8 * xsg3 + 984690 * xsg4 - 47040 * xom * xsg4 + 2168964 * xom2 * xsg4 - 7728108 * xom3 * xsg4 + 9064980 * xom4 * xsg4 - 4679860 * xom5 * xsg4 + 1158050 * xom6 * xsg4 - 132874 * xom7 * xsg4 + 5628 * xom8 * xsg4 - 663936 * xsg5 + 143472 * xom * xsg5 + 2626176 * xom2 * xsg5 - 9080708 * xom3 * xsg5 + 10372864 * xom4 * xsg5 - 5224948 * xom5 * xsg5 + 1255900 * xom6 * xsg5 - 139510 * xom7 * xsg5 + 5712 * xom8 * xsg5 + 203700 * xsg6 - 75432 * xom * xsg6 + 2255148 * xom2 * xsg6 - 7289324 * xom3 * xsg6 + 8089424 * xom4 * xsg6 - 3937152 * xom5 * xsg6 + 911678 * xom6 * xsg6 - 97650 * xom7 * xsg6 + 3864 * xom8 * xsg6 - 10512 * xsg7 - 47112 * xom * xsg7 + 1214424 * xom2 * xsg7 - 3944724 * xom3 * xsg7 + 4181968 * xom4 * xsg7 - 1932336 * xom5 * xsg7 + 426952 * xom6 * xsg7 - 43934 * xom7 * xsg7 + 1680 * xom8 * xsg7 - 15489 * xsg8 + 52560 * xom * xsg8 + 449808 * xom2 * xsg8 - 1368696 * xom3 * xsg8 + 1309476 * xom4 * xsg8 - 560392 * xom5 * xsg8 + 116984 * xom6 * xsg8 - 11528 * xom7 * xsg8 + 426 * xom8 * xsg8 - 20952 * xom * xsg9 + 129816 * xom2 * xsg9 - 239904 * xom3 * xsg9 + 189504 * xom4 * xsg9 - 73080 * xom5 * xsg9 + 14280 * xom6 * xsg9 - 1344 * xom7 * xsg9 + 48 * xom8 * xsg9 + 2 * L * (-2835 + 45360 * xsg + 22680 * xom * xsg + 14472 * xom4 + 141696 * xsg * xom4 - 8076 * xom5 - 78924 * xsg * xom5 + 2210 * xom6 + 21228 * xsg * xom6 - 282 * xom7 - 2670 * xsg * xom7 + 14 * xom8 + 128 * xsg * xom8 - 185220 * xsg2 - 83160 * xom * xsg2 + 623496 * xom4 * xsg2 - 344712 * xom5 * xsg2 + 91094 * xom6 * xsg2 - 11242 * xom7 * xsg2 + 520 * xom8 * xsg2 + 423360 * xsg3 + 158760 * xom * xsg3 + 1625520 * xom4 * xsg3 - 884184 * xom5 * xsg3 + 229296 * xom6 * xsg3 - 27622 * xom7 * xsg3 + 1232 * xom8 * xsg3 - 568890 * xsg4 - 105840 * xom * xsg4 + 2762100 * xom4 * xsg4 - 1471620 * xom5 * xsg4 + 373110 * xom6 * xsg4 - 43638 * xom7 * xsg4 + 1876 * xom8 * xsg4 + 486864 * xsg5 - 42336 * xom * xsg5 + 3177360 * xom4 * xsg5 - 1653156 * xom5 * xsg5 + 406900 * xom6 * xsg5 - 45962 * xom7 * xsg5 + 1904 * xom8 * xsg5 - 254772 * xsg6 + 152712 * xom * xsg6 + 2495808 * xom4 * xsg6 - 1256544 * xom5 * xsg6 + 297258 * xom6 * xsg6 - 32270 * xom7 * xsg6 + 1288 * xom8 * xsg6 + 76032 * xsg7 - 122040 * xom * xsg7 + 1308384 * xom4 * xsg7 - 623952 * xom5 * xsg7 + 140184 * xom6 * xsg7 - 14562 * xom7 * xsg7 + 560 * xom8 * xsg7 - 9819 * xsg8 + 46368 * xom * xsg8 + 420252 * xom4 * xsg8 - 183624 * xom5 * xsg8 + 38696 * xom6 * xsg8 - 3832 * xom7 * xsg8 + 142 * xom8 * xsg8 - 6984 * xom * xsg9 + 63168 * xom4 * xsg9 - 24360 * xom5 * xsg9 + 4760 * xom6 * xsg9 - 448 * xom7 * xsg9 + 16 * xom8 * xsg9 + 36 * xom2 * (97 + 970 * xsg + 2895 * xsg2 + 10170 * xsg3 + 16695 * xsg4 + 17976 * xsg5 + 17493 * xsg6 + 9546 * xsg7 + 2684 * xsg8 + 1202 * xsg9) - 12 * xom3 * (895 + 9063 * xsg + 41405 * xsg2 + 108075 * xsg3 + 186075 * xsg4 + 221193 * xsg5 + 178731 * xsg6 + 96621 * xsg7 + 34510 * xsg8 + 6664 * xsg9)) + eulergamma * (11133 + 5463 * xom - 84024 * xsg - 33201 * xom * xsg + 6984 * xom2 + 51336 * xsg * xom2 - 25392 * xom3 - 252720 * xsg * xom3 + 39012 * xom4 + 384072 * xsg * xom4 - 27276 * xom5 - 265260 * xsg * xom5 + 9692 * xom6 + 93288 * xsg * xom6 - 1804 * xom7 - 17092 * xsg * xom7 + 166 * xom8 + 1544 * xsg * xom8 - 6 * xom9 - 54 * xsg * xom9 + 271404 * xsg2 + 73980 * xom * xsg2 + 257112 * xom2 * xsg2 - 1129464 * xom3 * xsg2 + 1689984 * xom4 * xsg2 - 1155552 * xom5 * xsg2 + 400548 * xom6 * xsg2 - 72156 * xom7 * xsg2 + 6376 * xom8 * xsg2 - 216 * xom9 * xsg2 - 497448 * xsg3 - 67284 * xom * xsg3 + 695088 * xom2 * xsg3 - 2958600 * xom3 * xsg3 + 4390944 * xom4 * xsg3 - 2958144 * xom5 * xsg3 + 1008160 * xom6 * xsg3 - 178052 * xom7 * xsg3 + 15344 * xom8 * xsg3 - 504 * xom9 * xsg3 + 542430 * xsg4 - 34398 * xom * xsg4 + 1159704 * xom2 * xsg4 - 5092056 * xom3 * xsg4 + 7429776 * xom4 * xsg4 - 4911600 * xom5 * xsg4 + 1640388 * xom6 * xsg4 - 282884 * xom7 * xsg4 + 23716 * xom8 * xsg4 - 756 * xom9 * xsg4 - 370440 * xsg5 + 92610 * xom * xsg5 + 1418256 * xom2 * xsg5 - 5985000 * xom3 * xsg5 + 8510256 * xom4 * xsg5 - 5498736 * xom5 * xsg5 + 1789752 * xom6 * xsg5 - 299964 * xom7 * xsg5 + 24416 * xom8 * xsg5 - 756 * xom9 * xsg5 + 132300 * xsg6 - 79380 * xom * xsg6 + 1217160 * xom2 * xsg6 - 4801608 * xom3 * xsg6 + 6643728 * xom4 * xsg6 - 4160880 * xom5 * xsg6 + 1309148 * xom6 * xsg6 - 212212 * xom7 * xsg6 + 16744 * xom8 * xsg6 - 504 * xom9 * xsg6 - 22680 * xsg7 - 3780 * xom * xsg7 + 650160 * xom2 * xsg7 - 2593080 * xom3 * xsg7 + 3443328 * xom4 * xsg7 - 2055312 * xom5 * xsg7 + 618768 * xom6 * xsg7 - 96556 * xom7 * xsg7 + 7376 * xom8 * xsg7 - 216 * xom9 * xsg7 - 2835 * xsg8 + 14175 * xom * xsg8 + 241920 * xom2 * xsg8 - 899640 * xom3 * xsg8 + 1084860 * xom4 * xsg8 - 601524 * xom5 * xsg8 + 171360 * xom6 * xsg8 - 25632 * xom7 * xsg8 + 1894 * xom8 * xsg8 - 54 * xom9 * xsg8 - 8505 * xom * xsg9 + 68040 * xom2 * xsg9 - 158760 * xom3 * xsg9 + 158760 * xom4 * xsg9 - 79380 * xom5 * xsg9 + 21168 * xom6 * xsg9 - 3024 * xom7 * xsg9 + 216 * xom8 * xsg9 - 6 * xom9 * xsg9 + 2 * L * (-2835 - 2835 * xom + 22680 * xsg + 19845 * xom * xsg + 11340 * xom4 + 113400 * xsg * xom4 - 8316 * xom5 - 80892 * xsg * xom5 + 3024 * xom6 + 29232 * xsg * xom6 - 576 * xom7 - 5472 * xsg * xom7 + 54 * xom8 + 504 * xsg * xom8 - 2 * xom9 - 18 * xsg * xom9 - 79380 * xsg2 - 56700 * xom * xsg2 + 498960 * xom4 * xsg2 - 353808 * xom5 * xsg2 + 126000 * xom6 * xsg2 - 23184 * xom7 * xsg2 + 2088 * xom8 * xsg2 - 72 * xom9 * xsg2 + 158760 * xsg3 + 79380 * xom * xsg3 + 1300320 * xom4 * xsg3 - 910224 * xom5 * xsg3 + 318528 * xom6 * xsg3 - 57456 * xom7 * xsg3 + 5040 * xom8 * xsg3 - 168 * xom9 * xsg3 - 198450 * xsg4 - 39690 * xom * xsg4 + 2215080 * xom4 * xsg4 - 1519560 * xom5 * xsg4 + 521136 * xom6 * xsg4 - 91728 * xom7 * xsg4 + 7812 * xom8 * xsg4 - 252 * xom9 * xsg4 + 158760 * xsg5 - 39690 * xom * xsg5 + 2555280 * xom4 * xsg5 - 1713096 * xom5 * xsg5 + 572544 * xom6 * xsg5 - 97776 * xom7 * xsg5 + 8064 * xom8 * xsg5 - 252 * xom9 * xsg5 - 79380 * xsg6 + 79380 * xom * xsg6 + 2010960 * xom4 * xsg6 - 1309392 * xom5 * xsg6 + 422352 * xom6 * xsg6 - 69552 * xom7 * xsg6 + 5544 * xom8 * xsg6 - 168 * xom9 * xsg6 + 22680 * xsg7 - 56700 * xom * xsg7 + 1058400 * xom4 * xsg7 - 656208 * xom5 * xsg7 + 201600 * xom6 * xsg7 - 31824 * xom7 * xsg7 + 2448 * xom8 * xsg7 - 72 * xom9 * xsg7 - 2835 * xsg8 + 19845 * xom * xsg8 + 343980 * xom4 * xsg8 - 195804 * xom5 * xsg8 + 56448 * xom6 * xsg8 - 8496 * xom7 * xsg8 + 630 * xom8 * xsg8 - 18 * xom9 * xsg8 - 2835 * xom * xsg9 + 52920 * xom4 * xsg9 - 26460 * xom5 * xsg9 + 7056 * xom6 * xsg9 - 1008 * xom7 * xsg9 + 72 * xom8 * xsg9 - 2 * xom9 * xsg9 + 22680 * xom2 * (xsg + 2 * xsg2 + 8 * xsg3 + 14 * xsg4 + 14 * xsg5 + 14 * xsg6 + 8 * xsg7 + 2 * xsg8 + xsg9) - 7560 * xom3 * (1 + 9 * xsg + 42 * xsg2 + 110 * xsg3 + 188 * xsg4 + 224 * xsg5 + 182 * xsg6 + 98 * xsg7 + 35 * xsg8 + 7 * xsg9))))) / 2835.) * xsgplus_m10
        };
        c = c + c_b;

        const Weights c_c_star = {
            exp_xom_xsg * (-2 * L - 2 * xsg * L + 2 * log_xom + 2 * xsg * log_xom) * xsgplus_m2 + (2 * xom * exp_xom_xsgplus * hyp_plus_xom + 2 * xom * xsg * hyp_xom_xsg - 2 * expint_ei_xom_xsg * L + 2 * exp_xom_xsgplus * L + 2 * xsg * exp_xom_xsgplus * L + L2 - exp_xom_xsgplus * L2 - xom * exp_xom_xsgplus * L2 - xom * xsg * exp_xom_xsgplus * L2 - log2_xom + exp_xom_xsgplus * log2_xom + xom * exp_xom_xsgplus * log2_xom + xom * xsg * exp_xom_xsgplus * log2_xom + 2 * expint_ei_xom_xsg * log_xom + 2 * expint_ei_plus_xom * exp_xom_xsgplus * (-1 - xsg + (1 + xom + xom * xsg) * L - (1 + xom + xom * xsg) * log_xom) - 2 * eulergamma * (-L + log_xom + exp_xom_xsgplus * (-1 - xsg + (1 + xom + xom * xsg) * L - (1 + xom + xom * xsg) * log_xom)) + 2 * L * log_xsg - 2 * log_xom * log_xsg + 2 * exp_xom_xsgplus * hyp_plus_xom * xom2 + 2 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2) * xsgplus_m2,
            -(exp_xom_xsg * (2 + 4 * xsg - 4 * L - 2 * xom * L - 4 * xsg * L - 4 * xom * xsg * L + 4 * log_xom + 2 * xom * log_xom + 4 * xsg * log_xom + 4 * xom * xsg * log_xom + 2 * xsg2 - 2 * xom * L * xsg2 + 2 * xom * log_xom * xsg2) * xsgplus_m3) - (-2 * exp_xom_xsgplus - 4 * xsg * exp_xom_xsgplus + 4 * eulergamma * exp_xom_xsgplus + 4 * xom * eulergamma * exp_xom_xsgplus + 4 * xsg * eulergamma * exp_xom_xsgplus + 8 * xom * xsg * eulergamma * exp_xom_xsgplus - 4 * expint_ei_plus_xom * exp_xom_xsgplus - 4 * xom * expint_ei_plus_xom * exp_xom_xsgplus - 4 * xsg * expint_ei_plus_xom * exp_xom_xsgplus - 8 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 2 * xom * exp_xom_xsgplus * hyp_plus_xom - 2 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 2 * xom * xsg * hyp_xom_xsg + 2 * eulergamma * L - 2 * xsg * eulergamma * L - 2 * expint_ei_xom_xsg * L + 2 * xsg * expint_ei_xom_xsg * L + 4 * exp_xom_xsgplus * L + 4 * xom * exp_xom_xsgplus * L + 4 * xsg * exp_xom_xsgplus * L + 8 * xom * xsg * exp_xom_xsgplus * L - 2 * eulergamma * exp_xom_xsgplus * L - 2 * xom * eulergamma * exp_xom_xsgplus * L + 2 * xsg * eulergamma * exp_xom_xsgplus * L + 2 * expint_ei_plus_xom * exp_xom_xsgplus * L + 2 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 2 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + L2 - xsg * L2 - exp_xom_xsgplus * L2 - xom * exp_xom_xsgplus * L2 + xsg * exp_xom_xsgplus * L2 - log2_xom + xsg * log2_xom + exp_xom_xsgplus * log2_xom + xom * exp_xom_xsgplus * log2_xom - xsg * exp_xom_xsgplus * log2_xom - 2 * eulergamma * log_xom + 2 * xsg * eulergamma * log_xom + 2 * expint_ei_xom_xsg * log_xom - 2 * xsg * expint_ei_xom_xsg * log_xom + 2 * eulergamma * exp_xom_xsgplus * log_xom + 2 * xom * eulergamma * exp_xom_xsgplus * log_xom - 2 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 2 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 2 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 2 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 2 * L * log_xsg - 2 * xsg * L * log_xsg - 2 * log_xom * log_xsg + 2 * xsg * log_xom * log_xsg + 2 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 2 * eulergamma * exp_xom_xsgplus * L * xom2 - 4 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 + 2 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 4 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 - exp_xom_xsgplus * L2 * xom2 - 2 * xsg * exp_xom_xsgplus * L2 * xom2 + exp_xom_xsgplus * log2_xom * xom2 + 2 * xsg * exp_xom_xsgplus * log2_xom * xom2 + 2 * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 4 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 - 2 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 4 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 + 2 * exp_xom_xsgplus * hyp_plus_xom * xom3 + 4 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 - 2 * exp_xom_xsgplus * xsg2 + 4 * xom * eulergamma * exp_xom_xsgplus * xsg2 - 4 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 - 2 * xom * hyp_xom_xsg * xsg2 + 4 * xom * exp_xom_xsgplus * L * xsg2 + 2 * xom * eulergamma * exp_xom_xsgplus * L * xsg2 - 2 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + xom * exp_xom_xsgplus * L2 * xsg2 - xom * exp_xom_xsgplus * log2_xom * xsg2 - 2 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg2 + 2 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 - 2 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg2 - 2 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 + 2 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 - exp_xom_xsgplus * L2 * xom2 * xsg2 + exp_xom_xsgplus * log2_xom * xom2 * xsg2 + 2 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 2 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 2 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2) * xsgplus_m3,
            (exp_xom_xsg * (12 + 8 * xom + 24 * xsg + 24 * xom * xsg - 14 * L - 4 * xom * L - 6 * xsg * L + 14 * log_xom + 4 * xom * log_xom + 6 * xsg * log_xom - 4 * L * xom2 - 12 * xsg * L * xom2 + 4 * log_xom * xom2 + 12 * xsg * log_xom * xom2 + 12 * xsg2 + 24 * xom * xsg2 + 6 * L * xsg2 + 12 * xom * L * xsg2 - 6 * log_xom * xsg2 - 12 * xom * log_xom * xsg2 - 12 * L * xom2 * xsg2 + 12 * log_xom * xom2 * xsg2 + 8 * xom * xsg3 - 2 * L * xsg3 + 8 * xom * L * xsg3 + 2 * log_xom * xsg3 - 8 * xom * log_xom * xsg3 - 4 * L * xom2 * xsg3 + 4 * log_xom * xom2 * xsg3) * xsgplus_m4) / 3. + ((-12 * exp_xom_xsgplus - 12 * xom * exp_xom_xsgplus - 24 * xsg * exp_xom_xsgplus - 36 * xom * xsg * exp_xom_xsgplus + 14 * eulergamma * exp_xom_xsgplus + 12 * xom * eulergamma * exp_xom_xsgplus + 6 * xsg * eulergamma * exp_xom_xsgplus + 12 * xom * xsg * eulergamma * exp_xom_xsgplus - 14 * expint_ei_plus_xom * exp_xom_xsgplus - 12 * xom * expint_ei_plus_xom * exp_xom_xsgplus - 6 * xsg * expint_ei_plus_xom * exp_xom_xsgplus - 12 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 6 * xom * exp_xom_xsgplus * hyp_plus_xom - 12 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 6 * xom * xsg * hyp_xom_xsg + 6 * eulergamma * L - 12 * xsg * eulergamma * L - 6 * expint_ei_xom_xsg * L + 12 * xsg * expint_ei_xom_xsg * L + 14 * exp_xom_xsgplus * L + 12 * xom * exp_xom_xsgplus * L + 6 * xsg * exp_xom_xsgplus * L + 12 * xom * xsg * exp_xom_xsgplus * L - 6 * eulergamma * exp_xom_xsgplus * L - 6 * xom * eulergamma * exp_xom_xsgplus * L + 12 * xsg * eulergamma * exp_xom_xsgplus * L + 6 * xom * xsg * eulergamma * exp_xom_xsgplus * L + 6 * expint_ei_plus_xom * exp_xom_xsgplus * L + 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L - 6 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + 3 * L2 - 6 * xsg * L2 - 3 * exp_xom_xsgplus * L2 - 3 * xom * exp_xom_xsgplus * L2 + 6 * xsg * exp_xom_xsgplus * L2 + 3 * xom * xsg * exp_xom_xsgplus * L2 - 3 * log2_xom + 6 * xsg * log2_xom + 3 * exp_xom_xsgplus * log2_xom + 3 * xom * exp_xom_xsgplus * log2_xom - 6 * xsg * exp_xom_xsgplus * log2_xom - 3 * xom * xsg * exp_xom_xsgplus * log2_xom - 6 * eulergamma * log_xom + 12 * xsg * eulergamma * log_xom + 6 * expint_ei_xom_xsg * log_xom - 12 * xsg * expint_ei_xom_xsg * log_xom + 6 * eulergamma * exp_xom_xsgplus * log_xom + 6 * xom * eulergamma * exp_xom_xsgplus * log_xom - 12 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 6 * xom * xsg * eulergamma * exp_xom_xsgplus * log_xom - 6 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 6 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 6 * L * log_xsg - 12 * xsg * L * log_xsg - 6 * log_xom * log_xsg + 12 * xsg * log_xom * log_xsg + 12 * eulergamma * exp_xom_xsgplus * xom2 + 36 * xsg * eulergamma * exp_xom_xsgplus * xom2 - 12 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 - 36 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom2 + 6 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 6 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2 + 12 * exp_xom_xsgplus * L * xom2 + 36 * xsg * exp_xom_xsgplus * L * xom2 + 12 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 - 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 6 * xsg * exp_xom_xsgplus * L2 * xom2 - 6 * xsg * exp_xom_xsgplus * log2_xom * xom2 - 12 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 12 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 - 4 * eulergamma * exp_xom_xsgplus * L * xom3 - 12 * xsg * eulergamma * exp_xom_xsgplus * L * xom3 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 + 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 - 2 * exp_xom_xsgplus * L2 * xom3 - 6 * xsg * exp_xom_xsgplus * L2 * xom3 + 2 * exp_xom_xsgplus * log2_xom * xom3 + 6 * xsg * exp_xom_xsgplus * log2_xom * xom3 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom3 + 12 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom3 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 - 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 + 4 * exp_xom_xsgplus * hyp_plus_xom * xom4 + 12 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom4 - 12 * exp_xom_xsgplus * xsg2 - 36 * xom * exp_xom_xsgplus * xsg2 - 6 * eulergamma * exp_xom_xsgplus * xsg2 - 12 * xom * eulergamma * exp_xom_xsgplus * xsg2 + 6 * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 12 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 6 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg2 - 12 * xom * hyp_xom_xsg * xsg2 + 6 * eulergamma * L * xsg2 - 6 * expint_ei_xom_xsg * L * xsg2 - 6 * exp_xom_xsgplus * L * xsg2 - 12 * xom * exp_xom_xsgplus * L * xsg2 - 6 * eulergamma * exp_xom_xsgplus * L * xsg2 + 6 * xom * eulergamma * exp_xom_xsgplus * L * xsg2 + 6 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 - 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 3 * L2 * xsg2 - 3 * exp_xom_xsgplus * L2 * xsg2 + 3 * xom * exp_xom_xsgplus * L2 * xsg2 - 3 * log2_xom * xsg2 + 3 * exp_xom_xsgplus * log2_xom * xsg2 - 3 * xom * exp_xom_xsgplus * log2_xom * xsg2 - 6 * eulergamma * log_xom * xsg2 + 6 * expint_ei_xom_xsg * log_xom * xsg2 + 6 * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 6 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 6 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 6 * L * log_xsg * xsg2 - 6 * log_xom * log_xsg * xsg2 + 36 * eulergamma * exp_xom_xsgplus * xom2 * xsg2 - 36 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg2 - 6 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg2 + 36 * exp_xom_xsgplus * L * xom2 * xsg2 + 24 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 - 24 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 + 12 * exp_xom_xsgplus * L2 * xom2 * xsg2 - 12 * exp_xom_xsgplus * log2_xom * xom2 * xsg2 - 24 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 24 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 24 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2 - 12 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg2 + 12 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg2 - 6 * exp_xom_xsgplus * L2 * xom3 * xsg2 + 6 * exp_xom_xsgplus * log2_xom * xom3 * xsg2 + 12 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg2 - 12 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg2 + 12 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg2 - 12 * xom * exp_xom_xsgplus * xsg3 + 2 * eulergamma * exp_xom_xsgplus * xsg3 - 12 * xom * eulergamma * exp_xom_xsgplus * xsg3 - 2 * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 + 12 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 + 6 * xom * hyp_xom_xsg * xsg3 + 2 * exp_xom_xsgplus * L * xsg3 - 12 * xom * exp_xom_xsgplus * L * xsg3 - 6 * xom * eulergamma * exp_xom_xsgplus * L * xsg3 + 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 3 * xom * exp_xom_xsgplus * L2 * xsg3 + 3 * xom * exp_xom_xsgplus * log2_xom * xsg3 + 6 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg3 - 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 + 12 * eulergamma * exp_xom_xsgplus * xom2 * xsg3 - 12 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg3 + 6 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg3 + 12 * exp_xom_xsgplus * L * xom2 * xsg3 + 12 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg3 - 12 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg3 + 6 * exp_xom_xsgplus * L2 * xom2 * xsg3 - 6 * exp_xom_xsgplus * log2_xom * xom2 * xsg3 - 12 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg3 + 12 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg3 - 12 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg3 - 4 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg3 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg3 - 2 * exp_xom_xsgplus * L2 * xom3 * xsg3 + 2 * exp_xom_xsgplus * log2_xom * xom3 * xsg3 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg3 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg3 + 4 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg3) * xsgplus_m4) / 3.,
            -0.3333333333333333 * (exp_xom_xsg * (16 + 6 * xom + 28 * xsg + 8 * xom * xsg - 16 * L - 6 * xom * L - 4 * xom * xsg * L + 16 * log_xom + 6 * xom * log_xom + 4 * xom * xsg * log_xom + 6 * xom2 + 24 * xsg * xom2 + 2 * L * xom2 + 16 * xsg * L * xom2 - 2 * log_xom * xom2 - 16 * xsg * log_xom * xom2 - 2 * L * xom3 - 8 * xsg * L * xom3 + 2 * log_xom * xom3 + 8 * xsg * log_xom * xom3 + 12 * xsg2 - 12 * xom * xsg2 + 36 * xom2 * xsg2 + 36 * L * xom2 * xsg2 - 36 * log_xom * xom2 * xsg2 - 12 * L * xom3 * xsg2 + 12 * log_xom * xom3 * xsg2 + 4 * xsg3 - 24 * xom * xsg3 - 16 * L * xsg3 - 12 * xom * L * xsg3 + 16 * log_xom * xsg3 + 12 * xom * log_xom * xsg3 + 24 * xom2 * xsg3 + 32 * L * xom2 * xsg3 - 32 * log_xom * xom2 * xsg3 - 8 * L * xom3 * xsg3 + 8 * log_xom * xom3 * xsg3 + 4 * xsg4 - 10 * xom * xsg4 - 10 * xom * L * xsg4 + 10 * xom * log_xom * xsg4 + 6 * xom2 * xsg4 + 10 * L * xom2 * xsg4 - 10 * log_xom * xom2 * xsg4 - 2 * L * xom3 * xsg4 + 2 * log_xom * xom3 * xsg4) * xsgplus_m5) - ((-16 * exp_xom_xsgplus - 12 * xom * exp_xom_xsgplus - 28 * xsg * exp_xom_xsgplus - 24 * xom * xsg * exp_xom_xsgplus + 16 * eulergamma * exp_xom_xsgplus + 16 * xom * eulergamma * exp_xom_xsgplus + 16 * xom * xsg * eulergamma * exp_xom_xsgplus - 16 * expint_ei_plus_xom * exp_xom_xsgplus - 16 * xom * expint_ei_plus_xom * exp_xom_xsgplus - 16 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 6 * xom * exp_xom_xsgplus * hyp_plus_xom - 18 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 6 * xom * xsg * hyp_xom_xsg + 6 * eulergamma * L - 18 * xsg * eulergamma * L - 6 * expint_ei_xom_xsg * L + 18 * xsg * expint_ei_xom_xsg * L + 16 * exp_xom_xsgplus * L + 16 * xom * exp_xom_xsgplus * L + 16 * xom * xsg * exp_xom_xsgplus * L - 6 * eulergamma * exp_xom_xsgplus * L - 6 * xom * eulergamma * exp_xom_xsgplus * L + 18 * xsg * eulergamma * exp_xom_xsgplus * L + 12 * xom * xsg * eulergamma * exp_xom_xsgplus * L + 6 * expint_ei_plus_xom * exp_xom_xsgplus * L + 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 18 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L - 12 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + 3 * L2 - 9 * xsg * L2 - 3 * exp_xom_xsgplus * L2 - 3 * xom * exp_xom_xsgplus * L2 + 9 * xsg * exp_xom_xsgplus * L2 + 6 * xom * xsg * exp_xom_xsgplus * L2 - 3 * log2_xom + 9 * xsg * log2_xom + 3 * exp_xom_xsgplus * log2_xom + 3 * xom * exp_xom_xsgplus * log2_xom - 9 * xsg * exp_xom_xsgplus * log2_xom - 6 * xom * xsg * exp_xom_xsgplus * log2_xom - 6 * eulergamma * log_xom + 18 * xsg * eulergamma * log_xom + 6 * expint_ei_xom_xsg * log_xom - 18 * xsg * expint_ei_xom_xsg * log_xom + 6 * eulergamma * exp_xom_xsgplus * log_xom + 6 * xom * eulergamma * exp_xom_xsgplus * log_xom - 18 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 12 * xom * xsg * eulergamma * exp_xom_xsgplus * log_xom - 6 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 18 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 12 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 6 * L * log_xsg - 18 * xsg * L * log_xsg - 6 * log_xom * log_xsg + 18 * xsg * log_xom * log_xsg - 12 * exp_xom_xsgplus * xom2 - 48 * xsg * exp_xom_xsgplus * xom2 - 24 * xsg * eulergamma * exp_xom_xsgplus * xom2 + 24 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom2 + 6 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 12 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2 - 24 * xsg * exp_xom_xsgplus * L * xom2 - 6 * eulergamma * exp_xom_xsgplus * L * xom2 - 12 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 + 6 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 - 3 * exp_xom_xsgplus * L2 * xom2 - 6 * xsg * exp_xom_xsgplus * L2 * xom2 + 3 * exp_xom_xsgplus * log2_xom * xom2 + 6 * xsg * exp_xom_xsgplus * log2_xom * xom2 + 6 * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 12 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 - 6 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 12 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 + 8 * eulergamma * exp_xom_xsgplus * xom3 + 32 * xsg * eulergamma * exp_xom_xsgplus * xom3 - 8 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 - 32 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom3 + 6 * exp_xom_xsgplus * hyp_plus_xom * xom3 + 12 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 + 8 * exp_xom_xsgplus * L * xom3 + 32 * xsg * exp_xom_xsgplus * L * xom3 + 4 * eulergamma * exp_xom_xsgplus * L * xom3 + 24 * xsg * eulergamma * exp_xom_xsgplus * L * xom3 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 - 24 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 + 2 * exp_xom_xsgplus * L2 * xom3 + 12 * xsg * exp_xom_xsgplus * L2 * xom3 - 2 * exp_xom_xsgplus * log2_xom * xom3 - 12 * xsg * exp_xom_xsgplus * log2_xom * xom3 - 4 * eulergamma * exp_xom_xsgplus * log_xom * xom3 - 24 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom3 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 + 24 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 - 4 * exp_xom_xsgplus * hyp_plus_xom * xom4 - 24 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom4 - 2 * eulergamma * exp_xom_xsgplus * L * xom4 - 8 * xsg * eulergamma * exp_xom_xsgplus * L * xom4 + 2 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 + 8 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 - exp_xom_xsgplus * L2 * xom4 - 4 * xsg * exp_xom_xsgplus * L2 * xom4 + exp_xom_xsgplus * log2_xom * xom4 + 4 * xsg * exp_xom_xsgplus * log2_xom * xom4 + 2 * eulergamma * exp_xom_xsgplus * log_xom * xom4 + 8 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom4 - 2 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 - 8 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 + 2 * exp_xom_xsgplus * hyp_plus_xom * xom5 + 8 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom5 - 12 * exp_xom_xsgplus * xsg2 + 18 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg2 - 18 * xom * hyp_xom_xsg * xsg2 + 18 * eulergamma * L * xsg2 - 18 * expint_ei_xom_xsg * L * xsg2 - 18 * eulergamma * exp_xom_xsgplus * L * xsg2 + 18 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 9 * L2 * xsg2 - 9 * exp_xom_xsgplus * L2 * xsg2 - 9 * log2_xom * xsg2 + 9 * exp_xom_xsgplus * log2_xom * xsg2 - 18 * eulergamma * log_xom * xsg2 + 18 * expint_ei_xom_xsg * log_xom * xsg2 + 18 * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 18 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 18 * L * log_xsg * xsg2 - 18 * log_xom * log_xsg * xsg2 - 72 * exp_xom_xsgplus * xom2 * xsg2 - 72 * eulergamma * exp_xom_xsgplus * xom2 * xsg2 + 72 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg2 - 72 * exp_xom_xsgplus * L * xom2 * xsg2 - 24 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 + 24 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 - 12 * exp_xom_xsgplus * L2 * xom2 * xsg2 + 12 * exp_xom_xsgplus * log2_xom * xom2 * xsg2 + 24 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 24 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 48 * eulergamma * exp_xom_xsgplus * xom3 * xsg2 - 48 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg2 + 24 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2 + 48 * exp_xom_xsgplus * L * xom3 * xsg2 + 48 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg2 - 48 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg2 + 24 * exp_xom_xsgplus * L2 * xom3 * xsg2 - 24 * exp_xom_xsgplus * log2_xom * xom3 * xsg2 - 48 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg2 + 48 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg2 - 48 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg2 - 12 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg2 + 12 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg2 - 6 * exp_xom_xsgplus * L2 * xom4 * xsg2 + 6 * exp_xom_xsgplus * log2_xom * xom4 * xsg2 + 12 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg2 - 12 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg2 + 12 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg2 - 4 * exp_xom_xsgplus * xsg3 + 24 * xom * exp_xom_xsgplus * xsg3 + 16 * eulergamma * exp_xom_xsgplus * xsg3 + 16 * xom * eulergamma * exp_xom_xsgplus * xsg3 - 16 * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 16 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 6 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg3 + 18 * xom * hyp_xom_xsg * xsg3 - 6 * eulergamma * L * xsg3 + 6 * expint_ei_xom_xsg * L * xsg3 + 16 * exp_xom_xsgplus * L * xsg3 + 16 * xom * exp_xom_xsgplus * L * xsg3 + 6 * eulergamma * exp_xom_xsgplus * L * xsg3 - 12 * xom * eulergamma * exp_xom_xsgplus * L * xsg3 - 6 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 + 12 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 3 * L2 * xsg3 + 3 * exp_xom_xsgplus * L2 * xsg3 - 6 * xom * exp_xom_xsgplus * L2 * xsg3 + 3 * log2_xom * xsg3 - 3 * exp_xom_xsgplus * log2_xom * xsg3 + 6 * xom * exp_xom_xsgplus * log2_xom * xsg3 + 6 * eulergamma * log_xom * xsg3 - 6 * expint_ei_xom_xsg * log_xom * xsg3 - 6 * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 12 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 6 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 12 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 6 * L * log_xsg * xsg3 + 6 * log_xom * log_xsg * xsg3 - 48 * exp_xom_xsgplus * xom2 * xsg3 - 72 * eulergamma * exp_xom_xsgplus * xom2 * xsg3 + 72 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg3 + 12 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg3 - 72 * exp_xom_xsgplus * L * xom2 * xsg3 - 36 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg3 + 36 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg3 - 18 * exp_xom_xsgplus * L2 * xom2 * xsg3 + 18 * exp_xom_xsgplus * log2_xom * xom2 * xsg3 + 36 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg3 - 36 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg3 + 32 * eulergamma * exp_xom_xsgplus * xom3 * xsg3 - 32 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg3 + 36 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg3 + 32 * exp_xom_xsgplus * L * xom3 * xsg3 + 40 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg3 - 40 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg3 + 20 * exp_xom_xsgplus * L2 * xom3 * xsg3 - 20 * exp_xom_xsgplus * log2_xom * xom3 * xsg3 - 40 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg3 + 40 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg3 - 40 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg3 - 8 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg3 + 8 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg3 - 4 * exp_xom_xsgplus * L2 * xom4 * xsg3 + 4 * exp_xom_xsgplus * log2_xom * xom4 * xsg3 + 8 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg3 - 8 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg3 + 8 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg3 - 4 * exp_xom_xsgplus * xsg4 + 12 * xom * exp_xom_xsgplus * xsg4 + 16 * xom * eulergamma * exp_xom_xsgplus * xsg4 - 16 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 - 6 * xom * hyp_xom_xsg * xsg4 + 16 * xom * exp_xom_xsgplus * L * xsg4 + 6 * xom * eulergamma * exp_xom_xsgplus * L * xsg4 - 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 + 3 * xom * exp_xom_xsgplus * L2 * xsg4 - 3 * xom * exp_xom_xsgplus * log2_xom * xsg4 - 6 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg4 + 6 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 - 12 * exp_xom_xsgplus * xom2 * xsg4 - 24 * eulergamma * exp_xom_xsgplus * xom2 * xsg4 + 24 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg4 - 6 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg4 - 24 * exp_xom_xsgplus * L * xom2 * xsg4 - 18 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg4 + 18 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg4 - 9 * exp_xom_xsgplus * L2 * xom2 * xsg4 + 9 * exp_xom_xsgplus * log2_xom * xom2 * xsg4 + 18 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg4 - 18 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg4 + 8 * eulergamma * exp_xom_xsgplus * xom3 * xsg4 - 8 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg4 + 18 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg4 + 8 * exp_xom_xsgplus * L * xom3 * xsg4 + 12 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg4 - 12 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg4 + 6 * exp_xom_xsgplus * L2 * xom3 * xsg4 - 6 * exp_xom_xsgplus * log2_xom * xom3 * xsg4 - 12 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg4 + 12 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg4 - 12 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg4 - 2 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg4 + 2 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg4 - exp_xom_xsgplus * L2 * xom4 * xsg4 + exp_xom_xsgplus * log2_xom * xom4 * xsg4 + 2 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg4 - 2 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg4 + 2 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg4) * xsgplus_m5) / 3.,
            (exp_xom_xsg * (100 + 56 * xom + 160 * xsg + 140 * xom * xsg - 86 * L - 24 * xom * L + 50 * xsg * L + 40 * xom * xsg * L + 86 * log_xom + 24 * xom * log_xom - 50 * xsg * log_xom - 40 * xom * xsg * log_xom - 12 * xom2 - 120 * xsg * xom2 - 28 * L * xom2 - 120 * xsg * L * xom2 + 28 * log_xom * xom2 + 120 * xsg * log_xom * xom2 + 16 * xom3 + 80 * xsg * xom3 + 16 * L * xom3 + 100 * xsg * L * xom3 - 16 * log_xom * xom3 - 100 * xsg * log_xom * xom3 - 4 * L * xom4 - 20 * xsg * L * xom4 + 4 * log_xom * xom4 + 20 * xsg * log_xom * xom4 + 120 * xsg2 + 160 * xom * xsg2 - 60 * L * xsg2 + 80 * xom * L * xsg2 + 60 * log_xom * xsg2 - 80 * xom * log_xom * xsg2 - 360 * xom2 * xsg2 - 280 * L * xom2 * xsg2 + 280 * log_xom * xom2 * xsg2 + 160 * xom3 * xsg2 + 240 * L * xom3 * xsg2 - 240 * log_xom * xom3 * xsg2 - 40 * L * xom4 * xsg2 + 40 * log_xom * xom4 * xsg2 + 160 * xsg3 + 200 * xom * xsg3 - 140 * L * xsg3 + 140 * log_xom * xsg3 - 480 * xom2 * xsg3 - 400 * L * xom2 * xsg3 + 400 * log_xom * xom2 * xsg3 + 160 * xom3 * xsg3 + 280 * L * xom3 * xsg3 - 280 * log_xom * xom3 * xsg3 - 40 * L * xom4 * xsg3 + 40 * log_xom * xom4 * xsg3 + 100 * xsg4 + 200 * xom * xsg4 + 50 * L * xsg4 + 40 * xom * L * xsg4 - 50 * log_xom * xsg4 - 40 * xom * log_xom * xsg4 - 300 * xom2 * xsg4 - 300 * L * xom2 * xsg4 + 300 * log_xom * xom2 * xsg4 + 80 * xom3 * xsg4 + 160 * L * xom3 * xsg4 - 160 * log_xom * xom3 * xsg4 - 20 * L * xom4 * xsg4 + 20 * log_xom * xom4 * xsg4 + 76 * xom * xsg5 - 6 * L * xsg5 + 56 * xom * L * xsg5 + 6 * log_xom * xsg5 - 56 * xom * log_xom * xsg5 - 72 * xom2 * xsg5 - 88 * L * xom2 * xsg5 + 88 * log_xom * xom2 * xsg5 + 16 * xom3 * xsg5 + 36 * L * xom3 * xsg5 - 36 * log_xom * xom3 * xsg5 - 4 * L * xom4 * xsg5 + 4 * log_xom * xom4 * xsg5) * xsgplus_m6) / 15. + ((-100 * exp_xom_xsgplus - 100 * xom * exp_xom_xsgplus - 160 * xsg * exp_xom_xsgplus - 260 * xom * xsg * exp_xom_xsgplus + 86 * eulergamma * exp_xom_xsgplus + 80 * xom * eulergamma * exp_xom_xsgplus - 50 * xsg * eulergamma * exp_xom_xsgplus - 86 * expint_ei_plus_xom * exp_xom_xsgplus - 80 * xom * expint_ei_plus_xom * exp_xom_xsgplus + 50 * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 30 * xom * exp_xom_xsgplus * hyp_plus_xom - 120 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 30 * xom * xsg * hyp_xom_xsg + 30 * eulergamma * L - 120 * xsg * eulergamma * L - 30 * expint_ei_xom_xsg * L + 120 * xsg * expint_ei_xom_xsg * L + 86 * exp_xom_xsgplus * L + 80 * xom * exp_xom_xsgplus * L - 50 * xsg * exp_xom_xsgplus * L - 30 * eulergamma * exp_xom_xsgplus * L - 30 * xom * eulergamma * exp_xom_xsgplus * L + 120 * xsg * eulergamma * exp_xom_xsgplus * L + 90 * xom * xsg * eulergamma * exp_xom_xsgplus * L + 30 * expint_ei_plus_xom * exp_xom_xsgplus * L + 30 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 120 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L - 90 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + 15 * L2 - 60 * xsg * L2 - 15 * exp_xom_xsgplus * L2 - 15 * xom * exp_xom_xsgplus * L2 + 60 * xsg * exp_xom_xsgplus * L2 + 45 * xom * xsg * exp_xom_xsgplus * L2 - 15 * log2_xom + 60 * xsg * log2_xom + 15 * exp_xom_xsgplus * log2_xom + 15 * xom * exp_xom_xsgplus * log2_xom - 60 * xsg * exp_xom_xsgplus * log2_xom - 45 * xom * xsg * exp_xom_xsgplus * log2_xom - 30 * eulergamma * log_xom + 120 * xsg * eulergamma * log_xom + 30 * expint_ei_xom_xsg * log_xom - 120 * xsg * expint_ei_xom_xsg * log_xom + 30 * eulergamma * exp_xom_xsgplus * log_xom + 30 * xom * eulergamma * exp_xom_xsgplus * log_xom - 120 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 90 * xom * xsg * eulergamma * exp_xom_xsgplus * log_xom - 30 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 30 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 120 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 90 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 30 * L * log_xsg - 120 * xsg * L * log_xsg - 30 * log_xom * log_xsg + 120 * xsg * log_xom * log_xsg + 120 * xsg * exp_xom_xsgplus * xom2 + 80 * eulergamma * exp_xom_xsgplus * xom2 + 280 * xsg * eulergamma * exp_xom_xsgplus * xom2 - 80 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 - 280 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom2 + 30 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 90 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2 + 80 * exp_xom_xsgplus * L * xom2 + 280 * xsg * exp_xom_xsgplus * L * xom2 + 120 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 - 120 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 60 * xsg * exp_xom_xsgplus * L2 * xom2 - 60 * xsg * exp_xom_xsgplus * log2_xom * xom2 - 120 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 120 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 40 * exp_xom_xsgplus * xom3 - 200 * xsg * exp_xom_xsgplus * xom3 - 40 * eulergamma * exp_xom_xsgplus * xom3 - 280 * xsg * eulergamma * exp_xom_xsgplus * xom3 + 40 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 + 280 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom3 - 120 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 - 40 * exp_xom_xsgplus * L * xom3 - 280 * xsg * exp_xom_xsgplus * L * xom3 - 40 * eulergamma * exp_xom_xsgplus * L * xom3 - 200 * xsg * eulergamma * exp_xom_xsgplus * L * xom3 + 40 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 + 200 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 - 20 * exp_xom_xsgplus * L2 * xom3 - 100 * xsg * exp_xom_xsgplus * L2 * xom3 + 20 * exp_xom_xsgplus * log2_xom * xom3 + 100 * xsg * exp_xom_xsgplus * log2_xom * xom3 + 40 * eulergamma * exp_xom_xsgplus * log_xom * xom3 + 200 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom3 - 40 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 - 200 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 + 20 * eulergamma * exp_xom_xsgplus * xom4 + 100 * xsg * eulergamma * exp_xom_xsgplus * xom4 - 20 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 - 100 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom4 + 40 * exp_xom_xsgplus * hyp_plus_xom * xom4 + 200 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom4 + 20 * exp_xom_xsgplus * L * xom4 + 100 * xsg * exp_xom_xsgplus * L * xom4 + 20 * eulergamma * exp_xom_xsgplus * L * xom4 + 120 * xsg * eulergamma * exp_xom_xsgplus * L * xom4 - 20 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 - 120 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 + 10 * exp_xom_xsgplus * L2 * xom4 + 60 * xsg * exp_xom_xsgplus * L2 * xom4 - 10 * exp_xom_xsgplus * log2_xom * xom4 - 60 * xsg * exp_xom_xsgplus * log2_xom * xom4 - 20 * eulergamma * exp_xom_xsgplus * log_xom * xom4 - 120 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom4 + 20 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 + 120 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 - 20 * exp_xom_xsgplus * hyp_plus_xom * xom5 - 120 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom5 - 4 * eulergamma * exp_xom_xsgplus * L * xom5 - 20 * xsg * eulergamma * exp_xom_xsgplus * L * xom5 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 + 20 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 - 2 * exp_xom_xsgplus * L2 * xom5 - 10 * xsg * exp_xom_xsgplus * L2 * xom5 + 2 * exp_xom_xsgplus * log2_xom * xom5 + 10 * xsg * exp_xom_xsgplus * log2_xom * xom5 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom5 + 20 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom5 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 - 20 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 + 4 * exp_xom_xsgplus * hyp_plus_xom * xom6 + 20 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom6 - 120 * exp_xom_xsgplus * xsg2 - 280 * xom * exp_xom_xsgplus * xsg2 + 60 * eulergamma * exp_xom_xsgplus * xsg2 - 80 * xom * eulergamma * exp_xom_xsgplus * xsg2 - 60 * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 80 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 180 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg2 - 120 * xom * hyp_xom_xsg * xsg2 + 180 * eulergamma * L * xsg2 - 180 * expint_ei_xom_xsg * L * xsg2 + 60 * exp_xom_xsgplus * L * xsg2 - 80 * xom * exp_xom_xsgplus * L * xsg2 - 180 * eulergamma * exp_xom_xsgplus * L * xsg2 - 60 * xom * eulergamma * exp_xom_xsgplus * L * xsg2 + 180 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 60 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 90 * L2 * xsg2 - 90 * exp_xom_xsgplus * L2 * xsg2 - 30 * xom * exp_xom_xsgplus * L2 * xsg2 - 90 * log2_xom * xsg2 + 90 * exp_xom_xsgplus * log2_xom * xsg2 + 30 * xom * exp_xom_xsgplus * log2_xom * xsg2 - 180 * eulergamma * log_xom * xsg2 + 180 * expint_ei_xom_xsg * log_xom * xsg2 + 180 * eulergamma * exp_xom_xsgplus * log_xom * xsg2 + 60 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 180 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 - 60 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 180 * L * log_xsg * xsg2 - 180 * log_xom * log_xsg * xsg2 + 480 * exp_xom_xsgplus * xom2 * xsg2 + 560 * eulergamma * exp_xom_xsgplus * xom2 * xsg2 - 560 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg2 + 60 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg2 + 560 * exp_xom_xsgplus * L * xom2 * xsg2 + 240 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 - 240 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 + 120 * exp_xom_xsgplus * L2 * xom2 * xsg2 - 120 * exp_xom_xsgplus * log2_xom * xom2 * xsg2 - 240 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 240 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 400 * exp_xom_xsgplus * xom3 * xsg2 - 720 * eulergamma * exp_xom_xsgplus * xom3 * xsg2 + 720 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg2 - 240 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2 - 720 * exp_xom_xsgplus * L * xom3 * xsg2 - 480 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg2 + 480 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg2 - 240 * exp_xom_xsgplus * L2 * xom3 * xsg2 + 240 * exp_xom_xsgplus * log2_xom * xom3 * xsg2 + 480 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg2 - 480 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg2 + 200 * eulergamma * exp_xom_xsgplus * xom4 * xsg2 - 200 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg2 + 480 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg2 + 200 * exp_xom_xsgplus * L * xom4 * xsg2 + 280 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg2 - 280 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg2 + 140 * exp_xom_xsgplus * L2 * xom4 * xsg2 - 140 * exp_xom_xsgplus * log2_xom * xom4 * xsg2 - 280 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg2 + 280 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg2 - 280 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg2 - 40 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg2 + 40 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg2 - 20 * exp_xom_xsgplus * L2 * xom5 * xsg2 + 20 * exp_xom_xsgplus * log2_xom * xom5 * xsg2 + 40 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg2 - 40 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg2 + 40 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg2 - 160 * exp_xom_xsgplus * xsg3 - 280 * xom * exp_xom_xsgplus * xsg3 + 140 * eulergamma * exp_xom_xsgplus * xsg3 + 80 * xom * eulergamma * exp_xom_xsgplus * xsg3 - 140 * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 80 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 120 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg3 + 180 * xom * hyp_xom_xsg * xsg3 - 120 * eulergamma * L * xsg3 + 120 * expint_ei_xom_xsg * L * xsg3 + 140 * exp_xom_xsgplus * L * xsg3 + 80 * xom * exp_xom_xsgplus * L * xsg3 + 120 * eulergamma * exp_xom_xsgplus * L * xsg3 - 60 * xom * eulergamma * exp_xom_xsgplus * L * xsg3 - 120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 + 60 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 60 * L2 * xsg3 + 60 * exp_xom_xsgplus * L2 * xsg3 - 30 * xom * exp_xom_xsgplus * L2 * xsg3 + 60 * log2_xom * xsg3 - 60 * exp_xom_xsgplus * log2_xom * xsg3 + 30 * xom * exp_xom_xsgplus * log2_xom * xsg3 + 120 * eulergamma * log_xom * xsg3 - 120 * expint_ei_xom_xsg * log_xom * xsg3 - 120 * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 60 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 60 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 120 * L * log_xsg * xsg3 + 120 * log_xom * log_xsg * xsg3 + 720 * exp_xom_xsgplus * xom2 * xsg3 + 800 * eulergamma * exp_xom_xsgplus * xom2 * xsg3 - 800 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg3 + 60 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg3 + 800 * exp_xom_xsgplus * L * xom2 * xsg3 + 240 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg3 - 240 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg3 + 120 * exp_xom_xsgplus * L2 * xom2 * xsg3 - 120 * exp_xom_xsgplus * log2_xom * xom2 * xsg3 - 240 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg3 + 240 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg3 - 400 * exp_xom_xsgplus * xom3 * xsg3 - 880 * eulergamma * exp_xom_xsgplus * xom3 * xsg3 + 880 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg3 - 240 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg3 - 880 * exp_xom_xsgplus * L * xom3 * xsg3 - 640 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg3 + 640 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg3 - 320 * exp_xom_xsgplus * L2 * xom3 * xsg3 + 320 * exp_xom_xsgplus * log2_xom * xom3 * xsg3 + 640 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg3 - 640 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg3 + 200 * eulergamma * exp_xom_xsgplus * xom4 * xsg3 - 200 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg3 + 640 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg3 + 200 * exp_xom_xsgplus * L * xom4 * xsg3 + 320 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg3 - 320 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg3 + 160 * exp_xom_xsgplus * L2 * xom4 * xsg3 - 160 * exp_xom_xsgplus * log2_xom * xom4 * xsg3 - 320 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg3 + 320 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg3 - 320 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg3 - 40 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg3 + 40 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg3 - 20 * exp_xom_xsgplus * L2 * xom5 * xsg3 + 20 * exp_xom_xsgplus * log2_xom * xom5 * xsg3 + 40 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg3 - 40 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg3 + 40 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg3 - 100 * exp_xom_xsgplus * xsg4 - 260 * xom * exp_xom_xsgplus * xsg4 - 50 * eulergamma * exp_xom_xsgplus * xsg4 + 50 * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 + 30 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg4 - 120 * xom * hyp_xom_xsg * xsg4 + 30 * eulergamma * L * xsg4 - 30 * expint_ei_xom_xsg * L * xsg4 - 50 * exp_xom_xsgplus * L * xsg4 - 30 * eulergamma * exp_xom_xsgplus * L * xsg4 + 90 * xom * eulergamma * exp_xom_xsgplus * L * xsg4 + 30 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 - 90 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 + 15 * L2 * xsg4 - 15 * exp_xom_xsgplus * L2 * xsg4 + 45 * xom * exp_xom_xsgplus * L2 * xsg4 - 15 * log2_xom * xsg4 + 15 * exp_xom_xsgplus * log2_xom * xsg4 - 45 * xom * exp_xom_xsgplus * log2_xom * xsg4 - 30 * eulergamma * log_xom * xsg4 + 30 * expint_ei_xom_xsg * log_xom * xsg4 + 30 * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 90 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 30 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 90 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 30 * L * log_xsg * xsg4 - 30 * log_xom * log_xsg * xsg4 + 480 * exp_xom_xsgplus * xom2 * xsg4 + 640 * eulergamma * exp_xom_xsgplus * xom2 * xsg4 - 640 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg4 - 90 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg4 + 640 * exp_xom_xsgplus * L * xom2 * xsg4 + 240 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg4 - 240 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg4 + 120 * exp_xom_xsgplus * L2 * xom2 * xsg4 - 120 * exp_xom_xsgplus * log2_xom * xom2 * xsg4 - 240 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg4 + 240 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg4 - 200 * exp_xom_xsgplus * xom3 * xsg4 - 520 * eulergamma * exp_xom_xsgplus * xom3 * xsg4 + 520 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg4 - 240 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg4 - 520 * exp_xom_xsgplus * L * xom3 * xsg4 - 440 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg4 + 440 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg4 - 220 * exp_xom_xsgplus * L2 * xom3 * xsg4 + 220 * exp_xom_xsgplus * log2_xom * xom3 * xsg4 + 440 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg4 - 440 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg4 + 100 * eulergamma * exp_xom_xsgplus * xom4 * xsg4 - 100 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg4 + 440 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg4 + 100 * exp_xom_xsgplus * L * xom4 * xsg4 + 180 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg4 - 180 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg4 + 90 * exp_xom_xsgplus * L2 * xom4 * xsg4 - 90 * exp_xom_xsgplus * log2_xom * xom4 * xsg4 - 180 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg4 + 180 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg4 - 180 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg4 - 20 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg4 + 20 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg4 - 10 * exp_xom_xsgplus * L2 * xom5 * xsg4 + 10 * exp_xom_xsgplus * log2_xom * xom5 * xsg4 + 20 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg4 - 20 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg4 + 20 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg4 - 100 * xom * exp_xom_xsgplus * xsg5 + 6 * eulergamma * exp_xom_xsgplus * xsg5 - 80 * xom * eulergamma * exp_xom_xsgplus * xsg5 - 6 * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 + 80 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 + 30 * xom * hyp_xom_xsg * xsg5 + 6 * exp_xom_xsgplus * L * xsg5 - 80 * xom * exp_xom_xsgplus * L * xsg5 - 30 * xom * eulergamma * exp_xom_xsgplus * L * xsg5 + 30 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 - 15 * xom * exp_xom_xsgplus * L2 * xsg5 + 15 * xom * exp_xom_xsgplus * log2_xom * xsg5 + 30 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg5 - 30 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 + 120 * exp_xom_xsgplus * xom2 * xsg5 + 200 * eulergamma * exp_xom_xsgplus * xom2 * xsg5 - 200 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg5 + 30 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg5 + 200 * exp_xom_xsgplus * L * xom2 * xsg5 + 120 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg5 - 120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg5 + 60 * exp_xom_xsgplus * L2 * xom2 * xsg5 - 60 * exp_xom_xsgplus * log2_xom * xom2 * xsg5 - 120 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg5 + 120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg5 - 40 * exp_xom_xsgplus * xom3 * xsg5 - 120 * eulergamma * exp_xom_xsgplus * xom3 * xsg5 + 120 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg5 - 120 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg5 - 120 * exp_xom_xsgplus * L * xom3 * xsg5 - 120 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg5 + 120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg5 - 60 * exp_xom_xsgplus * L2 * xom3 * xsg5 + 60 * exp_xom_xsgplus * log2_xom * xom3 * xsg5 + 120 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg5 - 120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg5 + 20 * eulergamma * exp_xom_xsgplus * xom4 * xsg5 - 20 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg5 + 120 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg5 + 20 * exp_xom_xsgplus * L * xom4 * xsg5 + 40 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg5 - 40 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg5 + 20 * exp_xom_xsgplus * L2 * xom4 * xsg5 - 20 * exp_xom_xsgplus * log2_xom * xom4 * xsg5 - 40 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg5 + 40 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg5 - 40 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg5 - 4 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg5 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg5 - 2 * exp_xom_xsgplus * L2 * xom5 * xsg5 + 2 * exp_xom_xsgplus * log2_xom * xom5 * xsg5 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg5 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg5 + 4 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg5) * xsgplus_m6) / 15.,
            -0.022222222222222223 * (exp_xom_xsg * (346 + 140 * xom + 456 * xsg + 84 * xom * xsg - 276 * L - 90 * xom * L + 300 * xsg * L + 84 * xom * xsg * L + 276 * log_xom + 90 * xom * log_xom - 300 * xsg * log_xom - 84 * xom * xsg * log_xom + 152 * xom2 + 804 * xsg * xom2 + 48 * L * xom2 + 516 * xsg * L * xom2 - 48 * log_xom * xom2 - 516 * xsg * log_xom * xom2 - 72 * xom3 - 528 * xsg * xom3 - 92 * L * xom3 - 588 * xsg * L * xom3 + 92 * log_xom * xom3 + 588 * xsg * log_xom * xom3 + 20 * xom4 + 120 * xsg * xom4 + 32 * L * xom4 + 216 * xsg * L * xom4 - 32 * log_xom * xom4 - 216 * xsg * log_xom * xom4 - 4 * L * xom5 - 24 * xsg * L * xom5 + 4 * log_xom * xom5 + 24 * xsg * log_xom * xom5 + 570 * xsg2 - 360 * xom * xsg2 - 600 * L * xsg2 - 150 * xom * L * xsg2 + 600 * log_xom * xsg2 + 150 * xom * log_xom * xsg2 + 2100 * xom2 * xsg2 + 1500 * L * xom2 * xsg2 - 1500 * log_xom * xom2 * xsg2 - 1560 * xom3 * xsg2 - 1680 * L * xom3 * xsg2 + 1680 * log_xom * xom3 * xsg2 + 300 * xom4 * xsg2 + 600 * L * xom4 * xsg2 - 600 * log_xom * xom4 * xsg2 - 60 * L * xom5 * xsg2 + 60 * log_xom * xom5 * xsg2 + 1040 * xsg3 - 440 * xom * xsg3 - 600 * L * xsg3 - 600 * xom * L * xsg3 + 600 * log_xom * xsg3 + 600 * xom * log_xom * xsg3 + 3400 * xom2 * xsg3 + 2280 * L * xom2 * xsg3 - 2280 * log_xom * xom2 * xsg3 - 2400 * xom3 * xsg3 - 2680 * L * xom3 * xsg3 + 2680 * log_xom * xom3 * xsg3 + 400 * xom4 * xsg3 + 880 * L * xom4 * xsg3 - 880 * log_xom * xom4 * xsg3 - 80 * L * xom5 * xsg3 + 80 * log_xom * xom5 * xsg3 + 510 * xsg4 - 420 * xom * xsg4 + 300 * L * xsg4 - 150 * xom * L * xsg4 - 300 * log_xom * xsg4 + 150 * xom * log_xom * xsg4 + 3360 * xom2 * xsg4 + 2280 * L * xom2 * xsg4 - 2280 * log_xom * xom2 * xsg4 - 2040 * xom3 * xsg4 - 2460 * L * xom3 * xsg4 + 2460 * log_xom * xom3 * xsg4 + 300 * xom4 * xsg4 + 720 * L * xom4 * xsg4 - 720 * log_xom * xom4 * xsg4 - 60 * L * xom5 * xsg4 + 60 * log_xom * xom5 * xsg4 - 24 * xsg5 - 540 * xom * xsg5 - 276 * L * xsg5 - 60 * xom * L * xsg5 + 276 * log_xom * xsg5 + 60 * xom * log_xom * xsg5 + 1812 * xom2 * xsg5 + 1428 * L * xom2 * xsg5 - 1428 * log_xom * xom2 * xsg5 - 912 * xom3 * xsg5 - 1212 * L * xom3 * xsg5 + 1212 * log_xom * xom3 * xsg5 + 120 * xom4 * xsg5 + 312 * L * xom4 * xsg5 - 312 * log_xom * xom4 * xsg5 - 24 * L * xom5 * xsg5 + 24 * log_xom * xom5 * xsg5 + 46 * xsg6 - 256 * xom * xsg6 - 186 * xom * L * xsg6 + 186 * xom * log_xom * xsg6 + 404 * xom2 * xsg6 + 396 * L * xom2 * xsg6 - 396 * log_xom * xom2 * xsg6 - 168 * xom3 * xsg6 - 248 * L * xom3 * xsg6 + 248 * log_xom * xom3 * xsg6 + 20 * xom4 * xsg6 + 56 * L * xom4 * xsg6 - 56 * log_xom * xom4 * xsg6 - 4 * L * xom5 * xsg6 + 4 * log_xom * xom5 * xsg6) * xsgplus_m7) - ((-346 * exp_xom_xsgplus - 300 * xom * exp_xom_xsgplus - 456 * xsg * exp_xom_xsgplus - 480 * xom * xsg * exp_xom_xsgplus + 276 * eulergamma * exp_xom_xsgplus + 276 * xom * eulergamma * exp_xom_xsgplus - 300 * xsg * eulergamma * exp_xom_xsgplus - 24 * xom * xsg * eulergamma * exp_xom_xsgplus - 276 * expint_ei_plus_xom * exp_xom_xsgplus - 276 * xom * expint_ei_plus_xom * exp_xom_xsgplus + 300 * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 24 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 90 * xom * exp_xom_xsgplus * hyp_plus_xom - 450 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 90 * xom * xsg * hyp_xom_xsg + 90 * eulergamma * L - 450 * xsg * eulergamma * L - 90 * expint_ei_xom_xsg * L + 450 * xsg * expint_ei_xom_xsg * L + 276 * exp_xom_xsgplus * L + 276 * xom * exp_xom_xsgplus * L - 300 * xsg * exp_xom_xsgplus * L - 24 * xom * xsg * exp_xom_xsgplus * L - 90 * eulergamma * exp_xom_xsgplus * L - 90 * xom * eulergamma * exp_xom_xsgplus * L + 450 * xsg * eulergamma * exp_xom_xsgplus * L + 360 * xom * xsg * eulergamma * exp_xom_xsgplus * L + 90 * expint_ei_plus_xom * exp_xom_xsgplus * L + 90 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 450 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L - 360 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + 45 * L2 - 225 * xsg * L2 - 45 * exp_xom_xsgplus * L2 - 45 * xom * exp_xom_xsgplus * L2 + 225 * xsg * exp_xom_xsgplus * L2 + 180 * xom * xsg * exp_xom_xsgplus * L2 - 45 * log2_xom + 225 * xsg * log2_xom + 45 * exp_xom_xsgplus * log2_xom + 45 * xom * exp_xom_xsgplus * log2_xom - 225 * xsg * exp_xom_xsgplus * log2_xom - 180 * xom * xsg * exp_xom_xsgplus * log2_xom - 90 * eulergamma * log_xom + 450 * xsg * eulergamma * log_xom + 90 * expint_ei_xom_xsg * log_xom - 450 * xsg * expint_ei_xom_xsg * log_xom + 90 * eulergamma * exp_xom_xsgplus * log_xom + 90 * xom * eulergamma * exp_xom_xsgplus * log_xom - 450 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 360 * xom * xsg * eulergamma * exp_xom_xsgplus * log_xom - 90 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 90 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 450 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 360 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 90 * L * log_xsg - 450 * xsg * L * log_xsg - 90 * log_xom * log_xsg + 450 * xsg * log_xom * log_xsg - 300 * exp_xom_xsgplus * xom2 - 1440 * xsg * exp_xom_xsgplus * xom2 - 840 * xsg * eulergamma * exp_xom_xsgplus * xom2 + 840 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom2 + 90 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 360 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2 - 840 * xsg * exp_xom_xsgplus * L * xom2 - 90 * eulergamma * exp_xom_xsgplus * L * xom2 - 180 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 + 90 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 180 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 - 45 * exp_xom_xsgplus * L2 * xom2 - 90 * xsg * exp_xom_xsgplus * L2 * xom2 + 45 * exp_xom_xsgplus * log2_xom * xom2 + 90 * xsg * exp_xom_xsgplus * log2_xom * xom2 + 90 * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 180 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 - 90 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 180 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 + 120 * exp_xom_xsgplus * xom3 + 960 * xsg * exp_xom_xsgplus * xom3 + 280 * eulergamma * exp_xom_xsgplus * xom3 + 1680 * xsg * eulergamma * exp_xom_xsgplus * xom3 - 280 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 - 1680 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom3 + 90 * exp_xom_xsgplus * hyp_plus_xom * xom3 + 180 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 + 280 * exp_xom_xsgplus * L * xom3 + 1680 * xsg * exp_xom_xsgplus * L * xom3 + 120 * eulergamma * exp_xom_xsgplus * L * xom3 + 960 * xsg * eulergamma * exp_xom_xsgplus * L * xom3 - 120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 - 960 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 + 60 * exp_xom_xsgplus * L2 * xom3 + 480 * xsg * exp_xom_xsgplus * L2 * xom3 - 60 * exp_xom_xsgplus * log2_xom * xom3 - 480 * xsg * exp_xom_xsgplus * log2_xom * xom3 - 120 * eulergamma * exp_xom_xsgplus * log_xom * xom3 - 960 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom3 + 120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 + 960 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 - 60 * exp_xom_xsgplus * xom4 - 360 * xsg * exp_xom_xsgplus * xom4 - 120 * eulergamma * exp_xom_xsgplus * xom4 - 840 * xsg * eulergamma * exp_xom_xsgplus * xom4 + 120 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 + 840 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom4 - 120 * exp_xom_xsgplus * hyp_plus_xom * xom4 - 960 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom4 - 120 * exp_xom_xsgplus * L * xom4 - 840 * xsg * exp_xom_xsgplus * L * xom4 - 120 * eulergamma * exp_xom_xsgplus * L * xom4 - 780 * xsg * eulergamma * exp_xom_xsgplus * L * xom4 + 120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 + 780 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 - 60 * exp_xom_xsgplus * L2 * xom4 - 390 * xsg * exp_xom_xsgplus * L2 * xom4 + 60 * exp_xom_xsgplus * log2_xom * xom4 + 390 * xsg * exp_xom_xsgplus * log2_xom * xom4 + 120 * eulergamma * exp_xom_xsgplus * log_xom * xom4 + 780 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom4 - 120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 - 780 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 + 24 * eulergamma * exp_xom_xsgplus * xom5 + 144 * xsg * eulergamma * exp_xom_xsgplus * xom5 - 24 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 - 144 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom5 + 120 * exp_xom_xsgplus * hyp_plus_xom * xom5 + 780 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom5 + 24 * exp_xom_xsgplus * L * xom5 + 144 * xsg * exp_xom_xsgplus * L * xom5 + 36 * eulergamma * exp_xom_xsgplus * L * xom5 + 240 * xsg * eulergamma * exp_xom_xsgplus * L * xom5 - 36 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 - 240 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 + 18 * exp_xom_xsgplus * L2 * xom5 + 120 * xsg * exp_xom_xsgplus * L2 * xom5 - 18 * exp_xom_xsgplus * log2_xom * xom5 - 120 * xsg * exp_xom_xsgplus * log2_xom * xom5 - 36 * eulergamma * exp_xom_xsgplus * log_xom * xom5 - 240 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom5 + 36 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 + 240 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 - 36 * exp_xom_xsgplus * hyp_plus_xom * xom6 - 240 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom6 - 4 * eulergamma * exp_xom_xsgplus * L * xom6 - 24 * xsg * eulergamma * exp_xom_xsgplus * L * xom6 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 + 24 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 - 2 * exp_xom_xsgplus * L2 * xom6 - 12 * xsg * exp_xom_xsgplus * L2 * xom6 + 2 * exp_xom_xsgplus * log2_xom * xom6 + 12 * xsg * exp_xom_xsgplus * log2_xom * xom6 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom6 + 24 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom6 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 - 24 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 + 4 * exp_xom_xsgplus * hyp_plus_xom * xom7 + 24 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom7 - 570 * exp_xom_xsgplus * xsg2 - 60 * xom * exp_xom_xsgplus * xsg2 + 600 * eulergamma * exp_xom_xsgplus * xsg2 + 300 * xom * eulergamma * exp_xom_xsgplus * xsg2 - 600 * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 - 300 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 900 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg2 - 450 * xom * hyp_xom_xsg * xsg2 + 900 * eulergamma * L * xsg2 - 900 * expint_ei_xom_xsg * L * xsg2 + 600 * exp_xom_xsgplus * L * xsg2 + 300 * xom * exp_xom_xsgplus * L * xsg2 - 900 * eulergamma * exp_xom_xsgplus * L * xsg2 - 450 * xom * eulergamma * exp_xom_xsgplus * L * xsg2 + 900 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 450 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 450 * L2 * xsg2 - 450 * exp_xom_xsgplus * L2 * xsg2 - 225 * xom * exp_xom_xsgplus * L2 * xsg2 - 450 * log2_xom * xsg2 + 450 * exp_xom_xsgplus * log2_xom * xsg2 + 225 * xom * exp_xom_xsgplus * log2_xom * xsg2 - 900 * eulergamma * log_xom * xsg2 + 900 * expint_ei_xom_xsg * log_xom * xsg2 + 900 * eulergamma * exp_xom_xsgplus * log_xom * xsg2 + 450 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 900 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 - 450 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 900 * L * log_xsg * xsg2 - 900 * log_xom * log_xsg * xsg2 - 3420 * exp_xom_xsgplus * xom2 * xsg2 - 2760 * eulergamma * exp_xom_xsgplus * xom2 * xsg2 + 2760 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg2 + 450 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg2 - 2760 * exp_xom_xsgplus * L * xom2 * xsg2 - 990 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 + 990 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 - 495 * exp_xom_xsgplus * L2 * xom2 * xsg2 + 495 * exp_xom_xsgplus * log2_xom * xom2 * xsg2 + 990 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 990 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 3000 * exp_xom_xsgplus * xom3 * xsg2 + 4680 * eulergamma * exp_xom_xsgplus * xom3 * xsg2 - 4680 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg2 + 990 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2 + 4680 * exp_xom_xsgplus * L * xom3 * xsg2 + 2760 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg2 - 2760 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg2 + 1380 * exp_xom_xsgplus * L2 * xom3 * xsg2 - 1380 * exp_xom_xsgplus * log2_xom * xom3 * xsg2 - 2760 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg2 + 2760 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg2 - 900 * exp_xom_xsgplus * xom4 * xsg2 - 2400 * eulergamma * exp_xom_xsgplus * xom4 * xsg2 + 2400 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg2 - 2760 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg2 - 2400 * exp_xom_xsgplus * L * xom4 * xsg2 - 2220 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg2 + 2220 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg2 - 1110 * exp_xom_xsgplus * L2 * xom4 * xsg2 + 1110 * exp_xom_xsgplus * log2_xom * xom4 * xsg2 + 2220 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg2 - 2220 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg2 + 360 * eulergamma * exp_xom_xsgplus * xom5 * xsg2 - 360 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg2 + 2220 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg2 + 360 * exp_xom_xsgplus * L * xom5 * xsg2 + 660 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg2 - 660 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg2 + 330 * exp_xom_xsgplus * L2 * xom5 * xsg2 - 330 * exp_xom_xsgplus * log2_xom * xom5 * xsg2 - 660 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg2 + 660 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg2 - 660 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg2 - 60 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg2 + 60 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg2 - 30 * exp_xom_xsgplus * L2 * xom6 * xsg2 + 30 * exp_xom_xsgplus * log2_xom * xom6 * xsg2 + 60 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg2 - 60 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg2 + 60 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg2 - 1040 * exp_xom_xsgplus * xsg3 + 600 * eulergamma * exp_xom_xsgplus * xsg3 + 1200 * xom * eulergamma * exp_xom_xsgplus * xsg3 - 600 * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 1200 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 900 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg3 + 900 * xom * hyp_xom_xsg * xsg3 - 900 * eulergamma * L * xsg3 + 900 * expint_ei_xom_xsg * L * xsg3 + 600 * exp_xom_xsgplus * L * xsg3 + 1200 * xom * exp_xom_xsgplus * L * xsg3 + 900 * eulergamma * exp_xom_xsgplus * L * xsg3 - 900 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 450 * L2 * xsg3 + 450 * exp_xom_xsgplus * L2 * xsg3 + 450 * log2_xom * xsg3 - 450 * exp_xom_xsgplus * log2_xom * xsg3 + 900 * eulergamma * log_xom * xsg3 - 900 * expint_ei_xom_xsg * log_xom * xsg3 - 900 * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 900 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 900 * L * log_xsg * xsg3 + 900 * log_xom * log_xsg * xsg3 - 5280 * exp_xom_xsgplus * xom2 * xsg3 - 4080 * eulergamma * exp_xom_xsgplus * xom2 * xsg3 + 4080 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg3 - 4080 * exp_xom_xsgplus * L * xom2 * xsg3 - 1800 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg3 + 1800 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg3 - 900 * exp_xom_xsgplus * L2 * xom2 * xsg3 + 900 * exp_xom_xsgplus * log2_xom * xom2 * xsg3 + 1800 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg3 - 1800 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg3 + 4800 * exp_xom_xsgplus * xom3 * xsg3 + 7520 * eulergamma * exp_xom_xsgplus * xom3 * xsg3 - 7520 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg3 + 1800 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg3 + 7520 * exp_xom_xsgplus * L * xom3 * xsg3 + 4320 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg3 - 4320 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg3 + 2160 * exp_xom_xsgplus * L2 * xom3 * xsg3 - 2160 * exp_xom_xsgplus * log2_xom * xom3 * xsg3 - 4320 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg3 + 4320 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg3 - 1200 * exp_xom_xsgplus * xom4 * xsg3 - 3600 * eulergamma * exp_xom_xsgplus * xom4 * xsg3 + 3600 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg3 - 4320 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg3 - 3600 * exp_xom_xsgplus * L * xom4 * xsg3 - 3480 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg3 + 3480 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg3 - 1740 * exp_xom_xsgplus * L2 * xom4 * xsg3 + 1740 * exp_xom_xsgplus * log2_xom * xom4 * xsg3 + 3480 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg3 - 3480 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg3 + 480 * eulergamma * exp_xom_xsgplus * xom5 * xsg3 - 480 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg3 + 3480 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg3 + 480 * exp_xom_xsgplus * L * xom5 * xsg3 + 960 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg3 - 960 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg3 + 480 * exp_xom_xsgplus * L2 * xom5 * xsg3 - 480 * exp_xom_xsgplus * log2_xom * xom5 * xsg3 - 960 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg3 + 960 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg3 - 960 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg3 - 80 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg3 + 80 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg3 - 40 * exp_xom_xsgplus * L2 * xom6 * xsg3 + 40 * exp_xom_xsgplus * log2_xom * xom6 * xsg3 + 80 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg3 - 80 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg3 + 80 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg3 - 510 * exp_xom_xsgplus * xsg4 + 60 * xom * exp_xom_xsgplus * xsg4 - 300 * eulergamma * exp_xom_xsgplus * xsg4 + 300 * xom * eulergamma * exp_xom_xsgplus * xsg4 + 300 * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 - 300 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 + 450 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg4 - 900 * xom * hyp_xom_xsg * xsg4 + 450 * eulergamma * L * xsg4 - 450 * expint_ei_xom_xsg * L * xsg4 - 300 * exp_xom_xsgplus * L * xsg4 + 300 * xom * exp_xom_xsgplus * L * xsg4 - 450 * eulergamma * exp_xom_xsgplus * L * xsg4 + 450 * xom * eulergamma * exp_xom_xsgplus * L * xsg4 + 450 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 - 450 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 + 225 * L2 * xsg4 - 225 * exp_xom_xsgplus * L2 * xsg4 + 225 * xom * exp_xom_xsgplus * L2 * xsg4 - 225 * log2_xom * xsg4 + 225 * exp_xom_xsgplus * log2_xom * xsg4 - 225 * xom * exp_xom_xsgplus * log2_xom * xsg4 - 450 * eulergamma * log_xom * xsg4 + 450 * expint_ei_xom_xsg * log_xom * xsg4 + 450 * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 450 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 450 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 450 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 450 * L * log_xsg * xsg4 - 450 * log_xom * log_xsg * xsg4 - 5220 * exp_xom_xsgplus * xom2 * xsg4 - 4080 * eulergamma * exp_xom_xsgplus * xom2 * xsg4 + 4080 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg4 - 450 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg4 - 4080 * exp_xom_xsgplus * L * xom2 * xsg4 - 1350 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg4 + 1350 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg4 - 675 * exp_xom_xsgplus * L2 * xom2 * xsg4 + 675 * exp_xom_xsgplus * log2_xom * xom2 * xsg4 + 1350 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg4 - 1350 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg4 + 4200 * exp_xom_xsgplus * xom3 * xsg4 + 7080 * eulergamma * exp_xom_xsgplus * xom3 * xsg4 - 7080 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg4 + 1350 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg4 + 7080 * exp_xom_xsgplus * L * xom3 * xsg4 + 4200 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg4 - 4200 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg4 + 2100 * exp_xom_xsgplus * L2 * xom3 * xsg4 - 2100 * exp_xom_xsgplus * log2_xom * xom3 * xsg4 - 4200 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg4 + 4200 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg4 - 900 * exp_xom_xsgplus * xom4 * xsg4 - 3000 * eulergamma * exp_xom_xsgplus * xom4 * xsg4 + 3000 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg4 - 4200 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg4 - 3000 * exp_xom_xsgplus * L * xom4 * xsg4 - 3120 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg4 + 3120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg4 - 1560 * exp_xom_xsgplus * L2 * xom4 * xsg4 + 1560 * exp_xom_xsgplus * log2_xom * xom4 * xsg4 + 3120 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg4 - 3120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg4 + 360 * eulergamma * exp_xom_xsgplus * xom5 * xsg4 - 360 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg4 + 3120 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg4 + 360 * exp_xom_xsgplus * L * xom5 * xsg4 + 780 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg4 - 780 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg4 + 390 * exp_xom_xsgplus * L2 * xom5 * xsg4 - 390 * exp_xom_xsgplus * log2_xom * xom5 * xsg4 - 780 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg4 + 780 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg4 - 780 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg4 - 60 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg4 + 60 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg4 - 30 * exp_xom_xsgplus * L2 * xom6 * xsg4 + 30 * exp_xom_xsgplus * log2_xom * xom6 * xsg4 + 60 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg4 - 60 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg4 + 60 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg4 + 24 * exp_xom_xsgplus * xsg5 + 480 * xom * exp_xom_xsgplus * xsg5 + 276 * eulergamma * exp_xom_xsgplus * xsg5 - 24 * xom * eulergamma * exp_xom_xsgplus * xsg5 - 276 * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 + 24 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 - 90 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg5 + 450 * xom * hyp_xom_xsg * xsg5 - 90 * eulergamma * L * xsg5 + 90 * expint_ei_xom_xsg * L * xsg5 + 276 * exp_xom_xsgplus * L * xsg5 - 24 * xom * exp_xom_xsgplus * L * xsg5 + 90 * eulergamma * exp_xom_xsgplus * L * xsg5 - 360 * xom * eulergamma * exp_xom_xsgplus * L * xsg5 - 90 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 + 360 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 - 45 * L2 * xsg5 + 45 * exp_xom_xsgplus * L2 * xsg5 - 180 * xom * exp_xom_xsgplus * L2 * xsg5 + 45 * log2_xom * xsg5 - 45 * exp_xom_xsgplus * log2_xom * xsg5 + 180 * xom * exp_xom_xsgplus * log2_xom * xsg5 + 90 * eulergamma * log_xom * xsg5 - 90 * expint_ei_xom_xsg * log_xom * xsg5 - 90 * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 360 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 90 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 360 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 90 * L * log_xsg * xsg5 + 90 * log_xom * log_xsg * xsg5 - 2880 * exp_xom_xsgplus * xom2 * xsg5 - 2760 * eulergamma * exp_xom_xsgplus * xom2 * xsg5 + 2760 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg5 + 360 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg5 - 2760 * exp_xom_xsgplus * L * xom2 * xsg5 - 900 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg5 + 900 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg5 - 450 * exp_xom_xsgplus * L2 * xom2 * xsg5 + 450 * exp_xom_xsgplus * log2_xom * xom2 * xsg5 + 900 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg5 - 900 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg5 + 1920 * exp_xom_xsgplus * xom3 * xsg5 + 3600 * eulergamma * exp_xom_xsgplus * xom3 * xsg5 - 3600 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg5 + 900 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg5 + 3600 * exp_xom_xsgplus * L * xom3 * xsg5 + 2400 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg5 - 2400 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg5 + 1200 * exp_xom_xsgplus * L2 * xom3 * xsg5 - 1200 * exp_xom_xsgplus * log2_xom * xom3 * xsg5 - 2400 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg5 + 2400 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg5 - 360 * exp_xom_xsgplus * xom4 * xsg5 - 1320 * eulergamma * exp_xom_xsgplus * xom4 * xsg5 + 1320 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg5 - 2400 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg5 - 1320 * exp_xom_xsgplus * L * xom4 * xsg5 - 1500 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg5 + 1500 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg5 - 750 * exp_xom_xsgplus * L2 * xom4 * xsg5 + 750 * exp_xom_xsgplus * log2_xom * xom4 * xsg5 + 1500 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg5 - 1500 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg5 + 144 * eulergamma * exp_xom_xsgplus * xom5 * xsg5 - 144 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg5 + 1500 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg5 + 144 * exp_xom_xsgplus * L * xom5 * xsg5 + 336 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg5 - 336 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg5 + 168 * exp_xom_xsgplus * L2 * xom5 * xsg5 - 168 * exp_xom_xsgplus * log2_xom * xom5 * xsg5 - 336 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg5 + 336 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg5 - 336 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg5 - 24 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg5 + 24 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg5 - 12 * exp_xom_xsgplus * L2 * xom6 * xsg5 + 12 * exp_xom_xsgplus * log2_xom * xom6 * xsg5 + 24 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg5 - 24 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg5 + 24 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg5 - 46 * exp_xom_xsgplus * xsg6 + 300 * xom * exp_xom_xsgplus * xsg6 + 276 * xom * eulergamma * exp_xom_xsgplus * xsg6 - 276 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg6 - 90 * xom * hyp_xom_xsg * xsg6 + 276 * xom * exp_xom_xsgplus * L * xsg6 + 90 * xom * eulergamma * exp_xom_xsgplus * L * xsg6 - 90 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg6 + 45 * xom * exp_xom_xsgplus * L2 * xsg6 - 45 * xom * exp_xom_xsgplus * log2_xom * xsg6 - 90 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg6 + 90 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg6 - 660 * exp_xom_xsgplus * xom2 * xsg6 - 840 * eulergamma * exp_xom_xsgplus * xom2 * xsg6 + 840 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg6 - 90 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg6 - 840 * exp_xom_xsgplus * L * xom2 * xsg6 - 450 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg6 + 450 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg6 - 225 * exp_xom_xsgplus * L2 * xom2 * xsg6 + 225 * exp_xom_xsgplus * log2_xom * xom2 * xsg6 + 450 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg6 - 450 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg6 + 360 * exp_xom_xsgplus * xom3 * xsg6 + 760 * eulergamma * exp_xom_xsgplus * xom3 * xsg6 - 760 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg6 + 450 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg6 + 760 * exp_xom_xsgplus * L * xom3 * xsg6 + 600 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg6 - 600 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg6 + 300 * exp_xom_xsgplus * L2 * xom3 * xsg6 - 300 * exp_xom_xsgplus * log2_xom * xom3 * xsg6 - 600 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg6 + 600 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg6 - 60 * exp_xom_xsgplus * xom4 * xsg6 - 240 * eulergamma * exp_xom_xsgplus * xom4 * xsg6 + 240 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg6 - 600 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg6 - 240 * exp_xom_xsgplus * L * xom4 * xsg6 - 300 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg6 + 300 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg6 - 150 * exp_xom_xsgplus * L2 * xom4 * xsg6 + 150 * exp_xom_xsgplus * log2_xom * xom4 * xsg6 + 300 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg6 - 300 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg6 + 24 * eulergamma * exp_xom_xsgplus * xom5 * xsg6 - 24 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg6 + 300 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg6 + 24 * exp_xom_xsgplus * L * xom5 * xsg6 + 60 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg6 - 60 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg6 + 30 * exp_xom_xsgplus * L2 * xom5 * xsg6 - 30 * exp_xom_xsgplus * log2_xom * xom5 * xsg6 - 60 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg6 + 60 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg6 - 60 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg6 - 4 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg6 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg6 - 2 * exp_xom_xsgplus * L2 * xom6 * xsg6 + 2 * exp_xom_xsgplus * log2_xom * xom6 * xsg6 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg6 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg6 + 4 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg6) * xsgplus_m7) / 45.,
            (exp_xom_xsg * (2744 + 1392 * xom + 3024 * xsg + 2492 * xom * xsg - 2022 * L - 540 * xom * L + 3402 * xsg * L + 1848 * xom * xsg * L + 2022 * log_xom + 540 * xom * log_xom - 3402 * xsg * log_xom - 1848 * xom * xsg * log_xom - 484 * xom2 - 6272 * xsg * xom2 - 696 * L * xom2 - 3948 * xsg * L * xom2 + 696 * log_xom * xom2 + 3948 * xsg * log_xom * xom2 + 1088 * xom3 + 7952 * xsg * xom3 + 848 * L * xom3 + 6972 * xsg * L * xom3 - 848 * log_xom * xom3 - 6972 * xsg * log_xom * xom3 - 360 * xom4 - 2800 * xsg * xom4 - 492 * L * xom4 - 3724 * xsg * L * xom4 + 492 * log_xom * xom4 + 3724 * xsg * log_xom * xom4 + 48 * xom5 + 336 * xsg * xom5 + 104 * L * xom5 + 784 * xsg * L * xom5 - 104 * log_xom * xom5 - 784 * xsg * log_xom * xom5 - 8 * L * xom6 - 56 * xsg * L * xom6 + 8 * log_xom * xom6 + 56 * xsg * log_xom * xom6 + 7560 * xsg2 + 5544 * xom * xsg2 - 8190 * L * xsg2 + 252 * xom * L * xsg2 + 8190 * log_xom * xsg2 - 252 * xom * log_xom * xsg2 - 23436 * xom2 * xsg2 - 14784 * L * xom2 * xsg2 + 14784 * log_xom * xom2 * xsg2 + 26208 * xom3 * xsg2 + 23688 * L * xom3 * xsg2 - 23688 * log_xom * xom3 * xsg2 - 9240 * xom4 * xsg2 - 12348 * L * xom4 * xsg2 + 12348 * log_xom * xom4 * xsg2 + 1008 * xom5 * xsg2 + 2520 * L * xom5 * xsg2 - 2520 * log_xom * xom5 * xsg2 - 168 * L * xom6 * xsg2 + 168 * log_xom * xom6 * xsg2 + 14560 * xsg3 + 13860 * xom * xsg3 - 3150 * L * xsg3 + 3150 * log_xom * xsg3 - 45080 * xom2 * xsg3 - 30660 * L * xom2 * xsg3 + 30660 * log_xom * xom2 * xsg3 + 49840 * xom3 * xsg3 + 45220 * L * xom3 * xsg3 - 45220 * log_xom * xom3 * xsg3 - 16800 * xom4 * xsg3 - 23100 * L * xom4 * xsg3 + 23100 * log_xom * xom4 * xsg3 + 1680 * xom5 * xsg3 + 4480 * L * xom5 * xsg3 - 4480 * log_xom * xom5 * xsg3 - 280 * L * xom6 * xsg3 + 280 * log_xom * xom6 * xsg3 + 7560 * xsg4 + 14560 * xom * xsg4 + 3150 * L * xsg4 + 6300 * xom * L * xsg4 - 3150 * log_xom * xsg4 - 6300 * xom * log_xom * xsg4 - 54460 * xom2 * xsg4 - 36120 * L * xom2 * xsg4 + 36120 * log_xom * xom2 * xsg4 + 58240 * xom3 * xsg4 + 53760 * L * xom3 * xsg4 - 53760 * log_xom * xom3 * xsg4 - 18200 * xom4 * xsg4 - 26180 * L * xom4 * xsg4 + 26180 * log_xom * xom4 * xsg4 + 1680 * xom5 * xsg4 + 4760 * L * xom5 * xsg4 - 4760 * log_xom * xom5 * xsg4 - 280 * L * xom6 * xsg4 + 280 * log_xom * xom6 * xsg4 + 3024 * xsg5 + 7812 * xom * xsg5 - 5922 * L * xsg5 + 2520 * xom * L * xsg5 + 5922 * log_xom * xsg5 - 2520 * xom * log_xom * xsg5 - 43344 * xom2 * xsg5 - 27636 * L * xom2 * xsg5 + 27636 * log_xom * xom2 * xsg5 + 41328 * xom3 * xsg5 + 40068 * L * xom3 * xsg5 - 40068 * log_xom * xom3 * xsg5 - 11760 * xom4 * xsg5 - 17892 * L * xom4 * xsg5 + 17892 * log_xom * xom4 * xsg5 + 1008 * xom5 * xsg5 + 3024 * L * xom5 * xsg5 - 3024 * log_xom * xom5 * xsg5 - 168 * L * xom6 * xsg5 + 168 * log_xom * xom6 * xsg5 + 2744 * xsg6 + 4872 * xom * xsg6 + 1302 * L * xsg6 - 252 * xom * L * xsg6 - 1302 * log_xom * xsg6 + 252 * xom * log_xom * xsg6 - 20692 * xom2 * xsg6 - 14448 * L * xom2 * xsg6 + 14448 * log_xom * xom2 * xsg6 + 16352 * xom3 * xsg6 + 17192 * L * xom3 * xsg6 - 17192 * log_xom * xom3 * xsg6 - 4200 * xom4 * xsg6 - 6804 * L * xom4 * xsg6 + 6804 * log_xom * xom4 * xsg6 + 336 * xom5 * xsg6 + 1064 * L * xom5 * xsg6 - 1064 * log_xom * xom5 * xsg6 - 56 * L * xom6 * xsg6 + 56 * log_xom * xom6 * xsg6 + 2204 * xom * xsg7 - 90 * L * xsg7 + 1392 * xom * L * xsg7 + 90 * log_xom * xsg7 - 1392 * xom * log_xom * xsg7 - 4376 * xom2 * xsg7 - 3804 * L * xom2 * xsg7 + 3804 * log_xom * xom2 * xsg7 + 2768 * xom3 * xsg7 + 3228 * L * xom3 * xsg7 - 3228 * log_xom * xom3 * xsg7 - 640 * xom4 * xsg7 - 1108 * L * xom4 * xsg7 + 1108 * log_xom * xom4 * xsg7 + 48 * xom5 * xsg7 + 160 * L * xom5 * xsg7 - 160 * log_xom * xom5 * xsg7 - 8 * L * xom6 * xsg7 + 8 * log_xom * xom6 * xsg7) * xsgplus_m8) / 315. + ((-2744 * exp_xom_xsgplus - 2744 * xom * exp_xom_xsgplus - 3024 * xsg * exp_xom_xsgplus - 5768 * xom * xsg * exp_xom_xsgplus + 2022 * eulergamma * exp_xom_xsgplus + 1932 * xom * eulergamma * exp_xom_xsgplus - 3402 * xsg * eulergamma * exp_xom_xsgplus - 2100 * xom * xsg * eulergamma * exp_xom_xsgplus - 2022 * expint_ei_plus_xom * exp_xom_xsgplus - 1932 * xom * expint_ei_plus_xom * exp_xom_xsgplus + 3402 * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 2100 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 630 * xom * exp_xom_xsgplus * hyp_plus_xom - 3780 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 630 * xom * xsg * hyp_xom_xsg + 630 * eulergamma * L - 3780 * xsg * eulergamma * L - 630 * expint_ei_xom_xsg * L + 3780 * xsg * expint_ei_xom_xsg * L + 2022 * exp_xom_xsgplus * L + 1932 * xom * exp_xom_xsgplus * L - 3402 * xsg * exp_xom_xsgplus * L - 2100 * xom * xsg * exp_xom_xsgplus * L - 630 * eulergamma * exp_xom_xsgplus * L - 630 * xom * eulergamma * exp_xom_xsgplus * L + 3780 * xsg * eulergamma * exp_xom_xsgplus * L + 3150 * xom * xsg * eulergamma * exp_xom_xsgplus * L + 630 * expint_ei_plus_xom * exp_xom_xsgplus * L + 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 3780 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L - 3150 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + 315 * L2 - 1890 * xsg * L2 - 315 * exp_xom_xsgplus * L2 - 315 * xom * exp_xom_xsgplus * L2 + 1890 * xsg * exp_xom_xsgplus * L2 + 1575 * xom * xsg * exp_xom_xsgplus * L2 - 315 * log2_xom + 1890 * xsg * log2_xom + 315 * exp_xom_xsgplus * log2_xom + 315 * xom * exp_xom_xsgplus * log2_xom - 1890 * xsg * exp_xom_xsgplus * log2_xom - 1575 * xom * xsg * exp_xom_xsgplus * log2_xom - 630 * eulergamma * log_xom + 3780 * xsg * eulergamma * log_xom + 630 * expint_ei_xom_xsg * log_xom - 3780 * xsg * expint_ei_xom_xsg * log_xom + 630 * eulergamma * exp_xom_xsgplus * log_xom + 630 * xom * eulergamma * exp_xom_xsgplus * log_xom - 3780 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 3150 * xom * xsg * eulergamma * exp_xom_xsgplus * log_xom - 630 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 3780 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 3150 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 630 * L * log_xsg - 3780 * xsg * L * log_xsg - 630 * log_xom * log_xsg + 3780 * xsg * log_xom * log_xsg + 6720 * xsg * exp_xom_xsgplus * xom2 + 1932 * eulergamma * exp_xom_xsgplus * xom2 + 7644 * xsg * eulergamma * exp_xom_xsgplus * xom2 - 1932 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 - 7644 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom2 + 630 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 3150 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2 + 1932 * exp_xom_xsgplus * L * xom2 + 7644 * xsg * exp_xom_xsgplus * L * xom2 + 3780 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 - 3780 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 1890 * xsg * exp_xom_xsgplus * L2 * xom2 - 1890 * xsg * exp_xom_xsgplus * log2_xom * xom2 - 3780 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 3780 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 2240 * exp_xom_xsgplus * xom3 - 15680 * xsg * exp_xom_xsgplus * xom3 - 1960 * eulergamma * exp_xom_xsgplus * xom3 - 17640 * xsg * eulergamma * exp_xom_xsgplus * xom3 + 1960 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 + 17640 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom3 - 3780 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 - 1960 * exp_xom_xsgplus * L * xom3 - 17640 * xsg * exp_xom_xsgplus * L * xom3 - 1260 * eulergamma * exp_xom_xsgplus * L * xom3 - 8820 * xsg * eulergamma * exp_xom_xsgplus * L * xom3 + 1260 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 + 8820 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 - 630 * exp_xom_xsgplus * L2 * xom3 - 4410 * xsg * exp_xom_xsgplus * L2 * xom3 + 630 * exp_xom_xsgplus * log2_xom * xom3 + 4410 * xsg * exp_xom_xsgplus * log2_xom * xom3 + 1260 * eulergamma * exp_xom_xsgplus * log_xom * xom3 + 8820 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom3 - 1260 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 - 8820 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 + 840 * exp_xom_xsgplus * xom4 + 6720 * xsg * exp_xom_xsgplus * xom4 + 1820 * eulergamma * exp_xom_xsgplus * xom4 + 13580 * xsg * eulergamma * exp_xom_xsgplus * xom4 - 1820 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 - 13580 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom4 + 1260 * exp_xom_xsgplus * hyp_plus_xom * xom4 + 8820 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom4 + 1820 * exp_xom_xsgplus * L * xom4 + 13580 * xsg * exp_xom_xsgplus * L * xom4 + 1260 * eulergamma * exp_xom_xsgplus * L * xom4 + 10080 * xsg * eulergamma * exp_xom_xsgplus * L * xom4 - 1260 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 - 10080 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 + 630 * exp_xom_xsgplus * L2 * xom4 + 5040 * xsg * exp_xom_xsgplus * L2 * xom4 - 630 * exp_xom_xsgplus * log2_xom * xom4 - 5040 * xsg * exp_xom_xsgplus * log2_xom * xom4 - 1260 * eulergamma * exp_xom_xsgplus * log_xom * xom4 - 10080 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom4 + 1260 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 + 10080 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 - 168 * exp_xom_xsgplus * xom5 - 1176 * xsg * exp_xom_xsgplus * xom5 - 504 * eulergamma * exp_xom_xsgplus * xom5 - 3864 * xsg * eulergamma * exp_xom_xsgplus * xom5 + 504 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 + 3864 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom5 - 1260 * exp_xom_xsgplus * hyp_plus_xom * xom5 - 10080 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom5 - 504 * exp_xom_xsgplus * L * xom5 - 3864 * xsg * exp_xom_xsgplus * L * xom5 - 588 * eulergamma * exp_xom_xsgplus * L * xom5 - 4452 * xsg * eulergamma * exp_xom_xsgplus * L * xom5 + 588 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 + 4452 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 - 294 * exp_xom_xsgplus * L2 * xom5 - 2226 * xsg * exp_xom_xsgplus * L2 * xom5 + 294 * exp_xom_xsgplus * log2_xom * xom5 + 2226 * xsg * exp_xom_xsgplus * log2_xom * xom5 + 588 * eulergamma * exp_xom_xsgplus * log_xom * xom5 + 4452 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom5 - 588 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 - 4452 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 + 56 * eulergamma * exp_xom_xsgplus * xom6 + 392 * xsg * eulergamma * exp_xom_xsgplus * xom6 - 56 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 - 392 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom6 + 588 * exp_xom_xsgplus * hyp_plus_xom * xom6 + 4452 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom6 + 56 * exp_xom_xsgplus * L * xom6 + 392 * xsg * exp_xom_xsgplus * L * xom6 + 112 * eulergamma * exp_xom_xsgplus * L * xom6 + 840 * xsg * eulergamma * exp_xom_xsgplus * L * xom6 - 112 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 - 840 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 + 56 * exp_xom_xsgplus * L2 * xom6 + 420 * xsg * exp_xom_xsgplus * L2 * xom6 - 56 * exp_xom_xsgplus * log2_xom * xom6 - 420 * xsg * exp_xom_xsgplus * log2_xom * xom6 - 112 * eulergamma * exp_xom_xsgplus * log_xom * xom6 - 840 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom6 + 112 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 + 840 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 - 112 * exp_xom_xsgplus * hyp_plus_xom * xom7 - 840 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom7 - 8 * eulergamma * exp_xom_xsgplus * L * xom7 - 56 * xsg * eulergamma * exp_xom_xsgplus * L * xom7 + 8 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 + 56 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 - 4 * exp_xom_xsgplus * L2 * xom7 - 28 * xsg * exp_xom_xsgplus * L2 * xom7 + 4 * exp_xom_xsgplus * log2_xom * xom7 + 28 * xsg * exp_xom_xsgplus * log2_xom * xom7 + 8 * eulergamma * exp_xom_xsgplus * log_xom * xom7 + 56 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom7 - 8 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 - 56 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 + 8 * exp_xom_xsgplus * hyp_plus_xom * xom8 + 56 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom8 - 7560 * exp_xom_xsgplus * xsg2 - 10584 * xom * exp_xom_xsgplus * xsg2 + 8190 * eulergamma * exp_xom_xsgplus * xsg2 + 2268 * xom * eulergamma * exp_xom_xsgplus * xsg2 - 8190 * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 - 2268 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 9450 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg2 - 3780 * xom * hyp_xom_xsg * xsg2 + 9450 * eulergamma * L * xsg2 - 9450 * expint_ei_xom_xsg * L * xsg2 + 8190 * exp_xom_xsgplus * L * xsg2 + 2268 * xom * exp_xom_xsgplus * L * xsg2 - 9450 * eulergamma * exp_xom_xsgplus * L * xsg2 - 5670 * xom * eulergamma * exp_xom_xsgplus * L * xsg2 + 9450 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 4725 * L2 * xsg2 - 4725 * exp_xom_xsgplus * L2 * xsg2 - 2835 * xom * exp_xom_xsgplus * L2 * xsg2 - 4725 * log2_xom * xsg2 + 4725 * exp_xom_xsgplus * log2_xom * xsg2 + 2835 * xom * exp_xom_xsgplus * log2_xom * xsg2 - 9450 * eulergamma * log_xom * xsg2 + 9450 * expint_ei_xom_xsg * log_xom * xsg2 + 9450 * eulergamma * exp_xom_xsgplus * log_xom * xsg2 + 5670 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 9450 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 - 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 9450 * L * log_xsg * xsg2 - 9450 * log_xom * log_xsg * xsg2 + 30240 * exp_xom_xsgplus * xom2 * xsg2 + 27132 * eulergamma * exp_xom_xsgplus * xom2 * xsg2 - 27132 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg2 + 5670 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg2 + 27132 * exp_xom_xsgplus * L * xom2 * xsg2 + 7560 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 - 7560 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 + 3780 * exp_xom_xsgplus * L2 * xom2 * xsg2 - 3780 * exp_xom_xsgplus * log2_xom * xom2 * xsg2 - 7560 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 7560 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 50400 * exp_xom_xsgplus * xom3 * xsg2 - 61320 * eulergamma * exp_xom_xsgplus * xom3 * xsg2 + 61320 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg2 - 7560 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2 - 61320 * exp_xom_xsgplus * L * xom3 * xsg2 - 31500 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg2 + 31500 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg2 - 15750 * exp_xom_xsgplus * L2 * xom3 * xsg2 + 15750 * exp_xom_xsgplus * log2_xom * xom3 * xsg2 + 31500 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg2 - 31500 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg2 + 22680 * exp_xom_xsgplus * xom4 * xsg2 + 44940 * eulergamma * exp_xom_xsgplus * xom4 * xsg2 - 44940 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg2 + 31500 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg2 + 44940 * exp_xom_xsgplus * L * xom4 * xsg2 + 34020 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg2 - 34020 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg2 + 17010 * exp_xom_xsgplus * L2 * xom4 * xsg2 - 17010 * exp_xom_xsgplus * log2_xom * xom4 * xsg2 - 34020 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg2 + 34020 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg2 - 3528 * exp_xom_xsgplus * xom5 * xsg2 - 12600 * eulergamma * exp_xom_xsgplus * xom5 * xsg2 + 12600 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg2 - 34020 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg2 - 12600 * exp_xom_xsgplus * L * xom5 * xsg2 - 14700 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg2 + 14700 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg2 - 7350 * exp_xom_xsgplus * L2 * xom5 * xsg2 + 7350 * exp_xom_xsgplus * log2_xom * xom5 * xsg2 + 14700 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg2 - 14700 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg2 + 1176 * eulergamma * exp_xom_xsgplus * xom6 * xsg2 - 1176 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg2 + 14700 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg2 + 1176 * exp_xom_xsgplus * L * xom6 * xsg2 + 2688 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg2 - 2688 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg2 + 1344 * exp_xom_xsgplus * L2 * xom6 * xsg2 - 1344 * exp_xom_xsgplus * log2_xom * xom6 * xsg2 - 2688 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg2 + 2688 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg2 - 2688 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg2 - 168 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg2 + 168 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg2 - 84 * exp_xom_xsgplus * L2 * xom7 * xsg2 + 84 * exp_xom_xsgplus * log2_xom * xom7 * xsg2 + 168 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg2 - 168 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg2 + 168 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg2 - 14560 * exp_xom_xsgplus * xsg3 - 22120 * xom * exp_xom_xsgplus * xsg3 + 3150 * eulergamma * exp_xom_xsgplus * xsg3 + 6300 * xom * eulergamma * exp_xom_xsgplus * xsg3 - 3150 * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 6300 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 12600 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg3 + 9450 * xom * hyp_xom_xsg * xsg3 - 12600 * eulergamma * L * xsg3 + 12600 * expint_ei_xom_xsg * L * xsg3 + 3150 * exp_xom_xsgplus * L * xsg3 + 6300 * xom * exp_xom_xsgplus * L * xsg3 + 12600 * eulergamma * exp_xom_xsgplus * L * xsg3 + 3150 * xom * eulergamma * exp_xom_xsgplus * L * xsg3 - 12600 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 3150 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 6300 * L2 * xsg3 + 6300 * exp_xom_xsgplus * L2 * xsg3 + 1575 * xom * exp_xom_xsgplus * L2 * xsg3 + 6300 * log2_xom * xsg3 - 6300 * exp_xom_xsgplus * log2_xom * xsg3 - 1575 * xom * exp_xom_xsgplus * log2_xom * xsg3 + 12600 * eulergamma * log_xom * xsg3 - 12600 * expint_ei_xom_xsg * log_xom * xsg3 - 12600 * eulergamma * exp_xom_xsgplus * log_xom * xsg3 - 3150 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 12600 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 + 3150 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 12600 * L * log_xsg * xsg3 + 12600 * log_xom * log_xsg * xsg3 + 60480 * exp_xom_xsgplus * xom2 * xsg3 + 58380 * eulergamma * exp_xom_xsgplus * xom2 * xsg3 - 58380 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg3 - 3150 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg3 + 58380 * exp_xom_xsgplus * L * xom2 * xsg3 + 16380 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg3 - 16380 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg3 + 8190 * exp_xom_xsgplus * L2 * xom2 * xsg3 - 8190 * exp_xom_xsgplus * log2_xom * xom2 * xsg3 - 16380 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg3 + 16380 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg3 - 95200 * exp_xom_xsgplus * xom3 * xsg3 - 117320 * eulergamma * exp_xom_xsgplus * xom3 * xsg3 + 117320 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg3 - 16380 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg3 - 117320 * exp_xom_xsgplus * L * xom3 * xsg3 - 62580 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg3 + 62580 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg3 - 31290 * exp_xom_xsgplus * L2 * xom3 * xsg3 + 31290 * exp_xom_xsgplus * log2_xom * xom3 * xsg3 + 62580 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg3 - 62580 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg3 + 42000 * exp_xom_xsgplus * xom4 * xsg3 + 84700 * eulergamma * exp_xom_xsgplus * xom4 * xsg3 - 84700 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg3 + 62580 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg3 + 84700 * exp_xom_xsgplus * L * xom4 * xsg3 + 64680 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg3 - 64680 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg3 + 32340 * exp_xom_xsgplus * L2 * xom4 * xsg3 - 32340 * exp_xom_xsgplus * log2_xom * xom4 * xsg3 - 64680 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg3 + 64680 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg3 - 5880 * exp_xom_xsgplus * xom5 * xsg3 - 22680 * eulergamma * exp_xom_xsgplus * xom5 * xsg3 + 22680 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg3 - 64680 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg3 - 22680 * exp_xom_xsgplus * L * xom5 * xsg3 - 27300 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg3 + 27300 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg3 - 13650 * exp_xom_xsgplus * L2 * xom5 * xsg3 + 13650 * exp_xom_xsgplus * log2_xom * xom5 * xsg3 + 27300 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg3 - 27300 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg3 + 1960 * eulergamma * exp_xom_xsgplus * xom6 * xsg3 - 1960 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg3 + 27300 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg3 + 1960 * exp_xom_xsgplus * L * xom6 * xsg3 + 4760 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg3 - 4760 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg3 + 2380 * exp_xom_xsgplus * L2 * xom6 * xsg3 - 2380 * exp_xom_xsgplus * log2_xom * xom6 * xsg3 - 4760 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg3 + 4760 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg3 - 4760 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg3 - 280 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg3 + 280 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg3 - 140 * exp_xom_xsgplus * L2 * xom7 * xsg3 + 140 * exp_xom_xsgplus * log2_xom * xom7 * xsg3 + 280 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg3 - 280 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg3 + 280 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg3 - 7560 * exp_xom_xsgplus * xsg4 - 22120 * xom * exp_xom_xsgplus * xsg4 - 3150 * eulergamma * exp_xom_xsgplus * xsg4 - 6300 * xom * eulergamma * exp_xom_xsgplus * xsg4 + 3150 * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 + 6300 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 + 9450 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg4 - 12600 * xom * hyp_xom_xsg * xsg4 + 9450 * eulergamma * L * xsg4 - 9450 * expint_ei_xom_xsg * L * xsg4 - 3150 * exp_xom_xsgplus * L * xsg4 - 6300 * xom * exp_xom_xsgplus * L * xsg4 - 9450 * eulergamma * exp_xom_xsgplus * L * xsg4 + 3150 * xom * eulergamma * exp_xom_xsgplus * L * xsg4 + 9450 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 - 3150 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 + 4725 * L2 * xsg4 - 4725 * exp_xom_xsgplus * L2 * xsg4 + 1575 * xom * exp_xom_xsgplus * L2 * xsg4 - 4725 * log2_xom * xsg4 + 4725 * exp_xom_xsgplus * log2_xom * xsg4 - 1575 * xom * exp_xom_xsgplus * log2_xom * xsg4 - 9450 * eulergamma * log_xom * xsg4 + 9450 * expint_ei_xom_xsg * log_xom * xsg4 + 9450 * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 3150 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 9450 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 3150 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 9450 * L * log_xsg * xsg4 - 9450 * log_xom * log_xsg * xsg4 + 73920 * exp_xom_xsgplus * xom2 * xsg4 + 67620 * eulergamma * exp_xom_xsgplus * xom2 * xsg4 - 67620 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg4 - 3150 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg4 + 67620 * exp_xom_xsgplus * L * xom2 * xsg4 + 25200 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg4 - 25200 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg4 + 12600 * exp_xom_xsgplus * L2 * xom2 * xsg4 - 12600 * exp_xom_xsgplus * log2_xom * xom2 * xsg4 - 25200 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg4 + 25200 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg4 - 112000 * exp_xom_xsgplus * xom3 * xsg4 - 140280 * eulergamma * exp_xom_xsgplus * xom3 * xsg4 + 140280 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg4 - 25200 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg4 - 140280 * exp_xom_xsgplus * L * xom3 * xsg4 - 74340 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg4 + 74340 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg4 - 37170 * exp_xom_xsgplus * L2 * xom3 * xsg4 + 37170 * exp_xom_xsgplus * log2_xom * xom3 * xsg4 + 74340 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg4 - 74340 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg4 + 46200 * exp_xom_xsgplus * xom4 * xsg4 + 97300 * eulergamma * exp_xom_xsgplus * xom4 * xsg4 - 97300 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg4 + 74340 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg4 + 97300 * exp_xom_xsgplus * L * xom4 * xsg4 + 76020 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg4 - 76020 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg4 + 38010 * exp_xom_xsgplus * L2 * xom4 * xsg4 - 38010 * exp_xom_xsgplus * log2_xom * xom4 * xsg4 - 76020 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg4 + 76020 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg4 - 5880 * exp_xom_xsgplus * xom5 * xsg4 - 24360 * eulergamma * exp_xom_xsgplus * xom5 * xsg4 + 24360 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg4 - 76020 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg4 - 24360 * exp_xom_xsgplus * L * xom5 * xsg4 - 30660 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg4 + 30660 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg4 - 15330 * exp_xom_xsgplus * L2 * xom5 * xsg4 + 15330 * exp_xom_xsgplus * log2_xom * xom5 * xsg4 + 30660 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg4 - 30660 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg4 + 1960 * eulergamma * exp_xom_xsgplus * xom6 * xsg4 - 1960 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg4 + 30660 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg4 + 1960 * exp_xom_xsgplus * L * xom6 * xsg4 + 5040 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg4 - 5040 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg4 + 2520 * exp_xom_xsgplus * L2 * xom6 * xsg4 - 2520 * exp_xom_xsgplus * log2_xom * xom6 * xsg4 - 5040 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg4 + 5040 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg4 - 5040 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg4 - 280 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg4 + 280 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg4 - 140 * exp_xom_xsgplus * L2 * xom7 * xsg4 + 140 * exp_xom_xsgplus * log2_xom * xom7 * xsg4 + 280 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg4 - 280 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg4 + 280 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg4 - 3024 * exp_xom_xsgplus * xsg5 - 10584 * xom * exp_xom_xsgplus * xsg5 + 5922 * eulergamma * exp_xom_xsgplus * xsg5 - 2268 * xom * eulergamma * exp_xom_xsgplus * xsg5 - 5922 * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 + 2268 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 - 3780 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg5 + 9450 * xom * hyp_xom_xsg * xsg5 - 3780 * eulergamma * L * xsg5 + 3780 * expint_ei_xom_xsg * L * xsg5 + 5922 * exp_xom_xsgplus * L * xsg5 - 2268 * xom * exp_xom_xsgplus * L * xsg5 + 3780 * eulergamma * exp_xom_xsgplus * L * xsg5 - 5670 * xom * eulergamma * exp_xom_xsgplus * L * xsg5 - 3780 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 + 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 - 1890 * L2 * xsg5 + 1890 * exp_xom_xsgplus * L2 * xsg5 - 2835 * xom * exp_xom_xsgplus * L2 * xsg5 + 1890 * log2_xom * xsg5 - 1890 * exp_xom_xsgplus * log2_xom * xsg5 + 2835 * xom * exp_xom_xsgplus * log2_xom * xsg5 + 3780 * eulergamma * log_xom * xsg5 - 3780 * expint_ei_xom_xsg * log_xom * xsg5 - 3780 * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 5670 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 3780 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 3780 * L * log_xsg * xsg5 + 3780 * log_xom * log_xsg * xsg5 + 60480 * exp_xom_xsgplus * xom2 * xsg5 + 49812 * eulergamma * exp_xom_xsgplus * xom2 * xsg5 - 49812 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg5 + 5670 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg5 + 49812 * exp_xom_xsgplus * L * xom2 * xsg5 + 16380 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg5 - 16380 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg5 + 8190 * exp_xom_xsgplus * L2 * xom2 * xsg5 - 8190 * exp_xom_xsgplus * log2_xom * xom2 * xsg5 - 16380 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg5 + 16380 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg5 - 80640 * exp_xom_xsgplus * xom3 * xsg5 - 106680 * eulergamma * exp_xom_xsgplus * xom3 * xsg5 + 106680 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg5 - 16380 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg5 - 106680 * exp_xom_xsgplus * L * xom3 * xsg5 - 56700 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg5 + 56700 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg5 - 28350 * exp_xom_xsgplus * L2 * xom3 * xsg5 + 28350 * exp_xom_xsgplus * log2_xom * xom3 * xsg5 + 56700 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg5 - 56700 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg5 + 30240 * exp_xom_xsgplus * xom4 * xsg5 + 67620 * eulergamma * exp_xom_xsgplus * xom4 * xsg5 - 67620 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg5 + 56700 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg5 + 67620 * exp_xom_xsgplus * L * xom4 * xsg5 + 55440 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg5 - 55440 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg5 + 27720 * exp_xom_xsgplus * L2 * xom4 * xsg5 - 27720 * exp_xom_xsgplus * log2_xom * xom4 * xsg5 - 55440 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg5 + 55440 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg5 - 3528 * exp_xom_xsgplus * xom5 * xsg5 - 15624 * eulergamma * exp_xom_xsgplus * xom5 * xsg5 + 15624 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg5 - 55440 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg5 - 15624 * exp_xom_xsgplus * L * xom5 * xsg5 - 20748 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg5 + 20748 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg5 - 10374 * exp_xom_xsgplus * L2 * xom5 * xsg5 + 10374 * exp_xom_xsgplus * log2_xom * xom5 * xsg5 + 20748 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg5 - 20748 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg5 + 1176 * eulergamma * exp_xom_xsgplus * xom6 * xsg5 - 1176 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg5 + 20748 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg5 + 1176 * exp_xom_xsgplus * L * xom6 * xsg5 + 3192 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg5 - 3192 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg5 + 1596 * exp_xom_xsgplus * L2 * xom6 * xsg5 - 1596 * exp_xom_xsgplus * log2_xom * xom6 * xsg5 - 3192 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg5 + 3192 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg5 - 3192 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg5 - 168 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg5 + 168 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg5 - 84 * exp_xom_xsgplus * L2 * xom7 * xsg5 + 84 * exp_xom_xsgplus * log2_xom * xom7 * xsg5 + 168 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg5 - 168 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg5 + 168 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg5 - 2744 * exp_xom_xsgplus * xsg6 - 5768 * xom * exp_xom_xsgplus * xsg6 - 1302 * eulergamma * exp_xom_xsgplus * xsg6 + 2100 * xom * eulergamma * exp_xom_xsgplus * xsg6 + 1302 * expint_ei_plus_xom * exp_xom_xsgplus * xsg6 - 2100 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg6 + 630 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg6 - 3780 * xom * hyp_xom_xsg * xsg6 + 630 * eulergamma * L * xsg6 - 630 * expint_ei_xom_xsg * L * xsg6 - 1302 * exp_xom_xsgplus * L * xsg6 + 2100 * xom * exp_xom_xsgplus * L * xsg6 - 630 * eulergamma * exp_xom_xsgplus * L * xsg6 + 3150 * xom * eulergamma * exp_xom_xsgplus * L * xsg6 + 630 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg6 - 3150 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg6 + 315 * L2 * xsg6 - 315 * exp_xom_xsgplus * L2 * xsg6 + 1575 * xom * exp_xom_xsgplus * L2 * xsg6 - 315 * log2_xom * xsg6 + 315 * exp_xom_xsgplus * log2_xom * xsg6 - 1575 * xom * exp_xom_xsgplus * log2_xom * xsg6 - 630 * eulergamma * log_xom * xsg6 + 630 * expint_ei_xom_xsg * log_xom * xsg6 + 630 * eulergamma * exp_xom_xsgplus * log_xom * xsg6 - 3150 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg6 - 630 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg6 + 3150 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg6 + 630 * L * log_xsg * xsg6 - 630 * log_xom * log_xsg * xsg6 + 30240 * exp_xom_xsgplus * xom2 * xsg6 + 26964 * eulergamma * exp_xom_xsgplus * xom2 * xsg6 - 26964 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg6 - 3150 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg6 + 26964 * exp_xom_xsgplus * L * xom2 * xsg6 + 7560 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg6 - 7560 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg6 + 3780 * exp_xom_xsgplus * L2 * xom2 * xsg6 - 3780 * exp_xom_xsgplus * log2_xom * xom2 * xsg6 - 7560 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg6 + 7560 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg6 - 32480 * exp_xom_xsgplus * xom3 * xsg6 - 47320 * eulergamma * exp_xom_xsgplus * xom3 * xsg6 + 47320 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg6 - 7560 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg6 - 47320 * exp_xom_xsgplus * L * xom3 * xsg6 - 27300 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg6 + 27300 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg6 - 13650 * exp_xom_xsgplus * L2 * xom3 * xsg6 + 13650 * exp_xom_xsgplus * log2_xom * xom3 * xsg6 + 27300 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg6 - 27300 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg6 + 10920 * exp_xom_xsgplus * xom4 * xsg6 + 26180 * eulergamma * exp_xom_xsgplus * xom4 * xsg6 - 26180 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg6 + 27300 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg6 + 26180 * exp_xom_xsgplus * L * xom4 * xsg6 + 23100 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg6 - 23100 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg6 + 11550 * exp_xom_xsgplus * L2 * xom4 * xsg6 - 11550 * exp_xom_xsgplus * log2_xom * xom4 * xsg6 - 23100 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg6 + 23100 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg6 - 1176 * exp_xom_xsgplus * xom5 * xsg6 - 5544 * eulergamma * exp_xom_xsgplus * xom5 * xsg6 + 5544 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg6 - 23100 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg6 - 5544 * exp_xom_xsgplus * L * xom5 * xsg6 - 7812 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg6 + 7812 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg6 - 3906 * exp_xom_xsgplus * L2 * xom5 * xsg6 + 3906 * exp_xom_xsgplus * log2_xom * xom5 * xsg6 + 7812 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg6 - 7812 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg6 + 392 * eulergamma * exp_xom_xsgplus * xom6 * xsg6 - 392 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg6 + 7812 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg6 + 392 * exp_xom_xsgplus * L * xom6 * xsg6 + 1120 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg6 - 1120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg6 + 560 * exp_xom_xsgplus * L2 * xom6 * xsg6 - 560 * exp_xom_xsgplus * log2_xom * xom6 * xsg6 - 1120 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg6 + 1120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg6 - 1120 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg6 - 56 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg6 + 56 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg6 - 28 * exp_xom_xsgplus * L2 * xom7 * xsg6 + 28 * exp_xom_xsgplus * log2_xom * xom7 * xsg6 + 56 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg6 - 56 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg6 + 56 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg6 - 2744 * xom * exp_xom_xsgplus * xsg7 + 90 * eulergamma * exp_xom_xsgplus * xsg7 - 1932 * xom * eulergamma * exp_xom_xsgplus * xsg7 - 90 * expint_ei_plus_xom * exp_xom_xsgplus * xsg7 + 1932 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg7 + 630 * xom * hyp_xom_xsg * xsg7 + 90 * exp_xom_xsgplus * L * xsg7 - 1932 * xom * exp_xom_xsgplus * L * xsg7 - 630 * xom * eulergamma * exp_xom_xsgplus * L * xsg7 + 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg7 - 315 * xom * exp_xom_xsgplus * L2 * xsg7 + 315 * xom * exp_xom_xsgplus * log2_xom * xsg7 + 630 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg7 - 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg7 + 6720 * exp_xom_xsgplus * xom2 * xsg7 + 7812 * eulergamma * exp_xom_xsgplus * xom2 * xsg7 - 7812 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg7 + 630 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg7 + 7812 * exp_xom_xsgplus * L * xom2 * xsg7 + 3780 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg7 - 3780 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg7 + 1890 * exp_xom_xsgplus * L2 * xom2 * xsg7 - 1890 * exp_xom_xsgplus * log2_xom * xom2 * xsg7 - 3780 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg7 + 3780 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg7 - 5600 * exp_xom_xsgplus * xom3 * xsg7 - 9240 * eulergamma * exp_xom_xsgplus * xom3 * xsg7 + 9240 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg7 - 3780 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg7 - 9240 * exp_xom_xsgplus * L * xom3 * xsg7 - 6300 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg7 + 6300 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg7 - 3150 * exp_xom_xsgplus * L2 * xom3 * xsg7 + 3150 * exp_xom_xsgplus * log2_xom * xom3 * xsg7 + 6300 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg7 - 6300 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg7 + 1680 * exp_xom_xsgplus * xom4 * xsg7 + 4340 * eulergamma * exp_xom_xsgplus * xom4 * xsg7 - 4340 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg7 + 6300 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg7 + 4340 * exp_xom_xsgplus * L * xom4 * xsg7 + 4200 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg7 - 4200 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg7 + 2100 * exp_xom_xsgplus * L2 * xom4 * xsg7 - 2100 * exp_xom_xsgplus * log2_xom * xom4 * xsg7 - 4200 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg7 + 4200 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg7 - 168 * exp_xom_xsgplus * xom5 * xsg7 - 840 * eulergamma * exp_xom_xsgplus * xom5 * xsg7 + 840 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg7 - 4200 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg7 - 840 * exp_xom_xsgplus * L * xom5 * xsg7 - 1260 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg7 + 1260 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg7 - 630 * exp_xom_xsgplus * L2 * xom5 * xsg7 + 630 * exp_xom_xsgplus * log2_xom * xom5 * xsg7 + 1260 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg7 - 1260 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg7 + 56 * eulergamma * exp_xom_xsgplus * xom6 * xsg7 - 56 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg7 + 1260 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg7 + 56 * exp_xom_xsgplus * L * xom6 * xsg7 + 168 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg7 - 168 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg7 + 84 * exp_xom_xsgplus * L2 * xom6 * xsg7 - 84 * exp_xom_xsgplus * log2_xom * xom6 * xsg7 - 168 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg7 + 168 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg7 - 168 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg7 - 8 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg7 + 8 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg7 - 4 * exp_xom_xsgplus * L2 * xom7 * xsg7 + 4 * exp_xom_xsgplus * log2_xom * xom7 * xsg7 + 8 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg7 - 8 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg7 + 8 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg7) * xsgplus_m8) / 315.,
            -0.0031746031746031746 * (exp_xom_xsg * (3008 + 1218 * xom + 2392 * xsg - 292 * xom * xsg - 2112 * L - 630 * xom * L + 4704 * xsg * L + 1668 * xom * xsg * L + 2112 * log_xom + 630 * xom * log_xom - 4704 * xsg * log_xom - 1668 * xom * xsg * log_xom + 1402 * xom2 + 9300 * xsg * xom2 + 426 * L * xom2 + 5724 * xsg * L * xom2 - 426 * log_xom * xom2 - 5724 * xsg * log_xom * xom2 - 1412 * xom3 - 13136 * xsg * xom3 - 1222 * L * xom3 - 10436 * xsg * L * xom3 + 1222 * log_xom * xom3 + 10436 * xsg * log_xom * xom3 + 830 * xom4 + 7080 * xsg * xom4 + 852 * L * xom4 + 7520 * xsg * L * xom4 - 852 * log_xom * xom4 - 7520 * xsg * log_xom * xom4 - 174 * xom5 - 1488 * xsg * xom5 - 272 * L * xom5 - 2328 * xsg * L * xom5 + 272 * log_xom * xom5 + 2328 * xsg * log_xom * xom5 + 14 * xom6 + 112 * xsg * xom6 + 38 * L * xom6 + 320 * xsg * L * xom6 - 38 * log_xom * xom6 - 320 * xsg * log_xom * xom6 - 2 * L * xom7 - 16 * xsg * L * xom7 + 2 * log_xom * xom7 + 16 * xsg * log_xom * xom7 + 11928 * xsg2 - 1820 * xom * xsg2 - 14112 * L * xsg2 - 4116 * xom * L * xsg2 + 14112 * log_xom * xsg2 + 4116 * xom * log_xom * xsg2 + 35644 * xom2 * xsg2 + 20580 * L * xom2 * xsg2 - 20580 * log_xom * xom2 * xsg2 - 51744 * xom3 * xsg2 - 41244 * L * xom3 * xsg2 + 41244 * log_xom * xom3 * xsg2 + 26880 * xom4 * xsg2 + 29008 * L * xom4 * xsg2 - 29008 * log_xom * xom4 * xsg2 - 5544 * xom5 * xsg2 - 8792 * L * xom5 * xsg2 + 8792 * log_xom * xom5 * xsg2 + 392 * xom6 * xsg2 + 1176 * L * xom6 * xsg2 - 1176 * log_xom * xom6 * xsg2 - 56 * L * xom7 * xsg2 + 56 * log_xom * xom7 * xsg2 + 21784 * xsg3 - 1428 * xom * xsg3 - 5292 * xom * L * xsg3 + 5292 * xom * log_xom * xsg3 + 83972 * xom2 * xsg3 + 46956 * L * xom2 * xsg3 - 46956 * log_xom * xom2 * xsg3 - 116368 * xom3 * xsg3 - 94388 * L * xom3 * xsg3 + 94388 * log_xom * xom3 * xsg3 + 59080 * xom4 * xsg3 + 64512 * L * xom4 * xsg3 - 64512 * log_xom * xom4 * xsg3 - 11760 * xom5 * xsg3 - 19096 * L * xom5 * xsg3 + 19096 * log_xom * xom5 * xsg3 + 784 * xom6 * xsg3 + 2464 * L * xom6 * xsg3 - 2464 * log_xom * xom6 * xsg3 - 112 * L * xom7 * xsg3 + 112 * log_xom * xom7 * xsg3 + 11480 * xsg4 - 11480 * xom * xsg4 + 122640 * xom2 * xsg4 + 73080 * L * xom2 * xsg4 - 73080 * log_xom * xom2 * xsg4 - 166600 * xom3 * xsg4 - 136360 * L * xom3 * xsg4 + 136360 * log_xom * xom3 * xsg4 + 81900 * xom4 * xsg4 + 91000 * L * xom4 * xsg4 - 91000 * log_xom * xom4 * xsg4 - 15540 * xom5 * xsg4 - 26040 * L * xom5 * xsg4 + 26040 * log_xom * xom5 * xsg4 + 980 * xom6 * xsg4 + 3220 * L * xom6 * xsg4 - 3220 * log_xom * xom6 * xsg4 - 140 * L * xom7 * xsg4 + 140 * log_xom * xom7 * xsg4 + 10248 * xsg5 - 16492 * xom * xsg5 - 14112 * L * xsg5 - 8820 * xom * L * xsg5 + 14112 * log_xom * xsg5 + 8820 * xom * log_xom * xsg5 + 116732 * xom2 * xsg5 + 71316 * L * xom2 * xsg5 - 71316 * log_xom * xom2 * xsg5 - 156912 * xom3 * xsg5 - 129612 * L * xom3 * xsg5 + 129612 * log_xom * xom3 * xsg5 + 73080 * xom4 * xsg5 + 83552 * L * xom4 * xsg5 - 83552 * log_xom * xom4 * xsg5 - 13104 * xom5 * xsg5 - 22792 * L * xom5 * xsg5 + 22792 * log_xom * xom5 * xsg5 + 784 * xom6 * xsg5 + 2688 * L * xom6 * xsg5 - 2688 * log_xom * xom6 * xsg5 - 112 * L * xom7 * xsg5 + 112 * log_xom * xom7 * xsg5 + 7112 * xsg6 - 7812 * xom * xsg6 + 4704 * L * xsg6 - 5292 * xom * L * xsg6 - 4704 * log_xom * xsg6 + 5292 * xom * log_xom * xsg6 + 75460 * xom2 * xsg6 + 44604 * L * xom2 * xsg6 - 44604 * log_xom * xom2 * xsg6 - 94976 * xom3 * xsg6 - 80836 * L * xom3 * xsg6 + 80836 * log_xom * xom3 * xsg6 + 40880 * xom4 * xsg6 + 48720 * L * xom4 * xsg6 - 48720 * log_xom * xom4 * xsg6 - 6888 * xom5 * xsg6 - 12488 * L * xom5 * xsg6 + 12488 * log_xom * xom5 * xsg6 + 392 * xom6 * xsg6 + 1400 * L * xom6 * xsg6 - 1400 * log_xom * xom6 * xsg6 - 56 * L * xom7 * xsg6 + 56 * log_xom * xom7 * xsg6 - 632 * xsg7 - 4060 * xom * xsg7 - 2112 * L * xsg7 + 924 * xom * L * xsg7 + 2112 * log_xom * xsg7 - 924 * xom * log_xom * xsg7 + 31404 * xom2 * xsg7 + 19620 * L * xom2 * xsg7 - 19620 * log_xom * xom2 * xsg7 - 33584 * xom3 * xsg7 - 30524 * L * xom3 * xsg7 + 30524 * log_xom * xom3 * xsg7 + 13080 * xom4 * xsg7 + 16448 * L * xom4 * xsg7 - 16448 * log_xom * xom4 * xsg7 - 2064 * xom5 * xsg7 - 3912 * L * xom5 * xsg7 + 3912 * log_xom * xom5 * xsg7 + 112 * xom6 * xsg7 + 416 * L * xom6 * xsg7 - 416 * log_xom * xom6 * xsg7 - 16 * L * xom7 * xsg7 + 16 * log_xom * xom7 * xsg7 + 264 * xsg8 - 2378 * xom * xsg8 - 1482 * xom * L * xsg8 + 1482 * xom * log_xom * xsg8 + 6262 * xom2 * xsg8 + 4926 * L * xom2 * xsg8 - 4926 * log_xom * xom2 * xsg8 - 5268 * xom3 * xsg8 - 5298 * L * xom3 * xsg8 + 5298 * log_xom * xom3 * xsg8 + 1830 * xom4 * xsg8 + 2452 * L * xom4 * xsg8 - 2452 * log_xom * xom4 * xsg8 - 270 * xom5 * xsg8 - 536 * L * xom5 * xsg8 + 536 * log_xom * xom5 * xsg8 + 14 * xom6 * xsg8 + 54 * L * xom6 * xsg8 - 54 * log_xom * xom6 * xsg8 - 2 * L * xom7 * xsg8 + 2 * log_xom * xom7 * xsg8) * xsgplus_m9) - ((-3008 * exp_xom_xsgplus - 2744 * xom * exp_xom_xsgplus - 2392 * xsg * exp_xom_xsgplus - 3024 * xom * xsg * exp_xom_xsgplus + 2112 * eulergamma * exp_xom_xsgplus + 2112 * xom * eulergamma * exp_xom_xsgplus - 4704 * xsg * eulergamma * exp_xom_xsgplus - 2592 * xom * xsg * eulergamma * exp_xom_xsgplus - 2112 * expint_ei_plus_xom * exp_xom_xsgplus - 2112 * xom * expint_ei_plus_xom * exp_xom_xsgplus + 4704 * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 2592 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 630 * xom * exp_xom_xsgplus * hyp_plus_xom - 4410 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 630 * xom * xsg * hyp_xom_xsg + 630 * eulergamma * L - 4410 * xsg * eulergamma * L - 630 * expint_ei_xom_xsg * L + 4410 * xsg * expint_ei_xom_xsg * L + 2112 * exp_xom_xsgplus * L + 2112 * xom * exp_xom_xsgplus * L - 4704 * xsg * exp_xom_xsgplus * L - 2592 * xom * xsg * exp_xom_xsgplus * L - 630 * eulergamma * exp_xom_xsgplus * L - 630 * xom * eulergamma * exp_xom_xsgplus * L + 4410 * xsg * eulergamma * exp_xom_xsgplus * L + 3780 * xom * xsg * eulergamma * exp_xom_xsgplus * L + 630 * expint_ei_plus_xom * exp_xom_xsgplus * L + 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 4410 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L - 3780 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + 315 * L2 - 2205 * xsg * L2 - 315 * exp_xom_xsgplus * L2 - 315 * xom * exp_xom_xsgplus * L2 + 2205 * xsg * exp_xom_xsgplus * L2 + 1890 * xom * xsg * exp_xom_xsgplus * L2 - 315 * log2_xom + 2205 * xsg * log2_xom + 315 * exp_xom_xsgplus * log2_xom + 315 * xom * exp_xom_xsgplus * log2_xom - 2205 * xsg * exp_xom_xsgplus * log2_xom - 1890 * xom * xsg * exp_xom_xsgplus * log2_xom - 630 * eulergamma * log_xom + 4410 * xsg * eulergamma * log_xom + 630 * expint_ei_xom_xsg * log_xom - 4410 * xsg * expint_ei_xom_xsg * log_xom + 630 * eulergamma * exp_xom_xsgplus * log_xom + 630 * xom * eulergamma * exp_xom_xsgplus * log_xom - 4410 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 3780 * xom * xsg * eulergamma * exp_xom_xsgplus * log_xom - 630 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 4410 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 3780 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 630 * L * log_xsg - 4410 * xsg * L * log_xsg - 630 * log_xom * log_xsg + 4410 * xsg * log_xom * log_xsg - 2744 * exp_xom_xsgplus * xom2 - 15232 * xsg * exp_xom_xsgplus * xom2 - 9744 * xsg * eulergamma * exp_xom_xsgplus * xom2 + 9744 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom2 + 630 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 3780 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2 - 9744 * xsg * exp_xom_xsgplus * L * xom2 - 630 * eulergamma * exp_xom_xsgplus * L * xom2 - 1260 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 + 630 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 1260 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 - 315 * exp_xom_xsgplus * L2 * xom2 - 630 * xsg * exp_xom_xsgplus * L2 * xom2 + 315 * exp_xom_xsgplus * log2_xom * xom2 + 630 * xsg * exp_xom_xsgplus * log2_xom * xom2 + 630 * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 1260 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 - 630 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 1260 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 + 2240 * exp_xom_xsgplus * xom3 + 22400 * xsg * exp_xom_xsgplus * xom3 + 3248 * eulergamma * exp_xom_xsgplus * xom3 + 25984 * xsg * eulergamma * exp_xom_xsgplus * xom3 - 3248 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 - 25984 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom3 + 630 * exp_xom_xsgplus * hyp_plus_xom * xom3 + 1260 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 + 3248 * exp_xom_xsgplus * L * xom3 + 25984 * xsg * exp_xom_xsgplus * L * xom3 + 1260 * eulergamma * exp_xom_xsgplus * L * xom3 + 12600 * xsg * eulergamma * exp_xom_xsgplus * L * xom3 - 1260 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 - 12600 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 + 630 * exp_xom_xsgplus * L2 * xom3 + 6300 * xsg * exp_xom_xsgplus * L2 * xom3 - 630 * exp_xom_xsgplus * log2_xom * xom3 - 6300 * xsg * exp_xom_xsgplus * log2_xom * xom3 - 1260 * eulergamma * exp_xom_xsgplus * log_xom * xom3 - 12600 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom3 + 1260 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 + 12600 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 - 1960 * exp_xom_xsgplus * xom4 - 16520 * xsg * exp_xom_xsgplus * xom4 - 2800 * eulergamma * exp_xom_xsgplus * xom4 - 25200 * xsg * eulergamma * exp_xom_xsgplus * xom4 + 2800 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 + 25200 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom4 - 1260 * exp_xom_xsgplus * hyp_plus_xom * xom4 - 12600 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom4 - 2800 * exp_xom_xsgplus * L * xom4 - 25200 * xsg * exp_xom_xsgplus * L * xom4 - 1890 * eulergamma * exp_xom_xsgplus * L * xom4 - 16380 * xsg * eulergamma * exp_xom_xsgplus * L * xom4 + 1890 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 + 16380 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 - 945 * exp_xom_xsgplus * L2 * xom4 - 8190 * xsg * exp_xom_xsgplus * L2 * xom4 + 945 * exp_xom_xsgplus * log2_xom * xom4 + 8190 * xsg * exp_xom_xsgplus * log2_xom * xom4 + 1890 * eulergamma * exp_xom_xsgplus * log_xom * xom4 + 16380 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom4 - 1890 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 - 16380 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 + 504 * exp_xom_xsgplus * xom5 + 4368 * xsg * exp_xom_xsgplus * xom5 + 1232 * eulergamma * exp_xom_xsgplus * xom5 + 10528 * xsg * eulergamma * exp_xom_xsgplus * xom5 - 1232 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 - 10528 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom5 + 1890 * exp_xom_xsgplus * hyp_plus_xom * xom5 + 16380 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom5 + 1232 * exp_xom_xsgplus * L * xom5 + 10528 * xsg * exp_xom_xsgplus * L * xom5 + 1092 * eulergamma * exp_xom_xsgplus * L * xom5 + 9576 * xsg * eulergamma * exp_xom_xsgplus * L * xom5 - 1092 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 - 9576 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 + 546 * exp_xom_xsgplus * L2 * xom5 + 4788 * xsg * exp_xom_xsgplus * L2 * xom5 - 546 * exp_xom_xsgplus * log2_xom * xom5 - 4788 * xsg * exp_xom_xsgplus * log2_xom * xom5 - 1092 * eulergamma * exp_xom_xsgplus * log_xom * xom5 - 9576 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom5 + 1092 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 + 9576 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 - 56 * exp_xom_xsgplus * xom6 - 448 * xsg * exp_xom_xsgplus * xom6 - 224 * eulergamma * exp_xom_xsgplus * xom6 - 1904 * xsg * eulergamma * exp_xom_xsgplus * xom6 + 224 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 + 1904 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom6 - 1092 * exp_xom_xsgplus * hyp_plus_xom * xom6 - 9576 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom6 - 224 * exp_xom_xsgplus * L * xom6 - 1904 * xsg * exp_xom_xsgplus * L * xom6 - 308 * eulergamma * exp_xom_xsgplus * L * xom6 - 2632 * xsg * eulergamma * exp_xom_xsgplus * L * xom6 + 308 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 + 2632 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 - 154 * exp_xom_xsgplus * L2 * xom6 - 1316 * xsg * exp_xom_xsgplus * L2 * xom6 + 154 * exp_xom_xsgplus * log2_xom * xom6 + 1316 * xsg * exp_xom_xsgplus * log2_xom * xom6 + 308 * eulergamma * exp_xom_xsgplus * log_xom * xom6 + 2632 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom6 - 308 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 - 2632 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 + 16 * eulergamma * exp_xom_xsgplus * xom7 + 128 * xsg * eulergamma * exp_xom_xsgplus * xom7 - 16 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 - 128 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom7 + 308 * exp_xom_xsgplus * hyp_plus_xom * xom7 + 2632 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom7 + 16 * exp_xom_xsgplus * L * xom7 + 128 * xsg * exp_xom_xsgplus * L * xom7 + 40 * eulergamma * exp_xom_xsgplus * L * xom7 + 336 * xsg * eulergamma * exp_xom_xsgplus * L * xom7 - 40 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 - 336 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 + 20 * exp_xom_xsgplus * L2 * xom7 + 168 * xsg * exp_xom_xsgplus * L2 * xom7 - 20 * exp_xom_xsgplus * log2_xom * xom7 - 168 * xsg * exp_xom_xsgplus * log2_xom * xom7 - 40 * eulergamma * exp_xom_xsgplus * log_xom * xom7 - 336 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom7 + 40 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 + 336 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 - 40 * exp_xom_xsgplus * hyp_plus_xom * xom8 - 336 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom8 - 2 * eulergamma * exp_xom_xsgplus * L * xom8 - 16 * xsg * eulergamma * exp_xom_xsgplus * L * xom8 + 2 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 + 16 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 - exp_xom_xsgplus * L2 * xom8 - 8 * xsg * exp_xom_xsgplus * L2 * xom8 + exp_xom_xsgplus * log2_xom * xom8 + 8 * xsg * exp_xom_xsgplus * log2_xom * xom8 + 2 * eulergamma * exp_xom_xsgplus * log_xom * xom8 + 16 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom8 - 2 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 - 16 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 + 2 * exp_xom_xsgplus * hyp_plus_xom * xom9 + 16 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom9 - 11928 * exp_xom_xsgplus * xsg2 - 4816 * xom * exp_xom_xsgplus * xsg2 + 14112 * eulergamma * exp_xom_xsgplus * xsg2 + 9408 * xom * eulergamma * exp_xom_xsgplus * xsg2 - 14112 * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 - 9408 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 13230 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg2 - 4410 * xom * hyp_xom_xsg * xsg2 + 13230 * eulergamma * L * xsg2 - 13230 * expint_ei_xom_xsg * L * xsg2 + 14112 * exp_xom_xsgplus * L * xsg2 + 9408 * xom * exp_xom_xsgplus * L * xsg2 - 13230 * eulergamma * exp_xom_xsgplus * L * xsg2 - 8820 * xom * eulergamma * exp_xom_xsgplus * L * xsg2 + 13230 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 6615 * L2 * xsg2 - 6615 * exp_xom_xsgplus * L2 * xsg2 - 4410 * xom * exp_xom_xsgplus * L2 * xsg2 - 6615 * log2_xom * xsg2 + 6615 * exp_xom_xsgplus * log2_xom * xsg2 + 4410 * xom * exp_xom_xsgplus * log2_xom * xsg2 - 13230 * eulergamma * log_xom * xsg2 + 13230 * expint_ei_xom_xsg * log_xom * xsg2 + 13230 * eulergamma * exp_xom_xsgplus * log_xom * xsg2 + 8820 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 13230 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 - 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 13230 * L * log_xsg * xsg2 - 13230 * log_xom * log_xsg * xsg2 - 53312 * exp_xom_xsgplus * xom2 * xsg2 - 34608 * eulergamma * exp_xom_xsgplus * xom2 * xsg2 + 34608 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg2 + 8820 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg2 - 34608 * exp_xom_xsgplus * L * xom2 * xsg2 - 13860 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 + 13860 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 - 6930 * exp_xom_xsgplus * L2 * xom2 * xsg2 + 6930 * exp_xom_xsgplus * log2_xom * xom2 * xsg2 + 13860 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 13860 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 90720 * exp_xom_xsgplus * xom3 * xsg2 + 102144 * eulergamma * exp_xom_xsgplus * xom3 * xsg2 - 102144 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg2 + 13860 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2 + 102144 * exp_xom_xsgplus * L * xom3 * xsg2 + 47880 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg2 - 47880 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg2 + 23940 * exp_xom_xsgplus * L2 * xom3 * xsg2 - 23940 * exp_xom_xsgplus * log2_xom * xom3 * xsg2 - 47880 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg2 + 47880 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg2 - 62440 * exp_xom_xsgplus * xom4 * xsg2 - 98000 * eulergamma * exp_xom_xsgplus * xom4 * xsg2 + 98000 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg2 - 47880 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg2 - 98000 * exp_xom_xsgplus * L * xom4 * xsg2 - 64260 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg2 + 64260 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg2 - 32130 * exp_xom_xsgplus * L2 * xom4 * xsg2 + 32130 * exp_xom_xsgplus * log2_xom * xom4 * xsg2 + 64260 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg2 - 64260 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg2 + 16464 * exp_xom_xsgplus * xom5 * xsg2 + 39872 * eulergamma * exp_xom_xsgplus * xom5 * xsg2 - 39872 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg2 + 64260 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg2 + 39872 * exp_xom_xsgplus * L * xom5 * xsg2 + 36792 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg2 - 36792 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg2 + 18396 * exp_xom_xsgplus * L2 * xom5 * xsg2 - 18396 * exp_xom_xsgplus * log2_xom * xom5 * xsg2 - 36792 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg2 + 36792 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg2 - 1568 * exp_xom_xsgplus * xom6 * xsg2 - 7056 * eulergamma * exp_xom_xsgplus * xom6 * xsg2 + 7056 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg2 - 36792 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg2 - 7056 * exp_xom_xsgplus * L * xom6 * xsg2 - 9912 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg2 + 9912 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg2 - 4956 * exp_xom_xsgplus * L2 * xom6 * xsg2 + 4956 * exp_xom_xsgplus * log2_xom * xom6 * xsg2 + 9912 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg2 - 9912 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg2 + 448 * eulergamma * exp_xom_xsgplus * xom7 * xsg2 - 448 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg2 + 9912 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg2 + 448 * exp_xom_xsgplus * L * xom7 * xsg2 + 1232 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg2 - 1232 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg2 + 616 * exp_xom_xsgplus * L2 * xom7 * xsg2 - 616 * exp_xom_xsgplus * log2_xom * xom7 * xsg2 - 1232 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg2 + 1232 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg2 - 1232 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg2 - 56 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg2 + 56 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg2 - 28 * exp_xom_xsgplus * L2 * xom8 * xsg2 + 28 * exp_xom_xsgplus * log2_xom * xom8 * xsg2 + 56 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg2 - 56 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg2 + 56 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg2 - 21784 * exp_xom_xsgplus * xsg3 - 11536 * xom * exp_xom_xsgplus * xsg3 + 14112 * xom * eulergamma * exp_xom_xsgplus * xsg3 - 14112 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 22050 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg3 + 13230 * xom * hyp_xom_xsg * xsg3 - 22050 * eulergamma * L * xsg3 + 22050 * expint_ei_xom_xsg * L * xsg3 + 14112 * xom * exp_xom_xsgplus * L * xsg3 + 22050 * eulergamma * exp_xom_xsgplus * L * xsg3 + 8820 * xom * eulergamma * exp_xom_xsgplus * L * xsg3 - 22050 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 11025 * L2 * xsg3 + 11025 * exp_xom_xsgplus * L2 * xsg3 + 4410 * xom * exp_xom_xsgplus * L2 * xsg3 + 11025 * log2_xom * xsg3 - 11025 * exp_xom_xsgplus * log2_xom * xsg3 - 4410 * xom * exp_xom_xsgplus * log2_xom * xsg3 + 22050 * eulergamma * log_xom * xsg3 - 22050 * expint_ei_xom_xsg * log_xom * xsg3 - 22050 * eulergamma * exp_xom_xsgplus * log_xom * xsg3 - 8820 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 22050 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 + 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 22050 * L * log_xsg * xsg3 + 22050 * log_xom * log_xsg * xsg3 - 123424 * exp_xom_xsgplus * xom2 * xsg3 - 76944 * eulergamma * exp_xom_xsgplus * xom2 * xsg3 + 76944 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg3 - 8820 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg3 - 76944 * exp_xom_xsgplus * L * xom2 * xsg3 - 26460 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg3 + 26460 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg3 - 13230 * exp_xom_xsgplus * L2 * xom2 * xsg3 + 13230 * exp_xom_xsgplus * log2_xom * xom2 * xsg3 + 26460 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg3 - 26460 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg3 + 206080 * exp_xom_xsgplus * xom3 * xsg3 + 235648 * eulergamma * exp_xom_xsgplus * xom3 * xsg3 - 235648 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg3 + 26460 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg3 + 235648 * exp_xom_xsgplus * L * xom3 * xsg3 + 110040 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg3 - 110040 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg3 + 55020 * exp_xom_xsgplus * L2 * xom3 * xsg3 - 55020 * exp_xom_xsgplus * log2_xom * xom3 * xsg3 - 110040 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg3 + 110040 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg3 - 137480 * exp_xom_xsgplus * xom4 * xsg3 - 218960 * eulergamma * exp_xom_xsgplus * xom4 * xsg3 + 218960 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg3 - 110040 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg3 - 218960 * exp_xom_xsgplus * L * xom4 * xsg3 - 145740 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg3 + 145740 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg3 - 72870 * exp_xom_xsgplus * L2 * xom4 * xsg3 + 72870 * exp_xom_xsgplus * log2_xom * xom4 * xsg3 + 145740 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg3 - 145740 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg3 + 35280 * exp_xom_xsgplus * xom5 * xsg3 + 87136 * eulergamma * exp_xom_xsgplus * xom5 * xsg3 - 87136 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg3 + 145740 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg3 + 87136 * exp_xom_xsgplus * L * xom5 * xsg3 + 81480 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg3 - 81480 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg3 + 40740 * exp_xom_xsgplus * L2 * xom5 * xsg3 - 40740 * exp_xom_xsgplus * log2_xom * xom5 * xsg3 - 81480 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg3 + 81480 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg3 - 3136 * exp_xom_xsgplus * xom6 * xsg3 - 14896 * eulergamma * exp_xom_xsgplus * xom6 * xsg3 + 14896 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg3 - 81480 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg3 - 14896 * exp_xom_xsgplus * L * xom6 * xsg3 - 21448 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg3 + 21448 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg3 - 10724 * exp_xom_xsgplus * L2 * xom6 * xsg3 + 10724 * exp_xom_xsgplus * log2_xom * xom6 * xsg3 + 21448 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg3 - 21448 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg3 + 896 * eulergamma * exp_xom_xsgplus * xom7 * xsg3 - 896 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg3 + 21448 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg3 + 896 * exp_xom_xsgplus * L * xom7 * xsg3 + 2576 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg3 - 2576 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg3 + 1288 * exp_xom_xsgplus * L2 * xom7 * xsg3 - 1288 * exp_xom_xsgplus * log2_xom * xom7 * xsg3 - 2576 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg3 + 2576 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg3 - 2576 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg3 - 112 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg3 + 112 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg3 - 56 * exp_xom_xsgplus * L2 * xom8 * xsg3 + 56 * exp_xom_xsgplus * log2_xom * xom8 * xsg3 + 112 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg3 - 112 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg3 + 112 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg3 - 11480 * exp_xom_xsgplus * xsg4 + 22050 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg4 - 22050 * xom * hyp_xom_xsg * xsg4 + 22050 * eulergamma * L * xsg4 - 22050 * expint_ei_xom_xsg * L * xsg4 - 22050 * eulergamma * exp_xom_xsgplus * L * xsg4 + 22050 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 + 11025 * L2 * xsg4 - 11025 * exp_xom_xsgplus * L2 * xsg4 - 11025 * log2_xom * xsg4 + 11025 * exp_xom_xsgplus * log2_xom * xsg4 - 22050 * eulergamma * log_xom * xsg4 + 22050 * expint_ei_xom_xsg * log_xom * xsg4 + 22050 * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 22050 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 22050 * L * log_xsg * xsg4 - 22050 * log_xom * log_xsg * xsg4 - 178640 * exp_xom_xsgplus * xom2 * xsg4 - 126000 * eulergamma * exp_xom_xsgplus * xom2 * xsg4 + 126000 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg4 - 126000 * exp_xom_xsgplus * L * xom2 * xsg4 - 35280 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg4 + 35280 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg4 - 17640 * exp_xom_xsgplus * L2 * xom2 * xsg4 + 17640 * exp_xom_xsgplus * log2_xom * xom2 * xsg4 + 35280 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg4 - 35280 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg4 + 296800 * exp_xom_xsgplus * xom3 * xsg4 + 341600 * eulergamma * exp_xom_xsgplus * xom3 * xsg4 - 341600 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg4 + 35280 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg4 + 341600 * exp_xom_xsgplus * L * xom3 * xsg4 + 164640 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg4 - 164640 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg4 + 82320 * exp_xom_xsgplus * L2 * xom3 * xsg4 - 82320 * exp_xom_xsgplus * log2_xom * xom3 * xsg4 - 164640 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg4 + 164640 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg4 - 191800 * exp_xom_xsgplus * xom4 * xsg4 - 310800 * eulergamma * exp_xom_xsgplus * xom4 * xsg4 + 310800 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg4 - 164640 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg4 - 310800 * exp_xom_xsgplus * L * xom4 * xsg4 - 209160 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg4 + 209160 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg4 - 104580 * exp_xom_xsgplus * L2 * xom4 * xsg4 + 104580 * exp_xom_xsgplus * log2_xom * xom4 * xsg4 + 209160 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg4 - 209160 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg4 + 47040 * exp_xom_xsgplus * xom5 * xsg4 + 119840 * eulergamma * exp_xom_xsgplus * xom5 * xsg4 - 119840 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg4 + 209160 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg4 + 119840 * exp_xom_xsgplus * L * xom5 * xsg4 + 114240 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg4 - 114240 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg4 + 57120 * exp_xom_xsgplus * L2 * xom5 * xsg4 - 57120 * exp_xom_xsgplus * log2_xom * xom5 * xsg4 - 114240 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg4 + 114240 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg4 - 3920 * exp_xom_xsgplus * xom6 * xsg4 - 19600 * eulergamma * exp_xom_xsgplus * xom6 * xsg4 + 19600 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg4 - 114240 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg4 - 19600 * exp_xom_xsgplus * L * xom6 * xsg4 - 29120 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg4 + 29120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg4 - 14560 * exp_xom_xsgplus * L2 * xom6 * xsg4 + 14560 * exp_xom_xsgplus * log2_xom * xom6 * xsg4 + 29120 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg4 - 29120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg4 + 1120 * eulergamma * exp_xom_xsgplus * xom7 * xsg4 - 1120 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg4 + 29120 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg4 + 1120 * exp_xom_xsgplus * L * xom7 * xsg4 + 3360 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg4 - 3360 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg4 + 1680 * exp_xom_xsgplus * L2 * xom7 * xsg4 - 1680 * exp_xom_xsgplus * log2_xom * xom7 * xsg4 - 3360 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg4 + 3360 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg4 - 3360 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg4 - 140 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg4 + 140 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg4 - 70 * exp_xom_xsgplus * L2 * xom8 * xsg4 + 70 * exp_xom_xsgplus * log2_xom * xom8 * xsg4 + 140 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg4 - 140 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg4 + 140 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg4 - 10248 * exp_xom_xsgplus * xsg5 + 11536 * xom * exp_xom_xsgplus * xsg5 + 14112 * eulergamma * exp_xom_xsgplus * xsg5 + 14112 * xom * eulergamma * exp_xom_xsgplus * xsg5 - 14112 * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 - 14112 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 - 13230 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg5 + 22050 * xom * hyp_xom_xsg * xsg5 - 13230 * eulergamma * L * xsg5 + 13230 * expint_ei_xom_xsg * L * xsg5 + 14112 * exp_xom_xsgplus * L * xsg5 + 14112 * xom * exp_xom_xsgplus * L * xsg5 + 13230 * eulergamma * exp_xom_xsgplus * L * xsg5 - 8820 * xom * eulergamma * exp_xom_xsgplus * L * xsg5 - 13230 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 + 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 - 6615 * L2 * xsg5 + 6615 * exp_xom_xsgplus * L2 * xsg5 - 4410 * xom * exp_xom_xsgplus * L2 * xsg5 + 6615 * log2_xom * xsg5 - 6615 * exp_xom_xsgplus * log2_xom * xsg5 + 4410 * xom * exp_xom_xsgplus * log2_xom * xsg5 + 13230 * eulergamma * log_xom * xsg5 - 13230 * expint_ei_xom_xsg * log_xom * xsg5 - 13230 * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 8820 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 13230 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 13230 * L * log_xsg * xsg5 + 13230 * log_xom * log_xsg * xsg5 - 167104 * exp_xom_xsgplus * xom2 * xsg5 - 126000 * eulergamma * exp_xom_xsgplus * xom2 * xsg5 + 126000 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg5 + 8820 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg5 - 126000 * exp_xom_xsgplus * L * xom2 * xsg5 - 44100 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg5 + 44100 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg5 - 22050 * exp_xom_xsgplus * L2 * xom2 * xsg5 + 22050 * exp_xom_xsgplus * log2_xom * xom2 * xsg5 + 44100 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg5 - 44100 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg5 + 282240 * exp_xom_xsgplus * xom3 * xsg5 + 325248 * eulergamma * exp_xom_xsgplus * xom3 * xsg5 - 325248 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg5 + 44100 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg5 + 325248 * exp_xom_xsgplus * L * xom3 * xsg5 + 158760 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg5 - 158760 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg5 + 79380 * exp_xom_xsgplus * L2 * xom3 * xsg5 - 79380 * exp_xom_xsgplus * log2_xom * xom3 * xsg5 - 158760 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg5 + 158760 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg5 - 172760 * exp_xom_xsgplus * xom4 * xsg5 - 288400 * eulergamma * exp_xom_xsgplus * xom4 * xsg5 + 288400 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg5 - 158760 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg5 - 288400 * exp_xom_xsgplus * L * xom4 * xsg5 - 196980 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg5 + 196980 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg5 - 98490 * exp_xom_xsgplus * L2 * xom4 * xsg5 + 98490 * exp_xom_xsgplus * log2_xom * xom4 * xsg5 + 196980 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg5 - 196980 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg5 + 39984 * exp_xom_xsgplus * xom5 * xsg5 + 105952 * eulergamma * exp_xom_xsgplus * xom5 * xsg5 - 105952 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg5 + 196980 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg5 + 105952 * exp_xom_xsgplus * L * xom5 * xsg5 + 103992 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg5 - 103992 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg5 + 51996 * exp_xom_xsgplus * L2 * xom5 * xsg5 - 51996 * exp_xom_xsgplus * log2_xom * xom5 * xsg5 - 103992 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg5 + 103992 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg5 - 3136 * exp_xom_xsgplus * xom6 * xsg5 - 16464 * eulergamma * exp_xom_xsgplus * xom6 * xsg5 + 16464 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg5 - 103992 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg5 - 16464 * exp_xom_xsgplus * L * xom6 * xsg5 - 25368 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg5 + 25368 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg5 - 12684 * exp_xom_xsgplus * L2 * xom6 * xsg5 + 12684 * exp_xom_xsgplus * log2_xom * xom6 * xsg5 + 25368 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg5 - 25368 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg5 + 896 * eulergamma * exp_xom_xsgplus * xom7 * xsg5 - 896 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg5 + 25368 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg5 + 896 * exp_xom_xsgplus * L * xom7 * xsg5 + 2800 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg5 - 2800 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg5 + 1400 * exp_xom_xsgplus * L2 * xom7 * xsg5 - 1400 * exp_xom_xsgplus * log2_xom * xom7 * xsg5 - 2800 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg5 + 2800 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg5 - 2800 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg5 - 112 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg5 + 112 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg5 - 56 * exp_xom_xsgplus * L2 * xom8 * xsg5 + 56 * exp_xom_xsgplus * log2_xom * xom8 * xsg5 + 112 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg5 - 112 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg5 + 112 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg5 - 7112 * exp_xom_xsgplus * xsg6 + 4816 * xom * exp_xom_xsgplus * xsg6 - 4704 * eulergamma * exp_xom_xsgplus * xsg6 + 9408 * xom * eulergamma * exp_xom_xsgplus * xsg6 + 4704 * expint_ei_plus_xom * exp_xom_xsgplus * xsg6 - 9408 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg6 + 4410 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg6 - 13230 * xom * hyp_xom_xsg * xsg6 + 4410 * eulergamma * L * xsg6 - 4410 * expint_ei_xom_xsg * L * xsg6 - 4704 * exp_xom_xsgplus * L * xsg6 + 9408 * xom * exp_xom_xsgplus * L * xsg6 - 4410 * eulergamma * exp_xom_xsgplus * L * xsg6 + 8820 * xom * eulergamma * exp_xom_xsgplus * L * xsg6 + 4410 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg6 - 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg6 + 2205 * L2 * xsg6 - 2205 * exp_xom_xsgplus * L2 * xsg6 + 4410 * xom * exp_xom_xsgplus * L2 * xsg6 - 2205 * log2_xom * xsg6 + 2205 * exp_xom_xsgplus * log2_xom * xsg6 - 4410 * xom * exp_xom_xsgplus * log2_xom * xsg6 - 4410 * eulergamma * log_xom * xsg6 + 4410 * expint_ei_xom_xsg * log_xom * xsg6 + 4410 * eulergamma * exp_xom_xsgplus * log_xom * xsg6 - 8820 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg6 - 4410 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg6 + 8820 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg6 + 4410 * L * log_xsg * xsg6 - 4410 * log_xom * log_xsg * xsg6 - 107072 * exp_xom_xsgplus * xom2 * xsg6 - 76944 * eulergamma * exp_xom_xsgplus * xom2 * xsg6 + 76944 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg6 - 8820 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg6 - 76944 * exp_xom_xsgplus * L * xom2 * xsg6 - 26460 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg6 + 26460 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg6 - 13230 * exp_xom_xsgplus * L2 * xom2 * xsg6 + 13230 * exp_xom_xsgplus * log2_xom * xom2 * xsg6 + 26460 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg6 - 26460 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg6 + 173600 * exp_xom_xsgplus * xom3 * xsg6 + 205184 * eulergamma * exp_xom_xsgplus * xom3 * xsg6 - 205184 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg6 + 26460 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg6 + 205184 * exp_xom_xsgplus * L * xom3 * xsg6 + 99960 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg6 - 99960 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg6 + 49980 * exp_xom_xsgplus * L2 * xom3 * xsg6 - 49980 * exp_xom_xsgplus * log2_xom * xom3 * xsg6 - 99960 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg6 + 99960 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg6 - 97720 * exp_xom_xsgplus * xom4 * xsg6 - 170800 * eulergamma * exp_xom_xsgplus * xom4 * xsg6 + 170800 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg6 - 99960 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg6 - 170800 * exp_xom_xsgplus * L * xom4 * xsg6 - 120540 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg6 + 120540 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg6 - 60270 * exp_xom_xsgplus * L2 * xom4 * xsg6 + 60270 * exp_xom_xsgplus * log2_xom * xom4 * xsg6 + 120540 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg6 - 120540 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg6 + 21168 * exp_xom_xsgplus * xom5 * xsg6 + 58688 * eulergamma * exp_xom_xsgplus * xom5 * xsg6 - 58688 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg6 + 120540 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg6 + 58688 * exp_xom_xsgplus * L * xom5 * xsg6 + 59976 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg6 - 59976 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg6 + 29988 * exp_xom_xsgplus * L2 * xom5 * xsg6 - 29988 * exp_xom_xsgplus * log2_xom * xom5 * xsg6 - 59976 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg6 + 59976 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg6 - 1568 * exp_xom_xsgplus * xom6 * xsg6 - 8624 * eulergamma * exp_xom_xsgplus * xom6 * xsg6 + 8624 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg6 - 59976 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg6 - 8624 * exp_xom_xsgplus * L * xom6 * xsg6 - 13832 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg6 + 13832 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg6 - 6916 * exp_xom_xsgplus * L2 * xom6 * xsg6 + 6916 * exp_xom_xsgplus * log2_xom * xom6 * xsg6 + 13832 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg6 - 13832 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg6 + 448 * eulergamma * exp_xom_xsgplus * xom7 * xsg6 - 448 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg6 + 13832 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg6 + 448 * exp_xom_xsgplus * L * xom7 * xsg6 + 1456 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg6 - 1456 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg6 + 728 * exp_xom_xsgplus * L2 * xom7 * xsg6 - 728 * exp_xom_xsgplus * log2_xom * xom7 * xsg6 - 1456 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg6 + 1456 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg6 - 1456 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg6 - 56 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg6 + 56 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg6 - 28 * exp_xom_xsgplus * L2 * xom8 * xsg6 + 28 * exp_xom_xsgplus * log2_xom * xom8 * xsg6 + 56 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg6 - 56 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg6 + 56 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg6 + 632 * exp_xom_xsgplus * xsg7 + 3024 * xom * exp_xom_xsgplus * xsg7 + 2112 * eulergamma * exp_xom_xsgplus * xsg7 - 2592 * xom * eulergamma * exp_xom_xsgplus * xsg7 - 2112 * expint_ei_plus_xom * exp_xom_xsgplus * xsg7 + 2592 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg7 - 630 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg7 + 4410 * xom * hyp_xom_xsg * xsg7 - 630 * eulergamma * L * xsg7 + 630 * expint_ei_xom_xsg * L * xsg7 + 2112 * exp_xom_xsgplus * L * xsg7 - 2592 * xom * exp_xom_xsgplus * L * xsg7 + 630 * eulergamma * exp_xom_xsgplus * L * xsg7 - 3780 * xom * eulergamma * exp_xom_xsgplus * L * xsg7 - 630 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg7 + 3780 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg7 - 315 * L2 * xsg7 + 315 * exp_xom_xsgplus * L2 * xsg7 - 1890 * xom * exp_xom_xsgplus * L2 * xsg7 + 315 * log2_xom * xsg7 - 315 * exp_xom_xsgplus * log2_xom * xsg7 + 1890 * xom * exp_xom_xsgplus * log2_xom * xsg7 + 630 * eulergamma * log_xom * xsg7 - 630 * expint_ei_xom_xsg * log_xom * xsg7 - 630 * eulergamma * exp_xom_xsgplus * log_xom * xsg7 + 3780 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg7 + 630 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg7 - 3780 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg7 - 630 * L * log_xsg * xsg7 + 630 * log_xom * log_xsg * xsg7 - 45472 * exp_xom_xsgplus * xom2 * xsg7 - 34608 * eulergamma * exp_xom_xsgplus * xom2 * xsg7 + 34608 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg7 + 3780 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg7 - 34608 * exp_xom_xsgplus * L * xom2 * xsg7 - 8820 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg7 + 8820 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg7 - 4410 * exp_xom_xsgplus * L2 * xom2 * xsg7 + 4410 * exp_xom_xsgplus * log2_xom * xom2 * xsg7 + 8820 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg7 - 8820 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg7 + 62720 * exp_xom_xsgplus * xom3 * xsg7 + 79744 * eulergamma * exp_xom_xsgplus * xom3 * xsg7 - 79744 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg7 + 8820 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg7 + 79744 * exp_xom_xsgplus * L * xom3 * xsg7 + 41160 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg7 - 41160 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg7 + 20580 * exp_xom_xsgplus * L2 * xom3 * xsg7 - 20580 * exp_xom_xsgplus * log2_xom * xom3 * xsg7 - 41160 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg7 + 41160 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg7 - 31640 * exp_xom_xsgplus * xom4 * xsg7 - 58800 * eulergamma * exp_xom_xsgplus * xom4 * xsg7 + 58800 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg7 - 41160 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg7 - 58800 * exp_xom_xsgplus * L * xom4 * xsg7 - 44100 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg7 + 44100 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg7 - 22050 * exp_xom_xsgplus * L2 * xom4 * xsg7 + 22050 * exp_xom_xsgplus * log2_xom * xom4 * xsg7 + 44100 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg7 - 44100 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg7 + 6384 * exp_xom_xsgplus * xom5 * xsg7 + 18592 * eulergamma * exp_xom_xsgplus * xom5 * xsg7 - 18592 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg7 + 44100 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg7 + 18592 * exp_xom_xsgplus * L * xom5 * xsg7 + 19992 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg7 - 19992 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg7 + 9996 * exp_xom_xsgplus * L2 * xom5 * xsg7 - 9996 * exp_xom_xsgplus * log2_xom * xom5 * xsg7 - 19992 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg7 + 19992 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg7 - 448 * exp_xom_xsgplus * xom6 * xsg7 - 2576 * eulergamma * exp_xom_xsgplus * xom6 * xsg7 + 2576 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg7 - 19992 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg7 - 2576 * exp_xom_xsgplus * L * xom6 * xsg7 - 4312 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg7 + 4312 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg7 - 2156 * exp_xom_xsgplus * L2 * xom6 * xsg7 + 2156 * exp_xom_xsgplus * log2_xom * xom6 * xsg7 + 4312 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg7 - 4312 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg7 + 128 * eulergamma * exp_xom_xsgplus * xom7 * xsg7 - 128 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg7 + 4312 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg7 + 128 * exp_xom_xsgplus * L * xom7 * xsg7 + 432 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg7 - 432 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg7 + 216 * exp_xom_xsgplus * L2 * xom7 * xsg7 - 216 * exp_xom_xsgplus * log2_xom * xom7 * xsg7 - 432 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg7 + 432 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg7 - 432 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg7 - 16 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg7 + 16 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg7 - 8 * exp_xom_xsgplus * L2 * xom8 * xsg7 + 8 * exp_xom_xsgplus * log2_xom * xom8 * xsg7 + 16 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg7 - 16 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg7 + 16 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg7 - 264 * exp_xom_xsgplus * xsg8 + 2744 * xom * exp_xom_xsgplus * xsg8 + 2112 * xom * eulergamma * exp_xom_xsgplus * xsg8 - 2112 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg8 - 630 * xom * hyp_xom_xsg * xsg8 + 2112 * xom * exp_xom_xsgplus * L * xsg8 + 630 * xom * eulergamma * exp_xom_xsgplus * L * xsg8 - 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg8 + 315 * xom * exp_xom_xsgplus * L2 * xsg8 - 315 * xom * exp_xom_xsgplus * log2_xom * xsg8 - 630 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg8 + 630 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg8 - 9464 * exp_xom_xsgplus * xom2 * xsg8 - 9744 * eulergamma * exp_xom_xsgplus * xom2 * xsg8 + 9744 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg8 - 630 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg8 - 9744 * exp_xom_xsgplus * L * xom2 * xsg8 - 4410 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg8 + 4410 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg8 - 2205 * exp_xom_xsgplus * L2 * xom2 * xsg8 + 2205 * exp_xom_xsgplus * log2_xom * xom2 * xsg8 + 4410 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg8 - 4410 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg8 + 10080 * exp_xom_xsgplus * xom3 * xsg8 + 14448 * eulergamma * exp_xom_xsgplus * xom3 * xsg8 - 14448 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg8 + 4410 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg8 + 14448 * exp_xom_xsgplus * L * xom3 * xsg8 + 8820 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg8 - 8820 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg8 + 4410 * exp_xom_xsgplus * L2 * xom3 * xsg8 - 4410 * exp_xom_xsgplus * log2_xom * xom3 * xsg8 - 8820 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg8 + 8820 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg8 - 4480 * exp_xom_xsgplus * xom4 * xsg8 - 8960 * eulergamma * exp_xom_xsgplus * xom4 * xsg8 + 8960 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg8 - 8820 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg8 - 8960 * exp_xom_xsgplus * L * xom4 * xsg8 - 7350 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg8 + 7350 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg8 - 3675 * exp_xom_xsgplus * L2 * xom4 * xsg8 + 3675 * exp_xom_xsgplus * log2_xom * xom4 * xsg8 + 7350 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg8 - 7350 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg8 + 840 * exp_xom_xsgplus * xom5 * xsg8 + 2576 * eulergamma * exp_xom_xsgplus * xom5 * xsg8 - 2576 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg8 + 7350 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg8 + 2576 * exp_xom_xsgplus * L * xom5 * xsg8 + 2940 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg8 - 2940 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg8 + 1470 * exp_xom_xsgplus * L2 * xom5 * xsg8 - 1470 * exp_xom_xsgplus * log2_xom * xom5 * xsg8 - 2940 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg8 + 2940 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg8 - 56 * exp_xom_xsgplus * xom6 * xsg8 - 336 * eulergamma * exp_xom_xsgplus * xom6 * xsg8 + 336 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg8 - 2940 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg8 - 336 * exp_xom_xsgplus * L * xom6 * xsg8 - 588 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg8 + 588 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg8 - 294 * exp_xom_xsgplus * L2 * xom6 * xsg8 + 294 * exp_xom_xsgplus * log2_xom * xom6 * xsg8 + 588 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg8 - 588 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg8 + 16 * eulergamma * exp_xom_xsgplus * xom7 * xsg8 - 16 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg8 + 588 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg8 + 16 * exp_xom_xsgplus * L * xom7 * xsg8 + 56 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg8 - 56 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg8 + 28 * exp_xom_xsgplus * L2 * xom7 * xsg8 - 28 * exp_xom_xsgplus * log2_xom * xom7 * xsg8 - 56 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg8 + 56 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg8 - 56 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg8 - 2 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg8 + 2 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg8 - exp_xom_xsgplus * L2 * xom8 * xsg8 + exp_xom_xsgplus * log2_xom * xom8 * xsg8 + 2 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg8 - 2 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg8 + 2 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg8) * xsgplus_m9) / 315.,
            (exp_xom_xsg * (29448 + 13968 * xom + 15840 * xsg + 13464 * xom * xsg - 19638 * L - 5040 * xom * L + 55674 * xsg * L + 26352 * xom * xsg * L + 19638 * log_xom + 5040 * xom * log_xom - 55674 * xsg * log_xom - 26352 * xom * xsg * log_xom - 6192 * xom2 - 98208 * xsg * xom2 - 6984 * L * xom2 - 49680 * xsg * L * xom2 + 6984 * log_xom * xom2 + 49680 * xsg * log_xom * xom2 + 19760 * xom3 + 186696 * xsg * xom3 + 13344 * L * xom3 + 136152 * xsg * L * xom3 - 13344 * log_xom * xom3 - 136152 * xsg * log_xom * xom3 - 13100 * xom4 - 128880 * xsg * xom4 - 12336 * L * xom4 - 120024 * xsg * L * xom4 + 12336 * log_xom * xom4 + 120024 * xsg * log_xom * xom4 + 4192 * xom5 + 39996 * xsg * xom5 + 5088 * L * xom5 + 49320 * xsg * L * xom5 - 5088 * log_xom * xom5 - 49320 * xsg * log_xom * xom5 - 588 * xom6 - 5544 * xsg * xom6 - 1052 * L * xom6 - 10008 * xsg * L * xom6 + 1052 * log_xom * xom6 + 10008 * xsg * log_xom * xom6 + 32 * xom7 + 288 * xsg * xom7 + 104 * L * xom7 + 972 * xsg * L * xom7 - 104 * log_xom * xom7 - 972 * xsg * log_xom * xom7 - 4 * L * xom8 - 36 * xsg * L * xom8 + 4 * log_xom * xom8 + 36 * xsg * log_xom * xom8 + 171360 * xsg2 + 94464 * xom * xsg2 - 192024 * L * xsg2 - 29376 * xom * L * xsg2 + 192024 * log_xom * xsg2 + 29376 * xom * log_xom * xsg2 - 440064 * xom2 * xsg2 - 255744 * L * xom2 * xsg2 + 255744 * log_xom * xom2 * xsg2 + 821376 * xom3 * xsg2 + 599040 * L * xom3 * xsg2 - 599040 * log_xom * xom3 * xsg2 - 562320 * xom4 * xsg2 - 526752 * L * xom4 * xsg2 + 526752 * log_xom * xom4 * xsg2 + 170784 * xom5 * xsg2 + 213120 * L * xom5 * xsg2 - 213120 * log_xom * xom5 * xsg2 - 23184 * xom6 * xsg2 - 42480 * L * xom6 * xsg2 + 42480 * log_xom * xom6 * xsg2 + 1152 * xom7 * xsg2 + 4032 * L * xom7 * xsg2 - 4032 * log_xom * xom7 * xsg2 - 144 * L * xom8 * xsg2 + 144 * log_xom * xom8 * xsg2 + 288288 * xsg3 + 256032 * xom * xsg3 + 74088 * L * xsg3 + 42336 * xom * L * xsg3 - 74088 * log_xom * xsg3 - 42336 * xom * log_xom * xsg3 - 1145088 * xom2 * xsg3 - 671328 * L * xom2 * xsg3 + 671328 * log_xom * xom2 * xsg3 + 2145696 * xom3 * xsg3 + 1566432 * L * xom3 * xsg3 - 1566432 * log_xom * xom3 * xsg3 - 1438080 * xom4 * xsg3 - 1360800 * L * xom4 * xsg3 + 1360800 * log_xom * xom4 * xsg3 + 427728 * xom5 * xsg3 + 540288 * L * xom5 * xsg3 - 540288 * log_xom * xom5 * xsg3 - 56448 * xom6 * xsg3 - 105504 * L * xom6 * xsg3 + 105504 * log_xom * xom6 * xsg3 + 2688 * xom7 * xsg3 + 9744 * L * xom7 * xsg3 - 9744 * log_xom * xom7 * xsg3 - 336 * L * xom8 * xsg3 + 336 * log_xom * xom8 * xsg3 + 206640 * xsg4 + 288288 * xom * xsg4 - 79380 * L * xsg4 + 127008 * xom * L * xsg4 + 79380 * log_xom * xsg4 - 127008 * xom * log_xom * xsg4 - 2022048 * xom2 * xsg4 - 1127952 * L * xom2 * xsg4 + 1127952 * log_xom * xom2 * xsg4 + 3638880 * xom3 * xsg4 + 2689344 * L * xom3 * xsg4 - 2689344 * log_xom * xom3 * xsg4 - 2386440 * xom4 * xsg4 - 2282112 * L * xom4 * xsg4 + 2282112 * log_xom * xom4 * xsg4 + 691488 * xom5 * xsg4 + 887040 * L * xom5 * xsg4 - 887040 * log_xom * xom5 * xsg4 - 88200 * xom6 * xsg4 - 168840 * L * xom6 * xsg4 + 168840 * log_xom * xom6 * xsg4 + 4032 * xom7 * xsg4 + 15120 * L * xom7 * xsg4 - 15120 * log_xom * xom7 * xsg4 - 504 * L * xom8 * xsg4 + 504 * log_xom * xom8 * xsg4 + 288288 * xsg5 + 333648 * xom * xsg5 - 206388 * L * xsg5 + 206388 * log_xom * xsg5 - 2423232 * xom2 * xsg5 - 1378944 * L * xom2 * xsg5 + 1378944 * log_xom * xom2 * xsg5 + 4178160 * xom3 * xsg5 + 3127824 * L * xom3 * xsg5 - 3127824 * log_xom * xom3 * xsg5 - 2671200 * xom4 * xsg5 - 2587536 * L * xom4 * xsg5 + 2587536 * log_xom * xom4 * xsg5 + 747432 * xom5 * xsg5 + 978768 * L * xom5 * xsg5 - 978768 * log_xom * xom5 * xsg5 - 91728 * xom6 * xsg5 - 180432 * L * xom6 * xsg5 + 180432 * log_xom * xom6 * xsg5 + 4032 * xom7 * xsg5 + 15624 * L * xom7 * xsg5 - 15624 * log_xom * xom7 * xsg5 - 504 * L * xom8 * xsg5 + 504 * log_xom * xom8 * xsg5 + 171360 * xsg6 + 330624 * xom * xsg6 + 116424 * L * xsg6 + 84672 * xom * L * xsg6 - 116424 * log_xom * xsg6 - 84672 * xom * log_xom * xsg6 - 1951488 * xom2 * xsg6 - 1149120 * L * xom2 * xsg6 + 1149120 * log_xom * xom2 * xsg6 + 3284736 * xom3 * xsg6 + 2483712 * L * xom3 * xsg6 - 2483712 * log_xom * xom3 * xsg6 - 2017680 * xom4 * xsg6 - 1993824 * L * xom4 * xsg6 + 1993824 * log_xom * xom4 * xsg6 + 539616 * xom5 * xsg6 + 725760 * L * xom5 * xsg6 - 725760 * log_xom * xom5 * xsg6 - 63504 * xom6 * xsg6 - 128688 * L * xom6 * xsg6 + 128688 * log_xom * xom6 * xsg6 + 2688 * xom7 * xsg6 + 10752 * L * xom7 * xsg6 - 10752 * log_xom * xom7 * xsg6 - 336 * L * xom8 * xsg6 + 336 * log_xom * xom8 * xsg6 + 15840 * xsg7 + 141984 * xom * xsg7 - 84024 * L * xsg7 + 78624 * xom * L * xsg7 + 84024 * log_xom * xsg7 - 78624 * xom * log_xom * xsg7 - 1073664 * xom2 * xsg7 - 614304 * L * xom2 * xsg7 + 614304 * log_xom * xom2 * xsg7 + 1721376 * xom3 * xsg7 + 1329120 * L * xom3 * xsg7 - 1329120 * log_xom * xom3 * xsg7 - 990720 * xom4 * xsg7 - 1010592 * L * xom4 * xsg7 + 1010592 * log_xom * xom4 * xsg7 + 250704 * xom5 * xsg7 + 348480 * L * xom5 * xsg7 - 348480 * log_xom * xom5 * xsg7 - 28224 * xom6 * xsg7 - 59040 * L * xom6 * xsg7 + 59040 * log_xom * xom6 * xsg7 + 1152 * xom7 * xsg7 + 4752 * L * xom7 * xsg7 - 4752 * log_xom * xom7 * xsg7 - 144 * L * xom8 * xsg7 + 144 * log_xom * xom8 * xsg7 + 29448 * xsg8 + 42192 * xom * xsg8 + 13338 * L * xsg8 - 15984 * xom * L * xsg8 - 13338 * log_xom * xsg8 + 15984 * xom * log_xom * xsg8 - 395568 * xom2 * xsg8 - 229608 * L * xom2 * xsg8 + 229608 * log_xom * xom2 * xsg8 + 547056 * xom3 * xsg8 + 446112 * L * xom3 * xsg8 - 446112 * log_xom * xom3 * xsg8 - 286380 * xom4 * xsg8 - 306000 * L * xom4 * xsg8 + 306000 * log_xom * xom4 * xsg8 + 67968 * xom5 * xsg8 + 98208 * L * xom5 * xsg8 - 98208 * log_xom * xom5 * xsg8 - 7308 * xom6 * xsg8 - 15804 * L * xom6 * xsg8 + 15804 * log_xom * xom6 * xsg8 + 288 * xom7 * xsg8 + 1224 * L * xom7 * xsg8 - 1224 * log_xom * xom7 * xsg8 - 36 * L * xom8 * xsg8 + 36 * log_xom * xom8 * xsg8 + 24408 * xom * xsg9 - 630 * L * xsg9 + 13968 * xom * L * xsg9 + 630 * log_xom * xsg9 - 13968 * xom * log_xom * xsg9 - 75168 * xom2 * xsg9 - 55152 * L * xom2 * xsg9 + 55152 * log_xom * xom2 * xsg9 + 79880 * xom3 * xsg9 + 72024 * L * xom3 * xsg9 - 72024 * log_xom * xom3 * xsg9 - 37040 * xom4 * xsg9 - 42072 * L * xom4 * xsg9 + 42072 * log_xom * xom4 * xsg9 + 8188 * xom5 * xsg9 + 12360 * L * xom5 * xsg9 - 12360 * log_xom * xom5 * xsg9 - 840 * xom6 * xsg9 - 1880 * L * xom6 * xsg9 + 1880 * log_xom * xom6 * xsg9 + 32 * xom7 * xsg9 + 140 * L * xom7 * xsg9 - 140 * log_xom * xom7 * xsg9 - 4 * L * xom8 * xsg9 + 4 * log_xom * xom8 * xsg9) * xsgplus_m10) / 2835. + ((-29448 * exp_xom_xsgplus - 29448 * xom * exp_xom_xsgplus - 15840 * xsg * exp_xom_xsgplus - 45288 * xom * xsg * exp_xom_xsgplus + 19638 * eulergamma * exp_xom_xsgplus + 19008 * xom * eulergamma * exp_xom_xsgplus - 55674 * xsg * eulergamma * exp_xom_xsgplus - 42336 * xom * xsg * eulergamma * exp_xom_xsgplus - 19638 * expint_ei_plus_xom * exp_xom_xsgplus - 19008 * xom * expint_ei_plus_xom * exp_xom_xsgplus + 55674 * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 42336 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus + 5670 * xom * exp_xom_xsgplus * hyp_plus_xom - 45360 * xom * xsg * exp_xom_xsgplus * hyp_plus_xom + 5670 * xom * xsg * hyp_xom_xsg + 5670 * eulergamma * L - 45360 * xsg * eulergamma * L - 5670 * expint_ei_xom_xsg * L + 45360 * xsg * expint_ei_xom_xsg * L + 19638 * exp_xom_xsgplus * L + 19008 * xom * exp_xom_xsgplus * L - 55674 * xsg * exp_xom_xsgplus * L - 42336 * xom * xsg * exp_xom_xsgplus * L - 5670 * eulergamma * exp_xom_xsgplus * L - 5670 * xom * eulergamma * exp_xom_xsgplus * L + 45360 * xsg * eulergamma * exp_xom_xsgplus * L + 39690 * xom * xsg * eulergamma * exp_xom_xsgplus * L + 5670 * expint_ei_plus_xom * exp_xom_xsgplus * L + 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L - 45360 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L - 39690 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L + 2835 * L2 - 22680 * xsg * L2 - 2835 * exp_xom_xsgplus * L2 - 2835 * xom * exp_xom_xsgplus * L2 + 22680 * xsg * exp_xom_xsgplus * L2 + 19845 * xom * xsg * exp_xom_xsgplus * L2 - 2835 * log2_xom + 22680 * xsg * log2_xom + 2835 * exp_xom_xsgplus * log2_xom + 2835 * xom * exp_xom_xsgplus * log2_xom - 22680 * xsg * exp_xom_xsgplus * log2_xom - 19845 * xom * xsg * exp_xom_xsgplus * log2_xom - 5670 * eulergamma * log_xom + 45360 * xsg * eulergamma * log_xom + 5670 * expint_ei_xom_xsg * log_xom - 45360 * xsg * expint_ei_xom_xsg * log_xom + 5670 * eulergamma * exp_xom_xsgplus * log_xom + 5670 * xom * eulergamma * exp_xom_xsgplus * log_xom - 45360 * xsg * eulergamma * exp_xom_xsgplus * log_xom - 39690 * xom * xsg * eulergamma * exp_xom_xsgplus * log_xom - 5670 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom - 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 45360 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 39690 * xom * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom + 5670 * L * log_xsg - 45360 * xsg * L * log_xsg - 5670 * log_xom * log_xsg + 45360 * xsg * log_xom * log_xsg + 4 * exp_xom_xsgplus * hyp_plus_xom * xom10 + 36 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom10 + 109872 * xsg * exp_xom_xsgplus * xom2 + 19008 * eulergamma * exp_xom_xsgplus * xom2 + 83376 * xsg * eulergamma * exp_xom_xsgplus * xom2 - 19008 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 - 83376 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom2 + 5670 * exp_xom_xsgplus * hyp_plus_xom * xom2 - 39690 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom2 + 19008 * exp_xom_xsgplus * L * xom2 + 83376 * xsg * exp_xom_xsgplus * L * xom2 + 45360 * xsg * eulergamma * exp_xom_xsgplus * L * xom2 - 45360 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 + 22680 * xsg * exp_xom_xsgplus * L2 * xom2 - 22680 * xsg * exp_xom_xsgplus * log2_xom * xom2 - 45360 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom2 + 45360 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 - 36624 * exp_xom_xsgplus * xom3 - 329616 * xsg * exp_xom_xsgplus * xom3 - 29232 * eulergamma * exp_xom_xsgplus * xom3 - 321552 * xsg * eulergamma * exp_xom_xsgplus * xom3 + 29232 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 + 321552 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom3 - 45360 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom3 - 29232 * exp_xom_xsgplus * L * xom3 - 321552 * xsg * exp_xom_xsgplus * L * xom3 - 15120 * eulergamma * exp_xom_xsgplus * L * xom3 - 136080 * xsg * eulergamma * exp_xom_xsgplus * L * xom3 + 15120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 + 136080 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 - 7560 * exp_xom_xsgplus * L2 * xom3 - 68040 * xsg * exp_xom_xsgplus * L2 * xom3 + 7560 * exp_xom_xsgplus * log2_xom * xom3 + 68040 * xsg * exp_xom_xsgplus * log2_xom * xom3 + 15120 * eulergamma * exp_xom_xsgplus * log_xom * xom3 + 136080 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom3 - 15120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 - 136080 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 + 27720 * exp_xom_xsgplus * xom4 + 277200 * xsg * exp_xom_xsgplus * xom4 + 39816 * eulergamma * exp_xom_xsgplus * xom4 + 383544 * xsg * eulergamma * exp_xom_xsgplus * xom4 - 39816 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 - 383544 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom4 + 15120 * exp_xom_xsgplus * hyp_plus_xom * xom4 + 136080 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom4 + 39816 * exp_xom_xsgplus * L * xom4 + 383544 * xsg * exp_xom_xsgplus * L * xom4 + 22680 * eulergamma * exp_xom_xsgplus * L * xom4 + 226800 * xsg * eulergamma * exp_xom_xsgplus * L * xom4 - 22680 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 - 226800 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 + 11340 * exp_xom_xsgplus * L2 * xom4 + 113400 * xsg * exp_xom_xsgplus * L2 * xom4 - 11340 * exp_xom_xsgplus * log2_xom * xom4 - 113400 * xsg * exp_xom_xsgplus * log2_xom * xom4 - 22680 * eulergamma * exp_xom_xsgplus * log_xom * xom4 - 226800 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom4 + 22680 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 + 226800 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 - 11592 * exp_xom_xsgplus * xom5 - 110376 * xsg * exp_xom_xsgplus * xom5 - 21168 * eulergamma * exp_xom_xsgplus * xom5 - 206640 * xsg * eulergamma * exp_xom_xsgplus * xom5 + 21168 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 + 206640 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom5 - 22680 * exp_xom_xsgplus * hyp_plus_xom * xom5 - 226800 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom5 - 21168 * exp_xom_xsgplus * L * xom5 - 206640 * xsg * exp_xom_xsgplus * L * xom5 - 16632 * eulergamma * exp_xom_xsgplus * L * xom5 - 161784 * xsg * eulergamma * exp_xom_xsgplus * L * xom5 + 16632 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 + 161784 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 - 8316 * exp_xom_xsgplus * L2 * xom5 - 80892 * xsg * exp_xom_xsgplus * L2 * xom5 + 8316 * exp_xom_xsgplus * log2_xom * xom5 + 80892 * xsg * exp_xom_xsgplus * log2_xom * xom5 + 16632 * eulergamma * exp_xom_xsgplus * log_xom * xom5 + 161784 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom5 - 16632 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 - 161784 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 + 2016 * exp_xom_xsgplus * xom6 + 19152 * xsg * exp_xom_xsgplus * xom6 + 5712 * eulergamma * exp_xom_xsgplus * xom6 + 54432 * xsg * eulergamma * exp_xom_xsgplus * xom6 - 5712 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 - 54432 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom6 + 16632 * exp_xom_xsgplus * hyp_plus_xom * xom6 + 161784 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom6 + 5712 * exp_xom_xsgplus * L * xom6 + 54432 * xsg * exp_xom_xsgplus * L * xom6 + 6048 * eulergamma * exp_xom_xsgplus * L * xom6 + 58464 * xsg * eulergamma * exp_xom_xsgplus * L * xom6 - 6048 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 - 58464 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 + 3024 * exp_xom_xsgplus * L2 * xom6 + 29232 * xsg * exp_xom_xsgplus * L2 * xom6 - 3024 * exp_xom_xsgplus * log2_xom * xom6 - 29232 * xsg * exp_xom_xsgplus * log2_xom * xom6 - 6048 * eulergamma * exp_xom_xsgplus * log_xom * xom6 - 58464 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom6 + 6048 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 + 58464 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 - 144 * exp_xom_xsgplus * xom7 - 1296 * xsg * exp_xom_xsgplus * xom7 - 720 * eulergamma * exp_xom_xsgplus * xom7 - 6768 * xsg * eulergamma * exp_xom_xsgplus * xom7 + 720 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 + 6768 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom7 - 6048 * exp_xom_xsgplus * hyp_plus_xom * xom7 - 58464 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom7 - 720 * exp_xom_xsgplus * L * xom7 - 6768 * xsg * exp_xom_xsgplus * L * xom7 - 1152 * eulergamma * exp_xom_xsgplus * L * xom7 - 10944 * xsg * eulergamma * exp_xom_xsgplus * L * xom7 + 1152 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 + 10944 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 - 576 * exp_xom_xsgplus * L2 * xom7 - 5472 * xsg * exp_xom_xsgplus * L2 * xom7 + 576 * exp_xom_xsgplus * log2_xom * xom7 + 5472 * xsg * exp_xom_xsgplus * log2_xom * xom7 + 1152 * eulergamma * exp_xom_xsgplus * log_xom * xom7 + 10944 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom7 - 1152 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 - 10944 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 + 36 * eulergamma * exp_xom_xsgplus * xom8 + 324 * xsg * eulergamma * exp_xom_xsgplus * xom8 - 36 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 - 324 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * xom8 + 1152 * exp_xom_xsgplus * hyp_plus_xom * xom8 + 10944 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom8 + 36 * exp_xom_xsgplus * L * xom8 + 324 * xsg * exp_xom_xsgplus * L * xom8 + 108 * eulergamma * exp_xom_xsgplus * L * xom8 + 1008 * xsg * eulergamma * exp_xom_xsgplus * L * xom8 - 108 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 - 1008 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 + 54 * exp_xom_xsgplus * L2 * xom8 + 504 * xsg * exp_xom_xsgplus * L2 * xom8 - 54 * exp_xom_xsgplus * log2_xom * xom8 - 504 * xsg * exp_xom_xsgplus * log2_xom * xom8 - 108 * eulergamma * exp_xom_xsgplus * log_xom * xom8 - 1008 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom8 + 108 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 + 1008 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 - 108 * exp_xom_xsgplus * hyp_plus_xom * xom9 - 1008 * xsg * exp_xom_xsgplus * hyp_plus_xom * xom9 - 4 * eulergamma * exp_xom_xsgplus * L * xom9 - 36 * xsg * eulergamma * exp_xom_xsgplus * L * xom9 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 + 36 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 - 2 * exp_xom_xsgplus * L2 * xom9 - 18 * xsg * exp_xom_xsgplus * L2 * xom9 + 2 * exp_xom_xsgplus * log2_xom * xom9 + 18 * xsg * exp_xom_xsgplus * log2_xom * xom9 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom9 + 36 * xsg * eulergamma * exp_xom_xsgplus * log_xom * xom9 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 - 36 * xsg * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 - 171360 * exp_xom_xsgplus * xsg2 - 187200 * xom * exp_xom_xsgplus * xsg2 + 192024 * eulergamma * exp_xom_xsgplus * xsg2 + 108000 * xom * eulergamma * exp_xom_xsgplus * xsg2 - 192024 * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 - 108000 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg2 + 158760 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg2 - 45360 * xom * hyp_xom_xsg * xsg2 + 158760 * eulergamma * L * xsg2 - 158760 * expint_ei_xom_xsg * L * xsg2 + 192024 * exp_xom_xsgplus * L * xsg2 + 108000 * xom * exp_xom_xsgplus * L * xsg2 - 158760 * eulergamma * exp_xom_xsgplus * L * xsg2 - 113400 * xom * eulergamma * exp_xom_xsgplus * L * xsg2 + 158760 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 113400 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg2 + 79380 * L2 * xsg2 - 79380 * exp_xom_xsgplus * L2 * xsg2 - 56700 * xom * exp_xom_xsgplus * L2 * xsg2 - 79380 * log2_xom * xsg2 + 79380 * exp_xom_xsgplus * log2_xom * xsg2 + 56700 * xom * exp_xom_xsgplus * log2_xom * xsg2 - 158760 * eulergamma * log_xom * xsg2 + 158760 * expint_ei_xom_xsg * log_xom * xsg2 + 158760 * eulergamma * exp_xom_xsgplus * log_xom * xsg2 + 113400 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg2 - 158760 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 - 113400 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg2 + 158760 * L * log_xsg * xsg2 - 158760 * log_xom * log_xsg * xsg2 + 144 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg2 + 546336 * exp_xom_xsgplus * xom2 * xsg2 + 460512 * eulergamma * exp_xom_xsgplus * xom2 * xsg2 - 460512 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg2 + 113400 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg2 + 460512 * exp_xom_xsgplus * L * xom2 * xsg2 + 90720 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg2 - 90720 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg2 + 45360 * exp_xom_xsgplus * L2 * xom2 * xsg2 - 45360 * exp_xom_xsgplus * log2_xom * xom2 * xsg2 - 90720 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg2 + 90720 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg2 - 1429344 * exp_xom_xsgplus * xom3 * xsg2 - 1419264 * eulergamma * exp_xom_xsgplus * xom3 * xsg2 + 1419264 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg2 - 90720 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg2 - 1419264 * exp_xom_xsgplus * L * xom3 * xsg2 - 635040 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg2 + 635040 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg2 - 317520 * exp_xom_xsgplus * L2 * xom3 * xsg2 + 317520 * exp_xom_xsgplus * log2_xom * xom3 * xsg2 + 635040 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg2 - 635040 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg2 + 1219680 * exp_xom_xsgplus * xom4 * xsg2 + 1685376 * eulergamma * exp_xom_xsgplus * xom4 * xsg2 - 1685376 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg2 + 635040 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg2 + 1685376 * exp_xom_xsgplus * L * xom4 * xsg2 + 997920 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg2 - 997920 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg2 + 498960 * exp_xom_xsgplus * L2 * xom4 * xsg2 - 498960 * exp_xom_xsgplus * log2_xom * xom4 * xsg2 - 997920 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg2 + 997920 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg2 - 471744 * exp_xom_xsgplus * xom5 * xsg2 - 897120 * eulergamma * exp_xom_xsgplus * xom5 * xsg2 + 897120 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg2 - 997920 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg2 - 897120 * exp_xom_xsgplus * L * xom5 * xsg2 - 707616 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg2 + 707616 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg2 - 353808 * exp_xom_xsgplus * L2 * xom5 * xsg2 + 353808 * exp_xom_xsgplus * log2_xom * xom5 * xsg2 + 707616 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg2 - 707616 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg2 + 80640 * exp_xom_xsgplus * xom6 * xsg2 + 231840 * eulergamma * exp_xom_xsgplus * xom6 * xsg2 - 231840 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg2 + 707616 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg2 + 231840 * exp_xom_xsgplus * L * xom6 * xsg2 + 252000 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg2 - 252000 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg2 + 126000 * exp_xom_xsgplus * L2 * xom6 * xsg2 - 126000 * exp_xom_xsgplus * log2_xom * xom6 * xsg2 - 252000 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg2 + 252000 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg2 - 5184 * exp_xom_xsgplus * xom7 * xsg2 - 28224 * eulergamma * exp_xom_xsgplus * xom7 * xsg2 + 28224 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg2 - 252000 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg2 - 28224 * exp_xom_xsgplus * L * xom7 * xsg2 - 46368 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg2 + 46368 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg2 - 23184 * exp_xom_xsgplus * L2 * xom7 * xsg2 + 23184 * exp_xom_xsgplus * log2_xom * xom7 * xsg2 + 46368 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg2 - 46368 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg2 + 1296 * eulergamma * exp_xom_xsgplus * xom8 * xsg2 - 1296 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg2 + 46368 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg2 + 1296 * exp_xom_xsgplus * L * xom8 * xsg2 + 4176 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg2 - 4176 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg2 + 2088 * exp_xom_xsgplus * L2 * xom8 * xsg2 - 2088 * exp_xom_xsgplus * log2_xom * xom8 * xsg2 - 4176 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg2 + 4176 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg2 - 4176 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg2 - 144 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg2 + 144 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg2 - 72 * exp_xom_xsgplus * L2 * xom9 * xsg2 + 72 * exp_xom_xsgplus * log2_xom * xom9 * xsg2 + 144 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg2 - 144 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg2 - 288288 * exp_xom_xsgplus * xsg3 - 459648 * xom * exp_xom_xsgplus * xsg3 - 74088 * eulergamma * exp_xom_xsgplus * xsg3 + 42336 * xom * eulergamma * exp_xom_xsgplus * xsg3 + 74088 * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 42336 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg3 - 317520 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg3 + 158760 * xom * hyp_xom_xsg * xsg3 - 317520 * eulergamma * L * xsg3 + 317520 * expint_ei_xom_xsg * L * xsg3 - 74088 * exp_xom_xsgplus * L * xsg3 + 42336 * xom * exp_xom_xsgplus * L * xsg3 + 317520 * eulergamma * exp_xom_xsgplus * L * xsg3 + 158760 * xom * eulergamma * exp_xom_xsgplus * L * xsg3 - 317520 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 158760 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg3 - 158760 * L2 * xsg3 + 158760 * exp_xom_xsgplus * L2 * xsg3 + 79380 * xom * exp_xom_xsgplus * L2 * xsg3 + 158760 * log2_xom * xsg3 - 158760 * exp_xom_xsgplus * log2_xom * xsg3 - 79380 * xom * exp_xom_xsgplus * log2_xom * xsg3 + 317520 * eulergamma * log_xom * xsg3 - 317520 * expint_ei_xom_xsg * log_xom * xsg3 - 317520 * eulergamma * exp_xom_xsgplus * log_xom * xsg3 - 158760 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg3 + 317520 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 + 158760 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg3 - 317520 * L * log_xsg * xsg3 + 317520 * log_xom * log_xsg * xsg3 + 336 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg3 + 1443456 * exp_xom_xsgplus * xom2 * xsg3 + 1215648 * eulergamma * exp_xom_xsgplus * xom2 * xsg3 - 1215648 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg3 - 158760 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg3 + 1215648 * exp_xom_xsgplus * L * xom2 * xsg3 + 362880 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg3 - 362880 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg3 + 181440 * exp_xom_xsgplus * L2 * xom2 * xsg3 - 181440 * exp_xom_xsgplus * log2_xom * xom2 * xsg3 - 362880 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg3 + 362880 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg3 - 3731616 * exp_xom_xsgplus * xom3 * xsg3 - 3709440 * eulergamma * exp_xom_xsgplus * xom3 * xsg3 + 3709440 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg3 - 362880 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg3 - 3709440 * exp_xom_xsgplus * L * xom3 * xsg3 - 1663200 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg3 + 1663200 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg3 - 831600 * exp_xom_xsgplus * L2 * xom3 * xsg3 + 831600 * exp_xom_xsgplus * log2_xom * xom3 * xsg3 + 1663200 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg3 - 1663200 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg3 + 3134880 * exp_xom_xsgplus * xom4 * xsg3 + 4372704 * eulergamma * exp_xom_xsgplus * xom4 * xsg3 - 4372704 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg3 + 1663200 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg3 + 4372704 * exp_xom_xsgplus * L * xom4 * xsg3 + 2600640 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg3 - 2600640 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg3 + 1300320 * exp_xom_xsgplus * L2 * xom4 * xsg3 - 1300320 * exp_xom_xsgplus * log2_xom * xom4 * xsg3 - 2600640 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg3 + 2600640 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg3 - 1185408 * exp_xom_xsgplus * xom5 * xsg3 - 2284128 * eulergamma * exp_xom_xsgplus * xom5 * xsg3 + 2284128 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg3 - 2600640 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg3 - 2284128 * exp_xom_xsgplus * L * xom5 * xsg3 - 1820448 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg3 + 1820448 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg3 - 910224 * exp_xom_xsgplus * L2 * xom5 * xsg3 + 910224 * exp_xom_xsgplus * log2_xom * xom5 * xsg3 + 1820448 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg3 - 1820448 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg3 + 197568 * exp_xom_xsgplus * xom6 * xsg3 + 578592 * eulergamma * exp_xom_xsgplus * xom6 * xsg3 - 578592 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg3 + 1820448 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg3 + 578592 * exp_xom_xsgplus * L * xom6 * xsg3 + 637056 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg3 - 637056 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg3 + 318528 * exp_xom_xsgplus * L2 * xom6 * xsg3 - 318528 * exp_xom_xsgplus * log2_xom * xom6 * xsg3 - 637056 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg3 + 637056 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg3 - 12096 * exp_xom_xsgplus * xom7 * xsg3 - 68544 * eulergamma * exp_xom_xsgplus * xom7 * xsg3 + 68544 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg3 - 637056 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg3 - 68544 * exp_xom_xsgplus * L * xom7 * xsg3 - 114912 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg3 + 114912 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg3 - 57456 * exp_xom_xsgplus * L2 * xom7 * xsg3 + 57456 * exp_xom_xsgplus * log2_xom * xom7 * xsg3 + 114912 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg3 - 114912 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg3 + 3024 * eulergamma * exp_xom_xsgplus * xom8 * xsg3 - 3024 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg3 + 114912 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg3 + 3024 * exp_xom_xsgplus * L * xom8 * xsg3 + 10080 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg3 - 10080 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg3 + 5040 * exp_xom_xsgplus * L2 * xom8 * xsg3 - 5040 * exp_xom_xsgplus * log2_xom * xom8 * xsg3 - 10080 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg3 + 10080 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg3 - 10080 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg3 - 336 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg3 + 336 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg3 - 168 * exp_xom_xsgplus * L2 * xom9 * xsg3 + 168 * exp_xom_xsgplus * log2_xom * xom9 * xsg3 + 336 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg3 - 336 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg3 - 206640 * exp_xom_xsgplus * xsg4 - 494928 * xom * exp_xom_xsgplus * xsg4 + 79380 * eulergamma * exp_xom_xsgplus * xsg4 - 127008 * xom * eulergamma * exp_xom_xsgplus * xsg4 - 79380 * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 + 127008 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg4 + 396900 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg4 - 317520 * xom * hyp_xom_xsg * xsg4 + 396900 * eulergamma * L * xsg4 - 396900 * expint_ei_xom_xsg * L * xsg4 + 79380 * exp_xom_xsgplus * L * xsg4 - 127008 * xom * exp_xom_xsgplus * L * xsg4 - 396900 * eulergamma * exp_xom_xsgplus * L * xsg4 - 79380 * xom * eulergamma * exp_xom_xsgplus * L * xsg4 + 396900 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 + 79380 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg4 + 198450 * L2 * xsg4 - 198450 * exp_xom_xsgplus * L2 * xsg4 - 39690 * xom * exp_xom_xsgplus * L2 * xsg4 - 198450 * log2_xom * xsg4 + 198450 * exp_xom_xsgplus * log2_xom * xsg4 + 39690 * xom * exp_xom_xsgplus * log2_xom * xsg4 - 396900 * eulergamma * log_xom * xsg4 + 396900 * expint_ei_xom_xsg * log_xom * xsg4 + 396900 * eulergamma * exp_xom_xsgplus * log_xom * xsg4 + 79380 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg4 - 396900 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 - 79380 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg4 + 396900 * L * log_xsg * xsg4 - 396900 * log_xom * log_xsg * xsg4 + 504 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg4 + 2614752 * exp_xom_xsgplus * xom2 * xsg4 + 1953504 * eulergamma * exp_xom_xsgplus * xom2 * xsg4 - 1953504 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg4 + 79380 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg4 + 1953504 * exp_xom_xsgplus * L * xom2 * xsg4 + 635040 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg4 - 635040 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg4 + 317520 * exp_xom_xsgplus * L2 * xom2 * xsg4 - 317520 * exp_xom_xsgplus * log2_xom * xom2 * xsg4 - 635040 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg4 + 635040 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg4 - 6338304 * exp_xom_xsgplus * xom3 * xsg4 - 6412896 * eulergamma * exp_xom_xsgplus * xom3 * xsg4 + 6412896 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg4 - 635040 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg4 - 6412896 * exp_xom_xsgplus * L * xom3 * xsg4 - 2842560 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg4 + 2842560 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg4 - 1421280 * exp_xom_xsgplus * L2 * xom3 * xsg4 + 1421280 * exp_xom_xsgplus * log2_xom * xom3 * xsg4 + 2842560 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg4 - 2842560 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg4 + 5226480 * exp_xom_xsgplus * xom4 * xsg4 + 7365456 * eulergamma * exp_xom_xsgplus * xom4 * xsg4 - 7365456 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg4 + 2842560 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg4 + 7365456 * exp_xom_xsgplus * L * xom4 * xsg4 + 4430160 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg4 - 4430160 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg4 + 2215080 * exp_xom_xsgplus * L2 * xom4 * xsg4 - 2215080 * exp_xom_xsgplus * log2_xom * xom4 * xsg4 - 4430160 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg4 + 4430160 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg4 - 1926288 * exp_xom_xsgplus * xom5 * xsg4 - 3769920 * eulergamma * exp_xom_xsgplus * xom5 * xsg4 + 3769920 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg4 - 4430160 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg4 - 3769920 * exp_xom_xsgplus * L * xom5 * xsg4 - 3039120 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg4 + 3039120 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg4 - 1519560 * exp_xom_xsgplus * L2 * xom5 * xsg4 + 1519560 * exp_xom_xsgplus * log2_xom * xom5 * xsg4 + 3039120 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg4 - 3039120 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg4 + 310464 * exp_xom_xsgplus * xom6 * xsg4 + 931392 * eulergamma * exp_xom_xsgplus * xom6 * xsg4 - 931392 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg4 + 3039120 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg4 + 931392 * exp_xom_xsgplus * L * xom6 * xsg4 + 1042272 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg4 - 1042272 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg4 + 521136 * exp_xom_xsgplus * L2 * xom6 * xsg4 - 521136 * exp_xom_xsgplus * log2_xom * xom6 * xsg4 - 1042272 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg4 + 1042272 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg4 - 18144 * exp_xom_xsgplus * xom7 * xsg4 - 106848 * eulergamma * exp_xom_xsgplus * xom7 * xsg4 + 106848 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg4 - 1042272 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg4 - 106848 * exp_xom_xsgplus * L * xom7 * xsg4 - 183456 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg4 + 183456 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg4 - 91728 * exp_xom_xsgplus * L2 * xom7 * xsg4 + 91728 * exp_xom_xsgplus * log2_xom * xom7 * xsg4 + 183456 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg4 - 183456 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg4 + 4536 * eulergamma * exp_xom_xsgplus * xom8 * xsg4 - 4536 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg4 + 183456 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg4 + 4536 * exp_xom_xsgplus * L * xom8 * xsg4 + 15624 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg4 - 15624 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg4 + 7812 * exp_xom_xsgplus * L2 * xom8 * xsg4 - 7812 * exp_xom_xsgplus * log2_xom * xom8 * xsg4 - 15624 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg4 + 15624 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg4 - 15624 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg4 - 504 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg4 + 504 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg4 - 252 * exp_xom_xsgplus * L2 * xom9 * xsg4 + 252 * exp_xom_xsgplus * log2_xom * xom9 * xsg4 + 504 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg4 - 504 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg4 - 288288 * exp_xom_xsgplus * xsg5 - 494928 * xom * exp_xom_xsgplus * xsg5 + 206388 * eulergamma * exp_xom_xsgplus * xsg5 + 127008 * xom * eulergamma * exp_xom_xsgplus * xsg5 - 206388 * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 - 127008 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg5 - 317520 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg5 + 396900 * xom * hyp_xom_xsg * xsg5 - 317520 * eulergamma * L * xsg5 + 317520 * expint_ei_xom_xsg * L * xsg5 + 206388 * exp_xom_xsgplus * L * xsg5 + 127008 * xom * exp_xom_xsgplus * L * xsg5 + 317520 * eulergamma * exp_xom_xsgplus * L * xsg5 - 79380 * xom * eulergamma * exp_xom_xsgplus * L * xsg5 - 317520 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 + 79380 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg5 - 158760 * L2 * xsg5 + 158760 * exp_xom_xsgplus * L2 * xsg5 - 39690 * xom * exp_xom_xsgplus * L2 * xsg5 + 158760 * log2_xom * xsg5 - 158760 * exp_xom_xsgplus * log2_xom * xsg5 + 39690 * xom * exp_xom_xsgplus * log2_xom * xsg5 + 317520 * eulergamma * log_xom * xsg5 - 317520 * expint_ei_xom_xsg * log_xom * xsg5 - 317520 * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 79380 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg5 + 317520 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 79380 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg5 - 317520 * L * log_xsg * xsg5 + 317520 * log_xom * log_xsg * xsg5 + 504 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg5 + 3215520 * exp_xom_xsgplus * xom2 * xsg5 + 2395008 * eulergamma * exp_xom_xsgplus * xom2 * xsg5 - 2395008 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg5 + 79380 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg5 + 2395008 * exp_xom_xsgplus * L * xom2 * xsg5 + 635040 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg5 - 635040 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg5 + 317520 * exp_xom_xsgplus * L2 * xom2 * xsg5 - 317520 * exp_xom_xsgplus * log2_xom * xom2 * xsg5 - 635040 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg5 + 635040 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg5 - 7285824 * exp_xom_xsgplus * xom3 * xsg5 - 7513632 * eulergamma * exp_xom_xsgplus * xom3 * xsg5 + 7513632 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg5 - 635040 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg5 - 7513632 * exp_xom_xsgplus * L * xom3 * xsg5 - 3386880 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg5 + 3386880 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg5 - 1693440 * exp_xom_xsgplus * L2 * xom3 * xsg5 + 1693440 * exp_xom_xsgplus * log2_xom * xom3 * xsg5 + 3386880 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg5 - 3386880 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg5 + 5886720 * exp_xom_xsgplus * xom4 * xsg5 + 8393616 * eulergamma * exp_xom_xsgplus * xom4 * xsg5 - 8393616 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg5 + 3386880 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg5 + 8393616 * exp_xom_xsgplus * L * xom4 * xsg5 + 5110560 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg5 - 5110560 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg5 + 2555280 * exp_xom_xsgplus * L2 * xom4 * xsg5 - 2555280 * exp_xom_xsgplus * log2_xom * xom4 * xsg5 - 5110560 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg5 + 5110560 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg5 - 2095632 * exp_xom_xsgplus * xom5 * xsg5 - 4189248 * eulergamma * exp_xom_xsgplus * xom5 * xsg5 + 4189248 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg5 - 5110560 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg5 - 4189248 * exp_xom_xsgplus * L * xom5 * xsg5 - 3426192 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg5 + 3426192 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg5 - 1713096 * exp_xom_xsgplus * L2 * xom5 * xsg5 + 1713096 * exp_xom_xsgplus * log2_xom * xom5 * xsg5 + 3426192 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg5 - 3426192 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg5 + 324576 * exp_xom_xsgplus * xom6 * xsg5 + 1001952 * eulergamma * exp_xom_xsgplus * xom6 * xsg5 - 1001952 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg5 + 3426192 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg5 + 1001952 * exp_xom_xsgplus * L * xom6 * xsg5 + 1145088 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg5 - 1145088 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg5 + 572544 * exp_xom_xsgplus * L2 * xom6 * xsg5 - 572544 * exp_xom_xsgplus * log2_xom * xom6 * xsg5 - 1145088 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg5 + 1145088 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg5 - 18144 * exp_xom_xsgplus * xom7 * xsg5 - 110880 * eulergamma * exp_xom_xsgplus * xom7 * xsg5 + 110880 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg5 - 1145088 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg5 - 110880 * exp_xom_xsgplus * L * xom7 * xsg5 - 195552 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg5 + 195552 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg5 - 97776 * exp_xom_xsgplus * L2 * xom7 * xsg5 + 97776 * exp_xom_xsgplus * log2_xom * xom7 * xsg5 + 195552 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg5 - 195552 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg5 + 4536 * eulergamma * exp_xom_xsgplus * xom8 * xsg5 - 4536 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg5 + 195552 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg5 + 4536 * exp_xom_xsgplus * L * xom8 * xsg5 + 16128 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg5 - 16128 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg5 + 8064 * exp_xom_xsgplus * L2 * xom8 * xsg5 - 8064 * exp_xom_xsgplus * log2_xom * xom8 * xsg5 - 16128 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg5 + 16128 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg5 - 16128 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg5 - 504 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg5 + 504 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg5 - 252 * exp_xom_xsgplus * L2 * xom9 * xsg5 + 252 * exp_xom_xsgplus * log2_xom * xom9 * xsg5 + 504 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg5 - 504 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg5 - 171360 * exp_xom_xsgplus * xsg6 - 459648 * xom * exp_xom_xsgplus * xsg6 - 116424 * eulergamma * exp_xom_xsgplus * xsg6 - 42336 * xom * eulergamma * exp_xom_xsgplus * xsg6 + 116424 * expint_ei_plus_xom * exp_xom_xsgplus * xsg6 + 42336 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg6 + 158760 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg6 - 317520 * xom * hyp_xom_xsg * xsg6 + 158760 * eulergamma * L * xsg6 - 158760 * expint_ei_xom_xsg * L * xsg6 - 116424 * exp_xom_xsgplus * L * xsg6 - 42336 * xom * exp_xom_xsgplus * L * xsg6 - 158760 * eulergamma * exp_xom_xsgplus * L * xsg6 + 158760 * xom * eulergamma * exp_xom_xsgplus * L * xsg6 + 158760 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg6 - 158760 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg6 + 79380 * L2 * xsg6 - 79380 * exp_xom_xsgplus * L2 * xsg6 + 79380 * xom * exp_xom_xsgplus * L2 * xsg6 - 79380 * log2_xom * xsg6 + 79380 * exp_xom_xsgplus * log2_xom * xsg6 - 79380 * xom * exp_xom_xsgplus * log2_xom * xsg6 - 158760 * eulergamma * log_xom * xsg6 + 158760 * expint_ei_xom_xsg * log_xom * xsg6 + 158760 * eulergamma * exp_xom_xsgplus * log_xom * xsg6 - 158760 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg6 - 158760 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg6 + 158760 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg6 + 158760 * L * log_xsg * xsg6 - 158760 * log_xom * log_xsg * xsg6 + 336 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg6 + 2614752 * exp_xom_xsgplus * xom2 * xsg6 + 2038176 * eulergamma * exp_xom_xsgplus * xom2 * xsg6 - 2038176 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg6 - 158760 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg6 + 2038176 * exp_xom_xsgplus * L * xom2 * xsg6 + 635040 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg6 - 635040 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg6 + 317520 * exp_xom_xsgplus * L2 * xom2 * xsg6 - 317520 * exp_xom_xsgplus * log2_xom * xom2 * xsg6 - 635040 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg6 + 635040 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg6 - 5747616 * exp_xom_xsgplus * xom3 * xsg6 - 5991552 * eulergamma * exp_xom_xsgplus * xom3 * xsg6 + 5991552 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg6 - 635040 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg6 - 5991552 * exp_xom_xsgplus * L * xom3 * xsg6 - 2751840 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg6 + 2751840 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg6 - 1375920 * exp_xom_xsgplus * L2 * xom3 * xsg6 + 1375920 * exp_xom_xsgplus * log2_xom * xom3 * xsg6 + 2751840 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg6 - 2751840 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg6 + 4485600 * exp_xom_xsgplus * xom4 * xsg6 + 6519744 * eulergamma * exp_xom_xsgplus * xom4 * xsg6 - 6519744 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg6 + 2751840 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg6 + 6519744 * exp_xom_xsgplus * L * xom4 * xsg6 + 4021920 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg6 - 4021920 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg6 + 2010960 * exp_xom_xsgplus * L2 * xom4 * xsg6 - 2010960 * exp_xom_xsgplus * log2_xom * xom4 * xsg6 - 4021920 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg6 + 4021920 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg6 - 1524096 * exp_xom_xsgplus * xom5 * xsg6 - 3134880 * eulergamma * exp_xom_xsgplus * xom5 * xsg6 + 3134880 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg6 - 4021920 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg6 - 3134880 * exp_xom_xsgplus * L * xom5 * xsg6 - 2618784 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg6 + 2618784 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg6 - 1309392 * exp_xom_xsgplus * L2 * xom5 * xsg6 + 1309392 * exp_xom_xsgplus * log2_xom * xom5 * xsg6 + 2618784 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg6 - 2618784 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg6 + 225792 * exp_xom_xsgplus * xom6 * xsg6 + 719712 * eulergamma * exp_xom_xsgplus * xom6 * xsg6 - 719712 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg6 + 2618784 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg6 + 719712 * exp_xom_xsgplus * L * xom6 * xsg6 + 844704 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg6 - 844704 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg6 + 422352 * exp_xom_xsgplus * L2 * xom6 * xsg6 - 422352 * exp_xom_xsgplus * log2_xom * xom6 * xsg6 - 844704 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg6 + 844704 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg6 - 12096 * exp_xom_xsgplus * xom7 * xsg6 - 76608 * eulergamma * exp_xom_xsgplus * xom7 * xsg6 + 76608 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg6 - 844704 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg6 - 76608 * exp_xom_xsgplus * L * xom7 * xsg6 - 139104 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg6 + 139104 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg6 - 69552 * exp_xom_xsgplus * L2 * xom7 * xsg6 + 69552 * exp_xom_xsgplus * log2_xom * xom7 * xsg6 + 139104 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg6 - 139104 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg6 + 3024 * eulergamma * exp_xom_xsgplus * xom8 * xsg6 - 3024 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg6 + 139104 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg6 + 3024 * exp_xom_xsgplus * L * xom8 * xsg6 + 11088 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg6 - 11088 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg6 + 5544 * exp_xom_xsgplus * L2 * xom8 * xsg6 - 5544 * exp_xom_xsgplus * log2_xom * xom8 * xsg6 - 11088 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg6 + 11088 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg6 - 11088 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg6 - 336 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg6 + 336 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg6 - 168 * exp_xom_xsgplus * L2 * xom9 * xsg6 + 168 * exp_xom_xsgplus * log2_xom * xom9 * xsg6 + 336 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg6 - 336 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg6 - 15840 * exp_xom_xsgplus * xsg7 - 187200 * xom * exp_xom_xsgplus * xsg7 + 84024 * eulergamma * exp_xom_xsgplus * xsg7 - 108000 * xom * eulergamma * exp_xom_xsgplus * xsg7 - 84024 * expint_ei_plus_xom * exp_xom_xsgplus * xsg7 + 108000 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg7 - 45360 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg7 + 158760 * xom * hyp_xom_xsg * xsg7 - 45360 * eulergamma * L * xsg7 + 45360 * expint_ei_xom_xsg * L * xsg7 + 84024 * exp_xom_xsgplus * L * xsg7 - 108000 * xom * exp_xom_xsgplus * L * xsg7 + 45360 * eulergamma * exp_xom_xsgplus * L * xsg7 - 113400 * xom * eulergamma * exp_xom_xsgplus * L * xsg7 - 45360 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg7 + 113400 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg7 - 22680 * L2 * xsg7 + 22680 * exp_xom_xsgplus * L2 * xsg7 - 56700 * xom * exp_xom_xsgplus * L2 * xsg7 + 22680 * log2_xom * xsg7 - 22680 * exp_xom_xsgplus * log2_xom * xsg7 + 56700 * xom * exp_xom_xsgplus * log2_xom * xsg7 + 45360 * eulergamma * log_xom * xsg7 - 45360 * expint_ei_xom_xsg * log_xom * xsg7 - 45360 * eulergamma * exp_xom_xsgplus * log_xom * xsg7 + 113400 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg7 + 45360 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg7 - 113400 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg7 - 45360 * L * log_xsg * xsg7 + 45360 * log_xom * log_xsg * xsg7 + 144 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg7 + 1443456 * exp_xom_xsgplus * xom2 * xsg7 + 1065312 * eulergamma * exp_xom_xsgplus * xom2 * xsg7 - 1065312 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg7 + 113400 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg7 + 1065312 * exp_xom_xsgplus * L * xom2 * xsg7 + 362880 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg7 - 362880 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg7 + 181440 * exp_xom_xsgplus * L2 * xom2 * xsg7 - 181440 * exp_xom_xsgplus * log2_xom * xom2 * xsg7 - 362880 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg7 + 362880 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg7 - 3042144 * exp_xom_xsgplus * xom3 * xsg7 - 3233664 * eulergamma * exp_xom_xsgplus * xom3 * xsg7 + 3233664 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg7 - 362880 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg7 - 3233664 * exp_xom_xsgplus * L * xom3 * xsg7 - 1481760 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg7 + 1481760 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg7 - 740880 * exp_xom_xsgplus * L2 * xom3 * xsg7 + 740880 * exp_xom_xsgplus * log2_xom * xom3 * xsg7 + 1481760 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg7 - 1481760 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg7 + 2227680 * exp_xom_xsgplus * xom4 * xsg7 + 3348576 * eulergamma * exp_xom_xsgplus * xom4 * xsg7 - 3348576 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg7 + 1481760 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg7 + 3348576 * exp_xom_xsgplus * L * xom4 * xsg7 + 2116800 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg7 - 2116800 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg7 + 1058400 * exp_xom_xsgplus * L2 * xom4 * xsg7 - 1058400 * exp_xom_xsgplus * log2_xom * xom4 * xsg7 - 2116800 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg7 + 2116800 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg7 - 713664 * exp_xom_xsgplus * xom5 * xsg7 - 1522080 * eulergamma * exp_xom_xsgplus * xom5 * xsg7 + 1522080 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg7 - 2116800 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg7 - 1522080 * exp_xom_xsgplus * L * xom5 * xsg7 - 1312416 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg7 + 1312416 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg7 - 656208 * exp_xom_xsgplus * L2 * xom5 * xsg7 + 656208 * exp_xom_xsgplus * log2_xom * xom5 * xsg7 + 1312416 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg7 - 1312416 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg7 + 100800 * exp_xom_xsgplus * xom6 * xsg7 + 332640 * eulergamma * exp_xom_xsgplus * xom6 * xsg7 - 332640 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg7 + 1312416 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg7 + 332640 * exp_xom_xsgplus * L * xom6 * xsg7 + 403200 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg7 - 403200 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg7 + 201600 * exp_xom_xsgplus * L2 * xom6 * xsg7 - 201600 * exp_xom_xsgplus * log2_xom * xom6 * xsg7 - 403200 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg7 + 403200 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg7 - 5184 * exp_xom_xsgplus * xom7 * xsg7 - 33984 * eulergamma * exp_xom_xsgplus * xom7 * xsg7 + 33984 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg7 - 403200 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg7 - 33984 * exp_xom_xsgplus * L * xom7 * xsg7 - 63648 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg7 + 63648 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg7 - 31824 * exp_xom_xsgplus * L2 * xom7 * xsg7 + 31824 * exp_xom_xsgplus * log2_xom * xom7 * xsg7 + 63648 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg7 - 63648 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg7 + 1296 * eulergamma * exp_xom_xsgplus * xom8 * xsg7 - 1296 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg7 + 63648 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg7 + 1296 * exp_xom_xsgplus * L * xom8 * xsg7 + 4896 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg7 - 4896 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg7 + 2448 * exp_xom_xsgplus * L2 * xom8 * xsg7 - 2448 * exp_xom_xsgplus * log2_xom * xom8 * xsg7 - 4896 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg7 + 4896 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg7 - 4896 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg7 - 144 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg7 + 144 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg7 - 72 * exp_xom_xsgplus * L2 * xom9 * xsg7 + 72 * exp_xom_xsgplus * log2_xom * xom9 * xsg7 + 144 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg7 - 144 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg7 - 29448 * exp_xom_xsgplus * xsg8 - 45288 * xom * exp_xom_xsgplus * xsg8 - 13338 * eulergamma * exp_xom_xsgplus * xsg8 + 42336 * xom * eulergamma * exp_xom_xsgplus * xsg8 + 13338 * expint_ei_plus_xom * exp_xom_xsgplus * xsg8 - 42336 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg8 + 5670 * xom * exp_xom_xsgplus * hyp_plus_xom * xsg8 - 45360 * xom * hyp_xom_xsg * xsg8 + 5670 * eulergamma * L * xsg8 - 5670 * expint_ei_xom_xsg * L * xsg8 - 13338 * exp_xom_xsgplus * L * xsg8 + 42336 * xom * exp_xom_xsgplus * L * xsg8 - 5670 * eulergamma * exp_xom_xsgplus * L * xsg8 + 39690 * xom * eulergamma * exp_xom_xsgplus * L * xsg8 + 5670 * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg8 - 39690 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg8 + 2835 * L2 * xsg8 - 2835 * exp_xom_xsgplus * L2 * xsg8 + 19845 * xom * exp_xom_xsgplus * L2 * xsg8 - 2835 * log2_xom * xsg8 + 2835 * exp_xom_xsgplus * log2_xom * xsg8 - 19845 * xom * exp_xom_xsgplus * log2_xom * xsg8 - 5670 * eulergamma * log_xom * xsg8 + 5670 * expint_ei_xom_xsg * log_xom * xsg8 + 5670 * eulergamma * exp_xom_xsgplus * log_xom * xsg8 - 39690 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg8 - 5670 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg8 + 39690 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg8 + 5670 * L * log_xsg * xsg8 - 5670 * log_xom * log_xsg * xsg8 + 36 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg8 + 546336 * exp_xom_xsgplus * xom2 * xsg8 + 394848 * eulergamma * exp_xom_xsgplus * xom2 * xsg8 - 394848 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg8 - 39690 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg8 + 394848 * exp_xom_xsgplus * L * xom2 * xsg8 + 90720 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg8 - 90720 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg8 + 45360 * exp_xom_xsgplus * L2 * xom2 * xsg8 - 45360 * exp_xom_xsgplus * log2_xom * xom2 * xsg8 - 90720 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg8 + 90720 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg8 - 984816 * exp_xom_xsgplus * xom3 * xsg8 - 1113840 * eulergamma * exp_xom_xsgplus * xom3 * xsg8 + 1113840 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg8 - 90720 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg8 - 1113840 * exp_xom_xsgplus * L * xom3 * xsg8 - 529200 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg8 + 529200 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg8 - 264600 * exp_xom_xsgplus * L2 * xom3 * xsg8 + 264600 * exp_xom_xsgplus * log2_xom * xom3 * xsg8 + 529200 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg8 - 529200 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg8 + 652680 * exp_xom_xsgplus * xom4 * xsg8 + 1033704 * eulergamma * exp_xom_xsgplus * xom4 * xsg8 - 1033704 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg8 + 529200 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg8 + 1033704 * exp_xom_xsgplus * L * xom4 * xsg8 + 687960 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg8 - 687960 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg8 + 343980 * exp_xom_xsgplus * L2 * xom4 * xsg8 - 343980 * exp_xom_xsgplus * log2_xom * xom4 * xsg8 - 687960 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg8 + 687960 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg8 - 195048 * exp_xom_xsgplus * xom5 * xsg8 - 434448 * eulergamma * exp_xom_xsgplus * xom5 * xsg8 + 434448 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg8 - 687960 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg8 - 434448 * exp_xom_xsgplus * L * xom5 * xsg8 - 391608 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg8 + 391608 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg8 - 195804 * exp_xom_xsgplus * L2 * xom5 * xsg8 + 195804 * exp_xom_xsgplus * log2_xom * xom5 * xsg8 + 391608 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg8 - 391608 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg8 + 26208 * exp_xom_xsgplus * xom6 * xsg8 + 89712 * eulergamma * exp_xom_xsgplus * xom6 * xsg8 - 89712 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg8 + 391608 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg8 + 89712 * exp_xom_xsgplus * L * xom6 * xsg8 + 112896 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg8 - 112896 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg8 + 56448 * exp_xom_xsgplus * L2 * xom6 * xsg8 - 56448 * exp_xom_xsgplus * log2_xom * xom6 * xsg8 - 112896 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg8 + 112896 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg8 - 1296 * exp_xom_xsgplus * xom7 * xsg8 - 8784 * eulergamma * exp_xom_xsgplus * xom7 * xsg8 + 8784 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg8 - 112896 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg8 - 8784 * exp_xom_xsgplus * L * xom7 * xsg8 - 16992 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg8 + 16992 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg8 - 8496 * exp_xom_xsgplus * L2 * xom7 * xsg8 + 8496 * exp_xom_xsgplus * log2_xom * xom7 * xsg8 + 16992 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg8 - 16992 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg8 + 324 * eulergamma * exp_xom_xsgplus * xom8 * xsg8 - 324 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg8 + 16992 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg8 + 324 * exp_xom_xsgplus * L * xom8 * xsg8 + 1260 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg8 - 1260 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg8 + 630 * exp_xom_xsgplus * L2 * xom8 * xsg8 - 630 * exp_xom_xsgplus * log2_xom * xom8 * xsg8 - 1260 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg8 + 1260 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg8 - 1260 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg8 - 36 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg8 + 36 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg8 - 18 * exp_xom_xsgplus * L2 * xom9 * xsg8 + 18 * exp_xom_xsgplus * log2_xom * xom9 * xsg8 + 36 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg8 - 36 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg8 - 29448 * xom * exp_xom_xsgplus * xsg9 + 630 * eulergamma * exp_xom_xsgplus * xsg9 - 19008 * xom * eulergamma * exp_xom_xsgplus * xsg9 - 630 * expint_ei_plus_xom * exp_xom_xsgplus * xsg9 + 19008 * xom * expint_ei_plus_xom * exp_xom_xsgplus * xsg9 + 5670 * xom * hyp_xom_xsg * xsg9 + 630 * exp_xom_xsgplus * L * xsg9 - 19008 * xom * exp_xom_xsgplus * L * xsg9 - 5670 * xom * eulergamma * exp_xom_xsgplus * L * xsg9 + 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * L * xsg9 - 2835 * xom * exp_xom_xsgplus * L2 * xsg9 + 2835 * xom * exp_xom_xsgplus * log2_xom * xsg9 + 5670 * xom * eulergamma * exp_xom_xsgplus * log_xom * xsg9 - 5670 * xom * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xsg9 + 4 * exp_xom_xsgplus * hyp_plus_xom * xom10 * xsg9 + 109872 * exp_xom_xsgplus * xom2 * xsg9 + 106704 * eulergamma * exp_xom_xsgplus * xom2 * xsg9 - 106704 * expint_ei_plus_xom * exp_xom_xsgplus * xom2 * xsg9 + 5670 * exp_xom_xsgplus * hyp_plus_xom * xom2 * xsg9 + 106704 * exp_xom_xsgplus * L * xom2 * xsg9 + 45360 * eulergamma * exp_xom_xsgplus * L * xom2 * xsg9 - 45360 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom2 * xsg9 + 22680 * exp_xom_xsgplus * L2 * xom2 * xsg9 - 22680 * exp_xom_xsgplus * log2_xom * xom2 * xsg9 - 45360 * eulergamma * exp_xom_xsgplus * log_xom * xom2 * xsg9 + 45360 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom2 * xsg9 - 147504 * exp_xom_xsgplus * xom3 * xsg9 - 188496 * eulergamma * exp_xom_xsgplus * xom3 * xsg9 + 188496 * expint_ei_plus_xom * exp_xom_xsgplus * xom3 * xsg9 - 45360 * exp_xom_xsgplus * hyp_plus_xom * xom3 * xsg9 - 188496 * exp_xom_xsgplus * L * xom3 * xsg9 - 105840 * eulergamma * exp_xom_xsgplus * L * xom3 * xsg9 + 105840 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom3 * xsg9 - 52920 * exp_xom_xsgplus * L2 * xom3 * xsg9 + 52920 * exp_xom_xsgplus * log2_xom * xom3 * xsg9 + 105840 * eulergamma * exp_xom_xsgplus * log_xom * xom3 * xsg9 - 105840 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom3 * xsg9 + 85680 * exp_xom_xsgplus * xom4 * xsg9 + 145656 * eulergamma * exp_xom_xsgplus * xom4 * xsg9 - 145656 * expint_ei_plus_xom * exp_xom_xsgplus * xom4 * xsg9 + 105840 * exp_xom_xsgplus * hyp_plus_xom * xom4 * xsg9 + 145656 * exp_xom_xsgplus * L * xom4 * xsg9 + 105840 * eulergamma * exp_xom_xsgplus * L * xom4 * xsg9 - 105840 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom4 * xsg9 + 52920 * exp_xom_xsgplus * L2 * xom4 * xsg9 - 52920 * exp_xom_xsgplus * log2_xom * xom4 * xsg9 - 105840 * eulergamma * exp_xom_xsgplus * log_xom * xom4 * xsg9 + 105840 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom4 * xsg9 - 23688 * exp_xom_xsgplus * xom5 * xsg9 - 55440 * eulergamma * exp_xom_xsgplus * xom5 * xsg9 + 55440 * expint_ei_plus_xom * exp_xom_xsgplus * xom5 * xsg9 - 105840 * exp_xom_xsgplus * hyp_plus_xom * xom5 * xsg9 - 55440 * exp_xom_xsgplus * L * xom5 * xsg9 - 52920 * eulergamma * exp_xom_xsgplus * L * xom5 * xsg9 + 52920 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom5 * xsg9 - 26460 * exp_xom_xsgplus * L2 * xom5 * xsg9 + 26460 * exp_xom_xsgplus * log2_xom * xom5 * xsg9 + 52920 * eulergamma * exp_xom_xsgplus * log_xom * xom5 * xsg9 - 52920 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom5 * xsg9 + 3024 * exp_xom_xsgplus * xom6 * xsg9 + 10752 * eulergamma * exp_xom_xsgplus * xom6 * xsg9 - 10752 * expint_ei_plus_xom * exp_xom_xsgplus * xom6 * xsg9 + 52920 * exp_xom_xsgplus * hyp_plus_xom * xom6 * xsg9 + 10752 * exp_xom_xsgplus * L * xom6 * xsg9 + 14112 * eulergamma * exp_xom_xsgplus * L * xom6 * xsg9 - 14112 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom6 * xsg9 + 7056 * exp_xom_xsgplus * L2 * xom6 * xsg9 - 7056 * exp_xom_xsgplus * log2_xom * xom6 * xsg9 - 14112 * eulergamma * exp_xom_xsgplus * log_xom * xom6 * xsg9 + 14112 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom6 * xsg9 - 144 * exp_xom_xsgplus * xom7 * xsg9 - 1008 * eulergamma * exp_xom_xsgplus * xom7 * xsg9 + 1008 * expint_ei_plus_xom * exp_xom_xsgplus * xom7 * xsg9 - 14112 * exp_xom_xsgplus * hyp_plus_xom * xom7 * xsg9 - 1008 * exp_xom_xsgplus * L * xom7 * xsg9 - 2016 * eulergamma * exp_xom_xsgplus * L * xom7 * xsg9 + 2016 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom7 * xsg9 - 1008 * exp_xom_xsgplus * L2 * xom7 * xsg9 + 1008 * exp_xom_xsgplus * log2_xom * xom7 * xsg9 + 2016 * eulergamma * exp_xom_xsgplus * log_xom * xom7 * xsg9 - 2016 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom7 * xsg9 + 36 * eulergamma * exp_xom_xsgplus * xom8 * xsg9 - 36 * expint_ei_plus_xom * exp_xom_xsgplus * xom8 * xsg9 + 2016 * exp_xom_xsgplus * hyp_plus_xom * xom8 * xsg9 + 36 * exp_xom_xsgplus * L * xom8 * xsg9 + 144 * eulergamma * exp_xom_xsgplus * L * xom8 * xsg9 - 144 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom8 * xsg9 + 72 * exp_xom_xsgplus * L2 * xom8 * xsg9 - 72 * exp_xom_xsgplus * log2_xom * xom8 * xsg9 - 144 * eulergamma * exp_xom_xsgplus * log_xom * xom8 * xsg9 + 144 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom8 * xsg9 - 144 * exp_xom_xsgplus * hyp_plus_xom * xom9 * xsg9 - 4 * eulergamma * exp_xom_xsgplus * L * xom9 * xsg9 + 4 * expint_ei_plus_xom * exp_xom_xsgplus * L * xom9 * xsg9 - 2 * exp_xom_xsgplus * L2 * xom9 * xsg9 + 2 * exp_xom_xsgplus * log2_xom * xom9 * xsg9 + 4 * eulergamma * exp_xom_xsgplus * log_xom * xom9 * xsg9 - 4 * expint_ei_plus_xom * exp_xom_xsgplus * log_xom * xom9 * xsg9) * xsgplus_m10) / 2835.
        };
        c = c + c_c_star;

        auto [a_begin, a_end] = blcdas->coefficient_range(mu);

        double result = 0.0;

        // contribution from terms a*, b, c*
        result += std::inner_product(a_begin, a_end, c.begin(), 0.0);

        // The (d*) contribution has to be calculated numerically
        // Workaround for Clang: cannot presently capture structured bindings; hence pass iterators explicitly
        auto c_d_star_integrand = [&](const double & x, const auto & a_begin, const auto & a_end) -> double {
            const double x_plus_xsg_m1 = 1.0 / (x + xsg);
            const double x_plus_xsg_m2  = x_plus_xsg_m1  * x_plus_xsg_m1;
            const double x_plus_xsg_m3  = x_plus_xsg_m2  * x_plus_xsg_m1;
            const double x_plus_xsg_m4  = x_plus_xsg_m3  * x_plus_xsg_m1;
            const double x_plus_xsg_m5  = x_plus_xsg_m4  * x_plus_xsg_m1;
            const double x_plus_xsg_m6  = x_plus_xsg_m5  * x_plus_xsg_m1;
            const double x_plus_xsg_m7  = x_plus_xsg_m6  * x_plus_xsg_m1;
            const double x_plus_xsg_m8  = x_plus_xsg_m7  * x_plus_xsg_m1;
            const double x_plus_xsg_m9  = x_plus_xsg_m8  * x_plus_xsg_m1;
            const double x_plus_xsg_m10 = x_plus_xsg_m9  * x_plus_xsg_m1;
            const double x_plus_xsg_m11 = x_plus_xsg_m10 * x_plus_xsg_m1;

            const double x2  = x   * x;
            const double x3  = x2  * x;
            const double x4  = x3  * x;
            const double x5  = x4  * x;
            const double x6  = x5  * x;
            const double x7  = x6  * x;
            const double x8  = x7  * x;
            const double x9  = x8  * x;
            const double x10 = x9  * x;
            const double x11 = x10 * x;
            const double x12 = x11 * x;
            const double x13 = x12 * x;
            const double x14 = x13 * x;
            const double x15 = x14 * x;
            const double x16 = x15 * x;
            const double x17 = x16 * x;
            const double x18 = x17 * x;

            const double log_x = std::log(1.0 - x);
            const double exp_x  = std::exp(-xom * (x + xsg));

            const Weights kernel = {
                -2 * x_plus_xsg_m3 + exp_x * x_plus_xsg_m3 * (2 + 2 * x * xom + 2 * xom * xsg + 2 * x * xsg * xom2 + x2 * xom2 + xom2 * xsg2),
                (2 * x - 4 * xsg) * x_plus_xsg_m4 + exp_x * x_plus_xsg_m4 * (-2 * x + 4 * xsg + 2 * x * xom * xsg - 2 * xom * x2 - x3 * xom2 - 3 * xsg * x3 * xom3 - x4 * xom3 + 4 * xom * xsg2 + 3 * x * xom2 * xsg2 - 3 * x2 * xom3 * xsg2 + 2 * xom2 * xsg3 - x * xom3 * xsg3),
                (x_plus_xsg_m5 * (24 * x * xsg - 6 * x2 - 18 * xsg2))/3. + (exp_x * x_plus_xsg_m5 * (-24 * x * xsg + 6 * x2 - 18 * xom * xsg * x2 + 6 * xom * x3 - 6 * xsg * x3 * xom2 + 3 * x4 * xom2 - 16 * xsg * x4 * xom3 - 2 * x5 * xom3 + 8 * xsg * x5 * xom4 + 2 * x6 * xom4 + 18 * xsg2 - 6 * x * xom * xsg2 - 12 * x2 * xom2 * xsg2 - 36 * x3 * xom3 * xsg2 + 12 * x4 * xom4 * xsg2 + 18 * xom * xsg3 + 6 * x * xom2 * xsg3 - 32 * x2 * xom3 * xsg3 + 8 * x3 * xom4 * xsg3 + 9 * xom2 * xsg4 - 10 * x * xom3 * xsg4 + 2 * x2 * xom4 * xsg4))/3.,
                (x_plus_xsg_m6 * (-36 * xsg * x2 + 6 * x3 + 54 * x * xsg2 - 24 * xsg3))/3. + (exp_x * x_plus_xsg_m6 * (36 * xsg * x2 - 6 * x3 + 30 * xom * xsg * x3 - 6 * xom * x4 + 12 * xsg * x4 * xom2 - 3 * x5 * xom2 - 21 * xsg * x5 * xom3 - 5 * x6 * xom3 + 25 * xsg * x6 * xom4 + 4 * x7 * xom4 - 5 * xsg * x7 * xom5 - x8 * xom5 - 54 * x * xsg2 - 18 * xom * x2 * xsg2 + 6 * x3 * xom2 * xsg2 - 54 * x4 * xom3 * xsg2 + 60 * x5 * xom4 * xsg2 - 10 * x6 * xom5 * xsg2 + 24 * xsg3 - 30 * x * xom * xsg3 - 24 * x2 * xom2 * xsg3 - 86 * x3 * xom3 * xsg3 + 70 * x4 * xom4 * xsg3 - 10 * x5 * xom5 * xsg3 + 24 * xom * xsg4 - 3 * x * xom2 * xsg4 - 69 * x2 * xom3 * xsg4 + 40 * x3 * xom4 * xsg4 - 5 * x4 * xom5 * xsg4 + 12 * xom2 * xsg5 - 21 * x * xom3 * xsg5 + 9 * x2 * xom4 * xsg5 - x3 * xom5 * xsg5))/3.,
                (x_plus_xsg_m7 * (240 * xsg * x3 - 30 * x4 - 540 * x2 * xsg2 + 480 * x * xsg3 - 150 * xsg4))/15. + (exp_x * x_plus_xsg_m7 * (-240 * xsg * x3 + 30 * x4 - 210 * xom * xsg * x4 + 30 * xom * x5 - 90 * xsg * x5 * xom2 + 15 * x6 * xom2 - 200 * xsg * x6 * xom3 - 20 * x7 * xom3 + 260 * xsg * x7 * xom4 + 40 * x8 * xom4 - 108 * xsg * x8 * xom5 - 16 * x9 * xom5 + 2 * x10 * xom6 + 12 * xsg * x9 * xom6 + 540 * x2 * xsg2 + 300 * xom * x3 * xsg2 + 45 * x4 * xom2 * xsg2 - 540 * x5 * xom3 * xsg2 + 760 * x6 * xom4 * xsg2 - 300 * x7 * xom5 * xsg2 + 30 * x8 * xom6 * xsg2 - 480 * x * xsg3 + 60 * xom * x2 * xsg3 + 180 * x3 * xom2 * xsg3 - 800 * x4 * xom3 * xsg3 + 1240 * x5 * xom4 * xsg3 - 440 * x6 * xom5 * xsg3 + 40 * x7 * xom6 * xsg3 + 150 * xsg4 - 330 * x * xom * xsg4 - 135 * x2 * xom2 * xsg4 - 860 * x3 * xom3 * xsg4 + 1160 * x4 * xom4 * xsg4 - 360 * x5 * xom5 * xsg4 + 30 * x6 * xom6 * xsg4 + 150 * xom * xsg5 - 90 * x * xom2 * xsg5 - 600 * x2 * xom3 * xsg5 + 580 * x3 * xom4 * xsg5 - 156 * x4 * xom5 * xsg5 + 12 * x5 * xom6 * xsg5 + 75 * xom2 * xsg6 - 180 * x * xom3 * xsg6 + 120 * x2 * xom4 * xsg6 - 28 * x3 * xom5 * xsg6 + 2 * x4 * xom6 * xsg6))/15.,
                (x_plus_xsg_m8 * (-900 * xsg * x4 + 90 * x5 + 2700 * x3 * xsg2 - 3600 * x2 * xsg3 + 2250 * x * xsg4 - 540 * xsg5))/45. + (exp_x * x_plus_xsg_m8 * (900 * xsg * x4 - 90 * x5 + 810 * xom * xsg * x5 - 90 * xom * x6 + 360 * xsg * x6 * xom2 - 45 * x7 * xom2 - 615 * xsg * x7 * xom3 - 105 * x8 * xom3 + 1470 * xsg * x8 * xom4 + 180 * x9 * xom4 - 114 * x10 * xom5 - 870 * xsg * x9 * xom5 + 196 * xsg * x10 * xom6 + 26 * x11 * xom6 - 14 * xsg * x11 * xom7 - 2 * x12 * xom7 - 2700 * x3 * xsg2 - 1800 * xom * x4 * xsg2 - 495 * x5 * xom2 * xsg2 - 2565 * x6 * xom3 * xsg2 + 4980 * x7 * xom4 * xsg2 - 2910 * x8 * xom5 * xsg2 + 630 * x9 * xom6 * xsg2 - 42 * x10 * xom7 * xsg2 + 3600 * x2 * xsg3 + 900 * xom * x3 * xsg3 - 450 * x4 * xom2 * xsg3 - 5355 * x5 * xom3 * xsg3 + 9570 * x6 * xom4 * xsg3 - 5490 * x7 * xom5 * xsg3 + 1120 * x8 * xom6 * xsg3 - 70 * x9 * xom7 * xsg3 - 2250 * x * xsg4 + 1350 * xom * x2 * xsg4 + 1125 * x3 * xom2 * xsg4 - 6075 * x4 * xom3 * xsg4 + 11580 * x5 * xom4 * xsg4 - 6270 * x6 * xom5 * xsg4 + 1190 * x7 * xom6 * xsg4 - 70 * x8 * xom7 * xsg4 + 540 * xsg5 - 1710 * x * xom * xsg5 - 180 * x2 * xom2 * xsg5 - 4725 * x3 * xom3 * xsg5 + 8850 * x4 * xom4 * xsg5 - 4314 * x5 * xom5 * xsg5 + 756 * x6 * xom6 * xsg5 - 42 * x7 * xom7 * xsg5 + 540 * xom * xsg6 - 585 * x * xom2 * xsg6 - 2775 * x2 * xom3 * xsg6 + 3900 * x3 * xom4 * xsg6 - 1650 * x4 * xom5 * xsg6 + 266 * x5 * xom6 * xsg6 - 14 * x6 * xom7 * xsg6 + 270 * xom2 * xsg7 - 825 * x * xom3 * xsg7 + 750 * x2 * xom4 * xsg7 - 270 * x3 * xom5 * xsg7 + 40 * x4 * xom6 * xsg7 - 2 * x5 * xom7 * xsg7))/45.,
                (x_plus_xsg_m9 * (7560 * xsg * x5 - 630 * x6 - 28350 * x4 * xsg2 + 50400 * x3 * xsg3 - 47250 * x2 * xsg4 + 22680 * x * xsg5 - 4410 * xsg6))/315. + (exp_x * x_plus_xsg_m9 * (-7560 * xsg * x5 + 630 * x6 - 6930 * xom * xsg * x6 + 630 * xom * x7 - 3150 * xsg * x7 * xom2 + 315 * x8 * xom2 - 7560 * xsg * x8 * xom3 - 630 * x9 * xom3 + 1890 * x10 * xom4 + 16380 * xsg * x9 * xom4 - 13356 * xsg * x10 * xom5 - 1512 * x11 * xom5 + 4452 * xsg * x11 * xom6 + 518 * x12 * xom6 - 640 * xsg * x12 * xom7 - 76 * x13 * xom7 + 32 * xsg * x13 * xom8 + 4 * x14 * xom8 + 28350 * x4 * xsg2 + 20790 * xom * x5 * xsg2 + 6930 * x6 * xom2 * xsg2 - 25200 * x7 * xom3 * xsg2 + 65520 * x8 * xom4 * xsg2 - 51660 * x9 * xom5 * xsg2 + 16884 * x10 * xom6 * xsg2 - 2352 * x11 * xom7 * xsg2 + 112 * x12 * xom8 * xsg2 - 50400 * x3 * xsg3 - 22050 * xom * x4 * xsg3 - 630 * x5 * xom2 * xsg3 - 59640 * x6 * xom3 * xsg3 + 150780 * x7 * xom4 * xsg3 - 115500 * x8 * xom5 * xsg3 + 36820 * x9 * xom6 * xsg3 - 4928 * x10 * xom7 * xsg3 + 224 * x11 * xom8 * xsg3 + 47250 * x2 * xsg4 - 3150 * xom * x3 * xsg4 - 12600 * x4 * xom2 * xsg4 - 97020 * x5 * xom3 * xsg4 + 218820 * x6 * xom4 * xsg4 - 164220 * x7 * xom5 * xsg4 + 50400 * x8 * xom6 * xsg4 - 6440 * x9 * xom7 * xsg4 + 280 * x10 * xom8 * xsg4 - 22680 * x * xsg5 + 24570 * xom * x2 * xsg5 + 10710 * x3 * xom2 * xsg5 - 93240 * x4 * xom3 * xsg5 + 210420 * x5 * xom4 * xsg5 - 152292 * x6 * xom5 * xsg5 + 44268 * x7 * xom6 * xsg5 - 5376 * x8 * xom7 * xsg5 + 224 * x9 * xom8 * xsg5 + 4410 * xsg6 - 18270 * x * xom * xsg6 + 3150 * x2 * xom2 * xsg6 - 57120 * x3 * xom3 * xsg6 + 134400 * x4 * xom4 * xsg6 - 89796 * x5 * xom5 * xsg6 + 24332 * x6 * xom6 * xsg6 - 2800 * x7 * xom7 * xsg6 + 112 * x8 * xom8 * xsg6 + 4410 * xom * xsg7 - 6930 * x * xom2 * xsg7 - 27720 * x2 * xom3 * xsg7 + 52500 * x3 * xom4 * xsg7 - 30660 * x4 * xom5 * xsg7 + 7644 * x5 * xom6 * xsg7 - 832 * x6 * xom7 * xsg7 + 32 * x7 * xom8 * xsg7 + 2205 * xom2 * xsg8 - 8190 * x * xom3 * xsg8 + 9450 * x2 * xom4 * xsg8 - 4620 * x3 * xom5 * xsg8 + 1050 * x4 * xom6 * xsg8 - 108 * x5 * xom7 * xsg8 + 4 * x6 * xom8 * xsg8))/315.,
                (x_plus_xsg_m10 * (-8820 * xsg * x6 + 630 * x7 + 39690 * x5 * xsg2 - 88200 * x4 * xsg3 + 110250 * x3 * xsg4 - 79380 * x2 * xsg5 + 30870 * x * xsg6 - 5040 * xsg7))/315. + (exp_x * x_plus_xsg_m10 * (8820 * xsg * x6 - 630 * x7 + 8190 * xom * xsg * x7 - 630 * xom * x8 + 3780 * xsg * x8 * xom2 - 315 * x9 * xom2 - 945 * x10 * xom3 - 7245 * xsg * x9 * xom3 + 25515 * xsg * x10 * xom4 + 2520 * x11 * xom4 - 25263 * xsg * x11 * xom5 - 2583 * x12 * xom5 + 11284 * xsg * x12 * xom6 + 1162 * x13 * xom6 - 2422 * xsg * x13 * xom7 - 254 * x14 * xom7 + 243 * xsg * x14 * xom8 + 26 * x15 * xom8 - 9 * xsg * x15 * xom9 - x16 * xom9 - 39690 * x5 * xsg2 - 30870 * xom * x6 * xsg2 - 11340 * x7 * xom2 * xsg2 - 40320 * x8 * xom3 * xsg2 + 112140 * x9 * xom4 * xsg2 - 111384 * x10 * xom5 * xsg2 + 48888 * x11 * xom6 * xsg2 - 10304 * x12 * xom7 * xsg2 + 1008 * x13 * xom8 * xsg2 - 36 * x14 * xom9 * xsg2 + 88200 * x4 * xsg3 + 48510 * xom * x5 * xsg3 + 8820 * x6 * xom2 * xsg3 - 101640 * x7 * xom3 * xsg3 + 295260 * x8 * xom4 * xsg3 - 288960 * x9 * xom5 * xsg3 + 124376 * x10 * xom6 * xsg3 - 25648 * x11 * xom7 * xsg3 + 2436 * x12 * xom8 * xsg3 - 84 * x13 * xom9 * xsg3 - 110250 * x3 * xsg4 - 22050 * xom * x4 * xsg4 + 13230 * x5 * xom2 * xsg4 - 169050 * x6 * xom3 * xsg4 + 511140 * x7 * xom4 * xsg4 - 486990 * x8 * xom5 * xsg4 + 205100 * x9 * xom6 * xsg4 - 41132 * x10 * xom7 * xsg4 + 3780 * x11 * xom8 * xsg4 - 126 * x12 * xom9 * xsg4 + 79380 * x2 * xsg5 - 30870 * xom * x3 * xsg5 - 26460 * x4 * xom2 * xsg5 - 216090 * x5 * xom3 * xsg5 + 598290 * x6 * xom4 * xsg5 - 556038 * x7 * xom5 * xsg5 + 227472 * x8 * xom6 * xsg5 - 44044 * x9 * xom7 * xsg5 + 3906 * x10 * xom8 * xsg5 - 126 * x11 * xom9 * xsg5 - 30870 * x * xsg6 + 48510 * xom * x2 * xsg6 + 8820 * x3 * xom2 * xsg6 - 182280 * x4 * xom3 * xsg6 + 479220 * x5 * xom4 * xsg6 - 432768 * x6 * xom5 * xsg6 + 169624 * x7 * xom6 * xsg6 - 31472 * x8 * xom7 * xsg6 + 2688 * x9 * xom8 * xsg6 - 84 * x10 * xom9 * xsg6 + 5040 * xsg7 - 25830 * x * xom * xsg7 + 11340 * x2 * xom2 * xsg7 - 94080 * x3 * xom3 * xsg7 + 261660 * x4 * xom4 * xsg7 - 222264 * x5 * xom5 * xsg7 + 81928 * x6 * xom6 * xsg7 - 14464 * x7 * xom7 * xsg7 + 1188 * x8 * xom8 * xsg7 - 36 * x9 * xom9 * xsg7 + 5040 * xom * xsg8 - 10395 * x * xom2 * xsg8 - 37485 * x2 * xom3 * xsg8 + 91140 * x3 * xom4 * xsg8 - 68355 * x4 * xom5 * xsg8 + 23226 * x5 * xom6 * xsg8 - 3878 * x6 * xom7 * xsg8 + 306 * x7 * xom8 * xsg8 - 9 * x8 * xom9 * xsg8 + 2520 * xom2 * xsg9 - 11025 * x * xom3 * xsg9 + 15435 * x2 * xom4 * xsg9 - 9555 * x3 * xom5 * xsg9 + 2940 * x4 * xom6 * xsg9 - 462 * x5 * xom7 * xsg9 + 35 * x6 * xom8 * xsg9 - x7 * xom9 * xsg9))/315.,
                (x_plus_xsg_m11 * (90720 * xsg * x7 - 5670 * x8 - 476280 * x6 * xsg2 + 1270080 * x5 * xsg3 - 1984500 * x4 * xsg4 + 1905120 * x3 * xsg5 - 1111320 * x2 * xsg6 + 362880 * x * xsg7 - 51030 * xsg8))/2835. + (exp_x * x_plus_xsg_m11 * (-90720 * xsg * x7 + 5670 * x8 - 85050 * xom * xsg * x8 + 5670 * xom * x9 + 20 * xsg * x17 * xom10 + 2 * x18 * xom10 + 2835 * x10 * xom2 - 39690 * xsg * x9 * xom2 - 105840 * xsg * x10 * xom3 - 7560 * x11 * xom3 + 325080 * xsg * x11 * xom4 + 30240 * x12 * xom4 - 394632 * xsg * x12 * xom5 - 36288 * x13 * xom5 + 219240 * xsg * x13 * xom6 + 20412 * x14 * xom6 - 62784 * xsg * x14 * xom7 - 5904 * x15 * xom7 + 9432 * xsg * x15 * xom8 + 900 * x16 * xom8 - 700 * xsg * x16 * xom9 - 68 * x17 * xom9 + 2 * x8 * xom10 * xsg10 + 25515 * xom2 * xsg10 - 128520 * x * xom3 * xsg10 + 211680 * x2 * xom4 * xsg10 - 158760 * x3 * xom5 * xsg10 + 61740 * x4 * xom6 * xsg10 - 13104 * x5 * xom7 * xsg10 + 1512 * x6 * xom8 * xsg10 - 88 * x7 * xom9 * xsg10 + 476280 * x6 * xsg2 + 385560 * xom * x7 * xsg2 + 90 * x16 * xom10 * xsg2 + 150255 * x8 * xom2 * xsg2 - 430920 * x9 * xom3 * xsg2 + 1632960 * x10 * xom4 * xsg2 - 1936872 * x11 * xom5 * xsg2 + 1066716 * x12 * xom6 * xsg2 - 301392 * x13 * xom7 * xsg2 + 44568 * x14 * xom8 * xsg2 - 3240 * x15 * xom9 * xsg2 - 1270080 * x5 * xsg3 - 793800 * xom * x6 * xsg3 + 240 * x15 * xom10 * xsg3 - 204120 * x7 * xom2 * xsg3 - 1421280 * x8 * xom3 * xsg3 + 4838400 * x9 * xom4 * xsg3 - 5691168 * x10 * xom5 * xsg3 + 3094560 * x11 * xom6 * xsg3 - 860832 * x12 * xom7 * xsg3 + 124992 * x13 * xom8 * xsg3 - 8880 * x14 * xom9 * xsg3 + 1984500 * x4 * xsg4 + 714420 * xom * x5 * xsg4 + 420 * x14 * xom10 * xsg4 - 39690 * x6 * xom2 * xsg4 - 2887920 * x7 * xom3 * xsg4 + 9525600 * x8 * xom4 * xsg4 - 11098080 * x9 * xom5 * xsg4 + 5933592 * x10 * xom6 * xsg4 - 1620864 * x11 * xom7 * xsg4 + 230328 * x12 * xom8 * xsg4 - 15960 * x13 * xom9 * xsg4 - 1905120 * x3 * xsg5 + 79380 * xom * x4 * xsg5 + 504 * x13 * xom10 * xsg5 + 396900 * x5 * xom2 * xsg5 - 3810240 * x6 * xom3 * xsg5 + 13169520 * x7 * xom4 * xsg5 - 15020208 * x8 * xom5 * xsg5 + 7871472 * x9 * xom6 * xsg5 - 2102688 * x10 * xom7 * xsg5 + 291312 * x11 * xom8 * xsg5 - 19656 * x12 * xom9 * xsg5 + 1111320 * x2 * xsg6 - 793800 * xom * x3 * xsg6 + 420 * x12 * xom10 * xsg6 - 357210 * x4 * xom2 * xsg6 - 3916080 * x5 * xom3 * xsg6 + 12912480 * x6 * xom4 * xsg6 - 14342832 * x7 * xom5 * xsg6 + 7330680 * x8 * xom6 * xsg6 - 1903104 * x9 * xom7 * xsg6 + 256032 * x10 * xom8 * xsg6 - 16800 * x11 * xom9 * xsg6 - 362880 * x * xsg7 + 748440 * xom * x2 * xsg7 + 240 * x11 * xom10 * xsg7 - 22680 * x3 * xom2 * xsg7 - 2933280 * x4 * xom3 * xsg7 + 8890560 * x5 * xom4 * xsg7 - 9610272 * x6 * xom5 * xsg7 + 4739616 * x7 * xom6 * xsg7 - 1186272 * x8 * xom7 * xsg7 + 154368 * x9 * xom8 * xsg7 - 9840 * x10 * xom9 * xsg7 + 51030 * xsg8 - 311850 * x * xom * xsg8 + 90 * x10 * xom10 * xsg8 + 218295 * x2 * xom2 * xsg8 - 1338120 * x3 * xom3 * xsg8 + 4233600 * x4 * xom4 * xsg8 - 4360608 * x5 * xom5 * xsg8 + 2037420 * x6 * xom6 * xsg8 - 487152 * x7 * xom7 * xsg8 + 61092 * x8 * xom8 * xsg8 - 3780 * x9 * xom9 * xsg8 + 51030 * xom * xsg9 + 20 * x9 * xom10 * xsg9 - 130410 * x * xom2 * xsg9 - 438480 * x2 * xom3 * xsg9 + 1323000 * x3 * xom4 * xsg9 - 1217160 * x4 * xom5 * xsg9 + 525672 * x5 * xom6 * xsg9 - 118944 * x6 * xom7 * xsg9 + 14328 * x7 * xom8 * xsg9 - 860 * x8 * xom9 * xsg9))/2835.,
            };
            return log_x * log_x * std::inner_product(a_begin, a_end, kernel.begin(), 0.0);
        };

        // contribution from term d*
        result += integrate<GSL::QAGS>(std::bind(c_d_star_integrand, std::placeholders::_1, a_begin, a_end), 0.0, 1.0);

        return result;
    }

    double
    AnalyticFormFactorBToGammaQCDF::F_V(const double & Egamma) const
    {
        return F_leading_power(Egamma) + xi(Egamma) + delta_xi(Egamma);
    }

    double
    AnalyticFormFactorBToGammaQCDF::F_A(const double & Egamma) const
    {
        return F_leading_power(Egamma) + xi(Egamma) - delta_xi(Egamma);
    }

    Diagnostics
    AnalyticFormFactorBToGammaQCDF::diagnostics() const
    {
        Diagnostics results;

        // Leading-order functionals
        results.add({ this->L0(),                                       "L0()"                                       });
        results.add({ this->L0_incomplete(8.0),                         "L0_incomplete(8.0)"                         });
        results.add({ this->norm_incomplete(8.0),                       "norm_incomplete(8.0)"                       });
        results.add({ this->lapltr_incomplete(8.0, 4.0),                "lapltr_incomplete(8.0, 4.0)"                });
        results.add({ this->lapltr_incomplete_dsigma(8.0, 4.0),         "lapltr_incomplete_dsigma(8.0, 4.0)"         });

        // Next-to-leading order functionals
        results.add({ this->L0_effective(3.0),                          "L0_effective(3.0)"                          });
        results.add({ this->L0_effective(2.16),                         "L0_effective(2.16)"                         });
        results.add({ this->L0_incomplete_effective(3.0, 8.0),          "L0_incomplete_effective(3.0, 8.0)"          });
        results.add({ this->lapltr_incomplete_effective(3.0, 8.0, 4.0), "lapltr_effective_incomplete(3.0, 8.0, 4.0)" });

        // Pieces of the form factors
        const auto [C, K_inv, U] = C_K_inv_U(2.16);
        results.add({ C,                                                "C at Egamma=2.16"                           });
        results.add({ K_inv,                                            "K_inv at Egamma=2.16"                       });
        results.add({ U,                                                "U at Egamma=2.16"                           });
        results.add({ this->F_leading_power(2.16),                      "F_leading_power(2.16)"                      });
        results.add({ this->xi(2.16),                                   "xi(2.16)"                                   });
        results.add({ this->delta_xi(2.16),                             "delta_xi(2.16)"                             });

        return results;
    }
}
