%
% Boundary tracker for reaction front passing a continuous
% precipitate, Mg-Al system at 180degC
%
% (c) Joseph Robson, University of Manchester, 2023
% Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
% Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
% Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation 
% and/or other materials provided with the distribution.
%
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS AS IS AND ANY EXPRESS OR IMPLIED WARRANTIES, 
% INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
% IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
% OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
% OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, 
% OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
%
% Note: This code is provided without support.
% THIS CODE IS NOT VALIDATED OR TESTED. IT IS NOT QUANTITATIVELY ACCURATE.
% ITS PURPOSE IS TO ILLUSTRATE BOUNDARY MOTION RATHER THAN PREDICT IT
% IT CERTAINLY CONTAINS ERRORS - USE AT YOUR OWN RISK.
%
% Complex version with pinning, curvature drag, solute diffusion
%
% Predefine X, Z (coordinates), C (composition at X, Z)
% Can be loaded from 2dcompositiononly.mat
%
% Turn off polyval warning
id =  'MATLAB:polyfit:RepeatedPointsOrRescale';
warning('off',id)
tk = 180+273; % Temperature
%xb = 0.077; % Al in alloy, in atomic fraction
xb = 0.08; % atom fraction Al in AZ91
xa = solvus_mg(tk);
xc = 12/29;
vmol = 1.4e-5;
na = 6.022045e23;
gas=8.314408694;
vat = vmol./na;
amg = 0.32092e-9;
gbe = 0.28; % Grain boundary energy J/m^2
%
% Assume boundary width ~ 3 x Mg atomic diameter
% (Christian, p. 371)
delta = 3/2 * amg;
q=125e3;
d0=4e-4;
dgb=d0.*exp(-q./(gas*tk)) .* 1000; % Scaling from Das paper for GB diffusion
%
pos = [X(:,1) Z(:,1)]; % initial grain boundary conditions
gs = max(X(:,1));
%
M = 1.7753e-16; % Effective boundary mobility
%M = 1.7753e-14; % Effective boundary mobility
t = 0;
tend = 8000;
dt = 16; % optimum version - results are sensitive to this choice
posstore = []; % Z positions
%
% Particle - will apply Zener drag in range
% 
pminz = Z(1,49);
pmaxz = Z(1,51);
pminx = 0;
pmaxx = 14;
while (t < tend)
    t = t + dt;
    xbinst_grid = griddata(X,Z,C,pos(:,1),pos(:,2)); %This is the background composition
    %
    xbinst_grid(find(isnan(xbinst_grid))) = xb;
    newxbinst = updatesol(xbinst_grid',gs,dgb,dt); % solute diffusion down reaction front
    %
    xbinst = newxbinst';
    %
    % If outside range for X,Z,C - set composition to far field
    %
    freerun = find(pos(:,2)>max(max(Z(1,:))));
    xbinst(freerun) = max(max(C));
    %
    % Overall chemical driving force (ideal solution model) for unit volume
    dgvol = ((gas * tk)/vmol)*((xc.*log(xbinst./xa))+((1-xc).*log((1-xbinst)./(1-xa))));
    dgvol(isnan(dgvol)) = 0;
    dgvol(find(dgvol<0)) = 0;
    %
    % Fudge factor to fit to CALPHAD calculation and scale for remnant
    % supersaturation
    %
    dgvol = 0.057 .* dgvol;
    %
    y = pos(:,2);
    %
    % Calculate curvatures using kappa script
    %
    [k, n, e] = kappa([X(:,1), y]); % k is 1/r - r is radius of curvature
    k(find(isnan(k))) = 0; %flat boundary, no drag
    n(find(isnan(n))) = 1; %flat boundary, no normal
    %
    % Forward or backward drag depends on normal n(:,2) (in Z direction)
    % n +ve = forward drag (pull)
    % n -ve = backward drag
    % 
    k = k .* sign(n(:,2));
    cdrag = 2 * gbe * k;
    %
    % Zener pinning
    % Zener pinning pressure = max Zener pinning force/area
    % = gbe * pi * Rpart / pi * Rpart.^2
    Partpos = discretize(pos(:,2),[pminz pmaxz])==1;
    cdrag(Partpos) = cdrag(Partpos) + ((gbe)./0.5e-6);
    % Velocity = M * DG
    v = M .* (dgvol - cdrag);
    posstore = [posstore; pos(:,2)'];
    pos_messy = pos(:,2) + (v.* dt); % Move up the Z axis
    pos(:,2) = smoothdata(pos_messy,"gaussian",10); %smoothing needed
    plot(X(:,1),pos(:,2))
    axis([0 4e-6 -3.9e-6 (7e-6)]) 
    title(['Time ' num2str(t)])
    drawnow;
end