# Copyright 2022 Lawrence Livermore National Security, LLC and other
# This is part of Flux Framework. See the COPYRIGHT file for details.
#
# SPDX-License-Identifier: Apache-2.0

import copy
import os
from contextlib import contextmanager


@contextmanager
def working_dir(path):
    """
    Sets the cwd within the context
    """
    here = os.getcwd()
    try:
        os.chdir(path)
        yield
    finally:
        os.chdir(here)


def chunks(listing, chunk_size):
    """
    Yield successive chunks from listing. Chunkify!
    """
    for i in range(0, len(listing), chunk_size):
        yield listing[i : i + chunk_size]


def slugify(name):
    """
    Slugify a name, replacing spaces with - and lowercase.
    """
    for char in [" ", ":", "/", "\\"]:
        name = name.replace(char, "-")
    return name.lower()


def print_bytes(byt, suffix="B"):
    """
    Pretty format size in bytes
    """
    for unit in ["", "Ki", "Mi", "Gi", "Ti", "Pi", "Ei", "Zi"]:
        if abs(byt) < 1024.0:
            return f"{byt:3.1f} {unit}{suffix}"
        byt /= 1024.0
    return f"{byt:.1f} Yi{suffix}"


def mb_to_bytes(mb):
    """
    Convert mb to bytes, usually so we can derive a better format.
    """
    return mb * (1048576)


def get_hash(obj):
    """
    Get a hash for a random object (set, tuple, list, dict)

    All nested attributes must at least be hashable!
    """
    if isinstance(obj, (set, tuple, list)):
        return tuple([get_hash(o) for o in obj])
    if not isinstance(obj, dict):
        return hash(obj)
    copied = copy.deepcopy(obj)
    for k, v in copied.items():
        copied[k] = get_hash(v)
    return hash(tuple(frozenset(sorted(copied.items()))))
