# this file defines van-Genuchten Mualem model and PDI model in C++ for FEniCS

cpp_code_theta_VGM = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class theta_VGM : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, tau;

  theta_VGM(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = theta_s;
    }
    else {
        double m = 1 - 1/n;
        double psi = value[0];
        double S_e = pow((1 + pow((-alpha*psi), n)), -m);
        double theta = S_e * (theta_s - theta_r) + theta_r;
        value[0] = theta;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<theta_VGM, std::shared_ptr<theta_VGM>, dolfin::Expression>
    (m, "theta_VGM")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &theta_VGM::u)
    .def_readwrite("theta_r", &theta_VGM::theta_r)
    .def_readwrite("theta_s", &theta_VGM::theta_s)
    .def_readwrite("alpha", &theta_VGM::alpha)
    .def_readwrite("n", &theta_VGM::n)
    .def_readwrite("K_s", &theta_VGM::K_s)
    .def_readwrite("tau", &theta_VGM::tau);
}

"""

cpp_code_dtheta_VGM = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class dtheta_VGM : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, tau;

  dtheta_VGM(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = 0.0;
    }
    else {
        double m = 1 - 1/n;
        double psi = value[0];
        double outer = m*pow((1 + pow((-alpha*psi), n)), (-m - 1));
        double inner =  n*alpha*pow((-alpha*psi), (n-1));
        double dtheta = (theta_s - theta_r)*outer*inner;
        value[0] = dtheta;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<dtheta_VGM, std::shared_ptr<dtheta_VGM>, dolfin::Expression>
    (m, "dtheta_VGM")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &dtheta_VGM::u)
    .def_readwrite("theta_r", &dtheta_VGM::theta_r)
    .def_readwrite("theta_s", &dtheta_VGM::theta_s)
    .def_readwrite("alpha", &dtheta_VGM::alpha)
    .def_readwrite("n", &dtheta_VGM::n)
    .def_readwrite("K_s", &dtheta_VGM::K_s)
    .def_readwrite("tau", &dtheta_VGM::tau);
}

"""

cpp_code_K_VGM = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class K_VGM : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, tau;

  K_VGM(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = K_s;
    }
    else {
        double m = 1 - 1/n;
        double psi = value[0];
        double S_e = pow((1 + pow((-alpha*psi), n)), -m);
        double inner = 1 - pow((1-pow(S_e, 1/m)), m);
        double K = K_s*pow(S_e, tau)*pow(inner, 2);
        value[0] = K;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<K_VGM, std::shared_ptr<K_VGM>, dolfin::Expression>
    (m, "K_VGM")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &K_VGM::u)
    .def_readwrite("theta_r", &K_VGM::theta_r)
    .def_readwrite("theta_s", &K_VGM::theta_s)
    .def_readwrite("alpha", &K_VGM::alpha)
    .def_readwrite("n", &K_VGM::n)
    .def_readwrite("K_s", &K_VGM::K_s)
    .def_readwrite("tau", &K_VGM::tau);
}

"""

cpp_code_dK_VGM = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class dK_VGM : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, tau;

  dK_VGM(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = 0.0;
    }
    else {
        double m = 1 - 1/n;
        double psi = value[0];

        // dK/dtheta
        double S_e_0 = pow((1 + pow((-alpha*psi), n)), -m);
        double S_e = std::min(S_e_0, 1 - 1.0E-12);
        double term1 = 1 - pow((1 - pow(S_e, 1/m)), m);
        double term2 = 2*pow(S_e, (tau - 1 + 1/m))*term1*pow((1 - pow(S_e, 1/m)), (m-1));
        double dK_dtheta = K_s/(theta_s - theta_r)*(tau*pow(S_e, (tau-1))*pow(term1, 2) + term2);

        // dtheta/dpsi
        double outer = m*pow((1 + pow((-alpha*psi), n)), (-m - 1));
        double inner =  n*alpha*pow((-alpha*psi), (n-1));
        double dtheta = (theta_s - theta_r)*outer*inner;

        value[0] = dK_dtheta * dtheta;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<dK_VGM, std::shared_ptr<dK_VGM>, dolfin::Expression>
    (m, "dK_VGM")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &dK_VGM::u)
    .def_readwrite("theta_r", &dK_VGM::theta_r)
    .def_readwrite("theta_s", &dK_VGM::theta_s)
    .def_readwrite("alpha", &dK_VGM::alpha)
    .def_readwrite("n", &dK_VGM::n)
    .def_readwrite("K_s", &dK_VGM::K_s)
    .def_readwrite("tau", &dK_VGM::tau);
}

"""

cpp_code_theta_PDI = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class theta_PDI : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, K_sf, slope, tau;

  theta_PDI(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = theta_s;
    }
    else if (value[0] <= -pow(10, 6.8)){
        value[0] = 0.0;
    }
    else {
        double psi_0 = -pow(10, 6.8);
        double m = 1 - 1/n;
        double psi = value[0];

        double gamma = pow((1 + pow((-alpha*psi), n)), -m);
        double gamma_0 = pow((1 + pow((-alpha*psi_0), n)), -m);

        double S_c = (gamma - gamma_0)/(1 - gamma_0);

        double psi_a = -1/alpha;
        double x = log10(-psi);
        double x_a = log10(-psi_a);
        double x_0 = log10(-psi_0);
        double b = 0.1 + 0.2/pow(n, 2) * (1 - exp(-pow((theta_r/(theta_s - theta_r)), 2)));
        double S_nc = 1 + 1/(x_a - x_0) * (x - x_a + b * log(1 + exp((x_a - x)/b)));

        double theta = (theta_s - theta_r) * S_c + theta_r * S_nc;

        value[0] = theta;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<theta_PDI, std::shared_ptr<theta_PDI>, dolfin::Expression>
    (m, "theta_PDI")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &theta_PDI::u)
    .def_readwrite("theta_r", &theta_PDI::theta_r)
    .def_readwrite("theta_s", &theta_PDI::theta_s)
    .def_readwrite("alpha", &theta_PDI::alpha)
    .def_readwrite("n", &theta_PDI::n)
    .def_readwrite("K_s", &theta_PDI::K_s)
    .def_readwrite("K_sf", &theta_PDI::K_sf)
    .def_readwrite("slope", &theta_PDI::slope)
    .def_readwrite("tau", &theta_PDI::tau);
}

"""


cpp_code_dtheta_PDI = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class dtheta_PDI : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, K_sf, slope, tau;

  dtheta_PDI(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = 0.0;
    }
    else {
        double psi_0 = -pow(10, 6.8);
        double m = 1 - 1/n;
        double psi = value[0];

        double gamma = pow((1 + pow((-alpha*psi), n)), -m);
        double gamma_0 = pow((1 + pow((-alpha*psi_0), n)), -m);

        double A = pow((-alpha * psi), n) + 1;
        double numerator = (1 -n)*pow((-alpha * psi), n) * pow(A, (1/n - 2));
        double denominator = psi;
        double C_VGM = numerator/denominator;

        double psi_a = -1/alpha;
        double x = log10(-psi);
        double x_a = log10(-psi_a);
        double x_0 = log10(-psi_0);
        double b = 0.1 + 0.2/pow(n, 2) * (1 - exp(-pow((theta_r/(theta_s - theta_r)), 2)));
        double dx = 1/(psi * log(10));
        double B = exp((x_a - x)/b);
        double D = -B/(b * (1 + B));
        double C_film = 1/(x_a - x_0) * (dx + b * dx * D);
        double dtheta = (theta_s - theta_r)*C_VGM/(1 - gamma_0) + theta_r * C_film;

        value[0] = dtheta;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<dtheta_PDI, std::shared_ptr<dtheta_PDI>, dolfin::Expression>
    (m, "dtheta_PDI")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &dtheta_PDI::u)
    .def_readwrite("theta_r", &dtheta_PDI::theta_r)
    .def_readwrite("theta_s", &dtheta_PDI::theta_s)
    .def_readwrite("alpha", &dtheta_PDI::alpha)
    .def_readwrite("n", &dtheta_PDI::n)
    .def_readwrite("K_s", &dtheta_PDI::K_s)
    .def_readwrite("K_sf", &dtheta_PDI::K_sf)
    .def_readwrite("slope", &dtheta_PDI::slope)
    .def_readwrite("tau", &dtheta_PDI::tau);
}

"""

cpp_code_K_PDI = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class K_PDI : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, K_sf, slope, tau;

  K_PDI(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = K_s + K_sf;
    }
    else if (value[0] <= -pow(10, 6.8)){
        value[0] = 0.0;
    }
    else {
        double psi_0 = -pow(10, 6.8);
        double m = 1 - 1/n;
        double psi = value[0];

        double gamma = pow((1 + pow((-alpha*psi), n)), -m);
        double gamma_0 = pow((1 + pow((-alpha*psi_0), n)), -m);

        double S_c = (gamma - gamma_0)/(1 - gamma_0);

        double psi_a = -1/alpha;
        double x = log10(-psi);
        double x_a = log10(-psi_a);
        double x_0 = log10(-psi_0);
        double b = 0.1 + 0.2/pow(n, 2) * (1 - exp(-pow((theta_r/(theta_s - theta_r)), 2)));
        double S_nc = 1 + 1/(x_a - x_0) * (x - x_a + b * log(1 + exp((x_a - x)/b)));

        double theta = (theta_s - theta_r) * S_c + theta_r * S_nc;

        double K_rc = pow(S_c, tau) * pow((1 - pow(((1 - pow(gamma, (1/m)))/(1 - pow(gamma_0, (1/m)))), m)), 2);
        double K_rf = pow((psi_0/psi_a), (slope * (1 - S_nc)));
        double K = K_s*K_rc + K_sf*K_rf;


        value[0] = K;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<K_PDI, std::shared_ptr<K_PDI>, dolfin::Expression>
    (m, "K_PDI")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &K_PDI::u)
    .def_readwrite("theta_r", &K_PDI::theta_r)
    .def_readwrite("theta_s", &K_PDI::theta_s)
    .def_readwrite("alpha", &K_PDI::alpha)
    .def_readwrite("n", &K_PDI::n)
    .def_readwrite("K_s", &K_PDI::K_s)
    .def_readwrite("K_sf", &K_PDI::K_sf)
    .def_readwrite("slope", &K_PDI::slope)
    .def_readwrite("tau", &K_PDI::tau);
}

"""

cpp_code_dK_PDI = """

#include <pybind11/pybind11.h>
#include <pybind11/eigen.h>
namespace py = pybind11;

#include <dolfin/function/Expression.h>
#include <dolfin/function/Function.h>

class dK_PDI : public dolfin::Expression
{
public:

  std::shared_ptr<dolfin::Function> u;
  double theta_r, theta_s, alpha, n, K_s, K_sf, slope, tau;

  dK_PDI(std::shared_ptr<dolfin::Function> u_) : dolfin::Expression(){
     u = u_;
    }

  void eval(Eigen::Ref<Eigen::VectorXd> value, Eigen::Ref<const Eigen::VectorXd> x) const override
  {
    u->eval(value, x);
    if (value[0] >= 0.0){
        value[0] = 0.0;
    }
    else if (value[0] <= -pow(10, 6.8)){
        value[0] = 0.0;
    }
    else {
        double psi_0 = -pow(10, 6.8);
        double m = 1 - 1/n;
        double psi = value[0];

        double gamma_prev = pow((1 + pow((-alpha*psi), n)), -m);
        double gamma_0 = pow((1 + pow((-alpha*psi_0), n)), -m);
        double gamma = std::min(gamma_prev, 1 - 1.0E-12);

        double S_c_prev = (gamma - gamma_0)/(1 - gamma_0);
        double S_c = std::max(S_c_prev, 1.0E-30);

        double A = pow((-alpha * psi), n) + 1;
        double numerator = (1 -n)*pow((-alpha * psi), n) * pow(A, (1/n - 2));
        double denominator = psi;
        double C_VGM = numerator/denominator;

        double psi_a = -1/alpha;
        double x = log10(-psi);
        double x_a = log10(-psi_a);
        double x_0 = log10(-psi_0);
        double b = 0.1 + 0.2/pow(n, 2) * (1 - exp(-pow((theta_r/(theta_s - theta_r)), 2)));
        double S_nc = 1 + 1/(x_a - x_0) * (x - x_a + b * log(1 + exp((x_a - x)/b)));
        double dx = 1/(psi * log(10));
        double B = exp((x_a - x)/b);
        double D = -B/(b * (1 + B));
        double C_film = 1/(x_a - x_0) * (dx + b * dx * D);
        double dtheta = (theta_s - theta_r)*C_VGM/(1 - gamma_0) + theta_r * C_film;

        double E = 1 - pow(((1 - pow(gamma, (1/m)))/(1 - pow(gamma_0, (1/m)))), m);
        double F = 1 - pow(gamma_0, (1/m));
        double G = 1 - pow(gamma, (1/m));
        double term1 = tau*pow(S_c, (tau - 1))*C_VGM/(1 - gamma_0)*pow(E, 2);
        double term2 = pow(S_c, tau)*2*E*(pow(G, (m-1))/pow(F, m))*pow(gamma, (1/m - 1))*C_VGM;
        double dK_c = K_s*(term1 + term2);

        double H = slope * (1 - S_nc);
        double K_rf = pow((psi_0/psi_a), H);

        double dK_f = K_sf*pow((psi_0/psi_a), H)*log(psi_0/psi_a)*(-slope)*C_film;

        double dK = dK_c + dK_f;


        value[0] = dK;
    }
  }
};

PYBIND11_MODULE(SIGNATURE, m)
{
  py::class_<dK_PDI, std::shared_ptr<dK_PDI>, dolfin::Expression>
    (m, "dK_PDI")
    .def(py::init<std::shared_ptr<dolfin::Function>>())
    .def_readwrite("u", &dK_PDI::u)
    .def_readwrite("theta_r", &dK_PDI::theta_r)
    .def_readwrite("theta_s", &dK_PDI::theta_s)
    .def_readwrite("alpha", &dK_PDI::alpha)
    .def_readwrite("n", &dK_PDI::n)
    .def_readwrite("K_s", &dK_PDI::K_s)
    .def_readwrite("K_sf", &dK_PDI::K_sf)
    .def_readwrite("slope", &dK_PDI::slope)
    .def_readwrite("tau", &dK_PDI::tau);
}

"""
