#ifndef otbSegmentCharacteristicsArchiveWriter_txx
#define otbSegmentCharacteristicsArchiveWriter_txx

#include "otbSegmentCharacteristicsArchiveWriter.h"

namespace otb{

  template <class TInputImage, class TInputLabel>
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::SegmentCharacteristicsArchiveWriter():
    m_CovarianceMatrixFlag(false),
    m_PerimeterFlag(false),
    m_NbDivisions(1)
  {
  }

  template <class TInputImage, class TInputLabel>
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::~SegmentCharacteristicsArchiveWriter()
  {
  }

  template <class TInputImage, class TInputLabel>
  void
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::SetInput(InputImagePointer im){
    this->SetNthInput(0,im);
  }
  
  template <class TInputImage, class TInputLabel>
  void
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::SetImageSeg(LabelImagePointer im){
    this->SetNthInput(1,im);
  }

  template <class TInputImage, class TInputLabel>
  TInputImage *
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::GetInput(){
    return static_cast<TInputImage *>(this->itk::ProcessObject::GetInput(0));
  }

  template <class TInputImage, class TInputLabel>
  TInputLabel *
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::GetImageSeg(){
    return static_cast<TInputLabel *>(this->itk::ProcessObject::GetInput(1));
  }
  
  template <class TInputImage, class TInputLabel>
  template <class T>
  T
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::readArchive(std::string name) {
    T object;
    std::ifstream ifs(name);
    boost::archive::binary_iarchive ia(ifs);
    ia >> object;
    return object;
  }
  
  template <class TInputImage, class TInputLabel>
  template <class T>
  void
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::writeArchive(T const& object, std::string name){
    std::ofstream ofs(name);
    boost::archive::binary_oarchive oa(ofs);
    oa << object;
    ofs.close();
  }
  
  //Predicate if 2 regions are neighbors
  template <class TInputImage, class TInputLabel>
  bool
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::areNeighboringRegions(RegionType const r1, RegionType const r2){
    const typename RegionType::IndexType i1 = r1.GetIndex();
    const typename RegionType::IndexType i2 = r2.GetIndex();
    const typename RegionType::SizeType s1 = r1.GetSize();
    const typename RegionType::SizeType s2 = r2.GetSize();
    bool b0 = false;
    bool b1 = false;
    if (i1[0]-(int)s2[0] == i2[0] || i1[0] == i2[0] || i1[0]+(int)s1[0] == i2[0]){
      b0= true;
    }
    if (i1[1]-(int)s2[1] == i2[1] || i1[1] == i2[1] || i1[1]+(int)s1[1] == i2[1]){
      b1= true;
    }
  
    return b0 && b1 && r1 != r2;
  }
  
  template <class TInputImage, class TInputLabel>
  void
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::fixArchives(std::vector<RegionType> const& regions, unsigned nbComps, std::string prefix)const{
    const unsigned n = regions.size();
    for(unsigned int i = 0; i < n; i++) {
      const RegionType r = regions[i];
      std::stringstream countName;
      countName << prefix << "count_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
      LabeledIntContainerType count = readArchive<LabeledIntContainerType>(countName.str());

      LabeledIntContainerType perimeter;
      if(m_PerimeterFlag){
	std::stringstream perimeterName;
	perimeterName << prefix << "perimeter_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
	perimeter = readArchive<LabeledIntContainerType>(perimeterName.str());
      }
      
      std::stringstream meansName;
      meansName << prefix << "means_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
      LabeledSampleContainerType means = readArchive<LabeledSampleContainerType>(meansName.str());
       
      LabeledSampleContainerType vars;
      LabeledMatrixContainerType covars;
      if(m_CovarianceMatrixFlag){
	std::stringstream name;
	name << prefix << "covars_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
	covars = readArchive<LabeledMatrixContainerType>(name.str());
      }
      else{
	std::stringstream name;
	name << prefix << "vars_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
	vars = readArchive<LabeledSampleContainerType>(name.str());
      }
      for(unsigned int j = 0; j < n ; j++) {
	if(j != i){
	  RegionType r2 = regions[j];
	  if(areNeighboringRegions(r,r2)){
	    std::stringstream countName2;
	    countName2 << prefix << "count_" << r2.GetIndex()[0] << "_" << r2.GetIndex()[1];
	    LabeledIntContainerType count2 = readArchive<LabeledIntContainerType>(countName2.str());
	    LabeledIntContainerType perimeter2;
	    if(m_PerimeterFlag){
	    std::stringstream perimeterName2;
	    perimeterName2 << prefix << "perimeter_" << r2.GetIndex()[0] << "_" << r2.GetIndex()[1];
	    perimeter2 = readArchive<LabeledIntContainerType>(perimeterName2.str());
	    }
	  
	    std::stringstream meansName2;
	    meansName2 << prefix << "means_" << r2.GetIndex()[0] << "_" << r2.GetIndex()[1];
	    LabeledSampleContainerType means2 = readArchive<LabeledSampleContainerType>(meansName2.str());
	    LabeledSampleContainerType vars2;
	    LabeledMatrixContainerType covars2;
	    if(m_CovarianceMatrixFlag){
	      std::stringstream name;
	      name << prefix << "covars_" << r2.GetIndex()[0] << "_" << r2.GetIndex()[1];
	      covars2 = readArchive<LabeledMatrixContainerType>(name.str());
	    }
	    else{
	      std::stringstream name;
	      name << prefix << "vars_" << r2.GetIndex()[0] << "_" << r2.GetIndex()[1];
	      vars2 = readArchive<LabeledSampleContainerType>(name.str());
	    }
	  
	    for(typename LabeledIntContainerType::const_iterator count2it = count2.cbegin();
		count2it != count2.cend() ; count2it++){
	      const typename LabeledIntContainerType::iterator cit = count.find(count2it->first);
	      if(cit != count.end()){
		// std::cout << "fuse" << std::endl;
		const LabelType currentLabel = cit->first;
		//Fuse means
		//cast to double here
		const double c1 = cit->second;
		const double c2 = count2it->second;
		const LabeledSampleContainerIteratorType m1 = means.find(currentLabel);
		const SampleType m2 = means2[currentLabel];
		if (m_CovarianceMatrixFlag) {
		  const LabeledMatrixContainerIteratorType v1 = covars.find(currentLabel);
		  const Matrix v2 = covars2[currentLabel];
		  //Fuse covars
		  for(unsigned int i = 0 ; i < nbComps+2; ++i)
		    {
		      for(unsigned int j = 0 ; j < nbComps+2; ++j)
			{
			  v1->second[i][j] = ((c1-1)*v1->second[i][j] + (c2-1)*v2[i][j] + (m1->second[i]-m2[i])*(m1->second[j]-m2[j])*c1*c2/(c1+c2))/(c1+c2-1);
			}
		    }
		}
		else{
		  const LabeledSampleContainerIteratorType v1 = vars.find(currentLabel);
		  const SampleType v2 = vars2[currentLabel];
		  //Fuse vars
		  for(unsigned int i = 0 ; i < nbComps+2; ++i)
		    {
		      v1->second[i] = ((c1-1)*v1->second[i] + (c2-1)*v2[i] + (m1->second[i]-m2[i])*(m1->second[i]-m2[i])*c1*c2/(c1+c2))/(c1+c2-1);
		    }
		}
		//Fuse means
		for(unsigned int i = 0 ; i < nbComps+2 ; ++i)
		  {
		    m1->second[i] = (c1*m1->second[i] + c2*m2[i])/(c1+c2);
		  }	      
		//Fuse count
		cit->second += count2it->second;
		//Fuse perimeters
		if(m_PerimeterFlag){
		  perimeter[currentLabel] += perimeter2[currentLabel];
		}
	      }
	    }
	  }
	}
      }
      std::stringstream newCountName;
      newCountName << prefix << "count_fixed_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
      writeArchive<LabeledIntContainerType>(count,newCountName.str());

      std::stringstream newPerimeterName;
      newPerimeterName << prefix << "perimeter_fixed_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
      writeArchive<LabeledIntContainerType>(perimeter,newPerimeterName.str());
    
      std::stringstream newMeansName;
      newMeansName << prefix << "means_fixed_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
      writeArchive<LabeledSampleContainerType>(means,newMeansName.str());

      if(m_CovarianceMatrixFlag){
	std::stringstream newCovarsName;
	newCovarsName << prefix << "covars_fixed_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
	writeArchive<LabeledMatrixContainerType>(covars,newCovarsName.str());
      }
      else{
	std::stringstream newVarsName;
	newVarsName << prefix << "vars_fixed_" << r.GetIndex()[0] << "_" << r.GetIndex()[1];
	writeArchive<LabeledSampleContainerType>(vars,newVarsName.str());
      }
    }
  }


  template <class TInputImage, class TInputLabel>
  void
  SegmentCharacteristicsArchiveWriter<TInputImage, TInputLabel>
  ::Run()
  {
    typename StreamingCharFilterType::Pointer streamingCharFilter = StreamingCharFilterType::New();
    typename CharFilter::Pointer charfilter = CharFilter::New();
    const unsigned int nbComps = this->GetInput()->GetNumberOfComponentsPerPixel();
    charfilter->SetInput(this->GetInput());
    charfilter->SetImageSeg(this->GetImageSeg());
    charfilter->SetNumberOfThreads(1);
    charfilter->SetPrefix(m_Prefix);
    charfilter->SetPerimeterFlag(m_PerimeterFlag);
    charfilter->SetCovarianceMatrixFlag(m_CovarianceMatrixFlag);
    streamingCharFilter->SetFilter(charfilter);
    streamingCharFilter->GetStreamer()->SetNumberOfDivisionsTiledStreaming(m_NbDivisions);
    streamingCharFilter->Update();
    m_StreamingManager = streamingCharFilter->GetStreamer()->GetStreamingManager();
    const unsigned int n = m_StreamingManager->GetNumberOfSplits();
    std::vector<RegionType> regions(n);
    for (unsigned int i = 0; i < n; i++) {   
      regions[i] = streamingCharFilter->GetStreamer()->GetStreamingManager()->GetSplit(i);
      // std::cout << regions[i];
    }
    fixArchives(regions,nbComps, m_Prefix);
  }
  
}
#endif
