
# Measured and estimated Gas transport rate (ka)
# Pereira et al. 2023, Angiosperms follow a convex trade-off to optimize hydraulic safety and efficiency
# New Phytologist

library(dplyr)
library(tidyverse)
library(broom)
library(ggplot2)
library(ggpubr)

dat5 <- data.frame()

  id <- read.csv("sample_id_all.csv", header=T, sep=',', as.is=T)
  list_samples <-id$sample 
  namecsv <- paste(list_samples,".csv",sep="")
  loop <- length(list_samples)
  diameter <- id$diameter*0.8/1000   #in m
  
  list_species <-id$species 
  anat <- read.csv("xinyi_database.csv", header=T, sep=',', as.is=T)
  
  ka_est_mean <- ka_all <- Nve_all<- Ap_all<- Ap_xinyi <- c()

  for(i in 1:loop){
    tryCatch({
      raw <- read.csv(namecsv[i], header=T, sep=',', as.is=T)
      species <- list_species[i]
      anat2 <- anat[anat$species == species,] 
      
      PLC = 1
      Hcc = 1.83E-02	                   #Hcc Henry's law constant (weighted average for N2 & O2)
      Cg_air = 4.09E+01	                 #mM or mol m-3,Cg,air at standard atmospheric pressure
      Dgw = 6.2E-11                        #Dgw, m2 s-1 diffusion coef of gas in wet wood: Justification Sorz J, Hietz P (2006) Gas diffusion through wood: implications for oxygen supply. Trees 20: 34-41
      Dg = 1.65E-09	                        #1.65E-09???, Dg, m2 s-1 diffusion coef of gas in wet pit membrane
      dv = (anat2[[11]]*1e-6)	# D  - Xinyi ######## DIVIDED BY 2????        #m, dv mean vessel diameter, diameter shared when overlapping
      Vl = anat2[[9]]/100       # Lv - Xinyi         #m, mean vessel length
      Vlc = Vl/2                         #m, cut open vessel length
      Vf = anat2[[2]]	          # Fc - Xinyi         #% (in decimal), fraction of vessel wall surface in common between vessels
      
      Vlf = anat2[[27]]                 	 #% (in decimal) vessel lumen in stem cross section, ax
      Vv = pi*Vlc*(dv/2)^2               #m3, volume of cut open vessel, Vv
      Vone = pi*Vl*(dv/2)^2              #m3, volume of one vessel, Vone
      Pmf = anat2[[4]]          # Fpf - Xinyi     	   #fraction vessel pit field that is pit membrane
      Pmt = anat2[[17]]*1e-9    # TPM center - Xinyi    	   #dm m, pit membrane thickness
      Ap = pi*dv*Vl*Pmf*Vf               #Ap m2, total pit membrane surface area
      Vt = 2.28E-06	                     #m3 volume of external tubing, Vt
      dw = diameter[i] 	                 #m, diam of wood in stem,dw
      Nv = Vlf*((dw/dv)^2)               #Nv, Number of vessels= ax(0.015/dv)2
      Nve = Nv*PLC                       #Nve, Number of vessels embolized to right
      Vvt = Vv*Nv                        #vol all cut open vessels
      Vt_v = Vt+Vvt                      #Vt+v, volume of cut open vessel plus tubing in model
      dt = 1                             #dt, simulation
      
      Vt2 = 5E-07            #tubing volume of the second sensor (at the branch tip)
      Vt_v2 = Vt2+Vvt        #Vt2+v, volume of cut open vessel plus tubing sensor 2 (at the branch tip) 
      
      R = 8.3144621/1000    #gas constant
      temp = 298.15         #temperature kelvins (25?C)
      p_atm <- 101.3        # define atmospheric pressure (in kPa)
      
      ka = dt*Ap*Dg*Hcc/Pmt    #calculate axial transport coefficient (ka= dt PM_a Dg Hcc/ddm)
      
      # define initial (pi_s) and final (pf_s) pressures time desired (usually 1.5 and 30 seconds, respectively)
      pi_s <- 1.5
      pf_s <-10  
      
      
      PI <- raw[raw$time== pi_s*2, ]
      pf <- raw[raw$time==pf_s*2, ]
      pres <- merge(PI, pf, by="rep")
      #calculate air discharged (AD) in mols, uL, and the percentage of air discharged (PAD)
      pres$ad_mol <- (((p_atm-pres$pressure.y*1000)*(Vt))/(8.3144621*293.15))-(((p_atm-pres$pressure.x*1000)*(Vt))/(8.3144621*293.15))
      pres$ad_ul <- (pres$ad_mol*8.3144621*293.15/(p_atm*1000))*1000*1000*1000
      #pres <- pres[pres$ad_ul > 2 , ] # remove points when solenoid valve did not work
      pres$pad <- (100*(pres$ad_ul-(min(pres$ad_ul))))/((max(pres$ad_ul))-(min(pres$ad_ul)))
      
      pres2 <- pres %>% select(rep,pad)
      
      raw <- merge(raw, pres2, by="rep")
      
      raw <- arrange(raw, rep, time)
      raw$Cg1 <- ((p_atm-raw$pressure))/(R*temp)
      raw$Cg2 <- ((p_atm-raw$pressure2))/(R*temp)
      
      raw$n1 <- raw$Cg1*Vt_v/Nve
      raw$lag1<- lag(raw$n1, n = 2)
      raw$dn1 <- raw$n1-raw$lag1
      raw$ka_est <- (raw$dn1)/(raw$Cg2-raw$Cg1)
      
      raw$time2 <- raw$time*0.5
      raw1 <- filter(raw, time >6)
      raw1 <-raw1 %>% filter(time == min(time))

      #find the nearest value close to XX% PAD
      nearxx <- raw1[raw1$pad >=PLC*100, ]
      raw1 <-nearxx %>% filter(pad == min(pad))
      raw1 <- filter(raw1, ka_est >0)

      ka_est_mean[i] <- mean(raw1$ka_est)
      ka_all[i] <- ka
      Nve_all[i] <- Nve
      Ap_all[i] <- Ap
      Ap_xinyi[i] <- anat2[[14]]
  
    }, error=function(e){})
  }

  dat <- data.frame(id$species, id$sample, ka_est_mean, ka_all, Nve_all, Ap_all, Ap_xinyi)
  names(dat)[names(dat) == "id.species"] <- "species"
  dat <- na.omit(dat)

  dat5 <- rbind(dat5, dat)

model_data <- data.frame()
model_loop <- seq(min(Ap_all, na.rm = T), max(Ap_all, na.rm = T),
                  by = (max(Ap_all, na.rm = T)-min(Ap_all, na.rm = T))/10)

for (m in model_loop) {
  model_Pmt = 100E-09
  for(k in 1:30) {
    model_Ap <- m
    model_ka = dt*model_Ap*Dg*Hcc/model_Pmt    #calculate axial transport rate (ka= dt PM_a Dg Hcc/ddm)
    model_data1 <- data.frame(model_Pmt,model_Ap,model_ka)
    names(model_data1) <- c("model_Pmt","model_Ap","model_ka")
    model_data <-rbind(model_data, model_data1)
    model_Pmt <- model_Pmt+25E-09
    #print(m)
  }
}

pmt <- read.csv('Kaack_tableS1.csv', header=T, sep=',', as.is=T)
dat2 <- merge(dat5, pmt, by="species")

# REMOVE OUTLIERS ka < 1e-10
dat4 <- dat2 %>%  filter(ka_est_mean < 1e-10) %>%  
  group_by(species) %>% summarise(sd=sd(ka_est_mean),mean=mean(ka_est_mean), ka_all=mean(ka_all), 
                                  tpm_centre=mean(tpm_centre), Ap_all=mean(Ap_all), p50=mean(p50))

fig1<-#data2 %>% filter(ka_est_mean < 1e1) %>% #group_by(species) %>% summarise_all(funs(mean))%>%# filter(species != "Corylus_avellana") %>% 
  ggplot(dat4,aes(x = ka_all, y = mean,color= species))+
  geom_smooth(aes(x = ka_all, y = mean), method=lm , color="blue", fill="lightgray", se=TRUE, size=1) +
  geom_errorbar(aes(ymin=mean-sd, ymax=mean+sd),size=1) +
  geom_point(size=2) +
  geom_abline(intercept = 0, slope = 1) +
  scale_x_continuous(bquote('k'[a_estimated]*' ('~ m^3*')')) +
  scale_y_continuous(bquote('k'[a_measured]*' ('~ m^3*')')) +
  theme_bw() + theme(panel.border = element_blank(), panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
                     axis.line = element_line(colour = "black"), aspect.ratio = 1) + # formato final
  theme(plot.margin = unit(c(0.5,0.5,0.5,0.5), "cm"))+
  theme(legend.position = "right")+
  theme(text = element_text(size=20), axis.title.x = element_text(vjust = - 1.0),axis.title.y = element_text(vjust = + 1.0))
fig1


fig2<-ggplot(dat4,aes(x = tpm_centre, y = mean,color= Ap_all))+
  geom_line(data = model_data, aes(x = model_Pmt*10^9, y = model_ka, group=model_Ap, color=model_Ap), size=1) +
  geom_errorbar(aes(ymin=mean-sd, ymax=mean+sd),size=1) +
  geom_point(size=2) +
  #  geom_line(data= raw, aes(x = time, y = pressure2, group=rep, color=rep)) +
  scale_x_continuous("Pit membrane thickness (nm)") +
  scale_y_continuous(bquote('k'[a]*' ('~ m^3*')')) +
  theme_bw() + theme(panel.border = element_blank(), panel.grid.major = element_blank(), panel.grid.minor = element_blank(),
                     axis.line = element_line(colour = "black"), aspect.ratio = 1) + # formato final
  scale_colour_gradient(name = bquote('A'[P]*'('~m^2*')'), low = "blue", high = "red")+
  theme(plot.margin = unit(c(0.5,0.5,0.5,0.5), "cm"))+
  theme(legend.position = c(0.75,0.75))+
  theme(text = element_text(size=20), axis.title.x = element_text(vjust = - 1.0),axis.title.y = element_text(vjust = + 1.0))
fig2
