# Laminar flow through a pipe of rectangular cross-section with an oscillating pressure gradient
# Fan and Chao, 1965. Unsteady, Laminar, Incompressible Flow Through Rectangular Ducts

# Import libraries
import numpy as np
import matplotlib.pyplot as plt
import csv
import math
import scipy.special as sc

from fig_config import (
	add_grid,
	figure_features,
)	# <--- import customized functions

# Input dataset
# Plot figure
plotFigure = 1 # NO=0, YES=1
saveFig = int(input("Save figure ? (0 or 1), where 0=NO, 1=YES: "))
# saveTxt = int(input("Save txt file ? (0 or 1), where 0=NO, 1=YES: "))
# saveFig = 1 # NO=0, YES=1, save figure
# saveTxt = 1 # NO=0, YES=1, save txt file

# Physical
rho = 1000.0		# fluid density (kg/m3)
nu = 1.0e-4			# kinematic viscosity (m2/s) nu = mu / rho
# Pipe dimensions length x height x width (m x m x m)
l_ = 0.15
h_ = 0.1
w_ = 0.1
# Point to calculate the velocity
# rectangular channel of height -h_/2 ≤ y ≤ h_/2 and width -w_/2 ≤ z ≤ w_/2 are
y_ = 0.0
z_ = 0.0
# Sum terms
sum_terms = 20


plt.style.use('seaborn-whitegrid')
# seaborn-whitegrid
# seaborn-deep
# seaborn-dark-palette
# seaborn-colorblind
# fivethirtyeight
# ggplot
# seaborn
# seaborn-pastel
# seaborn-white
# tableau-colorblind10
# Solarize_Light2

# Font sizes
SMALL_SIZE = 12
MEDIUM_SIZE = 14
BIGGER_SIZE = 18
ANALYTIC_LINE = 1.0
FOAM_LINE = 1.0
MPS_LINE = 1.5
ANALYTIC_MARKER = 1.5
FOAM_MARKER = 4.0
MPS_MARKER = 5.0

plt.rcParams["font.family"] = "Times New Roman"
plt.rcParams['text.usetex'] = True
plt.rc('font', size=BIGGER_SIZE)          # controls default text sizes
plt.rc('axes', titlesize=BIGGER_SIZE)     # fontsize of the axes title
plt.rc('axes', labelsize=BIGGER_SIZE)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('ytick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('legend', fontsize=SMALL_SIZE)    # legend fontsize
plt.rc('figure', titlesize=BIGGER_SIZE)  # fontsize of the figure title

def readCsvMps(fileName, listTime, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listTime.append(row['Time'])
			listVel.append(row['avg(Velocity(0))'])
	return listTime, listVel

def readCsvFoam(fileName, listTime, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listTime.append(row['Time'])
			listVel.append(row['avg(U(0))'])
	return listTime, listVel

if __name__ == '__main__':

	# reading a value, converting to float
	# dp_ = float(input("Enter the pressure amplitude: "))
	dp_ = 700
	print('A: %.4f' % dp_)
	# per_ = float(input("Enter the period: "))
	per_ = 2.0
	freq_ = 1.0 / per_
	omega_ = freq_ * 2.0 * np.pi 
	print('Omega (frequency): %.2f' % omega_)
	# nper_ = int(input("Enter the number of periods: "))
	nper_ = 25
	tf_ = nper_ * per_
	print('tf: %.2f' % tf_)

	# Particle distance
	# reading a value, converting to int
	lo_id = int(input("Enter the particle distance (1 or 2), where 1: 0.010m, 2: 0.005m "))
	lo_ = '0p0100'
	lo_num = 0.01
	if lo_id == 2:
		lo_ = '0p0050'
		lo_num = 0.005
	print('lo: %.5f' % lo_num)

	# Time array
	k_ = int(nper_ * 50)
	t_ = np.linspace(0, tf_, k_)	# Time array
	# Velocity empty array
	uxANAL = []
	# Ctes
	Kp = dp_ / l_ * 16.0 / (omega_ * np.pi ** 2 * rho)
	mu = nu * rho
	a_ = w_ / 2.0
	b_ = h_ / 2.0
	pi2 = pow(np.pi, 2) / 4.0
	pi4 = pow(np.pi, 4) / 16.0
	
	## Analytical Velocity
	for i in range(k_):
		sumVal = 0.0
		for m_ in range(0, sum_terms):
			for n_ in range(0, sum_terms):
				
				mn_ = ((2.0 * m_ + 1.0) ** 2 / a_ ** 2) + ((2.0 * n_ + 1.0) ** 2 / b_ ** 2)
				omega_nu = omega_ ** 2 + nu ** 2 * pi4 * mn_ ** 2

				A1 = pow(-1.0, m_ + n_) / ((2.0 * m_ + 1.0)*(2.0 * n_ + 1.0))
				A2 = np.cos((2.0 * m_ + 1.0) * np.pi * z_ / (2.0 * a_)) * np.cos((2.0 * n_ + 1.0) * np.pi * y_ / (2.0 * b_))
				A3 = (omega_ * nu * pi2 * mn_ * np.cos(omega_ * t_[i] - np.pi / 2.0) + omega_ ** 2 * np.sin(omega_ * t_[i] - np.pi / 2.0)) / omega_nu
				A4 = (omega_ * nu * mn_ * np.exp(- pi2 * mn_ * nu * t_[i])) / omega_nu
				
				sumVal += A1 * A2 * (A3 - A4)

		# Velocity
		uxANAL.append(Kp * sumVal)


	print('\nRectangular pipe')
	print('Max Velocity: %.3f' % max(uxANAL))

	# Hydraulic diameter
	Dh_ = 4.0 * h_ * w_ / (2.0 * h_ + 2.0 * w_)
	# Reynolds number
	umax = max(uxANAL)
	uavg = 0.5 * umax
	Re_ = uavg * Dh_ / nu
	flow_regime = 'Laminar'
	if Re_ < 2300:
		print('Reynolds: %.3f < 2300 - %s flow' % (Re_, flow_regime))
	elif Re_ > 2900:
		flow_regime = 'Turbulent'
		print('Reynolds: %.3f > 2900 - %s flow' % (Re_, flow_regime))
	else:
		flow_regime = 'Intermittent'
		print('Reynolds: %.3f between 2300-2900 - %s flow' % (Re_, flow_regime))

	
	# # Plotting the analytical points 
	# plt.plot(t_, uxANAL, color='green', linestyle='dashed', linewidth = 2)

	# # naming the x axis
	# plt.xlabel('t (s)')
	# # naming the y axis
	# plt.ylabel('u (m/s)')
	# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
	# #plt.title(r'\TeX\ is Number $\displaystyle\sum_{n=1}^\infty \frac{-e^{i\pi}}{2^n}$!', fontsize=16, color='r')
	# # legend = plt.legend(loc=0, shadow=False, frameon=True)
	# # legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
	# # legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend
	# fig = plt.gcf() # get current figure
	# file_name = 'analytical_poiseuille_velocity_LxHxW_' + f'{l_*1000:.0f}' + '_' + f'{h_*1000:.0f}' + '_' + f'{w_*1000:.0f}' + '_dP_' + f'{dp_:.0f}' + '_' + flow_regime
	# if saveFig:
	#     fig.savefig(file_name + '.png')
	# plt.show()
	# plt.close(fig)    # close the figure window

	
	# # Parabolic flow - analytical (txt file)

	# # Linearly spaced vector of k points
	# k = 40
	# yp_ = np.linspace(0, h_, k)

	# # empty array
	# up_ = []

	# for i in range(k):
	# 	# Velocity 1st term
	# 	u1 = G * yp_[i] * (h_ - yp_[i]) / (2.0 * mu)
	# 	# Velocity 2nd term
	# 	sum2nd = 0.0
	# 	for n_ in range(1, sum_terms):
	# 		bn = (2.0 * n_ - 1.0) * np.pi / h_
	# 		sum2nd += 1.0 / ((2.0 * n_ - 1.0) ** 3) * ( np.sinh(bn * z_) + np.sinh(bn * (w_ - z_)) ) / (np.sinh(bn * w_)) * np.sin(bn * yp_[i])

	# 	u2 = - 4.0 * G * h_ ** 2 / (mu * np.pi ** 3) * sum2nd

	# 	# Velocity
	# 	up_.append(u1 + u2)

	# file_name = 'analytical_poiseuille_parabolic_velocity_LxHxW_' + f'{l_*1000:.0f}' + '_' + f'{h_*1000:.0f}' + '_' + f'{w_*1000:.0f}' + '_dP_' + f'{dp_:.0f}' + '_' + flow_regime

	# # plotting the points 
	# plt.plot(yp_, up_, color='green', linestyle='dashed', linewidth = 2)

	# # naming the x axis
	# plt.xlabel('x (m)')
	# # naming the y axis
	# plt.ylabel('u (m/s)')
	# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
	# #plt.title(r'\TeX\ is Number $\displaystyle\sum_{n=1}^\infty \frac{-e^{i\pi}}{2^n}$!', fontsize=16, color='r')
	# # legend = plt.legend(loc=0, shadow=False, frameon=True)
	# # legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
	# # legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend
	# fig = plt.gcf() # get current figure
	# if saveFig:
	#     fig.savefig(file_name + '.png')
	# plt.show()
	# plt.close(fig)    # close the figure window


	# if saveTxt:
	# 	# Write and save txt file
	# 	data = np.column_stack([yp_, up_])
	# 	datafile = file_name + ".txt"
	# 	with open(datafile, 'wb') as f:
	# 		np.savetxt(f, data, fmt=['%.5f','%.5f'], delimiter=',', comments='', header='"x(m)","u(m/s)"')


	## Analytical x Numerical

	# csv MPS file name
	filenameMPS = "InOutSquare_lo" + lo_ + "_Sin_DP" + f'{dp_:.0f}' + "-T2p0s-" + f'{tf_:.0f}' + "s-vel.txt"
	tMPS = []
	uxMPS = []
	readCsvMps(filenameMPS, tMPS, uxMPS)
	# Convert list to array
	tMPSa = np.array(tMPS)
	uxMPSa = np.array(uxMPS)
	# Convert to float
	tMPS  =  tMPSa.astype(np.float64)
	uxMPS = uxMPSa.astype(np.float64)

	# csv OpenFOAM file name
	filenameFOAM = "square-tube-sine-DP" + f'{dp_:.0f}' + "-vel.txt"
	tFOAM = []
	uxFOAM = []
	readCsvFoam(filenameFOAM, tFOAM, uxFOAM)
	# Convert list to array
	tFOAMa = np.array(tFOAM)
	uxFOAMa = np.array(uxFOAM)
	# Convert to float
	tFOAM  =  tFOAMa.astype(np.float64)
	uxFOAM = uxFOAMa.astype(np.float64)

	## Compute error
	# # Get sorted array index
	# sortIndexMPS = np.argsort(tMPS)
	# # Rearrange array based upon index array
	# tMPS = tMPS[sortIndexMPS]
	# uxMPS = uxMPS[sortIndexMPS]

	# Interpolate to k_ points
	xvals = np.linspace(min(t_), max(t_), k_)
	# u_new = np.interp(xvals, t_, uxANAL)
	uxMPSnew = np.interp(xvals, tMPS, uxMPS)
	uxFOAMnew = np.interp(xvals, tFOAM, uxFOAM)

	# Last 4 periods
	k2 = round(0.5*k_)
	k4 = round((nper_-4) * 50)
	# print(nper_)
	# print((nper_-4) * 50)
	# print(k4)
	
	# print('Max Velocity MPS: %.3f' % max(uxMPS[k4:-1]))
	# print('Max Velocity MFOAM: %.3f' % max(uxFOAM[k4:-1]))
	print('Max Velocity MPS-interp: %.3f' % max(uxMPSnew[k4:-1]))
	print('Max Velocity MFOAM-interp: %.3f' % max(uxFOAMnew[k4:-1]))

	# Calculate the difference between the estimated and the actual value using numpy.subtract() function.
	# Further, calculate the square of the above results using numpy.square() function.
	# Finally, calculate the mean of the squared value using numpy.mean() function. The output is the MSE score.
	# MSE = np.square(np.subtract(uxANAnew,uxMPSnew)).mean()
	# sumAna = np.sum(uxANAL[k4:-1])
	MSE = np.square(np.subtract(uxANAL[k4:-1],uxMPSnew[k4:-1])).mean()
	# At the end, calculate the square root of MSE using math.sqrt() function to get the RMSE value.
	RMSE = math.sqrt(MSE)
	print("RMSE MPS-ANA:", f'{RMSE:.4f}')

	MSE = np.square(np.subtract(uxMPSnew[k4:-1],uxFOAMnew[k4:-1])).mean()
	RMSE = math.sqrt(MSE)
	print("RMSE MPS-FOAM:", f'{RMSE:.4f}')

	# MSE = np.square(np.subtract(uxANAnew,uxFOAMnew)).mean()
	MSE = np.square(np.subtract(uxANAL[k4:-1],uxFOAMnew[k4:-1])).mean()
	RMSE = math.sqrt(MSE)
	print("RMSE FOAM-ANA:", f'{RMSE:.4f}')

	# Err = abs(max(uxANAnew) - max(uxMPSnew)) / max(uxANAnew) * 100
	Err = abs(max(uxANAL) - max(uxMPSnew[k4:-1])) / max(uxANAL) * 100
	print("Error MPS-ANA:", f'{Err:.2f}')
	Err = abs(max(uxFOAMnew[k4:-1]) - max(uxMPSnew[k4:-1])) / max(uxFOAMnew[k4:-1]) * 100
	print("Error MPS-FOAM:", f'{Err:.2f}')
	# Err = abs(max(uxANAnew) - max(uxFOAMnew)) / max(uxANAnew) * 100
	Err = abs(max(uxANAL) - max(uxFOAMnew[k4:-1])) / max(uxANAL) * 100
	print("Error FOAM-ANA:", f'{Err:.2f}')

	# Plot and save figure
	if plotFigure:

		t_til = [x / per_ for x in t_]
		tFOAM_til = [x / per_ for x in tFOAM]
		tMPS_til = [x / per_ for x in tMPS]

		ymin = round(min(uxANAL), 1)
		ymax = round(max(uxMPS), 1)
		if max(uxMPS) > 1:
			ymin = math.ceil(min(uxANAL)-1.0)
			ymax = math.ceil(max(uxMPS))

		figure_features()	# <--- Add this line to every figure
		ax = plt.axes(xlim = (0, max(t_til)), ylim = (ymin, ymax))
		add_grid(ax, locations=(per_*0.5, per_, 0.5, 1))	# <--- Add this line to every figure

		# plotting Analytic velocity
		labelName = "Analytical (Fan and Chao, 1965)"
		plt.plot(t_til, uxANAL, color='black', linestyle='dashed', linewidth=ANALYTIC_LINE, label=labelName)

		labelName = "OpenFOAM"
		# plotting FOAM velocity
		plt.plot(tFOAM_til, uxFOAM, color='orange', linestyle='none', marker='o', markerfacecolor='none', markersize=FOAM_MARKER, label=labelName)

		labelName = "MPS"
		# plotting MPS velocity
		plt.plot(tMPS_til, uxMPS, color='blue', linestyle='none', marker='x', markersize=MPS_MARKER, label=labelName)

		plt.xlabel(r'$t/T$')
		plt.ylabel(r'$u (y=' + f'{z_:.2f}' + ',z=' f'{y_:.2f}' ',t)$ (m/s)')
		# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
		# Add text at the top of plot
		subTxtPos = [0.05, 0.90] # Text Position
		plt.text(subTxtPos[0], subTxtPos[1], "$Re=" + f'{Re_:.0f}' + "$", 
						transform = ax.transAxes, fontsize=BIGGER_SIZE)
		legend = plt.legend(loc=0, shadow=False, frameon=True)
		legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
		legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend

		fig = plt.gcf() # get current figure
		plt.show()

		if saveFig:
			file_name = 'rectangular_sine_press_LxHxW_' + f'{l_*1000:.0f}' + '_' + f'{h_*1000:.0f}' + '_' + f'{w_*1000:.0f}' + '_dP_' + f'{dp_:.0f}' + '_lo' + lo_ + '_ANALYTIC_MPS_OPENFOAM_' + flow_regime
			fig.savefig(file_name + '.png')
		
		plt.close(fig)    # close the figure window