# Laminar flow through a pipe of recatngular cross-section with an cte pressure gradient
# https://en.wikipedia.org/wiki/Hagen%E2%80%93Poiseuille_equation
# Boussinesq, 1868. Mémoire sur l'influence des Frottements dans les Mouvements Réguliers des Fluids

# Import libraries
import numpy as np
import matplotlib.pyplot as plt
import csv
import math
import scipy.special as sc

from fig_config import (
	add_grid,
	figure_features,
)	# <--- import customized functions

# Input dataset
# Plot figure
plotFigure = 1 # NO=0, YES=1
saveFig = int(input("Save figure ? (0 or 1), where 0=NO, 1=YES: "))
saveTxt = int(input("Save txt file ? (0 or 1), where 0=NO, 1=YES: "))
# saveFig = 1 # NO=0, YES=1, save figure
# saveTxt = 1 # NO=0, YES=1, save txt file

# Physical
rho = 1000.0		# fluid density (kg/m3)
nu = 1.0e-4					# kinematic viscosity (m2/s) nu = mu / rho
# delta_press = 30.0			# Pa
# Pipe dimensions length x height x width (m x m x m)
l_ = 0.15
h_ = 0.1
w_ = 0.1
# Point to calculate the velocity
# rectangular channel of height 0 ≤ y ≤ h_ and width 0 ≤ z ≤ w_ are
y_ = h_ / 2.0
z_ = w_ / 2.0
# Sum terms
sum_terms = 40

# print('dP: %.2f' % delta_press)

plt.style.use('seaborn-whitegrid')
# seaborn-whitegrid
# seaborn-deep
# seaborn-dark-palette
# seaborn-colorblind
# fivethirtyeight
# ggplot
# seaborn
# seaborn-pastel
# seaborn-white
# tableau-colorblind10
# Solarize_Light2

# Font sizes
SMALL_SIZE = 12
MEDIUM_SIZE = 14
BIGGER_SIZE = 18
ANALYTIC_LINE = 1.0
FOAM_LINE = 1.0
MPS_LINE = 1.5
ANALYTIC_MARKER = 1.5
FOAM_MARKER = 6.0
MPS_MARKER = 7.0

plt.rcParams["font.family"] = "Times New Roman"
plt.rcParams['text.usetex'] = True
plt.rc('font', size=BIGGER_SIZE)          # controls default text sizes
plt.rc('axes', titlesize=BIGGER_SIZE)     # fontsize of the axes title
plt.rc('axes', labelsize=BIGGER_SIZE)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('ytick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('legend', fontsize=SMALL_SIZE)    # legend fontsize
plt.rc('figure', titlesize=BIGGER_SIZE)  # fontsize of the figure title

def readCsvMps(fileName, listPos, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listPos.append(row['Points:1'])
			listVel.append(row['Velocity:0'])
	return listPos, listVel

def readCsvFoam(fileName, listPos, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listPos.append(row['Points:2'])
			listVel.append(row['U:0'])
	return listPos, listVel

if __name__ == '__main__':

	# reading a value, converting to float
	delta_press = float(input("Enter the delta press (30, 40 or 50): "))
	print('dP: %.2f' % delta_press)

	# Particle distance
	# reading a value, converting to int
	# lo_id = int(input("Enter the particle distance (1 or 2), where 1: 0.010m, 2: 0.005m "))
	lo_id = 1
	lo_ = '0p0100'
	lo_num = 0.01
	if lo_id == 2:
		lo_ = '0p0050'
		lo_num = 0.005
	print('lo: %.5f' % lo_num)

	print('rho: %.3f nu: %.6f' % (rho, nu))

	# https://en.wikipedia.org/wiki/Hagen%E2%80%93Poiseuille_equation

	# Ctes
	G = delta_press / l_
	mu = nu * rho 				# dynamic viscosity [kg / (m * s)] mu = nu * rho

	# Velocity 1st term
	u1 = G * y_ * (h_ - y_) / (2.0 * mu)

	# Velocity 2nd term

	#Declare variable for sum
	sum2nd = 0.0
	for n_ in range(1, sum_terms):
		bn = (2.0 * n_ - 1.0) * np.pi / h_
		sum2nd += 1.0 / ((2.0 * n_ - 1.0) ** 3) * ( np.sinh(bn * z_) + np.sinh(bn * (w_ - z_)) ) / (np.sinh(bn * w_)) * np.sin(bn * y_)

	u2 = - 4.0 * G * h_ ** 2 / (mu * np.pi ** 3) * sum2nd

	# Velocity
	u_ = u1 + u2

	print('Velocity 1st term: %.3f' % u1)
	print('Velocity 2nd term: %.3f' % u2)
	print('Velocity: %.3f' % u_)

	# Hydraulic diameter
	Dh_ = 4.0 * h_ * w_ / (2.0 * h_ + 2.0 * w_)

	# Reynolds number
	umax = u_
	uavg = 0.5 * umax
	Re_ = uavg * Dh_ / nu
	flow_regime = 'Laminar'
	if Re_ < 2300:
		print('Reynolds: %.3f < 2300 - %s flow' % (Re_, flow_regime))
	elif Re_ > 2900:
		flow_regime = 'Turbulent'
		print('Reynolds: %.3f > 2900 - %s flow' % (Re_, flow_regime))
	else:
		flow_regime = 'Intermittent'
		print('Reynolds: %.3f between 2300-2900 - %s flow' % (Re_, flow_regime))


	# Parabolic flow
	# Linearly spaced vector of k points
	k = 40
	yp_ = np.linspace(0, h_, k)

	# empty array
	up_ = []

	for i in range(k):
		# Velocity 1st term
		u1 = G * yp_[i] * (h_ - yp_[i]) / (2.0 * mu)
		# Velocity 2nd term
		sum2nd = 0.0
		for n_ in range(1, sum_terms):
			bn = (2.0 * n_ - 1.0) * np.pi / h_
			sum2nd += 1.0 / ((2.0 * n_ - 1.0) ** 3) * ( np.sinh(bn * z_) + np.sinh(bn * (w_ - z_)) ) / (np.sinh(bn * w_)) * np.sin(bn * yp_[i])

		u2 = - 4.0 * G * h_ ** 2 / (mu * np.pi ** 3) * sum2nd

		# Velocity
		up_.append(u1 + u2)


	file_name = 'analytical_poiseuille_velocity_LxHxW_' + f'{l_*1000:.0f}' + '_' + f'{h_*1000:.0f}' + '_' + f'{w_*1000:.0f}' + '_dP_' + f'{delta_press:.0f}' + '_' + flow_regime

	# # plotting the points 
	# plt.plot(yp_, up_, color='green', linestyle='dashed', linewidth = 2)

	# # naming the x axis
	# plt.xlabel('x (m)')
	# # naming the y axis
	# plt.ylabel('u (m/s)')
	# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
	# #plt.title(r'\TeX\ is Number $\displaystyle\sum_{n=1}^\infty \frac{-e^{i\pi}}{2^n}$!', fontsize=16, color='r')
	# # legend = plt.legend(loc=0, shadow=False, frameon=True)
	# # legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
	# # legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend
	# fig = plt.gcf() # get current figure
	# if saveFig:
	#     fig.savefig(file_name + '.png')
	# plt.show()
	# plt.close(fig)    # close the figure window

	if saveTxt:
		# Write and save txt file
		data = np.column_stack([yp_, up_])
		datafile = file_name + ".txt"

		with open(datafile, 'wb') as f:
			np.savetxt(f, data, fmt=['%.5f','%.5f'], delimiter=',', comments='', header='"x(m)","u(m/s)"')


	## Analytical x Numerical

	# csv MPS file name
	# filenameMPS = "mps-InOut-Box-DP" + f'{delta_press:.0f}' + ".txt"
	filenameMPS = "mps-InOutSquare-Box-" + 'lo' + lo_ + "-DP" + f'{delta_press:.0f}' + ".txt"
	yMPS = []
	uxMPS = []
	readCsvMps(filenameMPS, yMPS, uxMPS)
	# Convert list to array
	yMPSa = np.array(yMPS)
	uxMPSa = np.array(uxMPS)
	# Convert to float
	yMPS  =  yMPSa.astype(np.float64) - 0.05
	uxMPS = uxMPSa.astype(np.float64)

	# csv OpenFOAM file name
	filenameFOAM = "openFoam-square-tube-DP" + f'{delta_press:.0f}' + ".txt"
	yFOAM = []
	uxFOAM = []
	readCsvFoam(filenameFOAM, yFOAM, uxFOAM)
	# Convert list to array
	yFOAMa = np.array(yFOAM)
	uxFOAMa = np.array(uxFOAM)
	# Convert to float
	yFOAM  =  yFOAMa.astype(np.float64)
	uxFOAM = uxFOAMa.astype(np.float64)

	## Compute error
	# Get sorted array index
	sortIndexMPS = np.argsort(yMPS)
	# Rearrange array based upon index array
	yMPS = yMPS[sortIndexMPS]
	uxMPS = uxMPS[sortIndexMPS]

	# Get sorted array index
	sortIndexFOAM = np.argsort(yFOAM)
	# Rearrange array based upon index array
	yFOAM  = yFOAM[sortIndexFOAM]
	uxFOAM = uxFOAM[sortIndexFOAM]

	# Interpolate to 50 points
	xvalsMPS = np.linspace(min(yMPS), max(yMPS), 50)
	uxANAnew = np.interp(xvalsMPS, yp_, up_)
	uxMPSnew = np.interp(xvalsMPS, yMPS, uxMPS)
	uxFOAMnew = np.interp(xvalsMPS, yFOAM, uxFOAM)

	# Calculate the difference between the estimated and the actual value using numpy.subtract() function.
	# Further, calculate the square of the above results using numpy.square() function.
	# Finally, calculate the mean of the squared value using numpy.mean() function. The output is the MSE score.
	MSE = np.square(np.subtract(uxANAnew,uxMPSnew)).mean()
	# MSE = (np.divide(np.square(np.subtract(uxANAnew,uxMPSnew)), np.square(uxANAnew)) ).mean()
	# At the end, calculate the square root of MSE using math.sqrt() function to get the RMSE value.
	RMSE = math.sqrt(MSE) * 100
	print("RMSE MPS-ANA:", f'{RMSE:.4f}')

	MSE = np.square(np.subtract(uxFOAMnew,uxMPSnew)).mean()
	# MSE = (np.divide( np.square(np.subtract(uxFOAMnew,uxMPSnew)), np.square(uxFOAMnew))).mean()
	RMSE = math.sqrt(MSE) * 100
	print("RMSE MPS-FOAM:", f'{RMSE:.4f}')
	
	MSE = np.square(np.subtract(uxANAnew,uxFOAMnew)).mean()
	# MSE = (np.divide(np.square(np.subtract(uxANAnew,uxFOAMnew)), np.square(uxANAnew)) ).mean()
	RMSE = math.sqrt(MSE) * 100
	print("RMSE FOAM-ANA:", f'{RMSE:.4f}')

	Err = abs(max(uxANAnew) - max(uxMPSnew)) / max(uxANAnew) * 100
	print("Error MPS-ANA:", f'{Err:.2f}')
	Err = abs(max(uxFOAMnew) - max(uxMPSnew)) / max(uxFOAMnew) * 100
	print("Error MPS-FOAM:", f'{Err:.2f}')
	Err = abs(max(uxANAnew) - max(uxFOAMnew)) / max(uxANAnew) * 100
	print("Error FOAM-ANA:", f'{Err:.2f}')

	# labelName = "Analytical (Stokes, 1845)"
	# plt.plot(xvalsMPS, uxANAnew, color='black', linestyle='dashed', linewidth=ANALYTIC_LINE, label=labelName)
	# labelName = "OpenFOAM"
	# plt.plot(xvalsMPS, uxFOAMnew, color='orange', linestyle='none', marker='o', markerfacecolor='none', markersize=FOAM_MARKER, label=labelName)
	# labelName = "MPS"
	# plt.plot(xvalsMPS, uxMPSnew, color='blue', linestyle='none', marker='x', markersize=MPS_MARKER, label=labelName)
	# plt.show()

	# Plot and save figure
	if plotFigure:

		yp_til = [(x - w_/2.0) / w_ for x in yp_]
		up_til = [x / u_ for x in up_]

		epsx = 0.1
		epsy = 0.2
		figure_features()	# <--- Add this line to every figure
		ax = plt.axes(xlim = (round(min(yp_til), 1)-epsx, round(max(yp_til), 1)+epsx), ylim = (round(min(up_til), 1), round(max(up_til), 1)+epsy))
		add_grid(ax, locations=(0.25, 0.5, 0.1, 0.2))	# <--- Add this line to every figure

		# plotting Analytic velocity
		labelName = "Analytical (Boussinesq, 1868)"
		plt.plot(yp_til, up_til, color='black', linestyle='dashed', linewidth=ANALYTIC_LINE, label=labelName)

		yFOAM_til = [(x - w_/2.0) / w_ for x in yFOAM]
		uxFOAM_til = [x / u_ for x in uxFOAM]
		labelName = "OpenFOAM"
		# plotting FOAM velocity
		plt.plot(yFOAM_til, uxFOAM_til, color='orange', linestyle='none', marker='o', markerfacecolor='none', markersize=FOAM_MARKER, label=labelName)

		yMPS_til = [(x - w_/2.0) / w_ for x in yMPS]
		uxMPS_til = [x / u_ for x in uxMPS]
		labelName = "MPS"
		# plotting MPS velocity
		plt.plot(yMPS_til, uxMPS_til, color='blue', linestyle='none', marker='x', markersize=MPS_MARKER, label=labelName)

		# plt.xlim([tmin, tmax])
		# plt.ylim([math.ceil(min(up_)), math.ceil(max(uxMPS))])
		# plt.xlabel('x (m)')
		# plt.ylabel('u (m/s)')
		
		# naming the axis
		plt.xlabel(r'$\tilde{x}$')
		# naming the y axis
		plt.ylabel(r'$\tilde{u} (\tilde{x}=' + f'{(z_-w_/2.0):.2f}' + r',\tilde{y}=' + f'{(y_-h_/2.0):.2f}' + ')$')
		# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
		# Add text at the top of plot
		subTxtPos = [0.05, 0.90] # Text Position
		plt.text(subTxtPos[0], subTxtPos[1], "$Re=" + f'{Re_:.0f}' + "$", 
						transform = ax.transAxes, fontsize=BIGGER_SIZE)
		legend = plt.legend(loc=0, shadow=False, frameon=True)
		legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
		legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend

		fig = plt.gcf() # get current figure
		plt.show()

		if saveFig:
			file_name = 'poiseuille_velocity_space_LxHxW_' + f'{l_*1000:.0f}' + '_' + f'{h_*1000:.0f}' + '_' + f'{w_*1000:.0f}' + '_dP_' + f'{delta_press:.0f}' + '_lo' + lo_ + '_ANALYTIC_MPS_OPENFOAM_' + flow_regime
			fig.savefig(file_name + '.png')

		plt.close(fig)    # close the figure window