# Laminar flow through a pipe of uniform circular cross-section with an oscillating pressure gradient
# https://en.wikipedia.org/wiki/Hagen%E2%80%93Poiseuille_equation
# Sexl, 1930. Über den von EG Richardson entdeckten 'Annulareffekt

# Import libraries
import numpy as np
import matplotlib.pyplot as plt
import csv
import math
import scipy.special as sc

from fig_config import (
	add_grid,
	figure_features,
)	# <--- import customized functions

# Input dataset
# Plot figure
plotFigure = 1 # NO=0, YES=1
saveFig = int(input("Save figure ? (0 or 1), where 0=NO, 1=YES: "))
# saveTxt = int(input("Save txt file ? (0 or 1), where 0=NO, 1=YES: "))
# saveFig = 1 # NO=0, YES=1, save figure
# saveTxt = 1 # NO=0, YES=1, save txt file

# Physical
rho = 1000.0		# fluid density (kg/m3)
nu = 1.0e-4			# kinematic viscosity (m2/s) nu = mu / rho
# Pipe dimensions length x radius (m x m)
l_ = 0.15
R_ = 0.05
# MPS deviation
RDev_ = 0.003
R_ = R_ + RDev_
# Point to calculate the velocity 0 ≤ r ≤ R_
r_ = 0.0
# Sum terms
sum_terms = 20


plt.style.use('seaborn-whitegrid')
# seaborn-whitegrid
# seaborn-deep
# seaborn-dark-palette
# seaborn-colorblind
# fivethirtyeight
# ggplot
# seaborn
# seaborn-pastel
# seaborn-white
# tableau-colorblind10
# Solarize_Light2

# Font sizes
SMALL_SIZE = 12
MEDIUM_SIZE = 14
BIGGER_SIZE = 18
ANALYTIC_LINE = 1.0
FOAM_LINE = 1.5
MPS_LINE = 1.5
ANALYTIC_MARKER = 1.5
FOAM_MARKER = 3.0
MPS_MARKER = 3.0

plt.rcParams["font.family"] = "Times New Roman"
plt.rcParams['text.usetex'] = True
plt.rc('font', size=BIGGER_SIZE)          # controls default text sizes
plt.rc('axes', titlesize=BIGGER_SIZE)     # fontsize of the axes title
plt.rc('axes', labelsize=BIGGER_SIZE)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('ytick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('legend', fontsize=SMALL_SIZE)    # legend fontsize
plt.rc('figure', titlesize=BIGGER_SIZE)  # fontsize of the figure title

def readCsvMps(fileName, listTime, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listTime.append(row['Time'])
			listVel.append(row['avg(Velocity(0))'])
	return listTime, listVel

def readCsvFoam(fileName, listTime, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listTime.append(row['Time'])
			listVel.append(row['avg(U(2))'])
	return listTime, listVel

def readProbeFoam(fileName, listTime, listVel):
	"""
	This function read Probe file 
	and fill the list
	"""
	with open(fileName) as file:
		"""skip 3 lines"""
		for _ in range(3):
			next(file)
		"""read lines"""
		while (line := file.readline().rstrip()):
			"""split values separated by spaces"""
			sp = line.split()
			listTime.append(sp[0])
			"""split values separated by ("""
			s1 = sp[1].split("(")
			s2 = sp[2]
			s3 = sp[3].split(")")
			listVel.append(s3[0])
			# print(sp[0], s3[0])
	return listTime, listVel

if __name__ == '__main__':

	delta_press = 0.0;
	# reading a value, converting to float
	# dp_ = float(input("Enter the pressure amplitude: "))
	dp_ = 700
	print('A: %.2f' % dp_)
	# per_ = float(input("Enter the period: "))
	per_ = 2.0
	freq_ = 1.0 / per_
	omega_ = freq_ * 2.0 * np.pi 
	print('Omega (frequency): %.2f' % omega_)
	# nper_ = int(input("Enter the number of periods: "))
	nper_ = 25
	tf_ = nper_ * per_
	print('tf: %.2f' % tf_)


	# Particle distance
	# reading a value, converting to int
	# lo_id = int(input("Enter the particle distance (1 or 2), where 1: 0.010m, 2: 0.005m "))
	lo_id = 2
	lo_ = '0p0100'
	lo_num = 0.01
	if lo_id == 2:
		lo_ = '0p0050'
		lo_num = 0.005
	print('lo: %.5f' % lo_num)

	# Time array
	k_ = int(nper_ * 50)
	t_ = np.linspace(0, tf_, k_)	# Time array

	# Ctes
	G = delta_press / l_
	mu = nu * rho				# dynamic viscosity [kg / (m * s)] mu = nu * rho

	# DP/Dx = G - alpha * cos(omega * t) - beta * sin(omega * t)
	alpha_ = 0.0
	beta_  = dp_ / l_

	# Velocity 1st term
	u1 = G * (R_ ** 2 - r_ ** 2) / (4.0 * mu)

	# Velocity 2nd term
	k1_ = math.sqrt(rho * omega_ / mu)
	kr  = k1_ * r_
	kR_ = k1_ * R_

	# bei and ber are Kelvin functions
	F1 = (sc.ber(kr) * sc.ber(kR_) + sc.bei(kr) * sc.bei(kR_) ) / (sc.ber(kR_) ** 2 + sc.bei(kR_) ** 2)
	F2 = (sc.ber(kr) * sc.bei(kR_) - sc.bei(kr) * sc.ber(kR_) ) / (sc.ber(kR_) ** 2 + sc.bei(kR_) ** 2)

	u2 = ((alpha_ * F2 + beta_ * (F1 - 1.0)) * np.cos(omega_ * t_) + (beta_ * F2 - alpha_ * (F1 - 1.0)) * np.sin(omega_ * t_)) / (rho * omega_)

	# Velocity
	uxANAL = u1 + u2

	print('\nCircular pipe')
	print('Velocity 1st term: %.5f' % u1)
	print('Max Velocity 2nd term: %.5f' % max(u2))
	print('Max Velocity: %.5f' % max(uxANAL))

	# Hydraulic diameter
	Dh_ = 2.0 * R_
	# Reynolds number
	umax = max(uxANAL)
	uavg = 0.5 * umax
	Re_ = uavg * Dh_ / nu
	flow_regime = 'Laminar'
	if Re_ < 2300:
		print('Reynolds: %.2f < 2300 - %s flow' % (Re_, flow_regime))
	elif Re_ > 2900:
		flow_regime = 'Turbulent'
		print('Reynolds: %.2f > 2900 - %s flow' % (Re_, flow_regime))
	else:
		flow_regime = 'Intermittent'
		print('Reynolds: %.2f between 2300-2900 - %s flow' % (Re_, flow_regime))

	
	# # Plotting the analytical points 
	# plt.plot(t_, uxANAL, color='green', linestyle='dashed', linewidth = 2)

	# # naming the x axis
	# plt.xlabel(r'$t/T$')
	# # naming the y axis
	# plt.ylabel(r'$u (r=' + f'{r_:.2f}' + ')$ (m/s)')
	# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
	# #plt.title(r'\TeX\ is Number $\displaystyle\sum_{n=1}^\infty \frac{-e^{i\pi}}{2^n}$!', fontsize=16, color='r')
	# # legend = plt.legend(loc=0, shadow=False, frameon=True)
	# # legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
	# # legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend
	# fig = plt.gcf() # get current figure
	# file_name = 'circular_sin_velocity_R_' + f'{R_*1000:.0f}' + '_dP_' + f'{dp_:.0f}' + '_' + flow_regime
	# if saveFig:
	#     fig.savefig(file_name + '.png')
	# plt.show()
	# plt.close(fig)    # close the figure window


	## Analytical x Numerical

	# csv MPS file name
	filenameMPS = "InOutCircle_lo" + lo_ + "_Sin_DP" + f'{dp_:.0f}' + "-T2p0s-" + f'{tf_:.0f}' + "s-vel.txt"
	tMPS = []
	uxMPS = []
	readCsvMps(filenameMPS, tMPS, uxMPS)
	# Convert list to array
	tMPSa = np.array(tMPS)
	uxMPSa = np.array(uxMPS)
	# Convert to float
	tMPS  =  tMPSa.astype(np.float64)
	uxMPS = uxMPSa.astype(np.float64)

	# csv OpenFOAM file name
	filenameFOAM = "circle-tube-sine-DP" + f'{dp_:.0f}' + "-vel.txt"
	tFOAM = []
	uxFOAM = []
	# readCsvFoam(filenameFOAM, tFOAM, uxFOAM)
	readProbeFoam(filenameFOAM, tFOAM, uxFOAM)
	# Convert list to array
	tFOAMa = np.array(tFOAM)
	uxFOAMa = np.array(uxFOAM)
	# Convert to float
	tFOAM  =  tFOAMa.astype(np.float64)
	uxFOAM = uxFOAMa.astype(np.float64)

	## FFT
	xMPS = uxMPS - np.mean(uxMPS) # make sure the average is zero, so we don't get a huge DC offset.
	dt = tMPS[2] - tMPS[1] #[s] 1/the sampling rate
	fftx = np.fft.fft(xMPS) # the frequency transformed part
	# now discard anything  that we do not need..
	fftx = fftx[range(int(len(fftx)/2))]
	# now create the frequency axis: it runs from 0 to the sampling rate /2
	freq_fftx = np.linspace(0,2/dt,len(fftx))
	# Get array index of the maximum value
	maxIndexFFTX = np.argmax(abs(fftx)**2)
	perMPS = freq_fftx[maxIndexFFTX]
	print("Period MPS:", f'{perMPS:.4f}')
	# and plot a power spectrum
	# plt.plot(freq_fftx,abs(fftx)**2)
	# plt.show()

	## Compute error
	# # Get sorted array index
	# sortIndexMPS = np.argsort(tMPS)
	# # Rearrange array based upon index array
	# tMPS = tMPS[sortIndexMPS]
	# uxMPS = uxMPS[sortIndexMPS]

	# Interpolate to k_ points
	xvals = np.linspace(min(t_), max(t_), k_)
	# u_new = np.interp(xvals, t_, uxANAL)
	uxMPSnew = np.interp(xvals, tMPS, uxMPS)
	uxFOAMnew = np.interp(xvals, tFOAM, uxFOAM)

	# Last 4 periods
	k2 = round(0.5*k_)
	k4 = round((nper_-4) * 50)
	# print(nper_)
	# print((nper_-4) * 50)
	# print(k4)

	# Calculate the difference between the estimated and the actual value using numpy.subtract() function.
	# Further, calculate the square of the above results using numpy.square() function.
	# Finally, calculate the mean of the squared value using numpy.mean() function. The output is the MSE score.
	# MSE = np.square(np.subtract(uxANAnew,uxMPSnew)).mean()
	MSE = np.square(np.subtract(uxANAL[k4:-1],uxMPSnew[k4:-1])).mean()
	# At the end, calculate the square root of MSE using math.sqrt() function to get the RMSE value.
	RMSE = math.sqrt(MSE)
	print("RMSE MPS-ANA:", f'{RMSE:.4f}')

	MSE = np.square(np.subtract(uxMPSnew[k4:-1],uxFOAMnew[k4:-1])).mean()
	RMSE = math.sqrt(MSE)
	print("RMSE MPS-FOAM:", f'{RMSE:.4f}')

	# MSE = np.square(np.subtract(uxANAnew,uxFOAMnew)).mean()
	MSE = np.square(np.subtract(uxANAL[k4:-1],uxFOAMnew[k4:-1])).mean()
	RMSE = math.sqrt(MSE)
	print("RMSE FOAM-ANA:", f'{RMSE:.4f}')

	# Err = abs(max(uxANAnew) - max(uxMPSnew)) / max(uxANAnew) * 100
	Err = abs(max(uxANAL) - max(uxMPSnew[k4:-1])) / max(uxANAL) * 100
	print("Error MPS-ANA:", f'{Err:.2f}')
	Err = abs(max(uxFOAMnew[k4:-1]) - max(uxMPSnew[k4:-1])) / max(uxFOAMnew[k4:-1]) * 100
	print("Error MPS-FOAM:", f'{Err:.2f}')
	# Err = abs(max(uxANAnew) - max(uxFOAMnew)) / max(uxANAnew) * 100
	Err = abs(max(uxANAL) - max(uxFOAMnew[k4:-1])) / max(uxANAL) * 100
	print("Error FOAM-ANA:", f'{Err:.2f}')

	# Plot and save figure
	if plotFigure:

		t_til = [x / per_ for x in t_]
		tFOAM_til = [x / per_ for x in tFOAM]
		tMPS_til = [x / per_ for x in tMPS]

		ymin = round(min(uxANAL), 1)
		ymax = round(max(uxMPS), 1)
		if max(uxMPS) > 1:
			ymin = math.ceil(min(uxANAL)-1.0)
			ymax = math.ceil(max(uxMPS))
		
		figure_features()	# <--- Add this line to every figure
		ax = plt.axes(xlim = (0, max(t_til)), ylim = (ymin, ymax))
		add_grid(ax, locations=(per_*0.5, per_, 0.5, 1))	# <--- Add this line to every figure

		# plotting Analytic velocity
		labelName = "Analytical (Sexl, 1930)"
		plt.plot(t_til, uxANAL, color='black', linestyle='dashed', linewidth=ANALYTIC_LINE, label=labelName)

		# Interpolate to N points
		tvals = np.linspace(min(t_til), max(t_til), int(k_/2))
		# tvals = np.linspace(min(t_), max(t_), len(uxMPS))
		uxFOAMk2 = np.interp(tvals, tFOAM_til, uxFOAM)
		uxMPSk2 = np.interp(tvals, tMPS_til, uxMPS)

		labelName = "OpenFOAM"
		# plotting FOAM velocity
		# plt.plot(tFOAM_til, uxFOAM, color='orange', linestyle='none', marker='o', markerfacecolor='none', markersize=FOAM_MARKER, label=labelName)
		plt.plot(tvals, uxFOAMk2, color='orange', linestyle='none', marker='o', markerfacecolor='none', markersize=FOAM_MARKER, label=labelName)

		labelName = "MPS"
		# plotting MPS velocity
		# plt.plot(tMPS_til, uxMPS, color='blue', linestyle='none', marker='x', markersize=MPS_MARKER, label=labelName)
		plt.plot(tvals, uxMPSk2, color='blue', linestyle='none', marker='x', markersize=MPS_MARKER, label=labelName)

		plt.xlabel(r'$t/T$')
		# plt.ylabel(r'$\tilde{u} (\tilde{r}=' + f'{r_:.2f}' + ')$')
		plt.ylabel(r'$u (r=' + f'{r_:.2f}' + ',t)$ (m/s)')
		# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
		# Add text at the top of plot
		subTxtPos = [0.05, 0.90] # Text Position
		plt.text(subTxtPos[0], subTxtPos[1], "$Re=" + f'{Re_:.0f}' + "$", 
						transform = ax.transAxes, fontsize=BIGGER_SIZE)
		legend = plt.legend(loc=0, shadow=False, frameon=True)
		legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
		legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend

		fig = plt.gcf() # get current figure
		plt.show()
		
		if saveFig:
			file_name = 'circular_sine_press_R_' + f'{R_*1000:.0f}' + '_dP_' + f'{dp_:.0f}' + '_lo' + lo_ + '_ANALYTIC_MPS_OPENFOAM_' + flow_regime
			fig.savefig(file_name + '.png')
		
		plt.close(fig)    # close the figure window