# Laminar flow through a pipe of uniform circular cross-section (Hagen–Poiseuille flow)
# https://en.wikipedia.org/wiki/Hagen%E2%80%93Poiseuille_equation
# Stokes, 1845. On the theories of the internal friction of fluids in motion, and of 
# the equilibrium and motion of elastic solids

# Import libraries
import numpy as np
import matplotlib.pyplot as plt
import csv
import math
import scipy.special as sc

from fig_config import (
	add_grid,
	figure_features,
)	# <--- import customized functions

# Input dataset
# Plot figure
plotFigure = 1 # NO=0, YES=1
saveFig = int(input("Save figure ? (0 or 1), where 0=NO, 1=YES: "))
saveTxt = int(input("Save txt file ? (0 or 1), where 0=NO, 1=YES: "))
# saveFig = 1 # NO=0, YES=1, save figure
# saveTxt = 1 # NO=0, YES=1, save txt file

# Physical
rho = 1000.0		# fluid density (kg/m3)
nu = 1.0e-4			# kinematic viscosity (m2/s) nu = mu / rho
# Pipe dimensions length x radius (m x m)
R_ = 0.05
l_ = 0.15
# l_ = 40.0 * R_
# MPS deviation
RDev_ = 0.003
R_ = R_ + RDev_
# Point to calculate the velocity 0 ≤ r ≤ R_
r_ = 0.0
# Sum terms
sum_terms = 20

plt.style.use('seaborn-whitegrid')
# seaborn-whitegrid
# seaborn-deep
# seaborn-dark-palette
# seaborn-colorblind
# fivethirtyeight
# ggplot
# seaborn
# seaborn-pastel
# seaborn-white
# tableau-colorblind10
# Solarize_Light2

# Font sizes
SMALL_SIZE = 12
MEDIUM_SIZE = 14
BIGGER_SIZE = 18
ANALYTIC_LINE = 1.0
FOAM_LINE = 1.0
MPS_LINE = 1.5
ANALYTIC_MARKER = 1.5
FOAM_MARKER = 6.0
MPS_MARKER = 7.0

plt.rcParams["font.family"] = "Times New Roman"
plt.rcParams['text.usetex'] = True
plt.rc('font', size=BIGGER_SIZE)          # controls default text sizes
plt.rc('axes', titlesize=BIGGER_SIZE)     # fontsize of the axes title
plt.rc('axes', labelsize=BIGGER_SIZE)    # fontsize of the x and y labels
plt.rc('xtick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('ytick', labelsize=MEDIUM_SIZE)    # fontsize of the tick labels
plt.rc('legend', fontsize=SMALL_SIZE)    # legend fontsize
plt.rc('figure', titlesize=BIGGER_SIZE)  # fontsize of the figure title
# Enable LaTeX rendering
plt.rc('font', family='serif')
plt.rc('text', usetex=True)

def readCsvMps(fileName, listPos, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listPos.append(row['Points:1'])
			listVel.append(row['Velocity:0'])
	return listPos, listVel

def readCsvFoam(fileName, listPos, listVel):
	"""
	This function read CSV file 
	and fill the list
	"""
	with open(fileName, 'r+') as f:
		reader = csv.DictReader(f, delimiter=',')
		for row in reader:
			listPos.append(row['Points:0'])
			listVel.append(row['U:2'])
	return listPos, listVel

if __name__ == '__main__':

	# reading a value, converting to float
	delta_press = float(input("Enter the delta press (30 or 50): "))
	print('dP: %.4f' % delta_press)

	# Particle distance
	# reading a value, converting to int
	# lo_id = int(input("Enter the particle distance (1 or 2), where 1: 0.010m, 2: 0.005m "))
	lo_id = 2
	lo_ = '0p0100'
	lo_num = 0.01
	if lo_id == 2:
		lo_ = '0p0050'
		lo_num = 0.005
	print('lo: %.5f' % lo_num)

	print('rho: %.3f nu: %.6f R: %.5f' % (rho, nu, R_))

	# https://en.wikipedia.org/wiki/Hagen%E2%80%93Poiseuille_equation

	# Ctes
	G = delta_press / l_
	mu = nu * rho				# dynamic viscosity [kg / (m * s)] mu = nu * rho

	tf_ = R_ ** 2 / nu 			# Non-dimensional time
	print('tf: %.2f' % tf_)
	k_ = 50
	t_ = np.linspace(0, 1.5 * tf_, k_)

	# Velocity empty array
	u_ = []

	# Velocity 1st term
	u1 = G * (R_ ** 2 - r_ ** 2) / (4.0 * mu)

	# Velocity 2nd term
	# Compute sum_terms roots of the Bessel function J0
	lambda_n = sc.jn_zeros(0, sum_terms)
	# print(lambda_n)

	c1 = 2.0 * G * R_ ** 2 / mu

	# for i in range(k_):

	# 	#Declare variable for sum
	# 	sum2nd = 0.0
	# 	for n_ in range(1, sum_terms):
	# 		ni_ = n_-1
	# 		ln = lambda_n[ni_]			# positive root of Bessel function of the first kind of order zero
	# 		J0 = sc.jv(0, ln * r_ / R_)	# Bessel function of the first kind of order zero
	# 		J1 = sc.jv(1, ln )			# Bessel function of the first kind of order one
	# 		sum2nd += 1.0 / (ln ** 3) * J0 / J1 * math.exp(- ln ** 2 * nu * t_[i] / R_ ** 2)

	# 	u2 = - c1 * sum2nd

	# 	# Velocity
	# 	u_.append(u1 + u2)

	print('Velocity 1st term: %.5f' % u1)
	# print('Velocity 2nd term: %.3f' % u2)
	# print('Velocity: %.3f' % u_)

	# Hydraulic diameter
	Dh_ = 2.0 * R_
	# Reynolds number
	umax = u1
	uavg = 0.5 * umax
	Re_ = uavg * Dh_ / nu
	flow_regime = 'Laminar'
	if Re_ < 2300:
		print('Reynolds: %.2f < 2300 - %s flow' % (Re_, flow_regime))
	elif Re_ > 2900:
		flow_regime = 'Turbulent'
		print('Reynolds: %.2f > 2900 - %s flow' % (Re_, flow_regime))
	else:
		flow_regime = 'Intermittent'
		print('Reynolds: %.2f between 2300-2900 - %s flow' % (Re_, flow_regime))


	# Plotting Velocity x Time
	file_name = 'analytical_poiseuille_velocity_time_R_' + f'{R_*1000:.0f}' + '_dP_' + f'{delta_press:.0f}' + '_lo' + lo_ + '_' + flow_regime
	
	# t_til = [x / tf_ for x in t_]
	# u_til = [x / umax for x in u_]

	# # plotting the points
	# plt.plot(t_til, u_til, color='green', linestyle='dashed', linewidth = 2)

	# # naming the x axis
	# plt.xlabel(r'$\tilde{t}$')
	# plt.ylabel(r'$\tilde{u} (\tilde{r}=' + f'{r_:.2f}' + ')$')
	# plt.title(r"Hagen–Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
	# #plt.title(r'\TeX\ is Number $\displaystyle\sum_{n=1}^\infty \frac{-e^{i\pi}}{2^n}$!', fontsize=16, color='r')
	# # legend = plt.legend(loc=0, shadow=False, frameon=True)
	# # legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
	# # legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend
	# fig = plt.gcf() # get current figure
	# file_name = 'poiseuille_velocity_time_R_' + f'{R_*1000:.0f}' + '_dP_' + f'{delta_press:.0f}' + '_lo' + lo_ + '_' + flow_regime
	# if saveFig:
	#     fig.savefig(file_name + '.png')
	# plt.show()
	# plt.close(fig)    # close the figure window

	## Parabolic flow
	# Linearly spaced vector of k points
	k = 40
	rp_ = np.linspace(-R_, R_, k)

	# empty array
	up_ = []

	for i in range(k):
		u1 = G * (R_ ** 2 - rp_[i] ** 2) / (4.0 * mu)
		# Velocity
		up_.append(u1)

	rp_til = [x / R_ for x in rp_]
	up_til = [x / umax for x in up_]

	if saveTxt:
		# Write and save txt file
		data = np.column_stack([rp_, up_])
		datafile = file_name + ".txt"

		with open(datafile, 'wb') as f:
			np.savetxt(f, data, fmt=['%.5f','%.5f'], delimiter=',', comments='', header='"x(m)","u(m/s)"')


	## Analytical x Numerical

	# csv MPS file name
	filenameMPS = "mps-InOutCircle-Box-" + 'lo' + lo_ + "-DP" + f'{delta_press:.0f}' + ".txt"
	rMPS  = []
	uxMPS = []
	readCsvMps(filenameMPS, rMPS, uxMPS)
	# Convert list to array
	rMPSa  = np.array(rMPS)
	uxMPSa = np.array(uxMPS)
	# Convert to float
	rMPS  =  rMPSa.astype(np.float64) - (R_ - RDev_)
	uxMPS = uxMPSa.astype(np.float64)

	# csv OpenFOAM file name
	filenameFOAM = "openFoam-circle-tube-DP" + f'{delta_press:.0f}' + ".txt"
	rFOAM  = []
	uxFOAM = []
	readCsvFoam(filenameFOAM, rFOAM, uxFOAM)
	# Convert list to array
	rFOAMa  = np.array(rFOAM)
	uxFOAMa = np.array(uxFOAM)
	# Convert to float
	rFOAM  =  rFOAMa.astype(np.float64)
	uxFOAM = uxFOAMa.astype(np.float64)

	## Compute error
	# Get sorted array index
	sortIndexMPS = np.argsort(rMPS)
	# Rearrange array based upon index array
	rMPS  = rMPS[sortIndexMPS]
	uxMPS = uxMPS[sortIndexMPS]

	# Get sorted array index
	sortIndexFOAM = np.argsort(rFOAM)
	# Rearrange array based upon index array
	rFOAM  = rFOAM[sortIndexFOAM]
	uxFOAM = uxFOAM[sortIndexFOAM]

	# print(min(rMPS), max(rMPS))
	# print(min(rp_), max(rp_))
	# print(min(rFOAM), max(rFOAM))

	# Interpolate to 50 points
	xvalsMPS  = np.linspace(min(rMPS), max(rMPS), 50)
	uxANAnew  = np.interp(xvalsMPS, rp_, up_)
	uxMPSnew  = np.interp(xvalsMPS, rMPS, uxMPS)
	uxFOAMnew = np.interp(xvalsMPS, rFOAM, uxFOAM)

	# Calculate the difference between the estimated and the actual value using numpy.subtract() function.
	# Further, calculate the square of the above results using numpy.square() function.
	# Finally, calculate the mean of the squared value using numpy.mean() function. The output is the MSE score.
	MSE = np.square(np.subtract(uxANAnew,uxMPSnew)).mean()
	# At the end, calculate the square root of MSE using math.sqrt() function to get the RMSE value.
	RMSE = math.sqrt(MSE) * 100
	print("RMSE MPS-ANA:", f'{RMSE:.4f}')

	MSE = np.square(np.subtract(uxFOAMnew,uxMPSnew)).mean()
	RMSE = math.sqrt(MSE) * 100
	print("RMSE MPS-FOAM:", f'{RMSE:.4f}')
	
	MSE = np.square(np.subtract(uxANAnew,uxFOAMnew)).mean()
	RMSE = math.sqrt(MSE) * 100
	print("RMSE FOAM-ANA:", f'{RMSE:.4f}')


	Err = abs(max(uxANAnew) - max(uxMPSnew)) / max(uxANAnew) * 100
	print("Error MPS-ANA:", f'{Err:.2f}')
	Err = abs(max(uxFOAMnew) - max(uxMPSnew)) / max(uxFOAMnew) * 100
	print("Error MPS-FOAM:", f'{Err:.2f}')
	Err = abs(max(uxANAnew) - max(uxFOAMnew)) / max(uxANAnew) * 100
	print("Error FOAM-ANA:", f'{Err:.2f}')


	# figure_features()	# <--- Add this line to every figure
	# ax = plt.axes(xlim=(-1.2, 1.2), ylim=(0, 1.2))
	# add_grid(ax, locations=(0.25, 0.5, 0.1, 0.2))	# <--- Add this line to every figure

	# labelName = "Analytical (Stokes, 1845)"
	# plt.plot(xvalsMPS, uxANAnew, color='black', linestyle='dashed', linewidth=ANALYTIC_LINE, label=labelName)
	# labelName = "OpenFOAM"
	# plt.plot(xvalsMPS, uxFOAMnew, color='orange', linestyle='none', marker='o', markerfacecolor='none', markersize=FOAM_MARKER, label=labelName)
	# labelName = "MPS"
	# plt.plot(xvalsMPS, uxMPSnew, color='blue', linestyle='none', marker='x', markersize=MPS_MARKER, label=labelName)
	# plt.show()
	

	# Plot and save figure
	if plotFigure:

		figure_features()	# <--- Add this line to every figure
		ax = plt.axes(xlim=(-1.2, 1.2), ylim=(0, 1.2))
		add_grid(ax, locations=(0.25, 0.5, 0.1, 0.2))	# <--- Add this line to every figure

		rp_til = [x / R_ for x in rp_]
		up_til = [x / umax for x in up_]
		# plotting Analytic velocity
		labelName = "Analytical (Stokes, 1845)"
		plt.plot(rp_til, up_til, color='black', linestyle='dashed', linewidth=ANALYTIC_LINE, label=labelName)

		rFOAM_til  = [x / R_ for x in rFOAM]
		uxFOAM_til = [x / umax for x in uxFOAM]
		# plotting FOAM velocity
		labelName = "OpenFOAM"
		plt.plot(rFOAM_til, uxFOAM_til, color='orange', linestyle='none', marker='o', markerfacecolor='none', markersize=FOAM_MARKER, label=labelName)

		rMPS_til = [x / R_ for x in rMPS]
		uxMPS_til = [x / umax for x in uxMPS]
		# plotting MPS velocity
		labelName = "MPS"
		plt.plot(rMPS_til, uxMPS_til, color='blue', linestyle='none', marker='x', markersize=MPS_MARKER, label=labelName)

		plt.xlabel(r'$\tilde{r}$')
		plt.ylabel(r'$\tilde{u} (\tilde{r}=' + f'{r_:.2f}' + ')$')
		# plt.title(r"Poiseuille velocity - $Re=" + f'{Re_:.0f}' + "$") # displaying the title
		# Add text at the top of plot
		subTxtPos = [0.05, 0.90] # Text Position
		plt.text(subTxtPos[0], subTxtPos[1], "$Re=" + f'{Re_:.0f}' + "$", 
						transform = ax.transAxes, fontsize=BIGGER_SIZE)
		legend = plt.legend(loc=0, shadow=False, frameon=True)
		legend.get_frame().set_facecolor('white') # Put a nicer background color on the legend
		legend.get_frame().set_linewidth(0.0) # Remove only the border of the box of the legend

		# Plot figure
		fig = plt.gcf() # get current figure
		plt.show()
		
		if saveFig:
			file_name = 'poiseuille_velocity_space_R_' + f'{R_*1000:.0f}' + '_dP_' + f'{delta_press:.0f}' + '_lo' + lo_ + '_ANALYTIC_MPS_OPENFOAM_' + flow_regime
			fig.savefig(file_name + '.png')
		plt.close(fig)	# close the figure window