
rm(list=ls())

d1 <- read.csv("LastheniaEstimatedModelCoefficients.csv")
d2 <- read.csv("LatheniaIndividuals.csv")

# merging the model coefficients for Lasthenia with the information on the phenology of Lasthenia individuals from the original experiment to create a new dataframe that gives the coefficients for each individual according to their observed phenology.
l.dat <- merge(d1,d2)

# assign estimated values for each of the species' other demographic and competitive rates
bromus.lambda <- 1710.203708
lactuca.lambda <- 23316.1771

lasthenia.germination <- 0.07
bromus.germination <- 0.35
lactuca.germination <- 0.12

lasthenia.seed.survival <- 0.2
bromus.seed.survival <- 0.02
lactuca.seed.survival <- 0.43

alpha.bromus.bromus <- 0.063342321
alpha.lactuca.lactuca <- 0.007610788

alpha.bromus.lasthenia <- 0.008619542
alpha.lactuca.lasthenia <- 0.000807498


# simulation function, which accounts for correlations between intra and interspecific alphas.
Lasthenia.Comp <- function(with.var = TRUE,
                           competitor,
                           lasthenia.invasion = TRUE,
                           lasthenia.comp.rates,
                           initial.invader = 10^-8,
                           g_l=lasthenia.germination,
                           s_l=lasthenia.seed.survival, 
                           l_2,
                           g_2,
                           s_2,
                           alpha_22,
                           alpha_2l,
                           time.steps=iterations){

  if (competitor=="bromus") {dat <- lasthenia.comp.rates[lasthenia.comp.rates$competitor=="bromus",]} else if ((competitor=="lactuca")) {dat <- lasthenia.comp.rates[lasthenia.comp.rates$competitor=="lactuca",]}
  
  
  # are you running the function with or without variation? If with, then alphas take an empirical distribution of possible values, otherwise the alphas take the mean of the distribution of values.
  if (with.var == TRUE) {rates <- dat[,c("lambda","intra","inter")]}
  else {rates <- cbind.data.frame(rep(mean(dat$lambda),nrow(dat)),rep(mean(dat$intra),nrow(dat)),rep(mean(dat$inter),nrow(dat)))}
  names(rates)<-c("lambda","intra","inter")

  # does the simulation track the invasion of Lasthenia, or the invasion of the competitor? If Lasthenia is invading, then the other species takes it's equilibrium carrying capacity, otherwise the other species takes the initial invader density. Remember, that there is no variation in the intraspecific competition coefficient of the species, other than for Lasthenia.
  if (lasthenia.invasion == TRUE) {N_2 <- (((g_2 * l_2) / (1 - s_2 * (1 - g_2))) - 1) / (alpha_22 * g_2)
  }else{N_2<-initial.invader}

  # if Lasthenia is invading, it takes initial invader population size, else it takes it's carrying capacity, according to either variable intraspecific competition coefficients, or mean competition coefficient. The carrying capacity is determined by simulation over 20 time steps.
  if (lasthenia.invasion == TRUE) {N_l <- initial.invader} else {
    N_l=1000
    for (jj in 1:20){
      
      samp <- sample(1:nrow(rates),size = 1000000,replace = T)
      
      N_l <- N_l * mean((rates$lambda[samp] * g_l) * (1 / (1 + N_l * g_l * rates$intra[samp])) + ((1-g_l) * s_l))
      jj = jj+1
    }}

  # empty vectors to hold the population trajectories.
  (N_2<-c(N_2,rep(NA,time.steps)))
  (N_l<-c(N_l,rep(NA,time.steps)))

  # run the simulation
  for (ii in 1:time.steps){

    samp <- sample(1:nrow(rates),size = 1000000,replace = T)
    
    # getting the mean growth rate assuming that the population takes a distribution of values for alpha.
    hold_l <- N_l[ii] * 
      mean(
        (rates$lambda[samp] * g_l) * 
          (1 / (1 + N_l[ii] * g_l * rates$intra[samp] + 
                  N_2[ii] * g_2 * rates$inter[samp])) + 
          ((1-g_l) * s_l)
        )

    hold_2 <- N_2[ii] * (l_2 * g_2 / (1 + N_2[ii] * g_2 * alpha_22 + N_l[ii] * g_l * alpha_2l) + (1-g_2)*s_2)

    N_l[ii+1]<-hold_l
    N_2[ii+1]<-hold_2
    cat("proportion done", ii/time.steps,"\n")
  }
  out<-cbind(N_l,N_2)
  return(out)
}

iterations = 50

# Simulate competitive population dynamics for lasthenia vs. bromus; bromus resident
# with variation
phenol.with.var_lasthenia.inv.against.bromus.cor <- 
  Lasthenia.Comp(with.var = TRUE,
                           competitor="bromus",
                           lasthenia.invasion = TRUE,          
                           lasthenia.comp.rates=l.dat,
                           initial.invader = 10^-8,
                           g_l=lasthenia.germination,
                           s_l=lasthenia.seed.survival, 
                           l_2=bromus.lambda,
                           g_2=bromus.germination,
                           s_2=bromus.seed.survival,
                           alpha_22=alpha.bromus.bromus,
                           alpha_2l=alpha.bromus.lasthenia,
                           time.steps=iterations)

# simulate dynamics without intraspecific variation
phenol.NO.var_lasthenia.inv.against.bromus.cor <- 
  Lasthenia.Comp(with.var = FALSE,
                 competitor="bromus",
                 lasthenia.invasion = TRUE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=bromus.lambda,
                 g_2=bromus.germination,
                 s_2=bromus.seed.survival,
                 alpha_22=alpha.bromus.bromus,
                 alpha_2l=alpha.bromus.lasthenia,
                 time.steps=iterations)
 

# lasthenia vs. bromus; lasthenia resident
phenol.with.var_bromus.inv.against.lasthenia.cor <- 
  Lasthenia.Comp(with.var = TRUE,
                 competitor="bromus",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=bromus.lambda,
                 g_2=bromus.germination,
                 s_2=bromus.seed.survival,
                 alpha_22=alpha.bromus.bromus,
                 alpha_2l=alpha.bromus.lasthenia,
                 time.steps=iterations)

phenol.NO.var_bromus.inv.against.lasthenia.cor <- 
  Lasthenia.Comp(with.var = FALSE,
                 competitor="bromus",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=bromus.lambda,
                 g_2=bromus.germination,
                 s_2=bromus.seed.survival,
                 alpha_22=alpha.bromus.bromus,
                 alpha_2l=alpha.bromus.lasthenia,
                 time.steps=iterations)

# lasthenia vs. lactuca; lactuca resident
phenol.with.var_lasthenia.inv.against.lactuca.cor <- 
  Lasthenia.Comp(with.var = TRUE,
                 competitor="lactuca",
                 lasthenia.invasion = TRUE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)

phenol.NO.var_lasthenia.inv.against.lactuca.cor <- 
  Lasthenia.Comp(with.var = FALSE,
                 competitor="lactuca",
                 lasthenia.invasion = TRUE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)


# lasthenia vs. lactuca; lasthenia resident
phenol.with.var_lactuca.inv.against.lasthenia.cor <- 
  Lasthenia.Comp(with.var = TRUE,
                 competitor="lactuca",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)

phenol.NO.var_lactuca.inv.against.lasthenia.cor <- 
  Lasthenia.Comp(with.var = FALSE,
                 competitor="lactuca",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)

#### instantaneous invasion growth rates ####
log(phenol.NO.var_lasthenia.inv.against.bromus.cor[2,1]/phenol.NO.var_lasthenia.inv.against.bromus.cor[1,1])
log(phenol.with.var_lasthenia.inv.against.bromus.cor[2,1]/phenol.with.var_lasthenia.inv.against.bromus.cor[1,1])

log(phenol.NO.var_lasthenia.inv.against.lactuca.cor[2,1]/phenol.NO.var_lasthenia.inv.against.lactuca.cor[1,1])
log(phenol.with.var_lasthenia.inv.against.lactuca.cor[2,1]/phenol.with.var_lasthenia.inv.against.lactuca.cor[1,1])



#### Figures ####
(Lasthenia.col = "#33a02c")
(bromus.col = "#a6cee3")
(lactuca.col = "#1f78b4")
(black = "#000000")

pdf(file="Invasion_Simulations_Figure.pdf",width=8.5,height=5)

par(mfrow=c(1,2))
par(mar=c(c(5, 4, 4, 1) + 0.1))

plot(0:iterations,phenol.with.var_lactuca.inv.against.lasthenia.cor[,2]/1000,ylim=c(0,(max(c(as.vector(phenol.with.var_lactuca.inv.against.lasthenia.cor),as.vector(phenol.with.var_lactuca.inv.against.lasthenia.cor)))+1000)/1000),xlab="",ylab="",type='l',lwd=3,main="",col=lactuca.col)
lines(0:iterations,phenol.with.var_lactuca.inv.against.lasthenia.cor[,1]/1000,col=Lasthenia.col,lwd=3)
lines(0:iterations,phenol.NO.var_lactuca.inv.against.lasthenia.cor[,2]/1000,lwd=3,lty="21",col=lactuca.col)
lines(0:iterations,phenol.NO.var_lactuca.inv.against.lasthenia.cor[,1]/1000,col=Lasthenia.col,lwd=3,lty="21")
legend("topright",legend = c("with variation","NO variation","Lasthenia","Lactuca","Bromus"),lty=c("solid","21","1919","1919","1919"),lwd=c(3,3,0,0,0),pch=c(NA,NA,15,15,15),col=c(black,black,Lasthenia.col,lactuca.col,bromus.col),y.intersp = 2,bty = "n",pt.cex=2,cex=1)

# x-axis
mtext(text=expression(paste("years")),side=1,line=3,cex=1.2)
# y-axis
mtext(text=expression(paste("population size ('000s)")),side=2,line=2.5,cex=1.2)
# plot label
mtext(text = "a.",cex=1.2,side = 3,adj = 0,line = 0.5)

par(mar=c(c(5, 2, 4, 3) + 0.1))
plot(0:iterations,phenol.with.var_bromus.inv.against.lasthenia.cor[,2]/1000,ylim=c(0,(max(c(as.vector(phenol.with.var_bromus.inv.against.lasthenia.cor),as.vector(phenol.with.var_bromus.inv.against.lasthenia.cor)))+1000)/1000),xlab="",ylab="",type='l',lwd=3,main="",col=bromus.col)
lines(0:iterations,phenol.with.var_bromus.inv.against.lasthenia.cor[,1]/1000,col=Lasthenia.col,lwd=3)
lines(0:iterations,phenol.NO.var_bromus.inv.against.lasthenia.cor[,2]/1000,lwd=3,lty="21",col=bromus.col)
lines(0:iterations,phenol.NO.var_bromus.inv.against.lasthenia.cor[,1]/1000,col=Lasthenia.col,lwd=3,lty="21")

mtext(text=expression(paste("years")),side=1,line=3,cex=1.2)
# plot label
mtext(text = "b.",cex=1.2,side = 3,adj = 0,line = 0.5)

dev.off()


# remove covariance
Lasthenia.Comp.NOcov <- function(with.var = TRUE,
                           competitor,
                           lasthenia.invasion = TRUE,
                           lasthenia.comp.rates,
                           initial.invader = 10^-8,
                           g_l=lasthenia.germination,
                           s_l=lasthenia.seed.survival, 
                           l_2,
                           g_2,
                           s_2,
                           alpha_22,
                           alpha_2l,
                           time.steps=iterations){
  
  if (competitor=="bromus") {dat <- lasthenia.comp.rates[lasthenia.comp.rates$competitor=="bromus",]} else if ((competitor=="lactuca")) {dat <- lasthenia.comp.rates[lasthenia.comp.rates$competitor=="lactuca",]}
  
  
  # are you running the function with or without variation? If with, then alphas take an empirical distribution of possible values, otherwise the alphas take the mean of the distribution of values.
  if (with.var == TRUE) {rates <- dat[,c("lambda","intra","inter")]}
  else {rates <- cbind.data.frame(rep(mean(dat$lambda),nrow(dat)),rep(mean(dat$intra),nrow(dat)),rep(mean(dat$inter),nrow(dat)))}
  names(rates)<-c("lambda","intra","inter")
  
  # does the simulation track the invasion of Lasthenia, or the invasion of the competitor? If Lasthenia is invading, then the other species takes it's equilibrium carrying capacity, otherwise the other species takes the initial invader density. Remember, that there is no variation in the intraspecific competition coefficient of the species, other than for Lasthenia.
  if (lasthenia.invasion == TRUE) {N_2 <- (((g_2 * l_2) / (1 - s_2 * (1 - g_2))) - 1) / (alpha_22 * g_2)
  }else{N_2<-initial.invader}
  
  # if Lasthenia is invading, it takes initial invader population size, else it takes it's carrying capacity, according to either variable intraspecific competition coefficients, or mean competition coefficient. The carrying capacity is determined by simulation over 20 time steps.
  if (lasthenia.invasion == TRUE) {N_l<- initial.invader} else {
    N_l=1000
    for (jj in 1:20){
      
      samp.l <- sample(1:nrow(rates),size = 1000000,replace = T)
      samp.intra <- sample(1:nrow(rates),size = 1000000,replace = T)
      
      N_l <- N_l * mean((rates$lambda[samp.l] * g_l) * (1 / (1 + N_l * g_l * rates$intra[samp.intra])) + ((1-g_l) * s_l))
      jj = jj+1
    }}
  
  
  # empty vectors to hold the population trajectories.
  (N_2<-c(N_2,rep(NA,time.steps)))
  (N_l<-c(N_l,rep(NA,time.steps)))
  
  # run the simulation
  for (ii in 1:time.steps){
    
    samp.l <- sample(1:nrow(rates),size = 1000000,replace = T)
    samp.intra <- sample(1:nrow(rates),size = 1000000,replace = T)
    samp.inter <- sample(1:nrow(rates),size = 1000000,replace = T)
    
    # getting the mean growth rate assuming that the population takes a distribution of values for alpha.
    hold_l <- N_l[ii] * 
      mean(
        (rates$lambda[samp.l] * g_l) * 
          (1 / (1 + N_l[ii] * g_l * rates$intra[samp.intra] + 
                  N_2[ii] * g_2 * rates$inter[samp.inter])) + 
          ((1-g_l) * s_l)
      )
    
    hold_2 <- N_2[ii] * (l_2 * g_2 / (1 + N_2[ii] * g_2 * alpha_22 + N_l[ii] * g_l * alpha_2l) + (1-g_2)*s_2)
    
    N_l[ii+1]<-hold_l
    N_2[ii+1]<-hold_2
    cat("proportion done", ii/time.steps,"\n")
  }
  out<-cbind(N_l,N_2)
  return(out)
}


# lasthenia vs. bromus; bromus resident
phenol.with.var_lasthenia.inv.against.bromus.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = TRUE,
                 competitor="bromus",
                 lasthenia.invasion = TRUE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=bromus.lambda,
                 g_2=bromus.germination,
                 s_2=bromus.seed.survival,
                 alpha_22=alpha.bromus.bromus,
                 alpha_2l=alpha.bromus.lasthenia,
                 time.steps=iterations)

phenol.NO.var_lasthenia.inv.against.bromus.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = FALSE,
                 competitor="bromus",
                 lasthenia.invasion = TRUE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=bromus.lambda,
                 g_2=bromus.germination,
                 s_2=bromus.seed.survival,
                 alpha_22=alpha.bromus.bromus,
                 alpha_2l=alpha.bromus.lasthenia,
                 time.steps=iterations)


# lasthenia vs. bromus; lasthenia resident
phenol.with.var_bromus.inv.against.lasthenia.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = TRUE,
                 competitor="bromus",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=bromus.lambda,
                 g_2=bromus.germination,
                 s_2=bromus.seed.survival,
                 alpha_22=alpha.bromus.bromus,
                 alpha_2l=alpha.bromus.lasthenia,
                 time.steps=iterations)

phenol.NO.var_bromus.inv.against.lasthenia.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = FALSE,
                 competitor="bromus",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=bromus.lambda,
                 g_2=bromus.germination,
                 s_2=bromus.seed.survival,
                 alpha_22=alpha.bromus.bromus,
                 alpha_2l=alpha.bromus.lasthenia,
                 time.steps=iterations)

# lasthenia vs. lactuca; lactuca resident
phenol.with.var_lasthenia.inv.against.lactuca.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = TRUE,
                 competitor="lactuca",
                 lasthenia.invasion = TRUE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)

phenol.NO.var_lasthenia.inv.against.lactuca.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = FALSE,
                 competitor="lactuca",
                 lasthenia.invasion = TRUE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)


# lasthenia vs. lactuca; lasthenia resident
phenol.with.var_lactuca.inv.against.lasthenia.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = TRUE,
                 competitor="lactuca",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)

phenol.NO.var_lactuca.inv.against.lasthenia.NOcov <- 
  Lasthenia.Comp.NOcov(with.var = FALSE,
                 competitor="lactuca",
                 lasthenia.invasion = FALSE,          
                 lasthenia.comp.rates=l.dat,
                 initial.invader = 10^-8,
                 g_l=lasthenia.germination,
                 s_l=lasthenia.seed.survival, 
                 l_2=lactuca.lambda,
                 g_2=lactuca.germination,
                 s_2=lactuca.seed.survival,
                 alpha_22=alpha.lactuca.lactuca,
                 alpha_2l=alpha.lactuca.lasthenia,
                 time.steps=iterations)


#### instantaneous invasion growth rates ####
log(phenol.with.var_lasthenia.inv.against.bromus.NOcov[2,1]/phenol.with.var_lasthenia.inv.against.bromus.NOcov[1,1])
log(phenol.NO.var_lasthenia.inv.against.bromus.NOcov[2,1]/phenol.NO.var_lasthenia.inv.against.bromus.NOcov[1,1])

log(phenol.with.var_lasthenia.inv.against.lactuca.NOcov[2,1]/phenol.with.var_lasthenia.inv.against.lactuca.NOcov[1,1])
log(phenol.NO.var_lasthenia.inv.against.lactuca.NOcov[2,1]/phenol.NO.var_lasthenia.inv.against.lactuca.NOcov[1,1])


#### Figures ####
(Lasthenia.col = "#33a02c")
(bromus.col = "#a6cee3")
(lactuca.col = "#1f78b4")
(black = "#000000")

pdf(file="Invasion_Simulations_Figure_NOcov.pdf",width=8.5,height=5)
# quartz()
par(mfrow=c(1,2))
par(mar=c(c(5, 4, 4, 1) + 0.1))

plot(0:iterations,phenol.with.var_lactuca.inv.against.lasthenia.NOcov[,2]/1000,ylim=c(0,(max(c(as.vector(phenol.with.var_lactuca.inv.against.lasthenia.NOcov),as.vector(phenol.with.var_lactuca.inv.against.lasthenia.NOcov)))+1000)/1000),xlab="",ylab="",type='l',lwd=3,main="",col=lactuca.col)
lines(0:iterations,phenol.with.var_lactuca.inv.against.lasthenia.NOcov[,1]/1000,col=Lasthenia.col,lwd=3)
lines(0:iterations,phenol.NO.var_lactuca.inv.against.lasthenia.NOcov[,2]/1000,lwd=3,lty="21",col=lactuca.col)
lines(0:iterations,phenol.NO.var_lactuca.inv.against.lasthenia.NOcov[,1]/1000,col=Lasthenia.col,lwd=3,lty="21")
legend("topright",legend = c("with variation","NO variation","Lasthenia","Lactuca","Bromus"),lty=c("solid","21","1919","1919","1919"),lwd=c(3,3,0,0,0),pch=c(NA,NA,15,15,15),col=c(black,black,Lasthenia.col,lactuca.col,bromus.col),y.intersp = 2,bty = "n",pt.cex=2,cex=1)

# x-axis
mtext(text=expression(paste("years")),side=1,line=3,cex=1.2)
# y-axis
mtext(text=expression(paste("population size ('000s)")),side=2,line=2.5,cex=1.2)
# plot label
mtext(text = "a.",cex=1.2,side = 3,adj = 0,line = 0.5)

par(mar=c(c(5, 2, 4, 3) + 0.1))
plot(0:iterations,phenol.with.var_bromus.inv.against.lasthenia.NOcov[,2]/1000,ylim=c(0,(max(c(as.vector(phenol.with.var_bromus.inv.against.lasthenia.NOcov),as.vector(phenol.with.var_bromus.inv.against.lasthenia.NOcov)))+1000)/1000),xlab="",ylab="",type='l',lwd=3,main="",col=bromus.col)
lines(0:iterations,phenol.with.var_bromus.inv.against.lasthenia.NOcov[,1]/1000,col=Lasthenia.col,lwd=3)
lines(0:iterations,phenol.NO.var_bromus.inv.against.lasthenia.NOcov[,2]/1000,lwd=3,lty="21",col=bromus.col)
lines(0:iterations,phenol.NO.var_bromus.inv.against.lasthenia.NOcov[,1]/1000,col=Lasthenia.col,lwd=3,lty="21")

mtext(text=expression(paste("years")),side=1,line=3,cex=1.2)
# plot label
mtext(text = "b.",cex=1.2,side = 3,adj = 0,line = 0.5)

dev.off()