rm(list=ls())

#### data ####
# data for Lasthenia as the focal species. 
data.x <- read.csv("Data_Lasthenia.csv")
str(data.x)
head(data.x)

# data for Bromus and Lactuca as the focal species. 
data.z <- read.csv("Data_BromusLactuca.csv")
str(data.z)
head(data.z)

#Lasthenia competing against BROMUS
(d1_bromus<- data.x[data.x$phen_date=="110-130",])
(d2_bromus<- data.x[data.x$phen_date=="140-150",])
(d3_bromus<- data.x[data.x$phen_date=="165-175" & data.x$background!="LASE",])

#Lasthenia competing against LACTUCA
(d1_lactuca<- data.x[data.x$phen_date=="120-130",])
(d2_lactuca<- data.x[data.x$phen_date=="145-160",])
(d3_lactuca<- data.x[data.x$phen_date=="165-175" & data.x$background!="BRMA",])

(d4_bromlact <- data.x[data.x$phen_date=="165-175",])

# data to calculate model parameters when Bromus or Lactuca are the focal species. 
(d1_bromus_focal <- data.z[data.z$species=="Bromus",])
(d1_lactuca_focal <- data.z[data.z$species=="Lactuca",])

head(d1_lactuca)
head(d1_lactuca_focal)

#### models ####
#model 1 - no effect of density (no competitive effects)
compmodel1<-function(par){
  
  ## mu (=lambda later) and sigma parameters for the normal distribution (assuming lognormal error- seed data are logged)
  lambda<-par[1]
  sigma<-par[2]
  
  #this the predictive model- here is just fitting a horizontal line through the data:
  pred<-rep(lambda, times=length(log_seeds))
  
  #these are the log likelihoods of the data given the model + parameters
  llik<-dnorm(log_seeds,log(pred), sd=sigma, log=TRUE)
  
  #return the sum of negative log likelihoods - what optim minimizes
  return(sum(-1*llik))
}

#model 2 - competition, but no difference between species
compmodel2<-function(par){
  
  lambda<-par[1] ## same as model 1
  alpha<-par[2]  ## new parameter introduced in model 2
  sigma<-par[3] ## same as model 1
  
  ## apply is used to get a single vector of densities rather than a matrix
  ## there is only one competitor at a time here- just adding each density to a
  ## column of other zeros:
  
  tot_density<-apply(dens, MARGIN=2, FUN=sum)
  
  ## predictive model:
  pred<-lambda/(1+alpha*(tot_density))
  
  ## log likelihoods of data given the model + parameters:
  llik<-dnorm(log_seeds,log(pred), sd=sigma, log=TRUE)
  
  ## return sum of negative log likelihoods:
  return(sum(-1*llik))
  
}


#### Lasthenia as the focal species, against Bromus ####
#model 3 - all have different competitive effects (when Lasthenia is competing with Bromus)
compmodel3<-function(par){
  lambda<-par[1] #same as model 2
  a_BRMA<-par[2]	
  a_LACA<-par[3]
  # a_LASE<-par[4]
  sigma<-par[4] ## same as model 2
  
  ##probably overkill to name all the alphas, but it helps keep things straight
  
  # same form as model 1, but super long given all the species:
  pred<- lambda / (1+ a_BRMA* dens['BRMA',] + a_LACA* dens['LACA',])
  
  
  # likelihood as before:
  llik<-dnorm(log_seeds,log(pred), sd=sigma, log=TRUE)
  
  # return sum of negative log likelihoods
  return(sum(-1*llik)) #sum of negative log likelihoods
}


#### Bromus 110-130 ####
d <- d1_bromus

pdf(file="alphas_and_lambdas_LASTHENIA_bromus_110_130.pdf", width=11, height=8)

# plants from plots designated as lambda plots:
subset(d, d$plot_type=="Lambda")->lam
str(lam)

# drop lambda rows without seed production:
lam<-lam[!is.na(lam$seeds),]
str(lam)

# intermediate subset- plants from plots designated as competition plots ('not lambda')
subset(d, d$plot_type!="Lambda")->nl
str(nl)
# drop rows with no seed production from the not lambda subset:
nl<-nl[!is.na(nl$seeds),]
str(nl)

# set up a data frame of just plants in competition with seed production (keep this):
subset(nl, nl$density>0)->comp

# two data frame to use at this point- 'comp' (plots con competition) and 'lam' (plot de lambda)
splist_all <- c("BRMA","LACA")

# list of target species
(splist<-unique(comp$target))
(splist<-paste(splist))

# alpha order splist:
(splist<-splist[order(splist)])

# same for background
(splist_back<-unique(comp$background))
(splist_back<-paste(splist_back))

# alpha order splist:
(splist_back<-splist_back[order(splist_back)])

# objects to hold the final parameter estimates from model 3:
(alpha_matrix<- matrix(0, nrow=length(splist), ncol=length(splist_back)))
(row.names(alpha_matrix)<-splist)
(colnames(alpha_matrix)<-splist_back)
lambda_est<-NULL
sigma_est<-NULL
convergence_code<-NULL

upper_lambda <-NULL
lower_lambda <-NULL

(lower_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(lower_alpha)<-splist_all)
(colnames(lower_alpha)<-splist_all)

(upper_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(upper_alpha)<-splist_all)
(colnames(upper_alpha)<-splist_all)

## for each species in turn as a target:
for(i in 1:length(splist)){

  # i=1  
  
  #set up grid for plotting and add first panel indicating focal sp:
  layout(matrix(1:4, nrow=2))
  plot(x=0, y=0, type='n')
  text(0, 0, paste(splist[i], "as focal", sep=" "), cex=1, col="red")
  
    ## subset out the rows that are needed from the competition df
  (comp_points<-subset(comp, comp$target==splist[i], drop=TRUE))
  
  ## and the correct lambda plants
  (lam_points<-subset(lam, lam$target==splist[i]))
  
  ## now need to build up a vector of nonzero seed production
  ## and corresponding density vectors (held in a matrix) for background species
  ## to use in model fitting
  
  ## start with the lambda seeds- will add on to this for each background:
    (seeds<-lam_points$seeds)
  
  ##build density matrix (each row will be density of a species)
  (dens<-matrix(0, nrow=length(splist_back), ncol=(nrow(lam_points)+ nrow(comp_points))))
  
  (row.names(dens)<-splist_back)
  
  ## use this counter to keep track of which column is next to have data added to
  ## set it to begin after the lambda points:
  
  start <- nrow(lam_points) +1
  
  
  for(j in 1:length(splist_back)){
    
    # j=1
    # j=2
    
    ## LOOP creates a seeds vector and a corresponding dens ("density") matrix with
    ## background species as rows, and columns corresponding to the seed production
    ## vector.
    ## beginning columns correspond to lambda plants
    
    #take just the rows pertaining to a specific background sp:
    (bg_points<- subset(comp_points, comp_points$background==splist_back[j]))
    
    ## which row of the density matrix corresponds?
    (rownum <- which(row.names(dens)==splist_back[j]))
    
    #column to end with:
    (end<-start + nrow(bg_points)-1)
    
    ## drop in density values into matrix:
    (dens[rownum, start:end]<-bg_points$density)
    
    ## add seed numbers into the seeds vector
    (seeds<-c(seeds, bg_points$seeds))
    
    ##increase start counter so that next species is offset from this one
    (start<-end+1)
    
  }
  
  ## should now have "seeds" vector and corresponding "dens"ity matrix
  ## can test ncol(dens)==length(seeds)
  
  ## we'll be working with log seeds (for giving a lognormal error structure):
  
  (log_seeds<-log(seeds))
  
  
  #model fitting using optim and earlier likelihood functions
  
  #############################
  ## model 1, no competition ##
  #############################
  
  ###recall parameters are lambda and sigma- initialize these with estimates from the data:
  (par1<-c(mean(log_seeds), sd(log_seeds)))
  
  
  ##repeat optimization until we get convergence (or we try 25 times)
  for(k in 1:25){
    testcomp1<- optim(par1,compmodel1)
    ##update start parameters to final estimate to use in next run in case of nonconvergence
    par1<-testcomp1$par
    if(testcomp1$convergence==0){
      print(paste(splist[i], "model 1 converged on rep", k, sep=" "))
      break
    }
    
  }
  
  
  #############################
  ## model 2, one alpha      ##
  #############################
  
  ## pars here are lambda, alpha and sigma- use lambda and sigma from model 1 as starting esimtates
  (par2<-c(testcomp1$par[1],0.001,testcomp1$par[2]))
  par.scale<- c(1.e02, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before:
  for(k in 1:25){
    ##now using a specific method that permits constrained optimization so that alpha has to be nonzero- this is an issue in some of the fits, especially in model 3. lower parameter has same order as par2
    testcomp2<-optim(par2,compmodel2, method="L-BFGS-B", lower=c(1,0,0.0000000001), control=list(maxit=1000, parscale=par.scale))
    par2<-testcomp2$par
    if(testcomp2$convergence==0){
      print(paste(splist[i],  "model 2 converged on rep", k, sep=" "))
      break
    }
  }
  
  #############################
  ## model 3, unique alphas  ##
  #############################
  
  ## pars here are lambda, 2 alphas and sigma- use lambda and sigma from model 2, and alphas from model 2 as starting estimate for each species
  (par3<-c(testcomp2$par[1], rep(testcomp2$par[2], times=2), testcomp2$par[3]))
  par.scale<- c(1.e02, 1.e-03, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before
    for(k in 1:25){
    ##now using a specific method that permits constained optimization so that alpha has to be nonzero-
    testcomp3<-optim(par3,compmodel3, method="L-BFGS-B", lower=c(1, rep(0, times=2),0.0000000001), control=list(maxit=1000, parscale=par.scale), hessian = T)
    par3<-testcomp3$par
    if(testcomp3$convergence==0){
      print(paste(splist[i], "model 3 converged on rep", k, sep=" "))
      break
    }
  }
  
  
  ##################################
  ### save estimates from model 3 ##
  ##################################
  
  (lambda_est<-c(lambda_est, testcomp3$par[1]))
  (sigma_est<-c(sigma_est, testcomp3$par[4]))
  convergence_code<-c(convergence_code, testcomp3$convergence)
  
  fisher_info<-solve(testcomp3$hessian)
  prop_sigma<-sqrt(diag(fisher_info))
  upper<-testcomp3$par+1.96*prop_sigma
  lower<-testcomp3$par-1.96*prop_sigma
  
  upper_lambda <- c(upper_lambda, upper[1])
  lower_lambda <- c(lower_lambda, lower[1]) 
  
  
  ## in keeping with Lotka Volterra notation, we'll use alpha1_2 to indicate effect of
  ## sp 2 on growth of 1.  Following convention, i refers to rows and j to cols in a matrix
  ## so each step of the loop here (for a target sp) corresponds to one row of this matrix:
  
  alphas<-testcomp3$par[2:3]
  
  alpha_matrix[i,]<-alphas

  errors_lower_alpha <-lower[2:3]
  lower_alpha[i,] <- errors_lower_alpha
  errors_upper_alpha <-upper[2:3]
  upper_alpha[i,] <- errors_upper_alpha
  
  ##note that in cases where there is no data for a particular species the alpha estimate for that species ends up as the starting value- we need to be careful of these as they are basically gargbage numbers.  Keeping them in up to now to keep the structure of the data constant, but will set them to NA here:
  
  #identify which species have no data in this fit:
  which(apply(dens, MARGIN=1, FUN=mean)==0)->no_data
  
  ## set their alphas to NA in the matrix:
  
  alpha_matrix[i,no_data]<-NA
  
  
  ###############################
  ## some diagnostics +  plots ##
  ###############################
  
  ## print an error to the console if any one of the three models failed to converge:
  
  if(testcomp1$convergence + testcomp2$convergence + testcomp3$convergence !=0){print(paste("at least one model did not converge for", splist[i], sep=" "))}
  
  ##################################
  ## plot observed vs predicted:
  ##################
  
  par<-testcomp3$par
  
  #from model 3 code:
  lambda<-par[1]
  a_BRMA<-par[2]
  
  a_LACA<-par[3]
  # a_LASE<-par[4]
  
  pred<- lambda / (1+ a_BRMA* dens['BRMA',] + a_LACA* dens['LACA',])
  
  
  plot(seeds, pred, xlim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), ylim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), log='xy', xlab="observed seeds", ylab="predicted seeds", main=splist[i] )
  
  abline(a=0, b=1, lwd=2)
  
  
  #####################
  #### plot each fit
  ##########
  
  names(alphas)<-splist_back
  
  names(apply(dens, MARGIN=1, FUN=mean)!=0)->plotlist
  
  for(l in 1:length(plotlist)){
    
    ## which columns in the density dataframe have nozero values for species l ?
    cols<-which(dens[l,]>0)
    
    x<-dens[l,cols]
    y<-seeds[cols]
    
    ##add lambdas:
    x<-c(x, rep(0,nrow(lam_points)))
    y<-c(y, lam_points$seeds)
    
    x_det<-seq(min(x), max(x), by=(  (max(x)-min(x))/1000  ))
    
    alpha_temp<-alphas[which(names(alphas)==plotlist[l])]
    y_pred<-lambda/(1 + alpha_temp*x_det)
    
    if(length(unique(x))>1){
      plot(y~x, xlab="germinants_background", ylab="seeds", main=plotlist[l])
    } else {
      
      plot(x=0, y=0, main=plotlist[l], type='n')
      text(0, 0, "no data")
      
    }
    
    lines(y_pred~x_det, col="red", lwd=2)
  }
  
  
}

dev.off()

(results<-data.frame(splist, lambda_est, lower_lambda, upper_lambda, sigma_est, convergence_code))
names(results)<-c("species", "lambda", "lower_error", "upper error", "sigma", "convergence_code")
write.csv(results, "lambda_estimates_LASTHENIA_bromus_110_130.csv")
write.csv(alpha_matrix, "alpha_estimates_row_is_target_LASTHENIA_bromus_110_130.csv")
write.csv(lower_alpha, "alpha_lower_errors_control_LASTHENIA_bromus_110_130.csv")
write.csv(upper_alpha, "alpha_upper_errors_control_LASTHENIA_bromus_110_130.csv")

#### Bromus 140-150 ####
d<-d2_bromus

pdf(file="alphas_and_lambdas_LASTHENIA_bromus_140_150.pdf", width=11, height=8)

# plants from plots designated as lambda plots:
subset(d, d$plot_type=="Lambda")->lam
str(lam)

# drop lambda rows without seed production:
lam<-lam[!is.na(lam$seeds),]
str(lam)

# intermediate subset- plants from plots designated as competition plots ('not lambda')
subset(d, d$plot_type!="Lambda")->nl
str(nl)
# drop rows with no seed production from the not lambda subset:
nl<-nl[!is.na(nl$seeds),]
str(nl)

# set up a data frame of just plants in competition with seed production (keep this):
subset(nl, nl$density>0)->comp

# two data frame to use at this point- 'comp' (plots con competition) and 'lam' (plot de lambda)
splist_all <- c("BRMA","LACA")

# list of target species
(splist<-unique(comp$target))
(splist<-paste(splist))

# alpha order splist:
(splist<-splist[order(splist)])

# same for background
(splist_back<-unique(comp$background))
(splist_back<-paste(splist_back))

# alpha order splist:
(splist_back<-splist_back[order(splist_back)])

# objects to hold the final parameter estimates from model 3:
(alpha_matrix<- matrix(0, nrow=length(splist), ncol=length(splist_back)))
(row.names(alpha_matrix)<-splist)
(colnames(alpha_matrix)<-splist_back)
lambda_est<-NULL
sigma_est<-NULL
convergence_code<-NULL

upper_lambda <-NULL
lower_lambda <-NULL

(lower_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(lower_alpha)<-splist_all)
(colnames(lower_alpha)<-splist_all)

(upper_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(upper_alpha)<-splist_all)
(colnames(upper_alpha)<-splist_all)

## for each species in turn as a target:

for(i in 1:length(splist)){
  
  # i=1  
  
  #set up grid for plotting and add first panel indiciating focal sp:
  layout(matrix(1:4, nrow=2))
  plot(x=0, y=0, type='n')
  text(0, 0, paste(splist[i], "as focal", sep=" "), cex=1, col="red")
  
  ## subset out the rows that are needed from the competition df
  (comp_points<-subset(comp, comp$target==splist[i], drop=TRUE))
  
  ## and the correct lambda plants
  (lam_points<-subset(lam, lam$target==splist[i]))
  
  ## now need to build up a vector of nonzero seed production
  ## and corresponding density vectors (held in a matrix) for background species
  ## to use in model fitting
  
  ## start with the lambda seeds- will add on to this for each background:
  (seeds<-lam_points$seeds)
  
  ##build density matrix (each row will be density of a species)
  (dens<-matrix(0, nrow=length(splist_back), ncol=(nrow(lam_points)+ nrow(comp_points))))
  
  (row.names(dens)<-splist_back)
  
  
  ## use this counter to keep track of which column is next to have data added to
  ## set it to begin after the lambda points:
  
  start <- nrow(lam_points) +1
  
  
  for(j in 1:length(splist_back)){
    
    # j=1
    
    ## LOOP creates a seeds vector and a corresponding dens ("density") matrix with
    ## background species as rows, and columns corresponding to the seed production
    ## vector.
    ## beginning columns correspond to lambda plants
    
    #take just the rows pertaining to a specific background sp:
    (bg_points<- subset(comp_points, comp_points$background==splist_back[j]))
    
    ## which row of the density matrix corresponds?
    (rownum <- which(row.names(dens)==splist_back[j]))
    
    #column to end with:
    (end<-start + nrow(bg_points)-1)
    
    ## drop in density values into matrix:
    (dens[rownum, start:end]<-bg_points$density)
    
    ## add seed numbers into the seeds vector
    (seeds<-c(seeds, bg_points$seeds))
    
    ##increase start counter so that next species is offset from this one
    (start<-end+1)
    
  }
  
  ## should now have "seeds" vector and corresponding "dens"ity matrix
  ## can test ncol(dens)==length(seeds)
  
  ## we'll be working with log seeds (for giving a lognormal error structure):
  
  (log_seeds<-log(seeds))
  
  
  #model fitting using optim and earlier likelihood functions
  
  #############################
  ## model 1, no competition ##
  #############################
  
  ###recall parameters are lambda and sigma- initialize these with estimates from the data:
  (par1<-c(mean(log_seeds), sd(log_seeds)))
  
  
  ##repeat optimization until we get convergence (or we try 25 times)
  for(k in 1:25){
    testcomp1<-optim(par1,compmodel1)
    ##update start parameters to final estimate to use in next run in case of nonconvergence
    par1<-testcomp1$par
    if(testcomp1$convergence==0){
      print(paste(splist[i], "model 1 converged on rep", k, sep=" "))
      break
    }
    
  }
  
  
  #############################
  ## model 2, one alpha      ##
  #############################
  
  ## pars here are lambda, alpha and sigma- use lambda and sigma from model 1 as starting esimtates
  (par2<-c(testcomp1$par[1],0.001,testcomp1$par[2]))
  par.scale<- c(1.e02, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before:
  for(k in 1:25){
    ##now using a specific method that permits constrained optimization so that alpha has to be nonzero- this is an issue in some of the fits, especially in model 3. lower parameter has same order as par2
    testcomp2<-optim(par2,compmodel2, method="L-BFGS-B", lower=c(1,0,0.0000000001), control=list(maxit=1000, parscale=par.scale))
    par2<-testcomp2$par
    if(testcomp2$convergence==0){
      print(paste(splist[i],  "model 2 converged on rep", k, sep=" "))
      break
    }
  }
  
  #############################
  ## model 3, unique alphas  ##
  #############################
  
  ## pars here are lambda, 3 alphas and sigma- use lambda and sigma from model 2, and alphas from model 2 as starting esimtate for each species
  
  (par3<-c(testcomp2$par[1], rep(testcomp2$par[2], times=2), testcomp2$par[3]))
  par.scale<- c(1.e02, 1.e-03, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  ##as before
  
  for(k in 1:25){
    ##now using a specific method that permits constained optimization so that alpha has to be nonzero-
    testcomp3<-optim(par3,compmodel3, method="L-BFGS-B", lower=c(1, rep(0, times=2),0.0000000001), control=list(maxit=1000, parscale=par.scale), hessian = T)
    par3<-testcomp3$par
    if(testcomp3$convergence==0){
      print(paste(splist[i], "model 3 converged on rep", k, sep=" "))
      break
    }
  }
  
  
  ##################################
  ### save estimates from model 3 ##
  ##################################
  
  (lambda_est<-c(lambda_est, testcomp3$par[1]))
  (sigma_est<-c(sigma_est, testcomp3$par[4]))
  convergence_code<-c(convergence_code, testcomp3$convergence)
  
  fisher_info<-solve(testcomp3$hessian)
  prop_sigma<-sqrt(diag(fisher_info))
  upper<-testcomp3$par+1.96*prop_sigma
  lower<-testcomp3$par-1.96*prop_sigma
  
  upper_lambda <- c(upper_lambda, upper[1])
  lower_lambda <- c(lower_lambda, lower[1]) 
  
  
  ## in keeping with Lotka Volterra notation, we'll use alpha1_2 to indicate effect of
  ## sp 2 on growth of 1.  Following convention, i refers to rows and j to cols in a matrix
  ## so each step of the loop here (for a target sp) corresponds to one row of this matrix:
  
  alphas<-testcomp3$par[2:3]
  
  alpha_matrix[i,]<-alphas
  
  errors_lower_alpha <-lower[2:3]
  lower_alpha[i,] <- errors_lower_alpha
  errors_upper_alpha <-upper[2:3]
  upper_alpha[i,] <- errors_upper_alpha
  
  ##note that in cases where there is no data for a particular species the alpha estimate for that species ends up as the starting value- we need to be careful of these as they are basically gargbage numbers.  Keeping them in up to now to keep the structure of the data constant, but will set them to NA here:
  
  #identify which species have no data in this fit:
  which(apply(dens, MARGIN=1, FUN=mean)==0)->no_data
  
  ## set their alphas to NA in the matrix:
  
  alpha_matrix[i,no_data]<-NA
  
  
  ###############################
  ## some diagnostics +  plots ##
  ###############################
  
  ## print an error to the console if any one of the three models failed to converge:
  
  if(testcomp1$convergence + testcomp2$convergence + testcomp3$convergence !=0){print(paste("at least one model did not converge for", splist[i], sep=" "))}
  
  ##################################
  ## plot observed vs predicted:
  ##################
  
  par<-testcomp3$par
  
  #from model 3 code:
  lambda<-par[1]
  a_BRMA<-par[2]
  
  a_LACA<-par[3]
  # a_LASE<-par[4]
  
  pred<- lambda / (1+ a_BRMA* dens['BRMA',] + a_LACA* dens['LACA',])
  
  
  plot(seeds, pred, xlim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), ylim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), log='xy', xlab="observed seeds", ylab="predicted seeds", main=splist[i] )
  
  abline(a=0, b=1, lwd=2)
  
  
  #####################
  #### plot each fit
  ##########
  
  names(alphas)<-splist_back
  
  names(apply(dens, MARGIN=1, FUN=mean)!=0)->plotlist
  
  for(l in 1:length(plotlist)){
    
    ## which columns in the density dataframe have nozero values for species l ?
    cols<-which(dens[l,]>0)
    
    x<-dens[l,cols]
    y<-seeds[cols]
    
    ##add lambdas:
    x<-c(x, rep(0,nrow(lam_points)))
    y<-c(y, lam_points$seeds)
    
    x_det<-seq(min(x), max(x), by=(  (max(x)-min(x))/1000  ))
    
    alpha_temp<-alphas[which(names(alphas)==plotlist[l])]
    y_pred<-lambda/(1 + alpha_temp*x_det)
    
    if(length(unique(x))>1){
      plot(y~x, xlab="germinants_background", ylab="seeds", main=plotlist[l])
    } else {
      
      plot(x=0, y=0, main=plotlist[l], type='n')
      text(0, 0, "no data")
      
    }
    
    lines(y_pred~x_det, col="red", lwd=2)
  }
  
}

dev.off()

results<-data.frame(splist, lambda_est, lower_lambda, upper_lambda, sigma_est, convergence_code)

names(results)<-c("species", "lambda", "lower_error", "upper error", "sigma", "convergence_code")

write.csv(results, "lambda_estimates_LASTHENIA_bromus_140_150.csv")
write.csv(alpha_matrix, "alpha_estimates_row_is_target_LASTHENIA_bromus_140_150.csv")
write.csv(lower_alpha, "alpha_lower_errors_control_LASTHENIA_bromus_140_150.csv")
write.csv(upper_alpha, "alpha_upper_errors_control_LASTHENIA_bromus_140_150.csv")


#### Lactuca ####
# model 3 - both species (lasthenia and lactuca) have different competitive effects
compmodel3<-function(par){
  lambda<-par[1] #same as model 2
  # a_BRMA<-par[2]	
  a_LACA<-par[2]
  a_LASE<-par[3]
  sigma<-par[4] ## same as model 2
  
  ##probably overkill to name all the alphas, but it helps keep things straight
  
  # same form as model 1, but super long given all the species:
  pred<- lambda / (1+ a_LACA* dens['LACA',] + a_LASE* dens['LASE',])
  
  
  # likelihood as before:
  llik<-dnorm(log_seeds,log(pred), sd=sigma, log=TRUE)
  
  # return sum of negative log likelihoods
  return(sum(-1*llik)) #sum of negative log likelihoods
}

#### Lactuca 120-130 ####
d<-d1_lactuca

pdf(file="alphas_and_lambdas_LASTHENIA_lactuca_120_130.pdf", width=11, height=8)

# plants from plots designated as lambda plots:
(subset(d, d$plot_type=="Lambda")->lam)
str(lam)

# drop lambda rows without seed production:
lam<-lam[!is.na(lam$seeds),]
str(lam)

# intermediate subset- plants from plots designated as competition plots ('not lambda')
subset(d, d$plot_type!="Lambda")->nl
str(nl)
# drop rows with no seed production from the not lambda subset:
nl<-nl[!is.na(nl$seeds),]
str(nl)

# set up a data frame of just plants in competition with seed production (keep this):
subset(nl, nl$density>0)->comp

# two data frame to use at this point- 'comp' (plots con competition) and 'lam' (plot de lambda)
splist_all <- c("LACA","LASE")

# list of target species
(splist<-unique(comp$target))
(splist<-paste(splist))

# alpha order splist:
(splist<-splist[order(splist)])

# same for background
(splist_back<-unique(comp$background))
(splist_back<-paste(splist_back))

# alpha order splist:
(splist_back<-splist_back[order(splist_back)])

# objects to hold the final parameter estimates from model 3:
(alpha_matrix<- matrix(0, nrow=length(splist), ncol=length(splist_back)))
(row.names(alpha_matrix)<-splist)
(colnames(alpha_matrix)<-splist_back)
lambda_est<-NULL
sigma_est<-NULL
convergence_code<-NULL

upper_lambda <-NULL
lower_lambda <-NULL

(lower_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(lower_alpha)<-splist_all)
(colnames(lower_alpha)<-splist_all)

(upper_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(upper_alpha)<-splist_all)
(colnames(upper_alpha)<-splist_all)

## for each species in turn as a target:
for(i in 1:length(splist)){
  
  # i=1  
  
  #set up grid for plotting and add first panel indiciating focal sp:
  layout(matrix(1:4, nrow=2))
  plot(x=0, y=0, type='n')
  text(0, 0, paste(splist[i], "as focal", sep=" "), cex=1, col="red")
  
  ## subset out the rows that are needed from the competition df
  (comp_points<-subset(comp, comp$target==splist[i], drop=TRUE))
  
  ## and the correct lambda plants
  (lam_points<-subset(lam, lam$target==splist[i]))
  
  ## now need to build up a vector of nonzero seed production
  ## and corresponding density vectors (held in a matrix) for background species
  ## to use in model fitting
  
  ## start with the lambda seeds- will add on to this for each background:
  (seeds<-lam_points$seeds)
  
  ##build density matrix (each row will be density of a species)
  (dens<-matrix(0, nrow=length(splist_back), ncol=(nrow(lam_points)+ nrow(comp_points))))
  
  (row.names(dens)<-splist_back)
  
  ## use this counter to keep track of which column is next to have data added to
  ## set it to begin after the lambda points:
  
  start <- nrow(lam_points) +1
  
  
  for(j in 1:length(splist_back)){
    
    # j=1
    # j=2
    
    ## LOOP creates a seeds vector and a corresponding dens ("density") matrix with
    ## background species as rows, and columns corresponding to the seed production
    ## vector.
    ## beginning columns correspond to lambda plants
    
    #take just the rows pertaining to a specific background sp:
    (bg_points<- subset(comp_points, comp_points$background==splist_back[j]))
    
    ## which row of the density matrix corresponds?
    (rownum <- which(row.names(dens)==splist_back[j]))
    
    #column to end with:
    (end<-start + nrow(bg_points)-1)
    
    ## drop in density values into matrix:
    (dens[rownum, start:end]<-bg_points$density)
    
    ## add seed numbers into the seeds vector
    (seeds<-c(seeds, bg_points$seeds))
    
    ##increase start counter so that next species is offset from this one
    (start<-end+1)
    
  }
  
  ## should now have "seeds" vector and corresponding "dens"ity matrix
  ## can test ncol(dens)==length(seeds)
  
  ## we'll be working with log seeds (for giving a lognormal error structure):
  
  (log_seeds<-log(seeds))
  
  
  #model fitting using optim and earlier likelihood functions
  
  #############################
  ## model 1, no competition ##
  #############################
  
  ###recall parameters are lambda and sigma- initialize these with estimates from the data:
  (par1<-c(mean(log_seeds), sd(log_seeds)))
  
  
  ##repeat optimization until we get convergence (or we try 25 times)
  for(k in 1:25){
    testcomp1<-optim(par1,compmodel1)
    ##update start parameters to final estimate to use in next run in case of nonconvergence
    par1<-testcomp1$par
    if(testcomp1$convergence==0){
      print(paste(splist[i], "model 1 converged on rep", k, sep=" "))
      break
    }
    
  }
  
  #############################
  ## model 2, one alpha      ##
  #############################
  
  ## pars here are lambda, alpha and sigma- use lambda and sigma from model 1 as starting esimtates
  (par2<-c(testcomp1$par[1],0.001,testcomp1$par[2]))
  par.scale<- c(1.e02, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before:
  for(k in 1:25){
    ##now using a specific method that permits constrained optimization so that alpha has to be nonzero- this is an issue in some of the fits, especially in model 3. lower parameter has same order as par2
    testcomp2<-optim(par2,compmodel2, method="L-BFGS-B", lower=c(1,0,0.0000000001), control=list(maxit=1000, parscale=par.scale))
    par2<-testcomp2$par
    if(testcomp2$convergence==0){
      print(paste(splist[i],  "model 2 converged on rep", k, sep=" "))
      break
    }
  }
  
  #############################
  ## model 3, unique alphas  ##
  #############################
  
  ## pars here are lambda, 3 alphas and sigma- use lambda and sigma from model 2, and alphas from model 2 as starting esimtate for each species
  
  (par3<-c(testcomp2$par[1], rep(testcomp2$par[2], times=2), testcomp2$par[3]))
  par.scale<- c(1.e02, 1.e-03, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  ##as before
  
  for(k in 1:25){
    ##now using a specific method that permits constained optimization so that alpha has to be nonzero-
    testcomp3<-optim(par3,compmodel3, method="L-BFGS-B", lower=c(1, rep(0, times=2),0.0000000001), control=list(maxit=1000, parscale=par.scale), hessian = T)
    par3<-testcomp3$par
    if(testcomp3$convergence==0){
      print(paste(splist[i], "model 3 converged on rep", k, sep=" "))
      break
    }
  }
  
  
  ##################################
  ### save estimates from model 3 ##
  ##################################
  
  (lambda_est<-c(lambda_est, testcomp3$par[1]))
  (sigma_est<-c(sigma_est, testcomp3$par[4]))
  convergence_code<-c(convergence_code, testcomp3$convergence)
  
  fisher_info<-solve(testcomp3$hessian)
  prop_sigma<-sqrt(diag(fisher_info))
  upper<-testcomp3$par+1.96*prop_sigma
  lower<-testcomp3$par-1.96*prop_sigma
  
  upper_lambda <- c(upper_lambda, upper[1])
  lower_lambda <- c(lower_lambda, lower[1]) 
  
  
  ## in keeping with Lotka Volterra notation, we'll use alpha1_2 to indicate effect of
  ## sp 2 on growth of 1.  Following convention, i refers to rows and j to cols in a matrix
  ## so each step of the loop here (for a target sp) corresponds to one row of this matrix:
  
  alphas<-testcomp3$par[2:3]
  
  alpha_matrix[i,]<-alphas
  
  errors_lower_alpha <-lower[2:3]
  lower_alpha[i,] <- errors_lower_alpha
  errors_upper_alpha <-upper[2:3]
  upper_alpha[i,] <- errors_upper_alpha
  
  ##note that in cases where there is no data for a particular species the alpha estimate for that species ends up as the starting value- we need to be careful of these as they are basically gargbage numbers.  Keeping them in up to now to keep the structure of the data constant, but will set them to NA here:
  
  #identify which species have no data in this fit:
  which(apply(dens, MARGIN=1, FUN=mean)==0)->no_data
  
  ## set their alphas to NA in the matrix:
  
  alpha_matrix[i,no_data]<-NA
  
  
  ###############################
  ## some diagnostics +  plots ##
  ###############################
  
  ## print an error to the console if any one of the three models failed to converge:
  
  if(testcomp1$convergence + testcomp2$convergence + testcomp3$convergence !=0){print(paste("at least one model did not converge for", splist[i], sep=" "))}
  
  ##################################
  ## plot observed vs predicted:
  ##################
  
  par<-testcomp3$par
  
  #from model 3 code:
  lambda<-par[1]
  # a_BRMA<-par[2]
  
  a_LACA<-par[2]
  a_LASE<-par[3]
  
  pred<- lambda / (1+ a_LACA* dens['LACA',] + a_LACA* dens['LASE',])
  
  
  plot(seeds, pred, xlim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), ylim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), log='xy', xlab="observed seeds", ylab="predicted seeds", main=splist[i] )
  
  abline(a=0, b=1, lwd=2)
  
  
  #####################
  #### plot each fit
  ##########
  
  names(alphas)<-splist_back
  
  names(apply(dens, MARGIN=1, FUN=mean)!=0)->plotlist
  
  for(l in 1:length(plotlist)){
    
    ## which columns in the density dataframe have nozero values for species l ?
    cols<-which(dens[l,]>0)
    
    x<-dens[l,cols]
    y<-seeds[cols]
    
    ##add lambdas:
    x<-c(x, rep(0,nrow(lam_points)))
    y<-c(y, lam_points$seeds)
    
    x_det<-seq(min(x), max(x), by=(  (max(x)-min(x))/1000  ))
    
    alpha_temp<-alphas[which(names(alphas)==plotlist[l])]
    y_pred<-lambda/(1 + alpha_temp*x_det)
    
    if(length(unique(x))>1){
      plot(y~x, xlab="germinants_background", ylab="seeds", main=plotlist[l])
    } else {
      
      plot(x=0, y=0, main=plotlist[l], type='n')
      text(0, 0, "no data")
      
    }
    
    lines(y_pred~x_det, col="red", lwd=2)
  }
  
}

dev.off()


results<-data.frame(splist, lambda_est, lower_lambda, upper_lambda, sigma_est, convergence_code)

names(results)<-c("species", "lambda", "lower_error", "upper error", "sigma", "convergence_code")
write.csv(results, "lambda_estimates_LASTHENIA_lactuca_120_130.csv")
write.csv(alpha_matrix, "alpha_estimates_row_is_target_LASTHENIA_lactuca_120_130.csv")
write.csv(lower_alpha, "alpha_lower_errors_control_LASTHENIA_lactuca_120_130.csv")
write.csv(upper_alpha, "alpha_upper_errors_control_LASTHENIA_lactuca_120_130.csv")

#### Lactuca 145-160 ####
(d<-d2_lactuca)

pdf(file="alphas_and_lambdas_LASTHENIA_lactuca_145_160.pdf", width=11, height=8)

# plants from plots designated as lambda plots:
(subset(d, d$plot_type=="Lambda")->lam)
str(lam)

# drop lambda rows without seed production:
lam<-lam[!is.na(lam$seeds),]
str(lam)

# intermediate subset- plants from plots designated as competition plots ('not lambda')
subset(d, d$plot_type!="Lambda")->nl
str(nl)
# drop rows with no seed production from the not lambda subset:
nl<-nl[!is.na(nl$seeds),]
str(nl)

# set up a data frame of just plants in competition with seed production (keep this):
subset(nl, nl$density>0)->comp

# two data frame to use at this point- 'comp' (plots con competition) and 'lam' (plot de lambda)
splist_all <- c("LACA","LASE")

# list of target species
(splist<-unique(comp$target))
(splist<-paste(splist))

# alpha order splist:
(splist<-splist[order(splist)])

# same for background
(splist_back<-unique(comp$background))
(splist_back<-paste(splist_back))

# alpha order splist:
(splist_back<-splist_back[order(splist_back)])

# objects to hold the final parameter estimates from model 3:
(alpha_matrix<- matrix(0, nrow=length(splist), ncol=length(splist_back)))
(row.names(alpha_matrix)<-splist)
(colnames(alpha_matrix)<-splist_back)
lambda_est<-NULL
sigma_est<-NULL
convergence_code<-NULL

upper_lambda <-NULL
lower_lambda <-NULL

(lower_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(lower_alpha)<-splist_all)
(colnames(lower_alpha)<-splist_all)

(upper_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(upper_alpha)<-splist_all)
(colnames(upper_alpha)<-splist_all)

## for each species in turn as a target:
for(i in 1:length(splist)){
  
  # i=1  
  
  #set up grid for plotting and add first panel indiciating focal sp:
  layout(matrix(1:4, nrow=2))
  plot(x=0, y=0, type='n')
  text(0, 0, paste(splist[i], "as focal", sep=" "), cex=1, col="red")
  
  ## subset out the rows that are needed from the competition df
  (comp_points<-subset(comp, comp$target==splist[i], drop=TRUE))
  
  ## and the correct lambda plants
  (lam_points<-subset(lam, lam$target==splist[i]))
  
  ## now need to build up a vector of nonzero seed production
  ## and corresponding density vectors (held in a matrix) for background species
  ## to use in model fitting
  
  ## start with the lambda seeds- will add on to this for each background:
  (seeds<-lam_points$seeds)
  
  ##build density matrix (each row will be density of a species)
  (dens<-matrix(0, nrow=length(splist_back), ncol=(nrow(lam_points)+ nrow(comp_points))))
  
  (row.names(dens)<-splist_back)
  
  
  ## use this counter to keep track of which column is next to have data added to
  ## set it to begin after the lambda points:
  
  start <- nrow(lam_points) +1
  
  
  for(j in 1:length(splist_back)){
    
    # j=1
    
    ## LOOP creates a seeds vector and a corresponding dens ("density") matrix with
    ## background species as rows, and columns corresponding to the seed production
    ## vector.
    ## beginning columns correspond to lambda plants
    
    #take just the rows pertaining to a specific background sp:
    (bg_points<- subset(comp_points, comp_points$background==splist_back[j]))
    
    ## which row of the density matrix corresponds?
    (rownum <- which(row.names(dens)==splist_back[j]))
    
    #column to end with:
    (end<-start + nrow(bg_points)-1)
    
    ## drop in density values into matrix:
    (dens[rownum, start:end]<-bg_points$density)
    
    ## add seed numbers into the seeds vector
    (seeds<-c(seeds, bg_points$seeds))
    
    ##increase start counter so that next species is offset from this one
    (start<-end+1)
    
  }
  
  ## should now have "seeds" vector and corresponding "dens"ity matrix
  ## can test ncol(dens)==length(seeds)
  
  ## we'll be working with log seeds (for giving a lognormal error structure):
  
  (log_seeds<-log(seeds))
  
  
  #model fitting using optim and earlier likelihood functions
  
  #############################
  ## model 1, no competition ##
  #############################
  
  ###recall parameters are lambda and sigma- initialize these with estimates from the data:
  (par1<-c(mean(log_seeds), sd(log_seeds)))
  
  
  ##repeat optimization until we get convergence (or we try 25 times)
  for(k in 1:25){
    testcomp1<-optim(par1,compmodel1)
    ##update start parameters to final estimate to use in next run in case of nonconvergence
    par1<-testcomp1$par
    if(testcomp1$convergence==0){
      print(paste(splist[i], "model 1 converged on rep", k, sep=" "))
      break
    }
    
  }
  
  
  #############################
  ## model 2, one alpha      ##
  #############################
  
  ## pars here are lambda, alpha and sigma- use lambda and sigma from model 1 as starting esimtates
  (par2<-c(testcomp1$par[1],0.001,testcomp1$par[2]))
  par.scale<- c(1.e02, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before:
  for(k in 1:25){
    ##now using a specific method that permits constrained optimization so that alpha has to be nonzero- this is an issue in some of the fits, especially in model 3. lower parameter has same order as par2
    testcomp2<-optim(par2,compmodel2, method="L-BFGS-B", lower=c(1,0,0.0000000001), control=list(maxit=1000, parscale=par.scale))
    par2<-testcomp2$par
    if(testcomp2$convergence==0){
      print(paste(splist[i],  "model 2 converged on rep", k, sep=" "))
      break
    }
  }
  
  #############################
  ## model 3, unique alphas  ##
  #############################
  
  ## pars here are lambda, 3 alphas and sigma- use lambda and sigma from model 2, and alphas from model 2 as starting esimtate for each species
  
  (par3<-c(testcomp2$par[1], rep(testcomp2$par[2], times=2), testcomp2$par[3]))
  par.scale<- c(1.e02, 1.e-03, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  ##as before
  
  for(k in 1:25){
    ##now using a specific method that permits constained optimization so that alpha has to be nonzero-
    testcomp3<-optim(par3,compmodel3, method="L-BFGS-B", lower=c(1, rep(0, times=2),0.0000000001), control=list(maxit=1000, parscale=par.scale), hessian = T)
    par3<-testcomp3$par
    if(testcomp3$convergence==0){
      print(paste(splist[i], "model 3 converged on rep", k, sep=" "))
      break
    }
  }
  
  
  ##################################
  ### save estimates from model 3 ##
  ##################################
  
  (lambda_est<-c(lambda_est, testcomp3$par[1]))
  (sigma_est<-c(sigma_est, testcomp3$par[4]))
  convergence_code<-c(convergence_code, testcomp3$convergence)
  
  fisher_info<-solve(testcomp3$hessian)
  prop_sigma<-sqrt(diag(fisher_info))
  upper<-testcomp3$par+1.96*prop_sigma
  lower<-testcomp3$par-1.96*prop_sigma
  
  upper_lambda <- c(upper_lambda, upper[1])
  lower_lambda <- c(lower_lambda, lower[1]) 
  
  
  ## in keeping with Lotka Volterra notation, we'll use alpha1_2 to indicate effect of
  ## sp 2 on growth of 1.  Following convention, i refers to rows and j to cols in a matrix
  ## so each step of the loop here (for a target sp) corresponds to one row of this matrix:
  
  alphas<-testcomp3$par[2:3]
  
  alpha_matrix[i,]<-alphas
  
  errors_lower_alpha <-lower[2:3]
  lower_alpha[i,] <- errors_lower_alpha
  errors_upper_alpha <-upper[2:3]
  upper_alpha[i,] <- errors_upper_alpha
  
  ##note that in cases where there is no data for a particular species the alpha estimate for that species ends up as the starting value- we need to be careful of these as they are basically gargbage numbers.  Keeping them in up to now to keep the structure of the data constant, but will set them to NA here:
  
  #identify which species have no data in this fit:
  which(apply(dens, MARGIN=1, FUN=mean)==0)->no_data
  
  ## set their alphas to NA in the matrix:
  
  alpha_matrix[i,no_data]<-NA
  
  
  ###############################
  ## some diagnostics +  plots ##
  ###############################
  
  ## print an error to the console if any one of the three models failed to converge:
  
  if(testcomp1$convergence + testcomp2$convergence + testcomp3$convergence !=0){print(paste("at least one model did not converge for", splist[i], sep=" "))}
  
  ##################################
  ## plot observed vs predicted:
  ##################
  
  par<-testcomp3$par
  
  #from model 3 code:
  lambda<-par[1]
  # a_BRMA<-par[2]
  
  a_LACA<-par[2]
  a_LASE<-par[3]
  
  pred<- lambda / (1+ a_LACA* dens['LACA',] + a_LACA* dens['LASE',])
  
  
  plot(seeds, pred, xlim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), ylim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), log='xy', xlab="observed seeds", ylab="predicted seeds", main=splist[i] )
  
  abline(a=0, b=1, lwd=2)
  
  
  #####################
  #### plot each fit
  ##########
  
  names(alphas)<-splist_back
  
  names(apply(dens, MARGIN=1, FUN=mean)!=0)->plotlist
  
  for(l in 1:length(plotlist)){
    
    ## which columns in the density dataframe have nozero values for species l ?
    cols<-which(dens[l,]>0)
    
    x<-dens[l,cols]
    y<-seeds[cols]
    
    ##add lambdas:
    x<-c(x, rep(0,nrow(lam_points)))
    y<-c(y, lam_points$seeds)
    
    x_det<-seq(min(x), max(x), by=(  (max(x)-min(x))/1000  ))
    
    alpha_temp<-alphas[which(names(alphas)==plotlist[l])]
    y_pred<-lambda/(1 + alpha_temp*x_det)
    
    if(length(unique(x))>1){
      plot(y~x, xlab="germinants_background", ylab="seeds", main=plotlist[l])
    } else {
      
      plot(x=0, y=0, main=plotlist[l], type='n')
      text(0, 0, "no data")
      
    }
    
    lines(y_pred~x_det, col="red", lwd=2)
  }
  
}

dev.off()

results<-data.frame(splist, lambda_est, lower_lambda, upper_lambda, sigma_est, convergence_code)
names(results)<-c("species", "lambda", "lower_error", "upper error", "sigma", "convergence_code")

write.csv(results, "lambda_estimates_LASTHENIA_lactuca_145_160.csv")
write.csv(alpha_matrix, "alpha_estimates_row_is_target_LASTHENIA_lactuca_145_160.csv")
write.csv(lower_alpha, "alpha_lower_errors_control_LASTHENIA_lactuca_145_160.csv")
write.csv(upper_alpha, "alpha_upper_errors_control_LASTHENIA_lactuca_145_160.csv")


#### 165-175 for Bromus and Lactuca together ####
#model 3 - all species have different competitive effects
compmodel3<-function(par){
  lambda<-par[1] #same as model 2
  a_BRMA<-par[2]
  a_LACA<-par[3]
  a_LASE<-par[4]
  sigma<-par[5] ## same as model 2
  
  ##probably overkill to name all the alphas, but it helps keep things straight
  
  # same form as model 1, but super long given all the species:
  pred<- lambda / (1 + a_BRMA* dens['BRMA',] + a_LACA* dens['LACA',] + a_LASE* dens['LASE',])
  
  
  # likelihood as before:
  llik<-dnorm(log_seeds,log(pred), sd=sigma, log=TRUE)
  
  # return sum of negative log likelihoods
  return(sum(-1*llik)) #sum of negative log likelihoods
}

(d<-d4_bromlact)

pdf(file="alphas_and_lambdas_LASTHENIA_bromlact_165_175.pdf", width=11, height=8)

# plants from plots designated as lambda plots:
subset(d, d$plot_type=="Lambda")->lam
str(lam)

# drop lambda rows without seed production:
lam<-lam[!is.na(lam$seeds),]
str(lam)

# intermediate subset- plants from plots designated as competition plots ('not lambda')
subset(d, d$plot_type!="Lambda")->nl
str(nl)
# drop rows with no seed production from the not lambda subset:
nl<-nl[!is.na(nl$seeds),]
str(nl)

# set up a data frame of just plants in competition with seed production (keep this):
subset(nl, nl$density>0)->comp

# two data frame to use at this point- 'comp' (plots con competition) and 'lam' (plot de lambda)
splist_all <- c("BRMA","LACA","LASE")

# list of target species
(splist<-unique(comp$target))
(splist<-paste(splist))

# alpha order splist:
(splist<-splist[order(splist)])

# same for background
(splist_back<-unique(comp$background))
(splist_back<-paste(splist_back))

# alpha order splist:
(splist_back<-splist_back[order(splist_back)])

# objects to hold the final parameter estimates from model 3:
(alpha_matrix<- matrix(0, nrow=length(splist), ncol=length(splist_back)))
(row.names(alpha_matrix)<-splist)
(colnames(alpha_matrix)<-splist_back)
lambda_est<-NULL
sigma_est<-NULL
convergence_code<-NULL

upper_lambda <-NULL
lower_lambda <-NULL

(lower_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(lower_alpha)<-splist_all)
(colnames(lower_alpha)<-splist_all)

(upper_alpha<- matrix(0, nrow=length(splist_all), ncol=length(splist_all)))
(row.names(upper_alpha)<-splist_all)
(colnames(upper_alpha)<-splist_all)

## for each species in turn as a target:
for(i in 1:length(splist)){
  
  # i=1  
  
  #set up grid for plotting and add first panel indiciating focal sp:
  layout(matrix(1:6, nrow=3))
  plot(x=0, y=0, type='n')
  text(0, 0, paste(splist[i], "as focal", sep=" "), cex=1, col="red")
  
  ## subset out the rows that are needed from the competition df
  (comp_points<-subset(comp, comp$target==splist[i], drop=TRUE))
  
  ## and the correct lambda plants
  (lam_points<-subset(lam, lam$target==splist[i]))
  
  ## now need to build up a vector of nonzero seed production
  ## and corresponding density vectors (held in a matrix) for background species
  ## to use in model fitting
  
  ## start with the lambda seeds- will add on to this for each background:
  (seeds<-lam_points$seeds)
  
  ##build density matrix (each row will be density of a species)
  (dens<-matrix(0, nrow=length(splist_back), ncol=(nrow(lam_points)+ nrow(comp_points))))
  
  (row.names(dens)<-splist_back)
  
  ## use this counter to keep track of which column is next to have data added to
  ## set it to begin after the lambda points:
  
  start <- nrow(lam_points) +1
  
  
  for(j in 1:length(splist_back)){
    
    # j=3
    
    ## LOOP creates a seeds vector and a corresponding dens ("density") matrix with
    ## background species as rows, and columns corresponding to the seed production
    ## vector.
    ## beginning columns correspond to lambda plants
    
    #take just the rows pertaining to a specific background sp:
    (bg_points<- subset(comp_points, comp_points$background==splist_back[j]))
    
    ## which row of the density matrix corresponds?
    (rownum <- which(row.names(dens)==splist_back[j]))
    
    #column to end with:
    (end<-start + nrow(bg_points)-1)
    
    ## drop in density values into matrix:
    (dens[rownum, start:end]<-bg_points$density)
    
    ## add seed numbers into the seeds vector
    (seeds<-c(seeds, bg_points$seeds))
    
    ##increase start counter so that next species is offset from this one
    (start<-end+1)
    
  }
  
  ## should now have "seeds" vector and corresponding "dens"ity matrix
  ## can test ncol(dens)==length(seeds)
  
  ## we'll be working with log seeds (for giving a lognormal error structure):
  
  (log_seeds<-log(seeds))
  
  
  #model fitting using optim and earlier likelihood functions
  
  #############################
  ## model 1, no competition ##
  #############################
  
  ###recall parameters are lambda and sigma- initialize these with estimates from the data:
  (par1<-c(mean(log_seeds), sd(log_seeds)))
  
  
  ##repeat optimization until we get convergence (or we try 25 times)
  for(k in 1:25){
    testcomp1<-optim(par1,compmodel1)
    ##update start parameters to final estimate to use in next run in case of nonconvergence
    par1<-testcomp1$par
    if(testcomp1$convergence==0){
      print(paste(splist[i], "model 1 converged on rep", k, sep=" "))
      break
    }
    
  }
  
  
  #############################
  ## model 2, one alpha      ##
  #############################
  
  ## pars here are lambda, alpha and sigma- use lambda and sigma from model 1 as starting esimtates
  (par2<-c(testcomp1$par[1],0.001,testcomp1$par[2]))
  par.scale<- c(1.e02, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before:
  for(k in 1:25){
    ##now using a specific method that permits constrained optimization so that alpha has to be nonzero- this is an issue in some of the fits, especially in model 3. lower parameter has same order as par2
    testcomp2<-optim(par2,compmodel2, method="L-BFGS-B", lower=c(1,0,0.0000000001), control=list(maxit=1000, parscale=par.scale))
    par2<-testcomp2$par
    if(testcomp2$convergence==0){
      print(paste(splist[i],  "model 2 converged on rep", k, sep=" "))
      break
    }
  }
  
  #############################
  ## model 3, unique alphas  ##
  #############################
  
  ## pars here are lambda, 3 alphas and sigma- use lambda and sigma from model 2, and alphas from model 2 as starting esimtate for each species
  
  (par3<-c(testcomp2$par[1], rep(testcomp2$par[2], times=3), testcomp2$par[3]))
  par.scale<- c(1.e02, 1.e-03, 1.e-03,1.e-03, 1) ##this is to avoid problems of convergence due to scale
  ##as before
  
  for(k in 1:25){
    ##now using a specific method that permits constained optimization so that alpha has to be nonzero-
    testcomp3<-optim(par3,compmodel3, method="L-BFGS-B", lower=c(1, rep(0, times=3),0.0000000001), control=list(maxit=1000, parscale=par.scale), hessian = T)
    par3<-testcomp3$par
    if(testcomp3$convergence==0){
      print(paste(splist[i], "model 3 converged on rep", k, sep=" "))
      break
    }
  }
  
  
  ##################################
  ### save estimates from model 3 ##
  ##################################
  
  (lambda_est<-c(lambda_est, testcomp3$par[1]))
  (sigma_est<-c(sigma_est, testcomp3$par[4]))
  convergence_code<-c(convergence_code, testcomp3$convergence)
  
  fisher_info<-solve(testcomp3$hessian)
  prop_sigma<-sqrt(diag(fisher_info))
  upper<-testcomp3$par+1.96*prop_sigma
  lower<-testcomp3$par-1.96*prop_sigma
  
  upper_lambda <- c(upper_lambda, upper[1])
  lower_lambda <- c(lower_lambda, lower[1]) 
  
  
  ## in keeping with Lotka Volterra notation, we'll use alpha1_2 to indicate effect of
  ## sp 2 on growth of 1.  Following convention, i refers to rows and j to cols in a matrix
  ## so each step of the loop here (for a target sp) corresponds to one row of this matrix:
  
  alphas<-testcomp3$par[2:4]
  
  alpha_matrix[i,]<-alphas
  
  errors_lower_alpha <-lower[2:4]
  lower_alpha[i,] <- errors_lower_alpha
  errors_upper_alpha <-upper[2:4]
  upper_alpha[i,] <- errors_upper_alpha
  
  ##note that in cases where there is no data for a particular species the alpha estimate for that species ends up as the starting value- we need to be careful of these as they are basically gargbage numbers.  Keeping them in up to now to keep the structure of the data constant, but will set them to NA here:
  
  #identify which species have no data in this fit:
  which(apply(dens, MARGIN=1, FUN=mean)==0)->no_data
  
  ## set their alphas to NA in the matrix:
  
  alpha_matrix[i,no_data]<-NA
  
  
  ###############################
  ## some diagnostics +  plots ##
  ###############################
  
  ## print an error to the console if any one of the three models failed to converge:
  
  if(testcomp1$convergence + testcomp2$convergence + testcomp3$convergence !=0){print(paste("at least one model did not converge for", splist[i], sep=" "))}
  
  ##################################
  ## plot observed vs predicted:
  ##################
  
  par<-testcomp3$par
  
  #from model 3 code:
  lambda<-par[1]
  a_BRMA<-par[2]
  a_LACA<-par[3]
  a_LASE<-par[4]
  
  pred<- lambda / (1+ a_BRMA * dens['BRMA',] + a_LACA* dens['LACA',] + a_LASE* dens['LASE',])
  
  
  plot(seeds, pred, xlim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), ylim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), log='xy', xlab="observed seeds", ylab="predicted seeds", main=splist[i] )
  
  abline(a=0, b=1, lwd=2)
  
  
  #####################
  #### plot each fit
  ##########
  
  names(alphas)<-splist_back
  
  names(apply(dens, MARGIN=1, FUN=mean)!=0)->plotlist
  
  for(l in 1:length(plotlist)){
    
    ## which columns in the density dataframe have nozero values for species l ?
    cols<-which(dens[l,]>0)
    
    x<-dens[l,cols]
    y<-seeds[cols]
    
    ##add lambdas:
    x<-c(x, rep(0,nrow(lam_points)))
    y<-c(y, lam_points$seeds)
    
    x_det<-seq(min(x), max(x), by=(  (max(x)-min(x))/1000  ))
    
    alpha_temp<-alphas[which(names(alphas)==plotlist[l])]
    y_pred<-lambda/(1 + alpha_temp*x_det)
    
    if(length(unique(x))>1){
      plot(y~x, xlab="germinants_background", ylab="seeds", main=plotlist[l])
    } else {
      
      plot(x=0, y=0, main=plotlist[l], type='n')
      text(0, 0, "no data")
      
    }
    
    lines(y_pred~x_det, col="red", lwd=2)
  }
  
}

dev.off()

results<-data.frame(splist, lambda_est, lower_lambda, upper_lambda, sigma_est, convergence_code)
names(results)<-c("species", "lambda", "lower_error", "upper error", "sigma", "convergence_code")

write.csv(results, "lambda_estimates_LASTHENIA_bromlact_165_175.csv")
write.csv(alpha_matrix, "alpha_estimates_row_is_target_LASTHENIA_bromlact_165_175.csv")
write.csv(lower_alpha, "alpha_lower_errors_control_LASTHENIA_bromlact_165_175.csv")
write.csv(upper_alpha, "alpha_upper_errors_control_LASTHENIA_bromlact_165_175.csv")


#### BROMUS as focal ####
# model 3 - all species have different competitive effects (when Bromus is the focal, competing against itself and with Lasthenia)
compmodel3<-function(par){
  lambda<-par[1] #same as model 2
  a_BRMA<-par[2]	
  a_LACA<-par[3]
  # a_LASE<-par[4]
  sigma<-par[4] ## same as model 2
  
  ##probably overkill to name all the alphas, but it helps keep things straight
  
  # same form as model 1, but super long given all the species:
  pred<- lambda / (1+ a_BRMA* dens['BRMA',] + a_LACA* dens['LACA',])
  
  
  # likelihood as before:
  llik<-dnorm(log_seeds,log(pred), sd=sigma, log=TRUE)
  
  # return sum of negative log likelihoods
  return(sum(-1*llik)) #sum of negative log likelihoods
}

(d<-d1_bromus_focal)

pdf(file="alphas_and_lambdas_BROMUS.pdf", width=11, height=8)

# plants from plots designated as lambda plots:
subset(d, d$plot_type=="Lambda")->lam
str(lam)

# drop lambda rows without seed production:
lam<-lam[!is.na(lam$seeds),]
str(lam)

# intermediate subset- plants from plots designated as competition plots ('not lambda')
subset(d, d$plot_type!="Lambda")->nl
str(nl)
# drop rows with no seed production from the not lambda subset:
nl<-nl[!is.na(nl$seeds),]
str(nl)

# set up a data frame of just plants in competition with seed production (keep this):
subset(nl, nl$density>0)->comp

# two data frame to use at this point- 'comp' (plots con competition) and 'lam' (plot de lambda)
splist_all <- c("BRMA","LACA","LASE")

# list of target species
(splist<-unique(comp$target))
(splist<-paste(splist))

# alpha order splist:
(splist<-splist[order(splist)])

# same for background
(splist_back<-unique(comp$background))
(splist_back<-paste(splist_back))

# alpha order splist:
(splist_back<-splist_back[order(splist_back)])

# objects to hold the final parameter estimates from model 3:
(alpha_matrix<- matrix(0, nrow=length(splist), ncol=length(splist_back)))
(row.names(alpha_matrix)<-splist)
(colnames(alpha_matrix)<-splist_back)
lambda_est<-NULL
sigma_est<-NULL
convergence_code<-NULL

upper_lambda <-NULL
lower_lambda <-NULL

(lower_alpha<- matrix(0, nrow=length(splist_back), ncol=length(splist_back)))
(row.names(lower_alpha)<-splist_back)
(colnames(lower_alpha)<-splist_back)

(upper_alpha<- matrix(0, nrow=length(splist_back), ncol=length(splist_back)))
(row.names(upper_alpha)<-splist_back)
(colnames(upper_alpha)<-splist_back)

## for each species in turn as a target:
for(i in 1:length(splist)){
  
  # i=1  
  
  #set up grid for plotting and add first panel indiciating focal sp:
  layout(matrix(1:4, nrow=2))
  plot(x=0, y=0, type='n')
  text(0, 0, paste(splist[i], "as focal", sep=" "), cex=1, col="red")
  
  ## subset out the rows that are needed from the competition df
  (comp_points<-subset(comp, comp$target==splist[i], drop=TRUE))
  
  ## and the correct lambda plants
  (lam_points<-subset(lam, lam$target==splist[i]))
  
  ## now need to build up a vector of nonzero seed production
  ## and corresponding density vectors (held in a matrix) for background species
  ## to use in model fitting
  
  ## start with the lambda seeds- will add on to this for each background:
  (seeds<-lam_points$seeds)
  
  ##build density matrix (each row will be density of a species)
  (dens<-matrix(0, nrow=length(splist_back), ncol=(nrow(lam_points)+ nrow(comp_points))))
  
  (row.names(dens)<-splist_back)
  
  ## use this counter to keep track of which column is next to have data added to
  ## set it to begin after the lambda points:
  
  start <- nrow(lam_points) +1
  
  
  for(j in 1:length(splist_back)){
    
    # j=2
    
    ## LOOP creates a seeds vector and a corresponding dens ("density") matrix with
    ## background species as rows, and columns corresponding to the seed production
    ## vector.
    ## beginning columns correspond to lambda plants
    
    #take just the rows pertaining to a specific background sp:
    (bg_points<- subset(comp_points, comp_points$background==splist_back[j]))
    
    ## which row of the density matrix corresponds?
    (rownum <- which(row.names(dens)==splist_back[j]))
    
    #column to end with:
    (end<-start + nrow(bg_points)-1)
    
    ## drop in density values into matrix:
    (dens[rownum, start:end]<-bg_points$density)
    
    ## add seed numbers into the seeds vector
    (seeds<-c(seeds, bg_points$seeds))
    
    ##increase start counter so that next species is offset from this one
    (start<-end+1)
    
  }
  
  ## should now have "seeds" vector and corresponding "dens"ity matrix
  ## can test ncol(dens)==length(seeds)
  
  ## we'll be working with log seeds (for giving a lognormal error structure):
  
  (log_seeds<-log(seeds))
  
  
  #model fitting using optim and earlier likelihood functions
  
  #############################
  ## model 1, no competition ##
  #############################
  
  ###recall parameters are lambda and sigma- initialize these with estimates from the data:
  (par1<-c(mean(log_seeds), sd(log_seeds)))
  
  ##repeat optimization until we get convergence (or we try 25 times)
  for(k in 1:25){
    testcomp1<-optim(par1,compmodel1)
    ##update start parameters to final estimate to use in next run in case of nonconvergence
    par1<-testcomp1$par
    if(testcomp1$convergence==0){
      print(paste(splist[i], "model 1 converged on rep", k, sep=" "))
      break
    }
    
  }
  
  
  #############################
  ## model 2, one alpha      ##
  #############################
    ## pars here are lambda, alpha and sigma- use lambda and sigma from model 1 as starting esimtates
  (par2<-c(testcomp1$par[1],0.001,testcomp1$par[2]))
  par.scale<- c(1.e02, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before:
  for(k in 1:25){
    ##now using a specific method that permits constrained optimization so that alpha has to be nonzero- this is an issue in some of the fits, especially in model 3. lower parameter has same order as par2
    testcomp2<-optim(par2,compmodel2, method="L-BFGS-B", lower=c(1,0,0.0000000001), control=list(maxit=1000, parscale=par.scale))
    par2<-testcomp2$par
    if(testcomp2$convergence==0){
      print(paste(splist[i],  "model 2 converged on rep", k, sep=" "))
      break
    }
  }
  
  #############################
  ## model 3, unique alphas  ##
  #############################
  
  ## pars here are lambda, TWO alphas and sigma- use lambda and sigma from model 2, and alphas from model 2 as starting esimtate for each species
  (par3<-c(testcomp2$par[1], rep(testcomp2$par[2], times=2), testcomp2$par[3]))
  par.scale<- c(1.e02, 1.e-03, 1.e-03,1) ##this is to avoid problems of convergence due to scale
  ##as before
  
  for(k in 1:25){
    # k =1
    ##now using a specific method that permits constained optimization so that alpha has to be nonzero-
    testcomp3<-optim(par3,compmodel3, method="L-BFGS-B", lower=c(1, rep(0, times=2),0.0000000001), control=list(maxit=1000, parscale=par.scale), hessian = T)
    par3<-testcomp3$par
    if(testcomp3$convergence==0){
      print(paste(splist[i], "model 3 converged on rep", k, sep=" "))
      break
    }
  }
  
  ##################################
  ### save estimates from model 3 ##
  ##################################
  
  (lambda_est<-c(lambda_est, testcomp3$par[1]))
  (sigma_est<-c(sigma_est, testcomp3$par[4]))
  convergence_code<-c(convergence_code, testcomp3$convergence)
  
  fisher_info<-solve(testcomp3$hessian)
  prop_sigma<-sqrt(diag(fisher_info))
  upper<-testcomp3$par+1.96*prop_sigma
  lower<-testcomp3$par-1.96*prop_sigma
  
  upper_lambda <- c(upper_lambda, upper[1])
  lower_lambda <- c(lower_lambda, lower[1]) 
  
  ## in keeping with Lotka Volterra notation, we'll use alpha1_2 to indicate effect of
  ## sp 2 on growth of 1.  Following convention, i refers to rows and j to cols in a matrix
  ## so each step of the loop here (for a target sp) corresponds to one row of this matrix:
  
  alphas<-testcomp3$par[2:3]
  
  alpha_matrix[i,]<-alphas
  
  errors_lower_alpha <-lower[2:3]
  lower_alpha[i,] <- errors_lower_alpha
  errors_upper_alpha <-upper[2:3]
  upper_alpha[i,] <- errors_upper_alpha
  
  ##note that in cases where there is no data for a particular species the alpha estimate for that species ends up as the starting value- we need to be careful of these as they are basically garbage numbers.  Keeping them in up to now to keep the structure of the data constant, but will set them to NA here:
  
  #identify which species have no data in this fit:
  which(apply(dens, MARGIN=1, FUN=mean)==0)->no_data
  
  ## set their alphas to NA in the matrix:
  
  alpha_matrix[i,no_data]<-NA
  
  
  ###############################
  ## some diagnostics +  plots ##
  ###############################
  
  ## print an error to the console if any one of the three models failed to converge:
  
  if(testcomp1$convergence + testcomp2$convergence + testcomp3$convergence !=0){print(paste("at least one model did not converge for", splist[i], sep=" "))}
  
  ##################################
  ## plot observed vs predicted:
  ##################
  
  par<-testcomp3$par
  
  #from model 3 code:
  lambda<-par[1]
  a_BRMA<-par[2]
  a_LACA<-par[3]
  # a_LASE<-par[4]
  
  pred<- lambda / (1+ a_BRMA * dens['BRMA',] + a_LACA* dens['LACA',])
  
  
  plot(seeds, pred, xlim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), ylim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), log='xy', xlab="observed seeds", ylab="predicted seeds", main=splist[i] )
  
  abline(a=0, b=1, lwd=2)
  
  
  #####################
  #### plot each fit
  ##########
  
  names(alphas)<-splist_back
  
  names(apply(dens, MARGIN=1, FUN=mean)!=0)->plotlist
  
  for(l in 1:length(plotlist)){
    
    ## which columns in the density dataframe have nozero values for species l ?
    cols<-which(dens[l,]>0)
    
    x<-dens[l,cols]
    y<-seeds[cols]
    
    ##add lambdas:
    x<-c(x, rep(0,nrow(lam_points)))
    y<-c(y, lam_points$seeds)
    
    x_det<-seq(min(x), max(x), by=(  (max(x)-min(x))/1000  ))
    
    alpha_temp<-alphas[which(names(alphas)==plotlist[l])]
    y_pred<-lambda/(1 + alpha_temp*x_det)
    
    if(length(unique(x))>1){
      plot(y~x, xlab="germinants_background", ylab="seeds", main=plotlist[l])
    } else {
      
      plot(x=0, y=0, main=plotlist[l], type='n')
      text(0, 0, "no data")
      
    }
    
    lines(y_pred~x_det, col="red", lwd=2)
  }
  
}

dev.off()

results<-data.frame(splist, lambda_est, lower_lambda, upper_lambda, sigma_est, convergence_code)
names(results)<-c("species", "lambda", "lower_error", "upper error", "sigma", "convergence_code")

write.csv(results, "lambda_estimates_BROMUS.csv")
write.csv(alpha_matrix, "alpha_estimates_row_is_target_BROMUS.csv")
write.csv(lower_alpha, "alpha_lower_errors_control_BROMUS.csv")
write.csv(upper_alpha, "alpha_upper_errors_control_BROMUS.csv")


#
#### LACTUCA as focal ####
# model 3 - all species have different competitive effects (when Bromus is the focal, competing against itself and with Lasthenia)
compmodel3<-function(par){
  lambda<-par[1] #same as model 2
  # a_BRMA<-par[2]	
  a_LACA<-par[2]
  a_LASE<-par[3]
  sigma<-par[4] ## same as model 2
  
  ##probably overkill to name all the alphas, but it helps keep things straight
  
  # same form as model 1, but super long given all the species:
  pred<- lambda / (1+ a_LACA* dens['LACA',] + a_LASE* dens['LASE',])
  
  
  # likelihood as before:
  llik<-dnorm(log_seeds,log(pred), sd=sigma, log=TRUE)
  
  # return sum of negative log likelihoods
  return(sum(-1*llik)) #sum of negative log likelihoods
}

(d<-d1_lactuca_focal)

pdf(file="alphas_and_lambdas_LACTUCA.pdf", width=11, height=8)

# plants from plots designated as lambda plots:
subset(d, d$plot_type=="Lambda")->lam
str(lam)

# drop lambda rows without seed production:
lam<-lam[!is.na(lam$seeds),]
str(lam)

# intermediate subset- plants from plots designated as competition plots ('not lambda')
subset(d, d$plot_type!="Lambda")->nl
str(nl)
# drop rows with no seed production from the not lambda subset:
nl<-nl[!is.na(nl$seeds),]
str(nl)

# set up a data frame of just plants in competition with seed production (keep this):
subset(nl, nl$density>0)->comp

# two data frame to use at this point- 'comp' (plots con competition) and 'lam' (plot de lambda)
splist_all <- c("LACA","LASE")

# list of target species
(splist<-unique(comp$target))
(splist<-paste(splist))

# alpha order splist:
(splist<-splist[order(splist)])

# same for background
(splist_back<-unique(comp$background))
(splist_back<-paste(splist_back))

# alpha order splist:
(splist_back<-splist_back[order(splist_back)])

# objects to hold the final parameter estimates from model 3:
(alpha_matrix<- matrix(0, nrow=length(splist), ncol=length(splist_back)))
(row.names(alpha_matrix)<-splist)
(colnames(alpha_matrix)<-splist_back)
lambda_est<-NULL
sigma_est<-NULL
convergence_code<-NULL

upper_lambda <-NULL
lower_lambda <-NULL

(lower_alpha<- matrix(0, nrow=length(splist_back), ncol=length(splist_back)))
(row.names(lower_alpha)<-splist_back)
(colnames(lower_alpha)<-splist_back)

(upper_alpha<- matrix(0, nrow=length(splist_back), ncol=length(splist_back)))
(row.names(upper_alpha)<-splist_back)
(colnames(upper_alpha)<-splist_back)

## for each species in turn as a target:
for(i in 1:length(splist)){
  
  # i=1  
  
  #set up grid for plotting and add first panel indiciating focal sp:
  layout(matrix(1:4, nrow=2))
  plot(x=0, y=0, type='n')
  text(0, 0, paste(splist[i], "as focal", sep=" "), cex=1, col="red")
  
  ## subset out the rows that are needed from the competition df
  (comp_points<-subset(comp, comp$target==splist[i], drop=TRUE))
  
  ## and the correct lambda plants
  (lam_points<-subset(lam, lam$target==splist[i]))
  
  ## now need to build up a vector of nonzero seed production
  ## and corresponding density vectors (held in a matrix) for background species
  ## to use in model fitting
  
  ## start with the lambda seeds- will add on to this for each background:
  (seeds<-lam_points$seeds)
  
  ##build density matrix (each row will be density of a species)
  (dens<-matrix(0, nrow=length(splist_back), ncol=(nrow(lam_points)+ nrow(comp_points))))
  
  (row.names(dens)<-splist_back)
  
  ## use this counter to keep track of which column is next to have data added to
  ## set it to begin after the lambda points:
  
  start <- nrow(lam_points) +1
  
  
  for(j in 1:length(splist_back)){
    
    # j=1
    # j=2
    
    ## LOOP creates a seeds vector and a corresponding dens ("density") matrix with
    ## background species as rows, and columns corresponding to the seed production
    ## vector.
    ## beginning columns correspond to lambda plants
    
    #take just the rows pertaining to a specific background sp:
    (bg_points<- subset(comp_points, comp_points$background==splist_back[j]))
    
    ## which row of the density matrix corresponds?
    (rownum <- which(row.names(dens)==splist_back[j]))
    
    #column to end with:
    (end<-start + nrow(bg_points)-1)
    
    ## drop in density values into matrix:
    (dens[rownum, start:end]<-bg_points$density)
    
    ## add seed numbers into the seeds vector
    (seeds<-c(seeds, bg_points$seeds))
    
    ##increase start counter so that next species is offset from this one
    (start<-end+1)
    
  }
  
  ## should now have "seeds" vector and corresponding "dens"ity matrix
  ## can test ncol(dens)==length(seeds)
  
  ## we'll be working with log seeds (for giving a lognormal error structure):
  
  (log_seeds<-log(seeds))
  
  
  #model fitting using optim and earlier likelihood functions
  
  #############################
  ## model 1, no competition ##
  #############################
  
  ###recall parameters are lambda and sigma- initialize these with estimates from the data:
  (par1<-c(mean(log_seeds), sd(log_seeds)))
  
  ##repeat optimization until we get convergence (or we try 25 times)
  for(k in 1:25){
    testcomp1<-optim(par1,compmodel1)
    ##update start parameters to final estimate to use in next run in case of nonconvergence
    par1<-testcomp1$par
    if(testcomp1$convergence==0){
      print(paste(splist[i], "model 1 converged on rep", k, sep=" "))
      break
    }
    
  }
  
  
  #############################
  ## model 2, one alpha      ##
  #############################
  ## pars here are lambda, alpha and sigma- use lambda and sigma from model 1 as starting esimtates
  (par2<-c(testcomp1$par[1],0.001,testcomp1$par[2]))
  par.scale<- c(1.e02, 1.e-03, 1) ##this is to avoid problems of convergence due to scale
  
  ##as before:
  for(k in 1:25){
    ##now using a specific method that permits constrained optimization so that alpha has to be nonzero- this is an issue in some of the fits, especially in model 3. lower parameter has same order as par2
    testcomp2<-optim(par2,compmodel2, method="L-BFGS-B", lower=c(1,0,0.0000000001), control=list(maxit=1000, parscale=par.scale))
    par2<-testcomp2$par
    if(testcomp2$convergence==0){
      print(paste(splist[i],  "model 2 converged on rep", k, sep=" "))
      break
    }
  }
  
  #############################
  ## model 3, unique alphas  ##
  #############################
  
  ## pars here are lambda, TWO alphas and sigma- use lambda and sigma from model 2, and alphas from model 2 as starting estimate for each species
  (par3<-c(testcomp2$par[1], rep(testcomp2$par[2], times=2), testcomp2$par[3]))
  par.scale<- c(1.e02, 1.e-03, 1.e-03,1) ##this is to avoid problems of convergence due to scale
  ##as before
  
  for(k in 1:25){
    # k =1
    ##now using a specific method that permits constained optimization so that alpha has to be nonzero-
    testcomp3<-optim(par3,compmodel3, method="L-BFGS-B", lower=c(1, rep(0, times=2),0.0000000001), control=list(maxit=1000, parscale=par.scale), hessian = T)
    par3<-testcomp3$par
    if(testcomp3$convergence==0){
      print(paste(splist[i], "model 3 converged on rep", k, sep=" "))
      break
    }
  }
  
  ##################################
  ### save estimates from model 3 ##
  ##################################
  
  (lambda_est<-c(lambda_est, testcomp3$par[1]))
  (sigma_est<-c(sigma_est, testcomp3$par[4]))
  convergence_code<-c(convergence_code, testcomp3$convergence)
  
  fisher_info<-solve(testcomp3$hessian)
  prop_sigma<-sqrt(diag(fisher_info))
  upper<-testcomp3$par+1.96*prop_sigma
  lower<-testcomp3$par-1.96*prop_sigma
  
  upper_lambda <- c(upper_lambda, upper[1])
  lower_lambda <- c(lower_lambda, lower[1]) 
  
  ## in keeping with Lotka Volterra notation, we'll use alpha1_2 to indicate effect of
  ## sp 2 on growth of 1.  Following convention, i refers to rows and j to cols in a matrix
  ## so each step of the loop here (for a target sp) corresponds to one row of this matrix:
  
  alphas<-testcomp3$par[2:3]
  
  alpha_matrix[i,]<-alphas
  
  errors_lower_alpha <-lower[2:3]
  lower_alpha[i,] <- errors_lower_alpha
  errors_upper_alpha <-upper[2:3]
  upper_alpha[i,] <- errors_upper_alpha
  
  ##note that in cases where there is no data for a particular species the alpha estimate for that species ends up as the starting value- we need to be careful of these as they are basically garbage numbers.  Keeping them in up to now to keep the structure of the data constant, but will set them to NA here:
  
  #identify which species have no data in this fit:
  which(apply(dens, MARGIN=1, FUN=mean)==0)->no_data
  
  ## set their alphas to NA in the matrix:
  
  alpha_matrix[i,no_data]<-NA
  
  
  ###############################
  ## some diagnostics +  plots ##
  ###############################
  
  ## print an error to the console if any one of the three models failed to converge:
  
  if(testcomp1$convergence + testcomp2$convergence + testcomp3$convergence !=0){print(paste("at least one model did not converge for", splist[i], sep=" "))}
  
  ##################################
  ## plot observed vs predicted:
  ##################
  
  par<-testcomp3$par
  
  #from model 3 code:
  lambda<-par[1]
  # a_BRMA<-par[2]
  a_LACA<-par[2]
  a_LASE<-par[3]
  
  pred<- lambda / (1+ a_LACA * dens['LACA',] + a_LASE* dens['LASE',])
  
  
  plot(seeds, pred, xlim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), ylim=c(min(c(seeds, pred)), max(c(seeds, pred)) ), log='xy', xlab="observed seeds", ylab="predicted seeds", main=splist[i] )
  
  abline(a=0, b=1, lwd=2)
  
  
  #####################
  #### plot each fit
  ##########
  
  names(alphas)<-splist_back
  
  names(apply(dens, MARGIN=1, FUN=mean)!=0)->plotlist
  
  for(l in 1:length(plotlist)){
    
    ## which columns in the density dataframe have nozero values for species l ?
    cols<-which(dens[l,]>0)
    
    x<-dens[l,cols]
    y<-seeds[cols]
    
    ##add lambdas:
    x<-c(x, rep(0,nrow(lam_points)))
    y<-c(y, lam_points$seeds)
    
    x_det<-seq(min(x), max(x), by=(  (max(x)-min(x))/1000  ))
    
    alpha_temp<-alphas[which(names(alphas)==plotlist[l])]
    y_pred<-lambda/(1 + alpha_temp*x_det)
    
    if(length(unique(x))>1){
      plot(y~x, xlab="germinants_background", ylab="seeds", main=plotlist[l])
    } else {
      
      plot(x=0, y=0, main=plotlist[l], type='n')
      text(0, 0, "no data")
      
    }
    
    lines(y_pred~x_det, col="red", lwd=2)
  }
  
}

dev.off()

results<-data.frame(splist, lambda_est, lower_lambda, upper_lambda, sigma_est, convergence_code)
names(results)<-c("species", "lambda", "lower_error", "upper error", "sigma", "convergence_code")

write.csv(results, "lambda_estimates_LACTUCA.csv")
write.csv(alpha_matrix, "alpha_estimates_row_is_target_LACTUCA.csv")
write.csv(lower_alpha, "alpha_lower_errors_control_LACTUCA.csv")
write.csv(upper_alpha, "alpha_upper_errors_control_LACTUCA.csv")

