#include <rand_gen.h>
#include <ctime>
#include <cmath>

const int IM1 = 2147483563;
const int IM2 = 2147483399;
const int IA1 = 40014;
const int IA2 = 40692;
const int IQ1 = 53668;
const int IQ2 = 53774;
const int IR1 = 12211;
const int IR2 = 3791;
const int IMM1 = (IM1-1);
const double EPS = 1.2e-7;
//const double EPS = 1e-16;
const double AM = (1.0/IM1);
const double NDIV = (1+(double)IMM1/(double)NTAB);
const double RNMAX = (1.0-EPS);

/******************************************/
/* Initialise the random number generator */
/* min : lower bound for random number    */
/* max : upper bound for random number    */
/* seed : initial seed.                   */
/*        -1 -> use current time          */
/******************************************/
rand_gen::rand_gen(double min, double max,long seed) :
  rand_seed1_(0),rand_seed2_(123456789),min_(min), max_(max), normal_iset_(0),
  normal_cap_iset_(0)
{ int j;
  long k;

  /* by default we take the current time as the seed */
  if (seed  < 1) rand_seed1_ = (long )time(0);
  else rand_seed1_ = seed;

  rand_seed1_ = rand_seed1_ & 0xffffffff;
  rand_seed2_ = rand_seed1_;

  for(j = NTAB+7; j >= 0; j--)
  { k = rand_seed1_/IQ1;
    rand_seed1_ = IA1*(rand_seed1_-k*IQ1)-k*IR1;
    if (rand_seed1_ < 0) rand_seed1_ += IM1;
    if (j < NTAB) iv_[j] = rand_seed1_;    
  }
  iy_ = iv_[0];

}

/*************************************************/
/* Return the next number in the range ]min,max[ */
/*************************************************/
double rand_gen::next()
{ int j;
  long k;
  double temp;
    
  k = rand_seed1_/IQ1;
  rand_seed1_ = IA1*(rand_seed1_-k*IQ1)-k*IR1;
  if (rand_seed1_ < 0) rand_seed1_ += IM1;
  k = rand_seed2_/IQ2;
  rand_seed2_ = IA2*(rand_seed2_-k*IQ2)-k*IR2;
  if (rand_seed2_ < 0) rand_seed2_ += IM2;
  
  j = ((int) (iy_/NDIV))%NTAB;
  iy_ = iv_[j]-rand_seed2_;
  iv_[j] = rand_seed1_;
  if(iy_ < 1) iy_ += IMM1;
  if((temp=AM*iy_) > RNMAX)  temp = RNMAX;
  if (temp <= 0) temp = EPS;
  // temp is in the range ]0,1[

  return(min_+(max_-min_)*temp);
}

/*********************************************/
/* Return the next number in the range ]0,1[ */
/*********************************************/
double rand_gen::next_0_1()
{ int j;
  long k;
  double temp;
    
  k = rand_seed1_/IQ1;
  rand_seed1_ = IA1*(rand_seed1_-k*IQ1)-k*IR1;
  if (rand_seed1_ < 0) rand_seed1_ += IM1;
  k = rand_seed2_/IQ2;
  rand_seed2_ = IA2*(rand_seed2_-k*IQ2)-k*IR2;
  if (rand_seed2_ < 0) rand_seed2_ += IM2;
  
  j = ((int) (iy_/NDIV))%NTAB;
  iy_ = iv_[j]-rand_seed2_;
  iv_[j] = rand_seed1_;
  if(iy_ < 1) iy_ += IMM1;
  if((temp=AM*iy_) > RNMAX)  temp = RNMAX;
  if (temp <= 0) temp = EPS;
  // temp is in the range ]0,1[

  return(temp);
}

/*************************************************/
/* Return the next number in the range ]-1,1[    */
/*************************************************/
double rand_gen::next_m1_1()
{ int j;
  long k;
  double temp;
    
  k = rand_seed1_/IQ1;
  rand_seed1_ = IA1*(rand_seed1_-k*IQ1)-k*IR1;
  if (rand_seed1_ < 0) rand_seed1_ += IM1;
  k = rand_seed2_/IQ2;
  rand_seed2_ = IA2*(rand_seed2_-k*IQ2)-k*IR2;
  if (rand_seed2_ < 0) rand_seed2_ += IM2;
  
  j = ((int) (iy_/NDIV))%NTAB;
  iy_ = iv_[j]-rand_seed2_;
  iv_[j] = rand_seed1_;
  if(iy_ < 1) iy_ += IMM1;
  if((temp=AM*iy_) > RNMAX)  temp = RNMAX;
  if (temp <= 0) temp = EPS;
  // temp is in the range ]0,1[

  return(-1+2*temp);
}

/******************************************/
/* Return double with increament 1e-14 dx */
/* in range  ]1e-14,1-1e-14[ */
/******************************************/
const double EPS_DBL = 1e-14;
const double RNMAX_DBL = 1-EPS_DBL;
double rand_gen::next_0_1_dbl()
{ double d2=next_m1_1();
  d2 = (next_0_1()-EPS)*(1-2*EPS_DBL)/(1-2*EPS)+EPS_DBL+d2*1.2e-7;
  if(d2 < EPS_DBL) {d2 += 1-2*EPS_DBL;}
  else if(d2 > RNMAX_DBL) { d2 -= 1-2*EPS_DBL;}

  return(d2);
}

/***************************************/
/* Return a random number in the range */
/* [imin, imax ]                       */
/***************************************/
int rand_gen_int::next()
{ return(imin_+(int)floor(rand_gen::next()*(imax_-imin_+1)));
}


