#!/bin/python3
import sys
import os
import numpy as np
from math import sqrt
import re

if (len(sys.argv) == 1) or \
   ((len(sys.argv) == 2) and ((sys.argv[1]=='-h') or (sys.argv[1]=='-help'))):
  print(sys.argv[0]," -h,-help           : display help message")
  print(sys.argv[0]," -P1=VAL -P2=VAL -t=TYPE -qmax=VAL -dqmax=VAL [-S=VAL] [-DATA_DIR=DIRS] [-CHECK]-l")
  print("   P1 : P1  or P1min,P1max")
  print("   P2 : P2  or P2min,P2max  or dP2min,dP2max")
  print("   TYPE: sp, Ato, Atco, Atid, DArd and DArt")
  print("   DATA_DIR: use data dir name to extract the parameters")
  sys.exit(0)

CTYPE = "sp"
P1 = 6
P2 = 6
QMAX= 100
DQMAX= 100
LIST = False
S0 = -1
DATA_DIR = "" # P and q parameters from directory name. Create 1 condor.txt
CHECK = False

argi = 1;
while(argi < len(sys.argv)):
  w = sys.argv[argi].split("=")
  if(w[0]=="-P1"):
    P1 = w[1]
  elif(w[0]=="-P2"):
    P2 = w[1]
  elif(w[0]=="-t"):
    CTYPE = w[1]
  elif(w[0]=="-qmax"):
    QMAX = int(w[1])
  elif(w[0]=="-dqmax"):
    DQMAX = int(w[1])
  elif(w[0]=="-S"):
    S0 = float(w[1])
  elif(w[0]=="-l"):
    LIST = True
  elif(w[0]=="-DATA_DIR"):
    DATA_DIR = w[1]
  elif(w[0]=="-CHECK"):
    CHECK = True
  else:
    print("Invalid argument : ",sys.argv[argi])
    sys.exit(5)
  argi += 1

#######################################################################
# Generate the text for the condor file  
#######################################################################
def mk_condor_text(Type, P1, P2, q, Q, ts, S1, S2):
  """ Generate the text for the condor file
  """
  fmt = "bi_sym_cages -type {} -P1 {} -P2 {} {} {} -ts {} -S1 {} -S2 {} "+\
        " -cc 500 -cpc 500 -T0 4e-3 -Tmin 1e-8 -track 500 -n_sweep 2000"+\
        " -nr 10 -ncla 100 -dx_min 1e-13 -use_grad_mc -cla_mc -pure_mc\n"
  
  qtxt = "-q1 {} -q2 {} -q3 {}".format(*q)
  if len(Q)==3:
     Qtxt = "-Q1 {} -Q2 {} -Q3 {}".format(*Q)
  if len(Q)==4:
     Qtxt = "-Q1 {} -Q2 {} -Q3 {} -Q4 {}".format(*Q)
  if len(Q)==5:
     Qtxt = "-Q1 {} -Q2 {} -Q3 {} -Q4 {} -Q5 {}".format(*Q)

  com = fmt.format(Type, P1, P2, qtxt, Qtxt, ts, S1, S2);
  return com

#######################################################################
# Compute the value of P from the q values
#######################################################################
def Pofq(ql):
  return sum(ql)+len(ql)

#######################################################################
# Merge list [l1, l2, l3] and [L1, L2, L3] as
#  [ [l1, L1], [l1, L2], [l1,L3], [l2, L1], [l2, L2], [l2,L3], [l2, L1],
#    [l2, L2], [l2,L3]  ]
#######################################################################
def merge_lists(QL1, QL2):
    lst = []
    for ql1 in QL1:
        for ql2 in QL2:
            lst .append([ql1, ql2])
    return(lst)

# Remove equivalent qQ for sp
def prune_sp_lists(lst):
  equi_list = []
  new_list = []
  for item in lst:
    if not item in equi_list:
      q,Q = item
      eq_item = [[Q[0],Q[2],Q[1]], [q[0],q[2],q[1]]] 
      equi_list.append(eq_item)
      new_list.append(item)
  return new_list
  
# Remove equivalent qQ ,-> Qq 
def prune_chiral_lists(lst):
  equi_list = []
  new_list = []
  for item in lst:
    #print(item,equi_list)
    if not item in equi_list:
    #if not is_in_list(item, equi_list):
      q,Q = item
      eq_item = [Q, q]
      equi_list.append(eq_item)
      new_list.append(item)
  return new_list

# Remove equivalent perumtations aA, bB, cC for sp
def prune_permutations_lists(lst):
  equi_list = []
  new_list = []
  for item in lst:
    #print(item,equi_list)
    if not item in equi_list:
    #if not is_in_list(item, equi_list):
      a,b,c = item[0]
      A,B,C = item[1]
      equi_list.append([[a,c,b],[A,C,B]])
      equi_list.append([[b,a,c],[B,A,C]])
      equi_list.append([[b,c,a],[B,C,A]])
      equi_list.append([[c,a,b],[C,A,B]])
      equi_list.append([[c,b,a],[C,B,A]])
      
      new_list.append(item)
  return new_list

# Remove pc-age wityh q,Q > QMAX
# and max(q)-min(q) > DQMAX
def prune_Qmax_list(lst):
  new_list = []
  for item in lst:      
      qL,QL = item
      ok = True
      for q in qL+QL:
        if q > QMAX:
          ok = False
      if max(qL)-min(qL) > DQMAX:  
          ok = False
      if max(QL)-min(QL) > DQMAX:  
          ok = False
      if ok:    
        new_list.append(item)
  return new_list



#######################################################################  
# Generate list of q1 q2 q3 for polyhedon P
# Exclude qi > P/2 .
# Returbn list [[q1, q2, q3], [q1,q2,q3] ...]
#######################################################################
def list_tri(P, equi_bc = False):
    Q_list = []
    q1max = min(P-3, P//2)
    for q1 in range(1, q1max+1):
        q2max = min(P-3-q1, P//2)
        for q2 in range(1, q2max+1):
           q3 = P-3-q1-q2
           if q3 > 0 and q3 <= P//2 and  ((not equi_bc) or (q3 <= q2)):  
               Q_list.append([q1, q2, q3])
    return(Q_list)

#######################################################################  
# Generate list of q1 q2 q3 q4 for polyhedon P
# Exclude qi > P/2 .
# Equivalence  q1 <-> q4   and q2 <-> q3
# Return list [[q1, q2, q3, q4], [q1, q2, q3, q4] ...]
#######################################################################
def list_quadri(P):
    Q_list = []
    q1max = min(P-4, P//2)
    for q1 in range(1, q1max+1):
        q2max = min(P-4-q1, P//2)
        for q2 in range(1, q2max+1):
            q3max = min(P-4-q1-q2, P//2)
            for q3 in range(1, q3max+1):
                q4 = P-4-q1-q2-q3
                if q4 > 0 and q4 <= P//2 and q4 <= q1 :
                  Q_list.append([q1, q2, q3, q4])
    return(Q_list)

#######################################################################  
# Generate list of [q1, q2, q1, q2] for polyhedon P
# Exclude qi > P/2 .
# Return list [[q1, q2, q1, q2], [q1, q2, q1, q2] ...]
#        [] is P is off
#######################################################################
def list_quadri_bi(P):
    if P%2 > 0 :
        return []
    Q_list = []
    q1max = min(P-4, P//2)
    for q1 in range(1, q1max+1):
        q2 = (P-4-2*q1)//2
        if q2 > 0 and q2 <= P//2 and q1 >= q2:  
               Q_list.append([q1, q2, q1, q2])
    return(Q_list)


#######################################################################  
# Generate list of q1 q2 q3 q4 q5 for polyhedon P
# Exclude qi > P/2 .
# Equivalence  q2 <->q5
# Return list [[q1, q2, q3, q4, q5], [q1, q2, q3, q4, q5] ...]
#######################################################################
def list_penta(P):
    Q_list = []
    q1max = min(P-5, P//2)
    for q1 in range(1, q1max+1):
        q2max = min(P-5-q1, P//2)
        for q2 in range(1, q2max+1):
           q3max = min(P-5-q1-q2, P//2)
           for q3 in range(1, q3max+1):
               q4max = min(P-5-q1-q2-q3, P//2)
               for q4 in range(1, q3max+1):
                   q5 = P-5-q1-q2-q3-q4
                   if q5 > 0 and q5 <= P//2 and q5 <= q1:
                      Q_list.append([q1, q2, q3, q4, q5])
    return(Q_list)



#########################################################
# Return [q]*valence  for face with multiple of valence edges
#        empty list if P%3 != 0
#########################################################
def list_unique(P, valence):
    if P%valence > 0 :
        return []
    q = (P-valence)//valence
    return [[q]*valence ]

#########################################################
# Make list for cube: sp_Pp_a-b-c_A-B-C
#########################################################
def list_sp(P1, P2):
    #Q1_list = list_tri(P1, equi_bc = True)
    Q1_list = list_tri(P1)
    Q2_list = list_tri(P2)
    QL = merge_lists(Q1_list, Q2_list)       
    return(QL)          

#########################################################
# Make list for truncated octahedron: Ato_Pp_a-b-c_A-B-C
#########################################################
def list_Ato(P1, P2):
    Q1_list = list_tri(P1)
    Q2_list = list_tri(P2)
    QL = merge_lists(Q1_list, Q2_list)       
    return(QL)          

#########################################################
# Make list for rhombicuboctahedron: Atco_Pp_a-b-c_A-B-C
#########################################################
def list_Atco(P1, P2):
    Q1_list = list_tri(P1)
    Q2_list = list_tri(P2)
    QL = merge_lists(Q1_list, Q2_list)       
    return(QL)          

#########################################################
# Make list for truncated icosidodecahedron: Atid_Pp_a-b-c_A-B-C
#########################################################
def list_Atid(P1, P2):
    Q1_list = list_tri(P1)
    Q2_list = list_tri(P2)
    QL = merge_lists(Q1_list, Q2_list)       
    return(QL)          

#########################################################
# Make list for truncated icosidodecahedron: DArd_Pp_a-a-a-A-A-A-A
#########################################################
def list_DArd(P1, P2):
    Q1_list = list_unique(P1, 3)
    if len(Q1_list) == 0:
          print("Error : P1 ({}) must be a multiple of 3".format(P1))
          exit(5)
    Q2_list = list_unique(P2, 4)
    QL = merge_lists(Q1_list, Q2_list)       
    return(QL)          

#########################################################
# Make list for rhombic tricontahedron: DArt_Pp_a-a-a-A-A-A-A-A
#########################################################
def list_DArt(P1, P2):
    Q1_list = list_unique(P1, 3)
    if len(Q1_list) == 0:
          print("Error : P1 ({}) must be a multiple of 3".format(P1))
          exit(5)
    Q2_list = list_unique(P2, 5)
    QL = merge_lists(Q1_list, Q2_list)       
    return(QL)          

#########################################################
# Make dir name from 
#########################################################
def dir_name(Type,P1, P2, q, Q):
  qtxt = "{}_{}_{}".format(*q)
  if len(Q)==3:
    Qtxt = "{}_{}_{}".format(*Q)
  elif len(Q)==4:
    Qtxt = "{}_{}_{}_{}".format(*Q)
  elif len(Q)==5:
    Qtxt = "{}_{}_{}_{}_{}".format(*Q)
    
  return("DATA_{}_P{}_P{}_{}-{}".format(Type,P1,P2,qtxt,Qtxt)) 
  
#########################################################
# Creates all the condor files
#########################################################
def init_files(Type, qQl, S, ts):
   for item in qQl:
     q,Q = item
     P1 = Pofq(q)
     P2 = Pofq(Q)
     d = dir_name(Type, P1, P2, q, Q)    
     if not os.path.isdir(d):
       os.mkdir(d)

     Coef = sqrt(max(P1,P2)/6)
     S1 = S2 = S*Coef
     ts *= Coef
     
     f = d+"/run.csh"
     fp = open(f,"w")
     fp.write("#!/bin/tcsh\n")
     run_txt =  mk_condor_text(Type, P1, P2, q, Q, ts, S1, S2)
     fp.write(run_txt)
       
#######################################################################  
def show_ql(QL):
    for q in QL:
        print(q)

#######################################################################  
# N,M  -> return (N, M)
# N    _> return (N, N)
def get_dbl_int(NcM):
  if "," in NcM:
    w = NcM.split(",")
    N = int(w[0])
    M = int(w[1])
  else:
      N = M = int(NcM)
  return N,M

#######################################################################  
# Check if DATA dirctory exist in . BAD or CROSSING
# mk_all_cages.py -P1=p -P2=pmin,pmax -t=CTYPE -qmax=VAL -dQmax=VAL
def check_dir(Ctype, QL):
  for item in QL:
    q,Q = item
    P1 = sum(q)+len(q)
    P2 = sum(Q)+len(q)

    dirname =  dir_name(Ctype,P1, P2, q, Q)
    if not (os.path.isdir(dirname) or os.path.isdir("BAD/"+dirname) or
            os.path.isdir("CROSSING/"+dirname)):
       print("Missing:", dirname)

# SINGLE CONDOR FILE FROM DATA_DIR
print(DATA_DIR)
if DATA_DIR != "":
  res= re.search("DATA_(.*)_P(\d+)_P(\d+)_(.*)-(.*)", DATA_DIR)
  TYPE = res.group(1)
  P1 = int(res.group(2))
  P2 = int(res.group(3))
  ql = res.group(4).split("_")
  QL = res.group(5).split("_")
  print(TYPE,P1,P2,ql,QL)
  ts = 1
  if TYPE == "sp" or TYPE == "DARD":
    ts = 0.5
  fp = open("condor.txt","w")
  Coef = sqrt(max(P1,P2)/6)
  if S0 < 0:
    S0 =2
  S1 = S2 = S0*Coef
  ts *= Coef
  condor_txt =  mk_condor_text(TYPE, P1, P2, ql, QL, ts, S1, S2)
  fp.write(condor_txt)
  exit(0)

  
# List of P1 values  P or Pmin,Pmax
P1min, P1max = get_dbl_int(P1) 
P1_list = range(P1min, P1max+1)

# P2=dPmin,Pmax range of Delta P2 : P2 = P1+dP2
deltaP2= False
if P2[0] == "d":
    deltaP2 = True
    P2min, P2max = get_dbl_int(P2[1:]) 
else :
    P2min, P2max = get_dbl_int(P2) 
P2_list = range(P2min, P2max+1)

for p1 in P1_list:
    for xp2 in P2_list:
      if deltaP2:
            p2 = p1 + xp2
      else:
            p2 = xp2

      if CTYPE == "sp":
        QL = list_sp(p1, p2)
        QL = prune_Qmax_list(QL)
        QL = prune_chiral_lists(QL)
        QL = prune_sp_lists(QL)
        QL = prune_permutations_lists(QL)
        if CHECK:
          check_dir(CTYPE, QL)
          
        elif not LIST:
          show_ql(QL)
          if S0 >0 :
            init_files(CTYPE, QL, S0, 0.5)
          else:
            init_files(CTYPE, QL, 2, 0.5)
        
      elif CTYPE == "Ato":
        QL = list_Ato(p1, p2)
        QL = prune_Qmax_list(QL)
        QL = prune_chiral_lists(QL)
        if CHECK:
          check_dir(CTYPE, QL)
          
        elif LIST:
          show_ql(QL)
        else: 
          if S0 >0 :
            init_files(CTYPE, QL, S0, 1)
          else:
            init_files(CTYPE, QL, 1, 1)
         
      elif CTYPE == "Atco":
        QL = list_Atco(p1, p2)
        QL = prune_Qmax_list(QL)
        QL = prune_chiral_lists(QL)
        if CHECK:
          check_dir(CTYPE, QL)
          
        elif LIST:
          show_ql(QL)
        else: 
          if S0 >0 :
            init_files(CTYPE, QL, S0, 1)
          else:
            init_files(CTYPE, QL, 1, 1)
         
      elif CTYPE == "Atid":
        QL = list_Atid(p1, p2)
        QL = prune_Qmax_list(QL)
        QL = prune_chiral_lists(QL)
        
        if CHECK:
          check_dir(CTYPE, QL)
          
        elif LIST:
          show_ql(QL)
        else:  
          if S0 >0 :
            init_files(CTYPE, QL, S0, 1)
          else:
            init_files(CTYPE, QL, 1.5, 1)
        
      elif CTYPE == "DArd":
        if p1%3 == 0 and p2%4 == 0:
          QL = list_DArd(p1, p2)
          QL = prune_Qmax_list(QL)
          #QL = prune_chiral_lists(QL)
          if CHECK:
            check_dir(CTYPE, QL)
          
          elif LIST:
             show_ql(QL)
          else:   
             if S0 >0 :
               init_files(CTYPE, QL, S0, 0.5)
             else:
               init_files(CTYPE, QL, 1, 0.5)
       
      elif CTYPE == "DArt":
        if p1%3 == 0 and p2%5 == 0:
          QL = list_DArt(p1, p2)
          QL = prune_Qmax_list(QL)
          #QL = prune_chiral_lists(QL)
          if CHECK:
            check_dir(CTYPE, QL)
          
          elif LIST:
            show_ql(QL)
          else:  
            if S0 >0 :
              init_files(CTYPE, QL, S0, 1)
            else:
              init_files(CTYPE, QL, 1, 1)
       
