#!/bin/python3
import sys
import numpy as np
from math import *
import matplotlib.pyplot as plt

if (len(sys.argv) == 1) or \
   ((len(sys.argv) == 2) and ((sys.argv[1]=='-h') or (sys.argv[1]=='-help'))):
  print(sys.argv[0]," -h,-help          : display help message")
  print(sys.argv[0]," filename [-o OUTNAME] [-f1] [-f2] [-pf] [-pf2] [-U]")
  sys.exit(0)

OUTNAME = ""
FNAME = sys.argv[1]
F1 = False
F2 = False
PF1 = False
PF2 = False
SHOW_U = False

PHI_G = (1+sqrt(5))/2;

argi = 2;
while(argi < len(sys.argv)):
  w = sys.argv[argi].split("=")
  if(w[0]=="-o"):
    OUTNAME  = w[1]
  elif(w[0]=="-f1"): # only show face 1
    F1 = True
  elif(w[0]=="-f2"): # only show face 1
    F2 = True
  elif(w[0]=="-pf"): # show planar face 1 with node numbers
    PF1 = True
  elif(w[0]=="-pf2"): # show planar face 2 with node numbers
    PF2 = True
  elif(w[0]=="-U"): # show V, v1, v2, W, w1, w2, U
    SHOW_U = True
  else:
    print("Invalid argument : ",sys.argv[argi])
    sys.exit(5)
  argi += 1
  
def read_file(fname):
    S,theta, phi, psi = [0,0,0,0]
    YZ = []
    fp = open(fname)
    line_no = 1
    face1 = []
    face2 = []
    N1 = 0
    for line in fp:
        if line[0] != '#':
           w = line.split()
           if line_no == 1:
             CTYPE = w[0]
           elif line_no == 2:
             P1 = int(w[0])
             P2 = int(w[1])
             qs = list(map(int, w[2:]))
           elif line_no == 3:
               S1, S2, *XYZ = list(map(float, w))
           elif line_no == 4:
             cl, ca, cc = w
           elif line_no == 5:
             err_l_max, err_a_max, err_l_av_max, err_l_av, err_a_av = w
           elif line_no == 6:
             err_l_max_1, err_a_max_1, err_l_av_max_1, err_l_av_1, err_a_av_1 = w
           elif line_no == 7:
             err_l_max_2, err_a_max_2, err_l_av_max_2, err_l_av_2, err_a_av_2 = w
           elif line_no == 8:
             E,Efc,Epc = w[0:3]
           elif line_no == 9:
             V = np.array(list(map(float,w)))
           elif line_no == 10:
             v1 = np.array(list(map(float,w[:3])))
             v2 = np.array(list(map(float,w[3:])))             
           elif line_no == 11:
             W1 = np.array(list(map(float,w)))
           elif line_no == 12:
             w11 = np.array(list(map(float,w[:3])))
             w12 = np.array(list(map(float,w[3:])))             
           elif line_no <= 12+P1:   # face type 1
             face1.append([float(w[0]), float(w[1]), float(w[2])])
           else: # face type 2
             face2.append([float(w[0]), float(w[1]), float(w[2])])
           line_no +=1

    pc = { 'TYPE': CTYPE,'qs': qs,
           'P1': P1,'P2': P2,
           'V': V, 'v1': v1, 'v2': v2,
           'W1': W1, 'w11': w11, 'w12': w12,
           'S1': S1, 'S2':S2, 'XYZ': XYZ,
           'YZ': YZ, 
           'cl': cl, 'ca': ca , 'cc': cc,
           'face1': face1,
           'face2': face2
          }        
    return(pc)
  
def show_pc(pc):
    for k in pc.keys():
      print("{} {}".format(k, pc[k]))
      
###################### TOOLS #############################
# Make rotation matrix of angle beta around v  
def make_rotate_vector(v, beta):
  W = np.zeros([3,3], dtype='float64')
  V = np.array(v)
  V = V/np.linalg.norm(V) # normalise the vector
  W[0,1] = -V[2]
  W[0,2] = V[1]
  W[1,2] = -V[0]
  W[1,0] = -W[0,1]
  W[2,0] = -W[0,2]
  W[2,1] = -W[1,2]
  return(np.identity(3)+W*np.sin(beta)+W@W*(1-np.cos(beta)))

# rotate face by theta around v
def rotate_face(f, v, theta):
    Rv = make_rotate_vector(v, theta)
    nf = []
    for n in f:
        nf.append(Rv@n)
    return(nf)

# rotate face by theta around v
def rotate_face_R(f, Rv):
    nf = []
    for n in f:
        nf.append(Rv@n)
    return(nf)
  
def projector(V):
  P = np.zeros([3,3])
  nv = np.linalg.norm(V)
  for i in range(3):
    for j in range(3):
      P[i,j]= V[i]*V[j]/(nv**2)
  return(P)

def mk_off(fname, p_cage):
  fp = open(fname, "w")
  nf = nn = ne = 0
  for f in p_cage:
    nf += 1
    nn += len(f)
    ne += len(f)

  fp.write("OFF\n#\n")
  fp.write("{} {} {}\n".format(nn, nf, ne))

  face_lines = ""
  node_index = 0
  for f in p_cage:
    P = len(f)
    line = "{}".format(P)
    for n in f:
      fp.write("{} {} {}\n".format(n[0], n[1], n[2]))
      line +=  " {}".format(node_index)
      node_index += 1
    face_lines += line+"\n"
  fp.write(face_lines)

def mk_face(pc):
    f = []
    for n in pc["face"]:
        nn = pc['V'] + pc['v1']*n[0]+ pc['v2']*n[1]
        f.append(nn)
        #print(nn)
    pc['face3d'] = f
    #print("V=", pc['V'])
    #print("v1=", pc['v1'])
    #print("v2=", pc['v2'])
    
def mk_face_V0(pc):
    f = []
    v1 = pc['v1']
    v2 = pc['v2']
    V0 = pc['V']
    p = np.cross(v1,v2)
    p = p/np.linalg.norm(p)
    V = (p@V0)*p
    for n in pc["face"]:
        nn = V + v1*n[0]+ v2*n[1]
        f.append(nn)
        #print(nn)
    pc['face3d'] = f


# Make thin face axis 
##########################
def mk_axis(vec, dv=[0.1,0.1,0]):
  f = []
  v = np.array(vec)
  dv = np.array(dv)
  #print("axis vec=",vec)

  f.append([0,0,0])
  f.append([vec[0],vec[1],vec[2]])
  f.append([vec[0]+dv[0],vec[1]+dv[1],vec[2]+dv[2]])
  f.append([dv[0],dv[1],dv[2]])

  return(f)

##############################################################
def show_planar_face1(pc):
   X = []
   Y = []
   v1 = pc['v1']
   v2 = pc['v2']
   print("f1=",pc['face1'])
   for f in  pc['face1']:
     print(f)
     X.append(v1@f)
     Y.append(v2@f)

   X.append(X[0])
   Y.append(Y[0])
   plt.axes().set_aspect('equal')
   plt.plot(X,Y)

   for i in  range(len(X)-1):
     plt.text(X[i], Y[i], str(i+1))
   plt.show()
   
def show_planar_face2(pc):
   X = []
   Y = []
   w11 = pc['w11']
   w12 = pc['w12']
   print("w11=",w11)
   print("w12=",w12)
   print("f2=",pc['face2'])
   for f in  pc['face2']:
     print(f)
     X.append(w11@f)
     Y.append(w12@f)

   X.append(X[0])
   Y.append(Y[0])
   plt.axes().set_aspect('equal')
   plt.plot(X,Y)

   for i in  range(len(X)-1):
     plt.text(X[i], Y[i], str(i+1))
   plt.show()
   
##############################################################
#
##############################################################
# PRISM
def mk_sp(pc):
    
    #g1 = np.array([0., 1., -1.])
    #g2 = np.array([2., 1., 1.])
    #R1 = make_rotate_vector(g1, pc['phi'])
    #R2 = make_rotate_vector(g2, pc['psi'])
    #R = R2@R1
 
    Rzpiot = make_rotate_vector([0, 0, 1.], pi/2)
    Rzpi = make_rotate_vector([0, 0, 1.], pi)
    Rypi = make_rotate_vector([0, 1., 0], pi)

    V = pc['V']
    f11 = pc['face1']
    f21 = pc['face2']

    p_cage = []
    if not F2:
      p_cage.append(f11)
      if not F1:
        p_cage.append(rotate_face(f11, [0.,0.,1.], pi))
        p_cage.append(rotate_face(f11, [1.,0.,0.], pi))
        p_cage.append(rotate_face(f11, [0.,1.,0.], pi))
    
    if not F1:
      p_cage.append(f21)
      if not F2:
        p_cage.append(rotate_face(f21, [0.,0.,1.], pi))
        p_cage.append(rotate_face(f21, [0.,1.,0.], pi))
        p_cage.append(rotate_face(f21, [1.,0.,0.], pi))

    return(p_cage)
  
def mk_Ato(pc):
    Rxpiot = make_rotate_vector([1.0, 0., 0.], pi/2)
    Rypiot = make_rotate_vector([0., 1., 0.], pi/2)
    Rzpiot = make_rotate_vector([0., 0., 1.], pi/2)
    Rympiot = make_rotate_vector([0., 1., 0.], -pi/2)
    Rxpi = make_rotate_vector([1., 0, 0.], pi)
    Rypi = make_rotate_vector([0, 1., 0], pi)
    Rzpi = make_rotate_vector([0, 0, 1.], pi)

    V = pc['V']
    f1x1 = pc['face1']
    f2x1 = pc['face2']

    p_cage = []
    if not F2:
      p_cage.append(f1x1)
      if True or not F1:
        f1x2 = rotate_face_R(f1x1, Rxpi)
        p_cage.append(f1x2) # f1x2
        p_cage.append(rotate_face_R(f1x1, Rypi)) # f1x3
        p_cage.append(rotate_face_R(f1x2, Rypi)) # f1x4
        f1y1 = rotate_face_R(f1x1, Rypiot@Rzpiot)
        f1y2 = rotate_face_R(f1x2, Rypiot@Rzpiot)
        p_cage.append(f1y1) # f1y1
        p_cage.append(f1y2) # f1y2
        p_cage.append(rotate_face_R(f1y1, Rzpi)) # f1y3
        p_cage.append(rotate_face_R(f1y2, Rzpi)) # f1y4
        f1z1 = rotate_face_R(f1x1, Rzpiot@Rypiot)
        f1z2 = rotate_face_R(f1x2, Rzpiot@Rypiot)
        p_cage.append(f1z1) # f1z1
        p_cage.append(f1z2) # f1z2
        p_cage.append(rotate_face_R(f1z1, Rxpi)) # f1z3
        p_cage.append(rotate_face_R(f1z2, Rxpi)) # f1z4
    
    if not F1:
      p_cage.append(f2x1)
      if not F2:
        f2x2 = rotate_face_R(f2x1, Rxpi)
        p_cage.append(f2x2) # f2x2
        p_cage.append(rotate_face_R(f2x1, Rypi)) # f2x3
        p_cage.append(rotate_face_R(f2x2, Rypi)) # f2x4
        f2y1 = rotate_face_R(f2x1, Rypiot@Rzpiot)
        f2y2 = rotate_face_R(f2x2, Rypiot@Rzpiot)
        p_cage.append(f2y1) # f2y1
        p_cage.append(f2y2) # f2y2
        p_cage.append(rotate_face_R(f2y1, Rzpi)) # f2y3
        p_cage.append(rotate_face_R(f2y2, Rzpi)) # f2y4
        f2z1 = rotate_face_R(f2x1, Rzpiot@Rypiot)
        f2z2 = rotate_face_R(f2x2, Rzpiot@Rypiot)
        p_cage.append(f2z1) # f2z1
        p_cage.append(f2z2) # f2z2
        p_cage.append(rotate_face_R(f2z1, Rxpi)) # f2z3
        p_cage.append(rotate_face_R(f2z2, Rxpi)) # f2z4

    return(p_cage)

def mk_Atco(pc):
    Rxpiot = make_rotate_vector([1.0, 0., 0.], pi/2)
    Rypiot = make_rotate_vector([0., 1., 0.], pi/2)
    Rzpiot = make_rotate_vector([0., 0., 1.], pi/2)
    Rxmpiot = make_rotate_vector([1., 0., 0.], -pi/2)
    Rympiot = make_rotate_vector([0., 1., 0.], -pi/2)
    Rzmpiot = make_rotate_vector([0., 0., 1.], -pi/2)
    Rxpi = make_rotate_vector([1., 0, 0.], pi)
    Rypi = make_rotate_vector([0, 1., 0], pi)
    Rzpi = make_rotate_vector([0, 0, 1.], pi)
    Rxpio4 = make_rotate_vector([1., 0, 0.], pi/4)
    Rxmpio4 = make_rotate_vector([1., 0, 0.], -pi/4)
    Rx3pio4 = make_rotate_vector([1., 0, 0.],  3*pi/4)

  

    V = pc['V']
    f1x1 = pc['face1']
    f2x2 = pc['face2']

    p_cage = []
    if not F2:
      p_cage.append(f1x1)
      if True or not F1:
        f1x3 = rotate_face_R(f1x1,Rxpiot)
        f1x5 = rotate_face_R(f1x1,Rxpi)
        f1x7 = rotate_face_R(f1x1,Rxmpiot)
        p_cage.append(f1x3) # f1x2
        p_cage.append(f1x5) # f1x5
        p_cage.append(f1x7) # f1x7

        for R in [Rympiot, Rzpiot, Rzpi, Rypiot, Rzmpiot]:
           p_cage.append(rotate_face_R(f1x1,R))
           p_cage.append(rotate_face_R(f1x3,R))
           p_cage.append(rotate_face_R(f1x5,R))
           p_cage.append(rotate_face_R(f1x7,R))

        
    if not F1:
      p_cage.append(f2x2)
      if True or not F2:  
         f2x4 = rotate_face_R(f2x2,Rxpiot)
         f2x6 = rotate_face_R(f2x2,Rxpi)
         f2x8 = rotate_face_R(f2x2,Rxmpiot)

         p_cage.append(f2x2) # f2x2
         p_cage.append(f2x4) # f2x4
         p_cage.append(f2x6) # f2x6
         p_cage.append(f2x8) # f2x8
         for R in [Rympiot, Rzpiot, Rzpi, Rypiot, Rzmpiot]:
             p_cage.append(rotate_face_R(f2x2,R))
             p_cage.append(rotate_face_R(f2x4,R))
             p_cage.append(rotate_face_R(f2x6,R))
             p_cage.append(rotate_face_R(f2x8,R))

    return(p_cage)

def mk_Atid(pc):
    g = [sqrt((3+sqrt(5))/(5+sqrt(5))), sqrt(2/(5+sqrt(5))), 0]
    Rg = make_rotate_vector(g, pi)            
    Rzpi = make_rotate_vector([0, 0, 1.], pi)
    Rx1po5 = make_rotate_vector([1., 0, 0.], pi/5)
    Rx2po5 = make_rotate_vector([1., 0, 0.], 2*pi/5)
    Rxm2po5 = make_rotate_vector([1., 0, 0.], -2*pi/5)

    V = pc['V']
    f11 = pc['face1']
    f12 = pc['face2']
    
    # for off file
    Fdeca1 = [f11]
    for i in range(1,5):
          R = make_rotate_vector([1., 0, 0.], i*2*pi/5)
          Fdeca1.append(rotate_face_R(f11,R))      

    Fdeca1.append(f12)
    for i in range(1,5):
          R = make_rotate_vector([1., 0, 0.], i*2*pi/5)
          Fdeca1.append(rotate_face_R(f12, R))      


    # Decagon 1
    Fdeca2 = []
    for f in Fdeca1:
       Fdeca2.append(rotate_face_R(f, Rg))

    Fdecatop = Fdeca1 + Fdeca2
    for f in Fdeca2:
       for i in range(1,5):
          R = make_rotate_vector([1., 0, 0.], i*2*pi/5)
          Fdecatop.append(rotate_face_R(f, R))      
       
    Fdecabottom = []
    for f in Fdecatop:
          Fdecabottom.append(rotate_face_R(f, Rzpi))
          
    p_cage = Fdecatop+ Fdecabottom
    return(p_cage)
    
def mk_DArd(pc):
    Rxpiot = make_rotate_vector([1.0, 0., 0.], pi/2)
    Rypiot = make_rotate_vector([0., 1., 0.], pi/2)
    Rzpiot = make_rotate_vector([0., 0., 1.], pi/2)
    Rxmpiot = make_rotate_vector([1., 0., 0.], -pi/2)
    Rympiot = make_rotate_vector([0., 1., 0.], -pi/2)
    Rzmpiot = make_rotate_vector([0., 0., 1.], -pi/2)
    Rxpi = make_rotate_vector([1., 0, 0.], pi)
    Rypi = make_rotate_vector([0, 1., 0], pi)
    Rzpi = make_rotate_vector([0, 0, 1.], pi)

    V = pc['V']
    f11 = pc['face1']
    f21 = pc['face2']

    p_cage = []
    if not F1:
      p_cage.append(f21)
      if True or not F2:
         p_cage.append(rotate_face_R(f21, Rzpiot))
         p_cage.append(rotate_face_R(f21, Rzpi))
         p_cage.append(rotate_face_R(f21, Rzmpiot))
         p_cage.append(rotate_face_R(f21, Rympiot))
         p_cage.append(rotate_face_R(f21, Rypiot))
    
    if not F2:
      p_cage.append(f11)
      if not F1:
         p_cage.append(rotate_face_R(f11, Rzpiot))
         p_cage.append(rotate_face_R(f11, Rzpi))
         p_cage.append(rotate_face_R(f11, Rzmpiot))
         p_cage.append(rotate_face_R(f11, Rypiot))
         p_cage.append(rotate_face_R(f11, Rypi))
         p_cage.append(rotate_face_R(f11, Rzpiot@Rypi))
         p_cage.append(rotate_face_R(f11, Rxpi))

    return(p_cage)
  
def mk_DArt(pc):
    gp = np.array([1+PHI_G**2, 0, 2*PHI_G+PHI_G**2])/5
    gt = np.array([0, PHI_G, PHI_G+PHI_G**2])/3
    g1 = np.array([1, PHI_G, PHI_G**2])/2
    g6 = np.array([-1, PHI_G, PHI_G**2])/2
    gz = (np.identity(3) - projector(gp))@np.array([0, 0, 1.])
    
    Rgp2piof = make_rotate_vector(gp, 2*pi/5)
    #print("Rgp2piof=",Rgp2piof)
    Rgt2piot = make_rotate_vector(gt, 2*pi/3)
    Rgtm2piot = make_rotate_vector(gt, -2*pi/3)
    Rg1piot = make_rotate_vector(g1, pi)
    Rg6piot = make_rotate_vector(g6, pi)
    Rgzpi = make_rotate_vector(gz, pi)
    Rgp2pio10 = make_rotate_vector(gp, pi/5)
    R = Rgp2pio10@Rgzpi 

    V = pc['V']
    f11 = pc['face1']
    f21 = pc['face2']

    p_cage = []
    if not F1:
      p_cage.append(f21)
      if not F2:
         f22 = rotate_face_R(f21, Rg1piot)
         p_cage.append(f22)
         f = f22
         for i in range(1,5):
           #print("f=",f, "i=",i, "R=",Rgp2piof)
           f = rotate_face_R(f.copy(), Rgp2piof)
           p_cage.append(f)
      
    if not F2:
      p_cage.append(f11)
      if not F1:
         # First circle of triangles
         f = f11
         for i in range(1,5):
           f = rotate_face_R(f, Rgp2piof)
           p_cage.append(f)

         # Second circle of triangles
         f16 = rotate_face_R(f11, Rg6piot)
         p_cage.append(f16)
         f = f16
         for i in range(1,5):
            f = rotate_face_R(f.copy(), Rgp2piof)
            p_cage.append(f)

    # Bottom half      
    p_cagee_top = p_cage.copy()
    for f in p_cagee_top:
      p_cage.append(rotate_face_R(f.copy(), R))

    return(p_cage)
  
def show_vec(pc):
   V = pc['V']
   v1 = pc['v1']
   v2 = pc['v2']
   W = pc['W1']
   w1 = pc['w11']
   w2 = pc['w12']
   S1 = pc['S1']
   S2 = pc['S2']

   #W= np.array([0.7236068,  0.,         1.17082039])
   #V= np.array([0.,         0.53934466, 1.41202266])
   p = np.cross(v1,v2)
   q = np.cross(w1,w2)
   u = np.cross(q,p)

   uV = u@V
   uv1 = u@v1
   uv2 = u@v2
   qv1 = q@v1
   qv2 = q@v2
   qWmV = q@(W-V)

   #if abs(uv2) < abs(uv1):
   #   a2 = (qWmV*uv1+uV*qv1)/(qv2*uv1-uv2*qv1)
   #   U = V+a2*(v2-v1*(uv2/uv1))-v1*(uV/uv1)
   #else:  
   a1 = (qWmV*uv2+uV*qv2)/(qv1*uv2-uv1*qv2)
   U = V+a1*(v1-v2*(uv1/uv2))-v2*(uV/uv2)
   print("V=",V)
   print("v1=",v1,"v2=",v2)
   print("W=",W)
   print("w1=",w1,"w2=",w2)
   print("p=",p,"q=",q)
   print("u*V=",u@V)
   print("u*v1=",u@v1)
   print("u*v2=",u@v2)
   print("q*V=",q@V)
   print("q*v1=",q@v1)
   print("q*v2=",q@v2)
   print("q*W=",q@W)
   
   print("u=",u)
   print("U=",U)
   print("S1=",S1)
   print("S2=",S2)
   print("|U-V|=", sqrt((U-V)@(U-V)))
   print("|U-W|=", sqrt((U-W)@(U-W)))
   
##############################################################

pc = read_file(FNAME)
#show_pc(pc)
pos = FNAME.rfind(".")
outfile = FNAME[:pos]+".off"

if PF1:
  show_planar_face1(pc)
  exit()
if PF2:
  show_planar_face2(pc)
  exit()

if SHOW_U:
  show_vec(pc);
  exit()
  
print(pc['TYPE'])
if pc['TYPE'] == "sp":
   p_cage = mk_sp(pc) 
elif pc['TYPE'] == "Ato":
   p_cage = mk_Ato(pc) 
elif pc['TYPE'] == "Atco":
   p_cage = mk_Atco(pc) 
elif pc['TYPE'] == "Atid":
   p_cage = mk_Atid(pc) 
elif pc['TYPE'] == "DArd":
   p_cage = mk_DArd(pc) 
elif pc['TYPE'] == "DArt":
   p_cage = mk_DArt(pc) 

mk_off(outfile, p_cage)
