#include <getargs.h>
#include <ctype.h>
#include <iostream>
#include <fstream>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <t_build_array.h>
#include <time.h>

/* makeparlist */
#define MPL_SET  1
#define MPLS_SET 2

using namespace std;

extern char* VERSION;

int scanArgList(int const argc,char *const *argv,Parameter *pars,
            const char **help, char ** &Argv,PArgsModes const mode,
            int & makeparlist);
void Check_Par_Read(Parameter *pars);
void Set_Default_Value(Parameter *pars);
Parameter *Get_Par_Details(Parameter *pars,char *name);
int Syntax(const char **p,Parameter *pars,PArgsModes const mode);
void display_help(const char *, Parameter *pars);
bool readBoolValue(char const *s);
void makeParList(Parameter *pars,string FileName);

void Read_Parameter_Keyboard(Parameter *pars);
void Read_Parameter_File(Parameter *pars,ifstream *ifs);
PArgsRead Read_One_line(ifstream *ifs,char *name,char *val,int &line_no);
ifstream *Open_File(char *Name);

/***************************************************************************/
/* Read the argument list and extra the parameters.                        */
/* Parameters are preceded with the - symbol. They can be of the following */
/* Types : Integer, Real, Char, String, Boolean, and Flag                  */
/* All the parameters are described in the Par array passed to GetArgs.    */
/* A new Argv list is created for all the unused parameters.               */
/* The corresponding Argc is returned.                                     */
/*                                                                         */
/* Example of Parameter array:                                             */
/* name type  p init_val flag alias help                                   */
/*  -name the argument name (without the -)                                */
/*  -type the type of variable T_INT, T_REAL, T_CHAR, T_STRING, T_BOOL,    */
/*                             T_FLAG, T_COMMENT                           */
/*  -p : pointer to the C variable to initialise                           */
/*  -init_val : Default value entered as a string                          */
/*           The string is converted to the correct type.                  */
/*           FLAGS : always OFF by default, => can be set to 0             */
/*  -flag : PAR_FLAG_READ   : paramater has been initialised.              */
/*          PAR_FLAG_NOHELP : no help entry for this par                   */
/*          PAR_FLAG_NOPARLIST : exclude from par list output (-mpl)       */
/*          PAR_FLAG_QUOTEPSTR : quote strings in par list (-mpl)          */
/*  -alias : an alias names. ("" or 0 if none).                            */
/*  -help : help description for that parameters.                          */
/* Parameter Pars[] = {                                                    */
/* { "short",T_FLAG,&flag,0,0,"s","Short value"},                          */
/* { "file",T_STRING,File,"Def_File",0,"F","File name"},                   */
/* { "No",T_INT,&n,"10",0,"n","No of lines"},                              */
/* { "  ANY TEXT",T_COMMENT,0,0,0,0,"================="},                  */
/* { 0,0,0,0,0,0},                                                         */
/* };                                                                      */
/* char *Help[] = {                                                        */
/* " comm [-short=s] [-F=file FILE_NAME] [-No=n N] src",                   */
/* " comm -help",                                                          */
/* };                                                                      */
/*                                                                         */
/* GetArgs(argc,argv,Pars,help,Argv,mode)                                  */
/* argc, argv : the argument list                                          */
/* Pars the array of Parameter describing all the parameters               */
/* help : char array : a help message displayed when the -help flag is on  */
/* Argv : pointer to array of cstring pointers for the extra arguments     */
/* mode : PA_BADARGOK : extra args are added to arg list                   */
/*        PA_BADPAROK : bad parameters [-?] are added to the argument list */
/*        PA_NEGNOOK  : -d... args are allowed if PA_BADARGOK set          */
/*        PA_MINUSOK  : - is added to the argument list (for example, when */
/*                      used to indicate stdin or stdout.)                 */
/*        PA_MINUSLAST : - force all the remaining argument to be added to */
/*                      argument list (Argv must be large!)                */
/*        PA_H_ALIASHELP : -h -> display syntax                            */
/*        PA_HELP_ALIASHELP : -help -> display syntax                      */
/*        PA_V_ALIASVERSION : -v -> display version (extren char *VERSION) */
/*        PA_VERSION_ALIASVERSION : -version -> display version            */
/*        PA_MPL_ALIASMPL : -mpl -> Make paramater list in Argv[0]+".par"  */
/*        PA_LS_ALIASMPL :   -l -> make parameter list in Argv[0]+".par"   */
/*        PA_MINUSMINUS_H : --h args : display help line for each args     */
/*                                                                         */
/* Return : Number of extra argument read (Argc).                          */
/* Throws : a string (Error messaage) when an error occurs.                */
/***************************************************************************/
int GetArgs(int const argc,char * const *argv,Parameter *pars,
            const char **help, char ** &Argv,PArgsModes const mode)
{ int Argc;
  int makeparlist=0;
  string s_err;

  if((mode & PA_MINUSMINUS_H)&&(!strcmp("--h",argv[1])))
  { int i=2;
    while(i < argc)
    { display_help(argv[i++],pars);
    }

    exit(1);
  }

  if(argc == 2)
  { if(((mode & PA_H_ALIASHELP) && !strcmp("-h",argv[1])) ||  
       ((mode & PA_HELP_ALIASHELP) && !strcmp("-help",argv[1]))) 
    { Syntax(help,pars,mode); /* Display help message and exit */
      exit(1);
    }

    if(((mode & PA_V_ALIAVERSION ) && !strcmp("-v",argv[1])) || 
       ((mode & PA_VERSION_ALIAVERSION ) && !strcmp("-version",argv[1]))) 
    { cerr << VERSION << "\n"; /* Display Version Number */
      exit(1);
    }
  }

  /* Set all the default values : throws on error */
  Set_Default_Value(pars);

  /* Scan the argument list */
  Argc = scanArgList(argc,argv,pars,help,Argv,mode,makeparlist);

  /* Check that every parameter has been set  */
  Check_Par_Read(pars); /* Throw if there is a problem */

  if(makeparlist&MPL_SET)
  { makeParList(pars,((string)argv[0])+".par");
    if(makeparlist&MPLS_SET) exit(0);
  }

  return(Argc);
} 

/*******************************************************************/
/* Scan the argument list                                          */
/* Fill the parameter lists and return the extra arguments in Argv */
/* Return : Argc : no of extra arguments                           */
/*******************************************************************/
int scanArgList(int const argc,char *const *argv,Parameter *pars,
            const char **help, char ** &Argv,PArgsModes const mode,
            int & makeparlist)
{ int getpars,i,Argc;
  Parameter *par;
  build_array<char *> barr;
  char *s;

  i = 1;
  getpars = 1; /* Set to 0 when pars must be added to the argument list */
  while(i < argc) /* Scan the arguments list */
  { s = const_cast<char *>(argv[i]);
   
    if (getpars && (s[0] == '-')) 
    { if(s[1] == 0) /* - without name: */
      { if(mode & PA_MINUSOK)
        { barr.add(s);
        }
        else if (mode & PA_MINUSLAST)
        { getpars = 0;
        }
        else
        { throw((string)" ERROR : Invalid parameter: "+s);
        }
      }

      else /* Par */
      { if(!(par = Get_Par_Details(pars,s+1)))
        { if(((mode & PA_MPL_ALIASMPL) && !strcmp("-mpl",s)) ||  
             ((mode & PA_L_ALIASMPL) && !strcmp("-l",s))) 
          { makeparlist |= MPL_SET;
          }
          else if(((mode & PA_MPLS_ALIASMPL) && !strcmp("-MPL",s)) ||  
             ((mode & PA_LS_ALIASMPL) && !strcmp("-L",s))) 
          { makeparlist |= MPL_SET | MPLS_SET;
          }
          else if(mode & PA_BADPAROK) 
          { /* Add par to the argument list */
            barr.add(s);
          }
          else if((mode&PA_NEGNOOK) && (mode&PA_BADARGOK) && isdigit(s[1])) 
          { /* :Negative no : Add par to the argument list */
            barr.add(s);
          }
          else
          { throw((string)" ERROR : Invalid parameter: "+s);
          }
        }
        else 
        { switch (par->type)
          { case T_INT :
                  i++; 
                  if(i >= argc) goto NoValue;
                  sscanf(argv[i],"%d",(int *)par->p);
                  par->flag |= PAR_FLAG_READ;
                  break;

            case T_REAL :
                  i++; 
                  if(i >= argc) goto NoValue; 
                  sscanf(argv[i],"%lg",(double *)par->p);
                  par->flag |= PAR_FLAG_READ;
                  break;

            case T_CHAR :
                  i++; 
                  if(i >= argc) goto NoValue; 
                  sscanf(argv[i],"%c",(char *)par->p);
                  par->flag |= PAR_FLAG_READ;
                  break;

            case T_STRING :
                  i++; 
                  if(i >= argc) goto NoValue;
                  strcpy((char *)par->p,argv[i]);
                  par->flag |= PAR_FLAG_READ;
                  break;

            case T_BOOL :
                  i++; 
                  if(i >= argc) goto NoValue;
                  try 
                  { *((bool *)par->p) = readBoolValue(argv[i]);
                  }
                  catch (int err)
                  { throw((string)"ERROR: Invalid boolean default value <"
                            + argv[i] + "> for parameter " + par->name);
                  }
                  par->flag |= PAR_FLAG_READ;
                  break;

            case T_FLAG :
                  *((int *)par->p) = 1;
                  par->flag |= PAR_FLAG_READ;
                  break;

            default : 
                  barr.add(s);
                  break;
          }
        }
      }
    }
    else
    { if(mode & PA_BADARGOK) 
      { /* Add par to the argument list */
        barr.add(s);
      }
      else 
      { throw((string)(" ERROR : Invalid parameter: ")+s);
      }
    }
    i++;
  }

  Argc = 0;
  Argv = barr.makeArray(Argc);
  return(Argc);

NoValue:
  throw((string)" ERROR : No Value for parameter: "+s);

}

/****************************************************/
/* Initialise each parameter with the default value */
/* (when there is one).                             */
/* The value of read is set to 1 for each parameter */
/* with a default value                             */
/* Return : 1 : OK                                  */
/* Throws a string (Error message) on error         */
/****************************************************/
void Set_Default_Value(Parameter *pars)
{ string s_err;

  /* The last element of pars is a dummy line with type NULL */ 
  while(pars->type)
  { /* By default Flags are always off */
    if(pars->type == T_FLAG) 
    { pars->flag |= PAR_FLAG_READ;
      *((int *)pars->p) = 0;
    }
    else
    { switch(pars->type)
      { case T_NONE: break;
        case T_INT:
              if(pars->init_val)
              { sscanf(pars->init_val,"%d",(int *)pars->p);
                pars->flag |= PAR_FLAG_READ;
              }
                    break;
        case T_REAL:
              if(pars->init_val)
              { sscanf(pars->init_val,"%lg",(double *)pars->p);
                pars->flag |= PAR_FLAG_READ;
              }
                    break;
        case T_CHAR:
              if(pars->init_val)
              { sscanf(pars->init_val,"%c",(char *)pars->p);
                pars->flag |= PAR_FLAG_READ;
              }
                    break;
        case T_STRING:
              if(pars->init_val)
              { strcpy((char *)pars->p,pars->init_val);
                pars->flag |= PAR_FLAG_READ;
              }
              else ((char *)pars->p)[0] = 0; /* Allways set a string to 0 */

                    break;
        case T_BOOL:
              if(pars->init_val)
              { try 
                { *((bool *)pars->p) = readBoolValue(pars->init_val);
                }
                catch (int err)
                { throw((string)"ERROR: Invalid boolean default value <" +
                          pars->init_val + "> for parameter " + pars->name);
                }
                pars->flag |= PAR_FLAG_READ;
              }
                    break;
        case T_COMMENT: break;
        default :
          throw((string)"Invalid parameter type for parameter "+ pars->name);
      }
    }
    pars++;
  }
}

/****************************************************************/ 
/* Search the parameter entry associated to name in Parameter g */
/* p : Pointer to an array of pointer of Parameters             */
/* Return : pointer to the Parameter                            */
/*          0 : not found                                       */
/****************************************************************/
Parameter *Get_Par_Details(Parameter *pars,char *name)
{
       /* Scan all the parameters in the table */
  while(pars->type) /* Last element of p is a dummy line with type NULL */
  {    /* when the parameter is found: return a pointer to its def. line */
    if(!strcmp(pars->name,name)) return(pars);
    if(pars->alias && !strcmp(pars->alias,name)) return(pars);
    pars++;
  }
  return(0); /* No match found */
}

/****************************************************/
/* Check that all the parameters have been read.    */
/* p : Pointer to an array of pointer of Parameters */
/* Prints the name of all unread parameters.        */
/* Return : 1 : OK                                  */
/*          0 : some parameters are not set.        */
/* Throw and error message if some parameters have  */
/*       no value                                   */
/****************************************************/
void Check_Par_Read(Parameter *pars)
{ int r;
  string s;
 
  r = 1;
 
      /* The last element of g is a dummy line with type NULL */ 
  while(pars->type)
  { if(!(pars->flag & PAR_FLAG_READ) &&  /* look for unread pars */
        (pars->type != T_COMMENT)) 
    { /* Merge unset parameters */ 
      s += pars->name; s+=" "; 
      r = 0;
    }
    pars++;
  }

  if(!r)
  { s = " The following parameters have not been set : "+s;
    throw(s);
  }
}

/**********************************************************/
/* Print the array of characters on stderr                */
/* p : pointer to an array of characters (lines to print) */
/*     the last line must be a null pointer.              */
/* return : 1                                             */
/**********************************************************/
int Syntax(const char **p,Parameter *pars,PArgsModes const mode)
{ 

  while(*p) /* Scan all the elements of p until a NULL entry is found! */
  { cerr << *p++ << "\n";
  }

  // Build the argument list
  while(pars->type)
  { if(!(pars->flag & PAR_FLAG_NOHELP))
    { cerr << ((pars->type==T_COMMENT) ? "": "  ") << pars->name;
      if(pars->alias && pars->alias[0])
      { cerr << "="<<pars->alias;
      }
      switch(pars->type)
      { case T_NONE   :                    break;
        case T_INT    : cerr << " INT";    break;
        case T_REAL   : cerr << " REAL";   break;
        case T_CHAR   : cerr << " CHAR";   break;
        case T_BOOL   : cerr << " BOOL";   break;
        case T_FLAG   :                    break;
        case T_STRING : cerr << " STRING"; break;
        case T_COMMENT:                    break;
        default       : cerr << " UNKNOWN"; break; 
      }
      if(pars->help)
      { if(pars->type!=T_COMMENT) cerr << "\t: ";
        cerr << pars->help;
      }
      cerr << "\n";
    }   
    pars++;
  }
  
  cerr << "Standard flags : ";
  if(mode & PA_H_ALIASHELP)         cerr << "-h ";
  if(mode & PA_HELP_ALIASHELP)      cerr << "-help ";
  if(mode & PA_V_ALIAVERSION)       cerr << "-v ";
  if(mode & PA_VERSION_ALIAVERSION) cerr << "-version ";
  if(mode & PA_MPL_ALIASMPL)        cerr << "-mpl ";
  if(mode & PA_L_ALIASMPL)          cerr << "-l ";
  if(mode & PA_MPLS_ALIASMPL)       cerr << "-MPL ";
  if(mode & PA_LS_ALIASMPL)         cerr << "-L ";
  cerr << "\n";

  return(1);
}

/*************************************************/
/* Display the help message for the argument arg */
/* Search help in pars                           */
/*************************************************/
void display_help(const char *arg, Parameter *pars)
{ while(pars->type)
  { if(!strcmp(pars->name,arg) || (pars->alias &&!strcmp(pars->alias,arg)) )
    { cerr << pars->name;
      if(pars->alias && pars->alias[0])
      { cerr << "="<<pars->alias;
      }
      switch(pars->type)
      { case T_NONE   :                    break;
        case T_INT    : cerr << " INT";    break;
        case T_REAL   : cerr << " REAL";   break;
        case T_CHAR   : cerr << " CHAR";   break;
        case T_BOOL   : cerr << " BOOL";   break;
        case T_FLAG   :                    break;
        case T_STRING : cerr << " STRING"; break;
        case T_COMMENT:                    break;
        default       : cerr << " UNKNOWN"; break; 
      }
      if(pars->help)
      { if(pars->type!=T_COMMENT) cerr << "\t: ";
        cerr << pars->help;
      }
      cerr << "\n";
    }
    pars++;
  }
}

/*************************************/
/* Convert a string to a boolean     */
/* 1,true,TRUE,True    -> true       */
/* 0,false,FALSE,False -> flase      */
/* throw (1): error : invalid string */
/*************************************/
bool readBoolValue(char const *s)
{ if(!strcmp(s,"1") || 
     !strcmp(s,"true") ||
     !strcmp(s,"TRUE") || 
     !strcmp(s,"True") 
     )
  { return(true);
  }
  else if(!strcmp(s,"0") || 
          !strcmp(s,"false") ||
          !strcmp(s,"FALSE") || 
          !strcmp(s,"False") 
         ) 
  { return(false);
  }
  else
  { throw (1);
  }
}

/*****************************************/
/* Putput the list of parameters on cout */
/* pars : the array of parameters        */
/* Flags are never listed                */
/* Format :                              */
/* # VERSION STRING                      */
/* # date                                */
/* par1 val1                             */
/* par2 val2                             */
/* ...                                   */
/*****************************************/
void makeParList(Parameter *pars,string FileName)
{ time_t now;
  time ( &now );
  fstream ofs;

  ofs.open(FileName.data(),ios::out);  

  ofs << "# "<<VERSION <<"\n";
  ofs << "# "<< ctime(&now);
  while(pars->type)
  { if((pars->type!=T_FLAG) && !(pars->flag & PAR_FLAG_NOPARLIST))
    { ofs <<pars->name << " ";
      
      switch(pars->type)
      { case T_INT    : ofs << *((int *)pars->p);    break;
        case T_REAL   : ofs << *((double *)pars->p);   break;
        case T_CHAR   : ofs << *((char *)pars->p);   break;
        case T_BOOL   : ofs << (*((bool *)pars->p) ? "true" : "false"); break;
        case T_STRING : if(pars->flag & PAR_FLAG_QUOTEPSTR)
                        { ofs << "\""<<((char *)pars->p)<< "\""; 
                        }
                        else
                        { ofs << ((char *)pars->p); 
                        }
                        break;
        case T_FLAG   :
        case T_COMMENT:                              break;
        default       : ofs << "UNKNOWN VALUE TYPE"; break; 
      }
      ofs << "\n";
    }   
    pars++;
  }    
}

/****************************************************************/

/************************************************************************/ 
/* The function which reads the paramaters in uses and array of         */
/* The Paramaters structure is the same as GetArgs.                     */
/************************************************************************/

/*************************************************************************/
/* Read the parameters or print a help message.                          */
/* Check that all the parameters have been initialised                   */
/* Print a help message or version number (1 arg : -h -help ...)         */
/* Read the parameters from a file argv[1] and scan the other arguments  */
/*  The command line argument values overwrite the file's values         */
/*  -l or -mpl : create a parameters file: argv[1]+"s"                   */
/*                                                                       */
/* argc: No of parameters                                                */
/* argv: array of parameters                                             */
/* help : array of help string (last one is 0)                           */
/* pars : array of Parameter definitions                                 */
/* Return : 1 : OK                                                       */
/*          0 : error                                                    */
/*************************************************************************/
int GetPars(int const  argc,char *const *argv,Parameter *pars,
            const char **help, char ** &Argv,PArgsModes const mode)
{ int makeparlist=0;
  int Argc=0;

  /* Read Default Values if any */
  Set_Default_Value(pars);

  if(argc == 1) /* Read from Keyboard */
  { Read_Parameter_Keyboard(pars);
  }
  else if(argc == 2)
  { if(((mode & PA_H_ALIASHELP) && !strcmp("-h",argv[1])) ||  
       ((mode & PA_HELP_ALIASHELP) && !strcmp("-help",argv[1]))) 
    { Syntax(help,pars,mode); /* Display help message and exit */
      exit(1);
    }

    if(((mode & PA_V_ALIAVERSION ) && !strcmp("-v",argv[1])) || 
       ((mode & PA_VERSION_ALIAVERSION ) && !strcmp("-version",argv[1]))) 
    { cerr << VERSION << "\n"; /* Display Version Number */
      exit(1);
    }
  }

    /* Read from file */
  Read_Parameter_File(pars,Open_File(argv[1]));

  /* Scan the argument list */
  /* Skipp argv[0]; argv[1]=ParFile => argv[0] */
  Argc = scanArgList(argc-1,&argv[1],pars,help,Argv,mode,makeparlist);

  /* Check that every parameter has been set  */
  Check_Par_Read(pars); /* Throw if there is a problem */

  if(makeparlist & MPL_SET)
  { makeParList(pars,((string)argv[1])+"s");
    if(makeparlist & MPLS_SET) exit(0);
  }

  return(Argc);
}

/********************************************************/
/* Reads in each parameter from the keyboard.           */
/* Promt for a value for each of them.                  */
/* pars : pointer to the list of Parameters definitions */
/********************************************************/
void Read_Parameter_Keyboard(Parameter *pars)
{ char buff[80];
  while(pars->type) /* Last element of p is a dummy line with type NULL */
  { cout << pars->name<< " : "; /* Prompt for each parameters */
    switch(pars->type) /* Reads in the value */
    { case T_BOOL :
                 cin >> buff;
                 try 
                 { *((bool *)(pars->p)) = readBoolValue(buff);
                 }
                 catch(int i)
                 { throw((string)" Syntax Error: invalid bool value <"+
                          buff+"> for parameter "+pars->name);
                 }
                 pars->flag |= PAR_FLAG_READ;
                 break;
      case T_INT :
                 cin >> *((int *)pars->p);
                 pars->flag |= PAR_FLAG_READ;
                 break;
      case T_CHAR :
                 cin >> *((char *)pars->p);
                 pars->flag |= PAR_FLAG_READ;
                 break;
      case T_REAL :
                 cin >> *((double *)pars->p);
                 pars->flag |= PAR_FLAG_READ;
                 break;
      case T_STRING :
                 cin >> (char *)pars->p;
                 pars->flag |= PAR_FLAG_READ;
                 break;
      case T_NONE :
      case T_FLAG : 
      case T_COMMENT :
                 throw((string)"ERROR in Read_Parameter_Keyboard"); break;
    }
    pars++;
  }
}

/*********************************************************************/
/* Read_Parameter_File(Parameter *pars,ifstream *ifs)                */
/* Reads the file ifs, and set the value of the Parameters listed    */
/* in the structure pointed to by pars to the values set in the file */
/* pars : Points to an array of Parameter.                           */
/* ifs : a pointer to an ifstream open for reading                   */
/* Throw : string :error message on error.                           */
/*********************************************************************/
void Read_Parameter_File(Parameter *pars,ifstream *ifs)
{ char Name[MAX_PAR_LINE_LENGTH],Val[MAX_PAR_LINE_LENGTH],Buff[80];
  PArgsRead read;
  Parameter *p;
  int line_no=0;

  try {
  while((read = Read_One_line(ifs,Name,Val,line_no)) != PA_READ_EOF)
  { 
    p = 0;
    if(!(p = Get_Par_Details(pars,Name)))
    { throw((string)"Invalid parameter "+Name);
    }

    switch(read)
    { case PA_READ_ONE_WORD :
               if (p->type == T_BOOL)
               { /* Reverse Value */
                 *(bool *)(p->p) = *(bool *)(p->p) ? false : true;
                 p->flag |= PAR_FLAG_READ;
               }
               else 
               { throw((string)"");
               }
               break;
      case PA_READ_TWO_WORD :
               if (p->type == T_BOOL)
               { try 
                 { *((bool *)(p->p)) = readBoolValue(Val);
                 }
                 catch(int i)
                 { throw((string)"invalid bool value <"+Val+">");
                 }
                 p->flag |= PAR_FLAG_READ;
               }
               else if (p->type == T_CHAR)
               { sscanf(Val,"%c",(char *)p->p);
                 p->flag |= PAR_FLAG_READ;
               }
               else if (p->type == T_INT)
               { sscanf(Val,"%d",(int *)p->p);
                 p->flag |= PAR_FLAG_READ;
               }
               else if (p->type == T_REAL)
               { sscanf(Val,"%lg",(double *)p->p);
                 p->flag |= PAR_FLAG_READ;
               }
               else if (p->type == T_STRING)
               { strcpy((char *)p->p,Val);
                 /* bug in strncpy !!*/
                 /* strncpy(p->p,Val,MAX_PAR_LINE_LENGTH);*/
                 p->flag |= PAR_FLAG_READ;
               }
               else 
               { throw((string)"");
               }
               break;
       case PA_READ_WORD_AND_STRING :
               if (p->type == T_STRING)
               { strcpy((char *)p->p,Val);
                 // strncpy((char *)p->p,Val,MAX_PAR_LINE_LENGTH);
                 p->flag |= PAR_FLAG_READ;
               }
               else 
               { throw((string)"");
               }
              break;
       case PA_READ_EOF :
       case PA_READ_EMPTY_LINE :
       case PA_READ_ERROR : 
              throw((string)"ERROR in Read_Parameter_File"); break;
 
    }
  }
  }
  catch (string s_err)
  { sprintf(Buff,"%d",line_no);
    throw((string)" Syntax error in par file: line "+Buff+" : "+s_err);
  }
}


/**************************************************************************/ 
/* Read one data line :                                                   */
/* Format : Name  DATA                                                    */
/*     Name and Data are separated by 1 or more spaces.                   */
/*     If the first char of data is ", data is read up to the next "      */
/*     \ is used as the escape character \" == ", \\ == \ \n == \n \t==\t */
/* Return : PA_READ_ONE_WORD : read 1 word (name)                         */
/*          PA_READ_TWO_WORD : read 2 words (name val)                    */
/*          PA_READ_WORD_AND_STRING : read 2 words (name string)          */
/*          PA_READ_TWO_WORD : empty line                                 */
/*          PA_READ_EOF : EOF, no more data                               */
/* Throw : a string (error message) on error                              */
/**************************************************************************/
PArgsRead Read_One_line(ifstream *ifs,char *name,char *val,int &line_no)
{ bool skip;
  int i;
  char c;

  name[0] = 0;
  val[0] = 0;

  if(!ifs->good()) return(PA_READ_EOF);

  skip = true;

  /* Skip all the comment lines */
  while(skip)
  { if(!ifs->good() || !ifs->get(c)) return(PA_READ_EOF);

    if(c == '#') /* Comment line : skip it */
    { while (ifs->get(c) && (c != '\n'));
      ++line_no;
    }
    else
    { if((c != '\n') && c) // skip empty lines
      { skip = false;
      }
    }
  }

  i = 0;
  name[i++] = c;

  // We will eventualy read 1 \n : increament lino now.
  ++line_no;

  /* Read char until next space */
  while(ifs->good() && ifs->get(c) && (c != ' ') && (c != '\n') && (c != '\t'))
  { name[i++] = c;
  }
  name[i] = 0;

  if(!c || (c == '\n') || !ifs->good()) return(PA_READ_ONE_WORD);

  /* Read separating spaces */
  while(ifs->good() && ifs->get(c) && (c == ' ')&& (c == '\t'));
  if(!c || (c == '\n') || !ifs->good()) return(PA_READ_ONE_WORD);

  /* Read Value */
  i = 0;
  if(c == '"') /* Read a string */
  { while(ifs->good() && ifs->get(c) && (c != '\n'))
    { if(c == '\\') /* Escape char */
      { if (!ifs->good() || !ifs->get(c) || (c == '\n'))
        { /* Syntax error : \ at end of line */
          throw((string)"escape character at end of line.");
        }
        switch (c)
        { case 'n' : c = '\n'; break;
          case 't' : c = '\t'; break;
        }
        val[i++] = c;
      }
      else if (c == '"') /* End of string */
      { val[i] = 0;
        /* Read end of line */    
        while(ifs->good() && ifs->get(c) && (c != '\n'));

        return(PA_READ_WORD_AND_STRING); /* Read 2 words : name + string */
      }
      else val[i++] = c;
    }
    /* Syntax error : string not terminated */
    val[i] = 0;
    throw((string)"non terminated string: "+val);
  }
  else /* Read a word */
  { val[i++] = c;
    while(ifs->good() && ifs->get(c) && (c != ' ') && (c!='\n') && (c!='\t'))
    { val[i++] = c;
    }
    val[i] = 0;
    if(!c || (c == '\n') || !ifs->good()) return(PA_READ_TWO_WORD);
    /* Read end of line */
    while(ifs->good() && ifs->get(c) && (c != '\n'));
    return(PA_READ_TWO_WORD); /* Read 2 words */
  }
}

/**********************************************************************/
/* Open the file Name for reading and throw a string if it fails.     */
/* Name : the name of the file to open.                               */
/* Return : ifstream pointer                                          */
/* Throw : string if open fails                                       */
/**********************************************************************/
ifstream *Open_File(char *Name)
{ ifstream *ifs;

  ifs = new ifstream(Name);
  if(!ifs->good())
  { throw((string)" Error: Can't open file : "+Name);
  }
  return(ifs);
}

/***********************************************************/
/* Return the index of string name in the string array ptr */
/* Return : index of name in string array                  */
/*          -1 : name not found                            */
/***********************************************************/
int string_index(const char **ptr,const char *name)
{ int i=0;
  
  while(*ptr)
  { if(!strcmp(*ptr++,name)) return(i);
    ++i;
  }
  return(-1);
}
