#include <getargs.h>
#include <m_except.h>
#include <fstream>
#include <iostream>
#include <vector>
#include <cstdlib>
#include <cstring>
#include "p_cages_bi_sp.h"
#include "p_cages_bi_Ato.h"
#include "p_cages_bi_Atco.h"
#include "p_cages_bi_Atid.h"
#include "p_cages_bi_DArd.h"
#include "p_cages_bi_DArt.h"

using namespace std;

const char *VERSION = "bi_sym_cages: v1.0; 3/3/23; B. Piette";

struct pars {
  char type[256];
  int P1;
  int P2;
  int q1;
  int q2;
  int q3;
  int q4;
  int q5;
  int Q1;
  int Q2;
  int Q3;
  int Q4;
  int Q5;
  double cl;
  double ca;
  double cc;
  double cpc;
  int cage_weight;
  double T0;
  double Tmin;
  double Tcoef;
  double low_target_rate;
  double dx_min;
  double dx_max;
  int n_sweep;
  int n_monitor;
  int track;
  double tol;
  int iter_max;
  double epsilon;

  double S1;
  double S2;
  double ts;
  //
  int n_relax; // number of relaxations to find best
  int n_relax_cla; // number of relaxations for cla MC 
  int cla_mc; // use MC for cl ca relaxation 
  int N_cl_ca;
  int quasi_newton; // use the qn gradient method
  int pure_mc; // Only use pure MC
  int use_grad_mc; //Use gradient MC every step
  double T0_cla;
  double Tmin_cla;
  double Tcoef_cla;
  double lambda0;
} par;


Parameter Pars[] = {
 {"type" ,T_STRING, par.type, 0, 0, 0, " Type of cage "},
 {"P1" ,T_INT, &par.P1, 0, 0, 0, " Face type 1 size"},
 {"P2" ,T_INT, &par.P2, "3", 0, 0, "Face type 2 size"},
 {"q1" ,T_INT, &par.q1, 0, 0, 0, " hole edge 1"},
 {"q2" ,T_INT, &par.q2, 0, 0, 0, " hole edge 2"},
 {"q3" ,T_INT, &par.q3, 0, 0, 0, " hole edge 3"},
 {"q4" ,T_INT, &par.q4, "0", 0, 0, " hole edge 4"},
 {"q5" ,T_INT, &par.q5, "0", 0, 0, " hole edge 5"},
 {"Q1" ,T_INT, &par.Q1, 0, 0, 0, " hole edge 1"},
 {"Q2" ,T_INT, &par.Q2, 0, 0, 0, " hole edge 2"},
 {"Q3" ,T_INT, &par.Q3, 0, 0, 0, " hole edge 3"},
 {"Q4" ,T_INT, &par.Q4, "0", 0, 0, " hole edge 4"},
 {"Q5" ,T_INT, &par.Q5, "0", 0, 0, " hole edge 5"},
 {"cl" ,T_REAL, &par.cl, "1", 0, 0, " Lenght weight"},
 {"ca" ,T_REAL, &par.ca, "1", 0, 0, " Angle weight"},
 {"cc" ,T_REAL, &par.cc, 0, 0, 0, " Face convexity weight"},
 {"cpc" ,T_REAL, &par.cc, 0, 0, 0, " P-cage convexity weight"},
 {"cage_weight" ,T_FLAG,&par.cage_weight,"0",0,"cw"," use full cage weights"},
 {"T0" ,T_REAL,&par.T0,"1",0,0," : initial Temperature"},
 {"Tmin" ,T_REAL,&par.Tmin,"1e-6",0,0," final Temperature"},
 {"Tcoef" ,T_REAL,&par.Tcoef,"0.95",0,0," Temperature decrease"},
 {"low_target_rate" ,T_REAL,&par.low_target_rate,"0.33",0,"ltr","Low target rate"},
 {"dx_min" ,T_REAL,&par.dx_min,"1e-15",0,0," Minimum value for dX"},
 {"dx_max" ,T_REAL,&par.dx_max,"1e15",0,0," Maximum value for dX"},
 {"n_sweep" ,T_INT,&par.n_sweep,"1000",0,"ns"," no of iteration per sweep"},
 {"n_monitor" ,T_INT,&par.n_monitor,"100",0,"nm"," no of steps between mons"},
 {"track" ,T_INT,&par.track,"0",0,0," track progres"},
 {"tol",T_REAL,&par.tol,"1e-8",0,0,
  " relative error for quasi newton"},
 {"iter_max",T_INT,&par.iter_max,"1000",0,"itm",
  " Maximum number of quasi newton iterations"},
 
 {"S1",T_REAL,&par.S1,"1",0,0, " Face 1 scale for initial shape"},
 {"S2",T_REAL,&par.S2,"1",0,0, " Face 2 scale for initial shape"},
 {"ts",T_REAL,&par.ts,"1",0,0, " node scale for initial shape"},

 {"epsilon",T_REAL,&par.epsilon,"1e-8",0,"eps",
  " dx for gradient"},
 {"n_relax" ,T_INT,&par.n_relax,"1",0,"nr",
  " Number of relaxation to find best minimum"},
 {"n_relax_cla" ,T_INT,&par.n_relax_cla,"1",0,"nrcla",
  " Number of relaxation for cl ca MC"},
 {"N_cl_ca" ,T_INT,&par.N_cl_ca,"-1",0,"ncla",
  " Number of steps to scan ca and cl with ca+cl=2"},
 {"cla_mc" ,T_FLAG,&par.cla_mc,"0",0,"Use MC for cl ca relaxation",
  " Number of steps to scan ca and cl with ca+cl=2"},
 {"quasi_newton" ,T_FLAG,&par.quasi_newton,"0",0,"qn",
  " Use the quasi newton method instead of the gradient method"},
 {"pure_mc" ,T_FLAG,&par.pure_mc,"0",0,0,
  " Use the pure MC, no refining"},
 {"use_grad_mc" ,T_FLAG,&par.use_grad_mc,"0",0,0,
  " Use gradient MC every step as well"},
 {"T0_cla" ,T_REAL,&par.T0_cla,"1e-5",0,0,
    " : initial Temperature for cl-ca scan"},
 {"Tmin_cla" ,T_REAL,&par.Tmin_cla,"1e-7",0,0,
    " final Temperature for cl-ca scan"},
 {"Tcoef_cla" ,T_REAL,&par.Tcoef_cla,"0.95",0,0,
  " Temperature decrease for cl-ca scan" },
 {"lambda0" ,T_REAL,&par.lambda0,"1.0",0,"l0",
  " Initial lambda for gradient method" },
 { 0, T_NONE,0,0,0,0,0},
};

 
const char * help[] = {
 "sym_cages -P1 N -P2 N -q1 N -q2 N -q3 N  [-q4 N] [-q5 N]  [-T0 R] [-Tmin R] [-n_sweep N] [-n_monitor N]",
 "  Try to generate a prism base polygonal cage",
 " "
};


const int MAX_CROSS_ITER = 10;
const double CROSS_TOL = 1e-6;

const char *p_cage_bi_types[] = { "sp", "Ato", "Atco", "Atid", "DArd", "DArt", 0 };  


int find_type(const char *t)
{ int i = 0;
  const char *p;
  while((p=p_cage_bi_types[i]))
  { if (strcmp(t,p) == 0) { return(i); }
    i++;
  }
  return(-1); // not found
}

/********************************************************/
/* Perform the minimisation for the provided parameters */
/* Save the date in a file and global data in ofs       */
/********************************************************/
void relax(p_cage_bi *pcp, double cl, double ca, ofstream &ofs)
{ pcp->ca_ = ca;
  pcp->cl_ = cl;

  cout<<"cl="<<cl<<" ca="<<ca<<"\n";
  pcp->new_geometry();
  if (par.cla_mc)
  { pcp->relax_multi(par.n_relax_cla, pcp->relax_type_,
	             par.T0_cla, par.Tmin_cla, par.Tcoef_cla,
		     par.tol, true);
  }
  else
  { pcp->minimise_grad(par.tol);
    cout<<"cl="<<cl<<" ca="<<ca<<" GR: E="<<pcp->E_<<"\n";
  }

  stringstream ss2;
  ss2 << pcp->filename_prefix() <<"_cl"<<cl<<"_ca"<<ca<<"_face.txt";
  pcp->write(ss2.str().c_str());

  ofs <<"cl="<<cl <<" ca="<<ca<<"\n";
  ofs <<" x1="<<pcp->x1_<<" y1="<<pcp->y1_<<" z1="<<pcp->z1_
      <<" x2="<<pcp->x2_<<" y2="<<pcp->y2_<<" z2="<<pcp->z2_<<"\n";
  ofs <<" E="<<pcp->E_<<"\n";
  ofs <<"Error: "<<pcp->err_l_max_<<" "<<pcp->err_a_max_
      <<" "<<pcp->err_l_av_max_<<" "<< (pcp->err_l_max_+pcp->err_a_max_)
      <<" "<<pcp->err_l_av_<<" "<<pcp->err_a_av_<<"\n";
  ofs <<"Error1: "<<pcp->err_l_max_1_<<" "<<pcp->err_a_max_1_
      <<" "<<pcp->err_l_av_max_1_<<" "<< (pcp->err_l_max_1_+pcp->err_a_max_1_)
      <<" "<<pcp->err_l_av_1_<<" "<<pcp->err_a_av_1_<<"\n";
  ofs <<"Error2: "<<pcp->err_l_max_2_<<" "<<pcp->err_a_max_2_
      <<" "<<pcp->err_l_av_max_2_<<" "<< (pcp->err_l_max_2_+pcp->err_a_max_2_)
      <<" "<<pcp->err_l_av_2_<<" "<<pcp->err_a_av_2_<<"\n";
 
}

/********************************************************/
/* Find the cl and ca where the error matches           */
/* d1 : err_l1-err_a1                                   */
/* d2 : err_l2-err_a2                                   */
/* ofs : global output file                             */
/* tol : tolerance on the error                         */
/* use_av_l : user average l instead of l               */
/********************************************************/
void find_crossing_point(p_cage_bi *pcp, double cl_ca,
			 double cl1, double d1,
			 double cl2, double d2,
			 ofstream &ofs, double tol, bool use_av_l = false)
{ double ln_ncl,ncl, nca, new_d;
  int n=0;
  cout<<"find_crossing_point()\n";

  if ((d1*d2)>0 || (cl1<0)) { return ;}
  
  while ((fabs(d1) > tol) && (fabs(d2) > tol) && (n++ < MAX_CROSS_ITER))
  { //ncl = cl1 + d1*(cl2-cl1)/(d1-d2);
    ln_ncl = log(cl1) + d1*(log(cl2/cl1))/(d1-d2);
    ncl = exp(ln_ncl);
    cout<<"cl1 = "<<cl1<<" ln_ncl="<<ln_ncl<<" ncl="<<ncl<<"\n";
    nca = cl_ca-ncl;
    cout<<"cl1="<<cl1<<" d1="<<d1<<" cl2="<<cl2<<" d2="<<d2<<"\n";

    pcp->new_geometry();
    relax(pcp, ncl, nca, ofs);
    //new_d = pcp->err_l_max_-pcp->err_a_max_;
    if (use_av_l)
    { new_d = pcp->err_l_av_max_-pcp->err_a_max_;
    }
    else
    { new_d = pcp->err_l_max_-pcp->err_a_max_;
    }
    cout<<"ncl="<<ncl<<" new_d="<<new_d<<"\n";
    
    if(new_d*d1 < 0) // replace cl2,d2
    { cl2 = ncl;
      d2 = new_d;
    }
    else
    { cl1 = ncl;
      d1 = new_d;
    }
  }
}
			 

int main(int argc, char **argv)
{ int Argc;
  char **Argv=0;
  //double Emin, *best_pars;
  stringstream ss;
  string output_file;
  
  
  try
  { int size;
    p_cage_bi *pcp;
    
    try
    { Argc=GetArgs(argc,argv,Pars,help,Argv,PA_GETARGS_STD);
      //Argc=GetPars(argc,argv,Pars,help,Argv,PA_GETPARS_STD);
    }
    catch(string s_err)
    { cerr << "sym_cages: "<< s_err<<"\n";
      exit(1);
    }

    // check parameters
    int valq = 3;
    if (par.q4 > 0)
    { valq +=1;
      if (par.q5 > 0 )
      { valq +=1;
      }
    }
    int valQ = 3;
    if (par.Q4 > 0)
    { valQ +=1;
      if (par.Q5 > 0 )
      { valQ +=1;
      }
    }
    
    if (par.q1 <1 ||  par.q2 <1 || par.q3 <1 || par.q4 <0 || par.q5 <0
	|| (par.q1+par.q2+par.q3+par.q4+par.q5+valq != par.P1))
    { cerr<<"Invalid qs parameters\n";
      exit(5);
    }
    
    if (par.Q1 <1 ||  par.Q2 <1 || par.Q3 <1 || par.Q4 <0 || par.Q5 <0
	|| (par.Q1+par.Q2+par.Q3+par.Q4+par.Q5+valQ != par.P2))
    { cerr<<"Invalid Qs parameters\n";
      cerr<<"Q1="<<par.Q1<<" Q2="<<par.Q2<<" Q3="<<par.Q3<<" Q4="<<par.Q4<<"\n";
      cerr<<"P2="<<par.P2<<"\n";
      exit(5);
    }


    double *V0=0;
    int pcage_type = find_type(par.type);

    cout<<"pcage_type ="<<pcage_type<<"\n";
    
    switch(pcage_type)
    { case TYPE_SP:
	   { p_cage_bi_sp pp0(par.P1, par.P2, par.q1, par.q2, par.q3,
			      par.Q1, par.Q2, par.Q3,
			      par.cl, par.ca, par.cc, par.cpc,
			      V0, 0, par.n_sweep,
			      par.n_monitor, 0.1);
	     V0 = pp0.init_pcage(par.q1, par.q2, par.q3, par.Q1, par.Q2, par.Q3,
				 par.P1, par.P2, size, par.S1, par.S2, par.ts);
	     cout<<"bi_sym_cages A size="<<size<<"\n"<<flush;
	     pcp = new p_cage_bi_sp(par.P1, par.P2, par.q1, par.q2, par.q3,
				    par.Q1, par.Q2, par.Q3,
				    par.cl, par.ca, par.cc, par.cpc,
				    V0, 0, par.n_sweep,
				    par.n_monitor, 0.1);
	     cout<<"bi_sym_cages B\n"<<flush;
	   }
           break;
	   
       case TYPE_Ato:
	   { p_cage_bi_Ato  pp0(par.P1, par.P2, par.q1, par.q2, par.q3,
			        par.Q1, par.Q2, par.Q3,
			        par.cl, par.ca, par.cc, par.cpc,
				V0, 0, par.n_sweep,
			        par.n_monitor, 0.1);
	     V0 = pp0.init_pcage(par.q1, par.q2, par.q3, par.Q1, par.Q2, par.Q3,
				 par.P1, par.P2, size, par.S1, par.S2, par.ts);
	     cout<<"bi_sym_cages A size="<<size<<"\n"<<flush;
	     pcp = new p_cage_bi_Ato(par.P1, par.P2, par.q1, par.q2, par.q3,
				     par.Q1, par.Q2, par.Q3,
				     par.cl, par.ca, par.cc, par.cpc,
				     V0, 0, par.n_sweep,
				     par.n_monitor, 0.1);
	     cout<<"bi_sym_cages B\n"<<flush;
	   }
           break;
       case TYPE_Atco:
	   { p_cage_bi_Atco  pp0(par.P1, par.P2, par.q1, par.q2, par.q3,
			         par.Q1, par.Q2, par.Q3,
			         par.cl, par.ca, par.cc, par.cpc,
				 V0, 0, par.n_sweep,
			         par.n_monitor, 0.1);
	     V0 = pp0.init_pcage(par.q1, par.q2, par.q3, par.Q1, par.Q2, par.Q3,
				 par.P1, par.P2, size, par.S1, par.S2, par.ts);
	     cout<<"bi_sym_cages A size="<<size<<"\n"<<flush;
	     pcp = new p_cage_bi_Atco(par.P1, par.P2, par.q1, par.q2, par.q3,
				      par.Q1, par.Q2, par.Q3,
				      par.cl, par.ca, par.cc, par.cpc,
				      V0, 0, par.n_sweep,
				      par.n_monitor, 0.1);
	     cout<<"bi_sym_cages B\n"<<flush;
	   }
           break;
       case TYPE_Atid:
	   { p_cage_bi_Atid  pp0(par.P1, par.P2, par.q1, par.q2, par.q3,
			         par.Q1, par.Q2, par.Q3,
			         par.cl, par.ca, par.cc, par.cpc,
				 V0, 0, par.n_sweep,
			         par.n_monitor, 0.1);
	     V0 = pp0.init_pcage(par.q1, par.q2, par.q3, par.Q1, par.Q2, par.Q3,
				 par.P1, par.P2, size, par.S1, par.S2, par.ts);
	     cout<<"bi_sym_cages A size="<<size<<"\n"<<flush;
	     pcp = new p_cage_bi_Atid(par.P1, par.P2, par.q1, par.q2, par.q3,
				      par.Q1, par.Q2, par.Q3,
				      par.cl, par.ca, par.cc, par.cpc,
				      V0, 0, par.n_sweep,
				      par.n_monitor, 0.1);
	     cout<<"bi_sym_cages B\n"<<flush;
	   }
           break;
       case TYPE_DArd:
	   { if ((par.q2 !=par.q1) || (par.q3 !=par.q1) )
	     { cout<<"Invalid values for qs\n.";
	       exit(5);
	     }
	     if ((par.Q2 !=par.Q1) || (par.Q3 !=par.Q1) || (par.Q4 !=par.Q1))
	     { cout<<"Invalid values for Qs\n.";
	       exit(5);
	     }
	      
	     p_cage_bi_DArd  pp0(par.P1, par.P2, par.q1, par.q2, par.q3,
			         par.Q1, par.Q2, par.Q3, par.Q4,
			         par.cl, par.ca, par.cc, par.cpc,
				 V0, 0, par.n_sweep,
			         par.n_monitor, 0.1);
	     V0 = pp0.init_pcage(par.q1, par.q2, par.q3,
				 par.Q1, par.Q2, par.Q3, par.Q4,
				 par.P1, par.P2, size, par.S1, par.S2, par.ts);
	     cout<<"bi_sym_cages A size="<<size<<"\n"<<flush;
	     pcp = new p_cage_bi_DArd(par.P1, par.P2, par.q1, par.q2, par.q3,
				      par.Q1, par.Q2, par.Q3, par.Q4,
				      par.cl, par.ca, par.cc, par.cpc,
				      V0, 0, par.n_sweep,
				      par.n_monitor, 0.1);
	     cout<<"bi_sym_cages B\n"<<flush;
	   }
           break;
       case TYPE_DArt:
	   { if ((par.q2 !=par.q1) || (par.q3 !=par.q1) )
	     { cout<<"Invalid values for qs\n.";
	       exit(5);
	     }
	     if ((par.Q2 !=par.Q1) || (par.Q3 !=par.Q1) || (par.Q4 !=par.Q1)
		 || (par.Q5 !=par.Q1))
	     { cout<<"Invalid values for Qs\n.";
	       exit(5);
	     }
	      
	     p_cage_bi_DArt  pp0(par.P1, par.P2, par.q1, par.q2, par.q3,
			         par.Q1, par.Q2, par.Q3, par.Q4, par.Q5,
			         par.cl, par.ca, par.cc, par.cpc,
				 V0, 0, par.n_sweep,
			         par.n_monitor, 0.1);
	     V0 = pp0.init_pcage(par.q1, par.q2, par.q3,
				 par.Q1, par.Q2, par.Q3, par.Q4, par.Q5,
				 par.P1, par.P2, size, par.S1, par.S2, par.ts);
	     cout<<"bi_sym_cages A size="<<size<<"\n"<<flush;
	     pcp = new p_cage_bi_DArt(par.P1, par.P2, par.q1, par.q2, par.q3,
				      par.Q1, par.Q2, par.Q3, par.Q4, par.Q5,
				      par.cl, par.ca, par.cc, par.cpc,
				      V0, 0, par.n_sweep,
				      par.n_monitor, 0.1);
	     cout<<"bi_sym_cages B\n"<<flush;
	   }
           break;
       default:
         cout<<"Invalid pcage type "<<par.type<<"\n";
         exit(5);
    }
    pcp->set_lambda0(par.lambda0);
    pcp->set_low_target_rate(par.low_target_rate);
    pcp->set_dx_min(par.dx_min);
    pcp->set_dx_max(par.dx_max);
    if (par.use_grad_mc)
    { pcp->set_use_grad_mc();
    }
    
    int relax_type = USE_GR;
    if (par.quasi_newton) { relax_type = USE_QN; }
    if (par.pure_mc) { relax_type = USE_NONE; }
    pcp->set_relax_type(relax_type);

    //p_cage_prism pcp(par.P, par.N, par.q1, par.q2, par.q3, par.cl, par.ca,
    //		     par.cc, V0, 0 ,par.n_sweep, par.n_monitor, 0.1);

    if (par.cage_weight)
    { pcp->use_cage_weights();
    }
    
    pcp->set_iter_max(par.iter_max);
    pcp->set_epsilon(par.epsilon);

    
    ss << pcp->filename_prefix()<<"_cl"<<par.cl<<"_ca"<<par.ca<<"_face.txt";
    output_file = ss.str();

    cout<<"ss\n"<<flush;
    
    for(int i=0; i < size; i++)
    { cout<<"X: "<<pcp->get_X()[i]<<" ";
    }
    cout<<"\n";
    if (par.track > 0)
    { pcp->set_tracking(par.track);
    }

    if(par.Tmin < par.T0)
    { cout<<"First relaxation\n";
      pcp->new_geometry();
      pcp->relax_multi(par.n_relax, pcp->relax_type_ ,
		       par.T0, par.Tmin, par.Tcoef, par.tol, true);
    }

    cout<<"N_cl_ca ="<<par.N_cl_ca<<"\n";
    if (par.N_cl_ca>0)
    { double cl, ca, ca_0 = 1, ca_last = 0.001;
      double cl_0 = 1, cl_last = 0.001;
      double cl_ca = 2;

      double prev_cl=-1, prev_ca=-1;
      double prev_err_l=-1, prev_err_l_av=-1, prev_err_a=-1;
      
      // SCAN ca DOWN
      ss.str("");
      ss <<pcp->filename_prefix()<<"_cl_fsum.txt";
      ofstream ofs;
      ofs.open(ss.str().c_str(),ios::out);
      ofs<<"#cl ca max_err_l max_err_a  max_err_l_av max_err_l+max_err_a av_err_l av_err_a E theta\n";
      
      double lndxa = (log(ca_0)-log(ca_last))/par.N_cl_ca;
      std::cerr<<"N_cl_ca="<<par.N_cl_ca<<" "<<log(ca_0)<<" "<<log(ca_last)<<"\n";
      std::cerr<<"ca="<<ca_0<<" ca_last="<<ca_last<<" lndxa="<<lndxa<<"\n";
      for (int n=0; n <= par.N_cl_ca; ++n)
      { ca =  n==0 ? ca_0: ca_last*exp((par.N_cl_ca-n)*lndxa);
	cl = cl_ca-ca;
	relax(pcp, cl, ca, ofs);

        // find point where pcp->err_l_max_ = pcp->err_a_max_
	// Check for sign change in difference
	double d = pcp->err_l_max_-pcp->err_a_max_;
	double prev_d = prev_err_l-prev_err_a;
	prev_err_l = pcp->err_l_max_;
	
	double d_av = pcp->err_l_av_max_-pcp->err_a_max_;
	double prev_d_av = prev_err_l_av-prev_err_a;
	prev_err_l_av = pcp->err_l_av_max_;
	
	prev_err_a = pcp->err_a_max_;
	if ( d*prev_d < 0.0)
	{ find_crossing_point(pcp, cl_ca, prev_cl, prev_d, cl, d, ofs,
			      CROSS_TOL);
	}
	if ( d_av*prev_d_av < 0.0)
	{ find_crossing_point(pcp, cl_ca, prev_cl, prev_d_av, cl, d_av, ofs,
			      CROSS_TOL, true);
	}
	
	prev_cl = cl;
	prev_ca = ca;
      }
      //std::cerr<<"ca="<<ca_0<<" ca_last="<<ca_last<<" lndxa="<<lndxa<<"\n"<<flush;

      // SCAN cl DOWN
      prev_cl = prev_ca = prev_err_l = prev_err_a = -1;
      
      ss.str("");
      ss <<pcp->filename_prefix()<<"_ca_fsum.txt";
      ofs.close();
      ofs.open(ss.str().c_str(),ios::out);
      ofs<<"#cl ca max_err_l max_err_a max_err_l_av max_err_l+max_err_a av_err_l av_err_a E theta\n";
      double lndxl = (log(cl_0)-log(cl_last))/par.N_cl_ca;
      std::cerr<<"cl="<<cl_0<<" cl_last="<<cl_last<<" lndxl="<<lndxl<<"\n";
      for (int n=0; n <= par.N_cl_ca; ++n)
      { cl = n==0 ? cl_0: cl_last*exp((par.N_cl_ca-n)*lndxl);
	ca = cl_ca-cl;
	relax(pcp, cl, ca, ofs);
	
	// find point where pcp->err_l_max_ = pcp->err_a_max_
	// Check for sign change in difference
	double d = pcp->err_l_max_-pcp->err_a_max_;
	double prev_d = prev_err_l-prev_err_a;
	prev_err_l = pcp->err_l_max_;
	
	double d_av = pcp->err_l_av_max_-pcp->err_a_max_;
	double prev_d_av = prev_err_l_av-prev_err_a;
	prev_err_l_av = pcp->err_l_av_max_;
	
	prev_err_a = pcp->err_a_max_;
 	if ( d*prev_d < 0.0)
	{ find_crossing_point(pcp, cl_ca, prev_cl, prev_d, cl, d, ofs,
			      CROSS_TOL);
	}
	if ( d_av*prev_d_av < 0.0)
	{ find_crossing_point(pcp, cl_ca, prev_cl, prev_d_av, cl, d_av, ofs,
			      CROSS_TOL, true);
	}
	
	prev_cl = cl;
	prev_ca = ca;
     }
    }
    else
    { pcp->write(output_file.c_str());
      pcp->show_details();
    }
    if (V0) { delete V0; }

  }
  catch (m_except e) { e(); }


}

