# Manuscript title:      Fever in focus: symptoms, diagnoses, and treatment of
#                         febrile children in Ghana, a longitudinal hospital study
# Last updated:          07.08.2023
# Creation date:         30.11.2021
# Author:                Lydia Rautman, MPH
# Session info:          R version 4.3.0 (2023-04-21)
#                        Platform: aarch64-apple-darwin20 (64-bit)
#                        Running under: macOS Monterey 12.3.1
#                        Matrix products: default
#                        BLAS:   /System/Library/Frameworks/Accelerate.framework/Versions/A/Frameworks/vecLib.framework/Versions/A/libBLAS.dylib 
#                        LAPACK: /Library/Frameworks/R.framework/Versions/4.3-arm64/Resources/lib/libRlapack.dylib;  LAPACK version 3.11.0
#                        locale:
#                          [1] en_US.UTF-8/en_US.UTF-8/en_US.UTF-8/C/en_US.UTF-8/en_US.UTF-8
#                        time zone: Europe/Berlin
#                        tzcode source: internal
#                        attached base packages:
#                          [1] stats     graphics  grDevices utils     datasets  methods   base     
#                        other attached packages:
#                          [1] tidyr_1.3.0        RColorBrewer_1.1-3 networkD3_0.4      stringr_1.5.0      dplyr_1.1.2
# Objective:             Provide a script to accompany a manuscript 
#                        detailing analyses for reproducibility.
# How to use:            Download data set "fws_all_obs.rdata" and revise 
#                           pathway under "Load data sets" as appropriate.
#                        Use R Index to navigate sections and run code as needed.
# Contents:              Set-up
#                        Results: in-text statistics
#                           Overview
#                           Longitudinal symptoms and diagnoses
#                           Treatment
#                        Tables and figures
#                           Table 1
#                           Supplementary Table 1
#                           Figure 1
#                           Figure 2
#                           Figure 3
#                           Figure 4
# Variable definitions:  
#    antimalarial_tf - antimalarial given at any point during hospital stay
#    antibiotic_tf - antibiotic given at any point during hospital stay
#    include - patient included in longitudinal analysese (inclusion criteria:
#        age ≥30 days, length of stay ≥2 days, ≤50% of longitudinal data missing)
#    length_stay_num - length of stay in days (numeric). Calculated as
#        date of discharge - date of admission
#    PREFIXES:
#     adm_ - data collected on admission
#     rev1_ - data collected on review day 1
#     rev2_ - data collected on review day 2
#     rev3_ - data collected on review day 3
#     rev5_ - data collected on review day 5
#     rev7_ - data collected on review day 7
#    SUFFIXES:
#     _parasitemia - malaria parasite count in blood
#     _fever - tympanic temperature ≥38 C
#     _temperature - tympanic temperature
#     _breathingdiff - difficulty breathing
#     _cough - cough
#     _vomiting - vomiting
#     _loosestool - loose stool
#     _abdompain - abdominal pain
#     _convulsions - convulsions
#
####################################

#### SET-UP ####

# Install packages if necessary
# lapply(c("dplyr", "stringr", "networkD3", "tidyr",
#         "RColorBrewer"), install.packages, character.only = TRUE)

# Load packages
lapply(c("dplyr", "stringr", "networkD3", "tidyr",
         "RColorBrewer"), require, character.only = TRUE)

# Load data set
setwd()
load("fws_all_obs.rdata")
  # All observations (frequency tables)
fws_an <- fws_all_obs %>% filter(include == 1)
  # Reduced data set (longitudinal analyses)

##############################################################

#### RESULTS: IN-TEXT STATISTICS ####

##############################################################

#### RESULTS: OVERVIEW ####

# Age in years
summary(fws_all_obs$ageinyears, na.rm = TRUE)
table(fws_all_obs$ageinyears < 5)
# Temperature at admission
summary(fws_all_obs$adm_temperature, na.rm = TRUE)
# Symptoms at admission
lapply(fws_all_obs[,c("adm_cough","adm_breathingdiff","adm_vomiting",
               "adm_loosestool","adm_abdompain", "adm_convulsions")],
       function(x) paste0(table(x[x != ""])["Yes"],"/", 
                          sum(table(x[x != ""])), " (", 
                          formattable::percent(as.character(prop.table(table(x[x != ""]))["Yes"]), digits = 0), ")"))
# Suspected diagnoses
lapply(c("Malaria","Sepsis","LRTI", "Gastroenteritis", "Meningitis",
         "UTI","Other infection", "Other non-infection"),
       function(x) table(str_detect(fws_all_obs$suspected_diag, x)))
# Malaria parasites at admission
table(fws_all_obs$adm_parasitemia > 0)

###

# Frequency of hospital stays >2 days
paste0(table(fws_all_obs$length_stay_num > 2)["TRUE"], "/", sum(!is.na(fws_all_obs$length_stay_num)), 
       " (", formattable::percent(as.character(prop.table(table(fws_all_obs$length_stay_num > 2))["TRUE"]), digits = 0), ")")

# Length of stay by diagnosis (corresponding to Figure 1)
lapply(c("Malaria","Gastroenteritis","UTI","Other non-infection",
         "Other infection","LRTI","Sepsis","Meningitis"),
       function(x) summary(fws_all_obs$length_stay_num[str_detect(fws_all_obs$discharge_diag, x)]))
###

# Cases excluded from analyses

# Observations with stay < 2 days or missing >50% of longitudinal data (include == 0)
table(fws_all_obs$length_stay_num < 2 | fws_all_obs$include == 0)
# Symptoms at admission by inclusion/exclusion status (corresponding to Supplementary Table 1)
lapply(fws_all_obs[,c("adm_cough","adm_breathingdiff","adm_vomiting",
               "adm_loosestool","adm_abdompain", "adm_convulsions")],
       function(x) prop.table(table(x[x != ""], fws_all_obs$include[x != ""]),2))
# Discharge diagnoses by inclusion/exclusion status (corresponding to Supplementary Table 1)
lapply(c("Malaria","Sepsis","LRTI", "Gastroenteritis", "Meningitis",
         "UTI","Other infection", "Other non-infection"),
       function(x) prop.table(table(str_detect(fws_all_obs$discharge_diag, x), fws_all_obs$include),2))


###############################################################################
#### RESULTS: LONGITUDINAL SYMPTOMS AND DIAGNOSES ####

# Frequency of symptoms over stay (corresponding to Figure 2)
n.rev <- as.data.frame(cbind(n_fever = paste0(sum(fws_an$adm_temperature > 37.99, na.rm = TRUE),"/",sum(fws_an$adm_temperature > 37.99)),
                             t(sapply(c("adm_cough","adm_breathingdiff","adm_vomiting",
                                        "adm_loosestool","adm_abdompain", "adm_convulsions"),
                                      function(x) paste0(sum(fws_an[,x] == "Yes"),"/",sum(fws_an[,x] == "Yes" | fws_an[,x] == "No"))))))
colnames(n.rev) <- c("n_fever","n_cough","n_breath","n_vom","n_stool","n_abdom","n_convuls")
n.rev1 =as.data.frame(NULL)
for(j in c("rev1","rev2","rev3","rev5","rev7")) {  
  n.rev1[j,"n_fever"] <-  paste0(sum(fws_an[,paste0(j,"_temperature")] > 37.99, na.rm = TRUE),"/",sum(!is.na(fws_an[,paste0(j,"_temperature")])))
  n.rev1[j,"n_cough"] <-  paste0(sum(fws_an[,paste0(j,"_cough")] == "Yes"),"/",sum(fws_an[,paste0(j,"_cough")] == "Yes" | fws_an[,paste0(j,"_cough")] == "No"))
  n.rev1[j,"n_breath"] <- paste0(sum(fws_an[,paste0(j,"_breathingdiff")] == "Yes"),"/",sum(fws_an[,paste0(j,"_breathingdiff")] == "Yes" | fws_an[,paste0(j,"_breathingdiff")] == "No"))
  n.rev1[j,"n_vom"] <-    paste0(sum(fws_an[,paste0(j,"_vomiting")] == "Yes"),"/",sum(fws_an[,paste0(j,"_vomiting")] == "Yes" | fws_an[,paste0(j,"_vomiting")] == "No"))
  n.rev1[j,"n_stool"] <-  paste0(sum(fws_an[,paste0(j,"_loosestool")] == "Yes"),"/",sum(fws_an[,paste0(j,"_loosestool")] == "Yes" | fws_an[,paste0(j,"_cough")] == "No"))
  n.rev1[j,"n_abdom"] <-  paste0(sum(fws_an[,paste0(j,"_abdompain")] == "Yes"),"/",sum(fws_an[,paste0(j,"_abdompain")] == "Yes" | fws_an[,paste0(j,"_abdompain")] == "No"))
  n.rev1[j,"n_convuls"] <-paste0(sum(fws_an[,paste0(j,"_convulsions")] == "Yes"),"/",sum(fws_an[,paste0(j,"_convulsions")] == "Yes" | fws_an[,paste0(j,"_convulsions")] == "No"))
}
n_symptoms <- rbind(adm = n.rev, n.rev1)
n_symptoms <- cbind("time" = c(0:3,5,7), n_symptoms[,1:7])
n_symptoms

###

# Prevalence of any symptoms at last review before discharge
fws_an <- fws_an %>%
  mutate(adm_anysymp = 
           ifelse(apply(fws_an[,c("adm_cough", "adm_breathingdiff", "adm_vomiting", "adm_loosestool",
                                       "adm_abdompain", "adm_convulsions", "adm_fever")], 1,
                        function(x) "Yes" %in% x), TRUE, 
                  ifelse(rowSums(fws_an[,c("rev1_cough", "rev1_breathingdiff",
                                           "rev1_vomiting", "rev1_loosestool", 
                                           "rev1_abdompain", "rev1_convulsions", 
                                           "rev1_fever")] == "") > 6, NA, FALSE)),
         rev1_anysymp = 
           ifelse(apply(fws_an[,c("rev1_cough", "rev1_breathingdiff","rev1_vomiting", 
                                  "rev1_loosestool","rev1_abdompain", "rev1_convulsions", "rev1_fever")], 1, 
                        function(x) "Yes" %in% x), TRUE, 
                  ifelse(rowSums(fws_an[,c("rev1_cough", "rev1_breathingdiff",
                                           "rev1_vomiting", "rev1_loosestool",
                                           "rev1_abdompain", "rev1_convulsions",
                                           "rev1_fever")] == "") > 6, NA, FALSE)),
         rev2_anysymp = 
           ifelse(apply(fws_an[,c("rev2_cough", "rev2_breathingdiff","rev2_vomiting", 
                                  "rev2_loosestool","rev2_abdompain", "rev2_convulsions", "rev2_fever")], 1,
                        function(x) "Yes" %in% x), TRUE, 
                  ifelse(rowSums(fws_an[,c("rev2_cough", "rev2_breathingdiff",
                                           "rev2_vomiting", "rev2_loosestool",
                                           "rev2_abdompain", "rev2_convulsions",
                                           "rev2_fever")] == "") > 6, NA, FALSE)),
         rev3_anysymp = 
           ifelse(apply(fws_an[,c("rev3_cough", "rev3_breathingdiff","rev3_vomiting", 
                                  "rev3_loosestool","rev3_abdompain", "rev3_convulsions", "rev3_fever")], 1,
                        function(x) "Yes" %in% x), TRUE,
                  ifelse(rowSums(fws_an[,c("rev3_cough", "rev3_breathingdiff",
                                           "rev3_vomiting", "rev3_loosestool",
                                           "rev3_abdompain", "rev3_convulsions", 
                                           "rev3_fever")] == "") > 6, NA, FALSE)),
         rev5_anysymp = 
           ifelse(apply(fws_an[,c("rev5_cough", "rev5_breathingdiff","rev5_vomiting", 
                                  "rev5_loosestool","rev5_abdompain", "rev5_convulsions", "rev5_fever")], 1, 
                        function(x) "Yes" %in% x), TRUE,
                  ifelse(rowSums(fws_an[,c("rev5_cough", "rev5_breathingdiff",
                                           "rev5_vomiting", "rev5_loosestool",
                                           "rev5_abdompain", "rev5_convulsions",
                                           "rev5_fever")] == "") > 6, NA, FALSE)),
         rev7_anysymp = 
           ifelse(apply(fws_an[,c("rev7_cough", "rev7_breathingdiff","rev7_vomiting", "rev7_loosestool",
                                  "rev7_abdompain", "rev7_convulsions", "rev7_fever")], 1,
                        function(x) "Yes" %in% x), TRUE,
                  ifelse(rowSums(fws_an[,c("rev7_cough", "rev7_breathingdiff",
                                           "rev7_vomiting", "rev7_loosestool",
                                           "rev7_abdompain", "rev7_convulsions",
                                           "rev7_fever")] == "") > 6, NA, FALSE)))
anysymp_varl <- c("adm_anysymp","rev1_anysymp","rev2_anysymp","rev3_anysymp",
               "rev5_anysymp","rev7_anysymp")
# Get prevalence of symptoms reported on last day of stay
lastrev <- apply(fws_an[,anysymp_varl], 1, function(x) {
  if(rev(which(!is.na(x)))[1]) {
    return(rev(!is.na(x))[1])
  }
})
table(lastrev)

###

# Frequency of malaria diagnosis changed from suspected to discharge diagnosis
paste0(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag, "Malaria")], "Malaria"))["FALSE"],
       "/",sum(str_detect(fws_an$suspected_diag, "Malaria")), " (",
       formattable::percent(as.character(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag, "Malaria")], "Malaria"))["FALSE"] /
                                           sum(str_detect(fws_an$suspected_diag, "Malaria"))), digits = 0),")")

# Proportion of children with positive malaria slides:
  # Among those with suspected malaria diagnosis
paste0(table(fws_an$adm_parasitemia[str_detect(fws_an$suspected_diag, "Malaria")] > 0)["TRUE"],"/",
       sum(str_detect(fws_an$suspected_diag, "Malaria"))," (",
       formattable::percent(as.character(table(fws_an$adm_parasitemia[str_detect(fws_an$suspected_diag, "Malaria")] > 0)["TRUE"] /
                                           sum(str_detect(fws_an$suspected_diag, "Malaria"))), digits = 0), ")")
  # Among those with malaria discharge diagnosis
paste0(table(fws_an$adm_parasitemia[str_detect(fws_an$discharge_diag, "Malaria")] > 0)["TRUE"],"/",
       sum(str_detect(fws_an$discharge_diag, "Malaria"))," (",
       formattable::percent(as.character(table(fws_an$adm_parasitemia[str_detect(fws_an$discharge_diag, "Malaria")] > 0)["TRUE"] /
                                           sum(str_detect(fws_an$discharge_diag, "Malaria"))), digits = 0), ")")


# Frequency of sepsis diagnosis changed from suspected to discharge diagnosis
paste0(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag, "Sepsis")], "Sepsis"))["FALSE"],
       "/",sum(str_detect(fws_an$suspected_diag, "Sepsis")), " (",
       formattable::percent(as.character(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag, "Sepsis")], "Sepsis"))["FALSE"] /
                                           sum(str_detect(fws_an$suspected_diag, "Sepsis"))), digits = 0),")")
# Proportion of children with positive bacteriology:
  # Among those with suspected sepsis diagnosis
paste0(table(fws_all_obs$sepsis_bact_result[str_detect(fws_all_obs$suspected_diag, "Sepsis") & fws_all_obs$include == 1 & fws_all_obs$sepsis_bact_result != ""] == "1 pos")["TRUE"],"/",
       sum(str_detect(fws_all_obs$suspected_diag, "Sepsis"))," (",
       formattable::percent(as.character(table(fws_all_obs$sepsis_bact_result[str_detect(fws_all_obs$suspected_diag, "Sepsis") & fws_all_obs$include == 1 & fws_all_obs$sepsis_bact_result != ""] == "1 pos")["TRUE"] /
                                           sum(str_detect(fws_all_obs$suspected_diag, "Sepsis"))), digits = 0), ")")
  # Among those with sepsis discharge diagnosis
paste0(table(fws_all_obs$sepsis_bact_result[str_detect(fws_all_obs$discharge_diag, "Sepsis") & fws_all_obs$include == 1 & fws_all_obs$sepsis_bact_result != ""] == "1 pos")["TRUE"],"/",
       sum(str_detect(fws_all_obs$discharge_diag, "Sepsis"))," (",
       formattable::percent(as.character(table(fws_all_obs$sepsis_bact_result[str_detect(fws_all_obs$discharge_diag, "Sepsis") & fws_all_obs$include == 1 & fws_all_obs$sepsis_bact_result != ""] == "1 pos")["TRUE"] /
                                           sum(str_detect(fws_all_obs$discharge_diag, "Sepsis"))), digits = 0), ")")

###

# Sensitivity and specificity of diagnoses
sens_spec1.n = as.data.frame(NULL)
for(i in c("Malaria","Sepsis","LRTI","Gastro","UTI","Meningitis","Other infection","Other non-infection")) {
  sens_spec1.n["sensitivity",i] <- paste0(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i))["TRUE"],"/",
                                          sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i))))
  sens_spec1.n["specificity",i] <- paste0(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i))["TRUE"],"/",
                                          sum(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i))))
}
sens_spec1.n

# Confidence intervals for sensitivity and specificity of diagnoses
sens_spec_ci = as.data.frame(NULL)
for(i in c("Malaria","Sepsis","LRTI","Gastro","UTI","Meningitis","Other infection","Other non-infection")) {
  p_hat <- table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i))["TRUE"]/
    sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i)))
  sens_spec_ci["sensitivity",i] <- paste0("(", round((p_hat-(qnorm(0.95)/2)*sqrt(p_hat*(1-p_hat)/sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i)))))*100, digits = 0),
                                          "%-", round((p_hat+(qnorm(0.95)/2)*sqrt(p_hat*(1-p_hat)/sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i)))))*100, digits = 0),
                                          "%)")
  p_hat1 <- table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i))["TRUE"]/
    sum(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i)))
  sens_spec_ci["specificity",i] <- paste0("(", round((p_hat1-(qnorm(0.95)/2)*sqrt(p_hat1*(1-p_hat)/sum(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i)))))*100, digits = 0),
                                          "%-", round((p_hat1+(qnorm(0.95)/2)*sqrt(p_hat1*(1-p_hat)/sum(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i)))))*100, digits = 0),
                                          "%)")
}
sens_spec_ci

###

# Prevalence of any symptoms on day 1 by diagnosis stability (changed vs. constant diagnosis)
table(fws_an$rev1_anysymp, fws_an$changed_dx)
margin.table(table(fws_an$rev1_anysymp[!is.na(fws_an$rev1_anysymp)], fws_an$changed_dx[!is.na(fws_an$rev1_anysymp)]),2)
table(fws_an$rev2_anysymp, fws_an$changed_dx)
margin.table(table(fws_an$rev2_anysymp[!is.na(fws_an$rev2_anysymp)], fws_an$changed_dx[!is.na(fws_an$rev2_anysymp)]),2)
table(fws_an$rev3_anysymp, fws_an$changed_dx)
margin.table(table(fws_an$rev3_anysymp[!is.na(fws_an$rev3_anysymp)], fws_an$changed_dx[!is.na(fws_an$rev3_anysymp)]),2)
table(fws_an$rev5_anysymp, fws_an$changed_dx)
margin.table(table(fws_an$rev5_anysymp[!is.na(fws_an$rev5_anysymp)], fws_an$changed_dx[!is.na(fws_an$rev5_anysymp)]),2)

# Length of stay by diagnosis stability
by(fws_an$length_stay_num, fws_an$changed_dx, summary)

###############################################################################
#### RESULTS: TREATMENT ####

# Prevalence of antibiotic and antimalarial treatment
table(fws_an$antibiotic_tf)
table(fws_an$antimalarial_tf)
table(fws_an$antibiotic_tf, fws_an$antimalarial_tf)

# Prevalence of treatment among children with malaria only
table(fws_an$antimalarial_tf[fws_an$discharge_diag == "Malaria" | fws_an$discharge_diag == "; Malaria"])
table(fws_an$antibiotic_tf[fws_an$discharge_diag == "Malaria" | fws_an$discharge_diag == "; Malaria"])

# Prevalence of treatment among children without malaria
table(fws_an$antimalarial_tf[!str_detect(fws_an$discharge_diag, "Malaria")])
table(fws_an$antibiotic_tf[!str_detect(fws_an$discharge_diag, "Malaria")])

# Prevalence of switched treatment among children receiving medication
table(str_detect(fws_an$tx_pattern[fws_an$tx_pattern != "No medication"], "Switched"))

# Concurrence of changed diagnosis and changed treatment among children receiving medication
prop.table(table(fws_an$changed_dx[fws_an$tx_pattern != "No medication"], str_detect(fws_an$tx_pattern[fws_an$tx_pattern != "No medication"], "Switched")),2)

# Prevalence changed diagnosis among children without malaria (among children receiving medication)
table(fws_an$antibiotic_tf[!str_detect(fws_an$discharge_diag, "Malaria") & fws_an$tx_pattern != "No medication"], 
      fws_an$changed_dx[!str_detect(fws_an$discharge_diag, "Malaria") & fws_an$tx_pattern != "No medication"])
table(fws_an$antimalarial_tf[!str_detect(fws_an$discharge_diag, "Malaria") & fws_an$tx_pattern != "No medication"], 
      fws_an$changed_dx[!str_detect(fws_an$discharge_diag, "Malaria") & fws_an$tx_pattern != "No medication"])


##############################################################

#### TABLES AND FIGURES ####

##############################################################

#### TABLE 1 ####

summary1 =as.data.frame(NULL)
for(j in c("total","< 1 year", "1 year", "2-4 years", "≥ 5 years")) {  
  summary1[j,"n"] <- paste0(sum(fws_all_obs$age_cat == j),"/", sum(nrow(fws_all_obs)), " (", 
                            formattable::percent(as.character(sum(fws_all_obs$age_cat == j)/sum(table(fws_all_obs$age_cat))), digits = 1), ")")
  summary1[j,"ageinyears"] <- paste0(median(fws_all_obs$ageinyears[fws_all_obs$age_cat == j], na.rm = TRUE), " (",
                                     quantile(fws_all_obs$ageinyears[fws_all_obs$age_cat == j], .25, na.rm = TRUE),"-",
                                     quantile(fws_all_obs$ageinyears[fws_all_obs$age_cat == j], .75, na.rm = TRUE),")")
  summary1[j, "sex_male"] <- paste0(sum(fws_all_obs$sex[fws_all_obs$age_cat == j] == "Male"),"/",sum(!is.na(fws_all_obs$sex[fws_all_obs$age_cat == j])), " (",
                                    formattable::percent(as.character(sum(fws_all_obs$sex[fws_all_obs$age_cat == j] == "Male")/
                                                                        sum(!is.na(fws_all_obs$sex[fws_all_obs$age_cat == j]))), digits = 1), ")")
  summary1[j,"length_stay"] <- paste0(round(median(fws_all_obs$length_stay_num[fws_all_obs$age_cat == j], na.rm = TRUE), digits = 1), " (",
                                      round(quantile(fws_all_obs$length_stay_num[fws_all_obs$age_cat == j], .25, na.rm = TRUE), digits = 1),"-",
                                      round(quantile(fws_all_obs$length_stay_num[fws_all_obs$age_cat == j], .75, na.rm = TRUE), digits = 1),")")
  summary1[j,"temperature"] <- paste0(round(median(fws_all_obs$adm_temperature[fws_all_obs$age_cat == j], na.rm = TRUE), digits = 1), " (",
                                      round(quantile(fws_all_obs$adm_temperature[fws_all_obs$age_cat == j], .25, na.rm = TRUE), digits = 1),"-",
                                      round(quantile(fws_all_obs$adm_temperature[fws_all_obs$age_cat == j], .75, na.rm = TRUE), digits = 1),")")
  summary1[j, "para_prev"] <- paste0(sum(fws_all_obs$adm_parasitemia[fws_all_obs$age_cat == j] > 0),"/",sum(!is.na(fws_all_obs$adm_parasitemia[fws_all_obs$age_cat == j])), " (",
                                     formattable::percent(as.character(sum(fws_all_obs$adm_parasitemia[fws_all_obs$age_cat == j]> 0)/
                                                                         sum(!is.na(fws_all_obs$adm_parasitemia[fws_all_obs$age_cat == j]))), digits = 1), ")")
  summary1[j,"blood_cult"] <- paste0(sum(fws_all_obs$sepsis_bact_result[fws_all_obs$age_cat == j] == "1 pos"),"/",sum(fws_all_obs$sepsis_bact_result[fws_all_obs$age_cat == j] == "1 pos" | 
                                                                                                             fws_all_obs$sepsis_bact_result[fws_all_obs$age_cat == j] == "0 neg"), " (",
                                     formattable::percent(as.character(sum(fws_all_obs$sepsis_bact_result[fws_all_obs$age_cat == j] == "1 pos")/
                                                                         sum(fws_all_obs$sepsis_bact_result[fws_all_obs$age_cat == j] == "1 pos" | 
                                                                               fws_all_obs$sepsis_bact_result[fws_all_obs$age_cat == j] == "0 neg")), digits = 1), ")")
  summary1[j,"antipyretic_prior"] <- paste0(sum(fws_all_obs$prior_antipyretic[fws_all_obs$age_cat == j] == "Yes"),"/",sum(fws_all_obs$prior_antipyretic[fws_all_obs$age_cat == j] == "Yes" | 
                                                                                                                            fws_all_obs$prior_antipyretic[fws_all_obs$age_cat == j] == "No"), " (",
                                            formattable::percent(as.character(sum(fws_all_obs$prior_antipyretic[fws_all_obs$age_cat == j] == "Yes")/
                                                                                sum(fws_all_obs$prior_antipyretic[fws_all_obs$age_cat == j] == "Yes" | 
                                                                                      fws_all_obs$prior_antipyretic[fws_all_obs$age_cat == j] == "No")), digits = 1), ")")
  summary1[j,"antibiotic_prior"] <- paste0(sum(fws_all_obs$prior_antibiotic[fws_all_obs$age_cat == j] == "Yes"), "/",sum(fws_all_obs$prior_antibiotic[fws_all_obs$age_cat == j] == "Yes" | 
                                                                                                                           fws_all_obs$prior_antibiotic[fws_all_obs$age_cat == j] == "No")," (",
                                           formattable::percent(as.character(sum(fws_all_obs$prior_antibiotic[fws_all_obs$age_cat == j] == "Yes")/
                                                                               sum(fws_all_obs$prior_antibiotic[fws_all_obs$age_cat == j] == "Yes" | 
                                                                                     fws_all_obs$prior_antibiotic[fws_all_obs$age_cat == j] == "No")), digits = 1), ")")
  summary1[j,"antimalarial_prior"] <- paste0(sum(fws_all_obs$prior_antimalarial[fws_all_obs$age_cat == j] == "Yes"),"/",sum(fws_all_obs$prior_antimalarial[fws_all_obs$age_cat == j] == "Yes" | 
                                                                                                                        fws_all_obs$prior_antimalarial[fws_all_obs$age_cat == j] == "No"), " (",
                                             formattable::percent(as.character(sum(fws_all_obs$prior_antimalarial[fws_all_obs$age_cat == j] == "Yes")/
                                                                                 sum(fws_all_obs$prior_antimalarial[fws_all_obs$age_cat == j] == "Yes" | 
                                                                                       fws_all_obs$prior_antimalarial[fws_all_obs$age_cat == j] == "No")), digits = 1), ")")
  summary1[j,"susp_mal"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Malaria")), "/", sum(fws_all_obs$age_cat == j), "(",
                                    formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Malaria")/
                                                                            sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"susp_sepsis"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Sepsis")), "/", sum(fws_all_obs$age_cat == j), " (",
                                       formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Sepsis")/
                                                                               sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"susp_lrti"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "LRTI")), "/", sum(fws_all_obs$age_cat == j), " (",
                                     formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "LRTI")/
                                                                             sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"susp_gastro"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Gastroenteritis")), "/", sum(fws_all_obs$age_cat == j), " (",
                                       formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Gastroenteritis")/
                                                                               sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"susp_UTI"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "UTI")), "/", sum(fws_all_obs$age_cat == j), " (",
                                    formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "UTI")/
                                                                            sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"susp_mening"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Meningitis")), "/", sum(fws_all_obs$age_cat == j), " (",
                                       formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Meningitis")/
                                                                               sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"susp_other_inf"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Other infection")), "/", sum(fws_all_obs$age_cat == j), " (",
                                          formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Other infection")/
                                                                                  sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"susp_other_noninf"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Other non-infection")), "/", sum(fws_all_obs$age_cat == j), " (",
                                             formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag[fws_all_obs$age_cat == j], "Other non-infection")/
                                                                                     sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_mal"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Malaria")), "/", sum(fws_all_obs$age_cat == j), " (",
                                         formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Malaria")/
                                                                                 sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_sepsis"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Sepsis")), "/", sum(fws_all_obs$age_cat == j), " (",
                                            formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Sepsis")/
                                                                                    sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_lrti"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "LRTI")), "/", sum(fws_all_obs$age_cat == j), " (",
                                          formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "LRTI")/
                                                                                  sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_gastro"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Gastroenteritis")), "/", sum(fws_all_obs$age_cat == j), " (",
                                            formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Gastroenteritis")/
                                                                                    sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_UTI"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "UTI")), "/", sum(fws_all_obs$age_cat == j), " (",
                                         formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "UTI")/
                                                                                 sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_mening"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Meningitis")), "/", sum(fws_all_obs$age_cat == j), " (",
                                            formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Meningitis")/
                                                                                    sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_other_inf"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Other infection")), "/", sum(fws_all_obs$age_cat == j), " (",
                                               formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Other infection")/
                                                                                       sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  summary1[j,"discharge_other_noninf"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Other non-infection")), "/", sum(fws_all_obs$age_cat == j), " (",
                                                  formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$age_cat == j], "Other non-infection")/
                                                                                          sum(fws_all_obs$age_cat == j))), digits = 1), ")")
  
  
  # Totals
  summary1["total","n"] <- paste0(nrow(fws_all_obs), " (", formattable::percent(as.character(nrow(fws_all_obs)/nrow(fws_all_obs)), digits = 1), ")")
  summary1["total","ageinyears"] <- paste0(median(fws_all_obs$ageinyears, na.rm = TRUE), " (",
                                           quantile(fws_all_obs$ageinyears, .25, na.rm = TRUE),"-",
                                           quantile(fws_all_obs$ageinyears, .75, na.rm = TRUE),")")
  summary1["total", "sex_male"] <- paste0(sum(fws_all_obs$sex == "Male"),"/",sum(!is.na(fws_all_obs$sex)), " (",
                                          formattable::percent(as.character(sum(fws_all_obs$sex == "Male")/
                                                                              sum(!is.na(fws_all_obs$sex))), digits = 1), ")")
  summary1["total","length_stay"] <- paste0(round(median(fws_all_obs$length_stay_num, na.rm = TRUE), digits = 1), " (",
                                            round(quantile(fws_all_obs$length_stay_num, .25, na.rm = TRUE), digits = 1),"-",
                                            round(quantile(fws_all_obs$length_stay_num, .75, na.rm = TRUE), digits = 1),")")
  summary1["total","temperature"] <- paste0(round(median(fws_all_obs$adm_temperature, na.rm = TRUE), digits = 1), " (",
                                            round(quantile(fws_all_obs$adm_temperature, .25, na.rm = TRUE), digits = 1),"-",
                                            round(quantile(fws_all_obs$adm_temperature, .75, na.rm = TRUE), digits = 1),")")
  summary1["total", "para_prev"] <- paste0(sum(fws_all_obs$adm_parasitemia > 0),"/",sum(!is.na(fws_all_obs$adm_parasitemia)), " (",
                                           formattable::percent(as.character(sum(fws_all_obs$adm_parasitemia> 0)/
                                                                               sum(!is.na(fws_all_obs$adm_parasitemia))), digits = 1), ")")
  summary1["total","blood_cult"] <- paste0(sum(fws_all_obs$sepsis_bact_result == "1 pos"),"/",sum(fws_all_obs$sepsis_bact_result == "1 pos" | 
                                                                                            fws_all_obs$sepsis_bact_result == "0 neg"), " (",
                                           formattable::percent(as.character(sum(fws_all_obs$sepsis_bact_result == "1 pos")/
                                                                               sum(fws_all_obs$sepsis_bact_result == "1 pos" | 
                                                                                     fws_all_obs$sepsis_bact_result == "0 neg")), digits = 1), ")")
  summary1["total","antipyretic_prior"] <- paste0(sum(fws_all_obs$prior_antipyretic == "Yes"),"/",sum(fws_all_obs$prior_antipyretic == "Yes" | 
                                                                                                           fws_all_obs$prior_antipyretic == "No"), " (",
                                                  formattable::percent(as.character(sum(fws_all_obs$prior_antipyretic == "Yes")/
                                                                                      sum(fws_all_obs$prior_antipyretic == "Yes" | 
                                                                                            fws_all_obs$prior_antipyretic == "No")), digits = 1), ")")
  summary1["total","antibiotic_prior"] <- paste0(sum(fws_all_obs$prior_antibiotic == "Yes"),"/",sum(fws_all_obs$prior_antibiotic == "Yes" | 
                                                                                                         fws_all_obs$prior_antibiotic == "No"), " (",
                                                 formattable::percent(as.character(sum(fws_all_obs$prior_antibiotic == "Yes")/
                                                                                     sum(fws_all_obs$prior_antibiotic == "Yes" | 
                                                                                           fws_all_obs$prior_antibiotic == "No")), digits = 1), ")")
  summary1["total","antimalarial_prior"] <- paste0(sum(fws_all_obs$prior_antimalarial == "Yes"),"/",sum(fws_all_obs$prior_antimalarial == "Yes" | 
                                                                                                       fws_all_obs$prior_antimalarial == "No"), " (",
                                                   formattable::percent(as.character(sum(fws_all_obs$prior_antimalarial == "Yes")/
                                                                                       sum(fws_all_obs$prior_antimalarial == "Yes" | 
                                                                                             fws_all_obs$prior_antimalarial == "No")), digits = 1), ")")
  summary1["total","susp_mal"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "Malaria")), "/", nrow(fws_all_obs), " (",
                                          formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "Malaria")/
                                                                                  nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","susp_sepsis"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "Sepsis")), "/", nrow(fws_all_obs), " (",
                                             formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "Sepsis")/
                                                                                     nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","susp_lrti"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "LRTI")), "/", nrow(fws_all_obs), " (",
                                           formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "LRTI")/
                                                                                   nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","susp_gastro"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "Gastroenteritis")), "/", nrow(fws_all_obs), " (",
                                             formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "Gastroenteritis")/
                                                                                     nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","susp_UTI"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "UTI")), "/", nrow(fws_all_obs), " (",
                                          formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "UTI")/
                                                                                  nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","susp_mening"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "Meningitis")), "/", nrow(fws_all_obs), " (",
                                             formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "Meningitis")/
                                                                                     nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","susp_other_inf"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "Other infection")), "/", nrow(fws_all_obs), " (",
                                                formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "Other infection")/
                                                                                        nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","susp_other_noninf"] <-  paste0(sum(str_detect(fws_all_obs$suspected_diag, "Other non-infection")), "/", nrow(fws_all_obs), " (",
                                                   formattable::percent(as.character(sum(str_detect(fws_all_obs$suspected_diag, "Other non-infection")/
                                                                                           nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_mal"] <-  paste0(sum(str_detect(fws_all_obs$discharge, "Malaria")), "/", nrow(fws_all_obs), " (",
                                               formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge, "Malaria")/
                                                                                       nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_sepsis"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag, "Sepsis")), "/", nrow(fws_all_obs), " (",
                                                  formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag, "Sepsis")/
                                                                                          nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_lrti"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag, "LRTI")), "/", nrow(fws_all_obs), " (",
                                                formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag, "LRTI")/
                                                                                        nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_gastro"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag, "Gastroenteritis")), "/", nrow(fws_all_obs), " (",
                                                  formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag, "Gastroenteritis")/
                                                                                          nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_UTI"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag, "UTI")), "/", nrow(fws_all_obs), " (",
                                               formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag, "UTI")/
                                                                                       nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_mening"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag, "Meningitis")), "/", nrow(fws_all_obs), " (",
                                                  formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag, "Meningitis")/
                                                                                          nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_other_inf"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag, "Other infection")), "/", nrow(fws_all_obs), " (",
                                                     formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag, "Other infection")/
                                                                                             nrow(fws_all_obs))), digits = 1), ")")
  summary1["total","discharge_other_noninf"] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag, "Other non-infection")), "/", nrow(fws_all_obs), " (",
                                                        formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag, "Other non-infection")/
                                                                                                nrow(fws_all_obs))), digits = 1), ")")
  
}
summary1 <- as.data.frame(t(summary1))
rownames(summary1) <- c("Number of children","Age (years)", "Sex (male)", "Length of stay (days)",
                        "Temperature (C)",
                        "Parasites present in blood","Positive blood culture",
                        "Antipyretic use in 24h before admission","Antibiotic use in 24h prior to admission",
                        "Antimalarial use in 24h prior to admission",
                        "Suspect: Malaria","Suspect: Sepsis","Suspect: LRTI",
                        "Suspect: Gastrointestinal infection","Suspect: UTI",
                        "Suspect: Meningitis","Suspect: Other infection",
                        "Suspect: Other non-infection","Discharge: Malaria",
                        "Discharge: Sepsis",
                        "Discharge: LRTI","Discharge: Gastrointestinal infection",
                        "Discharge: UTI","Discharge: Meningitis",
                        "Discharge: Other infection","Discharge: Other non-infection")
summary1
###############################################################################
###############################################################################

#### SUPPLEMENTARY TABLE 1 ####

supp1 =as.data.frame(NULL)
for(j in c("0","1")) {  
  supp1["n", j] <- paste0(sum(fws_all_obs$include == j, na.rm = TRUE),"/", nrow(fws_all_obs)," (",
                          formattable::percent(as.character(sum(fws_all_obs$include == j, na.rm = TRUE) / sum(table(fws_all_obs$include))), digits = 1),")")
  supp1["ageinyears", j] <- paste0(median(fws_all_obs$ageinyears[fws_all_obs$include == j], na.rm = TRUE), " (",
                                   quantile(fws_all_obs$ageinyears[fws_all_obs$include == j], .25, na.rm = TRUE),"-",
                                   quantile(fws_all_obs$ageinyears[fws_all_obs$include == j], .75, na.rm = TRUE),")")
  supp1[ "sex_male", j] <- paste0(sum(fws_all_obs$sex[fws_all_obs$include == j] == "Male", na.rm = TRUE),"/", 
                                  sum(!is.na(fws_all_obs$sex[fws_all_obs$include == j]), na.rm = TRUE), " (",
                                  formattable::percent(as.character(sum(fws_all_obs$sex[fws_all_obs$include == j] == "Male", na.rm = TRUE)/
                                                                      sum(!is.na(fws_all_obs$sex[fws_all_obs$include == j]), na.rm = TRUE)), digits = 1), ")")
  supp1["length_stay", j] <- paste0(median(fws_all_obs$length_stay_num[fws_all_obs$include == j], na.rm = TRUE), " (",
                                    quantile(fws_all_obs$length_stay_num[fws_all_obs$include == j], .25, na.rm = TRUE),"-",
                                    quantile(fws_all_obs$length_stay_num[fws_all_obs$include == j], .75, na.rm = TRUE),")")
  supp1["antipyretic_prior", j] <- paste0(sum(fws_all_obs$prior_antipyretic[fws_all_obs$include == j] == "Yes", na.rm = TRUE), "/",
                                          sum(fws_all_obs$prior_antipyretic[fws_all_obs$include == j] == "Yes" | 
                                                fws_all_obs$prior_antipyretic[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                          formattable::percent(as.character(sum(fws_all_obs$prior_antipyretic[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                              sum(fws_all_obs$prior_antipyretic[fws_all_obs$include == j] == "Yes" | 
                                                                                    fws_all_obs$prior_antipyretic[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["antibiotic_prior", j] <- paste0(sum(fws_all_obs$prior_antibiotic[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                         sum(fws_all_obs$prior_antibiotic[fws_all_obs$include == j] == "Yes" | 
                                               fws_all_obs$prior_antibiotic[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                         formattable::percent(as.character(sum(fws_all_obs$prior_antibiotic[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                             sum(fws_all_obs$prior_antibiotic[fws_all_obs$include == j] == "Yes" | 
                                                                                   fws_all_obs$prior_antibiotic[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["antimalarial_prior", j] <- paste0(sum(fws_all_obs$prior_antimalarial[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                           sum(fws_all_obs$prior_antimalarial[fws_all_obs$include == j] == "Yes" | 
                                                 fws_all_obs$prior_antimalarial[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                           formattable::percent(as.character(sum(fws_all_obs$prior_antimalarial[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                               sum(fws_all_obs$prior_antimalarial[fws_all_obs$include == j] == "Yes" | 
                                                                                     fws_all_obs$prior_antimalarial[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["sympt_cough", j] <-  paste0(sum(fws_all_obs$adm_cough[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                     sum(fws_all_obs$adm_cough[fws_all_obs$include == j] == "Yes" | 
                                           fws_all_obs$adm_cough[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                     formattable::percent(as.character(sum(fws_all_obs$adm_cough[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                         sum(fws_all_obs$adm_cough[fws_all_obs$include == j] == "Yes" | 
                                                                               fws_all_obs$adm_cough[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["sympt_breathe", j] <-  paste0(sum(fws_all_obs$adm_breathingdiff[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                       sum(fws_all_obs$adm_breathingdiff[fws_all_obs$include == j] == "Yes" | 
                                             fws_all_obs$adm_breathingdiff[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                       formattable::percent(as.character(sum(fws_all_obs$adm_breathingdiff[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                           sum(fws_all_obs$adm_breathingdiff[fws_all_obs$include == j] == "Yes" | 
                                                                                 fws_all_obs$adm_breathingdiff[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["sympt_vomit", j] <-  paste0(sum(fws_all_obs$adm_vomiting[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                     sum(fws_all_obs$adm_vomiting[fws_all_obs$include == j] == "Yes" | 
                                           fws_all_obs$adm_vomiting[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                     formattable::percent(as.character(sum(fws_all_obs$adm_vomiting[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                         sum(fws_all_obs$adm_vomiting[fws_all_obs$include == j] == "Yes" | 
                                                                               fws_all_obs$adm_vomiting[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["sympt_stool", j] <-  paste0(sum(fws_all_obs$adm_loosestool[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                     sum(fws_all_obs$adm_loosestool[fws_all_obs$include == j] == "Yes" | 
                                           fws_all_obs$adm_loosestool[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                     formattable::percent(as.character(sum(fws_all_obs$adm_loosestool[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                         sum(fws_all_obs$adm_loosestool[fws_all_obs$include == j] == "Yes" | 
                                                                               fws_all_obs$adm_loosestool[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["sympt_abdom", j] <-  paste0(sum(fws_all_obs$adm_abdompain[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                     sum(fws_all_obs$adm_abdompain[fws_all_obs$include == j] == "Yes" | 
                                           fws_all_obs$adm_abdompain[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                     formattable::percent(as.character(sum(fws_all_obs$adm_abdompain[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                         sum(fws_all_obs$adm_abdompain[fws_all_obs$include == j] == "Yes" | 
                                                                               fws_all_obs$adm_abdompain[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  supp1["sympt_convuls", j] <-  paste0(sum(fws_all_obs$adm_convulsions[fws_all_obs$include == j] == "Yes", na.rm = TRUE),"/",
                                       sum(fws_all_obs$adm_convulsions[fws_all_obs$include == j] == "Yes" | 
                                             fws_all_obs$adm_convulsions[fws_all_obs$include == j] == "No", na.rm = TRUE), " (",
                                       formattable::percent(as.character(sum(fws_all_obs$adm_convulsions[fws_all_obs$include == j] == "Yes", na.rm = TRUE)/
                                                                           sum(fws_all_obs$adm_convulsions[fws_all_obs$include == j] == "Yes" | 
                                                                                 fws_all_obs$adm_convulsions[fws_all_obs$include == j] == "No", na.rm = TRUE)), digits = 1), ")")
  
  supp1["discharge_mal", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Malaria"), na.rm = TRUE),"/",
                                       sum(fws_all_obs$include == j, na.rm = TRUE), " (",
                                       formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Malaria")/
                                                                               sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
  supp1["discharge_sepsis", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Sepsis"), na.rm = TRUE),"/",
                                          sum(fws_all_obs$include == j, na.rm = TRUE)," (",
                                          formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Sepsis")/
                                                                                  sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
  supp1["discharge_lrti", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "LRTI"), na.rm = TRUE),"/",
                                        sum(fws_all_obs$include == j, na.rm = TRUE), " (",
                                        formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "LRTI")/
                                                                                sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
  supp1["discharge_gastro", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Gastroenteritis"), na.rm = TRUE),"/",
                                          sum(fws_all_obs$include == j, na.rm = TRUE)," (",
                                          formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Gastroenteritis")/
                                                                                  sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
  supp1["discharge_UTI", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "UTI"), na.rm = TRUE),"/",
                                       sum(fws_all_obs$include == j, na.rm = TRUE)," (",
                                       formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "UTI")/
                                                                               sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
  supp1["discharge_mening", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Meningitis"), na.rm = TRUE),"/",
                                          sum(fws_all_obs$include == j, na.rm = TRUE)," (",
                                          formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Meningitis")/
                                                                                  sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
  supp1["discharge_other_inf", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Other infection"), na.rm = TRUE), "/",
                                             sum(fws_all_obs$include == j, na.rm = TRUE)," (",
                                             formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Other infection")/
                                                                                     sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
  supp1["discharge_other_noninf", j] <-  paste0(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Other non-infection"), na.rm = TRUE),"/",
                                                sum(fws_all_obs$include == j, na.rm = TRUE)," (",
                                                formattable::percent(as.character(sum(str_detect(fws_all_obs$discharge_diag[fws_all_obs$include == j], "Other non-infection")/
                                                                                        sum(fws_all_obs$include == j, na.rm = TRUE), na.rm = TRUE)), digits = 1), ")")
}
colnames(supp1) <- c("missing_exclude","missing_include")
rownames(supp1) <- c("Total cases","Age (years)", "Male sex", "Length of stay (days)",
                     "Antipyretic use in 24h before admission","Antibiotic use in 24h prior to admission",
                     "Antimalarial use in 24h prior to admission",
                     "Cough reported at admission","Difficulty breathing reported at admission",
                     "Vomiting reported at admission","Loose stool reported at admission","Abdominal pain reported at admission","Convulsions reported at admission",
                     "Discharge diagnosis: Malaria","Discharge diagnosis: Sepsis","Discharge diagnosis: LRTI","Discharge diagnosis: Gastrointestinal infection",
                     "Discharge diagnosis: UTI","Discharge diagnosis: Meningitis","Discharge diagnosis: Other infection","Discharge diagnosis: Other non-infection")
supp1
###############################################################################
###############################################################################

#### FIGURE 1: LENGTH OF STAY BY DIAGNOSIS ####

diag_rep <- fws_all_obs %>%
  mutate(malaria = ifelse(str_detect(discharge_diag,"Malaria"), 1,0),
         sepsis = ifelse(str_detect(discharge_diag,"Sepsis"), 1,0),
         gastro = ifelse(str_detect(discharge_diag,"Gastroenteritis"), 1,0),
         mening = ifelse(str_detect(discharge_diag,"Meningitis"), 1,0),
         uti = ifelse(str_detect(discharge_diag,"UTI"), 1,0),
         lrti = ifelse(str_detect(discharge_diag,"LRTI"), 1,0),
         other_inf = ifelse(str_detect(discharge_diag,"Other infection"), 1,0),
         other_noninf = ifelse(str_detect(discharge_diag,"Other non-infection"), 1,0)) %>%
  select(length_stay_num, malaria, sepsis,
         gastro, mening, uti, lrti, other_inf, other_noninf)
diag_rep_long <- gather(diag_rep, diagnosis, tf, 
                        c(malaria, sepsis, gastro, mening, uti, lrti, other_inf, other_noninf))
diag_rep_box <- diag_rep_long %>%
  filter(tf == 1)

################### FIGURE 1. 
par(mar = c(4,13,3,1),mgp = c(2.5,1,0))
plot(length_stay_num ~ with(diag_rep_box, reorder(diagnosis, -length_stay_num, median)), 
     data = diag_rep_box, horizontal = TRUE, ylim = c(0,20),
     las = 1, yaxt = "n", main = "Length of stay by discharge diagnosis",
     xlab = "", ylab = "Length of stay (days)")
mtext("Discharge diagnosis", side = 2, line = 12)
axis(2, at = 1:8, las = 2,
     labels = rev(c(paste0("Malaria (N = ",sum(diag_rep_box$diagnosis == "malaria"), ")"),
                    paste0("Gastroenteritis (N = ",sum(diag_rep_box$diagnosis == "gastro"), ")"),
                    paste0("UTI (N = ",sum(diag_rep_box$diagnosis == "uti"), ")"),
                    paste0("Other non-infection (N = ",sum(diag_rep_box$diagnosis == "other_noninf"), ")"),
                    paste0("Other infection (N = ",sum(diag_rep_box$diagnosis == "other_inf"), ")"),
                    paste0("LRTI (N = ",sum(diag_rep_box$diagnosis == "lrti"), ")"),
                    paste0("Sepsis (N = ",sum(diag_rep_box$diagnosis == "sepsis"), ")"),
                    paste0("Meningitis (N = ",sum(diag_rep_box$diagnosis == "mening"), ")"))))

###############################################################################
###############################################################################

##### FIGURE 2: PREVALENCE OF SYMPTOMS OVER TIME ####

p.rev <- as.data.frame(cbind(n_fever = sum(fws_an$adm_temperature > 37.99, na.rm = TRUE)/sum(fws_an$adm_temperature > 37.99),
                             t(sapply(c("adm_cough","adm_breathingdiff","adm_vomiting",
                                        "adm_loosestool","adm_abdompain", "adm_convulsions"),
                                      function(x) sum(fws_an[,x] == "Yes")/sum(fws_an[,x] == "Yes" | fws_an[,x] == "No")))))
colnames(p.rev) <- c("pct_fever","pct_cough","pct_breath","pct_vom","pct_stool","pct_abdom","pct_convuls")
p.rev1 =as.data.frame(NULL)
for(j in c("rev1","rev2","rev3","rev5","rev7")) {  
  p.rev1[j,"pct_fever"] <- sum(fws_an[,paste0(j,"_temperature")] > 37.99, na.rm = TRUE)/sum(!is.na(fws_an[,paste0(j,"_temperature")]))
  p.rev1[j,"pct_cough"] <- sum(fws_an[,paste0(j,"_cough")] == "Yes")/sum(fws_an[,paste0(j,"_cough")] == "Yes" | fws_an[,paste0(j,"_cough")] == "No")
  p.rev1[j,"pct_breath"] <- sum(fws_an[,paste0(j,"_breathingdiff")] == "Yes")/sum(fws_an[,paste0(j,"_breathingdiff")] == "Yes" | fws_an[,paste0(j,"_breathingdiff")] == "No")
  p.rev1[j,"pct_vom"] <- sum(fws_an[,paste0(j,"_vomiting")] == "Yes")/sum(fws_an[,paste0(j,"_vomiting")] == "Yes" | fws_an[,paste0(j,"_vomiting")] == "No")
  p.rev1[j,"pct_stool"] <- sum(fws_an[,paste0(j,"_loosestool")] == "Yes")/sum(fws_an[,paste0(j,"_loosestool")] == "Yes" | fws_an[,paste0(j,"_loosestool")] == "No")
  p.rev1[j,"pct_abdom"] <- sum(fws_an[,paste0(j,"_abdompain")] == "Yes")/sum(fws_an[,paste0(j,"_abdompain")] == "Yes" | fws_an[,paste0(j,"_abdompain")] == "No")
  p.rev1[j,"pct_convuls"] <- sum(fws_an[,paste0(j,"_convulsions")] == "Yes")/sum(fws_an[,paste0(j,"_convulsions")] == "Yes" | fws_an[,paste0(j,"_convulsions")] == "No")
}
pct <- rbind(adm = p.rev, p.rev1)
pct <- cbind("time" = c(0:3,5,7), pct[,1:7])

################### FIGURE 2. 
col_set <- c("#1f77b4", "#ff7f0e", "#2ca02c", "#d62728","#9467bd","#8c564b", "#e377c2")
par(mar=c(4,4,2,0.5), mgp = c(2.2,3.5,0))
matplot(pct$time, pct[,2:8], type ="l", col = col_set, lty = 1, lwd = 2,
        main = "Prevalence of symptoms over time",
        ylab = "",xaxt = "n", yaxt = "n", las = 2, xlab = "Review visit")
title(ylab = "Percent of all children in hospital at review", line = 3)
axis(1, at = c(0:3,5,7), labels = c("Admission", "Day 1","Day 2", "Day 3","Day 5","Day 7"), mgp = c(0,0.5,0))
axis(2, at = seq(0,1,0.2), labels = c("0%","20%","40%","60%","80%","100%"), las = 2, mgp = c(0,0.7,0))
legend("topright",c("Fever","Cough","Difficulty breathing","Vomiting",
                    "Loose stool","Abdominal pain","Convulsions"),
       title = "Symptoms reported", col=col_set, lty = c(1), cex = 0.7)

###########################################################################
###########################################################################

#### FIGURE 3: DIAGNOSIS SANKEY ####

sank1 <- rbind("Malaria" = cbind(table(str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis")
                                       & str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"],
                                 table(str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis")
                                       & str_detect(fws_an$discharge_diag, "Sepsis") & !str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                                 table(str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis")
                                       & str_detect(fws_an$discharge_diag, "Sepsis") & str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                                 table(str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis") &
                                         !str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"]),
               
               "Sepsis" = cbind(table(str_detect(fws_an$suspected_diag, "Sepsis") & !str_detect(fws_an$suspected_diag, "Malaria")
                                      & str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"],
                                table(str_detect(fws_an$suspected_diag, "Sepsis") & !str_detect(fws_an$suspected_diag, "Malaria")
                                      & str_detect(fws_an$discharge_diag, "Sepsis") & !str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                                table(str_detect(fws_an$suspected_diag, "Sepsis") & !str_detect(fws_an$suspected_diag, "Malaria")
                                      & str_detect(fws_an$discharge_diag, "Sepsis") & str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                                table(str_detect(fws_an$suspected_diag, "Sepsis") & !str_detect(fws_an$suspected_diag, "Malaria") &
                                        !str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"]),
               
               "Malaria/Sepsis" = cbind(table(str_detect(fws_an$suspected_diag, "Malaria") & str_detect(fws_an$suspected_diag, "Sepsis")
                                              & str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"],
                                        table(str_detect(fws_an$suspected_diag, "Malaria") & str_detect(fws_an$suspected_diag, "Sepsis")
                                              & str_detect(fws_an$discharge_diag, "Sepsis") & !str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                                        table(str_detect(fws_an$suspected_diag, "Malaria") & str_detect(fws_an$suspected_diag, "Sepsis")
                                              & str_detect(fws_an$discharge_diag, "Sepsis") & str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                                        table(str_detect(fws_an$suspected_diag, "Malaria") & str_detect(fws_an$suspected_diag, "Sepsis") &
                                                !str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"]),
               
               "Other" = cbind(table(!str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis")
                                     & str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"],
                               table(!str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis")
                                     & str_detect(fws_an$discharge_diag, "Sepsis") & !str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                               table(!str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis")
                                     & str_detect(fws_an$discharge_diag, "Sepsis") & str_detect(fws_an$discharge_diag, "Malaria"))["TRUE"],
                               table(!str_detect(fws_an$suspected_diag, "Malaria") & !str_detect(fws_an$suspected_diag, "Sepsis") &
                                       !str_detect(fws_an$discharge_diag, "Malaria") & !str_detect(fws_an$discharge_diag, "Sepsis"))["TRUE"]))
links_sd1 <- data.frame(
  source=c("Malaria ","Malaria ","Malaria ","Malaria ",
           "Sepsis ","Sepsis ","Sepsis ","Sepsis ",
           "Malaria/Sepsis ", "Malaria/Sepsis ", "Malaria/Sepsis ", "Malaria/Sepsis ", 
           "Other ","Other ","Other ","Other "),
  target=c("Malaria", "Sepsis","Malaria/Sepsis","Other",
           "Malaria", "Sepsis","Malaria/Sepsis","Other",
           "Malaria", "Sepsis","Malaria/Sepsis","Other",
           "Malaria", "Sepsis","Malaria/Sepsis","Other"),
  value=c(sank1[1,],sank1[2,],sank1[3,],sank1[4,]))

nodes_sd1 <- data.frame(name=c(as.character(links_sd1$source), 
         as.character(links_sd1$target)) %>% unique()
)
links_sd1$IDsource <- match(links_sd1$source, nodes_sd1$name)-1 
links_sd1$IDtarget <- match(links_sd1$target, nodes_sd1$name)-1
sd_sankey1 <- sankeyNetwork(Links = links_sd1, Nodes = nodes_sd1,
                            Source = "IDsource", Target = "IDtarget",
                            Value = "value", NodeID = "name", 
                            sinksRight = TRUE, iterations = 0,
                            width = 600, height = 450, fontSize = 12,
                            colourScale = 'd3.scaleOrdinal(["#D53E4F", 
                           "#FEE08B", "#99D594", "#3288BD"])')

# Add labels to the nodes (suspected and discharge diagnoses)
sd_sankey1 <- htmlwidgets::onRender(sd_sankey1, '
  function(el) { 
    var cols_x = this.sankey.nodes().map(d => d.x).filter((v, i, a) => a.indexOf(v) === i).sort(function(a, b){return a - b});
    var labels = ["Suspected diagnosis", "Discharge diagnosis"];
    cols_x.forEach((d, i) => {
      d3.select(el).select("svg")
        .append("text")
        .attr("x",d*.865)
        .attr("y", 12)
        .text(labels[i])
        .style("font-size", "16");
    })
  }
')
################### FIGURE 3. 
sd_sankey1

###############################################################################
###############################################################################

#### FIGURE 4: DIAGNOSIS SENSITIVITY/SPECIFICITY ####

sens_spec1 = as.data.frame(NULL)
for(i in c("Malaria","Sepsis","LRTI","Gastro","UTI","Meningitis","Other infection","Other non-infection")) {
  sens_spec1["sensitivity",i] <- prop.table(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i)))["TRUE"]
  sens_spec1["specificity",i] <- prop.table(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i)))["TRUE"]
}

sens_spec_ci = as.data.frame(NULL)
for(i in c("Malaria","Sepsis","LRTI","Gastro","UTI","Meningitis","Other infection","Other non-infection")) {
  p_hat <- table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i))["TRUE"]/
    sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i)))
  sens_spec_ci["sensitivity",i] <- 
    paste0("(", round((p_hat-(qnorm(0.95)/2)*sqrt(p_hat*(1-p_hat)/sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i)))))*100, digits = 0),
           "%-", round((p_hat+(qnorm(0.95)/2)*sqrt(p_hat*(1-p_hat)/sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,i)],i)))))*100, digits = 0), "%)")
  p_hat1 <- table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i))["TRUE"]/
    sum(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i)))
  sens_spec_ci["specificity",i] <- 
    paste0("(", round((p_hat1-(qnorm(0.95)/2)*sqrt(p_hat1*(1-p_hat)/sum(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i)))))*100, digits = 0),
           "%-", round((p_hat1+(qnorm(0.95)/2)*sqrt(p_hat1*(1-p_hat)/sum(table(str_detect(fws_an$discharge_diag[str_detect(fws_an$suspected_diag,i)],i)))))*100, digits = 0),"%)")
}
p_hat = table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,"Malaria")],"Malaria"))["TRUE"]/
  sum(table(str_detect(fws_an$suspected_diag[str_detect(fws_an$discharge_diag,"Malaria")],"Malaria")))
z = qnorm(1-0.05)/2

# Multiply by 100 to get percentage
sens_spec <- as.matrix(round(sens_spec1*100))
sens_spec_ci <- as.matrix(sens_spec_ci)
# Add white to color scale to represent zero
cols <- (brewer.pal(9, 'RdYlGn'))
rf <- colorRampPalette(cols)
r <- rf(100)

################### FIGURE 4. 
par(mar = c(5,6,3,3))
fields::image.plot(t(sens_spec[2:1,]), col = r,axes = FALSE, xlab = "",ylab = "",
                   legend.lab = "", smallplot = c(0.85, 0.88, 0.30, 0.85),
                   axis.args = list(at = c(0,20,40,60,80,94), 
                                    labels = paste0(c(0,20,40,60,80,100),"%"), cex.axis = .8))
text(matrix(rep(seq(0, 1.1, 0.143), 1), 2, 8, byrow = T), 
     matrix(rep(seq(1, 0, -1), 1), 2, 8, byrow = F),
     paste0(sens_spec,"%\n", sens_spec_ci), cex = 1)
axis(1, at = c(0,0.143,0.286,0.572,0.715), mgp=c(1,1,0),
     labels = c("Malaria", "Sepsis", "LRTI","UTI","Meningitis"))
axis(1, at = c(0.429, 0.858, 1.001), mgp=c(2,2,0),
     labels = c("Gastrointestinal\ninfection","Other\ninfection","Other\nnon-infection"))
axis(2, at = seq(1,0,-1), labels = c("Sensitivity","Specificity"), las = 2)
title(xlab = "Diagnosis",line = 3.5)
grid(nx = 8, ny = 2, col = "gray", lty = 1)
box()

###############################################################################
###############################################################################
