module elem
implicit none
public
!
!>  Array of atomic symbols
!
character(len=2),parameter :: asymb(118) = (/ &
&   'H ',                                                                                'He', &
&   'Li','Be',                                                  'B ','C ','N ','O ','F ','Ne', &
&   'Na','Mg',                                                  'Al','Si','P ','S ','Cl','Ar', &
&   'K ','Ca','Sc','Ti','V ','Cr','Mn','Fe','Co','Ni','Cu','Zn','Ga','Ge','As','Se','Br','Kr', &
&   'Rb','Sr','Y ','Zr','Nb','Mo','Tc','Ru','Rh','Pd','Ag','Cd','In','Sn','Sb','Te','I ','Xe', &
&   'Cs','Ba','La', &
&             'Ce','Pr','Nd','Pm','Sm','Eu','Gd','Tb','Dy','Ho','Er','Tm','Yb','Lu',           &
&                  'Hf','Ta','W ','Re','Os','Ir','Pt','Au','Hg','Tl','Pb','Bi','Po','At','Rn', &
&   'Fr','Ra','Ac', &
&             'Th','Pa','U ','Np','Pu','Am','Cm','Bk','Cf','Es','Fm','Md','No','Lr', &
                   'Rf','Db','Sg','Bh','Hs','Mt','Ds','Rg','Cn','Nh','Fl','Mc','Lv','Ts','Og' /)
end module elem

module tumme
use elem
use kintcm, only : itunnl, izct, isct, ilct
use perconparam, only : fu_tumme
!
!> Module for the TUMME - POLYRATE interface 
!==================================================================================
!
implicit none
public

!< Polyrate exception flag : polyrate terminated (0) normally (1) abnormally 
integer :: warning_code

!< reaction species that are present: (1) react1 (2) react2 (3) prod1 (4) prod2 (5) transition structure 
logical :: tumme_react_type(5)

!< total number of atoms
integer :: tumme_natoms

!< atomic symbols 
character(len=2),allocatable :: tumme_asymbols(:)

!< define molecule as data derived type variable
type molecule
   integer :: natoms, nfreq
   logical :: linear
   character(len=2),allocatable :: asymbol(:)
   double precision,allocatable :: geom(:,:), freq(:)
   double precision :: elec_ener(3) !< energy of electronic state
   integer :: elec_dege(3) !< degeneracy of electronic state
end type 

!< define MEP geometry as data derived type variable
type mep_geom
   integer :: natoms, nfreq
   double precision :: s_mep
   double precision,allocatable :: geom(:,:), freq(:)
   double precision :: VMEP, VaG 
end type 

!< reactants, products and transition structure
type(molecule) :: tumme_react(2), tumme_prod(2), tumme_ts

!< points along the MEP
type(mep_geom), allocatable :: tumme_mep(:) ! MEP points info

!< frequency scaling factor, reaction energy and reaction barrier
double precision :: tumme_freq_scal_factor
double precision :: tumme_react_energy
double precision :: tumme_react_barrier

!< tunneling energies and tunneling correction probabilities
double precision, allocatable :: tumme_tunn_ener(:), tumme_prob_zct(:), tumme_prob_sct(:), tumme_prob_lct(:), tumme_prob_muomt(:) 
 
  contains

  subroutine extract_atcoor(xr,iatsv,geom)
  !> store only the coordinates of the atoms that are listed
  !> reactants and products xr(:,1), xr(:,3), transition state xr(:,5)
  !> proper dimensions should be taken from geom and not xr or iatsv becasue
  !they are overdimensioned
  implicit none
  double precision, intent(in) :: xr(:)
  integer, intent(in) :: iatsv(:)
  double precision, intent(out) :: geom(:,:)
  double precision, allocatable :: xtemp(:)
  integer :: nat, n3, i, j
  nat=size(geom,dim=1)
  n3=size(xr(:))
  allocate(xtemp(3*nat))
  do i = 1,nat
    do j=1,3
        xtemp( 3*i-(3-j) ) = xr (3*iatsv(i)-(3-j) )
    enddo
  end do
  call vec2mat(xtemp(:),geom(:,:))
  deallocate(xtemp)
  end subroutine extract_atcoor

  subroutine vec2mat(vec,matrix) 
  !> transform geometry 1D vector(3*n) into 2D matrix(n,3)
  implicit none
  double precision :: vec(:)
  double precision :: matrix(:,:)
  integer :: n, i, j
  n = size(matrix,dim=1)
  do i=1,n
    do j=1,3
      matrix(i,j) = vec(3*i-(3-j))
    end do
  enddo
  end subroutine vec2mat

  subroutine getGeom(vec, amass, matrix) 
    !> transform mass-weighted-geometry 1D vector(3*n) into 2D matrix(n,3)
    implicit none
    double precision :: vec(:)
    double precision :: amass(:)
    double precision :: matrix(:,:)
    integer :: n, i, j
    n = size(matrix,dim=1)
    do i=1,n
      do j=1,3
        matrix(i,j) = vec(3*i-(3-j))/amass(3*i-(3-j))
      end do
    enddo
  end subroutine getGeom

  subroutine alloc_mol(molec,nat,ts)
  !> allocate molecule data derived type
  implicit none
  type(molecule), intent(inout) :: molec
  integer, intent(in) :: nat, ts ! ts = 1
  molec%natoms = nat
  if (nat == 1) then !< atomic
    molec%nfreq = 0 !; molec%linear= .True. 
  else if ( nat == 2 ) then !< diatomic
    molec%nfreq = 1 !; molec%linear= .True. 
  else if ( nat > 2 ) then !< polyatomic
    ! if (ts == 1) then
    !   molec%nfreq = 3*nat - 7 
    ! else
    !   molec%nfreq = 3*nat - 6 
    ! endif
    molec%nfreq = 3*nat - 6
  endif
  allocate(molec%geom(nat,3))
  allocate(molec%asymbol(nat))
  allocate(molec%freq(molec%nfreq))
  end subroutine alloc_mol

  subroutine alloc_mep(mep_point,nat)
  !> allocate mep point data derived type 
  implicit none
  type(mep_geom), intent(inout) :: mep_point 
  integer, intent(in) :: nat
  mep_point%natoms = nat
  mep_point%nfreq = 3*nat-7 
  allocate(mep_point%geom(mep_point%natoms,3))
  allocate(mep_point%freq(mep_point%nfreq))
  end subroutine alloc_mep
 
  subroutine write_tumme
  !
  !> This subroutine writes all the information needed by TUMME in fu100 
  !=====================================================================================
  !
  implicit none
  integer :: i, nmep, ntun
  
  !< [warningCode] Polyrate exception flag - code terminated 0 normally 1 abnormally
  write(fu_tumme,'(A,2X,I1)') "[WarningCode]", warning_code
  
  !< reactants
  if (tumme_react_type(1).or.tumme_react_type(2)) write(fu_tumme,'(A)') "[Reactants]"
  do i = 1, 2
    if (tumme_react_type(i)) then
      write(fu_tumme,'(2X,A,I1,A)') "[Reactant_",i,"]"
      call write_specie(tumme_react(i))
      write(fu_tumme,'(2X,A)') "[End]"
    end if
  end do
  write(fu_tumme,'(A)') "[End]"
  
  !< products
  if (tumme_react_type(3).or.tumme_react_type(4)) write(fu_tumme,'(A)') "[Products]"
  do i = 3, 4
    if (tumme_react_type(i)) then
      write(fu_tumme,'(2X,A,i1,A)') "[Product_",i-2,"]"
      call write_specie(tumme_prod(i-2))
      write(fu_tumme,'(2X,A)') "[End]"
    end if
  end do
  write(fu_tumme,'(A)') "[End]"
  
  !< transition state structure
  if (tumme_react_type(5)) then 
    write(fu_tumme,'(A)') "[TransitionState]"
    call write_specie(tumme_ts)
    write(fu_tumme,'(A)') "[End]"
  endif
 
  !< scaling factor and energies
  write(fu_tumme,'(A,F14.9)') "[FrequencyScalingFactor]", tumme_freq_scal_factor
  write(fu_tumme,'(A,F14.9)') "[ReactionEnergy(withZPE)]", tumme_react_energy
  write(fu_tumme,'(A,F14.9)') "[ReactionBarrier(withZPE)]", tumme_react_barrier
  
  !< MEP - write points along the MEP
  nmep = size(tumme_mep)
  write(fu_tumme,'(A)') "[MEP]"
  do i = 1, nmep
    call write_mep(tumme_mep(i))
  end do
  write(fu_tumme,'(A)') "[End]"
 
  !< tunneling probabilities
  if (itunnl .eq. 1) then 
   write(fu_tumme,'(A)') "[TransmissionProbability]"
    !< tunneling energies
    ntun = size(tumme_tunn_ener)
    call write_tunn("Energies",tumme_tunn_ener) 
    !< tunneling probabilities 
    if (izct .eq. 1) call write_tunn("ZCT",tumme_prob_zct) 
    if (isct .eq. 1) call write_tunn("SCT",tumme_prob_sct) 
    if (ilct .eq. 1) then
      call write_tunn("LCT",tumme_prob_lct) 
      call write_tunn("muOMT",tumme_prob_muomt) 
    endif
    write(fu_tumme,'(A)') "[End]"
  end if 
 
  end subroutine write_tumme
 
  subroutine write_specie(mol)
  !
  !> This subroutine writes the information of a molecular specie, reactant, product or transition structure
  !=========================================================================================================
  !
  use perconparam, only : fu_tumme
  implicit none
  type (molecule), intent(in) :: mol
  integer :: i, j
 
  !< linear or not
  if (mol%linear) then
    write(fu_tumme,'(4X,A,1X,A)') "[linear]","yes"
  else
    write(fu_tumme,'(4X,A,1X,A)') "[linear]","no"
  endif

  !< number of atoms 
  write(fu_tumme,'(4X,A,I4)') "[Natom]",mol%natoms
 
  !< atomic symbol
  write(fu_tumme,'(4X,A,99999(1x,A))') "[Symbol]",(mol%asymbol(i),i=1,mol%natoms)
 
  !< geometry
  write(fu_tumme,'(4X,A)') "[Geometry]"
  do i = 1, mol%natoms
    write(fu_tumme,'(6X,3(1x,F14.9))') (mol%geom(i,j),j=1,3) 
  end do
  write(fu_tumme,'(4X,A)') "[End]"

  !< frequencies
  if (mol%natoms > 1) then  ! only for molecules and not atoms
    write(fu_tumme,'(4X,A)') "[Frequency(cm-1)]"
    write(fu_tumme,'(99999(6X,3(1x,F14.9)/))') (mol%freq(i),i=1,mol%nfreq) 
    write(fu_tumme,'(4X,A)') "[End]"
  end if

  !< excited electronic states and degeneracies
  write(fu_tumme,'(4X,A)') "[Eele_level]"
  do i = 1, 3 
    write(fu_tumme,'(4X,I2,1x,F14.9)') mol%elec_dege(i), mol%elec_ener(i)
  end do
  write(fu_tumme,'(4X,A)') "[End]"
  end subroutine write_specie

  subroutine write_mep(mol)
  !> This subroutine writes the geometries along the MEP
  use perconparam, only : fu_tumme
  implicit none
  integer :: i, j
  type (mep_geom) :: mol
  write(fu_tumme,'(2X,A)') "[GeneralTransitionState]"
  write(fu_tumme,'(4X,A,1x,F9.4)') "[ReactionCoordinate]", mol%s_mep
  write(fu_tumme,'(4x,A)') "[Geometry]" 
  do i = 1, mol%natoms 
    write(fu_tumme,'(6X,3(1x,F14.9))') (mol%geom(i,j),j=1,3)
  end do
  write(fu_tumme,'(4x,A)') "[End]" 
  write(fu_tumme,'(4x,A)') "[Frequency(cm-1)]" 
  write(fu_tumme,'(99999(6X,3(1x,F14.9)/))') (mol%freq(i),i=1,mol%nfreq) 
  write(fu_tumme,'(4x,A)') "[End]" 
  write(fu_tumme,'(4x,A,F22.9)') "[VMEP]",mol%vmep
  write(fu_tumme,'(4x,A,F22.9)') "[VaG]",mol%vag
  write(fu_tumme,'(2x,A)') "[End]"
  end subroutine write_mep

  subroutine write_tunn(tunn_type,prob)
  !> Write tunneling probabilities 
  use perconparam, only : fu_tumme
  implicit none
  double precision,intent(in) :: prob(:)
  character(len=*),intent(in) :: tunn_type
  integer :: i, ntun
  ntun = size(prob, dim=1)
  write(fu_tumme,'(2X,A,A,A)') "[",trim(tunn_type),"]"
  write(fu_tumme,'(99999(5x,5(F14.9)/))') (prob(i),i=1,ntun) 
  write(fu_tumme,'(2X,A)') "[End]"
  end subroutine write_tunn

end module tumme

! Template of the TUMME - POLYRATE interface file
!===========================================================================================
!
![WarningCode]  0 
![Reactant]
!  [Reactant_1]
!    [linear] no
!    [Natom]   6
!    [Symbol] H H H H C H
!    [Geometry]  
!        0.000000E+00 1.058354E+01 0.000000E+00 
!        1.093972E+00 0.000000E+00 0.000000E+00 
!        -5.469252E-01 0.000000E+00 9.474360E-01 
!        -5.469252E-01 -5.348647E-12 -9.474360E-01 
!        0.000000E+00 0.000000E+00 0.000000E+00 
!        0.000000E+00 1.132545E+01 0.000000E+00
!    [End]
!    [Frequency] 
!        0.0144580 0.0144578 0.0136048  0.0063003 0.0063003 
!        0.0026428 0.0000316 -0.0000306 0.0000057 0.0000001 
!        0.0000001 -0.0000001
!    [End]
!    [Eele_level] 
!      0 2 
!      0.05 2
!    [End]
!  [End]
!  [Reactant_2]
!    [linear] no
!    [Natom]   6
!    [Symbol] H H H H C H
!    [Geometry]  
!        0.000000E+00 1.058354E+01 0.000000E+00 
!        1.093972E+00 0.000000E+00 0.000000E+00 
!        -5.469252E-01 0.000000E+00 9.474360E-01 
!        -5.469252E-01 -5.348647E-12 -9.474360E-01 
!        0.000000E+00 0.000000E+00 0.000000E+00 
!        0.000000E+00 1.132545E+01 0.000000E+00
!    [End]
!    [Frequency] 
!        0.0144580 0.0144578 0.0136048  0.0063003 0.0063003 
!        0.0026428 0.0000316 -0.0000306 0.0000057 0.0000001 
!        0.0000001 -0.0000001
!    [End]
!    [Eele_level] 
!      0 2 
!      0.05 2
!    [End]
!  [End]
![End]
![Product]
!  [Product_1]
!    [linear] no
!    [Natom]   6
!    [Symbol] H H H H C H
!    [Geometry]  
!        0.000000E+00 1.058354E+01 0.000000E+00 
!        1.093972E+00 0.000000E+00 0.000000E+00 
!        -5.469252E-01 0.000000E+00 9.474360E-01 
!        -5.469252E-01 -5.348647E-12 -9.474360E-01 
!        0.000000E+00 0.000000E+00 0.000000E+00 
!        0.000000E+00 1.132545E+01 0.000000E+00
!    [End]
!    [Frequency] 
!       0.0144580 0.0144578 0.0136048  0.0063003 0.0063003 
!        0.0026428 0.0000316 -0.0000306 0.0000057 0.0000001 
!        0.0000001 -0.0000001
!    [End]
!    [Eele_level] 
!      0 2 
!      0.05 2
!    [End]
!  [End]
!  [Product_2]
!    [linear] no
!    [Natom]   6
!    [Symbol] H H H H C H
!    [Geometry]  
!        0.000000E+00 1.058354E+01 0.000000E+00 
!        1.093972E+00 0.000000E+00 0.000000E+00 
!        -5.469252E-01 0.000000E+00 9.474360E-01 
!        -5.469252E-01 -5.348647E-12 -9.474360E-01 
!        0.000000E+00 0.000000E+00 0.000000E+00 
!        0.000000E+00 1.132545E+01 0.000000E+00
!    [End]
!    [Frequency] 
!        0.0144580 0.0144578 0.0136048  0.0063003 0.0063003 
!        0.0026428 0.0000316 -0.0000306 0.0000057 0.0000001 
!        0.0000001 -0.0000001
!    [End]
!    [Eele_level] 
!      0 2 
!      0.05 2
!    [End]
!  [End]
![End]
!
![TransitionState]
!  [linear] no
!  [Natom]   6
!  [Symbol] H H H H C H
!  [Geometry]  
!      0.000000E+00 1.058354E+01 0.000000E+00 
!      1.093972E+00 0.000000E+00 0.000000E+00 
!      -5.469252E-01 0.000000E+00 9.474360E-01 
!      -5.469252E-01 -5.348647E-12 -9.474360E-01 
!      0.000000E+00 0.000000E+00 0.000000E+00 
!      0.000000E+00 1.132545E+01 0.000000E+00
!  [End]
!  [Frequency] 
!      0.0144580 0.0144578 0.0136048  0.0063003 0.0063003 
!        0.0026428 0.0000316 -0.0000306 0.0000057 0.0000001 
!        0.0000001 -0.0000001
!  [End]
!  [Eele_level] 
!    0 2 
!    0.05 2
!  [End]
![End]
!
![FrequencyScalingFactor]  0.981
![ReactionEnergy(withZPE)] -0.012458
![ReactionBarrier(withZPE)] 0.0045832
!
![MEP]
!  [GeneralTransitionState]
!    [ReactionCoordinate]  -0.5
!    [Geometry]  
!      0.000000E+00 1.058354E+01 0.000000E+00 
!      1.093972E+00 0.000000E+00 0.000000E+00 
!      -5.469252E-01 0.000000E+00 9.474360E-01 
!      -5.469252E-01 -5.348647E-12 -9.474360E-01 
!      0.000000E+00 0.000000E+00 0.000000E+00 
!      0.000000E+00 1.132545E+01 0.000000E+00
!    [End]
!    [Frequency] 
!      0.0144580 0.0144578 0.0136048  0.0063003 0.0063003 
!        0.0026428 0.0000316 -0.0000306 0.0000057 0.0000001 
!        0.0000001 -0.0000001
!    [End]
!    [VMEP] 0.00057
!    [VaG] 0.00087    
!  [End]
!  [GeneralTransitionState]
!    [ReactionCoordinate] 0.0
!    [Geometry]  
!      0.000000E+00 1.058354E+01 0.000000E+00 
!      1.093972E+00 0.000000E+00 0.000000E+00 
!      -5.469252E-01 0.000000E+00 9.474360E-01 
!      -5.469252E-01 -5.348647E-12 -9.474360E-01 
!      0.000000E+00 0.000000E+00 0.000000E+00 
!      0.000000E+00 1.132545E+01 0.000000E+00
!    [End]
!    [Frequency] 
!      0.0144580 0.0144578 0.0136048  0.0063003 0.0063003 
!        0.0026428 0.0000316 -0.0000306 0.0000057 0.0000001 
!        0.0000001 -0.0000001
!    [End]
!    [VMEP] 0.00057
!    [VaG] 0.00087    
!  [End]
![End]
!
![TransmissionProbability]
!  [Energies]
!      0.00147 0.00148 0.00149 0.00150 0.00151
!      0.00152 0.00153
!  [End]
!  [ZCT]
!      0.001321 0.00158 0.00365 0.00454 0.0254
!      0.002414 0.02545
!  [End]
!  [SCT]
!      0.001321 0.00158 0.00365 0.00454 0.0254
!      0.002414 0.02545
!  [End]
!  [LCT]
!      0.001321 0.00158 0.00365 0.00454 0.0254
!      0.002414 0.02545
!  [End]
!  [muOMT]
!      0.001321 0.00158 0.00365 0.00454 0.0254
!      0.002414 0.02545
!  [End]
![End]
