!
      subroutine prepot
      use potcm4, only : nder
      use oh3cm; use oh3cm2
!
!   System:    OH + H2
!   Reference: G. C. Schatz and H. Elgersma
!              Chem. Phys. Lett. 73, 21 (1980).
!
!        PREPOT must be called once before any call to POTEN.
!        The potential parameters are included in DATA statements.
!        The coordinates, potential energy, and the derivatives of 
!        of the potential energy with respect to the coordinates are 
!        passed through the common block POTCM:
!                  /POTCM/ R(6), VTOT, DVDR(6).
!        The coordinates, potential energy, and the derivatives of
!        the potential energy with respect to the coordinates for
!        the LEPS part of the potential are passed through the 
!        common block POT2CM:
!                  /POT2CM/ R(4), ENERGY, DEDR(4)
!        All information passed through the common blocks POTCM and
!        POT2CM are in hartree atomic units.
!
!   The the flags that indicate what calculations should be carried out in 
!   the potential routine are passed through the common block POTCCM:
!                  /POTCCM/ NSURF, NDER, NDUM(8)
!   where:
!        NSURF - which electronic state should be used
!                This option is not used in this potential because 
!                only the ground electronic state is available.
!        NDER  = 0 => no derivatives should be calculated
!        NDER  = 1 => calculate first derivatives
!        NDUM  - these 8 integer values can be used to flag options
!                within the potential; in this potential these options 
!                are not used.
!
      implicit none
      integer :: i
!
!     COMMON /POT2CM/ R(4), ENERGY, DEDR(4)
!     COMMON /POTCCM/ NSURF, NDER, NDUM(8)
!     COMMON /CONCOM/ XDE(3), XBETA(3), XRE(3), SATO, GAM(3), REOH, 
!    *                REHH, CON(7), ALP(4), CLAM(4), ACON(2)
!
!     DIMENSION DE(3), BETA(3), RE(3), Z(3), ZPO(3), OP3Z(3), ZP3(3), 
!    *          TZP3(3), TOP3Z(3), DO4Z(3), B(3), X(3), COUL(3), EXCH(3) 
!
!     PARAMETER (R2 = 1.41421356D0)
!
!   Initialize the flag for the potential calculation
!
      NDER = 1
!
!   Set up the values of DE, BETA, and RE for the three-body LEPS potential.
!
      DE(1)   = XDE(1)
      BETA(1) = XBETA(1)
      RE(1)   = XRE(1)
      DE(2)   = XDE(1)
      BETA(2) = XBETA(1)
      RE(2)   = XRE(1)
      DE(3)   = XDE(3)
      BETA(3) = XBETA(3)
      RE(3)   = XRE(3)
!
!   Echo the potential energy surface parameters to unit 6.
!
      write(6,1000) XDE, XBETA, XRE, SATO
      write(6,1200) (GAM(i),i=1,3),REOH,REHH
      write(6,1300) CON
      write(6,1400) ALP,CLAM,ACON
!
      do i = 1, 3
         Z(i) = SATO
!   Compute useful constants.
         ZPO(i)   = 1.0D0+Z(i)
         OP3Z(i)  = 1.0D0+3.0D0*Z(i)
         TOP3Z(i) = 2.0D0*OP3Z(i)
         ZP3(i)   = Z(i)+3.0D0
         TZP3(i)  = 2.0D0*ZP3(i)
         DO4Z(i)  = DE(i)/4.0D0/ZPO(i)
         B(i)     = BETA(i)*DO4Z(i)*2.0D0
      enddo
!
1000  FORMAT (/,2X,T5,'OH + H2 potential energy function', &
              //, 2X, T5, 'Potential energy surface parameters ', &
                          'in hartree atomic units:', &
              /,2X,T5,'Morse and LEPS parameters:', &
              /,2X,T5,'Dissociation energies:', T31,1P,3E13.6, &
              /,2X,T5,'Morse betas:', T31,1P,3E13.6, &
              /,2X,T5,'Equilibrium bond lengths:',T31,1P,3E13.6, &
              /,2X,T5,'Sato parameter:', T31, 1PE13.6) 
1200  FORMAT (/,2X,T5,'GAM:',T20,1P,3E13.6, &
              /,2X,T5,'REOH, REHH:',T20,1P,2E13.6)
1300  FORMAT (/,2X,T5,'CON:',T20,1P,4E13.6,(/,2X,T20,1P,4E13.6))
1400  FORMAT (/,2X,T5,'ALP:',T20,1P,4E13.6, &
              /,2X,T5,'CLAM:',T20,1P,4E13.6, &
              /,2X,T5,'ACON:',T20,1P,2E13.6)
!
      RETURN
      END SUBROUTINE prepot
!
      SUBROUTINE POT
      use potcm4, only : nder
      use oh3cm; use oh3cm2
      implicit none
      double precision :: rad, s
      integer :: i
!
!   Initialize the variable used for storing the energy.
!
      ENERGY = 0.D0
!
      do i = 1, 3
         X(I)    = EXP(-BETA(I)*(RX(I)-RE(I)))
         COUL(I) = DO4Z(I)*(ZP3(I)*X(I)-TOP3Z(I))*X(I)
         EXCH(I) = DO4Z(I)*(OP3Z(I)*X(I)-TZP3(I))*X(I)
         ENERGY  = ENERGY + COUL(I)
      enddo
      RAD = SQRT((EXCH(1)-EXCH(2))**2+(EXCH(2)-EXCH(3))**2+(EXCH(3)-EXCH(1))**2)
      ENERGY = ENERGY - RAD/R2 
!
!   Compute the derivatives of the energy with respect to the internal 
!   coordinates.
         IF (NDER .EQ. 1) THEN
             S = EXCH(1) + EXCH(2) + EXCH(3)
             do i = 1, 3 
                   DEDR(I) = B(I)*X(I)*((3.0D0*EXCH(I)-S)/R2*  &
                             (OP3Z(I)*X(I)-ZP3(I))/RAD-  &
                             ZP3(I)*X(I)+OP3Z(I))
             enddo
         ENDIF
!
      return 
      end subroutine pot
!
      subroutine poten
      use potcm4
      use oh3cm
!
!   This subprogram evaluates the OH + H2 potential energy and derivatives
!   of the potential with respect to the internal coordinates.
!   The subprogram PREPEF must be called once before any calls to this 
!   subprogram. 
!   All calculations in this subprogram are in hartree atomic units.
!
      implicit none
      integer :: i
!
!
!   Zero the array which will contain the derivatives of the energy 
!   with respect to the internal coordinates.
!
      IF (NDER .EQ. 1) THEN
         do i = 1, 6
             DVDR(i) = 0.0D0
         enddo
      ENDIF
!
!   Calculate the Morse part of the potential energy.
      VTOT=VMOR(XDE(1),XBETA(1),XRE(1),R(1))+VMOR(XDE(2),XBETA(2),   &
       XRE(2),R(4))+VMOR(XDE(2),XBETA(2),XRE(2),R(5))
!   Calculate the derivatives of the Morse part of the potential energy.
      IF (NDER .EQ. 1) THEN
          DVDR(1) = DVDR(1)+DVMOR(XDE(1),XBETA(1),XRE(1),R(1))
          DVDR(4) = DVDR(4)+DVMOR(XDE(2),XBETA(2),XRE(2),R(4))
          DVDR(5) = DVDR(5)+DVMOR(XDE(2),XBETA(2),XRE(2),R(5))
      ENDIF
!   Initialize the coordinates for the three-body LEPS part of the potential.
      RX(1) = R(2)
      RX(2) = R(3)
      RX(3) = R(6)
!
!   Calculate the three-body LEPS portion of the potential and update the 
!   energy term.
      CALL POT
      VTOT = VTOT+ENERGY
!   Update the array containing the derivatives with the LEPS derivatives.
      IF (NDER .EQ. 1) THEN
          DVDR(2) = DVDR(2)+DEDR(1)
          DVDR(3) = DVDR(3)+DEDR(2)
          DVDR(6) = DVDR(6)+DEDR(3)
      ENDIF
!   Initialize the coordinates for the H2O part of the potential for H1 and H2.
      RX(1) = R(1)
      RX(2) = R(2)
      RX(3) = R(4)
!   Calculate the H2O part of the potential and update the energy term.
      CALL VH2O
      VTOT = VTOT+ENERGY
!   Update the array containing the derivatives with the H2O derivatives.
      IF (NDER .EQ. 1) THEN
          DVDR(1) = DVDR(1)+DEDR(1)
          DVDR(2) = DVDR(2)+DEDR(2)
          DVDR(4) = DVDR(4)+DEDR(3)
      ENDIF
!   Initialize the coordinates for the H2O part of the potential for H1 and H3.
      RX(1) = R(1)
      RX(2) = R(3)
      RX(3) = R(5)
!   Calculate the H2O part of the potential and update the energy term.
      CALL VH2O
      VTOT = VTOT+ENERGY
!   Update the array containing the derivatives with the H2O derivatives.
      IF (NDER .EQ. 1) THEN
         DVDR(1) = DVDR(1)+DEDR(1)
         DVDR(3) = DVDR(3)+DEDR(2)
         DVDR(5) = DVDR(5)+DEDR(3)
      ENDIF
!   Initialize the coordinates for the four-body part of the potential.
      RX(1) = R(2)
      RX(2) = R(3)
      RX(3) = R(4)
      RX(4) = R(5)
!   Calculate the four-body part of the potential and update the energy term.
      CALL V4POT
      VTOT = VTOT+ENERGY
!   Update the array containing the derivatives.
      IF (NDER .EQ. 1) THEN
          DVDR(2) = DVDR(2)+DEDR(1)
          DVDR(3) = DVDR(3)+DEDR(2)
          DVDR(4) = DVDR(4)+DEDR(3)
          DVDR(5) = DVDR(5)+DEDR(4)
      ENDIF
!
!   Adjust the potential to the correct zero, which corresponds to
!   OH(R=RE) and H2(R=RE) at infinity.
!
      VTOT = VTOT-2.0D0*XDE(2)+XDE(3)
!
      RETURN
!   Define the statement functions VMOR and DVMOR which evaluate the Morse
!   diatomic potential energy  and the derivatives of the Morse potential.
!    
      CONTAINS
!
          DOUBLE PRECISION FUNCTION VMOR(D,B,T,RR)
          IMPLICIT NONE
          REAL(8) :: D, B, T, RR
          VMOR = D*(1.0D0-EXP(-B*(RR-T)))**2
          RETURN
          END FUNCTION VMOR
!
          DOUBLE PRECISION FUNCTION DVMOR(D,B,T,RR)
          IMPLICIT NONE
          REAL(8) :: D, B, T, RR
          DVMOR = 2.0D0*B*D*(1.0D0-EXP(-B*(RR-T)))*EXP(-B*(RR-T))
          RETURN
          end function DVMOR

      END subroutine poten
!
!*****
      subroutine V4POT
      use potcm4,only : nder
      use oh3cm, only : rx,energy,dedr,alp,clam,acon
      implicit none
      double precision :: R(4),A(4),C(4),COF(2),t1,t2
!
      R(:) = RX(:)
      A(:) = ALP(:)
      C(:) = CLAM(:)
      COF(:) = ACON(:)
!
      T1 = EXP(-C(1)*(R(1)-A(1))**2-C(1)*(R(2)-A(1))**2-C(3)*(R(3)-A(3)) &
         **2-C(3)*(R(4)-A(3))**2)*COF(1)
      T2 = EXP(-C(2)*(R(1)-A(2))**2-C(2)*(R(2)-A(2))**2-C(4)*(R(3)-A(4)) &
         **2-C(4)*(R(4)-A(4))**2)*COF(2)
      ENERGY = T1+T2
         IF (NDER .EQ. 1) THEN
             DEDR(1) = -2.0D0*(T1*C(1)*(R(1)-A(1))+T2*C(2)*(R(1)-A(2)))
             DEDR(2) = -2.0D0*(T1*C(1)*(R(2)-A(1))+T2*C(2)*(R(2)-A(2)))
             DEDR(3) = -2.0D0*(T1*C(3)*(R(3)-A(3))+T2*C(4)*(R(3)-A(4)))
             DEDR(4) = -2.0D0*(T1*C(3)*(R(4)-A(3))+T2*C(4)*(R(4)-A(4)))
         ENDIF
!
      return 
      end subroutine v4pot
!*****
!
      subroutine vh2o
      use potcm4,only : nder, ndum
      use oh3cm, only : RX,ENERGY,DEDR,GAM,REOH,REHH,CON
      implicit none
      double precision :: s(3), q(3), dq(3), x(3), dp(3), c(7)
      double precision :: p, temp, trm1, xmax1, xmax2
      integer :: i
!
!     XMAX1 FOR WHEN TANH SET=1.0 ON VAX
!     XMAX2 FOR PREVENTING OVERFLOWS ON VAX
!
      DATA XMAX1,XMAX2 / 15.0D0,43.0D0 /
      C(:) = CON(:)
!
!     STATEMENT FUNCTION
!
      s(1) = RX(1)-REOH
      s(3) = RX(2)-REOH
      s(2) = RX(3)-REHH
      do i = 1, 3
          X(i) = 0.5D0*GAM(i)*S(i)
          Q(i) = 1.0D0-TANH(X(i))
          if (X(i).LT.XMAX1) then
              DQ(i) = -0.5D0*GAM(i)/COSH(X(i))**2
              cycle
          endif
          if (X(i).LT.XMAX2) then
              Q(i) = TANLG(X(i))
              DQ(i) = -0.5D0*GAM(i)/COSH(X(i))**2
              cycle
          endif  
          Q(i) = 0.0D0
          DQ(i) = 0.0D0
      enddo
!
      P = C(1)+C(2)*(S(1)+S(3))+C(3)*S(2)+0.5D0*C(4)*(S(1)*S(1)+S(3)*S(3))   &
          +0.5D0*C(5)*S(2)*S(2)+C(6)*S(2)*(S(1)+S(3))+C(7)*S(1)*S(3)
      ENERGY = Q(1)*Q(2)*Q(3)*P
!
      IF (NDER .EQ. 1) THEN 
          DP(1) = C(2)+C(4)*S(1)+C(6)*S(2)+C(7)*S(3)
          DP(2) = C(3)+C(5)*S(2)+C(6)*(S(1)+S(3))
          DP(3) = C(2)+C(4)*S(3)+C(6)*S(2)+C(7)*S(1)
          do i = 1, 3
              TRM1 = 0.0D0
              IF (Q(i).EQ.0.0D0) then
                  DEDR(i) = ENERGY*(TRM1+(DP(i)/P))
                  cycle 
              endif 
              TRM1 = DQ(i)/Q(i)
              DEDR(i) = ENERGY*(TRM1+(DP(i)/P))
          enddo
          TEMP = DEDR(2)
          DEDR(2) = DEDR(3)
          DEDR(3) = TEMP
      ENDIF
!
      return 
      contains 

         double precision function tanlg(xx) 
         implicit none 
         real(8), INTENT(IN) :: xx
         TANLG = 2.0D0/(1.0D0+DEXP(2.0D0*XX))
         return
         end function tanlg

      END subroutine VH2O
!*****
