      SUBROUTINE setup(N3TM)
!
!   Sys.: NH3
!   Ref.: B. Maessen, P. Bopp, D. R. McLaughlin, and M. Wolfsberg
!         Z. Naturforsch. Teil A 39, 1005-1006 (1984).
!
!        SETUP must be called once before any calls to SURF.
!        The potential parameters are initialized in the subprogram SETUP.
!        The cartesian coordinates, potential energy, and the derivatives of 
!        the potential energy with respect to the cartesian coordinates are 
!        passed in the call statement through the argument list:
!                  CALL SURF (V, X, DX, N3TM)
!        where X and DX are dimensioned N3TM.
!        All information passed through the parameter list is in 
!        hartree atomic units.  
!
      use nh3, only : pi,ij,ik,A,B,C,re,C2A,C2B,C2C,C2D,C2E,C2F, &
                      C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H, C4A, &
                      RT1D2, RT1D3, RT1D6, PID2, ALPHAE
!                        RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2,  &
!                        SRIJ, SAIJ, PRDR, PRDA, DAIJ, DRIJ, &
!                        C2A, C2B, C2C, C2D, C2E, C2F, &
!                        C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H, &
!                        C4A, &
!                        RT1D2, RT1D3, RT1D6, PI, PID2, &
!                        IJ, IK
      implicit none
      integer,intent(in) :: n3tm
      integer :: n3tmmn
!
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!      COMMON / POTCON / RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2, 
!     *                  SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),
!     *                  C2A, C2B, C2C, C2D, C2E, C2F,
!     *                  C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,
!     *                  C4A
!      COMMON /RTCON/ RT1D2, RT1D3, RT1D6, PI, PID2
!      COMMON /INDICE/ IJ(3), IK(3)
      DATA N3TMMN / 12/
!
      WRITE (6, 1000)
!
!   Check the value of N3TM passed by the calling program.
!
         IF (N3TM .LT. N3TMMN) THEN
             WRITE (6, 6000) 
             STOP 'SETUP 1'
         ENDIF
!
         IJ(1) = 2 
         IJ(2) = 3 
         IJ(3) = 1 
         IK(1) = 3 
         IK(2) = 1 
         IK(3) = 2 
!
         PI = 3.1415926536D0
!
! INITIALIZE NH3 POTENTIAL CONSTANTS (SEE WOLFSBERG AND
! MORINO ET. AL.
!
      RE = 1.024D0
      ALPHAE = (PI / 180.00D0) * 107.3D0
!
      A = 0.53190D-11
      B = 1.1406D-11
      C = 0.6140D0
!
! INITIALIZE POTENTIAL CONSTANTS
!
! QUADRATIC TERMS: C2
!
      C2A = 3.52595D0
      C2B = 0.22163D0
      C2C = -0.22163D0
      C2D = 0.01480D0
      C2E = -0.11620D0
      C2F = 0.05810D0
!
! CUBIC TERMS: C3
!
      C3A = -0.55000D0
      C3B = -0.54000D0
      C3C = -7.01000D0
      C3D = -0.03778D0
      C3E =  0.05667D0
      C3F = -0.22667D0
      C3G = -0.07000D0
      C3H =  0.07000D0
!
! QUARTIC TERM: C4A
!
      C4A = 11.07417D0
!
! DETERMINE OTHER USEFUL CONSTANTS
!
      RT1D2 = 1.0D0/SQRT(2.0D0)
      RT1D3 = 1.0D0/SQRT(3.0D0)
      RT1D6 = 1.0D0/SQRT(6.0D0)
      PID2 = PI / 2.0D0
!
1000  FORMAT(/,2X,T5,'Setup has been called for NH3, ', &
                      'surface no. 2 of Wolfsberg')
6000  FORMAT(/,2X,T5,'Error: The value of N3TM passed by the ', &
                     'calling program is smaller ', &
             /,2X,T12,'than the minimum value allowed.')
!
      RETURN
      END SUBROUTINE SETUP

      SUBROUTINE SURF(V, XB, DXB, N3TM)
!
! THE DRIVER ROUTINE SHOULD BE CALLED SURF
!
      use nh3, only :  dvdx, vnh3
!        VM, VS, VNH3, VQUAD, VCUBIC, VQUAR,  &
!        DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),  &
!                      DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),  &
!                      DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12),  &
!                      RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2,   &
!                      SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),  &
!                      C2A, C2B, C2C, C2D, C2E, C2F,  &
!                      C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,  &
!                      C4A
      implicit none
      integer, intent(in) :: n3tm
      double precision, intent(in) :: xb(n3tm)
      double precision, intent(out) :: V, dxb(n3tm)
      double precision :: eeq0
      integer :: i

!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!      COMMON /XANG/ X(30)
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!C
!      COMMON /POTVAL/ VM, VS, VNH3, VQUAD, VCUBIC, VQUAR
!C
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!C
!      COMMON / POTCON / RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2, 
!     *                  SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),
!     *                  C2A, C2B, C2C, C2D, C2E, C2F,
!     *                  C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,
!     *                  C4A
!
!         DIMENSION XB(N3TM), DXB(N3TM)
!
! EEQ0 - EQUILIBRIUM ENERGY OF NH3 IN C3V GEOMETRY
!
       DATA EEQ0 /0.2533613651D0/
!
! DRIVER ROUTINE THAT COMPUTES THE WOLFSBERG POTENTIAL AND
! DERIVATIVES FOR NH3 - FINAL DERIVATIVES IN CARTESIANS
!
!
      CALL CORDTR(XB, N3TM)
!
      CALL POTM
      CALL POTS
      CALL DVDINC
      CALL DRDXSB
      CALL DALPDR
      CALL DVMDXS
      CALL DNDXSB
      CALL DFDXSB
      CALL DGDXSB
      CALL DVSDXS
!
!  SUBTRACT OFF EEQ0 SO THAT EQUILIBRIUM GEOM OF NH3 HAS V=0.0
!
      V = VNH3 - EEQ0
!
      do i=1,12
        DXB(i) = DVDX(i) * 0.52917706D0
      enddo
!
      return 
      END SUBROUTINE SURF
!
      SUBROUTINE CORDTR(XB, N3TM)
      use nh3, only : x,rnh,rhh,ij,ik,saij,alpha,delrnh,delalp,pid2,rnormi,rnormj,rnormk,rkappa, &
                      alphae,pid2,prdr,prda,sda2,sdr,sdr2,srij,daij,drij,re,sda
!                      RNH(3), RHH(3), ALPHA(3), RKAPPA(3),  &
!                     DELRNH(3), DELALP(3),  &
!                     RNORMI, RNORMJ, RNORMK,  &
!                     RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2,   &
!                     SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),  &
!                     C2A, C2B, C2C, C2D, C2E, C2F,  &
!                     C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,  &
!                     C4A,  &
!                     RT1D2, RT1D3, RT1D6, PI, PID2,  &
!                     IJ(3), IK(3), X(30)
      implicit none
      integer, intent(in) :: n3tm
      double precision,intent(in) :: XB(n3tm)
      integer :: N3M6, i, j, k, ih, IOFF
!
! RNH(I) = N-HI BOND LENGTH IN ANGSTROM
!
! RHH(I) = H(J)-H(K) DISTANCE IN ANGSTROM
!
! DELRNH(I) = DELTA IN N-HI BOND LENGTH FROM
!     EQUILIBRIUM IN ANGSTROM
!
! ALPHA(I) = HJ-N-HK BOND ANGLE IN RADIANS
!
! DELALP(I) = DELTA IN HJ-N-HK BOND ANGLE FROM
!     EQUILIBRIUM IN RADIANS
!
! RKAPPA(I) = ANGLE BETWEEN H-H-H PLANE AND ITH
!     BOND IN RADIANS
!
! X(I) = CARTESIAN CORDINATES
!     I = 1 - 3 :   X, Y, Z, OF N
!     I = 4 - 6 :   X, Y, Z, OF H1
!     I = 7 - 9 :   X, Y, Z, OF H2
!     I = 10 - 12 : X, Y, Z, OF H3
!
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!C
!      COMMON / POTCON / RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2, 
!     *                  SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),
!     *                  C2A, C2B, C2C, C2D, C2E, C2F,
!     *                  C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,
!     *                  C4A
!C
!      COMMON / RTCON / RT1D2, RT1D3, RT1D6, PI, PID2
!C
!      COMMON / INDICE / IJ(3), IK(3)
!C
!      COMMON /XANG/ X(30)
!         DIMENSION XB(N3TM)
!
! CHANGE ALL COORDINATES FROM BOHR TO ANGSTROM
!
      do i=1,12
        X(i) = XB(i) * 0.52917706D0
      enddo
      N3M6=6
!
! CALCULATE RNH
!
      IOFF=3
      do ih=1,3
        RNH(ih) = SQRT ((X(IOFF+1) - X(1))**2  &
                    +   (X(IOFF+2) - X(2))**2  &
                    +   (X(IOFF+3) - X(3))**2)
      IOFF=IOFF+3
      enddo
!
! CALCULATE RHH
!
      do i=1,3
        J=IJ(i)
        K=IK(i)
        RHH(i) = SQRT ((X(3*J+1) - X(3*K+1))**2  &
                   +   (X(3*J+2) - X(3*K+2))**2  &
                   +   (X(3*J+3) - X(3*K+3))**2)
      enddo
!
! CALCULATE ALPHA
!
      do i=1,3
        J=IJ(i)
        K=IK(i)
        ALPHA(i) = ACOS((RNH(J)**2+RNH(K)**2-RHH(i)**2)/  &
                 (2.0D0*RNH(J)*RNH(K)))
      enddo
!
! CALCULATE DELRNH AND DELALP
!
      do i=1,3
        DELRNH(i) = RNH(i) - RE
        DELALP(i) = ALPHA(i) - ALPHAE
      enddo
!
! CALCULATE RKAPPA
!
! DETERMINE COMPONENTS OF VECTOR NORMAL TO HHH PLANE
!
      RNORMI = (X(8)-X(5))*(X(12)-X(6)) - (X(9)-X(6))*(X(11)-X(5))
      RNORMJ = -((X(7)-X(4))*(X(12)-X(6)) - (X(9)-X(6))*(X(10)-X(4)))
      RNORMK = (X(7)-X(4))*(X(11)-X(5)) - (X(8)-X(5))*(X(10)-X(4))
!
      do i=1,3
      RKAPPA(I) = PID2 - ACOS ((RNORMI*(X(1)-X(3*I+1))     &
                          +     RNORMJ*(X(2)-X(3*I+2))     &
                          +     RNORMK*(X(3)-X(3*I+3))) /  &
       (RNH(I)*SQRT(RNORMI**2+RNORMJ**2+RNORMK**2)))
      enddo
!
! DETERMINE USEFUL SUMS:
!
!...SDR = SUM DELRNH(I)
!...SDR2 = SUM DELRNH(I)**2
!...SDA = SUM DELALP(I)
!...SDA2 = SUM DELALP(I)**2
!...SRIJ = SUM DELRNH(I)*DELRNH(J)
!...SAIJ = SUM DELALP(I)*DELALP(J)
!...PRDR = PRODUCT DELRNH(I)
!...PRDA = PRODUCT DELALP(I)
!...DAIJ(K) = PRODUCT DELALP(I)*DELALP(J)
!...DRIJ(K) = PRODUCT DELRNH(I)*DELRNH(J)
!
      SDR = DELRNH(1)+DELRNH(2)+DELRNH(3)
      SDR2 = DELRNH(1)**2 + DELRNH(2)**2 + DELRNH(3)**2
      SDA = DELALP(1)+DELALP(2)+DELALP(3)
      SDA2 = DELALP(1)**2+DELALP(2)**2+DELALP(3)**2
      SRIJ = DELRNH(1)*DELRNH(2)+DELRNH(1)*DELRNH(3)+DELRNH(2)*DELRNH(3)
      SAIJ = DELALP(1)*DELALP(2)+DELALP(1)*DELALP(3)+DELALP(2)*DELALP(3)
      PRDR = DELRNH(1)*DELRNH(2)*DELRNH(3)
      PRDA = DELALP(1)*DELALP(2)*DELALP(3)
!
      do i=1,3
        J=IJ(i)
        K=IK(i)
        DAIJ(i) = DELALP(J)*DELALP(K)
        DRIJ(i) = DELRNH(J)*DELRNH(K)
      enddo
!
      RETURN 
      END SUBROUTINE CORDTR
!
      SUBROUTINE POTM
      use nh3, only : vquad,vcubic,vquar,vm,delrnh,delalp,saij,srij, &
                     C2A, C2B, C2C, C2D, C2E, C2F,  &
                     C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H, C4A, PRDA, &
                     PRDR, SDA2, SDR, SDR2, VM, A, B, C
!                    RNH(3), RHH(3), ALPHA(3), RKAPPA(3),  &
!                    DELRNH(3), DELALP(3),  &
!                    RNORMI, RNORMJ, RNORMK,  &
!                    RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2,   &
!                    SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),  &
!                    C2A, C2B, C2C, C2D, C2E, C2F,  &
!                    C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,  &
!                    C4A, VM, VS, VNH3, VQUAD, VCUBIC, VQUAR  
      implicit none
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
! THIS SUBROUTINE CALCULATES THE MORINO POTENTIAL FOR
! NH3 WITH WOLFSBERG'S MODIFICATIONS
!
!
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!
!      COMMON / POTCON / RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2, 
!     *                  SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),
!     *                  C2A, C2B, C2C, C2D, C2E, C2F,
!     *                  C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,
!     *                  C4A
!
!      COMMON /POTVAL/ VM, VS, VNH3, VQUAD, VCUBIC, VQUAR
!
! CALCULATE QUADRATIC TERM
!
      VQUAD = C2A * SDR2 + C2B * SDA2 + C2C * SAIJ + C2D * SRIJ  &
            + C2E * (DELRNH(1)*DELALP(1) + DELRNH(2)*DELALP(2)  &
                  +  DELRNH(3)*DELALP(3))  &
            + C2F * (DELRNH(1)*(DELALP(2) + DELALP(3))  &
                  +  DELRNH(2)*(DELALP(1) + DELALP(3))  &
                  +  DELRNH(3)*(DELALP(1) + DELALP(2)))
!
! CUBIC TERM
!
      VCUBIC = C3A * (DELRNH(1)*(DELRNH(2)**2 + DELRNH(3)**2)  &
                   +  DELRNH(2)*(DELRNH(1)**2 + DELRNH(3)**2)  &
                   +  DELRNH(3)*(DELRNH(1)**2 + DELRNH(2)**2))  &
             + C3B * PRDR  &
             + C3C * (DELRNH(1)**3 + DELRNH(2)**3 + DELRNH(3)**3)  &
             + C3D * (DELALP(1)**3 + DELALP(2)**3 + DELALP(3)**3)  &
             + C3E * (DELALP(1)*(DELALP(2)**2 + DELALP(3)**2)  &
                   +  DELALP(2)*(DELALP(1)**2 + DELALP(3)**2)  &
                   +  DELALP(3)*(DELALP(1)**2 + DELALP(2)**2))  &
             + C3F * PRDA + C3G * SDR * SDA2 + C3H * SDR * SAIJ
!
! QUARTIC TERM
!
      VQUAR = C4A * (DELRNH(1)**4 + DELRNH(2)**4 + DELRNH(3)**4)
!
!
      VM = VQUAD + VCUBIC + VQUAR
!
! CONVERT TO HARTREE
!
      VM = VM * .22937D0
!
      RETURN 
      END SUBROUTINE POTM
!
      SUBROUTINE POTS
      use nh3, only : rkappa,rt1d3,a,b,c,vs,vnh3,vm
!                     RNH(3), RHH(3), ALPHA(3), RKAPPA(3),  &
!                     DELRNH(3), DELALP(3),  &
!                     RNORMI, RNORMJ, RNORMK,  &
!                     RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2,   &
!                     SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),  &
!                     C2A, C2B, C2C, C2D, C2E, C2F,  &
!                     C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,  &
!                     C4A, RT1D2, RT1D3, RT1D6, PI, PID2,  &
!                     VM, VS, VNH3, VQUAD, VCUBIC, VQUAR

      implicit none
      double precision :: roe 
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
! THIS SUBROUTINE CALCULATES THE WOLFSBERG TERM
! IN THE WOLFSBERG NH3 POTENTIAL
!
!
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!
!      COMMON / POTCON / RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2, 
!     *                  SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),
!     *                  C2A, C2B, C2C, C2D, C2E, C2F,
!     *                  C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,
!     *                  C4A
!
!      COMMON / RTCON / RT1D2, RT1D3, RT1D6, PI, PID2
!
!      COMMON /POTVAL/ VM, VS, VNH3, VQUAD, VCUBIC, VQUAR
!
      ROE = RT1D3 * (RKAPPA(1)+RKAPPA(2)+RKAPPA(3))
!
      VS = A*(ROE**2) + (B*EXP(-C*(ROE**2)))
!
! CONVERT VS FROM ERGS TO HARTREE
!
      VS = VS * 2.2937D+10
!
      VNH3 = VM + VS
!
      RETURN 
      END SUBROUTINE POTS
!
      SUBROUTINE MULT(A,NA,MA,B,NB,MB,C)
      implicit none
      integer, intent(in) :: NA,MA,NB,MB
      double precision,intent(in) :: A(NA,MA), B(NB,MB)
      double precision,intent(out) :: C(NA,MB)
      double precision :: sum
      integer :: i,j,k
!
! THIS SUBROUTINE PERFORMS THE MATRIX MULTIPLICATION :
!
!         C = A X B
!
!      DIMENSION A(NA,MA), B(NB,MB), C(NA,MB)
!
      do i = 1, NA
         do j = 1, MB
           SUM=0.0D0
           do  k = 1, MA
             SUM = SUM + A(i,k) * B(k,j)
           enddo 
           C(i,j) = SUM
         enddo
!
      enddo
      RETURN 
      END SUBROUTINE MULT
!
      SUBROUTINE DALPDR
      use nh3, only : dadr,rnh,alpha,ij,ik,rhh
!                     RNH(3), RHH(3), ALPHA(3), RKAPPA(3), &
!                     DELRNH(3), DELALP(3), &
!                     RNORMI, RNORMJ, RNORMK, &
!                     DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12), &
!                     DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12), &
!                     DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12), &
!                     IJ(3), IK(3)
      implicit none
      integer :: i,j,k

!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
! THIS SUBROUTINE CALCULATES THE NON-ZERO
! PARTIAL DERIVATIVES OF ALPHA(I) W.R.T.
! RNH(I) AND RHH(I)
!
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!C
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!C
!      COMMON /INDICE/ IJ(3), IK(3)
!
      do i=1,3
        J = IJ(i)
        K = IK(i)
!
        DADR(i,J) = -(1.0D0/SIN(ALPHA(I))) *  &
                   (RNH(J)**2 - RNH(K)**2 + RHH(I)**2) /  &
                   (2.0D0 * RNH(K) * RNH(J)**2)
!
        DADR(i,K) = -(1.0D0/SIN(ALPHA(I))) *  &
                   (RNH(K)**2 - RNH(J)**2 + RHH(I)**2) /  &
                   (2.0D0 * RNH(J) * RNH(K)**2)
!
        DADR(i,3+i) = (1.0D0/SIN(ALPHA(I))) * RHH(I) /  &
                    (RNH(J) * RNH(K))
!
      enddo
!
      RETURN 
      END SUBROUTINE DALPDR
!
      SUBROUTINE DFDXSB
      use nh3, only : rnorm,dfdx,x,dndx
!     X(30),  &
!     DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),  &
!     DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),  &
!     DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12),  &
!     RNORM(3), FNORM(3), GNORM(3)
      implicit none
      integer :: i, ix, m
!
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
! THIS SUBROUTINE EVALUATES THE PARTIAL DERIVATIVES OF FNORM(I)
! W.R.T. THE CARTESIANS
!
!      COMMON /XANG/ X(30)
!C
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!C
!      COMMON /NORMAL/ RNORM(3), FNORM(3), GNORM(3)
!
! EVALUATE DFDX(I,J), I=1,3; J=1,3
!
      do i = 1,3
        do ix = 1,3
          DFDX(i,ix) = RNORM(ix)
        enddo 
      enddo
!
! EVALUATE DFDX(I,J), I=1,3; J=4,12
!
      do i = 1,3
        do ix = 1,3
          do m = 1,3
            DFDX(i, 3*IX + M) = (X(1) - X(3*I+1))*DNDX(1,3*IX+M)  &
                        + (X(2) - X(3*I+2))*DNDX(2,3*IX+M)  &
                        + (X(3) - X(3*I+3))*DNDX(3,3*IX+M)
            IF(I.EQ.IX) DFDX(I,3*IX+M) = DFDX(I,3*IX+M) - RNORM(M)
          enddo
        enddo
      enddo
!
      RETURN
      END SUBROUTINE DFDXSB
!
      SUBROUTINE DGDXSB
      use nh3, only : drdx, dgdx, rnorm, rnh, dndx
!        DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12), &
!        DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12), &
!        DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12), &
!        RNORM(3), FNORM(3), GNORM(3), &
!        RNH(3), RHH(3), ALPHA(3), RKAPPA(3), &
!        DELRNH(3), DELALP(3), &
!        RNORMI, RNORMJ, RNORMK 
      implicit none
      double precision :: xnorm
      integer :: i, ix
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
! THIS SUBROUTINE EVALUATES THE PARTIAL DERIVATIVES OF
! GNORM W.R.T. THE CARTESIANS
!
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!C
!      COMMON /NORMAL/ RNORM(3), FNORM(3), GNORM(3)
!C
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!
! EVALUATE THE NORM OF RNORM
!
      XNORM = SQRT(RNORM(1)**2 + RNORM(2)**2 + RNORM(3)**2)
!
! EVALUATE DGDX(I,J), I=1,3; J=1,3
!
      do I=1,3
        do IX=1,3
!
          DGDX(I,IX) = XNORM * (DRDX(I,IX))
!
        enddo
      enddo
!
! EVALUATE DGDX(I,J), I=1,3; J=4,12
!
      do I=1,3
         do IX = 4,12
!
           DGDX(I,IX) = RNH(I) * (RNORM(1) * DNDX(1,IX)  &
                               +  RNORM(2) * DNDX(2,IX)  &
                               +  RNORM(3) * DNDX(3,IX)) &
                               /  XNORM                  &
                               +  XNORM * (DRDX(I,IX))
!
         enddo
      enddo
!
      RETURN
      END SUBROUTINE DGDXSB

      SUBROUTINE DNDXSB
      use nh3, only :rnormi,rnormj,rnormk,rnorm,x,dndx,fnorm,gnorm,rnh
!       X(30),  &
!       DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),  &
!       DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),  &
!       DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12),  &
!       RNORM(3), FNORM(3), GNORM(3),  &
!       RNH(3), RHH(3), ALPHA(3), RKAPPA(3),  &
!       DELRNH(3), DELALP(3),  &
!       RNORMI, RNORMJ, RNORMK
      implicit none
      integer :: i
!
! THIS SUBROUTINE EVALUATES THE COMPONENTS OF THE
! VECTOR NORMAL TO THE H1-H-H PLANE AND THEIR
! DERIVATIVES W.R.T. THE CARTESIANS
!
!      COMMON /XANG/ X(30)
!C
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!C
!      COMMON /NORMAL/ RNORM(3), FNORM(3), GNORM(3)
!C
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!
! INITIALIZE RNORM
!
      RNORM(1) = RNORMI
      RNORM(2) = RNORMJ
      RNORM(3) = RNORMK
!
! EVALUATE THE NON-ZERO DERIVATIVES OF RNORM
!
      DNDX(1,5) = (X(9) - X(12))
      DNDX(2,4) = -DNDX(1,5)
!
      DNDX(1,6) = (X(11) - X(8))
      DNDX(3,4) = -DNDX(1,6)
!
      DNDX(1,8) = (X(12) - X(6))
      DNDX(2,7) = -DNDX(1,8)
!
      DNDX(1,9) = - (X(11) - X(5))
      DNDX(3,7) = - DNDX(1,9)
!
      DNDX(1,11) = - (X(9) - X(6))
      DNDX(2,10) = -DNDX(1,11)
!
      DNDX(1,12) = (X(8) - X(5))
      DNDX(3,10) = -DNDX(1,12)
!
      DNDX(2,6) = (X(7) - X(10))
      DNDX(3,5) = -DNDX(2,6)
!
      DNDX(2,9) = (X(10) - X(4))
      DNDX(3,8) = -DNDX(2,9)
!
      DNDX(2,12) = -(X(7) - X(4))
      DNDX(3,11) = -DNDX(2,12)
!
! EVALUATE FNORM(I)
!
      do i=1,3
!
        FNORM(I) = RNORM(1) * (X(1) - X(3*I + 1))  &
                 + RNORM(2) * (X(2) - X(3*I + 2))  &
                 + RNORM(3) * (X(3) - X(3*I + 3))
!
      enddo
!
! EVALUATE GNORM(I)
!
      do i=1,3
!
       GNORM(I) = RNH(I) * SQRT(RNORM(1)**2 + RNORM(2)**2 + RNORM(3)**2)
!
      enddo
!
      RETURN
      END SUBROUTINE DNDXSB
!
      SUBROUTINE DRDXSB
      use nh3, only : x,drdx,ij,ik,rhh,rnh
!                     X(30),   &  
!                     RNH(3), RHH(3), ALPHA(3), RKAPPA(3),  &
!                     DELRNH(3), DELALP(3),  &
!                     RNORMI, RNORMJ, RNORMK,   &
!                     DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),  &
!                     DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),  &
!                     DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12),  &
!                     IJ(3), IK(3)
      implicit none
      integer :: i, j, k, m
!
! THIS SUBROUTINE CALCULATES THE PARTIAL DERIVATIVES
! OF RNH(I) AND RHH(I) W.R.T. THE CARTESIAN COORDINATES
! X(J)
!
!      COMMON /XANG/ X(30)
!
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!
!      COMMON /INDICE/ IJ(3), IK(3)
!
! CALCULATE DRDX - R IS RNH
!
      do j=1,3
        do i=1,3
           DRDX(i,j) = -(X(3*i+j) - X(j)) / RNH(i)
           DRDX(i, 3*i + j) = -DRDX(i,j)
        enddo
      enddo
!
! CALCULATE DRDX - R IS RHH
!
      do i=1,3
        J = IJ(i)
        K = IK(i)
        do m=1,3
          DRDX(3+I,3*K+M) = (X(3*K+M) - X(3*J+M))/RHH(I)
          DRDX(3+I,3*J+M) = -DRDX(3+I,3*K+M)
        enddo
      enddo
!
      RETURN 
      END SUBROUTINE DRDXSB
!
      SUBROUTINE DVDINC
      use nh3, only : ij,ik,C2A,C2B,C2C,C2D,C2E,C2F,delrnh,delalp, &
                      C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H, SAIJ, SDR, &
                      DVDIC, RKAPPA, A, B, C, C4A, SDA2, DAIJ, DRIJ
!                     RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2,  &
!                     SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),  &
!                     C2A, C2B, C2C, C2D, C2E, C2F,  &
!                     C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,  &
!                     C4A, DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12), &
!                     DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),  &
!                     DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12),  &
!                     RNH(3), RHH(3), ALPHA(3), RKAPPA(3),  &
!                     DELRNH(3), DELALP(3),  &
!                     RNORMI, RNORMJ, RNORMK,  &
!                     IJ(3), IK(3)
      implicit none
      integer :: i, j, k
      double precision :: DV2(6), DV3(6), DV4(6), DKAPPA, RKSUM
!
! THIS SUBROUTINE CALCULATES THE NON-ZERO
! DERIVATIVES OF V(NH3) FROM WOLFSBERG
! W.R.T. THE INTERNAL COORDINATES:
! RNH(I), ALPHA(I), AND KAPPA(I)
!
! DVDIC(J), J=1,3 ARE DV/DRNH(I)
! DVDIC(J), J=4,6 ARE DV/DALPHA(I)
! DVDIC(J), J=7,9 ARE DV/DKAPPA(I)
!
!
!      COMMON / POTCON / RE, ALPHAE, A, B, C, SDR, SDR2, SDA, SDA2, 
!     *                  SRIJ, SAIJ, PRDR, PRDA, DAIJ(3), DRIJ(3),
!     *                  C2A, C2B, C2C, C2D, C2E, C2F,
!     *                  C3A, C3B, C3C, C3D, C3E, C3F, C3G, C3H,
!     *                  C4A
!C
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!C
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!C
!      COMMON /INDICE/ IJ(3), IK(3)
!C
!      DIMENSION DV2(6), DV3(6), DV4(6)
!
!
!  CALCULATE DVQUAD/DRNH AND DVQUAD/DALP
!
      do i=1,3
        J = IJ(I)
        K = IK(I)
!
        DV2(I) = 2.0D0 * C2A * DELRNH(I)  &
             + C2D * (DELRNH(J) + DELRNH(K))  &
             + C2E * DELALP(I)  &
             + C2F * (DELALP(J) + DELALP(K))  
!
        DV2(I+3) = 2.0D0 * C2B * DELALP(I)  &
               + C2C * (DELALP(J) + DELALP(K))  &
               + C2E * DELRNH(I)  &
               + C2F * (DELRNH(J) + DELRNH(K))  
!
      enddo
!
! CALCULATE DVCUBIC/DRNH AND DVCUBIC/DALP
!
      do i=1,3
        J = IJ(i)
        K = IK(i)
!
        DV3(I) = C3A * (DELRNH(J)**2 + DELRNH(K)**2  &
                   + 2.0D0*(DRIJ(J) + DRIJ(K)))  &
             + C3B * DRIJ(I)  &
             + 3.0D0 * C3C * DELRNH(I)**2  &
             + C3G * SDA2  +  C3H * SAIJ  
!
        DV3(I+3) = 3.0D0 * C3D * DELALP(I)**2  &
               + C3E * (DELALP(J)**2 + DELALP(K)**2  &
                     + 2.0D0*(DAIJ(J) + DAIJ(K)))  &
               + C3F * DAIJ(I)  &
               + 2.0D0 * C3G * DELALP(I) * SDR  &
               + C3H * (DELALP(J) + DELALP(K)) * SDR
!
      enddo
!
! CALCULATE DQUARTIC/DRNH
!
      do i=1,3
        DV4(I) = 4.0D0 * C4A * DELRNH(I)**3
      enddo
!
      do i=1,6
        DVDIC(I) = DV2(I) + DV3(I) + DV4(I)
      enddo
!
      RKSUM = RKAPPA(1) + RKAPPA(2) + RKAPPA(3)
      DKAPPA = (2.0D0*A*RKSUM - (2.0D0*C*B*RKSUM)  &
             * EXP((-C*RKSUM**2)/3.0D0))/3.0D0
!
      DVDIC(7) = DKAPPA
      DVDIC(8) = DKAPPA
      DVDIC(9) = DKAPPA
!
      RETURN
      END SUBROUTINE DVDINC
!
      SUBROUTINE DVMDXS
      use nh3, only : drdx,dadx,dvmdx,dvdic,dadr
!                     DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),  &
!                     DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),  &
!                     DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!
      implicit none
      double precision :: DICDX(6,12)
      integer :: i, j
!
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
! THIS SUBROUTINE CALCULATES THE PARTIAL DERIVATIVES
! OF VM W.R.T. THE CARTESIAN COORDINATES
!
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!C
!      DIMENSION DICDX(6,12)
!
! CALCULATE DADX
!
      CALL MULT(DADR, 3, 6, DRDX, 6, 12, DADX)
!
! MOVE DRDX AND DADX INTO ONE TEMP ARRAY DICDX
!
      do i=1,3
        do j=1,12
           DICDX(i,j) = DRDX(i,j)
           DICDX(i+3,j) = DADX(i,j)
        enddo
      enddo
!
! CALCULATE DVMDX
!
      CALL MULT(DVDIC, 1, 6, DICDX, 6, 12, DVMDX)
!
! CONVERT TO HARTREE
!
      do i=1,12
        DVMDX(I) = DVMDX(I) * 0.22937D0
      enddo
!
      RETURN 
      END SUBROUTINE DVMDXS
!
      SUBROUTINE DVSDXS
      use nh3, only : gnorm,dfdx,fnorm,dgdx,dudx,pid2,rkappa,dkdx,dvdic,dvsdx,dvdx,dvmdx
!                     DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),  &
!                     DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),  &
!                     DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12),  &
!                     RNORM(3), FNORM(3), GNORM(3),  &
!                     RT1D2, RT1D3, RT1D6, PI, PID2,  &
!                     RNH(3), RHH(3), ALPHA(3), RKAPPA(3),  &
!                     DELRNH(3), DELALP(3),  &
!                     RNORMI, RNORMJ, RNORMK

      implicit none
      double precision :: dkappa
      integer :: i, ix
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
! THIS SUBROUTINE EVALUATES THE PARTIAL DERIVATIVES OF
! U(FNORM,GNORM) W.R.T. THE CARTESIANS
!          AND
! THE PARTIAL DERIVATIVES OF KAPPA(I) W.R.T. THE CARTESIANS
!          AND
! THE PARTIAL DERIVATIVES OF VS W.R.T. THE CARTESIANS
!
!          AND ULTIMATELY
!
! THE PARTIAL DERIVATIVES OF VNH3 W.R.T. THE CARTESIANS
!
!      COMMON /DERIV/ DVDIC(9), DRDX(6,12), DADR(3,6), DUDX(3,12),
!     *               DADX(3,12), DVMDX(12), DNDX(3,12), DFDX(3,12),
!     *               DGDX(3,12), DKDX(3,12), DVSDX(12), DVDX(12)
!
!      COMMON /NORMAL/ RNORM(3), FNORM(3), GNORM(3)
!
!      COMMON /RTCON/ RT1D2, RT1D3, RT1D6, PI, PID2
!
!      COMMON /COORD/ RNH(3), RHH(3), ALPHA(3), RKAPPA(3),
!     *               DELRNH(3), DELALP(3),
!     *               RNORMI, RNORMJ, RNORMK
!
! EVALUATE DUDX
!
      do i=1,3
        do ix = 1,12
          DUDX(I,IX) = (GNORM(I) * DFDX(I,IX) - FNORM(I) * DGDX(I,IX)) / &
                    (GNORM(I))**2
        enddo
      enddo 
!
!
! EVALUATE DKDX
!
      do i=1,3
         do ix= 1,12
           DKDX(I,IX) = (1.0D0/SIN(PID2 - RKAPPA(I))) * DUDX(I,IX)
         enddo
      enddo
!
!
! EVALUATE DVSDX
!
      DKAPPA = DVDIC(7)
!
      do ix = 1,12
!
        DVSDX(IX) = DKAPPA * (DKDX(1,IX) + DKDX(2,IX) + DKDX(3,IX))
!
      enddo
!
! CONVERT TO HARTREE
!
      do ix = 1,12
        DVSDX(IX) = DVSDX(IX) * 2.2937D+10
      enddo
!
! EVALUATE DVDX
!
      do ix=1,12
        DVDX(IX) = DVMDX(IX) + DVSDX(IX)
      enddo
!
      RETURN
      END SUBROUTINE DVSDXS
