!
      SUBROUTINE prepot
      use potcm3
      use clhbr
!
!   System:           ClHBr
!   Functional form:  Extended LEPS (London-Erying-Polyani-Sato)
!
!   References for the potential parameters and the potential functional form:
!   Morse Parameters: D. J. Douglas, J. C. Polanyi, and J. J. Sloan
!                     Chem. Phys. 13, 15 (1976).
!   Sato Parameters:  V. K. Babamov, V. Lopez, and R. A. Marcus
!                     J. Chem. Phys. 78, 5621 (1983).
!   Functional form:  P. J. Kuntz, E. M. Nemth, J. C. Polanyi, S. D. Rosner,
!                     and C. E. Young 
!                     J. Chem. Phys. 44, 1168 (1966)
!
!   PREPOT must be called once before any calls to POT.
!   The potential parameters are included in DATA statements.
!   Coordinates, potential energy, and derivatives are passed 
!   through the common block POTCM:
!                  COMMON /POTCM/ R(3), ENERGY, DEDR(3)
!   The potential parameters are passed through the common blocks
!   SATOCM and LEPSCM.
!   All the information passed through the common blocks POTCM, SATOCM,
!   and LEPSCM are in hartree atomic units.  
!
!        This potential is written such that:
!                       R(1) = R(Cl-H)
!                       R(2) = R(H-Br)
!                       R(3) = R(Br-Cl)
!
!   The the flags that indicate what calculations should be carried out in 
!   the potential routine are passed through the common block POTCCM:
!                  /POTCCM/ NSURF, NDER, NDUM(8)
!   where:
!        NSURF - which electronic state should be used.
!                This option is not used for this potential as only the 
!                ground electronic state is available.
!        NDER  = 0 => no derivatives should be calculated
!        NDER  = 1 => calculate first derivatives
!        NDUM  - these 8 integer values can be used to flag options
!                within the potential; in this potential these options 
!                are not used.
!
        implicit none
        double precision, parameter :: CKCAL = 627.5095D0, CANGS = 0.529177106D0
        double precision :: DEI(3), REI(3), BETAI(3), ZI(3)
        integer :: i
!
!   Data statements for the potential parameters; the energy parameters are in
!   kcal/mol, and the lengths are in Angstroms.
!
         DATA DEI/ 106.4D0, 90.24D0, 52.09D0/
         DATA REI/ 1.275D0, 1.414D0, 2.136D0/
         DATA BETAI/1.867D0, 1.851D0, 1.923D0/
         DATA ZI/0.02D0, 0.02D0, 0.0D0/
!
!   Set the flags for the type of calculations to be carried out by the 
!   potential.
!        
         NDER = 1
         WRITE (6, 100) DEI, REI, BETAI, ZI
!
100   FORMAT (/, 2X, T5, 'PREPOT has been called for Cl + HBr', &
              /, 2X, T5, 'Extended LEPS function', &
              //, 2X, T5, 'Potential energy surface parameters:', &
              /, 2X, T5, 'Bond', T47, 'Cl-H', T58, 'H-Br', T69, 'Br-Cl',   &
              /, 2X, T5, 'Dissociation energies (kcal/mol):',  &
              T44, F10.5, T55, F10.5, T66, F10.5, &
              /, 2X, T5, 'Equilibrium bond lengths (Angstroms):',  &
              T44, F10.5, T55, F10.5, T66, F10.5, &
              /, 2X, T5, 'Morse beta parameters (Angstroms**-1):',  &
              T44, F10.5, T55, F10.5, T66, F10.5, &
              /, 2X, T5, 'Sato parameters:',  &
              T44, F10.5, T55, F10.5, T66, F10.5,/)
!
      DO I = 1,3
             Z(I)    = ZI(I)
!   Convert to atomic units
             D(I)    = DEI(I)/CKCAL
             RE(I)   = REI(I)/CANGS
             BETA(I) = BETAI(I)*CANGS
!   Compute useful constants
             ZPO(I)   = 1.0D0 + Z(I)
             OP3Z(I)  = 1.0D0 + 3.0D0*Z(I)
             TOP3Z(I) = 2.0D0*OP3Z(I)
             ZP3(I)   = Z(I) + 3.0D0
             TZP3(I)  = 2.0D0*ZP3(I)
             DO4Z(I)  = D(I)/4.0D0/ZPO(I)
             B(I)     = BETA(I)*DO4Z(I)*2.0D0
      ENDDO
!
      RETURN
      END subroutine prepot
!*****
!
         SUBROUTINE pot
         use potcm3
         use clhbr
!
!   System:          ABC
!   Functional form: Extended LEPS (London-Erying-Polyani-Sato)
!   References:      P. J. Kuntz, E. M. Nemth, J. C. Polanyi, S. D. Rosner,
!                    and C. E. Young 
!                    J. Chem. Phys. 44, 1168 (1966)
!
!        PREPOT must be called once before any calls to POT.
!        The potential parameters are initialized in the subprogram PREPOT.
!        Coordinates, potential energy, and derivatives are passed 
!        through the common block POTCM:
!                  COMMON /POTCM/ R(3), ENERGY, DEDR(3)
!        The constants for the potential are passed through the common blocks
!        SATOCM and LEPSCM.
!        All the information passed through the common blocks POTCM, SATOCM, 
!        and LEPSCM are in hartree atomic units.  
!
!        For the reaction: A + BC -> AB + C we write:
!                          R(1) = R(A-B)
!                          R(2) = R(B-C)
!                          R(3) = R(C-A)
!
!   NOTE: The potential energy at the reactant asympotote is not equal to
!         zero, it is set equal to the energy of the reactant diatomic.
!
!   The the flags that indicate what calculations should be carried out in 
!   the potential routine are passed through the common block POTCCM:
!                  /POTCCM/ NSURF, NDER, NDUM(8)
!   where:
!        NSURF - which electronic surface should be used.
!                In this potential this option is not used; the ground
!                electronic state is the only one available.
!        NDER  = 0 => no derivatives should be calculated
!        NDER  = 1 => calculate first derivatives
!        NDUM  - these 8 integer values can be used to flag options
!                within the potential; in this potential these options 
!                are not used.
!
         implicit none
         double precision :: x(3), coul(3), exch(3)
         double precision,parameter :: r2 =  1.41421356D0
         double precision :: rad, s
         integer :: i
!        COMMON /POTCM/R(3), ENERGY, DEDR(3)
!        COMMON /POTCCM/ NSURF, NDER, NDUM(8)
!        COMMON /SATOCM/ D(3), RE(3), BETA(3), Z(3) 
!        COMMON /LEPSCM/ ZPO(3), OP3Z(3), ZP3(3), TZP3(3), TOP3Z(3), 
!    *                   DO4Z(3), B(3)
!
!   Initialize the variable used in the calculation of the energy.
!
         ENERGY = 0.D0
!
!   Compute the energy.
!
         do i = 1, 3
               X(I)    = EXP(-BETA(I)*(R(I)-RE(I)))
               COUL(I) = DO4Z(I)*(ZP3(I)*X(I)-TOP3Z(I))*X(I)
               EXCH(I) = DO4Z(I)*(OP3Z(I)*X(I)-TZP3(I))*X(I)
               ENERGY  = ENERGY + COUL(I)
         enddo
!
         RAD = SQRT((EXCH(1)-EXCH(2))**2 + (EXCH(2)-EXCH(3))**2 +   &
                    (EXCH(3)-EXCH(1))**2)
!
         ENERGY = ENERGY - RAD/R2 
!
!   Compute the derivatives of the energy with respect to the internal
!   coordinates.
!
         IF (NDER .EQ. 1) THEN
             S = EXCH(1) + EXCH(2) + EXCH(3)
             do i = 1,3
                   DEDR(I) = B(I)*X(I)*((3.0D0*EXCH(I)-S)/R2*  &
                             (OP3Z(I)*X(I)-ZP3(I))/RAD-  &
                             ZP3(I)*X(I)+OP3Z(I))  
             enddo 
         ENDIF
!
      RETURN
      END subroutine pot
!*****
