!**************************************************************************

      SUBROUTINE SURF(V, COORD, DX, N3TM)
!
!   System:    CH4+O based on functional forms by
!              Jordan and Gilbert, JCP, 102, 5669 (1995).
!              JEG, March 1996.
!
!   SETUP must be called once before any calls to SURF.
!   The cartesian coordinates, potential energy, and derivatives of the energy
!   with respect to the cartesian coordinates are passed by the calling  
!   program in the argument list as follows:
!        CALL SURF (V, X, DX, N3TM)
!   where X and DX are arrays dimensioned N3TM, and N3TM must be greater
!   than or equal to 18 (3*number of atoms).  
!   All the information passed to and from the potential energy surface 
!   routine is in hartree atomic units.  
!
!        This potential is written such that:
!                       X(1)  - X(3)  : X, Y, Z for H1
!                       X(4)  - X(6)  : X, Y, Z for C 
!                       X(7)  - X(9)  : X, Y, Z for H3
!                       X(10) - X(12) : X, Y, Z for H4
!                       X(13) - X(15) : X, Y, Z for H2
!                       X(16) - X(18) : X, Y, Z for  O
!
      use ch4oinc
      use ch4oinc1
      implicit none
      integer, intent(in) :: n3tm
      double precision, intent(in) :: coord(n3tm)
      double precision, intent(out) :: v, dx(n3tm)
      double precision :: fact1, fact2, en, vstr, vop, vip, aa5, r0cb, rphi
      integer :: i, ind, icount, n
      character(len=79) :: comlin
!
!     PUT COORDINATES IN PROPER ARRAYS
!
! Initialization 
        DO i=1, 150
          q(i)=0.D0
          pdot(i)=0.D0
        ENDDO
!
! Changing to angstroms
! 
      DO i = 1, 18
         q(i) = COORD(i)*0.52918d0
      ENDDO
!
!  Reading the parameters for a leps-type pes for the
!  reaction ch4+o -> ch3 + oh from the input file CONST
!
!  The inputed constants are through FORMATTED read statements
!
       open(unit=4,file='potch4o.dat',status='old')
!
!  read in indices for the carbon and hydrogen atoms
!
!
       read(4,80)comlin
       read(4,80)comlin
!
       read(4,80)comlin
       read(4,910)nnc,nnb,(nnh(i),i=1,4)
!
80     format(a79)
910    format(6i4)
!
!  calculate indexes for coordinates
!
       do ind=1,3
         icount=ind-3
         nc(ind)=3*nnc+icount
         nhb(ind)=3*nnb+icount
         do i=1,4
           nh(i,ind)=3*nnh(i)+icount
         enddo
       enddo
!
!  read in parameters for the stretching term
!
       read(4,80)comlin
       read(4,930)r0ch,d1ch,d3ch
!
       read(4,80)comlin
       read(4,930)a1ch,b1ch,c1ch
!
       read(4,80)comlin
       read(4,930)r0hh,d1hh,d3hh,ahh
!
       read(4,80)comlin
       read(4,930)r0cb,d1cb,d3cb,acb
!
930    format(8f10.5)
!
!  read in parameters for the out of plane bending term
!
       read(4,80)comlin
       read(4,930)a3s,b3s,aphi,bphi,cphi,rphi
!
       read(4,80)comlin
       read(4,930)atheta,btheta,ctheta
!
       read(4,80)comlin
       read(4,930)fch3,hch3
!
!
!  read in parameters for the in plane bending term
!
       read(4,80)comlin
       read(4,930)fkinf,ak,bk,aa1,aa2,aa3,aa4,aa5
!
!  convert to integration units:(Valores en la superficie de Gilbert)
!
!  kcal/mol
!  mdyn A-1        -> 1.0d+05 j/mol...
!  mdyn A rad-1
!
!  NB integration units are:
!
!  energy   in 1.0d+05 j/mol
!  time     in 1.0d-14 s
!  distance in 1.0d-10 m
!  angles   in radians
!  mass     in amu
!
       fact1=0.041840d0
       fact2=6.022045d0
! 
       d1ch=d1ch*fact1
       d3ch=d3ch*fact1
       d1cb=d1cb*fact1
       d3cb=d3cb*fact1
       d1hh=d1hh*fact1
       d3hh=d3hh*fact1
       fch3=fch3*fact2
       hch3=hch3*fact2
       fkinf=fkinf*fact2
       ak=ak*fact2
!
       close(unit=4)
!
!
!  calculate relative coordinates and bond lengths
!
       en=0.0d0

       call coorden
!
!  calculate switching functions
!
       call switchf
!
!  calculate reference angles and their derivatives
!
       call refangles
!
!  calculate stretching potential
!
       call stretch(vstr)
!
!  calculate out of plane bending potential
!
       call opbend(vop)
!
!  calculate in plane bending potential
!
       call ipbend(vip)
!
!  nb: total potential energy is vstr+vop+vip
!
       en=vstr+vop+vip
!
! changing from 10(5) j/mol to au
!
       en = en*0.03812D0
       V = en 
!
!  copy the pdots to the
!  appropriate elements in dv. dx en POLYRATE.
!
       do n=1,N3TM  
           DX(n)=0.0d0
       enddo
!
! Reordering and transformation from 10(5)j/mol/A 
! to au
! 
       do ind=1,3
          DX(ind)=pdot(nh(4,ind))*0.0201723d0
       enddo
       do ind=1,3
          DX(ind+3)=pdot(nc(ind))*0.0201723d0
       enddo
       do ind=1,3
          DX(ind+6)=pdot(nh(1,ind))*0.0201723d0
       enddo
       do ind=1,3
          DX(ind+9)=pdot(nh(2,ind))*0.0201723d0
       enddo
       do ind=1,3
          DX(ind+12)=pdot(nh(3,ind))*0.0201723d0
       enddo
       do ind=1,3
          DX(ind+15)=pdot(nhb(ind))*0.0201723d0
       enddo
!
       return
       end
!
!******************************************************
!
       subroutine coorden
!
!  calculates relative coordinates and bond lengths
!
       use ch4oinc
       implicit none
       integer :: i,ind
!       implicit double precision (a-h,o-z)
!
!  calculate relative coordinates
!
       do ind=1,3
         tcb(ind)=q(nc(ind))-q(nhb(ind))
         do i=1,4
           tch(i,ind)=q(nc(ind))-q(nh(i,ind))
           tbh(i,ind)=q(nhb(ind))-q(nh(i,ind))
         enddo
       enddo
!
!  calculate bond lengths
!
       rcb=sqrt(tcb(1)*tcb(1)+tcb(2)*tcb(2)+tcb(3)*tcb(3))
       do i=1,4
         rch(i)=sqrt(tch(i,1)*tch(i,1)+tch(i,2)*tch(i,2)+ &
                      tch(i,3)*tch(i,3))
         rbh(i)=sqrt(tbh(i,1)*tbh(i,1)+tbh(i,2)*tbh(i,2)+ &
                      tbh(i,3)*tbh(i,3))
!      write (*,*) 'rch(i) y rbh(i) = ',rch(i),rbh(i)
       enddo
       return
       end
!
!******************************************************
!
!
       subroutine refangles
!
!  subroutine calculates reference angles for the "in-plane" potential
!
       use ch4oinc
       implicit none
       double precision :: tau, pi, halfpi, twopi
       integer :: i,j,k
!       implicit double precision (a-h,o-z)
!
       tau=acos(-1.0d0/3.0d0)
       pi=4.0d0*atan(1.0d0)
       halfpi=0.5d0*pi
       twopi=2.0d0*pi
!
!  set diagonal elements to zero
!
       do i=1,4
         theta0(i,i)=0.0d0
         do k=1,4
           dtheta0(i,i,k)=0.0d0
         enddo
       enddo
!
!  calculate reference angles
!
       theta0(1,2)=tau+(tau-halfpi)*(sphi(1)*sphi(2)-1.0d0) &
                   +(tau-twopi/3.0d0)*(stheta(3)*stheta(4)-1.0d0)
       theta0(1,3)=tau+(tau-halfpi)*(sphi(1)*sphi(3)-1.0d0) &
                   +(tau-twopi/3.0d0)*(stheta(2)*stheta(4)-1.0d0)
       theta0(1,4)=tau+(tau-halfpi)*(sphi(1)*sphi(4)-1.0d0) &
                   +(tau-twopi/3.0d0)*(stheta(2)*stheta(3)-1.0d0)
       theta0(2,3)=tau+(tau-halfpi)*(sphi(2)*sphi(3)-1.0d0) &
                   +(tau-twopi/3.0d0)*(stheta(1)*stheta(4)-1.0d0)
       theta0(2,4)=tau+(tau-halfpi)*(sphi(2)*sphi(4)-1.0d0) &
                   +(tau-twopi/3.0d0)*(stheta(1)*stheta(3)-1.0d0)
       theta0(3,4)=tau+(tau-halfpi)*(sphi(3)*sphi(4)-1.0d0) &
                   +(tau-twopi/3.0d0)*(stheta(1)*stheta(2)-1.0d0)
!
!  calculate the derivatives of theta0(i,j) in terms of rch(k)
!  quantity calulated is dtheta0(i,j,k)
!
!  derivatives wrt rch(1)
!
       dtheta0(1,2,1)=(tau-halfpi)*dsphi(1)*sphi(2)
       dtheta0(1,3,1)=(tau-halfpi)*dsphi(1)*sphi(3)
       dtheta0(1,4,1)=(tau-halfpi)*dsphi(1)*sphi(4)
       dtheta0(2,3,1)=(tau-twopi/3.0d0)*dstheta(1)*stheta(4)
       dtheta0(2,4,1)=(tau-twopi/3.0d0)*dstheta(1)*stheta(3)
       dtheta0(3,4,1)=(tau-twopi/3.0d0)*dstheta(1)*stheta(2)
!
!  derivatives wrt rch(2)
!
       dtheta0(1,2,2)=(tau-halfpi)*sphi(1)*dsphi(2)
       dtheta0(1,3,2)=(tau-twopi/3.0d0)*dstheta(2)*stheta(4)
       dtheta0(1,4,2)=(tau-twopi/3.0d0)*dstheta(2)*stheta(3)
       dtheta0(2,3,2)=(tau-halfpi)*dsphi(2)*sphi(3)
       dtheta0(2,4,2)=(tau-halfpi)*dsphi(2)*sphi(4)
       dtheta0(3,4,2)=(tau-twopi/3.0d0)*stheta(1)*dstheta(2)
!
!  derivatives wrt rch(3)
!
       dtheta0(1,2,3)=(tau-twopi/3.0d0)*dstheta(3)*stheta(4)
       dtheta0(1,3,3)=(tau-halfpi)*sphi(1)*dsphi(3)
       dtheta0(1,4,3)=(tau-twopi/3.0d0)*stheta(2)*dstheta(3)
       dtheta0(2,3,3)=(tau-halfpi)*sphi(2)*dsphi(3)
       dtheta0(2,4,3)=(tau-twopi/3.0d0)*stheta(1)*dstheta(3)
       dtheta0(3,4,3)=(tau-halfpi)*dsphi(3)*sphi(4)
!
!  derivatives wrt rch(4)
!
       dtheta0(1,2,4)=(tau-twopi/3.0d0)*stheta(3)*dstheta(4)
       dtheta0(1,3,4)=(tau-twopi/3.0d0)*stheta(2)*dstheta(4)
       dtheta0(1,4,4)=(tau-halfpi)*sphi(1)*dsphi(4)
       dtheta0(2,3,4)=(tau-twopi/3.0d0)*stheta(1)*dstheta(4)
       dtheta0(2,4,4)=(tau-halfpi)*sphi(2)*dsphi(4)
       dtheta0(3,4,4)=(tau-halfpi)*sphi(3)*dsphi(4)
!
!  fill in the other half of the matrix
!
        do i=1,3
          do j=i+1,4
            theta0(j,i)=theta0(i,j)
            do k=1,4
              dtheta0(j,i,k)=dtheta0(i,j,k)
            enddo
          enddo
        enddo
       return
       end
!
!******************************************************
!
!
       subroutine stretch(vstr)
!
!  subroutine to calculate leps-type stretching potential and its
!  derivatives
!
       use ch4oinc
       implicit none
       double precision, intent(out) :: vstr
       double precision :: vqch(4),vjch(4),vqbh(4),vjbh(4),vq(4),vj(4)
       double precision :: achdc(3),achdh(4,3), dumqcb
       double precision :: rav, arga, e1, e3, factj, ach
       double precision :: vqcb
       double precision :: vjcb
       double precision :: dumach 
       double precision :: dumqbh 
       double precision :: dumqch
       double precision :: dumjcb
       double precision :: dumjbh
       double precision :: dumjch
       double precision :: dumjhi
       double precision :: dumjhj
       double precision :: dumqhi
       double precision :: dumqhj
       integer :: i, j, k, ind
!        
!       implicit double precision (a-h,o-z)
!
!       dimension vqch(4),vjch(4),vqbh(4),vjbh(4),vq(4),vj(4),
!     *           achdc(3),achdh(4,3)
!
!  calculate avergage bond length for the methane moiety
!
       rav=(rch(1)+rch(2)+rch(3)+rch(4))/4.0d0
!
!  initialise:
!
       vstr=0.0d0
!
!  ach:
!
!  nb: in double precision tanh(19.0d0)=1.0d0 and we put the if statement
!  in to avoid overflow/underflow errors
!
       arga=c1ch*(rav-r0ch)
       if(arga.lt.19.0d0)then
         ach=a1ch+b1ch*(tanh(arga)+1.0d0)*0.5d0
         dumach=b1ch*c1ch/(2.0d0*cosh(arga)**2)
       else
         ach=a1ch+b1ch
         dumach=0.0d0
       endif
!
!  calculate singlet: e1, triplet: e3 energies and vq and vj
!  terms for each bond
!
       e1=d1cb*(exp(-2.0d0*acb*(rcb-r0ch))-2.0d0*exp(-acb*(rcb-r0ch)))
       e3=d3cb*(exp(-2.0d0*acb*(rcb-r0ch))+2.0d0*exp(-acb*(rcb-r0ch)))
       vqcb=(e1+e3)*0.5d0
       vjcb=(e1-e3)*0.5d0
       do i=1,4
         e1=d1ch*(exp(-2.0d0*ach*(rch(i)-r0ch))  &
                    -2.0d0*exp(-ach*(rch(i)-r0ch)))
         e3=d3ch*(exp(-2.0d0*ach*(rch(i)-r0ch))  &
                    +2.0d0*exp(-ach*(rch(i)-r0ch)))
         vqch(i)=(e1+e3)*0.5d0
         vjch(i)=(e1-e3)*0.5d0
         e1=d1hh*(exp(-2.0d0*ahh*(rbh(i)-r0hh))  &
                    -2.0d0*exp(-ahh*(rbh(i)-r0hh)))
         e3=d3hh*(exp(-2.0d0*ahh*(rbh(i)-r0hh))  &
                    +2.0d0*exp(-ahh*(rbh(i)-r0hh)))
         vqbh(i)=(e1+e3)*0.5d0
         vjbh(i)=(e1-e3)*0.5d0
!
!  calculate 3 body potential
!
         vq(i)=vqch(i)+vqcb+vqbh(i)
         vj(i)=-sqrt(((vjch(i)-vjcb)**2+(vjcb-vjbh(i))**2  &
                       +(vjbh(i)-vjch(i))**2)*0.5d0)
         vstr=vstr+vq(i)+vj(i)
       enddo
!
!  partial derivatives
!  first we need the derivative of ach:
!
       do ind=1,3
         achdc(ind)=dumach*(tch(1,ind)/rch(1)+tch(2,ind)/rch(2)  &
                  +tch(3,ind)/rch(3)+tch(4,ind)/rch(4))/4.0d0
         do i=1,4
           achdh(i,ind)=-dumach*tch(i,ind)/rch(i)/4.0d0
         enddo
       enddo
       dumqcb=-acb*((d1cb+d3cb)*exp(-2.0d0*acb*(rcb-r0ch))-  &
               (d1cb-d3cb)*exp(-acb*(rcb-r0ch)))/rcb
!
!  calculate cartesian derivatives:
!  looping over ch(i) and bh(i)
!
       do i=1,4
         dumqbh=-ahh*((d1hh+d3hh)*exp(-2.0d0*ahh*(rbh(i)-r0hh))-  &
                 (d1hh-d3hh)*exp(-ahh*(rbh(i)-r0hh)))/rbh(i)
         factj=0.5d0/vj(i)
         dumjcb=-acb*((d1cb-d3cb)*exp(-2.0d0*acb*(rcb-r0ch))  &
                  -(d1cb+d3cb)*exp(-acb*(rcb-r0ch)))*factj/rcb
         dumjbh=-ahh*((d1hh-d3hh)*exp(-2.0d0*ahh*(rbh(i)-r0hh))  &
                  -(d1hh+d3hh)*exp(-ahh*(rbh(i)-r0hh)))*factj/rbh(i)
         do ind=1,3
!
!  deriv wrt hb:
!
                  pdot(nhb(ind))=pdot(nhb(ind))  &
                   -tcb(ind)*dumqcb+tbh(i,ind)*dumqbh  &
                  +(vjch(i)-vjcb)*(dumjcb*tcb(ind))  &
                  +(vjcb-vjbh(i))*(-dumjcb*tcb(ind)-dumjbh*tbh(i,ind)) &
                  +(vjbh(i)-vjch(i))*dumjbh*tbh(i,ind)
!
!  dvqch(i)/dc
!
           dumqch=-(ach*tch(i,ind)/rch(i)+achdc(ind)*(rch(i)-r0ch)) &
                    *((d1ch+d3ch)*exp(-2.0d0*ach*(rch(i)-r0ch)) &
                       -(d1ch-d3ch)*exp(-ach*(rch(i)-r0ch))) 
               pdot(nc(ind))=pdot(nc(ind))+dumqch+tcb(ind)*dumqcb
!
!  dvqch(i)/dh(i)
!
           dumqhi=(ach*tch(i,ind)/rch(i)-achdh(i,ind)*(rch(i)-r0ch)) &
                    *((d1ch+d3ch)*exp(-2.0d0*ach*(rch(i)-r0ch)) &
                       -(d1ch-d3ch)*exp(-ach*(rch(i)-r0ch))) 
              pdot(nh(i,ind))=pdot(nh(i,ind))+dumqhi-tbh(i,ind)*dumqbh
!
!  dvjch(i)/dc
!
           dumjch=-(ach*tch(i,ind)/rch(i)+achdc(ind)*(rch(i)-r0ch)) &
                    *((d1ch-d3ch)*exp(-2.0d0*ach*(rch(i)-r0ch)) &
                     -(d1ch+d3ch)*exp(-ach*(rch(i)-r0ch)))*factj
!
!  dvj(i)/dnc(ind)
!
           pdot(nc(ind))=pdot(nc(ind)) &
                  +(vjch(i)-vjcb)*(dumjch-dumjcb*tcb(ind)) &
                  +(vjcb-vjbh(i))*dumjcb*tcb(ind) &
                  -(vjbh(i)-vjch(i))*dumjch
!
!  dvjch(i)/dh(i)
!
           dumjhi=(ach*tch(i,ind)/rch(i)-achdh(i,ind)*(rch(i)-r0ch)) &
                    *((d1ch-d3ch)*exp(-2.0d0*ach*(rch(i)-r0ch)) &
                     -(d1ch+d3ch)*exp(-ach*(rch(i)-r0ch)))*factj
!
!  dvj(i)/dnh(i,ind)
!
            pdot(nh(i,ind))=pdot(nh(i,ind)) &
                  +(vjch(i)-vjcb)*dumjhi &
                  +(vjcb-vjbh(i))*dumjbh*tbh(i,ind) &
                  +(vjbh(i)-vjch(i))*(-dumjbh*tbh(i,ind)-dumjhi)
!
!  dv(i)/dh(j)
!
           do k=1,3
             j=i+k
             if(j.gt.4)j=j-4
             dumqhj=-achdh(j,ind)*(rch(i)-r0ch) &
                       *((d1ch+d3ch)*exp(-2.0d0*ach*(rch(i)-r0ch)) &
                          -(d1ch-d3ch)*exp(-ach*(rch(i)-r0ch)))
             dumjhj=-achdh(j,ind)*(rch(i)-r0ch) &
                       *((d1ch-d3ch)*exp(-2.0d0*ach*(rch(i)-r0ch)) &
                        -(d1ch+d3ch)*exp(-ach*(rch(i)-r0ch)))*factj
             pdot(nh(j,ind))=pdot(nh(j,ind))+dumqhj &
                  +(vjch(i)-vjcb)*dumjhj &
                  -(vjbh(i)-vjch(i))*dumjhj
           enddo
         enddo
       enddo
       return
       end
!
!******************************************************
!
!
       subroutine opbend(vop)
!
!  subroutine calculates symmetrized vop potential and derivatives
!
       use ch4oinc
       implicit none
       double precision, intent(out) :: vop
       integer :: i,j,k,l,ind,itemp
       double precision :: sum2,sum4,ddr
       double precision :: sumd2(4),sumd4(4)
!       implicit double precision (a-h,o-z)
!
!       dimension sumd2(4),sumd4(4)
!
!
       vop=0.0d0
!
!  calculate force constants and their derivatives
!
       call opforce
!
!  calculate out-of-plane angle and derivatives
!
       do i=1,4
         j=i+1
         if(j.gt.4)j=j-4
         k=j+1
         if(k.gt.4)k=k-4
         l=k+1
         if(l.gt.4)l=l-4
!
!  if i is an even number then we have switched the vectors
!  from a right handed set to a left handed set
!
!  in this case we need to switch vectors k and l around
!
         if((i.eq.2).or.(i.eq.4))then
           itemp=k
           k=l
           l=itemp
         endif
!
!  subroutine performs sum over j, k, l
!  sum2 = sum delta**2
!  sum4 = sum delta**4
!
         call calcdelta(i,j,k,l,sum2,sum4)
         sumd2(i)=sum2
         sumd4(i)=sum4
         vop=vop+fdelta(i)*sumd2(i)+hdelta(i)*sumd4(i)
       enddo
       do i=1,4
         do j=1,4
!
!  overall derivatives of force constants i wrt the bond-length rch(j)
!
           ddr=dfdelta(i,j)*sumd2(i)+dhdelta(i,j)*sumd4(i)
!
!  calculate derivatives in terms of cartesian coordinates:
!
           do ind=1,3
             pdot(nh(j,ind))=pdot(nh(j,ind))-tch(j,ind)*ddr/rch(j)
             pdot(nc(ind))=pdot(nc(ind))+tch(j,ind)*ddr/rch(j)
           enddo
         enddo
       enddo
       return
       end
!
!******************************************************
!
!
       subroutine ipbend(vip)
!
!  subroutine calculates symmetrised in plane bend term
!  and its derivatives
!
       use ch4oinc
       implicit none
       integer :: i,j,k,ind
       double precision,intent(out) :: vip 
       double precision :: costh(4,4),theta(4,4),dth(4,4)
       double precision :: dth0k
       double precision :: dth0c
       double precision :: termth
       double precision :: dthi
       double precision :: dthj
       double precision :: dthc
!       implicit double precision (a-h,o-z)
!
!       dimension costh(4,4),theta(4,4),dth(4,4)
!
!  initialise
!
       vip=0.0d0
!
!  calculate force constants: fk0(i,j), f1(i)
!  and derivatives wrt rch(k) and rbh(k): dfdc(i,j,k), dfdh(i,j,k)
!
       call ipforce
!
!  calculate theta(i,j) and in plane bend potential
!
       do i=1,3
         do j=i+1,4
           costh(i,j)=tch(i,1)*tch(j,1)+tch(i,2)*tch(j,2) &
                             +tch(i,3)*tch(j,3)
           costh(i,j)=costh(i,j)/rch(i)/rch(j)
           theta(i,j)=acos(costh(i,j))
           dth(i,j)=theta(i,j)-theta0(i,j)
           vip=vip+0.5d0*fk0(i,j)*f1(i)*f1(j)*dth(i,j)**2
!
!  calculate partial derivatives wrt cartesian coordinates
!
!  calculate pdots wrt theta:
!
           termth=-1.0d0/sqrt(1.0d0-costh(i,j)*costh(i,j))
           do ind=1,3
             dthi=-tch(j,ind)/rch(i)/rch(j) &
                        +costh(i,j)*tch(i,ind)/rch(i)/rch(i)
             dthi=dthi*termth
             dthj=-tch(i,ind)/rch(i)/rch(j) &
                        +costh(i,j)*tch(j,ind)/rch(j)/rch(j)
             dthj=dthj*termth
             dthc=-(dthi+dthj)
             pdot(nh(i,ind))=pdot(nh(i,ind)) &
                             +fk0(i,j)*f1(i)*f1(j)*dthi*dth(i,j)
             pdot(nh(j,ind))=pdot(nh(j,ind)) &
                             +fk0(i,j)*f1(i)*f1(j)*dthj*dth(i,j)
             pdot(nc(ind))=pdot(nc(ind)) &
                             +fk0(i,j)*f1(i)*f1(j)*dthc*dth(i,j)
             do k=1,4
!
!  calculate pdots wrt force constants and wrt theta0
!
               dth0k=-dtheta0(i,j,k)*tch(k,ind)/rch(k)
               dth0c=-dth0k
               pdot(nh(k,ind))=pdot(nh(k,ind))  &
                      -0.5d0*tch(k,ind)*dfdc(i,j,k)*dth(i,j)**2/rch(k) &
                      -0.5d0*tbh(k,ind)*dfdh(i,j,k)*dth(i,j)**2/rbh(k) &
                            -fk0(i,j)*f1(i)*f1(j)*dth0k*dth(i,j)
               pdot(nc(ind))=pdot(nc(ind))  &
                      +0.5d0*tch(k,ind)*dfdc(i,j,k)*dth(i,j)**2/rch(k) &
                            -fk0(i,j)*f1(i)*f1(j)*dth0c*dth(i,j)
               pdot(nhb(ind))=pdot(nhb(ind))  &
                      +0.5d0*tbh(k,ind)*dfdh(i,j,k)*dth(i,j)**2/rbh(k)
             enddo
           enddo
         enddo
       enddo
       return
       end
!
!*************************************************************************
!
       subroutine calcdelta(i,j,k,l,sum2,sum4)
!
!  subroutine calculates out of plane angle delta, loops
!  through delta(i,j), delta(i,k), delta(i,l)
!
!   also calculates the derivatives wrt delta
!
       use ch4oinc
       implicit none
       integer, intent(in) :: i,j,k,l
       double precision, intent(out) :: sum2, sum4 
       double precision :: norma, deldot, atemp1, atemp3, atemp4, atemp5
       double precision :: delta(4),a(3),b(3),axb(3),c(4,3)
       integer :: in(3)
       double precision :: argd(4),daxb(4,3,3),cdot(4,3,3),atemp2(3)
       integer :: ii,jj,ind
!       implicit double precision (a-h,o-z)
!       double precision norma
!
!       dimension  delta(4),in(3),a(3),b(3),axb(3),c(4,3),argd(4),
!     *            daxb(4,3,3),cdot(4,3,3),atemp2(3)
!
!  initialise
!
       sum2=0.0d0
       sum4=0.0d0
!
!  set j,k,l indices
!
       in(1)=j
       in(2)=k
       in(3)=l
!
!  vector a is rk-rj, vector b is rl-rj
!
       do ind=1,3
         a(ind)=q(nh(k,ind))-q(nh(j,ind))
         b(ind)=q(nh(l,ind))-q(nh(j,ind))
       enddo
!
!  axb is vector a cross b
!
       axb(1)=a(2)*b(3)-a(3)*b(2)
       axb(2)=a(3)*b(1)-a(1)*b(3)
       axb(3)=a(1)*b(2)-a(2)*b(1)
       norma=axb(1)*axb(1)+axb(2)*axb(2)+axb(3)*axb(3)
       norma=sqrt(norma)
!
!  c is position vector of h(ii): calculate c(j),c(k),c(l)
!
       do ii=1,3
         do ind=1,3
           c(in(ii),ind)=-tch(in(ii),ind)/rch(in(ii))
         enddo
       enddo
!
!  argd is the dot product axb dot c
!
       do ii=1,3
         argd(in(ii))=axb(1)*c(in(ii),1)+axb(2)*c(in(ii),2) &
                                      +axb(3)*c(in(ii),3)
         argd(in(ii))=argd(in(ii))/norma
         delta(in(ii))=acos(argd(in(ii)))-theta0(i,in(ii))
!        write(*,*) 'theta,delta',theta0(i,in(ii)),delta(in(ii))
         sum2=sum2+delta(in(ii))**2
         sum4=sum4+delta(in(ii))**4
       enddo
!
!  derivatives of axb wrt hj:
!
       daxb(j,1,1)=0.0d0
       daxb(j,1,2)=b(3)-a(3)
       daxb(j,1,3)=-b(2)+a(2)
       daxb(j,2,1)=-b(3)+a(3)
       daxb(j,2,2)=0.0d0
       daxb(j,2,3)=b(1)-a(1)
       daxb(j,3,1)=b(2)-a(2)
       daxb(j,3,2)=-b(1)+a(1)
       daxb(j,3,3)=0.0d0
!
!  derivatives of axb wrt hk:
!
       daxb(k,1,1)=0.0d0
       daxb(k,1,2)=-b(3)
       daxb(k,1,3)=b(2)
       daxb(k,2,1)=b(3)
       daxb(k,2,2)=0.0d0
       daxb(k,2,3)=-b(1)
       daxb(k,3,1)=-b(2)
       daxb(k,3,2)=b(1)
       daxb(k,3,3)=0.0d0
!
!  derivatives of axb wrt hl:
!
       daxb(l,1,1)=0.0d0
       daxb(l,1,2)=a(3)
       daxb(l,1,3)=-a(2)
       daxb(l,2,1)=-a(3)
       daxb(l,2,2)=0.0d0
       daxb(l,2,3)=a(1)
       daxb(l,3,1)=a(2)
       daxb(l,3,2)=-a(1)
       daxb(l,3,3)=0.0d0
!
!   loop over cdot(in(ii),ind,jind) where we consider deriv of c(in(ii))
!   wrt h(in(ii),jind) with components jind
!
       do ii=1,3
!
!  deriv of cdot(in(ii),x) wrt x, y, z
!
         cdot(in(ii),1,1)=1.0d0/rch(in(ii)) &
                         +tch(in(ii),1)*c(in(ii),1)/rch(in(ii))**2 
         cdot(in(ii),1,2)=tch(in(ii),2)*c(in(ii),1)/rch(in(ii))**2
         cdot(in(ii),1,3)=tch(in(ii),3)*c(in(ii),1)/rch(in(ii))**2
!
!  deriv of cdot(in(ii),y) wrt x, y, z
!
         cdot(in(ii),2,1)=tch(in(ii),1)*c(in(ii),2)/rch(in(ii))**2
         cdot(in(ii),2,2)=1.0d0/rch(in(ii)) &
                         +tch(in(ii),2)*c(in(ii),2)/rch(in(ii))**2
         cdot(in(ii),2,3)=tch(in(ii),3)*c(in(ii),2)/rch(in(ii))**2
!
!  deriv of cdot(in(ii),z) wrt x, y, z
!
         cdot(in(ii),3,1)=tch(in(ii),1)*c(in(ii),3)/rch(in(ii))**2
         cdot(in(ii),3,2)=tch(in(ii),2)*c(in(ii),3)/rch(in(ii))**2
         cdot(in(ii),3,3)=1.0d0/rch(in(ii)) &
                         +tch(in(ii),3)*c(in(ii),3)/rch(in(ii))**2
       enddo
!
       do ii=1,3
         do ind=1,3
            deldot=-dtheta0(i,in(ii),i)
!
!  derivative wrt h(i,ind)
!  for  rch(i) only terms are from the derivatives of theta0
!
            deldot=-deldot*tch(i,ind)/rch(i)
            pdot(nh(i,ind))=pdot(nh(i,ind)) &
                         +2.0d0*fdelta(i)*delta(in(ii))*deldot &
                         +4.0d0*hdelta(i)*delta(in(ii))**3*deldot
!
!  derivative wrt c(ind)
!
            deldot=-deldot
            pdot(nc(ind))=pdot(nc(ind)) &
                         +2.0d0*fdelta(i)*delta(in(ii))*deldot &
                         +4.0d0*hdelta(i)*delta(in(ii))**3*deldot
           do jj=1,3
!
!  partial derivatives wrt h(in(jj),ind), loop over delta(i,in(ii))
!
!   atemp1 is axb dot daxb wrt h(in(jj))
!
            atemp1=axb(1)*daxb(in(jj),ind,1) &
                  +axb(2)*daxb(in(jj),ind,2) &
                  +axb(3)*daxb(in(jj),ind,3)
            atemp1=atemp1/(norma**3)
!
!  atemp2 is deriv of normalised axb
!
            atemp2(1)=daxb(in(jj),ind,1)/norma-atemp1*axb(1)
            atemp2(2)=daxb(in(jj),ind,2)/norma-atemp1*axb(2)
            atemp2(3)=daxb(in(jj),ind,3)/norma-atemp1*axb(3)
!
!  atemp3 is daxb dot c(in(ii))
!
            atemp3=atemp2(1)*c(in(ii),1)+atemp2(2)*c(in(ii),2)  &
                                   +atemp2(3)*c(in(ii),3)
!
!  atemp4 is axb dot cdot
!
            atemp4=0.0d0
            if(ii.eq.jj)then
!
!  ie deriv of c(in(ii)) wrt h(in(jj)) is non zero only for ii = jj
!
              atemp4=axb(1)*cdot(in(ii),1,ind)  &
                           +axb(2)*cdot(in(ii),2,ind)  &
                           +axb(3)*cdot(in(ii),3,ind)
              atemp4=atemp4/norma
            endif
!
!  atemp5 is deriv of theta0(i,in(ii)) wrt to nh(in(jj),ind)
!
            atemp5=-dtheta0(i,in(ii),in(jj))
!
!  deriv wrt h(in(jj)),ind):
!
            atemp5=-atemp5*tch(in(jj),ind)/rch(in(jj))
            deldot=atemp3+atemp4
            deldot=-1.0d0/sqrt(1.0d0-argd(in(ii))**2)*deldot
            deldot=deldot+atemp5
            pdot(nh(in(jj),ind))=pdot(nh(in(jj),ind))  &
                         +2.0d0*fdelta(i)*delta(in(ii))*deldot  &
                         +4.0d0*hdelta(i)*delta(in(ii))**3*deldot
!
!  for carbon the only contributions are from axb dot cdot term and
!  from theta0 and derivative cdot wrt carbon=-cdot wrt hydrogen
!
            deldot=1.0d0/sqrt(1.0d0-argd(in(ii))**2)*atemp4
            deldot=deldot-atemp5
            pdot(nc(ind))=pdot(nc(ind))  &
                         +2.0d0*fdelta(i)*delta(in(ii))*deldot  &
                         +4.0d0*hdelta(i)*delta(in(ii))**3*deldot
          enddo
        enddo
       enddo
       return
       end

!******************************************************
!
       subroutine opforce
!
!  calculates the out-of-plane bending force constants
!  and their derivatives
!
       use ch4oinc
       implicit none
       double precision :: switch(4),dswitch(4,4)
       integer :: i,j
!       implicit double precision (a-h,o-z)
!
!       dimension switch(4),dswitch(4,4)
!
!  calculate switching functions:
!
       switch(1)=(1.0d0-s3(1))*s3(2)*s3(3)*s3(4)
       switch(2)=(1.0d0-s3(2))*s3(3)*s3(4)*s3(1)
       switch(3)=(1.0d0-s3(3))*s3(4)*s3(1)*s3(2)
       switch(4)=(1.0d0-s3(4))*s3(1)*s3(2)*s3(3)
!
!  calculate derivatives:
!  derivative of switch(1) wrt the 4 rch bond lengths
!
       dswitch(1,1)=-ds3(1)*s3(2)*s3(3)*s3(4)
       dswitch(1,2)=(1.0d0-s3(1))*ds3(2)*s3(3)*s3(4)
       dswitch(1,3)=(1.0d0-s3(1))*s3(2)*ds3(3)*s3(4)
       dswitch(1,4)=(1.0d0-s3(1))*s3(2)*s3(3)*ds3(4)
!
!  derivative of switch(2) wrt the 4 rch bond lengths
!
       dswitch(2,1)=(1.0d0-s3(2))*s3(3)*s3(4)*ds3(1)
       dswitch(2,2)=-ds3(2)*s3(3)*s3(4)*s3(1)
       dswitch(2,3)=(1.0d0-s3(2))*ds3(3)*s3(4)*s3(1)
       dswitch(2,4)=(1.0d0-s3(2))*s3(3)*ds3(4)*s3(1)
!
!  derivative of switch(3) wrt the 4 rch bond lengths
!
       dswitch(3,1)=(1.0d0-s3(3))*s3(4)*ds3(1)*s3(2)
       dswitch(3,2)=(1.0d0-s3(3))*s3(4)*s3(1)*ds3(2)
       dswitch(3,3)=-ds3(3)*s3(4)*s3(1)*s3(2)
       dswitch(3,4)=(1.0d0-s3(3))*ds3(4)*s3(1)*s3(2)
!
!  derivative of switch(3) wrt the 4 rch bond lengths
!
       dswitch(4,1)=(1.0d0-s3(4))*ds3(1)*s3(2)*s3(3)
       dswitch(4,2)=(1.0d0-s3(4))*s3(1)*ds3(2)*s3(3)
       dswitch(4,3)=(1.0d0-s3(4))*s3(1)*s3(2)*ds3(3)
       dswitch(4,4)=-ds3(4)*s3(1)*s3(2)*s3(3)
!
!  calculate the force constants and their derivatives
!
       do i=1,4
         fdelta(i)=switch(i)*fch3
         hdelta(i)=switch(i)*hch3
         do j=1,4
           dfdelta(i,j)=dswitch(i,j)*fch3
           dhdelta(i,j)=dswitch(i,j)*hch3
         enddo
       enddo
       return
       end
!
!******************************************************
!
!
       subroutine ipforce
!
!  calculates the symmetrised in plane bend force constants and
!  all partial derivatives involving them
!
       use ch4oinc
       implicit none
       double precision :: dfk0(4,4,4),df1dc(4),df1dh(4)
       double precision :: f0, f2, arga1, arga2, a1, a2, duma1, duma2
       integer :: i
!       implicit double precision (a-h,o-z)
!
!       dimension dfk0(4,4,4),df1dc(4),df1dh(4)
!
!  set force constant at asymptotes
!
       f0=fkinf+ak
       f2=fkinf
!
       fk0(1,2)=f0+f0*(s1(1)*s1(2)-1.0d0)+(f0-f2)*(s2(3)*s2(4)-1.0d0)
       fk0(1,3)=f0+f0*(s1(1)*s1(3)-1.0d0)+(f0-f2)*(s2(2)*s2(4)-1.0d0)
       fk0(1,4)=f0+f0*(s1(1)*s1(4)-1.0d0)+(f0-f2)*(s2(2)*s2(3)-1.0d0)
       fk0(2,3)=f0+f0*(s1(2)*s1(3)-1.0d0)+(f0-f2)*(s2(1)*s2(4)-1.0d0)
       fk0(2,4)=f0+f0*(s1(2)*s1(4)-1.0d0)+(f0-f2)*(s2(1)*s2(3)-1.0d0)
       fk0(3,4)=f0+f0*(s1(3)*s1(4)-1.0d0)+(f0-f2)*(s2(1)*s2(2)-1.0d0)
!
!  derivative of fk0
!
       dfk0(1,2,1)=f0*ds1(1)*s1(2)
       dfk0(1,2,2)=f0*s1(1)*ds1(2)
       dfk0(1,2,3)=(f0-f2)*ds2(3)*s2(4)
       dfk0(1,2,4)=(f0-f2)*s2(3)*ds2(4)
!
       dfk0(1,3,1)=f0*ds1(1)*s1(3)
       dfk0(1,3,2)=(f0-f2)*ds2(2)*s2(4)
       dfk0(1,3,3)=f0*s1(1)*ds1(3)
       dfk0(1,3,4)=(f0-f2)*s2(2)*ds2(4)
!
       dfk0(1,4,1)=f0*ds1(1)*s1(4)
       dfk0(1,4,2)=(f0-f2)*ds2(2)*s2(3)
       dfk0(1,4,3)=(f0-f2)*s2(2)*ds2(3)
       dfk0(1,4,4)=f0*s1(1)*ds1(4)
!
       dfk0(2,3,1)=(f0-f2)*ds2(1)*s2(4)
       dfk0(2,3,2)=f0*ds1(2)*s1(3)
       dfk0(2,3,3)=f0*s1(2)*ds1(3)
       dfk0(2,3,4)=(f0-f2)*s2(1)*ds2(4)
!
       dfk0(2,4,1)=(f0-f2)*ds2(1)*s2(3)
       dfk0(2,4,2)=f0*ds1(2)*s1(4)
       dfk0(2,4,3)=(f0-f2)*s2(1)*ds2(3)
       dfk0(2,4,4)=f0*s1(2)*ds1(4)
!
       dfk0(3,4,1)=(f0-f2)*ds2(1)*s2(2)
       dfk0(3,4,2)=(f0-f2)*s2(1)*ds2(2)
       dfk0(3,4,3)=f0*ds1(3)*s1(4)
       dfk0(3,4,4)=f0*s1(3)*ds1(4)
!
!       argfk0=bk*((rch(1)-r0ch)**2+(rch(2)-r0ch)**2
!     *              +(rch(3)-r0ch)**2+(rch(4)-r0ch)**2)
!       fk0=fkinf+ak*exp(-argfk0)
       do i=1,4
!
!  calc derivatives of fk0 wrt each of the rch(i) bonds
!
!         dfk0(i)=-2.0d0*ak*bk*(rch(i)-r0ch)*exp(-argfk0)
!
!  calculate the terms f1(i)
!
         arga1=aa1*rbh(i)*rbh(i)
         arga2=aa4*(rbh(i)-r0hh)*(rbh(i)-r0hh)
         a1=1.0d0-exp(-arga1)
         a2=aa2+aa3*exp(-arga2)
         f1(i)=a1*exp(-a2*(rch(i)-r0ch)**2)
!
!  and calculate the derivatives wrt rch(i) and rbh(i)
!
         duma1=2.0d0*aa1*rbh(i)*exp(-arga1)
         duma2=-2.0d0*aa3*aa4*(rbh(i)-r0hh)*exp(-arga2)
         df1dc(i)=-2.0d0*(rch(i)-r0ch)*a1*a2*exp(-a2*(rch(i)-r0ch)**2)
         df1dh(i)=duma1*exp(-a2*(rch(i)-r0ch)**2)  &
                   -duma2*(rch(i)-r0ch)**2*a1*exp(-a2*(rch(i)-r0ch)**2)
       enddo
!
!  derivative of total force constant f(i,j) wrt bond length rch(k)
!  is given by dfdc(i,j,k)
!
      dfdc(1,2,1)=dfk0(1,2,1)*f1(1)*f1(2)+fk0(1,2)*df1dc(1)*f1(2)
      dfdc(1,2,2)=dfk0(1,2,2)*f1(1)*f1(2)+fk0(1,2)*f1(1)*df1dc(2)
      dfdc(1,2,3)=dfk0(1,2,3)*f1(1)*f1(2)
      dfdc(1,2,4)=dfk0(1,2,4)*f1(1)*f1(2)
!
      dfdc(1,3,1)=dfk0(1,3,1)*f1(1)*f1(3)+fk0(1,3)*df1dc(1)*f1(3)
      dfdc(1,3,2)=dfk0(1,3,2)*f1(1)*f1(3)
      dfdc(1,3,3)=dfk0(1,3,3)*f1(1)*f1(3)+fk0(1,3)*f1(1)*df1dc(3)
      dfdc(1,3,4)=dfk0(1,3,4)*f1(1)*f1(3)
!
      dfdc(1,4,1)=dfk0(1,4,1)*f1(1)*f1(4)+fk0(1,4)*df1dc(1)*f1(4)
      dfdc(1,4,2)=dfk0(1,4,2)*f1(1)*f1(4)
      dfdc(1,4,3)=dfk0(1,4,3)*f1(1)*f1(4)
      dfdc(1,4,4)=dfk0(1,4,4)*f1(1)*f1(4)+fk0(1,4)*f1(1)*df1dc(4)
!
      dfdc(2,3,1)=dfk0(2,3,1)*f1(2)*f1(3)
      dfdc(2,3,2)=dfk0(2,3,2)*f1(2)*f1(3)+fk0(2,3)*df1dc(2)*f1(3)
      dfdc(2,3,3)=dfk0(2,3,3)*f1(2)*f1(3)+fk0(2,3)*f1(2)*df1dc(3)
      dfdc(2,3,4)=dfk0(2,3,4)*f1(2)*f1(3)
!
      dfdc(2,4,1)=dfk0(2,4,1)*f1(2)*f1(4)
      dfdc(2,4,2)=dfk0(2,4,2)*f1(2)*f1(4)+fk0(2,4)*df1dc(2)*f1(4)
      dfdc(2,4,3)=dfk0(2,4,3)*f1(2)*f1(4)
      dfdc(2,4,4)=dfk0(2,4,4)*f1(2)*f1(4)+fk0(2,4)*f1(2)*df1dc(4)
!
      dfdc(3,4,1)=dfk0(3,4,1)*f1(3)*f1(4)
      dfdc(3,4,2)=dfk0(3,4,2)*f1(3)*f1(4)
      dfdc(3,4,3)=dfk0(3,4,3)*f1(3)*f1(4)+fk0(3,4)*df1dc(3)*f1(4)
      dfdc(3,4,4)=dfk0(3,4,4)*f1(3)*f1(4)+fk0(3,4)*f1(3)*df1dc(4)
!
!  derivative of total force constant f(i,j) wrt bond length rbh(k)
!  is given by dfdh(i,j,k)
!
!  nb only non-zero derivatives are those from rbh(i) and rbh(j)
!
       dfdh(1,2,1)=fk0(1,2)*df1dh(1)*f1(2)
       dfdh(1,2,2)=fk0(1,2)*f1(1)*df1dh(2)
       dfdh(1,2,3)=0.0d0
       dfdh(1,2,4)=0.0d0
!
       dfdh(1,3,1)=fk0(1,3)*df1dh(1)*f1(3)
       dfdh(1,3,2)=0.0d0
       dfdh(1,3,3)=fk0(1,3)*f1(1)*df1dh(3)
       dfdh(1,3,4)=0.0d0
!
       dfdh(1,4,1)=fk0(1,4)*df1dh(1)*f1(4)
       dfdh(1,4,2)=0.0d0
       dfdh(1,4,3)=0.0d0
       dfdh(1,4,4)=fk0(1,4)*f1(1)*df1dh(4)
!
       dfdh(2,3,1)=0.0d0
       dfdh(2,3,2)=fk0(2,3)*df1dh(2)*f1(3)
       dfdh(2,3,3)=fk0(2,3)*f1(2)*df1dh(3)
       dfdh(2,3,4)=0.0d0
!
       dfdh(2,4,1)=0.0d0
       dfdh(2,4,2)=fk0(2,4)*df1dh(2)*f1(4)
       dfdh(2,4,3)=0.0d0
       dfdh(2,4,4)=fk0(2,4)*f1(2)*df1dh(4)
!
       dfdh(3,4,1)=0.0d0
       dfdh(3,4,2)=0.0d0
       dfdh(3,4,3)=fk0(3,4)*df1dh(3)*f1(4)
       dfdh(3,4,4)=fk0(3,4)*f1(3)*df1dh(4)
!
       return
       end
!
!******************************************************
!
!
       subroutine switchf
!
!  calculates switching functions: s3,sphi,stheta
!  and their derivatives ds3,dsphi,dstheta
!
       use ch4oinc
       implicit none
       double precision :: argmax, args1, args2, args3
       double precision :: argstheta, argsphi
       integer :: i
!       implicit double precision (a-h,o-z)
!
!  nb remember that integration units are:
!  energy in   1.0d+05 j/mol
!  time in     1.0d-14 s
!
       a1s=1.5313681d-7
       b1s=-4.6696246d0
       a2s=1.0147402d-7
       b2s=-12.363798d0
!
!  use double precision criterion:
!
!  tanh(19.0d0)=1.0d0
!
       argmax=19.0d0
       do i=1,4
         args1=a1s*(rch(i)-r0ch)*(rch(i)-b1s)**8
         if(args1.lt.argmax)then
           s1(i)=1.0d0-tanh(args1)
           ds1(i)=a1s*((rch(i)-b1s)**8  &
                       +8.0d0*(rch(i)-r0ch)*(rch(i)-b1s)**7)
           ds1(i)=-ds1(i)/cosh(args1)**2
         else
           s1(i)=0.0d0
           ds1(i)=0.0d0
         endif
!
         args2=a2s*(rch(i)-r0ch)*(rch(i)-b2s)**6
         if(args2.lt.argmax)then
           s2(i)=1.0d0-tanh(args2)
           ds2(i)=a2s*((rch(i)-b2s)**6  &
                       +6.0d0*(rch(i)-r0ch)*(rch(i)-b2s)**5)
           ds2(i)=-ds2(i)/cosh(args2)**2
         else
           s2(i)=0.0d0
           ds2(i)=0.0d0
         endif
!
!  calculate s3 and ds3
!
         args3=a3s*(rch(i)-r0ch)*(rch(i)-b3s)**2
         if (args3.lt.argmax)then
           s3(i)=1.0d0-tanh(args3)
           ds3(i)=a3s*(3.0d0*rch(i)**2-2.0d0*rch(i)*(r0ch+2.0d0*b3s)  &
                +b3s*(b3s+2.0d0*r0ch))
           ds3(i)=-ds3(i)/cosh(args3)**2
         else
           s3(i)=0.0d0
           ds3(i)=0.0d0
         endif
!
!  calculate sphi and dsphi
!
!  condition here is on the bondlength rch(i)
!  st argsphi is lt approx 19.0d0
!
         if(rch(i).lt.3.8d0)then
           argsphi=aphi*(rch(i)-r0ch)*exp(bphi*(rch(i)-cphi)**3)
           sphi(i)=1.0d0-tanh(argsphi)
           dsphi(i)=aphi*(1.0d0+3.0d0*bphi*(rch(i)-r0ch) &
                            *(rch(i)-cphi)**2)
           dsphi(i)=dsphi(i)*exp(bphi*(rch(i)-cphi)**3)
           dsphi(i)=-dsphi(i)/cosh(argsphi)**2
         else
           sphi(i)=0.0d0
           dsphi(i)=0.0d0
         endif
!
!  calculate stheta and dstheta
!
         if(rch(i).lt.3.8d0)then
           argstheta=atheta*(rch(i)-r0ch)*exp(btheta*(rch(i)-ctheta)**3)
           stheta(i)=1.0d0-tanh(argstheta)
           dstheta(i)=atheta*(1.0d0+3.0d0*btheta*(rch(i)-r0ch) &
                 *(rch(i)-ctheta)**2)
           dstheta(i)=dstheta(i)*exp(btheta*(rch(i)-ctheta)**3)
           dstheta(i)=-dstheta(i)/cosh(argstheta)**2
         else
           stheta(i)=0.0d0
           dstheta(i)=0.0d0
         endif
       enddo
       return
       end
!
      SUBROUTINE SETUP(N3TM)
      implicit none
      integer, intent(in) :: N3TM
      integer, parameter :: N3TMMN=12

!
!      IMPLICIT DOUBLE PRECISION (A-H,O-Z)
!
!   N3TMMN = 3 * NATOMS
!   NATOMS = the number of atoms represented by this potential function
!
!   The variable N3TMMN is the minimum value of N3TM allowed to be 
!   passed by the calling routine for the number of cartesian 
!   coordinates needed to represent the full system represented by this 
!   potential energy surface routine.
!   N3TM must be greater than or equal to N3TMMN.
!
!      PARAMETER (N3TMMN = 18)
!
!     COMMON /PDATCM/ D1(3),D3(3),ALPH(3),RE(3),BETA(3),CC(3),AA(3),    
!    *   APARM(5),REFV,TAU,CP,B1,C1
!     COMMON /PDT2CM/ PI,APHI,BPHI,CPHI,PCH4,FCH3,HCH3,RNOT,A3,B3,DELT, 
!    *   DIJ,A,B,C0,D,REX,ATETHA,BTETHA,CTETHA
!     COMMON /PDT3CM/ FK0,FA,CA,CB

!     CHARACTER*5 SURFNM
!    the above line was commented because the variable is not used.     0423TA02
!
!  CHECK THE NUMBER OF CARTESIAN COORDINATES SET BY THE CALLING PROGRAM
!
      IF (N3TM .LT. N3TMMN) THEN
          WRITE (6, 6000) N3TM, N3TMMN
          STOP 'SETUP 1'
      ENDIF
!
      RETURN
!
6000  FORMAT(/,2X,T5,'Warning: N3TM is set equal to ',I3, &
                        ' but this potential routine', &
                /,2X,T14,'requires N3TM be greater than or ', &
                         'equal to ',I3,/)
!
      END SUBROUTINE SETUP
