      SUBROUTINE SURF(V, COORD, DX, N3TM)
!
!     System: CH3 
!     Reference: D.M. Medvedev, L.B. Harding, and S.K. Gray
!            Mol. Phys. 104, 2006, 73-81.
!
!     This is the driver routine for the CH3 Global potential 
!     energy surface.
!     
!     SETUP must be called once before any calls to SURF.
!     In this PES, only potential energy with respect to the 
!     Cartesian coordinate is passed by calling the routine. 
!     CALL SURF(V, X, DX, N3TM)
!     where X and DX are arrays dimensioned N3TM, and N3TM must 
!     be greater than or equal to 12 (3*number of atoms).
!     DX is a dummy array for derivatives of the energy, which is 
!     always zero.
!     All the information passed to and from the potential energy 
!     surface routine is in hartree atomic units.
!
!       This potential is written such that:
!               X(1)  -  X(3)  : X, Y, Z for C
!               X(4)  -  X(6)  : X, Y, Z for H1
!               X(7)  -  X(9)  : X, Y, Z for H2
!               X(10) -  X(12) : X, Y, Z for H3
!
!       Zero of energy is CH + H2
!
      implicit none
      integer, intent(in) :: n3tm
      double precision, intent(in) :: coord(n3tm) 
      double precision, intent(out) :: v, dx(n3tm)
      double precision :: xz(3,4), energy
      integer :: i, j
!
!     Transfer Cartesian Coordinate to XZ(i,j)
!     i=1,2,3 == x,y,z;  j = 1,2,3,4 == C,H,H,H.
!     XZ in Bohr
!              
      do j=1,4
         do i=1,3
             XZ(i,j) = COORD(3*j-3+i)
         enddo 
      enddo 
!
      CALL CH3_XYZ(XZ,ENERGY)     
      V = ENERGY 
      RETURN
      END
!
      SUBROUTINE setup(N3TM)
      implicit none
      integer :: n3tm
      RETURN
      END
!
!
!     The subroutine ch3_xyz is provided by Stephen K. Gray
!     Reference: D.M. Medvedev, L.B. Harding, and S.K. Gray
!            Mol. Phys. 104, 2006, 73-81.
!
      subroutine ch3_xyz(xz, en)
      implicit none
      real*8,intent(in) :: xz(3, 4)
      real*8,intent(out) :: en
      real*8 :: rij(6)
      integer :: index, i, ip1, j, k
!
!     global potential surface for ch3
!
!     cas(7/7)+1+2+qc/aug-cc-pvtz
!
!     xz array contains the cartesian coordinates (au) for the
!     four atoms, carbon first, followed by the three hydrogens
!
!     this interface subroutine calculates the six internuclear
!     distances from the cartesian coordinates and inputs them
!     to the potential surface subroutine
!
      index = 0
      do i = 1, 3
         ip1 = i + 1
         do j = ip1, 4
            index = index + 1
            rij(index) = 0.0d0
            do k = 1, 3
               rij(index) = rij(index) + (xz(k, i) - xz(k, j)) ** 2
            end do
            rij(index) = dsqrt(rij(index))
         end do
      end do

      call ch3_rij(rij, en)
!
! adjust so ch(re)+h2(re) is the zero of energy:
!
      en=en+39.586990d0

      return
      end


      subroutine ch3_rij(rij,energy)
      implicit none
      double precision, intent(in) :: rij(6)
      double precision, intent(out) :: energy
      double precision :: rijt(6), swch3, ech3, echph2, ech2ph
      double precision :: swich, swich2, swjch, swjch2, swkch, swkch2
      double precision :: eglch2ch,  eglobal, swch2ph, swchph2

      call ch3switch(rij,swch3)
      if(swch3 .gt. 0.0d0) then
        call methyl(rij,ech3)
        if(swch3 .ge. 1.0d0) then
          energy = ech3
          return
        endif
      endif

      echph2=0.0d0
      ech2ph=0.0d0
      
      rijt(1) = rij(1)
      rijt(2) = rij(2)
      rijt(3) = rij(3)
      rijt(4) = rij(4)
      rijt(5) = rij(5)
      rijt(6) = rij(6)
      call chswitch(rijt,swich)
      call ch2switch(rijt,swich2)
      if(swich  .gt. 0.0d0) call chPh2(rijt,echph2)
      if(swich2 .gt. 0.0d0) then
        if(swich .gt. 0.0d0) stop 'swich and swich2'
        call ch2Ph(rijt,ech2ph)
      endif

      rijt(1) = rij(2)
      rijt(2) = rij(1)
      rijt(3) = rij(3)
      rijt(4) = rij(4)
      rijt(5) = rij(6)
      rijt(6) = rij(5)
      call chswitch(rijt,swjch)
      call ch2switch(rijt,swjch2)
      if(swjch  .gt. 0.0d0) then
        if(swich2 .gt. 0.0) stop 'swich2 and swjch'
        if(swich  .gt. 0.0) stop 'swich and swjch'
        call chPh2(rijt,echph2)
      endif
      if(swjch2 .gt. 0.0d0) then
        if(swich2 .gt. 0.0) stop 'swich2 and swjch2'
        if(swich  .gt. 0.0) stop 'swich and swjch2'
        if(swjch  .gt. 0.0) stop 'swjch and swjch2'
        call ch2Ph(rijt,ech2ph)
      endif
!
      rijt(1) = rij(3)
      rijt(2) = rij(2)
      rijt(3) = rij(1)
      rijt(4) = rij(6)
      rijt(5) = rij(5)
      rijt(6) = rij(4)
      call chswitch(rijt,swkch)
      call ch2switch(rijt,swkch2)
      if(swkch  .gt. 0.0d0) then
        if(swich2 .gt. 0.0) stop 'swich2 and swkch'
        if(swich  .gt. 0.0) stop 'swich  and swkch'
        if(swjch2 .gt. 0.0) stop 'swjch2 and swkch'
        if(swjch  .gt. 0.0) stop 'swjch  and swkch'
        call chPh2(rijt,echph2)
      endif
      if(swkch2 .gt. 0.0d0) then
        if(swich2 .gt. 0.0) stop 'swich2 and swkch2'
        if(swich  .gt. 0.0) stop 'swich  and swkch2'
        if(swjch2 .gt. 0.0) stop 'swjch2 and swkch2'
        if(swjch  .gt. 0.0) stop 'swjch  and swkch2'
        if(swkch  .gt. 0.0) stop 'swkch  and swkch2'
        call ch2Ph(rijt,ech2ph)
      endif
      swch2ph = max(swich2,swjch2,swkch2)
      swchph2 = max(swich, swjch, swkch)
!
      if(swch2ph .ge. 1.0d0) then
        eglch2ch = ech2ph
      else if(swchph2 .ge. 1.0d0) then
        eglch2ch = echph2
      else
        call global_rij(rij,eglobal)
        if( swch2ph .le. 0.0d0 .and. swchph2 .le. 0.0d0) then
          eglch2ch = eglobal
        else if(swch2ph .gt. 0.0d0) then
          eglch2ch = eglobal*(1.0d0-swch2ph) + swch2ph*ech2ph
        else if(swchph2 .gt. 0.0d0) then
          eglch2ch = eglobal*(1.0d0-swchph2) + swchph2*echph2
        endif
      endif
! 
      energy = eglch2ch*(1.0d0-swch3) + swch3*ech3
!
!      write(10,20) rij
!20    format(' rij:      ',6f10.5)
!      write(10,21) swch3,swch2ph,swchph2
!21    format(' Switches :',3f10.5)
!      write(10,22) ech3,ech2ph,echph2,eglobal
!22    format(' Energies :',4f10.5)
!      write(10,22) eglch2ch,energy
!
      return
      end subroutine ch3_rij

      subroutine chswitch(rijt,swi)
      implicit none
      double precision,intent(in) :: rijt(6)
      double precision,intent(out) :: swi
      double precision :: delch, delchl, delchu, rchl, rchu, rhhl, rhhu
      double precision :: ri, rj, rk, rij, rik, rjk
      double precision :: swij, swik, swch, swchj, swchk
!
!     this switches on near the ch+h2 asymptote
!
!
!     1d switch ranges
!
      data delchl,  delchu  / 0.1d0, 0.2d0 /
      data rchl,    rchu    / 3.0d0, 4.0d0 /
      data rhhl,    rhhu    / 3.0d0, 4.0d0 /

      ri  = rijt(1)
      rj  = rijt(2)
      rk  = rijt(3)
      rij = rijt(4)
      rik = rijt(5)
      rjk = rijt(6)

      delch = ((ri-2.12d0)/2.12d0)**2  + ((rjk-1.40d0)/1.40d0)**2
      delch = sqrt(delch)

      call switch(delch,delchl,delchu,swch)
      call switch(rj,    rchl,   rchu,   swchj)
      call switch(rk,    rchl,   rchu,   swchk)
      call switch(rij,   rhhl,   rhhu,   swij)
      call switch(rik,   rhhl,   rhhu,   swik)
      swchj=1.0d0-swchj
      swchk=1.0d0-swchk
      swij=1.0d0-swij
      swik=1.0d0-swik
      
      swi=swch*swchj*swchk*swij*swik

      return
      end subroutine chswitch

      subroutine ch2switch(rijt,swi)
      implicit none
      double precision,intent(in) :: rijt(6)
      double precision,intent(out) :: swi
      double precision :: delch2, delch2l, delch2u 
      double precision :: rchl, rchu, rhhl, rhhu 
      double precision :: swch, swch2, swij, swik 
      double precision :: ri, rj, rk, rij, rik, rjk
!      implicit real*8 (a-h,o-z)
!      dimension rijt(6)
!
!     this switches on near the ch2+h asymptote
!
!
!     1d switch ranges
!
      data delch2l, delch2u / 0.1d0, 0.2d0 /
      data rchl,    rchu    / 4.0d0, 5.0d0 /
      data rhhl,    rhhu    / 3.5d0, 4.5d0 /

      ri  = rijt(1)
      rj  = rijt(2)
      rk  = rijt(3)
      rij = rijt(4)
      rik = rijt(5)
      rjk = rijt(6)

      delch2 = ((rj-2.04d0)/2.04d0)**2 + ((rk-2.04d0)/2.04d0)**2  &
               + ((rjk-3.75d0)/3.75d0)**2
      delch2 = sqrt(delch2)

      call switch(delch2,delch2l,delch2u,swch2)
      call switch(ri,    rchl,   rchu,   swch)
      call switch(rij,   rhhl,   rhhu,   swij)
      call switch(rik,   rhhl,   rhhu,   swik)
      swch=1.0d0-swch
      swij=1.0d0-swij
      swik=1.0d0-swik
      
      swi=swch2*swch*swij*swik

      return
      end subroutine ch2switch

      subroutine ch3switch(rijt,swi)
      double precision,intent(in) :: rijt(6)
      double precision,intent(out) :: swi
      double precision :: del, dell, delu, rc, rh 
      double precision :: rci, rcj, rck, rhij, rhik, rhjk 
!
!     this switches on near the ch3 minimum
!
!
!     1d switch ranges
!
      data dell,  delu  / 0.3d0,  0.5d0  /
      data rc,    rh    / 2.04d0, 3.53d0 /

      rci  = rijt(1)
      rcj  = rijt(2)
      rck  = rijt(3)
      rhij = rijt(4)
      rhik = rijt(5)
      rhjk = rijt(6)

      del = ((rci-rc)/rc)**2+((rcj-rc)/rc)**2+((rck-rc)/rc)**2    &
           +((rhij-rh)/rh)**2+((rhik-rh)/rh)**2+((rhjk-rh)/rh)**2
      del = sqrt(del)

      call switch(del,dell,delu,swi)
!
!     swi = 1 if "near" ch3
!         = 0 if not "near" ch3
!

      return
      end subroutine ch3switch

      subroutine switch(r,rl,ru,sw)
      implicit none
      double precision,intent(in) :: r, rl, ru 
      double precision,intent(out) :: sw 
      double precision :: x
!
!     r = switch variable
!     ru = upper bound of switch range
!     rl = lower bound of switch range
!
!     sw = 1 if r < rl
!     sw = 0 if r > ru
!

      x = (r-rl)/(ru-rl)

      if (x.LT.0.) then
        sw=1.0
      end if
      if (x.GT.1.) then
        sw=0.0
      end if
      if ((x.GE.0.).and.(x.LE.1.)) then
        sw=1.0d0-(10.0d0*x**3-15.0d0*x**4+6.0d0*x**5)
      end if

      return
      end subroutine switch

      subroutine global_rij(rij,en)
!
!     from w-76819-924.p
!
!     rij(1) = r(c-h1) (au)
!     rij(2) = r(c-h2)
!     rij(3) = r(c-h3)
!     rij(4) = r(h1-h2)
!     rij(5) = r(h1-h3)
!     rij(6) = r(h2-h3)
!
!     en = energy (hartree)
!
!
      implicit none
      double precision,intent(in) :: rij(6)
      double precision,intent(out) :: en
      double precision :: fk(924),fmr(6,6),shift(6),alpham(6)
      integer :: nmax, nvar, i, j
!
      data fk(1), fk(2) / -3.9404846408E+01 ,  7.2389225242E-02 /
      data fk(   3), fk(   4) /  7.2389225242E-02 ,  7.2389225242E-02 /
      data fk(   5), fk(   6) / -2.5472983393E-01 , -2.5472983393E-01 /
      data fk(   7), fk(   8) / -2.5472983393E-01 ,  3.4614608431E-01 /
      data fk(   9), fk(  10) /  3.4614608431E-01 ,  3.4614608431E-01 /
      data fk(  11), fk(  12) /  6.4163334579E-01 ,  6.4163334579E-01 /
      data fk(  13), fk(  14) /  6.4163334579E-01 , -2.7949517483E-01 /
      data fk(  15), fk(  16) / -2.7949517483E-01 ,  2.5014089108E-01 /
      data fk(  17), fk(  18) /  2.5014089108E-01 , -6.4784115455E-01 /
      data fk(  19), fk(  20) / -2.7949517483E-01 ,  2.5014089108E-01 /
      data fk(  21), fk(  22) / -6.4784115455E-01 ,  2.5014089108E-01 /
      data fk(  23), fk(  24) / -6.4784115455E-01 ,  2.5014089108E-01 /
      data fk(  25), fk(  26) /  2.5014089108E-01 , -3.9987752810E-02 /
      data fk(  27), fk(  28) / -3.9987752810E-02 , -3.9987752810E-02 /
      data fk(  29), fk(  30) / -1.0254853196E+00 , -1.0254853196E+00 /
      data fk(  31), fk(  32) / -1.0254853196E+00 , -1.9082461016E+00 /
      data fk(  33), fk(  34) / -1.9082461016E+00 , -1.9082461016E+00 /
      data fk(  35), fk(  36) / -8.7023182679E-02 , -8.7023182679E-02 /
      data fk(  37), fk(  38) / -3.4777464708E-01 , -3.4777464708E-01 /
      data fk(  39), fk(  40) /  1.0945554940E+00 , -8.7023182679E-02 /
      data fk(  41), fk(  42) / -8.7023182679E-02 , -3.4777464708E-01 /
      data fk(  43), fk(  44) /  1.0945554940E+00 , -3.4777464708E-01 /
      data fk(  45), fk(  46) / -8.7023182679E-02 , -8.7023182679E-02 /
      data fk(  47), fk(  48) /  1.0945554940E+00 , -3.4777464708E-01 /
      data fk(  49), fk(  50) / -3.4777464708E-01 , -5.0137729525E-01 /
      data fk(  51), fk(  52) / -5.0137729525E-01 , -1.8514902624E-01 /
      data fk(  53), fk(  54) /  6.9755742018E-02 ,  6.9755742018E-02 /
      data fk(  55), fk(  56) / -5.0137729525E-01 , -1.8514902624E-01 /
      data fk(  57), fk(  58) / -5.0137729525E-01 ,  6.9755742018E-02 /
      data fk(  59), fk(  60) /  6.9755742018E-02 , -1.8514902624E-01 /
      data fk(  61), fk(  62) / -5.0137729525E-01 , -5.0137729525E-01 /
      data fk(  63), fk(  64) /  6.9755742018E-02 ,  6.9755742018E-02 /
      data fk(  65), fk(  66) /  3.4278520012E-01 ,  2.0586892979E+00 /
      data fk(  67), fk(  68) / -1.5743761875E-01 , -1.5743761875E-01 /
      data fk(  69), fk(  70) / -1.5743761875E-01 ,  2.0586892979E+00 /
      data fk(  71), fk(  72) / -1.5743761875E-01 , -7.3254988245E-01 /
      data fk(  73), fk(  74) /  9.2745303715E-01 ,  9.2745303715E-01 /
      data fk(  75), fk(  76) / -1.5743761875E-01 , -1.5743761875E-01 /
      data fk(  77), fk(  78) /  2.0586892979E+00 ,  9.2745303715E-01 /
      data fk(  79), fk(  80) / -7.3254988245E-01 ,  9.2745303715E-01 /
      data fk(  81), fk(  82) /  9.2745303715E-01 ,  9.2745303715E-01 /
      data fk(  83), fk(  84) / -7.3254988245E-01 ,  5.8551525897E-01 /
      data fk(  85), fk(  86) /  1.5227318867E+00 ,  1.5227318867E+00 /
      data fk(  87), fk(  88) /  1.5227318867E+00 ,  6.1942337556E-01 /
      data fk(  89), fk(  90) /  6.1942337556E-01 ,  6.1942337556E-01 /
      data fk(  91), fk(  92) / -3.6653196255E-01 , -3.6653196255E-01 /
      data fk(  93), fk(  94) / -3.9051871685E-01 , -3.9051871685E-01 /
      data fk(  95), fk(  96) /  7.2499642100E-01 , -3.6653196255E-01 /
      data fk(  97), fk(  98) / -3.6653196255E-01 , -3.9051871685E-01 /
      data fk(  99), fk( 100) /  7.2499642100E-01 , -3.9051871685E-01 /
      data fk( 101), fk( 102) / -3.6653196255E-01 , -3.6653196255E-01 /
      data fk( 103), fk( 104) /  7.2499642100E-01 , -3.9051871685E-01 /
      data fk( 105), fk( 106) / -3.9051871685E-01 , -3.2507189403E-01 /
      data fk( 107), fk( 108) / -3.2507189403E-01 ,  1.6974284978E+00 /
      data fk( 109), fk( 110) /  5.2413047235E-01 ,  5.2413047235E-01 /
      data fk( 111), fk( 112) / -3.2507189403E-01 ,  1.6974284978E+00 /
      data fk( 113), fk( 114) / -3.2507189403E-01 ,  5.2413047235E-01 /
      data fk( 115), fk( 116) /  5.2413047235E-01 ,  1.6974284978E+00 /
      data fk( 117), fk( 118) / -3.2507189403E-01 , -3.2507189403E-01 /
      data fk( 119), fk( 120) /  5.2413047235E-01 ,  5.2413047235E-01 /
      data fk( 121), fk( 122) /  4.9859014648E-02 ,  4.9859014648E-02 /
      data fk( 123), fk( 124) /  2.0757225971E+00 ,  2.0757225971E+00 /
      data fk( 125), fk( 126) /  7.6132655928E-01 ,  4.9859014648E-02 /
      data fk( 127), fk( 128) /  2.0757225971E+00 ,  7.6132655928E-01 /
      data fk( 129), fk( 130) /  2.0757225971E+00 ,  7.6132655928E-01 /
      data fk( 131), fk( 132) /  2.0757225971E+00 ,  2.0757225971E+00 /
      data fk( 133), fk( 134) /  1.8670990625E+00 ,  1.8670990625E+00 /
      data fk( 135), fk( 136) /  1.8670990625E+00 ,  5.4888778733E-01 /
      data fk( 137), fk( 138) /  7.4533043410E-01 , -1.6395787505E-01 /
      data fk( 139), fk( 140) / -6.8203718465E-01 , -1.6395787505E-01 /
      data fk( 141), fk( 142) /  7.4533043410E-01 , -6.8203718465E-01 /
      data fk( 143), fk( 144) /  9.4780383554E-01 , -3.5326512682E+00 /
      data fk( 145), fk( 146) / -3.5326512682E+00 ,  5.4888778733E-01 /
      data fk( 147), fk( 148) /  7.4533043410E-01 , -6.8203718465E-01 /
      data fk( 149), fk( 150) / -1.6395787505E-01 , -1.6395787505E-01 /
      data fk( 151), fk( 152) / -6.8203718465E-01 ,  7.4533043410E-01 /
      data fk( 153), fk( 154) / -3.5326512682E+00 ,  9.4780383554E-01 /
      data fk( 155), fk( 156) / -3.5326512682E+00 ,  5.4888778733E-01 /
      data fk( 157), fk( 158) / -6.8203718465E-01 ,  7.4533043410E-01 /
      data fk( 159), fk( 160) / -1.6395787505E-01 , -6.8203718465E-01 /
      data fk( 161), fk( 162) / -1.6395787505E-01 ,  7.4533043410E-01 /
      data fk( 163), fk( 164) / -3.5326512682E+00 , -3.5326512682E+00 /
      data fk( 165), fk( 166) /  9.4780383554E-01 , -3.2359369880E+00 /
      data fk( 167), fk( 168) / -8.3630803229E-01 ,  1.8935841902E+00 /
      data fk( 169), fk( 170) / -3.5579989533E-02 , -8.3630803229E-01 /
      data fk( 171), fk( 172) / -3.5579989533E-02 ,  1.8935841902E+00 /
      data fk( 173), fk( 174) / -1.1589018393E+00 , -1.1589018393E+00 /
      data fk( 175), fk( 176) / -8.7170197993E-01 , -8.3630803229E-01 /
      data fk( 177), fk( 178) / -3.2359369880E+00 ,  1.8935841902E+00 /
      data fk( 179), fk( 180) / -3.5579989533E-02 , -8.3630803229E-01 /
      data fk( 181), fk( 182) / -1.1589018393E+00 , -1.1589018393E+00 /
      data fk( 183), fk( 184) / -3.5579989533E-02 ,  1.8935841902E+00 /
      data fk( 185), fk( 186) / -8.7170197993E-01 , -8.3630803229E-01 /
      data fk( 187), fk( 188) / -8.3630803229E-01 , -1.1589018393E+00 /
      data fk( 189), fk( 190) / -1.1589018393E+00 , -3.2359369880E+00 /
      data fk( 191), fk( 192) /  1.8935841902E+00 , -3.5579989533E-02 /
      data fk( 193), fk( 194) / -3.5579989533E-02 ,  1.8935841902E+00 /
      data fk( 195), fk( 196) / -8.7170197993E-01 , -1.1905917338E+00 /
      data fk( 197), fk( 198) / -1.1905917338E+00 , -1.0364707309E+00 /
      data fk( 199), fk( 200) / -1.0364707309E+00 ,  5.6970682725E+00 /
      data fk( 201), fk( 202) / -1.1905917338E+00 , -1.0364707309E+00 /
      data fk( 203), fk( 204) /  5.6970682725E+00 , -1.0364707309E+00 /
      data fk( 205), fk( 206) /  5.6970682725E+00 , -1.0364707309E+00 /
      data fk( 207), fk( 208) / -1.0364707309E+00 , -1.0269564086E+00 /
      data fk( 209), fk( 210) / -1.0269564086E+00 , -1.0269564086E+00 /
      data fk( 211), fk( 212) / -2.9595479492E-01 , -2.9595479492E-01 /
      data fk( 213), fk( 214) / -2.9595479492E-01 ,  5.1887780188E-01 /
      data fk( 215), fk( 216) /  5.1887780188E-01 ,  5.1887780188E-01 /
      data fk( 217), fk( 218) /  3.2364494381E-01 ,  3.2364494381E-01 /
      data fk( 219), fk( 220) / -1.7474027462E+00 , -1.7474027462E+00 /
      data fk( 221), fk( 222) /  1.0848305352E+00 ,  3.2364494381E-01 /
      data fk( 223), fk( 224) /  3.2364494381E-01 , -1.7474027462E+00 /
      data fk( 225), fk( 226) /  1.0848305352E+00 , -1.7474027462E+00 /
      data fk( 227), fk( 228) /  3.2364494381E-01 ,  3.2364494381E-01 /
      data fk( 229), fk( 230) /  1.0848305352E+00 , -1.7474027462E+00 /
      data fk( 231), fk( 232) / -1.7474027462E+00 ,  6.7174487501E-01 /
      data fk( 233), fk( 234) /  6.7174487501E-01 , -2.5539687398E+00 /
      data fk( 235), fk( 236) / -1.4680582381E+00 , -1.4680582381E+00 /
      data fk( 237), fk( 238) /  6.7174487501E-01 , -2.5539687398E+00 /
      data fk( 239), fk( 240) /  6.7174487501E-01 , -1.4680582381E+00 /
      data fk( 241), fk( 242) / -1.4680582381E+00 , -2.5539687398E+00 /
      data fk( 243), fk( 244) /  6.7174487501E-01 ,  6.7174487501E-01 /
      data fk( 245), fk( 246) / -1.4680582381E+00 , -1.4680582381E+00 /
      data fk( 247), fk( 248) / -3.9826627494E-01 , -3.9826627494E-01 /
      data fk( 249), fk( 250) /  2.0175346521E-01 ,  2.0175346521E-01 /
      data fk( 251), fk( 252) /  7.1528866038E-01 , -3.9826627494E-01 /
      data fk( 253), fk( 254) / -3.9826627494E-01 ,  2.0175346521E-01 /
      data fk( 255), fk( 256) /  7.1528866038E-01 ,  2.0175346521E-01 /
      data fk( 257), fk( 258) / -3.9826627494E-01 , -3.9826627494E-01 /
      data fk( 259), fk( 260) /  7.1528866038E-01 ,  2.0175346521E-01 /
      data fk( 261), fk( 262) /  2.0175346521E-01 ,  1.4877193149E+00 /
      data fk( 263), fk( 264) /  1.4877193149E+00 , -1.3192973546E+00 /
      data fk( 265), fk( 266) /  3.0327956981E-02 ,  3.0327956981E-02 /
      data fk( 267), fk( 268) /  1.4877193149E+00 , -1.3192973546E+00 /
      data fk( 269), fk( 270) /  1.4877193149E+00 ,  3.0327956981E-02 /
      data fk( 271), fk( 272) /  3.0327956981E-02 , -1.3192973546E+00 /
      data fk( 273), fk( 274) /  1.4877193149E+00 ,  1.4877193149E+00 /
      data fk( 275), fk( 276) /  3.0327956981E-02 ,  3.0327956981E-02 /
      data fk( 277), fk( 278) / -7.5891904418E-01 ,  9.0226566019E-01 /
      data fk( 279), fk( 280) /  5.3106315935E-01 ,  3.3647309039E-01 /
      data fk( 281), fk( 282) /  5.3106315935E-01 ,  9.0226566019E-01 /
      data fk( 283), fk( 284) /  3.3647309039E-01 ,  4.0403939569E+00 /
      data fk( 285), fk( 286) / -3.0047027682E+00 , -3.0047027682E+00 /
      data fk( 287), fk( 288) / -7.5891904418E-01 ,  9.0226566019E-01 /
      data fk( 289), fk( 290) /  3.3647309039E-01 ,  5.3106315935E-01 /
      data fk( 291), fk( 292) /  5.3106315935E-01 ,  3.3647309039E-01 /
      data fk( 293), fk( 294) /  9.0226566019E-01 , -3.0047027682E+00 /
      data fk( 295), fk( 296) /  4.0403939569E+00 , -3.0047027682E+00 /
      data fk( 297), fk( 298) / -7.5891904418E-01 ,  3.3647309039E-01 /
      data fk( 299), fk( 300) /  9.0226566019E-01 ,  5.3106315935E-01 /
      data fk( 301), fk( 302) /  3.3647309039E-01 ,  5.3106315935E-01 /
      data fk( 303), fk( 304) /  9.0226566019E-01 , -3.0047027682E+00 /
      data fk( 305), fk( 306) / -3.0047027682E+00 ,  4.0403939569E+00 /
      data fk( 307), fk( 308) / -1.4986002669E+00 ,  6.9761630796E-01 /
      data fk( 309), fk( 310) / -2.1881911875E+00 ,  5.1885518826E-01 /
      data fk( 311), fk( 312) /  6.9761630796E-01 ,  5.1885518826E-01 /
      data fk( 313), fk( 314) / -2.1881911875E+00 , -7.9397248411E-01 /
      data fk( 315), fk( 316) / -7.9397248411E-01 ,  5.1930979853E+00 /
      data fk( 317), fk( 318) /  6.9761630796E-01 , -1.4986002669E+00 /
      data fk( 319), fk( 320) / -2.1881911875E+00 ,  5.1885518826E-01 /
      data fk( 321), fk( 322) /  6.9761630796E-01 , -7.9397248411E-01 /
      data fk( 323), fk( 324) / -7.9397248411E-01 ,  5.1885518826E-01 /
      data fk( 325), fk( 326) / -2.1881911875E+00 ,  5.1930979853E+00 /
      data fk( 327), fk( 328) /  6.9761630796E-01 ,  6.9761630796E-01 /
      data fk( 329), fk( 330) / -7.9397248411E-01 , -7.9397248411E-01 /
      data fk( 331), fk( 332) / -1.4986002669E+00 , -2.1881911875E+00 /
      data fk( 333), fk( 334) /  5.1885518826E-01 ,  5.1885518826E-01 /
      data fk( 335), fk( 336) / -2.1881911875E+00 ,  5.1930979853E+00 /
      data fk( 337), fk( 338) /  1.4244773731E+00 , -6.5539916303E-01 /
      data fk( 339), fk( 340) /  7.3714772467E-01 ,  1.5426894093E+00 /
      data fk( 341), fk( 342) /  7.3714772467E-01 , -6.5539916303E-01 /
      data fk( 343), fk( 344) /  1.5426894093E+00 , -1.8728695967E-01 /
      data fk( 345), fk( 346) /  3.2585952403E+00 ,  3.2585952403E+00 /
      data fk( 347), fk( 348) /  1.4244773731E+00 , -6.5539916303E-01 /
      data fk( 349), fk( 350) /  1.5426894093E+00 ,  7.3714772467E-01 /
      data fk( 351), fk( 352) /  7.3714772467E-01 ,  1.5426894093E+00 /
      data fk( 353), fk( 354) / -6.5539916303E-01 ,  3.2585952403E+00 /
      data fk( 355), fk( 356) / -1.8728695967E-01 ,  3.2585952403E+00 /
      data fk( 357), fk( 358) /  1.4244773731E+00 ,  1.5426894093E+00 /
      data fk( 359), fk( 360) / -6.5539916303E-01 ,  7.3714772467E-01 /
      data fk( 361), fk( 362) /  1.5426894093E+00 ,  7.3714772467E-01 /
      data fk( 363), fk( 364) / -6.5539916303E-01 ,  3.2585952403E+00 /
      data fk( 365), fk( 366) /  3.2585952403E+00 , -1.8728695967E-01 /
      data fk( 367), fk( 368) /  4.9435593164E-01 , -4.9629360340E-01 /
      data fk( 369), fk( 370) / -4.8528797897E+00 , -9.4762101187E-01 /
      data fk( 371), fk( 372) / -4.9629360340E-01 , -9.4762101187E-01 /
      data fk( 373), fk( 374) / -4.8528797897E+00 , -9.2771171521E-02 /
      data fk( 375), fk( 376) / -9.2771171521E-02 , -6.9478694544E+00 /
      data fk( 377), fk( 378) / -4.9629360340E-01 ,  4.9435593164E-01 /
      data fk( 379), fk( 380) / -4.8528797897E+00 , -9.4762101187E-01 /
      data fk( 381), fk( 382) / -4.9629360340E-01 , -9.2771171521E-02 /
      data fk( 383), fk( 384) / -9.2771171521E-02 , -9.4762101187E-01 /
      data fk( 385), fk( 386) / -4.8528797897E+00 , -6.9478694544E+00 /
      data fk( 387), fk( 388) / -4.9629360340E-01 , -4.9629360340E-01 /
      data fk( 389), fk( 390) / -9.2771171521E-02 , -9.2771171521E-02 /
      data fk( 391), fk( 392) /  4.9435593164E-01 , -4.8528797897E+00 /
      data fk( 393), fk( 394) / -9.4762101187E-01 , -9.4762101187E-01 /
      data fk( 395), fk( 396) / -4.8528797897E+00 , -6.9478694544E+00 /
      data fk( 397), fk( 398) / -1.7266935341E+00 , -1.7266935341E+00 /
      data fk( 399), fk( 400) / -1.3072240750E+00 , -2.5835695992E+00 /
      data fk( 401), fk( 402) /  1.8573154781E+00 ,  1.0439463599E+00 /
      data fk( 403), fk( 404) / -2.5835695992E+00 ,  1.0439463599E+00 /
      data fk( 405), fk( 406) /  1.8573154781E+00 ,  1.3055040751E+01 /
      data fk( 407), fk( 408) / -1.7266935341E+00 , -1.3072240750E+00 /
      data fk( 409), fk( 410) / -1.7266935341E+00 ,  1.8573154781E+00 /
      data fk( 411), fk( 412) / -2.5835695992E+00 ,  1.0439463599E+00 /
      data fk( 413), fk( 414) /  1.0439463599E+00 , -2.5835695992E+00 /
      data fk( 415), fk( 416) /  1.8573154781E+00 ,  1.3055040751E+01 /
      data fk( 417), fk( 418) / -1.3072240750E+00 , -1.7266935341E+00 /
      data fk( 419), fk( 420) / -1.7266935341E+00 ,  1.8573154781E+00 /
      data fk( 421), fk( 422) /  1.0439463599E+00 , -2.5835695992E+00 /
      data fk( 423), fk( 424) /  1.0439463599E+00 ,  1.8573154781E+00 /
      data fk( 425), fk( 426) / -2.5835695992E+00 ,  1.3055040751E+01 /
      data fk( 427), fk( 428) /  7.9436621613E-01 ,  3.3213312280E+00 /
      data fk( 429), fk( 430) /  3.3213312280E+00 ,  1.1908621372E+00 /
      data fk( 431), fk( 432) / -4.0544328484E+00 , -6.0966167076E-01 /
      data fk( 433), fk( 434) / -4.0544328484E+00 ,  1.1908621372E+00 /
      data fk( 435), fk( 436) / -6.0966167076E-01 ,  4.0420903418E+00 /
      data fk( 437), fk( 438) /  7.9436621613E-01 ,  1.1908621372E+00 /
      data fk( 439), fk( 440) / -4.0544328484E+00 ,  3.3213312280E+00 /
      data fk( 441), fk( 442) /  3.3213312280E+00 , -6.0966167076E-01 /
      data fk( 443), fk( 444) / -4.0544328484E+00 ,  1.1908621372E+00 /
      data fk( 445), fk( 446) /  4.0420903418E+00 , -6.0966167076E-01 /
      data fk( 447), fk( 448) /  7.9436621613E-01 ,  1.1908621372E+00 /
      data fk( 449), fk( 450) / -4.0544328484E+00 , -4.0544328484E+00 /
      data fk( 451), fk( 452) /  1.1908621372E+00 ,  4.0420903418E+00 /
      data fk( 453), fk( 454) /  3.3213312280E+00 ,  3.3213312280E+00 /
      data fk( 455), fk( 456) / -6.0966167076E-01 , -6.0966167076E-01 /
      data fk( 457), fk( 458) /  4.0146071321E+00 ,  4.0146071321E+00 /
      data fk( 459), fk( 460) /  4.0146071321E+00 , -9.3146778774E+00 /
      data fk( 461), fk( 462) / -9.3146778774E+00 , -9.3146778774E+00 /
      data fk( 463), fk( 464) /  2.9832545515E-01 ,  2.9832545515E-01 /
      data fk( 465), fk( 466) /  2.9832545515E-01 ,  1.7150804225E+00 /
      data fk( 467), fk( 468) /  1.7150804225E+00 ,  1.7150804225E+00 /
      data fk( 469), fk( 470) / -2.6317054049E-01 , -2.6317054049E-01 /
      data fk( 471), fk( 472) / -5.8912155049E-02 , -5.8912155049E-02 /
      data fk( 473), fk( 474) /  2.6129441890E-01 , -2.6317054049E-01 /
      data fk( 475), fk( 476) / -2.6317054049E-01 , -5.8912155049E-02 /
      data fk( 477), fk( 478) /  2.6129441890E-01 , -5.8912155049E-02 /
      data fk( 479), fk( 480) / -2.6317054049E-01 , -2.6317054049E-01 /
      data fk( 481), fk( 482) /  2.6129441890E-01 , -5.8912155049E-02 /
      data fk( 483), fk( 484) / -5.8912155049E-02 , -8.1500034170E-01 /
      data fk( 485), fk( 486) / -8.1500034170E-01 ,  6.4455556445E-01 /
      data fk( 487), fk( 488) / -1.6368861574E+00 , -1.6368861574E+00 /
      data fk( 489), fk( 490) / -8.1500034170E-01 ,  6.4455556445E-01 /
      data fk( 491), fk( 492) / -8.1500034170E-01 , -1.6368861574E+00 /
      data fk( 493), fk( 494) / -1.6368861574E+00 ,  6.4455556445E-01 /
      data fk( 495), fk( 496) / -8.1500034170E-01 , -8.1500034170E-01 /
      data fk( 497), fk( 498) / -1.6368861574E+00 , -1.6368861574E+00 /
      data fk( 499), fk( 500) / -1.9061755028E-01 , -1.9061755028E-01 /
      data fk( 501), fk( 502) / -3.2302707780E-01 , -3.2302707780E-01 /
      data fk( 503), fk( 504) / -9.4688442650E-01 , -1.9061755028E-01 /
      data fk( 505), fk( 506) / -1.9061755028E-01 , -3.2302707780E-01 /
      data fk( 507), fk( 508) / -9.4688442650E-01 , -3.2302707780E-01 /
      data fk( 509), fk( 510) / -1.9061755028E-01 , -1.9061755028E-01 /
      data fk( 511), fk( 512) / -9.4688442650E-01 , -3.2302707780E-01 /
      data fk( 513), fk( 514) / -3.2302707780E-01 , -2.0555145793E+00 /
      data fk( 515), fk( 516) / -2.0555145793E+00 ,  4.3454820907E-01 /
      data fk( 517), fk( 518) / -8.6960826606E-01 , -8.6960826606E-01 /
      data fk( 519), fk( 520) / -2.0555145793E+00 ,  4.3454820907E-01 /
      data fk( 521), fk( 522) / -2.0555145793E+00 , -8.6960826606E-01 /
      data fk( 523), fk( 524) / -8.6960826606E-01 ,  4.3454820907E-01 /
      data fk( 525), fk( 526) / -2.0555145793E+00 , -2.0555145793E+00 /
      data fk( 527), fk( 528) / -8.6960826606E-01 , -8.6960826606E-01 /
      data fk( 529), fk( 530) / -1.7313735777E-01 , -1.7313735777E-01 /
      data fk( 531), fk( 532) /  4.7406364303E-01 ,  4.7406364303E-01 /
      data fk( 533), fk( 534) /  2.3544866466E-01 , -1.7313735777E-01 /
      data fk( 535), fk( 536) /  4.7406364303E-01 ,  2.3544866466E-01 /
      data fk( 537), fk( 538) /  4.7406364303E-01 ,  2.3544866466E-01 /
      data fk( 539), fk( 540) /  4.7406364303E-01 ,  4.7406364303E-01 /
      data fk( 541), fk( 542) /  2.7355644423E+00 ,  2.7355644423E+00 /
      data fk( 543), fk( 544) /  2.7355644423E+00 ,  2.7191032879E+01 /
      data fk( 545), fk( 546) /  2.0560828246E+00 ,  1.0527588969E+00 /
      data fk( 547), fk( 548) /  1.0527588969E+00 ,  1.0527588969E+00 /
      data fk( 549), fk( 550) /  2.0560828246E+00 ,  1.0527588969E+00 /
      data fk( 551), fk( 552) / -2.5036853561E-01 ,  1.1856525325E+00 /
      data fk( 553), fk( 554) /  1.1856525325E+00 ,  1.0527588969E+00 /
      data fk( 555), fk( 556) /  1.0527588969E+00 ,  2.0560828246E+00 /
      data fk( 557), fk( 558) /  1.1856525325E+00 , -2.5036853561E-01 /
      data fk( 559), fk( 560) /  1.1856525325E+00 ,  1.1856525325E+00 /
      data fk( 561), fk( 562) /  1.1856525325E+00 , -2.5036853561E-01 /
      data fk( 563), fk( 564) /  5.4687971763E-01 , -1.3228954165E-01 /
      data fk( 565), fk( 566) /  4.3005148899E-01 ,  5.5250243177E-01 /
      data fk( 567), fk( 568) / -3.0265783303E-01 ,  5.5250243177E-01 /
      data fk( 569), fk( 570) /  4.3005148899E-01 , -3.0265783303E-01 /
      data fk( 571), fk( 572) /  1.6388648096E+00 , -3.5906488422E-02 /
      data fk( 573), fk( 574) / -3.5906488422E-02 , -1.3228954165E-01 /
      data fk( 575), fk( 576) /  4.3005148899E-01 , -3.0265783303E-01 /
      data fk( 577), fk( 578) /  5.5250243177E-01 ,  5.5250243177E-01 /
      data fk( 579), fk( 580) / -3.0265783303E-01 ,  4.3005148899E-01 /
      data fk( 581), fk( 582) / -3.5906488422E-02 ,  1.6388648096E+00 /
      data fk( 583), fk( 584) / -3.5906488422E-02 , -1.3228954165E-01 /
      data fk( 585), fk( 586) / -3.0265783303E-01 ,  4.3005148899E-01 /
      data fk( 587), fk( 588) /  5.5250243177E-01 , -3.0265783303E-01 /
      data fk( 589), fk( 590) /  5.5250243177E-01 ,  4.3005148899E-01 /
      data fk( 591), fk( 592) / -3.5906488422E-02 , -3.5906488422E-02 /
      data fk( 593), fk( 594) /  1.6388648096E+00 ,  5.2762406119E+00 /
      data fk( 595), fk( 596) / -3.2859169809E-01 , -6.9902144354E-01 /
      data fk( 597), fk( 598) / -2.9016170467E-01 , -3.2859169809E-01 /
      data fk( 599), fk( 600) / -2.9016170467E-01 , -6.9902144354E-01 /
      data fk( 601), fk( 602) /  1.1844923449E+00 ,  1.1844923449E+00 /
      data fk( 603), fk( 604) /  6.8926165761E+00 , -3.2859169809E-01 /
      data fk( 605), fk( 606) /  5.2762406119E+00 , -6.9902144354E-01 /
      data fk( 607), fk( 608) / -2.9016170467E-01 , -3.2859169809E-01 /
      data fk( 609), fk( 610) /  1.1844923449E+00 ,  1.1844923449E+00 /
      data fk( 611), fk( 612) / -2.9016170467E-01 , -6.9902144354E-01 /
      data fk( 613), fk( 614) /  6.8926165761E+00 , -3.2859169809E-01 /
      data fk( 615), fk( 616) / -3.2859169809E-01 ,  1.1844923449E+00 /
      data fk( 617), fk( 618) /  1.1844923449E+00 ,  5.2762406119E+00 /
      data fk( 619), fk( 620) / -6.9902144354E-01 , -2.9016170467E-01 /
      data fk( 621), fk( 622) / -2.9016170467E-01 , -6.9902144354E-01 /
      data fk( 623), fk( 624) /  6.8926165761E+00 ,  1.2638264173E-01 /
      data fk( 625), fk( 626) /  9.3524795829E-01 ,  3.7575686486E-01 /
      data fk( 627), fk( 628) / -2.6955875557E-01 ,  1.2638264173E-01 /
      data fk( 629), fk( 630) /  3.7575686486E-01 ,  9.3524795829E-01 /
      data fk( 631), fk( 632) / -2.6955875557E-01 , -7.1668757407E-01 /
      data fk( 633), fk( 634) / -4.8835263692E-01 , -7.9900491644E-01 /
      data fk( 635), fk( 636) /  1.3869606008E+00 , -4.8835263692E-01 /
      data fk( 637), fk( 638) / -7.1668757407E-01 , -7.9900491644E-01 /
      data fk( 639), fk( 640) /  1.3869606008E+00 , -6.2781231534E-01 /
      data fk( 641), fk( 642) / -6.2781231534E-01 ,  5.5890803898E-01 /
      data fk( 643), fk( 644) /  5.5890803898E-01 ,  1.2638264173E-01 /
      data fk( 645), fk( 646) /  9.3524795829E-01 , -2.6955875557E-01 /
      data fk( 647), fk( 648) /  3.7575686486E-01 ,  1.2638264173E-01 /
      data fk( 649), fk( 650) /  3.7575686486E-01 , -2.6955875557E-01 /
      data fk( 651), fk( 652) /  9.3524795829E-01 , -7.1668757407E-01 /
      data fk( 653), fk( 654) / -4.8835263692E-01 ,  1.3869606008E+00 /
      data fk( 655), fk( 656) / -7.9900491644E-01 , -6.2781231534E-01 /
      data fk( 657), fk( 658) / -6.2781231534E-01 ,  5.5890803898E-01 /
      data fk( 659), fk( 660) /  5.5890803898E-01 , -4.8835263692E-01 /
      data fk( 661), fk( 662) / -7.1668757407E-01 , -7.9900491644E-01 /
      data fk( 663), fk( 664) /  1.3869606008E+00 ,  1.2638264173E-01 /
      data fk( 665), fk( 666) / -2.6955875557E-01 ,  9.3524795829E-01 /
      data fk( 667), fk( 668) /  3.7575686486E-01 ,  1.2638264173E-01 /
      data fk( 669), fk( 670) / -2.6955875557E-01 ,  3.7575686486E-01 /
      data fk( 671), fk( 672) /  9.3524795829E-01 , -6.2781231534E-01 /
      data fk( 673), fk( 674) / -6.2781231534E-01 ,  5.5890803898E-01 /
      data fk( 675), fk( 676) /  5.5890803898E-01 , -7.1668757407E-01 /
      data fk( 677), fk( 678) / -4.8835263692E-01 ,  1.3869606008E+00 /
      data fk( 679), fk( 680) / -7.9900491644E-01 , -4.8835263692E-01 /
      data fk( 681), fk( 682) / -7.1668757407E-01 ,  1.3869606008E+00 /
      data fk( 683), fk( 684) / -7.9900491644E-01 , -5.9305642448E-01 /
      data fk( 685), fk( 686) /  2.5623132936E-01 ,  1.6099728151E+00 /
      data fk( 687), fk( 688) /  1.0114964607E+00 , -5.9305642448E-01 /
      data fk( 689), fk( 690) /  2.5623132936E-01 ,  1.0114964607E+00 /
      data fk( 691), fk( 692) /  1.6099728151E+00 , -1.5967016681E-01 /
      data fk( 693), fk( 694) / -1.5967016681E-01 ,  3.2381222698E-01 /
      data fk( 695), fk( 696) /  3.2381222698E-01 ,  1.8005903721E+00 /
      data fk( 697), fk( 698) / -2.5780152111E+00 ,  1.8648436357E-01 /
      data fk( 699), fk( 700) / -6.1837644117E+00 , -2.5780152111E+00 /
      data fk( 701), fk( 702) /  1.8005903721E+00 ,  1.8648436357E-01 /
      data fk( 703), fk( 704) / -6.1837644117E+00 ,  2.5623132936E-01 /
      data fk( 705), fk( 706) / -5.9305642448E-01 ,  1.6099728151E+00 /
      data fk( 707), fk( 708) /  1.0114964607E+00 , -1.5967016681E-01 /
      data fk( 709), fk( 710) / -1.5967016681E-01 ,  3.2381222698E-01 /
      data fk( 711), fk( 712) /  3.2381222698E-01 , -5.9305642448E-01 /
      data fk( 713), fk( 714) /  2.5623132936E-01 ,  1.0114964607E+00 /
      data fk( 715), fk( 716) /  1.6099728151E+00 ,  1.8005903721E+00 /
      data fk( 717), fk( 718) /  1.8648436357E-01 , -2.5780152111E+00 /
      data fk( 719), fk( 720) / -6.1837644117E+00 , -2.5780152111E+00 /
      data fk( 721), fk( 722) /  1.8648436357E-01 ,  1.8005903721E+00 /
      data fk( 723), fk( 724) / -6.1837644117E+00 , -1.5967016681E-01 /
      data fk( 725), fk( 726) / -1.5967016681E-01 ,  3.2381222698E-01 /
      data fk( 727), fk( 728) /  3.2381222698E-01 ,  2.5623132936E-01 /
      data fk( 729), fk( 730) / -5.9305642448E-01 ,  1.6099728151E+00 /
      data fk( 731), fk( 732) /  1.0114964607E+00 ,  2.5623132936E-01 /
      data fk( 733), fk( 734) / -5.9305642448E-01 ,  1.0114964607E+00 /
      data fk( 735), fk( 736) /  1.6099728151E+00 ,  1.8648436357E-01 /
      data fk( 737), fk( 738) /  1.8005903721E+00 , -2.5780152111E+00 /
      data fk( 739), fk( 740) / -6.1837644117E+00 ,  1.8648436357E-01 /
      data fk( 741), fk( 742) / -2.5780152111E+00 ,  1.8005903721E+00 /
      data fk( 743), fk( 744) / -6.1837644117E+00 ,  6.1854118485E-02 /
      data fk( 745), fk( 746) /  6.1854118485E-02 ,  9.8167377643E-01 /
      data fk( 747), fk( 748) / -1.8957290186E+00 ,  2.9121750833E-04 /
      data fk( 749), fk( 750) /  9.6760049180E-01 , -1.8957290186E+00 /
      data fk( 751), fk( 752) /  9.6760049180E-01 ,  2.9121750833E-04 /
      data fk( 753), fk( 754) / -7.8710326121E-01 ,  6.1854118485E-02 /
      data fk( 755), fk( 756) /  9.8167377643E-01 ,  6.1854118485E-02 /
      data fk( 757), fk( 758) /  2.9121750833E-04 , -1.8957290186E+00 /
      data fk( 759), fk( 760) /  9.6760049180E-01 ,  9.6760049180E-01 /
      data fk( 761), fk( 762) / -1.8957290186E+00 ,  2.9121750833E-04 /
      data fk( 763), fk( 764) / -7.8710326121E-01 ,  9.8167377643E-01 /
      data fk( 765), fk( 766) /  6.1854118485E-02 ,  6.1854118485E-02 /
      data fk( 767), fk( 768) /  2.9121750833E-04 ,  9.6760049180E-01 /
      data fk( 769), fk( 770) / -1.8957290186E+00 ,  9.6760049180E-01 /
      data fk( 771), fk( 772) /  2.9121750833E-04 , -1.8957290186E+00 /
      data fk( 773), fk( 774) / -7.8710326121E-01 , -1.6949728942E-01 /
      data fk( 775), fk( 776) / -3.1182155678E+00 , -3.1182155678E+00 /
      data fk( 777), fk( 778) /  1.5909300850E-01 , -8.0160445750E-01 /
      data fk( 779), fk( 780) /  3.3298901892E+00 , -8.0160445750E-01 /
      data fk( 781), fk( 782) /  1.5909300850E-01 ,  3.3298901892E+00 /
      data fk( 783), fk( 784) / -6.7027407559E-01 , -1.6949728942E-01 /
      data fk( 785), fk( 786) /  1.5909300850E-01 , -8.0160445750E-01 /
      data fk( 787), fk( 788) / -3.1182155678E+00 , -3.1182155678E+00 /
      data fk( 789), fk( 790) /  3.3298901892E+00 , -8.0160445750E-01 /
      data fk( 791), fk( 792) /  1.5909300850E-01 , -6.7027407559E-01 /
      data fk( 793), fk( 794) /  3.3298901892E+00 , -1.6949728942E-01 /
      data fk( 795), fk( 796) /  1.5909300850E-01 , -8.0160445750E-01 /
      data fk( 797), fk( 798) / -8.0160445750E-01 ,  1.5909300850E-01 /
      data fk( 799), fk( 800) / -6.7027407559E-01 , -3.1182155678E+00 /
      data fk( 801), fk( 802) / -3.1182155678E+00 ,  3.3298901892E+00 /
      data fk( 803), fk( 804) /  3.3298901892E+00 , -1.7615391617E+00 /
      data fk( 805), fk( 806) / -5.8905975358E-01 , -5.8905975358E-01 /
      data fk( 807), fk( 808) / -1.6046105469E-01 , -1.6046105469E-01 /
      data fk( 809), fk( 810) / -1.0150372030E+00 , -5.8905975358E-01 /
      data fk( 811), fk( 812) / -1.7615391617E+00 , -5.8905975358E-01 /
      data fk( 813), fk( 814) / -1.6046105469E-01 , -1.0150372030E+00 /
      data fk( 815), fk( 816) / -1.6046105469E-01 ,  3.4652680479E-01 /
      data fk( 817), fk( 818) /  2.7393244557E+00 , -1.1142879534E+00 /
      data fk( 819), fk( 820) / -5.9223755604E-01 , -2.0704075684E+00 /
      data fk( 821), fk( 822) / -3.2608708860E+00 ,  3.4652680479E-01 /
      data fk( 823), fk( 824) / -5.9223755604E-01 , -2.0704075684E+00 /
      data fk( 825), fk( 826) /  2.7393244557E+00 , -1.1142879534E+00 /
      data fk( 827), fk( 828) / -3.2608708860E+00 , -4.7948128757E-01 /
      data fk( 829), fk( 830) / -1.4849145201E+00 ,  1.1981070563E-01 /
      data fk( 831), fk( 832) /  1.1981070563E-01 , -1.4849145201E+00 /
      data fk( 833), fk( 834) / -8.8993153684E-01 , -5.8905975358E-01 /
      data fk( 835), fk( 836) / -5.8905975358E-01 , -1.7615391617E+00 /
      data fk( 837), fk( 838) / -1.0150372030E+00 , -1.6046105469E-01 /
      data fk( 839), fk( 840) / -1.6046105469E-01 ,  3.4652680479E-01 /
      data fk( 841), fk( 842) / -1.1142879534E+00 ,  2.7393244557E+00 /
      data fk( 843), fk( 844) / -2.0704075684E+00 , -5.9223755604E-01 /
      data fk( 845), fk( 846) / -3.2608708860E+00 , -4.7948128757E-01 /
      data fk( 847), fk( 848) / -1.4849145201E+00 ,  1.1981070563E-01 /
      data fk( 849), fk( 850) /  1.1981070563E-01 , -1.4849145201E+00 /
      data fk( 851), fk( 852) / -8.8993153684E-01 ,  3.4652680479E-01 /
      data fk( 853), fk( 854) / -5.9223755604E-01 , -2.0704075684E+00 /
      data fk( 855), fk( 856) /  2.7393244557E+00 , -1.1142879534E+00 /
      data fk( 857), fk( 858) / -3.2608708860E+00 , -4.7948128757E-01 /
      data fk( 859), fk( 860) / -1.4849145201E+00 ,  1.1981070563E-01 /
      data fk( 861), fk( 862) /  1.1981070563E-01 , -1.4849145201E+00 /
      data fk( 863), fk( 864) / -8.8993153684E-01 ,  3.4652680479E-01 /
      data fk( 865), fk( 866) / -1.1142879534E+00 ,  2.7393244557E+00 /
      data fk( 867), fk( 868) / -2.0704075684E+00 , -5.9223755604E-01 /
      data fk( 869), fk( 870) / -3.2608708860E+00 ,  3.4652680479E-01 /
      data fk( 871), fk( 872) / -2.0704075684E+00 , -5.9223755604E-01 /
      data fk( 873), fk( 874) / -1.1142879534E+00 ,  2.7393244557E+00 /
      data fk( 875), fk( 876) / -3.2608708860E+00 , -1.2957609931E+00 /
      data fk( 877), fk( 878) / -1.2957609931E+00 , -4.6247669604E+00 /
      data fk( 879), fk( 880) /  4.2510689410E+00 , -2.9471458348E+00 /
      data fk( 881), fk( 882) / -2.9471458348E+00 , -1.2957609931E+00 /
      data fk( 883), fk( 884) /  4.2510689410E+00 , -2.9471458348E+00 /
      data fk( 885), fk( 886) / -1.2957609931E+00 , -4.6247669604E+00 /
      data fk( 887), fk( 888) / -2.9471458348E+00 ,  4.2510689410E+00 /
      data fk( 889), fk( 890) / -1.2957609931E+00 , -2.9471458348E+00 /
      data fk( 891), fk( 892) / -1.2957609931E+00 , -2.9471458348E+00 /
      data fk( 893), fk( 894) / -4.6247669604E+00 ,  1.6567315586E+00 /
      data fk( 895), fk( 896) /  2.3688012651E+00 ,  2.3688012651E+00 /
      data fk( 897), fk( 898) / -2.4352198680E-01 , -2.4352198680E-01 /
      data fk( 899), fk( 900) /  2.3688012651E+00 ,  1.6567315586E+00 /
      data fk( 901), fk( 902) /  2.3688012651E+00 , -2.4352198680E-01 /
      data fk( 903), fk( 904) / -2.4352198680E-01 ,  2.3688012651E+00 /
      data fk( 905), fk( 906) /  2.3688012651E+00 ,  1.6567315586E+00 /
      data fk( 907), fk( 908) / -2.4352198680E-01 , -2.4352198680E-01 /
      data fk( 909), fk( 910) / -7.9066738310E-01 , -7.9066738310E-01 /
      data fk( 911), fk( 912) /  5.3371929740E+00 ,  3.8362394105E+00 /
      data fk( 913), fk( 914) /  3.8362394105E+00 , -7.9066738310E-01 /
      data fk( 915), fk( 916) / -7.9066738310E-01 ,  3.8362394105E+00 /
      data fk( 917), fk( 918) /  5.3371929740E+00 ,  3.8362394105E+00 /
      data fk( 919), fk( 920) / -7.9066738310E-01 , -7.9066738310E-01 /
      data fk( 921), fk( 922) /  3.8362394105E+00 ,  3.8362394105E+00 /
      data fk( 923), fk( 924) /  5.3371929740E+00 , -1.1051861635E+01 /

      data shift  / 2.12d0, 2.12d0, 2.12d0, 1.41d0, 1.41d0, 1.41d0 /
      data alpham / 6*0.5d0 /
      data nmax,nvar /6,6/
   
!
      en=0.0d0
      do i=1,nvar
       fmr(i,1) = 1.0d0- exp(-alpham(i)*(rij(i)-shift(i)))
       do j=2,nmax
       fmr(i,j) = fmr(i,1)**j
       enddo
      enddo
!
      en=en+fk(1) 
      en=en+fk(2)*fmr(1,1) 
      en=en+fk(3)*fmr(2,1) 
      en=en+fk(4)*fmr(3,1) 
      en=en+fk(5)*fmr(4,1) 
      en=en+fk(6)*fmr(5,1) 
      en=en+fk(7)*fmr(6,1) 
      en=en+fk(8)*fmr(1,2) 
      en=en+fk(9)*fmr(2,2) 
      en=en+fk(10)*fmr(3,2) 
      en=en+fk(11)*fmr(4,2) 
      en=en+fk(12)*fmr(5,2) 
      en=en+fk(13)*fmr(6,2) 
      en=en+fk(14)*fmr(1,1)*fmr(2,1) 
      en=en+fk(15)*fmr(1,1)*fmr(3,1) 
      en=en+fk(16)*fmr(1,1)*fmr(4,1) 
      en=en+fk(17)*fmr(1,1)*fmr(5,1) 
      en=en+fk(18)*fmr(1,1)*fmr(6,1) 
      en=en+fk(19)*fmr(2,1)*fmr(3,1) 
      en=en+fk(20)*fmr(2,1)*fmr(4,1) 
      en=en+fk(21)*fmr(2,1)*fmr(5,1) 
      en=en+fk(22)*fmr(2,1)*fmr(6,1) 
      en=en+fk(23)*fmr(3,1)*fmr(4,1) 
      en=en+fk(24)*fmr(3,1)*fmr(5,1) 
      en=en+fk(25)*fmr(3,1)*fmr(6,1) 
      en=en+fk(26)*fmr(4,1)*fmr(5,1) 
      en=en+fk(27)*fmr(4,1)*fmr(6,1) 
      en=en+fk(28)*fmr(5,1)*fmr(6,1) 
      en=en+fk(29)*fmr(1,3) 
      en=en+fk(30)*fmr(2,3) 
      en=en+fk(31)*fmr(3,3) 
      en=en+fk(32)*fmr(4,3) 
      en=en+fk(33)*fmr(5,3) 
      en=en+fk(34)*fmr(6,3) 
      en=en+fk(35)*fmr(1,2)*fmr(2,1) 
      en=en+fk(36)*fmr(1,2)*fmr(3,1) 
      en=en+fk(37)*fmr(1,2)*fmr(4,1) 
      en=en+fk(38)*fmr(1,2)*fmr(5,1) 
      en=en+fk(39)*fmr(1,2)*fmr(6,1) 
      en=en+fk(40)*fmr(1,1)*fmr(2,2) 
      en=en+fk(41)*fmr(2,2)*fmr(3,1) 
      en=en+fk(42)*fmr(2,2)*fmr(4,1) 
      en=en+fk(43)*fmr(2,2)*fmr(5,1) 
      en=en+fk(44)*fmr(2,2)*fmr(6,1) 
      en=en+fk(45)*fmr(1,1)*fmr(3,2) 
      en=en+fk(46)*fmr(2,1)*fmr(3,2) 
      en=en+fk(47)*fmr(3,2)*fmr(4,1) 
      en=en+fk(48)*fmr(3,2)*fmr(5,1) 
      en=en+fk(49)*fmr(3,2)*fmr(6,1) 
      en=en+fk(50)*fmr(1,1)*fmr(4,2) 
      en=en+fk(51)*fmr(2,1)*fmr(4,2) 
      en=en+fk(52)*fmr(3,1)*fmr(4,2) 
      en=en+fk(53)*fmr(4,2)*fmr(5,1) 
      en=en+fk(54)*fmr(4,2)*fmr(6,1) 
      en=en+fk(55)*fmr(1,1)*fmr(5,2) 
      en=en+fk(56)*fmr(2,1)*fmr(5,2) 
      en=en+fk(57)*fmr(3,1)*fmr(5,2) 
      en=en+fk(58)*fmr(4,1)*fmr(5,2) 
      en=en+fk(59)*fmr(5,2)*fmr(6,1) 
      en=en+fk(60)*fmr(1,1)*fmr(6,2) 
      en=en+fk(61)*fmr(2,1)*fmr(6,2) 
      en=en+fk(62)*fmr(3,1)*fmr(6,2) 
      en=en+fk(63)*fmr(4,1)*fmr(6,2) 
      en=en+fk(64)*fmr(5,1)*fmr(6,2) 
      en=en+fk(65)*fmr(1,1)*fmr(2,1)*fmr(3,1) 
      en=en+fk(66)*fmr(1,1)*fmr(2,1)*fmr(4,1) 
      en=en+fk(67)*fmr(1,1)*fmr(2,1)*fmr(5,1) 
      en=en+fk(68)*fmr(1,1)*fmr(2,1)*fmr(6,1) 
      en=en+fk(69)*fmr(1,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(70)*fmr(1,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(71)*fmr(1,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(72)*fmr(1,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(73)*fmr(1,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(74)*fmr(1,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(75)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(76)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(77)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(78)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(79)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(80)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(81)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(82)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(83)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(84)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(85)*fmr(1,4) 
      en=en+fk(86)*fmr(2,4) 
      en=en+fk(87)*fmr(3,4) 
      en=en+fk(88)*fmr(4,4) 
      en=en+fk(89)*fmr(5,4) 
      en=en+fk(90)*fmr(6,4) 
      en=en+fk(91)*fmr(1,3)*fmr(2,1) 
      en=en+fk(92)*fmr(1,3)*fmr(3,1) 
      en=en+fk(93)*fmr(1,3)*fmr(4,1) 
      en=en+fk(94)*fmr(1,3)*fmr(5,1) 
      en=en+fk(95)*fmr(1,3)*fmr(6,1) 
      en=en+fk(96)*fmr(1,1)*fmr(2,3) 
      en=en+fk(97)*fmr(2,3)*fmr(3,1) 
      en=en+fk(98)*fmr(2,3)*fmr(4,1) 
      en=en+fk(99)*fmr(2,3)*fmr(5,1) 
      en=en+fk(100)*fmr(2,3)*fmr(6,1) 
      en=en+fk(101)*fmr(1,1)*fmr(3,3) 
      en=en+fk(102)*fmr(2,1)*fmr(3,3) 
      en=en+fk(103)*fmr(3,3)*fmr(4,1) 
      en=en+fk(104)*fmr(3,3)*fmr(5,1) 
      en=en+fk(105)*fmr(3,3)*fmr(6,1) 
      en=en+fk(106)*fmr(1,1)*fmr(4,3) 
      en=en+fk(107)*fmr(2,1)*fmr(4,3) 
      en=en+fk(108)*fmr(3,1)*fmr(4,3) 
      en=en+fk(109)*fmr(4,3)*fmr(5,1) 
      en=en+fk(110)*fmr(4,3)*fmr(6,1) 
      en=en+fk(111)*fmr(1,1)*fmr(5,3) 
      en=en+fk(112)*fmr(2,1)*fmr(5,3) 
      en=en+fk(113)*fmr(3,1)*fmr(5,3) 
      en=en+fk(114)*fmr(4,1)*fmr(5,3) 
      en=en+fk(115)*fmr(5,3)*fmr(6,1) 
      en=en+fk(116)*fmr(1,1)*fmr(6,3) 
      en=en+fk(117)*fmr(2,1)*fmr(6,3) 
      en=en+fk(118)*fmr(3,1)*fmr(6,3) 
      en=en+fk(119)*fmr(4,1)*fmr(6,3) 
      en=en+fk(120)*fmr(5,1)*fmr(6,3) 
      en=en+fk(121)*fmr(1,2)*fmr(2,2) 
      en=en+fk(122)*fmr(1,2)*fmr(3,2) 
      en=en+fk(123)*fmr(1,2)*fmr(4,2) 
      en=en+fk(124)*fmr(1,2)*fmr(5,2) 
      en=en+fk(125)*fmr(1,2)*fmr(6,2) 
      en=en+fk(126)*fmr(2,2)*fmr(3,2) 
      en=en+fk(127)*fmr(2,2)*fmr(4,2) 
      en=en+fk(128)*fmr(2,2)*fmr(5,2) 
      en=en+fk(129)*fmr(2,2)*fmr(6,2) 
      en=en+fk(130)*fmr(3,2)*fmr(4,2) 
      en=en+fk(131)*fmr(3,2)*fmr(5,2) 
      en=en+fk(132)*fmr(3,2)*fmr(6,2) 
      en=en+fk(133)*fmr(4,2)*fmr(5,2) 
      en=en+fk(134)*fmr(4,2)*fmr(6,2) 
      en=en+fk(135)*fmr(5,2)*fmr(6,2) 
      en=en+fk(136)*fmr(1,2)*fmr(2,1)*fmr(3,1) 
      en=en+fk(137)*fmr(1,2)*fmr(2,1)*fmr(4,1) 
      en=en+fk(138)*fmr(1,2)*fmr(2,1)*fmr(5,1) 
      en=en+fk(139)*fmr(1,2)*fmr(2,1)*fmr(6,1) 
      en=en+fk(140)*fmr(1,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(141)*fmr(1,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(142)*fmr(1,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(143)*fmr(1,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(144)*fmr(1,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(145)*fmr(1,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(146)*fmr(1,1)*fmr(2,2)*fmr(3,1) 
      en=en+fk(147)*fmr(1,1)*fmr(2,2)*fmr(4,1) 
      en=en+fk(148)*fmr(1,1)*fmr(2,2)*fmr(5,1) 
      en=en+fk(149)*fmr(1,1)*fmr(2,2)*fmr(6,1) 
      en=en+fk(150)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(151)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(152)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(153)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(154)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(155)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(156)*fmr(1,1)*fmr(2,1)*fmr(3,2) 
      en=en+fk(157)*fmr(1,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(158)*fmr(1,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(159)*fmr(1,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(160)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(161)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(162)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(163)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(164)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(165)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(166)*fmr(1,1)*fmr(2,1)*fmr(4,2) 
      en=en+fk(167)*fmr(1,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(168)*fmr(1,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(169)*fmr(1,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(170)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(171)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(172)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(173)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(174)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(175)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(176)*fmr(1,1)*fmr(2,1)*fmr(5,2) 
      en=en+fk(177)*fmr(1,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(178)*fmr(1,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(179)*fmr(1,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(180)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(181)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(182)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(183)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(184)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(185)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(186)*fmr(1,1)*fmr(2,1)*fmr(6,2) 
      en=en+fk(187)*fmr(1,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(188)*fmr(1,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(189)*fmr(1,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(190)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(191)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(192)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(193)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(194)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(195)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(196)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(197)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(198)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(199)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(200)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(201)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(202)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(203)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(204)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(205)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(206)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(207)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(208)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(209)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(210)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(211)*fmr(1,5) 
      en=en+fk(212)*fmr(2,5) 
      en=en+fk(213)*fmr(3,5) 
      en=en+fk(214)*fmr(4,5) 
      en=en+fk(215)*fmr(5,5) 
      en=en+fk(216)*fmr(6,5) 
      en=en+fk(217)*fmr(1,4)*fmr(2,1) 
      en=en+fk(218)*fmr(1,4)*fmr(3,1) 
      en=en+fk(219)*fmr(1,4)*fmr(4,1) 
      en=en+fk(220)*fmr(1,4)*fmr(5,1) 
      en=en+fk(221)*fmr(1,4)*fmr(6,1) 
      en=en+fk(222)*fmr(1,1)*fmr(2,4) 
      en=en+fk(223)*fmr(2,4)*fmr(3,1) 
      en=en+fk(224)*fmr(2,4)*fmr(4,1) 
      en=en+fk(225)*fmr(2,4)*fmr(5,1) 
      en=en+fk(226)*fmr(2,4)*fmr(6,1) 
      en=en+fk(227)*fmr(1,1)*fmr(3,4) 
      en=en+fk(228)*fmr(2,1)*fmr(3,4) 
      en=en+fk(229)*fmr(3,4)*fmr(4,1) 
      en=en+fk(230)*fmr(3,4)*fmr(5,1) 
      en=en+fk(231)*fmr(3,4)*fmr(6,1) 
      en=en+fk(232)*fmr(1,1)*fmr(4,4) 
      en=en+fk(233)*fmr(2,1)*fmr(4,4) 
      en=en+fk(234)*fmr(3,1)*fmr(4,4) 
      en=en+fk(235)*fmr(4,4)*fmr(5,1) 
      en=en+fk(236)*fmr(4,4)*fmr(6,1) 
      en=en+fk(237)*fmr(1,1)*fmr(5,4) 
      en=en+fk(238)*fmr(2,1)*fmr(5,4) 
      en=en+fk(239)*fmr(3,1)*fmr(5,4) 
      en=en+fk(240)*fmr(4,1)*fmr(5,4) 
      en=en+fk(241)*fmr(5,4)*fmr(6,1) 
      en=en+fk(242)*fmr(1,1)*fmr(6,4) 
      en=en+fk(243)*fmr(2,1)*fmr(6,4) 
      en=en+fk(244)*fmr(3,1)*fmr(6,4) 
      en=en+fk(245)*fmr(4,1)*fmr(6,4) 
      en=en+fk(246)*fmr(5,1)*fmr(6,4) 
      en=en+fk(247)*fmr(1,3)*fmr(2,2) 
      en=en+fk(248)*fmr(1,3)*fmr(3,2) 
      en=en+fk(249)*fmr(1,3)*fmr(4,2) 
      en=en+fk(250)*fmr(1,3)*fmr(5,2) 
      en=en+fk(251)*fmr(1,3)*fmr(6,2) 
      en=en+fk(252)*fmr(1,2)*fmr(2,3) 
      en=en+fk(253)*fmr(2,3)*fmr(3,2) 
      en=en+fk(254)*fmr(2,3)*fmr(4,2) 
      en=en+fk(255)*fmr(2,3)*fmr(5,2) 
      en=en+fk(256)*fmr(2,3)*fmr(6,2) 
      en=en+fk(257)*fmr(1,2)*fmr(3,3) 
      en=en+fk(258)*fmr(2,2)*fmr(3,3) 
      en=en+fk(259)*fmr(3,3)*fmr(4,2) 
      en=en+fk(260)*fmr(3,3)*fmr(5,2) 
      en=en+fk(261)*fmr(3,3)*fmr(6,2) 
      en=en+fk(262)*fmr(1,2)*fmr(4,3) 
      en=en+fk(263)*fmr(2,2)*fmr(4,3) 
      en=en+fk(264)*fmr(3,2)*fmr(4,3) 
      en=en+fk(265)*fmr(4,3)*fmr(5,2) 
      en=en+fk(266)*fmr(4,3)*fmr(6,2) 
      en=en+fk(267)*fmr(1,2)*fmr(5,3) 
      en=en+fk(268)*fmr(2,2)*fmr(5,3) 
      en=en+fk(269)*fmr(3,2)*fmr(5,3) 
      en=en+fk(270)*fmr(4,2)*fmr(5,3) 
      en=en+fk(271)*fmr(5,3)*fmr(6,2) 
      en=en+fk(272)*fmr(1,2)*fmr(6,3) 
      en=en+fk(273)*fmr(2,2)*fmr(6,3) 
      en=en+fk(274)*fmr(3,2)*fmr(6,3) 
      en=en+fk(275)*fmr(4,2)*fmr(6,3) 
      en=en+fk(276)*fmr(5,2)*fmr(6,3) 
      en=en+fk(277)*fmr(1,3)*fmr(2,1)*fmr(3,1) 
      en=en+fk(278)*fmr(1,3)*fmr(2,1)*fmr(4,1) 
      en=en+fk(279)*fmr(1,3)*fmr(2,1)*fmr(5,1) 
      en=en+fk(280)*fmr(1,3)*fmr(2,1)*fmr(6,1) 
      en=en+fk(281)*fmr(1,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(282)*fmr(1,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(283)*fmr(1,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(284)*fmr(1,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(285)*fmr(1,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(286)*fmr(1,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(287)*fmr(1,1)*fmr(2,3)*fmr(3,1) 
      en=en+fk(288)*fmr(1,1)*fmr(2,3)*fmr(4,1) 
      en=en+fk(289)*fmr(1,1)*fmr(2,3)*fmr(5,1) 
      en=en+fk(290)*fmr(1,1)*fmr(2,3)*fmr(6,1) 
      en=en+fk(291)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(292)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(293)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(294)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(295)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(296)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(297)*fmr(1,1)*fmr(2,1)*fmr(3,3) 
      en=en+fk(298)*fmr(1,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(299)*fmr(1,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(300)*fmr(1,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(301)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(302)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(303)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(304)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(305)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(306)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(307)*fmr(1,1)*fmr(2,1)*fmr(4,3) 
      en=en+fk(308)*fmr(1,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(309)*fmr(1,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(310)*fmr(1,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(311)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(312)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(313)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(314)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(315)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(316)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(317)*fmr(1,1)*fmr(2,1)*fmr(5,3) 
      en=en+fk(318)*fmr(1,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(319)*fmr(1,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(320)*fmr(1,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(321)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(322)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(323)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(324)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(325)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(326)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(327)*fmr(1,1)*fmr(2,1)*fmr(6,3) 
      en=en+fk(328)*fmr(1,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(329)*fmr(1,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(330)*fmr(1,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(331)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(332)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(333)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(334)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(335)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(336)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(337)*fmr(1,1)*fmr(2,2)*fmr(3,2) 
      en=en+fk(338)*fmr(1,1)*fmr(2,2)*fmr(4,2) 
      en=en+fk(339)*fmr(1,1)*fmr(2,2)*fmr(5,2) 
      en=en+fk(340)*fmr(1,1)*fmr(2,2)*fmr(6,2) 
      en=en+fk(341)*fmr(1,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(342)*fmr(1,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(343)*fmr(1,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(344)*fmr(1,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(345)*fmr(1,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(346)*fmr(1,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(347)*fmr(1,2)*fmr(2,1)*fmr(3,2) 
      en=en+fk(348)*fmr(1,2)*fmr(2,1)*fmr(4,2) 
      en=en+fk(349)*fmr(1,2)*fmr(2,1)*fmr(5,2) 
      en=en+fk(350)*fmr(1,2)*fmr(2,1)*fmr(6,2) 
      en=en+fk(351)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(352)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(353)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(354)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(355)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(356)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(357)*fmr(1,2)*fmr(2,2)*fmr(3,1) 
      en=en+fk(358)*fmr(1,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(359)*fmr(1,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(360)*fmr(1,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(361)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(362)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(363)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(364)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(365)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(366)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(367)*fmr(1,2)*fmr(2,2)*fmr(4,1) 
      en=en+fk(368)*fmr(1,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(369)*fmr(1,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(370)*fmr(1,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(371)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(372)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(373)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(374)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(375)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(376)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(377)*fmr(1,2)*fmr(2,2)*fmr(5,1) 
      en=en+fk(378)*fmr(1,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(379)*fmr(1,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(380)*fmr(1,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(381)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(382)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(383)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(384)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(385)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(386)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(387)*fmr(1,2)*fmr(2,2)*fmr(6,1) 
      en=en+fk(388)*fmr(1,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(389)*fmr(1,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(390)*fmr(1,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(391)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(392)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(393)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(394)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(395)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(396)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(397)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(398)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(399)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(400)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(401)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(402)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(403)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(404)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(405)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(406)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(407)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(408)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(409)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(410)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(411)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(412)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(413)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(414)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(415)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(416)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(417)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(418)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(419)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(420)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(421)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(422)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(423)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(424)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(425)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(426)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(427)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(428)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(429)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(430)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(431)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(432)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(433)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(434)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(435)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(436)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(437)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(438)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(439)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(440)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(441)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(442)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(443)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(444)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(445)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(446)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(447)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(448)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(449)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(450)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(451)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(452)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(453)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(454)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(455)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(456)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(457)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(458)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(459)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(460)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(461)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(462)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(463)*fmr(1,6) 
      en=en+fk(464)*fmr(2,6) 
      en=en+fk(465)*fmr(3,6) 
      en=en+fk(466)*fmr(4,6) 
      en=en+fk(467)*fmr(5,6) 
      en=en+fk(468)*fmr(6,6) 
      en=en+fk(469)*fmr(1,5)*fmr(2,1) 
      en=en+fk(470)*fmr(1,5)*fmr(3,1) 
      en=en+fk(471)*fmr(1,5)*fmr(4,1) 
      en=en+fk(472)*fmr(1,5)*fmr(5,1) 
      en=en+fk(473)*fmr(1,5)*fmr(6,1) 
      en=en+fk(474)*fmr(1,1)*fmr(2,5) 
      en=en+fk(475)*fmr(2,5)*fmr(3,1) 
      en=en+fk(476)*fmr(2,5)*fmr(4,1) 
      en=en+fk(477)*fmr(2,5)*fmr(5,1) 
      en=en+fk(478)*fmr(2,5)*fmr(6,1) 
      en=en+fk(479)*fmr(1,1)*fmr(3,5) 
      en=en+fk(480)*fmr(2,1)*fmr(3,5) 
      en=en+fk(481)*fmr(3,5)*fmr(4,1) 
      en=en+fk(482)*fmr(3,5)*fmr(5,1) 
      en=en+fk(483)*fmr(3,5)*fmr(6,1) 
      en=en+fk(484)*fmr(1,1)*fmr(4,5) 
      en=en+fk(485)*fmr(2,1)*fmr(4,5) 
      en=en+fk(486)*fmr(3,1)*fmr(4,5) 
      en=en+fk(487)*fmr(4,5)*fmr(5,1) 
      en=en+fk(488)*fmr(4,5)*fmr(6,1) 
      en=en+fk(489)*fmr(1,1)*fmr(5,5) 
      en=en+fk(490)*fmr(2,1)*fmr(5,5) 
      en=en+fk(491)*fmr(3,1)*fmr(5,5) 
      en=en+fk(492)*fmr(4,1)*fmr(5,5) 
      en=en+fk(493)*fmr(5,5)*fmr(6,1) 
      en=en+fk(494)*fmr(1,1)*fmr(6,5) 
      en=en+fk(495)*fmr(2,1)*fmr(6,5) 
      en=en+fk(496)*fmr(3,1)*fmr(6,5) 
      en=en+fk(497)*fmr(4,1)*fmr(6,5) 
      en=en+fk(498)*fmr(5,1)*fmr(6,5) 
      en=en+fk(499)*fmr(1,4)*fmr(2,2) 
      en=en+fk(500)*fmr(1,4)*fmr(3,2) 
      en=en+fk(501)*fmr(1,4)*fmr(4,2) 
      en=en+fk(502)*fmr(1,4)*fmr(5,2) 
      en=en+fk(503)*fmr(1,4)*fmr(6,2) 
      en=en+fk(504)*fmr(1,2)*fmr(2,4) 
      en=en+fk(505)*fmr(2,4)*fmr(3,2) 
      en=en+fk(506)*fmr(2,4)*fmr(4,2) 
      en=en+fk(507)*fmr(2,4)*fmr(5,2) 
      en=en+fk(508)*fmr(2,4)*fmr(6,2) 
      en=en+fk(509)*fmr(1,2)*fmr(3,4) 
      en=en+fk(510)*fmr(2,2)*fmr(3,4) 
      en=en+fk(511)*fmr(3,4)*fmr(4,2) 
      en=en+fk(512)*fmr(3,4)*fmr(5,2) 
      en=en+fk(513)*fmr(3,4)*fmr(6,2) 
      en=en+fk(514)*fmr(1,2)*fmr(4,4) 
      en=en+fk(515)*fmr(2,2)*fmr(4,4) 
      en=en+fk(516)*fmr(3,2)*fmr(4,4) 
      en=en+fk(517)*fmr(4,4)*fmr(5,2) 
      en=en+fk(518)*fmr(4,4)*fmr(6,2) 
      en=en+fk(519)*fmr(1,2)*fmr(5,4) 
      en=en+fk(520)*fmr(2,2)*fmr(5,4) 
      en=en+fk(521)*fmr(3,2)*fmr(5,4) 
      en=en+fk(522)*fmr(4,2)*fmr(5,4) 
      en=en+fk(523)*fmr(5,4)*fmr(6,2) 
      en=en+fk(524)*fmr(1,2)*fmr(6,4) 
      en=en+fk(525)*fmr(2,2)*fmr(6,4) 
      en=en+fk(526)*fmr(3,2)*fmr(6,4) 
      en=en+fk(527)*fmr(4,2)*fmr(6,4) 
      en=en+fk(528)*fmr(5,2)*fmr(6,4) 
      en=en+fk(529)*fmr(1,3)*fmr(2,3) 
      en=en+fk(530)*fmr(1,3)*fmr(3,3) 
      en=en+fk(531)*fmr(1,3)*fmr(4,3) 
      en=en+fk(532)*fmr(1,3)*fmr(5,3) 
      en=en+fk(533)*fmr(1,3)*fmr(6,3) 
      en=en+fk(534)*fmr(2,3)*fmr(3,3) 
      en=en+fk(535)*fmr(2,3)*fmr(4,3) 
      en=en+fk(536)*fmr(2,3)*fmr(5,3) 
      en=en+fk(537)*fmr(2,3)*fmr(6,3) 
      en=en+fk(538)*fmr(3,3)*fmr(4,3) 
      en=en+fk(539)*fmr(3,3)*fmr(5,3) 
      en=en+fk(540)*fmr(3,3)*fmr(6,3) 
      en=en+fk(541)*fmr(4,3)*fmr(5,3) 
      en=en+fk(542)*fmr(4,3)*fmr(6,3) 
      en=en+fk(543)*fmr(5,3)*fmr(6,3) 
      en=en+fk(544)*fmr(4,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(545)*fmr(3,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(546)*fmr(3,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(547)*fmr(3,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(548)*fmr(2,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(549)*fmr(2,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(550)*fmr(2,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(551)*fmr(2,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(552)*fmr(2,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(553)*fmr(2,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(554)*fmr(1,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(555)*fmr(1,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(556)*fmr(1,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(557)*fmr(1,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(558)*fmr(1,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(559)*fmr(1,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(560)*fmr(1,2)*fmr(2,2)*fmr(6,2) 
      en=en+fk(561)*fmr(1,2)*fmr(2,2)*fmr(5,2) 
      en=en+fk(562)*fmr(1,2)*fmr(2,2)*fmr(4,2) 
      en=en+fk(563)*fmr(1,2)*fmr(2,2)*fmr(3,2) 
      en=en+fk(564)*fmr(1,4)*fmr(2,1)*fmr(3,1) 
      en=en+fk(565)*fmr(1,4)*fmr(2,1)*fmr(4,1) 
      en=en+fk(566)*fmr(1,4)*fmr(2,1)*fmr(5,1) 
      en=en+fk(567)*fmr(1,4)*fmr(2,1)*fmr(6,1) 
      en=en+fk(568)*fmr(1,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(569)*fmr(1,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(570)*fmr(1,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(571)*fmr(1,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(572)*fmr(1,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(573)*fmr(1,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(574)*fmr(1,1)*fmr(2,4)*fmr(3,1) 
      en=en+fk(575)*fmr(1,1)*fmr(2,4)*fmr(4,1) 
      en=en+fk(576)*fmr(1,1)*fmr(2,4)*fmr(5,1) 
      en=en+fk(577)*fmr(1,1)*fmr(2,4)*fmr(6,1) 
      en=en+fk(578)*fmr(2,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(579)*fmr(2,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(580)*fmr(2,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(581)*fmr(2,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(582)*fmr(2,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(583)*fmr(2,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(584)*fmr(1,1)*fmr(2,1)*fmr(3,4) 
      en=en+fk(585)*fmr(1,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(586)*fmr(1,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(587)*fmr(1,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(588)*fmr(2,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(589)*fmr(2,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(590)*fmr(2,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(591)*fmr(3,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(592)*fmr(3,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(593)*fmr(3,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(594)*fmr(1,1)*fmr(2,1)*fmr(4,4) 
      en=en+fk(595)*fmr(1,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(596)*fmr(1,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(597)*fmr(1,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(598)*fmr(2,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(599)*fmr(2,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(600)*fmr(2,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(601)*fmr(3,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(602)*fmr(3,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(603)*fmr(4,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(604)*fmr(1,1)*fmr(2,1)*fmr(5,4) 
      en=en+fk(605)*fmr(1,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(606)*fmr(1,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(607)*fmr(1,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(608)*fmr(2,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(609)*fmr(2,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(610)*fmr(2,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(611)*fmr(3,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(612)*fmr(3,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(613)*fmr(4,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(614)*fmr(1,1)*fmr(2,1)*fmr(6,4) 
      en=en+fk(615)*fmr(1,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(616)*fmr(1,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(617)*fmr(1,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(618)*fmr(2,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(619)*fmr(2,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(620)*fmr(2,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(621)*fmr(3,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(622)*fmr(3,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(623)*fmr(4,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(624)*fmr(1,3)*fmr(2,2)*fmr(3,1) 
      en=en+fk(625)*fmr(1,3)*fmr(2,2)*fmr(4,1) 
      en=en+fk(626)*fmr(1,3)*fmr(2,2)*fmr(5,1) 
      en=en+fk(627)*fmr(1,3)*fmr(2,2)*fmr(6,1) 
      en=en+fk(628)*fmr(1,3)*fmr(2,1)*fmr(3,2) 
      en=en+fk(629)*fmr(1,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(630)*fmr(1,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(631)*fmr(1,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(632)*fmr(1,3)*fmr(2,1)*fmr(4,2) 
      en=en+fk(633)*fmr(1,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(634)*fmr(1,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(635)*fmr(1,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(636)*fmr(1,3)*fmr(2,1)*fmr(5,2) 
      en=en+fk(637)*fmr(1,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(638)*fmr(1,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(639)*fmr(1,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(640)*fmr(1,3)*fmr(2,1)*fmr(6,2) 
      en=en+fk(641)*fmr(1,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(642)*fmr(1,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(643)*fmr(1,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(644)*fmr(1,2)*fmr(2,3)*fmr(3,1) 
      en=en+fk(645)*fmr(1,2)*fmr(2,3)*fmr(4,1) 
      en=en+fk(646)*fmr(1,2)*fmr(2,3)*fmr(5,1) 
      en=en+fk(647)*fmr(1,2)*fmr(2,3)*fmr(6,1) 
      en=en+fk(648)*fmr(1,1)*fmr(2,3)*fmr(3,2) 
      en=en+fk(649)*fmr(2,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(650)*fmr(2,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(651)*fmr(2,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(652)*fmr(1,1)*fmr(2,3)*fmr(4,2) 
      en=en+fk(653)*fmr(2,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(654)*fmr(2,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(655)*fmr(2,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(656)*fmr(1,1)*fmr(2,3)*fmr(5,2) 
      en=en+fk(657)*fmr(2,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(658)*fmr(2,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(659)*fmr(2,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(660)*fmr(1,1)*fmr(2,3)*fmr(6,2) 
      en=en+fk(661)*fmr(2,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(662)*fmr(2,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(663)*fmr(2,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(664)*fmr(1,2)*fmr(2,1)*fmr(3,3) 
      en=en+fk(665)*fmr(1,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(666)*fmr(1,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(667)*fmr(1,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(668)*fmr(1,1)*fmr(2,2)*fmr(3,3) 
      en=en+fk(669)*fmr(2,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(670)*fmr(2,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(671)*fmr(2,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(672)*fmr(1,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(673)*fmr(2,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(674)*fmr(3,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(675)*fmr(3,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(676)*fmr(1,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(677)*fmr(2,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(678)*fmr(3,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(679)*fmr(3,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(680)*fmr(1,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(681)*fmr(2,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(682)*fmr(3,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(683)*fmr(3,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(684)*fmr(1,2)*fmr(2,1)*fmr(4,3) 
      en=en+fk(685)*fmr(1,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(686)*fmr(1,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(687)*fmr(1,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(688)*fmr(1,1)*fmr(2,2)*fmr(4,3) 
      en=en+fk(689)*fmr(2,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(690)*fmr(2,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(691)*fmr(2,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(692)*fmr(1,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(693)*fmr(2,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(694)*fmr(3,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(695)*fmr(3,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(696)*fmr(1,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(697)*fmr(2,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(698)*fmr(3,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(699)*fmr(4,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(700)*fmr(1,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(701)*fmr(2,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(702)*fmr(3,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(703)*fmr(4,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(704)*fmr(1,2)*fmr(2,1)*fmr(5,3) 
      en=en+fk(705)*fmr(1,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(706)*fmr(1,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(707)*fmr(1,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(708)*fmr(1,1)*fmr(2,2)*fmr(5,3) 
      en=en+fk(709)*fmr(2,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(710)*fmr(2,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(711)*fmr(2,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(712)*fmr(1,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(713)*fmr(2,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(714)*fmr(3,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(715)*fmr(3,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(716)*fmr(1,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(717)*fmr(2,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(718)*fmr(3,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(719)*fmr(4,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(720)*fmr(1,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(721)*fmr(2,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(722)*fmr(3,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(723)*fmr(4,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(724)*fmr(1,2)*fmr(2,1)*fmr(6,3) 
      en=en+fk(725)*fmr(1,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(726)*fmr(1,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(727)*fmr(1,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(728)*fmr(1,1)*fmr(2,2)*fmr(6,3) 
      en=en+fk(729)*fmr(2,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(730)*fmr(2,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(731)*fmr(2,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(732)*fmr(1,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(733)*fmr(2,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(734)*fmr(3,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(735)*fmr(3,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(736)*fmr(1,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(737)*fmr(2,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(738)*fmr(3,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(739)*fmr(4,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(740)*fmr(1,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(741)*fmr(2,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(742)*fmr(3,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(743)*fmr(4,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(744)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(745)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(746)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(747)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(748)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(749)*fmr(1,3)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(750)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(751)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(752)*fmr(1,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(753)*fmr(1,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(754)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(755)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(756)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(757)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(758)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(759)*fmr(1,1)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(760)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(761)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(762)*fmr(2,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(763)*fmr(2,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(764)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(765)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(766)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(767)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(768)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(769)*fmr(1,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(770)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(771)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(772)*fmr(2,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(773)*fmr(3,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(774)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(775)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(776)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(777)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(778)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(779)*fmr(1,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(780)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(781)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(782)*fmr(2,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(783)*fmr(3,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(784)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(785)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(786)*fmr(1,1)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(787)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(788)*fmr(1,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(789)*fmr(1,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(790)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(791)*fmr(2,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(792)*fmr(2,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(793)*fmr(3,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(794)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(795)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(796)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(797)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(798)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(799)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(800)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(801)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(802)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(803)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(804)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(805)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(806)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(807)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(808)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(809)*fmr(1,2)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(810)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(811)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(812)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(813)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(814)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(815)*fmr(1,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(816)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(817)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(818)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(819)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(820)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(821)*fmr(1,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(822)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(823)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(824)*fmr(1,2)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(825)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(826)*fmr(1,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(827)*fmr(1,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(828)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(829)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(830)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(831)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(832)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(833)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(834)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(835)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(836)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(837)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(838)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(839)*fmr(2,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(840)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(841)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(842)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(843)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(844)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(845)*fmr(2,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(846)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(847)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(848)*fmr(1,1)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(849)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(850)*fmr(2,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(851)*fmr(2,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(852)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(853)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(854)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(855)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(856)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(857)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(858)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(859)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(860)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(861)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(862)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(863)*fmr(3,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(864)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(865)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(866)*fmr(1,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(867)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(868)*fmr(2,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(869)*fmr(3,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(870)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(871)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(872)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(873)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(874)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(875)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(876)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(877)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(878)*fmr(1,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(879)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(880)*fmr(2,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(881)*fmr(3,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(882)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(883)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(884)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(885)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(886)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(887)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(888)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(889)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(890)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(891)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(892)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(893)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(894)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(895)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(896)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(897)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(898)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(899)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(900)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(901)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(902)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(903)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(904)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(905)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(906)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(907)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(908)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(909)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(910)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(911)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(912)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(913)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(914)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(915)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(916)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(917)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(918)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(919)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(920)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(921)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(922)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(923)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(924)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) &
        *fmr(6,1) 

      return
      end subroutine global_rij
!
      subroutine ch2ph(rij,en)
!
!     from ch2-35868-924.p
!
!     rij(1) = r(c-h1) (au)
!     rij(2) = r(c-h2)
!     rij(3) = r(c-h3)
!     rij(4) = r(h1-h2)
!     rij(5) = r(h1-h3)
!     rij(6) = r(h2-h3)
!
!     en = energy (hartree)
!
!
      implicit none
      double precision,intent(in) :: rij(6)
      double precision,intent(out) :: en
      double precision :: fk(924),fmr(6,6),shift(6),alpham(6)
      integer :: i, j, nmax, nvar
!
      data shift  / 2.12d0, 2.12d0, 2.12d0, 3.75d0, 3.75d0, 3.75d0 /
      data alpham / 6*0.5d0 /
      data nmax,nvar /6,6/
      data fk(   1), fk(   2) / -3.9776635445E+01 , -2.3572897484E-03 /
      data fk(   3), fk(   4) / -2.3572897484E-03 , -2.3572897484E-03 /
      data fk(   5), fk(   6) / -1.4552322358E-02 , -1.4552322358E-02 /
      data fk(   7), fk(   8) / -1.4552322358E-02 ,  6.4989196953E-01 /
      data fk(   9), fk(  10) /  6.4989196953E-01 ,  6.4989196953E-01 /
      data fk(  11), fk(  12) /  1.6572305378E-01 ,  1.6572305378E-01 /
      data fk(  13), fk(  14) /  1.6572305378E-01 , -2.4344974285E-01 /
      data fk(  15), fk(  16) / -2.4344974285E-01 ,  8.8747597545E-02 /
      data fk(  17), fk(  18) /  8.8747597545E-02 ,  3.3738288384E-01 /
      data fk(  19), fk(  20) / -2.4344974285E-01 ,  8.8747597545E-02 /
      data fk(  21), fk(  22) /  3.3738288384E-01 ,  8.8747597545E-02 /
      data fk(  23), fk(  24) /  3.3738288384E-01 ,  8.8747597545E-02 /
      data fk(  25), fk(  26) /  8.8747597545E-02 , -2.5690748435E-02 /
      data fk(  27), fk(  28) / -2.5690748435E-02 , -2.5690748435E-02 /
      data fk(  29), fk(  30) /  1.3409870141E+00 ,  1.3409870141E+00 /
      data fk(  31), fk(  32) /  1.3409870141E+00 ,  4.7726382775E-01 /
      data fk(  33), fk(  34) /  4.7726382775E-01 ,  4.7726382775E-01 /
      data fk(  35), fk(  36) /  4.3861654433E+00 ,  4.3861654433E+00 /
      data fk(  37), fk(  38) / -1.9531113343E+00 , -1.9531113343E+00 /
      data fk(  39), fk(  40) / -1.6446212417E+00 ,  4.3861654433E+00 /
      data fk(  41), fk(  42) /  4.3861654433E+00 , -1.9531113343E+00 /
      data fk(  43), fk(  44) / -1.6446212417E+00 , -1.9531113343E+00 /
      data fk(  45), fk(  46) /  4.3861654433E+00 ,  4.3861654433E+00 /
      data fk(  47), fk(  48) / -1.6446212417E+00 , -1.9531113343E+00 /
      data fk(  49), fk(  50) / -1.9531113343E+00 , -1.9971466755E-01 /
      data fk(  51), fk(  52) / -1.9971466755E-01 ,  1.6922664747E-01 /
      data fk(  53), fk(  54) /  1.5209236289E-01 ,  1.5209236289E-01 /
      data fk(  55), fk(  56) / -1.9971466755E-01 ,  1.6922664747E-01 /
      data fk(  57), fk(  58) / -1.9971466755E-01 ,  1.5209236289E-01 /
      data fk(  59), fk(  60) /  1.5209236289E-01 ,  1.6922664747E-01 /
      data fk(  61), fk(  62) / -1.9971466755E-01 , -1.9971466755E-01 /
      data fk(  63), fk(  64) /  1.5209236289E-01 ,  1.5209236289E-01 /
      data fk(  65), fk(  66) /  8.1574234685E+00 , -3.1882730266E+00 /
      data fk(  67), fk(  68) / -3.2852724111E+00 , -3.2852724111E+00 /
      data fk(  69), fk(  70) / -3.2852724111E+00 , -3.1882730266E+00 /
      data fk(  71), fk(  72) / -3.2852724111E+00 ,  6.3325226603E-01 /
      data fk(  73), fk(  74) /  4.5802527421E-01 ,  4.5802527421E-01 /
      data fk(  75), fk(  76) / -3.2852724111E+00 , -3.2852724111E+00 /
      data fk(  77), fk(  78) / -3.1882730266E+00 ,  4.5802527421E-01 /
      data fk(  79), fk(  80) /  6.3325226603E-01 ,  4.5802527421E-01 /
      data fk(  81), fk(  82) /  4.5802527421E-01 ,  4.5802527421E-01 /
      data fk(  83), fk(  84) /  6.3325226603E-01 ,  2.1063033423E-01 /
      data fk(  85), fk(  86) / -3.7718383894E+00 , -3.7718383894E+00 /
      data fk(  87), fk(  88) / -3.7718383894E+00 ,  9.5088620178E-01 /
      data fk(  89), fk(  90) /  9.5088620178E-01 ,  9.5088620178E-01 /
      data fk(  91), fk(  92) / -1.3436107752E+01 , -1.3436107752E+01 /
      data fk(  93), fk(  94) /  5.3326550193E+00 ,  5.3326550193E+00 /
      data fk(  95), fk(  96) /  3.0702791659E+00 , -1.3436107752E+01 /
      data fk(  97), fk(  98) / -1.3436107752E+01 ,  5.3326550193E+00 /
      data fk(  99), fk( 100) /  3.0702791659E+00 ,  5.3326550193E+00 /
      data fk( 101), fk( 102) / -1.3436107752E+01 , -1.3436107752E+01 /
      data fk( 103), fk( 104) /  3.0702791659E+00 ,  5.3326550193E+00 /
      data fk( 105), fk( 106) /  5.3326550193E+00 , -1.7648543572E+00 /
      data fk( 107), fk( 108) / -1.7648543572E+00 ,  5.3383830281E-02 /
      data fk( 109), fk( 110) /  3.6888360392E-02 ,  3.6888360392E-02 /
      data fk( 111), fk( 112) / -1.7648543572E+00 ,  5.3383830281E-02 /
      data fk( 113), fk( 114) / -1.7648543572E+00 ,  3.6888360392E-02 /
      data fk( 115), fk( 116) /  3.6888360392E-02 ,  5.3383830281E-02 /
      data fk( 117), fk( 118) / -1.7648543572E+00 , -1.7648543572E+00 /
      data fk( 119), fk( 120) /  3.6888360392E-02 ,  3.6888360392E-02 /
      data fk( 121), fk( 122) / -1.9410474424E+01 , -1.9410474424E+01 /
      data fk( 123), fk( 124) / -1.1057758537E+00 , -1.1057758537E+00 /
      data fk( 125), fk( 126) / -4.4016885711E-01 , -1.9410474424E+01 /
      data fk( 127), fk( 128) / -1.1057758537E+00 , -4.4016885711E-01 /
      data fk( 129), fk( 130) / -1.1057758537E+00 , -4.4016885711E-01 /
      data fk( 131), fk( 132) / -1.1057758537E+00 , -1.1057758537E+00 /
      data fk( 133), fk( 134) /  1.8878050338E-01 ,  1.8878050338E-01 /
      data fk( 135), fk( 136) /  1.8878050338E-01 , -3.5996064592E+01 /
      data fk( 137), fk( 138) /  1.3043837769E+01 ,  1.1213413645E+01 /
      data fk( 139), fk( 140) /  9.5239152888E+00 ,  1.1213413645E+01 /
      data fk( 141), fk( 142) /  1.3043837769E+01 ,  9.5239152888E+00 /
      data fk( 143), fk( 144) / -2.1680174581E+00 , -1.3926633962E+00 /
      data fk( 145), fk( 146) / -1.3926633962E+00 , -3.5996064592E+01 /
      data fk( 147), fk( 148) /  1.3043837769E+01 ,  9.5239152888E+00 /
      data fk( 149), fk( 150) /  1.1213413645E+01 ,  1.1213413645E+01 /
      data fk( 151), fk( 152) /  9.5239152888E+00 ,  1.3043837769E+01 /
      data fk( 153), fk( 154) / -1.3926633962E+00 , -2.1680174581E+00 /
      data fk( 155), fk( 156) / -1.3926633962E+00 , -3.5996064592E+01 /
      data fk( 157), fk( 158) /  9.5239152888E+00 ,  1.3043837769E+01 /
      data fk( 159), fk( 160) /  1.1213413645E+01 ,  9.5239152888E+00 /
      data fk( 161), fk( 162) /  1.1213413645E+01 ,  1.3043837769E+01 /
      data fk( 163), fk( 164) / -1.3926633962E+00 , -1.3926633962E+00 /
      data fk( 165), fk( 166) / -2.1680174581E+00 ,  5.9208165647E-01 /
      data fk( 167), fk( 168) / -1.2785993751E+00 , -6.2258595213E-01 /
      data fk( 169), fk( 170) / -1.1758555161E+00 , -1.2785993751E+00 /
      data fk( 171), fk( 172) / -1.1758555161E+00 , -6.2258595213E-01 /
      data fk( 173), fk( 174) / -1.0126443228E+00 , -1.0126443228E+00 /
      data fk( 175), fk( 176) /  7.1888034775E-01 , -1.2785993751E+00 /
      data fk( 177), fk( 178) /  5.9208165647E-01 , -6.2258595213E-01 /
      data fk( 179), fk( 180) / -1.1758555161E+00 , -1.2785993751E+00 /
      data fk( 181), fk( 182) / -1.0126443228E+00 , -1.0126443228E+00 /
      data fk( 183), fk( 184) / -1.1758555161E+00 , -6.2258595213E-01 /
      data fk( 185), fk( 186) /  7.1888034775E-01 , -1.2785993751E+00 /
      data fk( 187), fk( 188) / -1.2785993751E+00 , -1.0126443228E+00 /
      data fk( 189), fk( 190) / -1.0126443228E+00 ,  5.9208165647E-01 /
      data fk( 191), fk( 192) / -6.2258595213E-01 , -1.1758555161E+00 /
      data fk( 193), fk( 194) / -1.1758555161E+00 , -6.2258595213E-01 /
      data fk( 195), fk( 196) /  7.1888034775E-01 , -9.7484403664E-01 /
      data fk( 197), fk( 198) / -9.7484403664E-01 , -1.8274868008E+00 /
      data fk( 199), fk( 200) / -1.8274868008E+00 , -1.7756863611E+00 /
      data fk( 201), fk( 202) / -9.7484403664E-01 , -1.8274868008E+00 /
      data fk( 203), fk( 204) / -1.7756863611E+00 , -1.8274868008E+00 /
      data fk( 205), fk( 206) / -1.7756863611E+00 , -1.8274868008E+00 /
      data fk( 207), fk( 208) / -1.8274868008E+00 ,  2.1121874266E+01 /
      data fk( 209), fk( 210) /  2.1121874266E+01 ,  2.1121874266E+01 /
      data fk( 211), fk( 212) /  1.1407780594E+00 ,  1.1407780594E+00 /
      data fk( 213), fk( 214) /  1.1407780594E+00 ,  4.8969012692E-01 /
      data fk( 215), fk( 216) /  4.8969012692E-01 ,  4.8969012692E-01 /
      data fk( 217), fk( 218) /  1.4099815428E+01 ,  1.4099815428E+01 /
      data fk( 219), fk( 220) / -3.4329956995E+00 , -3.4329956995E+00 /
      data fk( 221), fk( 222) / -2.5516205977E+00 ,  1.4099815428E+01 /
      data fk( 223), fk( 224) /  1.4099815428E+01 , -3.4329956995E+00 /
      data fk( 225), fk( 226) / -2.5516205977E+00 , -3.4329956995E+00 /
      data fk( 227), fk( 228) /  1.4099815428E+01 ,  1.4099815428E+01 /
      data fk( 229), fk( 230) / -2.5516205977E+00 , -3.4329956995E+00 /
      data fk( 231), fk( 232) / -3.4329956995E+00 , -2.0155450552E+00 /
      data fk( 233), fk( 234) / -2.0155450552E+00 ,  1.1000677234E+00 /
      data fk( 235), fk( 236) / -6.0389637940E-01 , -6.0389637940E-01 /
      data fk( 237), fk( 238) / -2.0155450552E+00 ,  1.1000677234E+00 /
      data fk( 239), fk( 240) / -2.0155450552E+00 , -6.0389637940E-01 /
      data fk( 241), fk( 242) / -6.0389637940E-01 ,  1.1000677234E+00 /
      data fk( 243), fk( 244) / -2.0155450552E+00 , -2.0155450552E+00 /
      data fk( 245), fk( 246) / -6.0389637940E-01 , -6.0389637940E-01 /
      data fk( 247), fk( 248) /  2.6633465074E+01 ,  2.6633465074E+01 /
      data fk( 249), fk( 250) /  1.7034444531E+00 ,  1.7034444531E+00 /
      data fk( 251), fk( 252) /  5.0893578475E-01 ,  2.6633465074E+01 /
      data fk( 253), fk( 254) /  2.6633465074E+01 ,  1.7034444531E+00 /
      data fk( 255), fk( 256) /  5.0893578475E-01 ,  1.7034444531E+00 /
      data fk( 257), fk( 258) /  2.6633465074E+01 ,  2.6633465074E+01 /
      data fk( 259), fk( 260) /  5.0893578475E-01 ,  1.7034444531E+00 /
      data fk( 261), fk( 262) /  1.7034444531E+00 ,  1.3303978049E+00 /
      data fk( 263), fk( 264) /  1.3303978049E+00 ,  4.7949574514E-01 /
      data fk( 265), fk( 266) /  3.1194332590E-01 ,  3.1194332590E-01 /
      data fk( 267), fk( 268) /  1.3303978049E+00 ,  4.7949574514E-01 /
      data fk( 269), fk( 270) /  1.3303978049E+00 ,  3.1194332590E-01 /
      data fk( 271), fk( 272) /  3.1194332590E-01 ,  4.7949574514E-01 /
      data fk( 273), fk( 274) /  1.3303978049E+00 ,  1.3303978049E+00 /
      data fk( 275), fk( 276) /  3.1194332590E-01 ,  3.1194332590E-01 /
      data fk( 277), fk( 278) /  5.2858706503E+01 , -1.5610161175E+01 /
      data fk( 279), fk( 280) / -1.2404986571E+01 , -9.3577406563E+00 /
      data fk( 281), fk( 282) / -1.2404986571E+01 , -1.5610161175E+01 /
      data fk( 283), fk( 284) / -9.3577406563E+00 , -1.0498832951E-01 /
      data fk( 285), fk( 286) /  3.8541412658E-01 ,  3.8541412658E-01 /
      data fk( 287), fk( 288) /  5.2858706503E+01 , -1.5610161175E+01 /
      data fk( 289), fk( 290) / -9.3577406563E+00 , -1.2404986571E+01 /
      data fk( 291), fk( 292) / -1.2404986571E+01 , -9.3577406563E+00 /
      data fk( 293), fk( 294) / -1.5610161175E+01 ,  3.8541412658E-01 /
      data fk( 295), fk( 296) / -1.0498832951E-01 ,  3.8541412658E-01 /
      data fk( 297), fk( 298) /  5.2858706503E+01 , -9.3577406563E+00 /
      data fk( 299), fk( 300) / -1.5610161175E+01 , -1.2404986571E+01 /
      data fk( 301), fk( 302) / -9.3577406563E+00 , -1.2404986571E+01 /
      data fk( 303), fk( 304) / -1.5610161175E+01 ,  3.8541412658E-01 /
      data fk( 305), fk( 306) /  3.8541412658E-01 , -1.0498832951E-01 /
      data fk( 307), fk( 308) /  6.6298316454E+00 , -9.6788986404E-01 /
      data fk( 309), fk( 310) /  3.1462831469E-01 , -4.4275282298E-01 /
      data fk( 311), fk( 312) / -9.6788986404E-01 , -4.4275282298E-01 /
      data fk( 313), fk( 314) /  3.1462831469E-01 , -1.1695584548E+00 /
      data fk( 315), fk( 316) / -1.1695584548E+00 ,  9.1437981940E-01 /
      data fk( 317), fk( 318) / -9.6788986404E-01 ,  6.6298316454E+00 /
      data fk( 319), fk( 320) /  3.1462831469E-01 , -4.4275282298E-01 /
      data fk( 321), fk( 322) / -9.6788986404E-01 , -1.1695584548E+00 /
      data fk( 323), fk( 324) / -1.1695584548E+00 , -4.4275282298E-01 /
      data fk( 325), fk( 326) /  3.1462831469E-01 ,  9.1437981940E-01 /
      data fk( 327), fk( 328) / -9.6788986404E-01 , -9.6788986404E-01 /
      data fk( 329), fk( 330) / -1.1695584548E+00 , -1.1695584548E+00 /
      data fk( 331), fk( 332) /  6.6298316454E+00 ,  3.1462831469E-01 /
      data fk( 333), fk( 334) / -4.4275282298E-01 , -4.4275282298E-01 /
      data fk( 335), fk( 336) /  3.1462831469E-01 ,  9.1437981940E-01 /
      data fk( 337), fk( 338) /  7.2391896183E+01 , -1.7868376997E+00 /
      data fk( 339), fk( 340) / -1.5422119374E+00 ,  1.1290576594E+00 /
      data fk( 341), fk( 342) / -1.5422119374E+00 , -1.7868376997E+00 /
      data fk( 343), fk( 344) /  1.1290576594E+00 , -1.4246290732E+00 /
      data fk( 345), fk( 346) / -8.0613988452E-01 , -8.0613988452E-01 /
      data fk( 347), fk( 348) /  7.2391896183E+01 , -1.7868376997E+00 /
      data fk( 349), fk( 350) /  1.1290576594E+00 , -1.5422119374E+00 /
      data fk( 351), fk( 352) / -1.5422119374E+00 ,  1.1290576594E+00 /
      data fk( 353), fk( 354) / -1.7868376997E+00 , -8.0613988452E-01 /
      data fk( 355), fk( 356) / -1.4246290732E+00 , -8.0613988452E-01 /
      data fk( 357), fk( 358) /  7.2391896183E+01 ,  1.1290576594E+00 /
      data fk( 359), fk( 360) / -1.7868376997E+00 , -1.5422119374E+00 /
      data fk( 361), fk( 362) /  1.1290576594E+00 , -1.5422119374E+00 /
      data fk( 363), fk( 364) / -1.7868376997E+00 , -8.0613988452E-01 /
      data fk( 365), fk( 366) / -8.0613988452E-01 , -1.4246290732E+00 /
      data fk( 367), fk( 368) / -1.7750189131E+01 , -1.1861577548E+01 /
      data fk( 369), fk( 370) /  2.2066860731E+00 ,  1.6602979620E+00 /
      data fk( 371), fk( 372) / -1.1861577548E+01 ,  1.6602979620E+00 /
      data fk( 373), fk( 374) /  2.2066860731E+00 ,  3.9210435565E+00 /
      data fk( 375), fk( 376) /  3.9210435565E+00 , -2.0762537558E-01 /
      data fk( 377), fk( 378) / -1.1861577548E+01 , -1.7750189131E+01 /
      data fk( 379), fk( 380) /  2.2066860731E+00 ,  1.6602979620E+00 /
      data fk( 381), fk( 382) / -1.1861577548E+01 ,  3.9210435565E+00 /
      data fk( 383), fk( 384) /  3.9210435565E+00 ,  1.6602979620E+00 /
      data fk( 385), fk( 386) /  2.2066860731E+00 , -2.0762537558E-01 /
      data fk( 387), fk( 388) / -1.1861577548E+01 , -1.1861577548E+01 /
      data fk( 389), fk( 390) /  3.9210435565E+00 ,  3.9210435565E+00 /
      data fk( 391), fk( 392) / -1.7750189131E+01 ,  2.2066860731E+00 /
      data fk( 393), fk( 394) /  1.6602979620E+00 ,  1.6602979620E+00 /
      data fk( 395), fk( 396) /  2.2066860731E+00 , -2.0762537558E-01 /
      data fk( 397), fk( 398) / -3.1511597490E+01 , -3.1511597490E+01 /
      data fk( 399), fk( 400) / -2.4192610310E+01 , -1.8498782618E+00 /
      data fk( 401), fk( 402) / -3.8873474247E+00 , -9.8689662240E-01 /
      data fk( 403), fk( 404) / -1.8498782618E+00 , -9.8689662240E-01 /
      data fk( 405), fk( 406) / -3.8873474247E+00 ,  3.2979794796E+00 /
      data fk( 407), fk( 408) / -3.1511597490E+01 , -2.4192610310E+01 /
      data fk( 409), fk( 410) / -3.1511597490E+01 , -3.8873474247E+00 /
      data fk( 411), fk( 412) / -1.8498782618E+00 , -9.8689662240E-01 /
      data fk( 413), fk( 414) / -9.8689662240E-01 , -1.8498782618E+00 /
      data fk( 415), fk( 416) / -3.8873474247E+00 ,  3.2979794796E+00 /
      data fk( 417), fk( 418) / -2.4192610310E+01 , -3.1511597490E+01 /
      data fk( 419), fk( 420) / -3.1511597490E+01 , -3.8873474247E+00 /
      data fk( 421), fk( 422) / -9.8689662240E-01 , -1.8498782618E+00 /
      data fk( 423), fk( 424) / -9.8689662240E-01 , -3.8873474247E+00 /
      data fk( 425), fk( 426) / -1.8498782618E+00 ,  3.2979794796E+00 /
      data fk( 427), fk( 428) /  7.8744039282E-02 ,  4.8480128348E+00 /
      data fk( 429), fk( 430) /  4.8480128348E+00 ,  5.0593901833E+00 /
      data fk( 431), fk( 432) /  5.9525627166E+00 , -3.1979774572E+00 /
      data fk( 433), fk( 434) /  5.9525627166E+00 ,  5.0593901833E+00 /
      data fk( 435), fk( 436) / -3.1979774572E+00 , -1.2730157504E+00 /
      data fk( 437), fk( 438) /  7.8744039282E-02 ,  5.0593901833E+00 /
      data fk( 439), fk( 440) /  5.9525627166E+00 ,  4.8480128348E+00 /
      data fk( 441), fk( 442) /  4.8480128348E+00 , -3.1979774572E+00 /
      data fk( 443), fk( 444) /  5.9525627166E+00 ,  5.0593901833E+00 /
      data fk( 445), fk( 446) / -1.2730157504E+00 , -3.1979774572E+00 /
      data fk( 447), fk( 448) /  7.8744039282E-02 ,  5.0593901833E+00 /
      data fk( 449), fk( 450) /  5.9525627166E+00 ,  5.9525627166E+00 /
      data fk( 451), fk( 452) /  5.0593901833E+00 , -1.2730157504E+00 /
      data fk( 453), fk( 454) /  4.8480128348E+00 ,  4.8480128348E+00 /
      data fk( 455), fk( 456) / -3.1979774572E+00 , -3.1979774572E+00 /
      data fk( 457), fk( 458) / -7.5921515578E+00 , -7.5921515578E+00 /
      data fk( 459), fk( 460) / -7.5921515578E+00 ,  7.6401740814E+00 /
      data fk( 461), fk( 462) /  7.6401740814E+00 ,  7.6401740814E+00 /
      data fk( 463), fk( 464) /  1.1165322606E+00 ,  1.1165322606E+00 /
      data fk( 465), fk( 466) /  1.1165322606E+00 ,  6.2018387020E-02 /
      data fk( 467), fk( 468) /  6.2018387020E-02 ,  6.2018387020E-02 /
      data fk( 469), fk( 470) / -4.2400461824E+00 , -4.2400461824E+00 /
      data fk( 471), fk( 472) / -7.7032791820E-01 , -7.7032791820E-01 /
      data fk( 473), fk( 474) /  9.5180417648E-01 , -4.2400461824E+00 /
      data fk( 475), fk( 476) / -4.2400461824E+00 , -7.7032791820E-01 /
      data fk( 477), fk( 478) /  9.5180417648E-01 , -7.7032791820E-01 /
      data fk( 479), fk( 480) / -4.2400461824E+00 , -4.2400461824E+00 /
      data fk( 481), fk( 482) /  9.5180417648E-01 , -7.7032791820E-01 /
      data fk( 483), fk( 484) / -7.7032791820E-01 , -4.0906153902E-01 /
      data fk( 485), fk( 486) / -4.0906153902E-01 ,  3.2256675756E-01 /
      data fk( 487), fk( 488) / -2.3058704424E-01 , -2.3058704424E-01 /
      data fk( 489), fk( 490) / -4.0906153902E-01 ,  3.2256675756E-01 /
      data fk( 491), fk( 492) / -4.0906153902E-01 , -2.3058704424E-01 /
      data fk( 493), fk( 494) / -2.3058704424E-01 ,  3.2256675756E-01 /
      data fk( 495), fk( 496) / -4.0906153902E-01 , -4.0906153902E-01 /
      data fk( 497), fk( 498) / -2.3058704424E-01 , -2.3058704424E-01 /
      data fk( 499), fk( 500) / -1.1816369593E+01 , -1.1816369593E+01 /
      data fk( 501), fk( 502) /  4.1849342425E-02 ,  4.1849342425E-02 /
      data fk( 503), fk( 504) /  5.1751102212E-02 , -1.1816369593E+01 /
      data fk( 505), fk( 506) / -1.1816369593E+01 ,  4.1849342425E-02 /
      data fk( 507), fk( 508) /  5.1751102212E-02 ,  4.1849342425E-02 /
      data fk( 509), fk( 510) / -1.1816369593E+01 , -1.1816369593E+01 /
      data fk( 511), fk( 512) /  5.1751102212E-02 ,  4.1849342425E-02 /
      data fk( 513), fk( 514) /  4.1849342425E-02 ,  5.5861799007E-01 /
      data fk( 515), fk( 516) /  5.5861799007E-01 ,  1.0359416850E+00 /
      data fk( 517), fk( 518) /  1.4965287914E-01 ,  1.4965287914E-01 /
      data fk( 519), fk( 520) /  5.5861799007E-01 ,  1.0359416850E+00 /
      data fk( 521), fk( 522) /  5.5861799007E-01 ,  1.4965287914E-01 /
      data fk( 523), fk( 524) /  1.4965287914E-01 ,  1.0359416850E+00 /
      data fk( 525), fk( 526) /  5.5861799007E-01 ,  5.5861799007E-01 /
      data fk( 527), fk( 528) /  1.4965287914E-01 ,  1.4965287914E-01 /
      data fk( 529), fk( 530) / -1.3864073884E+01 , -1.3864073884E+01 /
      data fk( 531), fk( 532) /  2.0198703769E-01 ,  2.0198703769E-01 /
      data fk( 533), fk( 534) / -2.0024608840E-01 , -1.3864073884E+01 /
      data fk( 535), fk( 536) /  2.0198703769E-01 , -2.0024608840E-01 /
      data fk( 537), fk( 538) /  2.0198703769E-01 , -2.0024608840E-01 /
      data fk( 539), fk( 540) /  2.0198703769E-01 ,  2.0198703769E-01 /
      data fk( 541), fk( 542) /  9.3389558928E-02 ,  9.3389558928E-02 /
      data fk( 543), fk( 544) /  9.3389558928E-02 ,  8.3603516478E-01 /
      data fk( 545), fk( 546) /  2.8333783581E+00 ,  6.6891409541E-01 /
      data fk( 547), fk( 548) /  6.6891409541E-01 ,  6.6891409541E-01 /
      data fk( 549), fk( 550) /  2.8333783581E+00 ,  6.6891409541E-01 /
      data fk( 551), fk( 552) /  5.3566385591E+00 ,  4.1059774370E+00 /
      data fk( 553), fk( 554) /  4.1059774370E+00 ,  6.6891409541E-01 /
      data fk( 555), fk( 556) /  6.6891409541E-01 ,  2.8333783581E+00 /
      data fk( 557), fk( 558) /  4.1059774370E+00 ,  5.3566385591E+00 /
      data fk( 559), fk( 560) /  4.1059774370E+00 ,  4.1059774370E+00 /
      data fk( 561), fk( 562) /  4.1059774370E+00 ,  5.3566385591E+00 /
      data fk( 563), fk( 564) / -5.9359200616E+01 , -2.4399647501E+01 /
      data fk( 565), fk( 566) /  4.8525495488E+00 ,  2.9095868554E+00 /
      data fk( 567), fk( 568) /  2.2615509558E+00 ,  2.9095868554E+00 /
      data fk( 569), fk( 570) /  4.8525495488E+00 ,  2.2615509558E+00 /
      data fk( 571), fk( 572) /  3.5234725797E+00 ,  4.9538606369E-01 /
      data fk( 573), fk( 574) /  4.9538606369E-01 , -2.4399647501E+01 /
      data fk( 575), fk( 576) /  4.8525495488E+00 ,  2.2615509558E+00 /
      data fk( 577), fk( 578) /  2.9095868554E+00 ,  2.9095868554E+00 /
      data fk( 579), fk( 580) /  2.2615509558E+00 ,  4.8525495488E+00 /
      data fk( 581), fk( 582) /  4.9538606369E-01 ,  3.5234725797E+00 /
      data fk( 583), fk( 584) /  4.9538606369E-01 , -2.4399647501E+01 /
      data fk( 585), fk( 586) /  2.2615509558E+00 ,  4.8525495488E+00 /
      data fk( 587), fk( 588) /  2.9095868554E+00 ,  2.2615509558E+00 /
      data fk( 589), fk( 590) /  2.9095868554E+00 ,  4.8525495488E+00 /
      data fk( 591), fk( 592) /  4.9538606369E-01 ,  4.9538606369E-01 /
      data fk( 593), fk( 594) /  3.5234725797E+00 ,  1.8883719731E+00 /
      data fk( 595), fk( 596) / -1.4669451358E+00 ,  1.0263120920E+00 /
      data fk( 597), fk( 598) /  1.1751066819E+00 , -1.4669451358E+00 /
      data fk( 599), fk( 600) /  1.1751066819E+00 ,  1.0263120920E+00 /
      data fk( 601), fk( 602) / -3.8906819345E-01 , -3.8906819345E-01 /
      data fk( 603), fk( 604) / -9.4011417546E-02 , -1.4669451358E+00 /
      data fk( 605), fk( 606) /  1.8883719731E+00 ,  1.0263120920E+00 /
      data fk( 607), fk( 608) /  1.1751066819E+00 , -1.4669451358E+00 /
      data fk( 609), fk( 610) / -3.8906819345E-01 , -3.8906819345E-01 /
      data fk( 611), fk( 612) /  1.1751066819E+00 ,  1.0263120920E+00 /
      data fk( 613), fk( 614) / -9.4011417546E-02 , -1.4669451358E+00 /
      data fk( 615), fk( 616) / -1.4669451358E+00 , -3.8906819345E-01 /
      data fk( 617), fk( 618) / -3.8906819345E-01 ,  1.8883719731E+00 /
      data fk( 619), fk( 620) /  1.0263120920E+00 ,  1.1751066819E+00 /
      data fk( 621), fk( 622) /  1.1751066819E+00 ,  1.0263120920E+00 /
      data fk( 623), fk( 624) / -9.4011417546E-02 , -4.3868789220E+01 /
      data fk( 625), fk( 626) /  5.6919299246E+00 ,  3.4849616581E+00 /
      data fk( 627), fk( 628) /  2.6185053860E+00 , -4.3868789220E+01 /
      data fk( 629), fk( 630) /  3.4849616581E+00 ,  5.6919299246E+00 /
      data fk( 631), fk( 632) /  2.6185053860E+00 ,  6.2047993661E-01 /
      data fk( 633), fk( 634) /  1.4547433037E+00 , -3.4613770921E+00 /
      data fk( 635), fk( 636) / -3.2606239994E+00 ,  1.4547433037E+00 /
      data fk( 637), fk( 638) /  6.2047993661E-01 , -3.4613770921E+00 /
      data fk( 639), fk( 640) / -3.2606239994E+00 ,  7.2529256137E-01 /
      data fk( 641), fk( 642) /  7.2529256137E-01 , -1.3121122110E+00 /
      data fk( 643), fk( 644) / -1.3121122110E+00 , -4.3868789220E+01 /
      data fk( 645), fk( 646) /  5.6919299246E+00 ,  2.6185053860E+00 /
      data fk( 647), fk( 648) /  3.4849616581E+00 , -4.3868789220E+01 /
      data fk( 649), fk( 650) /  3.4849616581E+00 ,  2.6185053860E+00 /
      data fk( 651), fk( 652) /  5.6919299246E+00 ,  6.2047993661E-01 /
      data fk( 653), fk( 654) /  1.4547433037E+00 , -3.2606239994E+00 /
      data fk( 655), fk( 656) / -3.4613770921E+00 ,  7.2529256137E-01 /
      data fk( 657), fk( 658) /  7.2529256137E-01 , -1.3121122110E+00 /
      data fk( 659), fk( 660) / -1.3121122110E+00 ,  1.4547433037E+00 /
      data fk( 661), fk( 662) /  6.2047993661E-01 , -3.4613770921E+00 /
      data fk( 663), fk( 664) / -3.2606239994E+00 , -4.3868789220E+01 /
      data fk( 665), fk( 666) /  2.6185053860E+00 ,  5.6919299246E+00 /
      data fk( 667), fk( 668) /  3.4849616581E+00 , -4.3868789220E+01 /
      data fk( 669), fk( 670) /  2.6185053860E+00 ,  3.4849616581E+00 /
      data fk( 671), fk( 672) /  5.6919299246E+00 ,  7.2529256137E-01 /
      data fk( 673), fk( 674) /  7.2529256137E-01 , -1.3121122110E+00 /
      data fk( 675), fk( 676) / -1.3121122110E+00 ,  6.2047993661E-01 /
      data fk( 677), fk( 678) /  1.4547433037E+00 , -3.2606239994E+00 /
      data fk( 679), fk( 680) / -3.4613770921E+00 ,  1.4547433037E+00 /
      data fk( 681), fk( 682) /  6.2047993661E-01 , -3.2606239994E+00 /
      data fk( 683), fk( 684) / -3.4613770921E+00 , -3.2631154099E+00 /
      data fk( 685), fk( 686) /  2.3168513716E-01 ,  5.3570034872E-02 /
      data fk( 687), fk( 688) /  3.3847146742E-01 , -3.2631154099E+00 /
      data fk( 689), fk( 690) /  2.3168513716E-01 ,  3.3847146742E-01 /
      data fk( 691), fk( 692) /  5.3570034872E-02 , -3.0239806790E+00 /
      data fk( 693), fk( 694) / -3.0239806790E+00 ,  8.2899279886E-02 /
      data fk( 695), fk( 696) /  8.2899279886E-02 , -9.8252371531E-01 /
      data fk( 697), fk( 698) / -3.2513355704E-01 ,  1.6013203549E-01 /
      data fk( 699), fk( 700) / -3.9867497132E-01 , -3.2513355704E-01 /
      data fk( 701), fk( 702) / -9.8252371531E-01 ,  1.6013203549E-01 /
      data fk( 703), fk( 704) / -3.9867497132E-01 ,  2.3168513716E-01 /
      data fk( 705), fk( 706) / -3.2631154099E+00 ,  5.3570034872E-02 /
      data fk( 707), fk( 708) /  3.3847146742E-01 , -3.0239806790E+00 /
      data fk( 709), fk( 710) / -3.0239806790E+00 ,  8.2899279886E-02 /
      data fk( 711), fk( 712) /  8.2899279886E-02 , -3.2631154099E+00 /
      data fk( 713), fk( 714) /  2.3168513716E-01 ,  3.3847146742E-01 /
      data fk( 715), fk( 716) /  5.3570034872E-02 , -9.8252371531E-01 /
      data fk( 717), fk( 718) /  1.6013203549E-01 , -3.2513355704E-01 /
      data fk( 719), fk( 720) / -3.9867497132E-01 , -3.2513355704E-01 /
      data fk( 721), fk( 722) /  1.6013203549E-01 , -9.8252371531E-01 /
      data fk( 723), fk( 724) / -3.9867497132E-01 , -3.0239806790E+00 /
      data fk( 725), fk( 726) / -3.0239806790E+00 ,  8.2899279886E-02 /
      data fk( 727), fk( 728) /  8.2899279886E-02 ,  2.3168513716E-01 /
      data fk( 729), fk( 730) / -3.2631154099E+00 ,  5.3570034872E-02 /
      data fk( 731), fk( 732) /  3.3847146742E-01 ,  2.3168513716E-01 /
      data fk( 733), fk( 734) / -3.2631154099E+00 ,  3.3847146742E-01 /
      data fk( 735), fk( 736) /  5.3570034872E-02 ,  1.6013203549E-01 /
      data fk( 737), fk( 738) / -9.8252371531E-01 , -3.2513355704E-01 /
      data fk( 739), fk( 740) / -3.9867497132E-01 ,  1.6013203549E-01 /
      data fk( 741), fk( 742) / -3.2513355704E-01 , -9.8252371531E-01 /
      data fk( 743), fk( 744) / -3.9867497132E-01 ,  1.1286168406E+01 /
      data fk( 745), fk( 746) /  1.1286168406E+01 ,  1.0710439131E+01 /
      data fk( 747), fk( 748) /  5.5437493644E+00 ,  7.5190926482E+00 /
      data fk( 749), fk( 750) /  2.8575440529E+00 ,  5.5437493644E+00 /
      data fk( 751), fk( 752) /  2.8575440529E+00 ,  7.5190926482E+00 /
      data fk( 753), fk( 754) / -1.8177190818E+00 ,  1.1286168406E+01 /
      data fk( 755), fk( 756) /  1.0710439131E+01 ,  1.1286168406E+01 /
      data fk( 757), fk( 758) /  7.5190926482E+00 ,  5.5437493644E+00 /
      data fk( 759), fk( 760) /  2.8575440529E+00 ,  2.8575440529E+00 /
      data fk( 761), fk( 762) /  5.5437493644E+00 ,  7.5190926482E+00 /
      data fk( 763), fk( 764) / -1.8177190818E+00 ,  1.0710439131E+01 /
      data fk( 765), fk( 766) /  1.1286168406E+01 ,  1.1286168406E+01 /
      data fk( 767), fk( 768) /  7.5190926482E+00 ,  2.8575440529E+00 /
      data fk( 769), fk( 770) /  5.5437493644E+00 ,  2.8575440529E+00 /
      data fk( 771), fk( 772) /  7.5190926482E+00 ,  5.5437493644E+00 /
      data fk( 773), fk( 774) / -1.8177190818E+00 ,  2.3724897337E+00 /
      data fk( 775), fk( 776) / -1.5354165857E+00 , -1.5354165857E+00 /
      data fk( 777), fk( 778) /  2.2397034646E+00 ,  1.9457795442E+00 /
      data fk( 779), fk( 780) / -6.9600412209E-01 ,  1.9457795442E+00 /
      data fk( 781), fk( 782) /  2.2397034646E+00 , -6.9600412209E-01 /
      data fk( 783), fk( 784) /  4.3609718236E-01 ,  2.3724897337E+00 /
      data fk( 785), fk( 786) /  2.2397034646E+00 ,  1.9457795442E+00 /
      data fk( 787), fk( 788) / -1.5354165857E+00 , -1.5354165857E+00 /
      data fk( 789), fk( 790) / -6.9600412209E-01 ,  1.9457795442E+00 /
      data fk( 791), fk( 792) /  2.2397034646E+00 ,  4.3609718236E-01 /
      data fk( 793), fk( 794) / -6.9600412209E-01 ,  2.3724897337E+00 /
      data fk( 795), fk( 796) /  2.2397034646E+00 ,  1.9457795442E+00 /
      data fk( 797), fk( 798) /  1.9457795442E+00 ,  2.2397034646E+00 /
      data fk( 799), fk( 800) /  4.3609718236E-01 , -1.5354165857E+00 /
      data fk( 801), fk( 802) / -1.5354165857E+00 , -6.9600412209E-01 /
      data fk( 803), fk( 804) / -6.9600412209E-01 ,  1.6441827165E+01 /
      data fk( 805), fk( 806) /  9.5268112643E+00 ,  9.5268112643E+00 /
      data fk( 807), fk( 808) /  7.1398245793E+00 ,  7.1398245793E+00 /
      data fk( 809), fk( 810) /  4.7744504687E+00 ,  9.5268112643E+00 /
      data fk( 811), fk( 812) /  1.6441827165E+01 ,  9.5268112643E+00 /
      data fk( 813), fk( 814) /  7.1398245793E+00 ,  4.7744504687E+00 /
      data fk( 815), fk( 816) /  7.1398245793E+00 ,  3.2471973329E+00 /
      data fk( 817), fk( 818) / -3.9197228988E+00 , -5.8187341423E+00 /
      data fk( 819), fk( 820) / -6.0984402294E+00 , -5.5147687142E+00 /
      data fk( 821), fk( 822) /  1.7334987637E+00 ,  3.2471973329E+00 /
      data fk( 823), fk( 824) / -6.0984402294E+00 , -5.5147687142E+00 /
      data fk( 825), fk( 826) / -3.9197228988E+00 , -5.8187341423E+00 /
      data fk( 827), fk( 828) /  1.7334987637E+00 ,  9.9979590716E+00 /
      data fk( 829), fk( 830) / -2.6904632063E+00 , -3.0060012918E+00 /
      data fk( 831), fk( 832) / -3.0060012918E+00 , -2.6904632063E+00 /
      data fk( 833), fk( 834) /  2.0763119138E+00 ,  9.5268112643E+00 /
      data fk( 835), fk( 836) /  9.5268112643E+00 ,  1.6441827165E+01 /
      data fk( 837), fk( 838) /  4.7744504687E+00 ,  7.1398245793E+00 /
      data fk( 839), fk( 840) /  7.1398245793E+00 ,  3.2471973329E+00 /
      data fk( 841), fk( 842) / -5.8187341423E+00 , -3.9197228988E+00 /
      data fk( 843), fk( 844) / -5.5147687142E+00 , -6.0984402294E+00 /
      data fk( 845), fk( 846) /  1.7334987637E+00 ,  9.9979590716E+00 /
      data fk( 847), fk( 848) / -2.6904632063E+00 , -3.0060012918E+00 /
      data fk( 849), fk( 850) / -3.0060012918E+00 , -2.6904632063E+00 /
      data fk( 851), fk( 852) /  2.0763119138E+00 ,  3.2471973329E+00 /
      data fk( 853), fk( 854) / -6.0984402294E+00 , -5.5147687142E+00 /
      data fk( 855), fk( 856) / -3.9197228988E+00 , -5.8187341423E+00 /
      data fk( 857), fk( 858) /  1.7334987637E+00 ,  9.9979590716E+00 /
      data fk( 859), fk( 860) / -2.6904632063E+00 , -3.0060012918E+00 /
      data fk( 861), fk( 862) / -3.0060012918E+00 , -2.6904632063E+00 /
      data fk( 863), fk( 864) /  2.0763119138E+00 ,  3.2471973329E+00 /
      data fk( 865), fk( 866) / -5.8187341423E+00 , -3.9197228988E+00 /
      data fk( 867), fk( 868) / -5.5147687142E+00 , -6.0984402294E+00 /
      data fk( 869), fk( 870) /  1.7334987637E+00 ,  3.2471973329E+00 /
      data fk( 871), fk( 872) / -5.5147687142E+00 , -6.0984402294E+00 /
      data fk( 873), fk( 874) / -5.8187341423E+00 , -3.9197228988E+00 /
      data fk( 875), fk( 876) /  1.7334987637E+00 ,  1.5797852445E+00 /
      data fk( 877), fk( 878) /  1.5797852445E+00 ,  1.6944711759E+00 /
      data fk( 879), fk( 880) /  1.6000428759E-01 , -3.8844851007E-01 /
      data fk( 881), fk( 882) / -3.8844851007E-01 ,  1.5797852445E+00 /
      data fk( 883), fk( 884) /  1.6000428759E-01 , -3.8844851007E-01 /
      data fk( 885), fk( 886) /  1.5797852445E+00 ,  1.6944711759E+00 /
      data fk( 887), fk( 888) / -3.8844851007E-01 ,  1.6000428759E-01 /
      data fk( 889), fk( 890) /  1.5797852445E+00 , -3.8844851007E-01 /
      data fk( 891), fk( 892) /  1.5797852445E+00 , -3.8844851007E-01 /
      data fk( 893), fk( 894) /  1.6944711759E+00 ,  1.5644359360E+01 /
      data fk( 895), fk( 896) /  1.1112711042E+01 ,  1.1112711042E+01 /
      data fk( 897), fk( 898) / -9.2291613855E+00 , -9.2291613855E+00 /
      data fk( 899), fk( 900) /  1.1112711042E+01 ,  1.5644359360E+01 /
      data fk( 901), fk( 902) /  1.1112711042E+01 , -9.2291613855E+00 /
      data fk( 903), fk( 904) / -9.2291613855E+00 ,  1.1112711042E+01 /
      data fk( 905), fk( 906) /  1.1112711042E+01 ,  1.5644359360E+01 /
      data fk( 907), fk( 908) / -9.2291613855E+00 , -9.2291613855E+00 /
      data fk( 909), fk( 910) / -1.0816806291E+01 , -1.0816806291E+01 /
      data fk( 911), fk( 912) /  5.9821248489E+00 ,  1.1931915956E+00 /
      data fk( 913), fk( 914) /  1.1931915956E+00 , -1.0816806291E+01 /
      data fk( 915), fk( 916) / -1.0816806291E+01 ,  1.1931915956E+00 /
      data fk( 917), fk( 918) /  5.9821248489E+00 ,  1.1931915956E+00 /
      data fk( 919), fk( 920) / -1.0816806291E+01 , -1.0816806291E+01 /
      data fk( 921), fk( 922) /  1.1931915956E+00 ,  1.1931915956E+00 /
      data fk( 923), fk( 924) /  5.9821248489E+00 , -9.5992696774E+00 /
   
!
      en=0.0d0
      do i=1,nvar
          fmr(i,1) = 1.0d0- exp(-alpham(i)*(rij(i)-shift(i)))
          do j=2,nmax
              fmr(i,j) = fmr(i,1)**j
          enddo
      enddo
!
      en=en+fk(1) 
      en=en+fk(2)*fmr(1,1) 
      en=en+fk(3)*fmr(2,1) 
      en=en+fk(4)*fmr(3,1) 
      en=en+fk(5)*fmr(4,1) 
      en=en+fk(6)*fmr(5,1) 
      en=en+fk(7)*fmr(6,1) 
      en=en+fk(8)*fmr(1,2) 
      en=en+fk(9)*fmr(2,2) 
      en=en+fk(10)*fmr(3,2) 
      en=en+fk(11)*fmr(4,2) 
      en=en+fk(12)*fmr(5,2) 
      en=en+fk(13)*fmr(6,2) 
      en=en+fk(14)*fmr(1,1)*fmr(2,1) 
      en=en+fk(15)*fmr(1,1)*fmr(3,1) 
      en=en+fk(16)*fmr(1,1)*fmr(4,1) 
      en=en+fk(17)*fmr(1,1)*fmr(5,1) 
      en=en+fk(18)*fmr(1,1)*fmr(6,1) 
      en=en+fk(19)*fmr(2,1)*fmr(3,1) 
      en=en+fk(20)*fmr(2,1)*fmr(4,1) 
      en=en+fk(21)*fmr(2,1)*fmr(5,1) 
      en=en+fk(22)*fmr(2,1)*fmr(6,1) 
      en=en+fk(23)*fmr(3,1)*fmr(4,1) 
      en=en+fk(24)*fmr(3,1)*fmr(5,1) 
      en=en+fk(25)*fmr(3,1)*fmr(6,1) 
      en=en+fk(26)*fmr(4,1)*fmr(5,1) 
      en=en+fk(27)*fmr(4,1)*fmr(6,1) 
      en=en+fk(28)*fmr(5,1)*fmr(6,1) 
      en=en+fk(29)*fmr(1,3) 
      en=en+fk(30)*fmr(2,3) 
      en=en+fk(31)*fmr(3,3) 
      en=en+fk(32)*fmr(4,3) 
      en=en+fk(33)*fmr(5,3) 
      en=en+fk(34)*fmr(6,3) 
      en=en+fk(35)*fmr(1,2)*fmr(2,1) 
      en=en+fk(36)*fmr(1,2)*fmr(3,1) 
      en=en+fk(37)*fmr(1,2)*fmr(4,1) 
      en=en+fk(38)*fmr(1,2)*fmr(5,1) 
      en=en+fk(39)*fmr(1,2)*fmr(6,1) 
      en=en+fk(40)*fmr(1,1)*fmr(2,2) 
      en=en+fk(41)*fmr(2,2)*fmr(3,1) 
      en=en+fk(42)*fmr(2,2)*fmr(4,1) 
      en=en+fk(43)*fmr(2,2)*fmr(5,1) 
      en=en+fk(44)*fmr(2,2)*fmr(6,1) 
      en=en+fk(45)*fmr(1,1)*fmr(3,2) 
      en=en+fk(46)*fmr(2,1)*fmr(3,2) 
      en=en+fk(47)*fmr(3,2)*fmr(4,1) 
      en=en+fk(48)*fmr(3,2)*fmr(5,1) 
      en=en+fk(49)*fmr(3,2)*fmr(6,1) 
      en=en+fk(50)*fmr(1,1)*fmr(4,2) 
      en=en+fk(51)*fmr(2,1)*fmr(4,2) 
      en=en+fk(52)*fmr(3,1)*fmr(4,2) 
      en=en+fk(53)*fmr(4,2)*fmr(5,1) 
      en=en+fk(54)*fmr(4,2)*fmr(6,1) 
      en=en+fk(55)*fmr(1,1)*fmr(5,2) 
      en=en+fk(56)*fmr(2,1)*fmr(5,2) 
      en=en+fk(57)*fmr(3,1)*fmr(5,2) 
      en=en+fk(58)*fmr(4,1)*fmr(5,2) 
      en=en+fk(59)*fmr(5,2)*fmr(6,1) 
      en=en+fk(60)*fmr(1,1)*fmr(6,2) 
      en=en+fk(61)*fmr(2,1)*fmr(6,2) 
      en=en+fk(62)*fmr(3,1)*fmr(6,2) 
      en=en+fk(63)*fmr(4,1)*fmr(6,2) 
      en=en+fk(64)*fmr(5,1)*fmr(6,2) 
      en=en+fk(65)*fmr(1,1)*fmr(2,1)*fmr(3,1) 
      en=en+fk(66)*fmr(1,1)*fmr(2,1)*fmr(4,1) 
      en=en+fk(67)*fmr(1,1)*fmr(2,1)*fmr(5,1) 
      en=en+fk(68)*fmr(1,1)*fmr(2,1)*fmr(6,1) 
      en=en+fk(69)*fmr(1,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(70)*fmr(1,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(71)*fmr(1,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(72)*fmr(1,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(73)*fmr(1,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(74)*fmr(1,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(75)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(76)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(77)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(78)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(79)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(80)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(81)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(82)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(83)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(84)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(85)*fmr(1,4) 
      en=en+fk(86)*fmr(2,4) 
      en=en+fk(87)*fmr(3,4) 
      en=en+fk(88)*fmr(4,4) 
      en=en+fk(89)*fmr(5,4) 
      en=en+fk(90)*fmr(6,4) 
      en=en+fk(91)*fmr(1,3)*fmr(2,1) 
      en=en+fk(92)*fmr(1,3)*fmr(3,1) 
      en=en+fk(93)*fmr(1,3)*fmr(4,1) 
      en=en+fk(94)*fmr(1,3)*fmr(5,1) 
      en=en+fk(95)*fmr(1,3)*fmr(6,1) 
      en=en+fk(96)*fmr(1,1)*fmr(2,3) 
      en=en+fk(97)*fmr(2,3)*fmr(3,1) 
      en=en+fk(98)*fmr(2,3)*fmr(4,1) 
      en=en+fk(99)*fmr(2,3)*fmr(5,1) 
      en=en+fk(100)*fmr(2,3)*fmr(6,1) 
      en=en+fk(101)*fmr(1,1)*fmr(3,3) 
      en=en+fk(102)*fmr(2,1)*fmr(3,3) 
      en=en+fk(103)*fmr(3,3)*fmr(4,1) 
      en=en+fk(104)*fmr(3,3)*fmr(5,1) 
      en=en+fk(105)*fmr(3,3)*fmr(6,1) 
      en=en+fk(106)*fmr(1,1)*fmr(4,3) 
      en=en+fk(107)*fmr(2,1)*fmr(4,3) 
      en=en+fk(108)*fmr(3,1)*fmr(4,3) 
      en=en+fk(109)*fmr(4,3)*fmr(5,1) 
      en=en+fk(110)*fmr(4,3)*fmr(6,1) 
      en=en+fk(111)*fmr(1,1)*fmr(5,3) 
      en=en+fk(112)*fmr(2,1)*fmr(5,3) 
      en=en+fk(113)*fmr(3,1)*fmr(5,3) 
      en=en+fk(114)*fmr(4,1)*fmr(5,3) 
      en=en+fk(115)*fmr(5,3)*fmr(6,1) 
      en=en+fk(116)*fmr(1,1)*fmr(6,3) 
      en=en+fk(117)*fmr(2,1)*fmr(6,3) 
      en=en+fk(118)*fmr(3,1)*fmr(6,3) 
      en=en+fk(119)*fmr(4,1)*fmr(6,3) 
      en=en+fk(120)*fmr(5,1)*fmr(6,3) 
      en=en+fk(121)*fmr(1,2)*fmr(2,2) 
      en=en+fk(122)*fmr(1,2)*fmr(3,2) 
      en=en+fk(123)*fmr(1,2)*fmr(4,2) 
      en=en+fk(124)*fmr(1,2)*fmr(5,2) 
      en=en+fk(125)*fmr(1,2)*fmr(6,2) 
      en=en+fk(126)*fmr(2,2)*fmr(3,2) 
      en=en+fk(127)*fmr(2,2)*fmr(4,2) 
      en=en+fk(128)*fmr(2,2)*fmr(5,2) 
      en=en+fk(129)*fmr(2,2)*fmr(6,2) 
      en=en+fk(130)*fmr(3,2)*fmr(4,2) 
      en=en+fk(131)*fmr(3,2)*fmr(5,2) 
      en=en+fk(132)*fmr(3,2)*fmr(6,2) 
      en=en+fk(133)*fmr(4,2)*fmr(5,2) 
      en=en+fk(134)*fmr(4,2)*fmr(6,2) 
      en=en+fk(135)*fmr(5,2)*fmr(6,2) 
      en=en+fk(136)*fmr(1,2)*fmr(2,1)*fmr(3,1) 
      en=en+fk(137)*fmr(1,2)*fmr(2,1)*fmr(4,1) 
      en=en+fk(138)*fmr(1,2)*fmr(2,1)*fmr(5,1) 
      en=en+fk(139)*fmr(1,2)*fmr(2,1)*fmr(6,1) 
      en=en+fk(140)*fmr(1,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(141)*fmr(1,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(142)*fmr(1,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(143)*fmr(1,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(144)*fmr(1,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(145)*fmr(1,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(146)*fmr(1,1)*fmr(2,2)*fmr(3,1) 
      en=en+fk(147)*fmr(1,1)*fmr(2,2)*fmr(4,1) 
      en=en+fk(148)*fmr(1,1)*fmr(2,2)*fmr(5,1) 
      en=en+fk(149)*fmr(1,1)*fmr(2,2)*fmr(6,1) 
      en=en+fk(150)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(151)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(152)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(153)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(154)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(155)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(156)*fmr(1,1)*fmr(2,1)*fmr(3,2) 
      en=en+fk(157)*fmr(1,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(158)*fmr(1,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(159)*fmr(1,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(160)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(161)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(162)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(163)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(164)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(165)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(166)*fmr(1,1)*fmr(2,1)*fmr(4,2) 
      en=en+fk(167)*fmr(1,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(168)*fmr(1,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(169)*fmr(1,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(170)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(171)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(172)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(173)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(174)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(175)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(176)*fmr(1,1)*fmr(2,1)*fmr(5,2) 
      en=en+fk(177)*fmr(1,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(178)*fmr(1,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(179)*fmr(1,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(180)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(181)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(182)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(183)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(184)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(185)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(186)*fmr(1,1)*fmr(2,1)*fmr(6,2) 
      en=en+fk(187)*fmr(1,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(188)*fmr(1,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(189)*fmr(1,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(190)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(191)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(192)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(193)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(194)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(195)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(196)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(197)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(198)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(199)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(200)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(201)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(202)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(203)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(204)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(205)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(206)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(207)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(208)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(209)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(210)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(211)*fmr(1,5) 
      en=en+fk(212)*fmr(2,5) 
      en=en+fk(213)*fmr(3,5) 
      en=en+fk(214)*fmr(4,5) 
      en=en+fk(215)*fmr(5,5) 
      en=en+fk(216)*fmr(6,5) 
      en=en+fk(217)*fmr(1,4)*fmr(2,1) 
      en=en+fk(218)*fmr(1,4)*fmr(3,1) 
      en=en+fk(219)*fmr(1,4)*fmr(4,1) 
      en=en+fk(220)*fmr(1,4)*fmr(5,1) 
      en=en+fk(221)*fmr(1,4)*fmr(6,1) 
      en=en+fk(222)*fmr(1,1)*fmr(2,4) 
      en=en+fk(223)*fmr(2,4)*fmr(3,1) 
      en=en+fk(224)*fmr(2,4)*fmr(4,1) 
      en=en+fk(225)*fmr(2,4)*fmr(5,1) 
      en=en+fk(226)*fmr(2,4)*fmr(6,1) 
      en=en+fk(227)*fmr(1,1)*fmr(3,4) 
      en=en+fk(228)*fmr(2,1)*fmr(3,4) 
      en=en+fk(229)*fmr(3,4)*fmr(4,1) 
      en=en+fk(230)*fmr(3,4)*fmr(5,1) 
      en=en+fk(231)*fmr(3,4)*fmr(6,1) 
      en=en+fk(232)*fmr(1,1)*fmr(4,4) 
      en=en+fk(233)*fmr(2,1)*fmr(4,4) 
      en=en+fk(234)*fmr(3,1)*fmr(4,4) 
      en=en+fk(235)*fmr(4,4)*fmr(5,1) 
      en=en+fk(236)*fmr(4,4)*fmr(6,1) 
      en=en+fk(237)*fmr(1,1)*fmr(5,4) 
      en=en+fk(238)*fmr(2,1)*fmr(5,4) 
      en=en+fk(239)*fmr(3,1)*fmr(5,4) 
      en=en+fk(240)*fmr(4,1)*fmr(5,4) 
      en=en+fk(241)*fmr(5,4)*fmr(6,1) 
      en=en+fk(242)*fmr(1,1)*fmr(6,4) 
      en=en+fk(243)*fmr(2,1)*fmr(6,4) 
      en=en+fk(244)*fmr(3,1)*fmr(6,4) 
      en=en+fk(245)*fmr(4,1)*fmr(6,4) 
      en=en+fk(246)*fmr(5,1)*fmr(6,4) 
      en=en+fk(247)*fmr(1,3)*fmr(2,2) 
      en=en+fk(248)*fmr(1,3)*fmr(3,2) 
      en=en+fk(249)*fmr(1,3)*fmr(4,2) 
      en=en+fk(250)*fmr(1,3)*fmr(5,2) 
      en=en+fk(251)*fmr(1,3)*fmr(6,2) 
      en=en+fk(252)*fmr(1,2)*fmr(2,3) 
      en=en+fk(253)*fmr(2,3)*fmr(3,2) 
      en=en+fk(254)*fmr(2,3)*fmr(4,2) 
      en=en+fk(255)*fmr(2,3)*fmr(5,2) 
      en=en+fk(256)*fmr(2,3)*fmr(6,2) 
      en=en+fk(257)*fmr(1,2)*fmr(3,3) 
      en=en+fk(258)*fmr(2,2)*fmr(3,3) 
      en=en+fk(259)*fmr(3,3)*fmr(4,2) 
      en=en+fk(260)*fmr(3,3)*fmr(5,2) 
      en=en+fk(261)*fmr(3,3)*fmr(6,2) 
      en=en+fk(262)*fmr(1,2)*fmr(4,3) 
      en=en+fk(263)*fmr(2,2)*fmr(4,3) 
      en=en+fk(264)*fmr(3,2)*fmr(4,3) 
      en=en+fk(265)*fmr(4,3)*fmr(5,2) 
      en=en+fk(266)*fmr(4,3)*fmr(6,2) 
      en=en+fk(267)*fmr(1,2)*fmr(5,3) 
      en=en+fk(268)*fmr(2,2)*fmr(5,3) 
      en=en+fk(269)*fmr(3,2)*fmr(5,3) 
      en=en+fk(270)*fmr(4,2)*fmr(5,3) 
      en=en+fk(271)*fmr(5,3)*fmr(6,2) 
      en=en+fk(272)*fmr(1,2)*fmr(6,3) 
      en=en+fk(273)*fmr(2,2)*fmr(6,3) 
      en=en+fk(274)*fmr(3,2)*fmr(6,3) 
      en=en+fk(275)*fmr(4,2)*fmr(6,3) 
      en=en+fk(276)*fmr(5,2)*fmr(6,3) 
      en=en+fk(277)*fmr(1,3)*fmr(2,1)*fmr(3,1) 
      en=en+fk(278)*fmr(1,3)*fmr(2,1)*fmr(4,1) 
      en=en+fk(279)*fmr(1,3)*fmr(2,1)*fmr(5,1) 
      en=en+fk(280)*fmr(1,3)*fmr(2,1)*fmr(6,1) 
      en=en+fk(281)*fmr(1,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(282)*fmr(1,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(283)*fmr(1,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(284)*fmr(1,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(285)*fmr(1,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(286)*fmr(1,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(287)*fmr(1,1)*fmr(2,3)*fmr(3,1) 
      en=en+fk(288)*fmr(1,1)*fmr(2,3)*fmr(4,1) 
      en=en+fk(289)*fmr(1,1)*fmr(2,3)*fmr(5,1) 
      en=en+fk(290)*fmr(1,1)*fmr(2,3)*fmr(6,1) 
      en=en+fk(291)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(292)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(293)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(294)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(295)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(296)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(297)*fmr(1,1)*fmr(2,1)*fmr(3,3) 
      en=en+fk(298)*fmr(1,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(299)*fmr(1,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(300)*fmr(1,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(301)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(302)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(303)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(304)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(305)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(306)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(307)*fmr(1,1)*fmr(2,1)*fmr(4,3) 
      en=en+fk(308)*fmr(1,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(309)*fmr(1,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(310)*fmr(1,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(311)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(312)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(313)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(314)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(315)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(316)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(317)*fmr(1,1)*fmr(2,1)*fmr(5,3) 
      en=en+fk(318)*fmr(1,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(319)*fmr(1,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(320)*fmr(1,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(321)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(322)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(323)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(324)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(325)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(326)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(327)*fmr(1,1)*fmr(2,1)*fmr(6,3) 
      en=en+fk(328)*fmr(1,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(329)*fmr(1,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(330)*fmr(1,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(331)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(332)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(333)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(334)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(335)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(336)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(337)*fmr(1,1)*fmr(2,2)*fmr(3,2) 
      en=en+fk(338)*fmr(1,1)*fmr(2,2)*fmr(4,2) 
      en=en+fk(339)*fmr(1,1)*fmr(2,2)*fmr(5,2) 
      en=en+fk(340)*fmr(1,1)*fmr(2,2)*fmr(6,2) 
      en=en+fk(341)*fmr(1,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(342)*fmr(1,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(343)*fmr(1,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(344)*fmr(1,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(345)*fmr(1,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(346)*fmr(1,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(347)*fmr(1,2)*fmr(2,1)*fmr(3,2) 
      en=en+fk(348)*fmr(1,2)*fmr(2,1)*fmr(4,2) 
      en=en+fk(349)*fmr(1,2)*fmr(2,1)*fmr(5,2) 
      en=en+fk(350)*fmr(1,2)*fmr(2,1)*fmr(6,2) 
      en=en+fk(351)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(352)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(353)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(354)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(355)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(356)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(357)*fmr(1,2)*fmr(2,2)*fmr(3,1) 
      en=en+fk(358)*fmr(1,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(359)*fmr(1,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(360)*fmr(1,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(361)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(362)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(363)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(364)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(365)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(366)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(367)*fmr(1,2)*fmr(2,2)*fmr(4,1) 
      en=en+fk(368)*fmr(1,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(369)*fmr(1,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(370)*fmr(1,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(371)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(372)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(373)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(374)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(375)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(376)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(377)*fmr(1,2)*fmr(2,2)*fmr(5,1) 
      en=en+fk(378)*fmr(1,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(379)*fmr(1,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(380)*fmr(1,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(381)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(382)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(383)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(384)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(385)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(386)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(387)*fmr(1,2)*fmr(2,2)*fmr(6,1) 
      en=en+fk(388)*fmr(1,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(389)*fmr(1,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(390)*fmr(1,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(391)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(392)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(393)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(394)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(395)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(396)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(397)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(398)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(399)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(400)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(401)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(402)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(403)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(404)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(405)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(406)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(407)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(408)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(409)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(410)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(411)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(412)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(413)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(414)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(415)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(416)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(417)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(418)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(419)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(420)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(421)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(422)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(423)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(424)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(425)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(426)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(427)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(428)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(429)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(430)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(431)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(432)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(433)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(434)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(435)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(436)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(437)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(438)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(439)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(440)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(441)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(442)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(443)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(444)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(445)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(446)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(447)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(448)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(449)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(450)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(451)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(452)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(453)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(454)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(455)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(456)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(457)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(458)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(459)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(460)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(461)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(462)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(463)*fmr(1,6) 
      en=en+fk(464)*fmr(2,6) 
      en=en+fk(465)*fmr(3,6) 
      en=en+fk(466)*fmr(4,6) 
      en=en+fk(467)*fmr(5,6) 
      en=en+fk(468)*fmr(6,6) 
      en=en+fk(469)*fmr(1,5)*fmr(2,1) 
      en=en+fk(470)*fmr(1,5)*fmr(3,1) 
      en=en+fk(471)*fmr(1,5)*fmr(4,1) 
      en=en+fk(472)*fmr(1,5)*fmr(5,1) 
      en=en+fk(473)*fmr(1,5)*fmr(6,1) 
      en=en+fk(474)*fmr(1,1)*fmr(2,5) 
      en=en+fk(475)*fmr(2,5)*fmr(3,1) 
      en=en+fk(476)*fmr(2,5)*fmr(4,1) 
      en=en+fk(477)*fmr(2,5)*fmr(5,1) 
      en=en+fk(478)*fmr(2,5)*fmr(6,1) 
      en=en+fk(479)*fmr(1,1)*fmr(3,5) 
      en=en+fk(480)*fmr(2,1)*fmr(3,5) 
      en=en+fk(481)*fmr(3,5)*fmr(4,1) 
      en=en+fk(482)*fmr(3,5)*fmr(5,1) 
      en=en+fk(483)*fmr(3,5)*fmr(6,1) 
      en=en+fk(484)*fmr(1,1)*fmr(4,5) 
      en=en+fk(485)*fmr(2,1)*fmr(4,5) 
      en=en+fk(486)*fmr(3,1)*fmr(4,5) 
      en=en+fk(487)*fmr(4,5)*fmr(5,1) 
      en=en+fk(488)*fmr(4,5)*fmr(6,1) 
      en=en+fk(489)*fmr(1,1)*fmr(5,5) 
      en=en+fk(490)*fmr(2,1)*fmr(5,5) 
      en=en+fk(491)*fmr(3,1)*fmr(5,5) 
      en=en+fk(492)*fmr(4,1)*fmr(5,5) 
      en=en+fk(493)*fmr(5,5)*fmr(6,1) 
      en=en+fk(494)*fmr(1,1)*fmr(6,5) 
      en=en+fk(495)*fmr(2,1)*fmr(6,5) 
      en=en+fk(496)*fmr(3,1)*fmr(6,5) 
      en=en+fk(497)*fmr(4,1)*fmr(6,5) 
      en=en+fk(498)*fmr(5,1)*fmr(6,5) 
      en=en+fk(499)*fmr(1,4)*fmr(2,2) 
      en=en+fk(500)*fmr(1,4)*fmr(3,2) 
      en=en+fk(501)*fmr(1,4)*fmr(4,2) 
      en=en+fk(502)*fmr(1,4)*fmr(5,2) 
      en=en+fk(503)*fmr(1,4)*fmr(6,2) 
      en=en+fk(504)*fmr(1,2)*fmr(2,4) 
      en=en+fk(505)*fmr(2,4)*fmr(3,2) 
      en=en+fk(506)*fmr(2,4)*fmr(4,2) 
      en=en+fk(507)*fmr(2,4)*fmr(5,2) 
      en=en+fk(508)*fmr(2,4)*fmr(6,2) 
      en=en+fk(509)*fmr(1,2)*fmr(3,4) 
      en=en+fk(510)*fmr(2,2)*fmr(3,4) 
      en=en+fk(511)*fmr(3,4)*fmr(4,2) 
      en=en+fk(512)*fmr(3,4)*fmr(5,2) 
      en=en+fk(513)*fmr(3,4)*fmr(6,2) 
      en=en+fk(514)*fmr(1,2)*fmr(4,4) 
      en=en+fk(515)*fmr(2,2)*fmr(4,4) 
      en=en+fk(516)*fmr(3,2)*fmr(4,4) 
      en=en+fk(517)*fmr(4,4)*fmr(5,2) 
      en=en+fk(518)*fmr(4,4)*fmr(6,2) 
      en=en+fk(519)*fmr(1,2)*fmr(5,4) 
      en=en+fk(520)*fmr(2,2)*fmr(5,4) 
      en=en+fk(521)*fmr(3,2)*fmr(5,4) 
      en=en+fk(522)*fmr(4,2)*fmr(5,4) 
      en=en+fk(523)*fmr(5,4)*fmr(6,2) 
      en=en+fk(524)*fmr(1,2)*fmr(6,4) 
      en=en+fk(525)*fmr(2,2)*fmr(6,4) 
      en=en+fk(526)*fmr(3,2)*fmr(6,4) 
      en=en+fk(527)*fmr(4,2)*fmr(6,4) 
      en=en+fk(528)*fmr(5,2)*fmr(6,4) 
      en=en+fk(529)*fmr(1,3)*fmr(2,3) 
      en=en+fk(530)*fmr(1,3)*fmr(3,3) 
      en=en+fk(531)*fmr(1,3)*fmr(4,3) 
      en=en+fk(532)*fmr(1,3)*fmr(5,3) 
      en=en+fk(533)*fmr(1,3)*fmr(6,3) 
      en=en+fk(534)*fmr(2,3)*fmr(3,3) 
      en=en+fk(535)*fmr(2,3)*fmr(4,3) 
      en=en+fk(536)*fmr(2,3)*fmr(5,3) 
      en=en+fk(537)*fmr(2,3)*fmr(6,3) 
      en=en+fk(538)*fmr(3,3)*fmr(4,3) 
      en=en+fk(539)*fmr(3,3)*fmr(5,3) 
      en=en+fk(540)*fmr(3,3)*fmr(6,3) 
      en=en+fk(541)*fmr(4,3)*fmr(5,3) 
      en=en+fk(542)*fmr(4,3)*fmr(6,3) 
      en=en+fk(543)*fmr(5,3)*fmr(6,3) 
      en=en+fk(544)*fmr(4,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(545)*fmr(3,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(546)*fmr(3,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(547)*fmr(3,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(548)*fmr(2,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(549)*fmr(2,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(550)*fmr(2,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(551)*fmr(2,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(552)*fmr(2,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(553)*fmr(2,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(554)*fmr(1,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(555)*fmr(1,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(556)*fmr(1,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(557)*fmr(1,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(558)*fmr(1,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(559)*fmr(1,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(560)*fmr(1,2)*fmr(2,2)*fmr(6,2) 
      en=en+fk(561)*fmr(1,2)*fmr(2,2)*fmr(5,2) 
      en=en+fk(562)*fmr(1,2)*fmr(2,2)*fmr(4,2) 
      en=en+fk(563)*fmr(1,2)*fmr(2,2)*fmr(3,2) 
      en=en+fk(564)*fmr(1,4)*fmr(2,1)*fmr(3,1) 
      en=en+fk(565)*fmr(1,4)*fmr(2,1)*fmr(4,1) 
      en=en+fk(566)*fmr(1,4)*fmr(2,1)*fmr(5,1) 
      en=en+fk(567)*fmr(1,4)*fmr(2,1)*fmr(6,1) 
      en=en+fk(568)*fmr(1,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(569)*fmr(1,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(570)*fmr(1,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(571)*fmr(1,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(572)*fmr(1,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(573)*fmr(1,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(574)*fmr(1,1)*fmr(2,4)*fmr(3,1) 
      en=en+fk(575)*fmr(1,1)*fmr(2,4)*fmr(4,1) 
      en=en+fk(576)*fmr(1,1)*fmr(2,4)*fmr(5,1) 
      en=en+fk(577)*fmr(1,1)*fmr(2,4)*fmr(6,1) 
      en=en+fk(578)*fmr(2,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(579)*fmr(2,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(580)*fmr(2,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(581)*fmr(2,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(582)*fmr(2,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(583)*fmr(2,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(584)*fmr(1,1)*fmr(2,1)*fmr(3,4) 
      en=en+fk(585)*fmr(1,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(586)*fmr(1,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(587)*fmr(1,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(588)*fmr(2,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(589)*fmr(2,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(590)*fmr(2,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(591)*fmr(3,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(592)*fmr(3,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(593)*fmr(3,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(594)*fmr(1,1)*fmr(2,1)*fmr(4,4) 
      en=en+fk(595)*fmr(1,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(596)*fmr(1,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(597)*fmr(1,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(598)*fmr(2,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(599)*fmr(2,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(600)*fmr(2,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(601)*fmr(3,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(602)*fmr(3,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(603)*fmr(4,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(604)*fmr(1,1)*fmr(2,1)*fmr(5,4) 
      en=en+fk(605)*fmr(1,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(606)*fmr(1,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(607)*fmr(1,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(608)*fmr(2,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(609)*fmr(2,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(610)*fmr(2,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(611)*fmr(3,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(612)*fmr(3,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(613)*fmr(4,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(614)*fmr(1,1)*fmr(2,1)*fmr(6,4) 
      en=en+fk(615)*fmr(1,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(616)*fmr(1,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(617)*fmr(1,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(618)*fmr(2,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(619)*fmr(2,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(620)*fmr(2,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(621)*fmr(3,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(622)*fmr(3,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(623)*fmr(4,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(624)*fmr(1,3)*fmr(2,2)*fmr(3,1) 
      en=en+fk(625)*fmr(1,3)*fmr(2,2)*fmr(4,1) 
      en=en+fk(626)*fmr(1,3)*fmr(2,2)*fmr(5,1) 
      en=en+fk(627)*fmr(1,3)*fmr(2,2)*fmr(6,1) 
      en=en+fk(628)*fmr(1,3)*fmr(2,1)*fmr(3,2) 
      en=en+fk(629)*fmr(1,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(630)*fmr(1,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(631)*fmr(1,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(632)*fmr(1,3)*fmr(2,1)*fmr(4,2) 
      en=en+fk(633)*fmr(1,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(634)*fmr(1,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(635)*fmr(1,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(636)*fmr(1,3)*fmr(2,1)*fmr(5,2) 
      en=en+fk(637)*fmr(1,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(638)*fmr(1,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(639)*fmr(1,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(640)*fmr(1,3)*fmr(2,1)*fmr(6,2) 
      en=en+fk(641)*fmr(1,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(642)*fmr(1,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(643)*fmr(1,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(644)*fmr(1,2)*fmr(2,3)*fmr(3,1) 
      en=en+fk(645)*fmr(1,2)*fmr(2,3)*fmr(4,1) 
      en=en+fk(646)*fmr(1,2)*fmr(2,3)*fmr(5,1) 
      en=en+fk(647)*fmr(1,2)*fmr(2,3)*fmr(6,1) 
      en=en+fk(648)*fmr(1,1)*fmr(2,3)*fmr(3,2) 
      en=en+fk(649)*fmr(2,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(650)*fmr(2,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(651)*fmr(2,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(652)*fmr(1,1)*fmr(2,3)*fmr(4,2) 
      en=en+fk(653)*fmr(2,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(654)*fmr(2,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(655)*fmr(2,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(656)*fmr(1,1)*fmr(2,3)*fmr(5,2) 
      en=en+fk(657)*fmr(2,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(658)*fmr(2,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(659)*fmr(2,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(660)*fmr(1,1)*fmr(2,3)*fmr(6,2) 
      en=en+fk(661)*fmr(2,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(662)*fmr(2,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(663)*fmr(2,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(664)*fmr(1,2)*fmr(2,1)*fmr(3,3) 
      en=en+fk(665)*fmr(1,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(666)*fmr(1,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(667)*fmr(1,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(668)*fmr(1,1)*fmr(2,2)*fmr(3,3) 
      en=en+fk(669)*fmr(2,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(670)*fmr(2,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(671)*fmr(2,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(672)*fmr(1,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(673)*fmr(2,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(674)*fmr(3,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(675)*fmr(3,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(676)*fmr(1,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(677)*fmr(2,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(678)*fmr(3,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(679)*fmr(3,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(680)*fmr(1,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(681)*fmr(2,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(682)*fmr(3,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(683)*fmr(3,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(684)*fmr(1,2)*fmr(2,1)*fmr(4,3) 
      en=en+fk(685)*fmr(1,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(686)*fmr(1,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(687)*fmr(1,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(688)*fmr(1,1)*fmr(2,2)*fmr(4,3) 
      en=en+fk(689)*fmr(2,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(690)*fmr(2,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(691)*fmr(2,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(692)*fmr(1,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(693)*fmr(2,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(694)*fmr(3,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(695)*fmr(3,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(696)*fmr(1,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(697)*fmr(2,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(698)*fmr(3,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(699)*fmr(4,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(700)*fmr(1,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(701)*fmr(2,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(702)*fmr(3,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(703)*fmr(4,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(704)*fmr(1,2)*fmr(2,1)*fmr(5,3) 
      en=en+fk(705)*fmr(1,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(706)*fmr(1,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(707)*fmr(1,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(708)*fmr(1,1)*fmr(2,2)*fmr(5,3) 
      en=en+fk(709)*fmr(2,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(710)*fmr(2,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(711)*fmr(2,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(712)*fmr(1,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(713)*fmr(2,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(714)*fmr(3,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(715)*fmr(3,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(716)*fmr(1,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(717)*fmr(2,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(718)*fmr(3,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(719)*fmr(4,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(720)*fmr(1,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(721)*fmr(2,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(722)*fmr(3,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(723)*fmr(4,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(724)*fmr(1,2)*fmr(2,1)*fmr(6,3) 
      en=en+fk(725)*fmr(1,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(726)*fmr(1,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(727)*fmr(1,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(728)*fmr(1,1)*fmr(2,2)*fmr(6,3) 
      en=en+fk(729)*fmr(2,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(730)*fmr(2,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(731)*fmr(2,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(732)*fmr(1,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(733)*fmr(2,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(734)*fmr(3,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(735)*fmr(3,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(736)*fmr(1,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(737)*fmr(2,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(738)*fmr(3,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(739)*fmr(4,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(740)*fmr(1,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(741)*fmr(2,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(742)*fmr(3,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(743)*fmr(4,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(744)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(745)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(746)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(747)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(748)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(749)*fmr(1,3)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(750)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(751)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(752)*fmr(1,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(753)*fmr(1,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(754)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(755)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(756)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(757)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(758)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(759)*fmr(1,1)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(760)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(761)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(762)*fmr(2,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(763)*fmr(2,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(764)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(765)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(766)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(767)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(768)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(769)*fmr(1,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(770)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(771)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(772)*fmr(2,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(773)*fmr(3,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(774)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(775)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(776)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(777)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(778)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(779)*fmr(1,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(780)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(781)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(782)*fmr(2,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(783)*fmr(3,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(784)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(785)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(786)*fmr(1,1)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(787)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(788)*fmr(1,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(789)*fmr(1,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(790)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(791)*fmr(2,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(792)*fmr(2,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(793)*fmr(3,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(794)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(795)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(796)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(797)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(798)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(799)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(800)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(801)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(802)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(803)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(804)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(805)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(806)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(807)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(808)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(809)*fmr(1,2)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(810)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(811)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(812)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(813)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(814)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(815)*fmr(1,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(816)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(817)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(818)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(819)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(820)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(821)*fmr(1,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(822)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(823)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(824)*fmr(1,2)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(825)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(826)*fmr(1,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(827)*fmr(1,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(828)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(829)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(830)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(831)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(832)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(833)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(834)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(835)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(836)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(837)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(838)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(839)*fmr(2,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(840)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(841)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(842)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(843)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(844)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(845)*fmr(2,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(846)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(847)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(848)*fmr(1,1)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(849)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(850)*fmr(2,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(851)*fmr(2,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(852)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(853)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(854)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(855)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(856)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(857)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(858)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(859)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(860)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(861)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(862)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(863)*fmr(3,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(864)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(865)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(866)*fmr(1,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(867)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(868)*fmr(2,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(869)*fmr(3,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(870)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(871)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(872)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(873)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(874)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(875)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(876)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(877)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(878)*fmr(1,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(879)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(880)*fmr(2,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(881)*fmr(3,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(882)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(883)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(884)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(885)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(886)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(887)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(888)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(889)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(890)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(891)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(892)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(893)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(894)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(895)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(896)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(897)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(898)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(899)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(900)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(901)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(902)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(903)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(904)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(905)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(906)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(907)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(908)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(909)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(910)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(911)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(912)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(913)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(914)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(915)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(916)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(917)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(918)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(919)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(920)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(921)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(922)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(923)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(924)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) &
        *fmr(6,1) 

      return
      end subroutine ch2ph
!
      subroutine methyl(rij,en)
!
!     from ch3-10941-924.p
!
!     rij(1) = r(c-h1) (au)
!     rij(2) = r(c-h2)
!     rij(3) = r(c-h3)
!     rij(4) = r(h1-h2)
!     rij(5) = r(h1-h3)
!     rij(6) = r(h2-h3)
!
!     en = energy (hartree)
!
!
!      implicit real*8 (a-h,o-z)
!      dimension fk(924),fmr(6,6), shift(6),rij(6),alpham(6)
      implicit none
      double precision,intent(in) :: rij(6)
      double precision,intent(out) :: en
      double precision :: fk(924),fmr(6,6),shift(6),alpham(6)
      integer :: i, j, nmax, nvar
!
      data fk(   1), fk(   2) / -3.9764313274E+01 ,  3.0923910238E-02 /
      data fk(   3), fk(   4) /  3.0923910238E-02 ,  3.0923910238E-02 /
      data fk(   5), fk(   6) / -1.8186573371E-02 , -1.8186573371E-02 /
      data fk(   7), fk(   8) / -1.8186573371E-02 ,  9.3204446382E-01 /
      data fk(   9), fk(  10) /  9.3204446382E-01 ,  9.3204446382E-01 /
      data fk(  11), fk(  12) /  1.8445809690E-01 ,  1.8445809690E-01 /
      data fk(  13), fk(  14) /  1.8445809690E-01 ,  2.1924752671E-01 /
      data fk(  15), fk(  16) /  2.1924752671E-01 , -2.8898652104E-01 /
      data fk(  17), fk(  18) / -2.8898652104E-01 , -6.7787341133E-02 /
      data fk(  19), fk(  20) /  2.1924752671E-01 , -2.8898652104E-01 /
      data fk(  21), fk(  22) / -6.7787341133E-02 , -2.8898652104E-01 /
      data fk(  23), fk(  24) / -6.7787341133E-02 , -2.8898652104E-01 /
      data fk(  25), fk(  26) / -2.8898652104E-01 ,  4.7279132555E-02 /
      data fk(  27), fk(  28) /  4.7279132555E-02 ,  4.7279132555E-02 /
      data fk(  29), fk(  30) / -1.1066188862E+00 , -1.1066188862E+00 /
      data fk(  31), fk(  32) / -1.1066188862E+00 ,  3.3270682836E-01 /
      data fk(  33), fk(  34) /  3.3270682836E-01 ,  3.3270682836E-01 /
      data fk(  35), fk(  36) / -9.3115840298E-01 , -9.3115840298E-01 /
      data fk(  37), fk(  38) /  6.6161709582E-01 ,  6.6161709582E-01 /
      data fk(  39), fk(  40) /  2.4086761289E-01 , -9.3115840298E-01 /
      data fk(  41), fk(  42) / -9.3115840298E-01 ,  6.6161709582E-01 /
      data fk(  43), fk(  44) /  2.4086761289E-01 ,  6.6161709582E-01 /
      data fk(  45), fk(  46) / -9.3115840298E-01 , -9.3115840298E-01 /
      data fk(  47), fk(  48) /  2.4086761289E-01 ,  6.6161709582E-01 /
      data fk(  49), fk(  50) /  6.6161709582E-01 , -7.7672436503E-01 /
      data fk(  51), fk(  52) / -7.7672436503E-01 , -1.3670465325E-01 /
      data fk(  53), fk(  54) /  9.3288638613E-02 ,  9.3288638613E-02 /
      data fk(  55), fk(  56) / -7.7672436503E-01 , -1.3670465325E-01 /
      data fk(  57), fk(  58) / -7.7672436503E-01 ,  9.3288638613E-02 /
      data fk(  59), fk(  60) /  9.3288638613E-02 , -1.3670465325E-01 /
      data fk(  61), fk(  62) / -7.7672436503E-01 , -7.7672436503E-01 /
      data fk(  63), fk(  64) /  9.3288638613E-02 ,  9.3288638613E-02 /
      data fk(  65), fk(  66) / -1.1403861964E+00 ,  1.6353328356E+00 /
      data fk(  67), fk(  68) /  5.4606362632E-01 ,  5.4606362632E-01 /
      data fk(  69), fk(  70) /  5.4606362632E-01 ,  1.6353328356E+00 /
      data fk(  71), fk(  72) /  5.4606362632E-01 , -2.7490038606E-01 /
      data fk(  73), fk(  74) / -3.2794619436E-01 , -3.2794619436E-01 /
      data fk(  75), fk(  76) /  5.4606362632E-01 ,  5.4606362632E-01 /
      data fk(  77), fk(  78) /  1.6353328356E+00 , -3.2794619436E-01 /
      data fk(  79), fk(  80) / -2.7490038606E-01 , -3.2794619436E-01 /
      data fk(  81), fk(  82) / -3.2794619436E-01 , -3.2794619436E-01 /
      data fk(  83), fk(  84) / -2.7490038606E-01 ,  2.4597010634E-01 /
      data fk(  85), fk(  86) /  9.8613968044E-01 ,  9.8613968044E-01 /
      data fk(  87), fk(  88) /  9.8613968044E-01 ,  4.0207111962E-01 /
      data fk(  89), fk(  90) /  4.0207111962E-01 ,  4.0207111962E-01 /
      data fk(  91), fk(  92) /  2.0601493975E+00 ,  2.0601493975E+00 /
      data fk(  93), fk(  94) / -9.0795537616E-01 , -9.0795537616E-01 /
      data fk(  95), fk(  96) / -3.9760925210E-01 ,  2.0601493975E+00 /
      data fk(  97), fk(  98) /  2.0601493975E+00 , -9.0795537616E-01 /
      data fk(  99), fk( 100) / -3.9760925210E-01 , -9.0795537616E-01 /
      data fk( 101), fk( 102) /  2.0601493975E+00 ,  2.0601493975E+00 /
      data fk( 103), fk( 104) / -3.9760925210E-01 , -9.0795537616E-01 /
      data fk( 105), fk( 106) / -9.0795537616E-01 , -1.0447710191E+00 /
      data fk( 107), fk( 108) / -1.0447710191E+00 ,  9.1444627931E-02 /
      data fk( 109), fk( 110) / -1.9880175361E-02 , -1.9880175361E-02 /
      data fk( 111), fk( 112) / -1.0447710191E+00 ,  9.1444627931E-02 /
      data fk( 113), fk( 114) / -1.0447710191E+00 , -1.9880175361E-02 /
      data fk( 115), fk( 116) / -1.9880175361E-02 ,  9.1444627931E-02 /
      data fk( 117), fk( 118) / -1.0447710191E+00 , -1.0447710191E+00 /
      data fk( 119), fk( 120) / -1.9880175361E-02 , -1.9880175361E-02 /
      data fk( 121), fk( 122) /  3.4077281342E+00 ,  3.4077281342E+00 /
      data fk( 123), fk( 124) /  1.2351602566E+00 ,  1.2351602566E+00 /
      data fk( 125), fk( 126) /  3.9215210598E-01 ,  3.4077281342E+00 /
      data fk( 127), fk( 128) /  1.2351602566E+00 ,  3.9215210598E-01 /
      data fk( 129), fk( 130) /  1.2351602566E+00 ,  3.9215210598E-01 /
      data fk( 131), fk( 132) /  1.2351602566E+00 ,  1.2351602566E+00 /
      data fk( 133), fk( 134) /  2.0799111156E-01 ,  2.0799111156E-01 /
      data fk( 135), fk( 136) /  2.0799111156E-01 ,  5.1615989752E+00 /
      data fk( 137), fk( 138) / -3.8860764890E+00 , -1.6263935006E+00 /
      data fk( 139), fk( 140) / -1.6962892278E+00 , -1.6263935006E+00 /
      data fk( 141), fk( 142) / -3.8860764890E+00 , -1.6962892278E+00 /
      data fk( 143), fk( 144) /  5.5143029929E-01 ,  5.1853168873E-01 /
      data fk( 145), fk( 146) /  5.1853168873E-01 ,  5.1615989752E+00 /
      data fk( 147), fk( 148) / -3.8860764890E+00 , -1.6962892278E+00 /
      data fk( 149), fk( 150) / -1.6263935006E+00 , -1.6263935006E+00 /
      data fk( 151), fk( 152) / -1.6962892278E+00 , -3.8860764890E+00 /
      data fk( 153), fk( 154) /  5.1853168873E-01 ,  5.5143029929E-01 /
      data fk( 155), fk( 156) /  5.1853168873E-01 ,  5.1615989752E+00 /
      data fk( 157), fk( 158) / -1.6962892278E+00 , -3.8860764890E+00 /
      data fk( 159), fk( 160) / -1.6263935006E+00 , -1.6962892278E+00 /
      data fk( 161), fk( 162) / -1.6263935006E+00 , -3.8860764890E+00 /
      data fk( 163), fk( 164) /  5.1853168873E-01 ,  5.1853168873E-01 /
      data fk( 165), fk( 166) /  5.5143029929E-01 ,  3.2193872735E+00 /
      data fk( 167), fk( 168) /  5.8505545319E-01 , -3.0599414029E-01 /
      data fk( 169), fk( 170) / -2.7252139846E-01 ,  5.8505545319E-01 /
      data fk( 171), fk( 172) / -2.7252139846E-01 , -3.0599414029E-01 /
      data fk( 173), fk( 174) / -4.9790522062E-01 , -4.9790522062E-01 /
      data fk( 175), fk( 176) /  2.8084019515E-01 ,  5.8505545319E-01 /
      data fk( 177), fk( 178) /  3.2193872735E+00 , -3.0599414029E-01 /
      data fk( 179), fk( 180) / -2.7252139846E-01 ,  5.8505545319E-01 /
      data fk( 181), fk( 182) / -4.9790522062E-01 , -4.9790522062E-01 /
      data fk( 183), fk( 184) / -2.7252139846E-01 , -3.0599414029E-01 /
      data fk( 185), fk( 186) /  2.8084019515E-01 ,  5.8505545319E-01 /
      data fk( 187), fk( 188) /  5.8505545319E-01 , -4.9790522062E-01 /
      data fk( 189), fk( 190) / -4.9790522062E-01 ,  3.2193872735E+00 /
      data fk( 191), fk( 192) / -3.0599414029E-01 , -2.7252139846E-01 /
      data fk( 193), fk( 194) / -2.7252139846E-01 , -3.0599414029E-01 /
      data fk( 195), fk( 196) /  2.8084019515E-01 , -6.9012554723E-01 /
      data fk( 197), fk( 198) / -6.9012554723E-01 ,  1.5102957134E+00 /
      data fk( 199), fk( 200) /  1.5102957134E+00 ,  1.5622380051E+00 /
      data fk( 201), fk( 202) / -6.9012554723E-01 ,  1.5102957134E+00 /
      data fk( 203), fk( 204) /  1.5622380051E+00 ,  1.5102957134E+00 /
      data fk( 205), fk( 206) /  1.5622380051E+00 ,  1.5102957134E+00 /
      data fk( 207), fk( 208) /  1.5102957134E+00 , -3.9988369547E+00 /
      data fk( 209), fk( 210) / -3.9988369547E+00 , -3.9988369547E+00 /
      data fk( 211), fk( 212) / -1.1795169591E+00 , -1.1795169591E+00 /
      data fk( 213), fk( 214) / -1.1795169591E+00 ,  2.6455455633E-01 /
      data fk( 215), fk( 216) /  2.6455455633E-01 ,  2.6455455633E-01 /
      data fk( 217), fk( 218) / -3.6051075982E+00 , -3.6051075982E+00 /
      data fk( 219), fk( 220) /  1.4158762831E+00 ,  1.4158762831E+00 /
      data fk( 221), fk( 222) /  2.9368817537E-01 , -3.6051075982E+00 /
      data fk( 223), fk( 224) / -3.6051075982E+00 ,  1.4158762831E+00 /
      data fk( 225), fk( 226) /  2.9368817537E-01 ,  1.4158762831E+00 /
      data fk( 227), fk( 228) / -3.6051075982E+00 , -3.6051075982E+00 /
      data fk( 229), fk( 230) /  2.9368817537E-01 ,  1.4158762831E+00 /
      data fk( 231), fk( 232) /  1.4158762831E+00 , -7.1675379285E-01 /
      data fk( 233), fk( 234) / -7.1675379285E-01 ,  3.0584618341E-01 /
      data fk( 235), fk( 236) / -1.3546575196E-01 , -1.3546575196E-01 /
      data fk( 237), fk( 238) / -7.1675379285E-01 ,  3.0584618341E-01 /
      data fk( 239), fk( 240) / -7.1675379285E-01 , -1.3546575196E-01 /
      data fk( 241), fk( 242) / -1.3546575196E-01 ,  3.0584618341E-01 /
      data fk( 243), fk( 244) / -7.1675379285E-01 , -7.1675379285E-01 /
      data fk( 245), fk( 246) / -1.3546575196E-01 , -1.3546575196E-01 /
      data fk( 247), fk( 248) / -7.2159192197E+00 , -7.2159192197E+00 /
      data fk( 249), fk( 250) / -1.3834970168E+00 , -1.3834970168E+00 /
      data fk( 251), fk( 252) / -4.3933144153E-01 , -7.2159192197E+00 /
      data fk( 253), fk( 254) / -7.2159192197E+00 , -1.3834970168E+00 /
      data fk( 255), fk( 256) / -4.3933144153E-01 , -1.3834970168E+00 /
      data fk( 257), fk( 258) / -7.2159192197E+00 , -7.2159192197E+00 /
      data fk( 259), fk( 260) / -4.3933144153E-01 , -1.3834970168E+00 /
      data fk( 261), fk( 262) / -1.3834970168E+00 ,  1.0464998980E+00 /
      data fk( 263), fk( 264) /  1.0464998980E+00 ,  6.7288795130E-01 /
      data fk( 265), fk( 266) /  1.8367752638E-01 ,  1.8367752638E-01 /
      data fk( 267), fk( 268) /  1.0464998980E+00 ,  6.7288795130E-01 /
      data fk( 269), fk( 270) /  1.0464998980E+00 ,  1.8367752638E-01 /
      data fk( 271), fk( 272) /  1.8367752638E-01 ,  6.7288795130E-01 /
      data fk( 273), fk( 274) /  1.0464998980E+00 ,  1.0464998980E+00 /
      data fk( 275), fk( 276) /  1.8367752638E-01 ,  1.8367752638E-01 /
      data fk( 277), fk( 278) / -1.2654696365E+01 ,  5.8539698547E+00 /
      data fk( 279), fk( 280) /  3.5513253656E+00 ,  2.7722476583E+00 /
      data fk( 281), fk( 282) /  3.5513253656E+00 ,  5.8539698547E+00 /
      data fk( 283), fk( 284) /  2.7722476583E+00 , -1.3582165755E+00 /
      data fk( 285), fk( 286) /  2.2869353164E-01 ,  2.2869353164E-01 /
      data fk( 287), fk( 288) / -1.2654696365E+01 ,  5.8539698547E+00 /
      data fk( 289), fk( 290) /  2.7722476583E+00 ,  3.5513253656E+00 /
      data fk( 291), fk( 292) /  3.5513253656E+00 ,  2.7722476583E+00 /
      data fk( 293), fk( 294) /  5.8539698547E+00 ,  2.2869353164E-01 /
      data fk( 295), fk( 296) / -1.3582165755E+00 ,  2.2869353164E-01 /
      data fk( 297), fk( 298) / -1.2654696365E+01 ,  2.7722476583E+00 /
      data fk( 299), fk( 300) /  5.8539698547E+00 ,  3.5513253656E+00 /
      data fk( 301), fk( 302) /  2.7722476583E+00 ,  3.5513253656E+00 /
      data fk( 303), fk( 304) /  5.8539698547E+00 ,  2.2869353164E-01 /
      data fk( 305), fk( 306) /  2.2869353164E-01 , -1.3582165755E+00 /
      data fk( 307), fk( 308) /  2.8533515190E+00 , -7.9007890486E-02 /
      data fk( 309), fk( 310) / -6.7444526143E-02 ,  9.0729819485E-02 /
      data fk( 311), fk( 312) / -7.9007890486E-02 ,  9.0729819485E-02 /
      data fk( 313), fk( 314) / -6.7444526143E-02 , -5.5114568650E-01 /
      data fk( 315), fk( 316) / -5.5114568650E-01 ,  3.0089645175E-01 /
      data fk( 317), fk( 318) / -7.9007890486E-02 ,  2.8533515190E+00 /
      data fk( 319), fk( 320) / -6.7444526143E-02 ,  9.0729819485E-02 /
      data fk( 321), fk( 322) / -7.9007890486E-02 , -5.5114568650E-01 /
      data fk( 323), fk( 324) / -5.5114568650E-01 ,  9.0729819485E-02 /
      data fk( 325), fk( 326) / -6.7444526143E-02 ,  3.0089645175E-01 /
      data fk( 327), fk( 328) / -7.9007890486E-02 , -7.9007890486E-02 /
      data fk( 329), fk( 330) / -5.5114568650E-01 , -5.5114568650E-01 /
      data fk( 331), fk( 332) /  2.8533515190E+00 , -6.7444526143E-02 /
      data fk( 333), fk( 334) /  9.0729819485E-02 ,  9.0729819485E-02 /
      data fk( 335), fk( 336) / -6.7444526143E-02 ,  3.0089645175E-01 /
      data fk( 337), fk( 338) / -1.9818127098E+01 , -4.8445112901E+00 /
      data fk( 339), fk( 340) / -2.3762611625E+00 , -1.5054590547E+00 /
      data fk( 341), fk( 342) / -2.3762611625E+00 , -4.8445112901E+00 /
      data fk( 343), fk( 344) / -1.5054590547E+00 , -8.5607541596E-01 /
      data fk( 345), fk( 346) / -2.4451284357E-01 , -2.4451284357E-01 /
      data fk( 347), fk( 348) / -1.9818127098E+01 , -4.8445112901E+00 /
      data fk( 349), fk( 350) / -1.5054590547E+00 , -2.3762611625E+00 /
      data fk( 351), fk( 352) / -2.3762611625E+00 , -1.5054590547E+00 /
      data fk( 353), fk( 354) / -4.8445112901E+00 , -2.4451284357E-01 /
      data fk( 355), fk( 356) / -8.5607541596E-01 , -2.4451284357E-01 /
      data fk( 357), fk( 358) / -1.9818127098E+01 , -1.5054590547E+00 /
      data fk( 359), fk( 360) / -4.8445112901E+00 , -2.3762611625E+00 /
      data fk( 361), fk( 362) / -1.5054590547E+00 , -2.3762611625E+00 /
      data fk( 363), fk( 364) / -4.8445112901E+00 , -2.4451284357E-01 /
      data fk( 365), fk( 366) / -2.4451284357E-01 , -8.5607541596E-01 /
      data fk( 367), fk( 368) /  9.0108790681E+00 ,  5.5768085765E+00 /
      data fk( 369), fk( 370) /  9.3800118493E-01 ,  2.1953195800E-01 /
      data fk( 371), fk( 372) /  5.5768085765E+00 ,  2.1953195800E-01 /
      data fk( 373), fk( 374) /  9.3800118493E-01 , -2.9490815832E-01 /
      data fk( 375), fk( 376) / -2.9490815832E-01 , -1.1806749488E-01 /
      data fk( 377), fk( 378) /  5.5768085765E+00 ,  9.0108790681E+00 /
      data fk( 379), fk( 380) /  9.3800118493E-01 ,  2.1953195800E-01 /
      data fk( 381), fk( 382) /  5.5768085765E+00 , -2.9490815832E-01 /
      data fk( 383), fk( 384) / -2.9490815832E-01 ,  2.1953195800E-01 /
      data fk( 385), fk( 386) /  9.3800118493E-01 , -1.1806749488E-01 /
      data fk( 387), fk( 388) /  5.5768085765E+00 ,  5.5768085765E+00 /
      data fk( 389), fk( 390) / -2.9490815832E-01 , -2.9490815832E-01 /
      data fk( 391), fk( 392) /  9.0108790681E+00 ,  9.3800118493E-01 /
      data fk( 393), fk( 394) /  2.1953195800E-01 ,  2.1953195800E-01 /
      data fk( 395), fk( 396) /  9.3800118493E-01 , -1.1806749488E-01 /
      data fk( 397), fk( 398) /  1.0862065055E+01 ,  1.0862065055E+01 /
      data fk( 399), fk( 400) /  1.1003955405E+01 , -3.7309780038E+00 /
      data fk( 401), fk( 402) / -2.0414096564E+00 , -1.4493864166E+00 /
      data fk( 403), fk( 404) / -3.7309780038E+00 , -1.4493864166E+00 /
      data fk( 405), fk( 406) / -2.0414096564E+00 , -1.0984595411E-01 /
      data fk( 407), fk( 408) /  1.0862065055E+01 ,  1.1003955405E+01 /
      data fk( 409), fk( 410) /  1.0862065055E+01 , -2.0414096564E+00 /
      data fk( 411), fk( 412) / -3.7309780038E+00 , -1.4493864166E+00 /
      data fk( 413), fk( 414) / -1.4493864166E+00 , -3.7309780038E+00 /
      data fk( 415), fk( 416) / -2.0414096564E+00 , -1.0984595411E-01 /
      data fk( 417), fk( 418) /  1.1003955405E+01 ,  1.0862065055E+01 /
      data fk( 419), fk( 420) /  1.0862065055E+01 , -2.0414096564E+00 /
      data fk( 421), fk( 422) / -1.4493864166E+00 , -3.7309780038E+00 /
      data fk( 423), fk( 424) / -1.4493864166E+00 , -2.0414096564E+00 /
      data fk( 425), fk( 426) / -3.7309780038E+00 , -1.0984595411E-01 /
      data fk( 427), fk( 428) / -2.4908776057E+00 ,  6.7685304200E-01 /
      data fk( 429), fk( 430) /  6.7685304200E-01 ,  1.9866755136E+00 /
      data fk( 431), fk( 432) /  6.9784659497E-01 , -2.3210590003E-01 /
      data fk( 433), fk( 434) /  6.9784659497E-01 ,  1.9866755136E+00 /
      data fk( 435), fk( 436) / -2.3210590003E-01 ,  1.9961331703E-01 /
      data fk( 437), fk( 438) / -2.4908776057E+00 ,  1.9866755136E+00 /
      data fk( 439), fk( 440) /  6.9784659497E-01 ,  6.7685304200E-01 /
      data fk( 441), fk( 442) /  6.7685304200E-01 , -2.3210590003E-01 /
      data fk( 443), fk( 444) /  6.9784659497E-01 ,  1.9866755136E+00 /
      data fk( 445), fk( 446) /  1.9961331703E-01 , -2.3210590003E-01 /
      data fk( 447), fk( 448) / -2.4908776057E+00 ,  1.9866755136E+00 /
      data fk( 449), fk( 450) /  6.9784659497E-01 ,  6.9784659497E-01 /
      data fk( 451), fk( 452) /  1.9866755136E+00 ,  1.9961331703E-01 /
      data fk( 453), fk( 454) /  6.7685304200E-01 ,  6.7685304200E-01 /
      data fk( 455), fk( 456) / -2.3210590003E-01 , -2.3210590003E-01 /
      data fk( 457), fk( 458) / -5.4605155877E+00 , -5.4605155877E+00 /
      data fk( 459), fk( 460) / -5.4605155877E+00 ,  2.7641079460E-01 /
      data fk( 461), fk( 462) /  2.7641079460E-01 ,  2.7641079460E-01 /
      data fk( 463), fk( 464) /  3.0843187719E-01 ,  3.0843187719E-01 /
      data fk( 465), fk( 466) /  3.0843187719E-01 ,  7.1021243814E-02 /
      data fk( 467), fk( 468) /  7.1021243814E-02 ,  7.1021243814E-02 /
      data fk( 469), fk( 470) /  2.8325229268E+00 ,  2.8325229268E+00 /
      data fk( 471), fk( 472) / -6.2962084095E-01 , -6.2962084095E-01 /
      data fk( 473), fk( 474) / -6.2846244409E-01 ,  2.8325229268E+00 /
      data fk( 475), fk( 476) /  2.8325229268E+00 , -6.2962084095E-01 /
      data fk( 477), fk( 478) / -6.2846244409E-01 , -6.2962084095E-01 /
      data fk( 479), fk( 480) /  2.8325229268E+00 ,  2.8325229268E+00 /
      data fk( 481), fk( 482) / -6.2846244409E-01 , -6.2962084095E-01 /
      data fk( 483), fk( 484) / -6.2962084095E-01 , -1.9017361177E-01 /
      data fk( 485), fk( 486) / -1.9017361177E-01 ,  1.4886885300E-01 /
      data fk( 487), fk( 488) / -7.0553565863E-02 , -7.0553565863E-02 /
      data fk( 489), fk( 490) / -1.9017361177E-01 ,  1.4886885300E-01 /
      data fk( 491), fk( 492) / -1.9017361177E-01 , -7.0553565863E-02 /
      data fk( 493), fk( 494) / -7.0553565863E-02 ,  1.4886885300E-01 /
      data fk( 495), fk( 496) / -1.9017361177E-01 , -1.9017361177E-01 /
      data fk( 497), fk( 498) / -7.0553565863E-02 , -7.0553565863E-02 /
      data fk( 499), fk( 500) /  7.6609884462E+00 ,  7.6609884462E+00 /
      data fk( 501), fk( 502) /  6.2910771805E-01 ,  6.2910771805E-01 /
      data fk( 503), fk( 504) /  9.5105427237E-01 ,  7.6609884462E+00 /
      data fk( 505), fk( 506) /  7.6609884462E+00 ,  6.2910771805E-01 /
      data fk( 507), fk( 508) /  9.5105427237E-01 ,  6.2910771805E-01 /
      data fk( 509), fk( 510) /  7.6609884462E+00 ,  7.6609884462E+00 /
      data fk( 511), fk( 512) /  9.5105427237E-01 ,  6.2910771805E-01 /
      data fk( 513), fk( 514) /  6.2910771805E-01 ,  2.7420172589E-01 /
      data fk( 515), fk( 516) /  2.7420172589E-01 ,  3.8644973894E-01 /
      data fk( 517), fk( 518) /  3.3361403655E-02 ,  3.3361403655E-02 /
      data fk( 519), fk( 520) /  2.7420172589E-01 ,  3.8644973894E-01 /
      data fk( 521), fk( 522) /  2.7420172589E-01 ,  3.3361403655E-02 /
      data fk( 523), fk( 524) /  3.3361403655E-02 ,  3.8644973894E-01 /
      data fk( 525), fk( 526) /  2.7420172589E-01 ,  2.7420172589E-01 /
      data fk( 527), fk( 528) /  3.3361403655E-02 ,  3.3361403655E-02 /
      data fk( 529), fk( 530) /  9.8971279452E+00 ,  9.8971279452E+00 /
      data fk( 531), fk( 532) / -4.0805281006E-01 , -4.0805281006E-01 /
      data fk( 533), fk( 534) / -1.6062079917E-01 ,  9.8971279452E+00 /
      data fk( 535), fk( 536) / -4.0805281006E-01 , -1.6062079917E-01 /
      data fk( 537), fk( 538) / -4.0805281006E-01 , -1.6062079917E-01 /
      data fk( 539), fk( 540) / -4.0805281006E-01 , -4.0805281006E-01 /
      data fk( 541), fk( 542) /  1.2757980781E-01 ,  1.2757980781E-01 /
      data fk( 543), fk( 544) /  1.2757980781E-01 ,  2.4697045551E-01 /
      data fk( 545), fk( 546) /  1.1363313167E+00 , -1.0832656186E-02 /
      data fk( 547), fk( 548) / -1.0832656186E-02 , -1.0832656186E-02 /
      data fk( 549), fk( 550) /  1.1363313167E+00 , -1.0832656186E-02 /
      data fk( 551), fk( 552) /  6.4365301222E+00 ,  2.7197974544E+00 /
      data fk( 553), fk( 554) /  2.7197974544E+00 , -1.0832656186E-02 /
      data fk( 555), fk( 556) / -1.0832656186E-02 ,  1.1363313167E+00 /
      data fk( 557), fk( 558) /  2.7197974544E+00 ,  6.4365301222E+00 /
      data fk( 559), fk( 560) /  2.7197974544E+00 ,  2.7197974544E+00 /
      data fk( 561), fk( 562) /  2.7197974544E+00 ,  6.4365301222E+00 /
      data fk( 563), fk( 564) /  4.3929521244E+01 ,  1.4302816142E+01 /
      data fk( 565), fk( 566) / -4.0355372475E+00 , -3.3549030833E+00 /
      data fk( 567), fk( 568) / -4.0020002620E+00 , -3.3549030833E+00 /
      data fk( 569), fk( 570) / -4.0355372475E+00 , -4.0020002620E+00 /
      data fk( 571), fk( 572) /  9.0950954595E-01 ,  3.7170746452E-02 /
      data fk( 573), fk( 574) /  3.7170746452E-02 ,  1.4302816142E+01 /
      data fk( 575), fk( 576) / -4.0355372475E+00 , -4.0020002620E+00 /
      data fk( 577), fk( 578) / -3.3549030833E+00 , -3.3549030833E+00 /
      data fk( 579), fk( 580) / -4.0020002620E+00 , -4.0355372475E+00 /
      data fk( 581), fk( 582) /  3.7170746452E-02 ,  9.0950954595E-01 /
      data fk( 583), fk( 584) /  3.7170746452E-02 ,  1.4302816142E+01 /
      data fk( 585), fk( 586) / -4.0020002620E+00 , -4.0355372475E+00 /
      data fk( 587), fk( 588) / -3.3549030833E+00 , -4.0020002620E+00 /
      data fk( 589), fk( 590) / -3.3549030833E+00 , -4.0355372475E+00 /
      data fk( 591), fk( 592) /  3.7170746452E-02 ,  3.7170746452E-02 /
      data fk( 593), fk( 594) /  9.0950954595E-01 ,  9.4961217139E-01 /
      data fk( 595), fk( 596) / -2.2360779542E-01 ,  9.5175687795E-02 /
      data fk( 597), fk( 598) /  8.4142763965E-02 , -2.2360779542E-01 /
      data fk( 599), fk( 600) /  8.4142763965E-02 ,  9.5175687795E-02 /
      data fk( 601), fk( 602) / -2.4383173490E-01 , -2.4383173490E-01 /
      data fk( 603), fk( 604) /  1.7116614495E-01 , -2.2360779542E-01 /
      data fk( 605), fk( 606) /  9.4961217139E-01 ,  9.5175687795E-02 /
      data fk( 607), fk( 608) /  8.4142763965E-02 , -2.2360779542E-01 /
      data fk( 609), fk( 610) / -2.4383173490E-01 , -2.4383173490E-01 /
      data fk( 611), fk( 612) /  8.4142763965E-02 ,  9.5175687795E-02 /
      data fk( 613), fk( 614) /  1.7116614495E-01 , -2.2360779542E-01 /
      data fk( 615), fk( 616) / -2.2360779542E-01 , -2.4383173490E-01 /
      data fk( 617), fk( 618) / -2.4383173490E-01 ,  9.4961217139E-01 /
      data fk( 619), fk( 620) /  9.5175687795E-02 ,  8.4142763965E-02 /
      data fk( 621), fk( 622) /  8.4142763965E-02 ,  9.5175687795E-02 /
      data fk( 623), fk( 624) /  1.7116614495E-01 ,  2.7480375630E+01 /
      data fk( 625), fk( 626) / -1.0030234861E+01 , -6.6453932643E+00 /
      data fk( 627), fk( 628) / -7.3393390047E+00 ,  2.7480375630E+01 /
      data fk( 629), fk( 630) / -6.6453932643E+00 , -1.0030234861E+01 /
      data fk( 631), fk( 632) / -7.3393390047E+00 ,  2.9526899804E+00 /
      data fk( 633), fk( 634) /  1.6494675786E+00 , -8.1564475318E-01 /
      data fk( 635), fk( 636) /  5.5333984555E-01 ,  1.6494675786E+00 /
      data fk( 637), fk( 638) /  2.9526899804E+00 , -8.1564475318E-01 /
      data fk( 639), fk( 640) /  5.5333984555E-01 ,  1.9851698602E+00 /
      data fk( 641), fk( 642) /  1.9851698602E+00 , -6.9192703081E-01 /
      data fk( 643), fk( 644) / -6.9192703081E-01 ,  2.7480375630E+01 /
      data fk( 645), fk( 646) / -1.0030234861E+01 , -7.3393390047E+00 /
      data fk( 647), fk( 648) / -6.6453932643E+00 ,  2.7480375630E+01 /
      data fk( 649), fk( 650) / -6.6453932643E+00 , -7.3393390047E+00 /
      data fk( 651), fk( 652) / -1.0030234861E+01 ,  2.9526899804E+00 /
      data fk( 653), fk( 654) /  1.6494675786E+00 ,  5.5333984555E-01 /
      data fk( 655), fk( 656) / -8.1564475318E-01 ,  1.9851698602E+00 /
      data fk( 657), fk( 658) /  1.9851698602E+00 , -6.9192703081E-01 /
      data fk( 659), fk( 660) / -6.9192703081E-01 ,  1.6494675786E+00 /
      data fk( 661), fk( 662) /  2.9526899804E+00 , -8.1564475318E-01 /
      data fk( 663), fk( 664) /  5.5333984555E-01 ,  2.7480375630E+01 /
      data fk( 665), fk( 666) / -7.3393390047E+00 , -1.0030234861E+01 /
      data fk( 667), fk( 668) / -6.6453932643E+00 ,  2.7480375630E+01 /
      data fk( 669), fk( 670) / -7.3393390047E+00 , -6.6453932643E+00 /
      data fk( 671), fk( 672) / -1.0030234861E+01 ,  1.9851698602E+00 /
      data fk( 673), fk( 674) /  1.9851698602E+00 , -6.9192703081E-01 /
      data fk( 675), fk( 676) / -6.9192703081E-01 ,  2.9526899804E+00 /
      data fk( 677), fk( 678) /  1.6494675786E+00 ,  5.5333984555E-01 /
      data fk( 679), fk( 680) / -8.1564475318E-01 ,  1.6494675786E+00 /
      data fk( 681), fk( 682) /  2.9526899804E+00 ,  5.5333984555E-01 /
      data fk( 683), fk( 684) / -8.1564475318E-01 , -2.0027019128E+00 /
      data fk( 685), fk( 686) / -3.6081153936E-01 ,  2.4673557854E-01 /
      data fk( 687), fk( 688) / -2.1452438330E-01 , -2.0027019128E+00 /
      data fk( 689), fk( 690) / -3.6081153936E-01 , -2.1452438330E-01 /
      data fk( 691), fk( 692) /  2.4673557854E-01 , -1.3652345689E+00 /
      data fk( 693), fk( 694) / -1.3652345689E+00 , -1.7771224602E-02 /
      data fk( 695), fk( 696) / -1.7771224602E-02 , -4.2092904661E-01 /
      data fk( 697), fk( 698) /  7.1867504355E-02 , -9.1449937369E-02 /
      data fk( 699), fk( 700) / -1.3623838463E-01 ,  7.1867504355E-02 /
      data fk( 701), fk( 702) / -4.2092904661E-01 , -9.1449937369E-02 /
      data fk( 703), fk( 704) / -1.3623838463E-01 , -3.6081153936E-01 /
      data fk( 705), fk( 706) / -2.0027019128E+00 ,  2.4673557854E-01 /
      data fk( 707), fk( 708) / -2.1452438330E-01 , -1.3652345689E+00 /
      data fk( 709), fk( 710) / -1.3652345689E+00 , -1.7771224602E-02 /
      data fk( 711), fk( 712) / -1.7771224602E-02 , -2.0027019128E+00 /
      data fk( 713), fk( 714) / -3.6081153936E-01 , -2.1452438330E-01 /
      data fk( 715), fk( 716) /  2.4673557854E-01 , -4.2092904661E-01 /
      data fk( 717), fk( 718) / -9.1449937369E-02 ,  7.1867504355E-02 /
      data fk( 719), fk( 720) / -1.3623838463E-01 ,  7.1867504355E-02 /
      data fk( 721), fk( 722) / -9.1449937369E-02 , -4.2092904661E-01 /
      data fk( 723), fk( 724) / -1.3623838463E-01 , -1.3652345689E+00 /
      data fk( 725), fk( 726) / -1.3652345689E+00 , -1.7771224602E-02 /
      data fk( 727), fk( 728) / -1.7771224602E-02 , -3.6081153936E-01 /
      data fk( 729), fk( 730) / -2.0027019128E+00 ,  2.4673557854E-01 /
      data fk( 731), fk( 732) / -2.1452438330E-01 , -3.6081153936E-01 /
      data fk( 733), fk( 734) / -2.0027019128E+00 , -2.1452438330E-01 /
      data fk( 735), fk( 736) /  2.4673557854E-01 , -9.1449937369E-02 /
      data fk( 737), fk( 738) / -4.2092904661E-01 ,  7.1867504355E-02 /
      data fk( 739), fk( 740) / -1.3623838463E-01 , -9.1449937369E-02 /
      data fk( 741), fk( 742) /  7.1867504355E-02 , -4.2092904661E-01 /
      data fk( 743), fk( 744) / -1.3623838463E-01 , -1.1659398973E+01 /
      data fk( 745), fk( 746) / -1.1659398973E+01 , -1.2259359420E+01 /
      data fk( 747), fk( 748) /  2.3219900456E+00 ,  2.6520240365E+00 /
      data fk( 749), fk( 750) /  1.7916613593E+00 ,  2.3219900456E+00 /
      data fk( 751), fk( 752) /  1.7916613593E+00 ,  2.6520240365E+00 /
      data fk( 753), fk( 754) /  1.2120123094E+00 , -1.1659398973E+01 /
      data fk( 755), fk( 756) / -1.2259359420E+01 , -1.1659398973E+01 /
      data fk( 757), fk( 758) /  2.6520240365E+00 ,  2.3219900456E+00 /
      data fk( 759), fk( 760) /  1.7916613593E+00 ,  1.7916613593E+00 /
      data fk( 761), fk( 762) /  2.3219900456E+00 ,  2.6520240365E+00 /
      data fk( 763), fk( 764) /  1.2120123094E+00 , -1.2259359420E+01 /
      data fk( 765), fk( 766) / -1.1659398973E+01 , -1.1659398973E+01 /
      data fk( 767), fk( 768) /  2.6520240365E+00 ,  1.7916613593E+00 /
      data fk( 769), fk( 770) /  2.3219900456E+00 ,  1.7916613593E+00 /
      data fk( 771), fk( 772) /  2.6520240365E+00 ,  2.3219900456E+00 /
      data fk( 773), fk( 774) /  1.2120123094E+00 , -6.3709905237E-02 /
      data fk( 775), fk( 776) / -1.5054691049E-02 , -1.5054691049E-02 /
      data fk( 777), fk( 778) /  1.1302823247E+00 ,  2.0094974799E-01 /
      data fk( 779), fk( 780) / -2.5181324106E-01 ,  2.0094974799E-01 /
      data fk( 781), fk( 782) /  1.1302823247E+00 , -2.5181324106E-01 /
      data fk( 783), fk( 784) /  2.7247962813E-01 , -6.3709905237E-02 /
      data fk( 785), fk( 786) /  1.1302823247E+00 ,  2.0094974799E-01 /
      data fk( 787), fk( 788) / -1.5054691049E-02 , -1.5054691049E-02 /
      data fk( 789), fk( 790) / -2.5181324106E-01 ,  2.0094974799E-01 /
      data fk( 791), fk( 792) /  1.1302823247E+00 ,  2.7247962813E-01 /
      data fk( 793), fk( 794) / -2.5181324106E-01 , -6.3709905237E-02 /
      data fk( 795), fk( 796) /  1.1302823247E+00 ,  2.0094974799E-01 /
      data fk( 797), fk( 798) /  2.0094974799E-01 ,  1.1302823247E+00 /
      data fk( 799), fk( 800) /  2.7247962813E-01 , -1.5054691049E-02 /
      data fk( 801), fk( 802) / -1.5054691049E-02 , -2.5181324106E-01 /
      data fk( 803), fk( 804) / -2.5181324106E-01 , -1.7136922446E+01 /
      data fk( 805), fk( 806) / -1.8757036212E+01 , -1.8757036212E+01 /
      data fk( 807), fk( 808) /  3.7745076477E+00 ,  3.7745076477E+00 /
      data fk( 809), fk( 810) /  3.0977529174E+00 , -1.8757036212E+01 /
      data fk( 811), fk( 812) / -1.7136922446E+01 , -1.8757036212E+01 /
      data fk( 813), fk( 814) /  3.7745076477E+00 ,  3.0977529174E+00 /
      data fk( 815), fk( 816) /  3.7745076477E+00 ,  3.4344788295E+00 /
      data fk( 817), fk( 818) / -2.1602276421E-01 , -1.2672297362E+00 /
      data fk( 819), fk( 820) / -1.5762385225E+00 ,  4.1326489278E-01 /
      data fk( 821), fk( 822) / -1.0425501968E+00 ,  3.4344788295E+00 /
      data fk( 823), fk( 824) / -1.5762385225E+00 ,  4.1326489278E-01 /
      data fk( 825), fk( 826) / -2.1602276421E-01 , -1.2672297362E+00 /
      data fk( 827), fk( 828) / -1.0425501968E+00 ,  6.5321342988E+00 /
      data fk( 829), fk( 830) / -1.9173659305E-01 , -4.2333345677E-01 /
      data fk( 831), fk( 832) / -4.2333345677E-01 , -1.9173659305E-01 /
      data fk( 833), fk( 834) /  4.7173271937E-01 , -1.8757036212E+01 /
      data fk( 835), fk( 836) / -1.8757036212E+01 , -1.7136922446E+01 /
      data fk( 837), fk( 838) /  3.0977529174E+00 ,  3.7745076477E+00 /
      data fk( 839), fk( 840) /  3.7745076477E+00 ,  3.4344788295E+00 /
      data fk( 841), fk( 842) / -1.2672297362E+00 , -2.1602276421E-01 /
      data fk( 843), fk( 844) /  4.1326489278E-01 , -1.5762385225E+00 /
      data fk( 845), fk( 846) / -1.0425501968E+00 ,  6.5321342988E+00 /
      data fk( 847), fk( 848) / -1.9173659305E-01 , -4.2333345677E-01 /
      data fk( 849), fk( 850) / -4.2333345677E-01 , -1.9173659305E-01 /
      data fk( 851), fk( 852) /  4.7173271937E-01 ,  3.4344788295E+00 /
      data fk( 853), fk( 854) / -1.5762385225E+00 ,  4.1326489278E-01 /
      data fk( 855), fk( 856) / -2.1602276421E-01 , -1.2672297362E+00 /
      data fk( 857), fk( 858) / -1.0425501968E+00 ,  6.5321342988E+00 /
      data fk( 859), fk( 860) / -1.9173659305E-01 , -4.2333345677E-01 /
      data fk( 861), fk( 862) / -4.2333345677E-01 , -1.9173659305E-01 /
      data fk( 863), fk( 864) /  4.7173271937E-01 ,  3.4344788295E+00 /
      data fk( 865), fk( 866) / -1.2672297362E+00 , -2.1602276421E-01 /
      data fk( 867), fk( 868) /  4.1326489278E-01 , -1.5762385225E+00 /
      data fk( 869), fk( 870) / -1.0425501968E+00 ,  3.4344788295E+00 /
      data fk( 871), fk( 872) /  4.1326489278E-01 , -1.5762385225E+00 /
      data fk( 873), fk( 874) / -1.2672297362E+00 , -2.1602276421E-01 /
      data fk( 875), fk( 876) / -1.0425501968E+00 , -4.7313044689E-02 /
      data fk( 877), fk( 878) / -4.7313044689E-02 ,  1.0293214368E+00 /
      data fk( 879), fk( 880) / -1.8200706080E-01 , -1.6032054922E-01 /
      data fk( 881), fk( 882) / -1.6032054922E-01 , -4.7313044689E-02 /
      data fk( 883), fk( 884) / -1.8200706080E-01 , -1.6032054922E-01 /
      data fk( 885), fk( 886) / -4.7313044689E-02 ,  1.0293214368E+00 /
      data fk( 887), fk( 888) / -1.6032054922E-01 , -1.8200706080E-01 /
      data fk( 889), fk( 890) / -4.7313044689E-02 , -1.6032054922E-01 /
      data fk( 891), fk( 892) / -4.7313044689E-02 , -1.6032054922E-01 /
      data fk( 893), fk( 894) /  1.0293214368E+00 ,  4.9811824796E+00 /
      data fk( 895), fk( 896) /  3.9652060655E+00 ,  3.9652060655E+00 /
      data fk( 897), fk( 898) /  3.0846944087E-02 ,  3.0846944087E-02 /
      data fk( 899), fk( 900) /  3.9652060655E+00 ,  4.9811824796E+00 /
      data fk( 901), fk( 902) /  3.9652060655E+00 ,  3.0846944087E-02 /
      data fk( 903), fk( 904) /  3.0846944087E-02 ,  3.9652060655E+00 /
      data fk( 905), fk( 906) /  3.9652060655E+00 ,  4.9811824796E+00 /
      data fk( 907), fk( 908) /  3.0846944087E-02 ,  3.0846944087E-02 /
      data fk( 909), fk( 910) / -1.7196386858E+00 , -1.7196386858E+00 /
      data fk( 911), fk( 912) /  1.8614300189E+00 , -6.1562496217E-01 /
      data fk( 913), fk( 914) / -6.1562496217E-01 , -1.7196386858E+00 /
      data fk( 915), fk( 916) / -1.7196386858E+00 , -6.1562496217E-01 /
      data fk( 917), fk( 918) /  1.8614300189E+00 , -6.1562496217E-01 /
      data fk( 919), fk( 920) / -1.7196386858E+00 , -1.7196386858E+00 /
      data fk( 921), fk( 922) / -6.1562496217E-01 , -6.1562496217E-01 /
      data fk( 923), fk( 924) /  1.8614300189E+00 ,  1.0078338217E+00 /

      data shift  / 2.04d0, 2.04d0, 2.04d0, 3.53d0, 3.53d0, 3.53d0 /
      data alpham / 6*0.5d0 /
      data nmax,nvar /6,6/
   
!
      en=0.0d0
      do i=1,nvar
          fmr(i,1) = 1.0d0- exp(-alpham(i)*(rij(i)-shift(i)))
          do j=2,nmax
              fmr(i,j) = fmr(i,1)**j
          enddo
      enddo
!
      en=en+fk(1) 
      en=en+fk(2)*fmr(1,1) 
      en=en+fk(3)*fmr(2,1) 
      en=en+fk(4)*fmr(3,1) 
      en=en+fk(5)*fmr(4,1) 
      en=en+fk(6)*fmr(5,1) 
      en=en+fk(7)*fmr(6,1) 
      en=en+fk(8)*fmr(1,2) 
      en=en+fk(9)*fmr(2,2) 
      en=en+fk(10)*fmr(3,2) 
      en=en+fk(11)*fmr(4,2) 
      en=en+fk(12)*fmr(5,2) 
      en=en+fk(13)*fmr(6,2) 
      en=en+fk(14)*fmr(1,1)*fmr(2,1) 
      en=en+fk(15)*fmr(1,1)*fmr(3,1) 
      en=en+fk(16)*fmr(1,1)*fmr(4,1) 
      en=en+fk(17)*fmr(1,1)*fmr(5,1) 
      en=en+fk(18)*fmr(1,1)*fmr(6,1) 
      en=en+fk(19)*fmr(2,1)*fmr(3,1) 
      en=en+fk(20)*fmr(2,1)*fmr(4,1) 
      en=en+fk(21)*fmr(2,1)*fmr(5,1) 
      en=en+fk(22)*fmr(2,1)*fmr(6,1) 
      en=en+fk(23)*fmr(3,1)*fmr(4,1) 
      en=en+fk(24)*fmr(3,1)*fmr(5,1) 
      en=en+fk(25)*fmr(3,1)*fmr(6,1) 
      en=en+fk(26)*fmr(4,1)*fmr(5,1) 
      en=en+fk(27)*fmr(4,1)*fmr(6,1) 
      en=en+fk(28)*fmr(5,1)*fmr(6,1) 
      en=en+fk(29)*fmr(1,3) 
      en=en+fk(30)*fmr(2,3) 
      en=en+fk(31)*fmr(3,3) 
      en=en+fk(32)*fmr(4,3) 
      en=en+fk(33)*fmr(5,3) 
      en=en+fk(34)*fmr(6,3) 
      en=en+fk(35)*fmr(1,2)*fmr(2,1) 
      en=en+fk(36)*fmr(1,2)*fmr(3,1) 
      en=en+fk(37)*fmr(1,2)*fmr(4,1) 
      en=en+fk(38)*fmr(1,2)*fmr(5,1) 
      en=en+fk(39)*fmr(1,2)*fmr(6,1) 
      en=en+fk(40)*fmr(1,1)*fmr(2,2) 
      en=en+fk(41)*fmr(2,2)*fmr(3,1) 
      en=en+fk(42)*fmr(2,2)*fmr(4,1) 
      en=en+fk(43)*fmr(2,2)*fmr(5,1) 
      en=en+fk(44)*fmr(2,2)*fmr(6,1) 
      en=en+fk(45)*fmr(1,1)*fmr(3,2) 
      en=en+fk(46)*fmr(2,1)*fmr(3,2) 
      en=en+fk(47)*fmr(3,2)*fmr(4,1) 
      en=en+fk(48)*fmr(3,2)*fmr(5,1) 
      en=en+fk(49)*fmr(3,2)*fmr(6,1) 
      en=en+fk(50)*fmr(1,1)*fmr(4,2) 
      en=en+fk(51)*fmr(2,1)*fmr(4,2) 
      en=en+fk(52)*fmr(3,1)*fmr(4,2) 
      en=en+fk(53)*fmr(4,2)*fmr(5,1) 
      en=en+fk(54)*fmr(4,2)*fmr(6,1) 
      en=en+fk(55)*fmr(1,1)*fmr(5,2) 
      en=en+fk(56)*fmr(2,1)*fmr(5,2) 
      en=en+fk(57)*fmr(3,1)*fmr(5,2) 
      en=en+fk(58)*fmr(4,1)*fmr(5,2) 
      en=en+fk(59)*fmr(5,2)*fmr(6,1) 
      en=en+fk(60)*fmr(1,1)*fmr(6,2) 
      en=en+fk(61)*fmr(2,1)*fmr(6,2) 
      en=en+fk(62)*fmr(3,1)*fmr(6,2) 
      en=en+fk(63)*fmr(4,1)*fmr(6,2) 
      en=en+fk(64)*fmr(5,1)*fmr(6,2) 
      en=en+fk(65)*fmr(1,1)*fmr(2,1)*fmr(3,1) 
      en=en+fk(66)*fmr(1,1)*fmr(2,1)*fmr(4,1) 
      en=en+fk(67)*fmr(1,1)*fmr(2,1)*fmr(5,1) 
      en=en+fk(68)*fmr(1,1)*fmr(2,1)*fmr(6,1) 
      en=en+fk(69)*fmr(1,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(70)*fmr(1,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(71)*fmr(1,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(72)*fmr(1,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(73)*fmr(1,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(74)*fmr(1,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(75)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(76)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(77)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(78)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(79)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(80)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(81)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(82)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(83)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(84)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(85)*fmr(1,4) 
      en=en+fk(86)*fmr(2,4) 
      en=en+fk(87)*fmr(3,4) 
      en=en+fk(88)*fmr(4,4) 
      en=en+fk(89)*fmr(5,4) 
      en=en+fk(90)*fmr(6,4) 
      en=en+fk(91)*fmr(1,3)*fmr(2,1) 
      en=en+fk(92)*fmr(1,3)*fmr(3,1) 
      en=en+fk(93)*fmr(1,3)*fmr(4,1) 
      en=en+fk(94)*fmr(1,3)*fmr(5,1) 
      en=en+fk(95)*fmr(1,3)*fmr(6,1) 
      en=en+fk(96)*fmr(1,1)*fmr(2,3) 
      en=en+fk(97)*fmr(2,3)*fmr(3,1) 
      en=en+fk(98)*fmr(2,3)*fmr(4,1) 
      en=en+fk(99)*fmr(2,3)*fmr(5,1) 
      en=en+fk(100)*fmr(2,3)*fmr(6,1) 
      en=en+fk(101)*fmr(1,1)*fmr(3,3) 
      en=en+fk(102)*fmr(2,1)*fmr(3,3) 
      en=en+fk(103)*fmr(3,3)*fmr(4,1) 
      en=en+fk(104)*fmr(3,3)*fmr(5,1) 
      en=en+fk(105)*fmr(3,3)*fmr(6,1) 
      en=en+fk(106)*fmr(1,1)*fmr(4,3) 
      en=en+fk(107)*fmr(2,1)*fmr(4,3) 
      en=en+fk(108)*fmr(3,1)*fmr(4,3) 
      en=en+fk(109)*fmr(4,3)*fmr(5,1) 
      en=en+fk(110)*fmr(4,3)*fmr(6,1) 
      en=en+fk(111)*fmr(1,1)*fmr(5,3) 
      en=en+fk(112)*fmr(2,1)*fmr(5,3) 
      en=en+fk(113)*fmr(3,1)*fmr(5,3) 
      en=en+fk(114)*fmr(4,1)*fmr(5,3) 
      en=en+fk(115)*fmr(5,3)*fmr(6,1) 
      en=en+fk(116)*fmr(1,1)*fmr(6,3) 
      en=en+fk(117)*fmr(2,1)*fmr(6,3) 
      en=en+fk(118)*fmr(3,1)*fmr(6,3) 
      en=en+fk(119)*fmr(4,1)*fmr(6,3) 
      en=en+fk(120)*fmr(5,1)*fmr(6,3) 
      en=en+fk(121)*fmr(1,2)*fmr(2,2) 
      en=en+fk(122)*fmr(1,2)*fmr(3,2) 
      en=en+fk(123)*fmr(1,2)*fmr(4,2) 
      en=en+fk(124)*fmr(1,2)*fmr(5,2) 
      en=en+fk(125)*fmr(1,2)*fmr(6,2) 
      en=en+fk(126)*fmr(2,2)*fmr(3,2) 
      en=en+fk(127)*fmr(2,2)*fmr(4,2) 
      en=en+fk(128)*fmr(2,2)*fmr(5,2) 
      en=en+fk(129)*fmr(2,2)*fmr(6,2) 
      en=en+fk(130)*fmr(3,2)*fmr(4,2) 
      en=en+fk(131)*fmr(3,2)*fmr(5,2) 
      en=en+fk(132)*fmr(3,2)*fmr(6,2) 
      en=en+fk(133)*fmr(4,2)*fmr(5,2) 
      en=en+fk(134)*fmr(4,2)*fmr(6,2) 
      en=en+fk(135)*fmr(5,2)*fmr(6,2) 
      en=en+fk(136)*fmr(1,2)*fmr(2,1)*fmr(3,1) 
      en=en+fk(137)*fmr(1,2)*fmr(2,1)*fmr(4,1) 
      en=en+fk(138)*fmr(1,2)*fmr(2,1)*fmr(5,1) 
      en=en+fk(139)*fmr(1,2)*fmr(2,1)*fmr(6,1) 
      en=en+fk(140)*fmr(1,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(141)*fmr(1,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(142)*fmr(1,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(143)*fmr(1,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(144)*fmr(1,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(145)*fmr(1,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(146)*fmr(1,1)*fmr(2,2)*fmr(3,1) 
      en=en+fk(147)*fmr(1,1)*fmr(2,2)*fmr(4,1) 
      en=en+fk(148)*fmr(1,1)*fmr(2,2)*fmr(5,1) 
      en=en+fk(149)*fmr(1,1)*fmr(2,2)*fmr(6,1) 
      en=en+fk(150)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(151)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(152)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(153)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(154)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(155)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(156)*fmr(1,1)*fmr(2,1)*fmr(3,2) 
      en=en+fk(157)*fmr(1,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(158)*fmr(1,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(159)*fmr(1,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(160)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(161)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(162)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(163)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(164)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(165)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(166)*fmr(1,1)*fmr(2,1)*fmr(4,2) 
      en=en+fk(167)*fmr(1,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(168)*fmr(1,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(169)*fmr(1,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(170)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(171)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(172)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(173)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(174)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(175)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(176)*fmr(1,1)*fmr(2,1)*fmr(5,2) 
      en=en+fk(177)*fmr(1,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(178)*fmr(1,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(179)*fmr(1,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(180)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(181)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(182)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(183)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(184)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(185)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(186)*fmr(1,1)*fmr(2,1)*fmr(6,2) 
      en=en+fk(187)*fmr(1,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(188)*fmr(1,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(189)*fmr(1,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(190)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(191)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(192)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(193)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(194)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(195)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(196)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(197)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(198)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(199)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(200)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(201)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(202)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(203)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(204)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(205)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(206)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(207)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(208)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(209)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(210)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(211)*fmr(1,5) 
      en=en+fk(212)*fmr(2,5) 
      en=en+fk(213)*fmr(3,5) 
      en=en+fk(214)*fmr(4,5) 
      en=en+fk(215)*fmr(5,5) 
      en=en+fk(216)*fmr(6,5) 
      en=en+fk(217)*fmr(1,4)*fmr(2,1) 
      en=en+fk(218)*fmr(1,4)*fmr(3,1) 
      en=en+fk(219)*fmr(1,4)*fmr(4,1) 
      en=en+fk(220)*fmr(1,4)*fmr(5,1) 
      en=en+fk(221)*fmr(1,4)*fmr(6,1) 
      en=en+fk(222)*fmr(1,1)*fmr(2,4) 
      en=en+fk(223)*fmr(2,4)*fmr(3,1) 
      en=en+fk(224)*fmr(2,4)*fmr(4,1) 
      en=en+fk(225)*fmr(2,4)*fmr(5,1) 
      en=en+fk(226)*fmr(2,4)*fmr(6,1) 
      en=en+fk(227)*fmr(1,1)*fmr(3,4) 
      en=en+fk(228)*fmr(2,1)*fmr(3,4) 
      en=en+fk(229)*fmr(3,4)*fmr(4,1) 
      en=en+fk(230)*fmr(3,4)*fmr(5,1) 
      en=en+fk(231)*fmr(3,4)*fmr(6,1) 
      en=en+fk(232)*fmr(1,1)*fmr(4,4) 
      en=en+fk(233)*fmr(2,1)*fmr(4,4) 
      en=en+fk(234)*fmr(3,1)*fmr(4,4) 
      en=en+fk(235)*fmr(4,4)*fmr(5,1) 
      en=en+fk(236)*fmr(4,4)*fmr(6,1) 
      en=en+fk(237)*fmr(1,1)*fmr(5,4) 
      en=en+fk(238)*fmr(2,1)*fmr(5,4) 
      en=en+fk(239)*fmr(3,1)*fmr(5,4) 
      en=en+fk(240)*fmr(4,1)*fmr(5,4) 
      en=en+fk(241)*fmr(5,4)*fmr(6,1) 
      en=en+fk(242)*fmr(1,1)*fmr(6,4) 
      en=en+fk(243)*fmr(2,1)*fmr(6,4) 
      en=en+fk(244)*fmr(3,1)*fmr(6,4) 
      en=en+fk(245)*fmr(4,1)*fmr(6,4) 
      en=en+fk(246)*fmr(5,1)*fmr(6,4) 
      en=en+fk(247)*fmr(1,3)*fmr(2,2) 
      en=en+fk(248)*fmr(1,3)*fmr(3,2) 
      en=en+fk(249)*fmr(1,3)*fmr(4,2) 
      en=en+fk(250)*fmr(1,3)*fmr(5,2) 
      en=en+fk(251)*fmr(1,3)*fmr(6,2) 
      en=en+fk(252)*fmr(1,2)*fmr(2,3) 
      en=en+fk(253)*fmr(2,3)*fmr(3,2) 
      en=en+fk(254)*fmr(2,3)*fmr(4,2) 
      en=en+fk(255)*fmr(2,3)*fmr(5,2) 
      en=en+fk(256)*fmr(2,3)*fmr(6,2) 
      en=en+fk(257)*fmr(1,2)*fmr(3,3) 
      en=en+fk(258)*fmr(2,2)*fmr(3,3) 
      en=en+fk(259)*fmr(3,3)*fmr(4,2) 
      en=en+fk(260)*fmr(3,3)*fmr(5,2) 
      en=en+fk(261)*fmr(3,3)*fmr(6,2) 
      en=en+fk(262)*fmr(1,2)*fmr(4,3) 
      en=en+fk(263)*fmr(2,2)*fmr(4,3) 
      en=en+fk(264)*fmr(3,2)*fmr(4,3) 
      en=en+fk(265)*fmr(4,3)*fmr(5,2) 
      en=en+fk(266)*fmr(4,3)*fmr(6,2) 
      en=en+fk(267)*fmr(1,2)*fmr(5,3) 
      en=en+fk(268)*fmr(2,2)*fmr(5,3) 
      en=en+fk(269)*fmr(3,2)*fmr(5,3) 
      en=en+fk(270)*fmr(4,2)*fmr(5,3) 
      en=en+fk(271)*fmr(5,3)*fmr(6,2) 
      en=en+fk(272)*fmr(1,2)*fmr(6,3) 
      en=en+fk(273)*fmr(2,2)*fmr(6,3) 
      en=en+fk(274)*fmr(3,2)*fmr(6,3) 
      en=en+fk(275)*fmr(4,2)*fmr(6,3) 
      en=en+fk(276)*fmr(5,2)*fmr(6,3) 
      en=en+fk(277)*fmr(1,3)*fmr(2,1)*fmr(3,1) 
      en=en+fk(278)*fmr(1,3)*fmr(2,1)*fmr(4,1) 
      en=en+fk(279)*fmr(1,3)*fmr(2,1)*fmr(5,1) 
      en=en+fk(280)*fmr(1,3)*fmr(2,1)*fmr(6,1) 
      en=en+fk(281)*fmr(1,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(282)*fmr(1,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(283)*fmr(1,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(284)*fmr(1,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(285)*fmr(1,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(286)*fmr(1,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(287)*fmr(1,1)*fmr(2,3)*fmr(3,1) 
      en=en+fk(288)*fmr(1,1)*fmr(2,3)*fmr(4,1) 
      en=en+fk(289)*fmr(1,1)*fmr(2,3)*fmr(5,1) 
      en=en+fk(290)*fmr(1,1)*fmr(2,3)*fmr(6,1) 
      en=en+fk(291)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(292)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(293)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(294)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(295)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(296)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(297)*fmr(1,1)*fmr(2,1)*fmr(3,3) 
      en=en+fk(298)*fmr(1,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(299)*fmr(1,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(300)*fmr(1,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(301)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(302)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(303)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(304)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(305)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(306)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(307)*fmr(1,1)*fmr(2,1)*fmr(4,3) 
      en=en+fk(308)*fmr(1,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(309)*fmr(1,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(310)*fmr(1,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(311)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(312)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(313)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(314)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(315)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(316)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(317)*fmr(1,1)*fmr(2,1)*fmr(5,3) 
      en=en+fk(318)*fmr(1,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(319)*fmr(1,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(320)*fmr(1,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(321)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(322)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(323)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(324)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(325)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(326)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(327)*fmr(1,1)*fmr(2,1)*fmr(6,3) 
      en=en+fk(328)*fmr(1,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(329)*fmr(1,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(330)*fmr(1,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(331)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(332)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(333)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(334)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(335)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(336)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(337)*fmr(1,1)*fmr(2,2)*fmr(3,2) 
      en=en+fk(338)*fmr(1,1)*fmr(2,2)*fmr(4,2) 
      en=en+fk(339)*fmr(1,1)*fmr(2,2)*fmr(5,2) 
      en=en+fk(340)*fmr(1,1)*fmr(2,2)*fmr(6,2) 
      en=en+fk(341)*fmr(1,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(342)*fmr(1,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(343)*fmr(1,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(344)*fmr(1,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(345)*fmr(1,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(346)*fmr(1,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(347)*fmr(1,2)*fmr(2,1)*fmr(3,2) 
      en=en+fk(348)*fmr(1,2)*fmr(2,1)*fmr(4,2) 
      en=en+fk(349)*fmr(1,2)*fmr(2,1)*fmr(5,2) 
      en=en+fk(350)*fmr(1,2)*fmr(2,1)*fmr(6,2) 
      en=en+fk(351)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(352)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(353)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(354)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(355)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(356)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(357)*fmr(1,2)*fmr(2,2)*fmr(3,1) 
      en=en+fk(358)*fmr(1,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(359)*fmr(1,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(360)*fmr(1,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(361)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(362)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(363)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(364)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(365)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(366)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(367)*fmr(1,2)*fmr(2,2)*fmr(4,1) 
      en=en+fk(368)*fmr(1,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(369)*fmr(1,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(370)*fmr(1,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(371)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(372)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(373)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(374)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(375)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(376)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(377)*fmr(1,2)*fmr(2,2)*fmr(5,1) 
      en=en+fk(378)*fmr(1,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(379)*fmr(1,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(380)*fmr(1,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(381)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(382)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(383)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(384)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(385)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(386)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(387)*fmr(1,2)*fmr(2,2)*fmr(6,1) 
      en=en+fk(388)*fmr(1,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(389)*fmr(1,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(390)*fmr(1,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(391)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(392)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(393)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(394)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(395)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(396)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(397)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(398)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(399)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(400)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(401)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(402)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(403)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(404)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(405)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(406)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(407)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(408)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(409)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(410)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(411)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(412)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(413)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(414)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(415)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(416)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(417)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(418)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(419)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(420)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(421)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(422)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(423)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(424)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(425)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(426)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(427)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(428)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(429)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(430)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(431)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(432)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(433)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(434)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(435)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(436)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(437)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(438)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(439)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(440)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(441)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(442)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(443)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(444)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(445)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(446)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(447)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(448)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(449)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(450)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(451)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(452)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(453)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(454)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(455)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(456)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(457)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(458)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(459)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(460)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(461)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(462)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(463)*fmr(1,6) 
      en=en+fk(464)*fmr(2,6) 
      en=en+fk(465)*fmr(3,6) 
      en=en+fk(466)*fmr(4,6) 
      en=en+fk(467)*fmr(5,6) 
      en=en+fk(468)*fmr(6,6) 
      en=en+fk(469)*fmr(1,5)*fmr(2,1) 
      en=en+fk(470)*fmr(1,5)*fmr(3,1) 
      en=en+fk(471)*fmr(1,5)*fmr(4,1) 
      en=en+fk(472)*fmr(1,5)*fmr(5,1) 
      en=en+fk(473)*fmr(1,5)*fmr(6,1) 
      en=en+fk(474)*fmr(1,1)*fmr(2,5) 
      en=en+fk(475)*fmr(2,5)*fmr(3,1) 
      en=en+fk(476)*fmr(2,5)*fmr(4,1) 
      en=en+fk(477)*fmr(2,5)*fmr(5,1) 
      en=en+fk(478)*fmr(2,5)*fmr(6,1) 
      en=en+fk(479)*fmr(1,1)*fmr(3,5) 
      en=en+fk(480)*fmr(2,1)*fmr(3,5) 
      en=en+fk(481)*fmr(3,5)*fmr(4,1) 
      en=en+fk(482)*fmr(3,5)*fmr(5,1) 
      en=en+fk(483)*fmr(3,5)*fmr(6,1) 
      en=en+fk(484)*fmr(1,1)*fmr(4,5) 
      en=en+fk(485)*fmr(2,1)*fmr(4,5) 
      en=en+fk(486)*fmr(3,1)*fmr(4,5) 
      en=en+fk(487)*fmr(4,5)*fmr(5,1) 
      en=en+fk(488)*fmr(4,5)*fmr(6,1) 
      en=en+fk(489)*fmr(1,1)*fmr(5,5) 
      en=en+fk(490)*fmr(2,1)*fmr(5,5) 
      en=en+fk(491)*fmr(3,1)*fmr(5,5) 
      en=en+fk(492)*fmr(4,1)*fmr(5,5) 
      en=en+fk(493)*fmr(5,5)*fmr(6,1) 
      en=en+fk(494)*fmr(1,1)*fmr(6,5) 
      en=en+fk(495)*fmr(2,1)*fmr(6,5) 
      en=en+fk(496)*fmr(3,1)*fmr(6,5) 
      en=en+fk(497)*fmr(4,1)*fmr(6,5) 
      en=en+fk(498)*fmr(5,1)*fmr(6,5) 
      en=en+fk(499)*fmr(1,4)*fmr(2,2) 
      en=en+fk(500)*fmr(1,4)*fmr(3,2) 
      en=en+fk(501)*fmr(1,4)*fmr(4,2) 
      en=en+fk(502)*fmr(1,4)*fmr(5,2) 
      en=en+fk(503)*fmr(1,4)*fmr(6,2) 
      en=en+fk(504)*fmr(1,2)*fmr(2,4) 
      en=en+fk(505)*fmr(2,4)*fmr(3,2) 
      en=en+fk(506)*fmr(2,4)*fmr(4,2) 
      en=en+fk(507)*fmr(2,4)*fmr(5,2) 
      en=en+fk(508)*fmr(2,4)*fmr(6,2) 
      en=en+fk(509)*fmr(1,2)*fmr(3,4) 
      en=en+fk(510)*fmr(2,2)*fmr(3,4) 
      en=en+fk(511)*fmr(3,4)*fmr(4,2) 
      en=en+fk(512)*fmr(3,4)*fmr(5,2) 
      en=en+fk(513)*fmr(3,4)*fmr(6,2) 
      en=en+fk(514)*fmr(1,2)*fmr(4,4) 
      en=en+fk(515)*fmr(2,2)*fmr(4,4) 
      en=en+fk(516)*fmr(3,2)*fmr(4,4) 
      en=en+fk(517)*fmr(4,4)*fmr(5,2) 
      en=en+fk(518)*fmr(4,4)*fmr(6,2) 
      en=en+fk(519)*fmr(1,2)*fmr(5,4) 
      en=en+fk(520)*fmr(2,2)*fmr(5,4) 
      en=en+fk(521)*fmr(3,2)*fmr(5,4) 
      en=en+fk(522)*fmr(4,2)*fmr(5,4) 
      en=en+fk(523)*fmr(5,4)*fmr(6,2) 
      en=en+fk(524)*fmr(1,2)*fmr(6,4) 
      en=en+fk(525)*fmr(2,2)*fmr(6,4) 
      en=en+fk(526)*fmr(3,2)*fmr(6,4) 
      en=en+fk(527)*fmr(4,2)*fmr(6,4) 
      en=en+fk(528)*fmr(5,2)*fmr(6,4) 
      en=en+fk(529)*fmr(1,3)*fmr(2,3) 
      en=en+fk(530)*fmr(1,3)*fmr(3,3) 
      en=en+fk(531)*fmr(1,3)*fmr(4,3) 
      en=en+fk(532)*fmr(1,3)*fmr(5,3) 
      en=en+fk(533)*fmr(1,3)*fmr(6,3) 
      en=en+fk(534)*fmr(2,3)*fmr(3,3) 
      en=en+fk(535)*fmr(2,3)*fmr(4,3) 
      en=en+fk(536)*fmr(2,3)*fmr(5,3) 
      en=en+fk(537)*fmr(2,3)*fmr(6,3) 
      en=en+fk(538)*fmr(3,3)*fmr(4,3) 
      en=en+fk(539)*fmr(3,3)*fmr(5,3) 
      en=en+fk(540)*fmr(3,3)*fmr(6,3) 
      en=en+fk(541)*fmr(4,3)*fmr(5,3) 
      en=en+fk(542)*fmr(4,3)*fmr(6,3) 
      en=en+fk(543)*fmr(5,3)*fmr(6,3) 
      en=en+fk(544)*fmr(4,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(545)*fmr(3,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(546)*fmr(3,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(547)*fmr(3,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(548)*fmr(2,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(549)*fmr(2,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(550)*fmr(2,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(551)*fmr(2,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(552)*fmr(2,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(553)*fmr(2,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(554)*fmr(1,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(555)*fmr(1,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(556)*fmr(1,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(557)*fmr(1,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(558)*fmr(1,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(559)*fmr(1,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(560)*fmr(1,2)*fmr(2,2)*fmr(6,2) 
      en=en+fk(561)*fmr(1,2)*fmr(2,2)*fmr(5,2) 
      en=en+fk(562)*fmr(1,2)*fmr(2,2)*fmr(4,2) 
      en=en+fk(563)*fmr(1,2)*fmr(2,2)*fmr(3,2) 
      en=en+fk(564)*fmr(1,4)*fmr(2,1)*fmr(3,1) 
      en=en+fk(565)*fmr(1,4)*fmr(2,1)*fmr(4,1) 
      en=en+fk(566)*fmr(1,4)*fmr(2,1)*fmr(5,1) 
      en=en+fk(567)*fmr(1,4)*fmr(2,1)*fmr(6,1) 
      en=en+fk(568)*fmr(1,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(569)*fmr(1,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(570)*fmr(1,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(571)*fmr(1,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(572)*fmr(1,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(573)*fmr(1,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(574)*fmr(1,1)*fmr(2,4)*fmr(3,1) 
      en=en+fk(575)*fmr(1,1)*fmr(2,4)*fmr(4,1) 
      en=en+fk(576)*fmr(1,1)*fmr(2,4)*fmr(5,1) 
      en=en+fk(577)*fmr(1,1)*fmr(2,4)*fmr(6,1) 
      en=en+fk(578)*fmr(2,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(579)*fmr(2,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(580)*fmr(2,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(581)*fmr(2,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(582)*fmr(2,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(583)*fmr(2,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(584)*fmr(1,1)*fmr(2,1)*fmr(3,4) 
      en=en+fk(585)*fmr(1,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(586)*fmr(1,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(587)*fmr(1,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(588)*fmr(2,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(589)*fmr(2,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(590)*fmr(2,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(591)*fmr(3,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(592)*fmr(3,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(593)*fmr(3,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(594)*fmr(1,1)*fmr(2,1)*fmr(4,4) 
      en=en+fk(595)*fmr(1,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(596)*fmr(1,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(597)*fmr(1,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(598)*fmr(2,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(599)*fmr(2,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(600)*fmr(2,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(601)*fmr(3,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(602)*fmr(3,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(603)*fmr(4,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(604)*fmr(1,1)*fmr(2,1)*fmr(5,4) 
      en=en+fk(605)*fmr(1,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(606)*fmr(1,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(607)*fmr(1,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(608)*fmr(2,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(609)*fmr(2,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(610)*fmr(2,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(611)*fmr(3,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(612)*fmr(3,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(613)*fmr(4,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(614)*fmr(1,1)*fmr(2,1)*fmr(6,4) 
      en=en+fk(615)*fmr(1,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(616)*fmr(1,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(617)*fmr(1,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(618)*fmr(2,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(619)*fmr(2,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(620)*fmr(2,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(621)*fmr(3,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(622)*fmr(3,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(623)*fmr(4,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(624)*fmr(1,3)*fmr(2,2)*fmr(3,1) 
      en=en+fk(625)*fmr(1,3)*fmr(2,2)*fmr(4,1) 
      en=en+fk(626)*fmr(1,3)*fmr(2,2)*fmr(5,1) 
      en=en+fk(627)*fmr(1,3)*fmr(2,2)*fmr(6,1) 
      en=en+fk(628)*fmr(1,3)*fmr(2,1)*fmr(3,2) 
      en=en+fk(629)*fmr(1,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(630)*fmr(1,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(631)*fmr(1,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(632)*fmr(1,3)*fmr(2,1)*fmr(4,2) 
      en=en+fk(633)*fmr(1,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(634)*fmr(1,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(635)*fmr(1,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(636)*fmr(1,3)*fmr(2,1)*fmr(5,2) 
      en=en+fk(637)*fmr(1,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(638)*fmr(1,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(639)*fmr(1,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(640)*fmr(1,3)*fmr(2,1)*fmr(6,2) 
      en=en+fk(641)*fmr(1,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(642)*fmr(1,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(643)*fmr(1,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(644)*fmr(1,2)*fmr(2,3)*fmr(3,1) 
      en=en+fk(645)*fmr(1,2)*fmr(2,3)*fmr(4,1) 
      en=en+fk(646)*fmr(1,2)*fmr(2,3)*fmr(5,1) 
      en=en+fk(647)*fmr(1,2)*fmr(2,3)*fmr(6,1) 
      en=en+fk(648)*fmr(1,1)*fmr(2,3)*fmr(3,2) 
      en=en+fk(649)*fmr(2,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(650)*fmr(2,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(651)*fmr(2,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(652)*fmr(1,1)*fmr(2,3)*fmr(4,2) 
      en=en+fk(653)*fmr(2,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(654)*fmr(2,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(655)*fmr(2,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(656)*fmr(1,1)*fmr(2,3)*fmr(5,2) 
      en=en+fk(657)*fmr(2,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(658)*fmr(2,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(659)*fmr(2,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(660)*fmr(1,1)*fmr(2,3)*fmr(6,2) 
      en=en+fk(661)*fmr(2,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(662)*fmr(2,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(663)*fmr(2,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(664)*fmr(1,2)*fmr(2,1)*fmr(3,3) 
      en=en+fk(665)*fmr(1,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(666)*fmr(1,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(667)*fmr(1,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(668)*fmr(1,1)*fmr(2,2)*fmr(3,3) 
      en=en+fk(669)*fmr(2,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(670)*fmr(2,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(671)*fmr(2,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(672)*fmr(1,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(673)*fmr(2,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(674)*fmr(3,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(675)*fmr(3,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(676)*fmr(1,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(677)*fmr(2,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(678)*fmr(3,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(679)*fmr(3,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(680)*fmr(1,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(681)*fmr(2,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(682)*fmr(3,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(683)*fmr(3,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(684)*fmr(1,2)*fmr(2,1)*fmr(4,3) 
      en=en+fk(685)*fmr(1,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(686)*fmr(1,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(687)*fmr(1,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(688)*fmr(1,1)*fmr(2,2)*fmr(4,3) 
      en=en+fk(689)*fmr(2,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(690)*fmr(2,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(691)*fmr(2,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(692)*fmr(1,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(693)*fmr(2,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(694)*fmr(3,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(695)*fmr(3,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(696)*fmr(1,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(697)*fmr(2,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(698)*fmr(3,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(699)*fmr(4,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(700)*fmr(1,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(701)*fmr(2,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(702)*fmr(3,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(703)*fmr(4,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(704)*fmr(1,2)*fmr(2,1)*fmr(5,3) 
      en=en+fk(705)*fmr(1,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(706)*fmr(1,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(707)*fmr(1,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(708)*fmr(1,1)*fmr(2,2)*fmr(5,3) 
      en=en+fk(709)*fmr(2,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(710)*fmr(2,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(711)*fmr(2,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(712)*fmr(1,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(713)*fmr(2,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(714)*fmr(3,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(715)*fmr(3,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(716)*fmr(1,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(717)*fmr(2,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(718)*fmr(3,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(719)*fmr(4,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(720)*fmr(1,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(721)*fmr(2,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(722)*fmr(3,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(723)*fmr(4,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(724)*fmr(1,2)*fmr(2,1)*fmr(6,3) 
      en=en+fk(725)*fmr(1,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(726)*fmr(1,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(727)*fmr(1,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(728)*fmr(1,1)*fmr(2,2)*fmr(6,3) 
      en=en+fk(729)*fmr(2,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(730)*fmr(2,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(731)*fmr(2,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(732)*fmr(1,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(733)*fmr(2,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(734)*fmr(3,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(735)*fmr(3,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(736)*fmr(1,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(737)*fmr(2,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(738)*fmr(3,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(739)*fmr(4,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(740)*fmr(1,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(741)*fmr(2,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(742)*fmr(3,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(743)*fmr(4,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(744)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(745)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(746)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(747)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(748)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(749)*fmr(1,3)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(750)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(751)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(752)*fmr(1,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(753)*fmr(1,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(754)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(755)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(756)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(757)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(758)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(759)*fmr(1,1)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(760)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(761)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(762)*fmr(2,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(763)*fmr(2,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(764)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(765)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(766)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(767)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(768)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(769)*fmr(1,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(770)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(771)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(772)*fmr(2,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(773)*fmr(3,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(774)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(775)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(776)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(777)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(778)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(779)*fmr(1,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(780)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(781)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(782)*fmr(2,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(783)*fmr(3,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(784)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(785)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(786)*fmr(1,1)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(787)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(788)*fmr(1,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(789)*fmr(1,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(790)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(791)*fmr(2,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(792)*fmr(2,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(793)*fmr(3,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(794)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(795)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(796)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(797)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(798)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(799)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(800)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(801)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(802)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(803)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(804)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(805)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(806)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(807)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(808)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(809)*fmr(1,2)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(810)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(811)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(812)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(813)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(814)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(815)*fmr(1,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(816)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(817)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(818)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(819)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(820)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(821)*fmr(1,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(822)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(823)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(824)*fmr(1,2)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(825)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(826)*fmr(1,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(827)*fmr(1,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(828)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(829)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(830)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(831)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(832)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(833)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(834)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(835)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(836)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(837)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(838)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(839)*fmr(2,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(840)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(841)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(842)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(843)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(844)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(845)*fmr(2,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(846)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(847)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(848)*fmr(1,1)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(849)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(850)*fmr(2,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(851)*fmr(2,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(852)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(853)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(854)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(855)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(856)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(857)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(858)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(859)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(860)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(861)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(862)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(863)*fmr(3,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(864)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(865)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(866)*fmr(1,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(867)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(868)*fmr(2,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(869)*fmr(3,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(870)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(871)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(872)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(873)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(874)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(875)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(876)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(877)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(878)*fmr(1,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(879)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(880)*fmr(2,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(881)*fmr(3,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(882)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(883)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(884)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(885)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(886)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(887)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(888)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(889)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(890)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(891)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(892)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(893)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(894)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(895)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(896)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(897)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(898)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(899)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(900)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(901)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(902)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(903)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(904)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(905)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(906)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(907)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(908)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(909)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(910)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(911)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(912)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(913)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(914)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(915)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(916)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(917)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(918)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(919)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(920)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(921)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(922)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(923)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(924)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)  &
        *fmr(6,1) 

      return
      end subroutine methyl
!
      subroutine chph2(rij,en)
!
!     from ch-19429-924.p
!
!     rij(1) = r(c-h1) (au)
!     rij(2) = r(c-h2)
!     rij(3) = r(c-h3)
!     rij(4) = r(h1-h2)
!     rij(5) = r(h1-h3)
!     rij(6) = r(h2-h3)
!
!     en = energy (hartree)
!
!
!      implicit real*8 (a-h,o-z)
!      dimension fk(924),fmr(6,6), shift(6),rij(6),alpham(6)
      implicit none
      double precision,intent(in) :: rij(6)
      double precision,intent(out) :: en
      double precision :: fk(924),fmr(6,6),shift(6),alpham(6)
      integer :: i, j, nmax, nvar
  
      data fk(   1), fk(   2) / -4.2123404450E+01 , -2.5080524918E+00 /
      data fk(   3), fk(   4) / -2.5080524918E+00 , -2.5080524918E+00 /
      data fk(   5), fk(   6) /  1.2498315271E+01 ,  1.2498315271E+01 /
      data fk(   7), fk(   8) /  1.2498315271E+01 ,  1.8705201191E+00 /
      data fk(   9), fk(  10) /  1.8705201191E+00 ,  1.8705201191E+00 /
      data fk(  11), fk(  12) / -2.3649951279E+01 , -2.3649951279E+01 /
      data fk(  13), fk(  14) / -2.3649951279E+01 , -2.9515126178E+00 /
      data fk(  15), fk(  16) / -2.9515126178E+00 ,  9.7207182118E+00 /
      data fk(  17), fk(  18) /  9.7207182118E+00 ,  5.5512966226E+00 /
      data fk(  19), fk(  20) / -2.9515126178E+00 ,  9.7207182118E+00 /
      data fk(  21), fk(  22) /  5.5512966226E+00 ,  9.7207182118E+00 /
      data fk(  23), fk(  24) /  5.5512966226E+00 ,  9.7207182118E+00 /
      data fk(  25), fk(  26) /  9.7207182118E+00 , -4.6711293760E+01 /
      data fk(  27), fk(  28) / -4.6711293760E+01 , -4.6711293760E+01 /
      data fk(  29), fk(  30) /  8.2585551569E+00 ,  8.2585551569E+00 /
      data fk(  31), fk(  32) /  8.2585551569E+00 ,  9.8173698839E+00 /
      data fk(  33), fk(  34) /  9.8173698839E+00 ,  9.8173698839E+00 /
      data fk(  35), fk(  36) / -6.7627689509E+00 , -6.7627689509E+00 /
      data fk(  37), fk(  38) / -1.0346072311E+00 , -1.0346072311E+00 /
      data fk(  39), fk(  40) / -2.6797209531E+01 , -6.7627689509E+00 /
      data fk(  41), fk(  42) / -6.7627689509E+00 , -1.0346072311E+00 /
      data fk(  43), fk(  44) / -2.6797209531E+01 , -1.0346072311E+00 /
      data fk(  45), fk(  46) / -6.7627689509E+00 , -6.7627689509E+00 /
      data fk(  47), fk(  48) / -2.6797209531E+01 , -1.0346072311E+00 /
      data fk(  49), fk(  50) / -1.0346072311E+00 , -1.8974693591E+01 /
      data fk(  51), fk(  52) / -1.8974693591E+01 ,  1.9346624074E+01 /
      data fk(  53), fk(  54) /  7.3163327532E+01 ,  7.3163327532E+01 /
      data fk(  55), fk(  56) / -1.8974693591E+01 ,  1.9346624074E+01 /
      data fk(  57), fk(  58) / -1.8974693591E+01 ,  7.3163327532E+01 /
      data fk(  59), fk(  60) /  7.3163327532E+01 ,  1.9346624074E+01 /
      data fk(  61), fk(  62) / -1.8974693591E+01 , -1.8974693591E+01 /
      data fk(  63), fk(  64) /  7.3163327532E+01 ,  7.3163327532E+01 /
      data fk(  65), fk(  66) /  2.8384425356E+01 , -1.6804818405E+01 /
      data fk(  67), fk(  68) /  1.6586421191E+01 ,  1.6586421191E+01 /
      data fk(  69), fk(  70) /  1.6586421191E+01 , -1.6804818405E+01 /
      data fk(  71), fk(  72) /  1.6586421191E+01 , -3.8150925846E+00 /
      data fk(  73), fk(  74) / -2.2781610464E+01 , -2.2781610464E+01 /
      data fk(  75), fk(  76) /  1.6586421191E+01 ,  1.6586421191E+01 /
      data fk(  77), fk(  78) / -1.6804818405E+01 , -2.2781610464E+01 /
      data fk(  79), fk(  80) / -3.8150925846E+00 , -2.2781610464E+01 /
      data fk(  81), fk(  82) / -2.2781610464E+01 , -2.2781610464E+01 /
      data fk(  83), fk(  84) / -3.8150925846E+00 ,  9.9959384789E+01 /
      data fk(  85), fk(  86) /  7.1128979149E+00 ,  7.1128979149E+00 /
      data fk(  87), fk(  88) /  7.1128979149E+00 ,  2.3781255501E+01 /
      data fk(  89), fk(  90) /  2.3781255501E+01 ,  2.3781255501E+01 /
      data fk(  91), fk(  92) /  6.4443406021E+00 ,  6.4443406021E+00 /
      data fk(  93), fk(  94) / -2.5660750755E+01 , -2.5660750755E+01 /
      data fk(  95), fk(  96) / -4.2164713190E+01 ,  6.4443406021E+00 /
      data fk(  97), fk(  98) /  6.4443406021E+00 , -2.5660750755E+01 /
      data fk(  99), fk( 100) / -4.2164713190E+01 , -2.5660750755E+01 /
      data fk( 101), fk( 102) /  6.4443406021E+00 ,  6.4443406021E+00 /
      data fk( 103), fk( 104) / -4.2164713190E+01 , -2.5660750755E+01 /
      data fk( 105), fk( 106) / -2.5660750755E+01 ,  1.2191209725E+01 /
      data fk( 107), fk( 108) /  1.2191209725E+01 , -7.5995305148E+01 /
      data fk( 109), fk( 110) / -3.4852979709E+01 , -3.4852979709E+01 /
      data fk( 111), fk( 112) /  1.2191209725E+01 , -7.5995305148E+01 /
      data fk( 113), fk( 114) /  1.2191209725E+01 , -3.4852979709E+01 /
      data fk( 115), fk( 116) / -3.4852979709E+01 , -7.5995305148E+01 /
      data fk( 117), fk( 118) /  1.2191209725E+01 ,  1.2191209725E+01 /
      data fk( 119), fk( 120) / -3.4852979709E+01 , -3.4852979709E+01 /
      data fk( 121), fk( 122) /  2.4991192912E+01 ,  2.4991192912E+01 /
      data fk( 123), fk( 124) /  2.7954810017E+01 ,  2.7954810017E+01 /
      data fk( 125), fk( 126) /  8.5280002000E+01 ,  2.4991192912E+01 /
      data fk( 127), fk( 128) /  2.7954810017E+01 ,  8.5280002000E+01 /
      data fk( 129), fk( 130) /  2.7954810017E+01 ,  8.5280002000E+01 /
      data fk( 131), fk( 132) /  2.7954810017E+01 ,  2.7954810017E+01 /
      data fk( 133), fk( 134) / -6.6774786863E+01 , -6.6774786863E+01 /
      data fk( 135), fk( 136) / -6.6774786863E+01 , -1.2561566244E+01 /
      data fk( 137), fk( 138) /  2.1834669383E+01 , -4.0632020437E+01 /
      data fk( 139), fk( 140) / -1.9691212234E+00 , -4.0632020437E+01 /
      data fk( 141), fk( 142) /  2.1834669383E+01 , -1.9691212234E+00 /
      data fk( 143), fk( 144) / -1.2055436799E+01 ,  6.0654686721E+01 /
      data fk( 145), fk( 146) /  6.0654686721E+01 , -1.2561566244E+01 /
      data fk( 147), fk( 148) /  2.1834669383E+01 , -1.9691212234E+00 /
      data fk( 149), fk( 150) / -4.0632020437E+01 , -4.0632020437E+01 /
      data fk( 151), fk( 152) / -1.9691212234E+00 ,  2.1834669383E+01 /
      data fk( 153), fk( 154) /  6.0654686721E+01 , -1.2055436799E+01 /
      data fk( 155), fk( 156) /  6.0654686721E+01 , -1.2561566244E+01 /
      data fk( 157), fk( 158) / -1.9691212234E+00 ,  2.1834669383E+01 /
      data fk( 159), fk( 160) / -4.0632020437E+01 , -1.9691212234E+00 /
      data fk( 161), fk( 162) / -4.0632020437E+01 ,  2.1834669383E+01 /
      data fk( 163), fk( 164) /  6.0654686721E+01 ,  6.0654686721E+01 /
      data fk( 165), fk( 166) / -1.2055436799E+01 ,  2.4998841955E+01 /
      data fk( 167), fk( 168) / -8.8800653155E+00 ,  2.9027376570E+00 /
      data fk( 169), fk( 170) / -1.3674373240E+01 , -8.8800653155E+00 /
      data fk( 171), fk( 172) / -1.3674373240E+01 ,  2.9027376570E+00 /
      data fk( 173), fk( 174) / -3.3028553771E+01 , -3.3028553771E+01 /
      data fk( 175), fk( 176) / -1.0747313225E+02 , -8.8800653155E+00 /
      data fk( 177), fk( 178) /  2.4998841955E+01 ,  2.9027376570E+00 /
      data fk( 179), fk( 180) / -1.3674373240E+01 , -8.8800653155E+00 /
      data fk( 181), fk( 182) / -3.3028553771E+01 , -3.3028553771E+01 /
      data fk( 183), fk( 184) / -1.3674373240E+01 ,  2.9027376570E+00 /
      data fk( 185), fk( 186) / -1.0747313225E+02 , -8.8800653155E+00 /
      data fk( 187), fk( 188) / -8.8800653155E+00 , -3.3028553771E+01 /
      data fk( 189), fk( 190) / -3.3028553771E+01 ,  2.4998841955E+01 /
      data fk( 191), fk( 192) /  2.9027376570E+00 , -1.3674373240E+01 /
      data fk( 193), fk( 194) / -1.3674373240E+01 ,  2.9027376570E+00 /
      data fk( 195), fk( 196) / -1.0747313225E+02 ,  2.5630712517E+01 /
      data fk( 197), fk( 198) /  2.5630712517E+01 , -2.8561549026E-02 /
      data fk( 199), fk( 200) / -2.8561549026E-02 ,  5.7966466680E+01 /
      data fk( 201), fk( 202) /  2.5630712517E+01 , -2.8561549026E-02 /
      data fk( 203), fk( 204) /  5.7966466680E+01 , -2.8561549026E-02 /
      data fk( 205), fk( 206) /  5.7966466680E+01 , -2.8561549026E-02 /
      data fk( 207), fk( 208) / -2.8561549026E-02 , -3.2126935286E+01 /
      data fk( 209), fk( 210) / -3.2126935286E+01 , -3.2126935286E+01 /
      data fk( 211), fk( 212) /  1.8431031800E+00 ,  1.8431031800E+00 /
      data fk( 213), fk( 214) /  1.8431031800E+00 , -3.1808206452E+01 /
      data fk( 215), fk( 216) / -3.1808206452E+01 , -3.1808206452E+01 /
      data fk( 217), fk( 218) /  3.5855469161E+00 ,  3.5855469161E+00 /
      data fk( 219), fk( 220) / -1.4184172265E+01 , -1.4184172265E+01 /
      data fk( 221), fk( 222) / -1.2284637364E+01 ,  3.5855469161E+00 /
      data fk( 223), fk( 224) /  3.5855469161E+00 , -1.4184172265E+01 /
      data fk( 225), fk( 226) / -1.2284637364E+01 , -1.4184172265E+01 /
      data fk( 227), fk( 228) /  3.5855469161E+00 ,  3.5855469161E+00 /
      data fk( 229), fk( 230) / -1.2284637364E+01 , -1.4184172265E+01 /
      data fk( 231), fk( 232) / -1.4184172265E+01 ,  5.0210969455E+00 /
      data fk( 233), fk( 234) /  5.0210969455E+00 ,  8.5070541603E+01 /
      data fk( 235), fk( 236) / -2.1175850243E+01 , -2.1175850243E+01 /
      data fk( 237), fk( 238) /  5.0210969455E+00 ,  8.5070541603E+01 /
      data fk( 239), fk( 240) /  5.0210969455E+00 , -2.1175850243E+01 /
      data fk( 241), fk( 242) / -2.1175850243E+01 ,  8.5070541603E+01 /
      data fk( 243), fk( 244) /  5.0210969455E+00 ,  5.0210969455E+00 /
      data fk( 245), fk( 246) / -2.1175850243E+01 , -2.1175850243E+01 /
      data fk( 247), fk( 248) /  8.3749962237E+00 ,  8.3749962237E+00 /
      data fk( 249), fk( 250) /  3.3483411254E+01 ,  3.3483411254E+01 /
      data fk( 251), fk( 252) /  5.1991525100E+01 ,  8.3749962237E+00 /
      data fk( 253), fk( 254) /  8.3749962237E+00 ,  3.3483411254E+01 /
      data fk( 255), fk( 256) /  5.1991525100E+01 ,  3.3483411254E+01 /
      data fk( 257), fk( 258) /  8.3749962237E+00 ,  8.3749962237E+00 /
      data fk( 259), fk( 260) /  5.1991525100E+01 ,  3.3483411254E+01 /
      data fk( 261), fk( 262) /  3.3483411254E+01 , -4.5993795404E+01 /
      data fk( 263), fk( 264) / -4.5993795404E+01 , -9.3053636536E+01 /
      data fk( 265), fk( 266) / -1.3948666121E+00 , -1.3948666121E+00 /
      data fk( 267), fk( 268) / -4.5993795404E+01 , -9.3053636536E+01 /
      data fk( 269), fk( 270) / -4.5993795404E+01 , -1.3948666121E+00 /
      data fk( 271), fk( 272) / -1.3948666121E+00 , -9.3053636536E+01 /
      data fk( 273), fk( 274) / -4.5993795404E+01 , -4.5993795404E+01 /
      data fk( 275), fk( 276) / -1.3948666121E+00 , -1.3948666121E+00 /
      data fk( 277), fk( 278) / -2.0036797234E+01 ,  8.0996986505E+00 /
      data fk( 279), fk( 280) / -2.9118424753E+01 , -1.9513529196E+01 /
      data fk( 281), fk( 282) / -2.9118424753E+01 ,  8.0996986505E+00 /
      data fk( 283), fk( 284) / -1.9513529196E+01 ,  3.1180428283E+01 /
      data fk( 285), fk( 286) /  7.2948800468E+01 ,  7.2948800468E+01 /
      data fk( 287), fk( 288) / -2.0036797234E+01 ,  8.0996986505E+00 /
      data fk( 289), fk( 290) / -1.9513529196E+01 , -2.9118424753E+01 /
      data fk( 291), fk( 292) / -2.9118424753E+01 , -1.9513529196E+01 /
      data fk( 293), fk( 294) /  8.0996986505E+00 ,  7.2948800468E+01 /
      data fk( 295), fk( 296) /  3.1180428283E+01 ,  7.2948800468E+01 /
      data fk( 297), fk( 298) / -2.0036797234E+01 , -1.9513529196E+01 /
      data fk( 299), fk( 300) /  8.0996986505E+00 , -2.9118424753E+01 /
      data fk( 301), fk( 302) / -1.9513529196E+01 , -2.9118424753E+01 /
      data fk( 303), fk( 304) /  8.0996986505E+00 ,  7.2948800468E+01 /
      data fk( 305), fk( 306) /  7.2948800468E+01 ,  3.1180428283E+01 /
      data fk( 307), fk( 308) /  7.0018600938E+00 , -2.5733812886E+01 /
      data fk( 309), fk( 310) / -1.5581636705E+01 ,  6.5342905979E+01 /
      data fk( 311), fk( 312) / -2.5733812886E+01 ,  6.5342905979E+01 /
      data fk( 313), fk( 314) / -1.5581636705E+01 ,  1.0300392559E+02 /
      data fk( 315), fk( 316) /  1.0300392559E+02 ,  5.1417311463E+01 /
      data fk( 317), fk( 318) / -2.5733812886E+01 ,  7.0018600938E+00 /
      data fk( 319), fk( 320) / -1.5581636705E+01 ,  6.5342905979E+01 /
      data fk( 321), fk( 322) / -2.5733812886E+01 ,  1.0300392559E+02 /
      data fk( 323), fk( 324) /  1.0300392559E+02 ,  6.5342905979E+01 /
      data fk( 325), fk( 326) / -1.5581636705E+01 ,  5.1417311463E+01 /
      data fk( 327), fk( 328) / -2.5733812886E+01 , -2.5733812886E+01 /
      data fk( 329), fk( 330) /  1.0300392559E+02 ,  1.0300392559E+02 /
      data fk( 331), fk( 332) /  7.0018600938E+00 , -1.5581636705E+01 /
      data fk( 333), fk( 334) /  6.5342905979E+01 ,  6.5342905979E+01 /
      data fk( 335), fk( 336) / -1.5581636705E+01 ,  5.1417311463E+01 /
      data fk( 337), fk( 338) /  3.2277297652E+01 ,  2.7262965812E+00 /
      data fk( 339), fk( 340) /  3.3573918045E+01 ,  9.0469320372E+01 /
      data fk( 341), fk( 342) /  3.3573918045E+01 ,  2.7262965812E+00 /
      data fk( 343), fk( 344) /  9.0469320372E+01 ,  3.5279515873E+01 /
      data fk( 345), fk( 346) /  1.0456912376E+02 ,  1.0456912376E+02 /
      data fk( 347), fk( 348) /  3.2277297652E+01 ,  2.7262965812E+00 /
      data fk( 349), fk( 350) /  9.0469320372E+01 ,  3.3573918045E+01 /
      data fk( 351), fk( 352) /  3.3573918045E+01 ,  9.0469320372E+01 /
      data fk( 353), fk( 354) /  2.7262965812E+00 ,  1.0456912376E+02 /
      data fk( 355), fk( 356) /  3.5279515873E+01 ,  1.0456912376E+02 /
      data fk( 357), fk( 358) /  3.2277297652E+01 ,  9.0469320372E+01 /
      data fk( 359), fk( 360) /  2.7262965812E+00 ,  3.3573918045E+01 /
      data fk( 361), fk( 362) /  9.0469320372E+01 ,  3.3573918045E+01 /
      data fk( 363), fk( 364) /  2.7262965812E+00 ,  1.0456912376E+02 /
      data fk( 365), fk( 366) /  1.0456912376E+02 ,  3.5279515873E+01 /
      data fk( 367), fk( 368) / -5.9296352340E+01 , -5.9340215653E+01 /
      data fk( 369), fk( 370) / -3.9128073435E+01 , -1.3435080537E+02 /
      data fk( 371), fk( 372) / -5.9340215653E+01 , -1.3435080537E+02 /
      data fk( 373), fk( 374) / -3.9128073435E+01 , -1.0597788954E+02 /
      data fk( 375), fk( 376) / -1.0597788954E+02 ,  2.1153606929E+01 /
      data fk( 377), fk( 378) / -5.9340215653E+01 , -5.9296352340E+01 /
      data fk( 379), fk( 380) / -3.9128073435E+01 , -1.3435080537E+02 /
      data fk( 381), fk( 382) / -5.9340215653E+01 , -1.0597788954E+02 /
      data fk( 383), fk( 384) / -1.0597788954E+02 , -1.3435080537E+02 /
      data fk( 385), fk( 386) / -3.9128073435E+01 ,  2.1153606929E+01 /
      data fk( 387), fk( 388) / -5.9340215653E+01 , -5.9340215653E+01 /
      data fk( 389), fk( 390) / -1.0597788954E+02 , -1.0597788954E+02 /
      data fk( 391), fk( 392) / -5.9296352340E+01 , -3.9128073435E+01 /
      data fk( 393), fk( 394) / -1.3435080537E+02 , -1.3435080537E+02 /
      data fk( 395), fk( 396) / -3.9128073435E+01 ,  2.1153606929E+01 /
      data fk( 397), fk( 398) / -4.0803584187E+01 , -4.0803584187E+01 /
      data fk( 399), fk( 400) /  5.7849863620E+01 ,  8.7695376763E+01 /
      data fk( 401), fk( 402) / -4.9395175974E+01 ,  1.2552200572E+02 /
      data fk( 403), fk( 404) /  8.7695376763E+01 ,  1.2552200572E+02 /
      data fk( 405), fk( 406) / -4.9395175974E+01 , -3.8742898456E+01 /
      data fk( 407), fk( 408) / -4.0803584187E+01 ,  5.7849863620E+01 /
      data fk( 409), fk( 410) / -4.0803584187E+01 , -4.9395175974E+01 /
      data fk( 411), fk( 412) /  8.7695376763E+01 ,  1.2552200572E+02 /
      data fk( 413), fk( 414) /  1.2552200572E+02 ,  8.7695376763E+01 /
      data fk( 415), fk( 416) / -4.9395175974E+01 , -3.8742898456E+01 /
      data fk( 417), fk( 418) /  5.7849863620E+01 , -4.0803584187E+01 /
      data fk( 419), fk( 420) / -4.0803584187E+01 , -4.9395175974E+01 /
      data fk( 421), fk( 422) /  1.2552200572E+02 ,  8.7695376763E+01 /
      data fk( 423), fk( 424) /  1.2552200572E+02 , -4.9395175974E+01 /
      data fk( 425), fk( 426) /  8.7695376763E+01 , -3.8742898456E+01 /
      data fk( 427), fk( 428) / -2.6618169312E+01 , -6.1549222913E+01 /
      data fk( 429), fk( 430) / -6.1549222913E+01 ,  3.8887693158E+01 /
      data fk( 431), fk( 432) / -1.6962662581E+02 ,  7.8912992005E+01 /
      data fk( 433), fk( 434) / -1.6962662581E+02 ,  3.8887693158E+01 /
      data fk( 435), fk( 436) /  7.8912992005E+01 ,  7.1315291127E+00 /
      data fk( 437), fk( 438) / -2.6618169312E+01 ,  3.8887693158E+01 /
      data fk( 439), fk( 440) / -1.6962662581E+02 , -6.1549222913E+01 /
      data fk( 441), fk( 442) / -6.1549222913E+01 ,  7.8912992005E+01 /
      data fk( 443), fk( 444) / -1.6962662581E+02 ,  3.8887693158E+01 /
      data fk( 445), fk( 446) /  7.1315291127E+00 ,  7.8912992005E+01 /
      data fk( 447), fk( 448) / -2.6618169312E+01 ,  3.8887693158E+01 /
      data fk( 449), fk( 450) / -1.6962662581E+02 , -1.6962662581E+02 /
      data fk( 451), fk( 452) /  3.8887693158E+01 ,  7.1315291127E+00 /
      data fk( 453), fk( 454) / -6.1549222913E+01 , -6.1549222913E+01 /
      data fk( 455), fk( 456) /  7.8912992005E+01 ,  7.8912992005E+01 /
      data fk( 457), fk( 458) /  1.2660246079E+02 ,  1.2660246079E+02 /
      data fk( 459), fk( 460) /  1.2660246079E+02 , -1.7624535018E+02 /
      data fk( 461), fk( 462) / -1.7624535018E+02 , -1.7624535018E+02 /
      data fk( 463), fk( 464) /  5.8678547268E-01 ,  5.8678547268E-01 /
      data fk( 465), fk( 466) /  5.8678547268E-01 ,  1.2648786998E+01 /
      data fk( 467), fk( 468) /  1.2648786998E+01 ,  1.2648786998E+01 /
      data fk( 469), fk( 470) /  2.7641121338E-01 ,  2.7641121338E-01 /
      data fk( 471), fk( 472) / -2.0015669882E+00 , -2.0015669882E+00 /
      data fk( 473), fk( 474) / -2.4252259880E+00 ,  2.7641121338E-01 /
      data fk( 475), fk( 476) /  2.7641121338E-01 , -2.0015669882E+00 /
      data fk( 477), fk( 478) / -2.4252259880E+00 , -2.0015669882E+00 /
      data fk( 479), fk( 480) /  2.7641121338E-01 ,  2.7641121338E-01 /
      data fk( 481), fk( 482) / -2.4252259880E+00 , -2.0015669882E+00 /
      data fk( 483), fk( 484) / -2.0015669882E+00 , -5.3023430838E+00 /
      data fk( 485), fk( 486) / -5.3023430838E+00 , -3.3996188022E+01 /
      data fk( 487), fk( 488) /  1.6414313991E+01 ,  1.6414313991E+01 /
      data fk( 489), fk( 490) / -5.3023430838E+00 , -3.3996188022E+01 /
      data fk( 491), fk( 492) / -5.3023430838E+00 ,  1.6414313991E+01 /
      data fk( 493), fk( 494) /  1.6414313991E+01 , -3.3996188022E+01 /
      data fk( 495), fk( 496) / -5.3023430838E+00 , -5.3023430838E+00 /
      data fk( 497), fk( 498) /  1.6414313991E+01 ,  1.6414313991E+01 /
      data fk( 499), fk( 500) /  7.8578641205E-01 ,  7.8578641205E-01 /
      data fk( 501), fk( 502) /  6.7763808565E+00 ,  6.7763808565E+00 /
      data fk( 503), fk( 504) /  7.3329210646E+00 ,  7.8578641205E-01 /
      data fk( 505), fk( 506) /  7.8578641205E-01 ,  6.7763808565E+00 /
      data fk( 507), fk( 508) /  7.3329210646E+00 ,  6.7763808565E+00 /
      data fk( 509), fk( 510) /  7.8578641205E-01 ,  7.8578641205E-01 /
      data fk( 511), fk( 512) /  7.3329210646E+00 ,  6.7763808565E+00 /
      data fk( 513), fk( 514) /  6.7763808565E+00 ,  1.8222064684E+01 /
      data fk( 515), fk( 516) /  1.8222064684E+01 ,  3.7145977211E+01 /
      data fk( 517), fk( 518) /  1.9632013114E+01 ,  1.9632013114E+01 /
      data fk( 519), fk( 520) /  1.8222064684E+01 ,  3.7145977211E+01 /
      data fk( 521), fk( 522) /  1.8222064684E+01 ,  1.9632013114E+01 /
      data fk( 523), fk( 524) /  1.9632013114E+01 ,  3.7145977211E+01 /
      data fk( 525), fk( 526) /  1.8222064684E+01 ,  1.8222064684E+01 /
      data fk( 527), fk( 528) /  1.9632013114E+01 ,  1.9632013114E+01 /
      data fk( 529), fk( 530) / -3.4495549923E+00 , -3.4495549923E+00 /
      data fk( 531), fk( 532) / -1.3560905734E+01 , -1.3560905734E+01 /
      data fk( 533), fk( 534) / -2.1709667258E+01 , -3.4495549923E+00 /
      data fk( 535), fk( 536) / -1.3560905734E+01 , -2.1709667258E+01 /
      data fk( 537), fk( 538) / -1.3560905734E+01 , -2.1709667258E+01 /
      data fk( 539), fk( 540) / -1.3560905734E+01 , -1.3560905734E+01 /
      data fk( 541), fk( 542) /  1.7933116656E+01 ,  1.7933116656E+01 /
      data fk( 543), fk( 544) /  1.7933116656E+01 ,  4.5038938765E+01 /
      data fk( 545), fk( 546) /  2.2822440307E+01 ,  7.6401881588E+01 /
      data fk( 547), fk( 548) /  7.6401881588E+01 ,  7.6401881588E+01 /
      data fk( 549), fk( 550) /  2.2822440307E+01 ,  7.6401881588E+01 /
      data fk( 551), fk( 552) /  9.2429542319E+00 ,  2.6324501931E+01 /
      data fk( 553), fk( 554) /  2.6324501931E+01 ,  7.6401881588E+01 /
      data fk( 555), fk( 556) /  7.6401881588E+01 ,  2.2822440307E+01 /
      data fk( 557), fk( 558) /  2.6324501931E+01 ,  9.2429542319E+00 /
      data fk( 559), fk( 560) /  2.6324501931E+01 ,  2.6324501931E+01 /
      data fk( 561), fk( 562) /  2.6324501931E+01 ,  9.2429542319E+00 /
      data fk( 563), fk( 564) / -1.1513827328E+01 ,  7.5213840990E+00 /
      data fk( 565), fk( 566) / -1.0297579090E+01 , -4.3349675888E+00 /
      data fk( 567), fk( 568) / -2.4090412966E+00 , -4.3349675888E+00 /
      data fk( 569), fk( 570) / -1.0297579090E+01 , -2.4090412966E+00 /
      data fk( 571), fk( 572) /  2.1789894086E+01 ,  1.1480441398E+01 /
      data fk( 573), fk( 574) /  1.1480441398E+01 ,  7.5213840990E+00 /
      data fk( 575), fk( 576) / -1.0297579090E+01 , -2.4090412966E+00 /
      data fk( 577), fk( 578) / -4.3349675888E+00 , -4.3349675888E+00 /
      data fk( 579), fk( 580) / -2.4090412966E+00 , -1.0297579090E+01 /
      data fk( 581), fk( 582) /  1.1480441398E+01 ,  2.1789894086E+01 /
      data fk( 583), fk( 584) /  1.1480441398E+01 ,  7.5213840990E+00 /
      data fk( 585), fk( 586) / -2.4090412966E+00 , -1.0297579090E+01 /
      data fk( 587), fk( 588) / -4.3349675888E+00 , -2.4090412966E+00 /
      data fk( 589), fk( 590) / -4.3349675888E+00 , -1.0297579090E+01 /
      data fk( 591), fk( 592) /  1.1480441398E+01 ,  1.1480441398E+01 /
      data fk( 593), fk( 594) /  2.1789894086E+01 , -3.3084568301E+00 /
      data fk( 595), fk( 596) /  1.8700946016E+01 ,  1.0033761362E+00 /
      data fk( 597), fk( 598) / -3.6424086659E+01 ,  1.8700946016E+01 /
      data fk( 599), fk( 600) / -3.6424086659E+01 ,  1.0033761362E+00 /
      data fk( 601), fk( 602) / -5.4038368536E+01 , -5.4038368536E+01 /
      data fk( 603), fk( 604) /  6.0751785033E+00 ,  1.8700946016E+01 /
      data fk( 605), fk( 606) / -3.3084568301E+00 ,  1.0033761362E+00 /
      data fk( 607), fk( 608) / -3.6424086659E+01 ,  1.8700946016E+01 /
      data fk( 609), fk( 610) / -5.4038368536E+01 , -5.4038368536E+01 /
      data fk( 611), fk( 612) / -3.6424086659E+01 ,  1.0033761362E+00 /
      data fk( 613), fk( 614) /  6.0751785033E+00 ,  1.8700946016E+01 /
      data fk( 615), fk( 616) /  1.8700946016E+01 , -5.4038368536E+01 /
      data fk( 617), fk( 618) / -5.4038368536E+01 , -3.3084568301E+00 /
      data fk( 619), fk( 620) /  1.0033761362E+00 , -3.6424086659E+01 /
      data fk( 621), fk( 622) / -3.6424086659E+01 ,  1.0033761362E+00 /
      data fk( 623), fk( 624) /  6.0751785033E+00 ,  1.2308959430E+01 /
      data fk( 625), fk( 626) / -1.8945156224E+01 , -2.9173088405E+00 /
      data fk( 627), fk( 628) / -3.8890949159E+00 ,  1.2308959430E+01 /
      data fk( 629), fk( 630) / -2.9173088405E+00 , -1.8945156224E+01 /
      data fk( 631), fk( 632) / -3.8890949159E+00 ,  9.4412448671E-01 /
      data fk( 633), fk( 634) /  1.7170567040E+01 , -2.7359749688E+01 /
      data fk( 635), fk( 636) / -3.5016299450E+01 ,  1.7170567040E+01 /
      data fk( 637), fk( 638) /  9.4412448671E-01 , -2.7359749688E+01 /
      data fk( 639), fk( 640) / -3.5016299450E+01 ,  1.0271110266E+01 /
      data fk( 641), fk( 642) /  1.0271110266E+01 , -4.0234015144E+01 /
      data fk( 643), fk( 644) / -4.0234015144E+01 ,  1.2308959430E+01 /
      data fk( 645), fk( 646) / -1.8945156224E+01 , -3.8890949159E+00 /
      data fk( 647), fk( 648) / -2.9173088405E+00 ,  1.2308959430E+01 /
      data fk( 649), fk( 650) / -2.9173088405E+00 , -3.8890949159E+00 /
      data fk( 651), fk( 652) / -1.8945156224E+01 ,  9.4412448671E-01 /
      data fk( 653), fk( 654) /  1.7170567040E+01 , -3.5016299450E+01 /
      data fk( 655), fk( 656) / -2.7359749688E+01 ,  1.0271110266E+01 /
      data fk( 657), fk( 658) /  1.0271110266E+01 , -4.0234015144E+01 /
      data fk( 659), fk( 660) / -4.0234015144E+01 ,  1.7170567040E+01 /
      data fk( 661), fk( 662) /  9.4412448671E-01 , -2.7359749688E+01 /
      data fk( 663), fk( 664) / -3.5016299450E+01 ,  1.2308959430E+01 /
      data fk( 665), fk( 666) / -3.8890949159E+00 , -1.8945156224E+01 /
      data fk( 667), fk( 668) / -2.9173088405E+00 ,  1.2308959430E+01 /
      data fk( 669), fk( 670) / -3.8890949159E+00 , -2.9173088405E+00 /
      data fk( 671), fk( 672) / -1.8945156224E+01 ,  1.0271110266E+01 /
      data fk( 673), fk( 674) /  1.0271110266E+01 , -4.0234015144E+01 /
      data fk( 675), fk( 676) / -4.0234015144E+01 ,  9.4412448671E-01 /
      data fk( 677), fk( 678) /  1.7170567040E+01 , -3.5016299450E+01 /
      data fk( 679), fk( 680) / -2.7359749688E+01 ,  1.7170567040E+01 /
      data fk( 681), fk( 682) /  9.4412448671E-01 , -3.5016299450E+01 /
      data fk( 683), fk( 684) / -2.7359749688E+01 , -1.7803662588E+01 /
      data fk( 685), fk( 686) / -3.9999644918E+01 ,  4.4512714473E+01 /
      data fk( 687), fk( 688) /  5.5199602343E+01 , -1.7803662588E+01 /
      data fk( 689), fk( 690) / -3.9999644918E+01 ,  5.5199602343E+01 /
      data fk( 691), fk( 692) /  4.4512714473E+01 , -2.1651989245E+01 /
      data fk( 693), fk( 694) / -2.1651989245E+01 ,  6.7591811571E+01 /
      data fk( 695), fk( 696) /  6.7591811571E+01 , -2.0369116533E+01 /
      data fk( 697), fk( 698) / -6.7010320763E+01 , -7.4145806286E+01 /
      data fk( 699), fk( 700) /  1.8321439114E+01 , -6.7010320763E+01 /
      data fk( 701), fk( 702) / -2.0369116533E+01 , -7.4145806286E+01 /
      data fk( 703), fk( 704) /  1.8321439114E+01 , -3.9999644918E+01 /
      data fk( 705), fk( 706) / -1.7803662588E+01 ,  4.4512714473E+01 /
      data fk( 707), fk( 708) /  5.5199602343E+01 , -2.1651989245E+01 /
      data fk( 709), fk( 710) / -2.1651989245E+01 ,  6.7591811571E+01 /
      data fk( 711), fk( 712) /  6.7591811571E+01 , -1.7803662588E+01 /
      data fk( 713), fk( 714) / -3.9999644918E+01 ,  5.5199602343E+01 /
      data fk( 715), fk( 716) /  4.4512714473E+01 , -2.0369116533E+01 /
      data fk( 717), fk( 718) / -7.4145806286E+01 , -6.7010320763E+01 /
      data fk( 719), fk( 720) /  1.8321439114E+01 , -6.7010320763E+01 /
      data fk( 721), fk( 722) / -7.4145806286E+01 , -2.0369116533E+01 /
      data fk( 723), fk( 724) /  1.8321439114E+01 , -2.1651989245E+01 /
      data fk( 725), fk( 726) / -2.1651989245E+01 ,  6.7591811571E+01 /
      data fk( 727), fk( 728) /  6.7591811571E+01 , -3.9999644918E+01 /
      data fk( 729), fk( 730) / -1.7803662588E+01 ,  4.4512714473E+01 /
      data fk( 731), fk( 732) /  5.5199602343E+01 , -3.9999644918E+01 /
      data fk( 733), fk( 734) / -1.7803662588E+01 ,  5.5199602343E+01 /
      data fk( 735), fk( 736) /  4.4512714473E+01 , -7.4145806286E+01 /
      data fk( 737), fk( 738) / -2.0369116533E+01 , -6.7010320763E+01 /
      data fk( 739), fk( 740) /  1.8321439114E+01 , -7.4145806286E+01 /
      data fk( 741), fk( 742) / -6.7010320763E+01 , -2.0369116533E+01 /
      data fk( 743), fk( 744) /  1.8321439114E+01 , -6.0110395556E+00 /
      data fk( 745), fk( 746) / -6.0110395556E+00 , -1.5550581587E+01 /
      data fk( 747), fk( 748) /  3.0198375935E+01 ,  2.6360915532E+01 /
      data fk( 749), fk( 750) /  2.4691715068E+01 ,  3.0198375935E+01 /
      data fk( 751), fk( 752) /  2.4691715068E+01 ,  2.6360915532E+01 /
      data fk( 753), fk( 754) / -7.3768443406E+01 , -6.0110395556E+00 /
      data fk( 755), fk( 756) / -1.5550581587E+01 , -6.0110395556E+00 /
      data fk( 757), fk( 758) /  2.6360915532E+01 ,  3.0198375935E+01 /
      data fk( 759), fk( 760) /  2.4691715068E+01 ,  2.4691715068E+01 /
      data fk( 761), fk( 762) /  3.0198375935E+01 ,  2.6360915532E+01 /
      data fk( 763), fk( 764) / -7.3768443406E+01 , -1.5550581587E+01 /
      data fk( 765), fk( 766) / -6.0110395556E+00 , -6.0110395556E+00 /
      data fk( 767), fk( 768) /  2.6360915532E+01 ,  2.4691715068E+01 /
      data fk( 769), fk( 770) /  3.0198375935E+01 ,  2.4691715068E+01 /
      data fk( 771), fk( 772) /  2.6360915532E+01 ,  3.0198375935E+01 /
      data fk( 773), fk( 774) / -7.3768443406E+01 , -3.7838781268E+00 /
      data fk( 775), fk( 776) /  3.9316405668E+01 ,  3.9316405668E+01 /
      data fk( 777), fk( 778) /  8.4801994799E+00 ,  9.6281817844E+01 /
      data fk( 779), fk( 780) / -7.0788971241E+01 ,  9.6281817844E+01 /
      data fk( 781), fk( 782) /  8.4801994799E+00 , -7.0788971241E+01 /
      data fk( 783), fk( 784) / -5.5294917098E+01 , -3.7838781268E+00 /
      data fk( 785), fk( 786) /  8.4801994799E+00 ,  9.6281817844E+01 /
      data fk( 787), fk( 788) /  3.9316405668E+01 ,  3.9316405668E+01 /
      data fk( 789), fk( 790) / -7.0788971241E+01 ,  9.6281817844E+01 /
      data fk( 791), fk( 792) /  8.4801994799E+00 , -5.5294917098E+01 /
      data fk( 793), fk( 794) / -7.0788971241E+01 , -3.7838781268E+00 /
      data fk( 795), fk( 796) /  8.4801994799E+00 ,  9.6281817844E+01 /
      data fk( 797), fk( 798) /  9.6281817844E+01 ,  8.4801994799E+00 /
      data fk( 799), fk( 800) / -5.5294917098E+01 ,  3.9316405668E+01 /
      data fk( 801), fk( 802) /  3.9316405668E+01 , -7.0788971241E+01 /
      data fk( 803), fk( 804) / -7.0788971241E+01 ,  6.7026672636E+00 /
      data fk( 805), fk( 806) / -5.8011277911E+01 , -5.8011277911E+01 /
      data fk( 807), fk( 808) /  9.2359591982E+01 ,  9.2359591982E+01 /
      data fk( 809), fk( 810) /  5.4522990634E+01 , -5.8011277911E+01 /
      data fk( 811), fk( 812) /  6.7026672636E+00 , -5.8011277911E+01 /
      data fk( 813), fk( 814) /  9.2359591982E+01 ,  5.4522990634E+01 /
      data fk( 815), fk( 816) /  9.2359591982E+01 ,  6.7661429816E+01 /
      data fk( 817), fk( 818) / -2.0056343944E+01 , -1.1541455261E+01 /
      data fk( 819), fk( 820) / -1.1733005015E+02 , -9.7932048436E+01 /
      data fk( 821), fk( 822) /  7.7727403890E+01 ,  6.7661429816E+01 /
      data fk( 823), fk( 824) / -1.1733005015E+02 , -9.7932048436E+01 /
      data fk( 825), fk( 826) / -2.0056343944E+01 , -1.1541455261E+01 /
      data fk( 827), fk( 828) /  7.7727403890E+01 ,  8.8253682151E+00 /
      data fk( 829), fk( 830) / -1.9863976778E+01 , -7.7977362549E+01 /
      data fk( 831), fk( 832) / -7.7977362549E+01 , -1.9863976778E+01 /
      data fk( 833), fk( 834) /  9.7632173833E+01 , -5.8011277911E+01 /
      data fk( 835), fk( 836) / -5.8011277911E+01 ,  6.7026672636E+00 /
      data fk( 837), fk( 838) /  5.4522990634E+01 ,  9.2359591982E+01 /
      data fk( 839), fk( 840) /  9.2359591982E+01 ,  6.7661429816E+01 /
      data fk( 841), fk( 842) / -1.1541455261E+01 , -2.0056343944E+01 /
      data fk( 843), fk( 844) / -9.7932048436E+01 , -1.1733005015E+02 /
      data fk( 845), fk( 846) /  7.7727403890E+01 ,  8.8253682151E+00 /
      data fk( 847), fk( 848) / -1.9863976778E+01 , -7.7977362549E+01 /
      data fk( 849), fk( 850) / -7.7977362549E+01 , -1.9863976778E+01 /
      data fk( 851), fk( 852) /  9.7632173833E+01 ,  6.7661429816E+01 /
      data fk( 853), fk( 854) / -1.1733005015E+02 , -9.7932048436E+01 /
      data fk( 855), fk( 856) / -2.0056343944E+01 , -1.1541455261E+01 /
      data fk( 857), fk( 858) /  7.7727403890E+01 ,  8.8253682151E+00 /
      data fk( 859), fk( 860) / -1.9863976778E+01 , -7.7977362549E+01 /
      data fk( 861), fk( 862) / -7.7977362549E+01 , -1.9863976778E+01 /
      data fk( 863), fk( 864) /  9.7632173833E+01 ,  6.7661429816E+01 /
      data fk( 865), fk( 866) / -1.1541455261E+01 , -2.0056343944E+01 /
      data fk( 867), fk( 868) / -9.7932048436E+01 , -1.1733005015E+02 /
      data fk( 869), fk( 870) /  7.7727403890E+01 ,  6.7661429816E+01 /
      data fk( 871), fk( 872) / -9.7932048436E+01 , -1.1733005015E+02 /
      data fk( 873), fk( 874) / -1.1541455261E+01 , -2.0056343944E+01 /
      data fk( 875), fk( 876) /  7.7727403890E+01 ,  4.2393859188E+01 /
      data fk( 877), fk( 878) /  4.2393859188E+01 , -7.7668562993E+01 /
      data fk( 879), fk( 880) /  1.4887201633E+02 , -9.1377628916E+01 /
      data fk( 881), fk( 882) / -9.1377628916E+01 ,  4.2393859188E+01 /
      data fk( 883), fk( 884) /  1.4887201633E+02 , -9.1377628916E+01 /
      data fk( 885), fk( 886) /  4.2393859188E+01 , -7.7668562993E+01 /
      data fk( 887), fk( 888) / -9.1377628916E+01 ,  1.4887201633E+02 /
      data fk( 889), fk( 890) /  4.2393859188E+01 , -9.1377628916E+01 /
      data fk( 891), fk( 892) /  4.2393859188E+01 , -9.1377628916E+01 /
      data fk( 893), fk( 894) / -7.7668562993E+01 ,  1.1611047759E+00 /
      data fk( 895), fk( 896) /  5.8579576651E+01 ,  5.8579576651E+01 /
      data fk( 897), fk( 898) / -1.3364346510E+02 , -1.3364346510E+02 /
      data fk( 899), fk( 900) /  5.8579576651E+01 ,  1.1611047759E+00 /
      data fk( 901), fk( 902) /  5.8579576651E+01 , -1.3364346510E+02 /
      data fk( 903), fk( 904) / -1.3364346510E+02 ,  5.8579576651E+01 /
      data fk( 905), fk( 906) /  5.8579576651E+01 ,  1.1611047759E+00 /
      data fk( 907), fk( 908) / -1.3364346510E+02 , -1.3364346510E+02 /
      data fk( 909), fk( 910) / -1.0598322343E+02 , -1.0598322343E+02 /
      data fk( 911), fk( 912) /  9.2284766629E+01 ,  1.8117828669E+02 /
      data fk( 913), fk( 914) /  1.8117828669E+02 , -1.0598322343E+02 /
      data fk( 915), fk( 916) / -1.0598322343E+02 ,  1.8117828669E+02 /
      data fk( 917), fk( 918) /  9.2284766629E+01 ,  1.8117828669E+02 /
      data fk( 919), fk( 920) / -1.0598322343E+02 , -1.0598322343E+02 /
      data fk( 921), fk( 922) /  1.8117828669E+02 ,  1.8117828669E+02 /
      data fk( 923), fk( 924) /  9.2284766629E+01 , -7.3373972739E+01 /

      data shift  / 2.12d0, 2.12d0, 2.12d0, 1.40d0, 1.40d0, 1.40d0 /
      data alpham / 6*0.5d0 /
      data nmax,nvar /6,6/
   
!
      en=0.0d0
      do i=1,nvar
       fmr(i,1) = 1.0d0- exp(-alpham(i)*(rij(i)-shift(i)))
       do j=2,nmax
       fmr(i,j) = fmr(i,1)**j
       enddo
      enddo
!
      en=en+fk(1) 
      en=en+fk(2)*fmr(1,1) 
      en=en+fk(3)*fmr(2,1) 
      en=en+fk(4)*fmr(3,1) 
      en=en+fk(5)*fmr(4,1) 
      en=en+fk(6)*fmr(5,1) 
      en=en+fk(7)*fmr(6,1) 
      en=en+fk(8)*fmr(1,2) 
      en=en+fk(9)*fmr(2,2) 
      en=en+fk(10)*fmr(3,2) 
      en=en+fk(11)*fmr(4,2) 
      en=en+fk(12)*fmr(5,2) 
      en=en+fk(13)*fmr(6,2) 
      en=en+fk(14)*fmr(1,1)*fmr(2,1) 
      en=en+fk(15)*fmr(1,1)*fmr(3,1) 
      en=en+fk(16)*fmr(1,1)*fmr(4,1) 
      en=en+fk(17)*fmr(1,1)*fmr(5,1) 
      en=en+fk(18)*fmr(1,1)*fmr(6,1) 
      en=en+fk(19)*fmr(2,1)*fmr(3,1) 
      en=en+fk(20)*fmr(2,1)*fmr(4,1) 
      en=en+fk(21)*fmr(2,1)*fmr(5,1) 
      en=en+fk(22)*fmr(2,1)*fmr(6,1) 
      en=en+fk(23)*fmr(3,1)*fmr(4,1) 
      en=en+fk(24)*fmr(3,1)*fmr(5,1) 
      en=en+fk(25)*fmr(3,1)*fmr(6,1) 
      en=en+fk(26)*fmr(4,1)*fmr(5,1) 
      en=en+fk(27)*fmr(4,1)*fmr(6,1) 
      en=en+fk(28)*fmr(5,1)*fmr(6,1) 
      en=en+fk(29)*fmr(1,3) 
      en=en+fk(30)*fmr(2,3) 
      en=en+fk(31)*fmr(3,3) 
      en=en+fk(32)*fmr(4,3) 
      en=en+fk(33)*fmr(5,3) 
      en=en+fk(34)*fmr(6,3) 
      en=en+fk(35)*fmr(1,2)*fmr(2,1) 
      en=en+fk(36)*fmr(1,2)*fmr(3,1) 
      en=en+fk(37)*fmr(1,2)*fmr(4,1) 
      en=en+fk(38)*fmr(1,2)*fmr(5,1) 
      en=en+fk(39)*fmr(1,2)*fmr(6,1) 
      en=en+fk(40)*fmr(1,1)*fmr(2,2) 
      en=en+fk(41)*fmr(2,2)*fmr(3,1) 
      en=en+fk(42)*fmr(2,2)*fmr(4,1) 
      en=en+fk(43)*fmr(2,2)*fmr(5,1) 
      en=en+fk(44)*fmr(2,2)*fmr(6,1) 
      en=en+fk(45)*fmr(1,1)*fmr(3,2) 
      en=en+fk(46)*fmr(2,1)*fmr(3,2) 
      en=en+fk(47)*fmr(3,2)*fmr(4,1) 
      en=en+fk(48)*fmr(3,2)*fmr(5,1) 
      en=en+fk(49)*fmr(3,2)*fmr(6,1) 
      en=en+fk(50)*fmr(1,1)*fmr(4,2) 
      en=en+fk(51)*fmr(2,1)*fmr(4,2) 
      en=en+fk(52)*fmr(3,1)*fmr(4,2) 
      en=en+fk(53)*fmr(4,2)*fmr(5,1) 
      en=en+fk(54)*fmr(4,2)*fmr(6,1) 
      en=en+fk(55)*fmr(1,1)*fmr(5,2) 
      en=en+fk(56)*fmr(2,1)*fmr(5,2) 
      en=en+fk(57)*fmr(3,1)*fmr(5,2) 
      en=en+fk(58)*fmr(4,1)*fmr(5,2) 
      en=en+fk(59)*fmr(5,2)*fmr(6,1) 
      en=en+fk(60)*fmr(1,1)*fmr(6,2) 
      en=en+fk(61)*fmr(2,1)*fmr(6,2) 
      en=en+fk(62)*fmr(3,1)*fmr(6,2) 
      en=en+fk(63)*fmr(4,1)*fmr(6,2) 
      en=en+fk(64)*fmr(5,1)*fmr(6,2) 
      en=en+fk(65)*fmr(1,1)*fmr(2,1)*fmr(3,1) 
      en=en+fk(66)*fmr(1,1)*fmr(2,1)*fmr(4,1) 
      en=en+fk(67)*fmr(1,1)*fmr(2,1)*fmr(5,1) 
      en=en+fk(68)*fmr(1,1)*fmr(2,1)*fmr(6,1) 
      en=en+fk(69)*fmr(1,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(70)*fmr(1,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(71)*fmr(1,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(72)*fmr(1,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(73)*fmr(1,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(74)*fmr(1,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(75)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(76)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(77)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(78)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(79)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(80)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(81)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(82)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(83)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(84)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(85)*fmr(1,4) 
      en=en+fk(86)*fmr(2,4) 
      en=en+fk(87)*fmr(3,4) 
      en=en+fk(88)*fmr(4,4) 
      en=en+fk(89)*fmr(5,4) 
      en=en+fk(90)*fmr(6,4) 
      en=en+fk(91)*fmr(1,3)*fmr(2,1) 
      en=en+fk(92)*fmr(1,3)*fmr(3,1) 
      en=en+fk(93)*fmr(1,3)*fmr(4,1) 
      en=en+fk(94)*fmr(1,3)*fmr(5,1) 
      en=en+fk(95)*fmr(1,3)*fmr(6,1) 
      en=en+fk(96)*fmr(1,1)*fmr(2,3) 
      en=en+fk(97)*fmr(2,3)*fmr(3,1) 
      en=en+fk(98)*fmr(2,3)*fmr(4,1) 
      en=en+fk(99)*fmr(2,3)*fmr(5,1) 
      en=en+fk(100)*fmr(2,3)*fmr(6,1) 
      en=en+fk(101)*fmr(1,1)*fmr(3,3) 
      en=en+fk(102)*fmr(2,1)*fmr(3,3) 
      en=en+fk(103)*fmr(3,3)*fmr(4,1) 
      en=en+fk(104)*fmr(3,3)*fmr(5,1) 
      en=en+fk(105)*fmr(3,3)*fmr(6,1) 
      en=en+fk(106)*fmr(1,1)*fmr(4,3) 
      en=en+fk(107)*fmr(2,1)*fmr(4,3) 
      en=en+fk(108)*fmr(3,1)*fmr(4,3) 
      en=en+fk(109)*fmr(4,3)*fmr(5,1) 
      en=en+fk(110)*fmr(4,3)*fmr(6,1) 
      en=en+fk(111)*fmr(1,1)*fmr(5,3) 
      en=en+fk(112)*fmr(2,1)*fmr(5,3) 
      en=en+fk(113)*fmr(3,1)*fmr(5,3) 
      en=en+fk(114)*fmr(4,1)*fmr(5,3) 
      en=en+fk(115)*fmr(5,3)*fmr(6,1) 
      en=en+fk(116)*fmr(1,1)*fmr(6,3) 
      en=en+fk(117)*fmr(2,1)*fmr(6,3) 
      en=en+fk(118)*fmr(3,1)*fmr(6,3) 
      en=en+fk(119)*fmr(4,1)*fmr(6,3) 
      en=en+fk(120)*fmr(5,1)*fmr(6,3) 
      en=en+fk(121)*fmr(1,2)*fmr(2,2) 
      en=en+fk(122)*fmr(1,2)*fmr(3,2) 
      en=en+fk(123)*fmr(1,2)*fmr(4,2) 
      en=en+fk(124)*fmr(1,2)*fmr(5,2) 
      en=en+fk(125)*fmr(1,2)*fmr(6,2) 
      en=en+fk(126)*fmr(2,2)*fmr(3,2) 
      en=en+fk(127)*fmr(2,2)*fmr(4,2) 
      en=en+fk(128)*fmr(2,2)*fmr(5,2) 
      en=en+fk(129)*fmr(2,2)*fmr(6,2) 
      en=en+fk(130)*fmr(3,2)*fmr(4,2) 
      en=en+fk(131)*fmr(3,2)*fmr(5,2) 
      en=en+fk(132)*fmr(3,2)*fmr(6,2) 
      en=en+fk(133)*fmr(4,2)*fmr(5,2) 
      en=en+fk(134)*fmr(4,2)*fmr(6,2) 
      en=en+fk(135)*fmr(5,2)*fmr(6,2) 
      en=en+fk(136)*fmr(1,2)*fmr(2,1)*fmr(3,1) 
      en=en+fk(137)*fmr(1,2)*fmr(2,1)*fmr(4,1) 
      en=en+fk(138)*fmr(1,2)*fmr(2,1)*fmr(5,1) 
      en=en+fk(139)*fmr(1,2)*fmr(2,1)*fmr(6,1) 
      en=en+fk(140)*fmr(1,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(141)*fmr(1,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(142)*fmr(1,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(143)*fmr(1,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(144)*fmr(1,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(145)*fmr(1,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(146)*fmr(1,1)*fmr(2,2)*fmr(3,1) 
      en=en+fk(147)*fmr(1,1)*fmr(2,2)*fmr(4,1) 
      en=en+fk(148)*fmr(1,1)*fmr(2,2)*fmr(5,1) 
      en=en+fk(149)*fmr(1,1)*fmr(2,2)*fmr(6,1) 
      en=en+fk(150)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(151)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(152)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(153)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(154)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(155)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(156)*fmr(1,1)*fmr(2,1)*fmr(3,2) 
      en=en+fk(157)*fmr(1,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(158)*fmr(1,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(159)*fmr(1,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(160)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(161)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(162)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(163)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(164)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(165)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(166)*fmr(1,1)*fmr(2,1)*fmr(4,2) 
      en=en+fk(167)*fmr(1,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(168)*fmr(1,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(169)*fmr(1,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(170)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(171)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(172)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(173)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(174)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(175)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(176)*fmr(1,1)*fmr(2,1)*fmr(5,2) 
      en=en+fk(177)*fmr(1,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(178)*fmr(1,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(179)*fmr(1,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(180)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(181)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(182)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(183)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(184)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(185)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(186)*fmr(1,1)*fmr(2,1)*fmr(6,2) 
      en=en+fk(187)*fmr(1,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(188)*fmr(1,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(189)*fmr(1,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(190)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(191)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(192)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(193)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(194)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(195)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(196)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(197)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(198)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(199)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(200)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(201)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(202)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(203)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(204)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(205)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(206)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(207)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(208)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(209)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(210)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(211)*fmr(1,5) 
      en=en+fk(212)*fmr(2,5) 
      en=en+fk(213)*fmr(3,5) 
      en=en+fk(214)*fmr(4,5) 
      en=en+fk(215)*fmr(5,5) 
      en=en+fk(216)*fmr(6,5) 
      en=en+fk(217)*fmr(1,4)*fmr(2,1) 
      en=en+fk(218)*fmr(1,4)*fmr(3,1) 
      en=en+fk(219)*fmr(1,4)*fmr(4,1) 
      en=en+fk(220)*fmr(1,4)*fmr(5,1) 
      en=en+fk(221)*fmr(1,4)*fmr(6,1) 
      en=en+fk(222)*fmr(1,1)*fmr(2,4) 
      en=en+fk(223)*fmr(2,4)*fmr(3,1) 
      en=en+fk(224)*fmr(2,4)*fmr(4,1) 
      en=en+fk(225)*fmr(2,4)*fmr(5,1) 
      en=en+fk(226)*fmr(2,4)*fmr(6,1) 
      en=en+fk(227)*fmr(1,1)*fmr(3,4) 
      en=en+fk(228)*fmr(2,1)*fmr(3,4) 
      en=en+fk(229)*fmr(3,4)*fmr(4,1) 
      en=en+fk(230)*fmr(3,4)*fmr(5,1) 
      en=en+fk(231)*fmr(3,4)*fmr(6,1) 
      en=en+fk(232)*fmr(1,1)*fmr(4,4) 
      en=en+fk(233)*fmr(2,1)*fmr(4,4) 
      en=en+fk(234)*fmr(3,1)*fmr(4,4) 
      en=en+fk(235)*fmr(4,4)*fmr(5,1) 
      en=en+fk(236)*fmr(4,4)*fmr(6,1) 
      en=en+fk(237)*fmr(1,1)*fmr(5,4) 
      en=en+fk(238)*fmr(2,1)*fmr(5,4) 
      en=en+fk(239)*fmr(3,1)*fmr(5,4) 
      en=en+fk(240)*fmr(4,1)*fmr(5,4) 
      en=en+fk(241)*fmr(5,4)*fmr(6,1) 
      en=en+fk(242)*fmr(1,1)*fmr(6,4) 
      en=en+fk(243)*fmr(2,1)*fmr(6,4) 
      en=en+fk(244)*fmr(3,1)*fmr(6,4) 
      en=en+fk(245)*fmr(4,1)*fmr(6,4) 
      en=en+fk(246)*fmr(5,1)*fmr(6,4) 
      en=en+fk(247)*fmr(1,3)*fmr(2,2) 
      en=en+fk(248)*fmr(1,3)*fmr(3,2) 
      en=en+fk(249)*fmr(1,3)*fmr(4,2) 
      en=en+fk(250)*fmr(1,3)*fmr(5,2) 
      en=en+fk(251)*fmr(1,3)*fmr(6,2) 
      en=en+fk(252)*fmr(1,2)*fmr(2,3) 
      en=en+fk(253)*fmr(2,3)*fmr(3,2) 
      en=en+fk(254)*fmr(2,3)*fmr(4,2) 
      en=en+fk(255)*fmr(2,3)*fmr(5,2) 
      en=en+fk(256)*fmr(2,3)*fmr(6,2) 
      en=en+fk(257)*fmr(1,2)*fmr(3,3) 
      en=en+fk(258)*fmr(2,2)*fmr(3,3) 
      en=en+fk(259)*fmr(3,3)*fmr(4,2) 
      en=en+fk(260)*fmr(3,3)*fmr(5,2) 
      en=en+fk(261)*fmr(3,3)*fmr(6,2) 
      en=en+fk(262)*fmr(1,2)*fmr(4,3) 
      en=en+fk(263)*fmr(2,2)*fmr(4,3) 
      en=en+fk(264)*fmr(3,2)*fmr(4,3) 
      en=en+fk(265)*fmr(4,3)*fmr(5,2) 
      en=en+fk(266)*fmr(4,3)*fmr(6,2) 
      en=en+fk(267)*fmr(1,2)*fmr(5,3) 
      en=en+fk(268)*fmr(2,2)*fmr(5,3) 
      en=en+fk(269)*fmr(3,2)*fmr(5,3) 
      en=en+fk(270)*fmr(4,2)*fmr(5,3) 
      en=en+fk(271)*fmr(5,3)*fmr(6,2) 
      en=en+fk(272)*fmr(1,2)*fmr(6,3) 
      en=en+fk(273)*fmr(2,2)*fmr(6,3) 
      en=en+fk(274)*fmr(3,2)*fmr(6,3) 
      en=en+fk(275)*fmr(4,2)*fmr(6,3) 
      en=en+fk(276)*fmr(5,2)*fmr(6,3) 
      en=en+fk(277)*fmr(1,3)*fmr(2,1)*fmr(3,1) 
      en=en+fk(278)*fmr(1,3)*fmr(2,1)*fmr(4,1) 
      en=en+fk(279)*fmr(1,3)*fmr(2,1)*fmr(5,1) 
      en=en+fk(280)*fmr(1,3)*fmr(2,1)*fmr(6,1) 
      en=en+fk(281)*fmr(1,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(282)*fmr(1,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(283)*fmr(1,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(284)*fmr(1,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(285)*fmr(1,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(286)*fmr(1,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(287)*fmr(1,1)*fmr(2,3)*fmr(3,1) 
      en=en+fk(288)*fmr(1,1)*fmr(2,3)*fmr(4,1) 
      en=en+fk(289)*fmr(1,1)*fmr(2,3)*fmr(5,1) 
      en=en+fk(290)*fmr(1,1)*fmr(2,3)*fmr(6,1) 
      en=en+fk(291)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(292)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(293)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(294)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(295)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(296)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(297)*fmr(1,1)*fmr(2,1)*fmr(3,3) 
      en=en+fk(298)*fmr(1,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(299)*fmr(1,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(300)*fmr(1,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(301)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(302)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(303)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(304)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(305)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(306)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(307)*fmr(1,1)*fmr(2,1)*fmr(4,3) 
      en=en+fk(308)*fmr(1,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(309)*fmr(1,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(310)*fmr(1,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(311)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(312)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(313)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(314)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(315)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(316)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(317)*fmr(1,1)*fmr(2,1)*fmr(5,3) 
      en=en+fk(318)*fmr(1,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(319)*fmr(1,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(320)*fmr(1,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(321)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(322)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(323)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(324)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(325)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(326)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(327)*fmr(1,1)*fmr(2,1)*fmr(6,3) 
      en=en+fk(328)*fmr(1,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(329)*fmr(1,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(330)*fmr(1,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(331)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(332)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(333)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(334)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(335)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(336)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(337)*fmr(1,1)*fmr(2,2)*fmr(3,2) 
      en=en+fk(338)*fmr(1,1)*fmr(2,2)*fmr(4,2) 
      en=en+fk(339)*fmr(1,1)*fmr(2,2)*fmr(5,2) 
      en=en+fk(340)*fmr(1,1)*fmr(2,2)*fmr(6,2) 
      en=en+fk(341)*fmr(1,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(342)*fmr(1,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(343)*fmr(1,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(344)*fmr(1,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(345)*fmr(1,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(346)*fmr(1,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(347)*fmr(1,2)*fmr(2,1)*fmr(3,2) 
      en=en+fk(348)*fmr(1,2)*fmr(2,1)*fmr(4,2) 
      en=en+fk(349)*fmr(1,2)*fmr(2,1)*fmr(5,2) 
      en=en+fk(350)*fmr(1,2)*fmr(2,1)*fmr(6,2) 
      en=en+fk(351)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(352)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(353)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(354)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(355)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(356)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(357)*fmr(1,2)*fmr(2,2)*fmr(3,1) 
      en=en+fk(358)*fmr(1,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(359)*fmr(1,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(360)*fmr(1,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(361)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(362)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(363)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(364)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(365)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(366)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(367)*fmr(1,2)*fmr(2,2)*fmr(4,1) 
      en=en+fk(368)*fmr(1,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(369)*fmr(1,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(370)*fmr(1,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(371)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(372)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(373)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(374)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(375)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(376)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(377)*fmr(1,2)*fmr(2,2)*fmr(5,1) 
      en=en+fk(378)*fmr(1,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(379)*fmr(1,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(380)*fmr(1,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(381)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(382)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(383)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(384)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(385)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(386)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(387)*fmr(1,2)*fmr(2,2)*fmr(6,1) 
      en=en+fk(388)*fmr(1,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(389)*fmr(1,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(390)*fmr(1,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(391)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(392)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(393)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(394)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(395)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(396)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(397)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(398)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(399)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(400)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(401)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(402)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(403)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(404)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(405)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(406)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(407)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(408)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(409)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(410)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(411)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(412)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(413)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(414)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(415)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(416)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(417)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(418)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(419)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(420)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(421)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(422)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(423)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(424)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(425)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(426)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(427)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(428)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(429)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(430)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(431)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(432)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(433)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(434)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(435)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(436)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(437)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(438)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(439)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(440)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(441)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(442)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(443)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(444)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(445)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(446)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(447)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(448)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(449)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(450)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(451)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(452)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(453)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(454)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(455)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(456)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(457)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(458)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(459)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(460)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(461)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(462)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(463)*fmr(1,6) 
      en=en+fk(464)*fmr(2,6) 
      en=en+fk(465)*fmr(3,6) 
      en=en+fk(466)*fmr(4,6) 
      en=en+fk(467)*fmr(5,6) 
      en=en+fk(468)*fmr(6,6) 
      en=en+fk(469)*fmr(1,5)*fmr(2,1) 
      en=en+fk(470)*fmr(1,5)*fmr(3,1) 
      en=en+fk(471)*fmr(1,5)*fmr(4,1) 
      en=en+fk(472)*fmr(1,5)*fmr(5,1) 
      en=en+fk(473)*fmr(1,5)*fmr(6,1) 
      en=en+fk(474)*fmr(1,1)*fmr(2,5) 
      en=en+fk(475)*fmr(2,5)*fmr(3,1) 
      en=en+fk(476)*fmr(2,5)*fmr(4,1) 
      en=en+fk(477)*fmr(2,5)*fmr(5,1) 
      en=en+fk(478)*fmr(2,5)*fmr(6,1) 
      en=en+fk(479)*fmr(1,1)*fmr(3,5) 
      en=en+fk(480)*fmr(2,1)*fmr(3,5) 
      en=en+fk(481)*fmr(3,5)*fmr(4,1) 
      en=en+fk(482)*fmr(3,5)*fmr(5,1) 
      en=en+fk(483)*fmr(3,5)*fmr(6,1) 
      en=en+fk(484)*fmr(1,1)*fmr(4,5) 
      en=en+fk(485)*fmr(2,1)*fmr(4,5) 
      en=en+fk(486)*fmr(3,1)*fmr(4,5) 
      en=en+fk(487)*fmr(4,5)*fmr(5,1) 
      en=en+fk(488)*fmr(4,5)*fmr(6,1) 
      en=en+fk(489)*fmr(1,1)*fmr(5,5) 
      en=en+fk(490)*fmr(2,1)*fmr(5,5) 
      en=en+fk(491)*fmr(3,1)*fmr(5,5) 
      en=en+fk(492)*fmr(4,1)*fmr(5,5) 
      en=en+fk(493)*fmr(5,5)*fmr(6,1) 
      en=en+fk(494)*fmr(1,1)*fmr(6,5) 
      en=en+fk(495)*fmr(2,1)*fmr(6,5) 
      en=en+fk(496)*fmr(3,1)*fmr(6,5) 
      en=en+fk(497)*fmr(4,1)*fmr(6,5) 
      en=en+fk(498)*fmr(5,1)*fmr(6,5) 
      en=en+fk(499)*fmr(1,4)*fmr(2,2) 
      en=en+fk(500)*fmr(1,4)*fmr(3,2) 
      en=en+fk(501)*fmr(1,4)*fmr(4,2) 
      en=en+fk(502)*fmr(1,4)*fmr(5,2) 
      en=en+fk(503)*fmr(1,4)*fmr(6,2) 
      en=en+fk(504)*fmr(1,2)*fmr(2,4) 
      en=en+fk(505)*fmr(2,4)*fmr(3,2) 
      en=en+fk(506)*fmr(2,4)*fmr(4,2) 
      en=en+fk(507)*fmr(2,4)*fmr(5,2) 
      en=en+fk(508)*fmr(2,4)*fmr(6,2) 
      en=en+fk(509)*fmr(1,2)*fmr(3,4) 
      en=en+fk(510)*fmr(2,2)*fmr(3,4) 
      en=en+fk(511)*fmr(3,4)*fmr(4,2) 
      en=en+fk(512)*fmr(3,4)*fmr(5,2) 
      en=en+fk(513)*fmr(3,4)*fmr(6,2) 
      en=en+fk(514)*fmr(1,2)*fmr(4,4) 
      en=en+fk(515)*fmr(2,2)*fmr(4,4) 
      en=en+fk(516)*fmr(3,2)*fmr(4,4) 
      en=en+fk(517)*fmr(4,4)*fmr(5,2) 
      en=en+fk(518)*fmr(4,4)*fmr(6,2) 
      en=en+fk(519)*fmr(1,2)*fmr(5,4) 
      en=en+fk(520)*fmr(2,2)*fmr(5,4) 
      en=en+fk(521)*fmr(3,2)*fmr(5,4) 
      en=en+fk(522)*fmr(4,2)*fmr(5,4) 
      en=en+fk(523)*fmr(5,4)*fmr(6,2) 
      en=en+fk(524)*fmr(1,2)*fmr(6,4) 
      en=en+fk(525)*fmr(2,2)*fmr(6,4) 
      en=en+fk(526)*fmr(3,2)*fmr(6,4) 
      en=en+fk(527)*fmr(4,2)*fmr(6,4) 
      en=en+fk(528)*fmr(5,2)*fmr(6,4) 
      en=en+fk(529)*fmr(1,3)*fmr(2,3) 
      en=en+fk(530)*fmr(1,3)*fmr(3,3) 
      en=en+fk(531)*fmr(1,3)*fmr(4,3) 
      en=en+fk(532)*fmr(1,3)*fmr(5,3) 
      en=en+fk(533)*fmr(1,3)*fmr(6,3) 
      en=en+fk(534)*fmr(2,3)*fmr(3,3) 
      en=en+fk(535)*fmr(2,3)*fmr(4,3) 
      en=en+fk(536)*fmr(2,3)*fmr(5,3) 
      en=en+fk(537)*fmr(2,3)*fmr(6,3) 
      en=en+fk(538)*fmr(3,3)*fmr(4,3) 
      en=en+fk(539)*fmr(3,3)*fmr(5,3) 
      en=en+fk(540)*fmr(3,3)*fmr(6,3) 
      en=en+fk(541)*fmr(4,3)*fmr(5,3) 
      en=en+fk(542)*fmr(4,3)*fmr(6,3) 
      en=en+fk(543)*fmr(5,3)*fmr(6,3) 
      en=en+fk(544)*fmr(4,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(545)*fmr(3,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(546)*fmr(3,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(547)*fmr(3,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(548)*fmr(2,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(549)*fmr(2,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(550)*fmr(2,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(551)*fmr(2,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(552)*fmr(2,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(553)*fmr(2,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(554)*fmr(1,2)*fmr(5,2)*fmr(6,2) 
      en=en+fk(555)*fmr(1,2)*fmr(4,2)*fmr(6,2) 
      en=en+fk(556)*fmr(1,2)*fmr(4,2)*fmr(5,2) 
      en=en+fk(557)*fmr(1,2)*fmr(3,2)*fmr(6,2) 
      en=en+fk(558)*fmr(1,2)*fmr(3,2)*fmr(5,2) 
      en=en+fk(559)*fmr(1,2)*fmr(3,2)*fmr(4,2) 
      en=en+fk(560)*fmr(1,2)*fmr(2,2)*fmr(6,2) 
      en=en+fk(561)*fmr(1,2)*fmr(2,2)*fmr(5,2) 
      en=en+fk(562)*fmr(1,2)*fmr(2,2)*fmr(4,2) 
      en=en+fk(563)*fmr(1,2)*fmr(2,2)*fmr(3,2) 
      en=en+fk(564)*fmr(1,4)*fmr(2,1)*fmr(3,1) 
      en=en+fk(565)*fmr(1,4)*fmr(2,1)*fmr(4,1) 
      en=en+fk(566)*fmr(1,4)*fmr(2,1)*fmr(5,1) 
      en=en+fk(567)*fmr(1,4)*fmr(2,1)*fmr(6,1) 
      en=en+fk(568)*fmr(1,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(569)*fmr(1,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(570)*fmr(1,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(571)*fmr(1,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(572)*fmr(1,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(573)*fmr(1,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(574)*fmr(1,1)*fmr(2,4)*fmr(3,1) 
      en=en+fk(575)*fmr(1,1)*fmr(2,4)*fmr(4,1) 
      en=en+fk(576)*fmr(1,1)*fmr(2,4)*fmr(5,1) 
      en=en+fk(577)*fmr(1,1)*fmr(2,4)*fmr(6,1) 
      en=en+fk(578)*fmr(2,4)*fmr(3,1)*fmr(4,1) 
      en=en+fk(579)*fmr(2,4)*fmr(3,1)*fmr(5,1) 
      en=en+fk(580)*fmr(2,4)*fmr(3,1)*fmr(6,1) 
      en=en+fk(581)*fmr(2,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(582)*fmr(2,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(583)*fmr(2,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(584)*fmr(1,1)*fmr(2,1)*fmr(3,4) 
      en=en+fk(585)*fmr(1,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(586)*fmr(1,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(587)*fmr(1,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(588)*fmr(2,1)*fmr(3,4)*fmr(4,1) 
      en=en+fk(589)*fmr(2,1)*fmr(3,4)*fmr(5,1) 
      en=en+fk(590)*fmr(2,1)*fmr(3,4)*fmr(6,1) 
      en=en+fk(591)*fmr(3,4)*fmr(4,1)*fmr(5,1) 
      en=en+fk(592)*fmr(3,4)*fmr(4,1)*fmr(6,1) 
      en=en+fk(593)*fmr(3,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(594)*fmr(1,1)*fmr(2,1)*fmr(4,4) 
      en=en+fk(595)*fmr(1,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(596)*fmr(1,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(597)*fmr(1,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(598)*fmr(2,1)*fmr(3,1)*fmr(4,4) 
      en=en+fk(599)*fmr(2,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(600)*fmr(2,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(601)*fmr(3,1)*fmr(4,4)*fmr(5,1) 
      en=en+fk(602)*fmr(3,1)*fmr(4,4)*fmr(6,1) 
      en=en+fk(603)*fmr(4,4)*fmr(5,1)*fmr(6,1) 
      en=en+fk(604)*fmr(1,1)*fmr(2,1)*fmr(5,4) 
      en=en+fk(605)*fmr(1,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(606)*fmr(1,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(607)*fmr(1,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(608)*fmr(2,1)*fmr(3,1)*fmr(5,4) 
      en=en+fk(609)*fmr(2,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(610)*fmr(2,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(611)*fmr(3,1)*fmr(4,1)*fmr(5,4) 
      en=en+fk(612)*fmr(3,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(613)*fmr(4,1)*fmr(5,4)*fmr(6,1) 
      en=en+fk(614)*fmr(1,1)*fmr(2,1)*fmr(6,4) 
      en=en+fk(615)*fmr(1,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(616)*fmr(1,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(617)*fmr(1,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(618)*fmr(2,1)*fmr(3,1)*fmr(6,4) 
      en=en+fk(619)*fmr(2,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(620)*fmr(2,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(621)*fmr(3,1)*fmr(4,1)*fmr(6,4) 
      en=en+fk(622)*fmr(3,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(623)*fmr(4,1)*fmr(5,1)*fmr(6,4) 
      en=en+fk(624)*fmr(1,3)*fmr(2,2)*fmr(3,1) 
      en=en+fk(625)*fmr(1,3)*fmr(2,2)*fmr(4,1) 
      en=en+fk(626)*fmr(1,3)*fmr(2,2)*fmr(5,1) 
      en=en+fk(627)*fmr(1,3)*fmr(2,2)*fmr(6,1) 
      en=en+fk(628)*fmr(1,3)*fmr(2,1)*fmr(3,2) 
      en=en+fk(629)*fmr(1,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(630)*fmr(1,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(631)*fmr(1,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(632)*fmr(1,3)*fmr(2,1)*fmr(4,2) 
      en=en+fk(633)*fmr(1,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(634)*fmr(1,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(635)*fmr(1,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(636)*fmr(1,3)*fmr(2,1)*fmr(5,2) 
      en=en+fk(637)*fmr(1,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(638)*fmr(1,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(639)*fmr(1,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(640)*fmr(1,3)*fmr(2,1)*fmr(6,2) 
      en=en+fk(641)*fmr(1,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(642)*fmr(1,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(643)*fmr(1,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(644)*fmr(1,2)*fmr(2,3)*fmr(3,1) 
      en=en+fk(645)*fmr(1,2)*fmr(2,3)*fmr(4,1) 
      en=en+fk(646)*fmr(1,2)*fmr(2,3)*fmr(5,1) 
      en=en+fk(647)*fmr(1,2)*fmr(2,3)*fmr(6,1) 
      en=en+fk(648)*fmr(1,1)*fmr(2,3)*fmr(3,2) 
      en=en+fk(649)*fmr(2,3)*fmr(3,2)*fmr(4,1) 
      en=en+fk(650)*fmr(2,3)*fmr(3,2)*fmr(5,1) 
      en=en+fk(651)*fmr(2,3)*fmr(3,2)*fmr(6,1) 
      en=en+fk(652)*fmr(1,1)*fmr(2,3)*fmr(4,2) 
      en=en+fk(653)*fmr(2,3)*fmr(3,1)*fmr(4,2) 
      en=en+fk(654)*fmr(2,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(655)*fmr(2,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(656)*fmr(1,1)*fmr(2,3)*fmr(5,2) 
      en=en+fk(657)*fmr(2,3)*fmr(3,1)*fmr(5,2) 
      en=en+fk(658)*fmr(2,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(659)*fmr(2,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(660)*fmr(1,1)*fmr(2,3)*fmr(6,2) 
      en=en+fk(661)*fmr(2,3)*fmr(3,1)*fmr(6,2) 
      en=en+fk(662)*fmr(2,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(663)*fmr(2,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(664)*fmr(1,2)*fmr(2,1)*fmr(3,3) 
      en=en+fk(665)*fmr(1,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(666)*fmr(1,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(667)*fmr(1,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(668)*fmr(1,1)*fmr(2,2)*fmr(3,3) 
      en=en+fk(669)*fmr(2,2)*fmr(3,3)*fmr(4,1) 
      en=en+fk(670)*fmr(2,2)*fmr(3,3)*fmr(5,1) 
      en=en+fk(671)*fmr(2,2)*fmr(3,3)*fmr(6,1) 
      en=en+fk(672)*fmr(1,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(673)*fmr(2,1)*fmr(3,3)*fmr(4,2) 
      en=en+fk(674)*fmr(3,3)*fmr(4,2)*fmr(5,1) 
      en=en+fk(675)*fmr(3,3)*fmr(4,2)*fmr(6,1) 
      en=en+fk(676)*fmr(1,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(677)*fmr(2,1)*fmr(3,3)*fmr(5,2) 
      en=en+fk(678)*fmr(3,3)*fmr(4,1)*fmr(5,2) 
      en=en+fk(679)*fmr(3,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(680)*fmr(1,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(681)*fmr(2,1)*fmr(3,3)*fmr(6,2) 
      en=en+fk(682)*fmr(3,3)*fmr(4,1)*fmr(6,2) 
      en=en+fk(683)*fmr(3,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(684)*fmr(1,2)*fmr(2,1)*fmr(4,3) 
      en=en+fk(685)*fmr(1,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(686)*fmr(1,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(687)*fmr(1,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(688)*fmr(1,1)*fmr(2,2)*fmr(4,3) 
      en=en+fk(689)*fmr(2,2)*fmr(3,1)*fmr(4,3) 
      en=en+fk(690)*fmr(2,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(691)*fmr(2,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(692)*fmr(1,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(693)*fmr(2,1)*fmr(3,2)*fmr(4,3) 
      en=en+fk(694)*fmr(3,2)*fmr(4,3)*fmr(5,1) 
      en=en+fk(695)*fmr(3,2)*fmr(4,3)*fmr(6,1) 
      en=en+fk(696)*fmr(1,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(697)*fmr(2,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(698)*fmr(3,1)*fmr(4,3)*fmr(5,2) 
      en=en+fk(699)*fmr(4,3)*fmr(5,2)*fmr(6,1) 
      en=en+fk(700)*fmr(1,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(701)*fmr(2,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(702)*fmr(3,1)*fmr(4,3)*fmr(6,2) 
      en=en+fk(703)*fmr(4,3)*fmr(5,1)*fmr(6,2) 
      en=en+fk(704)*fmr(1,2)*fmr(2,1)*fmr(5,3) 
      en=en+fk(705)*fmr(1,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(706)*fmr(1,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(707)*fmr(1,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(708)*fmr(1,1)*fmr(2,2)*fmr(5,3) 
      en=en+fk(709)*fmr(2,2)*fmr(3,1)*fmr(5,3) 
      en=en+fk(710)*fmr(2,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(711)*fmr(2,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(712)*fmr(1,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(713)*fmr(2,1)*fmr(3,2)*fmr(5,3) 
      en=en+fk(714)*fmr(3,2)*fmr(4,1)*fmr(5,3) 
      en=en+fk(715)*fmr(3,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(716)*fmr(1,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(717)*fmr(2,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(718)*fmr(3,1)*fmr(4,2)*fmr(5,3) 
      en=en+fk(719)*fmr(4,2)*fmr(5,3)*fmr(6,1) 
      en=en+fk(720)*fmr(1,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(721)*fmr(2,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(722)*fmr(3,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(723)*fmr(4,1)*fmr(5,3)*fmr(6,2) 
      en=en+fk(724)*fmr(1,2)*fmr(2,1)*fmr(6,3) 
      en=en+fk(725)*fmr(1,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(726)*fmr(1,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(727)*fmr(1,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(728)*fmr(1,1)*fmr(2,2)*fmr(6,3) 
      en=en+fk(729)*fmr(2,2)*fmr(3,1)*fmr(6,3) 
      en=en+fk(730)*fmr(2,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(731)*fmr(2,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(732)*fmr(1,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(733)*fmr(2,1)*fmr(3,2)*fmr(6,3) 
      en=en+fk(734)*fmr(3,2)*fmr(4,1)*fmr(6,3) 
      en=en+fk(735)*fmr(3,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(736)*fmr(1,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(737)*fmr(2,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(738)*fmr(3,1)*fmr(4,2)*fmr(6,3) 
      en=en+fk(739)*fmr(4,2)*fmr(5,1)*fmr(6,3) 
      en=en+fk(740)*fmr(1,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(741)*fmr(2,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(742)*fmr(3,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(743)*fmr(4,1)*fmr(5,2)*fmr(6,3) 
      en=en+fk(744)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(4,1) 
      en=en+fk(745)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(5,1) 
      en=en+fk(746)*fmr(1,3)*fmr(2,1)*fmr(3,1)*fmr(6,1) 
      en=en+fk(747)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(748)*fmr(1,3)*fmr(2,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(749)*fmr(1,3)*fmr(2,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(750)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(751)*fmr(1,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(752)*fmr(1,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(753)*fmr(1,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(754)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(4,1) 
      en=en+fk(755)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(5,1) 
      en=en+fk(756)*fmr(1,1)*fmr(2,3)*fmr(3,1)*fmr(6,1) 
      en=en+fk(757)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(758)*fmr(1,1)*fmr(2,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(759)*fmr(1,1)*fmr(2,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(760)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(761)*fmr(2,3)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(762)*fmr(2,3)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(763)*fmr(2,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(764)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(4,1) 
      en=en+fk(765)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(5,1) 
      en=en+fk(766)*fmr(1,1)*fmr(2,1)*fmr(3,3)*fmr(6,1) 
      en=en+fk(767)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(768)*fmr(1,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(769)*fmr(1,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(770)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(5,1) 
      en=en+fk(771)*fmr(2,1)*fmr(3,3)*fmr(4,1)*fmr(6,1) 
      en=en+fk(772)*fmr(2,1)*fmr(3,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(773)*fmr(3,3)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(774)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,3) 
      en=en+fk(775)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(776)*fmr(1,1)*fmr(2,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(777)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(778)*fmr(1,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(779)*fmr(1,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(780)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(5,1) 
      en=en+fk(781)*fmr(2,1)*fmr(3,1)*fmr(4,3)*fmr(6,1) 
      en=en+fk(782)*fmr(2,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(783)*fmr(3,1)*fmr(4,3)*fmr(5,1)*fmr(6,1) 
      en=en+fk(784)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,3) 
      en=en+fk(785)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(786)*fmr(1,1)*fmr(2,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(787)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(788)*fmr(1,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(789)*fmr(1,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(790)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,3) 
      en=en+fk(791)*fmr(2,1)*fmr(3,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(792)*fmr(2,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(793)*fmr(3,1)*fmr(4,1)*fmr(5,3)*fmr(6,1) 
      en=en+fk(794)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(6,3) 
      en=en+fk(795)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(796)*fmr(1,1)*fmr(2,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(797)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(798)*fmr(1,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(799)*fmr(1,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(800)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,3) 
      en=en+fk(801)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(802)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(803)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,3) 
      en=en+fk(804)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(4,1) 
      en=en+fk(805)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(5,1) 
      en=en+fk(806)*fmr(1,2)*fmr(2,2)*fmr(3,1)*fmr(6,1) 
      en=en+fk(807)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(808)*fmr(1,2)*fmr(2,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(809)*fmr(1,2)*fmr(2,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(810)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(4,1) 
      en=en+fk(811)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(5,1) 
      en=en+fk(812)*fmr(1,2)*fmr(2,1)*fmr(3,2)*fmr(6,1) 
      en=en+fk(813)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(814)*fmr(1,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(815)*fmr(1,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(816)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,2) 
      en=en+fk(817)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(818)*fmr(1,2)*fmr(2,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(819)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(820)*fmr(1,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(821)*fmr(1,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(822)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,2) 
      en=en+fk(823)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(824)*fmr(1,2)*fmr(2,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(825)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(826)*fmr(1,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(827)*fmr(1,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(828)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(6,2) 
      en=en+fk(829)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(830)*fmr(1,2)*fmr(2,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(831)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(832)*fmr(1,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(833)*fmr(1,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(834)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(4,1) 
      en=en+fk(835)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(5,1) 
      en=en+fk(836)*fmr(1,1)*fmr(2,2)*fmr(3,2)*fmr(6,1) 
      en=en+fk(837)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(838)*fmr(2,2)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(839)*fmr(2,2)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(840)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,2) 
      en=en+fk(841)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(842)*fmr(1,1)*fmr(2,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(843)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(844)*fmr(2,2)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(845)*fmr(2,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(846)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,2) 
      en=en+fk(847)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(848)*fmr(1,1)*fmr(2,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(849)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(850)*fmr(2,2)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(851)*fmr(2,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(852)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(6,2) 
      en=en+fk(853)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(854)*fmr(1,1)*fmr(2,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(855)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(856)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(857)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(858)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,2) 
      en=en+fk(859)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(860)*fmr(1,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(861)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(5,1) 
      en=en+fk(862)*fmr(2,1)*fmr(3,2)*fmr(4,2)*fmr(6,1) 
      en=en+fk(863)*fmr(3,2)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(864)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,2) 
      en=en+fk(865)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(866)*fmr(1,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(867)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,2) 
      en=en+fk(868)*fmr(2,1)*fmr(3,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(869)*fmr(3,2)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(870)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(6,2) 
      en=en+fk(871)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(872)*fmr(1,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(873)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,2) 
      en=en+fk(874)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(875)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(876)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(877)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(878)*fmr(1,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(879)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,2) 
      en=en+fk(880)*fmr(2,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(881)*fmr(3,1)*fmr(4,2)*fmr(5,2)*fmr(6,1) 
      en=en+fk(882)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(883)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(884)*fmr(1,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(885)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,2) 
      en=en+fk(886)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(887)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,2) 
      en=en+fk(888)*fmr(1,1)*fmr(2,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(889)*fmr(1,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(890)*fmr(1,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(891)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(892)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(893)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,2) 
      en=en+fk(894)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(895)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(896)*fmr(1,2)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(897)*fmr(1,2)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(898)*fmr(1,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(899)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1) 
      en=en+fk(900)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(6,1) 
      en=en+fk(901)*fmr(1,1)*fmr(2,2)*fmr(3,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(902)*fmr(1,1)*fmr(2,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(903)*fmr(2,2)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(904)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1) 
      en=en+fk(905)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(6,1) 
      en=en+fk(906)*fmr(1,1)*fmr(2,1)*fmr(3,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(907)*fmr(1,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(908)*fmr(2,1)*fmr(3,2)*fmr(4,1)*fmr(5,1)*fmr(6,1) 
      en=en+fk(909)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1) 
      en=en+fk(910)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(6,1) 
      en=en+fk(911)*fmr(1,1)*fmr(2,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(912)*fmr(1,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(913)*fmr(2,1)*fmr(3,1)*fmr(4,2)*fmr(5,1)*fmr(6,1) 
      en=en+fk(914)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2) 
      en=en+fk(915)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(916)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(917)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(918)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,2)*fmr(6,1) 
      en=en+fk(919)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(6,2) 
      en=en+fk(920)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(921)*fmr(1,1)*fmr(2,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(922)*fmr(1,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(923)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)*fmr(6,2) 
      en=en+fk(924)*fmr(1,1)*fmr(2,1)*fmr(3,1)*fmr(4,1)*fmr(5,1)  &
        *fmr(6,1) 

      return
      end subroutine chph2
