# -*- coding: utf-8 -*-
#
# This file is for use with essm.
# Copyright (C) 2023 ETH Zurich, Swiss Data Science Center.
#
# essm is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# essm is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with essm; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA 02111-1307, USA.
"""Variables defined in Mathematical_model.ipynb and dependencies."""

from essm.variables import Variable
from sympy.physics.units import (
    joule, kelvin, kilogram, meter, mole, pascal, second, watt
)


class alpha_a(Variable):
    """Thermal diffusivity of dry air."""

    name = 'alpha_a'
    unit = meter ** 2 / second
    assumptions = {'real': True}
    latex_name = '\\alpha_a'
    default = None


class c_pa(Variable):
    """Specific heat of dry air."""

    name = 'c_pa'
    unit = joule / (kelvin * kilogram)
    assumptions = {'real': True}
    latex_name = 'c_{pa}'
    default = 1010.0


class c_pamol(Variable):
    """Molar specific heat of dry air.

    https://en.wikipedia.org/wiki/Heat_capacity#Specific_heat_capacity
    """

    name = 'c_pamol'
    unit = joule / (kelvin * mole)
    assumptions = {'real': True}
    latex_name = 'c_{pa,mol}'
    default = 29.19


class c_pv(Variable):
    """Specific heat of water vapour at 300 K.

    http://www.engineeringtoolbox.com/water-vapor-d_979.html
    """

    name = 'c_pv'
    unit = joule / (kelvin * kilogram)
    assumptions = {'real': True}
    latex_name = 'c_{pv}'
    default = 1864


class C_wa(Variable):
    """Concentration of water in air."""

    name = 'C_wa'
    unit = mole / meter ** 3
    assumptions = {'real': True}
    latex_name = 'C_{wa}'
    default = None


class D_va(Variable):
    """Binary diffusion coefficient of water vapour in air."""

    name = 'D_va'
    unit = meter ** 2 / second
    assumptions = {'real': True}
    latex_name = 'D_{va}'
    default = None


class g(Variable):
    """Gravitational acceleration."""

    name = 'g'
    unit = meter / second ** 2
    assumptions = {'real': True}
    latex_name = 'g'
    default = 9.81


class Gr(Variable):
    """Grashof number."""

    name = 'Gr'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'N_{Gr_L}'
    default = None


class h_c(Variable):
    """Average 1-sided convective heat transfer coefficient."""

    name = 'h_c'
    unit = joule / (kelvin * meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'h_c'
    default = None


class k_a(Variable):
    """Thermal conductivity of dry air."""

    name = 'k_a'
    unit = joule / (kelvin * meter * second)
    assumptions = {'real': True}
    latex_name = 'k_a'
    default = None


class lambda_E(Variable):
    """Latent heat of evaporation."""

    name = 'lambda_E'
    unit = joule / kilogram
    assumptions = {'real': True}
    latex_name = '\\lambda_E'
    default = 2450000.0


class Le(Variable):
    """Lewis number."""

    name = 'Le'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'N_{Le}'
    default = None


class M_air(Variable):
    """Molar mass of air.

    http://www.engineeringtoolbox.com/molecular-mass-air-d_679.html
    """

    name = 'M_air'
    unit = kilogram / mole
    assumptions = {'real': True}
    latex_name = 'M_{air}'
    default = 0.02897


class M_N2(Variable):
    """Molar mass of nitrogen."""

    name = 'M_N2'
    unit = kilogram / mole
    assumptions = {'real': True}
    latex_name = 'M_{N_2}'
    default = 0.028


class M_O2(Variable):
    """Molar mass of oxygen."""

    name = 'M_O2'
    unit = kilogram / mole
    assumptions = {'real': True}
    latex_name = 'M_{O_2}'
    default = 0.032


class M_w(Variable):
    """Molar mass of water."""

    name = 'M_w'
    unit = kilogram / mole
    assumptions = {'real': True}
    latex_name = 'M_w'
    default = 0.018


class nu_a(Variable):
    """Kinematic viscosity of dry air."""

    name = 'nu_a'
    unit = meter ** 2 / second
    assumptions = {'real': True}
    latex_name = '\\nu_a'
    default = None


class Nu(Variable):
    """Average Nusselt number over given length."""

    name = 'Nu'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'N_{Nu_L}'
    default = None


class P_a(Variable):
    """Air pressure."""

    name = 'P_a'
    unit = pascal
    assumptions = {'real': True}
    latex_name = 'P_a'
    default = None


class Pr(Variable):
    """Prandtl number (0.71 for air)."""

    name = 'Pr'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'N_{Pr}'
    default = None


class P_N2(Variable):
    """Partial pressure of nitrogen."""

    name = 'P_N2'
    unit = pascal
    assumptions = {'real': True}
    latex_name = 'P_{N2}'
    default = None


class P_O2(Variable):
    """Partial pressure of oxygen."""

    name = 'P_O2'
    unit = pascal
    assumptions = {'real': True}
    latex_name = 'P_{O2}'
    default = None


class P_wa(Variable):
    """Water vapour pressure in the atmosphere."""

    name = 'P_wa'
    unit = pascal
    assumptions = {'real': True}
    latex_name = 'P_{wa}'
    default = None


class P_was(Variable):
    """Saturation water vapour pressure at air temperature."""

    name = 'P_was'
    unit = pascal
    assumptions = {'real': True}
    latex_name = 'P_{was}'
    default = None


class R_d(Variable):
    """Downwelling global radiation."""

    name = 'R_d'
    unit = watt / meter ** 2
    assumptions = {'real': True}
    latex_name = 'R_d'
    default = None


class Re_c(Variable):
    """Critical Reynolds number for the onset of turbulence."""

    name = 'Re_c'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'N_{Re_c}'
    default = None


class Re(Variable):
    """Average Reynolds number over given length."""

    name = 'Re'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'N_{Re_L}'
    default = None


class rho_a(Variable):
    """Density of dry air."""

    name = 'rho_a'
    unit = kilogram / meter ** 3
    assumptions = {'real': True}
    latex_name = '\\rho_a'
    default = None


class R_u(Variable):
    """Upwelling global radiation."""

    name = 'R_u'
    unit = watt / meter ** 2
    assumptions = {'real': True}
    latex_name = 'R_u'
    default = None


class R_mol(Variable):
    """Molar gas constant."""

    name = 'R_mol'
    unit = joule / (kelvin * mole)
    assumptions = {'real': True}
    latex_name = 'R_{mol}'
    default = 8.314472


class R_s(Variable):
    """Solar shortwave flux per area."""

    name = 'R_s'
    unit = joule / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'R_s'
    default = None


class sigm(Variable):
    """Stefan-Boltzmann constant."""

    name = 'sigm'
    unit = joule / (kelvin ** 4 * meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = '\\sigma'
    default = 5.67e-08


class T0(Variable):
    """Freezing point in Kelvin."""

    name = 'T0'
    unit = kelvin
    assumptions = {'real': True}
    latex_name = 'T_0'
    default = 273.15


class T_a(Variable):
    """Air temperature."""

    name = 'T_a'
    unit = kelvin
    assumptions = {'real': True}
    latex_name = 'T_a'
    default = None


class v_w(Variable):
    """Wind velocity."""

    name = 'v_w'
    unit = meter / second
    assumptions = {'real': True}
    latex_name = 'v_w'
    default = None


class x_N2(Variable):
    """Mole fraction of nitrogen in dry air."""

    name = 'x_N2'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'x_{N2}'
    default = 0.79


class x_O2(Variable):
    """Mole fraction of oxygen in dry air."""

    name = 'x_O2'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'x_{O2}'
    default = 0.21


class alpha_l(Variable):
    """Leaf albedo, fraction of shortwave radiation reflected by the leaf."""

    name = 'alpha_l'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\alpha_l'
    default = None


class a_s(Variable):
    """Fraction of one-sided leaf area covered by stomata.

    (1 if stomata are on one side only, 2 if they are on both sides).
    """

    name = 'a_s'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'a_s'
    default = None


class a_sh(Variable):
    """Fraction of projected area exchanging sensible heat with the air."""

    name = 'a_sh'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'a_{sh}'
    default = 2.0


class C_wl(Variable):
    """Concentration of water in the leaf air space."""

    name = 'C_wl'
    unit = mole / meter ** 3
    assumptions = {'real': True}
    latex_name = 'C_{wl}'
    default = None


class E_lmol(Variable):
    """Transpiration rate in molar units."""

    name = 'E_lmol'
    unit = mole / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'E_{l,mol}'
    default = None


class E_l(Variable):
    """Latent heat flux from leaf."""

    name = 'E_l'
    unit = joule / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'E_l'
    default = None


class epsilon_l(Variable):
    """Longwave emmissivity of the leaf surface."""

    name = 'epsilon_l'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\epsilon_l'
    default = 1.0


class g_bw(Variable):
    """Boundary layer conductance to water vapour."""

    name = 'g_bw'
    unit = meter / second
    assumptions = {'real': True}
    latex_name = 'g_{bw}'
    default = None


class g_bwmol(Variable):
    """Boundary layer conductance to water vapour."""

    name = 'g_bwmol'
    unit = mole / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'g_{bw,mol}'
    default = None


class g_sw(Variable):
    """Stomatal conductance to water vapour."""

    name = 'g_sw'
    unit = meter / second
    assumptions = {'real': True}
    latex_name = 'g_{sw}'
    default = None


class g_swmol(Variable):
    """Stomatal conductance to water vapour."""

    name = 'g_swmol'
    unit = mole / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'g_{sw,mol}'
    default = None


class g_tw(Variable):
    """Total leaf conductance to water vapour."""

    name = 'g_tw'
    unit = meter / second
    assumptions = {'real': True}
    latex_name = 'g_{tw}'
    default = None


class g_twmol(Variable):
    """Total leaf layer conductance to water vapour."""

    name = 'g_twmol'
    unit = mole / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'g_{tw,mol}'
    default = None


class H_l(Variable):
    """Sensible heat flux from leaf."""

    name = 'H_l'
    unit = joule / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'H_l'
    default = None


class L_A(Variable):
    """Leaf area."""

    name = 'L_A'
    unit = meter ** 2
    assumptions = {'real': True}
    latex_name = 'L_A'
    default = None


class L_l(Variable):
    """Leaf width as characteristic length scale for convection."""

    name = 'L_l'
    unit = meter
    assumptions = {'real': True}
    latex_name = 'L_l'
    default = None


class P_wl(Variable):
    """Water vapour pressure inside the leaf."""

    name = 'P_wl'
    unit = pascal
    assumptions = {'real': True}
    latex_name = 'P_{wl}'
    default = None


class r_bw(Variable):
    """Boundary layer resistance to water vapour, inverse of $g_{bw}$."""

    name = 'r_bw'
    unit = second / meter
    assumptions = {'real': True}
    latex_name = 'r_{bw}'
    default = None


class r_sw(Variable):
    """Stomatal resistance to water vapour, inverse of $g_{sw}$."""

    name = 'r_sw'
    unit = second / meter
    assumptions = {'real': True}
    latex_name = 'r_{sw}'
    default = None


class r_tw(Variable):
    """Total leaf resistance to water vapour, $r_{bv} + r_{sv}$."""

    name = 'r_tw'
    unit = second / meter
    assumptions = {'real': True}
    latex_name = 'r_{tw}'
    default = None


class rho_al(Variable):
    """Density of air at the leaf surface."""

    name = 'rho_al'
    unit = kilogram / meter ** 3
    assumptions = {'real': True}
    latex_name = '\\rho_{al}'
    default = None


class R_la(Variable):
    """Longwave radiation absorbed by leaf."""

    name = 'R_la'
    unit = watt / meter ** 2
    assumptions = {'real': True}
    latex_name = 'R_{la}'
    default = None


class R_ll(Variable):
    """Longwave radiation away from leaf."""

    name = 'R_ll'
    unit = watt / meter ** 2
    assumptions = {'real': True}
    latex_name = 'R_{ll}'
    default = None


class R_ld(Variable):
    """Downwards emitted/reflected global radiation from leaf."""

    name = 'R_ld'
    unit = watt / meter ** 2
    assumptions = {'real': True}
    latex_name = 'R_{ld}'
    default = None


class R_lu(Variable):
    """Upwards emitted/reflected global radiation from leaf."""

    name = 'R_lu'
    unit = watt / meter ** 2
    assumptions = {'real': True}
    latex_name = 'R_{lu}'
    default = None


class T_l(Variable):
    """Leaf temperature."""

    name = 'T_l'
    unit = kelvin
    assumptions = {'real': True}
    latex_name = 'T_l'
    default = None


class T_w(Variable):
    """Radiative temperature of objects surrounding the leaf."""

    name = 'T_w'
    unit = kelvin
    assumptions = {'real': True}
    latex_name = 'T_w'
    default = None


class mu_a(Variable):
    """Molar flow rate of air out of the chamber"""

    name = 'mu_a'
    unit = mole / second
    assumptions = {'real': True}
    latex_name = '\\mu_a'
    default = None


class mu_abar(Variable):
    """Average molar flow rate of air out of the chamber"""

    name = 'mu_abar'
    unit = mole / second
    assumptions = {'real': True}
    latex_name = '\\overline{\\mu_a}'
    default = None


class mu_o(Variable):
    """Molar flow rate of air entering the chamber"""

    name = 'mu_o'
    unit = mole / second
    assumptions = {'real': True}
    latex_name = '\\mu_o'
    default = None


class mu_obar(Variable):
    """Average molar flow rate of air entering the chamber"""

    name = 'mu_obar'
    unit = mole / second
    assumptions = {'real': True}
    latex_name = '\\overline{\\mu_o}'
    default = None


class c_o(Variable):
    """Mole fraction of CO2 in incoming air"""

    name = 'c_o'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'c_o'
    default = None


class c_obar(Variable):
    """Average mole fraction of CO2 in incoming air"""

    name = 'c_obar'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\overline{c_o}'
    default = None


class c_a(Variable):
    """Mole fraction of CO2 in chamber air"""

    name = 'c_a'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'c_a'
    default = None


class c_abar(Variable):
    """Average mole fraction of CO2 in chamber air"""

    name = 'c_abar'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\overline{c_a}'
    default = None


class c_a1(Variable):
    """Mole fraction of CO2 in chamber air at the beginning"""

    name = 'c_a1'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'c_{a1}'
    default = None


class c_a2(Variable):
    """Mole fraction of CO2 in chamber air at the end"""

    name = 'c_a2'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'c_{a2}'
    default = None


class Delta_ca(Variable):
    """Difference in CO2 mole fractions in chamber"""

    name = 'Delta_ca'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\Delta_{ca}'
    default = None
    expr = -c_a1 + c_a2


class w_o(Variable):
    """Mole fraction of H20 in incoming air"""

    name = 'w_o'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'w_o'
    default = None


class w_obar(Variable):
    """Average mole fraction of H2O in incoming air"""

    name = 'w_obar'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\overline{w_o}'
    default = None


class w_a(Variable):
    """Mole fraction of H20 in chamber air"""

    name = 'w_a'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'w_a'
    default = None


class w_abar(Variable):
    """Average mole fraction of H2O in chamber air"""

    name = 'w_abar'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\overline{w_a}'
    default = None


class w_a1(Variable):
    """Mole fraction of H20 in chamber air at the beginning"""

    name = 'w_a1'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'w_{a1}'
    default = None


class w_a2(Variable):
    """Mole fraction of H20 in chamber air at the end"""

    name = 'w_a2'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'w_{a2}'
    default = None


class Delta_wa(Variable):
    """Difference in H2O mole fractions in chamber"""

    name = 'Delta_wa'
    unit = 1
    assumptions = {'real': True}
    latex_name = '\\Delta_{wa}'
    default = None
    expr = -w_a1 + w_a2


class s(Variable):
    """Leaf area (3.5 cm2 in standard cuvette)"""

    name = 's'
    unit = meter ** 2
    assumptions = {'real': True}
    latex_name = 's'
    default = None


class A(Variable):
    """Net CO2 assimilation per leaf area"""

    name = 'A'
    unit = mole / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'A'
    default = None


class E(Variable):
    """Transpiration (Water released into chamber air) per leaf area"""

    name = 'E'
    unit = mole / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'E'
    default = None


class sE(Variable):
    """Transpiration rate in chamber"""

    name = 'sE'
    unit = mole / second
    assumptions = {'real': True}
    latex_name = 'sE'
    default = None


class sA(Variable):
    """Net CO2 Assimilation in chamber"""

    name = 'sA'
    unit = mole / second
    assumptions = {'real': True}
    latex_name = 'sA'
    default = None


class V_c(Variable):
    """Air volume in chamber"""

    name = 'V_c'
    unit = meter ** 3
    assumptions = {'real': True}
    latex_name = 'V_c'
    default = None


class n_a(Variable):
    """Number of moles of air in chamber"""

    name = 'n_a'
    unit = mole
    assumptions = {'real': True}
    latex_name = 'n_a'
    default = None


class n_ca(Variable):
    """Number of moles of CO2 in chamber"""

    name = 'n_ca'
    unit = mole
    assumptions = {'real': True}
    latex_name = 'n_{ca}'
    default = None


class n_wa(Variable):
    """Number of moles of water vapour in chamber"""

    name = 'n_wa'
    unit = mole
    assumptions = {'real': True}
    latex_name = 'n_{wa}'
    default = None


class t(Variable):
    """Time"""

    name = 't'
    unit = second
    assumptions = {'real': True}
    latex_name = 't'
    default = None


class t1(Variable):
    """Time stamp 1"""

    name = 't1'
    unit = second
    assumptions = {'real': True}
    latex_name = '{t_1}'
    default = None


class t2(Variable):
    """Time stamp 2"""

    name = 't2'
    unit = second
    assumptions = {'real': True}
    latex_name = '{t_2}'
    default = None


class Delta_t(Variable):
    """Time interval"""

    name = 'Delta_t'
    unit = second
    assumptions = {'real': True}
    latex_name = '\\Delta t'
    default = None
    expr = -t1 + t2


class dca_dt(Variable):
    """Rate of change in c_a"""

    name = 'dca_dt'
    unit = 1 / second
    assumptions = {'real': True}
    latex_name = '\\frac{dc_a}{dt}'
    default = None


class dwa_dt(Variable):
    """Rate of change in w_a"""

    name = 'dwa_dt'
    unit = 1 / second
    assumptions = {'real': True}
    latex_name = '\\frac{dw_a}{dt}'
    default = None


class rho_am(Variable):
    """Molar air density"""

    name = 'rho_am'
    unit = mole / meter ** 3
    assumptions = {'real': True}
    latex_name = '\\rho_{am}'
    default = None


class A_dry(Variable):
    """Assimilation rate (dry equivalent)"""

    name = 'A_dry'
    unit = mole / (meter ** 2 * second)
    assumptions = {'real': True}
    latex_name = 'A_{dry}'
    default = None
    expr = A / (1 - w_o)


class c_odry(Variable):
    """Dry mole fraction of CO2 entering the chamber"""

    name = 'c_odry'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'c_{o,dry}'
    default = None
    expr = c_o / (1 - w_o)


class c_adry(Variable):
    """Dry mole fraction of CO2 exiting the chamber"""

    name = 'c_adry'
    unit = 1
    assumptions = {'real': True}
    latex_name = 'c_{a,dry}'
    default = None
    expr = c_a / (1 - w_a)


class dcadry_dt(Variable):
    """Rate of change in c_adry"""

    name = 'dcadry_dt'
    unit = 1 / second
    assumptions = {'real': True}
    latex_name = '\\frac{dc_{a,dry}}{dt}'
    default = None


__all__ = (
    'alpha_a',
    'c_pa',
    'c_pamol',
    'c_pv',
    'C_wa',
    'D_va',
    'g',
    'Gr',
    'h_c',
    'k_a',
    'lambda_E',
    'Le',
    'M_air',
    'M_N2',
    'M_O2',
    'M_w',
    'nu_a',
    'Nu',
    'P_a',
    'Pr',
    'P_N2',
    'P_O2',
    'P_wa',
    'P_was',
    'R_d',
    'Re_c',
    'Re',
    'rho_a',
    'R_u',
    'R_mol',
    'R_s',
    'sigm',
    'T0',
    'T_a',
    'v_w',
    'x_N2',
    'x_O2',
    'alpha_l',
    'a_s',
    'a_sh',
    'C_wl',
    'E_lmol',
    'E_l',
    'epsilon_l',
    'g_bw',
    'g_bwmol',
    'g_sw',
    'g_swmol',
    'g_tw',
    'g_twmol',
    'H_l',
    'L_A',
    'L_l',
    'P_wl',
    'r_bw',
    'r_sw',
    'r_tw',
    'rho_al',
    'R_la',
    'R_ll',
    'R_ld',
    'R_lu',
    'T_l',
    'T_w',
    'mu_a',
    'mu_abar',
    'mu_o',
    'mu_obar',
    'c_o',
    'c_obar',
    'c_a',
    'c_abar',
    'c_a1',
    'c_a2',
    'Delta_ca',
    'w_o',
    'w_obar',
    'w_a',
    'w_abar',
    'w_a1',
    'w_a2',
    'Delta_wa',
    's',
    'A',
    'E',
    'sE',
    'sA',
    'V_c',
    'n_a',
    'n_ca',
    'n_wa',
    't',
    't1',
    't2',
    'Delta_t',
    'dca_dt',
    'dwa_dt',
    'rho_am',
    'A_dry',
    'c_odry',
    'c_adry',
    'dcadry_dt',
)
