# -*- coding: utf-8 -*-
#
# This file is for use with essm.
# Copyright (C) 2023 ETH Zurich, Swiss Data Science Center.
#
# essm is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# essm is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with essm; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA 02111-1307, USA.
"""Equations defined in Mathematical_model.ipynb and dependencies."""

from __future__ import division

from __main__ import (
    A, A_dry, Delta_ca, Delta_t, Delta_wa, E, V_c, c_a, c_abar, c_adry, c_o,
    c_obar, c_odry, dca_dt, dcadry_dt, dwa_dt, mu_a, mu_abar, mu_o, mu_obar,
    n_a, n_wa, rho_am, s, sA, sE, t, t1, t2, w_a, w_abar, w_o, w_obar
)
from essm import Eq
from essm.equations import Equation
from essm.variables import Variable
from essm.variables.leaf.energy_water import (
    C_wl, E_l, E_lmol, Gr, H_l, L_l, P_wl, R_ll, T_l, T_w, a_s, a_sh,
    epsilon_l, g_bw, g_sw, g_tw, g_twmol, h_c, rho_al
)
from essm.variables.physics.thermodynamics import (
    C_wa, Le, M_w, Nu, P_a, P_wa, R_mol, R_s, Re, T_a, c_pa, g, k_a, lambda_E,
    nu_a, rho_a, sigm, v_w
)
from sympy import Derivative, Eq, Integral, exp
from sympy.physics.units import kelvin, pascal


class eq_Rs_enbal(Equation):
    """Calculate R_s from energy balance.

    (Eq. 1 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(R_s, E_l + H_l + R_ll)


class eq_Rll(Equation):
    """R_ll as function of T_l and T_w.

    (Eq. 2 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(R_ll, a_sh * epsilon_l * sigm * (T_l ** 4 - T_w ** 4))


class eq_Hl(Equation):
    """H_l as function of T_l.

    (Eq. 3 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(H_l, a_sh * h_c * (-T_a + T_l))


class eq_El(Equation):
    """E_l as function of E_lmol.

    (Eq. 4 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(E_l, E_lmol * M_w * lambda_E)


class eq_Elmol(Equation):
    """E_lmol as functino of g_tw and C_wl.

    (Eq. 5 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(E_lmol, g_tw * (-C_wa + C_wl))


class eq_gtw(Equation):
    """g_tw from g_sw and g_bw.

    (Eq. 6 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(g_tw, 1 / (1 / g_sw + 1 / g_bw))


class eq_gbw_hc(Equation):
    """g_bw as function of h_c.

    (Eq. B2 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(g_bw, a_s * h_c / (Le ** (2 / 3) * c_pa * rho_a))


class eq_Cwl(Equation):
    """C_wl as function of P_wl and T_l.

    (Eq. B4 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(C_wl, P_wl / (R_mol * T_l))


class eq_Pwl(Equation):
    """Clausius-Clapeyron P_wl as function of T_l.

    (Eq. B3 in :cite:`hartmann_global_1994`)
    """

    class p_CC1(Variable):
        """Internal parameter of eq_Pwl."""

        name = 'p_CC1'
        unit = pascal
        assumptions = {'real': True}
        latex_name = 'p_1'
        default = 611.0

    class p_CC2(Variable):
        """Internal parameter of eq_Pwl."""

        name = 'p_CC2'
        unit = kelvin
        assumptions = {'real': True}
        latex_name = 'p_2'
        default = 273.0

    expr = Eq(
        P_wl, p_CC1 * exp(-M_w * lambda_E * (-1 / p_CC2 + 1 / T_l) / R_mol)
    )


class eq_Elmol_conv(Equation):
    """E_lmol as function of g_twmol and P_wl.

    (Eq. B6 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(E_lmol, g_twmol * (-P_wa + P_wl) / P_a)


class eq_hc(Equation):
    """h_c as a function of Nu and L_l.

    (Eq. B10 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(h_c, Nu * k_a / L_l)


class eq_Re(Equation):
    """Re as a function of v_w and L_l.

    (Eq. B11 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(Re, L_l * v_w / nu_a)


class eq_Gr(Equation):
    """Gr as function of air density within and outside of leaf.

    (Eq. B12 in :cite:`schymanski_leaf-scale_2017`)
    """

    expr = Eq(Gr, L_l ** 3 * g * (rho_a - rho_al) / (nu_a ** 2 * rho_al))


class eq_na(Equation):
    """Number of moles of air in chamber"""

    expr = Eq(n_a, P_a * V_c / (R_mol * T_a))


class eq_mu_a(Equation):
    """Flow rate going out of the chamber, Eq. 9-1 in handbook."""

    expr = Eq(mu_a, mu_o + sE)


class eq_sE(Equation):
    """Transpiration rate in chamber, Eq 9-2 in handbook."""

    expr = Eq(sE, mu_a * w_a - mu_o * w_o)


class eq_E(Equation):
    """
    Transpiration per leaf area, by re-arranging eq_mu_a and eq_sE.
    Eq. 9-3 in hanndbook.
    """

    expr = Eq(E, mu_o * (w_a - w_o) / (s * (1 - w_a)))


class eq_sA(Equation):
    """ 
    Net CO2 assimilation in chamber, Eq. 9-4 in handbook.
    """

    expr = Eq(sA, -c_a * mu_a + c_o * mu_o)


class eq_A(Equation):
    """
    Net CO2 assimilation per area, with mu_o eliminated.
    Eq. 9-5 in handbook."""

    expr = Eq(A, mu_o * (-c_a * (1 - w_o) / (1 - w_a) + c_o) / s)


class eq_mua_sE_sA(Equation):
    """Outgoing air as a function of sE and sA"""

    expr = Eq(mu_a, mu_o - sA + sE)


class eq_sA_dt(Equation):
    """sA as a function of dca_dt, assuming everything else constant."""

    expr = Eq(sA, -c_a * mu_a + c_o * mu_o - dca_dt * n_a)


class eq_sE_dt(Equation):
    """sE as a function of dwa_dt, assuming everything else constant."""

    expr = Eq(sE, dwa_dt * n_a + mu_a * w_a - mu_o * w_o)


class eq_sE_mb(Equation):
    """Mass balance of water for air flowing through a single-volume, well mixed chamber, Eq. 9-103"""

    expr = Eq(sE, V_c * dwa_dt * rho_am + mu_a * w_a - mu_o * w_o)


class eq_E_dyn(Equation):
    """Transpiration rate, Eq. 9-104"""

    expr = Eq(
        E, (-V_c * dwa_dt * rho_am + mu_o * (-w_a + w_o)) / (s * (w_a - 1))
    )


class eq_Adry(Equation):
    """Assimilation dry rate, Eq. 9-105"""

    expr = Eq(A_dry * s, V_c * dcadry_dt * rho_am + mu_o * (-c_adry + c_odry))


class eq_A_dyn(Equation):
    """Assimilation rate, Eq. 9-106"""

    expr = Eq(
        A,
        mu_o * (1 - w_o) * (V_c * dcadry_dt * rho_am / mu_o - c_adry + c_odry)
        / s
    )


class eq_Cbalance(Equation):
    """Chamber CO2 balance between t1 and t2, assuming constant sA"""

    expr = Eq(
        Delta_ca * n_a,
        -Delta_t * sA + Integral(-c_a * mu_a + c_o * mu_o, (t, t1, t2))
    )


class eq_sA_int(Equation):
    """Average flux over time period"""

    expr = Eq(
        sA,
        (-Delta_ca * n_a + Integral(-c_a * mu_a + c_o * mu_o,
                                    (t, t1, t2))) / Delta_t
    )


class eq_sE_avg(Equation):
    """Average H2O flux over time period"""

    expr = Eq(
        sE, mu_abar * w_abar - mu_obar * w_obar + Delta_wa * n_a / Delta_t
    )


class eq_gtwmol_gtw(eq_Elmol_conv.definition, eq_Elmol.definition,
                    eq_Cwl.definition):
    """g_twmol as a function of g_tw.

    It uses eq_Elmol, eq_Cwl and eq_Elmol_conv.
    """

    expr = Eq(
        g_twmol,
        g_tw * (P_a * P_wa * T_l - P_a * P_wl * T_a) /
        (R_mol * T_a * T_l * (P_wa - P_wl))
    )


class eq_gtwmol_gtw_iso(eq_gtwmol_gtw.definition):
    """g_twmol as a function of g_tw at isothermal conditions."""

    expr = Eq(g_twmol, P_a * g_tw / (R_mol * T_a))


class eq_nwa(eq_na.definition):
    """Number of moles of water vapour in chamber"""

    expr = Eq(n_wa, P_a * V_c * w_a / (R_mol * T_a))


class eq_Pwa(eq_nwa.definition):
    """Vapour pressure in chamber"""

    expr = Eq(P_wa, P_a * w_a)


class eq_sE1(eq_sE.definition, eq_mu_a.definition):
    """
    Transpiration per leaf area, deduced from eq_sE 
    and eq_mu_a"""

    expr = Eq(sE, mu_o * (-w_a + w_o) / (w_a - 1))


class eq_mu_a1(eq_sE.definition, eq_mu_a.definition):
    """
    Flow out of the chamber, with sE eliminated.
    """

    expr = Eq(mu_a, mu_o * (w_o - 1) / (w_a - 1))


class eq_sA1(eq_mu_a1.definition, eq_sA.definition):
    """
    Total assimilation in the chamber, equivalent to Eq. 9-5 in handbook.
    """

    expr = Eq(sA, -c_a * mu_o * (w_o - 1) / (w_a - 1) + c_o * mu_o)


class eq_mua_c_a(eq_sE.definition, eq_sA.definition, eq_mua_sE_sA.definition):
    """Rate of air flow out of chamber, considering sE and sA"""

    expr = Eq(mu_a, (c_o * mu_o + mu_o * w_o - mu_o) / (c_a + w_a - 1))


class eq_sA_c_a(eq_sE.definition, eq_sA.definition, eq_mua_sE_sA.definition):
    """
    Net CO2 assimilation in chamber, 
    cosidering contributions of sE and sA to mu_o.
    """

    expr = Eq(
        sA, (-c_a * mu_o * w_o + c_a * mu_o + c_o * mu_o * w_a - c_o * mu_o) /
        (c_a + w_a - 1)
    )


class eq_sE_c_a(eq_sE.definition, eq_sA.definition, eq_mua_sE_sA.definition):
    """
    Net water vapour release in chamber, 
    cosidering contributions of sE and sA to mu_o.
    """

    expr = Eq(
        sE, (-c_a * mu_o * w_o + c_o * mu_o * w_a - mu_o * w_a + mu_o * w_o) /
        (c_a + w_a - 1)
    )


class eq_mua_dt1(eq_sE_dt.definition, eq_mua_sE_sA.definition,
                 eq_sA_dt.definition):
    """Non-steady state rate of air flow out of chamber"""

    expr = Eq(
        mu_a, (
            -P_a * V_c * dca_dt - P_a * V_c * dwa_dt + R_mol * T_a * c_o * mu_o
            + R_mol * T_a * mu_o * w_o - R_mol * T_a * mu_o
        ) / (R_mol * T_a * c_a + R_mol * T_a * w_a - R_mol * T_a)
    )


class eq_sA_dt1(eq_sE_dt.definition, eq_mua_sE_sA.definition,
                eq_sA_dt.definition):
    """Non-steady state rate of CO2 removal from chamber"""

    expr = Eq(
        sA, (
            P_a * V_c * c_a * dwa_dt - P_a * V_c * dca_dt * w_a + P_a * V_c *
            dca_dt - R_mol * T_a * c_a * mu_o * w_o + R_mol * T_a * c_a * mu_o
            + R_mol * T_a * c_o * mu_o * w_a - R_mol * T_a * c_o * mu_o
        ) / (R_mol * T_a * c_a + R_mol * T_a * w_a - R_mol * T_a)
    )


class eq_sE_dt1(eq_sE_dt.definition, eq_mua_sE_sA.definition,
                eq_sA_dt.definition):
    """Non-steady state rate of water vapour addition to chamber"""

    expr = Eq(
        sE, (
            P_a * V_c * c_a * dwa_dt - P_a * V_c * dca_dt * w_a -
            P_a * V_c * dwa_dt - R_mol * T_a * c_a * mu_o * w_o +
            R_mol * T_a * c_o * mu_o * w_a - R_mol * T_a * mu_o * w_a +
            R_mol * T_a * mu_o * w_o
        ) / (R_mol * T_a * c_a + R_mol * T_a * w_a - R_mol * T_a)
    )


class eq_sA_avg(eq_sA_int.definition):
    """Average flux over time period"""

    expr = Eq(
        sA, -Delta_ca * n_a / Delta_t - c_abar * mu_abar + c_obar * mu_obar
    )


class eq_mua_avg1(eq_sE_avg.definition, eq_sA_avg.definition, eq_na.definition,
                  eq_mua_sE_sA.definition):
    """Average outflow of air from chamber over time period"""

    expr = Eq(
        mu_abar, (
            -Delta_ca * P_a * V_c + Delta_t * (
                R_mol * T_a * c_obar * mu_obar + R_mol * T_a * mu_obar * w_obar
                - R_mol * T_a * mu_obar
            ) - Delta_wa * P_a * V_c
        ) / (
            Delta_t *
            (R_mol * T_a * c_abar + R_mol * T_a * w_abar - R_mol * T_a)
        )
    )


class eq_sA_avg1(eq_sE_avg.definition, eq_sA_avg.definition, eq_na.definition,
                 eq_mua_sE_sA.definition):
    """Average CO2 flux over time period, with mu_a and n_a eliminated"""

    expr = Eq(
        sA, (
            -Delta_ca * P_a * V_c * w_abar + Delta_ca * P_a * V_c + Delta_t * (
                -R_mol * T_a * c_abar * mu_obar * w_obar + R_mol * T_a * c_abar
                * mu_obar + R_mol * T_a * c_obar * mu_obar * w_abar -
                R_mol * T_a * c_obar * mu_obar
            ) + Delta_wa * P_a * V_c * c_abar
        ) / (
            Delta_t *
            (R_mol * T_a * c_abar + R_mol * T_a * w_abar - R_mol * T_a)
        )
    )


class eq_sE_avg1(eq_sE_avg.definition, eq_sA_avg.definition, eq_na.definition,
                 eq_mua_sE_sA.definition):
    """Average H2O flux over time period, with mu_a and n_a eliminated"""

    expr = Eq(
        sE, (
            -Delta_ca * P_a * V_c * w_abar + Delta_t * (
                -R_mol * T_a * c_abar * mu_obar * w_obar +
                R_mol * T_a * c_obar * mu_obar * w_abar -
                R_mol * T_a * mu_obar * w_abar + R_mol * T_a * mu_obar * w_obar
            ) + Delta_wa * P_a * V_c * c_abar - Delta_wa * P_a * V_c
        ) / (
            Delta_t *
            (R_mol * T_a * c_abar + R_mol * T_a * w_abar - R_mol * T_a)
        )
    )


__all__ = (
    'eq_Rs_enbal',
    'eq_Rll',
    'eq_Hl',
    'eq_El',
    'eq_Elmol',
    'eq_gtw',
    'eq_gbw_hc',
    'eq_Cwl',
    'eq_Pwl',
    'eq_Elmol_conv',
    'eq_hc',
    'eq_Re',
    'eq_Gr',
    'eq_na',
    'eq_mu_a',
    'eq_sE',
    'eq_E',
    'eq_sA',
    'eq_A',
    'eq_mua_sE_sA',
    'eq_sA_dt',
    'eq_sE_dt',
    'eq_sE_mb',
    'eq_E_dyn',
    'eq_Adry',
    'eq_A_dyn',
    'eq_Cbalance',
    'eq_sA_int',
    'eq_sE_avg',
    'eq_gtwmol_gtw',
    'eq_gtwmol_gtw_iso',
    'eq_nwa',
    'eq_Pwa',
    'eq_sE1',
    'eq_mu_a1',
    'eq_sA1',
    'eq_mua_c_a',
    'eq_sA_c_a',
    'eq_sE_c_a',
    'eq_mua_dt1',
    'eq_sA_dt1',
    'eq_sE_dt1',
    'eq_sA_avg',
    'eq_mua_avg1',
    'eq_sA_avg1',
    'eq_sE_avg1',
)
