# -*- coding: utf-8 -*-
#
# This file is for use with essm.
# Copyright (C) 2022 ETH Zurich, Swiss Data Science Center.
#
# essm is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# essm is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with essm; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA 02111-1307, USA.
"""Equations defined in notebooks/extra_calculations/Intrinsic_permeability.ipynb and dependencies."""

from __future__ import division

from __main__ import A, K, L, Q_m, Q_v, dP, k, mu, rho
from essm import Eq
from essm.equations import Equation
from sympy import Eq, Integral


class eq_Qm_sperry(Equation):
    """
    Flow as a function of conductivity, pressure drop, cross-sectional area
    and twig length (deduced from Sperry et al. 1988)
    """

    expr = Eq(Q_m, -A * K * dP / L)


class eq_Qv_darcy(Equation):
    """Darcy's Law"""

    expr = Eq(Q_v, -A * dP * k / (L * mu))


class eq_Qv_sperry(Equation):
    """
    Volumetric flow as a function of conductivity, pressure drop,
    cross-sectional and twig length, deduced from Sperry et al. (1988).
    """

    expr = Eq(Q_v, -A * K * dP / (L * rho))


class eq_kK(Equation):
    """Permeability k as a function of conductivity K derived from Darcy's law and Sperry et al. (1988)"""

    expr = Eq(k, K * mu / rho)


__all__ = (
    'eq_Qm_sperry',
    'eq_Qv_darcy',
    'eq_Qv_sperry',
    'eq_kK',
)
